# Soya 3D
# Copyright (C) 2001-2002 Jean-Baptiste LAMY -- jiba@tuxfamily.org
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import weakref
import _soya, soya, math, soya.widget
from math3d import Point, Vector

Element3D = _soya.Element3D

class GraphicElement(object):
  """GraphicElement

Base class for all graphical 3D elements."""
  
  def __add__(self, vector):
    """GraphicElement + Vector -> Point

Translates a GraphicElement's position and returns the result (as a new Point).
Coordinates system conversion is performed if needed (=if the GraphicElement and
VECTOR are not defined in the same coordinates system)."""
    if (not vector.parent) or (not self.parent) or (self.parent is vector.parent):
      return Point(self.parent, self.x + vector.x, self.y + vector.y, self.z + vector.z)
    else:
      x, y, z = self.parent.transform_vector(vector.x, vector.y, vector.z, vector.parent)
      return Point(self.parent, self.x + x, self.y + y, self.z + z)
  def __sub__(self, vector):
    """GraphicElement - Vector -> Point

Translates a GraphicElement's position and returns the result (as a new Point).
Coordinates system conversion is performed if needed (=if the GraphicElement and
VECTOR are not defined in the same coordinates system)."""
    if (not vector.parent) or (not self.parent) or (self.parent is vector.parent):
      return Point(self.parent, self.x - vector.x, self.y - vector.y, self.z - vector.z)
    else:
      x, y, z = self.parent.transform_vector(vector.x, vector.y, vector.z, vector.parent)
      return Point(self.parent, self.x - x, self.y - y, self.z - z)
    
  def __mod__(self, coordsyst):
    """GraphicElement % coordsyst -> Point

Converts a GraphicElement to the coordinates system COORDSYST and returns
the result as a Point.
The returned value may be the same GraphicElement if its coordinates system is
already COORDSYST, so you should be carreful if you modify it."""
    if (not self.parent) or (not coordsyst) or (self.parent is coordsyst): return self
    p = Point(self.parent, self.x, self.y, self.z)
    p.convert_to(coordsyst)
    return p
  
  def position(self):
    """GraphicElement.position() -> Point

Returns the position of a GraphicElement, as a new Point."""
    return Point(self.parent, self.x, self.y, self.z)
  
  def move(self, position):
    """GraphicElement.move(position)

Moves a GraphicElement to POSITION (a Point or another 3D object, such as
a World, a volume,...).
Coordinates system conversion is performed is needed (=if the GraphicElement
and POSITION are not defined in the same coordinates system)."""
    if (not position.parent) or (not self.parent) or (self.parent is position.parent):
      self.set_xyz(position.x, position.y, position.z)
    else:
      x, y, z = self.parent.transform_point(position.x, position.y, position.z, position.parent)
      self.set_xyz(x, y, z)

    #position = position % self.parent
    #self.set_xyz(position.x, position.y, position.z)
    
  def add_vector(self, vector):
    """GraphicElement.add_vector(vector)

Translates a GraphicElement IN PLACE.
Coordinates system conversion is performed is needed (=if the GraphicElement and
POSITION are not defined in the same coordinates system)."""
    if (not vector.parent) or (not self.parent) or (self.parent is vector.parent):
      self.translate(vector.x, vector.y, vector.z)
    else:
      x, y, z = self.parent.transform_vector(vector.x, vector.y, vector.z, vector.parent)
      self.translate(x, y, z)
      
    #vector = vector % self.parent
    #self.set_xyz(self.x + vector.x, self.y + vector.y, self.z + vector.z)
    return self
  __iadd__ = add_vector
  
  def add_mul_vector(self, k, vector):
    """GraphicElement.add_vector(vector)

Translates a GraphicElement IN PLACE, by K * VECTOR.
Coordinates system conversion is performed is needed (=if the GraphicElement and
POSITION are not defined in the same coordinates system)."""
    if (not vector.parent) or (not self.parent) or (self.parent is vector.parent):
      self.translate(k * vector.x, k * vector.y, k * vector.z)
    else:
      x, y, z = self.parent.transform_vector(vector.x, vector.y, vector.z, vector.parent)
      self.translate(k * x, k * y, k * z)
    return self
  
#  def distance_to(self, other):
#    """GraphicElement.distance_to(other) -> float
#
#Gets the distance between a GraphicElement and OTHER."""
    #if (not other.parent) or (not self.parent) or (self.parent is other.parent):
    #  return math.sqrt((self.x - other.x) ** 2 + (self.y - other.y) ** 2 + (self.z - other.z) ** 2)
    #else:
    #  x, y, z = self.parent.transform_point(other.x, other.y, other.z, other.parent)
    #  return math.sqrt((self.x - x) ** 2 + (self.y - y) ** 2 + (self.z - z) ** 2)
    
    #other = other % self.parent
    #return math.sqrt((self.x - other.x) ** 2 + (self.y - other.y) ** 2 + (self.z - other.z) ** 2)
    
  def vector_to(self, other):
    """GraphicElement.vector_to(other) -> Vector

Gets the vector that starts at a GraphicElement and ends at OTHER."""
    if (not other.parent) or (not self.parent) or (self.parent is other.parent):
      return Vector(self.parent, other.x - self.x, other.y - self.y, other.z - self.z)
    else:
      x, y, z = self.parent.transform_point(other.x, other.y, other.z, other.parent)
      return Vector(self.parent, x - self.x, y - self.y, z - self.z)
    
    #other = other % self.parent
    #return Vector(self.parent, other.x - self.x, other.y - self.y, other.z - self.z)
  __rshift__ = vector_to
  
  def begin_round(self): pass
  def advance_time(self, proportion = 1.0): pass
  def end_round(self): pass

#   def look_at(self, pos):
#     """GraphicElement.look_at(pos)

# Rotates a GraphicElement so as it looks toward POS. The GraphicElement is
# understood to "look" at its -Z direction."""
#     if isinstance(pos, Vector):
#       if pos.parent and not (pos.parent is self.parent):
#         x, y, z = self.parent.transform_vector(pos.x, pos.y, pos.z, pos.parent)
#       else:
#         x = pos.x
#         y = pos.y
#         z = pos.z
#       #print "look to vector", x, y, z, "was", pos.x, pos.y, pos.z
#       self.look_at_xyz(x + self.x, y + self.y, z + self.z)
#     else:
#       if pos.parent and not (pos.parent is self.parent):
#         x, y, z = self.parent.transform_point(pos.x, pos.y, pos.z, pos.parent)
#       else:
#         x = pos.x
#         y = pos.y
#         z = pos.z
#       self.look_at_xyz(x, y, z)
      
#     pos = pos % self.parent
#     if isinstance(pos, Vector):
#       vec = pos
#       pos = self.position()
#       pos += vec
#     self.look_at_xyz(pos.x, pos.y, pos.z)
    
  def inside(self, parent):
    return (parent is self) or (self.parent and self.parent.inside(parent))
    #return (parent is self.parent) or (self.parent and self.parent.inside(parent))
    


class Camera(soya._CObj, soya.widget.Widget, _soya._Camera):
#class Camera(soya._CObj, GraphicElement, soya.widget.Widget, _soya._Camera):

  """Camera

A Camera is a 3D element that can be used to view something in the 3D scene."""
  def __init__(self, parent = None, name = ""):
    """Camera(parent = None, name = "") -> Camera

Creates a new Camera in World PARENT, named NAME."""
    _soya._Camera.__init__(self)
    soya.widget.Widget.__init__(self)
    self.name = name
    self.master = None
    self.resize_style = soya.widget.WIDGET_RESIZE_MAXIMIZE
    
    if parent: parent.add(self)

  def __repr__(self):
    if self.name: return "<Camera %s>" % self.name
    return "<Camera>"
  
  get_screen_width  = staticmethod(soya.get_screen_width)
  get_screen_height = staticmethod(soya.get_screen_height)
  

class Portal(soya._CObj, _soya._Portal):
#class Portal(soya._CObj, GraphicElement, _soya._Portal):

  """Portal

A Portal is a 3D element that shows a part of another World (its "beyond"
attribute).
The "beyond" World is only visible through the
quad (-0.5, -0.5, 0.0) - (0.5, 0.5, 0.0); you should scale and/or rotate
the Portal if you want another quad.
"""
  def __init__(self, parent = None, name = ""):
    """Portal(parent = None, name = "") -> Portal

Creates a new Portal in World PARENT, named NAME."""
    _soya._Portal.__init__(self)
    self.name = name
    
    if parent: parent.add(self)
    
  def __repr__(self):
    if self.name: return "<Portal %s to %s>" % self.name % self.beyond
    return "<Portal to %s>" % self.beyond
  

class Light(soya._CObj, _soya._Light):
#class Light(soya._CObj, GraphicElement, _soya._Light):

  """Light

A Light is a 3D element that cast light around.

The "ambient", "diffuse" and "specular" attributes are the different
colors of the light (all are 4-element tupples)."""
  
  def __init__(self, parent = None, name = ""):
    """Light(parent = None, name = "") -> Light

Creates a new Light in World PARENT, named NAME."""
    _soya._Light.__init__(self)
    self.name  = name
    
    if parent: parent.add(self)
    
  def __repr__(self):
    if self.name: return "<Light %s>" % self.name
    return "<Light>"
  
  def is_directional(self): return self._w == 0.0
  
  def set_directional(self, directional):
    if directional: self._w = 0.0
    else:           self._w = 1.0

  directional = property(is_directional, set_directional)
  
  #def __getstate__(self):
  #  return (self.__dict__, self._getstate())
  #
  #def __setstate__(self, t):
  #  self._setstate(t[1])
  #  self.__dict__ = t[0]

    
class Volume(soya._CObj, _soya._Volume):
#class Volume(soya._CObj, _soya._Volume, GraphicElement):

  """Volume

A Volume is a 3D element that has a visible shape.
The shape can be obtained from a world, with the to_shape() method."""
  
  def __init__(self, parent = None, shape = None, name = ""):
    """Volume(parent = None, shape = None, name = "") -> Volume

Creates a new Volume in World PARENT, shape SHAPE and named NAME."""
    _soya._Volume.__init__(self)
    self.name = name
    
    if shape : self.set_shape(shape)
    if parent: parent.add(self)
    
# class Volume(soya._CObj, GraphicElement, _soya._Coordsys):
#   def __init__(self, parent = None, shape = None, name = ""):
#     _soya._Coordsys.__init__(self)
#     self.name  = name
#     self.shape = shape
    
#     if parent: parent.add(self)

  def set_shape(self, shape):
    self.shape = shape
    
#   def batch(self):
#     return 0
#     #if self.shape: self.shape.batch(self)
#     #self.batch_object(self.shape, self.shape.is_alpha())
#     #_soya.renderer_add(self.shape, self, self.shape.is_alpha())
  
#   def render(self):
#     pass
#     #self.shape.render()
    
  def __repr__(self):
    if self.shape:
      if self.name: return "<%s %s, shape %s>" % (self.__class__.__name__, self.name, repr(self.shape))
      return "<%s, shape %s>" % (self.__class__.__name__, repr(self.shape))
    else:
      if self.name: return "<%s %s, no shape>" % (self.__class__.__name__, self.name)
      return "<%s, no shape>" % self.__class__.__name__
    

## class Morph(soya._CObj, _soya._Morph, GraphicElement): 
##   """Morph

## A Morph is similar to a Volume, but it has a MorphShape instead of a Shape.
## Morphes are typically used for characters,..."""

##   def __init__(self, parent, morphshape, name = ""):
##     """DO NOT call this directly!!!"""
##     if not morphshape: raise TypeError, "MorphShape must not be None"
##     import copy
##     worlds = copy.deepcopy(morphshape.worlds)
##     for w in worlds:
##       if w.parent is None: w._parent = self
##     _soya._Morph.__init__(self, morphshape, worlds)
##     self.name = name
##     if parent: parent.add(self)
    
##     import warnings
##     warnings.warn("Morph are deprecated and might be removed later; use rather Cal3D!", DeprecationWarning, stacklevel = 2)
    
##   def __repr__(self): return "<Morph %s, morphshape %s>" % (self.name, self.data)
  
##   def get_shape(self): return self.data
##   shape = property(get_shape)
  
##   def recursive(self):
##     recursive = self.children[:]
##     for item in self.children:
##       #if isinstance(item, World): recursive.extend(item.recursive())
##       if hasattr(item, "children"): recursive.extend(item.recursive())
##     return recursive
  
##   def __iter__(self): return iter(self.children)
##   def __contains__(self, item): return item in self.children

  
class World(soya.SavedInAPath, soya._CObj, _soya._World):
#class World(soya.SavedInAPath, soya._CObj, GraphicElement, _soya._World):

  """World

A World is a 3D element that can contains other elements.
Elements can be added or removed with the add() and remove() method, and the
children attribute is the list of the children elements.

The World's "atmosphere" attribute can be set to an Atmosphere object to
enable environmental effects (fog, background color, ambient lighting,...).

See tutorial lesson 007 about World, and 107 about Atmosphere."""
  _alls = weakref.WeakValueDictionary() # Required by SavedInAPath
    
  def __init__(self, parent = None, shape = None, name = "", filename = ""):
    """World(parent = None, shape = None, name = "", filename = "") -> World

Creates a new World in World PARENT, shape SHAPE and named NAME.
FILENAME is the name of the file in which the world will be saved, by the save()
method. It MUST NOT be a complete filename (="/home/jiba/machin") but only the
second part of the filename (="machin"), as the file will be placed into the
World.PATH directory."""
    _soya._World.__init__(self)
    soya.SavedInAPath.__init__(self, filename)
    self.name         = name
    self.shapify_args = None
    
    if parent: parent.add(self)
    if shape: self.set_shape(shape)

  def set_shape(self, shape):
    self.shape = shape

  def add(self, child):
    child._parent = self
    self.children.append(child)
  append = add
  
  def insert(self, index, child):
    child._parent = self
    self.children.insert(index, child)

  def remove(self, child):
    self.children.remove(child)
    child._parent = None
  
  def shapify(self):
    """World.shapify() -> Shape

Turns a World into a shape. The exact effect depends on the value of the
"shapify_args" attribute of the World.
If shapify_args is None, a normal shape is returned.
Else, shapify_args must be a 2-element tuple; the first one is a string indicating
the type of shape (currently "normal", "tree" or "morph" only) and the second one is a
dict with options' values.

For "tree", options are:
 - collapse : children sphere must have a radius < collapse x parent radius else child
              and parent are gathered.
 - mode : 0 for fast computation
          1 for slow but most efficient computation
 - params : variable in function of mode:
          0 : 1 param : max children radius expressed in percentage of parent radius
          1 : no param
exemple: ('tree', {})

For "cell-shading", options are:
 - shader : the name of the material used for cell-shading lighting
 - line_color : color of the outline
 - line_width : this is the maximum line width when the object is nearest the camera
exemple: ('cell-shading', { 'shader':'shader2', 'line_color':(0.0, 0.0, 0.0, 1.0), 'line_width':5.0 })

The "shadow" option is supported by normal and tree shape. If set to 1, it enables
the used of static shadows (i.e. with static lighting; dynamic shadows are not
supported yet).
"""

    MESH_PLANE_EQUATION = (1<<14)
    MESH_NEIGHBORS      = (1<<15)
    MESH_SHADOW_CAST    = (1<<21)

    angle = 200.0
    mesh_opt = 0

    if self.shapify_args:
      args = self.shapify_args[1]
      try:
        angle = args['angle']
      except: pass
      try:
        a = args['shadowcast']
        mesh_opt = MESH_PLANE_EQUATION | MESH_NEIGHBORS | MESH_SHADOW_CAST
      except: pass
      
    shape = soya.model.Shape()
    shape.filename = self.filename
    
    if (not self.shapify_args) or (self.shapify_args[0] == "normal"):
      self._to_shape(shape, angle, mesh_opt)
      shape._build_sphere()
      shape._build_display_list()
      
    elif self.shapify_args[0] == "tree":
      self._to_shape(shape, angle, mesh_opt)
      args = self.shapify_args[1]
      shape._build_tree()
      shape._optimize_tree(args.get("collapse", 0.9), args.get("mode"    , 0), *args.get("params"  , [0.5]))
      
    elif self.shapify_args[0] == "cell-shading":
      self._to_shape(shape, angle, mesh_opt | MESH_PLANE_EQUATION | MESH_NEIGHBORS)
      args = self.shapify_args[1]
      try:
        shader = args['shader']
        if isinstance(shader, str): shader = soya.model.Material.get(shader)
      except: shader = None
      try:    line_color = args['line_color']
      except: line_color = (0.0, 0.0, 0.0, 1.0)
      try:    line_width = args['line_width']
      except: line_width = 5.0
      #shape.shadow_cast = 1
      shape._build_face_list()
      shape._build_cell_shading(shader, line_color, line_width)
      shape.lit = 0
       
    if self.shapify_args:
      args = self.shapify_args[1]
      try:
        a = args['shadowcast']
        shape.shadow_cast = 1
        shape._build_face_list()
      except: pass

    return shape
    
  def recursive(self):
    """World.recursive() -> list

Gets a recursive list of all the children elements in a World (=the World
children, + the children of its children + ...)."""
    recursive = self.children[:]
    for item in self.children:
      #if isinstance(item, World): recursive.extend(item.recursive())
      if hasattr(item, "children"): recursive.extend(item.recursive())
    return recursive
  
  def __iter__(self): return iter(self.children)
  def __contains__(self, item): return item in self.children
  def __getitem__(self, name):
    for item in self.children:
      if item.name == name: return item
      elif hasattr(item, "__getitem__"):
        i = item[name]
        if i: return i
        
  def subitem(self, namepath):
    """World.subitem(namepath) -> 3D element

Returns the 3D element denoted by NAMEPATH.
NAMEPATH is a string that contains elemnts' names, separated by ".", such as
"character.head.mouth"."""
    item = self
    for name in namepath.split("."): item = item[name]
    return item
  
  def search(self, regexp):
    """World.search(regexp) -> 3D element

Searches 3D element (recursively) in a World, whose name match REGEXP.
REGEXP must be a regexp object (see module re)."""
    for item in self.children:
      if item.name and regexp.match(item.name): return item
      if isinstance(item, World):
        subresult = item.search(regexp)
        if subresult: return subresult
    return None
  
  def search_all(self, regexp):
    """World.search_all(regexp) -> list of 3D element

Searches all 3D elements (recursively) in a World, whose name match REGEXP.
REGEXP must be a regexp object (see module re)."""
    return filter(lambda item: item.name and regexp.match(item.name), self.recursive())
  
  def __repr__(self):
    name = self.name or self.filename
    if self.shape:
      if name: return "<%s %s, shape %s, %s children>" % (self.__class__.__name__, name, repr(self.shape), len(self.children))
      return "<%s shape %s, %s children>" % (self.__class__.__name__, repr(self.shape), len(self.children))
    else:
      if name: return "<%s %s, %s children>" % (self.__class__.__name__, name, len(self.children))
      return "<%s %s children>" % (self.__class__.__name__, len(self.children))
      
  def save(self, filename = None):
    """SavedInAPath.save(filename = None)

Saves a World. If no FILENAME is given, the object is saved in the path,
using its filename attribute. If FILENAME is given, it is saved at this
location."""
    parent = self.parent # Don't save the parent
    self._parent = None
    
    soya.SavedInAPath.save(self, filename)
    
    self._parent = parent
    
  def begin_round(self):
    for child in self.children:
      child.begin_round()
      
  def advance_time(self, proportion = 1.0):
    for child in self.children: child.advance_time(proportion)
      
  def end_round(self):
    for child in self.children: child.end_round()
      
  def __delitem__(self, index): self.remove(self.children[index])



class Atmosphere(soya._CObj, _soya._Atmosphere):
  """Atmosphere

An Atmosphere is an object that encapsulates environmental properties, such as
background color, fog, ...

Atmosphere can be used with World (see the World.atmosphere attribute)."""
  def __init__(self, ambient = None):
    """Atmosphere(ambient = None) -> Atmosphere

Creates a new Atmosphere. AMBIENT is the ambient lighting values, it should be a
color (=a 4-element tupple)."""
    if ambient: self.ambient = ambient
    

class Sprite(soya._CObj, _soya._Sprite):
#class Sprite(soya._CObj, GraphicElement, _soya._Sprite):

  """Sprite"""
  
  def __init__(self, parent = None, name = ""):
    _soya._Sprite.__init__(self)
    self.name  = name
    
    if parent: parent.add(self)
    
  def __repr__(self):
    if self.name: return "<Sprite %s>" % self.name
    return "<Sprite>"

  
class CylinderSprite(soya._CObj, _soya._Cylinder):
#class CylinderSprite(soya._CObj, GraphicElement, _soya._Cylinder):

  """Cylinder sprite"""
  
  def __init__(self, parent = None, name = ""):
    _soya._Cylinder.__init__(self)
    self.name  = name
    
    if parent: parent.add(self)
    
  def __repr__(self):
    if self.name: return "<Cylinder sprite %s>" % self.name
    return "<Cylinder sprite>"
  
