/*
 * P3
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**********************************************
 * world.c
 * Copyright (C) 2001-2002 Bertrand 'blam' LAMY
 **********************************************/

#include "p3_base.h"
#include "math3d.h"
#include "util.h"
#include "frustum.h"
#include "renderer.h"
#include "atmosphere.h"
#include "coordsys.h"
//#include "mesh.h"
#include "world.h"

extern P3_renderer* renderer;


P3_class P3_class_world = { 
  P3_ID_WORLD,
  (batch_func)     P3_world_batch,
  (render_func)    0,
  (shadow_func)    P3_world_shadow,
  (raypick_func)   P3_world_raypick,
  (raypick_b_func) P3_world_raypick_b,
};


void P3_world_invalid (P3_world* c) {
  int i;
  P3_coordsys_invalid ((P3_coordsys*) c);
  /* invalid children */
  if (c->children != NULL) {
    for (i = 0; i < P3_children_size (c->children); i++) {
      P3_object_invalid (P3_children_get (c->children, i));
    }
  }
}

P3_world* P3_world_new (P3_world* c) {
  if (c == NULL) {
    c = (P3_world*) malloc (sizeof (P3_world));
  }
  P3_coordsys_initialize ((P3_coordsys*) c);
  c->class = &P3_class_world;
  c->raypick_data = -1;
  c->shape = NULL;
  c->children = P3_children_new(4);
  c->atmosphere = NULL;
  return c;
}

void P3_world_batch (P3_world* w, P3_instance* csys) {
  int i;
  P3_any_object* o;
  P3_context* ctxt;
  if (w->option & P3_OBJECT_HIDDEN) return;
  /* compute renderering matrix: enter in this coordsys */
  P3_multiply_matrix (w->render_matrix, renderer->c_camera->render_matrix, P3_coordsys_get_root_matrix ((P3_coordsys*) w));
//  P3_multiply_matrix (w->render_matrix, csys->render_matrix, w->m);
  w->frustum_data = -1;
  /* atmosphere and context */
  ctxt = renderer->c_context;
  if (w->atmosphere != NULL) {
    if (renderer->r_atmosphere == NULL) {
      renderer->c_context->atmosphere = w->atmosphere;
      renderer->r_atmosphere = w->atmosphere;
    } else {
      if (w->atmosphere != ctxt->atmosphere) {
        /* we must create a new context */
        renderer->c_context = P3_renderer_get_context ();
        renderer->c_context->atmosphere = w->atmosphere;
        /* copy lights from previous context */
        for (i = 0; i < ctxt->lights->nb; i++) {
          P3_list_add (renderer->c_context->lights, P3_list_get (ctxt->lights, i));
        }
      }
    }
  }
  /* batch shape */
  if (w->shape != NULL && w->shape->class->batch != 0) {
    w->shape->class->batch (w->shape, (P3_instance*) w);
  }
  /* batch children */
  for (i = 0; i < P3_children_size (w->children); i++) {
    o = P3_children_get (w->children, i);
    if (o->class->batch != 0) {
      o->class->batch (o, (P3_instance*) w);
    }
  }
  if (renderer->c_context != ctxt) {
    renderer->c_context = ctxt;
  }
}

int P3_world_shadow (P3_world* w, P3_instance* inst, P3_light* light) {
  P3_any_object* o;
  int i;
  int write = 0;
  /* shape shadow */
  if (w->shape != NULL && w->shape->class->shadow != 0) {
    write = w->shape->class->shadow (w->shape, (P3_instance*) w, light);
  }
  /* children shadow */
  for (i = 0; i < P3_children_size (w->children); i++) {
    o = P3_children_get (w->children, i);
    if (o->class->shadow != 0) {
      write |= o->class->shadow (o, (P3_instance*) w, light);
    }
  }
  return write;
}

void P3_world_raypick (P3_world* world, P3_raypick_data* data, P3_raypickable* parent) {
  int i;
  P3_any_object* o;
  if (world->option & P3_OBJECT_NON_SOLID) return;
  /* transform origin and direction into the world coordsys */
  if (world->shape != NULL && world->shape->class->raypick != 0) world->shape->class->raypick (world->shape, data, (P3_raypickable*) world);
  for (i = 0; i < P3_children_size (world->children); i++) {
    o = P3_children_get (world->children, i);
    if (o->class->raypick != 0) o->class->raypick (o, data, (P3_raypickable*) world);
  }
}

int P3_world_raypick_b (P3_world* world, P3_raypick_data* data, P3_raypickable* parent) {
  int i;
  P3_any_object* o;
  if (world->option & P3_OBJECT_NON_SOLID) return P3_FALSE;
  /* transform origin and direction into the world coordsys */
  if (world->shape != NULL && world->shape->class->raypick_b (world->shape, data, (P3_raypickable*) world) == P3_TRUE) return P3_TRUE;
  for (i = 0; i < P3_children_size (world->children); i++) {
    o = P3_children_get (world->children, i);
    if (o->class->raypick_b != 0 && o->class->raypick_b (o, data, (P3_raypickable*) world) == P3_TRUE) return P3_TRUE;
  }
  return P3_FALSE;
}

void P3_world_extract (P3_world* world, int class, P3_list* list) {
  int i;
  P3_any_object* obj = NULL;
  for (i = 0; i < P3_children_size (world->children); i++) {
    obj = P3_children_get (world->children, i);
    if (obj->class->id == class && !(obj->option & P3_OBJECT_HIDDEN)) {
      P3_list_add (list, obj);
    } 
    if (obj->class->id == P3_ID_WORLD) {
      P3_world_extract ((P3_world*) obj, class, list);
    }
  }
}

int P3_contains (P3_any_object* obj, void* ptr) {
  int i;
  if (obj == NULL) { 
    return P3_FALSE;
  } else {
    switch (obj->class->id) {
    case P3_ID_WORLD:
      if (P3_contains (((P3_world*) obj)->shape, ptr) == P3_TRUE) { return P3_TRUE; }
      for (i = 0; i < P3_children_size (((P3_world*) obj)->children); i++) {
        if (P3_contains ((P3_any_object*) P3_children_get (((P3_world*) obj)->children, i), ptr) == P3_TRUE) { return P3_TRUE; }
      }
      break;
    case P3_ID_VOLUME:
      if (P3_contains (((P3_volume*) obj)->shape, ptr) == P3_TRUE) { return P3_TRUE; }
      break;
    default:
      if (obj == ptr) { return P3_TRUE; }
      break;
    }
  }
  return P3_FALSE;
}

void P3_world_get_data (P3_world* w, P3_chunk* chunk) {
  P3_chunk_save (chunk, w->m, 19 * sizeof (GLfloat));
  P3_chunk_save_int (chunk, w->option);
}

void P3_world_set_data (P3_world* w, P3_chunk* chunk) {
  w->class = &P3_class_world;
  w->validity = P3_COORDSYS_INVALID;
  w->raypick_data = -1;
  w->parent = NULL;
  P3_chunk_load (chunk, w->m, 19 * sizeof (GLfloat));
  w->option = P3_chunk_load_int (chunk);
}
