/*
 * P3
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*****************************************
 * watercube.c
 * Copyright (C) 2003 Bertrand 'blam' LAMY
 *****************************************/

#include <math.h>

#include "p3_base.h"
#include "math3d.h"
#include "util.h"
#include "material.h"
#include "renderer.h"
#include "frustum.h"
#include "coordsys.h"
#include "watercube.h"

extern P3_renderer* renderer;
extern float delta_time;

// TO DO reduce number of call to realloc
// TO DO underwater material
// TO DO frustum test with infinite cylinder or cube
// TO DO get waterlevel + normal


/*===========+
 | WATERCUBE |
 +===========*/
/* watercube nb_points_width and nb_points_depth must be like (2 ^ n) + 1
 */

#define P3_watercube_get_point(water, x, z) \
  (water->points + x + (z) * water->nb_points_width)

#define P3_watercube_get_patch(water, i, j, x, z) \
  i = (int) (((x) * (water->nb_points_width - 1)) / (water->size[0] * water->patch_size)); \
  j = (int) (((z) * (water->nb_points_depth - 1)) / (water->size[2] * water->patch_size))

#define P3_watercube_patch_get_points(water, patch_x, patch_z, p1, p2, p3, p4) \
  p1 = P3_watercube_get_point(water, patch_x, patch_z); \
  p2 = p1 + water->patch_size; \
  p3 = p1 + water->patch_size * water->nb_points_width; \
  p4 = p3 + water->patch_size


P3_class P3_class_watercube = { 
  P3_ID_WATERCUBE,
  (batch_func)     P3_watercube_batch,
  (render_func)    P3_watercube_render,
  (shadow_func)    0,
  (raypick_func)   0,
  (raypick_b_func) 0,
};


P3_watercube* P3_watercube_new (P3_watercube* water) {
  if (water == NULL) {
    water = (P3_watercube*) malloc (sizeof (P3_watercube));
  }
  water->class = &P3_class_watercube;
  P3_coordsys_initialize ((P3_coordsys*) water);
  water->size[0] = 30.0;
  water->size[1] = 30.0;
  water->size[2] = 30.0;
  water->material = NULL;
  water->texture_factor = 1.0;
  water->color[0] = 0.0;
  water->color[1] = 0.0;
  water->color[2] = 1.0;
  water->color[3] = 0.5;
  water->wave_height = 1.0;
  water->speed = 0.1;
  water->nb_points_width = 17;
  water->nb_points_depth = 17;
  water->patch_size = 8;
  water->nb_phases = 9;
  water->lod_linear = 0.2;
  water->wave_lod_linear = 0.5;
  water->max_camera_level = 0;
  water->nb_waves = 0;
  water->waves = NULL;
  return water;
}

void P3_watercube_dealloc (P3_watercube* water) {
  free (water->points);
  free (water->waves);
  free (water->patch_levels);
  free (water->phases);
  water->phases = NULL;
}

void P3_watercube_set_nb_phases (P3_watercube* water, int nb_phases) {
  int i;
  water->nb_phases = nb_phases;
  water->phases = (GLfloat*) realloc (water->phases, water->nb_phases * sizeof (GLfloat));
  for (i = 0; i < water->nb_phases; i++) {
    water->phases[i] = P3_rnd * P3_2_pi;
  }
}

void P3_watercube_init (P3_watercube* water) {
  P3_watercube_point* p;
  int nb;
  int i;
  int j;
  GLfloat scale_x;
  GLfloat scale_z;
  nb = water->nb_points_width * water->nb_points_depth;
  water->points = (P3_watercube_point*) malloc (nb * sizeof (P3_watercube_point));
  scale_x = water->size[0] / (water->nb_points_width - 1);
  scale_z = water->size[2] / (water->nb_points_depth - 1);
  for (j = 0; j < water->nb_points_depth; j++) {
    for (i = 0; i < water->nb_points_width; i++) {
      p = P3_watercube_get_point (water, i, j);
      p->coord[0] = i * scale_x;
      p->coord[1] = 0.0;
      p->coord[2] = j * scale_z;
      p->texcoord[0] = p->coord[0] * water->texture_factor;
      p->texcoord[1] = p->coord[2] * water->texture_factor;
      p->normal[0] = 0.0;
      p->normal[1] = 1.0;
      p->normal[2] = 0.0;
      p->option = 0;
    }
  }
  for (j = 0; j < water->nb_points_depth; j += water->patch_size) {
    for (i = 0; i < water->nb_points_width; i += water->patch_size) {
      P3_watercube_get_point (water, i, j)->option |= P3_WATERCUBE_POINT_ALWAYS_ENABLED;
    }
  }
  water->nb_patches_width = (int) ((water->nb_points_width - 1) / water->patch_size);
  water->nb_patches_depth = (int) ((water->nb_points_depth - 1) / water->patch_size);
  nb = water->nb_patches_width * water->nb_patches_depth * sizeof (int);
  water->patch_levels = (int*) malloc (nb);
  memset (water->patch_levels, P3_exp_of_2 (water->patch_size) << 1, nb);
  P3_watercube_set_nb_phases (water, water->nb_phases);
  water->nb_waves = 0;
  water->waves = NULL;
  water->option |= P3_WATERCUBE_INITED;
}


/*=============+
 | PATCH (LOD) |
 +=============*/
/* The patches represent the surface of the water.
 * patch_size must be like (2 ^ n)
 */

static void P3_watercube_render_point (P3_watercube* water, P3_watercube_point* p) {
  glArrayElement (p - water->points);
}

static void P3_watercube_render_tri (P3_watercube* water, P3_watercube_point* p1, P3_watercube_point* p2, P3_watercube_point* p3) {
  P3_watercube_render_point (water, p1);
  P3_watercube_render_point (water, p3);
  P3_watercube_render_point (water, p2);
}

static P3_watercube_point* P3_watercube_tri_get_p4 (P3_watercube* water, P3_watercube_point* p2, P3_watercube_point* p3) {
  int d;
  /*       p1
   *     /  | \
   *   p3--p4--p2
   */
  if (p2 < p3) {
    d = p3 - p2;
    if (d == 1 || d == water->nb_points_width) {
      /* maximum LOD level reached */
      return NULL;
    } else {
      return p2 + (d >> 1);
    }
  } else {
    d = p2 - p3;
    if (d == 1 || d == water->nb_points_width) {
      /* maximum LOD level reached */
      return NULL;
    } else {
      return p3 + (d >> 1);
    }
  }
}

static void P3_watercube_patch_render_tri (P3_watercube* water, P3_watercube_point* p1, P3_watercube_point* p2, P3_watercube_point* p3, GLfloat* fbox, GLfloat* fpos) {
  P3_watercube_point* p4;
  if ((p1->coord[0] < fbox[0] && p2->coord[0] < fbox[0] && p3->coord[0] < fbox[0]) ||
      (p1->coord[0] > fbox[3] && p2->coord[0] > fbox[3] && p3->coord[0] > fbox[3]) ||
      (p1->coord[2] < fbox[2] && p2->coord[2] < fbox[2] && p3->coord[2] < fbox[2]) ||
      (p1->coord[2] > fbox[5] && p2->coord[2] > fbox[5] && p3->coord[2] > fbox[5])) {
    return;
  }
  if (abs (p1 - p2) == 1 || abs (p1 - p3) == 1) {
    /* maximum LOD level reached */
    p4 = NULL;
  } else {
    p4 = P3_watercube_tri_get_p4 (water, p2, p3);
  }
  if (p4 != NULL && p4->option & P3_WATERCUBE_POINT_ENABLED) {
    /* triangle have children => recurse */
    if ((fpos[0] - p1->coord[0]) * (p4->coord[2] - p1->coord[2]) - (fpos[1] - p1->coord[2]) * (p4->coord[0] - p1->coord[0]) < 0.0) {
      P3_watercube_patch_render_tri (water, p4, p3, p1, fbox, fpos);
      P3_watercube_patch_render_tri (water, p4, p1, p2, fbox, fpos);
    } else {
      P3_watercube_patch_render_tri (water, p4, p1, p2, fbox, fpos);
      P3_watercube_patch_render_tri (water, p4, p3, p1, fbox, fpos);
    }
  } else {
    /* draw triangle */
    if ((p1->coord[1] > fbox[1] || p2->coord[1] > fbox[1] || p3->coord[1] > fbox[1]) &&
        (p1->coord[1] < fbox[4] || p2->coord[1] < fbox[4] || p3->coord[1] < fbox[4])) {
      P3_watercube_render_tri (water, p1, p2, p3);
    }
  }
}

static void P3_watercube_patch_render (P3_watercube* water, int patch_x, int patch_z, int type, GLfloat* fbox, GLfloat* fpos) {
  P3_watercube_point* p1;
  P3_watercube_point* p2;
  P3_watercube_point* p3;
  P3_watercube_point* p4;
  P3_watercube_point* p5;
  /*   p1------p2
   *    |\    /|
   *    |  p5  |
   *    |/    \|
   *   p3------p4
   */
  /* patch_x and patch_z are (n * patch_size) */
  P3_watercube_patch_get_points (water, patch_x, patch_z, p1, p2, p3, p4);
  p5 = p1 + ((p4 - p1) >> 1);
  /* test if central point of the patch is enabled */
  if (p5->option & P3_WATERCUBE_POINT_ENABLED) {
    if (fpos[1] - p1->coord[2] < fpos[0] - p1->coord[0]) {
      /* camera is in bottom-left => draw top-right first */
      if (fpos[1] - p3->coord[2] < p3->coord[0] - fpos[0]) {
        /* camera is bottom-right => draw top-left first */
        P3_watercube_patch_render_tri (water, p5, p1, p2, fbox, fpos);
        P3_watercube_patch_render_tri (water, p5, p2, p4, fbox, fpos);
        P3_watercube_patch_render_tri (water, p5, p3, p1, fbox, fpos);
        P3_watercube_patch_render_tri (water, p5, p4, p3, fbox, fpos);
      } else {
        P3_watercube_patch_render_tri (water, p5, p2, p4, fbox, fpos);
        P3_watercube_patch_render_tri (water, p5, p1, p2, fbox, fpos);
        P3_watercube_patch_render_tri (water, p5, p4, p3, fbox, fpos);
        P3_watercube_patch_render_tri (water, p5, p3, p1, fbox, fpos);
      }
    } else {
      if (fpos[1] - p3->coord[2] < p3->coord[0] - fpos[0]) {
        /* camera is bottom-right => draw top-left first */
        P3_watercube_patch_render_tri (water, p5, p3, p1, fbox, fpos);
        P3_watercube_patch_render_tri (water, p5, p4, p3, fbox, fpos);
        P3_watercube_patch_render_tri (water, p5, p1, p2, fbox, fpos);
        P3_watercube_patch_render_tri (water, p5, p2, p4, fbox, fpos);
      } else {
        P3_watercube_patch_render_tri (water, p5, p4, p3, fbox, fpos);
        P3_watercube_patch_render_tri (water, p5, p3, p1, fbox, fpos);
        P3_watercube_patch_render_tri (water, p5, p2, p4, fbox, fpos);
        P3_watercube_patch_render_tri (water, p5, p1, p2, fbox, fpos);
      }
    }
  } else {
    if (type & 1) {
      /* |\| */
      if (fpos[1] - p1->coord[2] < fpos[0] - p1->coord[0]) {
        /* camera is in bottom-left => draw top-right first */
        P3_watercube_render_tri (water, p2, p4, p1);
        P3_watercube_render_tri (water, p3, p1, p4);
      } else {
        P3_watercube_render_tri (water, p3, p1, p4);
        P3_watercube_render_tri (water, p2, p4, p1);
      }
    } else {
      /* |/| */
      if (fpos[1] - p3->coord[2] < p3->coord[0] - fpos[0]) {
        /* camera is bottom-right => draw top-left first */
        P3_watercube_render_tri (water, p1, p2, p3);
        P3_watercube_render_tri (water, p4, p3, p2);
      } else {
        P3_watercube_render_tri (water, p4, p3, p2);
        P3_watercube_render_tri (water, p1, p2, p3);
      }
    }
  }
}

static void P3_watercube_check_points (P3_watercube* water, int* pbox) {
  P3_watercube_point* ptr;
  P3_watercube_point* p0;
  P3_watercube_point* p;
  int max_level;
  int level;
  int x_add;
  int z_add;
  int ptr_add;
  int x;
  int z;
  int n;
  /* assume pbox[0 to 3] are multiple of patch_size */
  /* the 2 larger levels are useless to check (level = max_level and level = max_level - 1) */
  max_level = P3_exp_of_2 (water->patch_size) << 1;
  p0 = P3_watercube_get_point (water, pbox[0], pbox[1]);
  for (level = 0; level < max_level - 1; level++) {
    if (level & 1) {
      /* square
       *     |
       *   - p -
       *     |
       */
      n = P3_2_pow_n ((level + 1) >> 1);
      x_add = (n >> 1);
      z_add = x_add * water->nb_points_width;
      ptr_add = n * water->nb_points_width;
      ptr = p0 + x_add + z_add;
      for (z = pbox[1] + x_add; z < pbox[3]; z += n) {
        p = ptr;
        for (x = pbox[0] + x_add; x < pbox[2]; x += n) {
          if (p->option & P3_WATERCUBE_POINT_ENABLED) {
            /* enable diagonal points
             *   x     x
             *    \   /
             *      p
             *    /   \
             *   x     x
             */
            (p - x_add - z_add)->option |= P3_WATERCUBE_POINT_ENABLE;
            (p - x_add + z_add)->option |= P3_WATERCUBE_POINT_ENABLE;
            (p + x_add - z_add)->option |= P3_WATERCUBE_POINT_ENABLE;
            (p + x_add + z_add)->option |= P3_WATERCUBE_POINT_ENABLE;
          }
          p += n;
        }
        ptr += ptr_add;
      }
    } else {
      /* lozenge
       *   \   /
       *     p
       *   /   \
       */
      x_add = P3_2_pow_n (level >> 1);
      z_add = x_add * water->nb_points_width;
      n = (x_add << 1);
      ptr_add = n * water->nb_points_width;
      ptr = p0 + x_add;
      for (z = pbox[1]; z < pbox[3]; z += n) {
        p = ptr;
        for (x = pbox[0] + x_add; x < pbox[2]; x += n) {
          if (p->option & P3_WATERCUBE_POINT_ENABLED) {
            /* enable vertical and horizontal points
             *      x
             *      |
             *   x--p--x
             *      |
             *      x
             */
            if (z > x_add) { (p - z_add)->option |= P3_WATERCUBE_POINT_ENABLE; }
            (p - x_add)->option |= P3_WATERCUBE_POINT_ENABLE;
            (p + x_add)->option |= P3_WATERCUBE_POINT_ENABLE;
            if (z < water->nb_points_depth - x_add) { (p + z_add)->option |= P3_WATERCUBE_POINT_ENABLE; }
          }
          p += n;
        }
        ptr += ptr_add;
      }
      ptr = p0 + z_add;
      for (z = pbox[1] + x_add; z < pbox[3]; z += n) {
        p = ptr;
        for (x = pbox[0]; x < pbox[2]; x += n) {
          if (p->option & P3_WATERCUBE_POINT_ENABLED) {
            (p - z_add)->option |= P3_WATERCUBE_POINT_ENABLE;
            if (x > x_add) { (p - x_add)->option |= P3_WATERCUBE_POINT_ENABLE; }
            if (x < water->nb_points_width - x_add) { (p + x_add)->option |= P3_WATERCUBE_POINT_ENABLE; }
            (p + z_add)->option |= P3_WATERCUBE_POINT_ENABLE;
          }
          p += n;
        }
        ptr += ptr_add;
      }
    }
  }
}

static void P3_watercube_reset_lod_level (P3_watercube* water, int* pbox) {
  P3_watercube_point* ptr;
  P3_watercube_point* p;
  int i;
  int j;
  ptr = P3_watercube_get_point (water, pbox[0], pbox[1]);;
  for (j = pbox[1]; j <= pbox[3]; j++) {
    p = ptr;
    for (i = pbox[0]; i <= pbox[2]; i++) {
      p->option &= (0xFFFFFFFF - P3_WATERCUBE_POINT_ENABLE);
      p++;
    }
    ptr += water->nb_points_width;
  }
  memset (water->patch_levels, P3_exp_of_2 (water->patch_size) << 1, water->nb_patches_width * water->nb_patches_depth * sizeof (int));
}

static void P3_watercube_patch_upgrade_level (P3_watercube* water, int patch_x, int patch_z, int level) {
  P3_watercube_point* ptr;
  P3_watercube_point* p0;
  P3_watercube_point* p;
  int* patch_level;
  int x_add;
  int z_add;
  int x;
  int z;
  /* The higher the level is, the less triangle you will have.
   * High level means few precision/details.
   */
  /* assume level < patch_level */
  /* pacth_x and patch_z must be (n * patch_size) */
  x = (int) (patch_x / water->patch_size);
  z = (int) (patch_z / water->patch_size);
  patch_level = water->patch_levels + x + z * water->nb_patches_width;
  if (*patch_level > level) {
    x_add = P3_exp_of_2 (water->patch_size) << 1;
    if (level > x_add) { level = x_add; }
    p0 = P3_watercube_get_point (water, patch_x, patch_z);
    if (level & 1) {
      /* lozenge
       *   \   /
       *     p
       *   /   \
       */
      x_add = P3_2_pow_n ((level + 1) >> 1);
      z_add = x_add * water->nb_points_width;
      ptr = p0;
      for (z = 0; z < water->patch_size; z += x_add) {
        p = ptr;
        for (x = 0; x < water->patch_size; x += x_add) {
          p->option |= P3_WATERCUBE_POINT_ENABLE;
          p += x_add;
        }
        ptr += z_add;
      }
      ptr = p0 + (x_add >> 1) * (water->nb_points_width + 1);
      for (z = 0; z < water->patch_size; z += x_add) {
        p = ptr;
        for (x = 0; x < water->patch_size; x += x_add) {
          p->option |= P3_WATERCUBE_POINT_ENABLE;
          p += x_add;
        }
        ptr += z_add;
      }
    } else {
      /* square
       *     |
       *   - p -
       *     |
       */
      x_add = P3_2_pow_n (level >> 1);
      z_add = x_add * water->nb_points_width;
      ptr = p0;
      for (z = 0; z <= water->patch_size; z += x_add) {
        p = ptr;
        for (x = 0; x <= water->patch_size; x += x_add) {
          p->option |= P3_WATERCUBE_POINT_ENABLE;
          p += x_add;
        }
        ptr += z_add;
      }
    }
    *patch_level = level;
  }
}

static GLfloat P3_watercube_tri_get_waterlevel (P3_watercube* water, 
                                                P3_watercube_point* p1, P3_watercube_point* p2, P3_watercube_point* p3, 
                                                GLfloat x, GLfloat z, GLfloat* normal) {
  P3_watercube_point* p4;
  if (abs (p1 - p2) == 1 || abs (p1 - p3) == 1) {
    /* maximum LOD level reached */
    p4 = NULL;
  } else {
    p4 = P3_watercube_tri_get_p4 (water, p2, p3);
  }
  if (p4 == NULL || !(p4->option & P3_WATERCUBE_POINT_ENABLED)) {
    /* compute interpolated height */
    GLfloat r[2] = { 0.0, 0.0 };
    P3_equation_2_2 (r, 
                     p2->coord[0] - p1->coord[0],
                     p3->coord[0] - p1->coord[0],
                     p1->coord[0] - x,
                     p2->coord[2] - p1->coord[2],
                     p3->coord[2] - p1->coord[2],
                     p1->coord[2] - z);
    if (normal != NULL) {
      P3_face_normal (normal, p1->coord, p2->coord, p3->coord);
    }
    return p1->coord[1] + r[0] * (p2->coord[1] - p1->coord[1]) + r[1] * (p3->coord[1] - p1->coord[1]);
  } else {
    /* tri has children => recurse */
    if ((x - p1->coord[0]) * (p4->coord[2] - p1->coord[2]) - (z - p1->coord[2]) * (p4->coord[0] - p1->coord[0]) > 0.0) {
      /* point is in triangle p4 p3 p1 */
      return P3_watercube_tri_get_waterlevel (water, p4, p1, p2, x, z, normal);
    } else {
      /* point is in triangle p4 p1 p2 */
      return P3_watercube_tri_get_waterlevel (water, p4, p3, p1, x, z, normal);
    }
  }
}

GLfloat P3_watercube_get_waterlevel (P3_watercube* water, GLfloat x, GLfloat z, GLfloat* normal) {
  P3_watercube_point* p1;
  P3_watercube_point* p2;
  P3_watercube_point* p3;
  P3_watercube_point* p4;
  GLfloat k;
  GLfloat t;
  int patch_x;
  int patch_z;
  P3_watercube_get_patch (water, patch_x, patch_z, x, z);
  if (patch_x == water->nb_patches_width) { patch_x--; }
  if (patch_z == water->nb_patches_depth) { patch_z--; }
  P3_watercube_patch_get_points (water, patch_x * water->patch_size, patch_z * water->patch_size, p1, p2, p3, p4);
  if (((patch_x & 1) && (patch_z & 1)) || (!(patch_x & 1) && !(patch_z & 1))) {
    /* |\| */
    k =   (x - p3->coord[0]) / (p4->coord[0] - p1->coord[0]);
    t = - (z - p3->coord[2]) / (p4->coord[2] - p1->coord[2]);
    if (k <= 1.0 && t <= 1.0 && k + t <= 1.0) {
      return P3_watercube_tri_get_waterlevel (water, p3, p1, p4, x, z, normal);
    } else {
      return P3_watercube_tri_get_waterlevel (water, p2, p4, p1, x, z, normal);
    }
  } else {
    /* |/| */
    k = (x - p1->coord[0]) / (p4->coord[0] - p1->coord[0]);
    t = (z - p1->coord[2]) / (p4->coord[2] - p1->coord[2]);
    if (k <= 1.0 && t <= 1.0 && k + t <= 1.0) {
      return P3_watercube_tri_get_waterlevel (water, p1, p2, p3, x, z, normal);
    } else {
      return P3_watercube_tri_get_waterlevel (water, p4, p3, p2, x, z, normal);
    }
  }
}

int P3_watercube_is_underwater (P3_watercube* water, GLfloat* point) {
  if (point[0] < 0.0 || point[0] > water->size[0] ||
      point[2] < 0.0 || point[2] > water->size[2] ||
      point[1] < - water->size[1]) {
    return P3_FALSE;
  } else {
    if (point[1] < P3_watercube_get_waterlevel (water, point[0], point[2], NULL)) {
      return P3_TRUE;
    } else {
      return P3_FALSE;
    }
  }
}

static void P3_watercube_compute_normals (P3_watercube* water, int* pbox) {
  P3_watercube_point* ptr;
  P3_watercube_point* p0;
  P3_watercube_point* p;
  P3_watercube_point* q;
  GLfloat y0, y1, y2, y3, y5, y6, y7, y8;
  GLfloat a, b, c;
  int nbp;
  int max_level;
  int level;
  int x_add;
  int z_add;
  int ptr_add;
  int x;
  int z;
  int n;
  /* assume pbox[0 to 3] are multiple of patch_size */
  max_level = P3_exp_of_2 (water->patch_size) << 1;
  p0 = P3_watercube_get_point (water, pbox[0], pbox[1]);
  for (level = 0; level <= max_level; level++) {
    if (level & 1) {
      /* square
       *     |
       *   - p -
       *     |
       */
      n = P3_2_pow_n ((level + 1) >> 1);
      x_add = (n >> 1);
      z_add = x_add * water->nb_points_width;
      ptr_add = n * water->nb_points_width;
      ptr = p0 + x_add + z_add;
      for (z = pbox[1] + x_add; z < pbox[3]; z += n) {
        p = ptr;
        for (x = pbox[0] + x_add; x < pbox[2]; x += n) {
          if (p->option & P3_WATERCUBE_POINT_ENABLED) {
            nbp = 4;
            /* get diagonal points */
            y0 = (p - x_add - z_add)->coord[1];
            y2 = (p - x_add + z_add)->coord[1];
            y6 = (p + x_add - z_add)->coord[1];
            y8 = (p + x_add + z_add)->coord[1];
            /* get vertical and horizontal points if enabled */
            q = p - z_add;
            if (q->option & P3_WATERCUBE_POINT_ENABLED) {
              y1 = q->coord[1];
              nbp++;
            } else {
              y1 = 0.0;
            }
            q = p - x_add;
            if (q->option & P3_WATERCUBE_POINT_ENABLED) {
              y3 = q->coord[1];
              nbp++;
            } else {
              y3 = 0.0;
            }
            q = p + x_add;
            if (q->option & P3_WATERCUBE_POINT_ENABLED) {
              y5 = q->coord[1];
              nbp++;
            } else {
              y5 = 0.0;
            }
            q = p + z_add;
            if (q->option & P3_WATERCUBE_POINT_ENABLED) {
              y7 = q->coord[1];
              nbp++;
            } else {
              y7 = 0.0;
            }
            /* compute normal */
            c = 1.0 / nbp;
            a = (-y0 + y2 + 2.0 * y5 + y8 - y6 - 2.0 * y3) * c;
            b = (-2.0 * y1 - y2 + y8 + 2.0 * y7 + y6 - y0) * c;
            c = 1.0 / sqrt (a * a + 1.0 + b * b);
            p->normal[0] = - a * c;
            p->normal[1] =       c;
            p->normal[2] = - b * c;
          }
          p += n;
        }
        ptr += ptr_add;
      }
    } else {
      /* lozenge
       *   \   /
       *     p
       *   /   \
       */
      x_add = P3_2_pow_n (level >> 1);
      z_add = x_add * water->nb_points_width;
      n = (x_add << 1);
      ptr_add = n * water->nb_points_width;
      ptr = p0 + x_add;
      for (z = pbox[1]; z < pbox[3]; z += n) {
        p = ptr;
        for (x = pbox[0] + x_add; x < pbox[2]; x += n) {
          if (p->option & P3_WATERCUBE_POINT_ENABLED) {
            /* get vertical and horizontal points, diagonal points if enabled */
            nbp = 2;
            if (z > x_add) { 
              y1 = (p - z_add)->coord[1];
              nbp++;
              q = p - z_add - x_add;
              if (q->option & P3_WATERCUBE_POINT_ENABLED) {
                y0 = q->coord[1];
                nbp++;
              } else {
                y0 = 0.0;
              }
              q = p - z_add + x_add;
              if (q->option & P3_WATERCUBE_POINT_ENABLED) {
                y2 = q->coord[1];
                nbp++;
              } else {
                y2 = 0.0;
              }
            } else {
              y0 = 0.0;
              y1 = 0.0;
              y2 = 0.0;
            }
            y3 = (p - x_add)->coord[1];
            y5 = (p + x_add)->coord[1];
            if (z < water->nb_points_depth - x_add) { 
              y7 = (p + z_add)->coord[1]; 
              nbp++;
              q = p + z_add - x_add;
              if (q->option & P3_WATERCUBE_POINT_ENABLED) {
                y6 = q->coord[1];
                nbp++;
              } else {
                y6 = 0.0;
              }
              q = p + z_add + x_add;
              if (q->option & P3_WATERCUBE_POINT_ENABLED) {
                y8 = q->coord[1];
                nbp++;
              } else {
                y8 = 0.0;
              }
            } else {
              y6 = 0.0;
              y7 = 0.0;
              y8 = 0.0;
            }
            /* compute normal */
            c = 1.0 / nbp;
            a = (-y0 + y2 + 2.0 * y5 + y8 - y6 - 2.0 * y3) * c;
            b = (-2.0 * y1 - y2 + y8 + 2.0 * y7 + y6 - y0) * c;
            c = 1.0 / sqrt (a * a + 1.0 + b * b);
            p->normal[0] = - a * c;
            p->normal[1] =       c;
            p->normal[2] = - b * c;
          }
          p += n;
        }
        ptr += ptr_add;
      }
      ptr = p0 + z_add;
      for (z = pbox[1] + x_add; z < pbox[3]; z += n) {
        p = ptr;
        for (x = pbox[0]; x < pbox[2]; x += n) {
          if (p->option & P3_WATERCUBE_POINT_ENABLED) {
            nbp = 2;
            y1 = (p - z_add)->coord[1];
            if (x > x_add) { 
              y3 = (p - x_add)->coord[1]; 
              nbp++;
              q = p - x_add - z_add;
              if (q->option & P3_WATERCUBE_POINT_ENABLED) {
                y0 = q->coord[1];
                nbp++;
              } else {
                y0 = 0.0;
              }
              q = p - x_add + z_add;
              if (q->option & P3_WATERCUBE_POINT_ENABLED) {
                y6 = q->coord[1];
                nbp++;
              } else {
                y6 = 0.0;
              }
            } else {
              y0 = 0.0;
              y3 = 0.0;
              y6 = 0.0;
            }
            if (x < water->nb_points_width - x_add) { 
              y5 = (p + x_add)->coord[1]; 
              nbp++;
              q = p + x_add - z_add;
              if (q->option & P3_WATERCUBE_POINT_ENABLED) {
                y2 = q->coord[1];
                nbp++;
              } else {
                y2 = 0.0;
              }
              q = p + x_add + z_add;
              if (q->option & P3_WATERCUBE_POINT_ENABLED) {
                y8 = q->coord[1];
                nbp++;
              } else {
                y8 = 0.0;
              }
            } else {
              y2 = 0.0;
              y5 = 0.0;
              y8 = 0.0;
            }
            y7 = (p + z_add)->coord[1];
            /* compute normal */
            c = 1.0 / nbp;
            a = (-y0 + y2 + 2.0 * y5 + y8 - y6 - 2.0 * y3) * c;
            b = (-2.0 * y1 - y2 + y8 + 2.0 * y7 + y6 - y0) * c;
            c = 1.0 / sqrt (a * a + 1.0 + b * b);
            p->normal[0] = - a * c;
            p->normal[1] =       c;
            p->normal[2] = - b * c;
          }
          p += n;
        }
        ptr += ptr_add;
      }
    }
  }
}


/*============+
 | UNDERWATER |
 +============*/
/* Here are the functions to draw the water as if it was a filled volume.
 * There are a lots of mathematics, don't ask me to explain how it works.
 */

static void P3_watercube_underwater_horizontal_clip (GLfloat* result, GLfloat* p, GLfloat* v, GLfloat clip) {
  result[1] = p[1] + (clip - p[0]) / v[0] * v[1];
  result[0] = clip;
}

static void P3_watercube_underwater_vertical_clip (GLfloat* result, GLfloat* p, GLfloat* v, GLfloat clip) {
  result[0] = p[0] + (clip - p[1]) / v[1] * v[0];
  result[1] = clip;
}

static void P3_watercube_underwater_intersect_line (GLfloat* result, GLfloat* p1, GLfloat* p2) {
  GLfloat k = (renderer->c_camera->front + p1[2]) / (p1[2] - p2[2]);
  result[0] = p1[0] + k * (p2[0] - p1[0]);
  result[1] = p1[1] + k * (p2[1] - p1[1]);
  result[2] = - renderer->c_camera->front;
}

void P3_watercube_underwater_intersect_face (P3_chunk* rezult, GLfloat* p1, GLfloat* p2, GLfloat* p3) {
//static void P3_watercube_underwater_intersect_face (GLfloat** result, int* nb_result, GLfloat* p1, GLfloat* p2, GLfloat* p3) {
  /* Compute intersection of a triangle with the camera front plane
   * Assume p1, p2 and p3 are given in the camera coordsys */
  GLfloat* frustum = renderer->c_camera->frustum->points;
  GLfloat intersec[6]; 
  GLfloat v[3];
  GLfloat front = - renderer->c_camera->front;
  GLfloat d;
  /* we can have only 2 intersection points (case where the 3 points 
   * are in the camera front plane is impossible) */
  int nb_intersec = 0;
  /* compute intersections */
  if (p1[2] == front) {
    memcpy (intersec + nb_intersec * 3, p1, 3 * sizeof (GLfloat));
    nb_intersec++;
  }
  if (p2[2] == front) {
    memcpy (intersec + nb_intersec * 3, p2, 3 * sizeof (GLfloat));
    nb_intersec++;
  }
  if (p3[2] == front) {
    memcpy (intersec + nb_intersec * 3, p3, 3 * sizeof (GLfloat));
    nb_intersec++;
  }
  if ((p1[2] > front && p2[2] < front) || (p1[2] < front && p2[2] > front)) {
    P3_watercube_underwater_intersect_line (intersec + nb_intersec * 3, p1, p2);
    nb_intersec++;
  }
  if ((p3[2] > front && p2[2] < front) || (p3[2] < front && p2[2] > front)) {
    P3_watercube_underwater_intersect_line (intersec + nb_intersec * 3, p2, p3);
    nb_intersec++;
  }
  if ((p1[2] > front && p3[2] < front) || (p1[2] < front && p3[2] > front)) {
    P3_watercube_underwater_intersect_line (intersec + nb_intersec * 3, p1, p3);
    nb_intersec++;
  }
  if (nb_intersec == 2) {
    GLfloat* ptr;
    GLfloat* max;
//    int i;
    /* compute intersection with the camera front rectangle => clip */
    while (1) {
      if ((intersec[0] < frustum[ 3] && intersec[3] < frustum[ 3]) ||
          (intersec[1] > frustum[ 4] && intersec[4] > frustum[ 4]) ||
          (intersec[0] > frustum[ 9] && intersec[3] > frustum[ 9]) ||
          (intersec[1] < frustum[10] && intersec[4] < frustum[10])) {
        /* intersections are out of front rectangle */
        return;
      }
      v[0] = intersec[3] - intersec[0];
      v[1] = intersec[4] - intersec[1];
      /* intersect with left line */
      d = frustum[3];
      if (intersec[0] < d && intersec[3] > d) { 
        P3_watercube_underwater_horizontal_clip (intersec, intersec, v, d);
        continue;
      }
      if (intersec[0] > d && intersec[3] < d) { 
        P3_watercube_underwater_horizontal_clip (intersec + 3, intersec, v, d); 
        continue;
      }
      /* intersect with right line */
      d = frustum[9];
      if (intersec[0] < d && intersec[3] > d) { 
        P3_watercube_underwater_horizontal_clip (intersec + 3, intersec, v, d);
        continue; 
      }
      if (intersec[0] > d && intersec[3] < d) { 
        P3_watercube_underwater_horizontal_clip (intersec, intersec, v, d); 
        continue;
      }
      /* intersect with top line */
      d = frustum[4];
      if (intersec[1] > d && intersec[4] < d) { 
        P3_watercube_underwater_vertical_clip (intersec, intersec, v, d);
        continue;
      }
      if (intersec[1] < d && intersec[4] > d) { 
        P3_watercube_underwater_vertical_clip (intersec + 3, intersec, v, d);
        continue; 
      }
      /* intersect with bottom line */
      d = frustum[10];
      if (intersec[1] > d && intersec[4] < d) { 
        P3_watercube_underwater_vertical_clip (intersec + 3, intersec, v, d);
        continue; 
      }
      if (intersec[1] < d && intersec[4] > d) { 
        P3_watercube_underwater_vertical_clip (intersec, intersec, v, d);
        continue; 
      }
      break;
    }
    /* determine the right order for the 2 intersections (must turn clockwise) */
    P3_face_normal (v, p1, p2, p3);
    d = v[0] * (intersec[4] - intersec[1]) - v[1] * (intersec[3] - intersec[0]);
    if (d < 0.0) {
      /* swap */
      memcpy (v, intersec + 3, 3 * sizeof (GLfloat));
      memcpy (intersec + 3, intersec, 3 * sizeof (GLfloat));
      memcpy (intersec, v, 3 * sizeof (GLfloat));
    }
    /* search if this result has already been added
     * this case appears when a triangle has 1 of its edge entirely on the front plane
     */
    ptr = (GLfloat*) rezult->content;
    max = (GLfloat*) (rezult->content + rezult->nb);
    while (ptr < max) {
      if (fabs (intersec[0] - ptr[0]) < P3_UPSILON && fabs (intersec[1] - ptr[1]) < P3_UPSILON &&
          fabs (intersec[3] - ptr[3]) < P3_UPSILON && fabs (intersec[4] - ptr[4]) < P3_UPSILON) {
        return;
      }
      ptr += 6;
    }
/*
    ptr = *result;
    for (i = 0; i < *nb_result; i += 2) {
      if (fabs (intersec[0] - ptr[0]) < P3_UPSILON && fabs (intersec[1] - ptr[1]) < P3_UPSILON &&
          fabs (intersec[3] - ptr[3]) < P3_UPSILON && fabs (intersec[4] - ptr[4]) < P3_UPSILON) {
        return;
      }
      ptr += 6;
    }
*/
    /* add results */
    P3_chunk_add (rezult, intersec, 6 * sizeof (GLfloat));
/*
    *result = (GLfloat*) realloc (*result, (*nb_result + 2) * 3 * sizeof (GLfloat));
    memcpy (*result + *nb_result * 3, intersec, 6 * sizeof (GLfloat));
    *nb_result += 2;
*/
  } else if (nb_intersec > 2) {
    P3_error ("underwater_intersect_face has found %i intersections", nb_intersec);
  }
}

static int P3_watercube_underwater_search_up    (P3_watercube_underwater*, int, GLfloat*, int);
static int P3_watercube_underwater_search_down  (P3_watercube_underwater*, int, GLfloat*, int);
static int P3_watercube_underwater_search_left  (P3_watercube_underwater*, int, GLfloat*, int);
static int P3_watercube_underwater_search_right (P3_watercube_underwater*, int, GLfloat*, int);

static int P3_watercube_underwater_search_down (P3_watercube_underwater* underwater, int nb_first, GLfloat* pts, int nb_pts) {
  GLfloat* frustum = renderer->c_camera->frustum->points;
  GLfloat* ptr = pts;
  GLfloat* best = NULL;
  GLfloat max_y = 0.0;
  GLfloat previous = underwater->points[(underwater->nb_points) * 3 - 2];
  while (ptr < pts + nb_pts * 3) {
    if (ptr[0] >= frustum[9] - P3_UPSILON && 
        ptr[1] < previous &&
        (best == NULL || ptr[1] > max_y)) {
      max_y = ptr[1];
      best = ptr;
    }
    ptr += 6;
  }
  ptr = underwater->points + nb_first;
  if (ptr[0] >= frustum[9] - P3_UPSILON && ptr[1] < previous && (best == NULL || ptr[1] > max_y)) {
    /* first point is the next point we are looking for */
    underwater->points = (GLfloat*) realloc (underwater->points, (underwater->nb_points + 1) * 3 * sizeof (GLfloat));
    memcpy (underwater->points + underwater->nb_points * 3, underwater->points + nb_first, 3 * sizeof (GLfloat));
    underwater->nb_points++;
    (underwater->seq_sizes[underwater->nb_seqs - 1])++;
    return P3_TRUE;
  }
  if (best == NULL) {
    /* add the bottom right corner */
    underwater->points = (GLfloat*) realloc (underwater->points, (underwater->nb_points + 1) * 3 * sizeof (GLfloat));
    memcpy (underwater->points + underwater->nb_points * 3, frustum + 9, 3 * sizeof (GLfloat));
    underwater->nb_points++;
    (underwater->seq_sizes[underwater->nb_seqs - 1])++;
    /* search left */
    return P3_watercube_underwater_search_left (underwater, nb_first, pts, nb_pts);
  } else {
    /* add the segment found */
    underwater->points = (GLfloat*) realloc (underwater->points, (underwater->nb_points + 2) * 3 * sizeof (GLfloat));
    memcpy (underwater->points + underwater->nb_points * 3, best, 6 * sizeof (GLfloat));
    underwater->nb_points += 2;
    underwater->seq_sizes[underwater->nb_seqs - 1] += 2;
    /* remove the segment */
    ptr = pts + (nb_pts - 2) * 3;
    if (best != ptr) {
      memcpy (best, ptr, 6 * sizeof (GLfloat));
    }
    return P3_FALSE;
  }
}

static int P3_watercube_underwater_search_up (P3_watercube_underwater* underwater, int nb_first, GLfloat* pts, int nb_pts) {
  GLfloat* frustum = renderer->c_camera->frustum->points;
  GLfloat* ptr = pts;
  GLfloat* best = NULL;
  GLfloat min_y = 0.0;
  GLfloat previous = underwater->points[(underwater->nb_points) * 3 - 2];
  while (ptr < pts + nb_pts * 3) {
    if (ptr[0] <= frustum[3] + P3_UPSILON && 
        ptr[1] > previous &&
        (best == NULL || ptr[1] < min_y)) {
      min_y = ptr[1];
      best = ptr;
    }
    ptr += 6;
  }
  ptr = underwater->points + nb_first;
  if (ptr[0] <= frustum[3] + P3_UPSILON && ptr[1] > previous && (best == NULL || ptr[1] < min_y)) {
    /* first point is the next point we are looking for */
    underwater->points = (GLfloat*) realloc (underwater->points, (underwater->nb_points + 1) * 3 * sizeof (GLfloat));
    memcpy (underwater->points + underwater->nb_points * 3, underwater->points + nb_first, 3 * sizeof (GLfloat));
    underwater->nb_points++;
    (underwater->seq_sizes[underwater->nb_seqs - 1])++;
    return P3_TRUE;
  }
  if (best == NULL) {
    /* add the upper left corner */
    underwater->points = (GLfloat*) realloc (underwater->points, (underwater->nb_points + 1) * 3 * sizeof (GLfloat));
    memcpy (underwater->points + underwater->nb_points * 3, frustum + 3, 3 * sizeof (GLfloat));
    underwater->nb_points++;
    (underwater->seq_sizes[underwater->nb_seqs - 1])++;
    /* search right */
    return P3_watercube_underwater_search_right (underwater, nb_first, pts, nb_pts);
  } else {
    /* add the segment found */
    underwater->points = (GLfloat*) realloc (underwater->points, (underwater->nb_points + 2) * 3 * sizeof (GLfloat));
    memcpy (underwater->points + underwater->nb_points * 3, best, 6 * sizeof (GLfloat));
    underwater->nb_points += 2;
    underwater->seq_sizes[underwater->nb_seqs - 1] += 2;
    /* remove the segment */
    ptr = pts + (nb_pts - 2) * 3;
    if (best != ptr) {
      memcpy (best, ptr, 6 * sizeof (GLfloat));
    }
    return P3_FALSE;
  }
}

static int P3_watercube_underwater_search_right (P3_watercube_underwater* underwater, int nb_first, GLfloat* pts, int nb_pts) {
  GLfloat* frustum = renderer->c_camera->frustum->points;
  GLfloat* ptr = pts;
  GLfloat* best = NULL;
  GLfloat min_x = 0.0;
  GLfloat previous = underwater->points[(underwater->nb_points) * 3 - 3];
  while (ptr < pts + nb_pts * 3) {
    if (ptr[1] >= frustum[4] - P3_UPSILON && 
        ptr[0] > previous &&
        (best == NULL || ptr[0] < min_x)) {
      min_x = ptr[0];
      best = ptr;
    }
    ptr += 6;
  }
  ptr = underwater->points + nb_first;
  if (ptr[1] >= frustum[4] - P3_UPSILON && ptr[0] > previous && (best == NULL || ptr[0] < min_x)) {
    /* first point is the next point we are looking for */
    underwater->points = (GLfloat*) realloc (underwater->points, (underwater->nb_points + 1) * 3 * sizeof (GLfloat));
    memcpy (underwater->points + underwater->nb_points * 3, underwater->points + nb_first, 3 * sizeof (GLfloat));
    underwater->nb_points++;
    (underwater->seq_sizes[underwater->nb_seqs - 1])++;
    return P3_TRUE;
  }
  if (best == NULL) {
    /* add the upper rigth corner */
    underwater->points = (GLfloat*) realloc (underwater->points, (underwater->nb_points + 1) * 3 * sizeof (GLfloat));
    memcpy (underwater->points + underwater->nb_points * 3, frustum, 3 * sizeof (GLfloat));
    underwater->nb_points++;
    (underwater->seq_sizes[underwater->nb_seqs - 1])++;
    /* search down */
    return P3_watercube_underwater_search_down (underwater, nb_first, pts, nb_pts);
  } else {
    /* add the segment found */
    underwater->points = (GLfloat*) realloc (underwater->points, (underwater->nb_points + 2) * 3 * sizeof (GLfloat));
    memcpy (underwater->points + underwater->nb_points * 3, best, 6 * sizeof (GLfloat));
    underwater->nb_points += 2;
    underwater->seq_sizes[underwater->nb_seqs - 1] += 2;
    /* remove the segment */
    ptr = pts + (nb_pts - 2) * 3;
    if (best != ptr) {
      memcpy (best, ptr, 6 * sizeof (GLfloat));
    }
    return P3_FALSE;
  }
}

static int P3_watercube_underwater_search_left (P3_watercube_underwater* underwater, int nb_first, GLfloat* pts, int nb_pts) {
  GLfloat* frustum = renderer->c_camera->frustum->points;
  GLfloat* ptr = pts;
  GLfloat* best = NULL;
  GLfloat max_x = 0.0;
  GLfloat previous = underwater->points[(underwater->nb_points) * 3 - 3];
  while (ptr < pts + nb_pts * 3) {
    if (ptr[1] <= frustum[10] + P3_UPSILON && 
        ptr[0] < previous &&
        (best == NULL || ptr[0] > max_x)) {
      max_x = ptr[0];
      best = ptr;
    }
    ptr += 6;
  }
  ptr = underwater->points + nb_first;
  if (ptr[1] <= frustum[10] + P3_UPSILON && ptr[0] < previous && (best == NULL || ptr[0] > max_x)) {
    /* first point is the next point we are looking for */
    underwater->points = (GLfloat*) realloc (underwater->points, (underwater->nb_points + 1) * 3 * sizeof (GLfloat));
    memcpy (underwater->points + underwater->nb_points * 3, underwater->points + nb_first, 3 * sizeof (GLfloat));
    underwater->nb_points++;
    (underwater->seq_sizes[underwater->nb_seqs - 1])++;
    return P3_TRUE;
  }
  if (best == NULL) {
    /* add the bottom left corner */
    underwater->points = (GLfloat*) realloc (underwater->points, (underwater->nb_points + 1) * 3 * sizeof (GLfloat));
    memcpy (underwater->points + underwater->nb_points * 3, frustum + 6, 3 * sizeof (GLfloat));
    underwater->nb_points++;
    (underwater->seq_sizes[underwater->nb_seqs - 1])++;
    /* search up */
    return P3_watercube_underwater_search_up (underwater, nb_first, pts, nb_pts);
  } else {
    /* add the segment found */
    underwater->points = (GLfloat*) realloc (underwater->points, (underwater->nb_points + 2) * 3 * sizeof (GLfloat));
    memcpy (underwater->points + underwater->nb_points * 3, best, 6 * sizeof (GLfloat));
    underwater->nb_points += 2;
    underwater->seq_sizes[underwater->nb_seqs - 1] += 2;
    /* remove the segment */
    ptr = pts + (nb_pts - 2) * 3;
    if (best != ptr) {
      memcpy (best, ptr, 6 * sizeof (GLfloat));
    }
    return P3_FALSE;
  }
}

static void P3_watercube_underwater_prolongate (GLfloat* from, GLfloat* to, GLfloat* frustum) {
  GLfloat v[2];
  v[0] = to[0] - from[0];
  v[1] = to[1] - from[1];
  if (v[0] == 0.0) {
    if (v[1] > 0.0) {
      P3_watercube_underwater_vertical_clip (to, to, v, frustum[4]);
    } else {
      P3_watercube_underwater_vertical_clip (to, to, v, frustum[10]);
    }
    if (to[0] <= frustum[3]) P3_watercube_underwater_horizontal_clip (to, to, v, frustum[3]);
    if (to[0] >= frustum[9]) P3_watercube_underwater_horizontal_clip (to, to, v, frustum[9]);
  } else {
    if (v[0] > 0.0) {
      P3_watercube_underwater_horizontal_clip (to, to, v, frustum[9]);
    } else {
      P3_watercube_underwater_horizontal_clip (to, to, v, frustum[3]);
    }
    if (to[1] >= frustum[ 4]) P3_watercube_underwater_vertical_clip (to, to, v, frustum[ 4]);
    if (to[1] <= frustum[10]) P3_watercube_underwater_vertical_clip (to, to, v, frustum[10]);
  }
}

P3_watercube_underwater* P3_watercube_underwater_join_segments (GLfloat* pts, int nb_pts) {
  GLfloat* frustum = renderer->c_camera->frustum->points;
  P3_watercube_underwater* underwater;
  GLfloat* ptr;
  GLfloat* last;
  GLfloat* first;
  int nb_first;
  int* size;

  underwater = (P3_watercube_underwater*) malloc (sizeof (P3_watercube_underwater));
  underwater->points = NULL;
  underwater->nb_points = 0;
  underwater->seq_sizes = NULL;
  underwater->nb_seqs = 0;

  while (nb_pts > 0) {
    /* create a new sequence */
    /* add the last segment */
    underwater->points = (GLfloat*) realloc (underwater->points, (underwater->nb_points + 2) * 3 * sizeof (GLfloat));
    nb_first = underwater->nb_points * 3;
    memcpy (underwater->points + nb_first, pts + (nb_pts - 2) * 3, 6 * sizeof (GLfloat));
    underwater->nb_points += 2;
    underwater->seq_sizes = (int*) realloc (underwater->seq_sizes, (underwater->nb_seqs + 1) * sizeof (int));
    size = underwater->seq_sizes + underwater->nb_seqs;
    *size = 2;
    (underwater->nb_seqs)++;
    nb_pts -= 2;

    /* search previous segments */
    while (1) {
      ptr = pts;
      first = underwater->points + nb_first;
      while (ptr < pts + nb_pts * 3) {
        if (fabs (ptr[3] - first[0]) < P3_UPSILON &&
            fabs (ptr[4] - first[1]) < P3_UPSILON) {
          /* previous segment found => add it */
          underwater->points = (GLfloat*) realloc (underwater->points, (underwater->nb_points + 1) * 3 * sizeof (GLfloat));
          memmove (underwater->points + nb_first + 3, underwater->points + nb_first, *size * 3 * sizeof (GLfloat));
          memcpy (underwater->points + nb_first, ptr, 3 * sizeof (GLfloat));
          (underwater->nb_points)++;
          (*size)++;
          /* remove the segment */
          last = pts + (nb_pts - 2) * 3;
          if (last != ptr) {
            memcpy (ptr, last, 6 * sizeof (GLfloat));
          }
          nb_pts -= 2;
          ptr = NULL;
          break;
        }
        ptr += 6;
      }
      if (ptr != NULL) {
        /* no previous segment found */
        break;
      }
    }

    /* test if we already have a loop, in that case sequence is finished */
    first = underwater->points + nb_first;
    last = underwater->points + (underwater->nb_points - 1) * 3;
    if (*size > 2 && 
        fabs (first[0] - last[0]) < P3_UPSILON && fabs (first[1] - last[1]) < P3_UPSILON) {
      continue;
    }

    /* if sequence starts itself not in the border of the front rectangle, prolongate it */
    first = underwater->points + nb_first;
    if (first[0] > frustum[ 3] + P3_UPSILON && 
        first[0] < frustum[ 9] - P3_UPSILON &&
        first[1] < frustum[ 4] - P3_UPSILON &&
        first[1] > frustum[10] + P3_UPSILON) {
      P3_watercube_underwater_prolongate (first + 3, first, frustum);
    }

    /* it's not a loop => must add some of the front rectangle corners */
    while (1) {
      /* search next segments */
      while (1) {
        ptr = pts;
        last = underwater->points + (underwater->nb_points - 1) * 3;
        while (ptr < pts + nb_pts * 3) {
          if (fabs (ptr[0] - last[0]) < P3_UPSILON &&
              fabs (ptr[1] - last[1]) < P3_UPSILON) {
            /* next segment found => add it */
            underwater->points = (GLfloat*) realloc (underwater->points, (underwater->nb_points + 1) * 3 * sizeof (GLfloat));
            memcpy (underwater->points + underwater->nb_points * 3, ptr + 3, 3 * sizeof (GLfloat));
            (underwater->nb_points)++;
            (*size)++;
            /* remove the segment */
            last = pts + (nb_pts - 2) * 3;
            if (last != ptr) {
              memcpy (ptr, last, 6 * sizeof (GLfloat));
            }
            nb_pts -= 2;
            ptr = NULL;
            last = underwater->points + (underwater->nb_points - 1) * 3;
            break;
          }
          ptr += 6;
        }
        if (ptr != NULL) {
          /* no next segment found */
          break;
        }
      }
      if (last[0] >= frustum[9] - P3_UPSILON) {
        if (P3_watercube_underwater_search_down (underwater, nb_first, pts, nb_pts) == P3_TRUE) {
          break;
        } else {
          nb_pts -= 2;
        }
      } else if (last[0] <= frustum[3] + P3_UPSILON) {
        if (P3_watercube_underwater_search_up (underwater, nb_first, pts, nb_pts) == P3_TRUE) {
          break;
        } else {
          nb_pts -= 2;
        }
      } else if (last[1] <= frustum[10] + P3_UPSILON) {
        if (P3_watercube_underwater_search_left (underwater, nb_first, pts, nb_pts) == P3_TRUE) {
          break;
        } else {
          nb_pts -= 2;
        }
      } else if (last[1] >= frustum[4] - P3_UPSILON) {
        if (P3_watercube_underwater_search_right (underwater, nb_first, pts, nb_pts) == P3_TRUE) {
          break;
        } else {
          nb_pts -= 2;
        }
      } else {
        P3_watercube_underwater_prolongate (last - 3, last, frustum);
      }
    }
  }
  return underwater;
}

void P3_watercube_underwater_draw_segments (P3_watercube_underwater* underwater) {
  int deads[underwater->nb_points];
  GLfloat* pts = underwater->points;
  GLfloat* p1;
  GLfloat* p2;
  GLfloat* p3;
  GLfloat* ptr;
  GLfloat* max;
  GLfloat v1[2];
  GLfloat v2[2];
  GLfloat d;
  GLfloat t1, t2;
  GLfloat a, b;
  int nb;
  int i, j, k;
  int pi1, pi2, pi3;
  int nb_pts;
  int drawed = 0;
  /* Push the points away from the front plane (some OpenGL won't draw a face
   * if its vertices are all inside the camera front plane
   */
  a = 0.5 * (1.0 + renderer->c_camera->back / renderer->c_camera->front);
  b = - 0.5 * (renderer->c_camera->back + renderer->c_camera->front);
  ptr = underwater->points;
  for (i = 0; i < underwater->nb_points; i++) {
    *ptr *= a;
    ptr++;
    *ptr *= a;
    ptr++;
    *ptr = b;
    ptr++;
  }
  /* This function is able to draw a complex concave polygon 
   * Points must be given in clockwise order to define the interior of the polygon
   */
  glBegin (GL_TRIANGLES);
  for (i = 0; i < underwater->nb_seqs; i++) {
    nb = underwater->seq_sizes[i];
    nb_pts = nb;
    max = pts + nb * 3;
    memset (deads, 0, nb * sizeof (int));
    while (nb_pts > 3) {
      pi1 = 0;
      while (deads[pi1] == 1) { pi1++; }
      p1 = pts + pi1 * 3;
      p2 = NULL;
      j = nb_pts;
      while (j >= 3) {
        if (p2 == NULL) {
          pi2 = pi1 + 1;
          while (deads[pi2] == 1) { pi2++; }
          p2 = pts + pi2 * 3;
        }
        pi3 = pi2 + 1;
        while (deads[pi3] == 1) { pi3++; }
        p3 = pts + pi3 * 3;
        /* find if triangle is concave or convex */
        v1[0] = p2[0] - p1[0];
        v1[1] = p2[1] - p1[1];
        v2[0] = p3[0] - p2[0];
        v2[1] = p3[1] - p2[1];
        d = v2[0] * v1[1] - v2[1] * v1[0];
        if (d > 0.0) {
          /* convex */
          /* test if 1 of the other points is inside the triangle */
          d = 1.0 / d;
          ptr = pts;
          /* nb - 1 cause last point = first point of the sequence (it's a loop) */
          for (k = 0; k < nb - 1; k++) {
            if (ptr != p1 && ptr != p2 && ptr != p3) {
              a = p2[0] - ptr[0];
              b = p2[1] - ptr[1];
              t1 = (v2[0] * b - v2[1] * a) * d;
              t2 = (v1[0] * b - v1[1] * a) * d;
              if (t1 > 0.0 && t2 > 0.0 && t1 < 1.0 && t2 < 1.0 && t1 + t2 < 1.0) {
                /* skip */
                k = -1;
                break;
              }
            }
            ptr += 3;
          }
          if (k != -1) {
            glVertex3fv (p1);
            glVertex3fv (p2);
            glVertex3fv (p3);
            deads[pi2] = 1;
            nb_pts--;
            pi1 = pi3;
            p1 = p3;
            p2 = NULL;
            j -= 2;
            drawed++;
            continue;
          }
        }
        p1 = p2;
        pi2 = pi3;
        p2 = p3;
        j--;
      }
      if (drawed == 0) {
        P3_error ("underwater drawed nothing!!!");
        break;
      }
      drawed = 0;
    }
    pts += nb * 3;
  }
  glEnd ();
}

static void P3_watercube_underwater_intersect_patch_face (P3_watercube* water,
                                                          P3_watercube_point* p1, P3_watercube_point* p2, P3_watercube_point* p3, 
                                                          GLfloat* frustum_box, P3_chunk* pts) {
//                                                          GLfloat* frustum_box, int* nb_pts, GLfloat** pts) {
  P3_watercube_point* p4;
  /* frustum test */
  if ((p1->coord[0] < frustum_box[0] && p2->coord[0] < frustum_box[0] && p3->coord[0] < frustum_box[0]) ||
      (p1->coord[0] > frustum_box[3] && p2->coord[0] > frustum_box[3] && p3->coord[0] > frustum_box[3]) ||
      (p1->coord[2] < frustum_box[2] && p2->coord[2] < frustum_box[2] && p3->coord[2] < frustum_box[2]) ||
      (p1->coord[2] > frustum_box[5] && p2->coord[2] > frustum_box[5] && p3->coord[2] > frustum_box[5])) {
    return;
  }
  if (abs (p1 - p2) == 1 || abs (p1 - p3) == 1) {
    /* maximum LOD level reached */
    p4 = NULL;
  } else {
    p4 = P3_watercube_tri_get_p4 (water, p2, p3);
  }
  if (p4 != NULL && p4->option & P3_WATERCUBE_POINT_ENABLED) {
    /* triangle have children => recurse */
    P3_watercube_underwater_intersect_patch_face (water, p4, p1, p2, frustum_box, pts);
    P3_watercube_underwater_intersect_patch_face (water, p4, p3, p1, frustum_box, pts);
//    P3_watercube_underwater_intersect_patch_face (water, p4, p1, p2, frustum_box, nb_pts, pts);
//    P3_watercube_underwater_intersect_patch_face (water, p4, p3, p1, frustum_box, nb_pts, pts);
  } else {
    /* intersect triangle */
    GLfloat p[9];
    GLfloat* m;

// TO DO ? optimizable by stocking computed points

    m = P3_coordsys_get_root_matrix ((P3_coordsys*) water);
    P3_point_by_matrix_copy (p,     p1->coord, m);
    P3_point_by_matrix_copy (p + 3, p2->coord, m);
    P3_point_by_matrix_copy (p + 6, p3->coord, m);
    m = P3_coordsys_get_inverted_root_matrix ((P3_coordsys*) renderer->c_camera);
    P3_point_by_matrix (p,     m);
    P3_point_by_matrix (p + 3, m);
    P3_point_by_matrix (p + 6, m);
    P3_watercube_underwater_intersect_face (pts, p, p + 3, p + 6);
//    P3_watercube_underwater_intersect_face (pts, nb_pts, p, p + 3, p + 6);
  }
}

static void P3_watercube_intersect_surface_side (P3_watercube* water, P3_watercube_point* start, int add, int max, P3_chunk* pts) {
//static void P3_watercube_intersect_surface_side (P3_watercube* water, P3_watercube_point* start, int add, int max, int* nb_pts, GLfloat** pts) {
  P3_watercube_point* p;
  GLfloat q[12];
  GLfloat* m1;
  GLfloat* m2;
  int i;
  m1 = P3_coordsys_get_root_matrix ((P3_coordsys*) water);
  m2 = P3_coordsys_get_inverted_root_matrix ((P3_coordsys*) renderer->c_camera);
  p = start;
  P3_point_by_matrix_copy (q, p->coord, m1);
  P3_point_by_matrix (q, m2);
  q[3] = p->coord[0];
  q[4] = - water->size[1];
  q[5] = p->coord[2];
  P3_point_by_matrix (q + 3, m1);
  P3_point_by_matrix (q + 3, m2);
  for (i = 0; i < max; i++) {
    p += add;
    if (!(p->option & P3_WATERCUBE_POINT_ENABLED)) {
      continue;
    }
    P3_point_by_matrix_copy (q + 6, p->coord, m1);
    P3_point_by_matrix (q + 6, m2);
    q[ 9] = p->coord[0];
    q[10] = - water->size[1];
    q[11] = p->coord[2];
    P3_point_by_matrix (q + 9, m1);
    P3_point_by_matrix (q + 9, m2);
    P3_watercube_underwater_intersect_face (pts, q, q + 6, q + 9);
    P3_watercube_underwater_intersect_face (pts, q, q + 9, q + 3);
//    P3_watercube_underwater_intersect_face (pts, nb_pts, q, q + 6, q + 9);
//    P3_watercube_underwater_intersect_face (pts, nb_pts, q, q + 9, q + 3);
    memcpy (q, q + 6, 6 * sizeof (GLfloat));
  }
}

static void P3_watercube_render_total_underwater (void) {
  GLfloat p[12];
  GLfloat* ptrf = renderer->c_camera->frustum->points;
  /* compute screen surface rectangle 
   * we can't just take the frustum points cause their depth are equal to
   * the front of the camera and some OpenGL implementation won't draw the quad
   */
  p[ 0] = (ptrf[0] + ptrf[12]) * 0.5;
  p[ 1] = (ptrf[1] + ptrf[13]) * 0.5;
  p[ 2] = (ptrf[2] + ptrf[14]) * 0.5;
  p[ 3] = - p[0];
  p[ 4] =   p[1];
  p[ 5] =   p[2];
  p[ 6] = - p[0];
  p[ 7] = - p[1];
  p[ 8] =   p[2];
  p[ 9] =   p[0];
  p[10] = - p[1];
  p[11] =   p[2];
  glEnableClientState (GL_VERTEX_ARRAY);
  glVertexPointer (3, GL_FLOAT, 0, p);
  glDrawArrays (GL_QUADS, 0, 4);
  glDisableClientState (GL_VERTEX_ARRAY);
}

static void P3_watercube_render_underwater (P3_watercube* water, GLfloat* fbox, GLfloat* front_points) {
  P3_watercube_point* p1;
  P3_watercube_point* p2;
  P3_watercube_point* p3;
  P3_watercube_point* p4;
//  GLfloat* pts;
//  int nb_pts;
  P3_chunk* pts;
  GLfloat surface_min;
  GLfloat surface_max;
  int x1;
  int z1;
  int x2; 
  int z2;
  int i;
  int j;
  int k;
  /* frustum box test */
  if (fbox[3] < 0.0 || fbox[5] < 0.0 || fbox[0] > water->size[0] || fbox[2] > water->size[2] || fbox[4] < - water->size[1]) 
    return;
  /* adjust box to patch size */
  P3_watercube_get_patch (water, x1, z1, fbox[0], fbox[2]);
  P3_watercube_get_patch (water, x2, z2, fbox[3], fbox[5]);
  x1 *= water->patch_size;
  z1 *= water->patch_size;
  x2 = (x2 + 1) * water->patch_size;
  z2 = (z2 + 1) * water->patch_size;
  if (x1 < 0) x1 = 0;
  if (z1 < 0) z1 = 0;
  if (x1 == water->nb_points_width - 1) x1 -= water->patch_size;
  if (z1 == water->nb_points_depth - 1) z1 -= water->patch_size;
  if (x2 >= water->nb_points_width) x2 = water->nb_points_width - 1;
  if (z2 >= water->nb_points_depth) z2 = water->nb_points_depth - 1;
  p1 = P3_watercube_get_point (water, x1, z1);
  surface_min =   P3_INFINITY;
  surface_max = - P3_INFINITY;
  for (j = 0; j <= z2 - z1; j++) {
    p2 = p1;
    for (i = 0; i <= x2 - x1; i++) {
      if (p2->option & P3_WATERCUBE_POINT_ENABLED) {
        if      (p2->coord[1] < surface_min) surface_min = p2->coord[1];
        else if (p2->coord[1] > surface_max) surface_max = p2->coord[1];
      }
      p2++;
    }
    p1 += water->nb_points_width;
  }
  if (fbox[1] < surface_max) {
//    glDepthMask (GL_FALSE);
    glDisable (GL_DEPTH_TEST);
    glDisable (GL_LIGHTING);
    glDisable (GL_TEXTURE_2D);
//    glEnable (GL_BLEND);
    glDisable (GL_CULL_FACE);
    glLoadIdentity ();
    glColor4fv (water->color);
    if (fbox[4] < surface_min) {
      /* totally under surface */
      pts = P3_get_chunk ();
//      nb_pts = 0;
//      pts = NULL;
      /* add side of watercube if necessary */
      if (fbox[0] < 0.0 || fbox[3] > water->size[0] ||
          fbox[2] < 0.0 || fbox[5] > water->size[2] ||
          fbox[1] < - water->size[1]) {
        GLfloat* m;
        GLfloat p[24] = { 0.0, surface_min, 0.0,
                          water->size[0], surface_min, 0.0,
                          water->size[0], - water->size[1], 0.0,
                          0.0, - water->size[1], 0.0,
                          0.0, surface_min, water->size[2],
                          water->size[0], surface_min, water->size[2],
                          water->size[0], - water->size[1], water->size[2],
                          0.0, - water->size[1], water->size[2] };
        m = P3_coordsys_get_root_matrix ((P3_coordsys*) water);
        for (i = 0; i < 24; i += 3) { P3_point_by_matrix (p + i, m); }
        m = P3_coordsys_get_inverted_root_matrix ((P3_coordsys*) renderer->c_camera);
        for (i = 0; i < 24; i += 3) { P3_point_by_matrix (p + i, m); }
        if (fbox[0] < 0.0) {
          P3_watercube_underwater_intersect_face (pts, p, p + 21, p +  9);
          P3_watercube_underwater_intersect_face (pts, p, p + 12, p + 21);
//          P3_watercube_underwater_intersect_face (&pts, &nb_pts, p, p + 21, p +  9);
//          P3_watercube_underwater_intersect_face (&pts, &nb_pts, p, p + 12, p + 21);
        }
        if (fbox[3] > water->size[0]) {
          P3_watercube_underwater_intersect_face (pts, p + 3, p + 18, p + 15);
          P3_watercube_underwater_intersect_face (pts, p + 3, p +  6, p + 18);
//          P3_watercube_underwater_intersect_face (&pts, &nb_pts, p + 3, p + 18, p + 15);
//          P3_watercube_underwater_intersect_face (&pts, &nb_pts, p + 3, p +  6, p + 18);
        }
        if (fbox[2] < 0.0) {
          P3_watercube_underwater_intersect_face (pts, p, p + 6, p + 3);
          P3_watercube_underwater_intersect_face (pts, p, p + 9, p + 6);
//          P3_watercube_underwater_intersect_face (&pts, &nb_pts, p, p + 6, p + 3);
//          P3_watercube_underwater_intersect_face (&pts, &nb_pts, p, p + 9, p + 6);
        }
        if (fbox[5] > water->size[2]) {
          P3_watercube_underwater_intersect_face (pts, p + 15, p + 21, p + 12);
          P3_watercube_underwater_intersect_face (pts, p + 15, p + 18, p + 21);
//          P3_watercube_underwater_intersect_face (&pts, &nb_pts, p + 15, p + 21, p + 12);
//          P3_watercube_underwater_intersect_face (&pts, &nb_pts, p + 15, p + 18, p + 21);
        }
        if (fbox[1] < - water->size[1]) {
          P3_watercube_underwater_intersect_face (pts, p + 9, p + 18, p +  6);
          P3_watercube_underwater_intersect_face (pts, p + 9, p + 21, p + 18);
//          P3_watercube_underwater_intersect_face (&pts, &nb_pts, p + 9, p + 18, p +  6);
//          P3_watercube_underwater_intersect_face (&pts, &nb_pts, p + 9, p + 21, p + 18);
        }
        if (pts->nb > 0) {
//        if (nb_pts > 0) {
          P3_watercube_underwater* underwater;
          underwater = P3_watercube_underwater_join_segments ((GLfloat*) pts->content, pts->nb / (3 * sizeof (GLfloat)));
//          underwater = P3_watercube_underwater_join_segments (pts, nb_pts);
          P3_watercube_underwater_draw_segments (underwater);
//          free (pts);
          P3_drop_chunk (pts);
          free (underwater->points);
          free (underwater->seq_sizes);
          free (underwater);
          return;
        }
      }
      P3_watercube_render_total_underwater ();
    } else {
      /* maybe partially under water */
      pts = P3_get_chunk ();
//      nb_pts = 0;
//      pts = NULL;
      /* add side faces if front box is partially outside the watercube */
      if (fbox[0] < 0.0) {
        P3_watercube_intersect_surface_side (water, P3_watercube_get_point (water, 0, z1), 
                                             water->nb_points_width, z2 - z1, pts);
//                                             water->nb_points_width, z2 - z1, &nb_pts, &pts);
      }
      if (fbox[3] > water->size[0]) {
        P3_watercube_intersect_surface_side (water, P3_watercube_get_point (water, water->nb_points_width - 1, z2), 
                                             - water->nb_points_width, z2 - z1, pts);
//                                             - water->nb_points_width, z2 - z1, &nb_pts, &pts);
      }
      if (fbox[2] < 0.0) {
        P3_watercube_intersect_surface_side (water, P3_watercube_get_point (water, x2, 0), 
                                             - 1, x2 - x1, pts);
//                                             - 1, x2 - x1, &nb_pts, &pts);
      }
      if (fbox[5] > water->size[2]) {
        P3_watercube_intersect_surface_side (water, P3_watercube_get_point (water, x1, water->nb_points_depth - 1),
                                             1, x2 - x1, pts);
//                                             1, x2 - x1, &nb_pts, &pts);
      }
      /* add potential surface points */
      i = (int) (x1 / water->patch_size);
      j = (int) (z1 / water->patch_size);
      if (((i & 1) && (j & 1)) || (!(i & 1) && !(j & 1))) {
        k = 1;
      } else {
        k = 0;
      }
      for (j = z1; j < z2; j += water->patch_size) {
        for (i = x1; i < x2; i += water->patch_size) {
          P3_watercube_patch_get_points (water, i, j, p1, p2, p3, p4);
          if (k & 1) {
            /* |\| */
            P3_watercube_underwater_intersect_patch_face (water, p2, p4, p1, fbox, pts);
            P3_watercube_underwater_intersect_patch_face (water, p3, p1, p4, fbox, pts);
//            P3_watercube_underwater_intersect_patch_face (water, p2, p4, p1, fbox, &nb_pts, &pts);
//            P3_watercube_underwater_intersect_patch_face (water, p3, p1, p4, fbox, &nb_pts, &pts);
          } else {
            /* |/| */
            P3_watercube_underwater_intersect_patch_face (water, p1, p2, p3, fbox, pts);
            P3_watercube_underwater_intersect_patch_face (water, p4, p3, p2, fbox, pts);
//            P3_watercube_underwater_intersect_patch_face (water, p1, p2, p3, fbox, &nb_pts, &pts);
//            P3_watercube_underwater_intersect_patch_face (water, p4, p3, p2, fbox, &nb_pts, &pts);
          }
          k++;
        }
        k++;
      }
      if (pts->nb > 0) {
//      if (nb_pts > 0) {
        P3_watercube_underwater* underwater;
        underwater = P3_watercube_underwater_join_segments ((GLfloat*) pts->content, pts->nb / (3 * sizeof (GLfloat)));
//        underwater = P3_watercube_underwater_join_segments (pts, nb_pts);
        P3_watercube_underwater_draw_segments (underwater);
        P3_drop_chunk (pts);
//        free (pts);
        free (underwater->points);
        free (underwater->seq_sizes);
        free (underwater);
      } else {
        /* the point must be inside water in X and Z !! */
        for (i = 0; i < 12; i += 3) {
          if (P3_watercube_is_underwater (water, front_points + i) == P3_TRUE) {
            P3_watercube_render_total_underwater ();
            break;
          }
        }
      }
    }
//    glDepthMask (GL_TRUE);
    glEnable (GL_DEPTH_TEST);
    glEnable (GL_LIGHTING);
    glEnable (GL_TEXTURE_2D);
//    glDisable (GL_BLEND);
    glEnable (GL_CULL_FACE);
  }
}


/*================+
 | ANIMATED WAVES |
 +================*/

static void P3_watercube_clip_border_wave (P3_watercube* water) {
  P3_watercube_point* p;
  int i;
  p = water->points;
  for (i = 0; i < water->nb_points_width; i++) {
    if (p->option & P3_WATERCUBE_POINT_ENABLED && p->coord[1] > 0.0) { p->coord[1] = 0.0; }
    p++;
  }
  p = water->points;
  for (i = 0; i < water->nb_points_depth; i++) {
    if (p->option & P3_WATERCUBE_POINT_ENABLED && p->coord[1] > 0.0) { p->coord[1] = 0.0; }
    p += water->nb_points_width;
  }
  p = water->points + water->nb_points_width - 1;
  for (i = 0; i < water->nb_points_depth; i++) {
    if (p->option & P3_WATERCUBE_POINT_ENABLED && p->coord[1] > 0.0) { p->coord[1] = 0.0; }
    p += water->nb_points_width;
  }
  p = water->points + (water->nb_points_depth - 1) * water->nb_points_width;
  for (i = 0; i < water->nb_points_width; i++) {
    if (p->option & P3_WATERCUBE_POINT_ENABLED && p->coord[1] > 0.0) { p->coord[1] = 0.0; }
    p++;
  }
}

static void P3_watercube_lap (P3_watercube* water, int* pbox) {
  P3_watercube_point* ptr;
  P3_watercube_point* p;
  GLfloat phases[water->nb_phases];
  int i;
  int j;
  int k;
  int k0;
  int c;
  /* update phases */
  for (i = 0; i < water->nb_phases; i++) {
    water->phases[i] += delta_time * water->speed;
//    water->phases[i] += water->delta_time * water->speed;
    if (water->phases[i] > P3_2_pi) { water->phases[i] -= P3_2_pi; }
    phases[i] = (sin (water->phases[i]) - 1.0) * water->wave_height;
  }
  /* determine 1rst phase */
  k0 = (pbox[0] + pbox[1] * water->nb_points_width) % water->nb_phases;
  c = water->nb_points_width % water->nb_phases;
  /* set new points y */
  ptr = P3_watercube_get_point (water, pbox[0], pbox[1]);
  for (j = pbox[1]; j <= pbox[3]; j++) {
    p = ptr;
    k = k0;
    for (i = pbox[0]; i <= pbox[2]; i++) {
      if (p->option & P3_WATERCUBE_POINT_ENABLED) {
        p->coord[1] += phases[k];
      }
      k++;
      if (k == water->nb_phases) { k = 0; }
      p++;
    }
    ptr += water->nb_points_width;
    k0 += c;
    if (k0 >= water->nb_phases) { k0 -= water->nb_phases; }
  }
}

static void P3_watercube_delete_wave (P3_watercube* water, int index) {
  if (index != water->nb_waves - 1) {
    memcpy (water->waves + index, water->waves + water->nb_waves - 1, sizeof (P3_watercube_wave));
  }
  (water->nb_waves)--;
  water->waves = (P3_watercube_wave*) realloc (water->waves, water->nb_waves * sizeof (P3_watercube_wave));
}

void P3_watercube_new_wave (P3_watercube* water, GLfloat x, GLfloat z,
                            GLfloat height, GLfloat radius, GLfloat decay, GLfloat wavelength,
                            GLfloat speed, GLfloat height_decay) {
  P3_watercube_wave* w;
  if (x < 0.0 || z < 0.0 || x > water->size[0] || z > water->size[2]) { return; }
  water->waves = (P3_watercube_wave*) realloc (water->waves, (water->nb_waves + 1) * sizeof (P3_watercube_wave));
  w = water->waves + water->nb_waves;
  (water->nb_waves)++;
  w->position[0] = x;
  w->position[1] = z;
  w->height = height;
  w->radius = radius;
  w->decay = decay;
  w->wavelength = wavelength;
  w->speed = speed;
  w->height_decay = height_decay;
  w->max_radius = w->radius + w->height / (w->decay * water->wave_height);
}

static void P3_watercube_update_wave (P3_watercube* water, P3_watercube_wave* wave, int* pbox) {
  P3_watercube_point* ptr;
  P3_watercube_point* p;
  GLfloat a;
  GLfloat b;
  GLfloat r;
  GLfloat addx;
  GLfloat addz;
  GLfloat u;
  GLfloat v;
  int max_level;
  int level;
  int xmin;
  int zmin;
  int xmax;
  int zmax;
  int x;
  int z;
  /* wave to patch box */
  P3_watercube_get_patch (water, xmin, zmin, wave->position[0] - wave->max_radius, wave->position[1] - wave->max_radius);
  P3_watercube_get_patch (water, xmax, zmax, wave->position[0] + wave->max_radius, wave->position[1] + wave->max_radius);
  if (xmin >= water->nb_patches_width) { xmin = water->nb_patches_width - 1; }
  if (zmin >= water->nb_patches_depth) { zmin = water->nb_patches_depth - 1; }
  if (xmax >= water->nb_patches_width) { xmax = water->nb_patches_width - 1; }
  if (zmax >= water->nb_patches_depth) { zmax = water->nb_patches_depth - 1; }
  if (xmin < 0) { xmin = 0; }
  if (zmin < 0) { zmin = 0; }
  if (xmax < 0) { xmax = 0; }
  if (zmax < 0) { zmax = 0; }
  xmin *= water->patch_size;
  zmin *= water->patch_size;
  xmax = (xmax + 1) * water->patch_size;
  zmax = (zmax + 1) * water->patch_size;
  /* clip with pbox */
  if (pbox[0] > xmin) { xmin = pbox[0]; }
  if (pbox[1] > zmin) { zmin = pbox[1]; }
  if (pbox[2] < xmax) { xmax = pbox[2]; }
  if (pbox[3] < zmax) { zmax = pbox[3]; }
  /* update LOD for wave */
  max_level = (int) (wave->radius * water->wave_lod_linear);
  addx = water->size[0] / water->nb_patches_width;
  addz = water->size[2] / water->nb_patches_depth;
  r = ((int) (xmin / water->patch_size) + 0.5) * addx;
  b = ((int) (zmin / water->patch_size) + 0.5) * addz;
  for (z = zmin; z < zmax; z += water->patch_size) {
    a = r;
    for (x = xmin; x < xmax; x += water->patch_size) {
      u = a - wave->position[0];
      v = b - wave->position[1];
      v = sqrt (u * u + v * v) - addx;
      if (v < 0.0) {
        P3_watercube_patch_upgrade_level (water, x, z, max_level);
      } else {
        level = (int) (v * water->wave_lod_linear);
        if (level < max_level) { level = max_level; }
        P3_watercube_patch_upgrade_level (water, x, z, level);
      }
      a += addx;
    }
    b += addz;
  }
  ptr = P3_watercube_get_point (water, xmin, zmin);
  for (z = zmin; z <= zmax; z++) {
    p = ptr;
    for (x = xmin; x <= zmax; x++) {
      a = p->coord[0] - wave->position[0];
      b = p->coord[2] - wave->position[1];
      r = sqrt (a * a + b * b);
      p->coord[1] += cos (wave->radius + wave->wavelength * r) / (wave->decay * fabs (r - wave->radius) + 1.0) * wave->height;
      p++;
    }
    ptr += water->nb_points_width;
  }
  /* update wave parameter */
  wave->height *= pow (wave->height_decay, delta_time);
  wave->radius += wave->speed * delta_time;
//  wave->height *= pow (wave->height_decay, water->delta_time);
//  wave->radius += wave->speed * water->delta_time;
}

static void P3_watercube_update_waves (P3_watercube* water, P3_frustum* frustum, int* pbox) {
  int i = 0;
  /* delete old waves */
  while (i < water->nb_waves) {
    if ((water->waves + i)->height <= water->wave_height) {
      P3_watercube_delete_wave (water, i);
    } else {
      i++;
    }
  }
  /* play each wave */
  for (i = 0; i < water->nb_waves; i++) {
    P3_watercube_update_wave (water, water->waves + i, pbox);
  }
}

static void P3_watercube_update (P3_watercube* water, P3_frustum* frustum, int* pbox) {
  P3_watercube_point* ptr;
  P3_watercube_point* p;
  GLfloat q[2];
  GLfloat f;
  GLfloat addx;
  GLfloat addz;
  GLfloat x;
  GLfloat y;
  GLfloat z;
  GLfloat d;
  int i;
  int j;
  int level;


// HACK
/*
  water->delta_time = 1.0;
  if (water->nb_waves == 0) {
    P3_watercube_new_wave (water, 14.0, 14.0, 
                           5.0, // height
                           0.0, // radius
                           2.0, // decay
                           2.0, // wavelength
                           0.5, // speed
                           0.85  // height_decay
                           );
  }
*/

  P3_watercube_reset_lod_level (water, pbox);
  /* update LOD level with camera position */
  addx = water->size[0] / water->nb_patches_width;
  addz = water->size[2] / water->nb_patches_depth;
  f =    ((int) (pbox[0] / water->patch_size) + 0.5) * addx;
  q[1] = ((int) (pbox[1] / water->patch_size) + 0.5) * addz;
  y = frustum->position[1] * frustum->position[1];
  for (j = pbox[1]; j < pbox[3]; j += water->patch_size) {
    q[0] = f;
    for (i = pbox[0]; i < pbox[2]; i += water->patch_size) {
      x = q[0] - frustum->position[0];
      z = q[1] - frustum->position[2];
      d = sqrt (x * x + y + z * z) - addx;
      if (d < 0.0) {
        P3_watercube_patch_upgrade_level (water, i, j, water->max_camera_level);
      } else {
        level = (int) (d * water->lod_linear);
        if (level < water->max_camera_level) { 
          P3_watercube_patch_upgrade_level (water, i, j, water->max_camera_level);
        } else {
          P3_watercube_patch_upgrade_level (water, i, j, level);
        }
      }
      q[0] += addx;
    }
    q[1] += addz;
  }
  ptr = P3_watercube_get_point (water, pbox[0], pbox[1]);
  for (j = pbox[1]; j <= pbox[3]; j++) {
    p = ptr;
    for (i = pbox[0]; i <= pbox[2]; i++) {
      p->coord[1] = 0.0;
      p++;
    }
    ptr += water->nb_points_width;
  }
  P3_watercube_update_waves (water, frustum, pbox);
  P3_watercube_check_points (water, pbox);
  P3_watercube_lap (water, pbox);
  P3_watercube_clip_border_wave (water);
// TO DO is it really interesting ?
  P3_watercube_compute_normals (water, pbox);
}

/*
void P3_watercube_advance_time (P3_watercube* water, float proportion) {
  water->delta_time += proportion;
}
*/


/*===========+
 | RENDERING |
 +===========*/

void P3_watercube_batch (P3_watercube* water, P3_instance* csys) {
  GLfloat* front_points;
  P3_frustum* frustum;
  void* ptr;
  GLfloat* ubox;
//  GLfloat fbox[6];
//  int pbox[4];
  int n;
  if (water->option & P3_OBJECT_HIDDEN) return;
  if (!(water->option & P3_WATERCUBE_INITED)) P3_watercube_init (water); 
  water->frustum_data = -1;
  /* frustum test */
  frustum = P3_renderer_get_frustum ((P3_instance*) water);
  /* box test */
  P3_frustum_to_box (frustum, water->fbox);
  /* test if frustum rectangle is totally out of water on side */
  if (water->fbox[3] < 0.0 || water->fbox[5] < 0.0 || water->fbox[0] > water->size[0] || water->fbox[2] > water->size[2] || 
      water->fbox[4] < - water->size[1]) return;
  /* compute new renderer stuff: enter in this coordsys */
  P3_multiply_matrix (water->render_matrix, renderer->c_camera->render_matrix, P3_coordsys_get_root_matrix ((P3_coordsys*) water));
  /* adjust box to patch size */
  P3_watercube_get_patch (water, water->pbox[0], water->pbox[1], water->fbox[0], water->fbox[2]);
  P3_watercube_get_patch (water, water->pbox[2], water->pbox[3], water->fbox[3], water->fbox[5]);
  water->pbox[0] *= water->patch_size;
  water->pbox[1] *= water->patch_size;
  water->pbox[2] = (water->pbox[2] + 1) * water->patch_size;
  water->pbox[3] = (water->pbox[3] + 1) * water->patch_size;
  if (water->pbox[0] < 0) water->pbox[0] = 0;
  if (water->pbox[1] < 0) water->pbox[1] = 0;
  if (water->pbox[2] >= water->nb_points_width) water->pbox[2] = water->nb_points_width - 1;
  if (water->pbox[3] >= water->nb_points_depth) water->pbox[3] = water->nb_points_depth - 1;
  /* update LOD, make waves animation */
  P3_watercube_update (water, frustum, water->pbox);
//  water->delta_time = 0;
  /* bacth */

  P3_list_add (renderer->watercubes, water);
  water->context = renderer->c_context;

  water->camera_position[0] = frustum->position[0];
  water->camera_position[1] = frustum->position[2];
  /* hack: frustum back plane normal is camera direction */
  water->camera_z_axis[0] = frustum->planes[20];
  water->camera_z_axis[1] = frustum->planes[22];
  /* compute box from frustum front points for underwater effect -> ubox */
  front_points = frustum->points;
  ubox = water->ubox;
  ubox[0] = front_points[0];
  ubox[1] = front_points[1];
  ubox[2] = front_points[2];
  ubox[3] = ubox[0];
  ubox[4] = ubox[1];
  ubox[5] = ubox[2];
  if      (front_points[ 3] < ubox[0]) ubox[0] = front_points[ 3];
  else if (front_points[ 3] > ubox[3]) ubox[3] = front_points[ 3];
  if      (front_points[ 4] < ubox[1]) ubox[1] = front_points[ 4];
  else if (front_points[ 4] > ubox[4]) ubox[4] = front_points[ 4];
  if      (front_points[ 5] < ubox[2]) ubox[2] = front_points[ 5];
  else if (front_points[ 5] > ubox[5]) ubox[5] = front_points[ 5];
  if      (front_points[ 6] < ubox[0]) ubox[0] = front_points[ 6];
  else if (front_points[ 6] > ubox[3]) ubox[3] = front_points[ 6];
  if      (front_points[ 7] < ubox[1]) ubox[1] = front_points[ 7];
  else if (front_points[ 7] > ubox[4]) ubox[4] = front_points[ 7];
  if      (front_points[ 8] < ubox[2]) ubox[2] = front_points[ 8];
  else if (front_points[ 8] > ubox[5]) ubox[5] = front_points[ 8];
  if      (front_points[ 9] < ubox[0]) ubox[0] = front_points[ 9];
  else if (front_points[ 9] > ubox[3]) ubox[3] = front_points[ 9];
  if      (front_points[10] < ubox[1]) ubox[1] = front_points[10];
  else if (front_points[10] > ubox[4]) ubox[4] = front_points[10];
  if      (front_points[11] < ubox[2]) ubox[2] = front_points[11];
  else if (front_points[11] > ubox[5]) ubox[5] = front_points[11];

  memcpy (water->front_points, front_points, 12 * sizeof (GLfloat));

#if 0

  P3_renderer_batch (renderer->specials, water, NULL, renderer->data->nb);
//  n = P3_chunk_register (renderer->specials, sizeof (P3_watercube*) + sizeof (P3_context*) + 4 * sizeof (int) + 28 * sizeof (GLfloat));
//  ptr = renderer->specials->content + n;
  n = P3_chunk_register (renderer->data, 4 * sizeof (int) + 28 * sizeof (GLfloat));
  ptr = renderer->data->content + n;
//  *((P3_watercube**) ptr) = water;
//  ptr += sizeof (P3_watercube*);
//  *((P3_context**) ptr) = renderer->c_context;
//  ptr += sizeof (P3_context*);
  memcpy (ptr, pbox, 4 * sizeof (int));
  ptr += 4 * sizeof (int);
  memcpy (ptr, fbox, 6 * sizeof (GLfloat));
  ptr += 6 * sizeof (GLfloat);
  *((GLfloat*) ptr) = frustum->position[0];
  ptr += sizeof (GLfloat);
  *((GLfloat*) ptr) = frustum->position[2];
  ptr += sizeof (GLfloat);
  /* hack: frustum back plane normal is camera direction */
  *((GLfloat*) ptr) = frustum->planes[20];
  ptr += sizeof (GLfloat);
  *((GLfloat*) ptr) = frustum->planes[22];
  ptr += sizeof (GLfloat);
  /* compute box from frustum front points */
  front_points = frustum->points;
  fbox[0] = front_points[0];
  fbox[1] = front_points[1];
  fbox[2] = front_points[2];
  fbox[3] = fbox[0];
  fbox[4] = fbox[1];
  fbox[5] = fbox[2];
  if      (front_points[ 3] < fbox[0]) fbox[0] = front_points[ 3];
  else if (front_points[ 3] > fbox[3]) fbox[3] = front_points[ 3];
  if      (front_points[ 4] < fbox[1]) fbox[1] = front_points[ 4];
  else if (front_points[ 4] > fbox[4]) fbox[4] = front_points[ 4];
  if      (front_points[ 5] < fbox[2]) fbox[2] = front_points[ 5];
  else if (front_points[ 5] > fbox[5]) fbox[5] = front_points[ 5];
  if      (front_points[ 6] < fbox[0]) fbox[0] = front_points[ 6];
  else if (front_points[ 6] > fbox[3]) fbox[3] = front_points[ 6];
  if      (front_points[ 7] < fbox[1]) fbox[1] = front_points[ 7];
  else if (front_points[ 7] > fbox[4]) fbox[4] = front_points[ 7];
  if      (front_points[ 8] < fbox[2]) fbox[2] = front_points[ 8];
  else if (front_points[ 8] > fbox[5]) fbox[5] = front_points[ 8];
  if      (front_points[ 9] < fbox[0]) fbox[0] = front_points[ 9];
  else if (front_points[ 9] > fbox[3]) fbox[3] = front_points[ 9];
  if      (front_points[10] < fbox[1]) fbox[1] = front_points[10];
  else if (front_points[10] > fbox[4]) fbox[4] = front_points[10];
  if      (front_points[11] < fbox[2]) fbox[2] = front_points[11];
  else if (front_points[11] > fbox[5]) fbox[5] = front_points[11];
  /* front camera box for underwater effect -> ubox */
  memcpy (ptr, fbox, 6 * sizeof (GLfloat));
  ptr += 6 * sizeof (GLfloat);
  memcpy (ptr, front_points, 12 * sizeof (GLfloat));

#endif

}

void P3_watercube_render (P3_watercube* water) {
//  P3_context* ctx;
//  GLfloat* camera_position;
//  GLfloat* camera_z_axis;
//  GLfloat* front_points;
//  GLfloat* ubox;
//  GLfloat* fbox;
  int* pbox = water->pbox;
  int cpx;
  int cpz;
  int pi;
  int pj;
  int i;
  int j;
  int k;
  /* get frustum box back from renderer data chunk */
/*
  pbox = (int*) (renderer->data->content + renderer->data->nb);
  renderer->data->nb += 4 * sizeof (int);
  fbox = (GLfloat*) (renderer->data->content + renderer->data->nb);
  renderer->data->nb += 6 * sizeof (GLfloat);
  camera_position = (GLfloat*) (renderer->data->content + renderer->data->nb);
  renderer->data->nb += 2 * sizeof (GLfloat);
  camera_z_axis = (GLfloat*) (renderer->data->content + renderer->data->nb);
  renderer->data->nb += 2 * sizeof (GLfloat);
  ubox = (GLfloat*) (renderer->data->content + renderer->data->nb);
  renderer->data->nb += 6 * sizeof (GLfloat);
  front_points = (GLfloat*) (renderer->data->content + renderer->data->nb);
  renderer->data->nb += 12 * sizeof (GLfloat);
*/
/*
  ctx = *((P3_context**) (renderer->specials->content + renderer->specials->nb));
  renderer->specials->nb += sizeof (P3_context*);
  pbox = (int*) (renderer->specials->content + renderer->specials->nb);
  renderer->specials->nb += 4 * sizeof (int);
  fbox = (GLfloat*) (renderer->specials->content + renderer->specials->nb);
  renderer->specials->nb += 6 * sizeof (GLfloat);
  camera_position = (GLfloat*) (renderer->specials->content + renderer->specials->nb);
  renderer->specials->nb += 2 * sizeof (GLfloat);
  camera_z_axis = (GLfloat*) (renderer->specials->content + renderer->specials->nb);
  renderer->specials->nb += 2 * sizeof (GLfloat);
  ubox = (GLfloat*) (renderer->specials->content + renderer->specials->nb);
  renderer->specials->nb += 6 * sizeof (GLfloat);
  front_points = (GLfloat*) (renderer->specials->content + renderer->specials->nb);
  renderer->specials->nb += 12 * sizeof (GLfloat);
*/
  /* Render underwater effect if needed */
  P3_watercube_render_underwater (water, water->ubox, water->front_points);
  /* Render the surface of the watercube.
   * The surface is rendered in front to back order with writing in depth buffer enabled
   * to prevent water to be drawed many times on itself.
   */
  P3_renderer_active_context_over (renderer->c_context, water->context);

// TO DO ? frustum test with surface min and max height

  /* draw each patch */
  glLoadMatrixf (water->render_matrix);
  glDisable (GL_CULL_FACE);
  glEnable (GL_NORMALIZE);
  glDepthMask (GL_TRUE);
  glInterleavedArrays (GL_T2F_N3F_V3F, sizeof (P3_watercube_point), water->points);
  P3_material_activate (water->material);
//  if (water->material->option & P3_MATERIAL_ALPHA) glEnable (GL_BLEND);
  glBegin (GL_TRIANGLES);

#define RENDER_PATCHES_TOP_TO_BOTTOM(start, stop) \
  pj = (int) (start / water->patch_size); \
  for (j = start; j < stop; j += water->patch_size)

#define RENDER_PATCHES_BOTTOM_TO_TOP(start, stop) \
  pj = (int) (start / water->patch_size); \
  for (j = start; j >= stop; j -= water->patch_size)

#define RENDER_PATCHES_LEFT_TO_RIGHT(start, stop) \
  pi = (int) (start / water->patch_size); \
  if (((pi & 1) && (pj & 1)) || (!(pi & 1) && !(pj & 1))) { \
    k = 1; \
  } else { \
    k = 0; \
  } \
  for (i = start; i < stop; i += water->patch_size)

#define RENDER_PATCHES_RIGHT_TO_LEFT(start, stop) \
  pi = (int) (start / water->patch_size); \
  if (((pi & 1) && (pj & 1)) || (!(pi & 1) && !(pj & 1))) { \
    k = 1; \
  } else { \
    k = 0; \
  } \
  for (i = start; i >= stop; i -= water->patch_size)

#define RENDER_PATCH \
  P3_watercube_patch_render (water, i, j, k, water->fbox, water->camera_position); \
  k++

#define RENDER_LEFT_BEFORE \
  RENDER_PATCHES_RIGHT_TO_LEFT(cpx, pbox[0]) { \
    RENDER_PATCH; \
  } \
  RENDER_PATCHES_LEFT_TO_RIGHT(cpx + water->patch_size, pbox[2]) { \
    RENDER_PATCH; \
  }

#define RENDER_RIGHT_BEFORE \
  RENDER_PATCHES_LEFT_TO_RIGHT(cpx, pbox[2]) { \
    RENDER_PATCH; \
  } \
  RENDER_PATCHES_RIGHT_TO_LEFT(cpx - water->patch_size, pbox[0]) { \
    RENDER_PATCH; \
  }

  P3_watercube_get_patch (water, cpx, cpz, water->camera_position[0], water->camera_position[1]);
  if (cpx >= water->nb_patches_width) cpx = water->nb_patches_width - 1;
  if (cpz >= water->nb_patches_depth) cpz = water->nb_patches_depth - 1;
  cpx *= water->patch_size;
  cpz *= water->patch_size;
  if (cpx < 0) cpx = 0;
  if (cpz < 0) cpz = 0;
  if (water->camera_z_axis[1] > 0.0) {
    /* draw patches (cpz -> pbox[1]) before (cpz -> pbox[3]) */
    if (water->camera_z_axis[0] > 0.0) {
      /* draw patches (cpx -> pbox[0]) before (cpx -> pbox[2]) */
      RENDER_PATCHES_BOTTOM_TO_TOP(cpz, pbox[1]) {
        RENDER_LEFT_BEFORE
      }
      cpz += water->patch_size;
      RENDER_PATCHES_TOP_TO_BOTTOM(cpz, pbox[3]) {
        RENDER_LEFT_BEFORE
      }
    } else {
      /* draw patches (cpx -> pbox[2]) before (cpx -> pbox[0]) */
      RENDER_PATCHES_BOTTOM_TO_TOP(cpz, pbox[1]) {
        RENDER_RIGHT_BEFORE
      }
      cpz += water->patch_size;
      RENDER_PATCHES_TOP_TO_BOTTOM(cpz, pbox[3]) {
        RENDER_RIGHT_BEFORE
      }
    }
  } else {
    /* draw patches (cpz -> pbox[3]) before (cpz -> pbox[1]) */
    if (water->camera_z_axis[0] > 0.0) {
      /* draw patches (cpx -> pbox[0]) before (cpx -> pbox[2]) */
      RENDER_PATCHES_TOP_TO_BOTTOM(cpz, pbox[3]) {
        RENDER_LEFT_BEFORE
      }
      cpz -= water->patch_size;
      RENDER_PATCHES_BOTTOM_TO_TOP(cpz, pbox[1]) {
        RENDER_LEFT_BEFORE
      }
    } else {
      /* draw patches (cpx -> pbox[2]) before (cpx -> pbox[0]) */
      RENDER_PATCHES_TOP_TO_BOTTOM(cpz, pbox[3]) {
        RENDER_RIGHT_BEFORE
      }
      cpz -= water->patch_size;
      RENDER_PATCHES_BOTTOM_TO_TOP(cpz, pbox[1]) {
        RENDER_RIGHT_BEFORE
      }
    }
  }

#undef RENDER_PATCHES_TOP_TO_BOTTOM
#undef RENDER_PATCHES_BOTTOM_TO_TOP
#undef RENDER_PATCHES_LEFT_TO_RIGHT
#undef RENDER_PATCHES_RIGHT_TO_LEFT
#undef RENDER_PATCH
#undef RENDER_LEFT_BEFORE
#undef RENDER_RIGHT_BEFORE

  glEnd ();
  glDisableClientState (GL_VERTEX_ARRAY);
  glDisableClientState (GL_NORMAL_ARRAY);
  glDisableClientState (GL_TEXTURE_COORD_ARRAY);
  glEnable (GL_CULL_FACE);
  glDisable (GL_NORMALIZE);
  glDisable (GL_BLEND);
}


/*========+
 | SAVING |
 +========*/

void P3_watercube_get_data (P3_watercube* water, P3_chunk* chunk) {
  P3_chunk_save (chunk, water->m, 19 * sizeof (GLfloat));
  P3_chunk_save_int (chunk, water->option);
  P3_chunk_save (chunk, water->size,  3 * sizeof (GLfloat));
  P3_chunk_save (chunk, water->color, 4 * sizeof (GLfloat));
  P3_chunk_save_int (chunk, water->nb_points_width);
  P3_chunk_save_int (chunk, water->nb_points_depth);
  P3_chunk_save_int (chunk, water->patch_size);
  P3_chunk_save_int (chunk, water->max_camera_level);
  P3_chunk_save_int (chunk, water->nb_phases);
  P3_chunk_save_float (chunk, water->lod_linear);
  P3_chunk_save_float (chunk, water->wave_lod_linear);
  P3_chunk_save_float (chunk, water->wave_height);
  P3_chunk_save_float (chunk, water->speed);
}

void P3_watercube_set_data (P3_watercube* water, P3_chunk* chunk) {
  water->class = &P3_class_watercube;
  water->validity = P3_COORDSYS_INVALID;
  P3_chunk_load (chunk, water->m, 19 * sizeof (GLfloat));
  water->option = P3_chunk_load_int (chunk);
  P3_chunk_load (chunk, water->size,  3 * sizeof (GLfloat));
  P3_chunk_load (chunk, water->color, 4 * sizeof (GLfloat));
  water->nb_points_width  = P3_chunk_load_int (chunk);
  water->nb_points_depth  = P3_chunk_load_int (chunk);
  water->patch_size       = P3_chunk_load_int (chunk);
  water->max_camera_level = P3_chunk_load_int (chunk);
  water->nb_phases        = P3_chunk_load_int (chunk);
  water->lod_linear       = P3_chunk_load_float (chunk);
  water->wave_lod_linear  = P3_chunk_load_float (chunk);
  water->wave_height      = P3_chunk_load_float (chunk);
  water->speed            = P3_chunk_load_float (chunk);
  water->nb_waves = 0;
  water->waves    = NULL;
}

