/*
 * P3 python wrapper
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*****************************************
 * Copyright (C) 2002 Bertrand 'blam' LAMY
 *****************************************/

static PyObject* PyP3_GenericReduce (PyObject* a);
static PyObject* PyP3_EmptyFunc (PyObject* a, PyObject* args);
static void PyP3Point_Dealloc (P3_point* a);
static int PyP3Point_Init (P3_point* a, PyObject* args);
static int PyP3Point_Traverse (P3_point* a, visitproc visit, void* arg);
static int PyP3Point_Clear (P3_point* a);
static PyObject* PyP3Point_ConvertTo (P3_point* a, PyObject* csys);
static PyObject* PyP3Vector_ConvertTo (P3_vector* a, PyObject* csys);
static PyObject* PyP3Point_SetXYZ (P3_point* a, PyObject* args);
static PyObject* PyP3Point_GetRoot (P3_point* a);
static PyObject* PyP3Point_Move (P3_point* a, PyObject* arg);
static PyObject* PyP3Vector_Move (P3_vector* a, P3_vector* p);
static PyObject* PyP3Point_Copy (P3_point* a);
static PyObject* PyP3Vector_Copy (P3_point* a);
static PyObject* PyP3Point_Clone (P3_point* a, PyObject* arg);
static PyObject* PyP3Point_Mod (P3_point* a, PyObject* arg);
static PyObject* PyP3Vector_Mod (P3_point* a, PyObject* arg);
static PyObject* PyP3Point_Add (P3_point* a, P3_vector* arg);
static PyObject* PyP3Vector_Add (P3_point* a, P3_vector* arg);
static PyObject* PyP3Point_Sub (P3_point* a, P3_vector* arg);
static PyObject* PyP3Vector_Sub (P3_point* a, P3_vector* arg);
static PyObject* PyP3Point_AddXYZ (P3_point* a, PyObject* args);
static PyObject* PyP3Point_AddVector (P3_point* a, P3_vector* arg);
static PyObject* PyP3Point_AddMulVector (P3_point* a, PyObject* args);
static PyObject* PyP3Point_DistanceTo (P3_point* a, PyObject* arg);
static PyObject* PyP3Point_VectorTo (P3_point* a, PyObject* arg);
static PyObject* PyP3Point_GetState (P3_point* a);
static PyObject* PyP3Point_SetState (P3_point* a, PyObject* arg);
static PyObject* PyP3Point_Repr (P3_point* a);
static PyObject* PyP3Vector_Repr (P3_point* a);
static PyObject* PyP3Point_Eq (P3_point* a, PyObject* arg);
static PyObject* PyP3Point_Ne (P3_point* a, PyObject* arg);
static PyObject* PyP3Vector_CrossProduct (P3_vector* a, PyObject* args);
static PyObject* PyP3Vector_DotProduct (P3_vector* a, P3_vector* b);
static PyObject* PyP3Vector_Length (P3_vector* a);
static PyObject* PyP3Vector_SetLength (P3_vector* a, PyObject* arg);
static PyObject* PyP3Vector_Mul (P3_vector* a, PyObject* arg);
static PyObject* PyP3Vector_IMul (P3_vector* a, PyObject* arg);
static PyObject* PyP3Vector_Div (P3_vector* a, PyObject* arg);
static PyObject* PyP3Vector_IDiv (P3_vector* a, PyObject* arg);
static PyObject* PyP3Vector_Neg (P3_vector* a, PyObject* arg);
static PyObject* PyP3Vector_Abs (P3_vector* a, PyObject* arg);
static PyObject* PyP3Vector_Normalize (P3_vector* a);
static PyObject* PyP3Vector_SetStartEnd (P3_vector* a, PyObject* args);
static PyObject* PyP3Vector_AngleTo (P3_vector* a, P3_vector* b);
static void PyP3Coordsys_Dealloc (P3_coordsys* a);
static int PyP3Coordsys_Init (PyObject* self, PyObject* args, PyObject* kwds);
static int PyP3Child_Traverse (P3_child* a, visitproc visit, void* arg);
static int PyP3Child_Clear (P3_child* a);
static PyObject* PyP3Coordsys_Add (P3_coordsys* a, P3_vector* arg);
static PyObject* PyP3Coordsys_Sub (P3_coordsys* a, P3_vector* arg);
static PyObject* PyP3Coordsys_Mod (P3_coordsys* a, PyObject* arg);
static PyObject* PyP3Coordsys_Position (P3_coordsys* a);
static PyObject* PyP3Coordsys_Move (P3_coordsys* a, PyObject* arg);
static PyObject* PyP3Coordsys_AddVector (P3_coordsys* a, P3_vector* arg);
static PyObject* PyP3Coordsys_AddMulVector (P3_coordsys* a, PyObject* args);
static PyObject* PyP3Coordsys_DistanceTo (P3_coordsys* a, PyObject* arg);
static PyObject* PyP3Coordsys_VectorTo (P3_coordsys* a, PyObject* arg);
static PyObject* PyP3Coordsys_TransformPoint (P3_any_object* o, PyObject* args);
static PyObject* PyP3Coordsys_TransformVector (P3_any_object* o, PyObject* args);
static PyObject* PyP3Coordsys_Translate (P3_coordsys* a, PyObject* arg);
static PyObject* PyP3Coordsys_Shift (P3_coordsys* a, PyObject* arg);
static PyObject* PyP3Coordsys_SetXYZ (P3_coordsys* a, PyObject* arg);
static PyObject* PyP3Coordsys_Scale (P3_coordsys* a, PyObject* arg);
static PyObject* PyP3Coordsys_SetIdentity (P3_coordsys* a);
static PyObject* PyP3Coordsys_LookAtZ (P3_coordsys* a, PyObject* arg);
static PyObject* PyP3Coordsys_LookAtY (P3_coordsys* a, PyObject* arg);
static PyObject* PyP3Coordsys_LookAtX (P3_coordsys* a, PyObject* arg);
static PyObject* PyP3Coordsys_TurnLateral (P3_coordsys* a, PyObject* arg);
static PyObject* PyP3Coordsys_RotateLateral (P3_coordsys* a, PyObject* arg);
static PyObject* PyP3Coordsys_TurnVertical (P3_coordsys* a, PyObject* arg);
static PyObject* PyP3Coordsys_RotateVertical (P3_coordsys* a, PyObject* arg);
static PyObject* PyP3Coordsys_TurnIncline (P3_coordsys* a, PyObject* arg);
static PyObject* PyP3Coordsys_RotateIncline (P3_coordsys* a, PyObject* arg);
static PyObject* PyP3Coordsys_RotateAxe (P3_coordsys* a, PyObject* args);
static PyObject* PyP3Coordsys_Rotate (P3_coordsys* a, PyObject* args);
static PyObject* PyP3Coordsys_GetRoot (P3_coordsys* a);
static PyObject* PyP3Coordsys_Inside (P3_coordsys* a, PyObject* parent);
static PyObject* PyP3Coordsys_Raypick (P3_any_object* w, PyObject* args);
static PyObject* PyP3Coordsys_RaypickB (P3_any_object* w, PyObject* args);
static PyObject* PyP3Coordsys_SetAnimState (P3_coordsys* a, PyObject* args);
static PyObject* PyP3Coordsys_GetState (P3_coordsys* a);
static PyObject* PyP3Coordsys_SetState (P3_coordsys* a, PyObject* arg);

static char point_doc[] = "A Point is a 3D position.\n\n"
"Point(parent = None, x = 0.0, y = 0.0, z = 0.0)\n\n"
"Creates a new Point, with coordinates X, Y and Z, defined in the coordinates "
"system PARENT.";

static char point_get_root_doc[] = "Point.get_root() -> World\n\n"
"Gets the root parent of a Point. The root parent is the root World of the "
"hierarchy (often called the 'scene' object).";

static char point_set_xyz_doc[] = "Point.set_xyz(x, y, z)\n\n"
"Sets the coordinates of a point to X, Y and Z.";

static char point_move_doc[] = "Point.move(position)\n\n"
"Moves a Point to POSITION (a Point or another 3D object, such as a World, "
"a volume,...).\n"
"Coordinates system conversion is performed if needed (=if the Point and "
"POSITION are not defined in the same coordinates system).";

static char point_convert_to_doc[] = "Point.convert_to(parent)\n\n"
"Converts a Point to the coordinates system PARENT in place. The x, y and z "
"coordinates are modified, and the Point's parent is set to PARENT.";

static char point_copy_doc[] = """Point.copy() -> Point\n\n"
"Returns a copy of a Point.";

static char point_clone_doc[] = "Point.clone(other)\n\n"
"Changes IN PLACE this Point so as it is a clone of OTHER.";

static char point_add_xyz_doc[] = "Point.add_xyz(x, y, z)\n\n"
"Translates the coordinates of a point by X, Y and Z.";

static char point_add_vector_doc[] = "Point.add_vector(vector)\n\n"
"Translates a Point IN PLACE.\n"
"Coordinates system conversion is performed if needed (=if the Point and "
"VECTOR are not defined in the same coordinates system).\n"
"For Vector, add_vector means vectorial addition (translating a vector does "
"nothing !).";

static char point_add_mul_vector_doc[] = "Point.add_mul_vector(k, vector)\n\n"
"Translates a Point IN PLACE, by K * VECTOR.\n"
"Coordinates system conversion is performed if needed (=if the Point and "
"VECTOR are not defined in the same coordinates system).\n"
"For Vector, add_vector means vectorial addition (translating a vector does "
"nothing !).";

static char point_distance_to_doc[] = "Point.distance_to(other) -> float\n\n"
"Gets the distance between a Point and an OTHER.";

static char point_vector_to_doc[] = "Point.vector_to(other) -> Vector\n\n"
"Gets the vector that starts at a Point and ends at OTHER.";

static PyMethodDef PyP3Point_Methods[] = {
  { "vector_to",      (PyCFunction) PyP3Point_VectorTo,     METH_O,       point_vector_to_doc },
  { "distance_to",    (PyCFunction) PyP3Point_DistanceTo,   METH_O,       point_distance_to_doc },
  { "add_mul_vector", (PyCFunction) PyP3Point_AddMulVector, METH_VARARGS, point_add_mul_vector_doc },
  { "add_vector",     (PyCFunction) PyP3Point_AddVector,    METH_O,       point_add_vector_doc },
  { "add_xyz",        (PyCFunction) PyP3Point_AddXYZ,       METH_VARARGS, point_add_xyz_doc },
  { "clone",          (PyCFunction) PyP3Point_Clone,        METH_O,       point_clone_doc },
  { "copy",           (PyCFunction) PyP3Point_Copy,         METH_NOARGS,  point_copy_doc },
  { "position",       (PyCFunction) PyP3Point_Copy,         METH_NOARGS,  point_copy_doc },
  { "move",           (PyCFunction) PyP3Point_Move,         METH_O,       point_move_doc },
  { "get_root",       (PyCFunction) PyP3Point_GetRoot,      METH_NOARGS,  point_get_root_doc },
  { "convert_to",     (PyCFunction) PyP3Point_ConvertTo,    METH_O,       point_convert_to_doc },
  { "set_xyz",        (PyCFunction) PyP3Point_SetXYZ,       METH_VARARGS, point_set_xyz_doc },
  { "__eq__",         (PyCFunction) PyP3Point_Eq,           METH_O,       NULL },
  { "__ne__",         (PyCFunction) PyP3Point_Ne,           METH_O,       NULL },
  { "__getstate__",   (PyCFunction) PyP3Point_GetState,     METH_NOARGS,  NULL },
  { "__setstate__",   (PyCFunction) PyP3Point_SetState,     METH_O,       NULL },
  { "__reduce__",     (PyCFunction) PyP3_GenericReduce,     METH_NOARGS,  NULL },
  { NULL, NULL }
};

static char vector_doc[] = "A Vector is a 3D vector (and not a kind of list or sequence ;-). Vectors are "
"usefull for 3D math computation.\n"
"Most of the math operator, such as +, -, *, /, abs,... work on Vectors and do "
"what they are intended to do ;-)\n"
"Vector inherits from Point for practical reasons, since both have x, y, z "
"coordinates and a parent.";

static char vector_move_doc[] = "Vector.move(position)\n\n"
"Moves a Vector to POSITION (a Vector).\n"
"Coordinates system conversion is performed if needed (=if the Vector and "
"POSITION are not defined in the same coordinates system).";

static char vector_convert_to_doc[] = "Vector.convert_to(parent)\n\n"
"Converts a Vector to the coordinates system PARENT in place. The x, y and z "
"coordinates are modified, and the Vector's parent is set to PARENT.";

static char vector_copy_doc[] = "Vector.copy() -> Vector\n\n"
"Returns a copy of a Vector.";
  
static char vector_cross_product_doc[] = "Vector.cross_product(VECTOR) -> Vector\n\n"
"Returns the cross product of a Vector with VECTOR.\n\n"
"Vector.cross_product(VECTOR, RESULT)\n\n"
"The cross product is computed IN PLACE of result.";
  
static char vector_dot_product_doc[] = "Vector.dot_product(VECTOR) -> float\n\n"
"Returns the dot product of a Vector with VECTOR.";
  
static char vector_length_doc[] = "Vector.length() -> float\n\n"
"Gets the length of a Vector.";
    
static char vector_set_length_doc[] = "Vector.set_length(new_length)\n\n"
"Sets the length of a Vector to NEW_LENGTH. The Vector's coordinates are "
"multiplicated as needed.";

static char vector_normalize_doc[] = "Vector.normalize()\n\n"
"Normalizes a Vector IN PLACE.";

static char vector_set_start_end_doc[] = "Vector.set_start_end(start, end)\n\n"
"Sets this vector IN PLACE so as it correspond to the vector start->end.";
      
static char vector_angle_to_doc[] = "Vector.angle_to(VECTOR) -> angle in degree\n\n"
"Computes the angle between this Vector and VECTOR.";

static PyMethodDef PyP3Vector_Methods[] = {
  { "move",          (PyCFunction) PyP3Vector_Move,         METH_O,       vector_move_doc },
  { "copy",          (PyCFunction) PyP3Vector_Copy,         METH_NOARGS,  vector_copy_doc },
  { "convert_to",    (PyCFunction) PyP3Vector_ConvertTo,    METH_O,       vector_convert_to_doc },
  { "cross_product", (PyCFunction) PyP3Vector_CrossProduct, METH_VARARGS, vector_cross_product_doc },
  { "dot_product",   (PyCFunction) PyP3Vector_DotProduct,   METH_O,       vector_dot_product_doc },
  { "length",        (PyCFunction) PyP3Vector_Length,       METH_NOARGS,  vector_length_doc },
  { "set_length",    (PyCFunction) PyP3Vector_SetLength,    METH_O,       vector_set_length_doc },
  { "normalize",     (PyCFunction) PyP3Vector_Normalize,    METH_NOARGS,  vector_normalize_doc },
  { "set_start_end", (PyCFunction) PyP3Vector_SetStartEnd,  METH_VARARGS, vector_set_start_end_doc },
  { "angle_to",      (PyCFunction) PyP3Vector_AngleTo,      METH_O,       vector_angle_to_doc },
  { NULL, NULL }
};

static char element_doc[] = "Element3D\n\n"
"Base class for all graphical 3D elements (but not necessary rendered).";

static char coordsys_position_doc[] = "Element3D.position() -> Point\n\n"
"Returns the position of a Element3D, as a new Point.";

static char coordsys_move_doc[] = "Element3D.move(position)\n\n"
"Moves a Element3D to POSITION (a Point or another 3D object, such as "
"a World, a volume,...).\n"
"Coordinates system conversion is performed is needed (=if the Element3D "
"and POSITION are not defined in the same coordinates system).";

static char coordsys_add_vector_doc[] = "Element3D.add_vector(vector)\n\n"
"Translates a Element3D IN PLACE.\n"
"Coordinates system conversion is performed is needed (=if the Element3D and "
"POSITION are not defined in the same coordinates system).";

static char coordsys_add_mul_vector_doc[] = "Element3D.add_vector(vector, k)\n\n"
"Translates a Element3D IN PLACE, by K * VECTOR.\n"
"Coordinates system conversion is performed is needed (=if the Element3D and "
"POSITION are not defined in the same coordinates system).";

static char coordsys_distance_to_doc[] = "Element3D.distance_to(other) -> float\n\n"
"Gets the distance between a Element3D and OTHER.";

static char coordsys_vector_to_doc[] = "Element3D.vector_to(other) -> Vector\n\n"
"Gets the vector that starts at a Element3D and ends at OTHER.";

static char coordsys_look_at_doc[] = "Element3D.look_at(pos)\n\n"
"Rotates a Element3D so as it looks toward POS. The Element3D is "
"understood to 'look' at its -Z direction.";

static PyMethodDef PyP3Element_Methods[] = {
  { "vector_to",        (PyCFunction) PyP3Coordsys_VectorTo,        METH_O,       coordsys_vector_to_doc },
  { "distance_to",      (PyCFunction) PyP3Coordsys_DistanceTo,      METH_O,       coordsys_distance_to_doc },
  { "add_mul_vector",   (PyCFunction) PyP3Coordsys_AddMulVector,    METH_VARARGS, coordsys_add_mul_vector_doc },
  { "add_vector",       (PyCFunction) PyP3Coordsys_AddVector,       METH_O,       coordsys_add_vector_doc },
  { "position",         (PyCFunction) PyP3Coordsys_Position,        METH_NOARGS,  coordsys_position_doc },
  { "move",             (PyCFunction) PyP3Coordsys_Move,            METH_O,       coordsys_move_doc },
  { "get_root",         (PyCFunction) PyP3Coordsys_GetRoot,         METH_NOARGS,  NULL },
  { "set_xyz",          (PyCFunction) PyP3Coordsys_SetXYZ,          METH_VARARGS, NULL },
  { "raypick",          (PyCFunction) PyP3Coordsys_Raypick,         METH_VARARGS, NULL },
  { "raypick_b",        (PyCFunction) PyP3Coordsys_RaypickB,        METH_VARARGS, NULL },
  { "get_root",         (PyCFunction) PyP3Coordsys_GetRoot,         METH_NOARGS,  NULL },
  { "translate",        (PyCFunction) PyP3Coordsys_Translate,       METH_VARARGS, NULL },
  { "shift",            (PyCFunction) PyP3Coordsys_Shift,           METH_VARARGS, NULL },
  { "set_xyz",          (PyCFunction) PyP3Coordsys_SetXYZ,          METH_VARARGS, NULL },
  { "rotate_incline",   (PyCFunction) PyP3Coordsys_RotateIncline,   METH_O,       NULL },
  { "rotate_vertical",  (PyCFunction) PyP3Coordsys_RotateVertical,  METH_O,       NULL },
  { "rotate_lateral",   (PyCFunction) PyP3Coordsys_RotateLateral,   METH_O,       NULL },
  { "rotate_axe",       (PyCFunction) PyP3Coordsys_RotateAxe,       METH_VARARGS, NULL },
  { "rotate",           (PyCFunction) PyP3Coordsys_Rotate,          METH_VARARGS, NULL },
  { "turn_incline",     (PyCFunction) PyP3Coordsys_TurnIncline,     METH_O,       NULL },
  { "turn_vertical",    (PyCFunction) PyP3Coordsys_TurnVertical,    METH_O,       NULL },
  { "turn_lateral",     (PyCFunction) PyP3Coordsys_TurnLateral,     METH_O,       NULL },
  { "look_at",          (PyCFunction) PyP3Coordsys_LookAtZ,         METH_O,       coordsys_look_at_doc },
  { "look_at_y",        (PyCFunction) PyP3Coordsys_LookAtY,         METH_O,       NULL },
  { "look_at_x",        (PyCFunction) PyP3Coordsys_LookAtX,         METH_O,       NULL },
  { "set_identity",     (PyCFunction) PyP3Coordsys_SetIdentity,     METH_NOARGS,  NULL },
  { "scale",            (PyCFunction) PyP3Coordsys_Scale,           METH_VARARGS, NULL },
  { "transform_point",  (PyCFunction) PyP3Coordsys_TransformPoint,  METH_VARARGS, NULL },
  { "transform_vector", (PyCFunction) PyP3Coordsys_TransformVector, METH_VARARGS, NULL },
  { "distance_to",      (PyCFunction) PyP3Coordsys_DistanceTo,      METH_O,       NULL },
  { "inside",           (PyCFunction) PyP3Coordsys_Inside,          METH_O,       NULL },
  { "set_anim_state",   (PyCFunction) PyP3Coordsys_SetAnimState,    METH_VARARGS, NULL },
  { "begin_round",      (PyCFunction) PyP3_EmptyFunc,               METH_VARARGS, NULL },
  { "advance_time",     (PyCFunction) PyP3_EmptyFunc,               METH_VARARGS, NULL },
  { "end_round",        (PyCFunction) PyP3_EmptyFunc,               METH_VARARGS, NULL },
  { "_getstate",        (PyCFunction) PyP3Coordsys_GetState,        METH_NOARGS,  NULL },
  { "_setstate",        (PyCFunction) PyP3Coordsys_SetState,        METH_O,       NULL },
  { NULL, NULL }
};


/*---------+
 | Get Set |
 +---------*/

PY_GET_SET_ON_FLOAT  (Point, P3_point*, X, coord[0])
PY_GET_SET_ON_FLOAT  (Point, P3_point*, Y, coord[1])
PY_GET_SET_ON_FLOAT  (Point, P3_point*, Z, coord[2])
PY_GET_SET_ON_OBJECT (Point, P3_point*, Parent, parent, P3_coordsys*)

static PyGetSetDef PyP3Point_GetSets[] = {
  { "parent", (getter) PyP3Point_GetParent, (setter) PyP3Point_SetParent, NULL },
  { "x",      (getter) PyP3Point_GetX,      (setter) PyP3Point_SetX,      NULL },
  { "y",      (getter) PyP3Point_GetY,      (setter) PyP3Point_SetY,      NULL },
  { "z",      (getter) PyP3Point_GetZ,      (setter) PyP3Point_SetZ,      NULL },
  { NULL }
};

PY_GET_SET_ON_FLOAT_ADD       (Coordsys, P3_coordsys*, X, m[12], P3_object_invalid ((P3_any_object*) a);)
PY_GET_SET_ON_FLOAT_ADD       (Coordsys, P3_coordsys*, Y, m[13], P3_object_invalid ((P3_any_object*) a);)
PY_GET_SET_ON_FLOAT_ADD       (Coordsys, P3_coordsys*, Z, m[14], P3_object_invalid ((P3_any_object*) a);)
PY_GET_SET_ON_FLOAT_ARRAY_ADD (Coordsys, P3_coordsys*, Matrix, m, 19, P3_object_invalid ((P3_any_object*) a);)
PY_GET_SET_ON_OBJECT_ADD      (Child, P3_child*, Parent, parent, P3_coordsys*, P3_object_invalid ((P3_any_object*) a);)

static PyObject* PyP3Coordsys_GetXScale (P3_coordsys* a, void* context) {
  return PyFloat_FromDouble((double) a->m[16]);
}

static PyObject* PyP3Coordsys_GetYScale (P3_coordsys* a, void* context) {
  return PyFloat_FromDouble((double) a->m[17]);
}

static PyObject* PyP3Coordsys_GetZScale (P3_coordsys* a, void* context) {
  return PyFloat_FromDouble((double) a->m[18]);
}

static int PyP3Coordsys_SetXScale (P3_coordsys* a, PyObject* value, void* context) {
  P3_matrix_scale (a->m, (GLfloat) PyFloat_AS_DOUBLE (value) / a->m[16], 1.0, 1.0);
  P3_object_invalid ((P3_any_object*) a);
  return 0;
}

static int PyP3Coordsys_SetYScale (P3_coordsys* a, PyObject* value, void* context) {
  P3_matrix_scale (a->m, 1.0, (GLfloat) PyFloat_AS_DOUBLE (value) / a->m[17], 1.0);
  P3_object_invalid ((P3_any_object*) a);
  return 0;
}

static int PyP3Coordsys_SetZScale (P3_coordsys* a, PyObject* value, void* context) {
  P3_matrix_scale (a->m, 1.0, 1.0, (GLfloat) PyFloat_AS_DOUBLE (value) / a->m[18]);
  P3_object_invalid ((P3_any_object*) a);
  return 0;
}

static int PyP3Child_ChangeParent (P3_child* a, PyObject* value, void* context) {
  PyObject* p;
  if (a->parent != NULL) {
    p = PyObject_CallMethod (((P3_world*) a->parent)->children, "remove", "O", a);
    Py_XDECREF (p);
  }
  if (value == Py_None || value == NULL) {
    a->parent = NULL;
  } else {
    a->parent = (P3_coordsys*) value;
    Py_INCREF (value);
    p = PyObject_CallMethod (((P3_world*) value)->children, "append", "O", a);
  }
  P3_object_invalid ((P3_any_object*) a);
  return 0;
}

static PyGetSetDef PyP3Element_GetSets[] = {
  { "visible", (getter) PyP3Object_GetVisible,  (setter) PyP3Object_SetVisible,   NULL },
  { "solid",   (getter) PyP3Object_GetSolid,    (setter) PyP3Object_SetSolid,     NULL },
  { "_parent", (getter) PyP3Child_GetParent,    (setter) PyP3Child_SetParent,     NULL },
  { "parent",  (getter) PyP3Child_GetParent,    (setter) PyP3Child_ChangeParent,  NULL },
  { "x",       (getter) PyP3Coordsys_GetX,      (setter) PyP3Coordsys_SetX,       NULL },
  { "y",       (getter) PyP3Coordsys_GetY,      (setter) PyP3Coordsys_SetY,       NULL },
  { "z",       (getter) PyP3Coordsys_GetZ,      (setter) PyP3Coordsys_SetZ,       NULL },
  { "scale_x", (getter) PyP3Coordsys_GetXScale, (setter) PyP3Coordsys_SetXScale,  NULL },
  { "scale_y", (getter) PyP3Coordsys_GetYScale, (setter) PyP3Coordsys_SetYScale,  NULL },
  { "scale_z", (getter) PyP3Coordsys_GetZScale, (setter) PyP3Coordsys_SetZScale,  NULL },
  { "_matrix", (getter) PyP3Coordsys_GetMatrix, (setter) PyP3Coordsys_SetMatrix,  NULL },
  { NULL }
};


/*------+
 | Type |
 +------*/

static PyNumberMethods PyP3Point_AsNumber = {
  (binaryfunc) PyP3Point_Add, /*nb_add*/
  (binaryfunc) PyP3Point_Sub, /*nb_subtract*/
  0, /*nb_multiply*/
  0, /*nb_divide*/
  (binaryfunc) PyP3Point_Mod, /*nb_remainder*/
  0, /*nb_divmod*/
  0, /*nb_power*/
  0, /*nb_negative*/
  0, /*nb_positive*/
  0, /*nb_absolute*/
  0, /*nb_nonzero*/
  0, /*nb_invert*/
  0, /*nb_lshift*/
  (binaryfunc) PyP3Point_VectorTo, /*nb_rshift*/
  0, /*nb_and*/
  0, /*nb_xor*/
  0, /*nb_or*/
  0, /*nb_coerce*/
  0, /*nb_int*/
  0, /*nb_long*/
  0, /*nb_float*/
  0, /* nb_oct */
  0, /* nb_hex */
  (binaryfunc) PyP3Point_AddVector, /* nb_inplace_add */
  0, /* nb_inplace_subtract */
  0, /* nb_inplace_multiply */
  0, /* nb_inplace_divide */
  (binaryfunc) PyP3Point_ConvertTo, /* nb_inplace_remainder */
  0, /* nb_inplace_power */
  0, /* nb_inplace_lshift */
  0, /* nb_inplace_rshift */
  0, /* nb_inplace_and */
  0, /* nb_inplace_xor */
  0, /* nb_inplace_or */
  0, /* nb_floor_divide */
  0, /* nb_true_divide */
  0, /* nb_inplace_floor_divide */
  0, /* nb_inplace_true_divide */
};

PyTypeObject PyP3Point_Type = {
  PyObject_HEAD_INIT(NULL)
  0,
  "_soya._Point",
  sizeof(P3_point),
  0,
  (destructor) PyP3Point_Dealloc,/* tp_dealloc */
  0,/* tp_print */
  0,/* tp_getattr */
  0,/* tp_setattr */
  0,/* tp_compare */
  (reprfunc) PyP3Point_Repr,/* tp_repr */
  (PyNumberMethods*) &PyP3Point_AsNumber,/* tp_as_number */
  0,/* tp_as_sequence */
  0,/* tp_as_mapping */
  0,/* tp_hash */
  0,/* tp_call */
  0,/* tp_str */
  PYP3_GENERIC_GETATTR,/* tp_getattro */
  PYP3_GENERIC_SETATTR,/* tp_setattro */
  0,/* tp_as_buffer */
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE | Py_TPFLAGS_CHECKTYPES | Py_TPFLAGS_HAVE_GC,/* tp_flags */
  point_doc,/* tp_doc */
  (traverseproc) PyP3Point_Traverse,/* tp_traverse */
  (inquiry) PyP3Point_Clear,/* tp_clear */
  0,/* tp_richcompare */
  0,/* tp_weaklistoffset */
  0,/* tp_iter */
  0,/* tp_iternext */
  (PyMethodDef*) &PyP3Point_Methods,/* tp_methods */
  0,/* tp_members */
  (PyGetSetDef*) &PyP3Point_GetSets,/* tp_getset */
  0,/* tp_base */
  0,/* tp_dict */
  0,/* tp_descr_get */
  0,/* tp_descr_set */
  0,/* tp_dictoffset */
  (initproc) PyP3Point_Init,/* tp_init */
  PYP3_GENERIC_ALLOC,/* tp_alloc */
  (newfunc) PYP3_GENERIC_NEW,/* tp_new */
  PYP3_GENERIC_GC_FREE,/* tp_free */
};

static PyNumberMethods PyP3Vector_AsNumber = {
  (binaryfunc) PyP3Vector_Add, /*nb_add*/
  (binaryfunc) PyP3Vector_Sub, /*nb_subtract*/
  (binaryfunc) PyP3Vector_Mul, /*nb_multiply*/
  (binaryfunc) PyP3Vector_Div, /*nb_divide*/
  (binaryfunc) PyP3Vector_Mod, /*nb_remainder*/
  0, /*nb_divmod*/
  0, /*nb_power*/
  (unaryfunc) PyP3Vector_Neg, /*nb_negative*/
  0, /*nb_positive*/
  (unaryfunc) PyP3Vector_Abs, /*nb_absolute*/
  0, /*nb_nonzero*/
  0, /*nb_invert*/
  0, /*nb_lshift*/
  0, /*nb_rshift*/
  0, /*nb_and*/
  0, /*nb_xor*/
  0, /*nb_or*/
  0, /*nb_coerce*/
  0, /*nb_int*/
  0, /*nb_long*/
  0, /*nb_float*/
  0, /* nb_oct */
  0, /* nb_hex */
  (binaryfunc) PyP3Point_AddVector, /* nb_inplace_add */
  0, /* nb_inplace_subtract */
  (binaryfunc) PyP3Vector_IMul, /* nb_inplace_multiply */
  (binaryfunc) PyP3Vector_IDiv, /* nb_inplace_divide */
  (binaryfunc) PyP3Vector_ConvertTo, /* nb_inplace_remainder */
  0, /* nb_inplace_power */
  0, /* nb_inplace_lshift */
  0, /* nb_inplace_rshift */
  0, /* nb_inplace_and */
  0, /* nb_inplace_xor */
  0, /* nb_inplace_or */
  0, /* nb_floor_divide */
  (binaryfunc) PyP3Vector_Div, /* nb_true_divide */
  0, /* nb_inplace_floor_divide */
  0, /* nb_inplace_true_divide */
};

PyTypeObject PyP3Vector_Type = {
  PyObject_HEAD_INIT(NULL)
  0,
  "_soya._Vector",
  sizeof(P3_vector),
  0,
  (destructor) PyP3Point_Dealloc,/* tp_dealloc */
  0,/* tp_print */
  0,/* tp_getattr */
  0,/* tp_setattr */
  0,/* tp_compare */
  (reprfunc) PyP3Vector_Repr,/* tp_repr */
  (PyNumberMethods*) &PyP3Vector_AsNumber,/* tp_as_number */
  0,/* tp_as_sequence */
  0,/* tp_as_mapping */
  0,/* tp_hash */
  0,/* tp_call */
  0,/* tp_str */
  PYP3_GENERIC_GETATTR,/* tp_getattro */
  PYP3_GENERIC_SETATTR,/* tp_setattro */
  0,/* tp_as_buffer */
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE | Py_TPFLAGS_CHECKTYPES | Py_TPFLAGS_HAVE_GC,/* tp_flags */
  vector_doc,/* tp_doc */
  (traverseproc) PyP3Point_Traverse,/* tp_traverse */
  (inquiry) PyP3Point_Clear,/* tp_clear */
  0,/* tp_richcompare */
  0,/* tp_weaklistoffset */
  0,/* tp_iter */
  0,/* tp_iternext */
  (PyMethodDef*) &PyP3Vector_Methods,/* tp_methods */
  0,/* tp_members */
  (PyGetSetDef*) 0,/* tp_getset */
  &PyP3Point_Type,/* tp_base */
  0,/* tp_dict */
  0,/* tp_descr_get */
  0,/* tp_descr_set */
  0,/* tp_dictoffset */
  (initproc) PyP3Point_Init,/* tp_init */
  PYP3_GENERIC_ALLOC,/* tp_alloc */
  (newfunc) PYP3_GENERIC_NEW,/* tp_new */
  PYP3_GENERIC_GC_FREE,/* tp_free */
};

static PyNumberMethods PyP3Coordsys_AsNumber = {
  (binaryfunc) PyP3Coordsys_Add, /*nb_add*/
  (binaryfunc) PyP3Coordsys_Sub, /*nb_subtract*/
  0, /*nb_multiply*/
  0, /*nb_divide*/
  (binaryfunc) PyP3Coordsys_Mod, /*nb_remainder*/
  0, /*nb_divmod*/
  0, /*nb_power*/
  0, /*nb_negative*/
  0, /*nb_positive*/
  0, /*nb_absolute*/
  0, /*nb_nonzero*/
  0, /*nb_invert*/
  0, /*nb_lshift*/
  (binaryfunc) PyP3Coordsys_VectorTo, /*nb_rshift*/
  0, /*nb_and*/
  0, /*nb_xor*/
  0, /*nb_or*/
  0, /*nb_coerce*/
  0, /*nb_int*/
  0, /*nb_long*/
  0, /*nb_float*/
  0, /* nb_oct */
  0, /* nb_hex */
  (binaryfunc) PyP3Coordsys_AddVector, /* nb_inplace_add */
  0, /* nb_inplace_subtract */
  0, /* nb_inplace_multiply */
  0, /* nb_inplace_divide */
  0, /* nb_inplace_remainder */
  0, /* nb_inplace_power */
  0, /* nb_inplace_lshift */
  0, /* nb_inplace_rshift */
  0, /* nb_inplace_and */
  0, /* nb_inplace_xor */
  0, /* nb_inplace_or */
  0, /* nb_floor_divide */
  0, /* nb_true_divide */
  0, /* nb_inplace_floor_divide */
  0, /* nb_inplace_true_divide */
};

PyTypeObject PyP3Element_Type = {
  PyObject_HEAD_INIT(NULL)
  0,
  "soya.soya3d.Element3D",
  sizeof(P3_coordsys),
  0,
  (destructor) PyP3Coordsys_Dealloc,/* tp_dealloc */
  0,/* tp_print */
  0,/* tp_getattr */
  0,/* tp_setattr */
  0,/* tp_compare */
  0,/* tp_repr */
  (PyNumberMethods*) &PyP3Coordsys_AsNumber,/* tp_as_number */
  0,/* tp_as_sequence */
  0,/* tp_as_mapping */
  0,/* tp_hash */
  0,/* tp_call */
  0,/* tp_str */
  PYP3_GENERIC_GETATTR,/* tp_getattro */
  PYP3_GENERIC_SETATTR,/* tp_setattro */
  0,/* tp_as_buffer */
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE | Py_TPFLAGS_CHECKTYPES | Py_TPFLAGS_HAVE_GC,/* tp_flags */
  element_doc,/* tp_doc */
  (traverseproc) PyP3Child_Traverse,/* tp_traverse */
  (inquiry) PyP3Child_Clear,/* tp_clear */
  0,/* tp_richcompare */
  0,/* tp_weaklistoffset */
  0,/* tp_iter */
  0,/* tp_iternext */
  (PyMethodDef*) &PyP3Element_Methods,/* tp_methods */
  0,/* tp_members */
  (PyGetSetDef*) &PyP3Element_GetSets,/* tp_getset */
  0,/* tp_base */
  0,/* tp_dict */
  0,/* tp_descr_get */
  0,/* tp_descr_set */
  0,/* tp_dictoffset */
  (initproc) PyP3Coordsys_Init,/* tp_init */
  PYP3_GENERIC_ALLOC,/* tp_alloc */
  (newfunc) PYP3_GENERIC_NEW,/* tp_new */
  PYP3_GENERIC_GC_FREE,/* tp_free */
};


/*---------+
 | Methods |
 +---------*/

static PyObject* PyP3_GenericReduce (PyObject* a) {
  PyObject* tuple;
  PyObject* tuple2;
  PyObject* soya;
  PyObject* dict;
  PyObject* reconstructor;
  tuple = PyTuple_New (3);
  soya = PyImport_ImportModule ("soya");
  if (soya == NULL) {
    P3_error ("Failed to open module soya");
    Py_INCREF (Py_None);
    return Py_None;
  } else {
    dict = PyModule_GetDict (soya);
    tuple2 = PyTuple_New (1);
    PyTuple_SET_ITEM (tuple2, 0, PyObject_Type (a));
    reconstructor = PyDict_GetItemString (dict, "_soya_reconstructor");
    Py_INCREF (reconstructor);
    PyTuple_SET_ITEM (tuple, 0, reconstructor);
    PyTuple_SET_ITEM (tuple, 1, tuple2);
    PyTuple_SET_ITEM (tuple, 2, PyObject_CallMethod (a, "__getstate__", NULL));
    Py_DECREF (soya);
    return tuple;
  }
}

static void PyP3Point_Dealloc (P3_point* a) {
  PyObject_GC_UnTrack ((PyObject*) a);
  Py_XDECREF (a->parent);
  a->ob_type->tp_free ((PyObject*) a);
}

static int PyP3Point_Init (P3_point* a, PyObject* args) {
  int nb;
  if (args == Py_None) {
    nb = 0;
  } else {
    nb = PySequence_Size (args);
  }
  if (nb == 0) {
    a->parent = NULL;
    a->coord[0] = 0.0;
    a->coord[1] = 0.0;
    a->coord[2] = 0.0;
  } else {
    a->parent = (P3_coordsys*) PySequence_Fast_GET_ITEM (args, 0);
    if ((PyObject*) a->parent == Py_None) {
      a->parent = NULL;
    } else {
      Py_INCREF ((PyObject*) a->parent);
    }
    if (nb == 4) {
      a->coord[0] = (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 1));
      a->coord[1] = (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 2));
      a->coord[2] = (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 3));
    } else {
      a->coord[0] = 0.0;
      a->coord[1] = 0.0;
      a->coord[2] = 0.0;
    }
  }
  return 0;
}

static int PyP3Point_Traverse (P3_point* a, visitproc visit, void* arg) {
	int err;
  if (a->parent != NULL) {
    err = visit ((PyObject*) a->parent, arg);
    if (err) { return err; }
  }
	return 0;
}

static int PyP3Point_Clear (P3_point* a) {
  Py_XDECREF (a->parent);
  a->parent = NULL;
  return 0;
}

static void PyP3_GetPosition (PyObject* a, GLfloat** coord, P3_coordsys** csys) {
  if (PyObject_IsInstance ((PyObject*) a, (PyObject*) &PyP3Point_Type) == 1) {
    *coord = ((P3_point*) a)->coord;
    *csys  = ((P3_point*) a)->parent;
  } else {
    /* assume it's a P3_any_object */
    int id = ((P3_any_object*) a)->class->id;
    if (id == P3_ID_SPRITE || id == P3_ID_CYLINDER) {
      *coord = ((P3_sprite*) a)->position;
    } else {
      /* assume it's a coordsys */
      *coord = ((P3_coordsys*) a)->m + 12;
    }
    *csys = ((P3_child*) a)->parent;
  }
}

static void PyP3_GetPositionInto (PyObject* a, P3_coordsys* csys, GLfloat* rez) {
  if (PyObject_IsInstance ((PyObject*) a, (PyObject*) &PyP3Point_Type) == 1) {
    P3_point* p = (P3_point*) a;
    if (p->parent == NULL || csys == NULL) {
      memcpy (rez, p->coord, 3 * sizeof (GLfloat));
    } else {
      P3_point_by_matrix_copy (rez, p->coord, P3_coordsys_get_root_matrix (p->parent));
      P3_point_by_matrix (rez, P3_coordsys_get_inverted_root_matrix (csys));
    }
  } else {
    /* assume it's a P3_any_object */
    int id = ((P3_any_object*) a)->class->id;
    if (id == P3_ID_SPRITE || id == P3_ID_CYLINDER) {
      P3_sprite* s = (P3_sprite*) a;
      if (s->parent == NULL || csys == NULL) {
        memcpy (rez, s->position, 3 * sizeof (GLfloat));
      } else {
        P3_point_by_matrix_copy (rez, s->position, P3_coordsys_get_root_matrix (s->parent));
        P3_point_by_matrix (rez, P3_coordsys_get_inverted_root_matrix (csys));
      }
    } else {
      /* assume it's a coordsys */
      P3_coordsys* c = (P3_coordsys*) a;
      if (c->parent == NULL || csys == NULL) {
        memcpy (rez, c->m + 12, 3 * sizeof (GLfloat));
      } else {
        P3_point_by_matrix_copy (rez, c->m + 12, P3_coordsys_get_root_matrix (c->parent));
        P3_point_by_matrix (rez, P3_coordsys_get_inverted_root_matrix (csys));
      }
    }
  }
}

static void PyP3_GetPositionIntoNull (PyObject* a, GLfloat* rez) {
  if (PyObject_IsInstance ((PyObject*) a, (PyObject*) &PyP3Point_Type) == 1) {
    P3_point* p = (P3_point*) a;
    if (p->parent == NULL) {
      memcpy (rez, p->coord, 3 * sizeof (GLfloat));
    } else {
      P3_point_by_matrix_copy (rez, p->coord, P3_coordsys_get_root_matrix (p->parent));
    }
  } else {
    /* assume it's a P3_any_object */
    int id = ((P3_any_object*) a)->class->id;
    if (id == P3_ID_SPRITE || id == P3_ID_CYLINDER) {
      P3_sprite* s = (P3_sprite*) a;
      if (s->parent == NULL) {
        memcpy (rez, s->position, 3 * sizeof (GLfloat));
      } else {
        P3_point_by_matrix_copy (rez, s->position, P3_coordsys_get_root_matrix (s->parent));
      }
    } else {
      /* assume it's a coordsys */
      P3_coordsys* c = (P3_coordsys*) a;
      if (c->parent == NULL) {
        memcpy (rez, c->m + 12, 3 * sizeof (GLfloat));
      } else {
        P3_point_by_matrix_copy (rez, c->m + 12, P3_coordsys_get_root_matrix (c->parent));
      }
    }
  }
}

static PyObject* PyP3Point_ConvertTo (P3_point* a, PyObject* csys) {
  if (csys != Py_None && a->parent != NULL) {
    P3_point_by_matrix (a->coord, P3_coordsys_get_root_matrix (a->parent));
    P3_point_by_matrix (a->coord, P3_coordsys_get_inverted_root_matrix ((P3_coordsys*) csys));
  }
  Py_XDECREF ((PyObject*) a->parent);
  Py_INCREF (csys);
  a->parent = (P3_coordsys*) csys;
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Vector_ConvertTo (P3_vector* a, PyObject* csys) {
  if (csys != Py_None && a->parent != NULL) {
    P3_vector_by_matrix (a->coord, P3_coordsys_get_root_matrix (a->parent));
    P3_vector_by_matrix (a->coord, P3_coordsys_get_inverted_root_matrix ((P3_coordsys*) csys));
  }
  Py_XDECREF ((PyObject*) a->parent);
  Py_INCREF (csys);
  a->parent = (P3_coordsys*) csys;
  Py_INCREF ((PyObject*) a);
  return (PyObject*) a;
}

static PyObject* PyP3Point_SetXYZ (P3_point* a, PyObject* args) {
  a->coord[0] = (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 0));
  a->coord[1] = (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 1));
  a->coord[2] = (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 2));
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Point_GetRoot (P3_point* a) {
  P3_coordsys* c;
  if (a->parent == NULL) {
    Py_INCREF (Py_None);
    return Py_None;
  } else {
    c = P3_coordsys_get_root (a->parent);
    if (c == NULL) {
      Py_INCREF (Py_None);
      return Py_None;
    } else {
      Py_INCREF ((PyObject*) c);
      return (PyObject*) c;
    }
  }
}

static PyObject* PyP3Point_Move (P3_point* a, PyObject* arg) {
  PyP3_GetPositionInto (arg, a->parent, a->coord);
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Vector_Move (P3_vector* a, P3_vector* p) {
  if (a->parent == NULL || p->parent == NULL) {
    memcpy (a->coord, p->coord, 3 * sizeof (GLfloat));
  } else {
    P3_vector_by_matrix_copy (a->coord, p->coord, P3_coordsys_get_root_matrix (p->parent));
    P3_vector_by_matrix (a->coord, P3_coordsys_get_inverted_root_matrix (a->parent));
  }
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Point_Copy (P3_point* a) {
  P3_point* new;
  new = (P3_point*) PyP3Point_Type.tp_alloc (&PyP3Point_Type, 0);
  new->parent = a->parent;
  if (a->parent != NULL) Py_INCREF ((PyObject*) a->parent);
  memcpy (new->coord, a->coord, 3 * sizeof (GLfloat));
  return (PyObject*) new;
}

static PyObject* PyP3Vector_Copy (P3_vector* a) {
  P3_vector* new;
  new = (P3_vector*) PyP3Vector_Type.tp_alloc (&PyP3Vector_Type, 0);
  new->parent = a->parent;
  if (a->parent != NULL) Py_INCREF ((PyObject*) a->parent);
  memcpy (new->coord, a->coord, 3 * sizeof (GLfloat));
  return (PyObject*) new;
}


static PyObject* PyP3Point_Clone (P3_point* a, PyObject* arg) {
  Py_XDECREF ((PyObject*) a->parent);
  a->parent = ((P3_point*) arg)->parent;
  if (a->parent != NULL) Py_INCREF ((PyObject*) a->parent);
  memcpy (a->coord, ((P3_point*) arg)->coord, 3 * sizeof (GLfloat));
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Point_Mod (P3_point* a, PyObject* arg) {
  if (arg == Py_None) arg = NULL;
  if ((PyObject*) a->parent == arg) {
    Py_INCREF ((PyObject*) a);
    return (PyObject*) a;
  } else {
    P3_point* new;
    new = (P3_point*) PyP3Point_Type.tp_alloc (&PyP3Point_Type, 0);
    new->parent = (P3_coordsys*) arg;
    if (arg != NULL) Py_INCREF (arg);
    if (a->parent == NULL || arg == NULL) {
      memcpy (new->coord, a->coord, 3 * sizeof (GLfloat));
    } else {
      P3_point_by_matrix_copy (new->coord, a->coord, P3_coordsys_get_root_matrix (a->parent));
      P3_point_by_matrix (new->coord, P3_coordsys_get_inverted_root_matrix ((P3_coordsys*) arg));
    }
    return (PyObject*) new;
  }
}

static PyObject* PyP3Vector_Mod (P3_vector* a, PyObject* arg) {
  if (arg == Py_None) arg = NULL;
  if ((PyObject*) a->parent == arg) {
    Py_INCREF ((PyObject*) a);
    return (PyObject*) a;
  } else {
    P3_vector* new;
    new = (P3_vector*) PyP3Vector_Type.tp_alloc (&PyP3Vector_Type, 0);
    new->parent = (P3_coordsys*) arg;
    if (arg != NULL) Py_INCREF (arg);
    if (a->parent == NULL || arg == NULL) {
      memcpy (new->coord, a->coord, 3 * sizeof (GLfloat));
    } else {
      P3_vector_by_matrix_copy (new->coord, a->coord, P3_coordsys_get_root_matrix (a->parent));
      P3_vector_by_matrix (new->coord, P3_coordsys_get_inverted_root_matrix ((P3_coordsys*) arg));
    }
    return (PyObject*) new;
  }
}

static PyObject* PyP3Point_Add (P3_point* a, P3_vector* arg) {
  P3_point* new;
  new = (P3_point*) PyP3Point_Type.tp_alloc (&PyP3Point_Type, 0);
  new->parent = a->parent;
  if (new->parent != NULL) Py_INCREF ((PyObject*) new->parent);
  if (arg->parent == NULL || a->parent == NULL || a->parent == arg->parent) {
    new->coord[0] = a->coord[0] + arg->coord[0];
    new->coord[1] = a->coord[1] + arg->coord[1];
    new->coord[2] = a->coord[2] + arg->coord[2];
  } else {
    P3_vector_by_matrix_copy (new->coord, arg->coord, P3_coordsys_get_root_matrix (arg->parent));
    P3_vector_by_matrix (new->coord, P3_coordsys_get_inverted_root_matrix (a->parent));
    new->coord[0] += a->coord[0];
    new->coord[1] += a->coord[1];
    new->coord[2] += a->coord[2];
  }
  return (PyObject*) new;
}

static PyObject* PyP3Vector_Add (P3_vector* a, P3_vector* arg) {
  P3_vector* new;
  new = (P3_vector*) PyP3Vector_Type.tp_alloc (&PyP3Vector_Type, 0);
  new->parent = a->parent;
  if (new->parent != NULL) Py_INCREF ((PyObject*) new->parent);
  if (arg->parent == NULL || a->parent == NULL || a->parent == arg->parent) {
    new->coord[0] = a->coord[0] + arg->coord[0];
    new->coord[1] = a->coord[1] + arg->coord[1];
    new->coord[2] = a->coord[2] + arg->coord[2];
  } else {
    P3_vector_by_matrix_copy (new->coord, arg->coord, P3_coordsys_get_root_matrix (arg->parent));
    P3_vector_by_matrix (new->coord, P3_coordsys_get_inverted_root_matrix (a->parent));
    new->coord[0] += a->coord[0];
    new->coord[1] += a->coord[1];
    new->coord[2] += a->coord[2];
  }
  return (PyObject*) new;
}

static PyObject* PyP3Point_Sub (P3_point* a, P3_vector* arg) {
  P3_point* new;
  new = (P3_point*) PyP3Point_Type.tp_alloc (&PyP3Point_Type, 0);
  new->parent = a->parent;
  if (new->parent != NULL) Py_INCREF ((PyObject*) new->parent);
  if (arg->parent == NULL || a->parent == NULL || a->parent == arg->parent) {
    new->coord[0] = a->coord[0] - arg->coord[0];
    new->coord[1] = a->coord[1] - arg->coord[1];
    new->coord[2] = a->coord[2] - arg->coord[2];
  } else {
    P3_vector_by_matrix_copy (new->coord, arg->coord, P3_coordsys_get_root_matrix (arg->parent));
    P3_vector_by_matrix (new->coord, P3_coordsys_get_inverted_root_matrix (a->parent));
    new->coord[0] = a->coord[0] - new->coord[0];
    new->coord[1] = a->coord[1] - new->coord[1];
    new->coord[2] = a->coord[2] - new->coord[2];
  }
  return (PyObject*) new;
}

static PyObject* PyP3Vector_Sub (P3_vector* a, P3_vector* arg) {
  P3_vector* new;
  new = (P3_vector*) PyP3Vector_Type.tp_alloc (&PyP3Vector_Type, 0);
  new->parent = a->parent;
  if (new->parent != NULL) Py_INCREF ((PyObject*) new->parent);
  if (arg->parent == NULL || a->parent == NULL || a->parent == arg->parent) {
    new->coord[0] = a->coord[0] - arg->coord[0];
    new->coord[1] = a->coord[1] - arg->coord[1];
    new->coord[2] = a->coord[2] - arg->coord[2];
  } else {
    P3_vector_by_matrix_copy (new->coord, arg->coord, P3_coordsys_get_root_matrix (arg->parent));
    P3_vector_by_matrix (new->coord, P3_coordsys_get_inverted_root_matrix (a->parent));
    new->coord[0] = a->coord[0] - new->coord[0];
    new->coord[1] = a->coord[1] - new->coord[1];
    new->coord[2] = a->coord[2] - new->coord[2];
  }
  return (PyObject*) new;
}

static PyObject* PyP3Point_AddXYZ (P3_point* a, PyObject* args) {
  a->coord[0] += (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 0));
  a->coord[1] += (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 1));
  a->coord[2] += (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 2));
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Point_AddVector (P3_point* a, P3_vector* arg) {
  if (arg->parent == NULL || a->parent == NULL || a->parent == arg->parent) {
    a->coord[0] += arg->coord[0];
    a->coord[1] += arg->coord[1];
    a->coord[2] += arg->coord[2];
  } else {
    GLfloat tmp[3];
    P3_vector_by_matrix_copy (tmp, arg->coord, P3_coordsys_get_root_matrix (arg->parent));
    P3_vector_by_matrix (tmp, P3_coordsys_get_inverted_root_matrix (a->parent));
    a->coord[0] += tmp[0];
    a->coord[1] += tmp[1];
    a->coord[2] += tmp[2];
  }
  Py_INCREF ((PyObject*) a);
  return (PyObject*) a;
}

static PyObject* PyP3Point_AddMulVector (P3_point* a, PyObject* args) {
  P3_vector* v = (P3_vector*) PySequence_Fast_GET_ITEM (args, 1);
  GLfloat k = (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 0));
  if (v->parent == NULL || a->parent == NULL || a->parent == v->parent) {
    a->coord[0] += k * v->coord[0];
    a->coord[1] += k * v->coord[1];
    a->coord[2] += k * v->coord[2];
  } else {
    GLfloat tmp[3];
    P3_vector_by_matrix_copy (tmp, v->coord, P3_coordsys_get_root_matrix (v->parent));
    P3_vector_by_matrix (tmp, P3_coordsys_get_inverted_root_matrix (a->parent));
    a->coord[0] += k * tmp[0];
    a->coord[1] += k * tmp[1];
    a->coord[2] += k * tmp[2];
  }
  Py_INCREF ((PyObject*) a);
  return (PyObject*) a;
}

static PyObject* PyP3Point_DistanceTo (P3_point* a, PyObject* arg) {
  GLfloat t[3];
  PyP3_GetPositionInto (arg, a->parent, t);
  t[0] -= a->coord[0];
  t[1] -= a->coord[1];
  t[2] -= a->coord[2];
  return PyFloat_FromDouble (sqrt (t[0] * t[0] + t[1] * t[1] + t[2] * t[2]));
}

static PyObject* PyP3Point_VectorTo (P3_point* a, PyObject* arg) {
  P3_vector* new = (P3_vector*) PyP3Vector_Type.tp_alloc (&PyP3Vector_Type, 0);
  new->parent = a->parent;
  if (new->parent != NULL) Py_INCREF (new->parent);
  PyP3_GetPositionInto (arg, new->parent, new->coord);
  new->coord[0] -= a->coord[0];
  new->coord[1] -= a->coord[1];
  new->coord[2] -= a->coord[2];
  return (PyObject*) new;
}

static PyObject* PyP3Point_GetState (P3_point* a) {
  PyObject* tuple;
  tuple = PyTuple_New (4);
  if (a->parent == NULL) {
    Py_INCREF (Py_None);
    PyTuple_SET_ITEM (tuple, 0, Py_None);
  } else {
    Py_INCREF ((PyObject*) a->parent);
    PyTuple_SET_ITEM (tuple, 0, (PyObject*) a->parent);
  }
  PyTuple_SET_ITEM (tuple, 1, PyFloat_FromDouble ((double) a->coord[0]));
  PyTuple_SET_ITEM (tuple, 2, PyFloat_FromDouble ((double) a->coord[1]));
  PyTuple_SET_ITEM (tuple, 3, PyFloat_FromDouble ((double) a->coord[2]));
  return tuple;
}

static PyObject* PyP3Point_SetState (P3_point* a, PyObject* arg) {
  if (PyDict_Check (arg)) {
    a->parent = (P3_coordsys*) PyDict_GetItemString (arg, "parent");
    a->coord[0] = (GLfloat) PyFloat_AS_DOUBLE (PyDict_GetItemString (arg, "x"));
    a->coord[1] = (GLfloat) PyFloat_AS_DOUBLE (PyDict_GetItemString (arg, "y"));
    a->coord[2] = (GLfloat) PyFloat_AS_DOUBLE (PyDict_GetItemString (arg, "z"));
  } else {
    a->parent = (P3_coordsys*) PySequence_Fast_GET_ITEM (arg, 0);
    a->coord[0] = (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (arg, 1));
    a->coord[1] = (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (arg, 2));
    a->coord[2] = (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (arg, 3));
  }
  if ((PyObject*) a->parent == Py_None) {
    a->parent = NULL;
  } else {
    Py_INCREF ((PyObject*) a->parent);
  }
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Point_Repr (P3_point* a) {
  char buf[50];
  PyObject* o;
  PyObject* r;
  if (a->parent == NULL) {
    snprintf (buf, 50, "<Point %f, %f, %f in None>", a->coord[0], a->coord[1], a->coord[2]);
    r = PyString_FromString (buf);
  } else {
    snprintf (buf, 50, "<Point %f, %f, %f", a->coord[0], a->coord[1], a->coord[2]);
    o = PyObject_Repr ((PyObject*) a->parent);
    r = PyString_FromFormat ("%s in %s>", buf, PyString_AS_STRING (o));
    Py_DECREF (o);
  }
  return r;
}

static PyObject* PyP3Vector_Repr (P3_point* a) {
  char buf[50];
  PyObject* o;
  PyObject* r;
  if (a->parent == NULL) {
    snprintf (buf, 50, "<Vector %f, %f, %f in None>", a->coord[0], a->coord[1], a->coord[2]);
    r = PyString_FromString (buf);
  } else {
    snprintf (buf, 50, "<Vector %f, %f, %f", a->coord[0], a->coord[1], a->coord[2]);
    o = PyObject_Repr ((PyObject*) a->parent);
    r = PyString_FromFormat ("%s in %s>", buf, PyString_AS_STRING (o));
    Py_DECREF (o);
  }
  return r;
}

static PyObject* PyP3Point_Eq (P3_point* a, PyObject* arg) {
  if (PyObject_IsInstance (arg, (PyObject*) &PyP3Point_Type) == 1) {
    P3_point* p = (P3_point*) arg;
    if (a->parent == p->parent &&
        fabs (a->coord[0] - p->coord[0]) < P3_EPSILON &&
        fabs (a->coord[1] - p->coord[1]) < P3_EPSILON &&
        fabs (a->coord[2] - p->coord[2]) < P3_EPSILON) {
      return PyInt_FromLong (1);
    }
  } else {
    if (PyObject_IsInstance (arg, (PyObject*) &PyP3Element_Type) == 1) {
      P3_coordsys* c = (P3_coordsys*) arg;
      if (a->parent == c->parent &&
          fabs (a->coord[0] - c->m[12]) < P3_EPSILON &&
          fabs (a->coord[1] - c->m[13]) < P3_EPSILON &&
          fabs (a->coord[2] - c->m[14]) < P3_EPSILON) {
        return PyInt_FromLong (1);
      }
    } else {
      /* assume it's a sprite */
      P3_sprite* s = (P3_sprite*) arg;
      if (a->parent == s->parent &&
          fabs (a->coord[0] - s->position[12]) < P3_EPSILON &&
          fabs (a->coord[1] - s->position[13]) < P3_EPSILON &&
          fabs (a->coord[2] - s->position[14]) < P3_EPSILON) {
        return PyInt_FromLong (1);
      }
    }
  }
  return PyInt_FromLong (0);
}

static PyObject* PyP3Point_Ne (P3_point* a, PyObject* arg) {
  if (PyObject_IsInstance (arg, (PyObject*) &PyP3Point_Type) == 1) {
    P3_point* p = (P3_point*) arg;
    if (a->parent != p->parent ||
        fabs (a->coord[0] - p->coord[0]) >= P3_EPSILON ||
        fabs (a->coord[1] - p->coord[1]) >= P3_EPSILON ||
        fabs (a->coord[2] - p->coord[2]) >= P3_EPSILON) {
      return PyInt_FromLong (1);
    }
  } else {
    if (PyObject_IsInstance (arg, (PyObject*) &PyP3Element_Type) == 1) {
      P3_coordsys* c = (P3_coordsys*) arg;
      if (a->parent != c->parent ||
          fabs (a->coord[0] - c->m[12]) >= P3_EPSILON ||
          fabs (a->coord[1] - c->m[13]) >= P3_EPSILON ||
          fabs (a->coord[2] - c->m[14]) >= P3_EPSILON) {
        return PyInt_FromLong (1);
      }
    } else {
      /* assume it's a sprite */
      P3_sprite* s = (P3_sprite*) arg;
      if (a->parent != s->parent ||
          fabs (a->coord[0] - s->position[12]) >= P3_EPSILON ||
          fabs (a->coord[1] - s->position[13]) >= P3_EPSILON ||
          fabs (a->coord[2] - s->position[14]) >= P3_EPSILON) {
        return PyInt_FromLong (1);
      }
    }
  }
  return PyInt_FromLong (0);
}

static PyObject* PyP3Vector_CrossProduct (P3_vector* a, PyObject* args) {
  P3_vector* b = (P3_vector*) PySequence_Fast_GET_ITEM (args, 0);
  P3_vector* new;
  if (PySequence_Size (args) == 1) {
    new = (P3_vector*) PyP3Vector_Type.tp_alloc (&PyP3Vector_Type, 0);
    new->parent = a->parent;
    if(new->parent != NULL) Py_INCREF ((PyObject*) new->parent);
    if (a->parent == NULL || b->parent == NULL) {
      P3_vector_cross_product (new->coord, a->coord, b->coord);
    } else {
      GLfloat tmp[3];
      P3_vector_by_matrix_copy (tmp, b->coord, P3_coordsys_get_root_matrix (b->parent));
      P3_vector_by_matrix (tmp, P3_coordsys_get_inverted_root_matrix (a->parent));
      P3_vector_cross_product (new->coord, a->coord, tmp);
    }
    return (PyObject*) new;
  } else {
    new = (P3_vector*) PySequence_Fast_GET_ITEM (args, 1);
    Py_XDECREF (new->parent);
    new->parent = a->parent;
    if(new->parent != NULL) Py_INCREF ((PyObject*) new->parent);
    if (a->parent == NULL || b->parent == NULL) {
      P3_vector_cross_product (new->coord, a->coord, b->coord);
    } else {
      GLfloat tmp[3];
      P3_vector_by_matrix_copy (tmp, b->coord, P3_coordsys_get_root_matrix (b->parent));
      P3_vector_by_matrix (tmp, P3_coordsys_get_inverted_root_matrix (a->parent));
      P3_vector_cross_product (new->coord, a->coord, tmp);
    }
    Py_INCREF (Py_None);
    return Py_None;
  }
}

static PyObject* PyP3Vector_DotProduct (P3_vector* a, P3_vector* b) {
  if (a->parent == NULL || b->parent == NULL) {
    return PyFloat_FromDouble (a->coord[0] * b->coord[0] + a->coord[1] * b->coord[1] + a->coord[2] * b->coord[2]);
  } else {
    GLfloat tmp[3];
    P3_vector_by_matrix_copy (tmp, b->coord, P3_coordsys_get_root_matrix (b->parent));
    P3_vector_by_matrix (tmp, P3_coordsys_get_inverted_root_matrix (a->parent));
    return PyFloat_FromDouble (a->coord[0] * tmp[0] + a->coord[1] * tmp[1] + a->coord[2] * tmp[2]);
  }
}

static PyObject* PyP3Vector_Length (P3_vector* a) {
  return PyFloat_FromDouble (sqrt (a->coord[0] * a->coord[0] + a->coord[1] * a->coord[1] + a->coord[2] * a->coord[2]));
}

static PyObject* PyP3Vector_SetLength (P3_vector* a, PyObject* arg) {
  double d = PyFloat_AS_DOUBLE (arg) / sqrt (a->coord[0] * a->coord[0] + a->coord[1] * a->coord[1] + a->coord[2] * a->coord[2]);
  a->coord[0] *= d;
  a->coord[1] *= d;
  a->coord[2] *= d;
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Vector_Mul (P3_vector* a, PyObject* arg) {
  double d = PyFloat_AS_DOUBLE (arg);
  P3_vector* new = (P3_vector*) PyP3Vector_Type.tp_alloc (&PyP3Vector_Type, 0);
  new->parent = a->parent;
  if(new->parent != NULL) Py_INCREF ((PyObject*) new->parent);
  new->coord[0] = a->coord[0] * d;
  new->coord[1] = a->coord[1] * d;
  new->coord[2] = a->coord[2] * d;
  return (PyObject*) new;
}

static PyObject* PyP3Vector_IMul (P3_vector* a, PyObject* arg) {
  double d = PyFloat_AS_DOUBLE (arg);
  a->coord[0] *= d;
  a->coord[1] *= d;
  a->coord[2] *= d;
  Py_INCREF ((PyObject*) a);
  return (PyObject*) a;
}

static PyObject* PyP3Vector_Div (P3_vector* a, PyObject* arg) {
  double d = 1.0 / PyFloat_AS_DOUBLE (arg);
  P3_vector* new = (P3_vector*) PyP3Vector_Type.tp_alloc (&PyP3Vector_Type, 0);
  new->parent = a->parent;
  if(new->parent != NULL) Py_INCREF ((PyObject*) new->parent);
  new->coord[0] = a->coord[0] * d;
  new->coord[1] = a->coord[1] * d;
  new->coord[2] = a->coord[2] * d;
  return (PyObject*) new;
}

static PyObject* PyP3Vector_IDiv (P3_vector* a, PyObject* arg) {
  double d = 1.0 / PyFloat_AS_DOUBLE (arg);
  a->coord[0] *= d;
  a->coord[1] *= d;
  a->coord[2] *= d;
  Py_INCREF ((PyObject*) a);
  return (PyObject*) a;
}

static PyObject* PyP3Vector_Neg (P3_vector* a, PyObject* arg) {
  P3_vector* new = (P3_vector*) PyP3Vector_Type.tp_alloc (&PyP3Vector_Type, 0);
  new->parent = a->parent;
  if(new->parent != NULL) Py_INCREF ((PyObject*) new->parent);
  new->coord[0] = -a->coord[0];
  new->coord[1] = -a->coord[1];
  new->coord[2] = -a->coord[2];
  return (PyObject*) new;
}

static PyObject* PyP3Vector_Abs (P3_vector* a, PyObject* arg) {
  P3_vector* new = (P3_vector*) PyP3Vector_Type.tp_alloc (&PyP3Vector_Type, 0);
  new->parent = a->parent;
  if(new->parent != NULL) Py_INCREF ((PyObject*) new->parent);
  new->coord[0] = fabs (a->coord[0]);
  new->coord[1] = fabs (a->coord[1]);
  new->coord[2] = fabs (a->coord[2]);
  return (PyObject*) new;
}

static PyObject* PyP3Vector_Normalize (P3_vector* a) {
  double d = 1.0 / sqrt (a->coord[0] * a->coord[0] + a->coord[1] * a->coord[1] + a->coord[2] * a->coord[2]);
  a->coord[0] *= d;
  a->coord[1] *= d;
  a->coord[2] *= d;
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Vector_SetStartEnd (P3_vector* a, PyObject* args) {
  P3_coordsys* start_csys;
  GLfloat* start_coord;
  PyP3_GetPosition (PySequence_Fast_GET_ITEM (args, 0), &start_coord, &start_csys);
  Py_XDECREF ((PyObject*) a->parent);
  a->parent = start_csys;
  if (a->parent != NULL) Py_INCREF ((PyObject*) a->parent);
  PyP3_GetPositionInto (PySequence_Fast_GET_ITEM (args, 1), start_csys, a->coord);
  a->coord[0] -= start_coord[0];
  a->coord[1] -= start_coord[1];
  a->coord[2] -= start_coord[2];
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Vector_AngleTo (P3_vector* a, P3_vector* b) {
  if (a->parent == NULL || b->parent == NULL || a->parent == b->parent) {
    return PyFloat_FromDouble (P3_to_degrees (P3_vector_angle (a->coord, b->coord)));
  } else {
    GLfloat tmp[3];
    P3_vector_by_matrix_copy (tmp, b->coord, P3_coordsys_get_root_matrix (b->parent));
    P3_vector_by_matrix (tmp, P3_coordsys_get_inverted_root_matrix (a->parent));
    return PyFloat_FromDouble (P3_to_degrees (P3_vector_angle (a->coord, tmp)));
  }
}

