/*
 * P3 python wrapper
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*****************************************
 * Copyright (C) 2003 Bertrand 'blam' LAMY
 *****************************************/

/*=======+
 | IMAGE |
 +=======*/

/*---------+
 | Methods |
 +---------*/

static int PyP3Image_Init (P3_image* a, PyObject* args, PyObject* kwds) {
  if (args == NULL || args == Py_None || PySequence_Size (args) == 0) {
    a->nb_color = 0;
    a->width = 0;
    a->height = 0;
    a->pixels = NULL;
  } else {
    if (PySequence_Size (args) == 1) {
      /* this is only a filename -> load image in C */
      P3_image_load (a, PyString_AS_STRING (PySequence_Fast_GET_ITEM(args, 0)));
    } else {
      int size;
      /* this is a sequence : (String, Width, Height, NbColor) */
      a->width    = (int) PyInt_AS_LONG (PySequence_Fast_GET_ITEM (args, 1));
      a->height   = (int) PyInt_AS_LONG (PySequence_Fast_GET_ITEM (args, 2));
      a->nb_color = (int) PyInt_AS_LONG (PySequence_Fast_GET_ITEM (args, 3));
      size = a->width * a->height * a->nb_color * sizeof (GLubyte);
      a->pixels   = (GLubyte*) malloc (size);
      memcpy (a->pixels, PyString_AS_STRING(PySequence_Fast_GET_ITEM(args, 0)), size);
    }
  }
  return 0;
}

static void PyP3Image_Dealloc (P3_image* a) {
  free (a->pixels);
  a->ob_type->tp_free ((PyObject*) a); 
}

/*---------+
 | Get Set |
 +---------*/

static PyObject* PyP3Image_GetNbColor (P3_image* a, void* context) {
  return PyInt_FromLong((long) a->nb_color);
}

static PyObject* PyP3Image_GetWidth (P3_image* a, void* context) {
  return PyInt_FromLong((long) a->width);
}

static PyObject* PyP3Image_GetHeight (P3_image* a, void* context) {
  return PyInt_FromLong((long) a->height);
}

static PyGetSetDef PyP3Image_GetSets[] = {
  { "nb_color", (getter) PyP3Image_GetNbColor, (setter) 0, NULL },
  { "width",    (getter) PyP3Image_GetWidth,   (setter) 0, NULL },
  { "height",   (getter) PyP3Image_GetHeight,  (setter) 0, NULL },
  { NULL }
};

/*------+
 | Type |
 +------*/

PyTypeObject PyP3Image_Type = {
  PyObject_HEAD_INIT(NULL)
  0,
  "_soya._Image",
  sizeof(P3_image),
  0,
  (destructor) PyP3Image_Dealloc,/* tp_dealloc */
  0,/* tp_print */
  0,/* tp_getattr */
  0,/* tp_setattr */
  0,/* tp_compare */
  0,/* tp_repr */
  0,/* tp_as_number */
  0,/* tp_as_sequence */
  0,/* tp_as_mapping */
  0,/* tp_hash */
  0,/* tp_call */
  0,/* tp_str */
  PYP3_GENERIC_GETATTR,/* tp_getattro */
  0,/* tp_setattro */
  0,/* tp_as_buffer */
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,/* tp_flags */
  0,/* tp_doc */
  (traverseproc) 0,/* tp_traverse */
  (inquiry) 0,/* tp_clear */
  0,/* tp_richcompare */
  0,/* tp_weaklistoffset */
  0,/* tp_iter */
  0,/* tp_iternext */
  (PyMethodDef*) 0,/* tp_methods */
  0,/* tp_members */
  (PyGetSetDef*) &PyP3Image_GetSets,/* tp_getset */
  0,/* tp_base */
  0,/* tp_dict */
  0,/* tp_descr_get */
  0,/* tp_descr_set */
  0,/* tp_dictoffset */
  (initproc) PyP3Image_Init,/* tp_init */
  PYP3_GENERIC_ALLOC,/* tp_alloc */
  (newfunc) PyP3Object_New,/* tp_new */
  PYP3_GENERIC_FREE,/* tp_free */
};

/*==========+
 | MATERIAL |
 +==========*/

/*---------+
 | Methods |
 +---------*/

static int PyP3Material_Init (P3_material* a, PyObject* args, PyObject* kwds) {
  P3_material_new (a);
  return 0;
}

static void PyP3Material_Dealloc (P3_material* a) {
  PyObject_GC_UnTrack ((PyObject*) a);
  Py_XDECREF (a->image);
  P3_material_dealloc (a);
  a->ob_type->tp_free ((PyObject*) a); 
}

static int PyP3Material_Traverse (P3_material* a, visitproc visit, void* arg) {
  int err;
  if (a->image != NULL) {
    err = visit ((PyObject*) a->image, arg);
    if (err) { return err; }
  }
  return 0;
}

static int PyP3Material_Clear (P3_material* a) {
  Py_XDECREF (a->image);
  a->image = NULL;
  return 0;
}

static PyObject* PyP3Material_GetState (P3_material* a) {
  P3_chunk* chunk = P3_chunk_new ();
  PyObject* tuple;
  P3_material_get_data (a, chunk);
  tuple = PyString_FromStringAndSize ((char*) chunk->content, chunk->nb);
  P3_chunk_dealloc (chunk);
  return tuple;
}

static PyObject* PyP3Material_SetState (P3_material* a, PyObject* args) {
  P3_chunk* chunk = P3_chunk_new ();
  chunk->content = PyString_AS_STRING (args);
  P3_material_set_data (a, chunk);
  if (a->option & P3_MATERIAL_IMAGE) {
    a->image = (P3_image*) PyP3Image_Type.tp_alloc (&PyP3Image_Type, 0);
    P3_image_set_data (a->image, chunk);
  } else {
    a->image = NULL;
  }
/*
  if (P3_chunk_get_char (chunk) == '\0') {
    a->image = NULL;
  } else {
    a->image = (P3_image*) PyP3Image_Type.tp_alloc (&PyP3Image_Type, 0);
    P3_image_set_data (a->image, chunk);
  }
*/
  free (chunk);
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Material_Activate (P3_material* a) {
  P3_material_activate (a);
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Material_IsAlpha (P3_material* a) {
  if (a->option & P3_MATERIAL_ALPHA) {
    return PyInt_FromLong (1);
  } else {
    return PyInt_FromLong (0);
  }
}

static PyMethodDef PyP3Material_Methods[] = {
  { "_getstate", (PyCFunction) PyP3Material_GetState, METH_NOARGS },
  { "_setstate", (PyCFunction) PyP3Material_SetState, METH_O },
  { "activate", (PyCFunction) PyP3Material_Activate, METH_NOARGS },
  { "is_alpha", (PyCFunction) PyP3Material_IsAlpha, METH_NOARGS },
  { NULL, NULL } /* sentinel */
};

/*---------+
 | Get Set |
 +---------*/

PY_GET_SET_ON_OPTION          (Material, P3_material*, SeparateSpecular, P3_MATERIAL_SEPARATE_SPECULAR)
PY_GET_SET_ON_OPTION          (Material, P3_material*, Clamp, P3_MATERIAL_CLAMP)
PY_GET_SET_ON_OPTION_ADD      (Material, P3_material*, AdditiveBlending,    P3_MATERIAL_ADDITIVE_BLENDING, P3_material_compute_alpha (a);)
PY_GET_SET_ON_FLOAT           (Material, P3_material*, Shininess, shininess)
//PY_GET_SET_ON_FLOAT_ARRAY_ADD (Material, P3_material*, Diffuse, colors, 4, P3_material_compute_alpha (a);)
//PY_GET_SET_ON_FLOAT_ARRAY     (Material, P3_material*, Specular, colors + 4, 4)
PY_GET_SET_ON_OBJECT_ADD      (Material, P3_material*, Image, image, P3_image*, P3_image_check (a->image);  P3_material_compute_alpha (a);  if (a->id != 0) { glDeleteTextures (1, &(a->id)); a->id = 0; }  if (a->image == NULL) a->option &= ~P3_MATERIAL_IMAGE; else a->option |= P3_MATERIAL_IMAGE; )

#define compare_color(c1, c2) \
  fabs (c1[0] - c2[0]) <= P3_EPSILON && fabs (c1[1] - c2[1]) <= P3_EPSILON && fabs (c1[2] - c2[2]) <= P3_EPSILON && fabs (c1[3] - c2[3]) <= P3_EPSILON

static PyObject* PyP3Material_GetDiffuse (P3_material* a, void* context) {
  PyObject* tuple = PyTuple_New (4);
  PyTuple_SET_ITEM (tuple, 0, PyFloat_FromDouble ((double) (a->diffuse)[0]));
  PyTuple_SET_ITEM (tuple, 1, PyFloat_FromDouble ((double) (a->diffuse)[1]));
  PyTuple_SET_ITEM (tuple, 2, PyFloat_FromDouble ((double) (a->diffuse)[2]));
  PyTuple_SET_ITEM (tuple, 3, PyFloat_FromDouble ((double) (a->diffuse)[3]));
  return tuple;
}

static int PyP3Material_SetDiffuse (P3_material* a, PyObject* value, void* context) {
  GLfloat color[4];
  color[0] = (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (value, 0));
  color[1] = (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (value, 1));
  color[2] = (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (value, 2));
  color[3] = (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (value, 3));
  if (a->option & P3_MATERIAL_DIFFUSE) free (a->diffuse);
  if (compare_color (color, white)) {
    a->option &= ~P3_MATERIAL_DIFFUSE;
    a->diffuse = white;
  } else {
    a->option |= P3_MATERIAL_DIFFUSE;
    a->diffuse = (GLfloat*) malloc (4 * sizeof (GLfloat));
    memcpy (a->diffuse, color, 4 * sizeof (GLfloat));
  }
  P3_material_compute_alpha (a);
  return 0;
}

static PyObject* PyP3Material_GetSpecular (P3_material* a, void* context) {
  PyObject* tuple = PyTuple_New (4);
  PyTuple_SET_ITEM (tuple, 0, PyFloat_FromDouble ((double) (a->specular)[0]));
  PyTuple_SET_ITEM (tuple, 1, PyFloat_FromDouble ((double) (a->specular)[1]));
  PyTuple_SET_ITEM (tuple, 2, PyFloat_FromDouble ((double) (a->specular)[2]));
  PyTuple_SET_ITEM (tuple, 3, PyFloat_FromDouble ((double) (a->specular)[3]));
  return tuple;
}

static int PyP3Material_SetSpecular (P3_material* a, PyObject* value, void* context) {
  GLfloat color[4];
  color[0] = (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (value, 0));
  color[1] = (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (value, 1));
  color[2] = (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (value, 2));
  color[3] = (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (value, 3));
  if (a->option & P3_MATERIAL_SPECULAR) free (a->specular);
  if (compare_color (color, black)) {
    a->option &= ~P3_MATERIAL_SPECULAR;
    a->specular = black;
  } else {
    a->option |= P3_MATERIAL_SPECULAR;
    a->specular = (GLfloat*) malloc (4 * sizeof (GLfloat));
    memcpy (a->specular, color, 4 * sizeof (GLfloat));
  }
  return 0;
}

static PyObject* PyP3Material_GetEmissive (P3_material* a, void* context) {
  PyObject* tuple = PyTuple_New (4);
  PyTuple_SET_ITEM (tuple, 0, PyFloat_FromDouble ((double) (a->emissive)[0]));
  PyTuple_SET_ITEM (tuple, 1, PyFloat_FromDouble ((double) (a->emissive)[1]));
  PyTuple_SET_ITEM (tuple, 2, PyFloat_FromDouble ((double) (a->emissive)[2]));
  PyTuple_SET_ITEM (tuple, 3, PyFloat_FromDouble ((double) (a->emissive)[3]));
  return tuple;
}

static int PyP3Material_SetEmissive (P3_material* a, PyObject* value, void* context) {
  GLfloat color[4];
  color[0] = (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (value, 0));
  color[1] = (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (value, 1));
  color[2] = (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (value, 2));
  color[3] = (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (value, 3));
  if (a->option & P3_MATERIAL_EMISSIVE) free (a->emissive);
  if (compare_color (color, black)) {
    a->option &= ~P3_MATERIAL_EMISSIVE;
    a->emissive = black;
  } else {
    a->option |= P3_MATERIAL_EMISSIVE;
    a->emissive = (GLfloat*) malloc (4 * sizeof (GLfloat));
    memcpy (a->emissive, color, 4 * sizeof (GLfloat));
  }
  return 0;
}

#undef compare_color

static PyGetSetDef PyP3Material_GetSets[] = {
  { "clamp",             (getter) PyP3Material_GetClamp,            (setter) PyP3Material_SetClamp,            NULL },
  { "shininess",         (getter) PyP3Material_GetShininess,        (setter) PyP3Material_SetShininess,        NULL },
  { "image",             (getter) PyP3Material_GetImage,            (setter) PyP3Material_SetImage,            NULL },
  { "diffuse",           (getter) PyP3Material_GetDiffuse,          (setter) PyP3Material_SetDiffuse,          NULL },
  { "specular",          (getter) PyP3Material_GetSpecular,         (setter) PyP3Material_SetSpecular,         NULL },
  { "emissive",          (getter) PyP3Material_GetEmissive,         (setter) PyP3Material_SetEmissive,         NULL },
  { "additive_blending", (getter) PyP3Material_GetAdditiveBlending, (setter) PyP3Material_SetAdditiveBlending, NULL },
  { "separate_specular", (getter) PyP3Material_GetSeparateSpecular, (setter) PyP3Material_SetSeparateSpecular, NULL },
  { NULL }
};

/*------+
 | Type |
 +------*/

PyTypeObject PyP3Material_Type = {
  PyObject_HEAD_INIT(NULL)
  0,
  "_soya._Material",
  sizeof(P3_material),
  0,
  (destructor) PyP3Material_Dealloc,/* tp_dealloc */
  0,/* tp_print */
  0,/* tp_getattr */
  0,/* tp_setattr */
  0,/* tp_compare */
  0,/* tp_repr */
  0,/* tp_as_number */
  0,/* tp_as_sequence */
  0,/* tp_as_mapping */
  0,/* tp_hash */
  0,/* tp_call */
  0,/* tp_str */
  PYP3_GENERIC_GETATTR,/* tp_getattro */
  0,/* tp_setattro */
  0,/* tp_as_buffer */
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE | Py_TPFLAGS_HAVE_GC,/* tp_flags */
  0,/* tp_doc */
  (traverseproc) PyP3Material_Traverse,/* tp_traverse */
  (inquiry) PyP3Material_Clear,/* tp_clear */
  0,/* tp_richcompare */
  0,/* tp_weaklistoffset */
  0,/* tp_iter */
  0,/* tp_iternext */
  (PyMethodDef*) &PyP3Material_Methods,/* tp_methods */
  0,/* tp_members */
  (PyGetSetDef*) &PyP3Material_GetSets,/* tp_getset */
  0,/* tp_base */
  0,/* tp_dict */
  0,/* tp_descr_get */
  0,/* tp_descr_set */
  0,/* tp_dictoffset */
  (initproc) PyP3Material_Init,/* tp_init */
  PYP3_GENERIC_ALLOC,/* tp_alloc */
  (newfunc) PyP3Object_New,/* tp_new */
  PYP3_GENERIC_GC_FREE,/* tp_free */
};

