/*
 * P3 python wrapper
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*****************************************
 * Copyright (C) 2003 Bertrand 'blam' LAMY
 *****************************************/

/*=======+
 | CMESH |
 +=======*/

/*---------+
 | Methods |
 +---------*/

static int PyP3Cmesh_Init (P3_cmesh* a, PyObject* args, PyObject* kwds) {
  P3_cmesh_new (a);
  return 0;
}

static void PyP3Cmesh_Dealloc (P3_cmesh* a) {
  int i;
  PyObject_GC_UnTrack ((PyObject*) a);
  P3_cmesh_dealloc (a);
//  for (i = 0; i < a->nb_packs; i++) {
//    Py_XDECREF ((a->packs + i)->material);
//  }
  a->ob_type->tp_free ((PyObject*) a); 
}

static int PyP3Cmesh_Traverse (P3_cmesh* a, visitproc visit, void* arg) {
  int err;
  int i;
  for (i = 0; i < a->nb_packs; i++) {
    if ((a->packs + i)->material != NULL) {
      err = visit ((PyObject*) (a->packs + i)->material, arg);
      if (err) { return err; }
    }
  }
  return 0;
}

static int PyP3Cmesh_Clear (P3_cmesh* a) {
  int i;
  for (i = 0; i < a->nb_packs; i++) {
    Py_XDECREF ((a->packs + i)->material);
    (a->packs + i)->material = NULL;
  }
  return 0;
}

static PyObject* PyP3Cmesh_GetState (P3_cmesh* a) {
  P3_chunk* chunk = P3_chunk_new ();
  PyObject* tuple;
  PyObject* materials;
  P3_cpack* p;
  int nb = 0;
  int i;
  P3_cmesh_get_data (a, chunk);
  tuple = PyTuple_New (2);
  PyTuple_SET_ITEM (tuple, 0, PyString_FromStringAndSize ((char*) chunk->content, chunk->nb));
  for (i = 0; i < a->nb_packs; i++) {
    if ((a->packs + i)->option & P3_R_TEXTURED) { nb++; }
  }
  materials = PyTuple_New (nb);
  nb = 0;
  for (i = 0; i < a->nb_packs; i++) {
    p = a->packs + i;
    if (p->option & P3_R_TEXTURED) {
      if (p->material == NULL) {
        Py_INCREF (Py_None);
        PyTuple_SET_ITEM (materials, nb, Py_None);
      } else {
        Py_INCREF ((PyObject*) p->material);
        PyTuple_SET_ITEM (materials, nb, (PyObject*) p->material);
      }
      nb++;
    }
  }
  PyTuple_SET_ITEM (tuple, 1, materials);
  P3_chunk_dealloc (chunk);
  return tuple;
}

static PyObject* PyP3Cmesh_SetState (P3_cmesh* a, PyObject* args) {
  P3_chunk* chunk = P3_chunk_new ();
  PyObject* materials;
  P3_cpack* p;
  int i;
  int nb = 0;
  chunk->content = PyString_AS_STRING (PySequence_Fast_GET_ITEM (args, 0));
  P3_cmesh_set_data (a, chunk);
  materials = PySequence_Fast_GET_ITEM (args, 1);
  for (i = 0; i < a->nb_packs; i++) {
    p = a->packs + i;
    if (p->option & P3_R_TEXTURED) {
      p->material = (P3_material*) PyTuple_GET_ITEM (materials, nb);
      if ((PyObject*) p->material == Py_None) {
        p->material = NULL;
      } else {
        Py_INCREF ((PyObject*) p->material);
      }
      nb++;
    } else {
      p->material = NULL;
    }
  }
  free (chunk);
  Py_INCREF (Py_None);
  return Py_None;
}

PyObject* PyP3Cmesh_StaticLight (P3_cmesh* a, PyObject* args) {
  if (args == Py_None || PySequence_Size (args) == 0 || PySequence_Fast_GET_ITEM (args, 0) == Py_None) {
    P3_cmesh_static_light (a, NULL, 0);
  } else {
    int shadow = P3_FALSE;
    int nb;
    int i;
    P3_list* l;
    if (PySequence_Size (args) == 2) {
      if (PyObject_IsTrue (PySequence_Fast_GET_ITEM (args, 1))) {
        shadow = P3_TRUE;
      }
    }
    args = PySequence_Fast_GET_ITEM (args, 0);
    nb = PySequence_Size (args);
    l = P3_list_new (nb);
    for (i = 0; i < nb; i++) {
      P3_list_add (l, PySequence_Fast_GET_ITEM (args, i));
    }
    P3_cmesh_static_light (a, l, shadow);
    P3_list_dealloc (l);
  }
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Cmesh_ToFaces (P3_cmesh* a, PyObject* arg) {
  P3_cmesh_to_faces (a, (P3_world*) arg);
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Cmesh_BuildTree (P3_cmesh* a) {
  P3_cmesh_build_tree (a);
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Cmesh_OptimizeTree (P3_cmesh* a, PyObject* args) {
  GLfloat param;
  int mode;
  mode = (int) PyInt_AS_LONG (PySequence_Fast_GET_ITEM (args, 1));
  if (mode == 0) {
    param = (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 2));
  }
  P3_cmesh_optimize_tree (a, 
                          (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 0)),
                          mode, param);
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Cmesh_SetAllVisibility (P3_cmesh* a, PyObject* arg) {
  P3_cmesh_set_visibility_all (a, (int) PyInt_AS_LONG (arg));
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Cmesh_SetVisibilitySphere (P3_cmesh* a, PyObject* args) {
  GLfloat s[4];
  s[0] = (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 1));
  s[1] = (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 2));
  s[2] = (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 3));
  s[3] = (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 4));
  P3_cmesh_set_visibility_in_sphere (a, (int) PyInt_AS_LONG (PySequence_Fast_GET_ITEM (args, 0)), s);
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Cmesh_SetVisibilityCylinder (P3_cmesh* a, PyObject* args) {
  GLfloat s[3];
  s[0] = (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 1));
  s[1] = (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 2));
  s[2] = (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 3));
  P3_cmesh_set_visibility_in_cylinder (a, (int) PyInt_AS_LONG (PySequence_Fast_GET_ITEM (args, 0)), s);
  Py_INCREF (Py_None);
  return Py_None;
}

static PyMethodDef PyP3Cmesh_Methods[] = {
  { "_getstate",                  (PyCFunction) PyP3Cmesh_GetState,              METH_NOARGS },
  { "_setstate",                  (PyCFunction) PyP3Cmesh_SetState,              METH_O },
  { "compute_static_light",       (PyCFunction) PyP3Cmesh_StaticLight,           METH_VARARGS },
  { "to_faces",                   (PyCFunction) PyP3Cmesh_ToFaces,               METH_O },
  { "build_tree",                 (PyCFunction) PyP3Cmesh_BuildTree,             METH_NOARGS },
  { "optimize_tree",              (PyCFunction) PyP3Cmesh_OptimizeTree,          METH_VARARGS },
  { "set_all_visibility",         (PyCFunction) PyP3Cmesh_SetAllVisibility,      METH_O },
  { "set_visibility_in_sphere",   (PyCFunction) PyP3Cmesh_SetVisibilitySphere,   METH_VARARGS },
  { "set_visibility_in_cylinder", (PyCFunction) PyP3Cmesh_SetVisibilityCylinder, METH_VARARGS },
  { NULL, NULL } /* sentinel */
};

/*---------+
 | Get Set |
 +---------*/

PY_GET_SET_ON_OPTION (Cmesh, P3_cmesh*, AlwaysVisible, P3_CMESH_ALWAYS_VISIBLE)

static PyObject* PyP3Cmesh_GetNormalize (P3_cmesh* a, void* context) {
  if (a->packs->option & P3_R_NORMALIZE) {
    return PyInt_FromLong (1);
  } else {
    return PyInt_FromLong (0);
  } 
}

static int PyP3Cmesh_SetNormalize (P3_cmesh* a, PyObject* value, void* context) {
  int i;
  if (PyObject_IsTrue (value) == 1) {
    for (i = 0; i < a->nb_packs; i++) {
      (a->packs + i)->option &= P3_R_NORMALIZE;
    }
  } else {
    for (i = 0; i < a->nb_packs; i++) {
      (a->packs + i)->option &= (0xFFFFFFFF - P3_R_NORMALIZE);
    }
  }
  return 0;
}

static PyObject* PyP3Cmesh_GetUseVertexVisibility (P3_cmesh* a, void* context) {
  if (a->option & P3_CMESH_HAS_VERTEX_VISIBILITY) {
    return PyInt_FromLong (1);
  } else {
    return PyInt_FromLong (0);
  } 
}

static int PyP3Cmesh_SetUseVertexVisibility (P3_cmesh* a, PyObject* value, void* context) {
  free (a->voptions);
  if (PyObject_IsTrue (value) == 1) {
    a->option |= P3_CMESH_HAS_VERTEX_VISIBILITY;
    a->voptions = (char*) malloc (a->nb_vcoords * sizeof (char));
  } else {
    if (a->option & P3_CMESH_HAS_VERTEX_VISIBILITY) {
      a->option -= P3_CMESH_HAS_VERTEX_VISIBILITY;
    }
  }
  return 0;
}

static PyGetSetDef PyP3Cmesh_GetSets[] = {
  { "normalize",             (getter) PyP3Cmesh_GetNormalize,           (setter) PyP3Cmesh_SetNormalize,           NULL },
  { "always_visible",        (getter) PyP3Cmesh_GetAlwaysVisible,       (setter) PyP3Cmesh_SetAlwaysVisible,       NULL },
  { "use_vertex_visibility", (getter) PyP3Cmesh_GetUseVertexVisibility, (setter) PyP3Cmesh_SetUseVertexVisibility, NULL },
  { NULL }
};

/*------+
 | Type |
 +------*/

PyTypeObject PyP3Cmesh_Type = {
  PyObject_HEAD_INIT(NULL)
  0,
  "_soya._Cmesh",
  sizeof(P3_cmesh),
  0,
  (destructor) PyP3Cmesh_Dealloc,/* tp_dealloc */
  0,/* tp_print */
  0,/* tp_getattr */
  0,/* tp_setattr */
  0,/* tp_compare */
  0,/* tp_repr */
  0,/* tp_as_number */
  0,/* tp_as_sequence */
  0,/* tp_as_mapping */
  0,/* tp_hash */
  0,/* tp_call */
  0,/* tp_str */
  PYP3_GENERIC_GETATTR,/* tp_getattro */
  0,/* tp_setattro */
  0,/* tp_as_buffer */
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE | Py_TPFLAGS_HAVE_GC,/* tp_flags */
  0,/* tp_doc */
  (traverseproc) PyP3Cmesh_Traverse,/* tp_traverse */
  (inquiry) PyP3Cmesh_Clear,/* tp_clear */
  0,/* tp_richcompare */
  0,/* tp_weaklistoffset */
  0,/* tp_iter */
  0,/* tp_iternext */
  (PyMethodDef*) &PyP3Cmesh_Methods,/* tp_methods */
  0,/* tp_members */
  (PyGetSetDef*) &PyP3Cmesh_GetSets,/* tp_getset */
  0,/* tp_base */
  0,/* tp_dict */
  0,/* tp_descr_get */
  0,/* tp_descr_set */
  0,/* tp_dictoffset */
  (initproc) PyP3Cmesh_Init,/* tp_init */
  PYP3_GENERIC_ALLOC,/* tp_alloc */
  (newfunc) PyP3Object_New,/* tp_new */
  PYP3_GENERIC_GC_FREE,/* tp_free */
};

