/*
 * P3 python wrapper
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*****************************************
 * Copyright (C) 2003 Bertrand 'blam' LAMY
 *****************************************/

/*============+
 | ATMOSPHERE |
 +============*/

/*---------+
 | Methods |
 +---------*/

static int PyP3Atmosphere_Init (P3_atmosphere* a, PyObject* args, PyObject* kwds) {
  P3_atmosphere_new (a);
  return 0;
}

static void PyP3Atmosphere_Dealloc (P3_atmosphere* a) {
  int i;
  PyObject_GC_UnTrack ((PyObject*) a);
  Py_XDECREF (a->cloud);
  if (a->option & P3_ATMOSPHERE_SKYBOX_5) {
    for (i = 0; i < 5; i++) { Py_XDECREF (a->skybox[i]); }
  } else if (a->option & P3_ATMOSPHERE_SKYBOX_6) {
    for (i = 0; i < 6; i++) { Py_XDECREF (a->skybox[i]); }
  }
  a->ob_type->tp_free ((PyObject*) a); 
}

static int PyP3Atmosphere_Traverse (P3_atmosphere* a, visitproc visit, void* arg) {
  int err;
  int i;
  if (a->cloud != NULL) {
    err = visit ((PyObject*) a->cloud, arg);
    if (err) { return err; }
  }
  if (a->option & P3_ATMOSPHERE_SKYBOX_5) {
    for (i = 0; i < 5; i++) { 
      err = visit ((PyObject*) a->skybox[i], arg);
      if (err) { return err; }
    }
  } else if (a->option & P3_ATMOSPHERE_SKYBOX_6) {
    for (i = 0; i < 6; i++) { 
      err = visit ((PyObject*) a->skybox[i], arg);
      if (err) { return err; }
    }
  }
  return 0;
}

static int PyP3Atmosphere_Clear (P3_atmosphere* a) {
  int i;
  Py_XDECREF (a->cloud);
  a->cloud = NULL;
  if (a->option & P3_ATMOSPHERE_SKYBOX_5) {
    for (i = 0; i < 5; i++) { Py_XDECREF (a->skybox[i]); a->skybox[i] = NULL; }
  } else if (a->option & P3_ATMOSPHERE_SKYBOX_6) {
    for (i = 0; i < 6; i++) { Py_XDECREF (a->skybox[i]); a->skybox[i] = NULL; }
  }
  return 0;
}

static PyObject* PyP3Atmosphere_SetSkybox (P3_atmosphere* a, PyObject* args) {
  int i;
  int nb = 0;
  if (a->option & P3_ATMOSPHERE_SKYBOX_5) {
    for (i = 0; i < 5; i++) { Py_XDECREF (a->skybox[i]); }
  } else if (a->option & P3_ATMOSPHERE_SKYBOX_6) {
    for (i = 0; i < 6; i++) { Py_XDECREF (a->skybox[i]); }
  }
  a->option &= ~(P3_ATMOSPHERE_SKYBOX_5 | P3_ATMOSPHERE_SKYBOX_6 | P3_ATMOSPHERE_SKYBOX_ALPHA);
  free (a->skybox);
  if (args == Py_None || PySequence_Size (args) == 1) {
    free (a->skybox);
    a->skybox = NULL;
  } else if (PySequence_Size (args) == 5) {
    a->option |= P3_ATMOSPHERE_SKYBOX_5;
    a->skybox = (P3_material**) malloc (5 * sizeof (P3_material*));
    nb = 5;
  } else if (PySequence_Size (args) == 6) {
    a->option |= P3_ATMOSPHERE_SKYBOX_6;
    a->skybox = (P3_material**) malloc (6 * sizeof (P3_material*));
    nb = 6;
  }
  for (i = 0; i < nb; i++) {
    a->skybox[i] = (P3_material*) PySequence_Fast_GET_ITEM (args, i);
    if ((PyObject*) a->skybox[i] == Py_None) {
      a->skybox[i] = NULL;
    } else {
      Py_INCREF (a->skybox[i]);
      if (a->skybox[i]->option & P3_MATERIAL_ALPHA) { a->option |= P3_ATMOSPHERE_SKYBOX_ALPHA; }
    }
  }
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Atmosphere_GetState (P3_atmosphere* a) {
  P3_chunk* chunk = P3_chunk_new ();
  PyObject* tuple;
  P3_atmosphere_get_data (a, chunk);
  if (a->option & (P3_ATMOSPHERE_SKYBOX_5 | P3_ATMOSPHERE_SKYBOX_6)) {
    PyObject* skybox;
    int nb = 0;
    int i;
    tuple = PyTuple_New (3);
    if (a->option & P3_ATMOSPHERE_SKYBOX_5) nb = 5;
    if (a->option & P3_ATMOSPHERE_SKYBOX_6) nb = 6;
    skybox = PyTuple_New (nb);
    for (i = 0; i < nb; i++) {
      if (a->skybox[i] == NULL) {
        Py_INCREF (Py_None);
        PyTuple_SET_ITEM (skybox, i, Py_None);
      } else {
        Py_INCREF ((PyObject*) a->skybox[i]);
        PyTuple_SET_ITEM (skybox, i, (PyObject*) a->skybox[i]);
      }
    }
    PyTuple_SET_ITEM (tuple, 2, skybox);
  } else {
    tuple = PyTuple_New (2);
  }
  PyTuple_SET_ITEM (tuple, 0, PyString_FromStringAndSize ((char*) chunk->content, chunk->nb));
  if (a->cloud == NULL) {
    Py_INCREF (Py_None);
    PyTuple_SET_ITEM (tuple, 1, Py_None);
  } else {
    Py_INCREF ((PyObject*) a->cloud);
    PyTuple_SET_ITEM (tuple, 1, (PyObject*) a->cloud);
  }
  P3_chunk_dealloc (chunk);
  return tuple;
}

static PyObject* PyP3Atmosphere_SetState (P3_atmosphere* a, PyObject* args) {
  P3_chunk* chunk = P3_chunk_new ();
  PyObject* o;
  int nb = 0;
  int i;
  o = PySequence_Fast_GET_ITEM (args, 0);
  chunk->content = PyString_AS_STRING (o);
  P3_atmosphere_set_data (a, chunk);
  a->cloud = (P3_material*) PySequence_Fast_GET_ITEM (args, 1);
  if ((PyObject*) a->cloud == Py_None) {
    a->cloud = NULL;
  } else {
    Py_INCREF ((PyObject*) a->cloud);
  }
  if (a->option & P3_ATMOSPHERE_SKYBOX_5) { nb = 5; }
  if (a->option & P3_ATMOSPHERE_SKYBOX_6) { nb = 6; }
  if (nb > 0) {
    PyObject* skybox = PySequence_Fast_GET_ITEM (args, 2);
    a->skybox = (P3_material**) malloc (nb * sizeof (P3_material*));
    for (i = 0; i < nb; i++) {
      a->skybox[i] = (P3_material*) PySequence_Fast_GET_ITEM (skybox, i);
      if ((PyObject*) a->skybox[i] == Py_None) {
        a->skybox[i] = NULL;
      } else {
        Py_INCREF (a->skybox[i]);
      }
    }
  }
  free (chunk);
  Py_INCREF (Py_None);
  return Py_None;
}

static PyMethodDef PyP3Atmosphere_Methods[] = {
  { "set_skybox", (PyCFunction) PyP3Atmosphere_SetSkybox, METH_VARARGS },
  { "_getstate",  (PyCFunction) PyP3Atmosphere_GetState,  METH_NOARGS },
  { "_setstate",  (PyCFunction) PyP3Atmosphere_SetState,  METH_O },
  { NULL, NULL } /* sentinel */
};

/*---------+
 | Get Set |
 +---------*/

PY_GET_SET_ON_OPTION      (Atmosphere, P3_atmosphere*, Fog, P3_ATMOSPHERE_FOG)
PY_GET_SET_ON_OPTION      (Atmosphere, P3_atmosphere*, Skyplane, P3_ATMOSPHERE_SKYPLANE)
PY_GET_SET_ON_FLOAT       (Atmosphere, P3_atmosphere*, FogStart, fog_start)
PY_GET_SET_ON_FLOAT       (Atmosphere, P3_atmosphere*, FogEnd, fog_end)
PY_GET_SET_ON_FLOAT       (Atmosphere, P3_atmosphere*, FogDensity, fog_density)
PY_GET_SET_ON_FLOAT_ARRAY (Atmosphere, P3_atmosphere*, BgColor, bg_color, 4)
PY_GET_SET_ON_FLOAT_ARRAY (Atmosphere, P3_atmosphere*, Ambient, ambient, 4)
PY_GET_SET_ON_FLOAT_ARRAY (Atmosphere, P3_atmosphere*, FogColor, fog_color, 4)
PY_GET_SET_ON_FLOAT_ARRAY (Atmosphere, P3_atmosphere*, SkyColor, sky_color, 4)
PY_GET_SET_ON_OBJECT      (Atmosphere, P3_atmosphere*, Cloud, cloud, P3_material*)

static PyObject* PyP3Atmosphere_GetClear (P3_atmosphere* a, void* context) {
  if (a->option & P3_ATMOSPHERE_DONT_CLEAR) {
    return PyInt_FromLong (0);
  } else {
    return PyInt_FromLong (1);
  }
}

static int PyP3Atmosphere_SetClear (P3_atmosphere* a, PyObject* value, void* context) {
  if (PyObject_IsTrue (value) == 1) {
    if (a->option & P3_ATMOSPHERE_DONT_CLEAR) {
      a->option -= P3_ATMOSPHERE_DONT_CLEAR;
    }
  } else {
    a->option |= P3_ATMOSPHERE_DONT_CLEAR;
  }
  return 0;
}

static PyObject* PyP3Atmosphere_GetFogType (P3_atmosphere* a, void* context) {
  switch (a->fog_type) {
  case GL_LINEAR:
    return PyInt_FromLong(0);
  case GL_EXP:
    return PyInt_FromLong(1);
  case GL_EXP2:
    return PyInt_FromLong(2);
  }
  return PyInt_FromLong((long) a->fog_type);
}

static int PyP3Atmosphere_SetFogType (P3_atmosphere* a, PyObject* value, void* context) {
  long i = PyInt_AS_LONG (value);
  switch (i) {
  case 0:
    a->fog_type = GL_LINEAR;
    break;
  case 1:
    a->fog_type = GL_EXP;
    break;
  case 2:
    a->fog_type = GL_EXP2;
    break;
  }
  return 0;
}

static PyGetSetDef PyP3Atmosphere_GetSets[] = {
  { "bg_color",    (getter) PyP3Atmosphere_GetBgColor,    (setter) PyP3Atmosphere_SetBgColor,    NULL },
  { "ambient",     (getter) PyP3Atmosphere_GetAmbient,    (setter) PyP3Atmosphere_SetAmbient,    NULL },
  { "fog",         (getter) PyP3Atmosphere_GetFog,        (setter) PyP3Atmosphere_SetFog,        NULL },
  { "fog_type",    (getter) PyP3Atmosphere_GetFogType,    (setter) PyP3Atmosphere_SetFogType,    NULL },
  { "fog_density", (getter) PyP3Atmosphere_GetFogDensity, (setter) PyP3Atmosphere_SetFogDensity, NULL },
  { "fog_start",   (getter) PyP3Atmosphere_GetFogStart,   (setter) PyP3Atmosphere_SetFogStart,   NULL },
  { "fog_end",     (getter) PyP3Atmosphere_GetFogEnd,     (setter) PyP3Atmosphere_SetFogEnd,     NULL },
  { "fog_color",   (getter) PyP3Atmosphere_GetFogColor,   (setter) PyP3Atmosphere_SetFogColor,   NULL },
  { "skyplane",    (getter) PyP3Atmosphere_GetSkyplane,   (setter) PyP3Atmosphere_SetSkyplane,   NULL },
  { "sky_color",   (getter) PyP3Atmosphere_GetSkyColor,   (setter) PyP3Atmosphere_SetSkyColor,   NULL },
  { "cloud",       (getter) PyP3Atmosphere_GetCloud,      (setter) PyP3Atmosphere_SetCloud,      NULL },
  { "clear",       (getter) PyP3Atmosphere_GetClear,      (setter) PyP3Atmosphere_SetClear,      NULL },
  { NULL }
};

/*------+
 | Type |
 +------*/

PyTypeObject PyP3Atmosphere_Type = {
  PyObject_HEAD_INIT(NULL)
  0,
  "_soya._Atmosphere",
  sizeof(P3_atmosphere),
  0,
  (destructor) PyP3Atmosphere_Dealloc,/* tp_dealloc */
  0,/* tp_print */
  0,/* tp_getattr */
  0,/* tp_setattr */
  0,/* tp_compare */
  0,/* tp_repr */
  0,/* tp_as_number */
  0,/* tp_as_sequence */
  0,/* tp_as_mapping */
  0,/* tp_hash */
  0,/* tp_call */
  0,/* tp_str */
  PYP3_GENERIC_GETATTR,/* tp_getattro */
  0,/* tp_setattro */
  0,/* tp_as_buffer */
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE | Py_TPFLAGS_HAVE_GC,/* tp_flags */
  0,/* tp_doc */
  (traverseproc) PyP3Atmosphere_Traverse,/* tp_traverse */
  (inquiry) PyP3Atmosphere_Clear,/* tp_clear */
  0,/* tp_richcompare */
  0,/* tp_weaklistoffset */
  0,/* tp_iter */
  0,/* tp_iternext */
  (PyMethodDef*) &PyP3Atmosphere_Methods,/* tp_methods */
  0,/* tp_members */
  (PyGetSetDef*) &PyP3Atmosphere_GetSets,/* tp_getset */
  0,/* tp_base */
  0,/* tp_dict */
  0,/* tp_descr_get */
  0,/* tp_descr_set */
  0,/* tp_dictoffset */
  (initproc) PyP3Atmosphere_Init,/* tp_init */
  PYP3_GENERIC_ALLOC,/* tp_alloc */
  (newfunc) PyP3Object_New,/* tp_new */
  PYP3_GENERIC_GC_FREE,/* tp_free */
};

