/*
 * P3
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**********************************************
 * xmesh_fx.c
 * Copyright (C) 2001-2003 Bertrand 'blam' LAMY
 **********************************************/

#include "p3_base.h"
#include "math3d.h"
#include "util.h"
#include "material.h"
#include "frustum.h"
#include "renderer.h"
#include "mesh.h"
#include "fx.h"

extern GLfloat white[4];
extern P3_renderer* renderer;
extern P3_material* light_shader;


/*===========+
 | VERTEX FX |
 +===========*/

void P3_mesh_fx_all (P3_mesh* mesh, P3_fx* fx) {
  int i;
  for (i = 0; i < mesh->nb_vertices; i++) fx->apply (fx, i);
}

static void P3_mesh_vertex_fx_in_sphere (P3_mesh* mesh, int index, P3_fx* fx, GLfloat sphere[4]) {
  if (!(mesh->vertex_options[index] & P3_VERTEX_MADE)) {
    mesh->vertex_options[index] |= P3_VERTEX_MADE;
    if (P3_point_is_in_sphere (sphere, mesh->vertex_coords[index]) == P3_TRUE) fx->apply (fx, index);
  }
}

static void P3_mesh_node_fx_in_sphere (P3_mesh* mesh, P3_xnode* node, P3_fx* fx, GLfloat sphere[4]) {
  P3_xface* face;
  int i;
  if (P3_point_distance_to (sphere, node->sphere) - sphere[3] - node->sphere[3] < 0.0) {
    for (i = 0; i < node->nb_faces; i++) {
      face = (P3_xface*) node->faces[i];
      P3_mesh_vertex_fx_in_sphere (mesh, face->v1, fx, sphere);
      P3_mesh_vertex_fx_in_sphere (mesh, face->v2, fx, sphere);
      P3_mesh_vertex_fx_in_sphere (mesh, face->v3, fx, sphere);
      if (face->option & P3_FACE_QUAD) P3_mesh_vertex_fx_in_sphere (mesh, face->v4, fx, sphere);
    }
    for (i = 0; i < node->nb_child; i++) {
      P3_mesh_node_fx_in_sphere (mesh, node->child[i], fx, sphere);
    }
  }
}

void P3_mesh_fx_in_sphere (P3_mesh* mesh, P3_fx* fx, GLfloat sphere[4]) {
  int i;
  if (mesh->option & P3_MESH_TREE) {
    for (i = 0; i < mesh->nb_vertices; i++) mesh->vertex_options[i] &= ~P3_VERTEX_MADE;
    P3_mesh_node_fx_in_sphere (mesh, (P3_xnode*) mesh->xtra1, fx, sphere);
  } else {
    for (i = 0; i < mesh->nb_vertices; i++) {
      if (P3_point_is_in_sphere (sphere, mesh->vertex_coords[i]) == P3_TRUE) fx->apply (fx, i);
    }
  }
}

static void P3_mesh_vertex_fx_in_cylinderY (P3_mesh* mesh, int index, P3_fx* fx, GLfloat cylinder[3]) {
  GLfloat* f;
  GLfloat x, z;
  if (!(mesh->vertex_options[index] & P3_VERTEX_MADE)) {
    mesh->vertex_options[index] |= P3_VERTEX_MADE;
    f = mesh->vertex_coords[index];
    x = f[0] - cylinder[0];
    z = f[2] - cylinder[1];
    if (x * x + z * z <= cylinder[2] * cylinder[2]) fx->apply (fx, index);
  }
}

static void P3_mesh_node_fx_in_cylinderY (P3_mesh* mesh, P3_xnode* node, P3_fx* fx, GLfloat cylinder[3]) {
  P3_xface* face;
  GLfloat x, z;
  int i;
  x = node->sphere[0] - cylinder[0];
  z = node->sphere[2] - cylinder[1];
  if (sqrt (x * x + z * z) - cylinder[2] - node->sphere[3] <= 0.0) {
    for (i = 0; i < node->nb_faces; i++) {
      face = (P3_xface*) node->faces[i];
      P3_mesh_vertex_fx_in_cylinderY (mesh, face->v1, fx, cylinder);
      P3_mesh_vertex_fx_in_cylinderY (mesh, face->v2, fx, cylinder);
      P3_mesh_vertex_fx_in_cylinderY (mesh, face->v3, fx, cylinder);
      if (face->option & P3_FACE_QUAD) P3_mesh_vertex_fx_in_cylinderY (mesh, face->v4, fx, cylinder);
    }
    for (i = 0; i < node->nb_child; i++) {
      P3_mesh_node_fx_in_cylinderY (mesh, node->child[i], fx, cylinder);
    }
  }
}

void P3_mesh_fx_in_cylinderY (P3_mesh* mesh, P3_fx* fx, GLfloat cylinder[3]) {
  int i;
  if (mesh->option & P3_MESH_TREE) {
    for (i = 0; i < mesh->nb_vertices; i++) mesh->vertex_options[i] &= ~P3_VERTEX_MADE;
    P3_mesh_node_fx_in_cylinderY (mesh, (P3_xnode*) mesh->xtra1, fx, cylinder);
  } else {
    GLfloat* f;
    GLfloat x, z;
    for (i = 0; i < mesh->nb_vertices; i++) {
      f = mesh->vertex_coords[i];
      x = f[0] - cylinder[0];
      z = f[2] - cylinder[1];
      if (sqrt (x * x + z * z) <= cylinder[2]) fx->apply (fx, i);
    }
  }
}

void P3_mesh_fx_in_layerY (P3_mesh* mesh, P3_fx* fx, GLfloat layer[2]) {
  GLfloat* f;
  int i;
  for (i = 0; i < mesh->nb_vertices; i++) {
    f = mesh->vertex_coords[i];
    if (f[1] >= layer[0] && f[1] <= layer[1]) fx->apply (fx, i);
  }
}


/*==================+
 | MESH FX INSTANCE |
 +==================*/

P3_class P3_class_mesh_fxinstance = {
  P3_ID_MESH_FXINSTANCE,
  (batch_func)     P3_mesh_fxinstance_batch,
  (render_func)    P3_mesh_fxinstance_render,
  (shadow_func)    0,//P3_mesh_fxinstance_shadow,
  (raypick_func)   P3_mesh_fxinstance_raypick,
  (raypick_b_func) P3_mesh_fxinstance_raypick_b,
};


P3_mesh_fxinstance* P3_mesh_fxinstance_new (P3_mesh_fxinstance* fxi) {
  if (fxi == NULL) fxi = (P3_mesh_fxinstance*) malloc (sizeof (P3_mesh_fxinstance));
  fxi->class = &P3_class_mesh_fxinstance;
  fxi->mesh = NULL;
  fxi->vertex_options = NULL;
  fxi->vertex_warfogs = NULL;
  /* init fx cause if we create a P3_mesh_fxinstance that's to use fx! */
  P3_fx_init ();
  return fxi;
}

void P3_mesh_fxinstance_dealloc (P3_mesh_fxinstance* fxi) {
  free (fxi->vertex_options);
  free (fxi->vertex_warfogs);
}

void P3_mesh_fxinstance_set_mesh (P3_mesh_fxinstance* fxi, P3_mesh* mesh) {
  GLfloat* color;
  int i;
  fxi->mesh = mesh;
  if (mesh == NULL) {
    fxi->vertex_options = NULL;
    fxi->vertex_warfogs = NULL;
  } else {
    fxi->vertex_options = (char*) calloc (mesh->nb_vertices, sizeof (char));
    fxi->vertex_warfogs = (GLfloat**) malloc (mesh->nb_vertices * sizeof (GLfloat*));
    if (mesh->option & P3_MESH_DIFFUSES) {
      for (i = 0; i < mesh->nb_vertices; i++) fxi->vertex_warfogs[i] = mesh->vertex_diffuses[i];
    } else {
      if (mesh->nb_values >= 4) {
        color = mesh->values;
      } else {
        i = P3_xmesh_register_value ((P3_xmesh*) mesh, white, 4);
        color = mesh->values + i;
      }
      for (i = 0; i < mesh->nb_vertices; i++) fxi->vertex_warfogs[i] = color;
    }
  }
}

GLfloat* P3_mesh_fxinstance_register_color (P3_mesh_fxinstance* fxi, GLfloat color[4]) {
  P3_mesh* mesh = fxi->mesh;
  GLfloat* old = mesh->values;
  GLfloat* max = mesh->values + mesh->nb_values;
  int n = P3_xmesh_register_value ((P3_xmesh*) mesh, color, 4);
  int i;
  if (mesh->values != old) {
    for (i = 0; i < mesh->nb_vertices; i++) {
      if (fxi->vertex_warfogs[i] >= old && fxi->vertex_warfogs[i] < max)
        fxi->vertex_warfogs[i] = mesh->values + (fxi->vertex_warfogs[i] - old);
    }
  }
  return mesh->values + n;
}

void P3_mesh_fxinstance_raypick (P3_mesh_fxinstance* fxi, P3_raypick_data* data, P3_raypickable* parent) {
  if (fxi->mesh != NULL) P3_mesh_raypick (fxi->mesh, data, parent);
}

int P3_mesh_fxinstance_raypick_b (P3_mesh_fxinstance* fxi, P3_raypick_data* data, P3_raypickable* parent) {
  /* always raypick with the most accurate mesh (= the 1rst one) */
  if (fxi->mesh == NULL) return P3_FALSE;
  return P3_mesh_raypick_b (fxi->mesh, data, parent);
}

static void P3_mesh_fxinstance_face_batch (P3_mesh_fxinstance* fxi, P3_xface* face) {
  if (fxi->vertex_options[face->v1] & P3_VERTEX_INVISIBLE &&
      fxi->vertex_options[face->v2] & P3_VERTEX_INVISIBLE &&
      fxi->vertex_options[face->v3] & P3_VERTEX_INVISIBLE) {
    if (face->option & P3_FACE_TRIANGLE) 
      return;
    else if (fxi->vertex_options[face->v4] & P3_VERTEX_INVISIBLE) return;
  }
  if (fxi->vertex_options[face->v1] & P3_VERTEX_ALPHA ||
      fxi->vertex_options[face->v2] & P3_VERTEX_ALPHA ||
      fxi->vertex_options[face->v3] & P3_VERTEX_ALPHA ||
      (face->option & P3_FACE_QUAD && fxi->vertex_options[face->v3] & P3_VERTEX_ALPHA)) {
    P3_xpack_batch_face (fxi, P3_xpack_get_alpha (face->pack), face);
    return;
  }
  P3_xpack_batch_face (fxi, face->pack, face);
}

static void P3_mesh_fxinstance_node_batch (P3_mesh_fxinstance* fxi, P3_xnode* node) {
  int i;
  /* frustum test */
  if (P3_sphere_in_frustum (renderer->c_frustum, node->sphere) == P3_TRUE) {
    /* batch all faces */
    for (i = 0; i < node->nb_faces; i++) P3_mesh_fxinstance_face_batch (fxi, node->faces[i]);
    /* recurse */
    for (i = 0; i < node->nb_child; i++) P3_mesh_fxinstance_node_batch (fxi, node->child[i]);
  }
}

void P3_mesh_fxinstance_batch (P3_mesh_fxinstance* fxi, P3_instance* inst) {
  P3_mesh* mesh = fxi->mesh;
  P3_frustum* frustum = P3_renderer_get_frustum (inst);
  if (mesh == NULL) return;
  if (mesh->option & P3_MESH_HAS_SPHERE && 
      P3_sphere_in_frustum (frustum, (GLfloat*) mesh->xtra2) == P3_FALSE) { 
    return; 
  }
  P3_xmesh_batch_start (inst);
  /* batch each face */
  if (mesh->option & P3_MESH_TREE) {
    P3_mesh_fxinstance_node_batch (fxi, (P3_xnode*) mesh->xtra1);
  } else if (mesh->option & P3_MESH_FACE_LIST) {
    P3_mesh_face_list* flist = (P3_mesh_face_list*) mesh->xtra2;
    int i;
    for (i = 0; i < flist->nb_faces; i++) {
      P3_mesh_fxinstance_face_batch (fxi, flist->faces[i]);
    }
  } else {
    void* face = mesh->faces;
    void* max  = mesh->faces + mesh->faces_size;
    while (face < max) {
      P3_mesh_fxinstance_face_batch (fxi, (P3_xface*) face);
      face += P3_xmesh_face_size ((P3_xmesh*) mesh, face);
    }
  }
  P3_xmesh_batch_end ();
  if (mesh->option & P3_MESH_CELL_SHADING) {
    P3_renderer_batch (renderer->secondpass, fxi, inst, renderer->data->nb);
    P3_mesh_batch_outline (mesh, inst, frustum);
  }
}

void P3_mesh_fxinstance_render (P3_mesh_fxinstance* fxi, P3_instance* inst) {
  P3_xmesh_option_activate (fxi->mesh->option);
  renderer->colors = fxi->vertex_warfogs;
  if (fxi->mesh->option & P3_MESH_CELL_SHADING) {
    if (renderer->state == P3_RENDERER_STATE_SECONDPASS) {
      P3_mesh_render_outline (fxi->mesh);
    } else {
      P3_chunk* chunk;
      GLfloat* shades;
      chunk = P3_get_chunk ();
      P3_chunk_register (chunk, fxi->mesh->nb_vnormals * sizeof (GLfloat));
      shades = (GLfloat*) chunk->content;
      P3_mesh_prepare_cell_shading (fxi->mesh, inst, shades);
      /* active shader */
      light_shader = ((P3_mesh_cell_shading*) fxi->mesh->xtra1)->shader;
      P3_xmesh_pack_render_cellshaded ((P3_xmesh*) fxi->mesh, shades);
      P3_drop_chunk (chunk);
    }
  } else {
    P3_xmesh_pack_render ((P3_xmesh*) fxi->mesh);
  }
  P3_xmesh_option_inactivate (fxi->mesh->option);
}

/*
void P3_mesh_fxinstance_get_data (P3_mesh_* fxi, P3_chunk* chunk) {
  P3_chunk_add_float (chunk, mesh->LOD_factor);
}

void P3_mesh_fxinstance_set_data (P3_mesh_* fxi, P3_chunk* chunk) {
  mesh->LOD_factor = P3_chunk_get_float (chunk);
}
*/
