/*
 * P3
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**********************************************
 * mesh.c
 * Copyright (C) 2001-2003 Bertrand 'blam' LAMY
 **********************************************/

#include "p3_base.h"
#include "math3d.h"
#include "util.h"
#include "raypick.h"
#include "material.h"
#include "frustum.h"
#include "coordsys.h"
#include "renderer.h"
#include "light.h"
#include "fx.h"
#include "watercube.h"
#include "shadow.h"
#include "mesh.h"

extern GLfloat white[4];
extern GLfloat black[4];
extern P3_renderer* renderer;
extern int engine_option;
extern P3_material* current_material;
extern P3_material* light_shader;
extern int shadow_display_list;


P3_class P3_class_mesh = {
  P3_ID_MESH,
  (batch_func)     P3_mesh_batch,
  (render_func)    P3_mesh_render,
  (shadow_func)    P3_mesh_shadow,
  (raypick_func)   P3_mesh_raypick,
  (raypick_b_func) P3_mesh_raypick_b,
};


P3_mesh* P3_mesh_new (P3_mesh* mesh) {
  if (mesh == NULL) mesh = (P3_mesh*) malloc (sizeof (P3_mesh));
  mesh->class = &P3_class_mesh;
  P3_xmesh_initialize ((P3_xmesh*) mesh);
  mesh->xtra1 = NULL;
  mesh->xtra2 = NULL;
  return mesh;
}

static void P3_mesh_face_list_dealloc (P3_mesh_face_list* flist) {
  free (flist->faces);
  free (flist);
}

static void P3_mesh_display_list_dealloc (P3_mesh_display_list* dlist) {
  int i;
  if (engine_option & P3_GL_INITED) {
    for (i = 0; i < dlist->nb_list_opaque + dlist->nb_list_alpha; i++) {
      glDeleteLists ((dlist->dlists + i)->dlist, 1);
    }
  }
  free (dlist->dlists);
  free (dlist);
}

static void P3_mesh_free_xtra1 (P3_mesh* mesh) {
  if (mesh->xtra1 != NULL) {
    if (mesh->option & P3_MESH_TREE) {
      P3_xnode_dealloc ((P3_xnode*) mesh->xtra1);
      mesh->option -= P3_MESH_TREE;
    } else if (mesh->option & P3_MESH_CELL_SHADING) {
      /* shader is not decrefed cause it belongs to the mesh->materials array */
//      P3_mesh_face_list_dealloc ((P3_mesh_face_list*) mesh->xtra1);
      mesh->option -= P3_MESH_CELL_SHADING;
//      mesh->option -= P3_MESH_FACE_LIST;
      mesh->option -= P3_MESH_HAS_SPHERE;
    } else if (mesh->option & P3_MESH_DISPLAY_LISTS) {
      P3_mesh_display_list_dealloc ((P3_mesh_display_list*) mesh->xtra1);
      mesh->option -= P3_MESH_DISPLAY_LISTS;
    }
  }
  mesh->xtra1 = NULL;
}

static void P3_mesh_free_xtra2 (P3_mesh* mesh) {
  if (mesh->xtra2 != NULL) {
    if (mesh->option & P3_MESH_FACE_LIST) {
      P3_mesh_face_list_dealloc ((P3_mesh_face_list*) mesh->xtra2);
      mesh->option -= P3_MESH_FACE_LIST;
      mesh->option -= P3_MESH_HAS_SPHERE;
    } else if (mesh->option & P3_MESH_HAS_SPHERE) {
      free (mesh->xtra2);
      mesh->option -= P3_MESH_HAS_SPHERE;
    }
  }
  mesh->xtra2 = NULL;
}

void P3_mesh_dealloc (P3_mesh* mesh) {
  P3_mesh_free_xtra1 (mesh);
  P3_mesh_free_xtra2 (mesh);
  P3_xmesh_free_data ((P3_xmesh*) mesh);
}

void P3_mesh_check_fx (P3_mesh* mesh) {
  GLfloat* color;
  int i;
  /* need fx */
  P3_fx_init ();
  if (mesh->option & P3_MESH_DISPLAY_LISTS) P3_mesh_free_xtra1 (mesh);
  if (!(mesh->option & P3_MESH_VERTEX_OPTIONS)) {
    mesh->vertex_options = (char*) calloc (mesh->nb_vertices, sizeof (char));
    mesh->option |= P3_MESH_VERTEX_OPTIONS;
  }
  if (!(mesh->option & P3_MESH_WARFOGS)) {
    mesh->vertex_warfogs = (GLfloat**) malloc (mesh->nb_vertices * sizeof (GLfloat*));
    mesh->option |= P3_MESH_WARFOGS;
    if (mesh->option & P3_MESH_DIFFUSES) {
      for (i = 0; i < mesh->nb_vertices; i++) mesh->vertex_warfogs[i] = mesh->vertex_diffuses[i];
    } else {
      if (mesh->nb_values >= 4) {
        color = mesh->values;
      } else {
        i = P3_xmesh_register_value ((P3_xmesh*) mesh, white, 4);
        color = mesh->values + i;
      }
      for (i = 0; i < mesh->nb_vertices; i++) mesh->vertex_warfogs[i] = color;
    }
  }
}


/*============+
 | RAYPICKING |
 +============*/

void P3_mesh_raypick (P3_mesh* mesh, P3_raypick_data* data, P3_raypickable* parent) {
  GLfloat* raydata;
  raydata = P3_raypickable_get_raypick_data (parent, data);
  if (mesh->option & P3_MESH_TREE) {
    P3_xmesh_node_raypick ((P3_xmesh*) mesh, (P3_xnode*) mesh->xtra1, raydata, data, parent);
  } else {
    if (mesh->option & P3_MESH_HAS_SPHERE &&
        P3_sphere_raypick (raydata, (GLfloat*) mesh->xtra2) == P3_FALSE) {
      return;
    }
    if (mesh->option & P3_MESH_FACE_LIST) {
      P3_mesh_face_list* flist = (P3_mesh_face_list*) mesh->xtra2;
      int i;
      for (i = 0; i < flist->nb_faces; i++) {
        P3_xmesh_face_raypick ((P3_xmesh*) mesh, flist->faces[i], raydata, data, parent);
      }
    } else {
      P3_xmesh_raypick ((P3_xmesh*) mesh, raydata, data, parent);
    }
  }
}

int P3_mesh_raypick_b (P3_mesh* mesh, P3_raypick_data* data, P3_raypickable* parent) {
  GLfloat* raydata;
  raydata = P3_raypickable_get_raypick_data (parent, data);
  if (mesh->option & P3_MESH_TREE) {
    return P3_xmesh_node_raypick_b ((P3_xmesh*) mesh, (P3_xnode*) mesh->xtra1, raydata, data);
  } else {
    if (mesh->option & P3_MESH_HAS_SPHERE &&
        P3_sphere_raypick (raydata, (GLfloat*) mesh->xtra2) == P3_FALSE) {
      return P3_FALSE;
    }
    if (mesh->option & P3_MESH_FACE_LIST) {
      P3_mesh_face_list* flist = (P3_mesh_face_list*) mesh->xtra2;
      int i;
      for (i = 0; i < flist->nb_faces; i++) {
        if (P3_xmesh_face_raypick_b ((P3_xmesh*) mesh, flist->faces[i], raydata, data) == P3_TRUE) {
          return P3_TRUE;
        }
      }
    } else {
      return P3_xmesh_raypick_b ((P3_xmesh*) mesh, raydata, data);
    }
  }
  return P3_FALSE;
}


/*===================+
 | XTRA CONSTRUCTORS |
 +===================*/

static void P3_mesh_create_faces_list (P3_mesh_face_list* flist, P3_mesh* mesh) {
  P3_xface* face;
  int i;
  flist->nb_faces = 0;
  face = (P3_xface*) mesh->faces;
  while ((void*) face < mesh->faces + mesh->faces_size) {
    face = ((void*) face) + P3_xmesh_face_size ((P3_xmesh*) mesh, face);
    flist->nb_faces++;
  }
  flist->faces = (P3_xface**) malloc (flist->nb_faces * sizeof (P3_xface*));
  i = 0;
  face = (P3_xface*) mesh->faces;
  while ((void*) face < mesh->faces + mesh->faces_size) {
    flist->faces[i] = face;
    face = ((void*) face) + P3_xmesh_face_size ((P3_xmesh*) mesh, face);
    i++;
  }
}

static void P3_mesh_display_list_init (P3_mesh* mesh) {
  P3_mesh_display_list* dlists;
  P3_display_list* dlist;
  P3_xface* face;
  int i;
  dlists = (P3_mesh_display_list*) mesh->xtra1;
  P3_xmesh_option_activate (mesh->option);
  for (i = 0; i < dlists->nb_list_opaque + dlists->nb_list_alpha; i++) {
    dlist = dlists->dlists + i;
    dlist->dlist = glGenLists (1);
    P3_material_activate (dlist->material);
    P3_xface_option_activate (dlist->option);
    glNewList (dlist->dlist, GL_COMPILE);
    if (dlist->option & P3_FACE_TRIANGLE) {
      glBegin (GL_TRIANGLES);
    } else if (dlist->option & P3_FACE_QUAD) {
      glBegin (GL_QUADS);
    }
    face = mesh->faces;
    while ((void*) face < mesh->faces + mesh->faces_size) {
      if ((face->option & P3_DISPLAY_LIST_OPTIONS) == dlist->option && face->pack->material == dlist->material) {
        if (face->option & P3_FACE_QUAD) P3_xmesh_quad_render     ((P3_xmesh*) mesh, face);
        else                             P3_xmesh_triangle_render ((P3_xmesh*) mesh, face);
      }
      face = ((void*) face) + P3_xmesh_face_size ((P3_xmesh*) mesh, face);
    }
    glEnd ();
    glEndList ();
    P3_xface_option_inactivate (dlist->option);
  }
  P3_xmesh_option_inactivate (mesh->option);
  mesh->option |= P3_MESH_INITED;
}

void P3_mesh_build_tree (P3_mesh* mesh) {
  P3_mesh_free_xtra1 (mesh);
  mesh->xtra1 = P3_xmesh_build_tree ((P3_xmesh*) mesh);
  mesh->option |= P3_MESH_TREE;
}

void P3_mesh_optimize_tree (P3_mesh* mesh, GLfloat collapse, int mode, GLfloat param) {
  if (mesh->option & P3_MESH_TREE)
    P3_xmesh_optimize_tree ((P3_xnode*) mesh->xtra1, collapse, mode, param);
}

void P3_mesh_set_xtra1_display_lists (P3_mesh* mesh) {
  P3_mesh_display_list* displist;
  P3_display_list* dl;
  P3_xface* face;
  void* max;
  int nb, i, k;
  P3_mesh_free_xtra1 (mesh);
  displist = (P3_mesh_display_list*) malloc (sizeof (P3_mesh_display_list));
  displist->nb_list_opaque = 0;
  displist->nb_list_alpha = 0;
  nb = 0;
  displist->dlists = NULL;
  for (k = 0; k < 2; k++) {
    face = (P3_xface*) mesh->faces;
    max = mesh->faces + mesh->faces_size;
    while ((void*) face < max) {
      if ((face->option & P3_FACE_ALPHA && k == 1) || (!(face->option & P3_FACE_ALPHA) && k == 0)) {
        for (i = 0; i < nb; i++) {
          dl = displist->dlists + i;
          if (dl->material == face->pack->material && dl->option == (face->option & P3_DISPLAY_LIST_OPTIONS)) {
            i = -1;
            break;
          }
        }
        if (i != -1) {
          displist->dlists = (P3_display_list*) realloc (displist->dlists, (nb + 1) * sizeof (P3_display_list));
          dl = displist->dlists + nb;
          dl->material = face->pack->material;
          dl->option = face->option & P3_DISPLAY_LIST_OPTIONS;
          if (dl->option & P3_FACE_ALPHA) (displist->nb_list_alpha)++;
          else                            (displist->nb_list_opaque)++;
          nb++;
        }
      }
      face = ((void*) face) + P3_xmesh_face_size ((P3_xmesh*) mesh, face);
    }
  }
  mesh->xtra1 = displist;
  mesh->option |= P3_MESH_DISPLAY_LISTS;
}

void P3_mesh_set_xtra1_cell_shading (P3_mesh* mesh, P3_material* shader, GLfloat* color, GLfloat line_width_factor) {
  P3_mesh_cell_shading* cshade;
  P3_mesh_free_xtra1 (mesh);
  cshade = (P3_mesh_cell_shading*) malloc (sizeof (P3_mesh_cell_shading));
  cshade->shader = shader;
  if (shader != NULL) P3_xmesh_register_material ((P3_xmesh*) mesh, shader);
  memcpy (cshade->line_color, color, 4 * sizeof (GLfloat));
  cshade->line_width_factor = line_width_factor;
  mesh->option |= P3_MESH_CELL_SHADING;
  mesh->xtra1 = cshade;
}

void P3_mesh_set_xtra2_sphere (P3_mesh* mesh) {
  P3_mesh_free_xtra2 (mesh);
  mesh->xtra2 = malloc (4 * sizeof (GLfloat));
  if (mesh->nb_coords > 0) {
    P3_sphere_from_points ((GLfloat*)  mesh->xtra2, mesh->coords, mesh->nb_coords);
  }
  mesh->option |= P3_MESH_HAS_SPHERE;
}

void P3_mesh_set_xtra2_face_list (P3_mesh* mesh) {
  P3_mesh_face_list* flist;
  flist = (P3_mesh_face_list*) malloc (sizeof (P3_mesh_face_list));
  P3_sphere_from_points (flist->sphere, mesh->coords, mesh->nb_coords);
  P3_mesh_create_faces_list (flist, mesh);
  P3_mesh_free_xtra2 (mesh);
  mesh->xtra2 = flist;
  mesh->option |= P3_MESH_FACE_LIST;
  mesh->option |= P3_MESH_HAS_SPHERE;
}


/*===========+
 | RENDERING |
 +===========*/

void P3_mesh_batch_outline (P3_mesh* mesh, P3_instance* inst, P3_frustum* frustum) {

// TO DO computation + rendering can be done during rendering ???

  P3_chain* face_batched;
  P3_chain* faces;
  P3_chain* max;
  GLfloat* plane;
  P3_xface* face;
  P3_xface** neighbors;
  P3_xface* n;
  GLfloat d;
  int nbv;
  int k;
  int nb = 0;
  if (((P3_mesh_cell_shading*) mesh->xtra1)->line_width_factor <= 0.0) {
    /* no outline to draw */
    return;
  }
  /* compute distance between camera and mesh to adjust outline width */
  if (mesh->option & P3_MESH_HAS_SPHERE) {
    d = P3_sphere_distance_point ((GLfloat*) mesh->xtra2, frustum->position);
    if (d < 1.0) 
      d = ((P3_mesh_cell_shading*) mesh->xtra1)->line_width_factor;
    else
      d = ((P3_mesh_cell_shading*) mesh->xtra1)->line_width_factor / d;
//      d = ((P3_mesh_cell_shading*) mesh->xtra1)->line_width_factor * 0.5 * (1.0 + 1.0 / d);
  } else {
    d = 2.0;
  }
  /* batch in secondpass */
  P3_chunk_add_float (renderer->data, d);
  /* compute line to draw */
  max = (P3_chain*) (renderer->faces->content + renderer->faces->nb);
  face_batched = (P3_chain*) (renderer->faces->content + renderer->face_start);
  faces = face_batched;
  while (faces < max) {
    face = (P3_xface*) faces->data;
    plane = face->normal;
    if (plane[0] * frustum->position[0] + plane[1] * frustum->position[1] + plane[2] * frustum->position[2] + plane[3] > 0.0)
      face->option |= P3_FACE_FRONT;
    else
      face->option |= P3_FACE_BACK;
    faces++;
  }
  /* find edges */    
  faces = face_batched;
  while (faces < max) {
    face = (P3_xface*) faces->data;
    if (face->option & P3_FACE_FRONT) {
      if (face->option & P3_FACE_QUAD) {
        neighbors = (P3_xface**) (&(face->v4) + 1);
        nbv = 4;
      } else {
        neighbors = (P3_xface**) (&(face->v4));
        nbv = 3;
      }
      /* test if neighbors are back */
      for (k = 0; k < nbv; k++) {
        n = neighbors[k];
        if (n == NULL || n->option & P3_FACE_BACK) {
          /* add edge k = vertices k and k + 1 */
          P3_chunk_add_int (renderer->data, (&(face->v1))[k]);
          if (k < nbv - 1) {
            P3_chunk_add_int (renderer->data, (&(face->v1))[k + 1]);
          } else {
            P3_chunk_add_int (renderer->data, face->v1);
          }
          nb++;
        }
      }
    }
    faces++;
  }
  P3_chunk_add_int (renderer->data, -1);
  /* remove face added option */
  faces = face_batched;
  while (faces < max) {
    ((P3_xface*) faces->data)->option &= ~(P3_FACE_FRONT | P3_FACE_BACK);
    faces++;
  }   
}

void P3_mesh_batch (P3_mesh* mesh, P3_instance* inst) {
  P3_frustum* frustum = P3_renderer_get_frustum (inst);
  if (mesh->option & P3_MESH_HAS_SPHERE && 
      P3_sphere_in_frustum (frustum, (GLfloat*) mesh->xtra2) == P3_FALSE) { 
    return; 
  }
  if (mesh->option & P3_MESH_DISPLAY_LISTS) {
    P3_mesh_display_list* dlist = (P3_mesh_display_list*) mesh->xtra1;
    if (dlist->nb_list_opaque > 0) P3_renderer_batch (renderer->opaque, mesh, inst, -1);
    if (dlist->nb_list_alpha  > 0) P3_renderer_batch (renderer->alpha,  mesh, inst, -1);
  } else {
    P3_xmesh_batch_start (inst);
    /* batch each face */
    if (mesh->option & P3_MESH_TREE) {
      P3_xmesh_node_batch ((P3_xmesh*) mesh, (P3_xnode*) mesh->xtra1, frustum);
    } else if (mesh->option & P3_MESH_FACE_LIST) {
      P3_mesh_face_list* flist = (P3_mesh_face_list*) mesh->xtra2;
      int i;
      for (i = 0; i < flist->nb_faces; i++) {
        P3_xmesh_face_batch ((P3_xmesh*) mesh, flist->faces[i]);
      }
    } else {
      void* face = mesh->faces;
      void* max  = mesh->faces + mesh->faces_size;
      while (face < max) {
        P3_xmesh_face_batch ((P3_xmesh*) mesh, (P3_xface*) face);
        face += P3_xmesh_face_size ((P3_xmesh*) mesh, face);
      }
    }
    P3_xmesh_batch_end ();
    if (mesh->option & P3_MESH_CELL_SHADING) {
      P3_renderer_batch (renderer->secondpass, mesh, inst, renderer->data->nb);
      P3_mesh_batch_outline (mesh, inst, frustum);
    }
  }
}

static void P3_mesh_prepare_cell_shading_shades (P3_mesh* mesh, GLfloat* shades, P3_list* lights) {
  P3_light* light;
  GLfloat* ptr1;
  GLfloat* ptr2;
  GLfloat v[3];
  GLfloat tmp;
  int i, j, k;
  if (mesh->option & P3_MESH_VERTEX_NORMALS) {
    /* 1 shade per vnormal */
    for (i = 0; i < lights->nb; i++) {
      light = (P3_light*) P3_list_get (lights, i);
      if (light->option & P3_LIGHT_DIRECTIONAL) {
        /* directional light */
        ptr1 = mesh->vnormals;
        for (j = 0; j < mesh->nb_vnormals; j++) {
          tmp = - P3_vector_dot_product (ptr1, light->data);
          if (tmp > 0.0) shades[j] += tmp;
          ptr1 += 3;
        }
      } else {
        /* positional light */
        ptr1 = mesh->vnormals;
        for (j = 0; j < mesh->nb_vnormals; j++) {
          /* search the coord that correspond to this vertex normal */
          for (k = 0; k < mesh->nb_vertices; k++) {
            if (mesh->vertex_normals[k] == ptr1) break;
          }
          ptr2 = mesh->vertex_coords[k];
          v[0] = light->data[0] - ptr2[0];
          v[1] = light->data[1] - ptr2[1];
          v[2] = light->data[2] - ptr2[2];
          P3_vector_normalize (v);
          tmp = P3_vector_dot_product (ptr1, v);
          if (tmp > 0.0) shades[j] += tmp;
          ptr1 += 3;
        }
      }
    }
  } else {
    /* 1 shade per coord */
    for (i = 0; i < lights->nb; i++) {
      light = (P3_light*) P3_list_get (lights, i);
      if (light->option & P3_LIGHT_DIRECTIONAL) {
        /* directional light */
        ptr1 = mesh->vnormals;
        for (j = 0; j < mesh->nb_vnormals; j++) {
          tmp = - P3_vector_dot_product (ptr1, light->data);
          if (tmp > 0.0) shades[j] += tmp;
          ptr1 += 3;
        }
      } else {
        /* positional light */
        ptr1 = mesh->vnormals;
        ptr2 = mesh->coords;
        for (j = 0; j < mesh->nb_vnormals; j++) {
          v[0] = light->data[0] - ptr2[0];
          v[1] = light->data[1] - ptr2[1];
          v[2] = light->data[2] - ptr2[2];
          P3_vector_normalize (v);
          tmp = P3_vector_dot_product (ptr1, v);
          if (tmp > 0.0) shades[j] += tmp; 
          ptr1 += 3;
          ptr2 += 3;
        }
      }
    }
  }
}

void P3_mesh_prepare_cell_shading (P3_mesh* mesh, P3_instance* inst, GLfloat* shades) {
  int n;
  P3_light_list_cast_into (renderer->top_lights, (P3_coordsys*) inst);
  P3_light_list_cast_into (renderer->c_context->lights, (P3_coordsys*) inst);
  if (mesh->nb_vnormals > 0) {
    for (n = 0; n < mesh->nb_vnormals; n++) shades[n] = 0.0;
    P3_mesh_prepare_cell_shading_shades (mesh, shades, renderer->top_lights);
    P3_mesh_prepare_cell_shading_shades (mesh, shades, renderer->c_context->lights);
    /* clip shade texcoord values */
    for (n = 0; n < mesh->nb_vnormals; n++) {
      /* do not clip with interval [0, 1] because smooth magnification of texture
       * causes visual bugs
       */
      if (shades[n] > 0.95) shades[n] = 0.95;
      if (shades[n] < 0.05) shades[n] = 0.05;
    }
  }
}

void P3_mesh_render_outline (P3_mesh* mesh) {
  void* data;
  int* ptr;
  int i;
  data = renderer->data->content + renderer->data->nb;
  glLineWidth (*((GLfloat*) data));
  glColor4fv (((P3_mesh_cell_shading*) mesh->xtra1)->line_color);
  glDisable (GL_LIGHTING);
  glDepthFunc (GL_LEQUAL);
  P3_material_inactivate (current_material);
  current_material = NULL;
  ptr = (int*) (data + sizeof (GLfloat));
  if (renderer->colors == NULL) {
    glBegin (GL_LINES);
    while (1) { 
      i = *ptr;
      if (i == -1) break;
      glVertex3fv (mesh->vertex_coords[i]);
      ptr++;
    }
    glEnd ();
  } else {
    GLfloat* color_ptr = ((P3_mesh_cell_shading*) mesh->xtra1)->line_color;
    GLfloat alpha = color_ptr[3];
    glEnable (GL_BLEND);
    glBegin (GL_LINES);
    while (1) { 
      i = *ptr;
      if (i == -1) break;
      color_ptr[3] = renderer->colors[i][3];
      glColor4fv (color_ptr);
      glVertex3fv (mesh->vertex_coords[i]);
      ptr++;
    }
    glEnd ();
    color_ptr[3] = alpha;
    glDisable (GL_BLEND);
  }
  glEnable (GL_LIGHTING);
  glDepthFunc (GL_LESS);
  glColor4fv (white);
}

void P3_mesh_render (P3_mesh* mesh, P3_instance* inst) {
  int i;
  P3_xmesh_option_activate (mesh->option);
  if (mesh->option & P3_MESH_WARFOGS) {
    renderer->colors = mesh->vertex_warfogs;
  } else if (mesh->option & P3_MESH_DIFFUSES) {
    renderer->colors = mesh->vertex_diffuses;
  } else {
    renderer->colors = NULL;
  }
  if (mesh->option & P3_MESH_DISPLAY_LISTS) {
    P3_mesh_display_list* displist = (P3_mesh_display_list*) mesh->xtra1;
    P3_display_list* dlists;
    P3_display_list* dlist;
    int nb;
    if (!(mesh->option & P3_MESH_INITED)) P3_mesh_display_list_init (mesh);
    if (renderer->state == P3_RENDERER_STATE_OPAQUE) {
      dlists = displist->dlists;
      nb = displist->nb_list_opaque;
    } else {
      dlists = displist->dlists + displist->nb_list_opaque;
      nb = displist->nb_list_alpha;
    }
    for (i = 0; i < nb; i++) {
      dlist = dlists + i;
      P3_xface_option_activate (dlist->option);
      P3_material_activate (dlist->material);
      glCallList (dlist->dlist);
      P3_xface_option_inactivate (dlist->option);
    }
  } else {
    if (mesh->option & P3_MESH_CELL_SHADING) {
      if (renderer->state == P3_RENDERER_STATE_SECONDPASS) {
        P3_mesh_render_outline (mesh);
      } else {
        P3_chunk* chunk;
        GLfloat* shades;
        chunk = P3_get_chunk ();
        P3_chunk_register (chunk, mesh->nb_vnormals * sizeof (GLfloat));
        shades = (GLfloat*) chunk->content;
        P3_mesh_prepare_cell_shading (mesh, inst, shades);
        /* active shader */
        light_shader = ((P3_mesh_cell_shading*) mesh->xtra1)->shader;
        P3_xmesh_pack_render_cellshaded ((P3_xmesh*) mesh, shades);
        P3_drop_chunk (chunk);
      }
    } else {
      P3_xmesh_pack_render ((P3_xmesh*) mesh);
    }
  }
  P3_xmesh_option_inactivate (mesh->option);
}


/*========+
 | SHADOW |
 +========*/

int P3_mesh_shadow (P3_mesh* mesh, P3_instance* inst, P3_light* light) {

// TO DO assume mesh has a face list

  P3_mesh_face_list* fl = (P3_mesh_face_list*) mesh->xtra2;
  P3_frustum* frustum;
  P3_xface** neighbors;
  P3_xface* face;
  GLfloat* coord_ptr;
  GLfloat coord[4];
  GLfloat cone[9];
  GLfloat b = renderer->c_camera->back;
  int nbv, i, k;
  if (!(mesh->option & P3_MESH_SHADOW_CAST)) return P3_FALSE;
  /* tag all faces front or back */
  P3_light_cast_into (light, (P3_coordsys*) inst);
  if (light->option & P3_LIGHT_DIRECTIONAL) {

printf ("-->\n");

    P3_cone_from_sphere_and_vector (cone, (GLfloat*) mesh->xtra2, light->data, b);
    for (i = 0; i < fl->nb_faces; i++) {
      face = fl->faces[i];
      if (mesh->option & P3_MESH_VERTEX_OPTIONS) {
        if (mesh->vertex_options[face->v1] & P3_VERTEX_INVISIBLE &&
            mesh->vertex_options[face->v2] & P3_VERTEX_INVISIBLE &&
            mesh->vertex_options[face->v3] & P3_VERTEX_INVISIBLE &&
            (face->option & P3_FACE_TRIANGLE || mesh->vertex_options[face->v4] & P3_VERTEX_INVISIBLE)) {
          continue;
        }
      }
      /* consider double sided faces as the others faces. that's not 100% working
       * but it's the best I can do
       */
      if (P3_vector_dot_product (light->data, face->normal) >= 0.0) {
        face->option |= P3_FACE_BACK;

printf ("back\n");

      } else {
        face->option |= P3_FACE_FRONT;

printf ("front\n");

      }
    }
  } else {
    if (P3_cone_from_sphere_and_origin (cone, (GLfloat*) mesh->xtra2, light->data, b) == P3_FALSE) return P3_FALSE;
    for (i = 0; i < fl->nb_faces; i++) {
      face = fl->faces[i];
      if (mesh->option & P3_MESH_VERTEX_OPTIONS) {
        if (mesh->vertex_options[face->v1] & P3_VERTEX_INVISIBLE &&
            mesh->vertex_options[face->v2] & P3_VERTEX_INVISIBLE &&
            mesh->vertex_options[face->v3] & P3_VERTEX_INVISIBLE &&
            (face->option & P3_FACE_TRIANGLE || mesh->vertex_options[face->v4] & P3_VERTEX_INVISIBLE)) {
          continue;
        }
      }
      if (light->data[0] * face->normal[0] + light->data[1] * face->normal[1] + light->data[2] * face->normal[2] + face->normal[3] > 0.0) {
        face->option |= P3_FACE_FRONT;
      } else {
        face->option |= P3_FACE_BACK;
      }
    }
  }

  /* draw shadow volume 1rst step */
  glStencilFunc (GL_ALWAYS, 1, 0xFFFFFFFF);
  glFrontFace (GL_CW);
  glStencilOp (GL_KEEP, GL_KEEP, GL_INCR);
  glLoadMatrixf (inst->render_matrix);
  glNewList (shadow_display_list, GL_COMPILE_AND_EXECUTE);
  glBegin (GL_QUADS);
  /* test if camera is inside the shadow */
  frustum = P3_renderer_get_frustum (inst);
  coord[0] = 0.5 * (frustum->points[0] + frustum->points[6]);
  coord[1] = 0.5 * (frustum->points[1] + frustum->points[7]);
  coord[2] = 0.5 * (frustum->points[2] + frustum->points[8]);
  coord[3] = P3_point_distance_to (coord, frustum->points);
  if (P3_sphere_is_in_cone (coord, cone) == P3_TRUE) {
    /* camera is inside the shadow => special case 
     * we must draw the intersection of the shadow volume with the camera front plane
     * by chance we already have functions to do such thing in the watercube ;)
     */
    P3_watercube_underwater* underwater;
    P3_chunk* chunk = P3_get_chunk ();
    GLfloat* coord_ptr2;
    GLfloat* m1;
    GLfloat* m2;
    GLfloat coord2[3];
    GLfloat coord3[3];
    GLfloat coord4[3];
    m1 = P3_coordsys_get_root_matrix ((P3_coordsys*) inst);
    m2 = P3_coordsys_get_inverted_root_matrix ((P3_coordsys*) renderer->c_camera);
    /* find edges and draw shadow volume */
    for (i = 0; i < fl->nb_faces; i++) {
      face = fl->faces[i];
      if (face->option & P3_FACE_BACK) {
        /* test if neighbors are front */
        if (face->option & P3_FACE_QUAD) { 
          neighbors = (P3_xface**) (&(face->v4) + 1);
          nbv = 4;
        } else {
          neighbors = (P3_xface**) (&(face->v4));
          nbv = 3;
        }
        for (k = 0; k < nbv; k++) {
          if (neighbors[k] == NULL || neighbors[k]->option & P3_FACE_FRONT) {

// TO DO avoid pushing 2 vertices with the same coord

            /* add edge k = vertices k and k + 1 */
            coord_ptr = mesh->vertex_coords[(&(face->v1))[k]];
            glVertex3fv (coord_ptr);
            /* push coord far away */
            if (light->option & P3_LIGHT_DIRECTIONAL) {
              coord[0] = coord_ptr[0] + b * light->data[0];
              coord[1] = coord_ptr[1] + b * light->data[1];
              coord[2] = coord_ptr[2] + b * light->data[2];
              glVertex3fv (coord);
            } else {
              coord[0] = coord_ptr[0] - light->data[0];
              coord[1] = coord_ptr[1] - light->data[1];
              coord[2] = coord_ptr[2] - light->data[2];
              P3_vector_normalize (coord);
              coord[0] = coord_ptr[0] + b * coord[0];
              coord[1] = coord_ptr[1] + b * coord[1];
              coord[2] = coord_ptr[2] + b * coord[2];
              glVertex3fv (coord);
            }
            if (k < nbv - 1) {
              coord_ptr2 = mesh->vertex_coords[(&(face->v1))[k + 1]];
            } else {
              coord_ptr2 = mesh->vertex_coords[face->v1];
            }
            /* push coord far away */
            if (light->option & P3_LIGHT_DIRECTIONAL) {
              coord2[0] = coord_ptr2[0] + b * light->data[0];
              coord2[1] = coord_ptr2[1] + b * light->data[1];
              coord2[2] = coord_ptr2[2] + b * light->data[2];
              glVertex3fv (coord2);
            } else {
              coord2[0] = coord_ptr2[0] - light->data[0];
              coord2[1] = coord_ptr2[1] - light->data[1];
              coord2[2] = coord_ptr2[2] - light->data[2];
              P3_vector_normalize (coord2);
              coord2[0] = coord_ptr2[0] + b * coord2[0];
              coord2[1] = coord_ptr2[1] + b * coord2[1];
              coord2[2] = coord_ptr2[2] + b * coord2[2];
              glVertex3fv (coord2);
            }
            glVertex3fv (coord_ptr2);
            /* convert point to camera coordsys */
            P3_point_by_matrix_copy (coord3, coord_ptr,  m1);
            P3_point_by_matrix_copy (coord4, coord_ptr2, m1);
            P3_point_by_matrix (coord,  m1);
            P3_point_by_matrix (coord2, m1);
            P3_point_by_matrix (coord,  m2);
            P3_point_by_matrix (coord2, m2);
            P3_point_by_matrix (coord3, m2);
            P3_point_by_matrix (coord4, m2);
            P3_watercube_underwater_intersect_face (chunk, coord3, coord, coord2);
            P3_watercube_underwater_intersect_face (chunk, coord2, coord4, coord3);
          }
        }
      }
    }
    glEnd ();
    glEndList ();
    glDisable (GL_CULL_FACE);
    glDisable (GL_DEPTH_TEST);
    glLoadIdentity ();
    if (chunk->nb == 0) {
      /* 2 cases: front plane entirely inside the shadow or entirely outside
       * use a raypicking to be sure
       */
      P3_raypick_data rdata;
      GLfloat f;
      rdata.option = P3_RAYPICK_HALF_LINE;
      rdata.raypicked = P3_get_list ();
      rdata.raypick_data = P3_get_chunk ();
      if (light->option & P3_LIGHT_DIRECTIONAL) {
        rdata.root_data[0] = renderer->r_frustum->position[0];
        rdata.root_data[1] = renderer->r_frustum->position[1];
        rdata.root_data[2] = renderer->r_frustum->position[2];
        coord_ptr = P3_coordsys_get_root_matrix ((P3_coordsys*) light);
        rdata.root_data[6] = renderer->c_camera->back;
        rdata.root_data[3] = rdata.root_data[0] + rdata.root_data[6] * coord_ptr[ 8];
        rdata.root_data[4] = rdata.root_data[1] + rdata.root_data[6] * coord_ptr[ 9];
        rdata.root_data[5] = rdata.root_data[2] + rdata.root_data[6] * coord_ptr[10];
      } else {
        coord_ptr = P3_coordsys_get_root_matrix ((P3_coordsys*) light);
        memcpy (rdata.root_data, coord_ptr + 12, 3 * sizeof (GLfloat));
        rdata.root_data[3] = renderer->r_frustum->position[0] - rdata.root_data[0];
        rdata.root_data[4] = renderer->r_frustum->position[1] - rdata.root_data[1];
        rdata.root_data[5] = renderer->r_frustum->position[2] - rdata.root_data[2];
        rdata.root_data[6] = P3_vector_length (rdata.root_data + 3);
        f = 1.0 / rdata.root_data[6];
        rdata.root_data[3] *= f;
        rdata.root_data[4] *= f;
        rdata.root_data[5] *= f;
      }
      if (P3_mesh_raypick_b (mesh, &rdata, (P3_raypickable*) inst) == P3_TRUE) {
        /* all the screen must be shadowed
         * that's cool I have the coordinate of the edge of the screen in the gl vertex array ;)
         */
        glDrawArrays (GL_QUADS, 0, 4);
      }
      for (i = 0; i < rdata.raypicked->nb; i++) {
        ((P3_raypickable*) P3_list_get (rdata.raypicked, i))->raypick_data = -1;
      }
      P3_drop_list  (rdata.raypicked);
      P3_drop_chunk (rdata.raypick_data);
    } else if (chunk->nb > 0) {
      underwater = P3_watercube_underwater_join_segments ((GLfloat*) chunk->content, chunk->nb / (3 * sizeof (GLfloat)));
      P3_watercube_underwater_draw_segments (underwater);
      free (underwater->points);
      free (underwater->seq_sizes);
      free (underwater);
    }
    P3_drop_chunk (chunk);
    glEnable (GL_CULL_FACE);
    glEnable (GL_DEPTH_TEST);
    /* draw shadow volume 2nd step */
    glLoadMatrixf (inst->render_matrix);
    glFrontFace (GL_CCW);
    glStencilFunc (GL_ALWAYS, 0, 0xFFFFFFFF);
    glStencilOp (GL_KEEP, GL_KEEP, GL_DECR);
    glCallList (shadow_display_list);
  } else {
    /* find edges and draw shadow volume */
    for (i = 0; i < fl->nb_faces; i++) {
      face = fl->faces[i];
      if (face->option & P3_FACE_BACK) {
        /* test if neighbors are front */
        if (face->option & P3_FACE_QUAD) { 
          neighbors = (P3_xface**) (&(face->v4) + 1);
          nbv = 4;
        } else {
          neighbors = (P3_xface**) (&(face->v4));
          nbv = 3;
        }
        for (k = 0; k < nbv; k++) {
          if (neighbors[k] == NULL || neighbors[k]->option & P3_FACE_FRONT) {

// TO DO avoid pushing 2 vertices with the same coord

            /* add edge k = vertices k and k + 1 */
            coord_ptr = mesh->vertex_coords[(&(face->v1))[k]];
            glVertex3fv (coord_ptr);
            /* push coord far away */
            if (light->option & P3_LIGHT_DIRECTIONAL) {
              glVertex3f (coord_ptr[0] + b * light->data[0], coord_ptr[1] + b * light->data[1], coord_ptr[2] + b * light->data[2]);
            } else {
              coord[0] = coord_ptr[0] - light->data[0];
              coord[1] = coord_ptr[1] - light->data[1];
              coord[2] = coord_ptr[2] - light->data[2];
              P3_vector_normalize (coord);
              glVertex3f (coord_ptr[0] + b * coord[0], coord_ptr[1] + b * coord[1], coord_ptr[2] + b * coord[2]);
            }
            if (k < nbv - 1) {
              coord_ptr = mesh->vertex_coords[(&(face->v1))[k + 1]];
            } else {
              coord_ptr = mesh->vertex_coords[face->v1];
            }
            /* push coord far away */
            if (light->option & P3_LIGHT_DIRECTIONAL) {
              glVertex3f (coord_ptr[0] + b * light->data[0], coord_ptr[1] + b * light->data[1], coord_ptr[2] + b * light->data[2]);
            } else {
              coord[0] = coord_ptr[0] - light->data[0];
              coord[1] = coord_ptr[1] - light->data[1];
              coord[2] = coord_ptr[2] - light->data[2];
              P3_vector_normalize (coord);
              glVertex3f (coord_ptr[0] + b * coord[0], coord_ptr[1] + b * coord[1], coord_ptr[2] + b * coord[2]);
            }
            glVertex3fv (coord_ptr);
          }
        }
      }
    }
    glEnd ();
    glEndList ();
    /* draw shadow volume 2nd step */
    glFrontFace (GL_CCW);
    glStencilFunc (GL_ALWAYS, 0, 0xFFFFFFFF);
    glStencilOp (GL_KEEP, GL_KEEP, GL_DECR);
    glCallList (shadow_display_list);
  }
  /* remove face tag */
  for (i = 0; i < fl->nb_faces; i++) fl->faces[i]->option &= ~(P3_FACE_FRONT | P3_FACE_BACK);
  return P3_TRUE;
}


/*==================+
 | SAVING / LOADING |
 +==================*/

static void P3_mesh_display_list_get_data (P3_mesh* mesh, P3_mesh_display_list* dlist, P3_chunk* chunk) {
  P3_display_list* dl;
  int i;
  P3_chunk_save_int (chunk, dlist->nb_list_opaque);
  P3_chunk_save_int (chunk, dlist->nb_list_alpha);
  for (i = 0; i < dlist->nb_list_opaque + dlist->nb_list_alpha; i++) {
    dl = dlist->dlists + i;
    P3_chunk_save_int (chunk, dl->option);
    P3_chunk_save_int (chunk, P3_xmesh_get_material_index ((P3_xmesh*) mesh, dl->material));
  }
}

static void P3_mesh_cell_shading_get_data (P3_mesh* mesh, P3_mesh_cell_shading* cshade, P3_chunk* chunk) {
  P3_chunk_save (chunk, cshade->line_color, 4 * sizeof (GLfloat));
  P3_chunk_save_float (chunk, cshade->line_width_factor);
  if (cshade->shader == NULL) {
    P3_chunk_save_int (chunk, -1);
  } else {
    P3_chunk_save_int (chunk, P3_xmesh_get_material_index ((P3_xmesh*) mesh, cshade->shader));
  }
}

void P3_mesh_get_data (P3_mesh* mesh, P3_chunk* chunk) {
  int i;
  /* option */
  i = mesh->option;
  if (i & P3_MESH_INITED) i -= P3_MESH_INITED;
  P3_chunk_save_int (chunk, i);
  /* xtra */
  if (mesh->option & P3_MESH_DISPLAY_LISTS) {
    P3_mesh_display_list_get_data (mesh, (P3_mesh_display_list*) mesh->xtra1, chunk);
  }
  if (mesh->option & P3_MESH_HAS_SPHERE) {
    P3_chunk_save (chunk, (GLfloat*) mesh->xtra2, 4 * sizeof (GLfloat));
  }
  if (mesh->option & P3_MESH_CELL_SHADING) {
    P3_mesh_cell_shading_get_data (mesh, (P3_mesh_cell_shading*) mesh->xtra1, chunk);
  }
  /* xmesh data */
  P3_xmesh_get_data ((P3_xmesh*) mesh, chunk);
  /* xtra */
  if (mesh->option & P3_MESH_TREE) {
    P3_xnode_get_data ((P3_xnode*) mesh->xtra1, (P3_xmesh*) mesh, chunk);
  }
}

static P3_mesh_display_list* P3_mesh_display_list_set_data (P3_mesh* mesh, P3_chunk* chunk) {
  P3_mesh_display_list* dlist;
  P3_display_list* dl;
  int nb;
  int i;
  int n;
  dlist = (P3_mesh_display_list*) malloc (sizeof (P3_mesh_display_list));
  dlist->nb_list_opaque     = P3_chunk_load_int (chunk);
  dlist->nb_list_alpha      = P3_chunk_load_int (chunk);
  nb = dlist->nb_list_opaque + dlist->nb_list_alpha;
  dlist->dlists = (P3_display_list*) malloc (nb * sizeof (P3_display_list));
  for (i = 0; i < nb; i++) {
    dl = dlist->dlists + i;
    dl->option   = P3_chunk_load_int (chunk);
    n = P3_chunk_load_int (chunk);
    if (n == -1) {
      dl->material = NULL;
    } else {
      dl->material = mesh->materials[n];
    }
  }
  return dlist;
}

static void P3_mesh_cell_shading_set_data (P3_mesh* mesh, P3_mesh_cell_shading* cshade, P3_chunk* chunk) {
  int n;
  P3_chunk_load (chunk, cshade->line_color, 4 * sizeof (GLfloat));
  cshade->line_width_factor = P3_chunk_load_float (chunk);
  n = P3_chunk_load_int (chunk);
  if (n == -1) {
    cshade->shader = NULL;
  } else {
    cshade->shader = mesh->materials[n];
  }
}

void P3_mesh_set_data (P3_mesh* mesh, P3_chunk* chunk) {
  /* option */
  mesh->option = P3_chunk_load_int (chunk);
  mesh->xtra1 = NULL;
  mesh->xtra2 = NULL;
  /* xtra */
  if (mesh->option & P3_MESH_DISPLAY_LISTS) mesh->xtra1 = P3_mesh_display_list_set_data (mesh, chunk);
  if (mesh->option & P3_MESH_FACE_LIST) {
    P3_mesh_face_list* flist = (P3_mesh_face_list*) malloc (sizeof (P3_mesh_face_list));
    P3_chunk_load (chunk, flist->sphere, 4 * sizeof (GLfloat));
    mesh->xtra2 = flist;
  } else if (mesh->option & P3_MESH_HAS_SPHERE) {
    GLfloat* sphere = (GLfloat*) malloc (4 * sizeof (GLfloat));
    P3_chunk_load (chunk, sphere, 4 * sizeof (GLfloat));
    mesh->xtra2 = sphere;
  }
  if (mesh->option & P3_MESH_CELL_SHADING) {
    P3_mesh_cell_shading* cshade = (P3_mesh_cell_shading*) malloc (sizeof (P3_mesh_cell_shading));
    P3_mesh_cell_shading_set_data (mesh, cshade, chunk);
    mesh->xtra1 = cshade;
  }
  /* xmesh data */
  P3_xmesh_set_data ((P3_xmesh*) mesh, chunk);
  /* xtra */
  if (mesh->option & P3_MESH_TREE) mesh->xtra1 = P3_xnode_set_data ((P3_xmesh*) mesh, chunk);
  /* xtra initialization */
  if (mesh->option & P3_MESH_FACE_LIST) P3_mesh_create_faces_list ((P3_mesh_face_list*) mesh->xtra2, mesh);
}

