/*
 * P3
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**********************************************
 * cmesh.c
 * Copyright (C) 2001-2002 Bertrand 'blam' LAMY
 **********************************************/

#include "p3_base.h"
#include "gladd.h"
#include "util.h"
#include "math3d.h"
#include "material.h"
#include "renderer.h"
#include "coordsys.h"
#include "frustum.h"
#include "light.h"
#include "world.h"
#include "raypick.h"
#include "cmesh.h"
#include "xmesh.h"

#ifdef COMPILE_FOR_PYTHON
#include "python/face_python.h"
#endif /* COMPILE_FOR_PYTHON */

extern int          engine_option;
extern P3_renderer* renderer;
extern GLfloat      black[4];
extern GLfloat      white[4];
extern GLfloat      transparency[4];


/*=======+
 | CMESH |
 +=======*/

P3_class P3_class_cmesh = { 
  P3_ID_CMESH,
  (batch_func)     P3_cmesh_batch,
  (render_func)    P3_cmesh_render,
  (raypick_func)   P3_cmesh_raypick,
  (raypick_b_func) P3_cmesh_raypick_b,
};

P3_cmesh* P3_cmesh_new (P3_cmesh* mesh) {
  if (mesh == NULL) {
    mesh = (P3_cmesh*) malloc (sizeof (P3_cmesh));
  }
  mesh->class = &P3_class_cmesh;
  mesh->nb_packs = 0;
  mesh->nb_packs_opaque = 0;
  mesh->nb_packs_alpha = 0;
  mesh->packs = NULL;
  mesh->nb_vcoords = 0;
  mesh->vcoords = NULL;
  mesh->voptions = NULL;
  mesh->nb_vnormals = 0;
  mesh->vnormals = NULL;
  mesh->nb_fnormals = 0;
  mesh->fnormals = NULL;
  mesh->nb_values = 0;
  mesh->values = NULL;
  mesh->nb_vertices = 0;
  mesh->vertices = NULL;
  mesh->dimension[0] = 0.0;
  mesh->dimension[1] = 0.0;
  mesh->dimension[2] = 0.0;
  mesh->dimension[3] = 0.0;
  mesh->dimension[4] = 0.0;
  mesh->dimension[5] = 0.0;
  mesh->sphere[0] = 0.0;
  mesh->sphere[1] = 0.0;
  mesh->sphere[2] = 0.0;
  mesh->sphere[3] = 0.0;
  mesh->tree = NULL;

  mesh->xm = NULL;

  return mesh;
}

static void P3_cnode_dealloc (P3_cnode* node) {
  int i;
  free (node->faces);
  for (i = 0; i < node->nb_child; i++) {
    P3_cnode_dealloc (node->child[i]);
  }
  free (node->child);
  free (node);
}

static void P3_cmesh_uninit (P3_cmesh* mesh) {
  int i;
  if (mesh->option & P3_CMESH_INITED) {
    for (i = 0; i < mesh->nb_packs; i++) {
      glDeleteLists ((mesh->packs + i)->data, 1);
    }
  }
}

void P3_cmesh_dealloc (P3_cmesh* mesh) {
  P3_cpack* pack;
  int i;
  int j;
  P3_cmesh_uninit (mesh);
  for (i = 0; i < mesh->nb_packs; i++) {
    pack = mesh->packs + i;
    for (j = 0; j < pack->nb_faces; j++) {
      free ((pack->faces + j)->vertices);
    }
    if (pack->material != NULL) { P3_material_decref (pack->material); }
  }
  free (mesh->packs);
  free (mesh->vcoords);
  free (mesh->voptions);
  free (mesh->vnormals);
  free (mesh->fnormals);
  free (mesh->values);
  free (mesh->vertices);
  if (mesh->tree != NULL) {
    P3_cnode_dealloc (mesh->tree);
  }
}


/*===========+
 | RENDERING |
 +===========*/

struct _p3_cmesh_chain {
  P3_cface* face;
  int next;
};

static void P3_cpack_face_render (P3_cpack* pack, P3_cface* face) {
  P3_cvertex* vertex;
  int i;
  if (!(pack->option & P3_R_SMOOTHLIT)) {
    glNormal3fv (face->normal);
  }
  for (i = 0; i < pack->vertices_per_face; i++) {
    vertex = face->vertices[i];
    if (pack->option & P3_R_COLORED) {
      glColor4fv (vertex->d_color);
    }
    if (pack->option & P3_R_EMISSIVE) {
      glMaterialfv (GL_FRONT_AND_BACK, GL_EMISSION, vertex->e_color);
    }
    if (pack->option & P3_R_TEXTURED) {
      glTexCoord2fv (vertex->texcoord);
    }
    glArrayElement (vertex->coord);
  }
}

static void P3_cpack_face_render_with_visibility (P3_cmesh* mesh, P3_cpack* pack, P3_cface* face) {
  P3_cvertex* vertex;
  int i;
  int n;
  if (!(face->option & P3_OBJECT_HIDDEN)) {
    if (face->option & P3_CFACE_HAS_INVISIBLE_VERTEX) {
      if (!(pack->option & P3_R_SMOOTHLIT)) {
        glNormal3fv (face->normal);
      }
      for (i = 0; i < pack->vertices_per_face; i++) {
        vertex = face->vertices[i];
        n = vertex - mesh->vertices;
        if (pack->option & P3_R_COLORED) {
          if (mesh->voptions[n] & P3_OBJECT_HIDDEN) {
            glColor4f (vertex->d_color[0], vertex->d_color[1], vertex->d_color[2], 0.0);
          } else {
            glColor4fv (vertex->d_color);
          }
        } else {
          if (mesh->voptions[n] & P3_OBJECT_HIDDEN) {
            glColor4fv (transparency);
          }
        }
        if (pack->option & P3_R_EMISSIVE) {
          glMaterialfv (GL_FRONT_AND_BACK, GL_EMISSION, vertex->e_color);
        }
        if (pack->option & P3_R_TEXTURED) {
          glTexCoord2fv (vertex->texcoord);
        }
        glArrayElement (vertex->coord);
        if (!(pack->option & P3_R_COLORED) && mesh->voptions[n] & P3_OBJECT_HIDDEN) {
          glColor4fv (white);
        }
      }
    } else {
      P3_cpack_face_render (pack, face);
    }
  }
}

static void P3_cmesh_init (P3_cmesh* mesh) {
  P3_cpack* pack;
  int i;
  int j;
  for (i = 0; i < mesh->nb_packs; i++) {
    pack = mesh->packs + i;
    if (pack->option & P3_R_SMOOTHLIT) {
      glEnableClientState (GL_NORMAL_ARRAY);
    } else {
      glDisableClientState (GL_NORMAL_ARRAY);
    }
    pack->data = glGenLists (1);
    glNewList (pack->data, GL_COMPILE);
    glStart (pack->vertices_per_face);
    for (j = 0; j < pack->nb_faces; j++) {
      P3_cpack_face_render (pack, pack->faces + j);
    }
    glEnd ();
    glEndList ();
  }
  mesh->option |= P3_CMESH_INITED;
}

static void P3_cnode_batch (P3_cnode* node, P3_frustum* frustum) {
  P3_cpack* pack;
  P3_cface* face;
  int old;
  int i;
  struct _p3_cmesh_chain* chain;
  P3_chunk* chunk;
  /* frustum test */
  if (P3_sphere_in_frustum (frustum, node->sphere) == P3_TRUE) {
    /* batch all faces */
    for (i = 0; i < node->nb_faces; i++) {
      face = node->faces[i];
      if (!(face->option & P3_OBJECT_HIDDEN)) {
        pack = face->pack;
        /* fill current data and register a new one */
        old = pack->data;
        if (pack->option & P3_R_ALPHA) {
          chunk = renderer->alpha;
        } else {
          chunk = renderer->opaque;
        }
        pack->data = P3_chunk_register (chunk, sizeof (struct _p3_cmesh_chain));
        chain = (struct _p3_cmesh_chain*) (chunk->content + old);
        chain->face = face;
        chain->next = pack->data;
      }
    }
    /* recurse */
    for (i = 0; i < node->nb_child; i++) {
      P3_cnode_batch (node->child[i], frustum);
    }
  }
}

void P3_cmesh_batch (P3_cmesh* mesh, P3_instance* inst) {

/*
  if (mesh->xm != NULL) {
    P3_xmesh_batch ((P3_xmesh*) mesh->xm, inst);
    return;
  }
*/

  if (mesh->tree == NULL) {
    if (mesh->option & P3_CMESH_ALWAYS_VISIBLE || 
        P3_sphere_in_frustum (P3_renderer_get_frustum ((P3_coordsys*) inst), mesh->sphere) == P3_TRUE) {
      if (mesh->nb_packs_opaque > 0) { P3_renderer_add       (mesh, inst); }
      if (mesh->nb_packs_alpha  > 0) { P3_renderer_add_alpha (mesh, inst); }
    }
  } else {
    int chunk;
    int i;
    /* batch myself */
    if (mesh->nb_packs_opaque > 0) { P3_renderer_add       (mesh, inst); }
    if (mesh->nb_packs_alpha  > 0) { P3_renderer_add_alpha (mesh, inst); }
    /* register chunk to stock visibility */
    chunk = P3_chunk_register (renderer->opaque, mesh->nb_packs_opaque * sizeof (struct _p3_cmesh_chain));
    for (i = 0; i < mesh->nb_packs_opaque; i++) {
      (mesh->packs + i)->data = chunk + i * sizeof (struct _p3_cmesh_chain);
    }
    chunk = P3_chunk_register (renderer->alpha, mesh->nb_packs_alpha * sizeof (struct _p3_cmesh_chain));
    for (i = 0; i < mesh->nb_packs_alpha; i++) {
      (mesh->packs + i + mesh->nb_packs_opaque)->data = chunk + i * sizeof (struct _p3_cmesh_chain);
    }
    /* traverse the sphere tree and compute visibility */
    P3_cnode_batch (mesh->tree, P3_renderer_get_frustum ((P3_coordsys*) inst));
    /* fix faces chain => must be terminated by a NULL */
    for (i = 0; i < mesh->nb_packs_opaque; i++) {
      *((P3_cface**) (renderer->opaque->content + (mesh->packs + i)->data)) = NULL;
    }
    for (i = 0; i < mesh->nb_packs_alpha; i++) {
      *((P3_cface**) (renderer->alpha->content + (mesh->packs + i + mesh->nb_packs_opaque)->data)) = NULL;
    }
  }
}

static void P3_cpack_render (P3_cpack* pack) {
  P3_enable_rendering_option (pack->option);
  P3_material_activate (pack->material);
  if (pack->option & P3_R_SMOOTHLIT) {
    glEnableClientState (GL_NORMAL_ARRAY);
  } else {
    glDisableClientState (GL_NORMAL_ARRAY);
  }
  glCallList (pack->data);
  P3_disable_rendering_option (pack->option);
}

static void P3_cpack_render_with_visibility (P3_cmesh* mesh, P3_cpack* pack) {
  int i;
  P3_enable_rendering_option (pack->option);
  P3_material_activate (pack->material);
  if (pack->option & P3_R_SMOOTHLIT) {
    glEnableClientState (GL_NORMAL_ARRAY);
  } else {
    glDisableClientState (GL_NORMAL_ARRAY);
  }
  glStart (pack->vertices_per_face);
  for (i = 0; i < pack->nb_faces; i++) {
    P3_cpack_face_render_with_visibility (mesh, pack, pack->faces + i);
  }
  glEnd ();
  P3_disable_rendering_option (pack->option);
}

static void P3_cpack_render_part (P3_cpack* pack, struct _p3_cmesh_chain* chain, P3_chunk* chunk) {
  P3_cface* face;
  if (chain->face != NULL) {
    P3_enable_rendering_option (pack->option);
    P3_material_activate (pack->material);
    if (pack->option & P3_R_SMOOTHLIT) {
      glEnableClientState (GL_NORMAL_ARRAY);
    } else {
      glDisableClientState (GL_NORMAL_ARRAY);
    }
    face = chain->face;
    glStart (pack->vertices_per_face);
    while (face != NULL) {
      P3_cpack_face_render (pack, face);
      chain = (struct _p3_cmesh_chain*) (chunk->content + chain->next);
      face = chain->face;
      chunk->nb += sizeof (struct _p3_cmesh_chain);
    }
    glEnd ();
    P3_disable_rendering_option (pack->option);
  }
  chunk->nb += sizeof (struct _p3_cmesh_chain);
}

static void P3_cpack_render_part_with_visibility (P3_cmesh* mesh, P3_cpack* pack, struct _p3_cmesh_chain* chain, P3_chunk* chunk) {
  P3_cface* face;
  if (chain->face != NULL) {
    P3_enable_rendering_option (pack->option);
    P3_material_activate (pack->material);
    if (pack->option & P3_R_SMOOTHLIT) {
      glEnableClientState (GL_NORMAL_ARRAY);
    } else {
      glDisableClientState (GL_NORMAL_ARRAY);
    }
    face = chain->face;
    glStart (pack->vertices_per_face);
    while (face != NULL) {
      P3_cpack_face_render_with_visibility (mesh, pack, face);
      chain = (struct _p3_cmesh_chain*) (chunk->content + chain->next);
      face = chain->face;
      chunk->nb += sizeof (struct _p3_cmesh_chain);
    }
    glEnd ();
    P3_disable_rendering_option (pack->option);
  }
  chunk->nb += sizeof (struct _p3_cmesh_chain);
}

void P3_cmesh_render (P3_cmesh* mesh, P3_instance* inst) {
  int i;
  glVertexPointer (3, GL_FLOAT, 0, mesh->vcoords);
  glNormalPointer (GL_FLOAT, 0, mesh->vnormals);
  glEnableClientState (GL_VERTEX_ARRAY);
  if (mesh->tree == NULL) {
    if (mesh->option & P3_CMESH_HAS_VISIBILITY) {
      if (mesh->option & P3_CMESH_HAS_VERTEX_VISIBILITY) { glEnable (GL_BLEND); }
      if (renderer->state == P3_RENDERER_STATE_OPAQUE) {
        for (i = 0; i < mesh->nb_packs_opaque; i++) {
          P3_cpack_render_with_visibility (mesh, mesh->packs + i);
        }
      } else {
        for (i = 0; i < mesh->nb_packs_alpha; i++) {
          P3_cpack_render_with_visibility (mesh, mesh->packs + i + mesh->nb_packs_opaque);
        }
      }
      if (mesh->option & P3_CMESH_HAS_VERTEX_VISIBILITY) { glDisable (GL_BLEND); }
    } else {
      if (!(mesh->option & P3_CMESH_INITED)) {
        P3_cmesh_init (mesh);
      }
      if (renderer->state == P3_RENDERER_STATE_OPAQUE) {
        for (i = 0; i < mesh->nb_packs_opaque; i++) {
          P3_cpack_render (mesh->packs + i);
        }
      } else {
        for (i = 0; i < mesh->nb_packs_alpha; i++) {
          P3_cpack_render (mesh->packs + i + mesh->nb_packs_opaque);
        }
      }
    }
  } else {
    struct _p3_cmesh_chain* ptr;
    if (mesh->option & P3_CMESH_HAS_VISIBILITY) {
      if (mesh->option & P3_CMESH_HAS_VERTEX_VISIBILITY) { glEnable (GL_BLEND); }
      if (renderer->state == P3_RENDERER_STATE_OPAQUE) {
        ptr = (struct _p3_cmesh_chain*) (renderer->opaque->content + renderer->opaque->nb);
        for (i = 0; i < mesh->nb_packs_opaque; i++) {
          P3_cpack_render_part_with_visibility (mesh, mesh->packs + i, ptr + i, renderer->opaque);
        }
      } else {
        ptr = (struct _p3_cmesh_chain*) (renderer->alpha->content + renderer->alpha->nb);
        for (i = 0; i < mesh->nb_packs_alpha; i++) {
          P3_cpack_render_part_with_visibility (mesh, mesh->packs + i + mesh->nb_packs_opaque, ptr + i, renderer->alpha);
        }
      }
      if (mesh->option & P3_CMESH_HAS_VERTEX_VISIBILITY) { glDisable (GL_BLEND); }
    } else {
      if (renderer->state == P3_RENDERER_STATE_OPAQUE) {
        ptr = (struct _p3_cmesh_chain*) (renderer->opaque->content + renderer->opaque->nb);
        for (i = 0; i < mesh->nb_packs_opaque; i++) {
          P3_cpack_render_part (mesh->packs + i, ptr + i, renderer->opaque);
        }
      } else {
        ptr = (struct _p3_cmesh_chain*) (renderer->alpha->content + renderer->alpha->nb);
        for (i = 0; i < mesh->nb_packs_alpha; i++) {
          P3_cpack_render_part (mesh->packs + i + mesh->nb_packs_opaque, ptr + i, renderer->alpha);
        }
      }
    }
  }
  glDisableClientState (GL_VERTEX_ARRAY);
  glDisableClientState (GL_NORMAL_ARRAY);
  glVertexPointer (3, GL_FLOAT, 0, NULL);
  glNormalPointer (GL_FLOAT, 0, NULL);
}


/*============+
 | VISIBILITY |
 +============*/

void P3_cmesh_set_visibility_all (P3_cmesh* mesh, int visibility) {
  P3_cpack* pack;
  P3_cface* face;
  int i;
  int j;
  if (visibility == P3_TRUE) {
    if (mesh->option & P3_CMESH_HAS_VISIBILITY) {
      mesh->option -= P3_CMESH_HAS_VISIBILITY;
    }
    if (mesh->option & P3_CMESH_HAS_VERTEX_VISIBILITY) {
      for (i = 0; i < mesh->nb_vcoords; i++) {
        if (mesh->voptions[i] & P3_OBJECT_HIDDEN) {
          mesh->voptions[i] -= P3_OBJECT_HIDDEN;
        }
      }
    }
  } else {
    mesh->option |= P3_CMESH_HAS_VISIBILITY;
    if (mesh->option & P3_CMESH_HAS_VERTEX_VISIBILITY) {
      for (i = 0; i < mesh->nb_vcoords; i++) {
        mesh->voptions[i] |= P3_OBJECT_HIDDEN;
      }
    }
  }
  for (i = 0; i < mesh->nb_packs; i++) {
    pack = mesh->packs + i;
    for (j = 0; j < pack->nb_faces; j++) {
      face = pack->faces + j;
      if (face->option & P3_CFACE_HAS_INVISIBLE_VERTEX) {
        face->option -= P3_CFACE_HAS_INVISIBLE_VERTEX;
      }
      if (visibility == P3_FALSE) {
        face->option |= P3_OBJECT_HIDDEN;
      } else {
        if (face->option & P3_OBJECT_HIDDEN) {
          face->option -= P3_OBJECT_HIDDEN;
        }
      }
    }
  }
}

static void P3_cface_set_visibility_in_sphere (P3_cmesh* mesh, P3_cpack* pack, P3_cface* face, int visibility, GLfloat sphere[4]) {
  int i;
  if (mesh->option & P3_CMESH_HAS_VERTEX_VISIBILITY) {
    P3_cvertex* vertex;
    int k;
    int n = 0;
    for (i = 0; i < pack->vertices_per_face; i++) {
      vertex = face->vertices[i];
      k = vertex - mesh->vertices;
      if (P3_point_is_in_sphere (sphere, mesh->vcoords + vertex->coord * 3) == P3_TRUE) {
        if (visibility == P3_TRUE) {
          if (mesh->voptions[k] & P3_OBJECT_HIDDEN) {
            mesh->voptions[k] -= P3_OBJECT_HIDDEN;
          }
        } else {
          mesh->voptions[k] |= P3_OBJECT_HIDDEN;
        }
      }
      if (mesh->voptions[k] & P3_OBJECT_HIDDEN) {
        n--;
      } else {
        n++;
      }
    }
    if (n == pack->vertices_per_face) {
      if (face->option & P3_OBJECT_HIDDEN) {
        face->option -= P3_OBJECT_HIDDEN;
      }
      if (face->option & P3_CFACE_HAS_INVISIBLE_VERTEX) {
        face->option -= P3_CFACE_HAS_INVISIBLE_VERTEX;
      }
    } else if (n == - pack->vertices_per_face) {
      face->option |= P3_OBJECT_HIDDEN;
      if (face->option & P3_CFACE_HAS_INVISIBLE_VERTEX) {
        face->option -= P3_CFACE_HAS_INVISIBLE_VERTEX;
      }
    } else {
      if (face->option & P3_OBJECT_HIDDEN) {
        face->option -= P3_OBJECT_HIDDEN;
      }
      face->option |= P3_CFACE_HAS_INVISIBLE_VERTEX;
    }
  } else {
    for (i = 0; i < pack->vertices_per_face; i++) {
      if (P3_point_is_in_sphere (sphere, mesh->vcoords + face->vertices[i]->coord * 3) == P3_TRUE) {
        /* 1 point is inside -> full face visibility is set */
        if (visibility == P3_TRUE) {
          if (face->option & P3_OBJECT_HIDDEN) {
            face->option -= P3_OBJECT_HIDDEN;
          }
        } else {
          face->option |= P3_OBJECT_HIDDEN;
        }
        break;
      }
    }
  }
}

static void P3_cnode_set_visibility_in_sphere (P3_cmesh* mesh, P3_cnode* node, int visibility, GLfloat sphere[4]) {
  P3_cface* face;
  int i;
  if (P3_point_distance_to (sphere, node->sphere) - sphere[3] - node->sphere[3] < 0.0) {
    for (i = 0; i < node->nb_faces; i++) {
      face = node->faces[i];
      P3_cface_set_visibility_in_sphere (mesh, face->pack, face, visibility, sphere);
    }
    for (i = 0; i < node->nb_child; i++) {
      P3_cnode_set_visibility_in_sphere (mesh, node->child[i], visibility, sphere);
    }
  }
}

void P3_cmesh_set_visibility_in_sphere (P3_cmesh* mesh, int visibility, GLfloat sphere[4]) {
  mesh->option |= P3_CMESH_HAS_VISIBILITY;
  if (mesh->tree == NULL) {
    P3_cpack* pack;
    int i;
    int j;
    for (i = 0; i < mesh->nb_packs; i++) {
      pack = mesh->packs + i;
      for (j = 0; j < pack->nb_faces; j++) {
        P3_cface_set_visibility_in_sphere (mesh, pack, pack->faces + j, visibility, sphere);
      }
    }
  } else {
    P3_cnode_set_visibility_in_sphere (mesh, mesh->tree, visibility, sphere);
  }
}

static void P3_cface_set_visibility_in_cylinder (P3_cmesh* mesh, P3_cpack* pack, P3_cface* face, int visibility, GLfloat cylinder[3]) {
  GLfloat x;
  GLfloat z;
  GLfloat* ptr;
  int i;
  if (mesh->option & P3_CMESH_HAS_VERTEX_VISIBILITY) {
    P3_cvertex* vertex;
    int k;
    int n = 0;
    for (i = 0; i < pack->vertices_per_face; i++) {
      vertex = face->vertices[i];
      k = vertex - mesh->vertices;
      ptr = mesh->vcoords + vertex->coord * 3;
      x = ptr[0] - cylinder[0];
      z = ptr[2] - cylinder[1];
      if (sqrt (x * x + z * z) <= cylinder[2]) {
        if (visibility == P3_TRUE) {
          if (mesh->voptions[k] & P3_OBJECT_HIDDEN) {
            mesh->voptions[k] -= P3_OBJECT_HIDDEN;
          }
        } else {
          mesh->voptions[k] |= P3_OBJECT_HIDDEN;
        }
      }
      if (mesh->voptions[k] & P3_OBJECT_HIDDEN) {
        n--;
      } else {
        n++;
      }
    }
    if (n == pack->vertices_per_face) {
      if (face->option & P3_OBJECT_HIDDEN) {
        face->option -= P3_OBJECT_HIDDEN;
      }
      if (face->option & P3_CFACE_HAS_INVISIBLE_VERTEX) {
        face->option -= P3_CFACE_HAS_INVISIBLE_VERTEX;
      }
    } else if (n == - pack->vertices_per_face) {
      face->option |= P3_OBJECT_HIDDEN;
      if (face->option & P3_CFACE_HAS_INVISIBLE_VERTEX) {
        face->option -= P3_CFACE_HAS_INVISIBLE_VERTEX;
      }
    } else {
      if (face->option & P3_OBJECT_HIDDEN) {
        face->option -= P3_OBJECT_HIDDEN;
      }
      face->option |= P3_CFACE_HAS_INVISIBLE_VERTEX;
    }
  } else {
    for (i = 0; i < pack->vertices_per_face; i++) {
      ptr = mesh->vcoords + face->vertices[i]->coord * 3;
      x = ptr[0] - cylinder[0];
      z = ptr[2] - cylinder[1];
      if (sqrt (x * x + z * z) <= cylinder[2]) {
        /* 1 point is inside -> full face visibility is set */
        if (visibility == P3_TRUE) {
          if (face->option & P3_OBJECT_HIDDEN) {
            face->option -= P3_OBJECT_HIDDEN;
          }
        } else {
          face->option |= P3_OBJECT_HIDDEN;
        }
        break;
      }
    }
  }
}

static void P3_cnode_set_visibility_in_cylinder (P3_cmesh* mesh, P3_cnode* node, int visibility, GLfloat cylinder[3]) {
  P3_cface* face;
  GLfloat x;
  GLfloat z;
  int i;
  x = node->sphere[0] - cylinder[0];
  z = node->sphere[2] - cylinder[1];
  if (sqrt (x * x + z * z) - cylinder[2] - node->sphere[3] < 0.0) {
    for (i = 0; i < node->nb_faces; i++) {
      face = node->faces[i];
      P3_cface_set_visibility_in_cylinder (mesh, face->pack, face, visibility, cylinder);
    }
    for (i = 0; i < node->nb_child; i++) {
      P3_cnode_set_visibility_in_cylinder (mesh, node->child[i], visibility, cylinder);
    }
  }
}

void P3_cmesh_set_visibility_in_cylinder (P3_cmesh* mesh, int visibility, GLfloat cylinder[3]) {
  mesh->option |= P3_CMESH_HAS_VISIBILITY;
  if (mesh->tree == NULL) {
    P3_cpack* pack;
    int i;
    int j;
    for (i = 0; i < mesh->nb_packs; i++) {
      pack = mesh->packs + i;
      for (j = 0; j < pack->nb_faces; j++) {
        P3_cface_set_visibility_in_cylinder (mesh, pack, pack->faces + j, visibility, cylinder);
      }
    }
  } else {
    P3_cnode_set_visibility_in_cylinder (mesh, mesh->tree, visibility, cylinder);
  }
}


/*===============+
 | TREE CREATION |
 +===============*/

static void P3_cmesh_face_get_sphere (P3_cmesh* mesh, P3_cface* face, GLfloat* s) {
  GLfloat p[3 * face->pack->vertices_per_face];
  int i;
  for (i = 0; i < face->pack->vertices_per_face; i++) {
    memcpy (p + i * 3, mesh->vcoords + face->vertices[i]->coord * 3, 3 * sizeof (GLfloat));
  }
  P3_sphere_from_points (s, p, face->pack->vertices_per_face);
}

static P3_cnode* P3_cnode_new (P3_cface* face, GLfloat sphere[4]) {
  P3_cnode* node;
  node = (P3_cnode*) malloc (sizeof (P3_cnode));
  node->nb_faces = 1;
  node->faces = (P3_cface**) malloc (sizeof (P3_cface*));
  node->faces[0] = face;
  node->nb_child = 0;
  node->child = NULL;
  memcpy (node->sphere, sphere, 4 * sizeof (GLfloat));
  return node;
}

static void P3_cnode_add_node (P3_cnode* node, P3_cnode* add) {
  node->child = (P3_cnode**) realloc (node->child, (node->nb_child + 1) * sizeof (P3_cnode*));
  node->child[node->nb_child] = add;
  (node->nb_child)++;
}

static void P3_cnode_register_node (P3_cnode* node, P3_cnode* add) {
  GLfloat d;
  int i;
  int added = P3_FALSE;
  /* test if ancient nodes can be contained into the added node */
  i = 0;
  while (i < node->nb_child) {
    if (node->child[i] == NULL) {
      if (added == P3_FALSE) {
        node->child[i] = add;
        added = P3_TRUE;
      }
      i++;
    } else {
      d = P3_point_distance_to (add->sphere, node->child[i]->sphere);
      if (d + node->child[i]->sphere[3] <= add->sphere[3]) {
        /* add child i into added node */
        P3_cnode_add_node (add, node->child[i]);
        if (added == P3_FALSE) {
          node->child[i] = add;
          added = P3_TRUE;
          i++;
        } else {
          (node->nb_child)--;
          node->child[i] = node->child[node->nb_child];
          node->child[node->nb_child] = NULL;
        }
      } else {
        i++;
      }
    }
  }
  if (added == P3_FALSE) { P3_cnode_add_node (node, add); }
}

static void P3_cnode_added (P3_cnode* node, P3_cnode* new) {
  GLfloat d;
  int i;
  /* test if ancient nodes can be contained into the new one */
  i = 0;
  while (i < node->nb_child) {
    if (node->child[i] != NULL && new != node->child[i]) {
      d = P3_point_distance_to (new->sphere, node->child[i]->sphere);
      if (d + node->child[i]->sphere[3] <= new->sphere[3]) {
        /* add child i into new node */
        P3_cnode_add_node (new, node->child[i]);
        (node->nb_child)--;
        node->child[i] = node->child[node->nb_child];
        node->child[node->nb_child] = NULL;
      } else {
        i++;
      }
    } else {
      i++;
    }
  }
}

static int P3_cnode_gather (P3_cnode* node, int mode, GLfloat param) {
  int best1 = -1;
  int best2 = -1;
  P3_cnode* n;
  GLfloat best_sphere[4];
  GLfloat sphere[4];
  int i; int j;
  /* return P3_FALSE if no more gather are possible */
  /* try different technics... */
  if (mode == 0) {
    /* take the smallest sphere */
    GLfloat min_radius = 100000.0f;
    GLfloat radius = param * node->sphere[3];
    n = NULL;
    for (i = 0; i < node->nb_child; i++) {
      if (n == NULL || node->child[i]->sphere[3] < min_radius) { 
        best1 = i;
        n = node->child[i];
        min_radius = n->sphere[3]; 
      }
    }
    if (min_radius >= radius) {
      return P3_FALSE;
    } else {
      /* find the 1rst sphere to gather with that produce a sphere with a radius <= radius */
      for (i = 0; i < node->nb_child; i++) {
        if (i != best1) { 
          P3_sphere_from_2_spheres (best_sphere, n->sphere, node->child[i]->sphere);
          if (best_sphere[3] <= radius) {
            best2 = i;
            break;
          }
        }
      }
      return P3_FALSE;
    }
  } else {
    /* compute the best tree possible but very slow */
    /* find the 2 children that produce the smallest sphere */
    for (i = 0; i < node->nb_child; i++) {
      n = node->child[i];
      if (n != NULL) {
        for (j = i + 1; j < node->nb_child; j++) {
          if (node->child[j] != NULL) {
            P3_sphere_from_2_spheres (sphere, n->sphere, node->child[j]->sphere);
            if (best1 < 0 || sphere[3] < best_sphere[3]) {
              memcpy (best_sphere, sphere, 4 * sizeof (GLfloat));
              best1 = i;
              best2 = j;
            }
          }
        }
      }
    }
  }
  if (best_sphere[3] >= node->sphere[3]) { return P3_FALSE; }
  /* gather best1 and best2 */
  n = (P3_cnode*) malloc (sizeof (P3_cnode));
  n->nb_faces = 0;
  n->faces = NULL;
  n->nb_child = 2;
  n->child = (P3_cnode**) malloc (2 * sizeof (P3_cnode*));
  n->child[0] = node->child[best1];
  n->child[1] = node->child[best2];
  memcpy (n->sphere, best_sphere, 4 * sizeof (GLfloat));
  (node->nb_child)--;
  node->child[best1] = n;
  node->child[best2] = node->child[node->nb_child];
  P3_cnode_added (node, n);
  return P3_TRUE;
}

static void P3_cnode_add_face (P3_cnode* node, P3_cface* face, GLfloat sphere[4]) {
  P3_cnode* new;
  /* create a new node for face and add the new node to our node */
  new = P3_cnode_new (face, sphere);
  P3_cnode_register_node (node, new);
  node->child = (P3_cnode**) realloc (node->child, node->nb_child * sizeof (P3_cnode*));
}

static void P3_cnode_join (P3_cnode* n1, P3_cnode* n2) {
  int i;
  n1->faces = (P3_cface**) realloc (n1->faces, (n1->nb_faces + n2->nb_faces) * sizeof (P3_cface*));
  for (i = 0; i < n2->nb_faces; i++) {
    n1->faces[n1->nb_faces + i] = n2->faces[i];
  }
  n1->nb_faces += n2->nb_faces;
  n1->child = (P3_cnode**) realloc (n1->child, (n1->nb_child + n2->nb_child) * sizeof (P3_cnode*));
  for (i = 0; i < n2->nb_child; i++) {
    n1->child[n1->nb_child + i] = n2->child[i];
  }
  n1->nb_child += n2->nb_child;
}

static void P3_cnode_collapse_with_child (P3_cnode* node, GLfloat collapse) {
  int i;
  for (i = 0; i < node->nb_child; i++) {
    if (node->child[i]->sphere[3] > collapse * node->sphere[3]) {
      P3_cnode_join (node, node->child[i]);
      (node->nb_child)--;
      node->child[i] = node->child[node->nb_child];
    }
  }
}

void P3_cnode_optimize (P3_cnode* node, GLfloat collapse, int mode, GLfloat param) {
  int i;
  while (node->nb_child > 2) {
    /* gather some children */
    if (P3_cnode_gather (node, mode, param) == P3_FALSE) { break; }
  }
  P3_cnode_collapse_with_child (node, collapse);
  node->child = (P3_cnode**) realloc (node->child, node->nb_child * sizeof (P3_cnode*));
  for (i = 0; i < node->nb_child; i++) {
    P3_cnode_optimize (node->child[i], collapse, mode, param);
  }
}

static void P3_cnode_register_inside_face (P3_cnode* node, P3_cface* face, GLfloat sphere[4]) {
  /* recurse to children */
  GLfloat d;
  int i;
  for (i = 0; i < node->nb_child; i++) {
    d = P3_point_distance_to (node->child[i]->sphere, sphere);
    if (d + sphere[3] <= node->child[i]->sphere[3]) {
      /* face is inside that child */
      P3_cnode_register_inside_face (node->child[i], face, sphere);
      return;
    }
  }
  /* no child found => face must be added to node */
  P3_cnode_add_face (node, face, sphere);
}

static P3_cnode* P3_cnode_register_face (P3_cnode* node, P3_cnode* parent, P3_cface* face, GLfloat sphere[4]) {
  GLfloat d;
  d = P3_point_distance_to (node->sphere, sphere);
  if (d + sphere[3] <= node->sphere[3]) {
    /* face is inside node */
    P3_cnode_register_inside_face (node, face, sphere);
  } else if (d + node->sphere[3] <= sphere[3]) {
    /* node is inside face */
    P3_cnode* n;
    n = (P3_cnode*) malloc (sizeof (P3_cnode));
    n->nb_faces = 1;
    n->faces = (P3_cface**) malloc (sizeof (P3_cface*));
    n->faces[0] = face;
    n->nb_child = 1;
    n->child = (P3_cnode**) malloc (sizeof (P3_cnode*));
    n->child[0] = node;
    memcpy (n->sphere, sphere, 4 * sizeof (GLfloat));
    return n;
  } else {
    if (parent == NULL) {
      /* create a new node with no face */
      P3_cnode* n;
      n = (P3_cnode*) malloc (sizeof (P3_cnode));
      n->nb_faces = 0;
      n->faces = NULL;
      n->nb_child = 2;
      n->child = (P3_cnode**) malloc (2 * sizeof (P3_cnode*));
      n->child[0] = node;
      n->child[1] = P3_cnode_new (face, sphere);
      P3_sphere_from_2_spheres (n->sphere, node->sphere, sphere);
      return n;
    } else {
      P3_cnode_add_face (parent, face, sphere);
    }
  }
  return node;
}

int P3_cnode_get_nb_level (P3_cnode* node) {
  int i; int nb = 0; int n;
  for (i = 0; i < node->nb_child; i++) {
    n = P3_cnode_get_nb_level (node->child[i]);
    if (n > nb) { nb = n; }
  }
  return nb + 1;
}

void P3_cmesh_build_tree (P3_cmesh* mesh) {
  GLfloat sphere[4];
  P3_cpack* pack;
  P3_cface* face;
  int i;
  int j;
  int nb = 0;
  P3_cmesh_uninit (mesh);
  if (mesh->tree != NULL) {
    P3_cnode_dealloc (mesh->tree);
  }

printf ("Building tree...\n");

  for (i = 0; i < mesh->nb_packs; i++) {
    pack = mesh->packs + i;
    for (j = 0; j < pack->nb_faces; j++) {
      face = pack->faces + j;
      P3_cmesh_face_get_sphere (mesh, face, sphere);
      if (mesh->tree == NULL) {
        mesh->tree = P3_cnode_new (face, sphere);
      } else {
        mesh->tree = P3_cnode_register_face (mesh->tree, NULL, face, sphere);
      }
      nb++;
    }
  }

printf ("  %i faces\n", nb);
printf ("  %i levels\n", P3_cnode_get_nb_level (mesh->tree));
printf ("  [DONE]\n");

}

void P3_cmesh_optimize_tree (P3_cmesh* mesh, GLfloat collapse, int mode, GLfloat param) {
  if (mesh->tree != NULL) {

printf ("Optimizing tree...\n");

    P3_cnode_optimize (mesh->tree, collapse, mode, param);

printf ("  %i levels\n", P3_cnode_get_nb_level (mesh->tree));
printf ("  [DONE]\n");

  }
}


/*============+
 | RAYPICKING |
 +============*/

void P3_cmesh_face_raypick (P3_cmesh* mesh, P3_cface* face, GLfloat* raydata, 
                            int option, GLfloat* result, GLfloat* norm, void** rc, P3_raypickable* parent) {
  GLfloat z;
  int r;
  P3_cpack* pack = face->pack;
  if (face->option & P3_OBJECT_NON_SOLID) { return; }
//  if (pack->option & P3_OBJECT_NON_SOLID) { return; }
  if (pack->option & P3_R_FRONT_AND_BACK && option & P3_RAYPICK_CULL_FACE) {
    option -= P3_RAYPICK_CULL_FACE;
  }
  if (pack->vertices_per_face == 3) {
    /* triangle */
    r = P3_triangle_raypick (raydata, 
                             mesh->vcoords + face->vertices[0]->coord * 3,
                             mesh->vcoords + face->vertices[1]->coord * 3,
                             mesh->vcoords + face->vertices[2]->coord * 3,
                             face->normal, option, &z);
  } else if (pack->vertices_per_face == 4) {
    /* quad */
    r = P3_quad_raypick (raydata, 
                         mesh->vcoords + face->vertices[0]->coord * 3,
                         mesh->vcoords + face->vertices[1]->coord * 3,
                         mesh->vcoords + face->vertices[2]->coord * 3,
                         mesh->vcoords + face->vertices[3]->coord * 3,
                         face->normal, option, &z);
  } else if (pack->vertices_per_face > 4) {
    /* polygons. TO DO ? */
    r = P3_FALSE;
  } else {
    /* line or point: do nothing */
    r = P3_FALSE;
  }
  if (r != P3_FALSE && (*rc == NULL || fabs (z) < fabs (*result))) {
    *result = z;
    *rc = parent;
    if (r == P3_RAYPICK_DIRECT) {
      memcpy (norm, face->normal, 3 * sizeof (GLfloat));
    } else if (r == P3_RAYPICK_INDIRECT) {
      if (pack->option & P3_R_FRONT_AND_BACK) {
        norm[0] = - face->normal[0];
        norm[1] = - face->normal[1];
        norm[2] = - face->normal[2];
      } else {
        memcpy (norm, face->normal, 3 * sizeof (GLfloat));
      }
    }
  }
}

int P3_cmesh_face_raypick_b (P3_cmesh* mesh, P3_cface* face, GLfloat* raydata, int option) {
  GLfloat z;
  P3_cpack* pack = face->pack;
  if (face->option & P3_OBJECT_NON_SOLID) { return P3_FALSE; }
//  if (pack->option & P3_OBJECT_NON_SOLID) { return P3_FALSE; }
  if (pack->option & P3_R_FRONT_AND_BACK && option & P3_RAYPICK_CULL_FACE) {
    option -= P3_RAYPICK_CULL_FACE;
  }
  if (pack->vertices_per_face == 3) {
    /* triangle */
    if (P3_triangle_raypick (raydata, 
                             mesh->vcoords + face->vertices[0]->coord * 3,
                             mesh->vcoords + face->vertices[1]->coord * 3,
                             mesh->vcoords + face->vertices[2]->coord * 3,
                             face->normal, option, &z) != P3_FALSE) { return P3_TRUE; }
  } else if (pack->vertices_per_face == 4) {
    /* quad */
    if (P3_quad_raypick (raydata, 
                         mesh->vcoords + face->vertices[0]->coord * 3,
                         mesh->vcoords + face->vertices[1]->coord * 3,
                         mesh->vcoords + face->vertices[2]->coord * 3,
                         mesh->vcoords + face->vertices[3]->coord * 3,
                         face->normal, option, &z) != P3_FALSE) { return P3_TRUE; }
  } else if (pack->vertices_per_face > 4) {
    /* polygons. TO DO ? */
  } else {
    /* line or point: do nothing */
    return P3_FALSE;
  }
  return P3_FALSE;
}

static void P3_cmesh_node_raypick (P3_cmesh* mesh, P3_cnode* node, GLfloat* raydata, int option, 
                                   GLfloat* result, GLfloat* norm, void** rc, P3_raypickable* parent) {
  P3_cface* f;
  int i;
  if (P3_sphere_raypick (raydata, node->sphere) == P3_TRUE) {
    /* raypick on faces */
    for (i = 0; i < node->nb_faces; i++) {
      f = node->faces[i];
//      if (!(f->pack->option & P3_OBJECT_NON_SOLID) && !(f->option & P3_OBJECT_NON_SOLID)) {
      P3_cmesh_face_raypick (mesh, f, raydata, option, result, norm, rc, parent);
//      }
    }
    /* recurse to children */
    for (i = 0; i < node->nb_child; i++) {
      P3_cmesh_node_raypick (mesh, node->child[i], raydata, option, result, norm, rc, parent);
    }
  }
}

static int P3_cmesh_node_raypick_b (P3_cmesh* mesh, P3_cnode* node, GLfloat* raydata, int option) {
  P3_cface* f;
  int i;
  if (P3_sphere_raypick (raydata, node->sphere) == P3_TRUE) {
    /* raypick on faces */
    for (i = 0; i < node->nb_faces; i++) {
      f = node->faces[i];
//      if (!(f->pack & P3_OBJECT_NON_SOLID) && !(f->option & P3_OBJECT_NON_SOLID)) {
      if (P3_cmesh_face_raypick_b (mesh, f, raydata, option) == P3_TRUE) { return P3_TRUE; }
//      }
    }
    /* recurse to children */
    for (i = 0; i < node->nb_child; i++) {
      if (P3_cmesh_node_raypick_b (mesh, node->child[i], raydata, option) == P3_TRUE) { return P3_TRUE; }
    }
  }
  return P3_FALSE;
}

void P3_cmesh_raypick (P3_cmesh* mesh, GLfloat* rdata, int option, 
                       GLfloat* result, GLfloat* norm, void** rc, P3_raypickable* parent) {
  GLfloat* raydata;
  raydata = P3_raypickable_get_raypick_data (parent, rdata);
  if (mesh->tree == NULL) {
    P3_cpack* pack;
    int i;
    int j;
    if (mesh->option & P3_CMESH_ALWAYS_VISIBLE || P3_sphere_raypick (raydata, mesh->sphere) == P3_TRUE) {
      for (i = 0; i < mesh->nb_packs; i++) {
        pack = mesh->packs + i;
//        if (!(pack->option & P3_OBJECT_NON_SOLID)) {
        for (j = 0; j < pack->nb_faces; j++) {
//      if (!((pack->faces + j)->option & P3_OBJECT_NON_SOLID)) {
          P3_cmesh_face_raypick (mesh, pack->faces + j, raydata, option, result, norm, rc, parent);
//      }
        }
//        }
      }
    }
  } else {
    P3_cmesh_node_raypick (mesh, mesh->tree, raydata, option, result, norm, rc, parent);
  }
}

int P3_cmesh_raypick_b (P3_cmesh* mesh, GLfloat* rdata, int option, P3_raypickable* parent) {
  GLfloat* raydata;
  raydata = P3_raypickable_get_raypick_data (parent, rdata);
  if (mesh->tree == NULL) {
    P3_cpack* pack;
    int i;
    int j;
    if (mesh->option & P3_CMESH_ALWAYS_VISIBLE || P3_sphere_raypick (raydata, mesh->sphere) == P3_TRUE) {
      for (i = 0; i < mesh->nb_packs; i++) {
        pack = mesh->packs + i;
//        if (!(pack->option & P3_OBJECT_NON_SOLID)) {
        for (j = 0; j < pack->nb_faces; j++) {
//          if (!((pack->faces + j)->option & P3_OBJECT_NON_SOLID)) {
          if (P3_cmesh_face_raypick_b (mesh, pack->faces + j, raydata, option) == P3_TRUE) {
            return P3_TRUE;
          }
//          }
        }
//      }
      }
    }
    return P3_FALSE;
  } else {
    return P3_cmesh_node_raypick_b (mesh, mesh->tree, raydata, option);
  }
}


/*========+
 | SAVING |
 +========*/

void P3_cmesh_node_get_data (P3_cmesh* mesh, P3_cnode* node, P3_chunk* chunk) {
  int i;
  P3_chunk_add_int (chunk, node->nb_faces);
  P3_chunk_add_int (chunk, node->nb_child);
  P3_chunk_add (chunk, node->sphere, 4 * sizeof (GLfloat));
  for (i = 0; i < node->nb_faces; i++) {
    P3_chunk_add_int (chunk, node->faces[i]->pack - mesh->packs);
    P3_chunk_add_int (chunk, node->faces[i] - node->faces[i]->pack->faces);
  }
  for (i = 0; i < node->nb_child; i++) {
    P3_cmesh_node_get_data (mesh, node->child[i], chunk);
  }
}

void P3_cmesh_get_data (P3_cmesh* mesh, P3_chunk* chunk) {
  P3_cvertex* v;
  P3_cface* f;
  P3_cpack* p;
  int i;
  int j;
  int k;
  char c;
  /* data */
  i = mesh->option;
  if (i & P3_CMESH_INITED) { i -= P3_CMESH_INITED; }
  P3_chunk_add_int (chunk, i);
  P3_chunk_add_int (chunk, mesh->nb_vcoords);
  P3_chunk_add_int (chunk, mesh->nb_vnormals);
  P3_chunk_add_int (chunk, mesh->nb_fnormals);
  P3_chunk_add_int (chunk, mesh->nb_values);
  P3_chunk_add_int (chunk, mesh->nb_vertices);
  P3_chunk_add_int (chunk, mesh->nb_packs);
  P3_chunk_add_int (chunk, mesh->nb_packs_opaque);
  P3_chunk_add_int (chunk, mesh->nb_packs_alpha);
  P3_chunk_add (chunk, mesh->vcoords,  mesh->nb_vcoords  * 3 * sizeof (GLfloat));
  P3_chunk_add (chunk, mesh->vnormals, mesh->nb_vnormals * 3 * sizeof (GLfloat));
  P3_chunk_add (chunk, mesh->fnormals, mesh->nb_fnormals * 3 * sizeof (GLfloat));
  P3_chunk_add (chunk, mesh->values,   mesh->nb_values   * sizeof (GLfloat));
  P3_chunk_add (chunk, mesh->dimension, 6 * sizeof (GLfloat));
  P3_chunk_add (chunk, mesh->sphere,    4 * sizeof (GLfloat));
  /* vertices */
  for (i = 0; i < mesh->nb_vertices; i++) {
    v = mesh->vertices + i;
    P3_chunk_add_int (chunk, v->coord);
    if (v->e_color == NULL) {
      P3_chunk_add_int (chunk, -1);
    } else {
      P3_chunk_add_int (chunk, v->e_color - mesh->values);
    }
    if (v->d_color == NULL) {
      P3_chunk_add_int (chunk, -1);
    } else {
      P3_chunk_add_int (chunk, v->d_color - mesh->values);
    }
    if (v->texcoord == NULL) {
      P3_chunk_add_int (chunk, -1);
    } else {
      P3_chunk_add_int (chunk, v->texcoord - mesh->values);
    }
  }
  /* packs */
  for (i = 0; i < mesh->nb_packs; i++) {
    p = mesh->packs + i;
    k = p->option;
    P3_chunk_add_int (chunk, k);
    P3_chunk_add_int (chunk, p->vertices_per_face);
    P3_chunk_add_int (chunk, p->nb_faces);
    /* faces */
    for (j = 0; j < p->nb_faces; j++) {
      f = p->faces + j;
      P3_chunk_add_int (chunk, f->normal - mesh->fnormals);
      c = f->option;
// TO DO
      if (c & P3_OBJECT_HIDDEN) { c -= P3_OBJECT_HIDDEN; }
      if (c & P3_CFACE_HAS_INVISIBLE_VERTEX) { c -= P3_CFACE_HAS_INVISIBLE_VERTEX; }
      P3_chunk_add_char (chunk, c);
      /* vertices */
      for (k = 0; k < p->vertices_per_face; k++) {
        P3_chunk_add_int (chunk, f->vertices[k] - mesh->vertices);
      }
    }
  }
  /* nodes */
  if (mesh->tree == NULL) {
    P3_chunk_add_char (chunk, '\0');
  } else {
    P3_chunk_add_char (chunk, '\1');
    P3_cmesh_node_get_data (mesh, mesh->tree, chunk);
  }
}

P3_cnode* P3_cmesh_node_set_data (P3_cmesh* mesh, P3_chunk* chunk) {
  P3_cnode* node;
  P3_cpack* p;
  int i;
  node = (P3_cnode*) malloc (sizeof (P3_cnode));
  node->nb_faces = P3_chunk_get_int (chunk);
  node->nb_child = P3_chunk_get_int (chunk);
  node->faces = (P3_cface**) malloc (node->nb_faces * sizeof (P3_cface*));
  node->child = (P3_cnode**) malloc (node->nb_child * sizeof (P3_cnode*));
  P3_chunk_get (chunk, node->sphere, 4 * sizeof (GLfloat));
  for (i = 0; i < node->nb_faces; i++) {
    p = mesh->packs + P3_chunk_get_int (chunk);
    node->faces[i] = p->faces + P3_chunk_get_int (chunk);
  }
  for (i = 0; i < node->nb_child; i++) {
    node->child[i] = P3_cmesh_node_set_data (mesh, chunk);
  }
  return node;
}

void P3_cmesh_set_data (P3_cmesh* mesh, P3_chunk* chunk) {
  P3_cvertex* v;
  P3_cface* f;
  P3_cpack* p;
  int i;
  int j;
  int k;
  mesh->class = &P3_class_cmesh;
  /* data */
  mesh->option      = P3_chunk_get_int (chunk);
  mesh->nb_vcoords  = P3_chunk_get_int (chunk);
  mesh->nb_vnormals = P3_chunk_get_int (chunk);
  mesh->nb_fnormals = P3_chunk_get_int (chunk);
  mesh->nb_values   = P3_chunk_get_int (chunk);
  mesh->nb_vertices = P3_chunk_get_int (chunk);
  mesh->nb_packs    = P3_chunk_get_int (chunk);
  mesh->nb_packs_opaque = P3_chunk_get_int (chunk);
  mesh->nb_packs_alpha  = P3_chunk_get_int (chunk);
  mesh->vcoords  = (GLfloat*) malloc (mesh->nb_vcoords  * 3 * sizeof (GLfloat));
  mesh->vnormals = (GLfloat*) malloc (mesh->nb_vnormals * 3 * sizeof (GLfloat));
  mesh->fnormals = (GLfloat*) malloc (mesh->nb_fnormals * 3 * sizeof (GLfloat));
  mesh->values   = (GLfloat*) malloc (mesh->nb_values   * sizeof (GLfloat));
  mesh->vertices = (P3_cvertex*) malloc (mesh->nb_vertices * sizeof (P3_cvertex));
  mesh->packs    = (P3_cpack*)   malloc (mesh->nb_packs    * sizeof (P3_cpack));
  P3_chunk_get (chunk, mesh->vcoords,  mesh->nb_vcoords  * 3 * sizeof (GLfloat));
  P3_chunk_get (chunk, mesh->vnormals, mesh->nb_vnormals * 3 * sizeof (GLfloat));
  P3_chunk_get (chunk, mesh->fnormals, mesh->nb_fnormals * 3 * sizeof (GLfloat));
  P3_chunk_get (chunk, mesh->values,   mesh->nb_values   * sizeof (GLfloat));
  P3_chunk_get (chunk, mesh->dimension, 6 * sizeof (GLfloat));
  P3_chunk_get (chunk, mesh->sphere,    4 * sizeof (GLfloat));
  if (mesh->option & P3_CMESH_HAS_VERTEX_VISIBILITY) {
    mesh->voptions = (char*) malloc (mesh->nb_vcoords * sizeof (char));
  } else {
    mesh->voptions = NULL;
  }
  /* vertices */
  for (i = 0; i < mesh->nb_vertices; i++) {
    v = mesh->vertices + i;
    v->coord = P3_chunk_get_int (chunk);
    j = P3_chunk_get_int (chunk);
    if (j == -1) {
      v->e_color = NULL;
    } else {
      v->e_color = mesh->values + j;
    }
    j = P3_chunk_get_int (chunk);
    if (j == -1) {
      v->d_color = NULL;
    } else {
      v->d_color = mesh->values + j;
    }
    j = P3_chunk_get_int (chunk);
    if (j == -1) {
      v->texcoord = NULL;
    } else {
      v->texcoord = mesh->values + j;
    }
  }
  /* packs */
  for (i = 0; i < mesh->nb_packs; i++) {
    p = mesh->packs + i;
    p->option            = P3_chunk_get_int (chunk);
    p->vertices_per_face = P3_chunk_get_int (chunk);
    p->nb_faces          = P3_chunk_get_int (chunk);
    p->faces = (P3_cface*) malloc (p->nb_faces * sizeof (P3_cface));
    /* faces */
    for (j = 0; j < p->nb_faces; j++) {
      f = p->faces + j;
      f->pack = p;
      f->normal = mesh->fnormals + P3_chunk_get_int (chunk);
      f->option = P3_chunk_get_char (chunk);
      f->vertices = (P3_cvertex**) malloc (p->vertices_per_face * sizeof (P3_cvertex*));
      /* vertices */
      for (k = 0; k < p->vertices_per_face; k++) {
        f->vertices[k] = mesh->vertices + P3_chunk_get_int (chunk);
      }
    }
  }
  /* nodes */
  if (P3_chunk_get_char (chunk) == '\0') {
    mesh->tree = NULL;
  } else {
    mesh->tree = P3_cmesh_node_set_data (mesh, chunk);
  }

  mesh->xm = NULL;

}


/*===================+
 | MESH CONSTRUCTION |
 +===================*/

/*
static void P3_cmesh_compute_alpha (P3_cmesh* mesh) {
  int i;
  if (mesh->option & P3_CMESH_HAS_ALPHA) { mesh->option -= P3_CMESH_HAS_ALPHA; }
  for (i = 0; i < mesh->nb_packs; i++) {
    if ((mesh->packs + i)->option & P3_R_ALPHA) {
      mesh->option |= P3_CMESH_HAS_ALPHA;
      return;
    }
  }
}
*/

static int P3_cmesh_register_vcoord (P3_cmesh* mesh, GLfloat* coord, int opt) {
  int i;
  for (i = 0; i < mesh->nb_vcoords; i++) {
    if (   fabs (coord[0] - mesh->vcoords[i * 3    ]) < P3_EPSILON
        && fabs (coord[1] - mesh->vcoords[i * 3 + 1]) < P3_EPSILON
        && fabs (coord[2] - mesh->vcoords[i * 3 + 2]) < P3_EPSILON) { return i; }
  }
  i = mesh->nb_vcoords;
  (mesh->nb_vcoords)++;
  mesh->vcoords = (GLfloat*) realloc (mesh->vcoords, mesh->nb_vcoords * 3 * sizeof (GLfloat));
  memcpy (mesh->vcoords + i * 3, coord, 3 * sizeof (GLfloat));
  if (opt & P3_R_SMOOTHLIT) {
    mesh->nb_vnormals = mesh->nb_vcoords;
    mesh->vnormals = (GLfloat*) realloc (mesh->vnormals, mesh->nb_vnormals * 3 * sizeof (GLfloat));
    memset (mesh->vnormals + i * 3, 0.0, 3 * sizeof (GLfloat));
  }
  return i;
}

static GLfloat* P3_cmesh_register_fnormal (P3_cmesh* mesh, GLfloat* normal) {
  GLfloat* old;
  int i;
  int r;
  for (i = 0; i < mesh->nb_fnormals; i++) {
    if (   fabs (normal[0] - mesh->fnormals[i * 3    ]) < P3_EPSILON
        && fabs (normal[1] - mesh->fnormals[i * 3 + 1]) < P3_EPSILON
        && fabs (normal[2] - mesh->fnormals[i * 3 + 2]) < P3_EPSILON) { return mesh->fnormals + i * 3; }
  }
  r = mesh->nb_fnormals;
  (mesh->nb_fnormals)++;
  old = mesh->fnormals;
  mesh->fnormals = (GLfloat*) realloc (mesh->fnormals, mesh->nb_fnormals * 3 * sizeof (GLfloat));
  memcpy (mesh->fnormals + r * 3, normal, 3 * sizeof (GLfloat));
  if (mesh->fnormals != old) {
    P3_cpack* pack;
    int j;
    for (i = 0; i < mesh->nb_packs; i++) {
      pack = mesh->packs + i;
      for (j = 0; j < pack->nb_faces; j++) {
        (pack->faces + j)->normal = mesh->fnormals + ((pack->faces + j)->normal - old);
      }
    }
  }
  return mesh->fnormals + r * 3;
}

static GLfloat* P3_cmesh_register_value (P3_cmesh* mesh, GLfloat* value, int nb) {
  GLfloat* old;
  int r;
  int i; int j; int k;
  for (i = 0; i <= mesh->nb_values - nb; i++) {
    k = 0;
    for (j = 0; j < nb; j++) {
      if (fabs (value[j] - mesh->values[i + j]) < P3_EPSILON) { k++; }
    }
    if (k == nb) { return mesh->values + i; }
  }
  r = mesh->nb_values;
  mesh->nb_values += nb;
  old = mesh->values;
  mesh->values = (GLfloat*) realloc (mesh->values, mesh->nb_values * sizeof (GLfloat));
  memcpy (mesh->values + r, value, nb * sizeof (GLfloat));
  if (mesh->values != old) {
    P3_cvertex* v;
    for (i = 0; i < mesh->nb_vertices; i++) {
      v = mesh->vertices + i;
      if (v->e_color  != NULL) { v->e_color  = mesh->values + (v->e_color  - old); }
      if (v->d_color  != NULL) { v->d_color  = mesh->values + (v->d_color  - old); }
      if (v->texcoord != NULL) { v->texcoord = mesh->values + (v->texcoord - old); }
    }
  }
  return mesh->values + r;
}

static P3_cvertex* P3_cmesh_register_vertex (P3_cmesh* mesh, P3_cvertex* v) {
  P3_cvertex* old;
  P3_cvertex* r = NULL;
  int found = P3_FALSE;
  int i;
  int n;
  for (i = 0; i < mesh->nb_vertices; i++) {
    r = mesh->vertices + i;
    if (r->coord == v->coord
        && (r->d_color  == NULL || v->d_color  == NULL || r->d_color  == v->d_color)
        && (r->e_color  == NULL || v->e_color  == NULL || r->e_color  == v->e_color)
        && (r->texcoord == NULL || v->texcoord == NULL || r->texcoord == v->texcoord)
        ) { 
      found = P3_TRUE;
      break; 
    }
  }
  if (found == P3_FALSE) {
    n = mesh->nb_vertices;
    (mesh->nb_vertices)++;
    old = mesh->vertices;
    mesh->vertices = (P3_cvertex*) realloc (mesh->vertices, mesh->nb_vertices * sizeof (P3_cvertex));
    memcpy (mesh->vertices + n, v, sizeof (P3_cvertex));
    if (mesh->vertices != old) {
      P3_cpack* pack;
      P3_cface* face;
      int j;
      int k;
      for (i = 0; i < mesh->nb_packs; i++) {
        pack = mesh->packs + i;
        for (j = 0; j < pack->nb_faces; j++) {
          face = pack->faces + j;
          for (k = 0; k < pack->vertices_per_face; k++) {
            face->vertices[k] = mesh->vertices + (face->vertices[k] - old);
          }
        }
      }
    }
    return mesh->vertices + n;
  } else {
    if (r->d_color  == NULL) { r->d_color  = v->d_color;  }
    if (r->e_color  == NULL) { r->e_color  = v->e_color;  }
    if (r->texcoord == NULL) { r->texcoord = v->texcoord; }
    return r;
  }
}

/*
void P3_cmesh_register_face_coords (P3_cmesh* mesh, P3_face* face) {
  P3_coordsys* csys;
  P3_vertex* pv;
  GLfloat* matrix;
  GLfloat p[4];
  int nbv;
  int i;
  int option;
  GLfloat* c;
  nbv = P3_face_get_vertices_number (face);
  option = P3_face_get_option (face);
  c = (GLfloat*) malloc (nbv * 3 * sizeof (GLfloat));
  for (i = 0; i < nbv; i++) {
    pv = P3_face_get_vertex (face, i);
    csys = P3_vertex_get_coordsys (pv);
    matrix = P3_coordsys_get_root_matrix (csys);
    c[i * 3]     = P3_vertex_get_x (pv);
    c[i * 3 + 1] = P3_vertex_get_y (pv);
    c[i * 3 + 2] = P3_vertex_get_z (pv);
    P3_point_by_matrix (c + i * 3, matrix);
    P3_cmesh_register_vcoord (mesh, c + i * 3, option);
    if (option & P3_R_COLORED) {
      p[0] = P3_vertex_get_r (pv);
      p[1] = P3_vertex_get_g (pv);
      p[2] = P3_vertex_get_b (pv);
      p[3] = P3_vertex_get_a (pv);
      P3_cmesh_register_value (mesh, p, 4);
    }
    if (option & P3_R_TEXTURED) {
      p[0] = P3_vertex_get_u (pv);
      p[1] = P3_vertex_get_v (pv);
      P3_cmesh_register_value (mesh, p, 2);
    }
  }
  P3_face_normal (p, c, c + 3, c + 6);
  P3_cmesh_register_fnormal (mesh, p);
  free (c);
}

P3_cvertex* P3_cmesh_register_face_vertex (P3_cmesh* mesh, P3_vertex* pv, int option) {
  P3_cvertex v;
  P3_coordsys* csys;
  GLfloat* matrix;
  GLfloat p[4];
  csys = P3_vertex_get_coordsys (pv);
  matrix = P3_coordsys_get_root_matrix (csys);
  p[0] = P3_vertex_get_x (pv);
  p[1] = P3_vertex_get_y (pv);
  p[2] = P3_vertex_get_z (pv);
  P3_point_by_matrix (p, matrix);
  v.coord = P3_cmesh_register_vcoord (mesh, p, option);
  if (option & P3_R_COLORED) {
    p[0] = P3_vertex_get_r (pv);
    p[1] = P3_vertex_get_g (pv);
    p[2] = P3_vertex_get_b (pv);
    p[3] = P3_vertex_get_a (pv);
    v.d_color = P3_cmesh_register_value (mesh, p, 4);
  } else {
    v.d_color = NULL;
  }
  v.e_color = NULL;
  if (option & P3_R_TEXTURED) {
    p[0] = P3_vertex_get_u (pv);
    p[1] = P3_vertex_get_v (pv);
    v.texcoord = P3_cmesh_register_value (mesh, p, 2);
  } else {
    v.texcoord = NULL;
  }
  return P3_cmesh_register_vertex (mesh, &v);
}

void P3_cmesh_register_face_vertices (P3_cmesh* mesh, P3_face* face) {
  int nbv;
  int option;
  int i;
  nbv = P3_face_get_vertices_number (face);
  option = P3_face_get_option (face);
  for (i = 0; i < nbv; i++) {
    P3_cmesh_register_face_vertex (mesh, P3_face_get_vertex (face, i), option);
  }
}

void P3_cmesh_add_face_to_pack (P3_cmesh* mesh, P3_cpack* pack, P3_face* face) {
  P3_cface* f;
  GLfloat p[3];
  int i;

  pack->faces = (P3_cface*) realloc (pack->faces, (pack->nb_faces + 1) * sizeof (P3_cface));
  f = pack->faces + pack->nb_faces;
  (pack->nb_faces)++;

  f->vertices = (P3_cvertex**) malloc (pack->vertices_per_face * sizeof (P3_cvertex*));

  for (i = 0; i < pack->vertices_per_face; i++) {
    f->vertices[i] = P3_cmesh_register_face_vertex (mesh, P3_face_get_vertex (face, i), pack->option);
  }

  P3_face_normal (p, 
                  mesh->vcoords + f->vertices[0]->coord * 3,
                  mesh->vcoords + f->vertices[1]->coord * 3,
                  mesh->vcoords + f->vertices[2]->coord * 3);
  f->normal = P3_cmesh_register_fnormal (mesh, p);;

  // vertex normal
  if (pack->option & P3_R_SMOOTHLIT) {
    GLfloat v1[3];
    GLfloat v2[3];
    GLfloat w;
    int j;
    int nb;
    for (i = 0; i < pack->vertices_per_face; i++) {
      nb = i + 1;
      if (nb >= pack->vertices_per_face) { nb = 0; } 
      P3_vector_from_points (v1,
                             mesh->vcoords + f->vertices[i]->coord * 3,
                             mesh->vcoords + f->vertices[nb]->coord * 3);
      nb = i - 1;
      if (nb < 0) { nb = pack->vertices_per_face - 1; }
      P3_vector_from_points (v2,
                             mesh->vcoords + f->vertices[i]->coord * 3,
                             mesh->vcoords + f->vertices[nb]->coord * 3);
      w = P3_vector_angle (v1, v2);
      for (j = 0; j < 3; j++) {
        (mesh->vnormals + f->vertices[i]->coord * 3)[j] += (w * f->normal[j]);
      }
    }
  }
}

void P3_cmesh_add_face (P3_cmesh* mesh, P3_face* face) {
  int nbv;
  int option;
  P3_material* material;
  P3_cpack* pack;
  int i;
  nbv = P3_face_get_vertices_number (face);
  option = P3_face_get_option (face);
  material = P3_face_get_material (face);
  for (i = 0; i < mesh->nb_packs; i++) {
    pack = mesh->packs + i;
    if (pack->option == option && pack->vertices_per_face == nbv && pack->material == material) {
      P3_cmesh_add_face_to_pack (mesh, pack, face);
      i = -1;
      break;
    }
  }
  if (i != -1) {
    mesh->packs = (P3_cpack*) realloc (mesh->packs, (mesh->nb_packs + 1) * sizeof (P3_cpack));
    pack = mesh->packs + mesh->nb_packs;
    (mesh->nb_packs)++;
    pack->material = material;
    if (material != NULL) { P3_material_incref (material); }
    pack->option = option;
    pack->vertices_per_face = nbv;
    pack->faces = NULL;
    pack->nb_faces = 0;
    P3_cmesh_add_face_to_pack (mesh, pack, face);
  }
}
*/

static P3_cvertex* P3_cmesh_add_vertex (P3_cmesh* mesh, P3_vertex* pv, int option) {
  P3_cvertex v;
  P3_coordsys* csys;
  GLfloat* matrix;
  GLfloat p[4];
  GLfloat* old;
  csys = P3_vertex_get_coordsys (pv);
  matrix = P3_coordsys_get_root_matrix (csys);
  p[0] = P3_vertex_get_x (pv);
  p[1] = P3_vertex_get_y (pv);
  p[2] = P3_vertex_get_z (pv);
  P3_point_by_matrix (p, matrix);
  v.coord = P3_cmesh_register_vcoord (mesh, p, option);
  if (option & P3_R_COLORED) {
    p[0] = P3_vertex_get_r (pv);
    p[1] = P3_vertex_get_g (pv);
    p[2] = P3_vertex_get_b (pv);
    p[3] = P3_vertex_get_a (pv);
    v.d_color = P3_cmesh_register_value (mesh, p, 4);
  } else {
    v.d_color = NULL;
  }
  v.e_color = NULL;
  old = mesh->values;
  if (option & P3_R_TEXTURED) {
    p[0] = P3_vertex_get_u (pv);
    p[1] = P3_vertex_get_v (pv);
    v.texcoord = P3_cmesh_register_value (mesh, p, 2);
    if (v.d_color != NULL && mesh->values != old) { v.d_color = mesh->values + (v.d_color - old); }
  } else {
    v.texcoord = NULL;
  }
  return P3_cmesh_register_vertex (mesh, &v);
}

static void P3_cmesh_add_face_to_pack (P3_cmesh* mesh, P3_cpack* pack, P3_face* face) {
  P3_cface* f;
  GLfloat p[3];
  int i;
  pack->faces = (P3_cface*) realloc (pack->faces, (pack->nb_faces + 1) * sizeof (P3_cface));
  f = pack->faces + pack->nb_faces;
  (pack->nb_faces)++;
  if (P3_face_is_solid (face) == P3_TRUE) {
    f->option = 0;
  } else {
    f->option = P3_OBJECT_NON_SOLID;
  }
  f->vertices = (P3_cvertex**) malloc (pack->vertices_per_face * sizeof (P3_cvertex*));
  for (i = 0; i < pack->vertices_per_face; i++) {
    f->vertices[i] = P3_cmesh_add_vertex (mesh, P3_face_get_vertex (face, i), pack->option);
  }
  if (pack->vertices_per_face > 2) {
    P3_face_normal (p,
                    mesh->vcoords + f->vertices[0]->coord * 3,
                    mesh->vcoords + f->vertices[1]->coord * 3,
                    mesh->vcoords + f->vertices[2]->coord * 3);
    P3_vector_normalize (p);
    f->normal = P3_cmesh_register_fnormal (mesh, p);;
    if (pack->option & P3_R_SMOOTHLIT) {
      GLfloat v1[3];
      GLfloat v2[3];
      GLfloat w;
      int j;
      int nb;
      for (i = 0; i < pack->vertices_per_face; i++) {
        nb = i + 1;
        if (nb >= pack->vertices_per_face) { nb = 0; } 
        P3_vector_from_points (v1,
                               mesh->vcoords + f->vertices[i]->coord * 3,
                               mesh->vcoords + f->vertices[nb]->coord * 3);
        nb = i - 1;
        if (nb < 0) { nb = pack->vertices_per_face - 1; }
        P3_vector_from_points (v2,
                               mesh->vcoords + f->vertices[i]->coord * 3,
                               mesh->vcoords + f->vertices[nb]->coord * 3);
        w = P3_vector_angle (v1, v2);
        for (j = 0; j < 3; j++) {
          (mesh->vnormals + f->vertices[i]->coord * 3)[j] += (w * f->normal[j]);
        }
      }
    }
  } else {
    f->normal = NULL;
  }
}

static int P3_cmesh_option_compatible (int opt1, int opt2) {

#define OPTION_EQUALS(flag) \
  ((opt1 & flag && opt2 & flag) || (!(opt1 & flag) && !(opt2 & flag)))

  if (OPTION_EQUALS (P3_R_SMOOTHLIT) &&
      OPTION_EQUALS (P3_R_STATIC_LIT) &&
      OPTION_EQUALS (P3_R_ALPHA) &&
      OPTION_EQUALS (P3_R_TEXTURED) &&
      OPTION_EQUALS (P3_R_COLORED) &&
      OPTION_EQUALS (P3_R_FRONT_AND_BACK) &&
      OPTION_EQUALS (P3_R_NEVER_LIT) &&
      OPTION_EQUALS (P3_R_EMISSIVE)) {
    return P3_TRUE;
  } else {
    return P3_FALSE;
  }

#undef OPTION_EQUALS

}

static int P3_cpack_option_from_face_option (int option) {
  return (option & P3_R_SMOOTHLIT) 
       + (option & P3_R_STATIC_LIT) 
       + (option & P3_R_ALPHA) 
       + (option & P3_R_TEXTURED) 
       + (option & P3_R_COLORED) 
       + (option & P3_R_FRONT_AND_BACK) 
       + (option & P3_R_NEVER_LIT) 
       + (option & P3_R_EMISSIVE); 
}

static void P3_cmesh_add_face (P3_cmesh* mesh, P3_face* face) {
  P3_material* material;
  P3_cpack* pack = NULL;
  int nbv;
  int option;
  int i;
  nbv = P3_face_get_vertices_number (face);
  option = P3_face_get_option (face);
  /* HACK TO DO review */
  if (option & P3_R_STATIC_LIT) { option -= P3_R_STATIC_LIT; }
  material = P3_face_get_material (face);
  for (i = 0; i < mesh->nb_packs; i++) {
    pack = mesh->packs + i;
    if (P3_cmesh_option_compatible (pack->option, option) == P3_TRUE && 
        pack->vertices_per_face == nbv && pack->material == material) {
      i = -1;
      break;
    }
  }
  if (i != -1) {
    mesh->packs = (P3_cpack*) realloc (mesh->packs, (mesh->nb_packs + 1) * sizeof (P3_cpack));
    pack = mesh->packs + mesh->nb_packs;
    (mesh->nb_packs)++;
    pack->material = material;
    if (material != NULL) { P3_material_incref (material); }
    pack->option = P3_cpack_option_from_face_option (option);
    pack->vertices_per_face = nbv;
    pack->faces = NULL;
    pack->nb_faces = 0;
  }
  P3_cmesh_add_face_to_pack (mesh, pack, face);
}

void P3_cmesh_compute_dimension (P3_cmesh* mesh) {
  int i; int j;
  GLfloat min_x = 0.0;
  GLfloat min_y = 0.0;
  GLfloat min_z = 0.0;
  GLfloat max_x = 0.0;
  GLfloat max_y = 0.0;
  GLfloat max_z = 0.0;
  j = 0;
  for (i = 0; i < mesh->nb_vcoords; i++) {
    if (i == 0) {
      min_x = mesh->vcoords[j++];
      min_y = mesh->vcoords[j++];
      min_z = mesh->vcoords[j++];
      max_x = min_x;
      max_y = min_y;
      max_z = min_z;
    } else {
      if (mesh->vcoords[j] < min_x) { min_x = mesh->vcoords[j]; }
      if (mesh->vcoords[j] > max_x) { max_x = mesh->vcoords[j]; }
      j++;
      if (mesh->vcoords[j] < min_y) { min_y = mesh->vcoords[j]; }
      if (mesh->vcoords[j] > max_y) { max_y = mesh->vcoords[j]; }
      j++;
      if (mesh->vcoords[j] < min_z) { min_z = mesh->vcoords[j]; }
      if (mesh->vcoords[j] > max_z) { max_z = mesh->vcoords[j]; }
      j++;
    }
  }
  mesh->dimension[0] = min_x;
  mesh->dimension[1] = min_y;
  mesh->dimension[2] = min_z;
  mesh->dimension[3] = max_x;
  mesh->dimension[4] = max_y;
  mesh->dimension[5] = max_z;
}

static void P3_cmesh_sort_packs (P3_cmesh* mesh) {
  P3_cpack* packs = (P3_cpack*) malloc (mesh->nb_packs * sizeof (P3_cpack));
  P3_cpack* pack;
  int i;
  int j = 0;
  mesh->nb_packs_opaque = 0;
  mesh->nb_packs_alpha = 0;
  for (i = 0; i < mesh->nb_packs; i++) {
    pack = mesh->packs + i;
    if (pack->option & P3_R_ALPHA) {
      (mesh->nb_packs_alpha)++;
    } else {
      (mesh->nb_packs_opaque)++;
      memcpy (packs + j, pack, sizeof (P3_cpack));
      j++;
    }
  }
  for (i = 0; i < mesh->nb_packs; i++) {
    pack = mesh->packs + i;
    if (pack->option & P3_R_ALPHA) {
      memcpy (packs + j, pack, sizeof (P3_cpack));
      j++;
    }
  }
  free (mesh->packs);
  mesh->packs = packs;
}

static void P3_cmesh_finalize (P3_cmesh* mesh) {
  P3_cpack* pack;
  int i;
  int j;
  for(i = 0; i < mesh->nb_vnormals; i++) {
    P3_vector_normalize (mesh->vnormals + i * 3);
  }
  P3_cmesh_sort_packs (mesh);
  for (i = 0; i < mesh->nb_packs; i++) {
    pack = mesh->packs + i;
    for (j = 0; j < pack->nb_faces; j++) {
      (pack->faces + j)->pack = pack;
    }
  }
  P3_sphere_from_points (mesh->sphere, mesh->vcoords, mesh->nb_vcoords);
  P3_cmesh_compute_dimension (mesh);
}

P3_cmesh* P3_cmesh_from_faces (P3_cmesh* mesh, P3_list* given_faces, P3_list* lights) {
//  P3_cvertex* v;
//  P3_cface* ff;
//  P3_cpack* p;
  P3_face* f;
  int i;
//  int j;
  int k;

printf ("Building mesh from faces...\n");
printf ("  %i faces, %i lights\n", given_faces->nb, lights->nb);

  /* create a new mesh data */
  if (mesh == NULL) { mesh = P3_cmesh_new (NULL); }


//mesh->xm = P3_xmesh_from_faces (NULL, given_faces);


  /* add smoothlit vertices first */
  for (i = 0; i < given_faces->nb; i++) {
    f = (P3_face*) P3_list_get (given_faces, i);
    if (P3_face_is_smoothlit (f) == P3_TRUE) {
      P3_cmesh_add_face (mesh, f);
    }
  }
  /* then add non smoothlit vertices */
  for (i = 0; i < given_faces->nb; i++) {
    f = (P3_face*) P3_list_get (given_faces, i);
    if (P3_face_is_smoothlit (f) == P3_FALSE) {
      P3_cmesh_add_face (mesh, f);
    }
  }
/*
  for (i = 0; i < mesh->nb_vertices; i++) {
    v = mesh->vertices + i;
    v->e_color  += mesh->values;
    v->d_color  += mesh->values;
    v->texcoord += mesh->values;
  }
  for (i = 0; i < mesh->nb_packs; i++) {
    p = mesh->packs + i;
    for (j = 0; j < pack->nb_faces; j++) {
      ff = pack->faces + j;
      for (k = 0; k < pack->vertices_per_face; k++) {
        ff[k] += mesh->vertices;
      }
    }
  }
*/

/*
  for (i = 0; i < given_faces->nb; i++) {
    f = (P3_face*) P3_list_get (given_faces, i);
    if (P3_face_is_smoothlit (f) == P3_TRUE) {
      P3_cmesh_register_face_coords (mesh, f);
    }
  }
  for (i = 0; i < given_faces->nb; i++) {
    f = (P3_face*) P3_list_get (given_faces, i);
    if (P3_face_is_smoothlit (f) == P3_FALSE) {
      P3_cmesh_register_face_coords (mesh, f);
    }
  }
  for (i = 0; i < given_faces->nb; i++) {
    f = (P3_face*) P3_list_get (given_faces, i);
    P3_cmesh_register_face_vertices (mesh, f);
  }
  for (i = 0; i < given_faces->nb; i++) {
    f = (P3_face*) P3_list_get (given_faces, i);
    P3_cmesh_add_face (mesh, f);
  }
*/

  /* compute static lights if needed */
// TO DO ?
  if (lights->nb > 0) {
    P3_cmesh_static_light (mesh, lights, P3_FALSE);
  }

  /* finalize */
  P3_cmesh_finalize (mesh);

printf ("  %i packs\n", mesh->nb_packs);
printf ("  option:");
k = 0;
for (i = 0; i < mesh->nb_packs; i++) {
  k |= (mesh->packs + i)->option;
}
if (k & P3_OBJECT_NON_SOLID) { printf (" NON_SOLID"); }
if (k & P3_R_SMOOTHLIT)      { printf (" SMOOTHLIT"); }
if (k & P3_R_STATIC_LIT)     { printf (" STATIC_LIT"); }
if (k & P3_R_ALPHA)          { printf (" ALPHA"); }
if (k & P3_R_TEXTURED)       { printf (" TEXTURED"); }
if (k & P3_R_COLORED)        { printf (" COLORED"); }
if (k & P3_R_FRONT_AND_BACK) { printf (" DOUBLE_SIDED"); }
if (k & P3_R_NEVER_LIT)      { printf (" NEVER_LIT"); }
if (k & P3_R_EMISSIVE)       { printf (" EMISSIVE"); }
printf ("\n");
printf ("  [DONE]\n");

  return mesh;
}

P3_cmesh* P3_cmesh_from_world (P3_cmesh* mesh, P3_world* world) {
  P3_list* faces;
  P3_list* lights;
  faces  = P3_list_new (100);
  lights = P3_list_new (8);
  /* take all faces and lights of the world */
  P3_world_extract (world, P3_ID_FACE,  faces);
  P3_world_extract (world, P3_ID_LIGHT, lights);
  /* create the mesh */
  mesh = P3_cmesh_from_faces (mesh, faces, lights);
  P3_list_dealloc (faces);
  P3_list_dealloc (lights);
  return mesh;
}

void P3_cmesh_to_faces (P3_cmesh* mesh, P3_world* world) {
  P3_cpack* pack;
  P3_cface* face;
  P3_cvertex* vertex;
  GLfloat* ptr;
  P3_vertex* pv;
  P3_face* pf;
  int i; int j; int k;
  for (i = 0; i < mesh->nb_packs; i++) {
    pack = mesh->packs + i;
    for (j = 0; j < pack->nb_faces; j++) {
      face = pack->faces + j;
      pf = P3_face_new (world, pack->material, 0);
      if (pack->option & P3_R_SMOOTHLIT)      { P3_face_set_smoothlit    (pf, 1); }
      if (pack->option & P3_R_FRONT_AND_BACK) { P3_face_set_double_sided (pf, 1); }
      if (pack->option & P3_OBJECT_NON_SOLID) { P3_face_set_solid        (pf, 0); }
      if (pack->option & P3_R_STATIC_LIT)     { P3_face_set_static_lit   (pf, 1); }
      if (pack->option & P3_R_NEVER_LIT)      { P3_face_set_can_be_lit   (pf, 0); }
      for (k = 0; k < pack->vertices_per_face; k++) {
        vertex = face->vertices[k];
        ptr = mesh->vcoords + vertex->coord * 3;
        pv = P3_vertex_new (world, ptr[0], ptr[1], ptr[2]);
        if (pack->option & P3_R_TEXTURED) { P3_vertex_set_texcoord (pv, vertex->texcoord); }
        if (pack->option & P3_R_COLORED)  { P3_vertex_set_color    (pv, vertex->d_color); }
        P3_face_add_vertex (pf, pv);
      }
    }
  }
}


/*=================+
 | STATIC LIGHTING |
 +=================*/

void P3_cmesh_static_light (P3_cmesh* mesh, P3_list* lights, int shadow) {
  P3_cpack* pack;
  P3_cface* face;
  P3_cvertex* vertex;
  P3_cvertex new_vertex;
  P3_light* light;
  GLfloat* normal;
  GLfloat color[4];
  GLfloat color_tmp[4];
  GLfloat v1[3];
  GLfloat v2[3];
  int i;
  int j;
  int k;
  int l;

  if (lights == NULL || lights->nb == 0) {

printf("Removing static lighting\n");

    for (i = 0; i < mesh->nb_packs; i++) {
      pack = mesh->packs + i;
      if (pack->option & P3_R_EMISSIVE)   { pack->option -= P3_R_EMISSIVE; }
      if (pack->option & P3_R_STATIC_LIT) { pack->option -= P3_R_STATIC_LIT; }
    }
// TO DO problem: emissive color values are not removed !!

  } else {

printf("Computing static lighting\n");
printf("  %i lights\n", lights->nb);

    for (i = 0; i < mesh->nb_packs; i++) {
      pack = mesh->packs + i;
      pack->option |= P3_R_STATIC_LIT;
      pack->option |= P3_R_EMISSIVE;
      for (j = 0; j < pack->nb_faces; j++) {
        face = pack->faces + j;
        normal = face->normal;
        for (k = 0; k < pack->vertices_per_face; k++) {
          vertex = face->vertices[k];
          if (pack->option & P3_R_SMOOTHLIT) {
            normal = mesh->vnormals + vertex->coord * 3;
          }
          color[0] = 0.0;
          color[1] = 0.0;
          color[2] = 0.0;
          color[3] = 0.0;
          for (l = 0; l < lights->nb; l++) {
            light = (P3_light*) P3_list_get (lights, l);
            if (light->parent != NULL) {
              P3_point_by_matrix_copy  (v1, mesh->vcoords + vertex->coord * 3, P3_coordsys_get_inverted_root_matrix ((P3_coordsys*) light->parent));
              P3_vector_by_matrix_copy (v2, normal,                            P3_coordsys_get_inverted_root_matrix ((P3_coordsys*) light->parent));
            } else {
              memcpy (v1, mesh->vcoords + vertex->coord * 3, 3 * sizeof (GLfloat));
              memcpy (v2, normal,                            3 * sizeof (GLfloat));
            }
            if (shadow == P3_FALSE || P3_light_get_shadow_at (light, v1) == P3_FALSE) {
              P3_light_get_color_at (light, v1, v2, P3_FALSE, color_tmp);
              color[0] += color_tmp[0];
              color[1] += color_tmp[1];
              color[2] += color_tmp[2];
              color[3] += color_tmp[3];
            }
          }
          if (vertex->e_color == NULL) {
            vertex->e_color = P3_cmesh_register_value (mesh, color, 4);
          } else {
            new_vertex.coord    = vertex->coord;
            new_vertex.d_color  = vertex->d_color;
            new_vertex.e_color  = P3_cmesh_register_value (mesh, color, 4);
            new_vertex.texcoord = vertex->texcoord;
            face->vertices[k] = P3_cmesh_register_vertex (mesh, &new_vertex);;
          }
        }
      }
    }
  }

printf("  [DONE]\n");

}



