# Soya 3D
# Copyright (C) 2001-2003 Jean-Baptiste LAMY -- jiba@tuxfamily.org
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""soya.Animation

Animation module; see tutorial lesson 11."""

import soya, _soya, soya.soya3d as soya3d, soya.model as model


class Animation(object):
  """Animation

An animation, used to animate a world. An animation is made of several states,
which occur at a given time. In a first step, you must create the animation and
provide the states. Then the animation object can interpolate between these states.

The following attributes are available:
 - first_time and last_time (read only): the first
 - cyclic_lap: for cyclic animation, this is the time duration (a float) between
   the end of the animation and the beginning.
   For non-cyclic animation, it must be 0.0

Use animation.insert_state(...) to add new state to the animation, and
del animation[state_index] to remove a state.

Use animation.set_to_state to interpolate to the state at a given time."""
  
  def __init__(self):
    self.anims = {}
    self.first_time = self.last_time = 0.0
    
  def _compute_boundary_time(self):
    if not self.anims:
      self.first_time = self.last_time = 0.0
      return
    anim = self.anims.values()[0]
    self.first_time = anim.get_time_for(0)
    self.last_time  = anim.get_time_for(anim.states_number - 1) + anim.cyclic_lap
  def get_cyclic_lap(self):
    if not self.anims: return 0.0
    return self.anims.values()[0].cyclic_lap
  def set_cyclic_lap(self, cyclic_lap):
    if not self.anims: raise ValueError("Cannot set cyclic_lap attribute before having inserted the first state of the animation!")
    for anim in self.anims.values(): anim.cyclic_lap = cyclic_lap
    self._compute_boundary_time()
  cyclic_lap = property(get_cyclic_lap, set_cyclic_lap)
  
  def insert_state(self, time, world, all_items = None):
    """Animation.insert_state(time, world, all_items = None)

Insert the current state (objects' positions) of WORLD at instant TIME in this
animation.

If given, ALL_ITEMS is the list of the item (and subitem) of WORLD to animate; it
defaults to all items recursively in WORLD (=WORLD.recursive()).

Only the items that were present when the first state was inserted are taken into
account."""
    if not all_items: all_items = world.recursive()
    
    if self.anims:
      for item in all_items:
        if hasattr(item, "_anim_id"):
          anim = self.anims.get(item._anim_id)
          if anim: anim.add(item, time)
    else:
      for item in all_items:
        if isinstance(item, soya3d.Element3D):
          if not hasattr(item, "_anim_id"): item._anim_id = hash(item)
          anim = self.anims.get(item._anim_id)
          if not anim: anim = self.anims[item._anim_id] = _soya._AnimCoordsys()
          anim.add(item, time)
          
    self._compute_boundary_time()
    
  def set_to_state(self, time, world, old_time = 0.0, all_items = None):
    """Animation.set_to_state(time, world, old_time = 0.0, all_items = None)

Set the items (and subitems) of WORLD to their positions at instant TIME of the
animation. animation.first_time <= TIME <= animation.last_time is required, but
TIME doesn't need to be one of the time value given to Animation.insert_state -- in
such a case the position is interpolated between the two nearest available states.

WORLD should be eitheir the world that has been used to build the animation (with
insert_state) or a deepcopy of this world. Actually, the different subitem of world
are identified by their "_anim_id" attribute, which is automaticaly set if needed.

If given, ALL_ITEMS is the list of the item (and subitem) of WORLD to animate; it
defaults to all items recursively in WORLD (=WORLD.recursive())."""
    for item in all_items or world.recursive():
      if hasattr(item, "_anim_id"):
        anim = self.anims.get(item._anim_id)
        if anim: item.set_anim_state(anim, old_time, time)
        
  def __len__(self):
    """Animation.__len__() -> int

Gets the number of states in this animation."""
    if not self.anims: return 0
    return self.anims.values()[0].states_number
  
  def __delitem__(self, index_or_time):
    """del Animation[index_or_time]

Removes the state corresponding to INDEX_OR_TIME. INDEX_OR_TIME is interpreted as
an index if it is an int, and a time value if it is a float."""
    for anim in self.anims.values(): anim.remove(index_or_time)
    
    self._compute_boundary_time()
    
  def get_time_of_state(self, index):
    """Animation.get_time_of_state(index) -> float

Get the time at which occurs the state INDEX."""
    if not self.anims: return 0
    return self.anims.values()[0].get_time_for(index)
  
  def set_time_of_state(self, index, time):
    """Animation.set_time_of_state(index, time)

Set the time at which occurs the state INDEX to TIME."""
    for anim in self.anims.values(): anim.set_time_for(index, time)
    
    self._compute_boundary_time()
    

