/*
 * ----------------------------------------------------
 *
 * Emulation of an USB device 
 * (C) 2006 Jochen Karrer 
 *
 * State:
 *	Nothing works, does not compile
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * ----------------------------------------------------
 */

#include <usbdevice.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>

#define STATE_IDLE	    (0)
#define STATE_DO_OUT        (1)
#define STATE_DO_IN         (2)
#define STATE_DO_ISOCHO     (3)
#define STATE_DO_BCINTO     (4)
#define STATE_HS_BCO        (5)
#define STATE_DO_ISOCHI     (6)
#define STATE_DO_BCINTI     (7)
#define STATE_HSPING        (8)

#define	STATE_BCINTO_DEV_WAIT_ODATA 	(4)
#if 0
#define	STATE_BCINTO_DCHKPKT2		(9)
#define STATE_BCINT0_DOPKT		(xx) overlaps with something

#endif
#define STATE_BCINTI_DEV_RESP		(11)	


/* 
 * -----------------------------------------------------------------
 * Implementation of DEV state machine 
 * -----------------------------------------------------------------
 */
static void
pid_out(UsbDevice *udev,UsbPacket *packet) 
{

	UsbDev_Endpoint *epnt;
	uint8_t addr = packet->addr;
	uint8_t epnum = packet->endp & 0xf;
	uint8_t crc5 = packet->crc5; 
	if(addr != udev->addr) {
		return;	
	} 
	epnt = udev->ep_out[epnum];
	if(!epnt) {
		fprintf(stderr,"Accessing nonexisting endpoint %d\n",epnum);
		return;
	}
	if(udev->ta_state != STATE_IDLE) {
		fprintf(stderr,"Outpid received in wrong transaction state\n");
		return;
	}
	if(epnt->type == EPNT_TYPE_ISO) {
		udev->ta_state = STATE_DO_ISOCHO;
	} else if((dev->speed != USB_SPEED_HIGH) && 
		((epnt->type == EPNT_BULK) || (epnt->type == EPNT_CONTROL))) {
		udev->ta_state = STATE_DO_BCINTO;
	} else if((dev->speed == USB_SPEED_HIGH) &&
		((epnt->type == EPNT_BULK) || (epnt->type == EPNT_CONTROL))) {
		udev->ta_state = STATE_HS_BCO;
	} else if(epnt->type == EPNT_TYPE_INT) {
		udev->ta_state = STATE_DO_BCINTO;
	} else {
		fprintf(stderr,"usbdevice: Unhandled case in device state machine\n");
	}	
	udev.token.pid = pid;
	udev.token.addr = addr;
	udev.token.epnum = epnum;
}  

static void
pid_in(UsbDevice *udev,UsbPacket *packet) 
{
	UsbDev_Endpoint *epnt;
	uint8_t addr = packet->addr & 0x7f;
	uint8_t endp = token->endp & 0xf;
	uint8_t crc5 = token->crc5 & 0x1f; 
	int result;

	/* Is this for me ? */	
	if(addr != udev->addr) {
		return;	
	} 
	epnt = udev->ep_in[epnum];
	if(!epnt) {
		fprintf(stderr,"USB device: accessing nonexisting endpoint %d\n",epnum);
		return;
	}
	if(udev->ta_state != STATE_IDLE) {
		fprintf(stderr,"Outpid received in wrong transaction state\n");
		return;
	}
	if(epnt->type == EPNT_TYPE_ISO) {
		udev->ta_state = STATE_ISOCHI;
	} else {
		udev->ta_state = STATE_BCINTI;
	}
	udev.token.pid = pid;
	udev.token.addr = addr;
	udev.token.epnt = addr;
	result = epnt->fromDevice(eptn,replypacket);
	if(result > 0) {
		/* send_data(udev,replypacket); */
		udev->ta_state = STATE_BCINTINTI_DEV_RESP;
	} else if(result == -NAK) {
		udev->ta_state = STATE_IDLE;
		issue_nack(udev);
	} else if(result == -STALL) {
		udev->ta_state = STATE_IDLE;
		issue_stall(udev);
	}	
} 

static void
pid_setup(UsbDevice *udev,UsbPacket *packet) 
{
	UsbDev_Endpoint *epnt;
	uint8_t addr = packet->addr & 0x7f;
	uint8_t epnum = packet->endp & 0xf;
	uint8_t crc5 = token->crc5 & 0x1f; 
	if(addr != udev->addr) {
		return;	
	} 
	epnt = udev->ep_out[epnum];
	if(!epnt) {
		fprintf(stderr,"Accessing nonexisting endpoint %d\n",epnum);
		return;
	}
	if(epnt->type != EPNT_TYPE_CTRL) {
		return;
	}
	if(udev->ta_state != STATE_IDLE) {
		fprintf(stderr,"UsbDevice: SETUP pid received in wrong state\n");
		return;
	}
	if(epnt->type == EPNT_TYPE_ISO) {
		udev->ta_state = STATE_DO_ISOCHO;
	} else if(epnt->type == EPNT_TYPE_INT) {

		udev->ta_state = STATE_DO_BCINTO;
	} else if((udev->speed != USB_SPEED_HIGH) &&
		((epnt->type == EPNT_TYPE_BULK) || (epnt->type == EPNT_TYPE_CTRL))) {

		udev->ta_state = STATE_DO_BCINTO;
	} else if((udev->speed == USB_SPEED_HIGH) &&
		((epnt->type == EPNT_TYPE_BULK) || (epnt->type == EPNT_TYPE_CTRL))) {
		udev->ta_state = STATE_HS_BCO
	} else {
		fprintf(stderr,"UsbDevice: wrong SETUP pid packet\n");
	}
	udev.token.pid = pid;
	udev.token.addr = addr;
	udev.token.epnum = epnum;
}

static void
pid_sof(UsbDevice *udev,UsbPacket *packet) 
{

}

static void 
pid_data_0_1(UsbDevice *udev,UsbPacket *packet,int toggle) 
{
	UsbDev_Endpoint *epnt;
	uint8_t epnum = udev.token.epnum & 0xf;
	int result;
	if(udev->ta_state == STATE_BCINTO_DEV_WAIT_ODATA) {
		epnt = udev->ep_out[epnum];
		/* check_crc16(packet); */
		if(!check_toggle(epnt,toggle)) {
			issue_packet_ack(udev);
		} else {
			epnt->toggle ^= 1;
			result = epnt->toDevice(epnt,packet->data,packet->datalen);
			if(result == NOSPACE) {
				issue_packet_nak(udev);
			} else if(result == TROUBLE) {
				issue_packet_stall(udev);
			} else {
				issue_packet_ack(udev);
			} 
		}
	}
}

static void
pid_ack(UsbDevice *udev,UsbPacket *packet) 
{
	/* Page 225 */
	if(udev->ta_state == STATE_BCINTI_DEV_RESP) {
		epnt->doNext(epnt);	
		udev->ta_state = STATE_IDLE;
	} else {
		fprintf(stderr,"pid_ack: something is missing in state machine\n");
	}
}

static void
pid_nack(UsbDevice *udev,UsbPacket *packet) 
{
	if(udev->ta_state == STATE_BCINTI_DEV_RESP) {
		//epnt->DontNextData(udev);	/* No action required */
		udev->ta_state = STATE_IDLE;
	}

}

static void
pid_stall(UsbDevice *udev,UsbPacket *packet) 
{
	if(udev->ta_state == STATE_BCINTI_DEV_RESP) {
		fprintf(stderr,"STALL shouldn't happen in state %d\n",udev->ta_state\n");
	}

}

int
UsbDevice_Xmit(UsbDevice *udev,UsbPacket *packet) 
{
	uint8_t pid = packet->pid;
	switch(pid) {
		/* Token packets */
		case USB_PID_OUT:
			pid_out(udev,packet);
			break;

		case USB_PID_IN:
			pid_in(udev,packet);
			break;

		case USB_PID_SOF:
			pid_sof(udev,packet);
			break;

		case USB_PID_SETUP:
			break;

		/* Data packets */
		case USB_PID_DATA0:
			pid_data_0_1(udev,packet,0);
			break;

		case USB_PID_DATA1:
			pid_data_0_1(udev,packet,1);
			break;

		case USB_PID_DATA2:
			break;
		case USB_PID_MDATA:
			break;

		/* Handshake packets */
		case USB_PID_ACK:
			pid_ack(udev,packet);
			break;
		case USB_PID_NACK:
			pid_nack(udev,packet);
			break;
		case USB_PID_STALL:
			pid_stall(udev,packet);
			break;
		case USB_PID_NYET:
			break;
		case USB_PID_PRE:
			break;

		/* Special packets */
		case USB_PID_ERR:
			break;
		case USB_PID_SPLIT:
			break;
		case USB_PID_PING:
			break;
		case USB_PID_RESERVED:
			break;

	} 
}  

UsbDevice *
UsbDevice_New(void) {
	UsbDevice *udev = malloc(sizeof(UsbDevice));
	if(!udev) {
		fprintf(stderr,"Out of memory allocating usb device\n");
		exit(1);
	}
	udev->state = USB_STATE_NOTATTACHED;
}

