/*
 * -------------------------------------------------------------
 *
 * Emulation on Logical Signal Level 
 * (C) 2004  Lightmaze Solutions AG
 *   Author: Jochen Karrer
 *
 * Status:
 *	Basically working, needs cleanup
 *
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * -------------------------------------------------------------
 */

#include <stdint.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <signode.h>
#include <xy_hash.h>
#include "compiler_extensions.h"
#include <stdarg.h>

static XY_HashTable signode_hash;
static SigStamp g_stamp=0;
static int active_trace_deleted_flag=0;

/*
 * ----------------------------------------------------
 * SigNode_Find
 * 	Find a signal by its name in the hash table
 * ----------------------------------------------------
 */

SigNode *
SigNode_Find(const char *format,...) {
	char name[512];
	va_list ap;
	XY_HashEntry *entryPtr;
	SigNode *signode;
	va_start (ap, format);	
	vsnprintf(name,512,format,ap);
	va_end(ap);
	entryPtr = XY_FindHashEntry(&signode_hash,name);
	if(!entryPtr) {
		return NULL;
	}
	signode = XY_GetHashValue(entryPtr);	
	return signode;
}

/*
 * -------------------------------------------------------------
 * Create a new Signal and enter its name into the Hash Table
 * The default value is SIG_OPEN
 * -------------------------------------------------------------
 */
SigNode *
SigNode_New(const char *format,...) {
	char name[512];
	va_list ap;
	SigNode *signode;
	int isnew;
	va_start (ap, format);	
	vsnprintf(name,512,format,ap);
	va_end(ap);
	if((signode=SigNode_Find(name))) {
		fprintf(stderr,"Node \"%s\" already exists\n",name);
		return NULL;
	}	
	signode=malloc(sizeof(SigNode));
	if(!signode) {
		fprintf(stderr,"Out of memory\n");
		exit(1456);
	}
	memset(signode,0,sizeof(SigNode));
	signode->name=strdup(name);
	if(!signode->name) {
		fprintf(stderr,"Can not allocate name \"%s\"\n",name);
		exit(3245);
	}
	signode->hash_entry=XY_CreateHashEntry(&signode_hash,signode->name,&isnew);
	if(!signode->hash_entry) {
		fprintf(stderr,"Can not create hash entry for %s\n",name);
		free(signode->name);
		free(signode);
		return NULL;
	}
	XY_SetHashValue(signode->hash_entry,signode);
	signode->selfval = SIG_OPEN;
	return signode;
}


/*
 * ---------------------------------------------
 * Calculate one Value from two signal values
 * ---------------------------------------------
 */
static int8_t 
calculate_sigval(int val1,int val2) 
{
	if(val1 == SIG_OPEN) {
		return val2;
	} else if(val1==SIG_LOW) {
		if(val2==SIG_HIGH) {
			return SIG_LOW | SIG_ILLEGAL;
		}
		return SIG_LOW;
	} else if(val1==SIG_HIGH) {
		if(val2==SIG_LOW) {
			return SIG_HIGH | SIG_ILLEGAL;
		}
		return SIG_HIGH;
	} else if(val1 == SIG_PULLUP) {
		if(val2==SIG_OPEN) {
			return val1;
		} else if (val2==SIG_LOW) {
			return val2;
		} else if(val2==SIG_HIGH) {
			return val2;
		} else if (val2==SIG_PULLDOWN) {
			return SIG_PULLDOWN | SIG_ILLEGAL;
		} else if (val2==SIG_PULLUP) {
			return SIG_PULLUP;
		} 
	} else if(val1 == SIG_PULLDOWN) {
		if(val2==SIG_OPEN) {
			return val1;
		} else if (val2==SIG_LOW) {
			return val2;
		} else if(val2==SIG_HIGH) {
			return val2;
		} else if (val2==SIG_PULLDOWN) {
			return SIG_PULLDOWN;
		} else if (val2==SIG_PULLUP) {
			return SIG_PULLUP | SIG_ILLEGAL;
		} 
	}
	return SIG_OPEN;
}

static int8_t sigval_tab[256];

static void 
init_sigval_tab(void) 
{
	int i;
	for(i=0;i<256;i++) {
		int val1 = i & 0xf;
		int val2 = (i>>4)&0xf;
		sigval_tab[i]=calculate_sigval(val1,val2);
	}
}

static inline int 
lookup_sigval(int val1,int val2) 
{
	uint8_t index = (val1 & 0xf) | (val2<<4);
	int8_t result = sigval_tab[index];
	if(unlikely(result<0)) {
		fprintf(stderr,"Signal level conflict\n");
	}
	return result & 0xf;
}

/*
 * -----------------------------------------------
 * Recursive
 * -----------------------------------------------
 */

static int
SigMeassure(SigNode *sig,SigStamp stamp,int sigval) 
{
	SigLink *cursor;
	sig->stamp = stamp;
	for(cursor=sig->linkList;cursor;cursor=cursor->next) {
		SigNode *partner = cursor->partner;
		if(partner->stamp!=stamp) {
			sigval = lookup_sigval(sigval,partner->selfval);
			if((sigval==SIG_LOW) || (sigval==SIG_HIGH)) {
				return sigval;
			}
			sigval = SigMeassure(partner,stamp,sigval);
		}
	}
	if(sigval == SIG_PULLUP) {
		return SIG_HIGH;
	} else if(sigval == SIG_PULLDOWN) {
		return SIG_LOW;
	}
	return sigval;
}

/*
 * -------------------------------------------------
 * Todo: Traces can only delete itself
 * from TraceProcs 
 * -------------------------------------------------
 */
static void 
InvokeTraces(SigNode *signode) {
	SigTrace *trace;
	SigTrace *next;
	for(trace=signode->sigTraceList;trace;trace=next) {
		next = trace->next;	
		/* Avoid recursion */
		if(!trace->isactive) {
			trace->isactive++;
			trace->proc(signode,signode->propval,trace->clientData);
			if(active_trace_deleted_flag) {
				active_trace_deleted_flag = 0;
			} else {
				trace->isactive--;
			}
		} else {
			// fprintf(stderr,"Warning: Signal trace recursion\n");
		}
	}
}
/*
 * -------------------------------------------------
 * Propagate a meassured value to 
 * all connected nodes
 * -------------------------------------------------
 */
static int
SigPropagate(SigNode *sig,SigStamp stamp,int sigval) 
{
	SigLink *cursor;
	int invoke_traces=0;
	sig->stamp = stamp;
	if(sig->propval!=sigval) {
		sig->propval = sigval;
		invoke_traces=1;
	} 
	for(cursor=sig->linkList;cursor;cursor=cursor->next) {
		SigNode *partner = cursor->partner;
		if(partner->stamp!=stamp) {
			sigval = SigPropagate(partner,stamp,sigval);
		}
	}
	if(invoke_traces) {
		InvokeTraces(sig);
	}
	return sigval;
}

static int
update_sigval(SigNode *signode) 
{
	int propval = SigMeassure(signode,++g_stamp,signode->selfval);
	propval = SigPropagate(signode,++g_stamp,propval);
	return propval;
}
/*
 * ---------------------------------------------------
 * Set a Signal 
 * ---------------------------------------------------
 */
int 
SigNode_Set(SigNode *signode,int sigval) 
{
	if(sigval==signode->selfval) {
		return signode->propval;
	}
	//fprintf(stderr,"Propagate new %d, old %d ",sigval,signode->selfval); //jk
	signode->selfval = sigval;		
	return update_sigval(signode);
}

/*
 * ---------------------------------
 * Link two signals (bidirectional
 * ---------------------------------
 */
int
SigNode_Link(SigNode *sig1,SigNode *sig2) 
{
	SigLink *link1,*link2;
	link1 = malloc(sizeof(SigLink));
	link2 = malloc(sizeof(SigLink));
	if(!link1 || !link2) {
		fprintf(stderr,"Out of memory in %s %d\n",__FILE__,__LINE__);
		exit(1490);
	}
	link1->partner = sig2;
	link1->next=sig1->linkList;
	sig1->linkList = link1;

	link2->partner = sig1;
	link2->next=sig2->linkList;
	sig2->linkList = link2; 
	update_sigval(sig1);
	return 0;
}

int
SigName_Link(const char *name1,const char *name2) 
{
	SigNode *sig1,*sig2;
	sig1 = SigNode_Find(name1);
	sig2 = SigNode_Find(name2);
	if(!sig1) {
		fprintf(stderr,"Node not found: \"%s\"\n",name1);
		sleep(1);
		return -1;
	}
	if(!sig2) {
		fprintf(stderr,"Node not found: \"%s\"\n",name2);
		sleep(1);
		return -2;
	}
	return SigNode_Link(sig1,sig2);
}
/*
 * ---------------------------------
 * Remove pair of Links 
 * ----------------------------------
 */
static int 
Sig_RemoveLink(SigNode *sig,SigLink *link) 
{
	SigNode *partner = link->partner;
	SigLink *cursor,*prev=NULL;
	for(cursor=sig->linkList;cursor;prev=cursor, cursor=cursor->next) {
		if(cursor==link) {
			break;
		}
	}	
	if(!cursor) {
		fprintf(stderr,"Bug: Link not found\n");
		return 0;
	}
	if(prev) {
		prev->next = cursor->next;
	} else {
		sig->linkList = cursor->next;
	}	
	free(cursor);
	prev=NULL;
	for(cursor=partner->linkList;cursor;prev=cursor, cursor=cursor->next) {
		if(cursor->partner == sig) {
			break;
		}
	}
	if(!cursor) {
		fprintf(stderr,"Bug: Backlink not found\n");
		return -1;
	}
	if(prev) {
		prev->next = cursor->next;
	} else {
		partner->linkList = cursor->next;
	}	
	update_sigval(sig);
	update_sigval(partner);
	free(cursor);
	return 1;
}

int
SigName_RemoveLink(const char *name1,const char *name2) 
{
	SigNode *sig1,*sig2;
	sig1 = SigNode_Find(name1);
	sig2 = SigNode_Find(name2);
	if(!sig1) {
		fprintf(stderr,"Node not found: \"%s\"\n",name1);
		sleep(1);
		return -1;
	}
	if(!sig2) {
		fprintf(stderr,"Node not found: \"%s\"\n",name2);
		sleep(1);
		return -2;
	}
	return SigNode_RemoveLink(sig1,sig2);
}

/*
 * --------------------------------------------------------------------
 * RemoveLink
 *
 * Return Value: 1 if a link was removed
 *		 0 if no link was removed 
 *		-1 when consistency check failed
 * --------------------------------------------------------------------
 */
int
SigNode_RemoveLink(SigNode *sig1,SigNode *sig2) 
{
	SigLink *cursor, *prev=NULL;
	for(cursor=sig1->linkList;cursor;prev=cursor, cursor=cursor->next) {
                if(cursor->partner == sig2) {
                        break;
                }
        }
	if(!cursor) {
		return 0;
	}
	if(prev) {
		prev->next = cursor->next;
	} else {
		sig1->linkList = cursor->next;
	}	
	for(cursor=sig2->linkList;cursor;prev=cursor, cursor=cursor->next) {
                if(cursor->partner == sig1) {
                        break;
                }
        }
	if(!cursor) {
		fprintf(stderr,"Bug: Unidirectional signal link\n");
		return -1;
	}
	if(prev) {
		prev->next = cursor->next;
	} else {
		sig2->linkList = cursor->next;
	}	
	update_sigval(sig1);
	update_sigval(sig2);
	return 1;
}
/*
 * ---------------------------------------
 * Remove all links from a Signode
 * ---------------------------------------
 */
static void 
SigNode_UnLink(SigNode *sig) 
{
	while(sig->linkList) {
		Sig_RemoveLink(sig,sig->linkList); 
	}
}

void
SigNode_Delete(SigNode *signode) {
	
	SigNode_UnLink(signode);
	XY_DeleteHashEntry(&signode_hash,signode->hash_entry);
	free(signode->name);
	free(signode);
}

 
SigTrace *
SigNode_Trace(SigNode *node,SigTraceProc *proc,void *clientData) {
	SigTrace *trace;
	trace=malloc(sizeof(SigTrace));
	if(!trace) {
		fprintf(stderr,"Out of memory\n");
		exit(3408);
	}
	memset(trace,0,sizeof(SigTrace));
	trace->clientData = clientData;
	trace->proc = proc;
	trace->next=node->sigTraceList;
	node->sigTraceList=trace;
	trace->isactive = 0;
	return trace;
}

int
SigNode_Untrace(SigNode *node,SigTrace *trace) {
	SigTrace *cursor,*prev;
	for(prev=0,cursor=node->sigTraceList;cursor;prev=cursor,cursor=cursor->next) {
		if(cursor == trace) {
			break;
		}
	}
	if(cursor) {
		if(prev) {
			prev->next=cursor->next;
		} else {
			node->sigTraceList=cursor->next;
		}	
		if(cursor->isactive) {
			active_trace_deleted_flag = 1;
		}
		free(cursor);
		return 0;
	} else {
		fprintf(stderr,"Bug: Deleting non existing trace\n");
		return -1;
	}
}


/*
 * -----------------------------------------------
 * Return the dominant signal node. 
 * Mainly used for debugging purposes
 * -----------------------------------------------
 */
static SigNode *
_SigNode_FindDominant(SigNode *sig,SigStamp stamp) 
{
	SigLink *cursor;
	sig->stamp = stamp;
	if((sig->selfval == SIG_HIGH) || (sig->selfval == SIG_LOW)) {
		return sig;
	}
	for(cursor=sig->linkList;cursor;cursor=cursor->next) {
		SigNode *partner = cursor->partner;
		if(partner->stamp!=stamp) {
			return _SigNode_FindDominant(partner,stamp);
		}
	}
	return NULL;
}

SigNode *
SigNode_FindDominant(SigNode *sig)  
{
	return _SigNode_FindDominant(sig,++g_stamp);
}
static void 
_SigNode_Dump(SigNode *sig,SigStamp stamp) 
{
	SigLink *cursor;
	sig->stamp = stamp;
	fprintf(stderr,"node %s self %d, prop %d\n",SigName(sig),sig->selfval,sig->propval);
	for(cursor=sig->linkList;cursor;cursor=cursor->next) {
		SigNode *partner = cursor->partner;
		if(partner->stamp!=stamp) {
			return _SigNode_Dump(partner,stamp);
		}
	}
}

void
SigNode_Dump(SigNode *sig)  
{
	_SigNode_Dump(sig,++g_stamp);
}

/*
 * --------------------------------------------
 * Setup hash tables for the signal table 
 * and create GND and VCC signals
 * --------------------------------------------
 */
void
SignodesInit() {
	SigNode *node;
	XY_InitHashTable(&signode_hash,XY_STRING_KEYS,1024);

	init_sigval_tab();
	node=SigNode_New("GND");
	if(!node) {
		fprintf(stderr,"Can not create GND\n");
		exit(43275);
	}
	SigNode_Set(node,SIG_LOW);
	node=SigNode_New("VCC");
	if(!node) {
		fprintf(stderr,"Can not create VCC\n");
		exit(43275);
	}
	SigNode_Set(node,SIG_HIGH);
	return;
}
