/*
 * ----------------------------------------------------
 *
 * Emulation of the NS9750 USB Controller 
 * (C) 2004  Lightmaze Solutions AG
 *   Author: Jochen Karrer
 *
 * Status:
 *	Enabling and disabling the OHCI host controller
 *	works. 
 *	Interrupt status and enable is implemented but
 *	untested
 *
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * ----------------------------------------------------
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <bus.h>
#include <ns9750_bbus.h>
#include <ns9750_usb.h>
#include <usb_ohci.h>

#if 0
#define dprintf(x...) { fprintf(stderr,x); }
#else
#define dprintf(x...)
#endif

typedef struct NS9750Usb {
	uint32_t gctrl;
	uint32_t dctrl;
	uint32_t gien;
	uint32_t gist;
	uint32_t dev_ipr;
	BusDevice bdev;	
	BusDevice *ohcidev;
	int interrupt_posted;
} NS9750Usb;

void
usb_update_interrupts(NS9750Usb *usb) {
	uint32_t mask=0;
	if(usb->gien & NS9750_USB_GIEN_GBL_EN ) {
		mask=0xffffffff;
	}
	if(!(usb->gien & NS9750_USB_GIEN_GBL_DMA)) {
		mask=mask & 0x1fff;
	}
        if(usb->gist & usb->gien & mask) {
		if(!usb->interrupt_posted) {
                	BBus_PostIRQ(BB_IRQ_USB);
			usb->interrupt_posted=1;
		}
        } else {
		if(usb->interrupt_posted) {
                	BBus_UnPostIRQ(BB_IRQ_USB);
			usb->interrupt_posted=0;
		}
        }
}

/*
 * ---------------------------------------------------------
 * USB Controller has Bridge functionality for Interrupts
 * ---------------------------------------------------------
 */
static void
Usb_PostIRQ(BusDevice *bdev,int subint) {
	NS9750Usb *usb= bdev->owner;
        usb->gist |=(1<<subint);
	usb_update_interrupts(usb);
}

static void
Usb_UnPostIRQ(BusDevice *bdev,int subint) {
	NS9750Usb *usb= bdev->owner;
        usb->gist &= ~(1<<subint);
	usb_update_interrupts(usb);
}

/*
 * -----------------------------------------------------------
 * Documentation about the GCTRL DRST Bits seems to be wrong
 * -----------------------------------------------------------
 */

static uint32_t 
usb_gctrl_read(void *clientData,uint32_t address,int rqlen) {
	NS9750Usb *usb=clientData;
	if(usb->gctrl & NS9750_USB_GCTRL_HSTDEV) {
		return  NS9750_USB_GCTRL_HSTDEV | 0x823;
	} else {
		return usb->gctrl | 0x602;
	}
}
static void 
usb_gctrl_write(void *clientData,uint32_t value,uint32_t address,int rqlen)  {
	NS9750Usb *usb=clientData;
	uint32_t diff=value^usb->gctrl;
	dprintf("GCTRL Write value %08x\n",value);
	usb->gctrl=value;
	if(diff & (NS9750_USB_GCTRL_HSTDEV )) {
		if(((value&NS9750_USB_GCTRL_HRST) == 0)) {
			OhciHC_Enable(usb->ohcidev);	
		} else {
			// Does a reset of all registers
			OhciHC_Disable(usb->ohcidev);	
		}
	}
	return;
}
static uint32_t 
usb_dctrl_read(void *clientData,uint32_t address,int rqlen) {
	NS9750Usb *usb=clientData;
	return usb->dctrl;
}
static void 
usb_dctrl_write(void *clientData,uint32_t value,uint32_t address,int rqlen)  {
	NS9750Usb *usb=clientData;
	usb->dctrl=value;
	return;
}
static uint32_t 
usb_gien_read(void *clientData,uint32_t address,int rqlen) {
	NS9750Usb *usb=clientData;
	return usb->gien;
}
static void 
usb_gien_write(void *clientData,uint32_t value,uint32_t address,int rqlen)  {
	NS9750Usb *usb=clientData;
	usb->gien=value;
	usb_update_interrupts(usb);
	return;
}
static uint32_t 
usb_gist_read(void *clientData,uint32_t address,int rqlen) {
	NS9750Usb *usb=clientData;
	if(usb->gist&0x07ffc000) {
		return usb->gist | NS9750_USB_GIEN_GBL_DMA;
	} else {
		return usb->gist;
	}
}
/*
 * -----------------------------------------------
 * Interrupt Status is toggled when writing a 1
 * -----------------------------------------------
 */
static void 
usb_gist_write(void *clientData,uint32_t value,uint32_t address,int rqlen)  {
	NS9750Usb *usb=clientData;
	uint32_t xor = value & 0x3c2;
	// what to do with the other fields ?	
	usb->gist^=xor;
	usb_update_interrupts(usb);
}

static uint32_t 
usb_dev_ipr_read(void *clientData,uint32_t address,int rqlen) {
	NS9750Usb *usb=clientData;
	return usb->dev_ipr;
}
static void 
usb_dev_ipr_write(void *clientData,uint32_t value,uint32_t address,int rqlen)  {
	NS9750Usb *usb=clientData;
	usb->dev_ipr=value;
}

/*
 * ------------------------------------------------------------------
 *  This ignores all given Baseaddresses because 
 *  the addresses are unchangeable
 * ------------------------------------------------------------------
 */
static void
NS9750Usb_Map(void *owner,uint32_t base,uint32_t mapsize,uint32_t flags) {
	NS9750Usb *usb=owner;
	if(base!=NS9750_USB_BASE) {
		fprintf(stderr,"Error: NS9750 USB is not remappable to address %08x\n",base);
		exit(3245);
	}
	IOH_New32(NS9750_USB_GCTRL,usb_gctrl_read,usb_gctrl_write,usb);
	IOH_New32(NS9750_USB_DCTRL,usb_dctrl_read,usb_dctrl_write,usb);
	IOH_New32(NS9750_USB_GIEN,usb_gien_read,usb_gien_write,usb);
	IOH_New32(NS9750_USB_GIST,usb_gist_read,usb_gist_write,usb);
	IOH_New32(NS9750_USB_DEV_IPR,usb_dev_ipr_read,usb_dev_ipr_write,usb);
}

static void
NS9750Usb_UnMap(void *owner,uint32_t base,uint32_t mapsize) {
	IOH_Delete32(NS9750_USB_GCTRL);
	IOH_Delete32(NS9750_USB_DCTRL);
	IOH_Delete32(NS9750_USB_GIEN);
	IOH_Delete32(NS9750_USB_GIST);
	IOH_Delete32(NS9750_USB_DEV_IPR);
}

BusDevice *
NS9750Usb_New(const char *name) {
	NS9750Usb *usb;
	usb=malloc(sizeof(NS9750Usb));
	if(!usb) {
		fprintf(stderr,"Out of memory\n");
		exit(5769);
	}
	memset(usb,0,sizeof(NS9750Usb));
	//usb->gctrl = 0xe03;	 // manual says this
	usb->gctrl = 0x823;	 // real device says this
	usb->dctrl = 0x0;	
	usb->gien=0;

	// GIST startup value with device 0xa1f, without 0x801, reset value 0x200
	usb->gist=0xa1f;
	usb->dev_ipr=0;
	usb->bdev.first_mapping=NULL;
        usb->bdev.Map=NS9750Usb_Map;
        usb->bdev.UnMap=NS9750Usb_UnMap;
        usb->bdev.owner=usb;
        usb->bdev.hw_flags=MEM_FLAG_WRITABLE|MEM_FLAG_READABLE;
	usb->bdev.postIRQ = Usb_PostIRQ; 
	usb->bdev.unPostIRQ = Usb_UnPostIRQ;
	usb->bdev.read32 = Bus_Read32; 
	usb->bdev.write32 = Bus_Write32;
	usb->ohcidev=OhciHC_New("ns9750_ohci",&usb->bdev,NS9750_USB_IRQ_OHCI);	
	if(!usb->ohcidev) {
		fprintf(stderr,"Can't create ohcidev\n");
		exit(4324);
	}
	/* We add the mapping here because we always have the same address */
	Mem_AreaAddMapping(&usb->bdev,NS9750_USB_BASE,0x20000,MEM_FLAG_WRITABLE|MEM_FLAG_READABLE);
	Mem_AreaAddMapping(usb->ohcidev,NS9750_OHCI_BASE,0x20000,MEM_FLAG_WRITABLE|MEM_FLAG_READABLE);
	fprintf(stderr,"NS9750 OHCI USB Host Controller created\n");
        return &usb->bdev;
}
