;;; s7 test suite
;;;
;;; sources include
;;;   clisp test suite
;;;   sbcl test suite
;;;   Paul Dietz's CL test suite (gcl/ansi-tests/*)
;;;   R Kelsey, W Clinger, and J Rees r5rs.html (and r7rs.html)
;;;   A Jaffer's r4rstest.scm (the inspiration for this...)
;;;   guile test suite
;;;   gauche test suite
;;;   sacla test suite
;;;   Kent Dybvig's "The Scheme Programming Language"
;;;   Brad Lucier and Peter Bex
;;;   GSL tests
;;;   Abramowitz and Stegun, "Handbook of Mathematical Functions"
;;;   the arprec package of David Bailey et al
;;;   Maxima, William Schelter et al
;;;   N Higham, "Accuracy and Stability of Numerical Algorithms"
;;;   various mailing lists and websites (see individual cases below)

(unless (defined? 'full-s7test)         ; snd-test has a variable named full-test, and includes this file
  (define full-s7test #f))              ; this includes some time-consuming stuff
(define with-bignums (provided? 'gmp))  ; scheme number has any number of bits
					; we assume s7_double is double, and s7_int is int64_t
                                        ;   a few of the bignum tests assume the default bignum-precision is 128
					;   using a different default affects number->string primarily
(define with-complex (provided? 'complex-numbers))
(define with-windows (provided? 'windows))
(unless (defined? 's7test-exits) (define s7test-exits #t))
(define asan-flags "") ; -fsanitize=address -lasan ") ;-fsanitize=bounds ")

(define username (getenv "USER"))

(unless (defined? 'most-positive-fixnum)
  (define-constant most-positive-fixnum (*s7* 'most-positive-fixnum))
  (define-constant most-negative-fixnum (*s7* 'most-negative-fixnum)))

(set! (*s7* 'print-length) 32) ; old default, changed to 12 to match Snd, 23-Jul-21

;(set! (hook-functions *load-hook*) (list (lambda (hook) (format () "loading ~S...~%" (hook 'name)))))

;;; to loop s7test under gdb until it crashes:
#|
gdb repl
set pagination off
break _exit
commands
run
end
;;; repl.c can randomize *s7* parameters before calling s7_load
|#

;;; ---------------- pure-s7 ----------------
(define pure-s7 (provided? 'pure-s7))
(when pure-s7
  (define (make-polar mag ang)
    (if (and (real? mag) (real? ang))
	(complex (* mag (cos ang)) (* mag (sin ang)))
	(error 'wrong-type-arg "make-polar args should be real")))
  (define make-rectangular complex)

  (define (char-ci=? . chars) (apply char=? (map char-upcase chars)))
  (define (char-ci<=? . chars) (apply char<=? (map char-upcase chars)))
  (define (char-ci>=? . chars) (apply char>=? (map char-upcase chars)))
  (define (char-ci<? . chars) (apply char<? (map char-upcase chars)))
  (define (char-ci>? . chars) (apply char>? (map char-upcase chars)))

  (define (string-ci=? . strs) (apply string=? (map string-upcase strs)))
  (define (string-ci<=? . strs) (apply string<=? (map string-upcase strs)))
  (define (string-ci>=? . strs) (apply string>=? (map string-upcase strs)))
  (define (string-ci<? . strs) (apply string<? (map string-upcase strs)))
  (define (string-ci>? . strs) (apply string>? (map string-upcase strs)))

  (define (list->string lst) (apply string lst))
  (define (list->vector lst) (apply vector lst))

  (define (let->list e)
    (if (let? e)
	(reverse! (map values e))
	(error 'wrong-type-arg "let->list argument should be an environment: ~A" str)))

  (define* (string->list str (start 0) end)
    (if (and (string? str)
	     (integer? start)
	     (not (negative? start))
	     (or (not end)
		 (and (integer? end)
		      (>= end start))))
	(map values (substring str start (or end (length str))))
	(error 'wrong-type-arg "string->list argument should be a string: ~A" str)))

  (define (string-length str)
    (if (string? str)
	(length str)
	(if (and (openlet? str)
		 (defined? 'string-length str #t))
	    ((let-ref str 'string-length) str)
	    (error 'wrong-type-arg "string-length argument should be a string: ~A" str))))

  (define (string-fill! str chr . args)
    (if (string? str)
	(apply fill! str chr args)
	(if (and (openlet? str)
		 (defined? 'string-fill str #t))
	    (apply (let-ref str 'string-fill!) str chr args)
	    (error 'wrong-type-arg "string-fill! argument should be a string: ~A" str))))

  (define* (vector->list vect (start 0) end)
    (if (and (vector? vect)
	     (integer? start)
	     (not (negative? start))
	     (or (not end)
		 (and (integer? end)
		      (>= end start))))
	(if start
	    (let ((stop (or end (length vect))))
	      (if (= start stop)
		  ()
		  (map values (subvector vect start stop))))
	    (map values vect))
	(error 'wrong-type-arg "vector->list argument should be a vector: ~A" vect)))

  (define (vector-length vect)
    (if (vector? vect)
	(length vect)
	(error 'wrong-type-arg "vector-length argument should be a vector: ~A" vect)))

  (define (vector-fill! vect val . args)
    (if (vector? vect)
	(apply fill! vect val args)
	(error 'wrong-type-arg "vector-fill! argument should be a vector: ~A" str)))

  (define (vector-append . args)
    (if (null? args)
	#()
	(if (vector? (car args))
	    (apply append args)
	    (error 'wrong-type-arg "vector-append arguments should be vectors: ~A" args))))

  (define* (char-ready? p)
    (and p (not (input-port? p))
	 (error 'wrong-type-arg "char-ready? arg should be an input port")))

  (define (set-current-output-port port) (error 'undefined-function "set-current-output-port is not in pure-s7"))
  (define (set-current-input-port port) (error 'undefined-function "set-current-input-port is not in pure-s7"))

  (define (exact? n)
    (if (number? n)
	(rational? n)
	(if (and (openlet? n)
		 (defined? 'exact? n #t))
	    ((let-ref n 'exact?) n)
	    (error 'wrong-type-arg "exact? argument should be a number: ~A" n))))

  (define (inexact? x)
    (if (number? x)
	(not (rational? x))
	(if (and (openlet? n)
		 (defined? 'inexact? n #t))
	    ((let-ref n 'inexact?) n)
	    (error 'wrong-type-arg "inexact? argument should be a number: ~A" x))))

  (define (inexact->exact x)
    (if (not (number? x))
	(error 'wrong-type-arg "inexact->exact argument should be a number: ~A" x)
	(if (rational? x)
	    x
	    (rationalize x))))

  (define (exact->inexact x)
    (if (number? x)
	(* x 1.0)
	(error 'wrong-type-arg "exact->inexact argument should be a number: ~A" x)))

  (define (integer-length i)
    (if (integer? i)
	(if (memv i '(9223372036854775807 -9223372036854775808))
	    63
	    (ceiling (log (if (< i 0) (- i) (+ i 1)) 2)))
	(if (and (openlet? i)
		 (defined? 'integer-length i #t))
	    ((let-ref i 'integer-length) i)
	    (error 'wrong-type-arg "integer-length argument should be an integer: ~A" x))))

  (define-macro (call-with-values producer consumer) `(,consumer (,producer)))

  (define-macro (multiple-value-bind vars expression . body)   ; named "receive" in srfi-8 which strikes me as perverse
    (if (or (symbol? vars) (negative? (length vars)))
	`((lambda ,vars ,@body) ,expression)
	`((lambda* (,@vars . ,(gensym)) ,@body) ,expression)))

  (define-macro (multiple-value-set! vars expr . body)
    (let ((local-vars (map (lambda (n) (gensym)) vars)))
      `((lambda* (,@local-vars . ,(gensym))
	  ,@(map (lambda (n ln) `(set! ,n ,ln)) vars local-vars)
	  ,@body)
	,expr)))

  (define-macro (cond-expand . clauses)
    (letrec ((traverse (lambda (tree)
			 (if (pair? tree)
			     (cons (traverse (car tree))
				   (if (null? (cdr tree)) () (traverse (cdr tree))))
			     (if (memq tree '(and or not else))
				 tree
				 (and (symbol? tree) (provided? tree)))))))
      `(cond ,@(map (lambda (clause)
		      (cons (traverse (car clause))
			    (if (null? (cdr clause)) '(#f) (cdr clause))))
		    clauses))))
  )
;;; ---------------- end pure-s7 ----------------

(define-macro (defmacro name args . body) `(define-macro ,(cons name args) ,@body))
(define-macro (defmacro* name args . body) `(define-macro* ,(cons name args) ,@body))

(define tmp-output-file "tmp1.r5rs")
(define tmp-data-file "test.dat")
(define bold-text (format #f "~C[1m" #\escape))
(define unbold-text (format #f "~C[22m" #\escape))
(set! (hook-functions *unbound-variable-hook*) ())
(set! (hook-functions *missing-close-paren-hook*) ())
(define s7test-output #f) ; if a string, it's treated as a logfile

;(set! (*s7* 'gc-stats) 4) ; 4=stack
;(set! (*s7* 'undefined-identifier-warnings) #t)

;(set! (*s7* 'debug) 2)
;(set! ((funclet trace-in) '*debug-port*) #f)
;(set! (*s7* 'profile) 1)

(define old-stdin *stdin*)
(define old-stdout *stdout*)
(define old-stderr *stderr*)
(define *max-arity* #x20000000)

(define (-s7-stack-top-) (*s7* 'stack-top))

(when full-s7test
  (system "rm libc_s7.*")
  (system "rm libgdbm_s7.*")
  (system "rm libgsl_s7.*")
  (system "rm libm_s7.*"))


;;; --------------------------------------------------------------------------------

(if (and (defined? 'current-time) ; in Snd
	 (defined? 'mus-rand-seed))
    (set! (mus-rand-seed) (current-time)))

(define (ok? otst ola oexp)
  (let ((result (catch #t ola
		       (lambda (type info)
			 (if (not (eq? oexp 'error))
			     (begin (apply format #t info) (newline)))
			 'error))))
    (if (not (equal? result oexp))
	(format #t "~A: ~A got ~S but expected ~S~%~%" (port-line-number) otst result oexp))))

(define original-test-macro #f)

(unless (defined? 'test)
  (set! original-test-macro #t)
  (define-macro (test tst expected) ;(display tst *stderr*) (newline *stderr*)
    ;; `(ok? ',tst (lambda () (eval-string (format #f "~S" ',tst))) ,expected))
    ;; `(ok? ',tst (lambda () (eval ',tst)) ,expected))
    ;; `(ok? ',tst (lambda () ,tst) ,expected))
    ;; `(ok? ',tst (lambda () (eval-string (object->string ,tst :readable))) ,expected))
    ;; `(ok? ',tst (let () (define (_s7_) ,tst)) ,expected))
    ;; `(ok? ',tst (lambda () (let ((_s7_ #f)) (set! _s7_ ,tst))) ,expected))
    ;; `(ok? ',tst (lambda () (let ((_s7_ ,tst)) _s7_)) ,expected))
    ;; `(ok? ',tst (catch #t (lambda () (lambda* ((_a_ ,tst)) _a_)) (lambda any (lambda () 'error))) ,expected))
    ;; `(ok? ',tst (lambda () (do ((_a_ ,tst)) (#t _a_))) ,expected))
    ;; `(ok? ',tst (lambda () (call-with-exit (lambda (_a_) (_a_ ,tst)))) ,expected))
    ;; `(ok? ',tst (lambda () (values ,tst)) ,expected))
    ;; `(ok? ',tst (lambda () (define (_s7_ _a_) _a_) (_s7_ ,tst)) ,expected))
    ;; `(ok? ',tst (lambda () (define* (_s7_ (_a_ #f)) (or _a_)) (_s7_ ,tst)) ,expected))
    ;; `(ok? ',tst (lambda () (caadr (catch 'receive (lambda () (throw 'receive ,tst)) (lambda any any)))) ,expected))
    ;; `(ok? ',tst (lambda () (stacktrace (- (random 100) 50) (- (random 100) 50) (- (random 100) 50) (- (random 100) 50) (> (random 100) 50)) ,tst) ,expected))
    (list 'ok? (list quote tst) (list-values lambda () tst) expected))
#|
      `(let ((_result_ #f))
	 (define (stest) (set! _result_ ,tst))
	 (catch #t stest
		(lambda args
		  (set! _result_ 'error)))
	 (if (not (equal? _result_ ,expected))
	     (format #t "~A: ~A got ~S but expected ~S~%~%" (port-line-number) ',tst _result_ ,expected))))
|#
    )

(define (tok? otst ola)
  (let* ((data #f)
	 (result (catch #t ola
			(lambda args
			 (set! data args)
			 'error))))
    (if (or (not result)
	    (eq? result 'error))
	(format #t "~A: ~A got ~S ~A~%~%" (port-line-number) otst result (or data "")))))

(define-macro (test-t tst) ;(display tst *stderr*) (newline *stderr*)
  `(tok? ',tst (lambda () ,tst)))

(define-macro (test-e tst op arg) ;(display tst *stderr*) (newline *stderr*)
  `(let ((result (catch #t (lambda () ,tst)
			(lambda args
			 'error))))
     (if (not (eq? result 'error))
	 (format #t "~A: (~A ~S) got ~S but expected 'error~%~%" (port-line-number) ,op ,arg result))))


(define (op-error op result expected)
  (case op
    ((acosh)
     (/ (magnitude (- (cosh result) (cosh expected)))
	(max 0.001 (magnitude (cosh expected)))))
    ((asin)
     (/ (min (magnitude (- (sin result) (sin expected)))
	     (magnitude (- result expected)))
	(max 0.001 (* 10 (magnitude (sin expected))))))
    ((acos)
     (/ (min (magnitude (- (cos result) (cos expected)))
	     (magnitude (- result expected)))
	(max 0.001 (magnitude (cos expected)))))
    ((asinh)
     (/ (magnitude (- (sinh result) (sinh expected)))
	(max 0.001 (magnitude (sinh expected)))))
    ((atanh)
     (/ (min (magnitude (- (tanh result) (tanh expected)))
	     (magnitude (- result expected)))
	(max 0.001 (magnitude (tanh expected)))))
    ((atan)
     (/ (min (magnitude (- (tan result) (tan expected)))
	     (magnitude (- result expected)))
	(max 0.001 (magnitude (tan expected)))))
    ((cosh)
     (/ (min (magnitude (- result expected))
	     (magnitude (+ result expected)))
	(max 0.001 (magnitude expected))))
    (else (/ (magnitude (- result expected)) (max 0.001 (magnitude expected))))))


;;; relative error (/ (abs (- x res) (abs x)))

(define (number-ok? tst result expected)
  (unless (eqv? result expected)
    (if (or (not (number? result))
	    (not (eq? (nan? expected) (nan? result)))
	    (and (pair? tst)
		 (> (/ (magnitude (- result expected)) (max 0.001 (magnitude expected))) 1e-5)
		 (> (op-error (car tst) result expected) 1e-5)))
      (format #t "~A: ~A got ~A~Abut expected ~A~%~%"
		     (port-line-number) tst result
		     (if (and (rational? result) (not (rational? expected)))
			 (format #f " (~A) " (* 1.0 result))
			 " ")
		     expected))))

(define (nok? otst ola oexp)
  (let ((result (catch #t ola
		       (lambda args
			 'error))))
    (number-ok? otst result oexp)))

(if (not (defined? 'num-test))
    (define-macro (num-test tst expected) ;(display tst *stderr*) (newline *stderr*)
      ;; `(nok? ',tst  (lambda () ,tst) ,expected))
      ;; `(nok? ',tst (let () (define (_s7_) ,tst)) ,expected))
      (list-values 'nok? (list-values quote tst) (list-values lambda () tst) expected)))

(define-macro (num-test-1 proc val tst expected)
  `(let ((result (catch #t (lambda () ,tst)
			(lambda args
			 'error))))
     (number-ok? (list ,proc ,val) result ,expected)))

(define-macro (num-test-2 proc val1 val2 tst expected)
  `(let ((result (catch #t (lambda () ,tst)
			(lambda args
			 'error))))
     (number-ok? (list ,proc ,val1 ,val2) result ,expected)))

(define (string-wi=? s1 s2) ; string=? ignoring white-space
  (let ((iter1 (make-iterator s1))
	(iter2 (make-iterator s2)))
    (let wi-loop ((i1 (iterate iter1)) (i2 (iterate iter2)))
      (if (eq? i1 i2)
	  (or (eq? i1 #<eof>)
	      (wi-loop (iterate iter1) (iterate iter2)))
	  (if (and (char? i1)
		   (char-whitespace? i1))
	      (wi-loop (iterate iter1) i2)
	      (and (char? i2)
		   (char-whitespace? i2)
		   (wi-loop i1 (iterate iter2))))))))

(test (string-wi=? "" "") #t)
(test (string-wi=? "" " ") #t)
(test (string-wi=? "" " a") #f)
(test (string-wi=? "a" " a") #t)
(test (string-wi=? "a " " a") #t)
(test (string-wi=? " a " "a") #t)
(test (string-wi=? " a " " a") #t)
(test (string-wi=? "\n a\n " "a") #t)
(test (string-wi=? "aa" " a") #f)
(test (string-wi=? "aa" " a a ") #t)
(test (string-wi=? "aa" "aa ") #t)

;; ----------------

(define-macro (test-wi tst res)
  `(let ((val ,tst))
     (unless (string-wi=? val ,res)
       (format *stderr* "~A: ~S got ~S but expected ~S~%" (port-line-number) ',tst val ,res))))

;; ----------------

(define (reinvert n op1 op2 arg)
  (let ((body (op2 (op1 arg))))
    (do ((i3 1 (+ i3 1)))
	((= i3 n) body)
      (set! body (op2 (op1 body))))))

(define (recompose n op arg)
  (define (recompose-1 n)
    (if (= n 1)
	(op arg)
	(op (recompose-1 (- n 1)))))
  (recompose-1 n))

(if (setter 'val) (set! (setter 'val) #f)) ; might get here from snd-test

(define _ht_ (make-hash-table))
(define _undef_ (car (with-input-from-string "(#_asdf 1 2)" read)))

;;; --------------------------------------------------------------------------------
;;; before starting, make a test c-object

(unless (defined? 'with-block)
  (define with-block (not (provided? 'windows))))

(if with-block
    (begin
      (call-with-output-file "s7test-block.c"
	(lambda (p)
	  (format p "
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>

#include \"s7.h\"
static s7_scheme *s7;

/* c-object tests */
typedef struct {
  size_t size;
  double *data;
} g_block;

static s7_int g_block_type = 0, g_simple_block_type = 0, g_c_tag_type = 0, g_c_tag1_type = 0, g_cycle_type = 0, block_gc_loc = 0;
static s7_pointer g_block_methods;

static s7_pointer g_block_let(s7_scheme *sc, s7_pointer args)
{
  #define g_block_let_help \"(block-let block) returns the block'e let.\"
  if (s7_c_object_type(s7_car(args)) != g_block_type)
    return(s7_wrong_type_arg_error(sc, \"block-let\", 1, s7_car(args), \"a block\"));
  return(s7_c_object_let(s7_car(args)));
}

static s7_int max_vector_length = 0;
static s7_pointer make_block_1(s7_scheme *sc, s7_int size, bool cleared)
{
  g_block *g;
  s7_pointer new_g;
  if ((size < 0) ||
      (size > max_vector_length))
     return(s7_out_of_range_error(sc, \"make-block\", 1, s7_make_integer(sc, size), \"it should be positive and less than (*s7* 'max-vector-length)\"));
  g = (cleared) ? (g_block *)calloc(1, sizeof(g_block) + (size * sizeof(double))) : (g_block *)malloc(sizeof(g_block) + (size * sizeof(double)));
  g->size = (size_t)size;
  if (g->size > 0)
    g->data = (double *)((void *)g + sizeof(g_block));
  else g->data = NULL;
  new_g = s7_make_c_object(sc, g_block_type, (void *)g);
  s7_c_object_set_let(sc, new_g, g_block_methods);
  s7_openlet(sc, new_g);
  return(new_g);
}

static s7_pointer make_block(s7_scheme *sc, s7_int size) {return(make_block_1(sc, size, true));}
static s7_pointer make_block_raw(s7_scheme *sc, s7_int size) {return(make_block_1(sc, size, false));}

static s7_pointer g_make_block(s7_scheme *sc, s7_pointer args)
{
  #define g_make_block_help \"(make-block size) returns a new block of the given size\"
  s7_pointer arg1 = s7_car(args);
  if (!s7_is_integer(arg1))
    return(s7_wrong_type_arg_error(sc, \"make-block\", 1, arg1, \"an integer\"));
  return(make_block(sc, s7_integer(arg1)));
}

static s7_pointer g_make_simple_block(s7_scheme *sc, s7_pointer args)
{
  #define g_make_simple_block_help \"(make-simple-block size) returns a new simple-block of the given size\"
  g_block *g;
  s7_pointer new_g;
  s7_int size;
  s7_pointer arg1 = s7_car(args);
  if (!s7_is_integer(arg1))
    return(s7_wrong_type_arg_error(sc, \"make-simple-block\", 1, arg1, \"an integer\"));
  size = s7_integer(arg1);
  if ((size < 0) ||
      (size > max_vector_length))
     return(s7_out_of_range_error(sc, \"make-simple-block\", 1, arg1, \"it should be positive and less than (*s7* 'max-vector-length)\"));
  g = (g_block *)calloc(1, sizeof(g_block) + (size * sizeof(double)));
  g->size = (size_t)size;
  if (g->size > 0)
    g->data = (double *)((void *)g + sizeof(g_block));
  else g->data = NULL;
  new_g = s7_make_c_object(sc, g_simple_block_type, (void *)g);
  return(new_g);
}

static s7_pointer g_make_c_tag(s7_scheme *sc, s7_pointer args)
{
  s7_int *tag = malloc(sizeof(s7_int));
  *tag = 23;
  return(s7_make_c_object(sc, g_c_tag_type, (void *)tag));
}

static void g_c_tag_free(void *val)
{
  free(val);
}

static s7_pointer g_make_c_tag1(s7_scheme *sc, s7_pointer args)
{
  s7_int *tag = malloc(sizeof(s7_int));
  *tag = 23;
  return(s7_make_c_object(sc, g_c_tag1_type, (void *)tag));
}

typedef struct {s7_pointer obj;} g_cycle;
static s7_pointer g_make_cycle(s7_scheme *sc, s7_pointer args)
{
  g_cycle *g = (g_cycle *)malloc(sizeof(g_cycle));
  g->obj = s7_car(args);
  return(s7_make_c_object(sc, g_cycle_type, (void *)g));
}

static s7_pointer g_cycle_ref(s7_scheme *sc, s7_pointer args)
{
  g_cycle *g;
  if (s7_list_length(sc, args) != 1)
    return(s7_wrong_number_of_args_error(sc, \"cycle-ref takes 1 argument: ~~S\", args));
  g = (g_cycle *)s7_c_object_value(s7_car(args));
  return(g->obj);
}

static s7_pointer g_cycle_to_list(s7_scheme *sc, s7_pointer args)
{
  g_cycle *g = (g_cycle *)s7_c_object_value(s7_car(args));
  return(s7_cons(sc, g->obj, s7_nil(sc)));
}

static s7_pointer g_cycle_set(s7_scheme *sc, s7_pointer args)
{
  g_cycle *g = (g_cycle *)s7_c_object_value(s7_car(args));
  g->obj = s7_cadr(args);
  return(g->obj);
}

static s7_pointer g_cycle_implicit_set(s7_scheme *sc, s7_pointer args)
{
  g_cycle *g;
  s7_pointer val;
  s7_int index;
  if (s7_list_length(sc, args) != 3)
    return(s7_wrong_number_of_args_error(sc, \"cycle-set! takes 3 arguments: ~~S\", args));
  g = (g_cycle *)s7_c_object_value(s7_car(args));
  if ((!s7_is_integer(s7_cadr(args))) ||
      (s7_integer(s7_cadr(args)) != 0))
    return(s7_out_of_range_error(sc, \"implicit cycle-set!\", 2, s7_cadr(args), \"it should be 0\"));
  g->obj = s7_caddr(args);
  return(g->obj);
}

static s7_pointer g_cycle_copy(s7_scheme *sc, s7_pointer args)
{
  s7_pointer obj = s7_car(args);
  g_cycle *g;
  if (s7_c_object_type(obj) != g_cycle_type)   /* obj might not be a cycle object if destination is one */
    return(s7_f(sc));
  g = (g_cycle *)s7_c_object_value(s7_car(args));
  return(g_make_cycle(sc, s7_list(sc, 1, g->obj)));
}

static void g_cycle_mark(void *val)
{
  s7_mark(((g_cycle *)val)->obj);
}

static void g_cycle_free(void *val)
{
  free(val);
}

static s7_pointer g_to_block(s7_scheme *sc, s7_pointer args)
{
  #define g_block_help \"(block ...) returns a block c-object with the arguments as its contents.\"
  s7_pointer p = args, b;
  size_t len = s7_list_length(sc, args);
  g_block *g = (g_block *)malloc(sizeof(g_block) + (len * sizeof(double)));
  g->size = (size_t)len;
  if (g->size > 0)
    g->data = (double *)((void *)g + sizeof(g_block));
  else g->data = NULL;
  b = s7_make_c_object(sc, g_block_type, (void *)g);
  s7_c_object_set_let(sc, b, g_block_methods);
  s7_openlet(sc, b);
  for (size_t i = 0; i < len; i++, p = s7_cdr(p))
    g->data[i] = s7_number_to_real(sc, s7_car(p));
  /* if (s7_is_openlet(s7_car(p))) g->data[i] = s7_number_to_real(sc, s7_let_ref(sc, p, s7_make_symbol(sc, \"value\"))) */
  return(b);
}

static s7_pointer block_p_d(s7_scheme *sc, s7_double x)
{
  g_block *g = (g_block *)malloc(sizeof(g_block) + sizeof(double));
  s7_pointer new_g;
  g->size = 1;
  g->data = (double *)((void *)g + sizeof(g_block));
  g->data[0] = x;
  new_g = s7_make_c_object(sc, g_block_type, (void *)g);
  s7_c_object_set_let(sc, new_g, g_block_methods);
  s7_openlet(sc, new_g);
  return(new_g);
}

static bool is_NaN(s7_double x) {return(x != x);}
#if __cplusplus
  #define is_inf(x) std::isinf(x)
#else
  #define is_inf(x) isinf(x)
#endif

static char *g_block_display(s7_scheme *sc, void *value)
{
  static s7_pointer ffp = NULL, pl = NULL, oom = NULL;
  g_block *b = (g_block *)value;
  s7_int i, len, old_len, loc, bytes, prec;
  char *buf;
  if (!ffp)
    {
      ffp = s7_make_symbol(sc, \"float-format-precision\");
      pl = s7_make_symbol(sc, \"print-length\");
      oom = s7_make_symbol(sc, \"out-of-memory\");
    }
  prec = s7_integer(s7_let_field_ref(sc, ffp));
  if (prec >= 16) prec = 3;
  len = b->size;
  old_len = s7_integer(s7_let_field_ref(sc, pl));
  if (len > old_len) len = old_len;
  buf = (char *)malloc((len + 1) * 64);
  if (!buf) s7_error(sc, oom, s7_list(sc, 1, s7_make_string(sc, \"unable to allocate string to display block\")));
  buf[0] = (char)0;
  loc = snprintf(buf, (len + 1) * 64, \"(block\");
  for (i = 0; i < len; i++)
    {
      char *flt = (char *)(buf + loc);
      if (is_NaN(b->data[i]))
        bytes = snprintf(flt, 64, \" +nan.0\");
      else
        if (is_inf(b->data[i]))
          bytes = snprintf(flt, 64, \" %cinf.0\", (b->data[i] >= 0.0) ? '+' : '-');
        else bytes = snprintf(flt, 64, \" %.*f\", (int)prec, b->data[i]);
      loc += (bytes > 64) ? 64 : bytes;
    }
  if (b->size > old_len) {buf[loc++] = ' '; buf[loc++] = '.'; buf[loc++] = '.'; buf[loc++] = '.';}
  buf[loc] = ')';
  buf[loc + 1] = 0;
  return(buf);
}

static char *g_block_display_readably(s7_scheme *sc, void *value)
{
  s7_int i, loc, bytes;
  g_block *b = (g_block *)value;
  s7_int len = b->size;
  char *buf = (char *)malloc((len + 1) * 64);
  buf[0] = (char)0;
  loc = snprintf(buf, (len + 1) * 64, \"(block\");
  for (i = 0; i < len; i++)
    {
      char *flt = (char *)(buf + loc);
      if (is_NaN(b->data[i]))
        bytes = snprintf(flt, 64, \" +nan.0\");
      else
        if (is_inf(b->data[i]))
          bytes = snprintf(flt, 64, \" %cinf.0\", (b->data[i] >= 0.0) ? '+' : '-');
        else bytes = snprintf(flt, 64, \" %.16g\", b->data[i]);
      loc += (bytes > 64) ? 64 : bytes;
    }
  buf[loc] = ')';
  buf[loc + 1] = 0;
  return(buf);
}

static s7_pointer g_block_to_string(s7_scheme *sc, s7_pointer args)
{
  s7_pointer obj = s7_car(args);
  s7_pointer choice;
  char *descr;

  if (s7_is_pair(s7_cdr(args)))
    choice = s7_cadr(args);
  else choice = s7_t(sc);
  if (choice == s7_make_keyword(sc, \"readable\"))
    descr = g_block_display_readably(sc, s7_c_object_value(obj));
  else descr = g_block_display(sc, s7_c_object_value(obj));
  obj = s7_make_string(sc, descr);
  free(descr);
  return(obj);
}

static s7_pointer g_block_gc_free(s7_scheme *sc, s7_pointer obj)
{
  free(s7_c_object_value(obj));
  return(NULL);
}

static bool g_blocks_are_eql(void *val1, void *val2)
{
  s7_int i, len;
  g_block *b1 = (g_block *)val1;
  g_block *b2 = (g_block *)val2;
  if (val1 == val2) return(true);
  len = b1->size;
  if (len != b2->size) return(false);
  if ((len & 1) == 0)
    for (i = 0; i < len; i++)
      {
        if (b1->data[i] != b2->data[i]) return(false); i++;
        if (b1->data[i] != b2->data[i]) return(false);
      }
  else
    for (i = 0; i < len; i++)
      if (b1->data[i] != b2->data[i]) return(false);
  return(true);
}

static s7_pointer g_blocks_are_equal(s7_scheme *sc, s7_pointer args)
{
  return(s7_make_boolean(sc, g_blocks_are_eql((void *)s7_c_object_value(s7_car(args)), (void *)s7_c_object_value(s7_cadr(args)))));
}

static s7_pointer g_block_gc_mark(s7_scheme *sc, s7_pointer p)
{
  /* nothing to mark because we protect g_block_methods below, and all blocks get the same let */
  return(p);
}

static s7_pointer g_is_block(s7_scheme *sc, s7_pointer args)
{
  #define g_is_block_help \"(block? obj) returns #t if obj is a block.\"
  #define g_is_block_sig s7_make_signature(sc, 2, s7_make_symbol(sc, \"boolean?\"), s7_t(sc))
  return(s7_make_boolean(sc, s7_c_object_type(s7_car(args)) == g_block_type));
}

static s7_pointer g_is_simple_block(s7_scheme *sc, s7_pointer args)
{
  #define g_is_simple_block_help \"(simple-block? obj) returns #t if obj is a simple-block.\"
  return(s7_make_boolean(sc, s7_c_object_type(s7_car(args)) == g_simple_block_type));
}

static s7_pointer block_ref_p_pp(s7_scheme *sc, s7_pointer obj, s7_pointer ind)
{
  g_block *g;
  size_t index;
  s7_int typ = s7_c_object_type(obj);
  if ((typ != g_block_type) && (typ != g_simple_block_type))
    return(s7_wrong_type_arg_error(sc, \"block-ref\", 1, obj, \"a block\"));
  g  = (g_block *)s7_c_object_value(obj);
  if (s7_is_integer(ind))
    index = (size_t)s7_integer(ind);
  else
    {
      if (s7_is_symbol(ind)) /* ((block 'empty) b) etc (i.e. block method access) */
	{
          s7_pointer val = s7_symbol_local_value(sc, ind, g_block_methods);
          if ((!s7_is_eq(s7_undefined(sc), val)) && (ind != val)) /* else! */
            return(val);
	}
      return(s7_wrong_type_arg_error(sc, \"block-ref\", 2, ind, \"an integer\"));
    }
  if (index < g->size)
    return(s7_make_real(sc, g->data[index]));
  return(s7_out_of_range_error(sc, \"(implicit) block-ref\", 2, ind, \"it should be less than block length\"));
}

static s7_pointer g_block_ref(s7_scheme *sc, s7_pointer args)
{
  #define g_block_ref_help \"(block-ref b i) returns the block value at index i.\"
  #define g_block_ref_sig s7_make_signature(sc, 3, s7_t(sc), s7_make_symbol(sc, \"block?\"), s7_make_symbol(sc, \"integer?\"))
  if (s7_list_length(sc, args) != 2)
    return(s7_wrong_number_of_args_error(sc, \"block-ref takes 2 arguments: ~~S\", args));
  return(block_ref_p_pp(sc, s7_car(args), s7_cadr(args)));
}

static s7_double block_ref_d_7pi(s7_scheme *sc, s7_pointer p, s7_int index)
{
  g_block *g;
  s7_int typ = s7_c_object_type(p); /* currently d_7pi_ok only checks float-vector-ref, so we need to check block-ref here */
  if ((typ != g_block_type) && (typ != g_simple_block_type))
    s7_wrong_type_arg_error(sc, \"block-ref\", 1, p, \"a block\");
  g  = (g_block *)s7_c_object_value(p);
  if ((index < 0) || (index >= g->size))
     s7_out_of_range_error(sc, \"block-ref\", 2, s7_make_integer(sc, index), (index >= 0) ? \"it should be less than block length\" : \"it should be non-negative\");
  return(g->data[index]);
}

static s7_pointer block_set_p_ppp(s7_scheme *sc, s7_pointer obj, s7_pointer ind, s7_pointer val)
{
  g_block *g;
  s7_int index;
  s7_int typ = s7_c_object_type(obj);
  if ((typ != g_block_type) && (typ != g_simple_block_type))
    return(s7_wrong_type_arg_error(sc, \"block-set!\", 1, obj, \"a block\"));
  if (s7_is_immutable(obj))
    return(s7_wrong_type_arg_error(sc, \"block-set!\", 1, obj, \"a mutable block\"));
  if (!s7_is_integer(ind))
    return(s7_wrong_type_arg_error(sc, \"block-set!\", 2, ind, \"an integer\"));
  g = (g_block *)s7_c_object_value(obj);
  index = s7_integer(ind);
  if ((index >= 0) && (index < g->size))
    {
      g->data[index] = s7_number_to_real(sc, val);
      return(val);
    }
  return(s7_out_of_range_error(sc, \"block-set\", 2, ind, \"it should be less than block length\"));
}

static s7_pointer g_block_set(s7_scheme *sc, s7_pointer args)
{
  #define g_block_set_help \"(block-set! b i x) sets the block value at index i to x.\"
  #define g_block_set_sig s7_make_signature(sc, 4, s7_make_symbol(sc, \"real?\"), s7_make_symbol(sc, \"block?\"), s7_make_symbol(sc, \"integer?\"), s7_make_symbol(sc, \"float?\"))
  /* real? as return type, not float? because we return caddr(args) below, not the floatified version of it */
  /* c_object_set functions need to check that they have been passed the correct number of arguments */
  if (s7_list_length(sc, args) != 3)
    return(s7_wrong_number_of_args_error(sc, \"block-set! takes 3 arguments: ~~S\", args));
  return(block_set_p_ppp(sc, s7_car(args), s7_cadr(args), s7_caddr(args)));
}

static s7_double block_set_d_7pid(s7_scheme *sc, s7_pointer p, s7_int index, s7_double x)
{
  g_block *g;
  s7_int typ = s7_c_object_type(p);
  if ((typ != g_block_type) && (typ != g_simple_block_type))
    s7_wrong_type_arg_error(sc, \"block-set!\", 1, p, \"a block\");
  g  = (g_block *)s7_c_object_value(p);
  if ((index < 0) || (index >= g->size))
    s7_out_of_range_error(sc, \"block-set!\", 2, s7_make_integer(sc, index), (index >= 0) ? \"it should be less than block length\" : \"it should be non-negative\");
  if (s7_is_immutable(p))
    s7_wrong_type_arg_error(sc, \"block-set!\", 1, p, \"a mutable block\");
  g->data[index] = x;
  return(x);
}

static s7_pointer g_block_length(s7_scheme *sc, s7_pointer args)
{
  g_block *g = (g_block *)s7_c_object_value(s7_car(args));
  return(s7_make_integer(sc, g->size));
}

static s7_int get_start_and_end(s7_scheme *sc, s7_pointer args, s7_int *start, s7_int end)
{
  if (s7_is_pair(s7_cdr(args)))
    {
      s7_pointer p = s7_cadr(args);
      if (s7_is_integer(p))
        {
          s7_int nstart = s7_integer(p);
          if ((nstart < 0) || (nstart >= end))
            {s7_out_of_range_error(sc, \"subblock\", 2, p, \"it should be less than block length\"); return(0);}
          *start = nstart;
        }
      if (s7_is_pair(s7_cddr(args)))
        {
          p = s7_caddr(args);
          if (s7_is_integer(p))
            {
              s7_int nend = s7_integer(p);
              if (nend <= *start)
                {s7_out_of_range_error(sc, \"subblock\", 3, p, \"it should be greater than the start point\"); return(0);}
              if (nend < end) end = nend;
            }}}
  return(end - *start);
}

static s7_pointer g_block_copy(s7_scheme *sc, s7_pointer args)
{
  s7_pointer new_g;
  g_block *g, *g1;
  size_t len;
  s7_int start = 0;
  s7_pointer obj = s7_car(args);
  if (s7_c_object_type(obj) != g_block_type)   /* obj might not be a block object if destination is one */
    {
      if (s7_is_float_vector(obj))
        {
          s7_pointer v;
          s7_pointer dest = s7_cadr(args);
          g = (g_block *)s7_c_object_value(dest);
          len = g->size;
	  if (s7_is_null(sc, s7_cddr(args)))
  	    {
              if (len > s7_vector_length(obj)) len = s7_vector_length(obj);
	      memcpy((void *)(g->data), (void *)(s7_float_vector_elements(obj)), len * sizeof(s7_double));
              return(dest);
	    }
	  v = s7_make_float_vector_wrapper(sc, len, g->data, 1, NULL, false);
	  s7_gc_protect_via_stack(sc, v);
          s7_copy(sc, s7_cons(sc, obj, s7_cons(sc, v, s7_cddr(args))));
	  s7_gc_unprotect_via_stack(sc, v);
	  return(dest);
        }
      return(s7_f(sc));
    }
  g = (g_block *)s7_c_object_value(obj);
  len = g->size;
  if (s7_is_pair(s7_cdr(args)))
    {
      new_g = s7_cadr(args);
      if (s7_is_immutable(new_g))
        return(s7_wrong_type_arg_error(sc, \"block-copy!\", 0, new_g, \"a mutable block\"));
      if (s7_c_object_type(new_g) != g_block_type) /* fall back on the float-vector code using a wrapper */
        {
          s7_pointer v = s7_make_float_vector_wrapper(sc, len, g->data, 1, NULL, false);
	  s7_gc_protect_via_stack(sc, v);
          new_g = s7_copy(sc, s7_cons(sc, v, s7_cdr(args)));
	  s7_gc_unprotect_via_stack(sc, v);
          return(new_g);
        }
      if (s7_is_pair(s7_cddr(args)))
        len = get_start_and_end(sc, s7_cdr(args), &start, len);
    }
  else new_g = make_block_raw(sc, len);
  g1 = (g_block *)s7_c_object_value(new_g);
  if (g1->size < len) len = g1->size;
  memcpy((void *)(g1->data), (void *)(g->data + start), len * sizeof(double));
  return(new_g);
}

static s7_pointer g_blocks_are_equivalent(s7_scheme *sc, s7_pointer args)
{
  #define g_blocks_are_equivalent_help \"(equivalent? block1 block2)\"
  s7_pointer v1, v2;
  g_block *g1, *g2;
  bool result;
  uint32_t gc1, gc2;
  size_t len;
  s7_pointer arg1 = s7_car(args);
  s7_pointer arg2 = s7_cadr(args);
  if (!s7_is_c_object(arg2))
    return(s7_f(sc));
  if (arg1 == arg2)
    return(s7_make_boolean(sc, true));
  if (s7_is_let(arg1))             /* (block-let (block)) */
    return(s7_make_boolean(sc, false));    /* checked == above */
  g1 = (g_block *)s7_c_object_value(arg1);
  if (s7_c_object_type(arg2) != g_block_type)
    return(s7_make_boolean(sc, false));
  g2 = (g_block *)s7_c_object_value(arg2);
  len = g1->size;
  if (len != g2->size)
    return(s7_make_boolean(sc, false));
  v1 = s7_make_float_vector_wrapper(sc, len, g1->data, 1, NULL, false);
  gc1 = s7_gc_protect(sc, v1);
  v2 = s7_make_float_vector_wrapper(sc, len, g2->data, 1, NULL, false);
  gc2 = s7_gc_protect(sc, v2);
  result = s7_is_equivalent(sc, v1, v2);
  s7_gc_unprotect_at(sc, gc1);
  s7_gc_unprotect_at(sc, gc2);
  return(s7_make_boolean(sc, result));
}

static s7_pointer g_block_append(s7_scheme *sc, s7_pointer args)
{
  #define g_block_append_help \"(append block...) returns a new block containing the argument blocks concatenated.\"
  s7_int i, len = 0;
  s7_pointer p, new_g;
  g_block *g;
  for (i = 1, p = args; s7_is_pair(p); p = s7_cdr(p), i++)
    {
      g_block *g1;
      if (s7_c_object_type(s7_car(p)) != g_block_type)
        return(s7_wrong_type_arg_error(sc, \"block-append\", i, s7_car(p), \"a block\"));
      g1 = (g_block *)s7_c_object_value(s7_car(p));
      len += g1->size;
    }
  new_g = make_block_raw(sc, len);
  g = (g_block *)s7_c_object_value(new_g);
  for (i = 0, p = args; s7_is_pair(p); p = s7_cdr(p))
    {
      g_block *g1;
      g1 = (g_block *)s7_c_object_value(s7_car(p));
      memcpy((void *)(g->data + i), (void *)(g1->data), g1->size * sizeof(double));
      i += g1->size;
    }
  return(new_g);
}

#if (((defined(SIZEOF_VOID_P)) && (SIZEOF_VOID_P == 4)) || ((defined(__SIZEOF_POINTER__)) && (__SIZEOF_POINTER__ == 4)) || (!defined(__LP64__)))
  #define Vectorized
#else
#if (defined(__GNUC__) && __GNUC__ >= 5)
  #define Vectorized __attribute__((optimize(\"tree-vectorize\")))
#else
  #define Vectorized
#endif
#endif

static s7_pointer g_block_reverse(s7_scheme *sc, s7_pointer args)
{
  size_t i, j;
  g_block *g, *g1;
  s7_pointer new_g;
  if (!s7_is_null(sc, s7_cdr(args)))
    return(s7_wrong_number_of_args_error(sc, \"(block-)reverse\", args));
  g = (g_block *)s7_c_object_value(s7_car(args));
  new_g = make_block_raw(sc, g->size);
  g1 = (g_block *)s7_c_object_value(new_g);
  for (i = 0, j = g->size - 1; i < g->size; i++, j--)
    g1->data[i] = g->data[j];
  return(new_g);
}

#define LOOP_8(Code) do {Code; Code; Code; Code; Code; Code; Code; Code;} while (0)

static s7_pointer g_block_reverse_in_place(s7_scheme *sc, s7_pointer args) /* Vectorized is slower */
{
  #define g_block_reverse_in_place_help \"(block-reverse! block) returns block with its data reversed.\"
  size_t i, j;
  g_block *g;
  double *d1, *d2;
  s7_pointer obj = s7_car(args);
  if (s7_c_object_type(obj) != g_block_type)
    return(s7_wrong_type_arg_error(sc, \"block-reverse!\", 0, obj, \"a block\"));
  if (s7_is_immutable(obj))
    return(s7_wrong_type_arg_error(sc, \"block-reverse!\", 0, obj, \"a mutable block\"));
  g = (g_block *)s7_c_object_value(obj);
  if (g->size < 2) return(obj);
  d1 = g->data;
  d2 = (double *)(d1 + g->size - 1);
  if ((g->size & 0x3f) == 0) /* need even number of 32's (we're moving two at a time) */
    {
      while (d1 < d2)
       {
	 s7_double c;
         LOOP_8(c = *d1; *d1++ = *d2; *d2-- = c);
         LOOP_8(c = *d1; *d1++ = *d2; *d2-- = c);
         LOOP_8(c = *d1; *d1++ = *d2; *d2-- = c);
         LOOP_8(c = *d1; *d1++ = *d2; *d2-- = c);
       }}
  else
    if ((g->size & 0xf) == 0)
      {
        while (d1 < d2)
         {
	   s7_double c;
           LOOP_8(c = *d1; *d1++ = *d2; *d2-- = c);
         }}
    else
      while (d1 < d2) {s7_double c; c = *d1; *d1++ = *d2; *d2-- = c;}

  return(obj);
}

static Vectorized void block_memclr64(double *data, size_t bytes)
{
  size_t i;
  for (i = 0; i < bytes; )
    {
      LOOP_8(data[i++] = 0.0);
    }
}

static s7_pointer g_block_fill(s7_scheme *sc, s7_pointer args)
{
  s7_pointer obj = s7_car(args);
  s7_pointer val;
  size_t i, len;
  s7_int start = 0;
  double fill_val;
  g_block *g;
  double *data;

  if (s7_is_immutable(obj))
    return(s7_wrong_type_arg_error(sc, \"block-fill!\", 0, obj, \"a mutable block\"));
  val = s7_cadr(args);
  g = (g_block *)s7_c_object_value(obj);
  fill_val = s7_number_to_real(sc, val);
  len = g->size;
  if (s7_is_pair(s7_cddr(args)))
    len = get_start_and_end(sc, s7_cdr(args), &start, len);
  data = (double *)(g->data + start);
  if (fill_val == 0.0)
    {
      if ((g->size & 0x7) == 0)
        block_memclr64(data, len);
      else memset((void *)data, 0, len * sizeof(double));
    }
  else
    if ((g->size & 0x3) == 0)
      for (i = 0; i < len; ) {data[i++] = fill_val; data[i++] = fill_val; data[i++] = fill_val; data[i++] = fill_val;}
    else
      for (i = 0; i < len; i++) data[i] = fill_val;
  return(obj);
}

static s7_pointer g_blocks(s7_scheme *sc, s7_pointer args)
{
  return(s7_copy(sc, s7_list(sc, 1, args)));
}

static s7_pointer g_subblock(s7_scheme *sc, s7_pointer args)
{
  #define g_subblock_help \"(subblock block (start 0) end) returns a portion of the block.\"
  s7_pointer p, new_g;
  s7_pointer obj = s7_car(args);
  s7_int start = 0, new_len, i;
  g_block *g, *g1;
  if (s7_c_object_type(obj) != g_block_type)
    return(s7_wrong_type_arg_error(sc, \"subblock\", 1, obj, \"a block\"));
  g = (g_block *)s7_c_object_value(obj);
  new_len = get_start_and_end(sc, args, &start, g->size);
  new_g = make_block_raw(sc, new_len);
  g1 = (g_block *)s7_c_object_value(new_g);
  memcpy((void *)(g1->data), (void *)(g->data + start), new_len * sizeof(double));
  return(new_g);
}

static s7_pointer g_block_release_methods(s7_scheme *sc, s7_pointer args)
{
  s7_gc_unprotect_at(sc, block_gc_loc);
  return(s7_f(sc));
}


/* function port tests */
static unsigned char *fout = NULL;
static unsigned int fout_size = 0, fout_loc = 0;
static void foutput(s7_scheme *sc, unsigned char c, s7_pointer port)
{
  if (fout_size == fout_loc)
    {
      if (fout_size == 0)
        {
          fout_size = 128;
          fout = (unsigned char *)malloc(fout_size * sizeof(unsigned char));
        }
      else
        {
          fout_size += 128;
          fout = (unsigned char *)realloc(fout, fout_size * sizeof(unsigned char));
        }}
  fout[fout_loc++] = c;
}

static s7_pointer fout_open(s7_scheme *sc, s7_pointer args)
{
  return(s7_open_output_function(sc, foutput));
}

static s7_pointer fout_get_output(s7_scheme *sc, s7_pointer args)
{
  foutput(sc, 0, s7_car(args)); /* make sure it's null-terminated */
  return(s7_make_string_with_length(sc, (const char *)fout, fout_loc - 1));
}

static s7_pointer fout_close(s7_scheme *sc, s7_pointer args)
{
  fout_loc = 0;
  return(s7_car(args));
}

static const char *fin = NULL;
static unsigned int fin_size = 0, fin_loc = 0;
static s7_pointer finput(s7_scheme *sc, s7_read_t peek, s7_pointer port)
{
  switch (peek)
    {
      case S7_READ_CHAR:
        return(s7_make_character(sc, fin[fin_loc++]));
      case S7_PEEK_CHAR:
        return(s7_make_character(sc, fin[fin_loc]));
      case S7_READ_LINE:
        {
          unsigned int i;
          s7_pointer result;
          for (i = fin_loc; (i < fin_size) && (fin[i] != '\\n'); i++);
          result = s7_make_string_with_length(sc, (char *)(fin + fin_loc), i - fin_loc);
          fin_loc = i + 1;
          return(result);
        }
      case S7_IS_CHAR_READY:
        return(s7_make_boolean(sc, fin_loc < fin_size));
      case S7_READ:
        return(s7_error(sc, s7_make_symbol(sc, \"read-error\"), s7_make_string(sc, \"can't read yet!\")));
      default:
        return(s7_error(sc, s7_make_symbol(sc, \"read-error\"), s7_make_string(sc, \"unknown s7_input_function choice\")));
    }
}

static s7_pointer fin_open(s7_scheme *sc, s7_pointer args)
{
  /* arg = string to read */
  s7_pointer str;
  fin_loc = 0;
  str = s7_car(args);
  if (!s7_is_string(str))
    return(s7_wrong_type_arg_error(sc, \"fin_open\", 1, s7_car(args), \"a string\"));
  fin = s7_string(str); /* assume caller will GC protect the string */
  fin_size = s7_string_length(str);
  return(s7_open_input_function(sc, finput));
}

/* dilambda test */
static s7_pointer g_dilambda_test(s7_scheme *sc, s7_pointer args) {return(s7_f(sc));}
static s7_pointer g_set_dilambda_test(s7_scheme *sc, s7_pointer args) {return(s7_car(args));}

/* hash-table tests */
static s7_pointer g_hloc(s7_scheme *sc, s7_pointer args) {return(s7_make_integer(sc, 0));}
static s7_pointer g_heq(s7_scheme *sc, s7_pointer args) {return(s7_make_boolean(sc, s7_is_eq(s7_car(args), s7_cadr(args))));}

/* optimizer tests */
static s7_pointer g_cf10(s7_scheme *sc, s7_pointer args) {return(s7_car(args));}
static s7_pointer g_cf11(s7_scheme *sc, s7_pointer args) {return(s7_car(args));}
static s7_pointer g_cs11(s7_scheme *sc, s7_pointer args) {return(s7_car(args));}

static s7_pointer g_cf20(s7_scheme *sc, s7_pointer args) {return(s7_car(args));}
static s7_pointer g_cf21(s7_scheme *sc, s7_pointer args) {return(s7_car(args));}
static s7_pointer g_cf22(s7_scheme *sc, s7_pointer args) {return(s7_cadr(args));}

static s7_pointer g_cf30(s7_scheme *sc, s7_pointer args) {return(s7_car(args));}
static s7_pointer g_cf31(s7_scheme *sc, s7_pointer args) {return(s7_car(args));}
static s7_pointer g_cf32(s7_scheme *sc, s7_pointer args) {return(s7_cadr(args));}
static s7_pointer g_cf33(s7_scheme *sc, s7_pointer args) {return(s7_caddr(args));}

static s7_pointer g_cf41(s7_scheme *sc, s7_pointer args) {return(s7_car(args));}
static s7_pointer g_cf42(s7_scheme *sc, s7_pointer args) {return(s7_cadr(args));}
static s7_pointer g_cf43(s7_scheme *sc, s7_pointer args) {return(s7_caddr(args));}
static s7_pointer g_cf44(s7_scheme *sc, s7_pointer args) {return(s7_cadddr(args));}
static s7_pointer g_rs11(s7_scheme *sc, s7_pointer args) {return(s7_make_integer(sc, s7_integer(s7_car(args)) + 1));}

static s7_pointer g_cf51(s7_scheme *sc, s7_pointer args) {return(s7_car(args));}

static s7_pointer sload(s7_scheme *sc, s7_pointer args)
{
  if (s7_is_string(s7_car(args)))
    {
      if (s7_is_pair(s7_cdr(args)))
	 {
	   if (s7_is_let(s7_cadr(args)))
             return(s7_load_with_environment(sc, s7_string(s7_car(args)), s7_cadr(args)));
           return(s7_wrong_type_arg_error(sc, \"load\", 2, s7_cadr(args), \"an environment\"));
         }
      return(s7_load(sc, s7_string(s7_car(args))));
    }
  return(s7_wrong_type_arg_error(sc, \"load\", 1, s7_car(args), \"file name\"));
}
static s7_pointer scall(s7_scheme *sc, s7_pointer args) {return(s7_call(sc, s7_car(args), s7_cadr(args)));}
static s7_pointer sread(s7_scheme *sc, s7_pointer args)
{
  if (s7_is_pair(args))
    return(s7_read(sc, s7_car(args)));
  return(s7_read(sc, s7_current_input_port(sc)));
}
static s7_pointer swind(s7_scheme *sc, s7_pointer args) {return(s7_dynamic_wind(sc, s7_car(args), s7_cadr(args), s7_caddr(args)));}
static s7_pointer seval(s7_scheme *sc, s7_pointer args)
{
  if (s7_is_pair(s7_cdr(args)))
    return(s7_eval(sc, s7_car(args), s7_cadr(args)));
  return(s7_eval(sc, s7_car(args), s7_curlet(sc)));
}
static s7_pointer sevalstr(s7_scheme *sc, s7_pointer args)
{
  if (s7_is_string(s7_car(args)))
    {
      if (s7_is_pair(s7_cdr(args)))
	 {
	   if (s7_is_let(s7_cadr(args)))
             return(s7_eval_c_string_with_environment(sc, s7_string(s7_car(args)), s7_cadr(args)));
           return(s7_wrong_type_arg_error(sc, \"eval-string\", 2, s7_cadr(args), \"an environment\"));
	 }
      return(s7_eval_c_string_with_environment(sc, s7_string(s7_car(args)), s7_curlet(sc)));
    }
  return(s7_wrong_type_arg_error(sc, \"eval-string\", 1, s7_car(args), \"string of code\"));
}

void block_init(s7_scheme *sc);
void block_init(s7_scheme *sc)
{
  s7_pointer cur_env, meq_func;
  max_vector_length = s7_integer(s7_let_ref(sc, s7_symbol_value(sc, s7_make_symbol(sc, \"*s7*\")), s7_make_symbol(sc, \"max-vector-length\")));
  cur_env = s7_outlet(sc, s7_curlet(sc));
  g_block_type = s7_make_c_type(sc, \"<block>\");
  s7_c_type_set_gc_free(sc, g_block_type, g_block_gc_free);
  s7_c_type_set_equal(sc, g_block_type, g_blocks_are_eql);
  s7_c_type_set_is_equal(sc, g_block_type, g_blocks_are_equal);
  s7_c_type_set_is_equivalent(sc, g_block_type, g_blocks_are_equivalent);
  s7_c_type_set_gc_mark(sc, g_block_type, g_block_gc_mark);
  s7_c_type_set_ref(sc, g_block_type, g_block_ref);
  s7_c_type_set_set(sc, g_block_type, g_block_set);
  s7_c_type_set_length(sc, g_block_type, g_block_length);
  s7_c_type_set_copy(sc, g_block_type, g_block_copy);
  s7_c_type_set_reverse(sc, g_block_type, g_block_reverse);
  s7_c_type_set_fill(sc, g_block_type, g_block_fill);
  s7_c_type_set_to_string(sc, g_block_type, g_block_to_string);
  s7_define_safe_function(sc, \"make-block\", g_make_block, 1, 0, false, g_make_block_help);
  s7_define_safe_function(sc, \"block\", g_to_block, 0, 0, true, g_block_help);
  s7_define_typed_function(sc, \"block-ref\", g_block_ref, 2, 0, false, g_block_ref_help, g_block_ref_sig);
  s7_c_type_set_getter(sc, g_block_type, s7_name_to_value(sc, \"block-ref\"));
  s7_define_typed_function(sc, \"block-set!\", g_block_set, 3, 0, false, g_block_set_help, g_block_set_sig);
  s7_c_type_set_setter(sc, g_block_type, s7_name_to_value(sc, \"block-set!\"));
  s7_define_safe_function(sc, \"block-let\", g_block_let, 1, 0, false, g_block_let_help);
  s7_define_safe_function(sc, \"subblock\", g_subblock, 1, 0, true, g_subblock_help);
  s7_define_safe_function(sc, \"block-append\", g_block_append, 0, 0, true, g_block_append_help);
  s7_define_safe_function(sc, \"block-reverse!\", g_block_reverse_in_place, 1, 0, false, g_block_reverse_in_place_help);
  s7_define_typed_function(sc, \"block?\", g_is_block, 1, 0, false, g_is_block_help, g_is_block_sig);
  s7_define_safe_function_star(sc, \"blocks1\", g_blocks, \"(frequency 4)\", \"test for function*\");
  s7_define_safe_function_star(sc, \"blocks\", g_blocks, \"(frequency 4) (scaler 1)\", \"test for function*\");
  s7_define_safe_function_star(sc, \"blocks3\", g_blocks, \"(frequency 4) (scaler 1) (asdf 32)\", \"test for function*\");
  s7_define_safe_function_star(sc, \"blocks4\", g_blocks, \"(frequency 4) (scaler 1) (asdf 32) etc\", \"test for function*\");
  s7_define_function_star(sc, \"unsafe-blocks1\", g_blocks, \"(frequency 4)\", \"test for function*\");
  s7_define_function_star(sc, \"unsafe-blocks\", g_blocks, \"(frequency 4) (scaler 1)\", \"test for function*\");
  s7_define_function_star(sc, \"unsafe-blocks3\", g_blocks, \"(frequency 4) (scaler 1) (asdf 32)\", \"test for function*\");
  s7_define_function_star(sc, \"unsafe-blocks4\", g_blocks, \"(frequency 4) (scaler 1) (asdf 32) etc\", \"test for function*\");
  s7_define_safe_function_star(sc, \"blocks5\", g_blocks, \"(frequency 4) :allow-other-keys\", \"test for function*\");
  g_block_methods = s7_eval_c_string(sc, \"(openlet (immutable! (inlet 'float-vector? (lambda (p) #t) \\\n\
								      'signature (lambda (p) (list #t 'block? 'integer?)) \\\n\
								      'arity (lambda (p) (cons 1 1)) \\\n\
								      'aritable? (lambda (p args) (= args 1)) \\\n\
								      'vector-dimensions (lambda (p) (list (length p))) \\\n\
						                      'empty (lambda (p) (zero? (length p))) \\\n\
								      'vector-ref block-ref \\\n\
								      'vector-set! block-set! \\\n\
                                                                      'subsequence subblock \\\n\
						                      'append block-append \\\n\
						                      'reverse! block-reverse!)))\");
  block_gc_loc = s7_gc_protect(sc, g_block_methods);
  s7_define_safe_function(sc, \"block-release-methods\", g_block_release_methods, 0, 0, false, NULL);

  g_simple_block_type = s7_make_c_type(sc, \"<simple-block>\");
  s7_define_safe_function(sc, \"make-simple-block\", g_make_simple_block, 1, 0, false, g_make_simple_block_help);
  s7_c_type_set_gc_free(sc, g_simple_block_type, g_block_gc_free);
  s7_c_type_set_gc_mark(sc, g_simple_block_type, g_block_gc_mark);
  s7_c_type_set_length(sc, g_simple_block_type, g_block_length);
  s7_c_type_set_ref(sc, g_simple_block_type, g_block_ref);
  s7_c_type_set_set(sc, g_simple_block_type, g_block_set);
  s7_define_safe_function(sc, \"simple-block?\", g_is_simple_block, 1, 0, false, g_is_simple_block_help);

  s7_set_p_d_function(sc, s7_name_to_value(sc, \"block\"), block_p_d);
  s7_set_d_7pi_function(sc, s7_name_to_value(sc, \"block-ref\"), block_ref_d_7pi);
  s7_set_d_7pid_function(sc, s7_name_to_value(sc, \"block-set!\"), block_set_d_7pid);
  s7_set_p_pp_function(sc, s7_name_to_value(sc, \"block-ref\"), block_ref_p_pp);
  s7_set_p_ppp_function(sc, s7_name_to_value(sc, \"block-set!\"), block_set_p_ppp);

  g_c_tag_type = s7_make_c_type(sc, \"c-tag\");
  s7_define_safe_function(sc, \"make-c-tag\", g_make_c_tag, 0, 0, false, \"no help here\");
  s7_c_type_set_free(sc, g_c_tag_type, g_c_tag_free);

  g_c_tag1_type = s7_make_c_type(sc, \"c-tag1\");
  s7_define_safe_function(sc, \"make-c-tag1\", g_make_c_tag1, 0, 0, false, \"no help here\");
  s7_c_type_set_free(sc, g_c_tag1_type, g_c_tag_free);
  s7_c_type_set_equal(sc, g_c_tag1_type, NULL);
  s7_c_type_set_is_equal(sc, g_c_tag1_type, NULL);
  s7_c_type_set_is_equivalent(sc, g_c_tag1_type, NULL);
  s7_c_type_set_ref(sc, g_c_tag1_type, NULL);
  s7_c_type_set_set(sc, g_c_tag1_type, NULL);
  s7_c_type_set_length(sc, g_c_tag1_type, NULL);
  s7_c_type_set_copy(sc, g_c_tag1_type, NULL);
  s7_c_type_set_fill(sc, g_c_tag1_type, NULL);
  s7_c_type_set_reverse(sc, g_c_tag1_type, NULL);
  s7_c_type_set_to_list(sc, g_c_tag1_type, NULL);
  s7_c_type_set_to_string(sc, g_c_tag1_type, NULL);
  s7_c_type_set_getter(sc, g_c_tag1_type, NULL);
  s7_c_type_set_setter(sc, g_c_tag1_type, NULL);

  g_cycle_type = s7_make_c_type(sc, \"<cycle>\");
  s7_define_safe_function(sc, \"make-cycle\", g_make_cycle, 1, 0, false, \"no help here\");
  s7_define_safe_function(sc, \"<cycle>\", g_make_cycle, 1, 0, false, \"no help here\"); /* for print readably */
  s7_c_type_set_mark(sc, g_cycle_type, g_cycle_mark);
  s7_c_type_set_free(sc, g_cycle_type, g_cycle_free);
  s7_c_type_set_to_list(sc, g_cycle_type, g_cycle_to_list);
  s7_c_type_set_copy(sc, g_cycle_type, g_cycle_copy);
  s7_c_type_set_set(sc, g_cycle_type, g_cycle_implicit_set);
  s7_define_safe_function(sc, \"cycle-ref\", g_cycle_ref, 1, 0, false, \"no help here\");
  s7_define_safe_function(sc, \"cycle-set!\", g_cycle_set, 2, 0, false, \"no help here\");

  s7_define_safe_function(sc, \"function-open-output\", fout_open, 0, 0, false, \"\");
  s7_define_safe_function(sc, \"function-get-output\", fout_get_output, 1, 0, false, \"\");
  s7_define_safe_function(sc, \"function-close-output\", fout_close, 1, 0, false, \"\");
  s7_define_safe_function(sc, \"function-open-input\", fin_open, 1, 0, false, \"\");

  s7_define_safe_function(sc, \"hash_heq\", g_heq, 2, 0, false, \"hash-table test\");
  s7_define_safe_function(sc, \"hash_hloc\", g_hloc, 1, 0, false, \"hash-table test\");

  s7_define_safe_function(sc, \"cf10\", g_cf10, 1, 0, false, \"\");
  s7_define_safe_function(sc, \"cf11\", g_cf11, 1, 0, false, \"\");
  s7_define_safe_function(sc, \"cs11\", g_cs11, 1, 0, false, \"\");
  s7_define_safe_function(sc, \"rs11\", g_rs11, 1, 0, false, \"\");

  s7_define_safe_function(sc, \"cf20\", g_cf20, 2, 0, false, \"\");
  s7_define_safe_function(sc, \"cf21\", g_cf21, 2, 0, false, \"\");
  s7_define_safe_function(sc, \"cf22\", g_cf22, 2, 0, false, \"\");

  s7_define_safe_function(sc, \"cf30\", g_cf30, 3, 0, false, \"\");
  s7_define_safe_function(sc, \"cf31\", g_cf31, 3, 0, false, \"\");
  s7_define_safe_function(sc, \"cf32\", g_cf32, 3, 0, false, \"\");
  s7_define_safe_function(sc, \"cf33\", g_cf33, 3, 0, false, \"\");

  s7_define_safe_function(sc, \"cf41\", g_cf41, 4, 0, false, \"\");
  s7_define_safe_function(sc, \"cf42\", g_cf42, 4, 0, false, \"\");
  s7_define_safe_function(sc, \"cf43\", g_cf43, 4, 0, false, \"\");
  s7_define_safe_function(sc, \"cf44\", g_cf44, 4, 0, false, \"\");

  s7_define_safe_function(sc, \"cf51\", g_cf51, 5, 0, false, \"\");

  s7_define_function(sc, \"sload\",  sload,        1, 1, false, \"test s7_load\");
  s7_define_function(sc, \"scall\",  scall,        2, 0, false, \"test s7_call\");
  s7_define_function(sc, \"sread\",  sread,        0, 1, false, \"test s7_read\");
  s7_define_function(sc, \"swind\",  swind,        3, 0, false, \"test s7_dynamic_wind\");
  s7_define_function(sc, \"seval\",  seval,        1, 1, false, \"test s7_eval\");
  s7_define_function(sc, \"sevalstr\",  sevalstr,  1, 1, false, \"test s7_eval_c_string\");

  s7_define_safe_function(sc, \"dilambda_test\", g_dilambda_test, 0, 0, false, \"\");
  s7_define_safe_function(sc, \"set_dilambda_test\", g_set_dilambda_test, 1, 0, false, \"\");
  s7_set_setter(sc, s7_name_to_value(sc, \"dilambda_test\"), s7_name_to_value(sc, \"set_dilambda_test\"));
}
")))

  (let ((flags (if (provided? 'debugging) "-g3" "-g -O2")))
    (cond ((provided? 'osx)
	   (system (string-append "gcc -c s7test-block.c " flags))
	   (system "gcc s7test-block.o -o s7test-block.so -dynamic -bundle -undefined suppress -flat_namespace"))

	  ((or (provided? 'freebsd)
	       (provided? 'netbsd))
	   (system (string-append "cc -fPIC -c s7test-block.c " flags))
	   (system "cc s7test-block.o -shared -o s7test-block.so -lm -lc"))

	  ((provided? 'openbsd)
	   (system (string-append "clang -fPIC -c s7test-block.c " flags))
	   (system "clang s7test-block.o -shared -o s7test-block.so -lm -lc"))

	  ((provided? 'solaris)
	   (system "gcc -fPIC -c s7test-block.c")
	   (system "gcc s7test-block.o -shared -o s7test-block.so -G -ldl -lm"))

	  (else
	   (system (string-append "gcc -fPIC -c s7test-block.c " flags))
	   (system "gcc s7test-block.o -shared -o s7test-block.so -ldl -lm -Wl,-export-dynamic"))))

  (let ((new-env (sublet (curlet) (cons 'init_func 'block_init)))) ; load calls init_func
    (load "s7test-block.so" new-env))

  (define _c_obj_ (make-block 16))
  (unless (immutable? (block-let _c_obj_)) (format *stderr* "~S's let is mutable~%" _c_obj_))) ; with-block
  ;; else...
  (define _c_obj_ (c-pointer 0))) ; not with-block
(define _null_ (c-pointer 0))

(when (provided? 'linux)
  (if (and (provided? 'system-extras)
	   (not (file-exists? "ffitest.c"))
	   (file-exists? "tools/ffitest.c"))
      (system "cp tools/ffitest.c ."))
  (if (provided? 'gmp)
      (system (string-append (if (provided? 'clang) "clang" "gcc") " -o ffitest ffitest.c -g3 -Wall -fPIC s7.o -DWITH_GMP -lgmp -lmpfr -lmpc " asan-flags " -lm -I. -ldl -Wl,-export-dynamic"))
      (system (string-append (if (provided? 'clang) "clang" "gcc") " -o ffitest ffitest.c -g3 -Wall -fPIC s7.o " asan-flags " -lm -I. -ldl -Wl,-export-dynamic")))
  (system "ffitest"))

#|
(define (ok1? otst ola oexp)
  (let ((result (catch 'all-done ola
		       (lambda args
			 (if (not (eq? oexp 'error))
			     (begin (display args) (newline)))
			 'error))))
    (if (not (equal? result oexp))
	(format #t "~A: ~A got ~S but expected ~S~%~%" (port-line-number) otst result oexp))))

(define-macro (test tst expected)
  `(ok1? ',tst (lambda ()
		 (if (null? (hook-functions *error-hook*))
		     (set! (hook-functions *error-hook*)
			   (list (lambda (hook)
				   (apply format *stderr* (hook 'data))
				   (newline *stderr*)
				   (set! (hook 'result) 'error)
				   (throw 'all-done)))))
		 ,tst)
	 ,expected))
|#


;;; --------------------------------------------------------------------------------
;;; eq?

(test (eq? 'a 3) #f)
(test (eq? #t 't) #f)
(test (eq? "abs" 'abc) #f)
(test (eq? "hi" '(hi)) #f)
(test (eq? "hi" "hi") #f)
(test (eq? "()" ()) #f)
(test (eq? '(1) '(1)) #f)
(test (eq? '(#f) '(#f)) #f)
(test (eq? #\a #\b) #f)
(test (eq? #t #t) #t)
(test (eq? #f #f) #t)
(test (eq? #f #t) #f)
(test (eq? (null? ()) #t) #t)
(test (eq? (null? '(a)) #f) #t)
(test (eq? (cdr '(a)) ()) #t)
(test (eq? 'a 'a) #t)
(test (eq? 'a 'b) #f)
(test (eq? 'a (string->symbol "a")) #t)
(test (eq? (symbol "a") (string->symbol "a")) #t)
(test (eq? :a :a) #t)
(test (eq? :a 'a) #f)
(test (eq? ':a 'a) #f)
(test (eq? ':a ':a) #t)
(test (eq? :a a:) #f)
(test (eq? ':a 'a:) #f)
(test (eq? 'a: 'a:) #t)
(test (eq? ':a: 'a:) #f)
(test (eq? 'a (symbol "a")) #t)
(test (eq? :: '::) #t)
(test (eq? ':a (symbol->keyword (symbol "a"))) #t) ; but not a:
(test (eq? '(a) '(b)) #f)
(test (let ((x '(a . b))) (eq? x x)) #t)
(test (let ((x (cons 'a 'b))) (eq? x x)) #t)
(test (eq? (cons 'a 'b) (cons 'a 'b)) #f)
(test (eq? "abc" "cba") #f)
(test (let ((x "hi")) (eq? x x)) #t)
(test (eq? (string #\h #\i) (string #\h #\i)) #f)
(test (eq? #(a) #(b)) #f)
(test (let ((x (vector 'a))) (eq? x x)) #t)
(test (eq? (vector 'a) (vector 'a)) #f)
(test (eq? car car) #t)
(test (eq? car cdr) #f)
(test (let ((x (lambda () 1))) (eq? x x)) #t)
(test (let ((x (lambda () 1))) (let ((y x)) (eq? x y))) #t)
(test (let ((x (lambda () 1))) (let ((y (lambda () 1))) (eq? x y))) #f)
(test (eq? 'abc 'abc) #t)
(test (eq? eq? eq?) #t)
(test (eq? (if #f 1) 1) #f)
(test (eq? () '(#||#)) #t)
(test (eq? () '(#|@%$&|#)) #t)
(test (eq? '#||#hi 'hi) #t) ; ??
(test (eq? '; a comment
         hi 'hi) #t) ; similar:
    (test (cadr '#| a comment |#(+ 1 2)) 1)
    (test `(+ 1 ,@#||#(list 2 3)) '(+ 1 2 3))
    (test `(+ 1 ,#||#(+ 3 4)) '(+ 1 7))
    ;; but not splitting the ",@" or splitting a number:
    (test (+ 1 2.0+#||#3i) 'error)
    (test `(+ 1 ,#||#@(list 2 3)) 'error)
(test (eq? #||# (#|%%|# append #|^|#) #|?|# (#|+|# list #|<>|#) #||#) #t)
(test (eq? '() ;a comment
	   '()) #t)
(test (eq? 3/4 3) #f)
(test (eq? '() '()) #t)
(test (eq? '() '(  )) #t)
(test (eq? '()'()) #t)
(test (eq? '()(list)) #t)
(test (eq? () (list)) #t)
(test (eq? (begin) (append)) #t)
(test (let ((lst (list 1 2 3))) (eq? lst (apply list lst))) #f) ; changed 26-Sep-11

;(test (eq? 1/0 1/0) #f)
;(test (let ((+nan.0 1/0)) (eq? +nan.0 +nan.0)) #f)
;; these are "unspecified" so any boolean value is ok

(test (eq? ''2 '2) #f)
(test (eq? '2 '2) #t) ; unspecified??
(test (eq? '2 2) #t)
(test (eq? ''2 ''2) #f)
(test (eq? ''#\a '#\a) #f)
(test (eq? '#\a #\a) #t) ; was #f
(test (eq? 'car car) #f)
(test (eq? '()()) #t)
(test (eq? ''() '()) #f)
(test (eq? '   () '
()) #t)
(test (eq? '#f #f) #t)
(test (eq? '#f '#f) #t)
(test (eq? #f '  #f) #t)
(test (eq? '()'()) #t) ; no space
(test (#||# eq? #||# #f #||# #f #||#) #t)
(test (eq? (current-input-port) (current-input-port)) #t)
(test (let ((f (lambda () (quote (1 . "H"))))) (eq? (f) (f))) #t)
(test (let ((f (lambda () (cons 1 (string #\H))))) (eq? (f) (f))) #f)
(test (eq? *stdin* *stdin*) #t)
(test (eq? *stdout* *stderr*) #f)
(test (eq? *stdin* *stderr*) #f)
(test (eq? else else) #t)
(test (eq? :else else) #f)
(test (eq? :else 'else) #f)
(test (eq? :if if) #f)
(test (eq? 'if 'if) #t)
(test (eq? :if :if) #t)

(test (eq? (string) (string)) #t) ; was #f
(test (eq? (string) "") #t) ; was #f -- changed 29-Jun-21
(test (eq? (vector) (vector)) #f)
(test (eq? (vector) #()) #f)
(test (eq? (list) (list)) #t)
(test (eq? (list) ()) #t)
(test (eq? (hash-table) (hash-table)) #f)
(test (eq? (curlet) (curlet)) #t)
(test (eq? (rootlet) (rootlet)) #t)
(test (eq? (funclet abs) (funclet abs)) #t) ; or any other built-in...
(test (eq? letrec* letrec*) #t)

(test (eq? (current-input-port) (current-input-port)) #t)
(test (eq? (current-error-port) (current-error-port)) #t)
(test (eq? (current-output-port) (current-output-port)) #t)
(test (eq? (current-input-port) (current-output-port)) #f)

(test (eq? (string #\a) (string #\a)) #f)
(test (eq? "a" "a") #f)
(test (eq? #(1) #(1)) #f)
(test (let ((a "hello") (b "hello")) (eq? a b)) #f)
(test (let ((a "foo")) (eq? a (copy a))) #f)
(test (let ((p (c-pointer 0))) (eq? p (copy p))) #f)
(test (let ((p (c-pointer 0))) (let ((p1 p)) (eq? p p1))) #t)
(test (let () (define (g x) x) (define (u x) g) (define (f) (eq? g (u g))) (f)) #t) ; guile mailing list

(begin #| ; |# (display ""))
(newline)

(test (;
       eq? ';!
       (;)()#
	);((")";
       ;"#|)#""
       '#|";"|#(#|;|#); ;#
	 ;\;"#"#f
	       )#t)

(test (+ #| this is a comment |# 2 #| and this is another |# 3) 5)
(test (eq? #| a comment |# #f #f) #t)
(test (eq? #| a comment |##f #f) #t)  ; ??
(test (eq? #| a comment | ##f|##f #f) #t) ; ??
(test (eq? #||##||##|a comment| ##f|##f #f) #t)

(test (+ ;#|
            3 ;|#
            4)
      7)
(test (+ #| ; |# 3
		 4)
      7)
#!
(format *stderr* "#! ignored?~%")
!#
#|
(format *stderr* "#| ignored?~%")
|#

(test (eq? (if #f #t) (if #f 3)) #t)

(test (eq?) 'error)           ; "this comment is missing a double-quote
(test (eq? #t) 'error)        #| "this comment is missing a double-quote |#
(test (eq? #t #t #t) 'error)  #| and this has redundant starts #| #| |#
(test (eq? #f . 1) 'error)
(test (eq #f #f) 'error)

(define (feq)
  (let ((things (vector #t #f #\space () "" 0 1 3/4 1+i 1.5 '(1 .2) #() (vector) (vector 1) (list 1) 'f 't #\t)))
    (let ((len (length things)))
      (do ((i 0 (+ i 1)))
	  ((= i (- len 1)))
        (do ((j (+ i 1) (+ j 1)))
	    ((= j len))
  	  (if (eq? (vector-ref things i) (vector-ref things j))
	      (format #t ";feq: (eq? ~A ~A) -> #t?~%" (vector-ref things i) (vector-ref things j))))))))
(feq)

;;; these are defined at user-level in s7 -- why are other schemes so coy about them?
(test (eq? (if #f #f) #<unspecified>) #t)
(test (eq? (symbol->value '_?__undefined__?_) #<undefined>) #t)
(test (eq? #<eof> #<eof>) #t)
(test (eq? #<undefined> #<undefined>) #t)
(test (eq? #<unspecified> #<unspecified>) #t)
(test (eq? #<eof> #<undefined>) #f)
(test (eq? #<eof> ()) #f)
(test (eq? #<undefined> _undef_) #f)
(test (eq? _undef_ _undef_) #t)

(test (let () (define-macro (hi a) `(+ 1 ,a)) (eq? hi hi)) #t)
(test (let () (define (hi a) (+ 1 a)) (eq? hi hi)) #t)
(test (let ((x (lambda* (hi (a 1)) (+ 1 a)))) (eq? x x)) #t)

(test (eq? quasiquote quasiquote) #t)
(test (eq? `quasiquote 'quasiquote) #t)
(test (eq? 'if (keyword->symbol :if)) #t)
(test (eq? 'if (string->symbol "if")) #t)
(test (eq? (copy lambda) (copy 'lambda)) #f)
(test (eq? if 'if) #f)
(test (eq? if `if) #f)
(test (eq? if (keyword->symbol :if)) #f)
(test (eq? if (string->symbol "if")) #f)
(test (eq? lambda and) #f)
(test (eq? let let*) #f)
(test (eq? quote quote) #t)
(test (eq? '"hi" '"hi") #f) ; guile also
;(test (eq? '"" "") #f)
;(test (eq? '"" '"") #f)
;(test (eq? "" "") #f)
(test (eq? #() #()) #f)
(test (eq? '#() #()) #f)
(test (eq? '#() '#()) #f)
(test (let ((v #())) (eq? v #())) #f)
(test (let ((v #())) (eq? v v)) #t)
(test (call/cc (lambda (return) (return (eq? return return)))) #t)
(test (let ((r #f)) (call/cc (lambda (return) (set! r return) #f)) (eq? r r)) #t)
(test (eq? _unbound_variable_ #f) 'error)

(when with-block
  (let ((b (make-block 4)))
    (test (eq? b b) #t)
    (test (equal? b b) #t)
    (test (block? b) #t)
    (test (block? #()) #f)
    (test (block? #f) #f)
    (set! (b 0) 32)
    (test (b 0) 32.0)
    (let () (define (hi b i) (b i)) (test (hi b 0) 32.0))
    (let () (define (hi b) (b 0)) (test (hi b) 32.0))
    (let () (define (hi b) (b)) (test (hi b) 'error)) ;was 32!?
    (test b (block 32.0 0.0 0.0 0.0))
    (test (let? (block-let b)) #t)
    (test (((block-let b) 'float-vector?) b) #t)
    (test (object->string b) "(block 32.000 0.000 0.000 0.000)")
    (let ((b1 (make-block 4)))
      (test (eq? b b1) #f))
    (let ((b (block 1 2)))
      (test (vector-ref b 0) 1.0)
      (test (vector-set! b 0 3) 3)
      (test b (block 3 2)))
    (test (map abs b) (list 32.0 0.0 0.0 0.0)))
  (let ((b (block 1 2 3)))
    (test (reverse b) (block 3 2 1))
    (test b (block 1 2 3))
    (test (reverse b 1) 'error)
    (test (reverse! b) (block 3 2 1))
    (test (reverse! b 1) 'error)
    (test b (block 3 2 1))
    (test (b 'a) 'error)
    (test ((block) 'a) 'error))

  (when full-s7test
    (define (iota-block len)
      (let ((b (make-block len)))
        (do ((i 0 (+ i 1)))
            ((= i len) b)
          (block-set! b i i))))
    (do ((i 0 (+ i 1)))
        ((= i 512))
      (let* ((b (iota-block i))
             (b1 (reverse b))
             (b2 (reverse! b)))
        (unless (equal? b1 b2)
          (format *stderr* "reverse iota-block ~D: ~A~%    ~A~%" i b1 b2)))))

  ;; check map/for-each
  (test (let () (define (func) ((lambda () (map (vector 0 1 2) '(1))))) (func)) '(1))
  (let ((b (block 1 2 3)))
    (test ((lambda () (map b '(1)))) '(2.0))
    (test (let () (define (func) ((lambda () (for-each b '(1))))) (func)) #<unspecified>)
    (test (let () (define (func) ((lambda () (map b '(1))))) (func)) '(2.0))
    (test (let () (define (func) ((lambda () (map b '(xyx 1))))) (func)) 'error)
    (test (let () (define (func) ((lambda () (map (vector 1 2) '(xyx 1))))) (func)) 'error)
    (test (let () (define (func) ((lambda () (map b '(3 1))))) (func)) 'error)
    (test (let () (define (func) ((lambda () (map b '(#(1) 1))))) (func)) 'error))

  (test (object->string (block +nan.0)) "(block +nan.0)")
  (test (object->string (block +inf.0)) "(block +inf.0)")
  (test (object->string (block -inf.0)) "(block -inf.0)")
  (test (object->string (block pi)) "(block 3.142)")
  (test (object->string (block +nan.0) :readable) "(block +nan.0)")
  (test (object->string (block +inf.0) :readable) "(block +inf.0)")
  (test (object->string (block -inf.0) :readable) "(block -inf.0)")
  (test (object->string (block pi) :readable) "(block 3.141592653589793)") ; (block pi) would be better
  (let-temporarily (((*s7* 'float-format-precision) 8))
    (test (object->string (block pi)) "(block 3.14159265)"))

  (let ((v (make-vector 2 (block 1.0) block?)))
    (test (block? (v 0)) #t)
    (vector-set! v 0 (block 2.0))
    (test (block-ref (vector-ref v 0) 0) 2.0)
    (test (#_block-ref (vector-ref v 0) 0) 2.0)
    (test (vector-set! v 0 #f) 'error)
    (test (signature v) (let ((lst (list 'block? 'vector? 'integer?))) (set-cdr! (cddr lst) (cddr lst)) lst)))

  (let ((h (make-hash-table 8 #f (cons symbol? block?))))
    (hash-table-set! h 'a (block 1.0))
    (test (block? (h 'a)) #t)
    (test (block-ref (h 'a) 0) 1.0)
    (test (hash-table-set! h 'b 'asdf) 'error)
    (test (hash-table-set! h "b" (block)) 'error)
    (test (signature h) '(block? hash-table? symbol?)))

  (let ((h (make-hash-table 8 #f (cons #t block?))))
    (hash-table-set! h 'a (block 2.0))
    (test (block? (h 'a)) #t)
    (test (block-ref (h 'a) 0) 2.0)
    (test (hash-table-set! h 'b 'asdf) 'error)
    (test (hash-table-set! h "b" (block)) (block))
    (test (signature h) '(block? hash-table? #t)))

  (let ((h (make-hash-table 8 #f (cons symbol? #t))))
    (hash-table-set! h 'a (block 2.0))
    (test (block? (h 'a)) #t)
    (test (block-ref (h 'a) 0) 2.0)
    (test (hash-table-set! h 'b 'asdf) 'asdf)
    (test (hash-table-set! h "b" (block)) 'error)
    (test (signature h) '(#t hash-table? symbol?))
    (test (block-ref (block 1.0 2.0 3.0) else) 'error))

  (test (make-hash-table 8 #f (cons #t #f)) 'error)
  (test (make-hash-table 8 #f ()) 'error)

  (let ((sig (list #t 'hash-table? #t)))
    (set-cdr! (cddr sig) (cddr sig))
    (test (signature (make-hash-table 8 #f (cons #t #t))) sig)) ; same as (signature (make-hash-table 8 #f [#f]))

  (test (blocks) (list 4 1))
  (test (blocks :frequency 2) (list 2 1))
  (let ((freq :frequency)) (test (blocks freq 2) (list 2 1)))
  (let ((freq :scaler)) (test (blocks freq 2) (list 4 2)))
  (let ((c #f)) (test (blocks (if c 100 :frequency) 10) (list 10 1)))
  (test (blocks :scaler 3 :frequency 2) (list 2 3))
  (test (blocks :scaler 3 :phase 1) 'error)
  (test (map blocks '(1 2 3)) '((1 1) (2 1) (3 1)))
  (test (map blocks '( 1 2 3) '(4 5 6)) '((1 4) (2 5) (3 6)))
  (test (documentation blocks) "test for function*")
  (test (apply blocks '(:frequency 5 :scaler 4)) '(5 4))
  (test (let () (define (b1) (blocks 100)) (b1)) '(100 1))
  (test (let () (define (b1) (blocks 10 2)) (b1)) '(10 2))
  (test (procedure? blocks) #t)
  (unless (or with-bignums (> (*s7* 'debug) 0)) ; debug turns off s7-optimize
    (test (s7-optimize '((block-append (make-block 2) (block)))) (block 0 0))) ; segfault due to plist overuse

  (let ((b1 (block 1)) (b2 (block 2)) (x1 2.0) (x2 3.0) (i 0) (j 0) (x 0.0))
    (define (g1) ; opt_d_dd_fff_rev
      (do ((i 0 (+ i 1))) ((= i 1) x) (set! x (+ (* x1 (block-ref b1 i)) (* x2 (block-ref b2 j))))))
    (test (g1) 8.0)
    (define (g2) ; opt_d_dd_fff
      (do ((i 0 (+ i 1))) ((= i 1) x) (set! x (+ (* (block-ref b1 i) x1) (* (block-ref b2 j) x2)))))
    (test (g2) 8.0))

  (test (unsafe-blocks) (list 4 1))
  (test (unsafe-blocks :frequency 2) (list 2 1))
  (test (unsafe-blocks :scaler 3 :frequency 2) (list 2 3))
  (test (unsafe-blocks :scaler 3 :phase 1) 'error)
  (test (map unsafe-blocks '(1 2 3)) '((1 1) (2 1) (3 1)))
  (test (map unsafe-blocks '( 1 2 3) '(4 5 6)) '((1 4) (2 5) (3 6)))
  (test (documentation unsafe-blocks) "test for function*")
  (test (apply unsafe-blocks '(:frequency 5 :scaler 4)) '(5 4))
  (test (let () (define (b1) (unsafe-blocks 100)) (b1)) '(100 1))
  (test (let () (define (b1) (unsafe-blocks 10 2)) (b1)) '(10 2))
  (test (procedure? unsafe-blocks) #t)

  (test (blocks3 (car (list :a))) 'error)
  (test (let () (define (func) (blocks3 (car (list :a)))) (func)) 'error)

  (test (let () (define (func) (unsafe-blocks :asdf)) (func)) 'error)   ; hop_safe_c_function_star_a bug
  (test (unsafe-blocks 3 :asdf) 'error)
  (test (let () (define (func) (unsafe-blocks 3 :asdf)) (func)) 'error) ; hop_c_aa bug

  (test (let () (define (func) (unsafe-blocks1 :asdf)) (func)) 'error) ; ??
  (test (let () (define (func) (unsafe-blocks1 :fdsa)) (func)) 'error) ; ??
  (test (unsafe-blocks1 3 :asdf) 'error)
  (test (let () (define (func) (unsafe-blocks1 3 :asdf)) (func)) 'error)
  (test (let () (define (func) (unsafe-blocks3 :asdf)) (func)) 'error)
  (test (unsafe-blocks3 3 :asdf) 'error)
  (test (let () (define (func) (unsafe-blocks3 3 :asdf)) (func)) 'error)

  (test (let () (define (func) (unsafe-blocks3 1 3 :asdf)) (func)) 'error)  ; ?? all 6
  (test (unsafe-blocks3 1 3 :asdf) 'error)
  (test (let () (define (func) (unsafe-blocks3 1 3 :asdf)) (func)) 'error)
  (test (let () (define (func) (unsafe-blocks3 1 3 :fdsa)) (func)) 'error)
  (test (unsafe-blocks3 1 3 :fdsa) 'error)
  (test (let () (define (func) (unsafe-blocks3 1 3 :fdsa)) (func)) 'error)

  (test (let () (define (func) (unsafe-blocks1 :asdf)) (func)) 'error)  ; h_c_a
  (test (let () (define (func) (unsafe-blocks1 :fdsa)) (func)) 'error)

  (test (let () (define (func) (unsafe-blocks3 1 3 :asdf)) (func)) 'error)  ; h_c_fx
  (test (unsafe-blocks3 1 3 :asdf) 'error)

  (test (let () (define (func) (unsafe-blocks3 1 3 :asdf)) (func)) 'error)
  (test (let () (define (func) (unsafe-blocks3 1 3 :fdsa)) (func)) 'error)
  (test (unsafe-blocks3 1 3 :fdsa) 'error)
  (test (let () (define (func) (unsafe-blocks3 1 3 :fdsa)) (func)) 'error)

  (test (let () (define (func) (blocks :asdf)) (func)) 'error)
  (test (blocks 3 :asdf) 'error)
  (test (let () (define (func) (blocks 3 :asdf)) (func)) 'error)

  (test (blocks1 :asdf) 'error)
  (test (let () (define (func) (blocks1 :asdf)) (func)) 'error)   ; h_safe_c_d
  (test (let () (define (func) (blocks1 :fdsa)) (func)) 'error)
  (test (let ((sk :rest)) (define (func) (blocks1  sk)) (func)) 'error)

  (test (blocks1 3 :asdf) 'error)
  (test (let () (define (func) (blocks1 3 :asdf)) (func)) 'error)
  (test (let () (define (func) (blocks3 :asdf)) (func)) 'error)
  (test (blocks3 3 :asdf) 'error)
  (test (let () (define (func) (blocks3 3 :asdf)) (func)) 'error)
  (test (let () (define (func) (blocks3 1 3 :asdf)) (func)) 'error)
  (test (blocks3 1 3 :asdf) 'error)

  (test (let () (define (func) (blocks3 1 3 :asdf)) (func)) 'error)  ; h_safe_c_d
  (test (let () (define (func) (blocks3 1 3 :fdsa)) (func)) 'error)
  (test (blocks3 1 3 :fdsa) 'error)

  (test (let () (define (func) (blocks4 :asdf)) (func)) 'error)
  (test (blocks4 1 2 3 :etc) 'error)
  (test (let () (define (func) (blocks4 3 :asdf)) (func)) 'error)
  (test (let () (define (func) (blocks4 1 2 3 :etc)) (func)) 'error)
  (test (blocks4 1 3 :fdsa) 'error)
  (test (let () (define (func) (blocks4 1 2 3 :fdsa)) (func)) 'error)
  (test (blocks4 1 2 3 :fdsa) 'error)

  (test (blocks5) '(4))
  (test (blocks5 :frequency 440) '(440))
  (test (blocks5 :frequency 440 :amplitude 1.0) '(440))
  (test (blocks5 1) '(1))
  (test (blocks5 1 2) 'error) ; error: blocks5: too many arguments: (1 2)
  (test (blocks5 :a 1 :b 2) '(4))
  (test (blocks5 :a 1 :b 2 :frequency 440 :c 3) '(440))
  (test (let () (define (f) (blocks5 :a 1 :frequency 440)) (f)) '(440))
  (test (blocks5 :x) 'error) ; value missing
  (test (let () (define (f) (object->string (blocks5 (values :ho)))) (f)) 'error)
  (test (let () (define (f) (object->string (blocks5 (car (list :ho))))) (f)) 'error)
  (test (let () (define (f) (object->string (blocks4 (car (list :ho))))) (f)) 'error)
  (test (let () (define (f) (blocks5 :ho)) (f)) 'error)
  (test (let () (define (f) (blocks5 (symbol->keyword 'oops))) (f)) 'error)
  (test (let () (define (f) (blocks5 (string->keyword 'oops))) (f)) 'error)

  (test (call/cc (setter (block))) 'error)
  (test (call-with-exit (setter (block))) 'error)
  (test (call-with-input-string "123" (setter (block))) 'error)

  (let ((b (make-simple-block 4)))
    (test (eq? b b) #t)
    (test (equal? b b) #t)
    (test (simple-block? b) #t)
    (test (simple-block? #()) #f)
    (test (simple-block? #f) #f)
    (set! (b 0) 32)
    (test (b 0) 32.0)
    (test (length b) 4)
    (test (substring (object->string b) 0 17) "#<<simple-block> ")
    (test (substring (object->string b :readable) 0 17) "#<<simple-block> ")
    (let ((iter (make-iterator b)))
      (test (iterate iter) 32.0)
      (test (iter) 0.0)
      (test (object->string iter) "#<iterator: <simple-block>>"))
    (test (copy b) 'error)
    (test (reverse b) 'error)
    (test (reverse! b) 'error)
    (test (fill! b) 'error)
    (let ((b1 (make-simple-block 4)))
      (copy b b1)
      (test (equal? b b1) #f)
      (test (b1 0) 32.0)
      (test (append b b1) 'error))
    (s7-optimize '((b 0))))

  (test (make-vector 12 "ho" simple-block?) 'error)
  (test (signature (make-vector 12 (make-simple-block 1) simple-block?))
	(let ((L (list 'simple-block? 'vector? 'integer?)))
	  (set-cdr! (cddr L) (cddr L))
	  L))
  (test (make-vector 12 "ho" block?) 'error)
  (test (signature (make-vector 12 (make-block 1) block?))
	(let ((L (list 'block? 'vector? 'integer?)))
	  (set-cdr! (cddr L) (cddr L))
	  L))
  (test (make-vector 12 "ho" c-tag?) 'error)
  (test (signature (make-vector 12 (make-c-tag) c-tag?)) 'error)

  (test (let ((h (make-hash-table 8 #f (cons symbol? simple-block?)))) (hash-table-set! h 'a 1234)) 'error)
  (test (let ((h (make-hash-table 8 #f (cons symbol? simple-block?)))) (signature h)) '(simple-block? hash-table? symbol?))
  (test (let ((h (make-hash-table 8 #f (cons simple-block? symbol?)))) (hash-table-set! h (make-simple-block 1) 'a)) 'a)
  (test (let ((h (make-hash-table 8 #f (cons simple-block? symbol?)))) (signature h)) '(symbol? hash-table? simple-block?))

  (let ((g (make-cycle "123")))
    (test (cycle-ref g) "123")
    (test (substring (object->string g) 0 10) "#<<cycle> ")
    (test (object->string g :readable) "(<cycle> \"123\")")
    (test (cycle-set! g "321") "321")
    (test (cycle-ref g) "321")
    (test (equal? g 21) #f)
    (test (equal? g g) #t)
    (test (equal? g (make-cycle #\a)) #f)
    (test (equal? g (make-cycle "321")) #t)
    (test (equal? g (<cycle> "321")) #t)
    (cycle-set! g g)
    (let ((g1 (make-cycle g)))
      (test (equal? g g1) #t))
    (set! (g 0) #f)
    (test (cycle-ref g) #f)
    (set! (g 0) g)
    (test (substring (object->string g) 0 13) "#1=#<<cycle> ")
    (test-wi (object->string g :readable) "(let ((<1> (<cycle> #f))) (set! (<1> 0) <1>) <1>)")
    (let ((L (list 1)))
      (cycle-set! g L)
      (set! (L 0) g)
      (test-wi (object->string g :readable)
	       "(let ((<1> (<cycle> #f)))
                  (set! (<1> 0) (let ((<L> (list #f)))
                                  (set-car! <L> <1>)
                                  <L>))
                  <1>)"))
    (let ((g2 (make-cycle g)))
      (test (equal? g g2) #f))
    (let ((L (list 1)))
      (set-cdr! L L)
      (test-wi (object->string (make-cycle L) :readable) "(let ((<2> (<cycle> #f)) (<1> (list 1))) (set! (<2> 0) <1>) (set-cdr! <1> <1>) <2>)"))
    (let ((L (list (<cycle> 2) 3)))
      (set-cdr! (cdr L) L)
      (test-wi (object->string L :readable) "(let ((<1> (list (<cycle> 2) 3))) (set-cdr! (cdr <1>) <1>) <1>)"))
    (let ((L2 (make-list 3 #f))
	  (C (<cycle> #f))
	  (V1 (make-vector 3 #f)))
      (set! (L2 0) V1)
      (set! (V1 0) C)
      (set! (C 0) C)
      (set! (V1 1) L2)
      (test-wi (object->string L2 :readable)
	       "(let ((<3> (list #f #f #f))
                      (<2> (vector #f #f #f))
                      (<1> (<cycle> #f)))
                  (set-car! <3> <2>)
                  (set! (<2> 0) <1>)
                  (set! (<2> 1) <3>)
                  (set! (<1> 0) <1>)
                  <3>)"))
    (let ((L (list #f))
	  (C (make-cycle #f)))
      (set! (L 0) C)
      (let ((IT (make-iterator L)))
	(set! (C 0) IT)
	(test-wi (object->string IT :readable)
		 "(let ((<1> #f)
                        (<3> (list #f))
                        (<2> (<cycle> #f)))
                    (set! <1> (make-iterator <3>))
                    (set-car! <3> <2>)
                    (set! (<2> 0) <1>)
                    <1>)")))
    (let ((cy (make-cycle #f))
	  (it (make-iterator (make-list 3 #f)))
	  (cp (c-pointer 1 (make-list 3 #f))))
      (set! (((object->let cp) 'c-type) 1) cy)
      (set! ((iterator-sequence it) 1) it)
      (set! (cy 0) it)
      (test-wi (object->string cp :readable)
	       "(let ((<4> (list #f #f #f))
                      (<3> (<cycle> #f))
                      (<1> #f)
                      (<2> (list #f #f #f)))
                  (set! <1> (make-iterator <2>))
                  (set! (<4> 1) <3>)
                  (set! (<3> 0) <1>)
                  (set! (<2> 1) <1>)
                  (c-pointer 1 <4> #f))")))

  (let ((b (make-c-tag)))
    (test (eq? b b) #t)
    (test (equal? b b) #t)
    (test (b 0) 'error)
    (test (length b) #f)
    (test (substring (object->string b :readable) 0 8) "#<c-tag ")
    (let ((str (object->string b)))
      (test (substring str 0 8) "#<c-tag ")
      (test (str (- (length str) 1)) #\>))
    (let ((iter (make-iterator b)))
      (test (iterate iter) #<eof>)
      (test (object->string iter) "#<iterator: c-tag>"))
    (test (copy b) 'error)
    (test (reverse b) 'error)
    (test (reverse! b) 'error)
    (test (fill! b) 'error)
    (let ((b1 (make-c-tag)))
      (copy b b1)
      (test (equal? b b1) #f)
      (test (append b b1) 'error))
    (test (setter (make-c-tag)) #f))

  (let ((b (make-c-tag1))) ; checking NULL fields
    (test (eq? b b) #t)
    (test (equal? b b) #t)
    (test (b 0) 'error)
    (test (length b) #f)
    (test (substring (object->string b :readable) 0 8) "#<c-tag1")
    (let ((str (object->string b)))
      (test (substring str 0 8) "#<c-tag1")
      (test (str (- (length str) 1)) #\>))
    (let ((iter (make-iterator b)))
      (test (iterate iter) #<eof>)
      (test (object->string iter) "#<iterator: c-tag1>"))
    (test (copy b) 'error)
    (test (reverse b) 'error)
    (test (reverse! b) 'error)
    (test (fill! b) 'error)
    (let ((b1 (make-c-tag)))
      (copy b b1)
      (test (equal? b b1) #f)
      (test (append b b1) 'error))
    (test (setter (make-c-tag1)) #f))

  (define (fv32)
    (let ((b (block 1 2 3 4))
	  (f (make-float-vector 4)))
      (do ((i 0 (+ i 1)))
	  ((= i 4) f)
	(set! (f i) (+ (b i) 1.0)))))
  (test (fv32) (float-vector 2.0 3.0 4.0 5.0))

  (define (fv33)
    (let ((b (block 1 2 3 4))
	  (f (make-block 4)))
      (do ((i 0 (+ i 1)))
	  ((= i 4) f)
	(set! (f i) (+ (b i) 1.0)))))
  (test (fv33) (block 2.0 3.0 4.0 5.0))

  (define (fv34)
    (let ((b (block 1 2 3 4))
	  (f (make-vector 4)))
      (do ((k 0 (+ k 1)))
	  ((= k 1) f)
	(do ((i 0 (+ i 1)))
	    ((= i 4))
	  (set! (f i) (b i))))))
  (test (fv34) (vector 1.0 2.0 3.0 4.0)))

(when with-block
  (test (pair? (*s7* 'c-types)) #t))

;;; a ridiculous optimizer typo...
(test (let ((sym 'a)) (define (hi a) (eq? (cdr a) sym)) (hi '(a a))) #f)
(test (let ((sym 'a)) (define (hi a) (eq? (cdr a) sym)) (hi '(a . a))) #t)
(test (let ((sym 'a)) (define (hi a) (eq? (cdr a) sym)) (hi '(a . b))) #f)

(for-each
 (lambda (arg)
   (let ((x arg)
	 (y arg))
     (if (not (eq? x x))
	 (format #t ";(eq? x x) of ~A -> #f?~%" x))
     (if (not (eq? x arg))
	 (format #t ";(eq? x arg) of ~A ~A -> #f?~%" x arg))
     (if (not (eq? x y))
	 (format #t ";(eq? x y) of ~A ~A -> #f?~%" x y))))
 ;; actually I hear that #f is ok here for numbers
 (list "hi" '(1 2) (integer->char 65) 1 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3/4 #\f (lambda (a) (+ a 1)) :hi (if #f #f) #<eof> #<undefined>))
;; this used to include 3.14 and 1+i but that means the (eq? x x) case differs from the (eq? 3.14 3.14) case

(define comment-start (port-line-number))
#|
:'(1(1))
(1 (1))
:'(1#(1))
(1# (1))
|#
(if (not (= (- (port-line-number) comment-start) 7)) (format *stderr* ";block comment newline counter: ~D ~D~%" comment-start (port-line-number)))
(test (eval-string "|#") 'error)

;;; this comes from G Sussman
(let ()
  (define (counter count)
    (lambda ()
      (set! count (+ 1 count))
      count))

  (define c1 (counter 0))
  (define c2 (counter 0))

  (test (eq? c1 c2) #f)
  (test (eq? c1 c1) #t)
  (test (eq? c2 c2) #t)

  (test (let ((p (lambda (x) x))) (eqv? p p)) #t)
  (for-each
   (lambda (arg)
     (if (not ((lambda (p) (eq? p p)) arg))
	 (format #t "~A not eq? to itself?~%" arg)))
   (list "hi" '(1 2) (integer->char 65) 1 'a-symbol (make-vector 3) abs quasiquote macroexpand 1/0 (log 0)
	 3.14 3/4 1.0+1.0i #\f (lambda (a) (+ a 1)) :hi (if #f #f) #<eof> #<undefined> '(1 2 . 3)
	 (let ((lst (list 1 2)))
	   (set! (cdr (cdr lst)) lst)
	   lst)
	 (vector) (string) (list)
	 (let ((x 3))
	   (lambda (y) (+ x y))))))

;;; this for r7rs
(test (eq? #t #true) #t)
(test (eq? #f #false) #t)
(test (eq? () (map values ())) #t)

(let () (define (f2) f2) (test (eq? f2 (f2)) #t))
(letrec ((f2 (lambda () f2))) (test (eq? f2 (f2)) #t))


;;; --------------------------------------------------------------------------------
;;; eqv?

(test (eqv? 'a 3) #f)
(test (eqv? #t 't) #f)
(test (eqv? "abs" 'abc) #f)
(test (eqv? "hi" '(hi)) #f)
(test (eqv? "()" ()) #f)
(test (eqv? '(1) '(1)) #f)
(test (eqv? '(#f) '(#f)) #f)
(test (eqv? #\a #\b) #f)
(test (eqv? #\a #\a) #t)
(test (eqv? (integer->char 255) (string-ref (string #\x (integer->char 255) #\x) 1)) #t)
(test (eqv? (integer->char #xf0) (integer->char #x70)) #f)
(test (eqv? #\space #\space) #t)
(test (let ((x (string-ref "hi" 0))) (eqv? x x)) #t)
(test (eqv? #t #t) #t)
(test (eqv? #f #f) #t)
(test (eqv? #f #t) #f)
(test (eqv? (null? ()) #t) #t)
(test (eqv? (null? '(a)) #f) #t)
(test (eqv? (cdr '(a)) '()) #t)
(test (eqv? 'a 'a) #t)
(test (eqv? 'a 'b) #f)
(test (eqv? 'a (string->symbol "a")) #t)
(test (eqv? '(a) '(b)) #f)
(test (let ((x '(a . b))) (eqv? x x)) #t)
(test (let ((x (cons 'a 'b))) (eqv? x x)) #t)
(test (eqv? (cons 'a 'b) (cons 'a 'b)) #f)
(test (eqv? "abc" "cba") #f)
(test (let ((x "hi")) (eqv? x x)) #t)
(test (eqv? (string #\h #\i) (string #\h #\i)) #f)
(test (eqv? #(a) #(b)) #f)
(test (let ((x (vector 'a))) (eqv? x x)) #t)
(test (eqv? (vector 'a) (vector 'a)) #f)
(test (eqv? car car) #t)
(test (eqv? car cdr) #f)
(test (let ((x (lambda () 1))) (eqv? x x)) #t)
(test (eqv? (lambda () 1) (lambda () 1)) #f)
(test (let () (define (make-adder x) (lambda (y) (+ x y))) (eqv? (make-adder 1) (make-adder 1))) #f)
(test (eqv? 9/2 9/2) #t)
(test (eqv? quote quote) #t)
(test (eqv? () ()) #t)
(test (eqv? () '()) #t)
;(test (eqv? "" "") #f)
(test (eqv? "hi" "hi") #f) ; unspecified
(test (eqv? #() #()) #f)   ; unspecified, but in s7 (eqv? () ()) is #t
(test (eqv? (vector) (vector)) #f)

(let ((c1 (let ((x 32))
	    (lambda () x)))
      (c2 (let ((x 123))
	    (lambda () x))))
  (test (eqv? c1 c2) #f)
  (test (eqv? c1 c1) #t))

(test (eqv? most-positive-fixnum most-positive-fixnum) #t)
(test (eqv? most-positive-fixnum most-negative-fixnum) #f)
(test (eqv? 9223372036854775807 9223372036854775806) #f)
(test (eqv? 9223372036854775807 -9223372036854775808) #f)
(test (eqv? -9223372036854775808 -9223372036854775808) #t)
(test (eqv? 123456789/2 123456789/2) #t)
(test (eqv? 123456789/2 123456787/2) #f)
(test (eqv? -123456789/2 -123456789/2) #t)
(test (eqv? 2/123456789 2/123456789) #t)
(test (eqv? -2/123456789 -2/123456789) #t)
(test (eqv? 2147483647/2147483646 2147483647/2147483646) #t)
(test (eqv? 3/4 12/16) #t)
(test (eqv? 1/1 1) #t)
(test (eqv? 312689/99532 833719/265381) #f)
(test (let ((x 3.141)) (eqv? x x)) #t)
(test (let ((x 1+i)) (eqv? x x)) #t)
(test (let* ((x 3.141) (y x)) (eqv? x y)) #t)
(test (let* ((x 1+i) (y x)) (eqv? x y)) #t)
(test (let* ((x 3/4) (y x)) (eqv? x y)) #t)
(test (eqv? 1.0 1.0) #t)
(test (eqv? 0.6 0.6) #t)
(test (eqv? 0.6 0.60) #t)
(test (eqv? 1+i 1+i) #t)
(test (eqv? -3.14 -3.14) #t)
(test (eqv? 1e2 1e2) #t)
(test (eqv? 1 1.0) #f)
(test (eqv? 1/2 0.5) #f)
(test (eqv? 1 1/1) #t)
(test (eqv? 0.5 5e-1) #t)
(test (eqv? 1/0 1/0) #f)
(test (eqv? +nan.0 +nan.0) #f)

(test (eqv? (cons 'a 'b) (cons 'a 'c)) #f)
(test (eqv? eqv? eqv?) #t)
(test (eqv? #(1) #(1)) #f)
(test (eqv? '(1) '(1)) #f)
(test (eqv? '() '()) #t)
(test (eqv? '() (list)) #t)
(test (eqv? '(()) '(())) #f)
(test (eqv? (list 'abs 'cons) '(abs cons)) #f)

(define (feqv)
  (let ((things (vector #t #f #\space '() "" 0 1 3/4 1+i 1.5 '(1 .2) #() (vector) (vector 1) (list 1) 'f 't #\t)))
    (let ((len (length things)))
      (do ((i 0 (+ i 1)))
	  ((= i (- len 1)))
        (do ((j (+ i 1) (+ j 1)))
	    ((= j len))
	  (if (eqv? (vector-ref things i) (vector-ref things j))
	      (format #t ";(eqv? ~A ~A) -> #t?~%" (vector-ref things i) (vector-ref things j))))))))
(feqv)

(test (eqv?) 'error)
(test (eqv? #t) 'error)
(test (eqv? #t #t #t) 'error)
(test (eqv #f #f) 'error)

(test (eqv? ''2 '2) #f)
(test (eqv? '2 '2) #t)
(test (eqv? '2 2) #t)
(test (eqv? ''2 ''2) #f)
(test (eqv? ''#\a '#\a) #f)
(test (eqv? '#\a #\a) #t)
(test (eqv? 'car car) #f)
(test (eqv? ''() '()) #f)
(test (eqv? '#f #f) #t)
(test (eqv? '#f '#f) #t)
(test (eqv? #<eof> #<eof>) #t)
(test (eqv? #<undefined> #<undefined>) #t)
(test (eqv? #<unspecified> #<unspecified>) #t)
(test (eqv? (if #f #f) #<unspecified>) #t)
(test (eqv? #<eof> #<undefined>) #f)
(test (eqv? #<eof> '()) #f)
(test (let () (define-macro (hi a) `(+ 1 ,a)) (eqv? hi hi)) #t)
(test (let () (define (hi a) (+ 1 a)) (eqv? hi hi)) #t)
(test (let ((x (lambda* (hi (a 1)) (+ 1 a)))) (eqv? x x)) #t)
(test (eqv? else else) #t)
(test (let ((p (lambda (x) x))) (eqv? p p)) #t)
(test (eqv? #<undefined> _undef_) #f)
(test (eqv? _undef_ _undef_) #t)

(test (eqv? :a 'a) #f)
(test (eqv? :a a:) #f)
(test (eqv? :a :a) #t)

(when with-bignums
  (test (eqv? (bignum "1+i") (bignum "1+i")) #t)
  (test (eqv? (bignum "1+i") 1+i) #t)
  (test (eqv? 1+i (bignum "1+i")) #t)
  (test (eqv? (bignum "2.0") (bignum "2.0")) #t)
  (test (eqv? (bignum "2.0") (bignum "1.0")) #f)
  (test (eqv? (bignum +nan.0) (bignum 2.0)) #f)
  (test (eqv? (bignum +nan.0) 2.0) #f)
  (test (eqv? (bignum +nan.0) +nan.0) #f)
  (test (eqv? (complex +nan.0 1.0) (bignum (complex +nan.0 1.0))) #f))

;; from M Weaver:
(test (list (eqv? +0.0 -0.0)
	    (eqv? (complex +0.0  1.0)
		  (complex -0.0  1.0))
	    (eqv? (complex  1.0 +0.0)
		  (complex  1.0 -0.0)))
      '(#t #t #t))
(test (list ;(eq? +0.0 -0.0)
        (eq? (complex  +0.0  1.0)
              (complex -0.0  1.0))
        (eq? (complex   1.0 +0.0)
              (complex  1.0 -0.0)))
      '(#f #f))
(test (list (eq? +0 -0)
        (eq? (complex  +0  1)
              (complex -0  1))
        (eq? (complex   1 +0)
              (complex  1 -0)))
      '(#t #f #t))




;;; --------------------------------------------------------------------------------
;;; equal?

(test (equal? 'a 3) #f)
(test (equal? #t 't) #f)
(test (equal? "abs" 'abc) #f)
(test (equal? "hi" '(hi)) #f)
(test (equal? "()" '()) #f)
(test (equal? '(1) '(1)) #t)
(test (equal? '(#f) '(#f)) #t)
(test (equal? '(()) '(() . ())) #t)
(test (equal? #\a #\b) #f)
(test (equal? #\a #\a) #t)
(test (let ((x (string-ref "hi" 0))) (equal? x x)) #t)
(test (equal? #t #t) #t)
(test (equal? #f #f) #t)
(test (equal? #f #t) #f)
(test (equal? (null? '()) #t) #t)
(test (equal? (null? '(a)) #f) #t)
(test (equal? (cdr '(a)) '()) #t)
(test (equal? 'a 'a) #t)
(test (equal? 'a 'b) #f)
(test (equal? 'a (string->symbol "a")) #t)
(test (equal? '(a) '(b)) #f)
(test (equal? '(a) '(a)) #t)
(test (let ((x '(a . b))) (equal? x x)) #t)
(test (let ((x (cons 'a 'b))) (equal? x x)) #t)
(test (equal? (cons 'a 'b) (cons 'a 'b)) #t)
(test (equal?(cons 'a 'b)(cons 'a 'b)) #t) ; no space
(test (equal? "abc" "cba") #f)
(test (equal? "abc" "abc") #t)
(test (let ((x "hi")) (equal? x x)) #t)
(test (equal? (string #\h #\i) (string #\h #\i)) #t)
(test (equal? #(a) #(b)) #f)
(test (equal? #(a) #(a)) #t)
(test (let ((x (vector 'a))) (equal? x x)) #t)
(test (equal? (vector 'a) (vector 'a)) #t)
(test (equal? #(1 2) (vector 1 2)) #t)
(test (equal? #(1.0 2/3) (vector 1.0 2/3)) #t)
(test (equal? #(1 2) (vector 1 2.0)) #f) ; 2 not equal 2.0!
(test (equal? '(1 . 2) (cons 1 2)) #t)
(test (equal? '(1 #||# . #||# 2) (cons 1 2)) #t)
(test (- '#||#1) -1) ; hmm
(test (equal? #(1 "hi" #\a) (vector 1 "hi" #\a)) #t)
(test (equal? #((1 . 2)) (vector (cons 1 2))) #t)
(test (equal? #(1 "hi" #\a (1 . 2)) (vector 1 "hi" #\a (cons 1 2))) #t)
(test (equal? #(#f hi (1 2) 1 "hi" #\a (1 . 2)) (vector #f 'hi (list 1 2) 1 "hi" #\a (cons 1 2))) #t)
(test (equal? #(#(1) #(1)) (vector (vector 1) (vector 1))) #t)
(test (equal? #(()) (vector '())) #t)
(test (equal? #("hi" "ho") (vector "hi" '"ho")) #t)
(test (equal? `#(1) #(1)) #t)
(test (equal? ``#(1) #(1)) #t)
(test (equal? '`#(1) #(1)) #t)
(test (equal? ''#(1) #(1)) #f)
(test (equal? ''#(1) '#(1)) #f)
(test (equal? '(1) '        (   1    )) #t)
(test (equal? (list 1 "hi" #\a) '(1 "hi" #\a)) #t)
(test (equal? (list 1.0 2/3) '(1.0 2/3)) #t)
(test (equal? (list 1 2) '(1 2.0)) #f)
(test (equal? #(1.0+1.0i) (vector 1.0+1.0i)) #t)
(test (equal? (list 1.0+1.0i) '(1.0+1.0i)) #t)
(test (equal? '((())) (list (list (list)))) #t)
(test (equal? '((())) (cons (cons () ()) ())) #t)
(test (equal? car car) #t)
(test (equal? car cdr) #f)
(test (let ((x (lambda () 1))) (equal? x x)) #t)
(test (equal? (lambda () 1) (lambda () 1)) #f)
(test (equal? 9/2 9/2) #t)
(test (equal? #((())) #((()))) #t)
(test (equal? "123""123") #t);no space
(test (equal? """") #t)#|nospace|#
(test (equal? #()#()) #t)
(test (equal? #()()) #f)
(test (equal? ()"") #f)
(test (equal? "hi""hi") #t)
(test (equal? #<eof> #<eof>) #t)
(test (equal? #<undefined> #<undefined>) #t)
(test (equal? #<unspecified> #<unspecified>) #t)
(test (equal? (if #f #f) #<unspecified>) #t)
(test (equal? #<eof> #<undefined>) #f)
(test (equal? (values) #<eof>) #f)
(test (equal? (values) (values)) #t)
(test (equal? #<eof> #<unspecified>) #f)
(test (equal? (values) #<unspecified>) #t)
(test (equal? #<unspecified> (values)) #t)
(test (equal? #<eof> ()) #f)
(test (let () (define-macro (hi a) `(+ 1 ,a)) (equal? hi hi)) #t)
(test (let () (define (hi a) (+ 1 a)) (equal? hi hi)) #t)
(test (let ((x (lambda* (hi (a 1)) (+ 1 a)))) (equal? x x)) #t)
(test (equal? ``"" '"") #t)
(test (let ((pws (dilambda (lambda () 1) (lambda (x) x)))) (equal? pws pws)) #t)
(test (equal? if :if) #f)
(test (equal? (list 'abs 'cons) '(abs cons)) #t)
(test (equal? '(1) '(list 1)) #f)
(test (equal? #<undefined> _undef_) #f)
(test (equal? _undef_ _undef_) #t)
(test (equal? (list (values)) (list #<unspecified>)) #t)

(test (equal? most-positive-fixnum most-positive-fixnum) #t)
(test (equal? most-positive-fixnum most-negative-fixnum) #f)
(test (equal? pi pi) #t)
(test (equal? 9223372036854775807 9223372036854775806) #f)
(test (equal? 9223372036854775807 -9223372036854775808) #f)
(test (equal? -9223372036854775808 -9223372036854775808) #t)
(test (equal? 123456789/2 123456789/2) #t)
(test (equal? 123456789/2 123456787/2) #f)
(test (equal? -123456789/2 -123456789/2) #t)
(test (equal? 2/123456789 2/123456789) #t)
(test (equal? -2/123456789 -2/123456789) #t)
(test (equal? 2147483647/2147483646 2147483647/2147483646) #t)
(test (equal? 3/4 12/16) #t)
(test (equal? 1/1 1) #t)
(test (equal? 312689/99532 833719/265381) #f)
(test (let ((x 3.141)) (equal? x x)) #t)
(test (let ((x 1+i)) (equal? x x)) #t)
(test (let* ((x 3.141) (y x)) (equal? x y)) #t)
(test (let* ((x 1+i) (y x)) (equal? x y)) #t)
(test (let* ((x 3/4) (y x)) (equal? x y)) #t)
(test (equal? '(+ '1) '(+ 1)) #f) ; !?

(test (equal? '(1/0) '(1/0)) #f)
(test (equal? '1/0 '1/0) #f)
(test (equal? '(+nan.0) '(+nan.0)) #f)
(test (equal? (list +nan.0) (list +nan.0)) #f)
(test (equal? (vector +nan.0) (vector +nan.0)) #f)
(test (equal? #(1/0) #(1/0)) #f)
(test (equal? #r(0.0) #r(-0.0)) #t)
(test (equal? (float-vector) (int-vector)) #t)
(test (equal? (vector) (byte-vector)) #t)
(test (equal? (string) (byte-vector)) #f)
(test (equal? (make-vector 0 #f) (make-float-vector 0 1.0)) #t)
(test (equal? (make-vector 0 #f boolean?) (make-float-vector 0 1.0)) #t)
(test (equal? (make-vector '(0 1) #f boolean?) (make-float-vector 0 1.0)) #f)
(test (equal? (make-vector '(0 1) #f boolean?) (make-float-vector '(0 1) 1.0)) #t)
(let ((it1 (make-iterator (int-vector 1 2 3)))
      (it2 (make-iterator (int-vector 1 2 3))))
  (test (equal? it1 it2) #t))
(let ((it1 (make-iterator (int-vector 1 2 3)))
      (it2 (make-iterator (vector pi))))
  (test (equal? it1 it2) #f))
(let ((it1 (make-iterator (int-vector 1 2 3)))
      (it2 (make-iterator (int-vector 1 2 3))))
  (test (equivalent? it1 it2) #t))
(let ((it1 (make-iterator (int-vector 1 2 3)))
      (it2 (make-iterator (vector 1.0 2.0 3.0))))
  (test (equivalent? it1 it2) #t))

(test (equal? 3 3) #t)
(test (equal? 3 3.0) #f)
(test (equal? 3.0 3.0) #t)
(test (equal? 3-4i 3-4i) #t)
(test (equal? (string #\c) "c") #t)
(test (equal? equal? equal?) #t)
(test (equal? (cons 1 (cons 2 3)) '(1 2 . 3)) #t)
(test (equal? '() '()) #t)
(test (equal? '() (list)) #t)
(test (equal? (cdr '   ''0) '((quote 0))) #t)
(test (equal? "\n" "\n") #t)
(test (equal? #f ((lambda () #f))) #t)
(test (equal? (+) 0) #t)
(test (equal? (recompose 32 list '(1)) (recompose 32 list (list 1))) #t)
(test (equal? (recompose 100 list '(1)) (recompose 100 list (list 1))) #t)
(test (equal? (recompose 32 vector 1) (recompose 32 vector 1)) #t)
(test (equal? (reinvert 32 list vector 1) (reinvert 32 list vector 1)) #t)
(test (equal? (recompose 32 (lambda (a) (cons 1 a)) ()) (recompose 32 (lambda (a) (cons 1 a)) ())) #t)
(test (equal? (recompose 32 (lambda (a) (list 1 a)) ()) (recompose 32 (lambda (a) (list 1 a)) ())) #t)

(test (equal? "asd""asd") #t) ; is this the norm?
(let ((streq (lambda (a b) (equal? a b)))) (test (streq "asd""asd") #t))

(define (fequal)
  (let ((things (vector #t #f #\space () "" 0 1 3/4 1+i 1.5 '(1 .2) #() (vector 1) (list 1) 'f 't #\t)))
    (let ((len (length things)))
      (do ((i 0 (+ i 1)))
	  ((= i (- len 1)))
        (do ((j (+ i 1) (+ j 1)))
	    ((= j len))
	  (if (equal? (vector-ref things i) (vector-ref things j))
	      (format #t ";(equal? ~A ~A) -> #t?~%" (vector-ref things i) (vector-ref things j))))))))
(fequal)

(test (equal?) 'error)
(test (equal? #t) 'error)
(test (equal? #t #t #t) 'error)
(test (equal #t #t) 'error)

(when with-block
  (let ((b (make-block 4)))
    (test (equal? b b) #t)
    (let ((b1 (make-block 4)))
      (test (equal? b b1) #t)
      (set! (b 1) 1.0)
      (test (equal? b b1) #f))))
(test (let ((p (c-pointer 0))) (equal? p (copy p))) #t)

(test (call-with-exit (lambda (return) (return (equal? return return)))) #t)
(test (call-with-exit (lambda (return) (call-with-exit (lambda (quit) (return (equal? return quit)))))) #f)
(test (call/cc (lambda (return) (return (equal? return return)))) #t)
(test (let hiho ((i 0)) (equal? hiho hiho)) #t)
(test (let hiho ((i 0)) (let hoho ((i 0)) (equal? hiho hoho))) #f)
(test (equal? + *) #f)
(test (equal? lambda lambda) #t)
(test (equal? lambda lambda*) #f)
(test (equal? let let) #t)
(test (equal? let letrec) #f)
(test (equal? define define) #t)
(test (equal? + ((lambda (a) a) +)) #t)
(test (let ((x "hi")) (define (hi) x) (equal? (hi) (hi))) #t)

;; so (eq? 3/4 3/4) is #f, (eqv? 3/4 3/4) is #t,
;;    (eqv? #(1) #(1)) is #f, (equal? #(1) #(1)) is #t
;;    (equal? 3 3.0) is #f, (= 3 3.0) is #t
;; in s7
;;    (eq? 0.0 0.0) is #t,
;;    (eq? 2.0 2.0) is #f
(test (equal? .0 0.) #t)
(test (equal?
       (list "hi" (integer->char 65) 1 'a-symbol (make-vector 3) (list) (cons 1 2) abs quasiquote 3 3/4 1.0+1.0i #\f (if #f #f) #<eof> #<undefined>)
       (list "hi" (integer->char 65) 1 'a-symbol (make-vector 3) (list) (cons 1 2) abs quasiquote 3 3/4 1.0+1.0i #\f (if #f #f) #<eof> #<undefined>))
      #t)
(test (equal?
       (vector "hi" (integer->char 65) 1 'a-symbol (make-vector 3) abs quasiquote 3 3/4 1.0+1.0i #\f (if #f #f) #<eof> #<undefined>)
       (vector "hi" (integer->char 65) 1 'a-symbol (make-vector 3) abs quasiquote 3 3/4 1.0+1.0i #\f (if #f #f) #<eof> #<undefined>))
      #t)
(test (equal? (make-string 3 #\null) (make-string 3 #\null)) #t)
(test (equal? (make-list 3) (make-list 3)) #t)
(test (equal? (make-vector 3) (make-vector 3)) #t)
(unless with-bignums (test (equal? (random-state 100) (random-state 100)) #t))

(test (equal? (list 1) (immutable! (list 1))) #t) ; we also ignore shared lists, and if safety>1, list constants via quote shouldn't change equal?
(test (equal? (list 1) '(1)) #t)

(test (equal? (current-input-port) (current-input-port)) #t)
(test (equal? (current-input-port) (current-output-port)) #f)
(test (equal? *stdin* *stderr*) #f)
(test (let ((l1 (list 'a 'b))
	    (l2 (list 'a 'b 'a 'b)))
	(set! (cdr (cdr l1)) l1)
	(set! (cdr (cdr (cdr (cdr l2)))) l2)
	(equal? l1 l2))
      #t)
(test (let ((l1 (list 'a 'b))
	    (l2 (list 'a 'b 'a)))
	(set! (cdr (cdr l1)) l1)
	(set! (cdr (cdr (cdr l2))) l2)
	(equal? l1 l2))
      #f)
(test (let ((v1 (vector 1 2 3))
	    (v2 (vector 1 2 3)))
	(set! (v1 1) v1)
	(set! (v2 1) v2)
	(equal? v1 v2))
      #t)
(test (let ((v1 (vector 1 2 3))
	    (v2 (vector 1 2 4)))
	(set! (v1 1) v1)
	(set! (v2 1) v2)
	(equal? v1 v2))
      #f)

(when with-bignums
  (test (equal? (/ (* 5 most-positive-fixnum) (* 3 most-negative-fixnum)) -46116860184273879035/27670116110564327424) #t)
  (test (equal? +nan.0 (bignum 2.0)) #f)
  (test (equal? +nan.0 (bignum 2)) #f)
  (test (equal? +nan.0 (bignum 1/2)) #f)
  (test (equal? +nan.0 (bignum 2.0+i)) #f)
  (test (equal? (bignum 2.0) +nan.0) #f)
  (test (equivalent? (bignum 2.0) +nan.0) #f)
  (test (equivalent? +nan.0 (bignum 2.0)) #f)
  (test (equal? 2.0 (bignum 2.0)) #t)
  (test (equal? 2 (bignum 2)) #t)
  (test (eqv? 2 (bignum 2)) #t)
  (test (= 2 (bignum 2)) #t)
  (test (equal? (bignum 2.0) 2.0) #t)
  (test (equal? (bignum 2) 2) #t)
  (test (eqv? (bignum 2) 2) #t)
  (test (= (bignum 2) 2) #t))

;;; cyclic hash key tests
(let* ((H1 (hash-table))
       (I1 (inlet 'a H1)))
  (set! (H1 'a) I1)
  (let* ((H2 (hash-table))
	 (I2 (inlet 'a H2)))
    (set! (H2 'a) I2)
    (test (equal? H1 H2) #t)
    (test (equivalent? H1 H2) #t)))

(let* ((H1 (hash-table))
       (I1 (immutable! (openlet (inlet 'a H1)))))
  (set! (H1 'a) I1)
  (let* ((H2 (hash-table))
	 (I2 (immutable! (openlet (inlet 'a H2)))))
    (set! (H2 'a) I2)
    (test (equal? H1 H2) #t)
    (test (equivalent? H1 H2) #t)))

(let* ((H1 (hash-table))
       (I1 (inlet 'a H1)))
  (set! (H1 I1) I1)
  (let* ((H2 (hash-table))
	 (I2 (inlet 'a H2)))
    (set! (H2 I2) I2)
    (test (equal? H1 H2) #t)
    (test (equivalent? H1 H2) #t)))

(let* ((H1 (hash-table))
       (I1 (inlet 'a H1))
       (V1 (make-vector 4 I1)))
  (set! (H1 V1) I1)
  (let* ((H2 (hash-table))
	 (I2 (inlet 'a H2))
	 (V2 (make-vector 4 I2)))
    (set! (H2 V2) I2)
    (test (equal? H1 H2) #t)
    (test (equivalent? H1 H2) #t)))

(let ((H1 (list 1))
      (H2 (list 1)))
  (let ((V1 (vector H1))
	(V2 (vector H2)))
    (set! (H2 0) V2)
    (set! (H1 0) V1)
    (test (equal? H1 H2) #t)
    (test (equivalent? H1 H2) #t)
    (test (object->string H1) "#1=(#(#1#))")))

(let ((H1 (inlet 'a 1))
      (H2 (inlet 'a 1)))
  (let ((V1 (vector H1))
	(V2 (vector H2)))
    (set! (H2 'a) V2)
    (set! (H1 'a) V1)
    (test (equal? H1 H2) #t)
    (test (equivalent? H1 H2) #t)
    (test (object->string H1) "#1=(inlet 'a #(#1#))")))

(let ((H1 (hash-table))
      (H2 (hash-table))
      (H3 (hash-table)))
  (let ((V1 (vector H1))
	(V2 (vector H2)))
    (set! (H2 V2) H2)
    (set! (H1 V1) H1)
    (set! (H3 H1) 2)
    (test (H3 H2) 2)))

(let ((H1 (hash-table))
      (H2 (hash-table)))
  (let ((V1 (vector H1))
	(V2 (vector H2)))
    (set! (H2 V2) H2)
    (set! (H1 V1) H1)
    (test (object->string V1) "#1=#(#2=(hash-table #1# #2#))")
    (test (equal? V1 V2) #t)
    (test (equivalent? V1 V2) #t)))

(let ((H1 (hash-table))
      (H2 (hash-table)))
  (let ((V1 (vector H1))
	(V2 (vector H2)))
    (set! (H2 V2) H2)
    (set! (H1 V1) H1)
    (test (object->string H1) "#1=(hash-table #(#1#) #1#)")
    (test (equal? H1 H2) #t)
    (test (equivalent? H1 H2) #t)))


;;; --------------------------------------------------------------------------------
;;; equivalent?

(test (equivalent? 'a 3) #f)
(test (equivalent? #t 't) #f)
(test (equivalent? "abs" 'abc) #f)
(test (equivalent? "hi" '(hi)) #f)
(test (equivalent? "()" '()) #f)
(test (equivalent? '(1) '(1)) #t)
(test (equivalent? '(#f) '(#f)) #t)
(test (equivalent? '(()) '(() . ())) #t)
(test (equivalent? #\a #\b) #f)
(test (equivalent? #\a #\a) #t)
(test (let ((x (string-ref "hi" 0))) (equivalent? x x)) #t)
(test (equivalent? #t #t) #t)
(test (equivalent? #f #f) #t)
(test (equivalent? #f #t) #f)
(test (equivalent? (null? '()) #t) #t)
(test (equivalent? (null? '(a)) #f) #t)
(test (equivalent? (cdr '(a)) '()) #t)
(test (equivalent? 'a 'a) #t)
(test (equivalent? 'a 'b) #f)
(test (equivalent? 'a (string->symbol "a")) #t)
(test (equivalent? '(a) '(b)) #f)
(test (equivalent? '(a) '(a)) #t)
(test (let ((x '(a . b))) (equivalent? x x)) #t)
(test (let ((x (cons 'a 'b))) (equivalent? x x)) #t)
(test (equivalent? (cons 'a 'b) (cons 'a 'b)) #t)
(test (equivalent?(cons 'a 'b)(cons 'a 'b)) #t) ; no space
(test (equivalent? "abc" "cba") #f)
(test (equivalent? "abc" "abc") #t)
(test (let ((x "hi")) (equivalent? x x)) #t)
(test (equivalent? (string #\h #\i) (string #\h #\i)) #t)
(test (equivalent? #(a) #(b)) #f)
(test (equivalent? #(a) #(a)) #t)
(test (let ((x (vector 'a))) (equivalent? x x)) #t)
(test (equivalent? (vector 'a) (vector 'a)) #t)
(test (equivalent? #(1 2) (vector 1 2)) #t)
(test (equivalent? #(1.0 2/3) (vector 1.0 2/3)) #t)
(test (equivalent? #(1 2) (vector 1 2.0)) #t)
(test (equivalent? '(1 . 2) (cons 1 2)) #t)
(test (equivalent? '(1 #||# . #||# 2) (cons 1 2)) #t)
(test (- '#||#1) -1) ; hmm
(test (equivalent? #(1 "hi" #\a) (vector 1 "hi" #\a)) #t)
(test (equivalent? #((1 . 2)) (vector (cons 1 2))) #t)
(test (equivalent? #(1 "hi" #\a (1 . 2)) (vector 1 "hi" #\a (cons 1 2))) #t)
(test (equivalent? #(#f hi (1 2) 1 "hi" #\a (1 . 2)) (vector #f 'hi (list 1 2) 1 "hi" #\a (cons 1 2))) #t)
(test (equivalent? #(#(1) #(1)) (vector (vector 1) (vector 1))) #t)
(test (equivalent? #(()) (vector '())) #t)
(test (equivalent? #("hi" "ho") (vector "hi" '"ho")) #t)
(test (equivalent? `#(1) #(1)) #t)
(test (equivalent? ``#(1) #(1)) #t)
(test (equivalent? '`#(1) #(1)) #t)
(test (equivalent? ''#(1) #(1)) #f)
(test (equivalent? ''#(1) '#(1)) #f)
(test (equivalent? (list 1 "hi" #\a) '(1 "hi" #\a)) #t)
(test (equivalent? (list 1.0 2/3) '(1.0 2/3)) #t)
(test (equivalent? (list 1 2) '(1 2.0)) #t)
(test (equivalent? #(1.0+1.0i) (vector 1.0+1.0i)) #t)
(test (equivalent? (list 1.0+1.0i) '(1.0+1.0i)) #t)
(test (equivalent? '((())) (list (list (list)))) #t)
(test (equivalent? car car) #t)
(test (equivalent? car cdr) #f)
(test (let ((x (lambda () 1))) (equivalent? x x)) #t)
(test (equivalent? (lambda () 1) (lambda () 1)) #t)
(test (equivalent? 9/2 9/2) #t)
(test (equivalent? #((())) #((()))) #t)
(test (equivalent? "123""123") #t);no space
(test (equivalent? """") #t)#|nospace|#
(test (equivalent? #()#()) #t)
(test (equivalent? #()()) #f)
(test (equivalent? ()"") #f)
(test (equivalent? "hi""hi") #t)
(test (equivalent? #<eof> #<eof>) #t)
(test (equivalent? #<undefined> #<undefined>) #t)
(test (equivalent? #<unspecified> #<unspecified>) #t)
(test (equivalent? (if #f #f) #<unspecified>) #t)
(test (equivalent? #<eof> #<undefined>) #f)
(test (equivalent? #<eof> '()) #f)
(test (equivalent? (values) #<eof>) #f)
(test (equivalent? #<eof> (values)) #f)
(test (equivalent? (values) (values)) #t)
(test (equivalent? #<eof> #<unspecified>) #f)
(test (equivalent? (values) #<unspecified>) #t)
(test (equivalent? #<unspecified> (values)) #t)
(test (equivalent? #<undefined> _undef_) #f)
(test (equivalent? _undef_ _undef_) #t)
(test (let () (define-macro (hi a) `(+ 1 ,a)) (equivalent? hi hi)) #t)
(test (let () (define (hi a) (+ 1 a)) (equivalent? hi hi)) #t)
(test (let ((x (lambda* (hi (a 1)) (+ 1 a)))) (equivalent? x x)) #t)
(test (equivalent? ``"" '"") #t)
(test (let ((pws (dilambda (lambda () 1) (lambda (x) x)))) (equivalent? pws pws)) #t)
(test (equivalent? if :if) #f)
(test (equivalent? (list 'abs 'cons) '(abs cons)) #t)
(test (equivalent? (make-int-vector 2 0) (vector 0 0)) #t)
(test (equivalent? (make-int-vector 2 0) (make-vector 2 0)) #t)
(test (equivalent? (make-int-vector 2 0) (make-int-vector 2 0)) #t)
(test (equivalent? (make-int-vector 2 0) (make-float-vector 2)) #t)
(test (equivalent? (vector 0.0 0) (make-float-vector 2 0.0)) #t)
(test (equivalent? (make-int-vector 2 0) (vector 0 1.0)) #f)
(test (equivalent? (make-float-vector 1 +nan.0) (make-float-vector 1 +nan.0)) #t)
(test (call-with-input-file "s7test.scm"
	(lambda (port)
	  (equivalent? (call-with-input-file "s7test.scm" (lambda (p) p)) port)))
      #f)

(test (equivalent? (make-iterator "") (make-iterator "")) #t)
(test (equivalent? (make-iterator "1") (make-iterator "1" (cons 1 1))) #t)
(test (equivalent? (make-iterator "1" (cons 3 4)) (make-iterator "1" (cons 1 1))) #t)
(test (equivalent? (make-iterator #()) (make-iterator #())) #t)
(test (equivalent? (make-iterator '(1 2 3)) (make-iterator '(1 2 3))) #t)
(test (equivalent? (make-iterator '(1 2 3)) (make-iterator '(1 2 3 4))) #f)

(test (equivalent? (int-vector 1 2) (byte-vector 1 2)) #t)
(test (equivalent? (vector 1 2) (byte-vector 1 2)) #t)       ; but (equivalent? (vector #\a) "a") -> #f
(test (equivalent? (float-vector 1 2) (byte-vector 1 2)) #t) ; (equivalent? 1 1.0) -> #t

(let ((str "123"))
  (let ((i1 (make-iterator str))
	(i2 (make-iterator str)))
    (test (equal? i1 i2) #t)
    (test (equivalent? i1 i2) #t)
    (iterate i1)
    (test (equal? i1 i2) #f)
    (test (equivalent? i1 i2) #f)
    (iterate i2)
    (test (equal? i1 i2) #t)
    (test (equivalent? i1 i2) #t)))

(let ((i1 (make-iterator "123"))
      (i2 (make-iterator "123")))
  (test (equivalent? i1 i2) #t)
  (iterate i1)
  (test (equivalent? i1 i2) #f)
  (iterate i2)
  (test (equivalent? i1 i2) #t))

(let ((i1 (make-iterator (vector 1 2 3)))
      (i2 (make-iterator (int-vector 1 2 3))))
  (test (equivalent? i1 i2) #t)
  (iterate i1)
  (test (equivalent? i1 i2) #f)
  (iterate i2)
  (test (equivalent? i1 i2) #t))

(let ((i1 (make-iterator (vector 1 2 3)))
      (i2 (make-iterator (vector 1 2 3))))
  (test (equal? i1 i2) #t)
  (test (equivalent? i1 i2) #t)
  (iterate i1)
  (test (equal? i1 i2) #f)
  (test (equivalent? i1 i2) #f)
  (iterate i2)
  (test (equal? i1 i2) #t)
  (test (equivalent? i1 i2) #t))

(let ((str (hash-table 'a 1 'b 2)))
  (let ((i1 (make-iterator str))
	(i2 (make-iterator str)))
    (test (equal? i1 i2) #t)
    (test (equivalent? i1 i2) #t)
    (iterate i1)
    (test (equal? i1 i2) #f)
    (test (equivalent? i1 i2) #f)
    (iterate i2)
    (test (equal? i1 i2) #t)
    (test (equivalent? i1 i2) #t)))

(let ((i1 (make-iterator (list 1 2 3)))
      (i2 (make-iterator (list 1 2 3))))
  (test (equivalent? i1 i2) #t)
  (iterate i1)
  (test (equivalent? i1 i2) #f)
  (iterate i2)
  (test (equivalent? i1 i2) #t))

;;; opt bug
(test (equivalent? ''(1) (list 1)) #f)
(test (equivalent? ''(1+i) '(1+i)) #f)
(test (equivalent? '(1) (list 1)) #t)
(test (equivalent? '(1) ''(1)) #f)
(test (equivalent? (list 1) ''(1)) #f)
(test (equivalent? (list 1) '(1)) #t)
(test (equivalent? ''(1) ''(1)) #t)
(test (equivalent? '''(1) ''(1)) #f)

(let ()
  (define-macro (mac a) `(+ 1 ,a))
  (define-macro (mac1 a) `(+ 1 ,a))
  (define-macro (mac2 a) `(+ 2 ,a))
  (define-macro (mac3 a b) `(+ ,b ,a))
  (test (equivalent? mac mac1) #t)
  (test (equivalent? mac mac2) #f)
  (test (equivalent? mac1 mac3) #f)
  (test (equivalent? mac3 mac3) #t)
  (let ()
    (define-macro (mac4 a) `(+ 1 ,a))
    (test (equivalent? mac mac4) #t)) ; was #f
  (define-bacro (mac5 a) `(+ 1 ,a))
  (test (equivalent? mac mac5) #f)
  (define-bacro (mac6 a) `(+ 1 ,a))
  (test (equivalent? mac5 mac6) #t))

(test (equivalent? most-positive-fixnum most-positive-fixnum) #t)
(test (equivalent? most-positive-fixnum most-negative-fixnum) #f)
(test (equivalent? pi pi) #t)
(test (equivalent? 9223372036854775807 9223372036854775806) #f)
(test (equivalent? 9223372036854775807 -9223372036854775808) #f)
(test (equivalent? -9223372036854775808 -9223372036854775808) #t)
(test (equivalent? 123456789/2 123456789/2) #t)
(test (equivalent? 123456789/2 123456787/2) #f)
(test (equivalent? -123456789/2 -123456789/2) #t)
(test (equivalent? 2/123456789 2/123456789) #t)
(test (equivalent? -2/123456789 -2/123456789) #t)
(test (equivalent? 2147483647/2147483646 2147483647/2147483646) #t)
(test (equivalent? 3/4 12/16) #t)
(test (equivalent? 1/1 1) #t)
(test (equivalent? 312689/99532 833719/265381) #f)
(test (let ((x 3.141)) (equivalent? x x)) #t)
(test (let ((x 1+i)) (equivalent? x x)) #t)
(test (let* ((x 3.141) (y x)) (equivalent? x y)) #t)
(test (let* ((x 1+i) (y x)) (equivalent? x y)) #t)
(test (let* ((x 3/4) (y x)) (equivalent? x y)) #t)
(test (equivalent? .1 1/10) #t)
(test (equivalent? pi '(1 2)) #f)
(if with-bignums
    (test (equivalent? (cosh (bignum "128")) 1.943854202997297546111336844178739036366E55) #t)
    (test (equivalent? (cosh 128) 1.943854202997297e+55) #f))
(test (equivalent? (float-vector (cosh 128)) (float-vector 1.943854202997297e+55)) #f) ; can't decide about this -- see floats_are_equivalent
;;; here (* (*s7* 'equivalent-float-epsilon) (cosh 128)) = 1.943854202997298e+40
;;;                    (- (cosh 128) 1.943854202997297e+55) = 2.722258935367508e+39
(when with-block
  (test (equivalent? (block (cosh 128)) (block 1.943854202997297e+55)) #f)
  (test (equivalent? (block) (inlet 'a 1)) #f))

(test (let ((x 3.141)) (equivalent? x x)) #t)
(test (equivalent? 3 3) #t)
(test (equivalent? 3 3.0) #t)
(test (equivalent? 3.0 3.0) #t)
(test (equivalent? 3-4i 3-4i) #t)
(test (equivalent? 1/0 0/0) #t)
(test (equivalent? 1/0 (- 1/0)) #t) ; but they print as +nan.0 (this is C based I think), and equal? here is #f
(test (equivalent? (real-part (log 0)) (- (real-part (log 0)))) #f)
(test (equivalent? (log 0) (log 0)) #t)
(test (equivalent? 0/0+i 0/0+i) #t)
(test (equivalent? 0/0+i 0/0-i) #f)

(test (equivalent? (list 3) (list 3.0)) #t)
(test (equivalent? (list 3.0) (list 3.0)) #t)
(test (equivalent? (list 3-4i) (list 3-4i)) #t)
(test (equivalent? (list 1/0) (list 0/0)) #t)
(test (equivalent? (list (log 0)) (list (log 0))) #t)
(test (equivalent? (list 0/0+i) (list 0/0+i)) #t)

(test (equivalent? (vector 3) (vector 3.0)) #t)
(test (equivalent? (vector 3.0) (vector 3.0)) #t)
(test (equivalent? (vector 3-4i) (vector 3-4i)) #t)
(test (equivalent? (vector 1/0) (vector 0/0)) #t)
(test (equivalent? (vector (log 0)) (vector (log 0))) #t)
(test (equivalent? (vector 0/0+i) (vector 0/0+i)) #t)

(test (equivalent? (string #\c) "c") #t)
(test (equivalent? equivalent? equivalent?) #t)
(test (equivalent? (cons 1 (cons 2 3)) '(1 2 . 3)) #t)
(test (equivalent? '() '()) #t)
(test (equivalent? '() (list)) #t)
(test (equivalent? (cdr '   ''0) '((quote 0))) #t)
(test (equivalent? "\n" "\n") #t)
(test (equivalent? #f ((lambda () #f))) #t)
(test (equivalent? (+) 0) #t)
(test (equivalent? (recompose 32 list '(1)) (recompose 32 list (list 1))) #t)
(test (equivalent? (recompose 100 list '(1)) (recompose 100 list (list 1))) #t)
(test (equivalent? (recompose 32 vector 1) (recompose 32 vector 1)) #t)
(test (equivalent? (reinvert 32 list vector 1) (reinvert 32 list vector 1)) #t)
(test (equivalent? (recompose 32 (lambda (a) (cons 1 a)) ()) (recompose 32 (lambda (a) (cons 1 a)) ())) #t)
(test (equivalent? (recompose 32 (lambda (a) (list 1 a)) ()) (recompose 32 (lambda (a) (list 1 a)) ())) #t)

(test (equivalent? "asd""asd") #t) ; is this the norm?
(let ((streq (lambda (a b) (equivalent? a b)))) (test (streq "asd""asd") #t))

(let ((things (vector #t #f #\space () "" 0 1 3/4 1+i 1.5 '(1 .2) #() (vector 1) (list 1) 'f 't #\t)))
  (let ((len (length things)))
    (do ((i 0 (+ i 1)))
	((= i (- len 1)))
      (do ((j (+ i 1) (+ j 1)))
	  ((= j len))
	(if (equivalent? (vector-ref things i) (vector-ref things j))
	    (format #t ";(equivalent? ~A ~A) -> #t?~%" (vector-ref things i) (vector-ref things j)))))))

(test (equivalent?) 'error)
(test (equivalent? #t) 'error)
(test (equivalent? #t #t #t) 'error)
(test (equal #t #t) 'error)

(test (call-with-exit (lambda (return) (return (equivalent? return return)))) #t)
(test (call-with-exit (lambda (return) (call-with-exit (lambda (quit) (return (equivalent? return quit)))))) #f)
(test (call/cc (lambda (return) (return (equivalent? return return)))) #t)
(test (let hiho ((i 0)) (equivalent? hiho hiho)) #t)
(test (let hiho ((i 0)) (let hoho ((i 0)) (equivalent? hiho hoho))) #f)
(test (equivalent? + *) #f)
(test (equivalent? lambda lambda) #t)
(test (equivalent? lambda lambda*) #f)
(test (equivalent? let let) #t)
(test (equivalent? let letrec) #f)
(test (equivalent? define define) #t)
(test (equivalent? + ((lambda (a) a) +)) #t)
(test (let ((x "hi")) (define (hi) x) (equivalent? (hi) (hi))) #t)

(test (equivalent?
       (list "hi" (integer->char 65) 1 'a-symbol (make-vector 3) (list) (cons 1 2) abs quasiquote 3 3/4 1.0+1.0i #\f (if #f #f) #<eof> #<undefined>)
       (list "hi" (integer->char 65) 1 'a-symbol (make-vector 3) (list) (cons 1 2) abs quasiquote 3 3/4 1.0+1.0i #\f (if #f #f) #<eof> #<undefined>))
      #t)
(test (equivalent?
       (vector "hi" (integer->char 65) 1 'a-symbol (make-vector 3) abs quasiquote 3 3/4 1.0+1.0i #\f (if #f #f) #<eof> #<undefined>)
       (vector "hi" (integer->char 65) 1 'a-symbol (make-vector 3) abs quasiquote 3 3/4 1.0+1.0i #\f (if #f #f) #<eof> #<undefined>))
      #t)
(test (equivalent? (make-string 3 #\null) (make-string 3 #\null)) #t)
(test (equivalent? (make-list 3) (make-list 3)) #t)
(test (equivalent? (make-vector 3) (make-vector 3)) #t)
(test (equivalent? (make-float-vector 3 1.0) (vector 1 1 1)) #t)
(test (equivalent? (int-vector 1) (int-vector 2)) #f)
(test (equivalent? (int-vector 1) (int-vector 1)) #t)
(test (equivalent? (float-vector 0.0) (float-vector +nan.0)) #f)
(test (equivalent? (float-vector +nan.0) (float-vector +nan.0)) #t)
(let-temporarily (((*s7* 'equivalent-float-epsilon) 0.0))
  (test (equivalent? (float-vector 0.0) (float-vector +nan.0)) #f)
  (test (equivalent? (float-vector +nan.0) (float-vector +nan.0)) #t)
  (test (equivalent? (float-vector 0.0) (float-vector 0.0)) #t)
  (test (equivalent? (float-vector 0.0) (float-vector 1e-15)) #f)
  (set! (*s7* 'equivalent-float-epsilon) 0.01)
  (test (equivalent? (float-vector 0.0) (float-vector 1e-15)) #t)
  (test (equivalent? (float-vector 0.0) (float-vector 0.005)) #t)
  (test (equivalent? (float-vector 0.0) (float-vector 0.02)) #f))
(test (equivalent? (float-vector 1) (float-vector 1)) #t)
(test (equivalent? (float-vector 1) (float-vector 0)) #f)
(test (equivalent? (float-vector 1 2 3 4 5 6) (float-vector 1 2 3 4 5 6)) #t)
(test (equivalent? (float-vector 1 2 3 4 5 6) (float-vector 1 2 0 4 5 6)) #f)
(let ()
  (define (fvequiv)
    (do ((i 1 (+ i 1)))
	((= i 20))
      (let ((v1 (make-float-vector i))
	    (v2 (make-float-vector i))
	    (v3 (make-float-vector i)))
	(do ((j 0 (+ j 1)))
	    ((= j i)
	     (set! (v3 (random i)) 32)
	     (unless (equivalent? v1 v2) (format *stderr* "~S != ~S~%" v1 v2))
	     (when (equivalent? v1 v3) (format *stderr* "~S == ~S~%" v1 v3)))
	  (set! (v1 j) j)
	  (set! (v2 j) j)
	  (set! (v3 j) j)))))
  (fvequiv))

(unless with-bignums (test (equivalent? (random-state 100) (random-state 100)) #t))

(test (equivalent? (current-input-port) (current-input-port)) #t)
(test (equivalent? (current-input-port) (current-output-port)) #f)
(test (equivalent? *stdin* *stderr*) #f)

(test (equivalent?
       (let ()
	 (define-macro* (a_func (an_arg (lambda () #t)))
	   `,an_arg)
	 (a_func))
       (let ()
	 (define-macro (a_func an_arg)
	   `,an_arg)
	 (a_func (lambda () #t))))
      #t) ; was #f

(test (equivalent? (- 4/3 1 -63.0) 190/3) #t)
(test (equivalent? 190/3 (- 4/3 1 -63.0)) #t)
(test (equivalent? (+ -1 4/3 63.0) 190/3) #t)

(unless with-bignums
  (set! (*s7* 'equivalent-float-epsilon) 1e-15) ; just in case
  (test (equivalent? (+ 5e-16 +nan.0) +nan.0) #t)
  (test (equivalent? (+ 0+5e-16i +nan.0) +nan.0) #t)
  (test (equivalent? (+ 1/0 0+5e-16i) 1/0) #t)
  (test (equivalent? 1/0 (+ 1/0 0+5e-16i)) #t)
  (test (equivalent? 0 (+ 0 5e-16)) #t)
  (test (equivalent? 0 (- 0 1/1428571428571429)) #t)
  (test (equivalent? 0 (+ 0 0+5e-16i)) #t)
  (test (equivalent? 0 (+ 0 0-1/1428571428571429i)) #t)
  (test (equivalent? 0 (+ 0 1e-11)) #f)
  (test (equivalent? 0 0) #t)
  (test (equivalent? 0 1/1000) #f)
  (test (equivalent? 0 0.0) #t)
  (test (equivalent? 0 1e-16) #t)
  (test (equivalent? 0 0+i) #f)
  (test (equivalent? 0 1e-16+i) #f)
  (test (equivalent? 0 0+1e-16i) #t)
  (test (equivalent? 0 1e-300) #t)
  (test (equivalent? 0 0+1e-300i) #t)
  (test (equivalent? 0 1/0) #f)
  (test (equivalent? 0 (- 0/0)) #f)
  (test (equivalent? 0 (log 0)) #f)
  (test (equivalent? 1 (+ 1 5e-16)) #t)
  (test (equivalent? 1 (- 1 1/1428571428571429)) #t)
  (test (equivalent? 1 (+ 1 0+5e-16i)) #t)
  (test (equivalent? 1 (+ 1 0-1/1428571428571429i)) #t)
  (test (equivalent? 1 (+ 1 1e-11)) #f)
  (test (equivalent? 1 1) #t)
  (test (equivalent? 1 1.0) #t)
  (test (equivalent? 1 1e-16) #f)
  (test (equivalent? 1 1e4) #f)
  (test (equivalent? 1 0+i) #f)
  (test (equivalent? 1 1e-16+i) #f)
  (test (equivalent? 1 (complex 1 1/0)) #f)
  (test (equivalent? 1 (complex (real-part (log 0)) 1)) #f)
  (test (equivalent? 1 (complex 1 (real-part (log 0)))) #f)
  (test (equivalent? 1000 (+ 1000 5e-16)) #t)
  (test (equivalent? 1000 (- 1000 1/1428571428571429)) #t)
  (test (equivalent? 1000 (+ 1000 0+5e-16i)) #t)
  (test (equivalent? 1000 (+ 1000 0-1/1428571428571429i)) #t)
  (test (equivalent? 1000 (+ 1000 1e-11)) #f)
  (test (equivalent? 1000 (+ 1000 1e-14)) #t)
  (test (equivalent? 1000 1000) #t)
  (test (equivalent? 1000 1/1000) #f)
  (test (equivalent? 1000 1e4) #f)
  (test (equivalent? 1/1000 (+ 1/1000 5e-16)) #t)
  (test (equivalent? 1/1000 (- 1/1000 1/1428571428571429)) #t)
  (test (equivalent? 1/1000 (+ 1/1000 0+5e-16i)) #t)
  (test (equivalent? 1/1000 (+ 1/1000 0-1/1428571428571429i)) #t)
  (test (equivalent? 1/1000 (+ 1/1000 1e-11)) #f)
  (test (equivalent? 1/1000 0) #f)
  (test (equivalent? 1/1000 1/1000) #t)
  (test (equivalent? 1/1000 0.0) #f)
  (test (equivalent? 1/1000 1e-16) #f)
  (test (equivalent? 1/1000 1e-16+i) #f)
  (test (equivalent? 1/1000 0+1e-16i) #f)
  (test (equivalent? 1/1000 1e-300) #f)
  (test (equivalent? 1/1000 0+1e-300i) #f)
  (test (equivalent? 1/1000 1/0) #f)
  (test (equivalent? 0.0 (+ 0.0 5e-16)) #t)
  (test (equivalent? 0.0 (- 0.0 1/1428571428571429)) #t)
  (test (equivalent? 0.0 (+ 0.0 0+5e-16i)) #t)
  (test (equivalent? 0.0 (+ 0.0 0-1/1428571428571429i)) #t)
  (test (equivalent? 0.0 (+ 0.0 1e-11)) #f)
  (test (equivalent? 0.0 0) #t)
  (test (equivalent? 0.0 1/1000) #f)
  (test (equivalent? 0.0 0.0) #t)
  (test (equivalent? 0.0 1e-16) #t)
  (test (equivalent? 0.0 0+i) #f)
  (test (equivalent? 0.0 1+i) #f)
  (test (equivalent? 0.0 1e-16+i) #f)
  (test (equivalent? 0.0 0+1e-16i) #t)
  (test (equivalent? 0.0 1e-300) #t)
  (test (equivalent? 0.0 0+1e-300i) #t)
  (test (equivalent? 0.0 1/0) #f)
  (test (equivalent? 0.0 (real-part (log 0))) #f)
  (test (equivalent? 0.0 (- (real-part (log 0)))) #f)
  (test (equivalent? 0.0 (- 0/0)) #f)
  (test (equivalent? 0.0 (log 0)) #f)
  (test (equivalent? 1.0 (+ 1.0 5e-16)) #t)
  (test (equivalent? 1.0 (- 1.0 1/1428571428571429)) #t)
  (test (equivalent? 1.0 (+ 1.0 0+5e-16i)) #t)
  (test (equivalent? 1.0 (+ 1.0 0-1/1428571428571429i)) #t)
  (test (equivalent? 1.0 (+ 1.0 1e-11)) #f)
  (test (equivalent? 1.0 1) #t)
  (test (equivalent? 1.0 1.0) #t)
  (test (equivalent? 1.0 1e-16+i) #f)
  (test (equivalent? 1.0 0+1e-16i) #f)
  (test (equivalent? 1.0 1e-300) #f)
  (test (equivalent? 1.0 0+1e-300i) #f)
  (test (equivalent? 1.0 1/0) #f)
  (test (equivalent? 1.0 (- 0/0)) #f)
  (test (equivalent? 1.0 (complex 1/0 1)) #f)
  (test (equivalent? 1.0 (complex 1 1/0)) #f)
  (test (equivalent? 1.0 (complex 1 (real-part (log 0)))) #f)
  (test (equivalent? 1e-16 (+ 1e-16 5e-16)) #t)
  (test (equivalent? 1e-16 (- 1e-16 1/1428571428571429)) #t)
  (test (equivalent? 1e-16 (+ 1e-16 0+5e-16i)) #t)
  (test (equivalent? 1e-16 (+ 1e-16 0-1/1428571428571429i)) #t)
  (test (equivalent? 1e-16 (+ 1e-16 1e-11)) #f)
  (test (equivalent? 1e-16 0) #t)
  (test (equivalent? 1e-16 1/1000) #f)
  (test (equivalent? 1e-16 0.0) #t)
  (test (equivalent? 1e-16 1e-16) #t)
  (test (equivalent? 1e-16 1e-16+i) #f)
  (test (equivalent? 1e-16 0+1e-16i) #t)
  (test (equivalent? 1e-16 1e-300) #t)
  (test (equivalent? 1e-16 0+1e-300i) #t)
  (test (equivalent? 1e-16 1/0) #f)
  (test (equivalent? 1e4 (+ 1e4 5e-16)) #t)
  (test (equivalent? 1e4 (- 1e4 1/1428571428571429)) #t)
  (test (equivalent? 1e4 (+ 1e4 0+5e-16i)) #t)
  (test (equivalent? 1e4 (+ 1e4 0-1/1428571428571429i)) #t)
  (test (equivalent? 1e4 (+ 1e4 1e-11)) #f)
  (test (equivalent? 1e4 1000) #f)
  (test (equivalent? 1e4 1/1000) #f)
  (test (equivalent? 1e4 1e-16) #f)
  (test (equivalent? 1e4 1e4) #t)
  (test (equivalent? 1e4 1e-16+i) #f)
  (test (equivalent? 1e4 0+1e-16i) #f)
  (test (equivalent? 1e4 1e-300) #f)
  (test (equivalent? 1e4 0+1e-300i) #f)
  (test (equivalent? 1e4 1/0) #f)
  (test (equivalent? 0+i (+ 0+i 5e-16)) #t)
  (test (equivalent? 0+i (- 0+i 1/1428571428571429)) #t)
  (test (equivalent? 0+i (+ 0+i 0+5e-16i)) #t)
  (test (equivalent? 0+i (+ 0+i 0-1/1428571428571429i)) #t)
  (test (equivalent? 0+i (+ 0+i 1e-11)) #f)
  (test (equivalent? 0+i 0) #f)
  (test (equivalent? 0+i 1/1000) #f)
  (test (equivalent? 0+i 0.0) #f)
  (test (equivalent? 0+i 1e-16) #f)
  (test (equivalent? 0+i 0+i) #t)
  (test (equivalent? 0+i 1+i) #f)
  (test (equivalent? 0+i 1e-16+i) #t)
  (test (equivalent? 0+i 0+1e-16i) #f)
  (test (equivalent? 0+i 1e-300) #f)
  (test (equivalent? 0+i 0+1e-300i) #f)
  (test (equivalent? 0+i 1/0) #f)
  (test (equivalent? 0+i (real-part (log 0))) #f)
  (test (equivalent? 0+i (- (real-part (log 0)))) #f)
  (test (equivalent? 0+i (- 0/0)) #f)
  (test (equivalent? 0+i (log 0)) #f)
  (test (equivalent? 0+i (complex 1/0 1)) #f)
  (test (equivalent? 0+i (complex 1 1/0)) #f)
  (test (equivalent? 0+i (complex 1/0 1/0)) #f)
  (test (equivalent? 0+i (complex (real-part (log 0)) 1/0)) #f)
  (test (equivalent? 1+i (+ 1+i 5e-16)) #t)
  (test (equivalent? 1+i (- 1+i 1/1428571428571429)) #t)
  (test (equivalent? 1+i (+ 1+i 0+5e-16i)) #t)
  (test (equivalent? 1+i (+ 1+i 0-1/1428571428571429i)) #t)
  (test (equivalent? 1+i (+ 1+i 1e-11)) #f)
  (test (equivalent? 1+i 0+i) #f)
  (test (equivalent? 1+i 1+i) #t)
  (test (equivalent? 1+i 1e-16+i) #f)
  (test (equivalent? 1+i 0+1e-16i) #f)
  (test (equivalent? 1+i 1e-300) #f)
  (test (equivalent? 1+i 0+1e-300i) #f)
  (test (equivalent? 1e-16+i (+ 1e-16+i 5e-16)) #t)
  (test (equivalent? 1e-16+i (- 1e-16+i 1/1428571428571429)) #t)
  (test (equivalent? 1e-16+i (+ 1e-16+i 0+5e-16i)) #t)
  (test (equivalent? 1e-16+i (+ 1e-16+i 0-1/1428571428571429i)) #t)
  (test (equivalent? 1e-16+i (+ 1e-16+i 1e-11)) #f)
  (test (equivalent? 1e-16+i 0) #f)
  (test (equivalent? 1e-16+i 1e-16) #f)
  (test (equivalent? 1e-16+i 1e4) #f)
  (test (equivalent? 1e-16+i 0+i) #t)
  (test (equivalent? 1e-16+i 1+i) #f)
  (test (equivalent? 1e-16+i 1e-16+i) #t)
  (test (equivalent? 1e-16+i 0+1e-16i) #f)
  (test (equivalent? 1e-16+i 1e-300) #f)
  (test (equivalent? 1e-16+i 0+1e-300i) #f)
  (test (equivalent? 1e-16+i 1/0) #f)
  (test (equivalent? 1e-16+i (real-part (log 0))) #f)
  (test (equivalent? 1e-16+i (- (real-part (log 0)))) #f)
  (test (equivalent? 1e-16+i (- 0/0)) #f)
  (test (equivalent? 1e-16+i (log 0)) #f)
  (test (equivalent? 1e-16+i (complex 1/0 1)) #f)
  (test (equivalent? 1e-16+i (complex 1 1/0)) #f)
  (test (equivalent? 1e-16+i (complex 1/0 1/0)) #f)
  (test (equivalent? 1e-16+i (complex (real-part (log 0)) 1/0)) #f)
  (test (equivalent? 0+1e-16i (+ 0+1e-16i 5e-16)) #t)
  (test (equivalent? 0+1e-16i (- 0+1e-16i 1/1428571428571429)) #t)
  (test (equivalent? 0+1e-16i (+ 0+1e-16i 0+5e-16i)) #t)
  (test (equivalent? 0+1e-16i (+ 0+1e-16i 0-1/1428571428571429i)) #t)
  (test (equivalent? 0+1e-16i (+ 0+1e-16i 1e-11)) #f)
  (test (equivalent? 0+1e-16i 0) #t)
  (test (equivalent? 0+1e-16i 1/1000) #f)
  (test (equivalent? 0+1e-16i 0.0) #t)
  (test (equivalent? 0+1e-16i 1e-16) #t)
  (test (equivalent? 0+1e-16i 0+i) #f)
  (test (equivalent? 0+1e-16i 1+i) #f)
  (test (equivalent? 0+1e-16i 1e-16+i) #f)
  (test (equivalent? 0+1e-16i 0+1e-16i) #t)
  (test (equivalent? 0+1e-16i 1e-300) #t)
  (test (equivalent? 0+1e-16i 0+1e-300i) #t)
  (test (equivalent? 0+1e-16i 1/0) #f)
  (test (equivalent? 0+1e-16i (real-part (log 0))) #f)
  (test (equivalent? 0+1e-16i (- (real-part (log 0)))) #f)
  (test (equivalent? 0+1e-16i (- 0/0)) #f)
  (test (equivalent? 0+1e-16i (log 0)) #f)
  (test (equivalent? 1e-300 (+ 1e-300 5e-16)) #t)
  (test (equivalent? 1e-300 (- 1e-300 1/1428571428571429)) #t)
  (test (equivalent? 1e-300 (+ 1e-300 0+5e-16i)) #t)
  (test (equivalent? 1e-300 (+ 1e-300 0-1/1428571428571429i)) #t)
  (test (equivalent? 1e-300 (+ 1e-300 1e-11)) #f)
  (test (equivalent? 1e-300 0) #t)
  (test (equivalent? 1e-300 1/1000) #f)
  (test (equivalent? 1e-300 0.0) #t)
  (test (equivalent? 1e-300 1e-16) #t)
  (test (equivalent? 1e-300 1e-16+i) #f)
  (test (equivalent? 1e-300 0+1e-16i) #t)
  (test (equivalent? 1e-300 1e-300) #t)
  (test (equivalent? 1e-300 0+1e-300i) #t)
  (test (equivalent? 1e-300 1/0) #f)
  (test (equivalent? 1e-300 (- 0/0)) #f)
  (test (equivalent? 1e-300 (log 0)) #f)
  (test (equivalent? 0+1e-300i (+ 0+1e-300i 5e-16)) #t)
  (test (equivalent? 0+1e-300i (- 0+1e-300i 1/1428571428571429)) #t)
  (test (equivalent? 0+1e-300i (+ 0+1e-300i 0+5e-16i)) #t)
  (test (equivalent? 0+1e-300i (+ 0+1e-300i 0-1/1428571428571429i)) #t)
  (test (equivalent? 0+1e-300i (+ 0+1e-300i 1e-11)) #f)
  (test (equivalent? 0+1e-300i 0) #t)
  (test (equivalent? 0+1e-300i 1000) #f)
  (test (equivalent? 0+1e-300i 1/1000) #f)
  (test (equivalent? 0+1e-300i 0.0) #t)
  (test (equivalent? 0+1e-300i 1e-16) #t)
  (test (equivalent? 0+1e-300i 0+i) #f)
  (test (equivalent? 0+1e-300i 1e-16+i) #f)
  (test (equivalent? 0+1e-300i 0+1e-16i) #t)
  (test (equivalent? 0+1e-300i 1e-300) #t)
  (test (equivalent? 0+1e-300i 0+1e-300i) #t)
  (test (equivalent? 0+1e-300i 1/0) #f)
  (test (equivalent? 0+1e-300i (- 0/0)) #f)
  (test (equivalent? 1/0 (+ 1/0 5e-16)) #t)
  (test (equivalent? 1/0 (- 1/0 1/1428571428571429)) #t)
  (test (equivalent? 1/0 (+ 1/0 0+5e-16i)) #t)
  (test (equivalent? 1/0 (+ 1/0 0-1/1428571428571429i)) #t)
  (test (equivalent? 1/0 0) #f)
  (test (equivalent? 1/0 1/0) #t)
  (test (equivalent? 1/0 (real-part (log 0))) #f)
  (test (equivalent? 1/0 (- (real-part (log 0)))) #f)
  (test (equivalent? 1/0 (- 0/0)) #t)
  (test (equivalent? 1/0 (log 0)) #f)
  (test (equivalent? 1/0 (complex 1/0 1)) #f)
  (test (equivalent? 1/0 (complex 1 1/0)) #f)
  (test (equivalent? 1/0 (complex 1/0 1/0)) #f)
  (test (equivalent? 1/0 (complex (real-part (log 0)) 1/0)) #f)
  (test (equivalent? 1/0 (complex 1/0 (real-part (log 0)))) #f)
  (test (equivalent? 1/0 (complex (real-part (log 0)) (real-part (log 0)))) #f)
  (test (equivalent? 1/0 (complex (real-part (log 0)) 1)) #f)
  (test (equivalent? 1/0 (complex 1 (real-part (log 0)))) #f)
  (test (equivalent? (real-part (log 0)) (+ (real-part (log 0)) 5e-16)) #t)
  (test (equivalent? (real-part (log 0)) (- (real-part (log 0)) 1/1428571428571429)) #t)
  (test (equivalent? (real-part (log 0)) (+ (real-part (log 0)) 0+5e-16i)) #t)
  (test (equivalent? (real-part (log 0)) (+ (real-part (log 0)) 0-1/1428571428571429i)) #t)
  (test (equivalent? (real-part (log 0)) 0) #f)
  (test (equivalent? (real-part (log 0)) 1e-16+i) #f)
  (test (equivalent? (real-part (log 0)) 0+1e-16i) #f)
  (test (equivalent? (real-part (log 0)) 1e-300) #f)
  (test (equivalent? (real-part (log 0)) 0+1e-300i) #f)
  (test (equivalent? (real-part (log 0)) 1/0) #f)
  (test (equivalent? (real-part (log 0)) (real-part (log 0))) #t)
  (test (equivalent? (real-part (log 0)) (- (real-part (log 0)))) #f)
  (test (equivalent? (real-part (log 0)) (- 0/0)) #f)
  (test (equivalent? (real-part (log 0)) (log 0)) #f)
  (test (equivalent? (real-part (log 0)) (complex 1/0 1)) #f)
  (test (equivalent? (real-part (log 0)) (complex 1 1/0)) #f)
  (test (equivalent? (real-part (log 0)) (complex 1/0 1/0)) #f)
  (test (equivalent? (real-part (log 0)) (complex (real-part (log 0)) 1/0)) #f)
  (test (equivalent? (real-part (log 0)) (complex 1/0 (real-part (log 0)))) #f)
  (test (equivalent? (real-part (log 0)) (complex (real-part (log 0)) (real-part (log 0)))) #f)
  (test (equivalent? (real-part (log 0)) (complex (real-part (log 0)) 1)) #f)
  (test (equivalent? (real-part (log 0)) (complex 1 (real-part (log 0)))) #f)
  (test (equivalent? (- (real-part (log 0))) (+ (- (real-part (log 0))) 5e-16)) #t)
  (test (equivalent? (- (real-part (log 0))) (- (- (real-part (log 0))) 1/1428571428571429)) #t)
  (test (equivalent? (- (real-part (log 0))) (+ (- (real-part (log 0))) 0+5e-16i)) #t)
  (test (equivalent? (- (real-part (log 0))) (+ (- (real-part (log 0))) 0-1/1428571428571429i)) #t)
  (test (equivalent? (- (real-part (log 0))) 1e-16+i) #f)
  (test (equivalent? (- (real-part (log 0))) 0+1e-16i) #f)
  (test (equivalent? (- (real-part (log 0))) 1e-300) #f)
  (test (equivalent? (- (real-part (log 0))) 0+1e-300i) #f)
  (test (equivalent? (- (real-part (log 0))) 1/0) #f)
  (test (equivalent? (- (real-part (log 0))) (real-part (log 0))) #f)
  (test (equivalent? (- (real-part (log 0))) (- (real-part (log 0)))) #t)
  (test (equivalent? (- (real-part (log 0))) (- 0/0)) #f)
  (test (equivalent? (- (real-part (log 0))) (log 0)) #f)
  (test (equivalent? (- (real-part (log 0))) (complex 1/0 1)) #f)
  (test (equivalent? (- (real-part (log 0))) (complex 1 1/0)) #f)
  (test (equivalent? (- (real-part (log 0))) (complex 1/0 1/0)) #f)
  (test (equivalent? (- (real-part (log 0))) (complex (real-part (log 0)) 1/0)) #f)
  (test (equivalent? (- (real-part (log 0))) (complex 1/0 (real-part (log 0)))) #f)
  (test (equivalent? (- (real-part (log 0))) (complex (real-part (log 0)) (real-part (log 0)))) #f)
  (test (equivalent? (- (real-part (log 0))) (complex (real-part (log 0)) 1)) #f)
  (test (equivalent? (- (real-part (log 0))) (complex 1 (real-part (log 0)))) #f)
  (test (equivalent? (- 0/0) (+ (- 0/0) 5e-16)) #t)
  (test (equivalent? (- 0/0) (- (- 0/0) 1/1428571428571429)) #t)
  (test (equivalent? (- 0/0) (+ (- 0/0) 0+5e-16i)) #t)
  (test (equivalent? (- 0/0) (+ (- 0/0) 0-1/1428571428571429i)) #t)
  (test (equivalent? (- 0/0) 0) #f)
  (test (equivalent? (- 0/0) 1e-300) #f)
  (test (equivalent? (- 0/0) 0+1e-300i) #f)
  (test (equivalent? (- 0/0) 1/0) #t)
  (test (equivalent? (- 0/0) (real-part (log 0))) #f)
  (test (equivalent? (- 0/0) (- (real-part (log 0)))) #f)
  (test (equivalent? (- 0/0) (- 0/0)) #t)
  (test (equivalent? (- 0/0) (log 0)) #f)
  (test (equivalent? (- 0/0) (complex 1/0 1)) #f)
  (test (equivalent? (- 0/0) (complex 1 1/0)) #f)
  (test (equivalent? (- 0/0) (complex 1/0 1/0)) #f)
  (test (equivalent? (- 0/0) (complex (real-part (log 0)) 1/0)) #f)
  (test (equivalent? (- 0/0) (complex 1/0 (real-part (log 0)))) #f)
  (test (equivalent? (- 0/0) (complex (real-part (log 0)) (real-part (log 0)))) #f)
  (test (equivalent? (- 0/0) (complex (real-part (log 0)) 1)) #f)
  (test (equivalent? (- 0/0) (complex 1 (real-part (log 0)))) #f)
  (test (equivalent? (log 0) (+ (log 0) 5e-16)) #t)
  (test (equivalent? (log 0) (- (log 0) 1/1428571428571429)) #t)
  (test (equivalent? (log 0) (+ (log 0) 0+5e-16i)) #t)
  (test (equivalent? (log 0) (+ (log 0) 0-1/1428571428571429i)) #t)
  (test (equivalent? (log 0) 0) #f)
  (test (equivalent? (log 0) 1/0) #f)
  (test (equivalent? (log 0) (real-part (log 0))) #f)
  (test (equivalent? (log 0) (- (real-part (log 0)))) #f)
  (test (equivalent? (log 0) (- 0/0)) #f)
  (test (equivalent? (log 0) (log 0)) #t)
  (test (equivalent? (log 0) (complex 1/0 1)) #f)
  (test (equivalent? (log 0) (complex 1 1/0)) #f)
  (test (equivalent? (log 0) (complex 1/0 1/0)) #f)
  (test (equivalent? (log 0) (complex (real-part (log 0)) 1/0)) #f)
  (test (equivalent? (log 0) (complex 1/0 (real-part (log 0)))) #f)
  (test (equivalent? (log 0) (complex (real-part (log 0)) (real-part (log 0)))) #f)
  (test (equivalent? (log 0) (complex (real-part (log 0)) 1)) #f)
  (test (equivalent? (log 0) (complex 1 (real-part (log 0)))) #f)
  (test (equivalent? (complex 1/0 1) (+ (complex 1/0 1) 5e-16)) #t)
  (test (equivalent? (complex 1/0 1) (- (complex 1/0 1) 1/1428571428571429)) #t)
  (test (equivalent? (complex 1/0 1) (+ (complex 1/0 1) 0+5e-16i)) #t)
  (test (equivalent? (complex 1/0 1) (+ (complex 1/0 1) 0-1/1428571428571429i)) #t)
  (test (equivalent? (complex 1/0 1) 0) #f)
  (test (equivalent? (complex 1/0 1) 1) #f)
  (test (equivalent? (complex 1/0 1) 1e-16+i) #f)
  (test (equivalent? (complex 1/0 1) 0+1e-16i) #f)
  (test (equivalent? (complex 1/0 1) 1e-300) #f)
  (test (equivalent? (complex 1/0 1) 0+1e-300i) #f)
  (test (equivalent? (complex 1/0 1) 1/0) #f)
  (test (equivalent? (complex 1/0 1) (real-part (log 0))) #f)
  (test (equivalent? (complex 1/0 1) (- (real-part (log 0)))) #f)
  (test (equivalent? (complex 1/0 1) (- 0/0)) #f)
  (test (equivalent? (complex 1/0 1) (log 0)) #f)
  (test (equivalent? (complex 1/0 1) (complex 1/0 1)) #t)
  (test (equivalent? (complex 1/0 1) (complex 1 1/0)) #f)
  (test (equivalent? (complex 1/0 1) (complex 1/0 1/0)) #f)
  (test (equivalent? (complex 1/0 1) (complex (real-part (log 0)) 1/0)) #f)
  (test (equivalent? (complex 1/0 1) (complex 1/0 (real-part (log 0)))) #f)
  (test (equivalent? (complex 1/0 1) (complex (real-part (log 0)) (real-part (log 0)))) #f)
  (test (equivalent? (complex 1/0 1) (complex (real-part (log 0)) 1)) #f)
  (test (equivalent? (complex 1/0 1) (complex 1 (real-part (log 0)))) #f)
  (test (equivalent? (complex 1 1/0) (+ (complex 1 1/0) 5e-16)) #t)
  (test (equivalent? (complex 1 1/0) (- (complex 1 1/0) 1/1428571428571429)) #t)
  (test (equivalent? (complex 1 1/0) (+ (complex 1 1/0) 0+5e-16i)) #t)
  (test (equivalent? (complex 1 1/0) (+ (complex 1 1/0) 0-1/1428571428571429i)) #t)
  (test (equivalent? (complex 1 1/0) 0) #f)
  (test (equivalent? (complex 1 1/0) 1) #f)
  (test (equivalent? (complex 1 1/0) 1e-300) #f)
  (test (equivalent? (complex 1 1/0) 0+1e-300i) #f)
  (test (equivalent? (complex 1 1/0) 1/0) #f)
  (test (equivalent? (complex 1 1/0) (real-part (log 0))) #f)
  (test (equivalent? (complex 1 1/0) (- (real-part (log 0)))) #f)
  (test (equivalent? (complex 1 1/0) (- 0/0)) #f)
  (test (equivalent? (complex 1 1/0) (log 0)) #f)
  (test (equivalent? (complex 1 1/0) (complex 1/0 1)) #f)
  (test (equivalent? (complex 1 1/0) (complex 1 1/0)) #t)
  (test (equivalent? (complex 1 1/0) (complex 1/0 1/0)) #f)
  (test (equivalent? (complex 1 1/0) (complex (real-part (log 0)) 1/0)) #f)
  (test (equivalent? (complex 1 1/0) (complex 1/0 (real-part (log 0)))) #f)
  (test (equivalent? (complex 1 1/0) (complex (real-part (log 0)) (real-part (log 0)))) #f)
  (test (equivalent? (complex 1 1/0) (complex (real-part (log 0)) 1)) #f)
  (test (equivalent? (complex 1 1/0) (complex 1 (real-part (log 0)))) #f)
  (test (equivalent? (complex 1/0 1/0) (+ (complex 1/0 1/0) 5e-16)) #t)
  (test (equivalent? (complex 1/0 1/0) (- (complex 1/0 1/0) 1/1428571428571429)) #t)
  (test (equivalent? (complex 1/0 1/0) (+ (complex 1/0 1/0) 0+5e-16i)) #t)
  (test (equivalent? (complex 1/0 1/0) (+ (complex 1/0 1/0) 0-1/1428571428571429i)) #t)
  (test (equivalent? (complex 1/0 1/0) 0) #f)
  (test (equivalent? (complex 1/0 1/0) 1/0) #f)
  (test (equivalent? (complex 1/0 1/0) (real-part (log 0))) #f)
  (test (equivalent? (complex 1/0 1/0) (- (real-part (log 0)))) #f)
  (test (equivalent? (complex 1/0 1/0) (- 0/0)) #f)
  (test (equivalent? (complex 1/0 1/0) (log 0)) #f)
  (test (equivalent? (complex 1/0 1/0) (complex 1/0 1)) #f)
  (test (equivalent? (complex 1/0 1/0) (complex 1 1/0)) #f)
  (test (equivalent? (complex 1/0 1/0) (complex 1/0 1/0)) #t)
  (test (equivalent? (complex 1/0 1/0) (complex (real-part (log 0)) 1/0)) #f)
  (test (equivalent? (complex 1/0 1/0) (complex 1/0 (real-part (log 0)))) #f)
  (test (equivalent? (complex 1/0 1/0) (complex (real-part (log 0)) (real-part (log 0)))) #f)
  (test (equivalent? (complex 1/0 1/0) (complex (real-part (log 0)) 1)) #f)
  (test (equivalent? (complex 1/0 1/0) (complex 1 (real-part (log 0)))) #f)
  (test (equivalent? (complex (real-part (log 0)) 1/0) (+ (complex (real-part (log 0)) 1/0) 5e-16)) #t)
  (test (equivalent? (complex (real-part (log 0)) 1/0) (- (complex (real-part (log 0)) 1/0) 1/1428571428571429)) #t)
  (test (equivalent? (complex (real-part (log 0)) 1/0) (+ (complex (real-part (log 0)) 1/0) 0+5e-16i)) #t)
  (test (equivalent? (complex (real-part (log 0)) 1/0) (+ (complex (real-part (log 0)) 1/0) 0-1/1428571428571429i)) #t)
  (test (equivalent? (complex (real-part (log 0)) 1/0) 0) #f)
  (test (equivalent? (complex (real-part (log 0)) 1/0) 1) #f)
  (test (equivalent? (complex (real-part (log 0)) 1/0) 1000) #f)
  (test (equivalent? (complex (real-part (log 0)) 1/0) 1/1000) #f)
  (test (equivalent? (complex (real-part (log 0)) 1/0) 0.0) #f)
  (test (equivalent? (complex (real-part (log 0)) 1/0) 1.0) #f)
  (test (equivalent? (complex (real-part (log 0)) 1/0) 1e-16) #f)
  (test (equivalent? (complex (real-part (log 0)) 1/0) 1e4) #f)
  (test (equivalent? (complex (real-part (log 0)) 1/0) 1/0) #f)
  (test (equivalent? (complex (real-part (log 0)) 1/0) (real-part (log 0))) #f)
  (test (equivalent? (complex (real-part (log 0)) 1/0) (- (real-part (log 0)))) #f)
  (test (equivalent? (complex (real-part (log 0)) 1/0) (- 0/0)) #f)
  (test (equivalent? (complex (real-part (log 0)) 1/0) (log 0)) #f)
  (test (equivalent? (complex (real-part (log 0)) 1/0) (complex 1/0 1)) #f)
  (test (equivalent? (complex (real-part (log 0)) 1/0) (complex 1 1/0)) #f)
  (test (equivalent? (complex (real-part (log 0)) 1/0) (complex 1/0 1/0)) #f)
  (test (equivalent? (complex (real-part (log 0)) 1/0) (complex (real-part (log 0)) 1/0)) #t)
  (test (equivalent? (complex (real-part (log 0)) 1/0) (complex 1/0 (real-part (log 0)))) #f)
  (test (equivalent? (complex (real-part (log 0)) 1/0) (complex (real-part (log 0)) (real-part (log 0)))) #f)
  (test (equivalent? (complex (real-part (log 0)) 1/0) (complex (real-part (log 0)) 1)) #f)
  (test (equivalent? (complex (real-part (log 0)) 1/0) (complex 1 (real-part (log 0)))) #f)
  (test (equivalent? (complex 1/0 (real-part (log 0))) (+ (complex 1/0 (real-part (log 0))) 5e-16)) #t)
  (test (equivalent? (complex 1/0 (real-part (log 0))) (- (complex 1/0 (real-part (log 0))) 1/1428571428571429)) #t)
  (test (equivalent? (complex 1/0 (real-part (log 0))) (+ (complex 1/0 (real-part (log 0))) 0+5e-16i)) #t)
  (test (equivalent? (complex 1/0 (real-part (log 0))) (+ (complex 1/0 (real-part (log 0))) 0-1/1428571428571429i)) #t)
  (test (equivalent? (complex 1/0 (real-part (log 0))) (real-part (log 0))) #f)
  (test (equivalent? (complex 1/0 (real-part (log 0))) (- (real-part (log 0)))) #f)
  (test (equivalent? (complex 1/0 (real-part (log 0))) (- 0/0)) #f)
  (test (equivalent? (complex 1/0 (real-part (log 0))) (log 0)) #f)
  (test (equivalent? (complex 1/0 (real-part (log 0))) (complex 1/0 1)) #f)
  (test (equivalent? (complex 1/0 (real-part (log 0))) (complex 1 1/0)) #f)
  (test (equivalent? (complex 1/0 (real-part (log 0))) (complex 1/0 1/0)) #f)
  (test (equivalent? (complex 1/0 (real-part (log 0))) (complex (real-part (log 0)) 1/0)) #f)
  (test (equivalent? (complex 1/0 (real-part (log 0))) (complex 1/0 (real-part (log 0)))) #t)
  (test (equivalent? (complex 1/0 (real-part (log 0))) (complex (real-part (log 0)) (real-part (log 0)))) #f)
  (test (equivalent? (complex 1/0 (real-part (log 0))) (complex (real-part (log 0)) 1)) #f)
  (test (equivalent? (complex 1/0 (real-part (log 0))) (complex 1 (real-part (log 0)))) #f)
  (test (equivalent? (complex (real-part (log 0)) (real-part (log 0))) (+ (complex (real-part (log 0)) (real-part (log 0))) 5e-16)) #t)
  (test (equivalent? (complex (real-part (log 0)) (real-part (log 0))) (- (complex (real-part (log 0)) (real-part (log 0))) 1/1428571428571429)) #t)
  (test (equivalent? (complex (real-part (log 0)) (real-part (log 0))) (+ (complex (real-part (log 0)) (real-part (log 0))) 0+5e-16i)) #t)
  (test (equivalent? (complex (real-part (log 0)) (real-part (log 0))) (+ (complex (real-part (log 0)) (real-part (log 0))) 0-1/1428571428571429i)) #t)
  (test (equivalent? (complex (real-part (log 0)) (real-part (log 0))) 0) #f)
  (test (equivalent? (complex (real-part (log 0)) (real-part (log 0))) 1/0) #f)
  (test (equivalent? (complex (real-part (log 0)) (real-part (log 0))) (real-part (log 0))) #f)
  (test (equivalent? (complex (real-part (log 0)) (real-part (log 0))) (- (real-part (log 0)))) #f)
  (test (equivalent? (complex (real-part (log 0)) (real-part (log 0))) (- 0/0)) #f)
  (test (equivalent? (complex (real-part (log 0)) (real-part (log 0))) (log 0)) #f)
  (test (equivalent? (complex (real-part (log 0)) (real-part (log 0))) (complex 1/0 1)) #f)
  (test (equivalent? (complex (real-part (log 0)) (real-part (log 0))) (complex 1 1/0)) #f)
  (test (equivalent? (complex (real-part (log 0)) (real-part (log 0))) (complex 1/0 1/0)) #f)
  (test (equivalent? (complex (real-part (log 0)) (real-part (log 0))) (complex (real-part (log 0)) 1/0)) #f)
  (test (equivalent? (complex (real-part (log 0)) (real-part (log 0))) (complex 1/0 (real-part (log 0)))) #f)
  (test (equivalent? (complex (real-part (log 0)) (real-part (log 0))) (complex (real-part (log 0)) (real-part (log 0)))) #t)
  (test (equivalent? (complex (real-part (log 0)) (real-part (log 0))) (complex (real-part (log 0)) 1)) #f)
  (test (equivalent? (complex (real-part (log 0)) (real-part (log 0))) (complex 1 (real-part (log 0)))) #f)
  (test (equivalent? (complex (real-part (log 0)) 1) (+ (complex (real-part (log 0)) 1) 5e-16)) #t)
  (test (equivalent? (complex (real-part (log 0)) 1) (- (complex (real-part (log 0)) 1) 1/1428571428571429)) #t)
  (test (equivalent? (complex (real-part (log 0)) 1) (+ (complex (real-part (log 0)) 1) 0+5e-16i)) #t)
  (test (equivalent? (complex (real-part (log 0)) 1) (+ (complex (real-part (log 0)) 1) 0-1/1428571428571429i)) #t)
  (test (equivalent? (complex (real-part (log 0)) 1) 0) #f)
  (test (equivalent? (complex (real-part (log 0)) 1) 1) #f)
  (test (equivalent? (complex (real-part (log 0)) 1) 0+1e-300i) #f)
  (test (equivalent? (complex (real-part (log 0)) 1) 1/0) #f)
  (test (equivalent? (complex (real-part (log 0)) 1) (real-part (log 0))) #f)
  (test (equivalent? (complex (real-part (log 0)) 1) (- (real-part (log 0)))) #f)
  (test (equivalent? (complex (real-part (log 0)) 1) (- 0/0)) #f)
  (test (equivalent? (complex (real-part (log 0)) 1) (log 0)) #f)
  (test (equivalent? (complex (real-part (log 0)) 1) (complex 1/0 1)) #f)
  (test (equivalent? (complex (real-part (log 0)) 1) (complex 1 1/0)) #f)
  (test (equivalent? (complex (real-part (log 0)) 1) (complex 1/0 1/0)) #f)
  (test (equivalent? (complex (real-part (log 0)) 1) (complex (real-part (log 0)) 1/0)) #f)
  (test (equivalent? (complex (real-part (log 0)) 1) (complex 1/0 (real-part (log 0)))) #f)
  (test (equivalent? (complex (real-part (log 0)) 1) (complex (real-part (log 0)) (real-part (log 0)))) #f)
  (test (equivalent? (complex (real-part (log 0)) 1) (complex (real-part (log 0)) 1)) #t)
  (test (equivalent? (complex (real-part (log 0)) 1) (complex 1 (real-part (log 0)))) #f)
  (test (equivalent? (complex 1 (real-part (log 0))) (+ (complex 1 (real-part (log 0))) 5e-16)) #t)
  (test (equivalent? (complex 1 (real-part (log 0))) (- (complex 1 (real-part (log 0))) 1/1428571428571429)) #t)
  (test (equivalent? (complex 1 (real-part (log 0))) (+ (complex 1 (real-part (log 0))) 0+5e-16i)) #t)
  (test (equivalent? (complex 1 (real-part (log 0))) (+ (complex 1 (real-part (log 0))) 0-1/1428571428571429i)) #t)
  (test (equivalent? (complex 1 (real-part (log 0))) (real-part (log 0))) #f)
  (test (equivalent? (complex 1 (real-part (log 0))) (- (real-part (log 0)))) #f)
  (test (equivalent? (complex 1 (real-part (log 0))) (- 0/0)) #f)
  (test (equivalent? (complex 1 (real-part (log 0))) (log 0)) #f)
  (test (equivalent? (complex 1 (real-part (log 0))) (complex 1/0 1)) #f)
  (test (equivalent? (complex 1 (real-part (log 0))) (complex 1 1/0)) #f)
  (test (equivalent? (complex 1 (real-part (log 0))) (complex 1/0 1/0)) #f)
  (test (equivalent? (complex 1 (real-part (log 0))) (complex (real-part (log 0)) 1/0)) #f)
  (test (equivalent? (complex 1 (real-part (log 0))) (complex 1/0 (real-part (log 0)))) #f)
  (test (equivalent? (complex 1 (real-part (log 0))) (complex (real-part (log 0)) (real-part (log 0)))) #f)
  (test (equivalent? (complex 1 (real-part (log 0))) (complex (real-part (log 0)) 1)) #f)
  (test (equivalent? (complex 1 (real-part (log 0))) (complex 1 (real-part (log 0)))) #t)) ; end with-bignums

(test (equivalent? 1.0e10 (+ 1.0e10 1.0e-6)) #f)
(test (equivalent? (open-input-file "s7test.scm") (open-input-file "s7test.scm")) #t)
(test (equivalent? (make-iterator (hash-table 'a 1)) (make-iterator (hash-table 'a 1))) #t)
(let ((i1 (make-iterator (hash-table 'a 1 'b 2)))
      (i2 (make-iterator (hash-table 'a 1 'b 2))))
  (i1) (i2)
  (test (equivalent? i1 i2) #t))
(let ((i1 (make-iterator (hash-table 'a 1 'b 2)))
      (i2 (make-iterator (hash-table 'a 1 'b 2))))
  (i1)
  (test (equivalent? i1 i2) #f))
(test (equivalent? (make-iterator (inlet 'a 1)) (make-iterator (inlet 'a 1))) #t)
(let ((i1 (make-iterator (inlet 'a 1 'b 2)))
      (i2 (make-iterator (inlet 'a 1 'b 2))))
  (test (equivalent? i1 i2) #t)
  (i1)
  (test (equivalent? i1 i2) #f)
  (i2)
  (test (equivalent? i1 i2) #t)
  (i2)
  (test (equivalent? i1 i2) #f))
(test (equivalent? (make-iterator (list unless 1+i 0/0+i #u())) (make-iterator (list unless 1+i 0/0+i #u()))) #t)
(let ((i1 (make-iterator (list +nan.0 +inf.0)))
      (i2 (make-iterator (list +nan.0 +inf.0))))
  (test (equivalent? i1 i2) #t)
  (i1)
  (test (equivalent? i1 i2) #f))

(when with-block
  (let-temporarily (((*s7* 'equivalent-float-epsilon) 1e-6))
    (test (equivalent? (make-iterator (block (/ pi 2))) (make-iterator (block 1.570796326794897))) #t)))

(when with-bignums
  (test (equivalent? (bignum +nan.0) 0/0+0/0i) #f)
  (test (equivalent? (bignum 0) 0+0/0i) #f)
  (test (equivalent? +nan.0 (bignum 0/0+0/0i)) #f)
  (test (equivalent? 0 (bignum 0+0/0i)) #f)
  (test (equivalent? 1/2 (bignum 1/2+0/0i)) #f)
  (test (equivalent? (bignum 1/2) 1/2+0/0i) #f)
  (test (equivalent? (bignum 1/2+0/0i) 1/2+0/0i) #t)
  (test (equivalent? (vector 1) (vector (bignum 1))) #t)
  (test (equivalent? (list 1.0) (list (bignum 1.0))) #t)
  (test (equivalent? (hash-table 'a 1/2) (hash-table 'a (bignum 1/2))) #t)
  (test (equivalent? (inlet 'a 1+i) (inlet 'a (bignum 1+i))) #t))

(let ((h1 (make-hash-table 8 equivalent?)))
  (set! (h1 (complex 0.0 +inf.0)) 1)
  (test (h1 (complex 0.0 +inf.0)) 1)
  (let ((h2 (make-hash-table 8 equivalent?)))
    (set! (h2 (complex 0.0 +inf.0)) 1)
    (test (equivalent? h1 h2) #t)
    (let ((h3 (copy h1)))
      (test (equivalent? h1 h3) #t)
      (set! (h3 (complex +nan.0 +inf.0)) 2)
      (set! (h1 (complex +nan.0 +inf.0)) 2)
      (test (equivalent? h1 h3) #t)
      (set! (h2 (complex +nan.0 0.0)) 2)
      (test (equivalent? h1 h2) #f)
      )))
(test (equivalent? (let ((h (make-hash-table 8 equivalent?)))
			(set! (h (lambda (x) (or x))) (log 0))
			h)
		      (eval-string (object->string (let ((h (make-hash-table 8 equivalent?)))
						     (set! (h (lambda (x) (or x))) (log 0))
						     h)
						   :readable)))
      #t)


;;; ----------------
;;; try a bunch of combinations

(define-expansion (format-with-line port str . args)
  `(format ,port ,str ,(port-line-number) ,@args))

(let ((lst1 ())
      (lst2 ()))
  (if (not (eq? lst1 lst2)) (format-with-line #t ";~A: nils are not eq?~%"))
  (if (not (eqv? lst1 lst2)) (format-with-line #t ";~A: nils are not eqv?~%"))
  (if (not (equal? lst1 lst2)) (format-with-line #t ";~A: nils are not equal?~%"))

  (let ((v1 (make-vector 100 #f))
	(v2 (make-vector 100 #f)))
    (if (not (equal? v1 v2)) (format-with-line #t ";~A: base vectors are not equal?~%"))

    (let ((h1 (make-hash-table))
	  (h2 (make-hash-table)))
      (if (not (equal? h1 h2)) (format-with-line #t ";~A: base hash-tables are not equal?~%"))

      (let ((e1 (sublet (curlet)))
	    (e2 (sublet (curlet))))
	(if (not (equal? e1 e2)) (format-with-line #t ";~A: base environments are not equal?~%"))

	(let ((ctr 0))
	  (for-each
	   (lambda (arg1 arg2)
	     ;; make sure the args are eq? to themselves
	     ;; if equal? and equal to copy place in lst1, place copy in lst2, check that they are still equal
	     ;;     similarly for vector, hash-table, envs
	   (let ((a1 arg1)
		 (a2 arg2))
	     (if (not (eq? a1 arg1))
		 (format-with-line #t ";~A: ~A is not eq? to itself? ~A~%" arg1 a1))
	     (if (and (eq? a1 a2) (not (eqv? a1 a2)))
		 (format-with-line #t ";~A: ~A is eq? but not eqv? ~A~%" a1 a2))

	     (if (equal? a1 a2)
		 (begin
		   (if (and (eq? a1 a2) (not (eqv? a1 a2)))
		       (format-with-line #t ";~A: ~A is eq? and equal? but not eqv?? ~A~%" a1 a2))
		   (if (not (equivalent? a1 a2))
		       (format-with-line #t ";~A: ~A is equal? but not equivalent? ~A~%" a1 a2))
		   (set! lst1 (cons a1 lst1))
		   (set! lst2 (cons a2 lst2))
		   (set! (v1 ctr) a1)
		   (set! (v2 ctr) a2)
		   (let* ((sym1 (symbol "symbol-" (number->string ctr)))
			  (sym2 (copy sym1)))
		     (set! (h1 sym1) a1)
		     (set! (h2 sym2) a2)
		     (varlet e1 (cons sym1 a1))
		     (varlet e2 (cons sym2 a2))

		     (if (not (equal? lst1 lst2))
			 (begin
			   (format-with-line #t ";~A: add ~A to lists, now not equal?~%" a1)
			   (set! lst1 (cdr lst1))
			   (set! lst2 (cdr lst2))))
		     (if (not (equal? v1 v2))
			 (begin
			   (format-with-line #t ";~A: add ~A to vectors, now not equal?~%" a1)
			   (set! (v1 ctr) #f)
			   (set! (v2 ctr) #f)))
		     (if (not (equal? h1 h2))
			 (begin
			   (format-with-line #t ";~A: add ~A to hash-tables, now not equal?~%" a1)
			   (set! (h1 sym1) #f)
			   (set! (h2 sym2) #f)))
		     (if (not (equal? e1 e2))
			 (begin
			   (format-with-line #t ";~A: add ~A to environments, now not equal?~% ~A~% ~A~%" a1 e1 e2)
			   (eval `(set! ,sym1 #f) e1)
			   (eval `(set! ,sym2 #f) e2)))
		     ))
		 (begin
		   (if (eq? a1 arg1) (format-with-line #t ";~A: ~A is eq? but not equal? ~A~%" a1 a2))
		   (if (eqv? a1 arg1) (format-with-line #t ";~A: ~A is eqv? but not equal? ~A~%" a1 a2))
		   (format-with-line #t ";~A: ~A is not equal to ~A~%" a1 a2)))

	     (set! ctr (+ ctr 1))))

	 (list "hi" ""
	       (integer->char 65) #\space #\newline #\null
	       1 3/4
	       ;; 1.0 1+i pi (real-part (log 0)) 1e18
	       most-negative-fixnum most-positive-fixnum
	       'a-symbol
	       (make-vector 3 #f) #() #2d((1 2) (3 4))
	       abs quasiquote macroexpand (log 0)
	       (hash-table 'a 1 'b 2) (hash-table)
	       (sublet (curlet) 'a 1) (rootlet)
	       #f #t :hi
	       #<eof> #<undefined> #<unspecified>
	       (cons 1 2) () '(1) (list (cons 1 2)) '(1 2 . 3)
	       (let ((lst (cons 1 2))) (set-cdr! lst lst) lst)
	       )
	 (list (string #\h #\i) (string)
	       #\A #\space #\newline (integer->char 0)
	       (- 2 1) (/ 3 4)
	       ;; 1.0 1+i pi (real-part (log 0)) 1e18
	       -9223372036854775808 9223372036854775807
	       (string->symbol "a-symbol")
	       (vector #f #f #f) (vector)  #2d((1 2) (3 4))
	       abs quasiquote macroexpand (log 0)
	       (let ((h (make-hash-table 31))) (set! (h 'a) 1) (set! (h 'b) 2) h) (make-hash-table 123)
	       (sublet (curlet) '(a . 1)) (rootlet)
	       #f #t :hi
	       #<eof> #<undefined> (if #f #f)
	       '(1 . 2) (list) (list 1) (list (cons 1 2)) '(1 2 . 3)
	       (let ((lst (cons 1 2))) (set-cdr! lst lst) lst)
	       ))

	  (set! (v1 ctr) lst1)
	  (set! (v2 ctr) lst2)
	  (set! ctr (+ ctr 1))
	  (if (not (equal? v1 v2))
	      (format-with-line #t ";~A: add lists to vectors, now vectors not equal?~%")
	      (begin
		(set! lst1 (cons v1 lst1))
		(set! lst2 (cons v2 lst2))
		(if (not (equal? lst1 lst2))
		    (begin
		      (format-with-line #t ";~A: add vectors to lists, now lists not equal?~%")
		      (set! (h1 'lst1) lst1)
		      (set! (h2 'lst2) lst2)
		      (if (not (equal? h1 h2))
			  (format-with-line #t ";~A: add lists to hash-tables, not hash-tables not equal?~%")
			  (begin
			    (set! (v1 ctr) v1)
			    (set! (v2 ctr) v2)
			    (set! ctr (+ ctr 1))
			    (if (not (equal? v1 v2))
				(format-with-line #t ";~A: add vectors to themselves, now vectors not equal?~%"))
			    (if (not (equal? lst1 lst2))
				(format-with-line #t ";~A: add vectors to themselves, now lists not equal?~%"))
			    (set! (h1 'h1) h1)
			    (set! (h2 'h2) h2)
			    (if (not (equal? h1 h2))
				(format-with-line #t ";~A: add hash-tables to themselves, not hash-tables not equal?~%"))
			    )))))))))))

(define old-readers *#readers*)
(set! *#readers* (cons (cons #\u (lambda (str) (string->number (substring str 1)))) ()))
(test (eval (with-input-from-string "(+ 10 #u12)" read)) 22)
(test (eval (with-input-from-string "(+ 10 #u87)" read)) 97)

(set! *#readers*
  (list (cons #\[
	      (lambda (str)
		(let ((h (make-hash-table)))
		  (do ((c (read) (read)))
		      ((eq? c ']#) h)
		    (set! (h (car c)) (cdr c))))))))

(eval-string "(let ((table #[(a . 1) (b . #[(c . 3)]#)]#))
  (test (hash-table? table) #t)
  (test (table 'a) 1)
  (test (hash-table? (table 'b)) #t)
  (test ((table 'b) 'c) 3))" (curlet))

(set! *#readers* old-readers)

(when with-block
  (let ((b (make-block 4)))
    (test (equivalent? b b) #t)
    (let ((b1 (make-block 4)))
      (test (equivalent? b b1) #t)
      (set! (b 1) 1.0)
      (test (equivalent? b b1) #f))))
(test (let ((p (c-pointer 0))) (equivalent? p (copy p))) #t)


;;; --------------------------------------------------------------------------------
;;; some clm opt coverage tests
;;; move these!

(let () (define (fc) (do ((count 0) (i 7 (+ i 1))) ((= i 10) count) (set! count (quotient i 3)))) (num-test (fc) (quotient 9 3)))
(let () (define (fc) (do ((count 0) (i 7 (+ i 1))) ((= i 10) count) (set! count (ash 3 3)))) (num-test (fc) (ash 3 3)))
(let () (define (fc) (do ((count 0) (i 7 (+ i 1))) ((= i 10) count) (set! count (ash 3 i)))) (num-test (fc) (ash 3 9)))
(let () (define (fc) (do ((count 0) (i 7 (+ i 1))) ((= i 10) count) (set! count (ash 3 (+ i 1))))) (num-test (fc) (ash 3 10)))
(let () (define (fc) (do ((count 0) (i 7 (+ i 1))) ((= i 10) count) (set! count (ash i (+ i 1))))) (num-test (fc) (ash 9 10)))
(let () (define (fc) (do ((count 0) (i 7 (+ i 1))) ((= i 10) count) (set! count (ash (+ i 1) (- i 1))))) (num-test (fc) (ash 10 8)))
(let () (define (fc) (do ((count 0) (j 3) (i 7 (+ i 1))) ((= i 10) count) (set! count (quotient i j)))) (num-test (fc) (quotient 9 3)))
(let () (define (fc) (do ((count 0.0) (i 7.0 (+ i 1.0))) ((>= i 10.0) count) (set! count (quotient i 3.0)))) (num-test (fc) (quotient 9.0 3.0)))
(let () (define (fc) (do ((count 0) (i 7 (+ i 1))) ((= i 10) count) (set! count (remainder i 3)))) (test (fc) (remainder 9 3)))
(let () (define (fc) (do ((count 0.0) (i 7.0 (+ i 1.0))) ((>= i 10.0) count) (set! count (remainder i 3.0)))) (test (fc) (remainder 9.0 3.0)))
(let () (define (fc) (do ((count 0) (j 3) (i 7 (+ i 1))) ((= i 10) count) (set! count (remainder i j)))) (test (fc) (remainder 9 3)))
(let () (define (fc) (do ((count 0) (i 7 (+ i 1))) ((= i 10) count) (set! count (modulo i 3)))) (test (fc) (modulo 9 3)))
(let () (define (fc) (do ((count 0.0) (i 7.0 (+ i 1.0))) ((>= i 10.0) count) (set! count (modulo i 3.0)))) (test (fc) (modulo 9.0 3.0)))
(let () (define (fc) (do ((count 0) (j 3) (i 7 (+ i 1))) ((= i 10) count) (set! count (modulo i j)))) (test (fc) (modulo 9 3)))
(let () (define (fc) (do ((count 0) (i 7 (+ i 1))) ((= i 10) count) (set! count (max i 3)))) (test (fc) (max 9 3)))
(let () (define (fc) (do ((count 0.0) (i 7.0 (+ i 1.0))) ((>= i 10.0) count) (set! count (max i 3.0)))) (test (fc) (max 9.0 3.0)))
(let () (define (fc) (do ((count 0) (j 3) (i 7 (+ i 1))) ((= i 10) count) (set! count (max i j)))) (test (fc) (max 9 3)))
(let () (define (fc) (do ((count 0) (i 7 (+ i 1))) ((= i 10) count) (set! count (min i 3)))) (test (fc) (min 9 3)))
(let () (define (fc) (do ((count 0.0) (i 7.0 (+ i 1.0))) ((>= i 10.0) count) (set! count (min i 3.0)))) (test (fc) (min 9.0 3.0)))
(let () (define (fc) (do ((count 0) (j 3) (i 7 (+ i 1))) ((= i 10) count) (set! count (min i j)))) (test (fc) (min 9 3)))
(let () (define (fc) (do ((count 0) (i 7 (+ i 1))) ((= i 10) count) (set! count (logior i 3)))) (test (fc) (logior 9 3)))
(let () (define (fc) (do ((count 0) (j 3) (i 7 (+ i 1))) ((= i 10) count) (set! count (logior i j)))) (test (fc) (logior 9 3)))
(let () (define (fc) (do ((count 0) (i 7 (+ i 1))) ((= i 10) count) (set! count (logand i 3)))) (test (fc) (logand 9 3)))
(let () (define (fc) (do ((count 0) (j 3) (i 7 (+ i 1))) ((= i 10) count) (set! count (logand i j)))) (test (fc) (logand 9 3)))
(let () (define (fc) (do ((count 0) (i 7 (+ i 1))) ((= i 10) count) (set! count (logxor i 3)))) (test (fc) (logxor 9 3)))
(let () (define (fc) (do ((count 0) (j 3) (i 7 (+ i 1))) ((= i 10) count) (set! count (logxor i j)))) (test (fc) (logxor 9 3)))
(let () (define (fc) (do ((count 0) (i 7 (+ i 1))) ((= i 10) count) (set! count (lognot i)))) (test (fc) (lognot 9)))
(let () (define (fc) (do ((count 0) (i 7 (+ i 1))) ((= i 10) count) (set! count (ash i 3)))) (test (fc) (ash 9 3)))
(let () (define (fc) (do ((count 0) (j 3) (i 7 (+ i 1))) ((= i 10) count) (set! count (ash i j)))) (test (fc) (ash 9 3)))
(let () (define (fc) (do ((count 0) (i 7 (+ i 1))) ((= i 10) count) (set! count (ash i -3)))) (test (fc) (ash 9 -3)))
(let ((lt (inlet 'a 10))) (define (fc) (do ((i 0 (+ i 1))) ((= i 1) (let-ref lt 'a)) (let-set! lt 'a i))) (test (fc) 0))
(let ((lt (inlet 'a 10)) (sym 'a)) (define (fc) (do ((i 0 (+ i 1))) ((= i 1) (let-ref lt 'a)) (let-set! lt sym i))) (test (fc) 0))
(let ((dfn (vector #f))) (define (fc) (do ((i 0 (+ i 1))) ((= i 1) dfn) (if (defined? 'abs) (vector-set! dfn 0 #t)))) (test (fc) #(#t)))
(let ((dfn (vector #f))) (define (fc) (do ((i 0 (+ i 1))) ((= i 1) dfn) (if (defined? 'abs (curlet)) (vector-set! dfn 0 #t)))) (test (fc) #(#t)))
(let ((dfn (vector ""))) (define (fc) (do ((i 0 (+ i 1))) ((= i 1) dfn) (vector-set! dfn 0 (number->string i)))) (test (fc) (vector (number->string 0))))
(let ((dfn (vector ""))) (define (fc) (do ((i 0 (+ i 1))) ((= i 1) dfn) (vector-set! dfn 0 (number->string i 8)))) (test (fc) (vector (number->string 0 8))))
(let ((dfn (float-vector 0.0))) (define (fc) (do ((i 0.0 (+ i 1.0))) ((>= i 1.0) dfn) (float-vector-set! dfn 0 (magnitude i)))) (test (fc) (float-vector (magnitude 0.0))))
(let ((dfn (float-vector 0.0))) (define (fc) (do ((i 0.0 (+ i 1.0))) ((>= i 1.0) dfn) (float-vector-set! dfn 0 (angle i)))) (test (fc) (float-vector (angle 0.0))))
(let ((dfn (vector 0.0))) (define (fc) (do ((i 0 (+ i 1))) ((>= i 1) dfn) (vector-set! dfn 0 (complex i i)))) (test (fc) (vector 0)))
(let ((dfn (float-vector 0.0))) (define (fc) (do ((i 0.0 (+ i 1.0))) ((>= i 1.0) dfn) (float-vector-set! dfn 0 (sin i)))) (test (fc) (float-vector (sin 0.0))))
(let ((dfn (float-vector 0.0))) (define (fc) (do ((i 0.0 (+ i 1.0))) ((>= i 1.0) dfn) (float-vector-set! dfn 0 (cos i)))) (test (fc) (float-vector (cos 0.0))))
(let ((dfn (float-vector 0.0))) (define (fc) (do ((i 0.0 (+ i 1.0))) ((>= i 1.0) dfn) (float-vector-set! dfn 0 (tan i)))) (test (fc) (float-vector (tan 0.0))))
(let ((dfn (float-vector 0.0))) (define (fc) (do ((i 0.0 (+ i 1.0))) ((>= i 1.0) dfn) (float-vector-set! dfn 0 (atan i i)))) (test (fc) (float-vector (atan 0.0 0.0))))
(let ((dfn (float-vector 0.0))) (define (fc) (do ((i 0.0 (+ i 1.0))) ((>= i 1.0) dfn) (float-vector-set! dfn 0 (sinh i)))) (test (fc) (float-vector (sinh 0.0))))
(let ((dfn (float-vector 0.0))) (define (fc) (do ((i 0.0 (+ i 1.0))) ((>= i 1.0) dfn) (float-vector-set! dfn 0 (cosh i)))) (test (fc) (float-vector (cosh 0.0))))
(let ((dfn (float-vector 0.0))) (define (fc) (do ((i 0.0 (+ i 1.0))) ((>= i 1.0) dfn) (float-vector-set! dfn 0 (tanh i)))) (test (fc) (float-vector (tanh 0.0))))
(let ((dfn (float-vector 0))) (define (fc) (do ((i 0.0 (+ i 1.0))) ((>= i 1.0) dfn) (float-vector-set! dfn 0 (floor i)))) (test (fc) (float-vector (floor 0.0))))
(let ((dfn (int-vector 0))) (define (fc) (do ((i 0 (+ i 1))) ((= i 1) dfn) (int-vector-set! dfn 0 (floor (sqrt i))))) (test (fc) (int-vector (sqrt 0))))
(let ((dfn (float-vector 0))) (define (fc) (do ((i 0.0 (+ i 1.0))) ((>= i 1.0) dfn) (float-vector-set! dfn 0 (ceiling i)))) (test (fc) (float-vector (ceiling 0.0))))
(let ((dfn (int-vector 0))) (define (fc) (do ((i 0 (+ i 1))) ((= i 1) dfn) (int-vector-set! dfn 0 (ceiling (sqrt i))))) (test (fc) (int-vector (ceiling (sqrt 0.0)))))
(let ((dfn (int-vector 0))) (define (fc) (do ((i 0 (+ i 1))) ((= i 1) dfn) (int-vector-set! dfn 0 (ceiling i)))) (test (fc) (int-vector (ceiling 0))))
(let ((dfn (int-vector 0))) (define (fc) (do ((i 0.0 (+ i 1.0))) ((>= i 1.0) dfn) (int-vector-set! dfn 0 (ceiling i)))) (test (fc) (int-vector (ceiling 0.0))))
(let ((dfn (float-vector 0))) (define (fc) (do ((i 0.0 (+ i 1.0))) ((>= i 1.0) dfn) (float-vector-set! dfn 0 (round i)))) (test (fc) (float-vector (round 0.0))))
(let ((dfn (int-vector 0))) (define (fc) (do ((i 0 (+ i 1))) ((= i 1) dfn) (int-vector-set! dfn 0 (round (sqrt i))))) (test (fc) (int-vector (round (sqrt 0)))))
(let ((dfn (int-vector 0))) (define (fc) (do ((i 0 (+ i 1))) ((= i 1) dfn) (int-vector-set! dfn 0 (round i)))) (test (fc) (int-vector (round 0))))
(let ((dfn (int-vector 0))) (define (fc) (do ((i 0.0 (+ i 1.0))) ((>= i 1.0) dfn) (int-vector-set! dfn 0 (round i)))) (test (fc) (int-vector (round 0.0))))
(let ((dfn (int-vector 0))) (define (fc) (do ((i 0 (+ i 1))) ((= i 1) dfn) (int-vector-set! dfn 0 (truncate i)))) (test (fc) (int-vector (truncate 0))))
(let ((dfn (vector #f))) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (vector-set! dfn 0 (= i count)))) (test (fc) #(#t))) ;equal_p_ii
(let ((dfn (vector #f))) (define (fc) (do ((i 0.0 (+ i 1.0))) ((>= i 1.0) dfn) (vector-set! dfn 0 (= i 0.0)))) (test (fc) #(#t))) ;equal_p_dd
(let ((dfn #f)) (define (fc) (do ((i 0 (+ i 1))) ((= i 1) dfn) (set! dfn (= i 0)))) (test (fc) #t)) ;equal_p_pp
(let ((dfn #f)) (define (fc) (do ((count '(0)) (i 0 (+ i 1))) ((= i 1) dfn) (set! dfn (= (car count) 0)))) (test (fc) #t)) ;equal_p_pi
(let ((dfn (vector #f))) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (vector-set! dfn 0 (>= i count)))) (test (fc) #(#t)))
(let ((dfn (vector #f))) (define (fc) (do ((i 0.0 (+ i 1.0))) ((>= i 1.0) dfn) (vector-set! dfn 0 (>= i 0.0)))) (test (fc) #(#t)))
(let ((dfn (vector #f))) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (vector-set! dfn 0 (<= i count)))) (test (fc) #(#t)))
(let ((dfn (vector #f))) (define (fc) (do ((i 0.0 (+ i 1.0))) ((>= i 1.0) dfn) (vector-set! dfn 0 (<= i 0.0)))) (test (fc) #(#t)))
(let ((dfn (vector #f))) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (vector-set! dfn 0 (> i count)))) (test (fc) #(#f)))
(let ((dfn (vector #f))) (define (fc) (do ((i 0.0 (+ i 1.0))) ((>= i 1.0) dfn) (vector-set! dfn 0 (> i 0.0)))) (test (fc) #(#f)))
(let ((dfn (vector #f))) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (vector-set! dfn 0 (< i count)))) (test (fc) #(#f)))
(let ((dfn (vector #f))) (define (fc) (do ((i 0.0 (+ i 1.0))) ((>= i 1.0) dfn) (vector-set! dfn 0 (< i 0.0)))) (test (fc) #(#f)))
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (if (<= i (sqrt count)) (set! dfn #t)))) (test (fc) #t)) ;leq_b_pp
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (if (>= i (sqrt count)) (set! dfn #t))))  (test (fc) #t))
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (if (> i (sqrt count)) (set! dfn #t))))  (test (fc) #f))
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (if (< i (sqrt count)) (set! dfn #t))))  (test (fc) #f))
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (if (>= i count) (set! dfn #t)))) (test (fc) #t))
(let ((dfn #f)) (define (fc) (do ((i 0.0 (+ i 1.0))) ((>= i 1.0) dfn) (if (>= i 0.0) (set! dfn #t)))) (test (fc) #t))
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (if (<= i count) (set! dfn #t)))) (test (fc) #t))
(let ((dfn #f)) (define (fc) (do ((i 0.0 (+ i 1.0))) ((>= i 1.0) dfn) (if (<= i 0.0) (set! dfn #t)))) (test (fc) #t))
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (if (> i count) (set! dfn #t)))) (test (fc) #f))
(let ((dfn #f)) (define (fc) (do ((i 0.0 (+ i 1.0))) ((>= i 1.0) dfn) (if (> i 0.0) (set! dfn #t)))) (test (fc) #f))
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (if (< i count) (set! dfn #t)))) (test (fc) #f))
(let ((dfn #f)) (define (fc) (do ((i 0.0 (+ i 1.0))) ((>= i 1.0) dfn) (if (< i 0.0) (set! dfn #t)))) (test (fc) #f))
(let ((d #f)) (define (fc) (do ((c #\b) (i 0 (+ i 1))) ((= i 1) d) (if (char=? #\b c) (set! d #t)))) (test (fc) #t)) ;char_eq_b_direct
(let ((d #f) (e "b")) (define (fc) (do ((c #\b) (i 0 (+ i 1))) ((= i 1) d) (if (char=? (string-ref e 0) c) (set! d #t)))) (test (fc) #t)) ;char_eq_b
(let ((d #f)) (define (fc) (do ((c #\b) (i 0 (+ i 1))) ((= i 1) d) (if (char<=? #\b c) (set! d #t)))) (test (fc) #t))
(let ((d #f) (e "b")) (define (fc) (do ((c #\b) (i 0 (+ i 1))) ((= i 1) d) (if (char<=? (string-ref e 0) c) (set! d #t)))) (test (fc) #t))
(let ((d #f)) (define (fc) (do ((c #\b) (i 0 (+ i 1))) ((= i 1) d) (if (char>=? #\b c) (set! d #t)))) (test (fc) #t))
(let ((d #f) (e "b")) (define (fc) (do ((c #\b) (i 0 (+ i 1))) ((= i 1) d) (if (char>=? (string-ref e 0) c) (set! d #t)))) (test (fc) #t))
(let ((d #f)) (define (fc) (do ((c #\b) (i 0 (+ i 1))) ((= i 1) d) (if (char<? #\b c) (set! d #t)))) (test (fc) #f))
(let ((d #f) (e "b")) (define (fc) (do ((c #\b) (i 0 (+ i 1))) ((= i 1) d) (if (char<? (string-ref e 0) c) (set! d #t)))) (test (fc) #f))
(let ((d #f)) (define (fc) (do ((c #\b) (i 0 (+ i 1))) ((= i 1) d) (if (char>? #\b c) (set! d #t)))) (test (fc) #f))
(let ((d #f) (e "b")) (define (fc) (do ((c #\b) (i 0 (+ i 1))) ((= i 1) d) (if (char>? (string-ref e 0) c) (set! d #t)))) (test (fc) #f))
(let ((d #f)) (define (fc) (do ((c #\b) (i 0 (+ i 1))) ((= i 1) d) (if (char-ci=? #\b c) (set! d #t)))) (test (fc) #t))
(let ((d #f) (e "b")) (define (fc) (do ((c #\b) (i 0 (+ i 1))) ((= i 1) d) (if (char-ci=? (string-ref e 0) c) (set! d #t)))) (test (fc) #t))
(let ((d #f)) (define (fc) (do ((c #\b) (i 0 (+ i 1))) ((= i 1) d) (if (char-ci<=? #\b c) (set! d #t)))) (test (fc) #t))
(let ((d #f) (e "b")) (define (fc) (do ((c #\b) (i 0 (+ i 1))) ((= i 1) d) (if (char-ci<=? (string-ref e 0) c) (set! d #t)))) (test (fc) #t))
(let ((d #f)) (define (fc) (do ((c #\b) (i 0 (+ i 1))) ((= i 1) d) (if (char-ci>=? #\b c) (set! d #t)))) (test (fc) #t))
(let ((d #f) (e "b")) (define (fc) (do ((c #\b) (i 0 (+ i 1))) ((= i 1) d) (if (char-ci>=? (string-ref e 0) c) (set! d #t)))) (test (fc) #t))
(let ((d #f)) (define (fc) (do ((c #\b) (i 0 (+ i 1))) ((= i 1) d) (if (char-ci<? #\b c) (set! d #t)))) (test (fc) #f))
(let ((d #f) (e "b")) (define (fc) (do ((c #\b) (i 0 (+ i 1))) ((= i 1) d) (if (char-ci<? (string-ref e 0) c) (set! d #t)))) (test (fc) #f))
(let ((d #f)) (define (fc) (do ((c #\b) (i 0 (+ i 1))) ((= i 1) d) (if (char-ci>? #\b c) (set! d #t)))) (test (fc) #f))
(let ((d #f) (e "b")) (define (fc) (do ((c #\b) (i 0 (+ i 1))) ((= i 1) d) (if (char-ci>? (string-ref e 0) c) (set! d #t)))) (test (fc) #f))
(let ((d #f)) (define (fc) (do ((c "b") (i 0 (+ i 1))) ((= i 1) d) (if (string=? "b" c) (set! d #t)))) (test (fc) #t))
(let ((d #f) (e #("b"))) (define (fc) (do ((c "b") (i 0 (+ i 1))) ((= i 1) d) (if (string=? (vector-ref e 0) c) (set! d #t)))) (test (fc) #t))
(let ((d #f)) (define (fc) (do ((c "b") (i 0 (+ i 1))) ((= i 1) d) (if (string<=? "b" c) (set! d #t)))) (test (fc) #t))
(let ((d #f) (e #("b"))) (define (fc) (do ((c "b") (i 0 (+ i 1))) ((= i 1) d) (if (string<=? (vector-ref e 0) c) (set! d #t)))) (test (fc) #t))
(let ((d #f)) (define (fc) (do ((c "b") (i 0 (+ i 1))) ((= i 1) d) (if (string>=? "b" c) (set! d #t)))) (test (fc) #t))
(let ((d #f) (e #("b"))) (define (fc) (do ((c "b") (i 0 (+ i 1))) ((= i 1) d) (if (string>=? (vector-ref e 0) c) (set! d #t)))) (test (fc) #t))
(let ((d #f)) (define (fc) (do ((c "b") (i 0 (+ i 1))) ((= i 1) d) (if (string<? "b" c) (set! d #t)))) (test (fc) #f))
(let ((d #f) (e #("b"))) (define (fc) (do ((c "b") (i 0 (+ i 1))) ((= i 1) d) (if (string<? (vector-ref e 0) c) (set! d #t)))) (test (fc) #f))
(let ((d #f)) (define (fc) (do ((c "b") (i 0 (+ i 1))) ((= i 1) d) (if (string>? "b" c) (set! d #t)))) (test (fc) #f))
(let ((d #f) (e #("b"))) (define (fc) (do ((c "b") (i 0 (+ i 1))) ((= i 1) d) (if (string>? (vector-ref e 0) c) (set! d #t)))) (test (fc) #f))
(let ((d #f)) (define (fc) (do ((c "b") (i 0 (+ i 1))) ((= i 1) d) (if (string-ci=? "b" c) (set! d #t)))) (test (fc) #t))
(let ((d #f) (e #("b"))) (define (fc) (do ((c "b") (i 0 (+ i 1))) ((= i 1) d) (if (string-ci=? (vector-ref e 0) c) (set! d #t)))) (test (fc) #t))
(let ((d #f)) (define (fc) (do ((c "b") (i 0 (+ i 1))) ((= i 1) d) (if (string-ci<=? "b" c) (set! d #t)))) (test (fc) #t))
(let ((d #f) (e #("b"))) (define (fc) (do ((c "b") (i 0 (+ i 1))) ((= i 1) d) (if (string-ci<=? (vector-ref e 0) c) (set! d #t)))) (test (fc) #t))
(let ((d #f)) (define (fc) (do ((c "b") (i 0 (+ i 1))) ((= i 1) d) (if (string-ci>=? "b" c) (set! d #t)))) (test (fc) #t))
(let ((d #f) (e #("b"))) (define (fc) (do ((c "b") (i 0 (+ i 1))) ((= i 1) d) (if (string-ci>=? (vector-ref e 0) c) (set! d #t)))) (test (fc) #t))
(let ((d #f)) (define (fc) (do ((c "b") (i 0 (+ i 1))) ((= i 1) d) (if (string-ci<? "b" c) (set! d #t)))) (test (fc) #f))
(let ((d #f) (e #("b"))) (define (fc) (do ((c "b") (i 0 (+ i 1))) ((= i 1) d) (if (string-ci<? (vector-ref e 0) c) (set! d #t)))) (test (fc) #f))
(let ((d #f)) (define (fc) (do ((c "b") (i 0 (+ i 1))) ((= i 1) d) (if (string-ci>? "b" c) (set! d #t)))) (test (fc) #f))
(let ((d #f) (e #("b"))) (define (fc) (do ((c "b") (i 0 (+ i 1))) ((= i 1) d) (if (string-ci>? (vector-ref e 0) c) (set! d #t)))) (test (fc) #f))
(let () (define (fc) (do ((count 0.0) (i 0.0 (+ i 1.0))) ((>= i 3.0) count) (set! count (+ i)))) (test (fc) 2.0)) ;add_d_d
(let () (define (fc) (do ((count 0.0) (i 0.0 (+ i 1.0))) ((>= i 3.0) count) (set! count (magnitude (+ i 1.0))))) (test (fc) 3.0)) ;add_p_dd
(let () (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 3) count) (set! count (- i)))) (test (fc) -2)) ;subtract_i_i
(let () (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 3) count) (set! count (- i i i)))) (test (fc) -2)) ;subtract_i_iii
(let () (define (fc) (do ((count 0.0) (i 0.0 (+ i 1.0))) ((>= i 3.0) count) (set! count (- i)))) (test (fc) -2.0)) ;subtract_d_d
(let () (define (fc) (do ((count 0.0) (i 0.0 (+ i 1.0))) ((>= i 3.0) count) (set! count (- i i)))) (test (fc) 0.0)) ;subtract_d_dd
(let () (define (fc) (do ((count 0.0) (i 0.0 (+ i 1.0))) ((>= i 3.0) count) (set! count (- i i i)))) (test (fc) -2.0)) ;subtract_d_ddd
(let () (define (fc) (do ((count 0.0) (i 0.0 (+ i 1.0))) ((>= i 3.0) count) (set! count (- i i i i)))) (test (fc) -4.0)) ;subtract_d_dddd
(let () (define (fc) (do ((count 0.0) (i 0.0 (+ i 1.0))) ((>= i 3.0) count) (set! count (magnitude (- i 1.0))))) (test (fc) 1.0)) ;sub_p_dd
(let ((fv #r(1.0)))
  (define (fc) (do ((count 0.0) (i 0.0 (+ i 1.0))) ((>= i 3.0) count) (set! count (magnitude (- i (* 0.5 (float-vector-ref fv 0))))))) (test (fc) 1.5)) ;subtract_p_pp
(let () (define (fc) (do ((count 0.0) (i 0.0 (+ i 1.0))) ((>= i 3.0) count) (set! count (* i)))) (test (fc) 2.0)) ;multiply_d_d
(let () (define (fc) (do ((count 0.0) (i 0.0 (+ i 1.0))) ((>= i 3.0) count) (set! count (* i i)))) (test (fc) 4.0)) ;multiply_d_dd
(let () (define (fc) (do ((count 0.0) (i 0.0 (+ i 1.0))) ((>= i 3.0) count) (set! count (* i i i)))) (test (fc) 8.0)) ;multiply_d_ddd
(let () (define (fc) (do ((count 0.0) (i 0.0 (+ i 1.0))) ((>= i 3.0) count) (set! count (* i i i i)))) (test (fc) 16.0)) ;multiply_d_dddd
(let () (define (fc) (do ((count 0.0) (i 0.0 (+ i 1.0))) ((>= i 3.0) count) (set! count (magnitude (* i 1.0))))) (test (fc) 2.0)) ;mul_p_dd
(let ((fv #r(1.0)))
  (define (fc) (do ((count 0.0) (i 0.0 (+ i 1.0))) ((>= i 3.0) count) (set! count (magnitude (* i (* 0.5 (float-vector-ref fv 0))))))) (test (fc) 1.0)) ;multiply_p_pp
(let () (define (fc) (do ((count 0.0) (i 1.0 (+ i 1.0))) ((>= i 3.0) count) (set! count (/ i)))) (test (fc) 0.5)) ;divide_d_d
(let () (define (fc) (do ((count 0.0) (i 1.0 (+ i 1.0))) ((>= i 3.0) count) (set! count (/ i i)))) (test (fc) 1.0)) ;divide_d_dd
(let () (define (fc) (do ((count 0.0) (i 1.0 (+ i 1.0))) ((>= i 3.0) count) (set! count (/ i i i)))) (test (fc) 0.5)) ;divide_d_ddd
(let () (define (fc) (do ((count 0.0) (i 1.0 (+ i 1.0))) ((>= i 3.0) count) (set! count (/ i i i i)))) (test (fc) 0.25)) ;divide_d_dddd
(let () (define (fc) (do ((count 0.0) (i 1 (+ i 1))) ((= i 3) count) (set! count (magnitude (/ i i))))) (test (fc) 1)) ;divide_p_ii
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (set! dfn (<= i (sqrt count))))) (test (fc) #t)) ;leq_p_pp
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (set! dfn (<= (sqrt count) 1)))) (test (fc) #t)) ;leq_p_pi
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (set! dfn (< (sqrt count) 1)))) (test (fc) #t)) ;lt_p_pi
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (set! dfn (>= (sqrt count) 1)))) (test (fc) #f)) ;geq_p_pi
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (set! dfn (> (sqrt count) 1)))) (test (fc) #f)) ;gt_p_pi
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (if (<= (sqrt count) 1) (set! dfn #t)))) (test (fc) #t)) ;leq_b_pi
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (if (< (sqrt count) 1) (set! dfn #t)))) (test (fc) #t)) ;lt_b_pi
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (if (>= (sqrt count) 1) (set! dfn #t)))) (test (fc) #f)) ;geq_b_pi
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (if (> (sqrt count) 1) (set! dfn #t)))) (test (fc) #f)) ;gt_b_pi
(let () (define (fc) (do ((count 0.0) (i 1.0 (+ i 1.0))) ((>= i 3.0) count) (set! count (imag-part (complex i i))))) (test (fc) 2.0)) ;imag_part_d_p
(let () (define (fc) (do ((count 0.0) (i 1.0 (+ i 1.0))) ((>= i 3.0) count) (set! count (real-part (complex i i))))) (test (fc) 2.0)) ;real_part_d_p
(let () (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) count) (set! count (numerator (/ i 2))))) (test (fc) 0)) ;numerator_i
(let () (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) count) (set! count (denominator (/ i 2))))) (test (fc) 1)) ;denominator_i
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (if (nan? i) (set! dfn #t)))) (test (fc) #f)) ;is_nan_b
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (if (infinite? i) (set! dfn #t)))) (test (fc) #f)) ;is_infinite_b
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (if (even? i) (set! dfn #t)))) (test (fc) #t)) ;is_even_i
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (if (odd? i) (set! dfn #t)))) (test (fc) #f)) ;is_odd_i
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (if (even? (magnitude i)) (set! dfn #t)))) (test (fc) #t)) ;is_even_b
(let ((dfn #f))  (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (if (odd? (magnitude i)) (set! dfn #t)))) (test (fc) #f)) ;is_odd_b
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (if (zero? i) (set! dfn #t)))) (test (fc) #t)) ;is_zero_i
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (if (zero? (magnitude i)) (set! dfn #t)))) (test (fc) #t)) ;is_zero_d
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (if (zero? (sqrt i)) (set! dfn #t)))) (test (fc) #t)) ;is_zero_b
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (if (negative? i) (set! dfn #t)))) (test (fc) #f)) ;is_negative_i
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (if (negative? (magnitude i)) (set! dfn #t)))) (test (fc) #f)) ;is_negative_d
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (if (negative? (sqrt i)) (set! dfn #t)))) (test (fc) #f)) ;is_negative_b
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (if (positive? i) (set! dfn #t)))) (test (fc) #f)) ;is_positive_i
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (if (positive? (magnitude i)) (set! dfn #t)))) (test (fc) #f)) ;is_positive_d
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (if (positive? (sqrt i)) (set! dfn #t)))) (test (fc) #f)) ;is_positive_b
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (if (exact? (sqrt i)) (set! dfn #t)))) (test (fc) #t)) ;is_exact_b
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (if (inexact? (sqrt i)) (set! dfn #t)))) (test (fc) #f)) ;is_inexact_b
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) count) (set! count (integer-length i)))) (test (fc) 0)) ;integer_length_i_i
(let () (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) count) (set! count (random i)))) (test (fc) 0)) ;random_i_i
(let () (define (fc) (do ((count 0.0) (i 0.0 (+ i 1.0))) ((>= i 1.0) count) (set! count (random i)))) (test (fc) 0.0)) ;random_d_d
(let () (define (fc) (do ((count (complex 1 1)) (i 0 (+ i 1))) ((= i 1) count) (set! count (random (complex i i))))) (test (fc) 0)) ;random_p_p
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (if (random-state? i) (set! dfn #t)))) (test (fc) #f)) ;is_random_state_b
(let ((str "123qasde")) (define (fc) (do ((i 0 (+ i 1))) ((= i 1)) (char-position #\a str 0))) (test (fc) #t)) ;char_position_p_ppi
(let ((str "123qasde")) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) count) (set! count (string-length str)))) (test (fc) 8)) ;string_length_i
(let ((str #("123qasde")))
  (define (fc) (catch #t (lambda () (do ((count #\a) (i 0 (+ i 1))) ((= i 1) count) (set! count (string-ref str i)))) (lambda args 'error))) (test (fc) 'error)) ;string_ref_p_pi
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (if (macro? i) (set! dfn #t)))) (test (fc) #f)) ;is_macro_b
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (if (float? i) (set! dfn #t)))) (test (fc) #f)) ;is_float_b
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (if (port-closed? *stdout*) (set! dfn #t)))) (test (fc) #f)) ;is_port_closed_b
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (if (directory? "s7test.scm") (set! dfn #t)))) (test (fc) #f)) ;is_directory_b
(let ((dfn #f)) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (if (sequence? i) (set! dfn #t)))) (test (fc) #f)) ;is_sequence_b
(let ((p '(1 2 (3 4)))) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) count) (set! count (tree-leaves p)))) (test (fc) 4)) ; tree_leaves_i
(let () (define (fc) (do ((i 0 (+ i 1))) ((= i 1)) (newline))) (let-temporarily (((current-output-port) #f)) (test (fc) #t)))
(let () (define (fc) (do ((i 0 (+ i 1))) ((= i 1)) (newline #f))) (test (fc) #t))
(let () (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1)) (set! count (port-line-number (current-input-port))))) (test (fc) #t)) ; port_line_number_i_p
(let ((dfn #f)) (define (fc) (do ((i 0 (+ i 1))) ((= i 1) dfn) (if (provided? 'asdf) (set! dfn #t)))) (test (fc) #f)) ; is_provided_b
(let ((str #u(1 2 3))) (define (fc) (do ((count 0) (i 0 (+ i 1))) ((= i 1) count) (set! count (byte-vector-ref str i)))) (test (fc) 1)) ; byte_vector_ref_i
(let ((str #u(1 2 3))) (define (fc) (do ((i 0 (+ i 1))) ((= i 1) (str 0)) (byte-vector-set! str i 4))) (test (fc) 4)) ; byte_vector_set_i

(let () (define (f1) (let ((dfn #f)) (do ((count 0) (i 0 (+ i 1))) ((= i 1) dfn) (if (logbit? i count) (set! dfn #t))))) (test (f1) #f)) ; opt_b_ii_ss
(let () (define (f2) (let ((dfn #f)) (do ((i 0 (+ i 1))) ((= i 1) dfn) (if (logbit? i 0) (set! dfn #t))))) (test (f2) #f)) ; opt_b_ii_sc_bit
(let () (define (f3) (let ((dfn #f)) (do ((count 0.0) (i 0.0 (+ i 1.0))) ((= i 1.0) dfn) (if (> i count) (set! dfn #t))))) (test (f3) #f)) ; opt_b_dd_ss_gt
(let () (define (f4) (let ((dfn #f)) (do ((count 1.0) (i 0.0 (+ i 1.0))) ((= i 1.0) dfn) (if (< i count) (set! dfn #t))))) (test (f4) #t)) ; opt_b_dd_ss_lt
(let () (define (f5) (let ((dfn #f)) (do ((count 1.0) (i 0.0 (+ i 1.0))) ((= i 1.0) dfn) (if (< i 1.0) (set! dfn #t))))) (test (f5) #t)) ; opt_b_dd_sc_lt
(let () (define (f6) (let ((dfn #f)) (do ((count 1.0) (i 0.0 (+ i 1.0))) ((= i 1.0) dfn) (if (>= i 0.0) (set! dfn #t))))) (test (f6) #t)) ; opt_b_dd_sc_geq
(let () (define (f7) (let ((dfn #f)) (do ((count 1.0) (i 0.0 (+ i 1.0))) ((= i 1.0) dfn) (if (<= i 1.0) (set! dfn #t))))) (test (f7) #t)) ; opt_b_dd_sc
(let () (define (f8) (let ((dfn #f)) (do ((count 1.0) (i 0.0 (+ i 1.0))) ((>= i 1.0) dfn) (if (= i 1.0) (set! dfn #t))))) (test (f8) #f)) ; opt_b_dd_sc_eq
(let () (define (f9) (let ((dfn #f)) (do ((count 1.0) (i 0.0 (+ i 1.0))) ((>= i 1.0) dfn) (if (= i (+ count 1.0)) (set! dfn #t))))) (test (f9) #f)) ; opt_b_dd_sf
(let () (define (f10) (let ((dfn #f)) (do ((count 1.0) (i 0.0 (+ i 1.0))) ((>= i 1.0) dfn) (if (= (+ i 1.0) (+ count 1.0)) (set! dfn #t))))) (test (f10) #f)) ; opt_b_dd_ff
(let () (define (f11) (do ((x 1.0) (i 0 (+ i 1))) ((= i 1)) (if (negative? (+ x 1.0)) (* x 2) (- x 3)))) (test (f11) #t)) ; opt_b_d_f
(let () (define (f12) (let ((dfn #\c)) (do ((i 0 (+ i 1))) ((= i 1) dfn) (if (char=? dfn #\a) (set! dfn #\b))))) (test (f12) #\c)) ; opt_b_7pp_sc
(let () (define (f13) (let ((dfn #\c)) (do ((i 0 (+ i 1))) ((= i 1) dfn) (if (eq? dfn #\a) (set! dfn #\b))))) (test (f13) #\c)) ; opt_b_pp_sc

(let () (define (f14) (let ((dfn #f)) (do ((count 0) (i 0 (+ i 1))) ((= i 1) count) (set! count (random 1))))) (test (f14) 0)) ; opt_i_7i_c
(let () (define (f15) (let ((dfn #f)) (do ((count 0) (i 0 (+ i 1))) ((= i 1) count) (set! count (ash (+ i 1) 3))))) (test (f15) 8)) ; opt_i_7ii_fc

;;; coverage for op_let_a_fx_old -- why does it have to be at top level?
(define (___f fv) (let ((fv-copy (copy fv))) (reverse! fv) (reverse! fv))) (___f (int-vector 1 2 3)) (test (___f (int-vector 1 2 3)) #i(1 2 3))

;;; fx coverage
(define _gfx_ 3)
(define _vfx_ (vector (vector 0)))
(define _vfxi_ (vector 0))
(let () ; fx_* coverage
  (define (f1 x) (and (pair? (cddr x)) (symbol? (cadr x))))
  (test (f1 (list 1 2 3)) #f)
  (test (f1 (list 1 'a 3)) #t)
  (test (f1 (list 1 'a)) #f)

  (define (f2 x) (and (not (null? x)) (pair? (car x))))
  (test (f2 (list 1 2)) #f)
  (test (f2 (list (list 1) 2)) #t)
  (test (f2 (list)) #f)

  (define (f3 x y) (or (< x y) (<= x y)))
  (test (f3 3 2) #f)
  (test (f3 3 3) #t)
  (test (f3 1 2) #t)

  (define (f4 x y) (or (>= x y) (> x _gfx_)))
  (test (f4 4 5) #t)
  (test (f4 3 3) #t)
  (test (f4 2 3) #f)

  (define (f5 fv z) (let ((x (vector-ref fv 0))) (when (< x z) (vector-set! fv 0 (+ x 1)) (f5 fv z))))
  (test (f5 (vector 0) 2) #<unspecified>)

  (define (f6 fv z) (let ((x (length fv))) (when (eqv? x z) (f6 (cons x fv) z))))
  (test (f6 (list 0) 2) #<unspecified>)

  (define (f7 x y) (let ((z x)) (if (zero? z) (f7 (- x 1) (cons z y)))))
  (test (f7 2 ()) #<unspecified>)

  (define (f8 x y z) (or (proper-list? z) (hash-table? x) (integer? z)))
  (test (f8 0 0 (list 1)) #t)
  (test (f8 0 0 1) #t)
  (test (f8 0 0 (vector 1)) #f)

  (define (f9 x y) (or (vector? x) (not x) (vector? y)))
  (test (f9 #f 0) #t)
  (test (f9 #(0) 0) #t)
  (test (f9 () ()) #f)

  (define (f10 x) (or (= x _gfx_) (eqv? x _gfx_)))
  (test (f10 1) #f)
  (test (f10 _gfx_) #t)

  (define (f11 x y z) (or (not (eq? (car z) 'a)) (null? (cddr z)) (eqv? x y)))
  (test (f11 1 2 (list 1 2)) #t)
  (test (f11 1 1 (list 'a 2)) #t)

  (define (f12 x y) (if (not (> y x)) (not (eqv? y x))))
  (test (f12 1 2) #<unspecified>)
  (test (f12 1 1) #f)

  (define (f13 x y q r) (if (zero? (- (* q r) (* r q))) 32 12) (if (< (- q r) (- r q)) 32 12))
  (test (f13 1 2 3 4) 32)

  (define (f14 x y) (let ((z (+ x y))) (cond ((= z 0) pi) ((< z 0) 'oops) (else (f14 (- x 1) (- y 1))))))
  (test (f14 1 2) 'oops)

  (define (f15 lst) (let loop ((p lst) (sum 0)) (if (null? p) sum (loop (cdr p) (+ sum (car p))))))
  (test (f15 (list 0 1 2)) 3)

  (define (f16 x y z) (+ (* 3.0 x) (- 3.0 x) (- z 3.0)))
  (test (f16 3 4 5) 11.0)

  (define (f17 x y z) (let ((v (vector 'a))) (if (eq? z (vector-ref v x)) 0 1)))
  (test (f17 0 0 'a) 0)

  (define (f18 x y z) (let ((v (vector 0))) (if (>= z (vector-ref v x)) 0 1)))
  (test (f18 0 0 0) 0)

  (define (f19 x y z) (let ((v (vector 0))) (if (> (vector-ref v x) z) 0 1)))
  (test (f19 0 0 0) 1)

  (define (f20 x y z) (let ((v (vector 0))) (+ (* z (vector-ref v x)) (- z (vector-ref v y)))))
  (test (f20 0 0 0) 0)

  (define (f21 x y z) (let ((v (vector 0))) (if (> (+ z (vector-ref v x)) 1) 0 1)))
  (test (f21 0 0 0) 1)

  (define (len=2? x) (and (pair? x) (pair? (cdr x)) (null? (cddr x))))
  (define (f22 x) (and (list? x) (len=2? x)))
  (test (f22 (list 1 2)) #t)

  (define (len>2? x) (and (pair? x) (pair? (cdr x)) (pair? (cddr x))))
  (define (f23 x) (and (list? x) (len>2? x)))
  (test (f23 (list 1 2 3)) #t)

  (define (f24 x) (let ((h (hash-table))) (hash-table-set! h x (+ (or (hash-table-ref h x) 0) 1))))
  (test (f24 'a) 1)

  (define (f25 x) (if (or (not (symbol? x)) (keyword? x)) 1 0))
  (test (f25 'a) 0)
  (test (f25 :a) 1)
  (test (f25 #f) 1)

  (define (f26 x)  (if (> (+ _gfx_ (* x 2) 32) 0) 1 0))
  (test (f26 3) 1)

  (define (f27 x) (let ((y 3)) (if (zero? (remainder x y)) 0 1)))
  (test (f27 4) 1)
  (test (f27 6) 0)

  (define (f28 x y) (if (= (remainder (car y) x) 0) 0 (f28 (- x 1) y)))
  (test (f28 2 '(3)) 0)
  (test (f28 3 '(3)) 0)

  (define (f29) (let ((v (vector 1 2)) (i 0) (j 1)) (if (zero? (- (vector-ref v i) (vector-ref v j))) 0 1)))
  (test (f29) 1)

  (define (f30 x) (if (eq? (string-ref (symbol->string (car x)) 0) #\a) 0 1))
  (test (f30 '(abc)) 0)
  (test (f30 '(bcd)) 1)

  (define (f31 x) (do ((y 3 (+ y 1))) ((or (zero? x) (>= y x)) 0)))
  (test (f31 4) 0)
  (test (f31 0) 0)

  (define (f32 x y z) (if (vector-ref x (+ y z)) 1 0))
  (test (f32 (vector #f #t) 1 0) 1)

  (define (f33 x y) (if (string? (number->string (+ 1 (car x) (car x)) y)) 1 0))
  (test (f33 '(0) 10) 1)

  (define (f34 x y q r) (eqv? (vector-ref (vector-ref q r) y) 0))
  (test (f34 0 0 (vector (vector 1)) 0) #f)
  (test (f34 0 0 (vector (vector 0)) 0) #t)

  (define (f35 x y q r) (eqv? (vector-ref (vector-ref x y) q) 0))
  (test (f35 (vector (vector 1)) 0 0 0) #f)
  (test (f35 (vector (vector 0)) 0 0 0) #t)

  (define (f36 x y) (eqv? (vector-ref (vector-ref _vfx_ y) x) 0))
  (test (f36 0 0) #t)
  (test (f36 0 1) 'error)

  (define (f37 x) (eqv? (vector-ref _vfx_ (vector-ref _vfxi_ x)) 0))
  (test (f37 0) #f)

  (define (f38 x y) (eqv? (+ (* x x) (* y y)) 1))
  (test (f38 1 2) #f)
  (test (f38 1 0) #t)

  (define (f39 x y z) (eqv? (vector-ref (vector-ref x y) z) 0))
  (test (f39 (vector (vector 0)) 0 0) #t)
  (test (f39 (vector (vector 1)) 0 0) #f)

  (define (f40 items sequence)
    (cond ((not (pair? sequence)) sequence) ((memq (car sequence) items) (f40 items (cdr sequence))) (else (cons (car sequence) (f40 items (cdr sequence))))))
  (test (f40 '(a b c) '(a d f e b c)) '(d f e))

  (define (f41 row dist placed) (or (null? placed) (and (not (= (car placed) (+ row dist))) (not (= (car placed) (- row dist))) (f41 row (+ dist 1) (cdr placed)))))
  (test (f41 0 0 '(0 1 2)) #f)
  (test (f41 0 1 '(0 1 2)) #t)

  (define (f42 v i j y)
    (if (and (or (> (vector-ref v i) y)
                 (>= y (vector-ref v j)))
             (or (> (vector-ref v j) y)
                 (>= y (vector-ref v i))))
	0 1))
  (test (f42 (vector 1 2 3 4) 1 2 3) 0)
  (test (f42 (vector 1 2 3 4) 1 2 2) 1)

  (define-constant (f43 x)
    (and (pair? x) (pair? (cdr x))))
  (define (g)
    (let ((x (list 1 2)))
      (if (f43 x) 0 1)))
  (test (g) 0)

  (define (f44 fv z)
    (let ((x (vector-ref fv 0)))
      (when (< x 30)
	(vector-set! fv 0 z)
	(f44 fv (+ z 1)))))
  (test (f44 (vector 0) 0) #<unspecified>)

  (define (f45 x y z q)
    (zero? (* x (hash-table-ref y (vector-ref z q)))))
  (test (f45 2.0 (hash-table 'a 3.0) (vector 'a) 0) #f)

  (define (f46 x y z)
    (zero? (- (string->number (vector-ref x y)) z)))
  (test (f46 (vector "3.0") 0 1.0) #f)

  (define (f47 x y)
    (if (number? y)
        (or (positive? y)
            (/ 3.5 y))
        (cddr y)))
  (test (f47 0 1) #t)
  (test (f47 0 -1) -3.5)
  (test (f47 -1 (list 1 2)) ())

  (let () ; tshoot
    (define (palindrome? string)
      (or (< (string-length string) 2)
	  (and (char=? (string-ref string 0)
		       (string-ref string (- (string-length string) 1)))
  	       (palindrome? (substring string 1 (- (string-length string) 1))))))
    (define (pal-test) (test (palindrome? "abcdefgfedcba") #t))
    (pal-test))

  (let () ; primes.scm benchmark
    (define  (iotar m n) (if (> m n) () (cons m (iotar (+ 1 m) n))))
    (define (erat l)
      (letrec ((only-p (lambda (n l)
		         (if (null? l) ()
			     (if (= (remainder (car l) n) 0)
			         (only-p n (cdr l))
			         (cons (car l) (only-p n (cdr l))))))))
        (if (null? l) () (cons (car l) (erat (only-p (car l) (cdr l)))))))
    (define (f49) (erat (iotar 2 20)))
    (test (f49) '(2 3 5 7 11 13 17 19)))
)

(let ()
  ;; unknown_g
  (define (f3 f x) (f x))
  (define (f2 x) (+ x 1))
  (define (f4 x) (list x) (+ x 2))
  (define (f5 x) (call-with-exit (lambda (return) (return (+ x 1)))))
  (define f6 (vector 1 2 3))
  (define f7 "123")
  (test (f3 f2 1) 2)
  (test (f3 f4 1) 3)
  (test (f3 f5 1) 2)
  (test (f3 f6 1) 2)
  (test (f3 f7 1) #\2)

  (define (f8 g x) (g x))
  (test (f8 f2 1) 2)
  (test (f8 (lambda (x) (+ x 1)) 1) 2)
  (test (f8 abs -1) 1)
  (test (f8 + 1) 1)
  (test (f8 (list 1 2 3) 1) 2)

  ;; unknown_ss
  (define (f9 g x y) (g x y))
  (define (f10 x y) (+ x y))
  (define (f11 x y) (list x y) (+ x y))
  (define (f12 x y) (call-with-exit (lambda (return) (return (+ x y)))))
  (define f13 #2r((1 2 3 4 5 6 7) (8 9 10 11 12 13 14)))
  (define-macro (f14 x y) `(+ ,x ,y))
  (test (f9 f10 1 2) 3)
  (test (f9 f11 3 4) 7)
  (test (f9 f12 3 4) 7)
  (test (f9 f13 1 2) 10.0)
  (test (f9 f14 7 8) 15)

  ;; unknown_a
  (define (f13 f x) (f (* x 3)))
  (define (f12 x) (+ x 1))
  (define (f14 x) (list x) (+ x 2))
  (define (f15 x) (call-with-exit (lambda (return) (return (+ x 1)))))
  (define f16 (vector 1 2 3 4 5 6))
  (define f17 "123456")
  (test (f13 f12 1) 4)
  (test (f13 f14 1) 5)
  (test (f13 f15 1) 4)
  (test (f13 f16 1) 4)
  (test (f13 f17 1) #\4)

  (define (f18 g x) (g (* 3 x)))
  (test (f18 f12 1) 4)
  (test (f18 (lambda (x) (+ x 1)) 1) 4)
  (test (f18 abs -1) 3)
  (test (f18 + 1) 3)
  (test (f18 (list 1 2 3 4 5 6) 1) 4))

;; -------- unknown:
(test (let ((e-1 (lambda (x) (x)))) (e-1 +)) 0)
(test (let ((e-1 (lambda (x) (x)))) (e-1 (lambda () 3))) 3)
(test (let ((e-1 (lambda (x) (x)))) (e-1 (lambda* ((d 32)) d))) 32)
(test (let ((e-1 (lambda (x) (x)))) (call-with-exit (lambda (return) (e-1 return) 123))) ())
(test (let ((e-1 (lambda (x) (x)))) (e-1 (macro () `(+ 1 2)))) 3)
(test (let ((e-1 (lambda (x) (x)))) (e-1 (macro* ((d 32)) `(+ 1 ,d)))) 33)
(test (let ((e-1 (lambda (x) (x)))) (let ((iter (make-iterator '(1 2 3)))) (e-1 iter))) 1)

;; -------- unknown_g:
(test (let ((f-1 (lambda (x a) (x a)))) (f-1 abs 2)) 2)
(test (let ((f-1 (lambda (x a) (x a)))) (f-1 / 2)) 1/2)
(test (let ((f-1 (lambda (x a) (x a)))) (f-1 #(0 1 2 3) 2)) 2)
(test (let ((f-1 (lambda (x a) (x a)))) (f-1 #r(1 2 3 4) 2)) 3.0)
(test (let ((f-1 (lambda (x a) (x a)))) (f-1 (lambda (b) (+ b 1)) 2)) 3)
(test (let ((f-1 (lambda (x a) (x a)))) (f-1 (lambda* (b (d 3)) (+ b d)) 2)) 5)
(test (let ((f-1 (lambda (x a) (x a)))) (f-1 (macro (x) `(+ ,x 10)) 2)) 12)
(test (let ((f-1 (lambda (x a) (x a)))) (f-1 (macro* (x (d 10)) `(+ ,x ,d)) 2)) 12)
(test (let ((f-1 (lambda (x a) (x a)))) (f-1 (list 1 2 3 4) 2)) 3)
(test (let ((f-1 (lambda (x a) (x a)))) (f-1 (hash-table 1 2 2 4) 2)) 4)
(test (let ((f-1 (lambda (x a) (x a)))) (f-1 "asdf" 2)) #\d)
(test (let ((f-1 (lambda (x a) (x a)))) (f-1 (block 1 2 3 4) 2)) 3.0)
(test (let ((f-1 (lambda (x a) (x a)))) (f-1 (bacro (x) `(+ ,x 10)) 2)) 12) ; no fixup [-> op_s_s in this case]
(test (let ((f-1 (lambda (x a) (x a)))) (f-1 abs 2)) 2) ; op_s_s
(test (let ((f-1 (lambda (x a) (x a)))) (call-with-exit (lambda (return) (f-1 return 32) 2))) 32)
(test (let ((f-1 (lambda (x a) (x a)))) (call/cc (lambda (return) (f-1 return 32) 2))) 32)
(test (let ((f-1 (lambda (x a) (x a)))) (f-1 (inlet 'a 1 'b 2) 'b)) 2)

;;; -------- unknown_gg:
(test (let ((g-1 (lambda (x a b) (x a b)))) (g-1 + 2 3)) 5)
(test (let ((g-1 (lambda (x a b) (x a b)))) (g-1 / 2 3)) 2/3)
(test (let ((g-1 (lambda (x a b) (x a b)))) (g-1 #2d((0 1 2 3) (4 5 6 7)) 1 2)) 6)
(test (let ((g-1 (lambda (x a b) (x a b)))) (g-1 (lambda (b c) (+ b c)) 2 3)) 5)
(test (let ((g-1 (lambda (x a b) (x a b)))) (g-1 (lambda* (b (d 3)) (+ b d)) 2 1)) 3)
(test (let ((g-1 (lambda (x a b) (x a b)))) (g-1 (macro (x y) `(+ ,x ,y)) 2 3)) 5)
(test (let ((g-1 (lambda (x a b) (x a b)))) (g-1 (macro* (x (d 10)) `(+ ,x ,d)) 2 1)) 3)

;; -------- unknown_a:
(test (let ((f-1 (lambda (x a) (x (+ a 1))))) (f-1 abs 2)) 3)
(test (let ((f-1 (lambda (x a) (x (+ a 1))))) (f-1 / 2)) 1/3)
(test (let ((f-1 (lambda (x a) (x (+ a 1))))) (f-1 #(0 1 2 3) 2)) 3)
(test (let ((f-1 (lambda (x a) (x (+ a 1))))) (f-1 #r(1 2 3 4) 2)) 4.0)
(test (let ((f-1 (lambda (x a) (x (+ a 1))))) (f-1 (lambda (b) (+ b 1)) 2)) 4)
(test (let ((f-1 (lambda (x a) (x (+ a 1))))) (f-1 (lambda* (b (d 3)) (+ b d)) 2)) 6)
(test (let ((f-1 (lambda (x a) (x (+ a 1))))) (f-1 (macro (x) `(+ ,x 10)) 2)) 13)
(test (let ((f-1 (lambda (x a) (x (+ a 1))))) (f-1 (macro* (x (d 10)) `(+ ,x ,d)) 2)) 13)
(test (let ((f-1 (lambda (x a) (x (+ a 1))))) (f-1 (list 1 2 3 4) 2)) 4)
(test (let ((f-1 (lambda (x a) (x (+ a 1))))) (f-1 (hash-table 1 2 3 4) 2)) 4)
(test (let ((f-1 (lambda (x a) (x (+ a 1))))) (f-1 "asdf" 2)) #\f)
(test (let ((f-1 (lambda (x a) (x (+ a 1))))) (f-1 (block 1 2 3 4) 2)) 4.0)
(test (let ((f-1 (lambda (x a) (x (+ a 1))))) (f-1 (bacro (x) `(+ ,x 10)) 2)) 13) ; no fixup [-> op_s_s in this case]
(test (let ((f-1 (lambda (x a) (x (+ a 1))))) (f-1 abs 2)) 3) ; op_s_s
(test (let ((f-1 (lambda (x a) (x (+ a 1))))) (call-with-exit (lambda (return) (f-1 return 32) 2))) 33)
(test (let ((f-1 (lambda (x a) (x (+ a 1))))) (call/cc (lambda (return) (f-1 return 32) 2))) 33)
(test (let ((f-1 (lambda (x a) (x (car '(a)))))) (f-1 (inlet 'a 1 'b 2) 'b)) 1)

;;; -------- unknown_aa:
(test (let ((g-1 (lambda (x a b) (x (+ a 1) (- b 1))))) (g-1 + 2 3)) 5)
(test (let ((g-1 (lambda (x a b) (x (+ a 1) (- b 1))))) (g-1 / 2 3)) 3/2)
(test (let ((g-1 (lambda (x a b) (x (+ a 1) (- b 1))))) (g-1 #2d((0 1 2 3) (4 5 6 7)) 0 3)) 6)
(test (let ((g-1 (lambda (x a b) (x (+ a 1) (- b 1))))) (g-1 (lambda (b c) (+ b c)) 2 3)) 5)
(test (let ((g-1 (lambda (x a b) (x (+ a 1) (- b 1))))) (g-1 (lambda* (b (d 3)) (+ b d)) 2 1)) 3)
(test (let ((g-1 (lambda (x a b) (x (+ a 1) (- b 1))))) (g-1 (macro (x y) `(+ ,x ,y)) 2 3)) 5)
(test (let ((g-1 (lambda (x a b) (x (+ a 1) (- b 1))))) (g-1 (macro* (x (d 10)) `(+ ,x ,d)) 2 1)) 3)

;;; -------- unknown_all_s:
(test (let ((g-1 (lambda (x a b c) (x a b c)))) (g-1 + 2 3 -1)) 4)
(test (let ((g-1 (lambda (x a b c) (x a b c)))) (g-1 / 2 3 2)) 1/3)
(test (let ((g-1 (lambda (x a b c) (x a b c)))) (g-1 (lambda (b c d) (+ b c d)) 2 3 1)) 6)
(test (let ((g-1 (lambda (x a b c) (x a b c)))) (g-1 (lambda* (b (d 3) (e 0)) (+ b d e)) 2 1 3)) 6)
(test (let ((g-1 (lambda (x a b c) (x a b c)))) (g-1 (macro (x y z) `(+ ,x ,y ,z)) 2 3 4)) 9)
(test (let ((g-1 (lambda (x a b c) (x a b c)))) (g-1 (macro* (x (d 10) (e 0)) `(+ ,x ,d ,e)) 2 1 3)) 6)

;;; -------- unknown_all_a:
(test (let ((g-1 (lambda (x a b c) (x (+ a 1) (- b 1) (* c 2))))) (g-1 + 2 3 5)) 15)
(test (let ((g-1 (lambda (x a b c) (x (+ a 1) (- b 1) (* c 2))))) (g-1 / 2 3 4)) 3/16)
(test (let ((g-1 (lambda (x a b c) (x (+ a 1) (- b 1) (* c 2))))) (g-1 (lambda (b c d) (+ b c d)) 2 3 5)) 15)
(test (let ((g-1 (lambda (x a b c) (x (+ a 1) (- b 1) (* c 2))))) (g-1 (lambda* (b (d 3) (e 0)) (+ b d e)) 2 1 3)) 9)
(test (let ((g-1 (lambda (x a b c) (x (+ a 1) (- b 1) (* c 2))))) (g-1 (macro (x y z) `(+ ,x ,y ,z)) 2 3 5)) 15)
(test (let ((g-1 (lambda (x a b c) (x (+ a 1) (- b 1) (* c 2))))) (g-1 (macro* (x (d 10) (e 0)) `(+ ,x ,d ,e)) 2 1 3)) 9)

;;; -------- unknown_all_fp:
(test (let ((g-1 (lambda (x a b c) (x (+ a 1) (- b 1) (values c 2))))) (g-1 + 2 3 5)) 12)
(test (let ((g-1 (lambda (x a b c) (x (+ a 1) (- b 1) (values c 2))))) (g-1 / 2 3 4)) 3/16)
(test (let ((g-1 (lambda (x a b c) (x (+ a 1) (- b 1) (values c 2))))) (g-1 (lambda (b c d e) (+ b c d e)) 2 3 5)) 12)
(test (let ((g-1 (lambda (x a b c) (x (+ a 1) (- b 1) (values c 2))))) (g-1 (lambda* (b c (d 3) (e 0)) (+ b c d e)) 2 3 5)) 12)
(test (let ((g-1 (lambda (x a b c) (x (+ a 1) (- b 1) 2 (values c 2))))) (g-1 (macro (x y z w) `(+ ,x ,y ,z ,w)) 2 3 5)) 14)
(test (let ((g-1 (lambda (x a b c) (x (+ a 1) (- b 1) 2 (values c 2))))) (g-1 (macro* (x c (d 10) (e 0)) `(+ ,x ,c ,d ,e)) 2 1 3)) 10)


;;; --------------------------------------------------------------------------------
;;; c-object?
;;; c-object-type
;;; c-pointer
;;; c-pointer?
;;; c-pointer->list
;;; c-pointer-info
;;; c-pointer-type
;;; c-pointer-weak1|2

(when with-block
  (test (c-object? (block)) #t)
  (test (integer? (c-object-type (block))) #t)
  (test ((*s7* 'c-types) (c-object-type (block))) "<block>")) ; perhaps return block? instead?

(test (c-pointer? 0) #f)
(test (c-pointer? _null_) #t)
(test (c-pointer->list (c-pointer 0 1 2)) '(0 1 2))
(if with-block
    (test (c-object? _c_obj_) #t)
    (test (c-pointer? _c_obj_) #t))

(for-each
 (lambda (arg)
   (test (c-pointer? arg) #f)
   (test (c-object? arg) #f)
   (test (c-object-type arg) 'error)
   (test (c-pointer arg) 'error)
   (test (c-pointer-info arg) 'error)
   (test (c-pointer-type arg) 'error)
   (test (c-pointer->list arg) 'error))
 (list "hi" () (integer->char 65) #f #t 0+i '(1 2) _ht_ _undef_ 'a-symbol (cons 1 2) (make-vector 3) abs
       #<eof> '(1 2 3) #\newline (lambda (a) (+ a 1)) #<unspecified> #<undefined>))

(test (c-pointer?) 'error)
(test (c-object?) 'error)
(test (c-pointer? _c_obj_ 2 3) 'error)
(test (c-object? _c_obj_ 2) 'error)
(test (c-pointer 1 2 3 4 5 6) 'error)
(test (c-pointer? (openlet (inlet 'c-pointer? (lambda (p) #t)))) #t)
(test (c-pointer? (c-pointer 2 'integer?) 'integer?) #t)
(test (c-pointer? (c-pointer 2 'integer?) 'symbol?) #f)

(test (c-pointer->list (c-pointer 0)) '(0 #f #f))
(test (c-pointer->list (c-pointer 123 'null?)) '(123 null? #f))
(test (c-pointer-info (copy (c-pointer 1 2 3 4 5))) 3)
(test (c-pointer-weak1 (copy (c-pointer 1 2 3 4 5))) 4)
(test (c-pointer-weak2 (copy (c-pointer 1 2 3 4 5))) 5)

(test (equal? (c-pointer 1) 1) #f)
(test (equal? (c-pointer 1) (c-pointer 1)) #t)
(test (equal? (c-pointer 1) (c-pointer 0)) #f)
(test (equal? (c-pointer 1 (vector)) (c-pointer 1 (vector))) #t)
(test (equal? (c-pointer 1 (vector)) (c-pointer 1 (vector 0))) #f)
(test (equal? (c-pointer 1 vector? (vector)) (c-pointer 1 vector? (vector))) #t)
(test (equal? (c-pointer 1 vector? (vector)) (c-pointer 1 vector? (vector 0))) #f)
(test (equal? (c-pointer 1 +nan.0) (c-pointer 1 +nan.0)) #t) ; nan's are eq?
(test (equivalent? (c-pointer 1) 1) #f)
(test (equivalent? (c-pointer 1) (c-pointer 1)) #t)
(test (equivalent? (c-pointer 1) (c-pointer 0)) #f)
(test (equivalent? (c-pointer 1 (vector)) (c-pointer 1 (vector))) #t)
(test (equivalent? (c-pointer 1 (vector)) (c-pointer 1 (vector 0))) #f)
(test (equivalent? (c-pointer 1 vector? (vector)) (c-pointer 1 vector? (vector))) #t)
(test (equivalent? (c-pointer 1 vector? (vector)) (c-pointer 1 vector? (vector 0))) #f)
(test (equivalent? (c-pointer 1 +nan.0) (c-pointer 1 +nan.0)) #t)
(test (equivalent? (c-pointer 1 +nan.0) (c-pointer 1 +inf.0)) #f)
(test (equal? (copy (c-pointer 0 vector?)) (c-pointer 0 vector?)) #t)

(test (object->string (c-pointer 1 (vector)) :readable) "(c-pointer 1 #() #f)")
(test (object->string (c-pointer 1 (vector))) "#<c_pointer 0x1>") ; ??
(test (object->string (copy (c-pointer 1 2 3 4 5)) :readable) "(c-pointer 1 2 3)") ; see s7.html, same for equal? and equivalent?

(test (c-pointer-info) 'error)
(test (c-pointer-info #f) 'error)
(test (c-pointer-info (c-pointer 0)) #f)
(test (c-pointer-info (c-pointer 0) #f) 'error)
(test (c-pointer-info (c-pointer 0)) #f)
(test (c-pointer-info (c-pointer 0 1 2)) 2)

(test (c-pointer-type) 'error)
(test (c-pointer-type #f) 'error)
(test (c-pointer-type (c-pointer 0)) #f)
(test (c-pointer-type (c-pointer 0) #f) 'error)
(test (c-pointer-type (c-pointer 0)) #f)
(test (c-pointer-type (c-pointer 0 1 2)) 1)

(test (object->string (c-pointer 123123123 (symbol (make-string 130 #\a)))) "#<c_pointer 0x756b5b3>")

(test (type-of (c-pointer-weak1 (c-pointer 1))) 'boolean?)
(test (c-pointer-weak1 (copy (let ((<1> #f) (<2> (list #f (hash-table 'a 1 'c 3 'b 2) #f))) (set! <1> (c-pointer 1 <2> #f)) (set-car! <2> <1>) <1>))) #f)

(when with-bignums
  (test (c-pointer? (c-pointer (bignum "12341234"))) #t)
  (test (c-pointer (bignum "1.4")) 'error))

(let ((ptr (c-pointer 1 'abc (inlet 'object->string
				  (lambda (obj . args)
				    (let ((lt (object->let obj)))
				      (format #f "I am pointer ~A of type '~A!"
					      (lt 'pointer)
					      (lt 'c-type))))))))
  (openlet ptr)
  (test (object->string ptr) "I am pointer 1 of type 'abc!"))

(test (openlet (c-pointer #b101 (setter car) (rootlet))) 'error)
(test (abs (openlet (c-pointer 3 'asdf (inlet 'abs (lambda (val) 12))))) 12)
(test (with-let (c-pointer 3 'asdf (openlet (inlet 'abs (lambda (val) 12)))) (abs 32)) 12)
(test (with-let (c-pointer 3 'asdf (openlet (sublet (inlet) 'let-ref-fallback (lambda (lt sym) 12)))) asdf) 12)


;;; --------------------------------------------------------------------------------
;;; type-of

(test (type-of) 'error)
(test (type-of 1 2) 'error)
(test (type-of #f) 'boolean?)
(test (type-of ()) 'null?)
(test (type-of (list 1)) 'pair?)
(test (type-of 1) 'integer?)
(test (type-of 1/2) 'rational?)
(test (type-of 1.0) 'float?)
(test (type-of 1+i) 'complex?)
(test (type-of #<unspecified>) 'unspecified?)
(test (type-of (values)) 'unspecified?)
(test (type-of #<undefined>) 'undefined?)
(test (type-of _undef_) 'undefined?)
(test (type-of #<eof>) 'eof-object?)
(test (type-of (hash-table)) 'hash-table?)
(test (type-of (weak-hash-table)) 'hash-table?)
(test (type-of #(1)) 'vector?)
(test (type-of #r(1.0)) 'float-vector?)
(test (type-of #i(1)) 'int-vector?)
(test (type-of (byte-vector 1 2)) 'byte-vector?)
(test (type-of "") 'string?)
(test (type-of #\a) 'char?)
(test (type-of 'a) 'symbol?)
(test (type-of :a) 'symbol?)
(test (type-of (gensym)) 'symbol?)
(test (type-of (inlet 'a 1)) 'let?)
(test (type-of *stderr*) 'output-port?)
(test (type-of *stdin*) 'input-port?)
(test (type-of abs) 'procedure?)
(test (type-of +) 'procedure?)
(test (type-of (lambda () 1)) 'procedure?)
(test (type-of (lambda* ((a 1)) a)) 'procedure?)
(test (type-of quasiquote) 'macro?)
(test (type-of (define-bacro (_b_ x) `(+ ,x 1))) 'macro?) ; bacro? undefined
(test (type-of (c-pointer 0)) 'c-pointer?)
(test (type-of (random-state 123)) 'random-state?)
(test (type-of lambda) 'syntax?)
(test (type-of (call-with-exit (lambda (g) g))) 'goto?)
(test (type-of (call/cc (lambda (g) g))) 'continuation?)
(test (type-of (make-iterator '(1 2))) 'iterator?)
(when with-block
  (test (type-of (block)) 'c-object?))

(let ()
  (define (remove obj seq) ; remove-all?
    (case (type-of seq)

      ((pair?)
       (if (proper-list? seq)
	   (let loop ((lst seq) (res ()))
	     (if (null? lst)
		 (reverse! res)
		 (loop (cdr lst)
		       (if (equal? obj (car lst))
			   res
			   (cons (car lst) res)))))
	   seq))

      ((string? vector? float-vector? int-vector? byte-vector?)
       (let ((len (length seq)))
	 (if (zero? len)
	     seq
	     (do ((v (copy seq))
		  (j -1)
		  (i 0 (+ i 1)))
		 ((= i len)
		  (if (string? seq)
		      (copy v (make-string (+ j 1)))
		      (if (byte-vector? seq)
			  (copy v (make-byte-vector (+ j 1)))
			  (subvector v 0 (+ j 1)))))
	       (if (not (equal? obj (seq i)))
		   (set! (v (set! j (+ j 1))) (seq i)))))))

      ((hash-table?)
       (let ((ht (copy seq)))
	 (hash-table-set! ht obj #f)
	 ht))

      ((let?)
       (let ((lt (copy seq)))
	 (if (and (openlet? lt)
		  (let-ref lt 'remove))
	     ((let-ref lt 'remove) obj lt)
	     (cutlet lt obj))
	 lt))

      (else seq)))

  (test (remove #\a "abcdabcd") "bcdbcd")
  (test (remove 1 (vector 0 1 2 1 1)) #(0 2))
  (test (remove 1 (vector 1)) #())
  (test (remove 1 (byte-vector 0 1 2 1 1)) #u(0 2))
  (test (remove 1 (int-vector 1 2 1 3 4 1 5)) #i(2 3 4 5))
  (test (remove 1.0 (float-vector 1.0 3.0 1.0)) #r(3.0))
  (test (remove 1 '(1 2 3 4 1)) '(2 3 4))
  (test (remove 'a (hash-table 'a 1 'b 2)) (hash-table 'b 2))
  (test (remove 'a (inlet 'a 1 'b 2)) (inlet 'b 2))
  (let ((lt1 (remove 'a (openlet (inlet 'a 1 'b 2
					'remove (lambda (obj lt)
						  (set! (lt 'b) 3)
						  (cutlet lt obj)))))))
    (test (lt1 'b) 3))
  (test (remove 1 (vector)) #())

  (define (hi)
    (let ((v (byte-vector 1 2 3))
	  (sv (make-vector 3)))
      (do ((i 0 (+ i 1)))
	  ((= i 3) sv)
	(set! (sv i) (v i)))))
  (test (hi) #(1 2 3))

  (define (hi)
    (let ((v (byte-vector 1 2 3))
	  (sv (make-byte-vector 3)))
      (do ((i 0 (+ i 1)))
	  ((= i 3) sv)
	(set! (sv i) (v i)))))
  (test (hi) #u(1 2 3)))

(let ()
  (define (the type expr)
    (if (eq? type (symbol->value (type-of expr)))
	expr
	(error 'bad-type "~S is ~S but should be ~S in (the ~S ~S)" expr (type-of expr) type type expr)))
  (test (+ 1 (the integer? 3)) 4)
  (test (+ 1 (the integer? 3.0)) 'error))


;;; --------------------------------------------------------------------------------
;;; boolean?

(test (boolean? #f) #t)
(test (boolean? #t) #t)
(test (boolean? 0) #f)
(test (boolean? 1) #f)
(test (boolean? "") #f)
(test (boolean? #\0) #f)
(test (boolean? ()) #f)
(test (boolean? #()) #f)
(test (boolean? 't) #f)
(test (boolean? (list)) #f)
(test ( boolean? #t) #t)
(test (boolean? boolean?) #f)
(test (boolean? or) #f)
(test (   ; a comment
       boolean?  ;;; and another
       #t
       )
      #t)

(for-each
 (lambda (arg)
   (if (boolean? arg)
       (format #t ";(boolean? ~A) -> #t?~%" arg)))
 (list "hi" '(1 2) (integer->char 65) 1 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #\f (lambda (a) (+ a 1)) :hi (if #f #f) #<eof> #<undefined>))

(test (recompose 12 boolean? #f) #t)

(test (boolean?) 'error)
(test (boolean? #f #t) 'error)
(test (boolean #f) 'error)
(test (boolean? (lambda (x) #f)) #f)
(test (boolean? and) #f)
(test (boolean? if) #f)
(test (boolean? (values)) #f)
;(test (boolean? else) #f) ; this could also be an error -> unbound variable, like (symbol? else)




;;; --------------------------------------------------------------------------------
;;; not

(test (not #f) #t)
(test (not #t) #f)
(test (not (not #t)) #t)
(test (not 0) #f)
(test (not 1) #f)
(test (not ()) #f)
(test (not 't) #f)
(test (not (list)) #f)
(test (not (list 3)) #f)
(test (not 'nil) #f)
(test (not not) #f)
(test (not "") #f)
(test (not lambda) #f)
(test (not quote) #f)

(for-each
 (lambda (arg)
   (if (not arg)
       (format #t ";(not ~A) -> #t?~%" arg)))
 (list "hi" (integer->char 65) 1 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #\f (lambda (a) (+ a 1)) :hi #<eof> #<undefined> (if #f #f)))

(test (recompose 12 not #f) #f)

(test (not) 'error)
(test (not #f #t) 'error)
(test (not and) #f)
(test (not case) #f)

(let () ; check some optimizer branches
  (define (f1 sym) (not (symbol? sym))) (test (f1 'hi) #f) (test (f1 "hi") #t)
  (define (f2 sym) (not (integer? sym))) (test (f2 2) #f) (test (f2 'hi) #t)
  (define (f3 sym) (not (char? sym))) (test (f3 2) #t) (test (f3 #\a) #f)
  (define (f4 sym) (not (list? sym))) (test (f4 2) #t) (test (f4 '(1 2 3)) #f)
  (define (f5 sym) (not (boolean? sym))) (test (f5 2) #t) (test (f5 #f) #f)
  (define (f6 sym) (not (eof-object? sym))) (test (f6 2) #t) (test (f6 #<eof>) #f)
  (define (f7 sym) (not (pair? (car sym)))) (test (f7 '(hi)) #t) (test (f7 '((1))) #f)
  (define (f8 sym) (not (eq? sym 'q))) (test (f8 'a) #t) (test (f8 'q) #f)
  (define (f9 sym) (pair? (cadr sym))) (test (f9 '(1 2 3)) #f) (test (f9 '(1 (2 3) 4)) #t)
  (define (f10 lst val) (eq? (car lst) val)) (test (f10 '(#f) #f) #t) (test (f10 '(a) 32) #f)
  (define (f11 lst) (eq? (caar lst) 'q)) (test (f11 '((a))) #f) (test (f11 '((q))) #t)
  (define (f12 lst) (= (length lst) 2)) (test (f12 '(1 2)) #t) (test (f12 '(1 2 3)) #f)
  (define (f13 lst) (< (length lst) 2)) (test (f13 '(1 2)) #f) (test (f13 '(1)) #t)
  (define (f14 lst) (negative? (length lst))) (test (f14 '(1 2)) #f) (test (f14 '(1 . 3)) #t)
  (define (f15 lst) (memq (car lst) '(a b c))) (test (f15 '(a)) '(a b c)) (test (f15 '(d)) #f)
  (define (f16 a b) (if a (begin (+ b a) (format #f "~A" a) (+ a a)))) (test (f16 1 2) 2)
  (define (f17 a) (aritable? a 1)) (test (f17 abs) #t)
  (define (f22) (begin (display ":") (display (object->string 2)) (display ":"))) (test (with-output-to-string (lambda () (f22))) ":2:")
  (define (f23 a b) (list a b))
  (define (f24 x y) (f23 (car x) (car y)))
  (define (f25 x y) (f23 (cdr x) (cdr y)))
  (test (f24 '(1 2) '(3 4)) '(1 3)) (test (f25 '(1 2) '(3 4)) '((2) (4)))
  (define (f24a s1 s2 s3) (+ (* s1 s2) (* (- 1.0 s1) s3))) (test (f24a 2.0 3.0 4.0) 2.0)
  (let () (define (a b) (define c 1) (+ b c)) (define (tst) (a 2)) (tst) (test (tst) 3))
  (define (f25)
    (let ((x 0.0) (y 1.0))
      (call-with-exit
       (lambda (return)
	 (do ((i y (+ i 1))) ((= i 6))
	   (do ((i i (+ i 1))) ((>= i 7))
	     (set! x (+ x i))
	     (if (> x 123.0) (return x))))))
      x))
  (test (f25) 85.0)
  )
(let ()
  (test (let () (define (ho a) (+ a 2)) (define (hi) (+ (ho 1) (ho 2))) (hi)) 7)
  (test (let () (define (ho a) (+ a 2)) (define (hi) (+ (ho 1) (values 3 4))) (hi)) 10)
  (test (let () (define (ho a) (+ a 2)) (define (hi) (+ (values 3 4) (ho 1))) (hi)) 10)
  (test (let () (define (hi) (+ (values 1 2) (values 3 4))) (hi)) 10)
  (test (let () (define (ho a) (values a 1)) (define (hi) (- (ho 2))) (hi)) 1)
  (test (let () (define (ho1) (*s7* 'version)) (define (ho2) (ho1)) (string? (ho2))) #t)
  (test (let () (define (hi) (vector 0)) (define (ho) (hi)) (ho)) #(0)))
(let ()
  (define (make-it . names) (apply vector names))
  (define (hi) (make-it pi pi pi pi))
  (test (hi) (vector pi pi pi pi)))
(test (let () (define (hi a b c d) (+ a (* (- b c) d))) (define (ho) (hi 1 2 3 4)) (ho)) -3)
(test (let () (define (hi a b c d) (+ a (* d (- b c)))) (define (ho) (hi 1 2 3 4)) (ho)) -3)
(test (let () (define (hi) (let ((x (values 1 2))) (if x (list x)))) (define (ho) (hi)) (catch #t (lambda () (ho)) (lambda args #f)) (ho)) 'error) ; let_one_p_new
(test (let () (define (hi) (let ((x (values 1 2))) (display x) (if x (list x)))) (define (ho) (hi)) (catch #t (lambda () (ho)) (lambda args #f)) (ho)) 'error) ; let_one_new
(test (let () (define (hi a b) (- (+ a (abs b)))) (define (ho) (hi 1 -2)) (ho)) -3)

(let () (define (e1) (((lambda () list)) 'a 'b 'c)) (define (e2) (e1)) (e2) (test (e2) '(a b c)))
(let () (define (c1 s i) (case (string-ref s i) ((#\a) 1) (else 2))) (define (c3 s i) (c1 s i)) (c3 "hiho" 1) (test (c3 "hiho" 1) 2))
(let () (define (c1 s i) (case (string-ref s i) ((#\a) 1) ((#\i) 2))) (define (c3 s i) (c1 s i)) (c3 "hiho" 1) (test (c3 "hiho" 1) 2))
(let () (define (c1 s i) (case (string-ref s i) ((#\a #\h) 1) ((#\i #\o) 2))) (define (c3 s i) (c1 s i)) (c3 "hiho" 1) (test (c3 "hiho" 1) 2))
(let () (define (c1 s i) (case (string-ref s i) ((#\a #\h) 1) (else 2))) (define (c3 s i) (c1 s i)) (c3 "hiho" 1) (test (c3 "hiho" 1) 2))
(let () (define (d1) (do ((lst () (cons i lst)) (i 0 (+ i 1))) ((> i 6) (reverse lst)))) (define (d2) (d1)) (d2) (test (d2) '(0 1 2 3 4 5 6)))
(let () (define (d3) ((define (hi a) (+ a 1)) 2)) (define (d4) (d3)) (d4) (test (d4) 3))
(let () (define (fif) (if (< 2 3) (quote . -1))) (catch #t fif (lambda args 'error)) (test (catch #t fif (lambda args 'error)) 'error))
;(let () (define (fcond) (cond ((< 2 3) ((lambda (x) x 1 . 5) 2)))) (catch #t fcond (lambda args 'error)) (test (fcond) 'error))
;(let () (define (fcond1) (cond ((< 2 3) ((lambda* (x) x . 5) 2)))) (catch #t fcond1 (lambda args 'error)) (test (fcond1) 'error))
; those aren't what they appear to be: the catch does the stray dot check/error, then a call simply does what it can
(let () (define (incsaa k i) (let ((sum 1)) (set! sum (+ sum (expt k i) (expt (- k) i))) sum)) (define (f1) (incsaa 3 2)) (test (f1) 19))
(let () (define (unks v1 i) (let ((x 0)) (set! x (v1 i)) x)) (define (f1) (unks (vector 1 2 3) 2)) (test (f1) 3))

(test (let () (define (func) (catch #t (lambda () (when (not abs cond) #f)) (lambda args 'err))) (define (hi) (func) (func)) (hi) (hi)) 'err) ; set_if_opts


;;; --------------------------------------------------------------------------------
;;; symbol?

(test (symbol? 't) #t)
(test (symbol? "t") #f)
(test (symbol? '(t)) #f)
(test (symbol? #t) #f)
(test (symbol? 4) #f)
(test (symbol? 'foo) #t)
(test (symbol? (car '(a b))) #t)
(test (symbol? 'nil) #t)
(test (symbol? ()) #f)
(test (symbol? #()) #f)
(test (symbol? #f) #f)
(test (symbol? 'car) #t)
(test (symbol? car) #f)
(test (symbol? '#f) #f)
(test (symbol? #()) #f)
(test (symbol? :hi) #t)
(test (symbol? hi:) #t)
(test (symbol? :hi:) #t)
(test (symbol? '::) #t)
(test (symbol? ':) #t)
(test (symbol? '|) #t)
(test (symbol? '|') #t)
(test (symbol? '@) #t)
;(test (symbol? '#:) #t) ; confusable given guile-style keywords
(test (symbol? #b1) #f)
(test (symbol? 'sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789) #t) ;M Gran
(test (symbol? (vector-ref #(1 a 34) 1)) #t)
(test (if (symbol? '1+) (symbol? '0e) #t) #t)
(test (symbol? 'begin) #t)
(test (symbol? 'if) #t)
(test (symbol? (keyword->symbol :if)) #t)
(test (symbol? (string->symbol "if")) #t)
(test (symbol? if) #f)
(test (symbol? quote) #f)
(test (symbol? '(AB\c () xyz)) #f)

(for-each
 (lambda (arg)
   (if (symbol? arg)
       (format #t ";(symbol? ~A) -> #t?~%" arg)))
 (list "hi" (integer->char 65) 1 (list 1 2) '#t '3 (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #\f (lambda (a) (+ a 1)) #<eof> #<undefined>))

(test (symbol?) 'error)
(test (symbol? 'hi 'ho) 'error)
(test (symbol? 'hi 3) 'error)
(test (symbol? 3 3) 'error)
(test (symbol? 3 'hi) 'error)
(test (symbol 'hi) 'error) ; symbol takes a string

;;; "Returns #t if obj is a symbol, otherwise returns #f" (r5|6rs.html)
(test (symbol? begin) #f) ; this is an error in Guile, it was #t in s7
(test (symbol? expt) #f)
(test (symbol? if) #f)
(test (symbol? and) #f)
(test (symbol? lambda) #f)
(test (symbol? 'let) #t)
(test (symbol? call/cc) #f)
(test (symbol? '1.2.3) #t)
(test (symbol? '1.2) #f)
(test (symbol? ''1.2) #f)
(test (symbol? '"hi") #f)

(test (let ((sym000000000000000000000 3))
	(let ((sym000000000000000000001 4))
	  (+ sym000000000000000000000 sym000000000000000000001)))
      7)
(test (let ((11-1 10)
	    (2012-4-19 21)
	    (1+the-road 18)
	    (-1+2 1)
	    (1e. 2)
	    (0+i' 3)
	    (0.. 4))
	(+ 11-1 2012-4-19 1+the-road -1+2 1e. 0+i' 0..))
      59)

(test (let ((name "hiho"))
	(string-set! name 2 #\null)
	(symbol? (string->symbol name)))
      #t)


;;; syntax?
(test (syntax? 'lambda) #f)
(test (syntax? lambda) #t)
(test (syntax? if) #t)
(test (syntax? macroexpand) #t)
(test (syntax? 1) #f)
(for-each
 (lambda (arg)
   (if (syntax? arg)
       (format #t ";(syntax? ~A) -> #t?~%" arg)))
 (list "hi" (integer->char 65) (list 1 2) '#t '3 (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote 1/0 (log 0)
       3.14 3/4 1.0+1.0i #\f (lambda (a) (+ a 1)) #<eof> #<undefined>))
(test (syntax?) 'error)
(test (syntax? 'hi 'ho) 'error)
(test (syntax? 'hi 3) 'error)
(test (syntax? 3 3) 'error)
(test (syntax? 3 'hi) 'error)
(test (syntax? 'else) #f)
(test (syntax? '=>) #f)
(test (syntax? else) #f)
(let ()
  (define (syntactic x)
    (if x (case x ((1) 1) (else 2))))
  (syntactic 1)
  (let ((source (procedure-source syntactic)))
    (test (syntax? (car source)) #f))) ; 'lambda from (lambda (x) (if x (case x ((1) 1) (else 2))))


;;; --------------------------------------------------------------------------------
;;; procedure?

(test (procedure? car) #t)
(test (procedure? procedure?) #t)
(test (procedure? 'car) #f)
(test (procedure? (lambda (x) x)) #t)
(test (procedure? '(lambda (x) x)) #f)
(test (call/cc procedure?) #t)
(test (let ((a (lambda (x) x)))	(procedure? a)) #t)
(test (letrec ((a (lambda () (procedure? a)))) (a)) #t)
(test (let ((a 1)) (let ((a (lambda () (procedure? a)))) (a))) #f)
(test (let () (define (hi) 1) (procedure? hi)) #t)
(test (let () (define-macro (hi a) `(+ ,a 1)) (procedure? hi)) #f)
(test (procedure? begin) #f)
(test (procedure? lambda) #f)
(test (procedure? (lambda* ((a 1)) a)) #t)
(test (procedure? and) #f)
(test (procedure? 'let) #f)
(test (procedure? (dilambda (lambda () 1) (lambda (x) x))) #t)
(if with-bignums (test (procedure? (bignum "1e100")) #f))
(test (procedure? quasiquote) #f)
(let () (define-macro (hi a) `(+ ,a 1)) (test (procedure? hi) #f))
(test (procedure? (random-state 1234)) #f)
(test (procedure? pi) #f)
(test (procedure? cond) #f)
(test (procedure? do) #f)
(test (procedure? set!) #f)

(for-each
 (lambda (arg)
   (if (procedure? arg)
       (format #t ";(procedure? ~A) -> #t?~%" arg)))
 (list "hi" _ht_ _undef_ _null_ :hi (integer->char 65) 1 (list 1 2) '#t '3 (make-vector 3) 3.14 3/4 1.0+1.0i #\f #() (if #f #f)))

(test (procedure?) 'error)
(test (procedure? abs car) 'error)
(test (procedure abs) 'error)

;; these are questionable -- an applicable object is a procedure
(test (procedure? "hi") #f)
(test (procedure? '(1 2)) #f)
(test (procedure? #(1 2)) #f)





;;; --------------------------------------------------------------------------------
;;; CHARACTERS
;;; --------------------------------------------------------------------------------

(test (eqv? '#\  #\space) #t)
(test (eqv? #\newline '#\newline) #t)

;;; --------------------------------------------------------------------------------
;;; char?

(test (char? #\a) #t)
(test (char? #\() #t)
(test (char? #\space) #t)
(test (char? '#\newline) #t)
(test (char? #\1) #t)
(test (char? #\$) #t)
(test (char? #\.) #t)
(test (char? #\\) #t)
(test (char? #\)) #t)
(test (char? #\%) #t)
(test (char? '#\space) #t)
(test (char? '#\ ) #t)
(test (char? '#\newline) #t)
(test (char? '#\a) #t)
(test (char? '#\8) #t)
(test (char? #\-) #t)
(test (char? #\n) #t)
(test (char? #\() #t)
(test (char? #\#) #t)
(test (char? #\x) #t)
(test (char? #\o) #t)
(test (char? #\b) #t)
(test (char? #b101) #f)
(test (char? #o73) #f)
(test (char? #x73) #f)
(test (char? 'a) #f)
(test (char? 97) #f)
(test (char? "a") #f)
(test (char? (string-ref "hi" 0)) #t)
(test (char? (string-ref (make-string 1) 0)) #t)
(test (char? #\") #t)
(test (char? #\') #t)
(test (char? #\`) #t)
(test (char? #\@) #t)
(test (char? #<eof>) #f)
(test (char? '1e311) #f)

(for-each
 (lambda (arg)
   (if (char? arg)
       (format #t ";(char? ~A) -> #t?~%" arg)))
 (list "hi" () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #f #t (if #f #f) :hi (lambda (a) (+ a 1))))

(test (char? begin) #f)

(do ((i 0 (+ i 1)))
    ((= i 256))
  (if (not (char? (integer->char i)))
      (format #t ";(char? (integer->char ~A)) -> #f?~%" i)))

(test (char?) 'error)
(test (char? #\a #\b) 'error)
(test (char #\a) 'error)

(test (char? #\x65) #t)
(test (char? #\x000000000065) #t)
(test (char? #\x0) #t)
(test (char=? #\x000 #\null) #t)
(test (char=? #\x08 #\x8) #t)
(test (char=? #\x0e #\xe) #t) ; Guile thinks both of these names are bogus
(test (char=? #\x00e #\xe) #t)
(test (char=? #\x0000e #\xe) #t)
(test (char=? #\x00000000e #\xe) #t) ; hmmm -- surely this is a bug
(test (char? #\xff) #t)
;; any larger number is a reader error

;(test (eval-string "(char? #\xbdca2cbec)") 'error) ; needs 2 \\
(test (eval-string "(char? #\\xbdca2cbec)") #f)
(test (eval-string "(char? #\\100)") #f)
(test (eval-string "(char? #\\x-65)") #f)
(test (eval-string "(char? #\\x6.5)") #f)
(test (eval-string "(char? #\\x6/5)") #f)
(test (eval-string "(char? #\\x6/3)") #f)
(test (eval-string "(char? #\\x6+i)") #f)
(test (eval-string "(char? #\\x6asd)") #f)
(test (eval-string "(char? #\\x6#)") #f)
(test (eval-string "(char? #\\x#b0)") #f)
(test (eval-string "(char? #\\x#b0") 'error) ; missing )
(test (eval-string "(char? #\\x-0)") #f)
(test (eval-string "(char? #\\x1.4)") #f)
(test (eval-string "(char? #\\x#b0)") #f)
(test (eval-string "(char? #\\x-0)") #f)
(test (eval-string "(char? #\\x1.4)") #f)

(test (char=? #\x6a #\j) #t)

(test (char? #\return) #t)
(test (char? #\null) #t)
(test (char? #\nul) #t)
(test (char? #\linefeed) #t)
(test (char? #\tab) #t)
(test (char? #\space) #t)
(test (char=? #\null #\nul) #t)
(test (char=? #\newline #\linefeed) #t)
(test (char=? #\return #\xd) #t)
(test (char=? #\nul #\x0) #t)
;(test (char? #\ÿ) #t) ; this seems to involve unwanted translations in emacs?
(test (eval-string (string-append "(char? " (format #f "#\\~C" (integer->char 255)) ")")) #t)
(test (eval-string (string-append "(char? " (format #f "#\\~C" (integer->char 127)) ")")) #t)
(test (apply char? (list (integer->char 255))) #t)

(test (char? #\escape) #t)
(test (char? #\alarm) #t)
(test (char? #\backspace) #t)
(test (char? #\delete) #t)
(test (char=? #\delete #\backspace) #f)

(num-test (let ((str (make-string 258 #\space)))
	    (do ((i 1 (+ i 1)))
		((= i 256))
	      (string-set! str i (integer->char i)))
	    (string-set! str 257 (integer->char 0))
	    (string-length str))
	  258)


(let ((a-to-z (list #\a #\b #\c #\d #\e #\f #\g #\h #\i #\j #\k #\l #\m #\n #\o #\p #\q #\r #\s #\t #\u #\v #\x #\y #\z))
      (cap-a-to-z (list #\A #\B #\C #\D #\E #\F #\G #\H #\I #\J #\K #\L #\M #\N #\O #\P #\Q #\R #\S #\T #\U #\V #\X #\Y #\Z))
      (mixed-a-to-z (list #\a #\B #\c #\D #\e #\F #\g #\H #\I #\j #\K #\L #\m #\n #\O #\p #\Q #\R #\s #\t #\U #\v #\X #\y #\Z))
      (digits (list #\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9)))

;;; --------------------------------------------------------------------------------
;;; char-upper-case?

  (test (char-upper-case? #\a) #f)
  (test (char-upper-case? #\A) #t)

  (for-each
   (lambda (arg)
     (if (not (char-upper-case? arg))
	 (format #t ";(char-upper-case? ~A) -> #f?~%" arg)))
   cap-a-to-z)

  (for-each
   (lambda (arg)
     (if (char-upper-case? arg)
	 (format #t ";(char-upper-case? ~A) -> #t?~%" arg)))
   a-to-z)

  (test (char-upper-case? (integer->char 192)) #t) ; 192..208 for unicode
  ;; non-alpha chars are "unspecified" here

  (test (char-upper-case? 1) 'error)
  (test (char-upper-case?) 'error)
  (test (char-upper-case? 1) 'error)
  (test (char-upper-case?) 'error)
  (test (char-upper-case? #\a #\b) 'error)
  (test (char-upper-case #\a) 'error)



;;; --------------------------------------------------------------------------------
;;; char-lower-case?

  (test (char-lower-case? #\A) #f)
  (test (char-lower-case? #\a) #t)

  (for-each
   (lambda (arg)
     (if (not (char-lower-case? arg))
	 (format #t ";(char-lower-case? ~A) -> #f?~%" arg)))
   a-to-z)

  (for-each
   (lambda (arg)
     (if (char-lower-case? arg)
	 (format #t ";(char-lower-case? ~A) -> #t?~%" arg)))
   cap-a-to-z)

  (test (char-lower-case? 1) 'error)
  (test (char-lower-case?) 'error)
  (test (char-lower-case? 1) 'error)
  (test (char-lower-case?) 'error)
  (test (char-lower-case? #\a #\b) 'error)
  (test (char-lower-case #\a) 'error)

;;  (test (char-lower-case? #\xb5) #t)  ; what is this?  in Snd it's #t, in ex1 it's #f -- is this a locale choice?
  (test (char-lower-case? #\xb6) #f)

  (for-each
   (lambda (c)
     (test (and (not (char-upper-case? c))
		(not (char-lower-case? c))) #t))
   (map integer->char (list 0 1 2 3 32 33 34 170 182 247)))



;;; --------------------------------------------------------------------------------
;;; char-upcase

  (test (char-upcase #\A) #\A)
  (test (char-upcase #\a) #\A)
  (test (char-upcase #\?) #\?)
  (test (char-upcase #\$) #\$)
  (test (char-upcase #\.) #\.)
  (test (char-upcase #\\) #\\)
  (test (char-upcase #\5) #\5)
  (test (char-upcase #\)) #\))
  (test (char-upcase #\%) #\%)
  (test (char-upcase #\0) #\0)
  (test (char-upcase #\_) #\_)
  (test (char-upcase #\?) #\?)
  (test (char-upcase #\space) #\space)
  (test (char-upcase #\newline) #\newline)
  (test (char-upcase #\null) #\null)
  (test (char-upper-case? (char-upcase #\?)) #f) ; !
  (test (char-lower-case? (char-downcase #\?)) #f)
  (test (char-upper-case? (char-upcase #\_)) #f)
  (test (or (char-upper-case? #\?) (char-lower-case? #\?)) #f)

  (for-each
   (lambda (arg1 arg2)
     (if (not (char=? (char-upcase arg1) arg2))
	 (format #t ";(char-upcase ~A) != ~A?~%" arg1 arg2)))
   a-to-z
   cap-a-to-z)

  (do ((i 1 (+ i 1)))
      ((= i 256))
    (if (and (not (char=? (integer->char i) (char-upcase (integer->char i))))
	     (not (char-alphabetic? (integer->char i))))
	(format #t ";(char-upcase ~A) -> ~A but not alphabetic?~%" (integer->char i) (char-upcase (integer->char i)))))

  (test (recompose 12 char-upcase #\a) #\A)
  (test (reinvert 12 char-upcase char-downcase #\a) #\a)

  (test (char-upcase) 'error)
  (test (char-upcase #\a #\b) 'error)
  (test (char-upcase #<eof>) 'error)
  (test (char-upcase #f) 'error)
  (test (char-upcase (list)) 'error)



;;; --------------------------------------------------------------------------------
;;; char-downcase

  (test (char-downcase #\A) #\a)
  (test (char-downcase #\a) #\a)
  (test (char-downcase #\?) #\?)
  (test (char-downcase #\$) #\$)
  (test (char-downcase #\.) #\.)
  (test (char-downcase #\_) #\_)
  (test (char-downcase #\\) #\\)
  (test (char-downcase #\5) #\5)
  (test (char-downcase #\)) #\))
  (test (char-downcase #\%) #\%)
  (test (char-downcase #\0) #\0)
  (test (char-downcase #\space) #\space)

  (for-each
   (lambda (arg1 arg2)
     (if (not (char=? (char-downcase arg1) arg2))
	 (format #t ";(char-downcase ~A) != ~A?~%" arg1 arg2)))
   cap-a-to-z
   a-to-z)

  (test (recompose 12 char-downcase #\A) #\a)

  (test (char-downcase) 'error)
  (test (char-downcase #\a #\b) 'error)


;;; --------------------------------------------------------------------------------
;;; char-numeric?

  (test (char-numeric? #\a) #f)
  (test (char-numeric? #\5) #t)
  (test (char-numeric? #\A) #f)
  (test (char-numeric? #\z) #f)
  (test (char-numeric? #\Z) #f)
  (test (char-numeric? #\0) #t)
  (test (char-numeric? #\9) #t)
  (test (char-numeric? #\space) #f)
  (test (char-numeric? #\;) #f)
  (test (char-numeric? #\.) #f)
  (test (char-numeric? #\-) #f)
  (test (char-numeric? (integer->char 200)) #f)
  (test (char-numeric? (integer->char 128)) #f)
  (test (char-numeric? (integer->char 216)) #f) ; 0 slash
  (test (char-numeric? (integer->char 189)) #f) ; 1/2

  (for-each
   (lambda (arg)
     (if (char-numeric? arg)
	 (format #t ";(char-numeric? ~A) -> #t?~%" arg)))
   cap-a-to-z)

  (for-each
   (lambda (arg)
     (if (char-numeric? arg)
	 (format #t ";(char-numeric? ~A) -> #t?~%" arg)))
   a-to-z)

  (test (char-numeric?) 'error)
  (test (char-numeric? #\a #\b) 'error)


;;; --------------------------------------------------------------------------------
;;; char-whitespace?

  (test (char-whitespace? #\a) #f)
  (test (char-whitespace? #\A) #f)
  (test (char-whitespace? #\z) #f)
  (test (char-whitespace? #\Z) #f)
  (test (char-whitespace? #\0) #f)
  (test (char-whitespace? #\9) #f)
  (test (char-whitespace? #\space) #t)
  (test (char-whitespace? #\tab) #t)
  (test (char-whitespace? #\newline) #t)
  (test (char-whitespace? #\return) #t)
  (test (char-whitespace? #\linefeed) #t)
  (test (char-whitespace? #\null) #f)
  (test (char-whitespace? #\;) #f)
  (test (char-whitespace? #\xb) #t)
  (test (char-whitespace? #\x0b) #t)
  (test (char-whitespace? #\xc) #t)
  (test (char-whitespace? #\xd) #t) ; #\return
  (test (char-whitespace? #\xe) #f)

  ;; unicode whitespace apparently:
  (test (char-whitespace? (integer->char 9)) #t)
  (test (char-whitespace? (integer->char 10)) #t)
  (test (char-whitespace? (integer->char 11)) #t)
  (test (char-whitespace? (integer->char 12)) #t)
  (test (char-whitespace? (integer->char 13)) #t)
  (test (char-whitespace? (integer->char 32)) #t)
  (test (char-whitespace? (integer->char 133)) #t)
  (test (char-whitespace? (integer->char 160)) #t)

  (for-each
   (lambda (arg)
     (if (char-whitespace? arg)
	 (format #t ";(char-whitespace? ~A) -> #t?~%" arg)))
   mixed-a-to-z)

  (for-each
   (lambda (arg)
     (if (char-whitespace? arg)
	 (format #t ";(char-whitespace? ~A) -> #t?~%" arg)))
   digits)

  (test (char-whitespace?) 'error)
  (test (char-whitespace? #\a #\b) 'error)


;;; --------------------------------------------------------------------------------
;;; char-alphabetic?

  (test (char-alphabetic? #\a) #t)
  (test (char-alphabetic? #\$) #f)
  (test (char-alphabetic? #\A) #t)
  (test (char-alphabetic? #\z) #t)
  (test (char-alphabetic? #\Z) #t)
  (test (char-alphabetic? #\0) #f)
  (test (char-alphabetic? #\9) #f)
  (test (char-alphabetic? #\space) #f)
  (test (char-alphabetic? #\;) #f)
  (test (char-alphabetic? #\.) #f)
  (test (char-alphabetic? #\-) #f)
  (test (char-alphabetic? #\_) #f)
  (test (char-alphabetic? #\^) #f)
  (test (char-alphabetic? #\[) #f)

  ;(test (char-alphabetic? (integer->char 200)) #t) ; ??
  (test (char-alphabetic? (integer->char 127)) #f)  ; backspace

  (for-each
   (lambda (arg)
     (if (char-alphabetic? arg)
	 (format #t ";(char-alphabetic? ~A) -> #t?~%" arg)))
   digits)

  (for-each
   (lambda (arg)
     (if (not (char-alphabetic? arg))
	 (format #t ";(char-alphabetic? ~A) -> #f?~%" arg)))
   mixed-a-to-z)

  (test (char-alphabetic?) 'error)
  (test (char-alphabetic? #\a #\b) 'error)

  (for-each
   (lambda (op)
     (for-each
      (lambda (arg)
	(test (op arg) 'error))
      (list "hi" () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
	    3.14 3/4 1.0+1.0i #f #t :hi (if #f #f) (lambda (a) (+ a 1)))))
   (list char-upper-case? char-lower-case? char-upcase char-downcase char-numeric? char-whitespace? char-alphabetic?))

  (test
   (let ((unhappy ()))
     (do ((i 0 (+ i 1)))
	 ((= i 256))
       (let* ((ch (integer->char i))
	      (chu (char-upcase ch))
	      (chd (char-downcase ch)))

	 (if (and (not (char=? ch chu))
		  (not (char-upper-case? chu)))
	     (format #t ";(char-upper-case? (char-upcase ~C)) is #f~%" ch))

	 (if (and (not (char=? ch chd))
		  (not (char-lower-case? chd)))
	     (format #t ";(char-lower-case? (char-downcase ~C)) is #f~%" ch))

	 (if (or (and (not (char=? ch chu))
		      (not (char=? ch (char-downcase chu))))
		 (and (not (char=? ch chd))
		      (not (char=? ch (char-upcase chd))))
		 (and (not (char=? ch chd))
		      (not (char=? ch chu)))
		 (not (char-ci=? chu chd))
		 (not (char-ci=? ch chu))
		 (and (char-alphabetic? ch)
		      (or (not (char-alphabetic? chd))
			  (not (char-alphabetic? chu))))
		 (and (char-numeric? ch)
		      (or (not (char-numeric? chd))
			  (not (char-numeric? chu))))
		 (and (char-whitespace? ch)
		      (or (not (char-whitespace? chd))
			  (not (char-whitespace? chu))))
		 (and (char-alphabetic? ch)
		      (char-whitespace? ch))
		 (and (char-numeric? ch)
		      (char-whitespace? ch))
		 (and (char-alphabetic? ch)
		      (char-numeric? ch)))
	     ;; there are characters that are alphabetic but the result of char-upcase is not an upper-case character
	     ;; 223 for example, or 186 for lower case
	     (set! unhappy (cons (format #f "~C: ~C ~C (~D)~%" ch chu chd i) unhappy)))))
     unhappy)
   ())

  (for-each
   (lambda (op)
     (for-each
      (lambda (arg)
	(test (op #\a arg) 'error))
      (list "hi" () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
	    3.14 3/4 1.0+1.0i #f #t :hi (if #f #f) (lambda (a) (+ a 1)))))
   (list char=? char<? char<=? char>? char>=? char-ci=? char-ci<? char-ci<=? char-ci>? char-ci>=?))

  (for-each
   (lambda (op)
     (for-each
      (lambda (arg)
	(test (op arg #\a) 'error))
      (list "hi" () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
	    3.14 3/4 1.0+1.0i #f #t :hi (if #f #f) (lambda (a) (+ a 1)))))
   (list char=? char<? char<=? char>? char>=? char-ci=? char-ci<? char-ci<=? char-ci>? char-ci>=?))

  (let () ; check direct opts
    (define (fc str)
      (let ((len (length str))
	    (count 0))
	(do ((i 0 (+ i 1)))
	    ((= i len) count)
	  (if (char-whitespace? (string-ref str i))
	      (set! count (+ count 1)))
	  (if (char-alphabetic? (string-ref str i))
	      (set! count (+ count 1)))
	  (if (char-numeric? (string-ref str i))
	      (set! count (+ count 1)))
	  (if (char-lower-case? (string-ref str i))
	      (set! count (+ count 1)))
	  (if (char-upper-case? (string-ref str i))
	      (set! count (+ count 1))))))
    (test (fc "123 4Ab.2") 10))


;;; --------------------------------------------------------------------------------
;;; char=?

  (test (char=? #\d #\d) #t)
  (test (char=? #\A #\a) #f)
  (test (char=? #\d #\x) #f)
  (test (char=? #\d #\D) #f)
  (test (char=? #\a #\a) #t)
  (test (char=? #\A #\B) #f)
  (test (char=? #\a #\b) #f)
  (test (char=? #\9 #\0) #f)
  (test (char=? #\A #\A) #t)
  (test (char=? #\  #\space) #t)
  (let ((i (char->integer #\space)))
    (test (char=? (integer->char i) #\space) #t))
  (test (char=? (integer->char (char->integer #\")) #\") #t)
  (test (char=? #\x65 #\e) #t)

  (test (char=? #\d #\d #\d #\d) #t)
  (test (char=? #\d #\d #\x #\d) #f)
  (test (char=? #\d #\y #\x #\c) #f)
  (test (apply char=? cap-a-to-z) #f)
  (test (apply char=? mixed-a-to-z) #f)
  (test (apply char=? digits) #f)
  (test (char=? #\d #\c #\d) #f)

  (test (char=? #\a) 'error)
  (test (char=?) 'error)
  (test (char=? #\a 0) 'error)
  (test (char=? #\a #\b 0) 'error)
  (test (char=? 90 (integer->char 90)) 'error)
  (test (char=? 90 #\Z) 'error)
  (test (char=? #\Z 90) 'error)
  (test (char=? 1 1) 'error)


;;; --------------------------------------------------------------------------------
;;; char<?

  (test (char<? #\z #\0) #f)
  (test (char<? #\d #\x) #t)
  (test (char<? #\d #\d) #f)
  (test (char<? #\d #\x) #t)
  (test (char<? #\A #\B) #t)
  (test (char<? #\a #\b) #t)
  (test (char<? #\9 #\0) #f)
  (test (char<? #\A #\A) #f)
  (test (char<? #\space #\space) #f)

  (test (char<? #\a #\e #\y #\z) #t)
  (test (char<? #\a #\e #\e #\y) #f)
  (test (apply char<? a-to-z) #t)
  (test (apply char<? cap-a-to-z) #t)
  (test (apply char<? mixed-a-to-z) #f)
  (test (apply char<? digits) #t)
  (test (apply char<? (reverse a-to-z)) #f)
  (test (apply char<? (reverse cap-a-to-z)) #f)
  (test (apply char<? (reverse mixed-a-to-z)) #f)
  (test (apply char<? (reverse digits)) #f)
  (test (char<? #\b #\c #\a) #f)
  (test (char<? #\B #\B #\A) #f)
  (test (char<? #\b #\c #\e) #t)
  (test (char<? (integer->char #xf0) (integer->char #x70)) #f)

  (test (char<?) 'error)
  (test (char<? #\b #\a "hi") 'error)
  (test (char<? #\b #\a 0) 'error)
  (test (char<? (integer->char 0) (integer->char 255)) #t)
  (test (char<? 90 (integer->char 90)) 'error)
  (test (char<? 90 #\Z) 'error)
  (test (char<? #\Z 90) 'error)



;;; --------------------------------------------------------------------------------
;;; char<=?

  (test (char<=? #\d #\x) #t)
  (test (char<=? #\d #\d) #t)

  (test (char<=? #\a #\e #\y #\z) #t)
  (test (char<=? #\a #\e #\e #\y) #t)
  (test (char<=? #\A #\B) #t)
  (test (char<=? #\a #\b) #t)
  (test (char<=? #\9 #\0) #f)
  (test (char<=? #\A #\A) #t)
  (test (char<=? #\space #\space) #t)

  (test (char<=? #\a #\e #\y #\z) #t)
  (test (char<=? #\a #\e #\e #\y) #t)
  (test (char<=? #\e #\e #\d #\y) #f)
  (test (apply char<=? a-to-z) #t)
  (test (apply char<=? cap-a-to-z) #t)
  (test (apply char<=? mixed-a-to-z) #f)
  (test (apply char<=? digits) #t)
  (test (apply char<=? (reverse a-to-z)) #f)
  (test (apply char<=? (reverse cap-a-to-z)) #f)
  (test (apply char<=? (reverse mixed-a-to-z)) #f)
  (test (apply char<=? (reverse digits)) #f)
  (test (char<=? #\b #\c #\a) #f)
  (test (char<=? #\B #\B #\A) #f)
  (test (char<=? #\b #\c #\e) #t)

  (test (char<=? #\b #\a "hi") 'error)
  (test (char<=? #\b #\a 0) 'error)
  (test (char<=?) 'error)
  (test (char<=? 90 (integer->char 90)) 'error)
  (test (char<=? 90 #\Z) 'error)
  (test (char<=? #\Z 90) 'error)



;;; --------------------------------------------------------------------------------
;;; char>?

  (test (char>? #\e #\d) #t)
  (test (char>? #\z #\a) #t)
  (test (char>? #\A #\B) #f)
  (test (char>? #\a #\b) #f)
  (test (char>? #\9 #\0) #t)
  (test (char>? #\A #\A) #f)
  (test (char>? #\space #\space) #f)

  (test (char>? #\d #\c #\b #\a) #t)
  (test (char>? #\d #\d #\c #\a) #f)
  (test (char>? #\e #\d #\b #\c #\a) #f)
  (test (apply char>? a-to-z) #f)
  (test (apply char>? cap-a-to-z) #f)
  (test (apply char>? mixed-a-to-z) #f)
  (test (apply char>? digits) #f)
  (test (apply char>? (reverse a-to-z)) #t)
  (test (apply char>? (reverse cap-a-to-z)) #t)
  (test (apply char>? (reverse mixed-a-to-z)) #f)
  (test (apply char>? (reverse digits)) #t)
  (test (char>? #\d #\c #\a) #t)
  (test (char>? #\d #\c #\c) #f)
  (test (char>? #\B #\B #\C) #f)
  (test (char>? #\b #\c #\e) #f)
  (test (char>? (integer->char #xf0) (integer->char #x70)) #t)

  (test (char>? #\a #\b "hi") 'error)
  (test (char>? #\a #\b 0) 'error)
  (test (char>?) 'error)
  (test (char>? 90 (integer->char 90)) 'error)
  (test (char>? 90 #\Z) 'error)
  (test (char>? #\Z 90) 'error)



;;; --------------------------------------------------------------------------------
;;; char>=?

  (test (char>=? #\e #\d) #t)
  (test (char>=? #\A #\B) #f)
  (test (char>=? #\a #\b) #f)
  (test (char>=? #\9 #\0) #t)
  (test (char>=? #\A #\A) #t)
  (test (char>=? #\space #\space) #t)

  (test (char>=? #\d #\c #\b #\a) #t)
  (test (char>=? #\d #\d #\c #\a) #t)
  (test (char>=? #\e #\d #\b #\c #\a) #f)
  (test (apply char>=? a-to-z) #f)
  (test (apply char>=? cap-a-to-z) #f)
  (test (apply char>=? mixed-a-to-z) #f)
  (test (apply char>=? digits) #f)
  (test (apply char>=? (reverse a-to-z)) #t)
  (test (apply char>=? (reverse cap-a-to-z)) #t)
  (test (apply char>=? (reverse mixed-a-to-z)) #f)
  (test (apply char>=? (reverse digits)) #t)
  (test (char>=? #\d #\c #\a) #t)
  (test (char>=? #\d #\c #\c) #t)
  (test (char>=? #\B #\B #\C) #f)
  (test (char>=? #\b #\c #\e) #f)

  (test (char>=? #\a #\b "hi") 'error)
  (test (char>=? #\a #\b 0) 'error)
  (test (char>=?) 'error)
  (test (char>=? 90 (integer->char 90)) 'error)
  (test (char>=? 90 #\Z) 'error)
  (test (char>=? #\Z 90) 'error)



;;; --------------------------------------------------------------------------------
;;; char-ci=?

  (test (char-ci=? #\A #\B) #f)
  (test (char-ci=? #\a #\B) #f)
  (test (char-ci=? #\A #\b) #f)
  (test (char-ci=? #\a #\b) #f)
  (test (char-ci=? #\9 #\0) #f)
  (test (char-ci=? #\A #\A) #t)
  (test (char-ci=? #\A #\a) #t)
  (test (char-ci=? #\a #\A) #t)
  (test (char-ci=? #\space #\space) #t)

  (test (char-ci=? #\d #\D #\d #\d) #t)
  (test (char-ci=? #\d #\d #\X #\d) #f)
  (test (char-ci=? #\d #\Y #\x #\c) #f)
  (test (apply char-ci=? cap-a-to-z) #f)
  (test (apply char-ci=? mixed-a-to-z) #f)
  (test (apply char-ci=? digits) #f)
  (test (char-ci=? #\d #\c #\d) #f)

  (test (char-ci=?) 'error)
  (test (char-ci=? #\a #\b 0) 'error)



;;; --------------------------------------------------------------------------------
;;; char-ci<?

  (test (char-ci<? #\A #\B) #t)
  (test (char-ci<? #\a #\B) #t)
  (test (char-ci<? #\A #\b) #t)
  (test (char-ci<? #\a #\b) #t)
  (test (char-ci<? #\9 #\0) #f)
  (test (char-ci<? #\0 #\9) #t)
  (test (char-ci<? #\A #\A) #f)
  (test (char-ci<? #\A #\a) #f)
  (test (char-ci<? #\Y #\_) #t)
  (test (char-ci<? #\\ #\J) #f)
  (test (char-ci<? #\_ #\e) #f)
  (test (char-ci<? #\t #\_) #t)
  (test (char-ci<? #\a #\]) #t)
  (test (char-ci<? #\z #\^) #t)

  (test (char-ci<? #\b #\a "hi") 'error)
  (test (char-ci<? #\b #\a 0) 'error)
  (test (char-ci>? (integer->char #xf0) (integer->char #x70)) #t)

#|
    ;; this tries them all:
    (do ((i 0 (+ i 1)))
	((= i 256))
      (do ((k 0 (+ k 1)))
	  ((= k 256))
	(let ((c1 (integer->char i))
	      (c2 (integer->char k)))
	  (for-each
	   (lambda (op1 op2)
	     (if (not (eq? (op1 c1 c2) (op2 (string c1) (string c2))))
		 (format #t ";(~A|~A ~A ~A) -> ~A|~A~%" op1 op2 c1 c2 (op1 c1 c2) (op2 (string c1) (string c2)))))
	   (list char=? char<? char<=? char>? char>=? char-ci=? char-ci<? char-ci<=? char-ci>? char-ci>=?)
	   (list string=? string<? string<=? string>? string>=? string-ci=? string-ci<? string-ci<=? string-ci>? string-ci>=?)))))
|#

  (test (char-ci<? #\d #\D #\d #\d) #f)
  (test (char-ci<? #\d #\d #\X #\d) #f)
  (test (char-ci<? #\d #\Y #\x #\c) #f)
  (test (apply char-ci<? cap-a-to-z) #t)
  (test (apply char-ci<? mixed-a-to-z) #t)
  (test (apply char-ci<? digits) #t)
  (test (char-ci<? #\d #\c #\d) #f)
  (test (char-ci<? #\b #\c #\a) #f)
  (test (char-ci<? #\b #\C #\e) #t)
  (test (char-ci<? #\3 #\? #\Z #\[) #t)

  (test (char-ci>? #\a #\b "hi") 'error)
  (test (char-ci>? #\a #\b 0) 'error)



;;; --------------------------------------------------------------------------------
;;; char-ci>?

  (test (char-ci>? #\A #\B) #f)
  (test (char-ci>? #\a #\B) #f)
  (test (char-ci>? #\A #\b) #f)
  (test (char-ci>? #\a #\b) #f)
  (test (char-ci>? #\9 #\0) #t)
  (test (char-ci>? #\A #\A) #f)
  (test (char-ci>? #\A #\a) #f)
  (test (char-ci>? #\^ #\a) #t)
  (test (char-ci>? #\_ #\e) #t)
  (test (char-ci>? #\[ #\S) #t)
  (test (char-ci>? #\\ #\l) #t)
  (test (char-ci>? #\t #\_) #f)
  (test (char-ci>? #\a #\]) #f)
  (test (char-ci>? #\z #\^) #f)
  (test (char-ci>? #\] #\X) #t)

  (test (char-ci>? #\d #\D #\d #\d) #f)
  (test (char-ci>? #\d #\d #\X #\d) #f)
  (test (char-ci>? #\d #\Y #\x #\c) #f)
  (test (apply char-ci>? cap-a-to-z) #f)
  (test (apply char-ci>? mixed-a-to-z) #f)
  (test (apply char-ci>? (reverse mixed-a-to-z)) #t)
  (test (apply char-ci>? digits) #f)
  (test (char-ci>? #\d #\c #\d) #f)
  (test (char-ci>? #\b #\c #\a) #f)
  (test (char-ci>? #\d #\C #\a) #t)


;;; --------------------------------------------------------------------------------
;;; char-ci<=?

  (test (char-ci<=? #\A #\B) #t)
  (test (char-ci<=? #\a #\B) #t)
  (test (char-ci<=? #\A #\b) #t)
  (test (char-ci<=? #\a #\b) #t)
  (test (char-ci<=? #\9 #\0) #f)
  (test (char-ci<=? #\A #\A) #t)
  (test (char-ci<=? #\A #\a) #t)
  (test (char-ci<=? #\` #\H) #f)
  (test (char-ci<=? #\[ #\m) #f)
  (test (char-ci<=? #\j #\`) #t)
  (test (char-ci<=? #\\ #\E) #f)
  (test (char-ci<=? #\t #\_) #t)
  (test (char-ci<=? #\a #\]) #t)
  (test (char-ci<=? #\z #\^) #t)

  (test (char-ci<=? #\d #\D #\d #\d) #t)
  (test (char-ci<=? #\d #\d #\X #\d) #f)
  (test (char-ci<=? #\d #\Y #\x #\c) #f)
  (test (apply char-ci<=? cap-a-to-z) #t)
  (test (apply char-ci<=? mixed-a-to-z) #t)
  (test (apply char-ci<=? digits) #t)
  (test (char-ci<=? #\d #\c #\d) #f)
  (test (char-ci<=? #\b #\c #\a) #f)
  (test (char-ci<=? #\b #\c #\C) #t)
  (test (char-ci<=? #\b #\C #\e) #t)

  (test (char-ci<=? #\b #\a "hi") 'error)
  (test (char-ci<=? #\b #\a 0) 'error)



;;; --------------------------------------------------------------------------------
;;; char-ci>=?

  (test (char-ci>=? #\A #\B) #f)
  (test (char-ci>=? #\a #\B) #f)
  (test (char-ci>=? #\A #\b) #f)
  (test (char-ci>=? #\a #\b) #f)
  (test (char-ci>=? #\9 #\0) #t)
  (test (char-ci>=? #\A #\A) #t)
  (test (char-ci>=? #\A #\a) #t)
  (test (char-ci>=? #\Y #\_) #f)
  (test (char-ci>=? #\` #\S) #t)
  (test (char-ci>=? #\[ #\Y) #t)
  (test (char-ci>=? #\t #\_) #f)
  (test (char-ci>=? #\a #\]) #f)
  (test (char-ci>=? #\z #\^) #f)

  (test (char-ci>=? #\d #\D #\d #\d) #t)
  (test (char-ci>=? #\d #\d #\X #\d) #f)
  (test (char-ci>=? #\d #\Y #\x #\c) #f)
  (test (apply char-ci>=? cap-a-to-z) #f)
  (test (apply char-ci>=? mixed-a-to-z) #f)
  (test (apply char-ci>=? (reverse mixed-a-to-z)) #t)
  (test (apply char-ci>=? (reverse mixed-a-to-z)) #t)
  (test (apply char-ci>=? digits) #f)
  (test (char-ci>=? #\d #\c #\d) #f)
  (test (char-ci>=? #\b #\c #\a) #f)
  (test (char-ci>=? #\d #\D #\a) #t)
  (test (char-ci>=? #\\ #\J #\+) #t)

  (test (char-ci>=? #\a #\b "hi") 'error)
  (test (char-ci>=? #\a #\b 0) 'error)

  ) ; end let with a-to-z



;;; --------------------------------------------------------------------------------
;;; integer->char
;;; char->integer

(test (integer->char (char->integer #\.)) #\.)
(test (integer->char (char->integer #\A)) #\A)
(test (integer->char (char->integer #\a)) #\a)
(test (integer->char (char->integer #\space)) #\space)
(test (char->integer (integer->char #xf0)) #xf0)

(do ((i 0 (+ i 1)))
    ((= i 256))
  (if (not (= (char->integer (integer->char i)) i))
      (format #t ";char->integer ~D ~A != ~A~%" i (integer->char i) (char->integer (integer->char i)))))

(test (reinvert 12 integer->char char->integer 60) 60)

(test (char->integer 33) 'error)
(test (char->integer) 'error)
(test (integer->char) 'error)
(test (integer->char (expt 2 31)) 'error)
(test (integer->char (expt 2 32)) 'error)
(test (integer->char 12 14) 'error)
(test (char->integer #\a #\b) 'error)
;(test (char->integer #\ÿ) 255) ; emacs confusion?
(test (eval-string (string-append "(char->integer " (format #f "#\\~C" (integer->char 255)) ")")) 255)

(for-each
 (lambda (arg)
   (test (char->integer arg) 'error))
 (list -1 1 0 123456789 "hi" () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (integer->char arg) 'error))
 (list -1 257 123456789 -123456789 #\a "hi" () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi most-positive-fixnum 1/0 (if #f #f) (lambda (a) (+ a 1))))

(test (#\a) 'error)
(test (#\newline 1) 'error)



;;; --------------------------------------------------------------------------------
;;; STRINGS
;;; --------------------------------------------------------------------------------

;;; --------------------------------------------------------------------------------
;;; string?

(test (string? "abc") #t)
(test (string? ':+*/-) #f)
(test (string? "das ist einer der teststrings") #t)
(test (string? '(das ist natuerlich falsch)) #f)
(test (string? "aaaaaa") #t)
(test (string? #\a) #f)
(test (string? "\"\\\"") #t)
(test (string? lambda) #f)
(test (string? format) #f)

(for-each
 (lambda (arg)
   (test (string? arg) #f))
 (list #\a () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (string?) 'error)
(test (string? "hi" "ho") 'error)
(test (string? #\null) #f)



;;; --------------------------------------------------------------------------------
;;; string=?

(test (string=? "foo" "foo") #t)
(test (string=? "foo" "FOO") #f)
(test (string=? "foo" "bar") #f)
(test (string=? "FOO" "FOO") #t)
(test (string=? "A" "B") #f)
(test (string=? "a" "b") #f)
(test (string=? "9" "0") #f)
(test (string=? "A" "A") #t)
(test (string=? "" "") #t)
(test (string=? (string #\newline) (string #\newline)) #t)

(test (string=? "A" "B" "a") #f)
(test (string=? "A" "A" "a") #f)
(test (string=? "A" "A" "A") #t)
(test (string=? "foo" "foo" "foo") #t)
(test (string=? "foo" "foo" "") #f)
(test (string=? "foo" "foo" "fOo") #f)

(test (string=? "foo" "FOO" 1.0) 'error)
(let ((str "01234567"))
  (test (length str) 8)
  (set! str (reverse! str))
  (test (string=? "76543210" str) #t) ; checking the bswap case in s7
  (set! str "012345670123456701234567012345670123456701234567")
  (test (length str) 48)
  (set! str (reverse! str))
  (string=? str "765432107654321076543210765432107654321076543210")
  (set! str "0123456")
  (set! str (reverse! str))
  (test (string=? "6543210" str) #t))

(test (let ((str (string #\" #\1 #\\ #\2 #\")))	(string=? str "\"1\\2\"")) #t)
(test (let ((str (string #\\ #\\ #\\)))	(string=? str "\\\\\\")) #t)
(test (let ((str (string #\")))	(string=? str "\"")) #t)
(test (let ((str (string #\\ #\"))) (string=? str "\\\"")) #t)
(test (let ((str (string #\space #\? #\)))) (string=? str " ?)")) #t)
(test (let ((str (string #\# #\\ #\t))) (string=? str "#\\t")) #t)
(test (string=? (string #\x (integer->char #xf0) #\x) (string #\x (integer->char #x70) #\x)) #f)
(test (string=? (string #\x (integer->char #xf0) #\x) (string #\x (integer->char #xf0) #\x)) #t)
(test (string=? "\x65;\x65;" "ee") #t)
(test (string=? "\"\\\n\t\r\/\b\f\x65;\"" "\x22;\x5c;\xa;\x09;\xd;\x2f;\x8;\xc;e\x22;") #t)

(test (string=? (string) "") #t)
(test (string=? (string) (make-string 0)) #t)
(test (string=? (string-copy (string)) (make-string 0)) #t)
(test (string=? "" (make-string 0)) #t)
(test (string=? "" (string-append)) #t)
(test (string=? (string #\space #\newline) " \n") #t)

(test (string=? "......" "...\
...") #t)
(test (string=? "\n" (string #\newline)) #t)
(test (string=? "\
\
\
\
" "") #t)
(test (string=? "" (string #\null)) #f)
(test (string=? (string #\null #\null) (string #\null)) #f)
(test (string=? "" "asd") #f)
(test (string=? "asd" "") #f)
(test (string=? "xx" (make-string 2 #\x) (string #\x #\x) (list->string (list #\x #\x)) (substring "axxb" 1 3) (string-append "x" "x")) #t)
(test (let ((s1 "1234") (s2 "1245")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string=? s1 s2)) #f)
(test (let ((s1 "1234") (s2 "1234")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string=? s1 s2)) #t)
(test (let ((s1 "1234") (s2 "124")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string=? s1 s2)) #f)
(test (string=? (make-string 3 #\space) (let ((s (make-string 4 #\space))) (set! (s 3) #\null) s)) #f)
(test "\x3012;" "0\x12;") ; \x30 = 48 = #\0

(for-each
 (lambda (arg)
   (test (string=? "hi" arg) 'error)
   (test (string=? arg "hi") 'error))
 (list #\a () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))


;; this strikes me as highly dubious
(test (call-with-input-string "1\n2" (lambda (p) (read p))) 1)
(test (call-with-input-string "1\\ \n2" (lambda (p) (read p))) (symbol "1\\"))

(test (eval (with-input-from-string "(symbol \"(\\\")\")" read)) (symbol "(\")"))

(test (call-with-input-string "1\
2" (lambda (p) (read p))) 12)

;; do we guarantee that read takes place in the current environment? no...
(test (call-with-input-string "fl\
oor" read) 'floor)

(test (call-with-input-string "p\
i" (lambda (p) (eval (read p)))) pi)

(test (call-with-input-string "(+ 1;\
 this is presumably a comment
 1)" (lambda (p) (eval (read p)))) 2)

(test (call-with-input-string "(+ 1;\
 this is presumably a comment;\
 and more commentary
 1)" (lambda (p) (eval (read p)))) 2)

(test (string=? (string #\1) (byte-vector 2)) 'error) ; changed 18-June-18
(let () (define (func) (let ((x #f) (i 0)) (if (not x) (string=? (immutable! "asdf") :frequency)))) (test (func) 'error))


;;; --------------------------------------------------------------------------------
;;; string<?

(test (string<? "aaaa" "aaab") #t)
(test (string<? "aaaa" "aaaaa") #t)
(test (string<? "" "abcdefgh") #t)
(test (string<? "a" "abcdefgh") #t)
(test (string<? "abc" "abcdefgh") #t)
(test (string<? "cabc" "abcdefgh") #f)
(test (string<? "abcdefgh" "abcdefgh") #f)
(test (string<? "xyzabc" "abcdefgh") #f)
(test (string<? "abc" "xyzabcdefgh") #t)
(test (string<? "abcdefgh" "") #f)
(test (string<? "abcdefgh" "a") #f)
(test (string<? "abcdefgh" "abc") #f)
(test (string<? "abcdefgh" "cabc") #t)
(test (string<? "abcdefgh" "xyzabc") #t)
(test (string<? "xyzabcdefgh" "abc") #f)
(test (string<? "abcdef" "bcdefgh") #t)
(test (string<? "" "") #f)
(test (string<? "A" "B") #t)
(test (string<? "a" "b") #t)
(test (string<? "9" "0") #f)
(test (string<? "A" "A") #f)

(test (string<? "A" "B" "A") #f)
(test (string<? "A" "A" "B") #f)
(test (string<? "A" "A" "A") #f)
(test (string<? "B" "B" "C") #f)
(test (string<? "foo" "foo" "foo") #f)
(test (string<? "foo" "foo" "") #f)
(test (string<? "foo" "foo" "fOo") #f)

(test (string<? "foo" "fo" 1.0) 'error)
(test (let ((s1 "1234") (s2 "1245")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string<? s1 s2)) #t)
(test (let ((s1 "1234") (s2 "123")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string<? s1 s2)) #f)
(test (let ((s1 "123") (s2 "1234")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string<? s1 s2)) #t)
(test (not (string<? "foo\x0a;" "foo\x0a;")) #t)
(test (string<? "foo\x0a;" "foo\x0b;") #t)

(test (string<? (string (integer->char #xf0)) (string (integer->char #x70))) #f)

(for-each
 (lambda (arg)
   (test (string<? "hi" arg) 'error)
   (test (string<? arg "hi") 'error))
 (list #\a () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))



;;; --------------------------------------------------------------------------------
;;; string>?

(test (string>? "aaab" "aaaa") #t)
(test (string>? "aaaaa" "aaaa") #t)
(test (string>? "" "abcdefgh") #f)
(test (string>? "a" "abcdefgh") #f)
(test (string>? "abc" "abcdefgh") #f)
(test (string>? "cabc" "abcdefgh") #t)
(test (string>? "abcdefgh" "abcdefgh") #f)
(test (string>? "xyzabc" "abcdefgh") #t)
(test (string>? "abc" "xyzabcdefgh") #f)
(test (string>? "abcdefgh" "") #t)
(test (string>? "abcdefgh" "a") #t)
(test (string>? "abcdefgh" "abc") #t)
(test (string>? "abcdefgh" "cabc") #f)
(test (string>? "abcdefgh" "xyzabc") #f)
(test (string>? "xyzabcdefgh" "abc") #t)
(test (string>? "abcde" "bc") #f)
(test (string>? "bcdef" "abcde") #t)
(test (string>? "bcdef" "abcdef") #t)
(test (string>? "" "") #f)
(test (string>? "A" "B") #f)
(test (string>? "a" "b") #f)
(test (string>? "9" "0") #t)
(test (string>? "A" "A") #f)

(test (string>? "A" "B" "a") #f)
(test (string>? "C" "B" "A") #t)
(test (string>? "A" "A" "A") #f)
(test (string>? "B" "B" "A") #f)
(test (string>? "foo" "foo" "foo") #f)
(test (string>? "foo" "foo" "") #f)
(test (string>? "foo" "foo" "fOo") #f)

(test (string>? "foo" "fooo" 1.0) 'error)
(test (let ((s1 "1234") (s2 "1245")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string>? s1 s2)) #f)
(test (let ((s1 "1234") (s2 "123")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string>? s1 s2)) #t)
(test (let ((s1 "123") (s2 "1234")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string>? s1 s2)) #f)

(test (string>? (string (integer->char #xf0)) (string (integer->char #x70))) #t) ; ??

(for-each
 (lambda (arg)
   (test (string>? "hi" arg) 'error)
   (test (string>? arg "hi") 'error))
 (list #\a () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))



;;; --------------------------------------------------------------------------------
;;; string<=?

(test (string<=? "aaa" "aaaa") #t)
(test (string<=? "aaaaa" "aaaa") #f)
(test (string<=? "a" "abcdefgh") #t)
(test (string<=? "abc" "abcdefgh") #t)
(test (string<=? "aaabce" "aaabcdefgh") #f)
(test (string<=? "cabc" "abcdefgh") #f)
(test (string<=? "abcdefgh" "abcdefgh") #t)
(test (string<=? "xyzabc" "abcdefgh") #f)
(test (string<=? "abc" "xyzabcdefgh") #t)
(test (string<=? "abcdefgh" "") #f)
(test (string<=? "abcdefgh" "a") #f)
(test (string<=? "abcdefgh" "abc") #f)
(test (string<=? "abcdefgh" "cabc") #t)
(test (string<=? "abcdefgh" "xyzabc") #t)
(test (string<=? "xyzabcdefgh" "abc") #f)
(test (string<=? "abcdef" "bcdefgh") #t)
(test (string<=? "" "") #t)
(test (string<=? "A" "B") #t)
(test (string<=? "a" "b") #t)
(test (string<=? "9" "0") #f)
(test (string<=? "A" "A") #t)

(test (string<=? "A" "B" "C") #t)
(test (string<=? "C" "B" "A") #f)
(test (string<=? "A" "B" "B") #t)
(test (string<=? "A" "A" "A") #t)
(test (string<=? "B" "B" "A") #f)
(test (string<=? "foo" "foo" "foo") #t)
(test (string<=? "foo" "foo" "") #f)
(test (string<=? "foo" "foo" "fooo") #t)

(test (string<=? "foo" "fo" 1.0) 'error)
(test (let ((s1 "1234") (s2 "1245")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string<=? s1 s2)) #t)
(test (let ((s1 "1234") (s2 "123")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string<=? s1 s2)) #f)
(test (let ((s1 "123") (s2 "1234")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string<=? s1 s2)) #t)
(test (let ((s1 "1234") (s2 "1234")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string<=? s1 s2)) #t)

(for-each
 (lambda (arg)
   (test (string<=? "hi" arg) 'error)
   (test (string<=? arg "hi") 'error))
 (list #\a () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))



;;; --------------------------------------------------------------------------------
;;; string>=?

(test (string>=? "aaaaa" "aaaa") #t)
(test (string>=? "aaaa" "aaaa") #t)
(test (string>=? "aaa" "aaaa") #f)
(test (string>=? "" "abcdefgh") #f)
(test (string>=? "a" "abcdefgh") #f)
(test (string>=? "abc" "abcdefgh") #f)
(test (string>=? "cabc" "abcdefgh") #t)
(test (string>=? "abcdefgh" "abcdefgh") #t)
(test (string>=? "xyzabc" "abcdefgh") #t)
(test (string>=? "abc" "xyzabcdefgh") #f)
(test (string>=? "abcdefgh" "") #t)
(test (string>=? "abcdefgh" "a") #t)
(test (string>=? "abcdefgh" "abc") #t)
(test (string>=? "abcdefgh" "cabc") #f)
(test (string>=? "abcdefgh" "xyzabc") #f)
(test (string>=? "xyzabcdefgh" "abc") #t)
(test (string>=? "bcdef" "abcdef") #t)
(test (string>=? "A" "B") #f)
(test (string>=? "a" "b") #f)
(test (string>=? "9" "0") #t)
(test (string>=? "A" "A") #t)
(test (string>=? "" "") #t)

(test (string>=? "A" "B" "C") #f)
(test (string>=? "C" "B" "A") #t)
(test (string>=? "C" "B" "B") #t)
(test (string>=? "A" "B" "B") #f)
(test (string>=? "A" "A" "A") #t)
(test (string>=? "B" "B" "A") #t)
(test (string>=? "B" "B" "C") #f)
(test (string>=? "foo" "foo" "foo") #t)
(test (string>=? "foo" "foo" "") #t)
(test (string>=? "foo" "foo" "fo") #t)

(test (string>=? "fo" "foo" 1.0) 'error)
(test (let ((s1 "1234") (s2 "1245")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string>=? s1 s2)) #f)
(test (let ((s1 "1234") (s2 "123")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string>=? s1 s2)) #t)
(test (let ((s1 "123") (s2 "1234")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string>=? s1 s2)) #f)
(test (let ((s1 "1234") (s2 "1234")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string>=? s1 s2)) #t)

(for-each
 (lambda (arg)
   (test (string>=? "hi" arg) 'error)
   (test (string>=? arg "hi") 'error))
 (list #\a () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))



;;; --------------------------------------------------------------------------------
;;; string-ci=?

(test (string-ci=? "A" "B") #f)
(test (string-ci=? "a" "B") #f)
(test (string-ci=? "A" "b") #f)
(test (string-ci=? "a" "b") #f)
(test (string-ci=? "9" "0") #f)
(test (string-ci=? "A" "A") #t)
(test (string-ci=? "A" "a") #t)
(test (string-ci=? "" "") #t)
(test (string-ci=? "aaaa" "AAAA") #t)
(test (string-ci=? "aaaa" "Aaaa") #t)

(test (string-ci=? "A" "B" "a") #f)
(test (string-ci=? "A" "A" "a") #t)
(test (string-ci=? "A" "A" "a") #t)
(test (string-ci=? "foo" "foo" "foo") #t)
(test (string-ci=? "foo" "foo" "") #f)
(test (string-ci=? "foo" "Foo" "fOo") #t)

(test (string-ci=? "foo" "GOO" 1.0) 'error)
(test (let ((s1 "1234") (s2 "1245")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci=? s1 s2)) #f)
(test (let ((s1 "1234") (s2 "1234")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci=? s1 s2)) #t)
(test (let ((s1 "1234") (s2 "124")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci=? s1 s2)) #f)
(test (let ((s1 "abcd") (s2 "ABCD")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci=? s1 s2)) #t)
(test (let ((s1 "abcd") (s2 "ABCE")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci=? s1 s2)) #f)
(test (let ((s1 "abcd") (s2 "ABC")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci=? s1 s2)) #f)

(for-each
 (lambda (arg)
   (test (string-ci=? "hi" arg) 'error)
   (test (string-ci=? arg "hi") 'error))
 (list #\a () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))


(when full-s7test
  (let ((size 15)
	(tries 10000))
    (let ((str1 (make-string size))
	  (str2 (make-string size)))
      (do ((i 0 (+ i 1)))
	  ((= i tries))
	(do ((k 0 (+ k 1)))
	    ((= k size))
	  (set! (str1 k) (integer->char (random 128)))
	  (if (> (random 10) 4)
	      (set! (str2 k) (char-upcase (str1 k)))
	      (set! (str2 k) (char-downcase (str1 k)))))
	(if (not (string-ci=? str1 str2))
	    (format #t "not =: ~S ~S~%" str1 str2))
	(if (and (string-ci<? str1 str2)
		 (string-ci>=? str1 str2))
	    (format #t "< : ~S ~S~%" str1 str2))
	(if (and (string-ci>? str1 str2)
		 (string-ci<=? str1 str2))
	    (format #t "> : ~S ~S~%" str1 str2))))))



;;; --------------------------------------------------------------------------------
;;; string-ci<?

(test (string-ci<? "a" "Aa") #t)
(test (string-ci<? "A" "B") #t)
(test (string-ci<? "a" "B") #t)
(test (string-ci<? "A" "b") #t)
(test (string-ci<? "a" "b") #t)
(test (string-ci<? "9" "0") #f)
(test (string-ci<? "0" "9") #t)
(test (string-ci<? "A" "A") #f)
(test (string-ci<? "A" "a") #f)
(test (string-ci<? "" "") #f)

(test (string-ci<? "t" "_") #t)
(test (string-ci<? "a" "]") #t)
(test (string-ci<? "z" "^") #t)
(test (string-ci<? "]4.jVKo\\\\^:\\A9Z4" "MImKA[mNv1`") #f)

(test (string-ci<? "A" "B" "A") #f)
(test (string-ci<? "A" "A" "B") #f)
(test (string-ci<? "A" "A" "A") #f)
(test (string-ci<? "B" "B" "C") #f)
(test (string-ci<? "B" "b" "C") #f)
(test (string-ci<? "foo" "foo" "foo") #f)
(test (string-ci<? "foo" "foo" "") #f)
(test (string-ci<? "foo" "foo" "fOo") #f)
(test (string-ci<? "34ZsfQD<obff33FBPFl" "7o" "9l7OM" "FC?M63=" "rLM5*J") #t)
(test (string-ci<? "NX7" "-;h>P" "DMhk3Bg") #f)
(test (string-ci<? "+\\mZl" "bE7\\e(HaW5CDXbPi@U_" "B_") #t)

(if (char-ci<? (integer->char #xf0) (integer->char #x70))
    (test (string-ci<? (string (integer->char #xf0)) (string (integer->char #x70))) #t))

(test (string-ci<? "foo" "fo" 1.0) 'error)
(test (let ((s1 "1234") (s2 "1245")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<? s1 s2)) #t)
(test (let ((s1 "1234") (s2 "1234")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<? s1 s2)) #f)
(test (let ((s1 "1234") (s2 "124")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<? s1 s2)) #t)
(test (let ((s1 "123") (s2 "12")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<? s1 s2)) #f)
(test (let ((s1 "abcd") (s2 "ABCD")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<? s1 s2)) #f)
(test (let ((s1 "abcd") (s2 "ABCE")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<? s1 s2)) #t)
(test (let ((s1 "abcd") (s2 "ABC")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<? s1 s2)) #f)
(test (let ((s1 "abc") (s2 "ABCD")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<? s1 s2)) #t)

(for-each
 (lambda (arg)
   (test (string-ci<? "hi" arg) 'error)
   (test (string-ci<? arg "hi") 'error))
 (list #\a () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

;;; from scheme bboard
(test (string-ci<=? "test" "tes") #f)
(test (string-ci<? "test" "tes") #f)
(test (string-ci>=? "test" "tes") #t)
(test (string-ci>? "test" "tes") #t)
(test (string-ci<=? "tes" "test") #t)
(test (string-ci<? "tes" "test") #t)
(test (string-ci>=? "tes" "test") #f)
(test (string-ci>? "tes" "test") #f)



;;; --------------------------------------------------------------------------------
;;; string-ci>?

(test (string-ci>? "Aaa" "AA") #t)
(test (string-ci>? "A" "B") #f)
(test (string-ci>? "a" "B") #f)
(test (string-ci>? "A" "b") #f)
(test (string-ci>? "a" "b") #f)
(test (string-ci>? "9" "0") #t)
(test (string-ci>? "A" "A") #f)
(test (string-ci>? "A" "a") #f)
(test (string-ci>? "" "") #f)
(test (string-ci>? "Z" "DjNTl0") #t)
(test (string-ci>? "2399dt7BVN[,A" "^KHboHV") #f)

(test (string-ci>? "t" "_") #f)
(test (string-ci>? "a" "]") #f)
(test (string-ci>? "z" "^") #f)
(test (string-ci>? "R*95oG.k;?" "`2?J6LBbLG^alB[fMD") #f)
(test (string-ci>? "]" "X") #t)

(test (string-ci>? "A" "B" "a") #f)
(test (string-ci>? "C" "b" "A") #t)
(test (string-ci>? "a" "A" "A") #f)
(test (string-ci>? "B" "B" "A") #f)
(test (string-ci>? "foo" "foo" "foo") #f)
(test (string-ci>? "foo" "foo" "") #f)
(test (string-ci>? "foo" "foo" "fOo") #f)
(test (string-ci>? "ZNiuEa@/V" "KGbKliYMY" "9=69q3ica" ":]") #f)
(test (string-ci>? "^" "aN@di;iEO" "7*9q6uPmX9)PaY,6J" "15vH") #t)

(test (string-ci>? "foo" "fooo" 1.0) 'error)
(test (let ((s1 "abcd") (s2 "ABCD")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci>? s1 s2)) #f)
(test (let ((s1 "abcd") (s2 "ABCE")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci>? s1 s2)) #f)
(test (let ((s1 "abcd") (s2 "ABC")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci>? s1 s2)) #t)
(test (let ((s1 "abc") (s2 "ABCD")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci>? s1 s2)) #f)

(for-each
 (lambda (arg)
   (test (string-ci>? "hi" arg) 'error)
   (test (string-ci>? arg "hi") 'error))
 (list #\a () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))



;;; --------------------------------------------------------------------------------
;;; string-ci<=?

(test (string-ci<=? "A" "B") #t)
(test (string-ci<=? "a" "B") #t)
(test (string-ci<=? "A" "b") #t)
(test (string-ci<=? "a" "b") #t)
(test (string-ci<=? "9" "0") #f)
(test (string-ci<=? "A" "A") #t)
(test (string-ci<=? "A" "a") #t)
(test (string-ci<=? "" "") #t)
(test (string-ci<=? ":LPC`" ",O0>affA?(") #f)

(test (string-ci<=? "t" "_") #t)
(test (string-ci<=? "a" "]") #t)
(test (string-ci<=? "z" "^") #t)
(test (string-ci<=? "G888E>beF)*mwCNnagP" "`2uTd?h") #t)

(test (string-ci<=? "A" "b" "C") #t)
(test (string-ci<=? "c" "B" "A") #f)
(test (string-ci<=? "A" "B" "B") #t)
(test (string-ci<=? "a" "A" "A") #t)
(test (string-ci<=? "B" "b" "A") #f)
(test (string-ci<=? "foo" "foo" "foo") #t)
(test (string-ci<=? "foo" "foo" "") #f)
(test (string-ci<=? "FOO" "fOo" "fooo") #t)
(test (string-ci<=? "78mdL82*" "EFaCrIdm@_D+" "eMu\\@dSSY") #t)
(test (string-ci<=? "`5pNuFc3PM<rNs" "e\\Su_raVNk6HD" "vXnuN7?S0?S(w+M?p") #f)

(test (string-ci<=? "fOo" "fo" 1.0) 'error)
(test (let ((s1 "abcd") (s2 "ABCD")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<=? s1 s2)) #t)
(test (let ((s1 "abcd") (s2 "ABCE")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<=? s1 s2)) #t)
(test (let ((s1 "abcd") (s2 "ABC")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<=? s1 s2)) #f)
(test (let ((s1 "abc") (s2 "ABCD")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<=? s1 s2)) #t)

(for-each
 (lambda (arg)
   (test (string-ci<=? "hi" arg) 'error)
   (test (string-ci<=? arg "hi") 'error))
 (list #\a () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))



;;; --------------------------------------------------------------------------------
;;; string-ci>=?

(test (string-ci>=? "A" "B") #f)
(test (string-ci>=? "a" "B") #f)
(test (string-ci>=? "A" "b") #f)
(test (string-ci>=? "a" "b") #f)
(test (string-ci>=? "9" "0") #t)
(test (string-ci>=? "A" "A") #t)
(test (string-ci>=? "A" "a") #t)
(test (string-ci>=? "" "") #t)
(test (string-ci>=? "5d7?[o[:hop=ktv;9)" "p^r9;TAXO=^") #f)

(test (string-ci>=? "t" "_") #f)
(test (string-ci>=? "a" "]") #f)
(test (string-ci>=? "z" "^") #f)
(test (string-ci>=? "jBS" "`<+s[[:`l") #f)

(test (string-ci>=? "A" "b" "C") #f)
(test (string-ci>=? "C" "B" "A") #t)
(test (string-ci>=? "C" "B" "b") #t)
(test (string-ci>=? "a" "B" "B") #f)
(test (string-ci>=? "A" "A" "A") #t)
(test (string-ci>=? "B" "B" "A") #t)
(test (string-ci>=? "B" "b" "C") #f)
(test (string-ci>=? "foo" "foo" "foo") #t)
(test (string-ci>=? "foo" "foo" "") #t)
(test (string-ci>=? "foo" "foo" "fo") #t)
(test (string-ci>=? "tF?8`Sa" "NIkMd7" "f`" "1td-Z?teE" "-ik1SK)hh)Nq].>") #t)
(test (string-ci>=? "Z6a8P" "^/VpmWwt):?o[a9\\_N" "8[^h)<KX?[utsc") #f)

(test (string-ci>=? "fo" "foo" 1.0) 'error)
(test (let ((s1 "abcd") (s2 "ABCD")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci>=? s1 s2)) #t)
(test (let ((s1 "abcd") (s2 "ABCE")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci>=? s1 s2)) #f)
(test (let ((s1 "abcd") (s2 "ABC")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci>=? s1 s2)) #t)
(test (let ((s1 "abc") (s2 "ABCD")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci>=? s1 s2)) #f)

(for-each
 (lambda (arg)
   (test (string-ci>=? "hi" arg) 'error)
   (test (string-ci>=? arg "hi") 'error))
 (list #\a () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))



;;; --------------------------------------------------------------------------------
;;; string-length

(test (string-length "abc") 3)
(test (string-length "") 0)
(test (string-length (string)) 0)
(test (string-length "\"\\\"") 3)
(test (string-length (string #\newline)) 1)
(test (string-length "hi there") 8)
(test (string-length "\"") 1)
(test (string-length "\\") 1)
(test (string-length "\n") 1)
(test (string-length (make-string 100 #\a)) 100)
(test (string-length "1\\2") 3)
(test (string-length "1\\") 2)
(test (string-length "hi\\") 3)
(test (string-length "\\\\\\\"") 4)
(test (string-length "A ; comment") 11)
(test (string-length "#| comment |#") 13)
(test (string-length "'123") 4)
(test (string-length '"'123") 4)
(test (let ((str (string #\# #\\ #\t))) (string-length str)) 3)

(test (string-length "#\\(") 3)
(test (string-length ")()") 3)
(test (string-length "(()") 3)
(test (string-length "(string #\\( #\\+ #\\space #\\1 #\\space #\\3 #\\))") 44)
(test (string-length) 'error)
(test (string-length "hi" "ho") 'error)
(test (string-length (string #\null)) 1) ; ?
(test (string-length (string #\null #\null)) 2) ; ?
(test (string-length (string #\null #\newline)) 2) ; ?
(test (string-length ``"hi") 2) ; ?? and in s7 ,"hi" is "hi" as with numbers

(test (string-length ";~S ~S") 6)
(test (string-length "\n;~S ~S") 7)
(test (string-length "\n\t") 2)
(test (string-length "#\newline") 8)
(test (string-length "#\tab") 4)
(test (string-length "a\x00;b") 3)

(test (string-length "123\
456") 6)
(test (string-length"123\n
456") 8)
(test (string-length"123\n\
456") 7)

(for-each
 (lambda (arg)
   (test (string-length arg) 'error))
 (list #\a () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))


(test (string? "[a-zA-Z_\x7f-\xff][a-zA-Z0-9_\x7f-\xff]*") #t)
(test (format #f "\x7f;~a" 1) (string #\delete #\1))
(test (format #f "\x00;~a\xff;" 1) (string #\null #\1 #\xff))
(test (format #f "\x44;") "D")
(test (format #f "\x44") "D")
(test (format #f "\x44;o") "Do")
(test (format #f "\x9") (string #\tab))

(test (string? "\x7") #t)
(test (string? "\x7f") #t)
(test (string? "\x7ff") #t)
(test (string? "\x7fff") #t)
(test (string? "\x7fff") #t)
;(test (string? "\xH") 'error)
;(string? "\x7H") ; an error in Guile
(test (string? "\x7fH") #t)
(test (string? "\x7ffH") #t)
(test (string? "\x7fffH") #t)
(test (string? "\x7fffH") #t)

;;; what is correct here? r7rs seems to say the semicolon is needed if we're in a string constant??
;;;    tests changed to include semicolon 1-Mar-22


;;; --------------------------------------------------------------------------------
;;; string

(for-each
 (lambda (arg)
   (test (string #\a arg) 'error)
   (test (string #\a #\null arg) 'error)
   (test (string arg) 'error))
 (list () (list 1) '(1 . 2) "a" #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (string) "")
(test (string #\a #\b #\c) "abc")
(test (string #\a) "a")
(test (map string '(#\a #\b)) '("a" "b"))
(test (map string '(#\a #\b) '(#\c #\d)) '("ac" "bd"))
(test (map string '(#\a #\b #\c) '(#\d #\e #\f) '(#\g #\h #\i)) '("adg" "beh" "cfi"))
(test (map string "abc" "def" "ghi") '("adg" "beh" "cfi"))
(test (string #\" #\# #\") "\"#\"")
(test (string #\\ #\\ #\# #\\ #\# #\#) "\\\\#\\##")
(test (string #\' #\' #\` #\") '"''`\"")
;;; some schemes accept \' and other such sequences in a string, but the spec only mentions \\ and \"
(test (string ()) 'error)
(test (string "j" #\a) 'error)
(test (string (values #\a #\b #\c)) "abc")




;;; --------------------------------------------------------------------------------
;;; make-string

(test (make-string 0) "")
(test (make-string 3 #\a) "aaa")
(test (make-string 0 #\a) "")
(test (make-string 3 #\space) "   ")
(test (let ((hi (make-string 3 #\newline))) (string-length hi)) 3)
(test (make-string (* 8796093022208 8796093022208)) 'error)
(test (make-string 8796093022208) 'error)

(test (make-string -1) 'error)
(test (make-string -0) "")
(test (make-string 2 #\a #\b) 'error)
(test (make-string) 'error)
(test (make-string most-positive-fixnum) 'error)
(test (make-string most-negative-fixnum) 'error)
(let () (define (hi size) (make-string size (integer->char (+ 1 (random 255))))) (string? (hi 3)))

(for-each
 (lambda (arg)
   (test (make-string 3 arg) 'error))
 (list "hi" () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (make-string arg #\a) 'error))
 (list #\a "hi" () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (make-string arg) 'error))
 (list #\a "hi" () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (make-string 0 #f) 'error)


;;; --------------------------------------------------------------------------------
;;; string-ref

(test (string-ref "abcdef-dg1ndh" 0) #\a)
(test (string-ref "abcdef-dg1ndh" 1) #\b)
(test (string-ref "abcdef-dg1ndh" 6) #\-)
(test (string-ref "\"\\\"" 1) #\\)
(test (string-ref "\"\\\"" 2) #\")

(test (let ((str (make-string 3 #\x))) (set! (string-ref str 1) #\a) str) "xax")

(test (string-ref "abcdef-dg1ndh" 20) 'error)
(test (string-ref "abcdef-dg1ndh") 'error)
(test (string-ref "abcdef-dg1ndh" -3) 'error)
(test (string-ref) 'error)
(test (string-ref 2) 'error)
(test (string-ref "\"\\\"" 3) 'error)
(test (string-ref "" 0) 'error)
(test (string-ref "" 1) 'error)
(test (string-ref "hiho" (expt 2 32)) 'error)
(test (char=? (string-ref (string #\null) 0) #\null) #t)
(test (char=? (string-ref (string #\1 #\null #\2) 1) #\null) #t)
(test (char=? ("1\x002;" 1) #\null) #t)
(test (char=? (string-ref (string #\newline) 0) #\newline) #t)
(test (char=? (string-ref (string #\space) 0) #\space) #t)

(for-each
 (lambda (arg)
   (test (string-ref arg 0) 'error))
 (list #\a 1 () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (string-ref "hiho" arg) 'error))
 (list #\a -1 123 4 "hi" () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test ("hi" 1) #\i)
(test (("hi" 1) 0) 'error)
(test ("hi" 1 2) 'error)
(test ("" 0) 'error)
(test (set! ("" 0) #\a) 'error)
(test (set! ("hi" 1 2) #\a) 'error)
(test (set! ("hi" 1) #\a #\b) 'error)
(test ("hi") 'error)
(test ("") 'error)
(test ((let () "hi")) 'error)
(test ((let () "hi") 0) #\h)

(test ("abs" most-negative-fixnum) 'error)
(test (string-ref "abs" most-negative-fixnum) 'error)
(test ("abs" (+ 1 most-negative-fixnum)) 'error)
(test ("abs" most-positive-fixnum) 'error)
(test (catch #t (lambda () ("hi" 1 2)) (lambda (t i) (apply format #f i)))  "string ref: too many indices: (\"hi\" 1 2)")


;;; --------------------------------------------------------------------------------
;;; string-copy

(test (let ((hi (string-copy "hi"))) (string-set! hi 0 #\H) hi) "Hi")
(test (let ((hi (string-copy "hi"))) (string-set! hi 1 #\H) hi) "hH")
(test (let ((hi (string-copy "\"\\\""))) (string-set! hi 0 #\a) hi) "a\\\"")
(test (let ((hi (string-copy "\"\\\""))) (string-set! hi 1 #\a) hi) "\"a\"")
(test (let ((hi (string #\a #\newline #\b))) (string-set! hi 1 #\c) hi) "acb")
(test (string-copy "ab") "ab")
(test (string-copy "") "")
(test (string-copy "\"\\\"") "\"\\\"")
(test (let ((hi "abc")) (eq? hi (string-copy hi))) #f)
(test (let ((hi (string-copy (make-string 8 (integer->char 0))))) (string-fill! hi #\a) hi) "aaaaaaaa")
(test (string-copy (string-copy (string-copy "a"))) "a")
(test (string-copy (string-copy (string-copy ""))) "")
(test (string-copy "a\x00;b") "a\x00;b") ; prints normally as "a" however
(test (string-copy (string #\1 #\null #\2)) (string #\1 #\null #\2))
(test (string-copy) 'error)
(test (string-copy "hi" (immutable! "ho")) 'error)
(test (string-copy "hi" (make-string 4 #\.) 0) "hi..")
(test (string-copy "hi" (make-string 4 #\.) 1) ".hi.")
(test (string-copy "hi" (make-string 4 #\.) 2) "..hi")
(test (string-copy "hi" (make-string 4 #\.) 3) "...h")
(test (string-copy "hi" (make-string 4 #\.) 4) "....")
(test (string-copy "hi" (make-string 4 #\.) 1 1) "....")
(test (string-copy "hi" (make-string 4 #\.) 1 2) ".h..")
(test (string-copy "hi" (make-string 4 #\.) 1 3) ".hi.")
(test (string-copy "hi" (make-string 4 #\.) 1 4) ".hi.")
(test (string-copy "hi" "asdf") "hidf")
(test (string-copy "" (make-string 4 #\.)) "....")
(test (string-copy "ho" "") "")
(test (string-copy "ho" ".") "h")
(test (let () (define (func) (string-copy "abc" (substring "0123" 1))) (func)) "abc") ; string_substring_chooser incorrectly from string_copy

(for-each
 (lambda (arg)
   (test (string-copy arg) 'error)
   (test (string-copy "hi" arg) 'error))
 (list #\a 1 () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (length (string-copy (string #\null))) 1)
(test (string-copy "hi" (make-string 4 #\.) "ho") 'error)
(test (string-copy "hi" (make-string 4 #\.) 1 "ho") 'error)


;;; --------------------------------------------------------------------------------
;;; string-set!

(let ((str (make-string 10 #\x)))
  (string-set! str 3 (integer->char 0))
  (test (string=? str "xxx") #f)
  (test (char=? (string-ref str 4) #\x) #t)
  (string-set! str 4 #\a)
  (test (string=? str "xxx") #f)
  (test (char=? (string-ref str 4) #\a) #t)
  (string-set! str 3 #\x)
  (test (string=? str "xxxxaxxxxx") #t))

(test (string-set! "hiho" 1 #\c) #\c)
(test (set! ("hi" 1 2) #\i) 'error)
(test (set! ("hi" 1) "ho") 'error)
(test (set! ("hi") #\i) 'error)
(test (let ((x "hi") (y 'x)) (string-set! y 0 #\x) x) 'error)
(test (let ((str "ABS")) (set! (str 0) #\a)) #\a)
(test (let ((str "ABS")) (string-set! str 0 #\a)) #\a)
(test (let ((str "ABS")) (set! (string-ref str 0) #\a)) #\a)

(test (let ((hi (make-string 3 #\a)))
	(string-set! hi 1 (let ((ho (make-string 4 #\x)))
			    (string-set! ho 1 #\b)
			    (string-ref ho 0)))
	hi)
      "axa")

(test (string-set! "hiho" (expt 2 32) #\a) 'error)

(test (let ((hi (string-copy "hi"))) (string-set! hi 2 #\H) hi) 'error)
(test (let ((hi (string-copy "hi"))) (string-set! hi -1 #\H) hi) 'error)
(test (let ((g (lambda () "***"))) (string-set! (g) 0 #\?)) #\?)
(test (string-set! "" 0 #\a) 'error)
(test (string-set! "" 1 #\a) 'error)
(test (string-set! (string) 0 #\a) 'error)
(test (string-set! (symbol->string 'lambda) 0 #\a) #\a)
(test (let ((ho (make-string 0 #\x))) (string-set! ho 0 #\a) ho) 'error)
(test (let ((str "hi")) (string-set! (let () str) 1 #\a) str) "ha") ; (also in Guile)
(test (let ((x 2) (str "hi")) (string-set! (let () (set! x 3) str) 1 #\a) (list x str)) '(3 "ha"))
(test (let ((str "hi")) (set! ((let () str) 1) #\b) str) "hb")
(test (let ((str "hi")) (string-set! (let () (string-set! (let () str) 0 #\x) str) 1 #\x) str) "xx")
(test (let ((str "hi")) (string-set! (let () (set! str "hiho") str) 3 #\x) str) "hihx") ; ! (this works in Guile also)

(for-each
 (lambda (arg)
   (test (string-set! arg 0 #\a) 'error))
 (list #\a 1 () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (string-set! "hiho" arg #\a) 'error))
 (list #\a -1 123 4 "hi" () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (string-set! "hiho" 0 arg) 'error))
 (list 1 "hi" () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (equal? (let ((str "hiho")) (string-set! str 2 #\null) str) "hi") #f)
(test (string=? (let ((str "hiho")) (string-set! str 2 #\null) str) "hi") #f)
(test (let* ((s1 "hi") (s2 s1)) (string-set! s2 1 #\x) s1) "hx")
(test (let* ((s1 "hi") (s2 (copy s1))) (string-set! s2 1 #\x) s1) "hi")

(test (eq? (car (catch #t (lambda () (set! ("hi") #\a)) (lambda args args))) 'wrong-number-of-args) #t)
(test (eq? (car (catch #t (lambda () (set! ("hi" 0 0) #\a)) (lambda args args))) 'wrong-number-of-args) #t) ; (vector-set! 1 ...)
(test (eq? (car (catch #t (lambda () (set! (("hi" 0) 0) #\a)) (lambda args args))) 'no-setter) #t) ; (set! (1 ...))

(test (let ((s "012345")) (set! (apply s 2) #\a) s) 'error)
(test (string-set! #u(0 1 0) 0 -9223372036854775808) 'error)

(let () (define (hi) (let ((str "123")) (set! (str 0) (values #\a #\b)) str)) (test (begin (hi) (hi)) 'error))
(let () (define (hi) (let ((str "123")) (set! (str 0) (values #\a #\b)) str)) (test (begin (catch #t hi (lambda a a)) (hi)) 'error))



;;; --------------------------------------------------------------------------------
;;; string-fill!

(test (string-fill! "hiho" #\c) #\c)
(test (string-fill! "" #\a) #\a)
(test (string-fill! "hiho" #\a) #\a)
(test (let ((g (lambda () "***"))) (string-fill! (g) #\?)) #\?)
(test (string-fill!) 'error)
(test (string-fill! "hiho" #\a #\b) 'error)

(test (let ((hi (string-copy "hi"))) (string-fill! hi #\s) hi) "ss")
(test (let ((hi (string-copy ""))) (string-fill! hi #\x) hi) "")
(test (let ((str (make-string 0))) (string-fill! str #\a) str) "")
(test (let ((hi (make-string 8 (integer->char 0)))) (string-fill! hi #\a) hi) "aaaaaaaa") ; is this result widely accepted?
(test (recompose 12 string-copy "xax") "xax")
(test (let ((hi (make-string 3 #\x))) (recompose 12 (lambda (a) (string-fill! a #\a) a) hi)) "aaa")
(test (let ((hi (make-string 3 #\x))) (recompose 12 (lambda (a) (string-fill! hi a)) #\a) hi) "aaa")
(test (let ((str (string #\null #\null))) (fill! str #\x) str) "xx")

(for-each
 (lambda (arg)
   (test (let ((hiho "hiho")) (string-fill! hiho arg) hiho) 'error))
 (list 1 "hi" () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (string-fill! arg #\a) 'error))
 (list #\a 1 () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (let ((str "1234567890")) (string-fill! str #\a 0) str) "aaaaaaaaaa")
(test (let ((str "1234567890")) (string-fill! str #\a 0 10) str) "aaaaaaaaaa")
(test (let ((str "1234567890")) (string-fill! str #\a 0 0) str) "1234567890")
(test (let ((str "1234567890")) (string-fill! str #\a 4 4) str) "1234567890")
(test (let ((str "1234567890")) (string-fill! str #\a 10 10) str) "1234567890")
(test (let ((str "1234567890")) (string-fill! str #\a 0 4) str) "aaaa567890")
(test (let ((str "1234567890")) (string-fill! str #\a 3 4) str) "123a567890")
(test (let ((str "1234567890")) (string-fill! str #\a 1 9) str) "1aaaaaaaa0")
(test (let ((str "1234567890")) (string-fill! str #\a 8) str) "12345678aa")
(test (let ((str "1234567890")) (string-fill! str #\a 1 9 0) str) 'error)
(test (let ((str "1234567890")) (string-fill! str #\a 1 0) str) 'error)
(test (let ((str "1234567890")) (string-fill! str #\a 11) str) 'error)
(test (let ((str "1234567890")) (string-fill! str #\a 9 11) str) 'error)
(test (string-fill! "" 0 "hi") 'error)
(test (string-fill! "" 0 -1 3) 'error)
(test (string-fill! "" 0 1) 'error)
(test (string-fill! "" 0 0 4/3) 'error)
(test (string-fill! "aaa" #\b #f) 'error)
(test (string-fill! "aaa" #\b 1 #f) 'error)
(test (string-fill! "aaa" #\b #f 1) 'error)



;;; --------------------------------------------------------------------------------
;;; string-upcase
;;; string-downcase

(test (string-downcase "") "")
(test (string-downcase "a") "a")
(test (string-downcase "A") "a")
(test (string-downcase "AbC") "abc")
(test (string-downcase "\"\\\"") "\"\\\"")
(test (let ((hi "abc")) (eq? hi (string-downcase hi))) #f)
(test (string-downcase (string-upcase (string-downcase "a"))) "a")
(test (string-downcase "a\x00;b") "a\x00;b")
(test (string-downcase (string #\1 #\null #\2)) (string #\1 #\null #\2))
(test (string-downcase) 'error)
(test (string-downcase "hi" "ho") 'error)

(test (string-upcase "") "")
(test (string-upcase "a") "A")
(test (string-upcase "A") "A")
(test (string-upcase "AbC") "ABC")
(test (string-upcase "\"\\\"") "\"\\\"")
(test (let ((hi "ABC")) (eq? hi (string-upcase hi))) #f)
(test (string-upcase (string-downcase (string-upcase "a"))) "A")
(test (string-upcase "a\x00;b") "A\x00;B")
(test (string-upcase (string #\1 #\null #\2)) (string #\1 #\null #\2))
(test (string-upcase) 'error)
 (test (string-upcase "hi" "ho") 'error)

(for-each
 (lambda (arg)
   (test (string-downcase arg) 'error)
   (test (string-upcase arg) 'error))
 (list #\a 1 () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

;;; for r7rs, these need to be unicode-aware


;;; --------------------------------------------------------------------------------
;;; substring

(test (substring "ab" 0 0) "")
(test (substring "ab" 1 1) "")
(test (substring "ab" 2 2) "")
(test (substring "ab" 0 1) "a")
(test (substring "ab" 1 2) "b")
(test (substring "ab" 0 2) "ab")
(test (substring "hi there" 3 6) "the")
(test (substring "hi there" 0 (string-length "hi there")) "hi there")
(test (substring "" 0 0) "")
(let ((str "012345"))
  (let ((str1 (substring str 2 4)))
    (string-set! str1 1 #\x)
    (test (string=? str "012345") #t)
    (let ((str2 (substring str1 1)))
      (set! (str2 0) #\z)
      (test (string=? str "012345") #t)
      (test (string=? str1 "2x") #t)
      (test (string=? str2 "z") #t))))
(test (substring (substring "hiho" 0 2) 1) "i")
(test (substring (substring "hiho" 0 2) 2) "")
(test (substring (substring "hiho" 0 2) 0 1) "h")
(test (substring "hi\nho" 3 5) "ho")
(test (substring (substring "hi\nho" 1 4) 2) "h")
(test (substring (substring "hi\nho" 3 5) 1 2) "o")
(test (substring "hi\"ho" 3 5) "ho")
(test (substring (substring "hi\"ho" 1 4) 2) "h")
(test (substring (substring "hi\"ho" 3 5) 1 2) "o")
(test (let* ((s1 "0123456789") (s2 (substring s1 1 3))) (string-set! s2 1 #\x) s1) "0123456789")
(test (substring (substring "" 0 0) 0 0) "")
(test (substring (format #f "") 0 0) "")
(test (string=? (substring (substring (substring "01234567" 1) 1) 1) "34567") #t)
(let ()
  (define (hi) (string=? (substring (substring (substring "01234567" 1) 1) 1) "34567"))
  (define (ho) (hi)) (ho)
  (test (ho) #t))

(test (substring "012" 3) "")
(test (substring "012" 10) 'error)
(test (substring "012" most-positive-fixnum) 'error)
(test (substring "012" -1) 'error)
(test (substring "012" 3 3) "")
(test (substring "012" 3 4) 'error)
(test (substring "012" 3 2) 'error)
(test (substring "012" 3 -2) 'error)
(test (substring "012" 3 0) 'error)
(test (substring "012" 0) "012")
(test (substring "012" 2) "2")
(test (substring "" 0) "")

(test (recompose 12 (lambda (a) (substring a 0 3)) "12345") "123")
(test (reinvert 12 (lambda (a) (substring a 0 3)) (lambda (a) (string-append a "45")) "12345") "12345")

(test (substring "ab" 0 3) 'error)
(test (substring "ab" 3 3) 'error)
(test (substring "ab" 2 3) 'error)
(test (substring "" 0 1) 'error)
(test (substring "" -1 0) 'error)
(test (substring "abc" -1 0) 'error)
(test (substring "hiho" (expt 2 32) (+ 2 (expt 2 32))) 'error)
(test (substring) 'error)
(test (substring "hiho" 0 1 2) 'error)
(test (substring "1234" -1 -1) 'error)
(test (substring "1234" 1 0) 'error)
(test (substring "" most-positive-fixnum 1) 'error)

(let ((str "0123456789"))
  (string-set! str 5 #\null)
  (test (substring str 6) "6789")
  (test (substring str 5 5) "")
  (test (substring str 4 5) "4")
  (test (substring str 5 6) "\x00;")
  (test (substring str 5 7) "\x00;6")
  (test (substring str 4 7) "4\x00;6"))

(for-each
 (lambda (arg)
   (test (substring "hiho" arg 0) 'error))
 (list "hi" #\a 1 () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (substring "0123" arg) 'error)
   (test (substring "hiho" 1 arg) 'error))
 (list "hi" #\a -1 () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (substring arg 1 2) 'error))
 (list #\a 1 () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(define (substring? pattern target) ; taken from net somewhere (umich?) with changes for s7 (which now has string-position, so this is unneeded)
  (define (build-shift-vector pattern)
    (let* ((pat-len (length pattern))
	   (shift-vec (make-vector 256 (+ pat-len 1)))
	   (max-pat-index (- pat-len 1)))
      (let loop ((index 0))
	(set! (shift-vec (char->integer (pattern index))) (- pat-len index))
	(if (< index max-pat-index)
	    (loop (+ index 1))
	    shift-vec))))
  (if (or (not (string? pattern))
	  (not (string? target)))
      (error 'wrong-type-arg "substring? args should be strings: ~S ~S" pattern target)
      (let ((pat-len (length pattern)))
	(if (zero? pat-len)
	    0
	    (let ((shift-vec (build-shift-vector pattern)))
	      (let* ((tar-len (length target))
		     (max-tar-index (- tar-len 1))
		     (max-pat-index (- pat-len 1)))
		(let outer ((start-index 0))
		  (and (<= (+ pat-len start-index) tar-len)
		       (let inner ((p-ind 0) (t-ind start-index))
			 (cond
			  ((> p-ind max-pat-index) #f)           ; nothing left to check
			  ((char=? (pattern p-ind) (target t-ind))
			   (if (= p-ind max-pat-index)
			       start-index                       ; success -- return start index of match
			       (inner (+ p-ind 1) (+ t-ind 1)))) ; keep checking
			  ((> (+ pat-len start-index) max-tar-index) #f) ; fail
			  (else (outer (+ start-index (shift-vec (char->integer (target (+ start-index pat-len)))))))))))))))))

(test (substring? "hiho" "test hiho test") 5)
(test (substring? "hiho" "test hihptest") #f)
(test (substring? "hiho" "test hih") #f)
(test (substring? "hiho" "") #f)
(test (substring? "hiho" "hiho") 0)
(test (substring? "" "hiho") 0)
(test (substring? "abc" 'abc) 'error)
(test (substring "123345" (ash 1 32)) 'error)
(test (substring "123345" 0 (ash 1 32)) 'error)
(test (substring "123345" 8796093022208) 'error)




;;; --------------------------------------------------------------------------------
;;; string-append

(test (string-append "hi" "ho") "hiho")
(test (string-append "hi") "hi")
(test (string-append "hi" "") "hi")
(test (string-append "hi" "" "ho") "hiho")
(test (string-append "" "hi") "hi")
(test (string-append) "")
(test (string-append "a" (string-append (string-append "b" "c") "d") "e") "abcde")
(test (string-append "a" "b" "c" "d" "e") "abcde")
(test (string-append (string-append) (string-append (string-append))) "")
(test (let ((hi "hi")) (let ((ho (string-append hi))) (eq? hi ho))) #f)
(test (let ((hi "hi")) (let ((ho (string-append hi))) (string-set! ho 0 #\a) hi)) "hi")
(test (let ((hi "hi")) (set! hi (string-append hi hi hi hi)) hi) "hihihihi")
(test (string-append ()) 'error)
(test (string=? (string-append "012" (string #\null) "456")
		(let ((str "0123456")) (string-set! str 3 #\null) str))
      #t)
(test (string=? (string-append "012" (string #\null) "356")
		(let ((str "0123456")) (string-set! str 3 #\null) str))
      #f)
(test (string-append """hi""ho""") "hiho")
(test (let* ((s1 "hi") (s2 (string-append s1 s1))) (string-set! s2 1 #\x) s1) "hi")
(test (let* ((s1 "hi") (s2 (string-append s1))) (string-set! s2 1 #\x) s1) "hi")
(test (length (string-append (string #\x #\y (integer->char 127) #\z) (string #\a (integer->char 0) #\b #\c))) 8)

(test (length (string-append "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc")) 915)
(test (length (string-append (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c))) 915)


(num-test (letrec ((hi (lambda (str n)
			 (if (= n 0)
			     str
			     (hi (string-append str "a") (- n 1))))))
	    (string-length (hi "" 100)))
	  100)

(test (let* ((str "hiho")
	     (str1 "ha")
	     (str2 (string-append str1 str)))
	(string-set! str2 1 #\x)
	(string-set! str2 4 #\x)
	(and (string=? str "hiho")
	     (string=? str1 "ha")
	     (string=? str2 "hxhixo")))
      #t)
(test (let* ((str (string-copy "hiho"))
	     (str1 (string-copy "ha"))
	     (str2 (string-append str1 str)))
	(string-set! str1 1 #\x)
	(string-set! str 2 #\x)
	(and (string=? str "hixo")
	     (string=? str1 "hx")
	     (string=? str2 "hahiho")))
      #t)

(let ((s1 (string #\x #\null #\y))
      (s2 (string #\z #\null)))
  (test (string=? (string-append s1 s2) (string #\x #\null #\y #\z #\null)) #t)
  (test (string=? (string-append s2 s1) (string #\z #\null #\x #\null #\y)) #t))

(test (recompose 12 string-append "x") "x")
(test (recompose 12 (lambda (a) (string-append a "x")) "a") "axxxxxxxxxxxx")
(test (recompose 12 (lambda (a) (string-append "x" a)) "a") "xxxxxxxxxxxxa")

(test (length (string-append "\\?" "hi")) 4)
(test (string-append "hi" 1) 'error)
(test (eval-string "(string-append \"\\?\")") 'error) ; guile mailing list
(test (eval-string "(string-append \"\\?\" \"hi\")") 'error) ; guile mailing list
(for-each
 (lambda (arg)
   (test (string-append "hiho" arg) 'error)
   (test (string-append arg "hi") 'error)
   (test (string-append "a" "b" arg) 'error))
 (list #\a 1 () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(let ()
  (define (f) (string-append (string-append (string))))
  (test (byte-vector? (f)) #f)
  (test (byte-vector? (f)) #f))

(let ()
  (define (f) (string-append (string-append (string #\a #\b))))
  (test (byte-vector? (f)) #f)
  (test (byte-vector? (f)) #f))

(test (append #u(0 1 2 3) #u(4 5 6)) #u(0 1 2 3 4 5 6))



;;; --------------------------------
(test (let ((str (make-string 4 #\x))
	    (ctr 0))
	(for-each
	 (lambda (c)
	   (string-set! str ctr c)
	   (set! ctr (+ ctr 1)))
	 "1234")
	str)
      "1234")

(test (let ((str (make-string 8 #\x))
	    (ctr 0))
	(for-each
	 (lambda (c1 c2)
	   (string-set! str ctr c1)
	   (string-set! str (+ ctr 1) c2)
	   (set! ctr (+ ctr 2)))
	 "1234"
	 "hiho")
	str)
      "1h2i3h4o")

#|
(let ((size 1024))
  (let ((str (make-string size)))
    (do ((i 0 (+ i 1)))
	((= i size))
      (set! (str i) (integer->char (+ 1 (modulo i 255)))))
    (let ((str1 (string-copy str)))
      (test (string? str1) #t)
      (test (string-length str1) 1024)
      (test (string-ref str1 556) (string-ref str 556))
      (test (string=? str str1) #t)
      (test (string<=? str str1) #t)
      (test (string>=? str str1) #t)
      (test (string-ci=? str str1) #t)
      (test (string-ci<=? str str1) #t)
      (test (string-ci>=? str str1) #t)
      (test (string<? str str1) #f)
      (test (string>? str str1) #f)
      (test (string-ci<? str str1) #f)
      (test (string-ci>? str str1) #f)
      (test (substring str 123 321) (substring str1 123 321))

      (string-set! str1 1000 #\space)
      (test (string=? str str1) #f)
      (test (string<=? str str1) #f)
      (test (string>=? str str1) #t)
      (test (string-ci=? str str1) #f)
      (test (string-ci<=? str str1) #f)
      (test (string-ci>=? str str1) #t)
      (test (string<? str str1) #f)
      (test (string>? str str1) #t)
      (test (string-ci<? str str1) #f)
      (test (string-ci>? str str1) #t)

      (test (string-length (string-append str str1)) 2048)
      ))))
|#



;;; --------------------------------------------------------------------------------
;;; string->list
;;; list->string

(test (string->list "abc") (list #\a #\b #\c))
(test (string->list "") ())
(test (string->list (make-string 0)) ())
(test (string->list (string #\null)) '(#\null))
(test (string->list (string)) ())
(test (string->list (substring "hi" 0 0)) ())
(test (string->list (list->string (list #\a #\b #\c))) (list #\a #\b #\c))
(test (string->list (list->string ())) ())
(test (list->string (string->list "abc")) "abc")
(test (list->string (string->list "hi there")) "hi there")
(test (list->string (string->list "&*#%^@%$)~@")) "&*#%^@%$)~@")
(test (list->string (string->list "")) "")
(test (let* ((str "abc")
	     (lst (string->list str)))
	(and (string=? str "abc")
	     (equal? lst (list #\a #\b #\c))))
      #t)
(test (list->string ()) "")

(test (list->string (list #\a #\b #\c)) "abc")
(test (list->string (list)) "")

(test (list->string (list #\" #\# #\")) "\"#\"")
(test (list->string (list #\\ #\\ #\# #\\ #\# #\#)) "\\\\#\\##")
(test (list->string (list #\' #\' #\` #\")) '"''`\"")

(test (reinvert 12 string->list list->string "12345") "12345")

(test (string->list) 'error)
(test (list->string) 'error)
(test (string->list "hi" "ho") 'error)
(test (list->string () '(1 2)) 'error)
(test (apply list->string '(#\a . #\b)) 'error)
(test (list->string #\a . #\b) 'error)
(test (let ((lst (cons #\a #\b))) (list->string lst)) 'error)
(test (string->list " hi ") '(#\space #\h #\i #\space))
(test (string->list (string (integer->char #xf0) (integer->char #x70))) (list (integer->char #xf0) (integer->char #x70)))

(for-each
 (lambda (arg)
   (test (string->list arg) 'error))
 (list #\a 1 () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (let ((x (cons #\a #\b))) (set-cdr! x x) (list->string x)) 'error)
(test (let ((lst (list #\a #\b))) (set! (cdr (cdr lst)) lst) (list->string lst)) 'error)
(test (let ((lst (list #\a #\b))) (set! (cdr (cdr lst)) lst) (apply string lst)) 'error)

(for-each
 (lambda (arg)
   (test (list->string arg) 'error))
 (list "hi" #\a 1 ''foo '(1 . 2) (cons #\a #\b) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(let ((str (list->string '(#\x #\space #\null #\x))))
  (test (length str) 4)
  (test (str 1) #\space)
  (test (str 2) #\null)
  (test (str 3) #\x)
  (test (object->string str) "\"x \\x00;x\"")
  (let ((lst (string->list str)))
    (test lst '(#\x #\space #\null #\x))))

(let ((strlen 8))
  (let ((str (make-string strlen)))
    (do ((i 0 (+ i 1)))
	((= i 10))
      (do ((k 0 (+ k 1)))
	  ((= k strlen))
	(set! (str k) (integer->char (random 256))))
      (let ((lst (string->list str)))
	(let ((newstr (list->string lst)))
	  (let ((lstlen (length lst))
		(newstrlen (length newstr)))
	    (if (or (not (= lstlen strlen newstrlen))
		    (not (string=? newstr str)))
		(format #t ";string->list->string: ~S -> ~A -> ~S~%" str lst newstr))))))))

(when full-s7test
  (let ()
    (define (all-strs len file)
      (let* ((funny-chars (list #\` #\# #\, #\@ #\' #\" #\. #\( #\) #\\))
	     (num-chars (length funny-chars)))
	(let ((ctrs (make-vector len 0)))

	  (do ((i 0 (+ i 1)))
	      ((= i (expt num-chars len)))
	    (let ((carry #t))
	      (do ((k 0 (+ k 1)))
		  ((or (= k len)
		       (not carry)))
		(vector-set! ctrs k (+ 1 (vector-ref ctrs k)))
		(if (= (vector-ref ctrs k) num-chars)
		    (vector-set! ctrs k 0)
		    (set! carry #f)))

	      (let ((strlst ()))
		(do ((k 0 (+ k 1)))
		    ((= k len))
		  (let ((c (list-ref funny-chars (vector-ref ctrs k))))
		    (set! strlst (cons c strlst))))

		(let ((str (list->string strlst)))
		  (format file "(test (and (string=? ~S (string ~{#\\~C~^ ~})) (equal? '~A (string->list ~S))) #t)~%" str strlst strlst str))))))))

    (call-with-output-file "strtst.scm"
      (lambda (p)
	(do ((len 3 (+ len 1)))
	    ((= len 5))
	  (all-strs len p))))

    (load "strtst.scm")))


(test (and (string=? "\"" (string #\")) (equal? '(#\") (string->list "\""))) #t)
(test (and (string=? "#\\" (string #\# #\\)) (equal? '(#\# #\\) (string->list "#\\"))) #t)
(test (and (string=? "#(" (string #\# #\()) (equal? '(#\# #\() (string->list "#("))) #t)
(test (and (string=? "\"@" (string #\" #\@)) (equal? '(#\" #\@) (string->list "\"@"))) #t)
(test (and (string=? "\";" (string #\" #\;)) (equal? '(#\" #\;) (string->list "\";"))) #t)
(test (and (string=? ")(" (string #\) #\()) (equal? '(#\) #\() (string->list ")("))) #t)
(test (and (string=? "`)#" (string #\` #\) #\#)) (equal? '(#\` #\) #\#) (string->list "`)#"))) #t)
(test (and (string=? "##\\" (string #\# #\# #\\)) (equal? '(#\# #\# #\\) (string->list "##\\"))) #t)
(test (and (string=? "#\"(" (string #\# #\" #\()) (equal? '(#\# #\" #\() (string->list "#\"("))) #t)
(test (and (string=? "#.@" (string #\# #\. #\@)) (equal? '(#\# #\. #\@) (string->list "#.@"))) #t)
(test (and (string=? ",`@" (string #\, #\` #\@)) (equal? '(#\, #\` #\@) (string->list ",`@"))) #t)
(test (and (string=? "',@" (string #\' #\, #\@)) (equal? '(#\' #\, #\@) (string->list "',@"))) #t)
(test (and (string=? "\"#@" (string #\" #\# #\@)) (equal? '(#\" #\# #\@) (string->list "\"#@"))) #t)
(test (and (string=? "\")\"" (string #\" #\) #\")) (equal? '(#\" #\) #\") (string->list "\")\""))) #t)
(test (and (string=? ")#(" (string #\) #\# #\()) (equal? '(#\) #\# #\() (string->list ")#("))) #t)
(test (and (string=? "`(,@" (string #\` #\( #\, #\@)) (equal? '(#\` #\( #\, #\@) (string->list "`(,@"))) #t)
(test (and (string=? "`)#\"" (string #\` #\) #\# #\")) (equal? '(#\` #\) #\# #\") (string->list "`)#\""))) #t)
(test (and (string=? "#\"'#" (string #\# #\" #\' #\#)) (equal? '(#\# #\" #\' #\#) (string->list "#\"'#"))) #t)
(test (and (string=? "#(@\\" (string #\# #\( #\@ #\\)) (equal? '(#\# #\( #\@ #\\) (string->list "#(@\\"))) #t)
(test (and (string=? "#(\\\\" (string #\# #\( #\\ #\\)) (equal? '(#\# #\( #\\ #\\) (string->list "#(\\\\"))) #t)
(test (and (string=? ",,.@" (string #\, #\, #\. #\@)) (equal? '(#\, #\, #\. #\@) (string->list ",,.@"))) #t)
(test (and (string=? ",@`\"" (string #\, #\@ #\` #\")) (equal? '(#\, #\@ #\` #\") (string->list ",@`\""))) #t)
(test (and (string=? "\"'\")" (string #\" #\' #\" #\))) (equal? '(#\" #\' #\" #\)) (string->list "\"'\")"))) #t)
(test (and (string=? "\")#\"" (string #\" #\) #\# #\")) (equal? '(#\" #\) #\# #\") (string->list "\")#\""))) #t)
(test (and (string=? "(\\`)" (string #\( #\\ #\` #\))) (equal? '(#\( #\\ #\` #\)) (string->list "(\\`)"))) #t)
(test (and (string=? "))\"'" (string #\) #\) #\" #\')) (equal? '(#\) #\) #\" #\') (string->list "))\"'"))) #t)
(test (and (string=? "\\,\\\"" (string #\\ #\, #\\ #\")) (equal? '(#\\ #\, #\\ #\") (string->list "\\,\\\""))) #t)
(test (and (string=? "\\\"`\"" (string #\\ #\" #\` #\")) (equal? '(#\\ #\" #\` #\") (string->list "\\\"`\""))) #t)
(test (and (string=? "\\\\#\"" (string #\\ #\\ #\# #\")) (equal? '(#\\ #\\ #\# #\") (string->list "\\\\#\""))) #t)

(test (string->list "" 0 10) 'error)
(test (string->list "1" 0 2) 'error)
(test (string->list "" 0 0) ())
(test (string->list "1" 1) ())
(test (string->list "1" 0) '(#\1))
(test (string->list "" #\null) 'error)
(test (string->list "" 0 #\null) 'error)
(test (string->list "" -1) 'error)
(test (string->list "1" -1) 'error)
(test (string->list "1" 0 -1) 'error)
(test (string->list "1" -2 -1) 'error)
(test (string->list "1" most-negative-fixnum) 'error)
(test (string->list "1" 2) 'error)

(for-each
 (lambda (arg)
   (test (string->list "012345" arg) 'error)
   (test (string->list "012345" 1 arg) 'error))
 (list #\a "hi" () (list 1) '(1 . 2) 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (string->list "12345" 0) '(#\1 #\2 #\3 #\4 #\5))
(test (string->list "12345" 0 5) '(#\1 #\2 #\3 #\4 #\5))
(test (string->list "12345" 5 5) ())
(test (string->list "12345" 4 5) '(#\5))
(test (string->list "12345" 2 4) '(#\3 #\4))
(test (string->list "12345" 2 1) 'error)
(test (string->list "12345" 2 3 4) 'error)
(test (string->list (make-string 3 #\null) 2 3) '(#\null))



;;; --------------------------------------------------------------------------------
;;; char-position
;;; string-position

(test (char-position) 'error)
(test (char-position #\a) 'error)
(test (char-position #\a "abc" #\0) 'error)
(test (char-position #\a "abc" 0 1) 'error)
(test (string-position) 'error)
(test (string-position #\a) 'error)
(test (string-position "a" "abc" #\0) 'error)
(test (string-position "a" "abc" 0 1) 'error)

(for-each
 (lambda (arg)
   (test (string-position arg "abc") 'error)
   (test (char-position arg "abc") 'error)
   (test (string-position "a" arg) 'error)
   (test (char-position #\a arg) 'error)
   (test (string-position "a" "abc" arg) 'error)
   (test (char-position #\a "abc" arg) 'error))
 (list () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 -1 most-negative-fixnum 1.0+1.0i :hi (if #f #f) (lambda (a) (+ a 1))))
(test (char-position #\a "abc" most-positive-fixnum) #f)
(test (char-position "a" "abc" most-positive-fixnum) #f)
(test (string-position "a" "abc" most-positive-fixnum) #f)

(test (char-position #\b "abc") 1)
(test (char-position #\b "abc" 0) 1)
(test (char-position #\b "abc" 1) 1)
(test (char-position "b" "abc") 1)
(test (char-position "b" "abc" 1) 1)
(test (char-position "b" "abc") 1)
(test (string-position "b" "abc") 1)
(test (string-position "b" "abc" 1) 1)
(test (string-position "b" "abc" 2) #f)
(test (string-position "b" "abc" 3) #f)
(test (char-position "b" "abc" 2) #f)
(test (char-position "b" "abc" 3) #f)
(test (char-position #\b "abc" 2) #f)
(test (char-position #\b "abc" 3) #f)
(test (char-position "ab" "abcd") 0)
(test (char-position "ab" "ffbcd") 2)
(test (char-position "ab" "ffacd") 2)
(test (string-position "ab" "ffacd") #f)
(test (string-position "ab" "ffabd") 2)
(test (string-position "ab" "ffabab" 2) 2)
(test (string-position "ab" "ffabab" 3) 4)
(test (string-position "ab" "ffabab" 4) 4)
(test (string-position "ab" "ffabab" 5) #f)
(test (string-position "abc" "ab") #f)
(test (string-position "abc" "") #f)
(test (string-position "" "") #f)
(test (char-position "\"" "a") #f)
(test (char-position "\"" "a\"b") 1)
(test (char-position #\" "a\"b") 1)
(test (string-position "\"hiho\"" "hiho") #f)
(test (string-position "\"hiho\"" "\"\"hiho\"") 1)

(test (string-position "" "a") #f) ; this is a deliberate choice in s7.c
(test (char-position "" "a") #f)
(test (char-position #\null "a") 1)  ; ??
(test (char-position #\null "") #f)  ; ??
(test (string-position (string #\null) "a") 0) ; ??
(test (string-position (string #\null) "") #f) ; ??
(test (char-position #\null (string #\null)) 0) ; ??
(test (char-position #\null (string #\a #\null #\n)) 1)
(test (char-position "" (string #\a #\null #\n)) #f)
;(test (char-position #\n (string #\a #\null #\n)) 2)   ; ?? returns #f due to assumption of C-style strings
;(test (char-position "n" (string #\a #\null #\n)) 1)   ; oops!
;(test (string-position "n" (string #\a #\null #\n)) 2) ; oops!
(test (char-position "" (string #\a #\n)) #f)
(test (char-position #(1) "asdasd" 63) 'error)
(test (let ((i 0)) (sort! (vector 3 2 4 5 1) (lambda (a b) (+ (char-position #\xff abs (+ i 1))) (> a b)))) 'error)

;; if "" as string-pos first, -> #f so same for char-pos, even if string contains a null

(let ()
  ;; actually more of a string-append/temp substring test
  (define (fixit str)
    (let ((pos (char-position #\& str)))
      (if (not pos)
	  str
	  (string-append (substring str 0 pos)
			 (let ((epos (char-position #\; str pos)))
			   (let ((substr (substring str (+ pos 1) epos)))
			     (let ((replacement (cond ((string=? substr "gt") ">")
						      ((string=? substr "lt") "<")
						      ((string=? substr "mdash") "-")
						      (else (format #t "unknown: ~A~%" substr)))))
			       (string-append replacement
					      (fixit (substring str (+ epos 1)))))))))))
  (test (fixit "(let ((f (hz-&gt;radians 100)) (g (hz-&gt;radians 200))) (&lt; f g))")
	"(let ((f (hz->radians 100)) (g (hz->radians 200))) (< f g))"))

;;; opt bug
(test (apply char-position '(#\a #u() #f)) 'error)
(test (let () (define (f1) (do ((i 0 (+ i 1))) ((= i 1) (char-position #\a #u() #f)) (char-position #\a #u() #f))) (f1)) 'error)




;;; --------------------------------------------------------------------------------
;;; symbol->string
;;; string->symbol
;;; symbol

(test (symbol->string 'hi) "hi")
(test (string->symbol (symbol->string 'hi)) 'hi)
(test (eq? (string->symbol "hi") 'hi) #t)
(test (eq? (string->symbol "hi") (string->symbol "hi")) #t)

(test (string->symbol "hi") 'hi)

(test (let ((str (symbol->string 'hi)))
	(catch #t (lambda () (string-set! str 1 #\x)) (lambda args 'error)) ; can be disallowed
	(symbol->string 'hi))
      "hi")

(test (symbol->string 'sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789)
      "sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789")
(test (string->symbol "sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789")
      'sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789)
(test (let ((sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789 32))
	(+ sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789 1))
      33)

(test (symbol->string (string->symbol "hi there")) "hi there")
(test (symbol->string (string->symbol "Hi There")) "Hi There")
(test (symbol->string (string->symbol "HI THERE")) "HI THERE")
(test (symbol->string (string->symbol "")) 'error) ; this fluctuates
(test (symbol? (string->symbol "(weird name for a symbol!)")) #t)
(test (symbol->string (string->symbol "()")) "()")
(test (symbol->string (string->symbol (string #\"))) "\"")
(test (symbol->string 'quote) "quote")
(test (symbol->string if) 'error)
(test (symbol->string quote) 'error)

(test (symbol? (string->symbol "0")) #t)
(test (symbol? (symbol "0")) #t)
(test (symbol? (symbol ".")) #t) ; hmmm
(test (let () (define |.| 1) (+ |.| 2)) 3)
(test (string->symbol "0e") '0e)
(test (string->symbol "1+") '1+)
(test (symbol? (string->symbol "1+i")) #t)
(test (string->symbol ":0") ':0)
(test (symbol? (string->symbol " hi")) #t)
(test (symbol? (string->symbol "hi ")) #t)
(test (keyword? (string->symbol ":asdf")) #t)

(test (reinvert 12 string->symbol symbol->string "hiho") "hiho")

(test (symbol->string) 'error)
(test (string->symbol) 'error)
(test (symbol->string 'hi 'ho) 'error)
(test (string->symbol "hi" "ho") 'error)

(test (symbol? (string->symbol (string #\x (integer->char 255) #\x))) #t)
(test (symbol? (string->symbol (string #\x (integer->char 8) #\x))) #t)
(test (symbol? (string->symbol (string #\x (integer->char 128) #\x))) #t)
(test (symbol? (string->symbol (string #\x (integer->char 200) #\x))) #t)
(test (symbol? (string->symbol (string #\x (integer->char 255) #\x))) #t)
(test (symbol? (string->symbol (string #\x (integer->char 20) #\x))) #t)
(test (symbol? (string->symbol (string #\x (integer->char 2) #\x))) #t)
(test (symbol? (string->symbol (string #\x (integer->char 7) #\x))) #t)
(test (symbol? (string->symbol (string #\x (integer->char 17) #\x))) #t)
(test (symbol? (string->symbol (string #\x (integer->char 170) #\x))) #t)
(test (symbol? (string->symbol (string #\x (integer->char 0) #\x))) #t)       ; but the symbol's name here is "x"
;(test (eq? (string->symbol (string #\x (integer->char 0) #\x)) 'x) #t)        ;   hmmm...
(test (symbol? (string->symbol (string #\x #\y (integer->char 127) #\z))) #t) ; xy(backspace)z

(test (symbol "abc" ()) 'error)
(test (symbol () "abc") 'error)
(test (symbol "a" #(#\a)) 'error)
(test (symbol "a" "" "b") 'ab)
(test (symbol "" ()) 'error)

(test (symbol? (string->symbol (string #\; #\" #\)))) #t)
(test (let (((symbol ";")) 3) (symbol ";")) 'error)
(test ((lambda () (do ((__i__ 0 (+ __i__ 1))) ((= __i__ 1) 3) (let (((symbol ";")) 3) (symbol ";"))))) 'error)
(test (symbol "") 'error)
(test (symbol "" "") 'error)
(test (symbol "" "" "") 'error)
(test (symbol (string #\null) "ho") (symbol "\x00;ho"))
(test (let () (define (func) (symbol (string #\null) "ho")) (define (hi) (func) (func)) (hi) (hi)) (symbol "\x00;ho"))
(test (symbol "a" (string #\null) "b") (symbol "a\x00;b"))

(test (object->string (symbol "(ab)")) "(symbol \"(ab)\")")
(test (object->string (symbol "(ab")) "(symbol \"(ab\")")
(test (object->string (symbol "()")) "(symbol \"()\")")
(test (object->string (symbol (string #\return))) "(symbol \"\\r\")")
(test (object->string (symbol (string #\return #\a #\tab))) "(symbol \"\\ra\\t\")")
(test (object->string (symbol (string #\return #\tab))) "(symbol \"\\r\\t\")")
(test (object->string (symbol (string #\a #\b #\return #\tab))) "(symbol \"ab\\r\\t\")")
(test (object->string (symbol (string #\return #\tab #\a #\b))) "(symbol \"\\r\\tab\")")
(test (object->string (symbol (string #\return #\tab #\a #\null))) "(symbol \"\\r\\ta\\x00;\")")

(test (symbol->string (openlet (inlet 'symbol->string (lambda (s) "a symbol")))) "a symbol")
(test (let () (apply define (list (symbol "#g")) (list 1)) (eval (list (symbol "#g")))) 1)

(for-each
 (lambda (arg)
   (test (symbol->string arg) 'error))
 (list #\a 1 "hi" () (list 1) '(1 . 2) #f (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (string->symbol arg) 'error)
   (test (symbol arg) 'error)
   (test (symbol "a" arg) 'error))
 (list #\a 1 () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (symbol? (string->symbol (string arg))) #t)
   (test (symbol? (symbol (string arg))) #t))
 (list #\; #\, #\. #\) #\( #\" #\' #\` #\x33 #\xff #\x7f #\# #\]))

(test (symbol) 'error)
(test (symbol "hi" "ho") 'hiho)

(let ()
  (define-macro (string-case selector . clauses)
    `(case (symbol ,selector)
       ,@(map (lambda (clause)
		(if (pair? (car clause))
		    `(,(map symbol (car clause)) ,@(cdr clause))
		    clause))
	      clauses)))

  (test (let ((str "hi"))
	  (string-case str
            (("hi" "ho") 1 2 3)
	    (("hiho") 2)
	    (else 4)))
	3))

(let ()
  (apply define (list (symbol "(#)") 3))
  (test (eval (symbol "(#)") (curlet)) 3))

(let ()
  (define (immutabl obj) (string->symbol (object->string obj :readable)))
  (define (symbol->object sym) (eval-string (symbol->string sym)))
  (test (symbol->object (immutabl (list 1 2 3))) (list 1 2 3))
  (test (symbol->object (immutabl "hi")) "hi"))



;;; --------------------------------------------------------------------------------
;;; symbol->value
;;; symbol->dynamic-value

(let ((sym 0))
  (test (symbol->value 'sym) 0)
  (test (symbol->dynamic-value 'sym) 0)
  (for-each
   (lambda (arg)
     (set! sym arg)
     (test (symbol->value 'sym) arg)
     (test (symbol->dynamic-value 'sym) arg))
   (list #\a 1 () (list 1) '(1 . 2) #f (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand (log 0)
	 3.14 3/4 1.0+1.0i #t (if #f #f) #<eof> (lambda (a) (+ a 1)))))

(for-each
 (lambda (arg)
   (test (symbol->value arg) 'error)
   (test (symbol->value 'abs arg) 'error)
   (test (symbol->dynamic-value arg) 'error)
   (test (symbol->dynamic-value 'abs arg) 'error))
 (list #\a 1 () (list 1) "hi" '(1 . 2) #f (make-vector 3) abs _ht_ _undef_ _null_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t (if #f #f) #<eof>))

(test (symbol->value) 'error)
(test (symbol->value 'hi 'ho) 'error)
(test (symbol->dynamic-value) 'error)
(test (symbol->dynamic-value 'hi 'ho) 'error)

(test (symbol->value 'abs (unlet)) abs)
(test (symbol->value 'abs (rootlet)) abs)
(test (symbol->value 'lambda) lambda)
(test (symbol->value 'do) do)
(test (symbol->value do) 'error)
(test (symbol->value 'macroexpand) macroexpand)
(test (symbol->value 'quasiquote) quasiquote)
(test (symbol->value 'else) else)
(test (symbol->value :hi) :hi)
(test (symbol->value hi:) hi:)

(test (symbol->dynamic-value 'lambda) lambda)
(test (symbol->dynamic-value 'do) do)
(test (symbol->dynamic-value do) 'error)
(test (symbol->dynamic-value 'macroexpand) macroexpand)
(test (symbol->dynamic-value 'quasiquote) quasiquote)
(test (symbol->dynamic-value 'else) else)
(test (symbol->dynamic-value :hi) :hi)
(test (symbol->dynamic-value hi:) hi:)

(test (symbol->value '#<eof>) 'error) ; because it's not a symbol:
(test (symbol? '#<eof>) #f)
(test (let ((a1 32)) (let () (symbol->value 'a1 (curlet)))) 32)
(test (let ((a1 32)) (let ((a1 0)) (symbol->value 'a1 (curlet)))) 0)
(test (let ((a1 32)) (let ((a1 0)) (symbol->value 'b1 (curlet)))) #<undefined>)
(test (symbol->value 'abs ()) 'error)
(test (let ((a1 (let ((b1 32)) (lambda () b1)))) (symbol->value 'b1 (funclet a1))) 32)
(test (let ((x #f)) (set! x (let ((a1 (let ((b1 32)) (lambda () b1)))) a1)) (symbol->value 'b1 (funclet x))) 32)
(test (symbol->value 'if) if)
(test (symbol->value if) 'error)
(test ((define (hi a) (+ a 1)) 2) 3)
(test ((define-macro (hi a) `(+ ,a 1)) 2) 3)
(test (let ((mac (define-macro (hi a) `(+ ,a 1)))) (mac 3)) 4)

(test (eq? #_abs (symbol->value 'abs 'unlet)) #t)
(test (eq? #_lambda (symbol->value 'lambda 'unlet)) #t)
(test (let ((b 2)) (let ((e (curlet))) (let ((a 1)) (symbol->value 'a e)))) #<undefined>)
(test (let ((a 2)) (let ((e (curlet))) (let ((a 1)) (symbol->value 'a e)))) 2)
(test (let ((a 2)) (let ((e (curlet))) (let ((a 1)) (symbol->value 'a)))) 1)

(let ()
  (define *ds* 0)
  (define (get-ds) (list *ds* (symbol->dynamic-value '*ds*)))
  (test (get-ds) '(0 0))
  (let ((*ds* 32))
    (test (values (get-ds)) '(0 32)))
  (let ((*ds* 3))
    (define (gds) (list *ds* (symbol->dynamic-value '*ds*)))
    (test (list (get-ds) (gds)) '((0 3) (3 3)))
    (let ((*ds* 123))
      (test (list (get-ds) (gds)) '((0 123) (3 123)))))
  (let ((*ds* 3))
    (define (gds) (list *ds* (symbol->dynamic-value '*ds*)))
    (let ((*ds* 123))
      (set! *ds* 321)
      (test (list (get-ds) (gds)) '((0 321) (3 321))))))

(test (symbol->dynamic-value 'asdasfasdasfg) #<undefined>)

(let ((x 32))
  (define (gx) (symbol->dynamic-value 'x))
  (let ((x 12))
    (test (values (gx)) 12)))

(let ((x "hi")
      (y 0)
      (z '(1 2 3)))
  (define (gx) (+ (symbol->dynamic-value 'x) (symbol->dynamic-value 'z)))
  (let ((x 32)
	(z (+ 123 (car z))))
    (test (values (gx)) 156)))

(let ((x 32))
  (define (gx) (symbol->dynamic-value 'x))
  (let ((x 100))
    (let ((x 12))
      (test (values (gx)) 12))))

(let ((x 32))
  (define (gx) ; return both bindings of 'x
    (list x (symbol->value 'x) (symbol->dynamic-value 'x)))
  (let ((x 100))
    (let ((x 12))
      (test (values (gx)) '(32 32 12)))))

(test (let () (define (func) (int-vector-set! (subvector #i2d((1 2) (3 4)) 0 4) 1 2 (arity (symbol->dynamic-value (gensym "g_123"))))) (define (hi) (func)) (hi)) 'error)
(test (let () (define (func) (list (string-append) 42 (integer->char 255) (symbol->dynamic-value (gensym)))) (define (hi) (func)) (hi)) (list "" 42 #\xff #<undefined>))

(let ((bindings ()))
  ;; taken from the MIT_Scheme documentation (changing fluid-let to let)

  (define (write-line v)
    (set! bindings (cons v bindings)))

  (define (complicated-dynamic-binding)
    (let ((variable 1)
	  (inside-continuation #f))
      (write-line variable)
      (call-with-current-continuation
       (lambda (outside-continuation)
	 (let ((variable 2))
	   (write-line variable)
	   (set! variable 3)
	   (call-with-current-continuation
	    (lambda (k)
	      (set! inside-continuation k)
	      (outside-continuation #t)))
	   (write-line variable)
	   (set! inside-continuation #f))))
      (write-line variable)
      (if inside-continuation
	  (begin
	    (set! variable 4)
	    (inside-continuation #f)))))

  (complicated-dynamic-binding)
  (test (reverse bindings) '(1 2 1 3 4)))

;;; (define (func x) (call-with-output-file "/dev/null" (symbol->dynamic-value (*function* (curlet)))))
;;;    gets either stack overflow or error: open-output-file: Too many open files "/dev/null"
;;;    because (symbol->dynamic-value (*function* (curlet))) is the calling function (func) so we have
;;;    an infinite recursion.

(when with-block
  (test (defined? 'subsequence (block) #t) #t)
  (test ((block) 'subsequence) subblock)
  (test (let-ref (block) 'subsequence) subblock)
  (test (with-let (block) subsequence) subblock)
  (test (symbol->value 'subsequence (block)) subblock)
  (let () (define (func) (do () (#t (with-let (block 1.0 2.0 3.0) (subsequence abs abs abs))))) (test (func) 'error))
  (let () (define (func) (do () (#t (with-let (block 1.0 2.0 3.0) (subsequence 0-i abs abs))))) (test (func) 'error))
  (let () (define (func) (do () (#t (with-let (block 1.0 2.0 3.0) (subsequence abs 0-i abs))))) (test (func) 'error))
  (let () (define (func) (do () (#t (with-let (block 1.0 2.0 3.0) (subsequence abs abs 0-i))))) (test (func) 'error))
  (let () (define (func) (do () (#t (with-let (block 1.0 2.0 3.0) (subsequence 0-i 0-i 0-i))))) (test (func) 'error))
  (let () (define (func) (do () (#t (with-let (block 1.0 2.0 3.0) (subsequence 0-i 0-i abs))))) (test (func) 'error))
  (let () (define (func) (do () (#t (with-let (block 1.0 2.0 3.0) (subsequence 0-i abs 0-i))))) (test (func) 'error))
  (let () (define (func) (do () (#t (with-let (block 1.0 2.0 3.0) (subsequence abs 0-i 0-i))))) (test (func) 'error))
)


;;; --------------------------------------------------------------------------------
;;; BYTE-VECTORS
;;; --------------------------------------------------------------------------------

(let ((bv #u(1 0 3)))
  (test bv #u(1 0 3))
  (test (object->string bv) "#u(1 0 3)")
  (test (equal? bv #u(1 0 3)) #t)
  (test (eq? bv bv) #t)
  (test (eqv? bv bv) #t)
  (test (equal? (byte-vector 1 0 3) #u(1 0 3)) #t)
  (test (byte-vector? bv) #t)
  (test (equal? (make-byte-vector 3 0) #u(0 0 0)) #t)
  (test (string-ref #u(64 65 66) 1) 'error)
  (test (byte-vector-ref #u(64 65 66) 1) 65)
  (test (let ((nbv (copy bv))) (equal? nbv bv)) #t)
  (test (let ((rbv (reverse bv))) (equal? rbv #u(3 0 1))) #t)
  (test (length bv) 3)
  )

(test (eval-string "#u(-1)") 'error)
(test (eval-string "#u(1.0)") 'error)
(test (eval-string "#u(3/2)") 'error)
(test (eval-string "#u(1+i)") 'error)
(test (eval-string "#u((32))") 'error)
(test (eval-string "#u(#\\a)") 'error)
(test (eval-string "#u(256)") 'error)
(test (eval-string "#u(1/0)") 'error)
(test (eval-string "#u(9223372036854775807)") 'error)
(test (eval-string "#u(-9223372036854775808)") 'error)
(test #u(#b11 #x8) #u(3 8))
(test (eval-string "#u(1 2 . 3)") 'error)

(test #u(255) (byte-vector 255))
(test (byte-vector 256) 'error)
(test (byte-vector -1) 'error)
(test (object->string #u()) "#u()")
(test (object->string #u(255)) "#u(255)")
(test (object->string #u(255 255)) "#u(255 255)")
(test (object->string #u(128)) "#u(128)")
(test (object->string #u(128 128)) "#u(128 128)")

(test (length #u(0)) 1)
(test (length #u(0 0)) 2)
(test (length #u()) 0)
(test (length (byte-vector)) 0)
(test (byte-vector? #u()) #t)
(test (equal? (let ((bv #u(1 0 3))) (set! (bv 2) 64) bv) #u(1 0 64)) #t)
(test (let ((bv #u(1 0 3))) (map values bv)) '(1 0 3))
(test (let ((bv #u(1 0 3)) (lst ())) (for-each (lambda (x) (set! lst (cons x lst))) bv) lst) '(3 0 1))
(test (let ((bv #u(1 2 3))) (bv 1)) 2)
(test (let ((bv #u(1 2 3))) (reverse bv)) #u(3 2 1))
(test (let ((bv #u(1 2 3))) (object->string (reverse bv))) "#u(3 2 1)")
(test (let ((bv #u(1 2 3))) (copy bv)) #u(1 2 3))
(test (#u(1 2 3) 2) 3)
(test (let ((v #u(0 1 2))) (let ((v1 (reverse! v))) (eq? v v1))) #t)
(test (let ((v #u(0 1 2))) (reverse! v)) #u(2 1 0))
;; should (vector? #u(1 2)) be #t?
(test (format #f "~{~A ~}" (byte-vector 255 0)) "255 0 ")

;;; string->byte-vector -- why is this needed? -- why not use copy instead?
(test (byte-vector? (string->byte-vector (string #\0))) #t)
(test (byte-vector? (string->byte-vector "")) #t)
(test (byte-vector? (string->byte-vector "1230")) #t)
(test (byte-vector? (string->byte-vector (string->byte-vector (string #\0)))) 'error)
(test (byte-vector? (string->byte-vector (string))) #t)
(test (byte-vector? (string->byte-vector #u(1 2))) 'error)
(test (byte-vector? (string->byte-vector #u())) 'error)
(test (byte-vector? (string->byte-vector #(1 2))) 'error)
(for-each
 (lambda (arg)
   (test (string->byte-vector arg) 'error)
   (test (byte-vector? arg) #f))
 (list #\a () (list 1) '(1 . 2) #f (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t (if #f #f) #<eof> (lambda (a) (+ a 1))))

(test (string->byte-vector #x010203) 'error)
(test (string->byte-vector (immutable! "123")) #u(49 50 51))
(let ((str (immutable! "123")))
  (string->byte-vector str)
  (test str "123")
  (test (immutable? str) #t))

(test (let ((str "123")) (string->byte-vector str) (byte-vector? str)) #f)
(test (let ((str (string #\a))) (string->byte-vector str) (byte-vector? str)) #f)
(test (let () (define (func) (string->byte-vector "ho")) (define (hi) (func) (func)) (hi) (hi)) #u(104 111))
(test (let () (define (func) (byte-vector->string #u(65))) (define (hi) (func) (func)) (hi) (hi)) "A")

;;; byte-vector->string
(test (string? (byte-vector->string #u(0))) #t)
(test (string? (byte-vector->string #u())) #t)
(test (string? (byte-vector->string #u(1 2 3))) #t)
(test (string? (byte-vector->string (string->byte-vector (string #\0)))) #t)
(test (string? (byte-vector->string (byte-vector))) #t)
(test (string? (byte-vector->string "asd")) 'error)
(test (string? (byte-vector->string "")) 'error)
(test (string? (byte-vector->string #(1 2))) 'error)
(for-each
 (lambda (arg)
   (test (byte-vector->string arg) 'error))
 (list #\a () (list 1) '(1 . 2) #f (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t (if #f #f) #<eof> (lambda (a) (+ a 1))))

(test (byte-vector->string #x010203) 'error)
(test (byte-vector->string (immutable! #u(49 50 51))) "123")
(let ((bv (immutable! #u(49 50 51))))
  (byte-vector->string bv)
  (test bv #u(49 50 51))
  (test (immutable? bv) #t))


;;; make-byte-vector
(test (equal? (make-byte-vector 0) #u()) #t)
(test (equal? (make-byte-vector 0 32) #u()) #t)
(test (equal? (make-byte-vector 1 32) #u(32)) #t)
(test (make-byte-vector 0 -32) 'error)
(test (make-byte-vector 1 -32) 'error)
(test (make-byte-vector 1 256) 'error)
(test (make-byte-vector 1 3.0) 'error)
(for-each
 (lambda (arg)
   (if (not (eq? 'error (catch #t (lambda () (make-byte-vector arg)) (lambda args 'error))))
       (format *stderr* ";(make-byte-vector ~S) returns a byte-vector?\n" arg))
   (if (not (eq? 'error (catch #t (lambda () (make-byte-vector 1 arg)) (lambda args 'error))))
       (format *stderr* ";(make-byte-vector 1 ~S) returns a byte-vector?\n" arg)))
 (list #\a () '(1 . 2) #f (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t (if #f #f) #<eof> (lambda (a) (+ a 1))))

(test (let () (define (f) (do ((i 0 (+ i 1))) ((= i 1)) (make-byte-vector most-positive-fixnum 0))) (f)) 'error)

;;; byte-vector
(test (byte-vector) #u())
(test (byte-vector 32) (make-byte-vector 1 32))
(test (byte-vector 0 256) 'error)
(test (byte-vector -1) 'error)
(for-each
 (lambda (arg)
   (test (byte-vector arg) 'error))
 (list #\a () (list 1)  '(1 . 2) #f (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t (if #f #f) #<eof> (lambda (a) (+ a 1))))

(test (map append #u(0 1 2)) '(0 1 2))
(test (format #f "~{#x~X~| ~}" #u(49 50 51)) "#x31 #x32 #x33")
(test (format #f "~{~D~| ~}" (string->byte-vector "abcd")) "97 98 99 100")
(test (let ((lst ())) (for-each (lambda (c) (set! lst (cons c lst))) #u(90 91 92)) (reverse lst)) '(90 91 92))
(test (integer? (#u(1 2 3) 0)) #t)
(test (integer? ((string->byte-vector "abc") 1)) #t)

(test ((vector (byte-vector 1)) 0 0) 1) ; i.e. not a character
(test (let () (define (f) (or (#_byte-vector 1) (#_documentation cons))) (f)) #u(1)) ; optimizer bug
#|
(test (byte-vector 0
		   (openlet
		    (inlet 'byte-vector (lambda args
					  (append #u(1) (apply #_byte-vector (cdr args))))))
		   2)
      #u(0 1 2))
|#
(let ((bv (byte-vector 0 1 2 3)))
  (fill! bv 4)
  (test bv #u(4 4 4 4))
  (fill! bv 1 1 3)
  (test bv #u(4 1 1 4))
  (let ((bv1 (copy bv)))
    (test bv1 #u(4 1 1 4))
    (fill! bv 1)
    (copy bv bv1)
    (test bv1 #u(1 1 1 1))
    (fill! bv 255)
    (copy bv bv1 1 3)
    (test bv1 #u(255 255 1 1)))) ; copy and fill do not interpret their indices in the same way (one is source, the other destination)

(test (equal? (byte-vector (char->integer #\a)) (string #\a)) #f)
(test (equivalent? (byte-vector (char->integer #\a)) (string #\a)) #f)

(test (byte-vector? (copy "12")) #f)
(test (byte-vector? (copy #u(0))) #t)
(test (byte-vector? (reverse (byte-vector 0 1))) #t)
(test (let ((v (byte-vector 0))) (fill! v #\a)) 'error)
(test (let ((v (byte-vector 0))) (fill! v 1) v) #u(1))
(test (byte-vector? (append #u(0 1) (byte-vector 2 3))) #t)

;;; should string->byte-vector insist on string (not bv) arg?  similarly for string-ref et al?

;;; byte-vector-ref
;;; byte-vector-set!

(test (let ((str "123")) (byte-vector-ref str 0)) 'error)
(test (let ((str "123")) (byte-vector-set! str 0 1)) 'error)
(test (let ((str "123")) (byte-vector-set! str 0 #\1)) 'error)
(test (let ((str (byte-vector 0 1 2))) (byte-vector-ref str 0)) 0)
(test (let ((str (byte-vector 0 1 2))) (char? (byte-vector-ref str 0))) #f)
(test (let ((str (byte-vector 0 1 2))) (byte-vector-set! str 0 1)) 1)
(test (let ((str (byte-vector 0 1 2))) (byte-vector-set! str 0 #\1)) 'error)
(test (byte-vector-ref #u(0 1 2)) 'error)
(test (byte-vector-ref) 'error)
(test (byte-vector-ref #u(0 1 2) 1 1) 'error)
(test (byte-vector-ref #u(0 1 2) -1) 'error)
(test (byte-vector-set! #u(0 1 2)) 'error)
(test (byte-vector-set! #u(0 1 2) 0) 'error)
(test (byte-vector-set!) 'error)
(test (byte-vector-set! #u(0 1 2) 1 1 2) 'error)
(test (byte-vector-set! #u(0 1 2) -1 1) 'error)
(test (let ((iv #i2d((0 1) (2 3)))) (define (func) (do ((i 0 (+ i 1))) ((= i 1)) (do ((j 0 (+ j 1))) ((= j 1)) (byte-vector-set! iv 1 0)))) (func)) 'error)

(for-each
 (lambda (arg)
   (test (byte-vector-ref arg 0) 'error)
   (test (byte-vector-set! arg 0 0) 'error)
   (test (byte-vector-ref #u(0 1 2) arg) 'error)
   (test (byte-vector-set! #u(0 1 2) arg 0) 'error)
   (test (byte-vector-set! #u(0 1 2) 0 arg) 'error)
   (test (let ((v #u(0 1 2))) (v arg)) 'error)
   (test (let ((v #u(0 1 2))) (set! (v arg) 0)) 'error))
 (list #\a () (list 1) "str" "" '(1 . 2) #f (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t (if #f #f) #<eof> (lambda (a) (+ a 1))))

(test (let ((v #u(0 1 2))) (v 1)) 1)
(test (let ((v (byte-vector 0 1 2))) (set! (v 1) 3) v) #u(0 3 2))

(test (let ((bv (byte-vector #\1))) bv) 'error)
(test (let ((bv (string 1))) bv) 'error)
(let ((bv #u(0 1))) (test (string->byte-vector (byte-vector->string bv)) bv))
(test (let ((bv (byte-vector 1))) (define (f) (do ((i 0 (+ i 1))) ((= i 1)) (byte-vector-set! bv 0 #\1))) (f)) 'error)
(test (let ((bc #u(0 1))) (set! (bc 0) #\1)) 'error)
(test (let ((bc #u(0 1))) (byte-vector-set! bc 0 #\1)) 'error)
(test (let ((bc #u(0 1))) (string-set! bc 0 #\1)) 'error)
(test (let ((bc "123")) (set! (bc 0) 1)) 'error)
(test (let ((bc "123")) (string-set! bc 0 1)) 'error)
(test (let ((bc "123")) (byte-vector-set! bc 0 1)) 'error)
(test (let ((bv (byte-vector 1))) (define (f) (do ((i 0 (+ i 1))) ((= i 1)) (set! (bv 0) #\1))) (f)) 'error)
(test (let ((bv (string #\1))) (define (f) (do ((i 0 (+ i 1))) ((= i 1)) (set! (bv 0) 1))) (f)) 'error)
(test (let ((bv (string #\1))) (define (f) (do ((i 0 (+ i 1))) ((= i 1)) (string-set! bv 0 1))) (f)) 'error)
(test (let () (define (hi) (do ((i 0 (+ i 1))) ((= i 1)) (char-alphabetic? (string-ref #u(0 1) 1)))) (hi)) 'error)

(let ((fv (byte-vector 0 1)))
  (test (byte-vector-ref fv 0 0) 'error)
  (test (vector-ref fv 0 0) 'error)
  (test (fv 0 0) 'error))
(let ((fv #i2d((1 2) (3 4))))
  (test (byte-vector-ref fv 0 0 0) 'error)
  (test (byte-vector-ref fv 0 0 0 0) 'error)
  (test (vector-ref fv 0 0 0) 'error)
  (test (fv 0 0 0) 'error))

(let ((bv (byte-vector 0 1 2)))
  (test (byte-vector-ref bv bv) 'error)
  (test (bv bv)	'error)
  (test (call-with-input-file (apply bv bv (list))) 'error))

(let ((bv (make-byte-vector '(2 3))))
  (test (vector-length bv) 6)
  (test (length bv) 6)
  (test (vector-dimensions bv) '(2 3))
  (byte-vector-set! bv 0 0 32)
  (test (byte-vector-ref bv 0 0) 32)
  (set! (bv 0 1) 33)
  (test (bv 0 1) 33)
  (fill! bv 1)
  (test bv (make-byte-vector '(2 3) 1))
  (test (bv 0) #u(1 1 1))
  (set! (bv 0 0) 2)
  (test (copy bv (make-byte-vector 4)) #u(2 1 1 1)))

(let ((bv #u(1 2 3)))
  (test (fill! bv #\a) 'error)
  (test (vector-fill! bv #\a) 'error)
  (test (fill! bv -1) 'error)
  (test (vector-fill! bv 256) 'error)
  (test (bv 3) 'error)
  (test (vector-ref bv 3) 'error)
  (test (vector-ref bv 0) 1)
  (vector-set! bv 1 32)
  (test (bv 1) 32)
  (test (copy #i(1 2 321) (make-byte-vector 3)) 'error))

(test (copy #i(31 32 321) (make-string 3)) 'error)
(test (copy #(31 32 321) (make-byte-vector 3)) 'error)
(test (copy '(31 32 321) (make-byte-vector 3)) 'error)
(test (copy #(31 32 321) (make-string 3)) 'error)
(test (copy '(31 32 321) (make-string 3)) 'error)

(let ((bv (make-byte-vector '(2 3) 95)))
  (test (byte-vector->string bv) "______")
  (test (vector->list bv) '(95 95 95 95 95 95)))

(let ((bv #u(0 1 2 3 4 5)))
  (test (subvector bv 0 2) #u(0 1))
  (test (subvector bv 2 4) #u(2 3))
  (test (subvector bv 0 4 '(2 2)) (let ((b (make-byte-vector '(2 2)))) (copy bv b) b))
  (test (vector-append bv #i(6 7)) #u(0 1 2 3 4 5 6 7))
  (test (append bv #i(6 7)) #u(0 1 2 3 4 5 6 7)))

(test ((make-byte-vector '(2 3) 1) 1) #u(1 1 1))
(test ((make-byte-vector '(2 4) 1) 1) #u(1 1 1 1))
(test ((make-byte-vector '(2 3) 1) 0) #u(1 1 1))
(test ((make-byte-vector '(2 4) 12) 0) #u(12 12 12 12))
(test ((make-byte-vector '(3 2) 12) 1) #u(12 12))
(test ((make-byte-vector '(3 2) 12) 2) #u(12 12))
(test (byte-vector-ref (make-byte-vector '(3 2) 12) 2) #u(12 12))

(test (let ((bv (make-byte-vector '(2 3) 0))) (copy #u(1 2 3 4 5 6) bv) (subvector bv 2 6)) #u(3 4 5 6))
(test (object->string #u2d((1 2) (3 4))) "#u2d((1 2) (3 4))")
(test (object->string #u8(1 2)) "#u(1 2)")
(let ((x #u2d((1 2) (3 4))) (lst ()))
  (test (do ((iter (make-iterator x)) (lst ())) ((iterator-at-end? iter) (reverse lst)) (set! lst (cons (iter) lst))) '(1 2 3 4 #<eof>)))

(test (equal? #i(1) #u(1)) #t)
(test (equivalent? #i(1) #u(1)) #t)
(test (equal? #u(1) #i(1)) #t)
(test (equivalent? #u(1) #i(1)) #t)
(test (equal? #u(1 2 3 4) #u2d((1 2) (3 4))) #f)
(test (equal? #i(1 2 3 4) #i2d((1 2) (3 4))) #f)
(test (equivalent? #u(1 2 3 4) #u2d((1 2) (3 4))) #f)
(test (subvector #u(1 2 3 4) 0 3) #u(1 2 3))
(test (equivalent? #u2d((1 2) (3 4)) #i2d((1 2) (3 4))) #t)


;;; --------------------------------------------------------------------------------
;;; LISTS
;;; --------------------------------------------------------------------------------


;;; --------------------------------------------------------------------------------
;;; cons

(test (cons 'a ()) '(a))
(test (cons '(a) '(b c d)) '((a) b c d))
(test (cons "a" '(b c)) '("a" b c))
(test (cons 'a 3) '(a . 3))
(test (cons '(a b) 'c) '((a b) . c))
(test (cons () ()) '(()))
(test (cons () 1) '(() . 1))
(test (cons 1 2) '(1 . 2))
(test (cons 1 ()) '(1))
(test (cons () 2) '(() . 2))
(test (cons 1 (cons 2 (cons 3 (cons 4 ())))) '(1 2 3 4))
(test (cons 'a 'b) '(a . b))
(test (cons 'a (cons 'b (cons 'c ()))) '(a b c))
(test (cons 'a (list 'b 'c 'd)) '(a b c d))
(test (cons 'a (cons 'b (cons 'c 'd))) '(a b c . d))
(test '(a b c d e) '(a . (b . (c . (d . (e . ()))))))
(test (cons (cons 1 2) (cons 3 4)) '((1 . 2) 3 . 4))
(test (list (cons 1 2) (cons 3 4)) '((1 . 2) (3 . 4)))
(test (cons (cons 1 (cons 2 3)) 4) '((1 . (2 . 3)) . 4))
(test (cons (cons 1 (cons 2 ())) (cons 1 2)) '((1 2) . (1 . 2)))
(test (let ((lst (list 1 2))) (list (apply cons lst) lst)) '((1 . 2) (1 2)))
(test (let ((lst (list 1 2))) (list lst (apply cons lst))) '((1 2) (1 . 2)))
(test (cdadr (let ((lst (list 1 2))) (list (apply cons lst) lst))) '(2))
(test (cons '+ '=) '(+ . =))
(test (cons .(cadddr 10)) (cons cadddr 10))
(test (#_cons 1 2) '(1 . 2))
(test (cons 1 ()) '(
                      1
		       ))



;;; --------------------------------------------------------------------------------
;;; car

(test (car (list 1 2 3)) 1)
(test (car (cons 1 2)) 1)
(test (car (list 1)) 1)
(test (car '(1 2 3)) 1)
(test (car '(1)) 1)
(test (car '(1 . 2)) 1)
(test (car '((1 2) 3)) '(1 2))
(test (car '(((1 . 2) . 3) 4)) '((1 . 2) . 3))
(test (car (list (list) (list 1 2))) ())
(test (car '(a b c)) 'a)
(test (car '((a) b c d)) '(a))
(test (car (reverse (list 1 2 3 4))) 4)
(test (car (list 'a 'b 'c 'd 'e 'f 'g)) 'a)
(test (car '(a b c d e f g)) 'a)
(test (car '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '((((1 2 3) 4) 5) (6 7)))
(test (car '(a)) 'a)
(test (car '(1 ^ 2)) 1)
(test (car '(1 .. 2)) 1)
(test (car ''foo) 'quote)
(test (car '(1 2 . 3)) 1)
(test (car (cons 1 ())) 1)
(test (car (if #f #f)) 'error)
(test (car ()) 'error)
(test (car #(1 2)) 'error)
(test (car #(1 2)) 'error)
(let ((L (list 1 2 3))) (set! (car L) 32) (test L '(32 2 3)))
(let ((L (list 1 2 3))) (set! (#_car L) 12) (test L '(12 2 3)))

(for-each
 (lambda (arg)
   (if (not (equal? (car (cons arg ())) arg))
       (format #t ";(car '(~A)) returned ~A?~%" arg (car (cons arg ()))))
   (test (car arg) 'error))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand (log 0)
       3.14 3/4 1.0+1.0i #\f #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (reinvert 12 car (lambda (a) (cons a ())) '(1)) '(1))



;;; --------------------------------------------------------------------------------
;;; cdr

(test (cdr (list 1 2 3)) '(2 3))
(test (cdr (cons 1 2)) 2)
(test (cdr (list 1)) ())
(test (cdr '(1 2 3)) '(2 3))
(test (cdr '(1)) ())
(test (cdr '(1 . 2)) 2)
(test (cdr '((1 2) 3)) '(3))
(test (cdr '(((1 . 2) . 3) 4)) '(4))
(test (cdr (list (list) (list 1 2))) '((1 2)))
(test (cdr '(a b c)) '(b c))
(test (cdr '((a) b c d)) '(b c d))
(test (equal? (cdr (reverse (list 1 2 3 4))) 4) #f)
(test (equal? (cdr (list 'a 'b 'c 'd 'e 'f 'g)) 'a) #f)
(test (cdr '((((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f) g)) '(g))
(test (cdr '(a)) ())
(test (cdr '(a b c d e f g)) '(b c d e f g))
(test (cdr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '((((u v w) x) y) ((q w e) r) (a b c) e f g))
(test (cdr ''foo) '(foo))
(test (cdr (cons (cons 1 2) (cons 3 4))) '(3 . 4))
(test (cdr '(1 2 . 3)) '(2 . 3))
(test (cdr (if #f #f)) 'error)
(test (cdr ()) 'error)

(for-each
 (lambda (arg)
   (if (not (equal? (cdr (cons () arg)) arg))
       (format #t ";(cdr '(() ~A) -> ~A?~%" arg (cdr (cons () arg))))
   (test (cdr arg) 'error))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand (log 0)
       3.14 3/4 1.0+1.0i #\f #t :hi (if #f #f) (lambda (a) (+ a 1))))

(let* ((a (list 1 2 3))
       (b a))
  (set! (car a) (cadr a))
  (set! (cdr a) (cddr a))
  (test a (list 2 3))
  (test b a))

(define (cons-r a b n) (if (= 0 n) (cons a b) (cons (cons-r (+ a 1) (+ b 1) (- n 1)) (cons-r (- a 1) (- b 1) (- n 1)))))
(define (list-r a b n) (if (= 0 n) (list a b) (list (list-r (+ a 1) (+ b 1) (- n 1)) (list-r (- a 1) (- b 1) (- n 1)))))

(define lists (list (list 1 2 3)
		    (cons 1 2)
		    (list 1)
		    (list)
		    (list (list 1 2) (list 3 4))
		    (list (list 1 2) 3)
		    '(1 . 2)
		    '(a b c)
		    '((a) b (c))
		    '((1 2) (3 4))
		    '((1 2 3) (4 5 6) (7 8 9))
		    '(((1) (2) (3)) ((4) (5) (6)) ((7) (8) (9)))
		    '((((1 123) (2 124) (3 125) (4 126)) ((5) (6) (7) (8)) ((9) (10) (11) (12)) ((13) (14) (15) (16)))
		      (((21 127) (22 128) (23 129) (24 130)) ((25) (26) (27) (28)) ((29) (30) (31) (32)) ((33) (34) (35) (36)))
		      (((41 131) (42 132) (43 133) (44 134)) ((45) (46) (47) (48)) ((49) (50) (51) (52)) ((53) (54) (55) (56)))
		      (((61 135) (62 136) (63 137) (64 138)) ((65) (66) (67) (68)) ((69) (70) (71) (72)) ((73) (74) (75) (76)))
		      321)
		    (cons 1 (cons 2 (cons 3 4)))
		    (cons (cons 2 (cons 3 4)) 5)
		    (cons () 1)
		    (cons 1 ())
		    (cons () ())
		    (list 1 2 (cons 3 4) 5 (list (list 6) 7))
		    (cons-r 0 0 4)
		    (cons-r 0 0 5)
		    (cons-r 0 0 10)
		    (list-r 0 0 3)
		    (list-r 0 0 7)
		    (list-r 0 0 11)
		    ''a
		    ))


;;; --------------------------------------------------------------------------------
;;; cxr

(define (caar-1 x) (car (car x)))
(define (cadr-1 x) (car (cdr x)))
(define (cdar-1 x) (cdr (car x)))
(define (cddr-1 x) (cdr (cdr x)))
(define (caaar-1 x) (car (car (car x))))
(define (caadr-1 x) (car (car (cdr x))))
(define (cadar-1 x) (car (cdr (car x))))
(define (caddr-1 x) (car (cdr (cdr x))))
(define (cdaar-1 x) (cdr (car (car x))))
(define (cdadr-1 x) (cdr (car (cdr x))))
(define (cddar-1 x) (cdr (cdr (car x))))
(define (cdddr-1 x) (cdr (cdr (cdr x))))
(define (caaaar-1 x) (car (car (car (car x)))))
(define (caaadr-1 x) (car (car (car (cdr x)))))
(define (caadar-1 x) (car (car (cdr (car x)))))
(define (caaddr-1 x) (car (car (cdr (cdr x)))))
(define (cadaar-1 x) (car (cdr (car (car x)))))
(define (cadadr-1 x) (car (cdr (car (cdr x)))))
(define (caddar-1 x) (car (cdr (cdr (car x)))))
(define (cadddr-1 x) (car (cdr (cdr (cdr x)))))
(define (cdaaar-1 x) (cdr (car (car (car x)))))
(define (cdaadr-1 x) (cdr (car (car (cdr x)))))
(define (cdadar-1 x) (cdr (car (cdr (car x)))))
(define (cdaddr-1 x) (cdr (car (cdr (cdr x)))))
(define (cddaar-1 x) (cdr (cdr (car (car x)))))
(define (cddadr-1 x) (cdr (cdr (car (cdr x)))))
(define (cdddar-1 x) (cdr (cdr (cdr (car x)))))
(define (cddddr-1 x) (cdr (cdr (cdr (cdr x)))))

(for-each
 (lambda (name op1 op2)
   (for-each
    (lambda (lst)
      (let ((val1 (catch #t (lambda () (op1 lst)) (lambda args 'error)))
	    (val2 (catch #t (lambda () (op2 lst)) (lambda args 'error))))
	(if (not (equal? val1 val2))
	    (format #t ";(~A ~S) -> ~S, (~A-1): ~S?~%" name lst val1 name val2))))
    lists))
 (list 'caar 'cadr 'cdar 'cddr 'caaar 'caadr 'cadar 'cdaar 'caddr 'cdddr 'cdadr 'cddar
       'caaaar 'caaadr 'caadar 'cadaar 'caaddr 'cadddr 'cadadr 'caddar 'cdaaar
       'cdaadr 'cdadar 'cddaar 'cdaddr 'cddddr 'cddadr 'cdddar)

 (list caar cadr cdar cddr caaar caadr cadar cdaar caddr cdddr cdadr cddar
       caaaar caaadr caadar cadaar caaddr cadddr cadadr caddar cdaaar
       cdaadr cdadar cddaar cdaddr cddddr cddadr cdddar)

 (list caar-1 cadr-1 cdar-1 cddr-1 caaar-1 caadr-1 cadar-1 cdaar-1 caddr-1 cdddr-1 cdadr-1 cddar-1
       caaaar-1 caaadr-1 caadar-1 cadaar-1 caaddr-1 cadddr-1 cadadr-1 caddar-1 cdaaar-1
       cdaadr-1 cdadar-1 cddaar-1 cdaddr-1 cddddr-1 cddadr-1 cdddar-1))

(test (equal? (cadr (list 'a 'b 'c 'd 'e 'f 'g)) 'b) #t)
(test (equal? (cddr (list 'a 'b 'c 'd 'e 'f 'g)) '(c d e f g)) #t)
(test (equal? (caddr (list 'a 'b 'c 'd 'e 'f 'g)) 'c) #t)
(test (equal? (cdddr (list 'a 'b 'c 'd 'e 'f 'g)) '(d e f g)) #t)
(test (equal? (cadddr (list 'a 'b 'c 'd 'e 'f 'g)) 'd) #t)
(test (equal? (cddddr (list 'a 'b 'c 'd 'e 'f 'g)) '(e f g)) #t)
(test (equal? (caadr (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '((u v w) x)) #t)
(test (equal? (cadar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(6 7)) #t)
(test (equal? (cdaar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(5)) #t)
(test (equal? (cdadr (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(y)) #t)
(test (equal? (cddar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) ()) #t)
(test (equal? (caaaar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(1 2 3)) #t)
(test (equal? (caadar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) 6) #t)
(test (equal? (caaddr (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(q w e)) #t)
(test (equal? (cadaar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) 5) #t)
(test (equal? (cadadr (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) 'y) #t)
(test (equal? (caddar (list (list (list (list (list 1 2 3) 4) 5) 1 6 (list 5 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) 6) #t)
(test (equal? (cadddr (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(a b c)) #t)
(test (equal? (cdaaar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(4)) #t)
(test (equal? (cdaadr (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(x)) #t)
(test (equal? (cdadar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(7)) #t)

(test (caar '((a) b c d e f g)) 'a)
(test (cadr '(a b c d e f g)) 'b)
(test (cdar '((a b) c d e f g)) '(b))
(test (cddr '(a b c d e f g)) '(c d e f g))
(test (caaar '(((a)) b c d e f g)) 'a)
(test (caadr '(a (b) c d e f g)) 'b)
(test (cadar '((a b) c d e f g)) 'b)
(test (caddr '(a b c d e f g)) 'c)
(test (cdaar '(((a b)) c d e f g)) '(b))
(test (cdadr '(a (b c) d e f g)) '(c))
(test (cddar '((a b c) d e f g)) '(c))
(test (cdddr '(a b c d e f g)) '(d e f g))
(test (caaaar '((((a))) b c d e f g)) 'a)
(test (caaadr '(a ((b)) c d e f g)) 'b)
(test (caadar '((a (b)) c d e f g)) 'b)
(test (caaddr '(a b (c) d e f g)) 'c)
(test (cadaar '(((a b)) c d e f g)) 'b)
(test (cadadr '(a (b c) d e f g)) 'c)
(test (caddar '((a b c) d e f g)) 'c)
(test (cadddr '(a b c d e f g)) 'd)
(test (cdaaar '((((a b))) c d e f g)) '(b))
(test (cdaadr '(a ((b c)) d e f g)) '(c))
(test (cdadar '((a (b c)) d e f g)) '(c))
(test (cdaddr '(a b (c d) e f g)) '(d))
(test (cddaar '(((a b c)) d e f g)) '(c))
(test (cddadr '(a (b c d) e f g)) '(d))
(test (cdddar '((a b c d) e f g)) '(d))
(test (cddddr '(a b c d e f g)) '(e f g))
(test (cadr '(1 2 . 3)) 2)
(test (cddr '(1 2 . 3)) 3)
(test (cadadr '''1) 1)
(test (cdadr '''1) '(1))

;; sacla
(test (caar '((a) b c)) 'a)
(test (cadr '(a b c)) 'b)
(test (cdar '((a . aa) b c)) 'aa)
(test (cddr '(a b . c)) 'c)
(test (caaar '(((a)) b c)) 'a)
(test (caadr '(a (b) c)) 'b)
(test (cadar '((a aa) b c)) 'aa)
(test (caddr '(a b c)) 'c)
(test (cdaar '(((a . aa)) b c)) 'aa)
(test (cdadr '(a (b . bb) c)) 'bb)
(test (cddar '((a aa . aaa) b c)) 'aaa)
(test (cdddr '(a b c . d)) 'd)
(test (caaaar '((((a))) b c)) 'a)
(test (caaadr '(a ((b)) c)) 'b)
(test (caadar '((a (aa)) b c)) 'aa)
(test (caaddr '(a b (c))) 'c)
(test (cadaar '(((a aa)) b c)) 'aa)
(test (cadadr '(a (b bb) c)) 'bb)
(test (caddar '((a aa aaa) b c)) 'aaa)
(test (cadddr '(a b c d)) 'd)
(test (cdaaar '((((a . aa))) b c)) 'aa)
(test (cdaadr '(a ((b . bb)) c)) 'bb)
(test (cdadar '((a (aa . aaa)) b c)) 'aaa)
(test (cdaddr '(a b (c . cc))) 'cc)
(test (cddaar '(((a aa . aaa)) b c)) 'aaa)
(test (cddadr '(a (b bb . bbb) c)) 'bbb)
(test (cdddar '((a aa aaa . aaaa) b c)) 'aaaa)
(test (cddddr '(a b c d . e)) 'e)

(test (caar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(((1 2 3) 4) 5))
(test (cadr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(((u v w) x) y))
(test (cdar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '((6 7)))
(test (cddr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(((q w e) r) (a b c) e f g))
(test (caaar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '((1 2 3) 4))
(test (caadr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '((u v w) x))
(test (cadar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(6 7))
(test (caddr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '((q w e) r))
(test (cdaar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(5))
(test (cdadr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(y))
(test (cddar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) ())
(test (cdddr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '((a b c) e f g))
(test (caaaar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(1 2 3))
(test (caaadr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(u v w))
(test (caadar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) 6)
(test (caaddr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(q w e))
(test (cadaar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) 5)
(test (cadadr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) 'y)
(test (cadddr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(a b c))
(test (cdaaar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(4))
(test (cdaadr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(x))
(test (cdadar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(7))
(test (cdaddr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(r))
(test (cddaar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) ())
(test (cddadr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) ())
(test (cddddr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(e f g))

(test (cadr '(a b c d e f g)) 'b)
(test (cddr '(a b c d e f g)) '(c d e f g))
(test (caddr '(a b c d e f g)) 'c)
(test (cdddr '(a b c d e f g)) '(d e f g))
(test (cadddr '(a b c d e f g)) 'd)
(test (cddddr '(a b c d e f g)) '(e f g))

(test (caar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '((a . b) c . d))
(test (caar '(((a . b) c . d) (e . f) g . h)) '(a . b))
(test (caar '((a . b) c . d)) 'a)
(test (cadr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '((i . j) k . l))
(test (cadr '(((a . b) c . d) (e . f) g . h)) '(e . f))
(test (cadr '((a . b) c . d)) 'c)
(test (cdar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '((e . f) g . h))
(test (cdar '(((a . b) c . d) (e . f) g . h)) '(c . d))
(test (cdar '((a . b) c . d)) 'b)
(test (cddr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '((m . n) o . p))
(test (cddr '(((a . b) c . d) (e . f) g . h)) '(g . h))
(test (cddr '((a . b) c . d)) 'd)
(test (caaar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '(a . b))
(test (caaar '(((a . b) c . d) (e . f) g . h)) 'a)
(test (caadr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '(i . j))
(test (caadr '(((a . b) c . d) (e . f) g . h)) 'e)
(test (cddar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '(g . h))
(test (cddar '(((a . b) c . d) (e . f) g . h)) 'd)
(test (cdddr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '(o . p))
(test (cdddr '(((a . b) c . d) (e . f) g . h)) 'h)
(test (caaaar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'a)
(test (caaadr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'i)
(test (caddar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'g)
(test (cadddr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'o)
(test (cdaaar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'b)
(test (cdaadr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'j)
(test (cdddar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'h)
(test (cddddr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'p)
(test (cadr ''foo) 'foo)

(let ((lst '((((A . B) C . D) (E . F) G . H) ((I . J) K . L) (M . N) O . P))) ; from comp.lang.lisp I think
  (test (car lst) '(((A . B) C . D) (E . F) G . H))
  (test (cdr lst) '(((I . J) K . L) (M . N) O . P))
  (test (caar lst) '((A . B) C . D))
  (test (cadr lst) '((I . J) K . L))
  (test (cdar lst) '((E . F) G . H))
  (test (cddr lst) '((M . N) O . P))
  (test (caaar lst) '(A . B))
  (test (caadr lst) '(I . J))
  (test (cadar lst) '(E . F))
  (test (caddr lst) '(M . N))
  (test (cdaar lst) '(C . D))
  (test (cdadr lst) '(K . L))
  (test (cddar lst) '(G . H))
  (test (cdddr lst) '(O . P))
  (test (caaaar lst) 'A)
  (test (caaadr lst) 'I)
  (test (caadar lst) 'E)
  (test (caaddr lst) 'M)
  (test (cadaar lst) 'C)
  (test (cadadr lst) 'K)
  (test (caddar lst) 'G)
  (test (cadddr lst) 'O)
  (test (cdaaar lst) 'B)
  (test (cdaadr lst) 'J)
  (test (cdadar lst) 'F)
  (test (cdaddr lst) 'N)
  (test (cddaar lst) 'D)
  (test (cddadr lst) 'L)
  (test (cdddar lst) 'H)
  (test (cddddr lst) 'P))

(test (recompose 10 cdr '(1 2 3 4 5 6 7 8 9 10 11 12)) '(11 12))
(test (recompose 10 car '(((((((((((1 2 3)))))))))))) '(1 2 3))

(test (cons 1 . 2) 'error)
(test (eval-string "(1 . 2 . 3)") 'error)
(test (car (list)) 'error)
(test (car ()) 'error)
(test (cdr (list)) 'error)
(test (cdr ()) 'error)
(test (caddar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) 'error)
(test (cdddar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) 'error)
(test (caar '(a b c d e f g)) 'error)
(test (cdar '(a b c d e f g)) 'error)
(test (caaar '(a b c d e f g)) 'error)
(test (caadr '(a b c d e f g)) 'error)
(test (cadar '(a b c d e f g)) 'error)
(test (cdaar '(a b c d e f g)) 'error)
(test (cdadr '(a b c d e f g)) 'error)
(test (cddar '(a b c d e f g)) 'error)
(test (caaaar '(a b c d e f g)) 'error)
(test (caaadr '(a b c d e f g)) 'error)
(test (caadar '(a b c d e f g)) 'error)
(test (caaddr '(a b c d e f g)) 'error)
(test (cadaar '(a b c d e f g)) 'error)
(test (cadadr '(a b c d e f g)) 'error)
(test (caddar '(a b c d e f g)) 'error)
(test (cdaaar '(a b c d e f g)) 'error)
(test (cdaadr '(a b c d e f g)) 'error)
(test (cdadar '(a b c d e f g)) 'error)
(test (cdaddr '(a b c d e f g)) 'error)
(test (cddaar '(a b c d e f g)) 'error)
(test (cddadr '(a b c d e f g)) 'error)
(test (cdddar '(a b c d e f g)) 'error)
(test (caar 'a) 'error)
(test (caar '(a)) 'error)
(test (cadr 'a) 'error)
(test (cadr '(a . b)) 'error)
(test (cdar 'a) 'error)
(test (cdar '(a . b)) 'error)
(test (cddr 'a) 'error)
(test (cddr '(a . b)) 'error)
(test (caaar 'a) 'error)
(test (caaar '(a)) 'error)
(test (caaar '((a))) 'error)
(test (caadr 'a) 'error)
(test (caadr '(a . b)) 'error)
(test (caadr '(a b)) 'error)
(test (cadar 'a) 'error)
(test (cadar '(a . b)) 'error)
(test (cadar '((a . c) . b)) 'error)
(test (caddr 'a) 'error)
(test (caddr '(a . b)) 'error)
(test (caddr '(a c . b)) 'error)
(test (cdaar 'a) 'error)
(test (cdaar '(a)) 'error)
(test (cdaar '((a . b))) 'error)
(test (cdadr 'a) 'error)
(test (cdadr '(a . b)) 'error)
(test (cdadr '(a b . c)) 'error)
(test (cddar 'a) 'error)
(test (cddar '(a . b)) 'error)
(test (cddar '((a . b) . b)) 'error)
(test (cdddr 'a) 'error)
(test (cdddr '(a . b)) 'error)
(test (cdddr '(a c . b)) 'error)
(test (caaaar 'a) 'error)
(test (caaaar '(a)) 'error)
(test (caaaar '((a))) 'error)
(test (caaaar '(((a)))) 'error)
(test (caaadr 'a) 'error)
(test (caaadr '(a . b)) 'error)
(test (caaadr '(a b)) 'error)
(test (caaadr '(a (b))) 'error)
(test (caadar 'a) 'error)
(test (caadar '(a . b)) 'error)
(test (caadar '((a . c) . b)) 'error)
(test (caadar '((a c) . b)) 'error)
(test (caaddr 'a) 'error)
(test (caaddr '(a . b)) 'error)
(test (caaddr '(a c . b)) 'error)
(test (caaddr '(a c b)) 'error)
(test (cadaar 'a) 'error)
(test (cadaar '(a)) 'error)
(test (cadaar '((a . b))) 'error)
(test (cadaar '((a b))) 'error)
(test (cadadr 'a) 'error)
(test (cadadr '(a . b)) 'error)
(test (cadadr '(a b . c)) 'error)
(test (cadadr '(a (b . e) . c)) 'error)
(test (caddar 'a) 'error)
(test (caddar '(a . b)) 'error)
(test (caddar '((a . b) . b)) 'error)
(test (caddar '((a b . c) . b)) 'error)
(test (cadddr 'a) 'error)
(test (cadddr '(a . b)) 'error)
(test (cadddr '(a c . b)) 'error)
(test (cadddr '(a c e . b)) 'error)
(test (cdaaar 'a) 'error)
(test (cdaaar '(a)) 'error)
(test (cdaaar '((a))) 'error)
(test (cdaaar '(((a . b)))) 'error)
(test (cdaadr 'a) 'error)
(test (cdaadr '(a . b)) 'error)
(test (cdaadr '(a b)) 'error)
(test (cdaadr '(a (b . c))) 'error)
(test (cdadar 'a) 'error)
(test (cdadar '(a . b)) 'error)
(test (cdadar '((a . c) . b)) 'error)
(test (cdadar '((a c . d) . b)) 'error)
(test (cdaddr 'a) 'error)
(test (cdaddr '(a . b)) 'error)
(test (cdaddr '(a c . b)) 'error)
(test (cdaddr '(a c b . d)) 'error)
(test (cddaar 'a) 'error)
(test (cddaar '(a)) 'error)
(test (cddaar '((a . b))) 'error)
(test (cddaar '((a b))) 'error)
(test (cddadr 'a) 'error)
(test (cddadr '(a . b)) 'error)
(test (cddadr '(a b . c)) 'error)
(test (cddadr '(a (b . e) . c)) 'error)
(test (cdddar 'a) 'error)
(test (cdddar '(a . b)) 'error)
(test (cdddar '((a . b) . b)) 'error)
(test (cdddar '((a b . c) . b)) 'error)
(test (cddddr 'a) 'error)
(test (cddddr '(a . b)) 'error)
(test (cddddr '(a c . b)) 'error)
(test (cddddr '(a c e . b)) 'error)

(test (caar '((1))) 1)
(test (cadr '(1 2)) 2)
(test (cdar '((1 . 2))) 2)
(test (cddr '(1 2 . 3)) 3)
(test (caaar '(((1)))) 1)
(test (caadr '(1 (2))) 2)
(test (cadar '((1 2))) 2)
(test (cdaar '(((1 . 2)))) 2)
(test (caddr '(1 2 3)) 3)
(test (cdddr '(1 2 3 . 4)) 4)
(test (cdadr '(1 (2 . 3))) 3)
(test (cddar '((1 2 . 3))) 3)
(test (caaaar '((((1))))) 1)
(test (caaadr '(1 ((2)))) 2)
(test (caadar '((1 (2)))) 2)
(test (cadaar '(((1 2)))) 2)
(test (caaddr '(1 2 (3))) 3)
(test (cadddr '(1 2 3 4)) 4)
(test (cadadr '(1 (2 3))) 3)
(test (caddar '((1 2 3))) 3)
(test (cdaaar '((((1 . 2))))) 2)
(test (cdaadr '(1 ((2 . 3)))) 3)
(test (cdadar '((1 (2 . 3)))) 3)
(test (cddaar '(((1 2 . 3)))) 3)
(test (cdaddr '(1 2 (3 . 4))) 4)
(test (cddddr '(1 2 3 4 . 5)) 5)
(test (cddadr '(1 (2 3 . 4))) 4)
(test (cdddar '((1 2 3 . 4))) 4)

(let () (define (f1 x) (eq? (car x) 'y)) (let ((z 1)) (test (f1 z) 'error)))
(let () (define (f1 x) (eq? (cdr x) 'y)) (let ((z 1)) (test (f1 z) 'error)))
(let () (define (f1 x) (eq? (caar x) 'y)) (let ((z (list 1 2))) (test (f1 z) 'error)))
(let () (define (f1 x) (eq? (cadr x) 'y)) (let ((z (cons 1 2))) (test (f1 z) 'error)))
(let () (define (f1 x) (eq? (cdar x) 'y)) (let ((z (list 1 2))) (test (f1 z) 'error)))
(let () (define (f1 x) (eq? (cddr x) 'y)) (let ((z (cons 1 2))) (test (f1 z) 'error)))



;;; --------------------------------------------------------------------------------
;;; length

(test (length (list 'a 'b 'c 'd 'e 'f)) 6)
(test (length (list 'a 'b 'c 'd)) 4)
(test (length (list 'a (list 'b 'c) 'd)) 3)
(test (length ()) 0)
(test (length '(this-that)) 1)
(test (length '(this - that)) 3)
(test (length '(a b)) 2)
(test (length '(a b c)) 3)
(test (length '(a (b) (c d e))) 3)
(test (length (list 1 (cons 1 2))) 2)
(test (length (list 1 (cons 1 ()))) 2)

(for-each
 (lambda (arg)
   (test (length arg) #f))
 (list (integer->char 65) #f 'a-symbol abs quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #\f #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (length 'x) #f)
(test (length (cons 1 2)) -1)
(let ((x (list 1 2)))
  (set-cdr! x x)
  (test (infinite? (length x)) #t))
(test (length '(1 2 . 3)) -2)
(test (length) 'error)
(test (length '(1 2 3) #(1 2 3)) 'error)
(test (integer? (length (funclet cons))) #t)
(test (> (length (rootlet)) 200) #t)

(test (length '((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((0))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))) 1)

(test (let ((f (lambda (a) (values a (+ a 1)))))
	(define (f1 x)
	  (hash-table-entries (and (< (length x) 100) x)))
	(f1 f))
      'error)
(test (let ((f (lambda (a) (values a (+ a 1)))))
	(define (f1 x)
	  (hash-table-entries (and (= (length x) 100) x)))
	(f1 f))
      'error)

(test (call-with-input-string "01234" (lambda (p) (length p))) 5)
(test (call-with-input-file "s7test.scm" (lambda (p) (> (length p) 5000000))) #t)
(unless pure-s7
  (test (length (call-with-input-file "s7test.scm" (dilambda (lambda* (a b) a) (lambda* (a b c) c)))) #f)) ; (length <closed-port>)
(test (let ((len 0)) (call-with-output-string (lambda (p) (display "0123" p) (set! len (length p)))) len) 4)


;;; --------------------------------------------------------------------------------
;;; reverse

(test (reverse '(a b c d)) '(d c b a))
(test (reverse '(a b c))  '(c b a))
(test (reverse '(a (b c) d (e (f))))  '((e (f)) d (b c) a))
(test (reverse ()) ())
(test (reverse (list 1 2 3)) '(3 2 1))
(test (reverse (list 1)) '(1))
(test (reverse (list)) (list))
(test (reverse '(1 2 3)) (list 3 2 1))
(test (reverse '(1)) '(1))
(test (reverse '((1 2) 3)) '(3 (1 2)))
(test (reverse '(((1 . 2) . 3) 4)) '(4 ((1 . 2) . 3)))
(test (reverse (list (list) (list 1 2))) '((1 2) ()))
(test (reverse '((a) b c d)) '(d c b (a)))
(test (reverse (reverse (list 1 2 3 4))) (list 1 2 3 4))
(test (reverse ''foo) '(foo quote))
(test (let ((x (list 1 2 3 4)))
	(let ((y (reverse x)))
	  (and (equal? x (list 1 2 3 4))
	       (equal? y (list 4 3 2 1)))))
      #t)
(test (letrec ((hi (lambda (lst n)
		     (if (= n 0)
			 lst
			 (hi (reverse lst) (- n 1))))))
	(hi (list 1 2 3) 100))
      (list 1 2 3))
(test (let ((var (list 1 2 3))) (reverse (cdr var)) var) (list 1 2 3))
(test (let ((var '(1 2 3))) (reverse (cdr var)) var) '(1 2 3))
(test (let ((var (list 1 (list 2 3)))) (reverse (cdr var)) var) (list 1 (list 2 3)))
(test (let ((var '(1 (2 3)))) (reverse (cdr var)) var) '(1 (2 3)))
(test (let ((var (list (list 1 2) (list 3 4 5)))) (reverse (car var)) var) '((1 2) (3 4 5)))
(test (let ((x '(1 2 3))) (list (reverse x) x)) '((3 2 1) (1 2 3)))
(test (reverse '(1 2)) '(2 1))
(test (reverse '(1 2 3)) '(3 2 1))
(test (reverse '(1 2 3 4)) '(4 3 2 1))

(when with-block
  (test (block? (reverse _c_obj_)) #t)
  (let ((b (block 1 2 3 4)))
    (let ((b1 (reverse b)))
      (test b1 (block 4 3 2 1))
      (test b (block 1 2 3 4)))))

(for-each
 (lambda (lst)
   (if (proper-list? lst)
       (if (not (equal? lst (reverse (reverse lst))))
	   (format #t ";(reverse (reverse ~A)) -> ~A?~%" lst (reverse (reverse lst))))))
 lists)

(for-each
 (lambda (lst)
   (if (proper-list? lst)
       (if (not (equal? lst (reverse (reverse (reverse (reverse lst))))))
	   (format #t ";(reverse...(4x) ~A) -> ~A?~%" lst (reverse (reverse (reverse (reverse lst))))))))
 lists)

(test (let ((x (list 1 2 3))) (list (recompose 32 reverse x) x)) '((1 2 3) (1 2 3)))
(test (let ((x (list 1 2 3))) (list (recompose 31 reverse x) x)) '((3 2 1) (1 2 3)))

(test (reverse (cons 1 2)) '(2 . 1))
(test (reverse '(1 . 2)) '(2 . 1))
(test (reverse '(1 2 . 3)) '(3 2 1))
(test (reverse) 'error)
(test (reverse '(1 2 3) '(3 2 1)) 'error)
(test (reverse (subvector (make-int-vector '(2 3) 0) 0 6 '(6))) (make-int-vector 6 0))
(test (reverse (make-float-vector 6 0.0)) (make-float-vector 6 0.0))

(for-each
 (lambda (arg)
   (test (reverse arg) 'error))
 (list (integer->char 65) #f 'a-symbol abs quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #\f #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (reverse "hi") "ih")
(test (reverse "") "")
(test (reverse "123") "321")
(test (reverse "1234") "4321")
(test (reverse "12") "21")
(test (reverse "a\x00;b") "b\x00;a")
(test (reverse #()) #())
(test (reverse #(1)) #(1))
(test (reverse #(1 2)) #(2 1))
(test (reverse #(1 2 3)) #(3 2 1))
(test (reverse #(1 2 3 4)) #(4 3 2 1))
(test (reverse #2d((1 2) (3 4))) #2d((4 3) (2 1)))
(test (reverse (string #\a #\null #\b)) "b\x00;a")
(test (reverse abs) 'error)
(test (vector->list (reverse (let ((v (make-int-vector 3))) (set! (v 1) 1) (set! (v 2) 2) v))) '(2 1 0))
(test (reverse (int-vector)) #())
(test (reverse (int-vector 1)) (int-vector 1))
(test (reverse (int-vector 1 2)) (int-vector 2 1))
(test (reverse (int-vector 1 2 3)) (int-vector 3 2 1))
(test (reverse (int-vector 1 2 3 4)) (int-vector 4 3 2 1))
(test (reverse (float-vector)) #())
(test (reverse (float-vector 1)) (float-vector 1))
(test (reverse (float-vector 1 2)) (float-vector 2 1))
(test (reverse (float-vector 1 2 3)) (float-vector 3 2 1))
(test (reverse (float-vector 1 2 3 4)) (float-vector 4 3 2 1))
(test (let ((v #(1 2 3))) (reverse v) v) #(1 2 3))
(test (reverse #u(1 2 3)) #u(3 2 1))
(test (reverse #u(1 2)) #u(2 1))
(test (reverse #u(1 2 3 4)) #u(4 3 2 1))

(when with-block
  (let ((b (block 1.0 2.0 3.0)))
    (set! (b 1) 32.0)
    (test (b 1) 32.0)
    (let ((b1 (reverse b)))
      (test b1 (block 3.0 32.0 1.0)))))




;;; --------------------------------------------------------------------------------
;;; reverse!

(test (reverse! '(1 . 2)) 'error)
(test (reverse! (cons 1 2)) 'error)
(test (reverse! (cons 1 (cons 2 3))) 'error)
(test (reverse!) 'error)
(test (reverse! '(1 2 3) '(3 2 1)) 'error)

(test (reverse! '(a b c d)) '(d c b a))
(test (reverse! '(a b c))  '(c b a))
(test (reverse! '(a (b c) d (e (f))))  '((e (f)) d (b c) a))
(test (reverse! ()) ())
(test (reverse! (list 1 2 3)) '(3 2 1))
(test (reverse! (list 1)) '(1))
(test (reverse! (list)) (list))
(test (reverse! '(1 2 3)) (list 3 2 1))
(test (reverse! '(1)) '(1))
(test (reverse! '((1 2) 3)) '(3 (1 2)))
(test (reverse! '(((1 . 2) . 3) 4)) '(4 ((1 . 2) . 3)))
(test (reverse! (list (list) (list 1 2))) '((1 2) ()))
(test (reverse! '((a) b c d)) '(d c b (a)))
(test (reverse! (reverse! (list 1 2 3 4))) (list 1 2 3 4))
(test (reverse! (reverse! (sort! (list 1 2 3 4) >))) (sort! (list 1 2 3 4) >))
(test (reverse! ''foo) '(foo quote))
(test (reverse (reverse! (list 1 2 3))) (list 1 2 3))
(test (reverse (reverse! (reverse! (reverse (list 1 2 3))))) (list 1 2 3))

;; check dumb optimization oversight
(when with-block
  (let ((b (block 0 1 2 3 4 5 6 7)))
    (set! b (reverse! b))
    (test b (block 7 6 5 4 3 2 1 0))))
(test (reverse! #r(0 1 2 3 4 5 6 7)) #r(7 6 5 4 3 2 1 0))
(test (reverse! #i(0 1 2 3 4 5 6 7)) #i(7 6 5 4 3 2 1 0))
(test (reverse! #(0 1 2 3 4 5 6 7)) #(7 6 5 4 3 2 1 0))
(test (reverse! #u(0 1 2 3 4 5 6 7)) #u(7 6 5 4 3 2 1 0))
(test (reverse! "01234567") "76543210")

(do ((i 0 (+ i 1)))
    ((= i 33))
  (let ((b (make-vector i))
	(s (make-string i)))
    (do ((j 0 (+ j 1)))
	((= i j))
      (set! (b j) j)
      (set! (s j) (integer->char (+ 32 j))))
    (set! b (reverse! b))
    (set! s (reverse! s))
    (do ((j 0 (+ j 1)))
	((= i j))
      (if (not (= (b j) (- i j 1)))
	  (format *stderr* "~A at ~A ~A~%" b i j))
      (if (not (char=? (s j) (integer->char (+ 32 (- i j 1)))))
	  (format *stderr* "~S at ~A ~A~%" s i j)))))


(test (let ((x (list 1 2 3))) (recompose 31 reverse! x)) '(3 2 1))
(test (reverse! '(1 2 . 3)) 'error)

(let* ((lst1 (list 1 2 3))
       (lst2 (apply list '(4 5 6)))
       (lst3 (sort! (reverse! (append lst1 lst2)) <)))
  (test lst3 '(1 2 3 4 5 6))
  (define (lt . args)
    args)
  (set! lst3 (sort! (apply reverse! (lt lst3)) >))
  (test lst3 '(6 5 4 3 2 1)))

(for-each
 (lambda (arg)
   (test (reverse! arg) 'error))
 (list (integer->char 65) #f 'a-symbol abs _ht_ _undef_ _null_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #\f #t (if #f #f) (lambda (a) (+ a 1))))

(test (let ((str "1234")) (reverse! str) str) "4321")
(test (let ((str "123")) (reverse! str) str) "321")
(test (let ((str "")) (reverse! str) str) "")
(test (let ((v #(1 2 3))) (reverse! v) v) #(3 2 1))
(test (let ((v #(1 2 3 4))) (reverse! v) v) #(4 3 2 1))
(test (let ((v #())) (reverse! v) v) #())
(test (let ((v (float-vector 1.0 2.0 3.0))) (reverse! v) v) (float-vector 3.0 2.0 1.0))
(test (let ((v (float-vector 1.0 2.0 3.0 4.0))) (reverse! v) v) (float-vector 4.0 3.0 2.0 1.0))
(test (let ((v (float-vector))) (reverse! v) v) #())
(test (let ((v (int-vector 1 2 3))) (reverse! v) v) (int-vector 3 2 1))
(test (let ((v (int-vector 1 2 3 4))) (reverse! v) v) (int-vector 4 3 2 1))
(test (let ((v (int-vector))) (reverse! v) v) #())
(when with-block
  (test (block? (reverse! _c_obj_)) #t)
  (let ((b (block 1 2 3 4)))
    (reverse! b)
    (test b (block 4 3 2 1)))
  (let ((b (block 1 2 3 4 5 6 7 8 9)))
    (set! b (reverse! b))
    (test b (block 9 8 7 6 5 4 3 2 1))))
(test (let ((v (make-int-vector 3 1))) (set! (v 1) 2) (set! (v 2) 3) (reverse! v) v) (let ((v (make-int-vector 3 3))) (set! (v 1) 2) (set! (v 2) 1) v))

(when full-s7test
  (let ()
    ;; some sequence tests

    (define (fv-tst len)
      (let ((fv (make-float-vector len)))
	(if (not (= (length fv) len))
	    (format *stderr* "float-vector length ~A: ~A~%" fv (length fv)))
	(fill! fv 0.0)
	(let ((fv-orig (copy fv)))
	  (do ((i 0 (+ i 1)))
	      ((= i len))
	    (set! (fv i) (- (random 1000.0) 500.0)))
	  (let ((fv-ran (copy fv))
		(fv-ran1 (copy fv)))
	    (sort! fv <)
	    (call-with-exit
	     (lambda (quit)
	       (do ((i 1 (+ i 1)))
		   ((= i len))
		 (when (> (fv (- i 1)) (fv i))
		   (format *stderr* "float-vector: ~A > ~A at ~D~%" (fv (- i 1)) (fv i) i)
		   (quit)))))
	    (sort! fv-ran (lambda (a b) (< a b)))
	    (if (not (equivalent? fv fv-ran))
		(format *stderr* "float-vector closure not equal~%"))
	    (sort! fv-ran1 (lambda (a b) (cond ((< a b) #t) (#t #f))))
	    (if (not (equivalent? fv fv-ran1))
		(format *stderr* "float-vector cond closure not equal~%")))

	  (let ((fv-copy (copy fv)))
	    (reverse! fv)
	    (if (and (not (equivalent? fv-copy fv))
		     (equivalent? fv fv-orig))
		(format *stderr* "float-vector reverse!: ~A ~A~%" fv fv-orig))
	    (reverse! fv)
	    (if (not (equivalent? fv-copy fv))
		(format *stderr* "float-vector reverse! twice: ~A ~A~%" fv fv-copy))
	    (let ((fv1 (apply float-vector (make-list len 1.0))))
	      (if (or (not (= (length fv1) len))
		      (not (= (fv1 (- len 1)) 1.0)))
		  (format *stderr* "float-vector apply: ~A ~A~%" len (fv (- len 1)))))
	    ))))

    (define (iv-tst len)
      (let ((fv (make-int-vector len 0)))
	(if (not (= (length fv) len))
	    (format *stderr* "int-vector length ~A: ~A~%" fv (length fv)))
	(fill! fv 0)
	(let ((fv-orig (copy fv)))
	  (do ((i 0 (+ i 1)))
	      ((= i len))
	    (set! (fv i) (- (random 1000000) 500000)))
	  (let ((fv-ran (copy fv))
		(fv-ran1 (copy fv)))
	    (sort! fv <)
	    (call-with-exit
	     (lambda (quit)
	       (do ((i 1 (+ i 1)))
		   ((= i len))
		 (when (> (fv (- i 1)) (fv i))
		   (format *stderr* "int-vector: ~A > ~A at ~D~%" (fv (- i 1)) (fv i) i)
		   (quit)))))
	    (sort! fv-ran (lambda (a b) (< a b)))
	    (if (not (equivalent? fv fv-ran))
		(format *stderr* "int-vector closure not equal~%"))
	    (sort! fv-ran1 (lambda (a b) (cond ((< a b) #t) (#t #f))))
	    (if (not (equivalent? fv fv-ran1))
		(format *stderr* "int-vector cond closure not equal~%")))

	  (let ((fv-copy (copy fv)))
	    (reverse! fv)
	    (if (and (not (equivalent? fv-copy fv))
		     (equivalent? fv fv-orig))
		(format *stderr* "int-vector reverse!: ~A ~A~%" fv fv-orig))
	    (reverse! fv)
	    (if (not (equivalent? fv-copy fv))
		(format *stderr* "int-vector reverse! twice: ~A ~A~%" fv fv-copy))
	    ))))

    (define (v-tst len)
      (let ((fv (make-vector len)))
	(if (not (= (length fv) len))
	    (format *stderr* "vector length ~A: ~A~%" fv (length fv)))
	(fill! fv 0)
	(let ((fv-orig (copy fv)))
	  (do ((i 0 (+ i 1)))
	      ((= i len))
	    (set! (fv i) (- (random 1000000) 500000)))
	  (let ((fv-ran (copy fv))
		(fv-ran1 (copy fv)))
	    (sort! fv <)
	    (call-with-exit
	     (lambda (quit)
	       (do ((i 1 (+ i 1)))
		   ((= i len))
		 (when (> (fv (- i 1)) (fv i))
		   (format *stderr* "vector: ~A > ~A at ~D~%" (fv (- i 1)) (fv i) i)
		   (quit)))))
	    (sort! fv-ran (lambda (a b) (< a b)))
	    (if (not (equivalent? fv fv-ran))
		(format *stderr* "vector closure not equal~%"))
	    (sort! fv-ran1 (lambda (a b) (cond ((< a b) #t) (#t #f))))
	    (if (not (equivalent? fv fv-ran1))
		(format *stderr* "vector cond closure not equal~%")))

	  (let ((fv-copy (copy fv)))
	    (reverse! fv)
	    (if (and (not (equivalent? fv-copy fv))
		     (equivalent? fv fv-orig))
		(format *stderr* "vector reverse!: ~A ~A~%" fv fv-orig))
	    (reverse! fv)
	    (if (not (equivalent? fv-copy fv))
		(format *stderr* "vector reverse! twice: ~A ~A~%" fv fv-copy))
	    (let ((fv1 (apply vector (make-list len 1))))
	      (if (or (not (= (length fv1) len))
		      (not (= (fv1 (- len 1)) 1)))
		  (format *stderr* "vector apply: ~A ~A~%" len (fv (- len 1)))))
	    ))))

    (define (s-tst len)
      (let ((fv (make-string len)))
	(if (not (= (length fv) len))
	    (format *stderr* "string length ~A: ~A~%" fv (length fv)))
	(fill! fv #\a)
	(let ((fv-orig (copy fv)))
	  (do ((i 0 (+ i 1)))
	      ((= i len))
	    (set! (fv i) (integer->char (+ 20 (random 100)))))
	  (let ((fv-ran (copy fv))
		(fv-ran1 (copy fv)))
	    (sort! fv char<?)
	    (call-with-exit
	     (lambda (quit)
	       (do ((i 1 (+ i 1)))
		   ((= i len))
		 (when (char>? (fv (- i 1)) (fv i))
		   (format *stderr* "string: ~A > ~A at ~D~%" (fv (- i 1)) (fv i) i)
		   (quit)))))
	    (sort! fv-ran (lambda (a b) (char<? a b)))
	    (if (not (equivalent? fv fv-ran))
		(format *stderr* "string closure not equal~%"))
	    (sort! fv-ran1 (lambda (a b) (cond ((char<? a b) #t) (#t #f))))
	    (if (not (equivalent? fv fv-ran))
		(format *stderr* "string cond closure not equal~%")))

	  (let ((fv-copy (copy fv)))
	    (reverse! fv)
	    (if (and (not (equivalent? fv-copy fv))
		     (equivalent? fv fv-orig))
		(format *stderr* "string reverse!: ~A ~A~%" fv fv-orig))
	    (reverse! fv)
	    (if (not (equivalent? fv-copy fv))
		(format *stderr* "string reverse! twice: ~A ~A~%" fv fv-copy))
	    (let ((fv1 (apply string (make-list len #\a))))
	      (if (or (not (= (length fv1) len))
		      (not (char=? (fv1 (- len 1)) #\a)))
		  (format *stderr* "string apply: ~A ~A~%" len (fv (- len 1)))))
	    ))))

    (define (p-tst len)
      (let ((fv (make-list len)))
	(if (not (= (length fv) len))
	    (format *stderr* "list length ~A: ~A~%" fv (length fv)))
	(fill! fv 0)
	(let ((fv-orig (copy fv)))
	  (do ((p fv (cdr p)))
	      ((null? p))
	    (set-car! p (- (random 100000) 50000)))
	  (let ((fv-ran (copy fv)))
	    (sort! fv <)
	    (call-with-exit
	     (lambda (quit)
	       (do ((p0 fv (cdr p0))
		    (p1 (cdr fv) (cdr p1))
		    (i 1 (+ i 1)))
		   ((null? p1))
		 (when (> (car p0) (car p1))
		   (format *stderr* "list: ~A > ~A at ~D~%" (car p0) (car p1) i)
		   (quit)))))
	    (sort! fv-ran (lambda (a b) (< a b)))
	    (if (not (equivalent? fv fv-ran))
		(format *stderr* "pair closure not equal~%")))

	  (let ((fv-copy (copy fv)))
	    (set! fv (reverse! fv))
	    (if (and (not (equivalent? fv-copy fv))
		     (equivalent? fv fv-orig))
		(format *stderr* "list reverse!: ~A ~A~%" fv fv-orig))
	    (set! fv (reverse! fv))
	    (if (not (equivalent? fv-copy fv))
		(format *stderr* "list reverse! twice: ~A ~A~%" fv fv-copy))
	    ))))

    (for-each
     (lambda (b p)
       (do ((k 0 (+ k 1)))
	   ((= k 1000))
	 (fv-tst b)
	 (iv-tst b)
	 (v-tst b)
	 (s-tst b)
	 (p-tst b))
       (do ((i 0 (+ i 1)))
	   ((= i p))
	 (format *stderr* "~D fv " (expt b i))
	 (fv-tst (expt b i))
	 (format *stderr* "iv ")
	 (iv-tst (expt b i))
	 (format *stderr* "v ")
	 (v-tst (expt b i))
	 (format *stderr* "s ")
	 (s-tst (expt b i))
	 (format *stderr* "p ")
	 (p-tst (expt b i))
	 (newline *stderr*)
	 ))
     (list 2 3 4 7 10)
     (list 12 4 3 6 6))
    ))

(test (let ((v (vector 1 2 3 4))) (let ((sv (subvector v 0 3))) (reverse! sv) v)) #(3 2 1 4))
(test (let ((v (vector 1 2 3 4))) (let ((sv (subvector v 1 4))) (reverse! sv) v)) #(1 4 3 2))
(test (let ((v (vector 1 2 3 4))) (let ((sv (subvector v 1 4))) (fill! sv 5) v)) #(1 5 5 5))
(test (let ((v (vector 1 2 3 4))) (let ((sv (subvector v 1 4))) (reverse sv) v)) #(1 2 3 4))
(test (let ((v (vector 1 2 3 4))) (let ((sv (subvector v 1 4))) (sort! sv >) v)) #(1 4 3 2))
(test (let ((v (make-int-vector '(3 3) 1))) (let ((sv (v 1))) (fill! sv 2) v)) (subvector (int-vector 1 1 1 2 2 2 1 1 1) 0 9 '(3 3)))
(test (immutable? (subvector (immutable! (vector 1 2 3 4)) 0 4 '(2 2))) #t)

(test (let ((v (make-int-vector '(3 3) 1)))
	(do ((i 0 (+ i 1)))
	    ((= i 3))
	  (do ((j 0 (+ j 1)))
	      ((= j 3))
	    (set! (v i j) (+ j (* i 3)))))
	(let ((sv (v 1)))
	  (fill! sv 2)
	  v))
      (subvector (int-vector 0 1 2 2 2 2 6 7 8) 0 9 '(3 3)))

(test (let ((v (make-int-vector '(3 3) 1)))
	(do ((i 0 (+ i 1)))
	    ((= i 3))
	  (do ((j 0 (+ j 1)))
	      ((= j 3))
	    (set! (v i j) (+ j (* i 3)))))
	(let ((sv (v 1)))
	  (sort! sv >)
	  v))
      (subvector (int-vector 0 1 2 5 4 3 6 7 8) 0 9 '(3 3)))

(test (let ((v (make-int-vector '(3 3) 1)))
	(do ((i 0 (+ i 1)))
	    ((= i 3))
	  (do ((j 0 (+ j 1)))
	      ((= j 3))
	    (set! (v i j) (+ j (* i 3)))))
	(let ((sv (v 1)))
	  (reverse! sv)
	  v))
      (subvector (int-vector 0 1 2 5 4 3 6 7 8) 0 9 '(3 3)))

(test (catch #t
	(lambda ()
	  (reverse! (catch #t 1 cons))
	  (reverse! (catch #t 1 cons))) ; this will clobber sc->wrong_type_arg_info if not caught
	(lambda args 'error))
      'error)
(test (reverse! `((1) . x)) 'error)
(if with-block (test (reverse! (immutable! (block 0 0))) 'error))

(when full-s7test
  (let ()
    (define (revstr size)
      (let ((str (make-string size)))
	(do ((i 0 (+ i 1)))
	    ((= i size))
	  (string-set! str i (integer->char (+ i 20))))
	(let ((rstr (reverse! str)))
	  (do ((i 0 (+ i 1))
	       (j (- size 1) (- j 1)))
	      ((= i size))
	    (unless (char=? (string-ref str j) (integer->char (+ i 20)))
	      (format *stderr* "revstr ~D: ~A ~A~%" i (string-ref str j) (integer->char (+ i 20))))))))

    (do ((i 1 (+ i 1)))
	((= i 8))
      (revstr (expt 2 i)))

    (define (revint size)
      (let ((str (make-int-vector size)))
	(do ((i 0 (+ i 1)))
	    ((= i size))
	  (int-vector-set! str i i))
	(let ((rstr (reverse! str)))
	  (do ((i 0 (+ i 1))
	       (j (- size 1) (- j 1)))
	      ((= i size))
	    (unless (= (int-vector-ref str j) i)
	      (format *stderr* "revint ~D: ~A ~A~%" i (int-vector-ref str j) i))))))

    (do ((i 3 (+ i 1)))
	((= i 10))
      (revint (expt 2 i)))
    ))


;;; --------------------------------------------------------------------------------
;;; pair?

(test (pair? 'a) #f)
(test (pair? '()) #f)
(test (pair? ()) #f)
(test (pair? '(a b c)) #t)
(test (pair? (cons 1 2)) #t)
(test (pair? ''()) #t)
(test (pair? #f) #f)
(test (pair? (make-vector 6)) #f)
(test (pair? #t) #f)
(test (pair? '(a . b)) #t)
(test (pair? #(a b))  #f)
(test (pair? (list 1 2)) #t)
(test (pair? (list)) #f)
(test (pair? ''foo) #t)
(test (pair? (list 'a 'b 'c 'd 'e 'f)) #t)
(test (pair? '(this-that)) #t)
(test (pair? '(this - that)) #t)
(let ((x (list 1 2)))
  (set-cdr! x x)
  (test (pair? x) #t))
(test (pair? (list 1 (cons 1 2))) #t)
(test (pair? (list 1 (cons 1 ()))) #t)
(test (pair? (cons 1 ())) #t)
(test (pair? (cons () ())) #t)
(test (pair? (cons () 1)) #t)
(test (pair? (list (list))) #t)
(test (pair? '(())) #t)
(test (pair? (cons 1 (cons 2 3))) #t)
(test (pair?) 'error)
(test (pair? `'1) #t)
(test (pair? begin) #f)
(test (pair? 'begin) #f)
(test (pair? ''begin) #t)
(test (pair? list) #f)

(for-each
 (lambda (arg)
   (if (pair? arg)
       (format #t ";(pair? ~A) -> #t?~%" arg)))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #\f #t :hi (if #f #f) (lambda (a) (+ a 1))))


;;; pair-line-number

(test (pair-line-number) 'error)
(test (pair-line-number () ()) 'error)
(for-each
 (lambda (arg)
   (test (pair-line-number arg) 'error))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #\f #t :hi (if #f #f) (lambda (a) (+ a 1))))

;; pair-filename

(test (pair-filename) 'error)
(test (pair-filename () ()) 'error)
(for-each
 (lambda (arg)
   (test (pair-filename arg) 'error))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #\f #t :hi (if #f #f) (lambda (a) (+ a 1))))

(let ()
  (call-with-output-file "t559.scm"
    (lambda (p)
      (format p "~%~%~%(define (pair-fl-test)~%  (+ 1 2))~%~%(list (port-line-number) (port-filename))~%~%")))
  (let ((loc (load "t559.scm" (curlet))))
    (test loc (list 7 "t559.scm")))
  (let ((s (procedure-source pair-fl-test)))
    (if (pair? s)
        (begin
          (test (pair-filename (cddr s)) "t559.scm")
          (test (pair-line-number (cddr s)) 4))
        (format *stderr* "t559.scm no source~%"))))


;;; --------------------------------------------------------------------------------
;;; list?

(test (list? 'a) #f)
(test (list? ()) #t)
(test (list? '(a b c)) #t)
(test (list? (cons 1 2)) #t)
(test (list? ''()) #t)
(test (list? #f) #f)
(test (list? (make-vector 6)) #f)
(test (list? #t) #f)
(test (list? '(a . b)) #t)
(test (list? #(a b))  #f)
(test (list? (list 1 2)) #t)
(test (list? (list)) #t)
(test (list? ''foo) #t)
(test (list? ''2) #t)
(test (list? (list 'a 'b 'c 'd 'e 'f)) #t)
(test (list? '(this-that)) #t)
(test (list? '(this - that)) #t)
(let ((x (list 1 2)))
  (set-cdr! x x)
  (test (proper-list? x) #f)
  (test (list? x) #t))
(test (list? (list 1 (cons 1 2))) #t)
(test (list? (list 1 (cons 1 ()))) #t)
(test (list? (cons 1 ())) #t)
(test (list? (cons () ())) #t)
(test (list? (cons () 1)) #t)
(test (list? (list (list))) #t)
(test (list? '(())) #t)
(test (list? '(1 2 . 3)) #t)
(test (list? (cons 1 (cons 2 3))) #t)
(test (list? '(1 . ())) #t)

(test (list? '(1 2) ()) 'error)
(test (list?) 'error)
(for-each
 (lambda (arg)
   (if (list? arg)
       (format #t ";(list? ~A) -> #t?~%" arg)))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #\f #t :hi (if #f #f) (lambda (a) (+ a 1))))


;;; proper-list?

(test (proper-list? 'a) #f)
(test (proper-list? ()) #t)
(test (proper-list? '(a b c)) #t)
(test (proper-list? (cons 1 2)) #f)
(test (proper-list? ''()) #t)
(test (proper-list? #f) #f)
(test (proper-list? (make-vector 6)) #f)
(test (proper-list? #t) #f)
(test (proper-list? '(a . b)) #f)
(test (proper-list? #(a b))  #f)
(test (proper-list? (list 1 2)) #t)
(test (proper-list? (list)) #t)
(test (proper-list? ''foo) #t)
(test (proper-list? ''2) #t)
(test (proper-list? (list 'a 'b 'c 'd 'e 'f)) #t)
(test (proper-list? '(this-that)) #t)
(test (proper-list? '(this - that)) #t)
(let ((x (list 1 2)))
  (set-cdr! x x)
  (test (proper-list? x) #f))
(test (proper-list? (list 1 (cons 1 2))) #t)
(test (proper-list? (list 1 (cons 1 ()))) #t)
(test (proper-list? (cons 1 ())) #t)
(test (proper-list? (cons () ())) #t)
(test (proper-list? (cons () 1)) #f)
(test (proper-list? (list (list))) #t)
(test (proper-list? '(())) #t)
(test (proper-list? '(1 2 . 3)) #f)
(test (proper-list? (cons 1 (cons 2 3))) #f)
(test (proper-list? '(1 . ())) #t)

(test (proper-list? (let ((lst (list 1))) (set-cdr! lst lst) lst)) #f)
(test (proper-list? (let ((lst (list 1 2))) (set-cdr! (cdr lst) lst) lst)) #f)
(test (proper-list? (let ((lst (list 1 2 3))) (set-cdr! (cddr lst) lst) lst)) #f)
(test (proper-list? (let ((lst (list 1 2 3 4))) (set-cdr! (cdddr lst) lst) lst)) #f)
(test (proper-list? (let ((lst (list 1 2 3 4 5))) (set-cdr! (cdr (cdddr lst)) lst) lst)) #f)

(test (proper-list? '(1 2) ()) 'error)
(test (proper-list?) 'error)
(for-each
 (lambda (arg)
   (if (proper-list? arg)
       (format #t ";(list? ~A) -> #t?~%" arg)))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #\f #t :hi (if #f #f) (lambda (a) (+ a 1))))


;;; --------------------------------------------------------------------------------
;;; null?

(test (null? 'a) '#f)
(test (null? ()) #t)
(test (null? ()) #t)
(test (null? '(a b c)) #f)
(test (null? (cons 1 2)) #f)
(test (null? ''()) #f)
(test (null? #f) #f)
(test (null? (make-vector 6)) #f)
(test (null? #t) #f)
(test (null? '(a . b)) #f)
(test (null? #(a b))  #f)
(test (null? (list 1 2)) #f)
(test (null? (list)) #t)
(test (null? ''foo) #f)
(test (null? (list 'a 'b 'c 'd 'e 'f)) #f)
(test (null? '(this-that)) #f)
(test (null? '(this - that)) #f)
(let ((x (list 1 2)))
  (set-cdr! x x)
  (test (null? x) #f))
(test (null? (list 1 (cons 1 2))) #f)
(test (null? (list 1 (cons 1 ()))) #f)
(test (null? (cons 1 ())) #f)
(test (null? (cons () ())) #f)
(test (null? (cons () 1)) #f)
(test (null? (list (list))) #f)
(test (null? '(())) #f)
(test (null? #()) #f)
(test (null? (make-vector '(2 0 3))) #f)
(test (null? "") #f)
(test (null? lambda) #f)
(test (null? cons) #f)
(test (null? (begin)) #t)
(test (null? (cdr (list 1))) #t)
(test (null? (cdr (cons () '(())))) #f)

(test (null? () ()) 'error)
(test (null?) 'error)

(for-each
 (lambda (arg)
   (if (null? arg)
       (format #t ";(null? ~A) -> #t?~%" arg)))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #\f #t (if #f #f) :hi #<eof> #<undefined> (values) (lambda (a) (+ a 1))))



;;; --------------------------------------------------------------------------------
;;; set-car!

(test (let ((x (cons 1 2))) (set-car! x 3) x) (cons 3 2))
(test (let ((x (list 1 2))) (set-car! x 3) x) (list 3 2))
(test (let ((x (list (list 1 2) 3))) (set-car! x 22) x) (list 22 3))
(test (let ((x (cons 1 2))) (set-car! x ()) x) (cons () 2))
(test (let ((x (list 1 (list 2 3 4)))) (set-car! x (list 5 (list 6))) x) (list (list 5 (list 6)) (list 2 3 4)))
(test (let ((x '(((1) 2) (3)))) (set-car! x '((2) 1)) x) '(((2) 1) (3)))
(test (let ((x ''foo)) (set-car! x "hi") x) (list "hi" 'foo))
(test (let ((x '((1 . 2) . 3))) (set-car! x 4) x) '(4 . 3))
(test (let ((x '(1 . 2))) (set-car! x (cdr x)) x) '(2 . 2))
(test (let ((x '(1 . 2))) (set-car! x x) (proper-list? x)) #f)
(test (let ((x (list 1))) (set-car! x ()) x) '(()))
(test (let ((x '(((1 2) . 3) 4))) (set-car! x 1) x) '(1 4))
(test (let ((lst (cons 1 (cons 2 3)))) (set-car! (cdr lst) 4) lst) (cons 1 (cons 4 3)))
(test (let ((lst (cons 1 (cons 2 3)))) (set-car! lst 4) lst) (cons 4 (cons 2 3)))
(test (let ((x (list 1 2))) (set! (car x) 0) x) (list 0 2))
(test (let ((x (cons 1 2))) (set! (cdr x) 0) x) (cons 1 0))
(test (let ((x (list 1 2))) (set-car! x (list 3 4)) x) '((3 4) 2))
(test (let ((x (cons 1 2))) (set-car! x (list 3 4)) x) '((3 4) . 2))
(test (let ((x (cons (list 1 2) 3))) (set-car! (car x) (list 3 4)) x) '(((3 4) 2) . 3))
(test (let ((lst (list 1 2 3))) (set! (car lst) 32) lst) '(32 2 3))

(test (set-car! '() 32) 'error)
(test (set-car! () 32) 'error)
(test (set-car! (list) 32) 'error)
(test (set-car! 'x 32) 'error)
(test (set-car! #f 32) 'error)
(test (set-car!) 'error)
(test (set-car! '(1 2) 1 2) 'error)
(test (let ((lst (list 1 2))) (set-car! lst (values 2 3)) lst) 'error)
(test (let ((lst '(1 2))) (set-car! lst 32)) 32)
(test (let ((lst '(1 2))) (set! (car lst) 32)) 32)

(test (let ((c (cons 1 2))) (set-car! c #\a) (car c)) #\a)
(test (let ((c (cons 1 2))) (set-car! c #()) (car c)) #())
(test (let ((c (cons 1 2))) (set-car! c #f) (car c)) #f)
(test (let ((c (cons 1 2))) (set-car! c _ht_) (car c)) _ht_)



;;; --------------------------------------------------------------------------------
;;; set-cdr!

(test (let ((x (cons 1 2))) (set-cdr! x 3) x) (cons 1 3))
(test (let ((x (list 1 2))) (set-cdr! x 3) x) (cons 1 3))
(test (let ((x (list (list 1 2) 3))) (set-cdr! x 22) x) '((1 2) . 22))
(test (let ((x (cons 1 2))) (set-cdr! x '()) x) (list 1))
(test (let ((x (list 1 (list 2 3 4)))) (set-cdr! x (list 5 (list 6))) x) '(1 5 (6)))
(test (let ((x '(((1) 2) (3)))) (set-cdr! x '((2) 1)) x) '(((1) 2) (2) 1))
(test (let ((x ''foo)) (set-cdr! x "hi") x) (cons 'quote "hi"))
(test (let ((x '((1 . 2) . 3))) (set-cdr! x 4) x) '((1 . 2) . 4))
(test (let ((x '(1 . 2))) (set-cdr! x (cdr x)) x) '(1 . 2))
(test (let ((x '(1 . 2))) (set-cdr! x x) (proper-list? x)) #f)
(test (let ((x (list 1))) (set-cdr! x '()) x) (list 1))
(test (let ((x '(1 . (2 . (3 (4 5)))))) (set-cdr! x 4) x) '(1 . 4))
(test (let ((lst (cons 1 (cons 2 3)))) (set-cdr! (cdr lst) 4) lst) (cons 1 (cons 2 4)))
(test (let ((x (cons (list 1 2) 3))) (set-cdr! (car x) (list 3 4)) x) '((1 3 4) . 3))
(test (let ((x (list 1 2))) (set-cdr! x (list 4 5)) x) '(1 4 5))
(test (let ((x (cons 1 2))) (set-cdr! x (list 4 5)) x) '(1 4 5)) ;!
(test (let ((x (cons 1 2))) (set-cdr! x (cons 4 5)) x) '(1 4 . 5))
(test (let ((lst (list 1 2 3))) (set! (cdr lst) 32) lst) (cons 1 32))

(test (set-cdr! '() 32) 'error)
(test (set-cdr! () 32) 'error)
(test (set-cdr! (list) 32) 'error)
(test (set-cdr! 'x 32) 'error)
(test (set-cdr! #f 32) 'error)
(test (set-cdr!) 'error)
(test (set-cdr! '(1 2) 1 2) 'error)
(test (let ((lst '(1 2))) (set-cdr! lst 32)) 32)
(test (let ((lst '(1 2))) (set! (cdr lst) 32)) 32)

(test (let ((c (cons 1 2))) (set-cdr! c #\a) (cdr c)) #\a)
(test (let ((c (cons 1 2))) (set-cdr! c #()) (cdr c)) #())
(test (let ((c (cons 1 2))) (set-cdr! c #f) (cdr c)) #f)
(test (let ((c (cons 1 2))) (set-cdr! c _ht_) (cdr c)) _ht_)
(test (let ((c (cons 1 2))) (set-cdr! c (list 3)) c) '(1 3))

;;; this is a version of the (list-set! '(1 2 3) ...) problem
;;;   (let () (define (func) (set-cdr! (quasiquote (int-vector)) imb)) (define (hi) (func) (func)) (hi) (hi))


;;; --------------------------------------------------------------------------------
;;; list-ref

(test (list-ref (list 1 2) 1) 2)
(test (list-ref '(a b c d) 2) 'c)
(test (list-ref (cons 1 2) 0) 1) ; !!
(test (list-ref ''foo 0) 'quote)
(test (list-ref '((1 2) (3 4)) 1) '(3 4))
(test (list-ref (list-ref (list (list 1 2) (list 3 4)) 1) 1) 4)
(test (let ((x (list 1 2 3))) (list-ref x (list-ref x 1))) 3)
(test (list-ref '(1 2 . 3) 1) 2)
(test (list-ref '(1 2 . 3) 2) 'error) ; hmm...
(test ('(1 2 . 3) 0) 1)
(test ('(1 . 2) 0) 1)

(test (let ((lst (list 1 2))) (set! (list-ref lst 1) 0) lst) (list 1 0))
(test (((lambda () list)) 'a 'b 'c) '(a b c))
(test (apply ((lambda () list)) (list 'a 'b 'c) (list 'c 'd 'e)) '((a b c) c d e))
(test (((lambda () (values list))) 1 2 3) '(1 2 3))
(test (apply list 'a 'b '(c)) '(a b c))

(for-each
 (lambda (name op1 op2)
   (for-each
    (lambda (lst)
      (let ((val1 (catch #t (lambda () (op1 lst)) (lambda args 'error)))
	    (val2 (catch #t (lambda () (op2 lst)) (lambda args 'error))))
	(if (not (equal? val1 val2))
	    (format #t ";(~A ~A) -> ~A ~A?~%" name lst val1 val2))))
    lists))
 (list 'list-ref:0 'list-ref:1 'list-ref:2 'list-ref:3)
 (list car cadr caddr cadddr)
 (list (lambda (l) (list-ref l 0)) (lambda (l) (list-ref l 1)) (lambda (l) (list-ref l 2)) (lambda (l) (list-ref l 3))))

(for-each
 (lambda (arg)
   (test (list-ref (list 1 arg) 1) arg))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand (log 0)
       3.14 3/4 1.0+1.0i #\f #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (let ((x '(1 . 2))) (set-cdr! x x) (list-ref x 0)) 1)
(test (let ((x '(1 . 2))) (set-cdr! x x) (list-ref x 1)) 1)
(test (let ((x '(1 . 2))) (set-cdr! x x) (list-ref x 100)) 1)

(test (list-ref '((1 2 3) (4 5 6)) 1) '(4 5 6))
(test (list-ref '((1 2 3) (4 5 6)) 1 2) 6)
(test (list-ref '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1) '((7 8 9) (10 11 12)))
(test (list-ref '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1 0) '(7 8 9))
(test (list-ref '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1 0 2) 9)
(test (list-ref '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1 0 3) 'error)
(test (list-ref '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1 0 2 0) 'error)

(test ('((1 2 3) (4 5 6)) 1) '(4 5 6))
(test ('((1 2 3) (4 5 6)) 1 2) 6)
(test ('(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1) '((7 8 9) (10 11 12)))
(test ('(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1 0) '(7 8 9))
(test ('(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1 0 2) 9)
(test ('(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1 0 3) 'error)
(test ('(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1 0 2 0) 'error)

(test (let ((L '((1 2 3) (4 5 6)))) (L 1)) '(4 5 6))
(test (let ((L '((1 2 3) (4 5 6)))) (L 1 2)) 6)
(test (let ((L '((1 2 3) (4 5 6)))) (L 1 2 3)) 'error)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (L 1)) '((7 8 9) (10 11 12)))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (L 1 0)) '(7 8 9))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (L 1 0 2)) 9)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (L 1 0 2 3)) 'error)

(test (let ((L '((1 2 3) (4 5 6)))) ((L 1) 2)) 6)
(test (let ((L '((1 2 3) (4 5 6)))) (((L 1) 2) 3)) 'error)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((L 1) 0)) '(7 8 9))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (((L 1) 0) 2)) 9)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((L 1 0) 2)) 9)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((L 1) 0 2)) 9)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((((L 1) 0) 2) 3)) 'error)

(test (let ((L '((1 2 3) (4 5 6)))) (list-ref (L 1) 2)) 6)
(test (let ((L '((1 2 3) (4 5 6)))) (list-ref ((L 1) 2) 3)) 'error)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-ref (L 1) 0)) '(7 8 9))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((list-ref (L 1) 0) 2)) 9)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-ref (((L 1) 0) 2) 3)) 'error)

(let ((zero 0)
      (one 1)
      (two 2)
      (three 3))
  (test (list-ref '((1 2 3) (4 5 6)) one) '(4 5 6))
  (test (list-ref '((1 2 3) (4 5 6)) 1 two) 6)
  (test (list-ref '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) one) '((7 8 9) (10 11 12)))
  (test (list-ref '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) one zero) '(7 8 9))
  (test (list-ref '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) one zero two) 9)
  (test (list-ref '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) one zero three) 'error)

  (test ('((1 2 3) (4 5 6)) one) '(4 5 6))
  (test ('((1 2 3) (4 5 6)) 1 two) 6)
  (test ('(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) one) '((7 8 9) (10 11 12)))
  (test ('(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) one zero) '(7 8 9))
  (test ('(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) one zero two) 9)
  (test ('(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) one zero three) 'error)

  (test (let ((L '((1 2 3) (4 5 6)))) (L one)) '(4 5 6))
  (test (let ((L '((1 2 3) (4 5 6)))) (L 1 two)) 6)
  (test (let ((L '((1 2 3) (4 5 6)))) (L 1 2 3)) 'error)
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (L one)) '((7 8 9) (10 11 12)))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (L one zero)) '(7 8 9))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (L one zero two)) 9)

  (test (let ((L '((1 2 3) (4 5 6)))) ((L one) two)) 6)
  (test (let ((L '((1 2 3) (4 5 6)))) (((L one) two) 3)) 'error)
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((L one) zero)) '(7 8 9))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (((L one) zero) two)) 9)
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((L one zero) two)) 9)
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((L one) 0 two)) 9)

  (test (let ((L '((1 2 3) (4 5 6)))) (list-ref (L one) two)) 6)
  (test (let ((L '((1 2 3) (4 5 6)))) (list-ref ((L one) two) 3)) 'error)
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-ref (L one) zero)) '(7 8 9))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((list-ref (L one) zero) two)) 9))


(test (list-ref () 0) 'error)
(test (list-ref (list 1 2) 2) 'error)
(test (list-ref (list 1 2) -1) 'error)
(test (list-ref (list 1 2) 1.3) 'error)
(test (list-ref (list 1 2) 1/3) 'error)
(test (list-ref (list 1 2) 1+2.0i) 'error)
(test (list-ref (cons 1 2) 1) 'error)
(test (list-ref (cons 1 2) 2) 'error)
(test (list-ref (list 1 2 3) (expt 2 32)) 'error)
(test (list-ref '(1 2 3) 1 2) 'error)
(test (list-ref) 'error)
(test (list-ref '(1 2)) 'error)
(test ('(0)) 'error)
(test ((0)) 'error)
(test (list-ref '((1 2) (3 4)) 1 1) 4)
(test ('(1 2 3) 1) 2)
(test ((list 1 2 3) 2) 3)
(test ((list)) 'error)
(test ((list 1) 0 0) 'error)
(test ((list 1 (list 2 3)) 1 1) 3)
(test ((append '(3) () '(1 2)) 0) 3)
(test ((append '(3) () 1) 0) 3)
(test ((append '(3) () 1) 1) 'error)
;; this works with 0 because:
(test ((cons 1 2) 0) 1)
(test (list-ref (cons 1 2) 0) 1)
(test (((list (list 1 2 3)) 0) 0) 1)
(test (((list (list 1 2 3)) 0) 1) 2)
(test (((list (list 1 2 3)) 0 1)) 'error) ; see below
(test (let ((lst (list (list 1 2 3)))) (lst 0 1)) 2)
(test ((list (list 1 2 3)) 0 1) 2)
(test (list-ref (list (list 1 2)) 0 ()) 'error)
(test (((list +) 0) 1 2 3) 6)


(let ((lst (list 1 2)))
  (for-each
   (lambda (arg)
     (test (list-ref (list 1 2) arg) 'error)
     (test ((list 1 2) arg) 'error)
     (test (lst arg) 'error))
   (list "hi" (integer->char 65) #f '(1 2) () 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
	 3.14 3/4 1.0+1.0i #\f #t :hi (if #f #f) (lambda (a) (+ a 1)))))




;;; --------------------------------------------------------------------------------
;;; list-set!

(test (let ((x (list 1))) (list-set! x 0 2) x) (list 2))
(test (let ((x (cons 1 2))) (list-set! x 0 3) x) '(3 . 2))
(test (let ((x (cons 1 2))) (list-set! x 1 3) x) 'error)
(test (let ((x '((1) 2))) (list-set! x 0 1) x) '(1 2))
(test (let ((x '(1 2))) (list-set! x 1 (list 3 4)) x) '(1 (3 4)))
(test (let ((x ''foo)) (list-set! x 0 "hi") x ) '("hi" foo))
(test (let ((x (list 1 2))) (list-set! x 0 x) (list? x)) #t)
(test (let ((x (list 1 2))) (list-set! x 1 x) (list? x)) #t)
(test (let ((x 2) (lst '(1 2))) (list-set! (let () (set! x 3) lst) 1 23) (list x lst)) '(3 (1 23)))
(test (apply list-set! '((1 2) (3 2)) 1 '(1 2)) 2)

(test (list-set! '(1 2 3) 1 4) 4)
(test (set-car! '(1 2) 4) 4)
(test (set-cdr! '(1 2) 4) 4)
(test (fill! (list 1 2) 4) 4)
(test (fill! () 1) 1)
(test (list-set! '(1 2 . 3) 1 23) 23)
(test (list-set! '(1 2 . 3) 2 23) 'error)
(test (set! ('(1 2 . 3) 1) 23) 23)
(test (let ((lst '(1 2 3))) (list-set! lst 0 32)) 32)
(test (let ((lst '(1 2 3))) (set! (lst 0) 32)) 32)
(test (let ((lst '(1 2 3))) (set! (list-ref lst 0) 32)) 32)

(for-each
 (lambda (arg)
   (test (let ((x (list 1 2))) (list-set! x 0 arg) (list-ref x 0)) arg))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand (log 0)
       3.14 3/4 1.0+1.0i #\f #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (let ((L '((1 2 3) (4 5 6)))) (list-set! L 1 32) L) '((1 2 3) 32))
(test (let ((L '((1 2 3) (4 5 6)))) (list-set! L 1 0 32) L) '((1 2 3) (32 5 6)))
(test (let ((L '((1 2 3) (4 5 6)))) (list-set! L 1 0 2 32) L) 'error)
(test (let ((L '((1 2 3) (4 5 6)))) (list-set! L 1 3 32) L) 'error)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-set! L 1 32) L) '(((1 2 3) (4 5 6)) 32))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-set! L 1 0 32) L) '(((1 2 3) (4 5 6)) (32 (10 11 12))))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-set! L 1 0 2 32) L) '(((1 2 3) (4 5 6)) ((7 8 32) (10 11 12))))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-set! L 1 0 2 1 32) L) 'error)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-set! L 1 4 2 32) L) 'error)

(test (let ((L '((1 2 3) (4 5 6)))) (set! (L 1) 32) L) '((1 2 3) 32))
(test (let ((L '((1 2 3) (4 5 6)))) (set! (L 1 0) 32) L) '((1 2 3) (32 5 6)))
(test (let ((L '((1 2 3) (4 5 6)))) (set! (L 1 0 2) 32) L) 'error)
(test (let ((L '((1 2 3) (4 5 6)))) (set! (L 1 3) 32) L) 'error)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (L 1) 32) L) '(((1 2 3) (4 5 6)) 32))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (L 1 0) 32) L) '(((1 2 3) (4 5 6)) (32 (10 11 12))))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (L 1 0 2) 32) L) '(((1 2 3) (4 5 6)) ((7 8 32) (10 11 12))))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (L 1 0 2 1) 32) L) 'error)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (L 1 4 2) 32) L) 'error)

(test (let ((L '((1 2 3) (4 5 6)))) (set! ((L 1) 0) 32) L) '((1 2 3) (32 5 6)))
(test (let ((L '((1 2 3) (4 5 6)))) (set! (((L 1) 0) 2) 32) L) 'error)
(test (let ((L '((1 2 3) (4 5 6)))) (set! ((L 1) 3) 32) L) 'error)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! ((L 1) 0) 32) L) '(((1 2 3) (4 5 6)) (32 (10 11 12))))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (((L 1) 0) 2) 32) L) '(((1 2 3) (4 5 6)) ((7 8 32) (10 11 12))))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! ((((L 1) 0) 2) 1) 32) L) 'error)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (((L 1) 4) 2) 32) L) 'error)
(test (let ((L '(((1 2 3))))) (set! ((L 0) 0 1) 32) L) '(((1 32 3))))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! ((L 1 0) 2) 32) L) '(((1 2 3) (4 5 6)) ((7 8 32) (10 11 12))))
(test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! (L 0 0 1) 32) L) '((((1 2 3) 32) ((7 8 9) (10 11 12))) (13 14 15)))
(test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! ((L 0) 0 1 2) 32) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
(test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! ((L 0 0) 1 2) 32) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
(test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! ((L 0 0 1) 2) 32) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
(test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! (((L 0) 0) 1 2) 32) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
(test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! (((L 0 0) 1) 2) 32) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
(test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! ((((L 0) 0) 1) 2) 32) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
(test (let ((L '(1 2 3))) (let ((L1 (list L))) (set! ((car L1) 1) 32) L)) '(1 32 3))

(let ((zero 0)
      (one 1)
      (two 2)
      (three 3)
      (thirty-two 32))
  (test (let ((L '((1 2 3) (4 5 6)))) (list-set! L one thirty-two) L) '((1 2 3) 32))
  (test (let ((L '((1 2 3) (4 5 6)))) (list-set! L one zero thirty-two) L) '((1 2 3) (32 5 6)))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-set! L one thirty-two) L) '(((1 2 3) (4 5 6)) 32))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-set! L one zero thirty-two) L) '(((1 2 3) (4 5 6)) (32 (10 11 12))))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-set! L one zero two thirty-two) L) '(((1 2 3) (4 5 6)) ((7 8 32) (10 11 12))))

  (test (let ((L '((1 2 3) (4 5 6)))) (set! (L one) thirty-two) L) '((1 2 3) 32))
  (test (let ((L '((1 2 3) (4 5 6)))) (set! (L one zero) thirty-two) L) '((1 2 3) (32 5 6)))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (L one) thirty-two) L) '(((1 2 3) (4 5 6)) 32))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (L one zero) thirty-two) L) '(((1 2 3) (4 5 6)) (32 (10 11 12))))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (L one zero two) thirty-two) L) '(((1 2 3) (4 5 6)) ((7 8 32) (10 11 12))))

  (test (let ((L '((1 2 3) (4 5 6)))) (set! ((L one) zero) thirty-two) L) '((1 2 3) (32 5 6)))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! ((L one) zero) thirty-two) L) '(((1 2 3) (4 5 6)) (32 (10 11 12))))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (((L one) zero) two) thirty-two) L) '(((1 2 3) (4 5 6)) ((7 8 32) (10 11 12))))
  (test (let ((L '(((1 2 3))))) (set! ((L zero) zero one) thirty-two) L) '(((1 32 3))))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! ((L one zero) two) thirty-two) L) '(((1 2 3) (4 5 6)) ((7 8 32) (10 11 12))))
  (test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! (L zero zero one) thirty-two) L) '((((1 2 3) 32) ((7 8 9) (10 11 12))) (13 14 15)))
  (test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! ((L zero) zero one two) thirty-two) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
  (test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! ((L zero zero) one two) thirty-two) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
  (test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! ((L zero 0 one) two) thirty-two) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
  (test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! (((L zero) zero) one two) thirty-two) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
  (test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! (((L zero zero) one) two) thirty-two) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
  (test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! ((((L zero) zero) one) two) thirty-two) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
  (test (let ((L '(1 2 3))) (let ((L1 (list L))) (set! ((car L1) one) thirty-two) L)) '(1 32 3)))

(test (let ((x '(1)) (y '(2))) (set! ((if #t x y) 0) 32) (list x y)) '((32) (2)))

(test (list-set! () 0 1) 'error)
(test (list-set! () -1 1) 'error)
(test (list-set! '(1) 1 2) 'error)
(test (list-set! '(1 2 3) -1 2) 'error)
(test (list-set! '(1) 1.5 2) 'error)
(test (list-set! '(1) 3/2 2) 'error)
(test (list-set! '(1) 1+3i 2) 'error)
(test (list-set! '(1 2 3) 1 2 3) 'error)
(test (list-set! (list 1 2 3) (expt 2 32)  0) 'error)
(test (list-set! (list 1 2) () 1) 'error)

(for-each
 (lambda (arg)
   (test (list-set! (list 1 2) arg arg) 'error)
   (test (list-set! arg 1 2) 'error)
   (test (list-set! (list 1 2) arg 1) 'error))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #\f #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (catch #t
       (lambda () (let ((L1 (list 1))) (list-set! L1 3 0)))
       (lambda (type info) (apply format #f info)))
      "list-set! second argument, 3, is out of range (it is too large)")



;;; --------------------------------------------------------------------------------
;;; list

(test (let ((tree1 (list 1 (list 1 2) (list (list 1 2 3)) (list (list (list 1 2 3 4)))))) tree1) '(1 (1 2) ((1 2 3)) (((1 2 3 4)))))
(test (let ((tree2 (list "one" (list "one" "two") (list (list "one" "two" "three"))))) tree2) '("one" ("one" "two") (("one" "two" "three"))))
(test (let ((tree1 (list 1 (list 1 2) (list 1 2 3) (list 1 2 3 4)))) tree1) '(1 (1 2) (1 2 3) (1 2 3 4)))
(test (let ((tree1 (list 1 (list 1 2))) (tree2 (list 1 (list 1 2)))) tree2) '(1 (1 2)))
(test (let ((tree1 (list 1 (list 1 2))) (tree2 (list 1 (list 1 2)))) (eqv? tree1 tree2)) #f)
(test (let ((tree1 (list ''a (list ''b ''c))) (tree2 (list ''a (list ''b ''c)))) tree2) '('a ('b 'c)))
(test (let ((lst (list 1 (list 2 3)))) lst) '(1 (2 3)))
(test (let* ((lst (list 1 (list 2 3))) (slst lst)) slst) '(1 (2 3)))
(test (list 1) '(1))
(test (let ((a 1)) (list a 2)) '(1 2))
(test (let ((a 1)) (list 'a '2)) '(a 2))
(test (let ((a 1)) (list 'a 2)) '(a 2))
(test (list) ())
(test (let ((a (list 1 2))) a) '(1 2))
(test (let ((a (list 1 2))) (list 3 4 'a (car (cons 'b 'c)) (+ 6 -2))) '(3 4 a b 4))
(test (list) ())
(test (length (list quote do map call/cc lambda define if begin set! let let* cond and or for-each)) 15)
(test (list 1(list 2)) '(1(2)))
(test (list 1 2 . 3) 'error)
;(test (list 1 2 , 3) 'error) ; ,3 -> 3 in the reader now
(test (list 1 2 ,@ 3) 'error)




;;; --------------------------------------------------------------------------------
;;; list-tail

(test (list-tail '(1 2 3) 0) '(1 2 3))
(test (list-tail '(1 2 3) 2) '(3))
(test (list-tail '(1 2 3) 3) ())
(test (list-tail '(1 2 3 . 4) 2) '(3 . 4))
(test (list-tail '(1 2 3 . 4) 3) 4)
(test (let ((x (list 1 2 3))) (eq? (list-tail x 2) (cddr x))) #t)
(test (list-tail () 0) ())
(test (list-tail () 1) 'error)
(test (list-tail '(1 2 3) 4) 'error)
(test (list-tail () -1) 'error)
(test (list-tail (list 1 2) 2) ())
(test (list-tail (cons 1 2) 0) '(1 . 2))
(test (list-tail (cons 1 2) 1) 2)
(test (list-tail (cons 1 2) 2) 'error)
(test (list-tail (cons 1 2) -1) 'error)
(test (list-tail ''foo 1) '(foo))
(test (list-tail '((1 2) (3 4)) 1) '((3 4)))
(test (list-tail (list-tail '(1 2 3) 1) 1) '(3))
(test (list-tail (list-tail (list-tail '(1 2 3 4) 1) 1) 1) '(4))
(test (list-tail '(1 2) (list-tail '(0 . 1) 1)) '(2))

(let ((x '(1 . 2))) (set-cdr! x x) (test (list-tail x 0) x))
(let ((x '(1 . 2))) (set-cdr! x x) (test (list-tail x 1) (cdr x)))
(let ((x '(1 . 2))) (set-cdr! x x) (test (list-tail x 100) x))

(let ((x (list 1 2 3)))
  (let ((y (list-tail x 1)))
    (set! (y 1) 32)
    (test (equal? y '(2 32)) #t)
    (test (equal? x '(1 2 32)) #t))) ; list-tail is not like substring

(for-each
 (lambda (name op1 op2)
   (for-each
    (lambda (lst)
      (let ((val1 (catch #t (lambda () (op1 lst)) (lambda args 'error)))
	    (val2 (catch #t (lambda () (op2 lst)) (lambda args 'error))))
	(if (not (equal? val1 val2))
	    (format #t ";(~A ~A) -> ~A ~A?~%" name lst val1 val2))))
    lists))
 (list 'list-tail:0 'list-tail:1 'list-tail:2 'list-tail:3 'list-tail:4)
 (list (lambda (l) l) cdr cddr cdddr cddddr)
 (list (lambda (l) (list-tail l 0)) (lambda (l) (list-tail l 1)) (lambda (l) (list-tail l 2)) (lambda (l) (list-tail l 3)) (lambda (l) (list-tail l 4))))

(test (list-tail (list 1 2) 3) 'error)
(test (list-tail (list 1 2) -1) 'error)
(test (list-tail (list 1 2) 1.3) 'error)
(test (list-tail (list 1 2) 1/3) 'error)
(test (list-tail (list 1 2) 1+2.0i) 'error)
(test (list-tail '(1 2 . 3)) 'error)
(test (list-tail '(1 2 . 3) 1) '(2 . 3))
(test (list-tail '(1 2 . 3) 0) '(1 2 . 3))
(test (list-tail (list 1 2 3) (+ 1 (expt 2 32))) 'error)
(test (list-tail) 'error)
(test (list-tail '(1)) 'error)
(test (list-tail '(1) 1 2) 'error)
(test (set! (list-tail (list 1 2 3)) '(32)) 'error) ; should this work?

(for-each
 (lambda (arg)
   (test (list-tail (list 1 2) arg) 'error)
   (test (list-tail arg 0) 'error))
 (list "hi" -1 3 most-negative-fixnum most-positive-fixnum
       (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #\f #t (if #f #f) #<eof> #() #(1 2 3) (lambda (a) (+ a 1))))


;;; --------------------------------------------------------------------------------
;;; make-list

(test (make-list 0) ())
(test (make-list 0 123) ())
(test (make-list 1) '(#f))
(test (make-list 1 123) '(123))
(test (make-list 1 ()) '(()))
(test (make-list 2) '(#f #f))
(test (make-list 2 1) '(1 1))
(test (make-list 2/1 1) '(1 1))
(test (make-list 2 (make-list 1 1)) '((1) (1)))
(test (make-list -1) 'error)
(test (make-list -0) ())
(test (make-list most-negative-fixnum) 'error)
(test (make-list most-positive-fixnum) 'error)
(test (make-list (* 8796093022208 8796093022208)) 'error)
(test (make-list 8796093022208) 'error)
(test (make-list 0 #\a) ())
(test (make-list 1 #\a) '(#\a))

(for-each
 (lambda (arg)
   (test (make-list arg) 'error))
 (list #\a #(1 2 3) 3.14 3/4 1.0+1.0i 0.0 1.0 () #t 'hi #(()) (list 1 2 3) '(1 . 2) "hi" (- (real-part (log 0.0)))))

(for-each
 (lambda (arg)
   (test ((make-list 1 arg) 0) arg))
 (list #\a #(1 2 3) 3.14 3/4 1.0+1.0i () #f 'hi #(()) (list 1 2 3) '(1 . 2) "hi"))

(test (make-list) 'error)
(test (make-list 1 2 3) 'error)
(test (let ((lst (make-list 2 (make-list 1 0)))) (eq? (lst 0) (lst 1))) #t)




;;; --------------------------------------------------------------------------------
;;; assq

(let ((e '((a 1) (b 2) (c 3))))
  (test (assq 'a e) '(a 1))
  (test (assq 'b e) '(b 2))
  (test (assq 'd e) #f))
(test (assq (list 'a) '(((a)) ((b)) ((c))))  #f)

(let ((xcons (cons 1 2))
      (xvect (vector 1 2))
      (xlambda (lambda () 1))
      (xstr "abs"))
  (let ((e (list (list #t 1) (list #f 2) (list 'a 3) (list xcons 4) (list xvect 5) (list xlambda 6) (list xstr 7) (list car 8))))
    (test (assq #t e) (list #t 1))
    (test (assq #f e) (list #f 2))
    (test (assq 'a e) (list 'a 3))
    (test (assq xcons e) (list xcons 4))
    (test (assq xvect e) (list xvect 5))
    (test (assq xlambda e) (list xlambda 6))
    (test (assq xstr e) (list xstr 7))
    (test (assq car e) (list car 8))))

(let ((e '((1+i 1) (3.0 2) (5/3 3))))
  (test (assq 1+i e) #f)
  (test (assq 3.0 e) #f)
  (test (assq 5/3 e) #f))

(test (assq 'x (cdr (assq 'a '((b . 32) (a . ((a . 12) (b . 32) (x . 1))) (c . 1))))) '(x . 1))

(test (assq #f '(#f 2 . 3)) #f)
(test (assq #f '((#f 2) . 3)) '(#f 2))
(test (assq () '((() 1) (#f 2))) '(() 1))
(test (assq () '((1) (#f 2))) #f)
(test (assq #() '((#f 1) (() 2) (#() 3))) #f)  ; (eq? #() #()) -> #f

(test (assq 'b '((a . 1) (b . 2) () (c . 3) #f)) '(b . 2))
(test (assq 'c '((a . 1) (b . 2) () (c . 3) #f)) '(c . 3))
(test (assq 'b '((a . 1) (b . 2) () (c . 3) . 4)) '(b . 2))
(test (assq 'c '((a . 1) (b . 2) () (c . 3) . 4)) '(c . 3))
(test (assq 'b (list '(a . 1) '(b . 2) () '(c . 3) #f)) '(b . 2))
(test (assq 'asdf (list '(a . 1) '(b . 2) () '(c . 3) #f)) #f)
(test (assq "" (list '("a" . 1) '("" . 2) '(#() . 3))) '("" . 2)) ; was #f
(test (assq 'a '((a . 1) (a . 2) (a . 3))) '(a . 1)) ; is this specified?
(test (assq 'a '((b . 1) (a . 2) (a . 3))) '(a . 2))

;; check the even/odd cases
(let ((odd '((3 . 1) (a . 2) (3.0 . 3) (b . 4) (3/4 . 5) (c . 6) (#(1) . 7) (d . 8)))
      (even '((e . 1) (3 . 2) (a . 3) (3.0 . 4) (b . 5) (3/4 . 6) (c . 7) (#(1) . 8) (d . 9))))
  (test (assq 'a odd) '(a . 2))
  (test (assq 'a even) '(a . 3))
  (test (assq 3/4 odd) #f)
  (test (assq 3/4 even) #f)
  (test (assq 3.0 odd) #f)
  (test (assq 3.0 even) #f)
  (test (assq #(1) odd) #f)
  (test (assq #(1) even) #f))



;;; --------------------------------------------------------------------------------
;;; assv

(test (assv 1 '(1 2 . 3)) #f)
(test (assv 1 '((1 2) . 3)) '(1 2))

(let ((e '((a 1) (b 2) (c 3))))
  (test (assv 'a e) '(a 1))
  (test (assv 'b e) '(b 2))
  (test (assv 'd e) #f))
(test (assv (list 'a) '(((a)) ((b)) ((c))))  #f)

(let ((xcons (cons 1 2))
      (xvect (vector 1 2))
      (xlambda (lambda () 1))
      (xstr "abs"))
  (let ((e (list (list #t 1) (list #f 2) (list 'a 3) (list xcons 4) (list xvect 5) (list xlambda 6) (list xstr 7) (list car 8))))
    (test (assv #t e) (list #t 1))
    (test (assv #f e) (list #f 2))
    (test (assv 'a e) (list 'a 3))
    (test (assv xcons e) (list xcons 4))
    (test (assv xvect e) (list xvect 5))
    (test (assv xlambda e) (list xlambda 6))
    (test (assv xstr e) (list xstr 7))
    (test (assv car e) (list car 8))))

(let ((e '((1+i 1) (3.0 2) (5/3 3) (#\a 4) ("hiho" 5))))
  (test (assv 1+i e) '(1+i 1))
  (test (assv 3.0 e) '(3.0 2))
  (test (assv 5/3 e) '(5/3 3))
  (test (assv #\a e) '(#\a 4))
  (test (assv "hiho" e) #f))

(let ((e '(((a) 1) (#(a) 2) ("c" 3))))
  (test (assv '(a) e) #f)
  (test (assv #(a) e) #f)
  (test (assv (string #\c) e) #f))

(let ((lst '((2 . a) (3 . b))))
  (set-cdr! (assv 3 lst) 'c)
  (test lst '((2 . a) (3 . c))))

(test (assv () '((() 1) (#f 2))) '(() 1))
(test (assv () '((1) (#f 2))) #f)
(test (assv #() '((#f 1) (() 2) (#() 3))) #f)  ; (eqv? #() #()) -> #f ??

(test (assv 'b '((a . 1) (b . 2) () (c . 3) #f)) '(b . 2))
(test (assv 'c '((a . 1) (b . 2) () (c . 3) #f)) '(c . 3))
(test (assv 'b '((a . 1) (b . 2) () (c . 3) . 4)) '(b . 2))
(test (assv 'c '((a . 1) (b . 2) () (c . 3) . 4)) '(c . 3))
(test (assv 'asdf '((a . 1) (b . 2) () (c . 3) . 4)) #f)
(test (assv 'd '((a . 1) (b . 2) () (c . 3) (d . 5))) '(d . 5))
(test (assv 'a '((a . 1) (a . 2) (a . 3))) '(a . 1)) ; is this specified?
(test (assv 'a '((b . 1) (a . 2) (a . 3))) '(a . 2))

(let ((odd '((3 . 1) (a . 2) (3.0 . 3) (b . 4) (3/4 . 5) (c . 6) (#(1) . 7) (d . 8)))
      (even '((e . 1) (3 . 2) (a . 3) (3.0 . 4) (b . 5) (3/4 . 6) (c . 7) (#(1) . 8) (d . 9))))
  (test (assv 'a odd) '(a . 2))
  (test (assv 'a even) '(a . 3))
  (test (assv 3 odd) '(3 . 1))
  (test (assv 3 even) '(3 . 2))
  (test (assv 3/4 odd) '(3/4 . 5))
  (test (assv 3/4 even) '(3/4 . 6))
  (test (assv 3.0 odd) '(3.0 . 3))
  (test (assv 3.0 even) '(3.0 . 4))
  (test (assv #(1) odd) #f)
  (test (assv #(1) even) #f))

(test (assv 1/0 '((1/0 . 1) (1.0 . 3))) #f)
(test (pair? (assv (real-part (log 0)) (list (cons 1/0 1) (cons (real-part (log 0)) 2) (cons -1 3)))) #t)
(test (pair? (assv (- (real-part (log 0))) (list (cons 1/0 1) (cons (real-part (log 0)) 2) (cons -1 3)))) #f)



;;; --------------------------------------------------------------------------------
;;; assoc

(let ((e '((a 1) (b 2) (c 3))))
  (test (assoc 'a e) '(a 1))
  (test (assoc 'b e) '(b 2))
  (test (assoc 'd e) #f))
(test (assoc (list 'a) '(((a)) ((b)) ((c))))  '((a)))

(let ((xcons (cons 1 2))
      (xvect (vector 1 2))
      (xlambda (lambda () 1))
      (xstr "abs"))
  (let ((e (list (list #t 1) (list #f 2) (list 'a 3) (list xcons 4) (list xvect 5) (list xlambda 6) (list xstr 7) (list car 8))))
    (test (assoc #t e) (list #t 1))
    (test (assoc #f e) (list #f 2))
    (test (assoc 'a e) (list 'a 3))
    (test (assoc xcons e) (list xcons 4))
    (test (assoc xvect e) (list xvect 5))
    (test (assoc xlambda e) (list xlambda 6))
    (test (assoc xstr e) (list xstr 7))
    (test (assoc car e) (list car 8))))

(let ((e '((1+i 1) (3.0 2) (5/3 3) (#\a 4) ("hiho" 5))))
  (test (assoc 1+i e) '(1+i 1))
  (test (assoc 3.0 e) '(3.0 2))
  (test (assoc 5/3 e) '(5/3 3))
  (test (assoc #\a e) '(#\a 4))
  (test (assoc "hiho" e) '("hiho" 5)))

(let ((e '(((a) 1) (#(a) 2) ("c" 3))))
  (test (assoc '(a) e) '((a) 1))
  (test (assoc #(a) e) '(#(a) 2))
  (test (assoc (string #\c) e) '("c" 3)))

(test (assoc 'a '((b c) (a u) (a i))) '(a u))
(test (assoc 'a '((b c) ((a) u) (a i))) '(a i))
(test (assoc (list 'a) '(((a)) ((b)) ((c))))  '((a)))
(test (assoc 5 '((2 3) (5 7) (11 13))) '(5 7))
(test (assoc 'key ()) #f)
(test (assoc 'key '(() ())) 'error)
(test (assoc () ()) #f)
(test (assoc 1 '((1 (2)) (((3) 4)))) '(1 (2)))
(test (assoc #f () 1/9223372036854775807) 'error)

(test (assoc () 1) 'error)
(test (assoc (cons 1 2) 1) 'error)
(test (assoc (let ((x (cons 1 2))) (set-cdr! x x)) 1) 'error)
(test (assoc '((1 2) .3) 1) 'error)
(test (assoc ''foo quote) 'error)
(test (assoc 3 '((a . 3)) abs =) 'error)
(test (assoc 1 '(1 2 . 3)) 'error)
(test (assoc 1 '((1 2) . 3)) '(1 2))
(test (assoc 1 '((1) (1 3) (1 . 2))) '(1))
(test (assoc 1 '((1 2 . 3) (1 . 2))) '(1 2 . 3))
(test (assoc '(((1 2))) '((1 2) ((1 2) 3) (((1 2) 3) 4) ((((1 2) 3) 4) 5))) #f)
(test (assoc '(((1 2))) '((1 2) ((1 2)) (((1 2))) ((((1 2)))))) '((((1 2)))))
(test (assoc 'a '((a . 1) (a . 2) (a . 3))) '(a . 1)) ; is this specified?
(test (assoc 'a '((b . 1) (a . 2) (a . 3))) '(a . 2))

(test (assoc () '((() 1) (#f 2))) '(() 1))
(test (assoc () '((1) (#f 2))) #f)
(test (assoc #() '((#f 1) (() 2) (#() 3))) '(#() 3))
(test (assoc #<unspecified> (list (cons (apply values ()) #f))) '(#<unspecified> . #f))
(test (assoc #<a> '((1 2) (#<a> 3))) '(#<a> 3))
(test (assoc #<...> '((1 x) (#<...> 4)) equivalent?) '(#<...> 4))
(test (assoc #<> '((1 2) (#<> 3))) '(#<> 3))
(test (assoc #<a> '((1 2) (#<b> 3))) #f)
(test (assoc #<undefined> '((#<undefined> 3))) '(#<undefined> 3))
(test (assoc #<eof> '((1 2) (#<eof> 3))) '(#<eof> 3))

(for-each
 (lambda (arg)
   (test (assoc arg (list (list 1 2) (list arg 3))) (list arg 3)))
 (list "hi" (integer->char 65) #f 'a-symbol #() abs 3/4 #\f #t (if #f #f)))

(test (assoc 'b '((a . 1) (b . 2) () (c . 3) #f)) '(b . 2))
(test (assoc 'c '((a . 1) (b . 2) () (c . 3) #f)) '(c . 3))
(test (assoc 'b '((a . 1) (b . 2) () (c . 3) . 4)) '(b . 2))
(test (assoc 'c '((a . 1) (b . 2) () (c . 3) . 4)) '(c . 3))
(test (assoc 'c '((a . 1) (b . 2) () (c . 3) (c . 4) . 4)) '(c . 3))
(test (assoc 'asdf '((a . 1) (b . 2) () (c . 3) (c . 4) . 4)) #f)
(test (assoc "" (list '("a" . 1) '("" . 2) '(#() . 3))) '("" . 2))
(test (assoc assoc (list (cons abs 1) (cons assoc 2) (cons + 3))) (cons assoc 2))

(let ((odd '((3 . 1) (a . 2) (3.0 . 3) (b . 4) (3/4 . 5) (c . 6) (#(1) . 7) (d . 8)))
      (even '((e . 1) (3 . 2) (a . 3) (3.0 . 4) (b . 5) (3/4 . 6) (c . 7) (#(1) . 8) (d . 9))))
  (test (assoc 'a odd) '(a . 2))
  (test (assoc 'a even) '(a . 3))
  (test (assoc 3 odd) '(3 . 1))
  (test (assoc 3 even) '(3 . 2))
  (test (assoc 3/4 odd) '(3/4 . 5))
  (test (assoc 3/4 even) '(3/4 . 6))
  (test (assoc 3.0 odd =) '(3 . 1))
  (test (assoc 3.0 odd) '(3.0 . 3))
  (test (assoc 3.0 even) '(3.0 . 4))
  (test (assoc #(1) odd) '(#(1) . 7))
  (test (assoc #(1) even) '(#(1) . 8)))

(test (assoc 3 '((1 . a) (2 . b) (3 . c) (4 . d)) =) '(3 . c))
(test (assoc 3 '((1 . a) (2 . b) (31 . c) (4 . d)) =) #f)
(test (assoc 3 () =) #f)
(test (assoc 3.0 '((1 . a) (2 . b) (3 . c) (4 . d)) =) '(3 . c))
(test (assoc #\a '((#\A . 1) (#\b . 2)) char=?) #f)
(test (assoc #\a '((#\A . 1) (#\b . 2)) char-ci=?) '(#\A . 1))
(test (assoc #\a '((#\A . 1) (#\b . 2)) (lambda (a b) (char-ci=? a b))) '(#\A . 1))
(test (assoc 3 '((1 . a) (2 . b) (3 . c) (4 . d)) #(1)) 'error)
(test (assoc 3 '((1 . a) (2 . b) (3 . c) (4 . d)) abs) 'error)
(test (assoc 3 '((1 . a) (2 . b) (3 . c) (4 . d)) quasiquote) 'error)
(test (assoc 3 '((1 . a) (2 . b) (3 . c) (4 . d)) (lambda (a b c) (= a b))) 'error)
(test (assoc 3.0 '((1 . a) (2 . b) (3 . c) (4 . d)) (lambda* (a b c) (= a b))) '(3 . c))
(test (assoc 3 '((1 . a) (2 . b) (3 . c) (4 . d)) (lambda (a) (= a 1))) 'error)
(test (assoc 4.0 '((1 . a) (2 . b) (3 . c) (4 . d)) (dilambda = =)) '(4 . d))
(test (catch #t (lambda () (assoc 4.0 '((1 . a) (2 . b) (3 . c) (4 . d)) (lambda (a b) (error 'assoc a)))) (lambda args (car args))) 'assoc)
(test (call-with-exit (lambda (go) (assoc 4.0 '((1 . a) (2 . b) (3 . c) (4 . d)) (lambda (a b) (go 'assoc))))) 'assoc)
(test (assoc 3 '((#\a . 3) (#() . 2) (3.0 . 1) ("3" . 0))) #f)
(test (assoc 3 '((#\a . 3) (#() . 2) (3.0 . 1) ("3" . 0)) (lambda (a b) (= a b))) 'error)
(test (assoc 3 '((#\a . 3) (#() . 2) (3.0 . 1) ("3" . 0)) (lambda (a b) (and (number? b) (= a b)))) '(3.0 . 1)) ; is this order specified?
(test (let ((lst (list (cons 1 2) (cons 3 4) (cons 5 6)))) (set! (cdr (cdr lst)) lst) (assoc 3 lst)) '(3 . 4))
(test (let ((lst '((1 . 2) (3 . 4) . 5))) (assoc 3 lst)) '(3 . 4))
(test (let ((lst '((1 . 2) (3 . 4) . 5))) (assoc 5 lst)) #f)
(test (let ((lst '((1 . 2) (3 . 4) . 5))) (assoc 3 lst =)) '(3 . 4))
(test (let ((lst '((1 . 2) (3 . 4) . 5))) (assoc 5 lst =)) #f)
(test (assoc 3 '((1 . 2) . 3)) #f)
(test (assoc 1 '((1 . 2) . 3)) '(1 . 2))
(test (assoc 3 '((1 . 2) . 3) =) #f)
(test (assoc 1 '((1 . 2) . 3) =) '(1 . 2))
(test (let ((lst (list (cons 1 2) (cons 2 3) (cons 3 4)))) (and (assoc 2 lst =) lst)) '((1 . 2) (2 . 3) (3 . 4)))
(test (let ((lst (list (cons 1 2) (cons 2 3) (cons 3 4)))) (set! (cdr (cdr lst)) lst) (assoc 2 lst)) '(2 . 3))
(test (let ((lst (list (cons 1 2) (cons 2 3) (cons 3 4)))) (set! (cdr (cdr lst)) lst) (assoc 2 lst =)) '(2 . 3))
(test (let ((lst (list (cons 1 2) (cons 2 3) (cons 3 4)))) (set! (cdr (cdr lst)) lst) (assoc 4 lst)) #f)
(test (let ((lst (list (cons 1 2) (cons 2 3) (cons 3 4)))) (set! (cdr (cdr lst)) lst) (assoc 4 lst =)) #f)
(test (let ((lst (list (cons 1 2) (cons 2 3) (cons 3 4)))) (set! (cdr (cdr (cdr lst))) lst) (assoc 3 lst =)) '(3 . 4))
(test (assoc '(1 2) '((a . 3) ((1 2) . 4))) '((1 2) . 4))
(test (assoc '(1 2) '((a . 3) ((1 2) . (3 4)))) '((1 2) 3 4))
(test (assoc '(1 2) '((a . 3) ((1 2) . (3 . 4)))) '((1 2) 3 . 4))
(test (cdr (assoc '(1 2) '((a . 3) ((1 2) . (3 . 4))))) (cons 3 4))

(test (assoc #t (list 1 2) #()) 'error)
(test (assoc #t (list 1 2) (integer->char 127)) 'error)
(test (assoc #t (list 1 2) (lambda (x y) (+ x 1))) 'error) ; (+ #t 1)
(test (assoc #t (list 1 2) abs) 'error)
(test (assoc #t (list 1 2) (lambda args args)) 'error)
(test (assoc 1 '((3 . 2) 3) =) 'error)
(test (assoc 1 '((1 . 2) 3) =) '(1 . 2)) ; this is like other trailing error unchecked cases -- should we check?

(test (assoc 'a '((c 3) (a 1) (b 2)) (lambda (a) (eq? a b))) 'error)
(test (assoc 'a '((c 3) (a 1) (b 2)) (lambda (a b) (eq? a b))) '(a 1))
(test (assoc 'a '((c 3) (a 1) (b 2)) (lambda (a b c) (eq? a b))) 'error)
(test (assoc 'a '((c 3) (a 1) (b 2)) (lambda (a b c . d) (eq? a b))) 'error)
(test (assoc 'a '((c 3) (a 1) (b 2)) (lambda (a b . c) (eq? a b))) '(a 1))
(test (assoc 'a '((c 3) (a 1) (b 2)) (lambda a (apply eq? a))) '(a 1))
(test (assoc 'a '((c 3) (a 1) (b 2)) (lambda (a . b) (eq? a (car b)))) '(a 1))

(test (assoc 'a '((c 3) (a 1) (b 2)) (lambda* (a) (eq? a b))) 'error)
(test (assoc 'a '((c 3) (a 1) (b 2)) (lambda* (a b) (eq? a b))) '(a 1))
(test (assoc 'a '((c 3) (a 1) (b 2)) (lambda* (a b c) (eq? a b))) '(a 1))
(test (assoc 'a '((c 3) (a 1) (b 2)) (lambda* (a b c . d) (eq? a b))) '(a 1))
(test (assoc 'a '((c 3) (a 1) (b 2)) (lambda* (a b . c) (eq? a b))) '(a 1))
(test (assoc 'a '((c 3) (a 1) (b 2)) (lambda* a (apply eq? a))) '(a 1))
(test (assoc 'a '((c 3) (a 1) (b 2)) (lambda* (a . b) (eq? a (car b)))) '(a 1))
(test (assoc 'a '((c 3) (a 1) (b 2)) (lambda* (a :rest b) (eq? a (car b)))) '(a 1))
(test (assoc 'a '((c 3) (a 1) (b 2)) (lambda* (a :rest b :rest c) (eq? a (car b)))) '(a 1))
(test (assoc 'a '((c 3) (a 1) (b 2)) (lambda* (:rest a) (apply eq? a))) '(a 1))
(test (assoc 'a '((a 1) (b 2) (c 3)) (define-macro (_m_ a b) `(eq? ',a ',b))) '(a 1))
(test (assoc 'c '((a 1) (b 2) (c 3)) (define-macro (_m_ a b) `(eq? ',a ',b))) '(c 3))

(when with-bignums
  (test (assoc 1 (list (list (bignum 1) 2))) '(1 2))
  (test (assoc 1 (list (list (bignum 1) 2)) =) '(1 2))
  (test (assoc 1 (list (list (bignum 1) 2)) eqv?) '(1 2))
  (test (assoc 1 (list (list (bignum 1) 2)) equivalent?) '(1 2))
  (test (assv 1 (list (list (bignum 1) 2))) '(1 2))
  (test (assv (bignum 1) (list (list 1 3))) '(1 3)))

(let ()
  (define (atest a b)
    (eq? a b))
  (atest 1 1)
  (let ((lst (list (cons 'a 1) (cons 'b 2))))
    (test (assoc 'b lst atest) '(b . 2))))

(for-each
  (lambda (arg lst)
    (test (assoc arg lst eq?) (assq arg lst))
    (test (assoc arg lst eqv?) (assv arg lst))
    (test (assoc arg lst equal?) (assoc arg lst)))
  (list 'a #f (list 'a) 'a 1 3/4 #(1) "hi")
  (list '((a . 1) (b . 2) (c . 3)) '((1 . 1) ("hi" . 2) (#t . 4) (#f . 5) (2 . 3))
	'((b . 1) ((a) . 2) (c . 3)) '((d . 1) (a . 2) (b . 4) . (c . 3))
	'((1 . 1) (3/4 . 2) (23 . 3)) '((a . 1) (1 . 2) (#(1) . 4) (23 . 3))
	'((1 . 1) ("hi" . 2) (23 . 3))))

(test (catch #t (lambda () (assoc 1 (list (list 3 2) (list 2)) (lambda (a b) (catch #t 1 cons)))) (lambda args 'error)) '(3 2))
(test (catch #t (lambda () (member 1 (list 3 2) (lambda (a b) (catch #t 1 cons)))) (lambda args 'error)) '(3 2))
;;; are those correct??  at least it doesn't die.


;;; --------------------------------------------------------------------------------
;;; memq

(test (memq 'a '(a b c)) '(a b c))
(test (memq 'a (list 'a 'b 'c)) '(a b c))
(test (memq 'b '(a b c)) '(b c))
(test (memq 'a '(b c d)) #f)
(test (memq (list 'a) '(b (a) c))  #f)
(test (memq 'a '(b a c a d a)) '(a c a d a))
(let ((v (vector 'a))) (test (memq v (list 'a 1.2 v "hi")) (list v "hi")))
(test (memq #f '(1 a #t "hi" #f 2)) '(#f 2))
(test (memq eq? (list 2 eqv? 1 eq?)) (list eq?))
(test (memq eq? (list 2 eqv? 2)) #f)
(test (memq 6 (memq 5 (memq 4 (memq 3 (memq 2 (memq 1 '(1 2 3 4 5 6))))))) '(6))
(test (memq 1/2 (list (/ 2.0) .5 1/2)) #f)
(test (memq 'a (cons 'a 'b)) '(a . b))
(test (memq) 'error)
(test (memq 'a) 'error)
(test (memq 'a 'b) 'error)
(test (memq 'a '(a b . c)) '(a b . c))
(test (memq 'b '(a b . c)) '(b . c))
(test (memq 'c '(a b . c)) #f) ; or should it be 'c?
(test (memq () '(1 () 3)) '(() 3))
(test (memq () '(1 2)) #f)
(test (memq 'a '(c d a b c)) '(a b c))
(test (memq 'a '(c d f b c)) #f)
(test (memq 'a ()) #f)
(test (memq 'a '(c d a b . c)) '(a b . c))
(test (memq 'a '(c d f b . c)) #f)
(test (memq #f '(1 "hi" #t)) #f)
(test (memq () ()) #f)
(test (memq () (list)) #f)
(test (memq () (list ())) '(()))
(test (let ((x (cons 1 2))) (memq x (list x (cons 3 4)))) '((1 . 2) (3 . 4)))
(test (pair? (let ((x (lambda () 1))) (memq x (list 1 2 x 3)))) #t)
(test (memq memq (list abs + memq car)) (list memq car))
(test (memq 'a '(a a a)) '(a a a)) ;?
(test (memq 'a '(b a a)) '(a a))
(test (memq "hi" '(1 "hi" 2)) #f)
(test (let ((str "hi")) (memq str (list 1 str 2))) '("hi" 2))
(test (memq #\a '(1 #f #\a 2)) '(#\a 2))

(test (let* ((x (vector 1 2 3)) (lst (list 1 "hi" x (vector 1 2)))) (memq x lst)) '(#(1 2 3) #(1 2)))
(test (let* ((x (vector 1 2 3)) (lst (list 1 "hi" (vector 1 2 3)))) (memq x lst)) #f)

(let ((odd '(3 a 3.0 b 3/4 c #(1) d))
      (even '(e 3 a 3.0 b 3/4 c #(1) d)))
  (test (memq 'a odd) '(a 3.0 b 3/4 c #(1) d))
  (test (memq 'a even) '(a 3.0 b 3/4 c #(1) d))
  (test (memq 3/4 odd) #f)
  (test (memq 3/4 even) #f)
  (test (memq 3.0 odd) #f)
  (test (memq 3.0 even) #f)
  (test (memq #(1) odd) #f)
  (test (memq #(1) even) #f))

;;; but (memq pi (list 1 pi 2)) -> '(3.1415926535898 2)

(test (memq (values #\a '(#\A 97 a))) #f)
(test (memq (values #\a '(#\A 97 #\a))) '(#\a))
(test (memq #\a (values #\a '(#\A 97 #\a))) 'error)
(test (memq #\a (values '(#\A 97 #\a))) '(#\a))
(test (memq #\a '(1 2) (values)) 'error) ; hmmm
(test ((values memq (values #\a '(#\A 97 #\a)))) '(#\a))



;;; --------------------------------------------------------------------------------
;;; memv

(test (memv 101 '(100 101 102)) '(101 102))
(test (memv 101 (list 100 101 102)) '(101 102))
(test (memv 3.4 '(1.2 2.3 3.4 4.5)) '(3.4 4.5))
(test (memv 3.4 '(1.3 2.5 3.7 4.9)) #f)
(test (memv 1/2 (list (/ 2.0) .5 1/2)) '(1/2))
(test (memv 1.0 '(1 2 3)) #f)
(test (memv 1/0 '(1/0 1.0 3)) #f)
(test (pair? (memv (real-part (log 0)) (list 1/0 (real-part (log 0)) -1))) #t)
(test (pair? (memv (- (real-part (log 0))) (list 1/0 (real-part (log 0)) -1))) #f)

(let ((ls (list 'a 'b 'c)))
  (set-car! (memv 'b ls) 'z)
  (test ls '(a z c)))
(test (memv 1 (cons 1 2)) '(1 . 2))
(test (memv 'a (list 'a 'b . 'c)) 'error)
(test (memv 'a '(a b . c)) '(a b . c))
(test (memv 'asdf '(a b . c)) #f)
(test (memv) 'error)
(test (memv 'a) 'error)
(test (memv 'a 'b) 'error)
(test (memv 'c '(a b c)) '(c))
(test (memv 'c '(a b . c)) #f)
(test (memv ''a '('a b c)) #f)
(test (let ((x (cons 1 2))) (memv x (list (cons 1 2) (cons 3 4)))) #f)
(test (let ((x (cons 1 2))) (memv x (list x (cons 3 4)))) '((1 . 2) (3 . 4)))
(test (memv 'a '(a a a)) '(a a a)) ;?
(test (memv 'a '(b a a)) '(a a))
(test (memv "hi" '(1 "hi" 2)) #f)
(test (memv #\a '(1 #f #\a 2)) '(#\a 2))
(test (memv cons (list car cdr cons +)) (list cons +))
(test (memv (apply values ()) (list #<unspecified>)) (list #<unspecified>))

(let ((odd '(3 a 3.0 b 3/4 c #(1) d))
      (even '(e 3 a 3.0 b 3/4 c #(1) d)))
  (test (memv 'a odd) '(a 3.0 b 3/4 c #(1) d))
  (test (memv 'a even) '(a 3.0 b 3/4 c #(1) d))
  (test (memv 3/4 odd) '(3/4 c #(1) d))
  (test (memv 3/4 even) '(3/4 c #(1) d))
  (test (memv 3.0 odd) '(3.0 b 3/4 c #(1) d))
  (test (memv 3.0 even) '(3.0 b 3/4 c #(1) d))
  (test (memv #(1) odd) #f)
  (test (memv #(1) even) #f))
(test (memv #(1) '(1 #(1) 2)) #f)
(test (memv () '(1 () 2)) '(() 2))
(test (let* ((x (vector 1 2 3)) (lst (list 1 "hi" x (vector 1 2)))) (memv x lst)) '(#(1 2 3) #(1 2)))
(test (let* ((x (vector 1 2 3)) (lst (list 1 "hi" (vector 1 2 3)))) (memv x lst)) #f)



;;; --------------------------------------------------------------------------------
;;; member

(test (member (list 'a) '(b (a) c)) '((a) c))
(test (member "b" '("a" "c" "b")) '("b"))
(test (member 1 '(3 2 1 4)) '(1 4))
(test (member 1 (list 3 2 1 4)) '(1 4))
(test (member car (list abs car modulo)) (list car modulo))
(test (member do (list quote map do)) (list do))
(test (member 5/2 (list 1/3 2/4 5/2)) '(5/2))
(test (member 'a '(a b c d)) '(a b c d))
(test (member 'b '(a b c d)) '(b c d))
(test (member 'c '(a b c d)) '(c d))
(test (member 'd '(a b c d)) '(d))
(test (member 'e '(a b c d)) #f)
(test (member 1 (cons 1 2)) '(1 . 2))
(test (member 1 '(1 2 . 3)) '(1 2 . 3))
(test (member 2 '(1 2 . 3)) '(2 . 3))
(test (member 3 '(1 2 . 3)) #f)
(test (member 4 '(1 2 . 3)) #f)
(test (member 1/2 (list (/ 2.0) .5 1/2)) '(1/2))
(test (member) 'error)
(test (member 'a) 'error)
(test (member 'a 'b) 'error)
(test (member () '(1 2 3)) #f)
(test (member () '(1 2 ())) '(()))
(test (member #() '(1 () 2 #() 3)) '(#() 3))
(test (member #2d((1 2) (3 4)) '(1 #() #2d((1 2) (1 2)))) #f)
(test (member #2d((1 2) (3 4)) '(1 #() #2d((1 2) (3 4)))) '(#2d((1 2) (3 4))))
(test (let ((x (cons 1 2))) (member x (list (cons 1 2) (cons 3 4)))) '((1 . 2) (3 . 4)))
(test (let ((x (list 1 2))) (member x (list (cons 1 2) (list 1 2)))) '((1 2)))
(test (member ''a '('a b c)) '('a b c))
(test (member 'a '(a a a)) '(a a a)) ;?
(test (member 'a '(b a a)) '(a a))
(test (member (member 3 '(1 2 3 4)) '((1 2) (2 3) (3 4) (4 5))) '((3 4) (4 5)))
(test (member "hi" '(1 "hi" 2)) '("hi" 2))
(test (member #\a '(1 #f #\a 2)) '(#\a 2))
(test (let* ((x (vector 1 2 3)) (lst (list 1 "hi" x (vector 1 2)))) (member x lst)) '(#(1 2 3) #(1 2)))
(test (let* ((x (vector 1 2 3)) (lst (list 1 "hi" (vector 1 2 3)))) (member x lst)) '(#(1 2 3)))
(test (member #<a> '(1 2 #<a> 3)) '(#<a> 3))
(test (member #<...> '(1 x #<...> 4 5) equivalent?) '(#<...> 4 5))
(test (member #<> '(1 2 #<>)) '(#<>))
(test (member #<a> '(1 #<b> 2)) #f)
(test (member #<undefined> '(1 #<undefined> 3)) '(#<undefined> 3))
(test (member #<eof> '(1 #<eof> 3)) '(#<eof> 3))
(test (member #<unspecified> '(1 #<unspecified> 3)) '(#<unspecified> 3))

(for-each
 (lambda (arg)
   (test (member arg (list 1 2 arg 3)) (list arg 3)))
 (list "hi" (integer->char 65) #f 'a-symbol abs 3/4 #\f #t (if #f #f) '(1 2 (3 (4))) most-positive-fixnum))

(test (member 3 . (1 '(2 3))) 'error)
(test (member 3 '(1 2 3) = =) 'error)
(test (member 3 . ('(1 2 3))) '(3))
(test (member 3 . ('(1 2 3 . 4))) '(3 . 4))
(test (member . (3 '(1 2 3))) '(3))
(test (member '(1 2) '(1 2)) #f)
(test (member '(1 2) '((1 2))) '((1 2)))
(test (member . '(quote . ((quote)))) #f)
(test (member . '(quote . ((quote) .()))) #f)
(test (member '(((1))) '((((1).()).()).())) '((((1)))))
(test (member '((1)) '(1 (1) ((1)) (((1))))) '(((1)) (((1)))))
(test (member member (list abs car memq member +)) (list member +))
(test (member () () "") 'error)

(let ((odd '(3 a 3.0 b 3/4 c #(1) d))
      (even '(e 3 a 3.0 b 3/4 c #(1) d)))
  (test (member 'a odd) '(a 3.0 b 3/4 c #(1) d))
  (test (member 'a even) '(a 3.0 b 3/4 c #(1) d))
  (test (member 3/4 odd) '(3/4 c #(1) d))
  (test (member 3/4 even) '(3/4 c #(1) d))
  (test (member 3.0 odd) '(3.0 b 3/4 c #(1) d))
  (test (member 3.0 even) '(3.0 b 3/4 c #(1) d))
  (test (member #(1) odd) '(#(1) d))
  (test (member #(1) even) '(#(1) d)))

(test (member 3 '(1 2 3 4) =) '(3 4))
(test (member 3 () =) #f)
(test (member 3 '(1 2 4 5) =) #f)
(test (member 4.0 '(1 2 4 5) =) '(4 5))
(test (member #\a '(#\b #\A #\c) char=?) #f)
(test (member #\a '(#\b #\A #\c) char-ci=?) '(#\A #\c))
(test (member #\a '(#\b #\A #\c) (lambda (a b) (char-ci=? a b))) '(#\A #\c))
(test (char=? (car (member #\a '(#\b #\a))) #\a) #t)
(test (char=? (car (member #\a '(#\b #\a) (lambda (a b) (char=? a b)))) #\a) #t)
(test (member 3 '(1 2 3 4) <) '(4))
(test (member 3 '((1 2) (3 4)) member) '((3 4)))
(test (member 3 '(((1 . 2) (4 . 5)) ((3 . 4))) assoc) '(((3 . 4))))
(test (member '(#f #f #t) '(0 1 2) list-ref) '(2))
(test (let ((v (vector 1 2 3))) (member v (list 0 v) vector-fill!)) '(0 #(0 0 0)))

(test (member 3 '(1 2 3) abs) 'error)
(test (member 3 '(1 2 3) quasiquote) 'error)
(test (member 3 '(1 2 3) (lambda (a b c) (= a b))) 'error)
(test (member 3 '(1 2 3) (lambda* (a b c) (= a b))) '(3))
(test (member 3 '(1 2 3 4) (dilambda = =)) '(3 4))
(test (catch #t (lambda () (member 3 '(1 2 3) (lambda (a b) (error 'member a)))) (lambda args (car args))) 'member)
(test (call-with-exit (lambda (go) (member 3 '(1 2 3) (lambda (a b) (go 'member))))) 'member)
(test (member 'a '(a a a) eq?) '(a a a))
(test (member 'a '(b a a) eqv?) '(a a))
(test (member 3.0 '(1 #\a (3 . 3) abs #() 3+i)) #f)
(test (member 3.0 '(1 #\a (3 . 3) abs #() 3+i) (lambda (a b) (= (real-part a) b))) 'error)
(test (member 3.0 '(1 #\a (3 . 3) abs #() 3+i) (lambda (a b) (and (number? b) (= (real-part b) a)))) '(3+i))
;; is it guaranteed that in the comparison function the value is first and the list member 2nd?
(test (member 4 '((1 2 3) (4 5 6) (7 8 9)) member) '((4 5 6) (7 8 9)))
(test (member 4 '(1 2 3) member) 'error)
(test (member 4 '((1 2) (3 5) 7) (lambda (a b) (member a (map (lambda (c) (+ c 1)) b)))) '((3 5) 7))
(test (member 4 '((1 2) (3 5) 7) (lambda (a b) (assoc a (map (lambda (c) (cons (+ c 1) c)) b)))) '((3 5) 7))
(test (let ((f #f)) (member 'a '(a b c) (lambda (a b) (if (eq? b 'a) (set! f (lambda () b))) (eq? a 123))) (f)) 'a)
(test (let ((i 0) (f (make-vector 3))) (member 'a '(a b c) (lambda (a b) (vector-set! f i b) (set! i (+ i 1)) (eq? a 123))) f) #(a b c))
(test (member 1 '(0 1 2) (lambda (a b . c) (= a b))) '(1 2))
(test (member 1 '(0 1 2) (lambda* (a b c) (= a b))) '(1 2))
(test (member 1 '(0 1 2) (lambda (a) (= a b))) 'error)
(test (member 1 '(0 1 2) (lambda a (= (car a) (cadr a)))) '(1 2))

(test (member 'a '(c 3 a 1 b 2) (lambda (a) (eq? a b))) 'error)
(test (member 'a '(c 3 a 1 b 2) (lambda (a b) (eq? a b))) '(a 1 b 2))
(test (member 'a '(c 3 a 1 b 2) (lambda (a b c) (eq? a b))) 'error)
(test (member 'a '(c 3 a 1 b 2) (lambda (a b c . d) (eq? a b))) 'error)
(test (member 'a '(c 3 a 1 b 2) (lambda (a b . c) (eq? a b))) '(a 1 b 2))
(test (member 'a '(c 3 a 1 b 2) (lambda a (apply eq? a))) '(a 1 b 2))
(test (member 'a '(c 3 a 1 b 2) (lambda (a . b) (eq? a (car b)))) '(a 1 b 2))

(test (member 'a '(c 3 a 1 b 2) (lambda* (a) (eq? a b))) 'error)
(test (member 'a '(c 3 a 1 b 2) (lambda* (a b) (eq? a b))) '(a 1 b 2))
(test (member 'a '(c 3 a 1 b 2) (lambda* (a b c) (eq? a b))) '(a 1 b 2))
(test (member 'a '(c 3 a 1 b 2) (lambda* (a b c . d) (eq? a b))) '(a 1 b 2))
(test (member 'a '(c 3 a 1 b 2) (lambda* (a b . c) (eq? a b))) '(a 1 b 2))
(test (member 'a '(c 3 a 1 b 2) (lambda* a (apply eq? a))) '(a 1 b 2))
(test (member 'a '(c 3 a 1 b 2) (lambda* (a . b) (eq? a (car b)))) '(a 1 b 2))
(test (member 'a '(c 3 a 1 b 2) (lambda* (a :rest b) (eq? a (car b)))) '(a 1 b 2))
(test (member 'a '(c 3 a 1 b 2) (lambda* (a :rest b :rest c) (eq? a (car b)))) '(a 1 b 2))
(test (member 'a '(c 3 a 1 b 2) (lambda* (:rest a) (apply eq? a))) '(a 1 b 2))
(test (member 'a '(a b c) (define-macro (_m_ a b) `(eq? ',a ',b))) '(a b c))
(test (member 'c '(a b c) (define-macro (_m_ a b) `(eq? ',a ',b))) '(c))

(test (member 4 '(1 2 3 4 . 5)) '(4 . 5))
(test (member 4 '(1 2 3 4 . 5) =) '(4 . 5))
(test (member 4 '(1 2 3 . 4)) #f)
(test (member 4 '(1 2 3 . 4) =) #f)
(test (let ((lst (list 1 2 3))) (and (member 2 lst =) lst)) '(1 2 3))
(test (pair? (let ((lst (list 1 2 3))) (set! (cdr (cdr lst)) lst) (member 2 lst))) #t)
(test (pair? (let ((lst (list 1 2 3))) (set! (cdr (cdr lst)) lst) (member 2 lst =))) #t)
(test (let ((lst (list 1 2 3))) (set! (cdr (cdr lst)) lst) (member 4 lst)) #f)
(test (let ((lst (list 1 2 3))) (set! (cdr (cdr lst)) lst) (member 4 lst =)) #f)
(test (pair? (let ((lst (list 1 2 3))) (set! (cdr (cdr (cdr lst))) lst) (member 3 lst =))) #t)
(test (pair? (let ((lst (list 1 2 3 4))) (set! (cdr (cdr (cdr lst))) (cdr (cdr lst))) (member 3 lst =))) #t)
(test (let ((lst (list 1 2 3 4))) (set! (cdr (cdr (cdr lst))) (cdr (cdr lst))) (member 5 lst =)) #f)
(test (let ((lst (list 1 2 3 4))) (set! (cdr (cdr (cdr lst))) (cdr lst)) (member 4 lst =)) #f)
(test (let ((lst '(1 2 3 5 6 9 10))) (member 3 lst (let ((last (car lst))) (lambda (a b) (let ((result (= (- b last) a))) (set! last b) result))))) '(9 10))
(test (let ((lst '(1 2 3 5 6 9 10))) (member 2 lst (let ((last (car lst))) (lambda (a b) (let ((result (= (- b last) a))) (set! last b) result))))) '(5 6 9 10))
(test (member 1 () =) #f)
(test (member 1 #(1) =) 'error)
(test (member 3 '(5 4 3 2 1) >) '(2 1))
(test (member 3 '(5 4 3 2 1) >=) '(3 2 1))
(test (member '(1 2) '((1) (1 . 2) (1 2 . 3) (1 2 3) (1 2) 1 . 2)) '((1 2) 1 . 2))
(test (member '(1 2 . 3) '((1) (1 . 2) (1 2 . 3) (1 2 3) (1 2) 1 . 2)) '((1 2 . 3) (1 2 3) (1 2) 1 . 2))

(when with-bignums
  (test (memv 1 (list (bignum 1))) '(1))
  (test (memv (bignum 1) (list 1)) '(1))
  (test (member 1 (list (bignum 1)) eqv?) '(1))
  (test (member 1 (list (bignum 1)) =) '(1))
  (test (member 1 (list (bignum 1))) '(1))
  (test (member 1 (list (bignum 1)) equivalent?) '(1))
  (test (member 1 (list (bignum 1)) equal?) '(1))
  (test (member 1.0 (list (bignum 1)) equivalent?) '(1))
  (test (member (+ 1.0 1e-15) (list (bignum 1)) equivalent?) #f)
  (test (assoc 1 (list (list (bignum 1) 'ok))) '(1 ok))
  (test (let-temporarily (((*s7* 'equivalent-float-epsilon) 1e-6)) (member (+ 1 1e-10) (list (bignum 1)) equivalent?)) '(1))
  (test (let-temporarily (((*s7* 'equivalent-float-epsilon) 1e-6)) (member (+ 1 1e-10) (list (bignum 1)) =)) #f))

(when with-block
  (test (member 1 (list 2 3) (lambda (a b) (float-vector? (block)))) '(2 3)))

(let ()
  (define (sfind obj lst)
    (member obj lst (lambda (a b)
		      (catch #t
			(lambda ()
			  (and (equal? a b)
			       (member obj lst (lambda (a b)
						 (catch #t
						   (lambda ()
						     (error 'oops))
						   (lambda args
						     (equal? a b)))))))
			(lambda args
			  'oops)))))
  (test (sfind 'a '(b c a d)) '(a d)))

(let* ((records (list ; from Woody Douglass
		 (inlet :person "oscar meyer")
		 (inlet :person "howard johnson")
		 (inlet :person "betty crocker")))
       (match-record (lambda (name record)
		       (display name #f) ;(format (current-error-port) "COMPARE '~A' '~A'~%" name (record 'person))
		       (let ((vname (record 'person)))
			 (equal? name (substring vname 0 (min (length name) (length vname)))))))
       (find-record (lambda (name)
		      (display name #f) ;(format (current-error-port) "FINDING ~A~%" name)
		      (let* ((found (member name records match-record)))
			(if found
			    (car found)
			    #f)))))
  (test (find-record "betty") (inlet :person "betty crocker")))

(let ()
  (define-macro (do-list lst . body)
    `(member #t ,(cadr lst) (lambda (a b)
			      (let ((,(car lst) b))
				,@body
				#f))))
  (let ((sum 0))
    (do-list (x '(1 2 3)) (set! sum (+ sum x)))
    (test (= sum 6) #t)))

(let ()
  (define (tree-member a lst)
    (member a lst (lambda (c d)
		    (if (pair? d)
			(tree-member c d)
			(equal? c d)))))
  (test (tree-member 1 '(2 3 (4 1) 5)) '((4 1) 5))
  (test (tree-member -1 '(2 3 (4 1) 5)) #f)
  (test (tree-member 1 '(2 3 ((4 (1) 5)))) '(((4 (1) 5)))))

(let ((lst (list 1 2 3)))
  (set! (cdr (cdr (cdr lst))) lst)
  (test (member 2 lst) (member 2 lst equal?)))

(let ((lst (list 1 2 3)))
  (set! (cdr (cdr (cdr lst))) lst)
  (test (member 4 lst) (member 4 lst equal?)))

(let ((lst (list 1 2 3 4)))
  (set! (cdr (cdr (cdr (cdr lst)))) lst)
  (test (member 4 lst) (member 4 lst equal?)))

(let ((lst (list 1 2 3 4)))
  (set! (cdr (cdr (cdr (cdr lst)))) (cdr lst))
  (test (member 4 lst) (member 4 lst equal?)))

(for-each
  (lambda (arg lst)
    (test (member arg lst eq?) (memq arg lst))
    (test (member arg lst eqv?) (memv arg lst))
    (test (member arg lst equal?) (member arg lst)))
  (list 'a #f (list 'a) 'a 1 3/4 #(1) "hi")
  (list '(a b c) '(1 "hi" #t #f 2) '(b (a) c) '(d a b . c) '(1 3/4 23) '(1 3/4 23) '(a 1 #(1) 23) '(1 "hi" 23)))

(for-each
 (lambda (op)
   (test (op) 'error)
   (for-each
    (lambda (arg)
      (let ((result (catch #t (lambda () (op arg)) (lambda args 'error))))
	(if (not (eq? result 'error))
	    (format #t ";(~A ~A) returned ~A?~%" op arg result))
	(test (op arg () arg) 'error)
	(test (op arg) 'error)))
    (list () "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
	  3.14 3/4 1.0+1.0i #\f #t :hi (if #f #f) (lambda (a) (+ a 1)))))
 (list cons car cdr set-car! set-cdr! caar cadr cdar cddr caaar caadr cadar cdaar caddr cdddr cdadr cddar
       caaaar caaadr caadar cadaar caaddr cadddr cadadr caddar cdaaar cdaadr cdadar cddaar cdaddr cddddr cddadr cdddar
       assq assv memq memv list-ref list-tail))

(for-each
 (lambda (op)
   (test (op '(1) '(2)) 'error))
 (list reverse car cdr caar cadr cdar cddr caaar caadr cadar cdaar caddr cdddr cdadr cddar
       caaaar caaadr caadar cadaar caaddr cadddr cadadr caddar cdaaar cdaadr cdadar cddaar cdaddr cddddr cddadr cdddar
       list-ref list-tail list-set!))

(for-each
 (lambda (op)
   (for-each
    (lambda (arg)
      (let ((result (catch #t (lambda () (op #f arg)) (lambda args 'error))))
	(if (not (eq? result 'error))
	    (format #t ";(~A #f ~A) returned ~A?~%" op arg result))))
    (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
	  3.14 3/4 1.0+1.0i #\f #t :hi (if #f #f) (lambda (a) (+ a 1)))))
 (list assq assv assoc memq memv member))




;;; --------------------------------------------------------------------------------
;;; append

(test (append '(a b c) ()) '(a b c))
(test (append () '(a b c)) '(a b c))
(test (append '(a b) '(c d)) '(a b c d))
(test (append '(a b) 'c) '(a b . c))
(test (equal? (append (list 'a 'b 'c) (list 'd 'e 'f) () '(g)) '(a b c d e f g)) #t)
(test (append (list 'a 'b 'c) (list 'd 'e 'f) () (list 'g)) '(a b c d e f g))
(test (append (list 'a 'b 'c) 'd) '(a b c . d))
(test (append () ()) ())
(test (append () (list 'a 'b 'c)) '(a b c))
(test (append) ())
(test (append () 1) 1)
(test (append 'a) 'a)
(test (append '(x) '(y))  '(x y))
(test (append '(a) '(b c d)) '(a b c d))
(test (append '(a (b)) '((c)))  '(a (b) (c)))
(test (append '(a b) '(c . d))  '(a b c . d))
(test (append () 'a)  'a)
(test (append '(a b) (append (append '(c)) '(e) 'f)) '(a b c e . f))
(test (append ''foo 'foo) '(quote foo . foo))
(test (append () (cons 1 2)) '(1 . 2))
(test (append () () ()) ())
(test (append (cons 1 2)) '(1 . 2))
(test (append (list 1) 2) '(1 . 2))

(test (append #f) #f)
(test (append () #f) #f)
(test (append '(1 2) #f) '(1 2 . #f))
(test (append () () #f) #f)
(test (append () '(1 2) #f) '(1 2 . #f))
(test (append '(1 2) () #f) '(1 2 . #f))
(test (append '(1 2) '(3 4) #f) '(1 2 3 4 . #f))
(test (append () () () #f) #f)
(test (append '(1 2) '(3 4) '(5 6) #f) '(1 2 3 4 5 6 . #f))
(test (append () () #()) #())
(test (append () ((lambda () #f))) #f)

(test (append (begin) do) do)
(test (append if) if)
(test (append quote) quote)
(test (append 0) 0) ; is this correct?
(test (append () 0) 0)
(test (append () () 0) 0)
(test (let* ((x '(1 2 3)) (y (append x ()))) (eq? x y)) #f) ; check that append returns a new list
(test (let* ((x '(1 2 3)) (y (append x ()))) (equal? x y)) #t)
(test (let* ((x (list 1 2 3)) (y (append x (list)))) (eq? x y)) #f)
(test (append '(1) 2) '(1 . 2))
(let ((x (list 1 2 3)))
  (let ((y (append x ())))
    (set-car! x 0)
    (test (= (car y) 1) #t)))
(let ((x (list 1 2 3)))
  (let ((y (append x ())))
    (set-cdr! x 0)
    (test (and (= (car y) 1)
	       (= (cadr y) 2)
	       (= (caddr y) 3))
	  #t)))

(test (let ((xx (list 1 2))) (recompose 12 (lambda (x) (append (list (car x)) (cdr x))) xx)) '(1 2))

(test (append 'a 'b) 'error)
(test (append 'a ()) 'error)
(test (append (cons 1 2) ()) 'error)
(test (append '(1) 2 '(3)) 'error)
(test (append '(1) 2 3) 'error)
(test (let ((lst (list 1 2 3))) (append lst lst)) '(1 2 3 1 2 3))
(test (append ''1 ''1) '(quote 1 quote 1))
(test (append '(1 2 . 3) '(4)) 'error)
(test (append '(1 2 . 3)) '(1 2 . 3))
(test (append '(4) '(1 2 . 3)) '(4 1 2 . 3))
(test (append () 12 . ()) 12)
(test (append '(1) 12) '(1 . 12))
(test (append '(1) 12 . ()) '(1 . 12))
(test (append () () '(1) 12) '(1 . 12))
(test (append '(1) '(2) '(3) 12) '(1 2 3 . 12))
(test (append '(((1))) '(((2)))) '(((1)) ((2))))
(test (append () . (2)) 2)
(test (append . (2)) 2)
(test (append ''() ()) ''())
(test (let ((i 1)) (logior 123 (append i))) 123) ; !
(test (let ((x "hi")) (eq? (append x "") x)) #f)
(test (let ((x #(1 2 3))) (eq? (append x #()) x)) #f)
(test (let ((x '(1 2 3))) (eq? (append x ()) x)) #f)
(test (let ((x '(1 2 3))) (eq? (append x) x)) #t)
(test (append (rootlet) #f) 'error) ; "append first argument, (rootlet), is a let but should be a sequence other than the rootlet"
(test (append () (make-hash-table) (byte-vector)) #u())
(test (append (append () (make-hash-table)) (byte-vector)) (hash-table))

(for-each
 (lambda (arg)
   (test (append arg) arg)
   (test (append () arg) arg)
   (test (append () () () arg) arg))
 (list "hi" #\a #f 'a-symbol _ht_ _undef_ _null_ (make-vector 3) abs 1 3.14 3/4 1.0+1.0i #t #<unspecified> #<eof> () #() (list 1 2) (cons 1 2) #(0) (lambda (a) (+ a 1))))
(test (append not) not)

(test (let ((l0 (list 0))
	    (l1 (list 0)))
	(let ((m0 (append '(2) l0))
	      (m1 (append '(2) l1 '())))
	  (and (equal? l0 l1)
	       (equal? m0 m1)
	       (let ()
		 (list-set! m0 1 3)
		 (list-set! m1 1 3)
		 (list l0 l1)))))
      '((3) (0)))

;;; generic append
(test (append "asdasd" '("asd")) 'error)
(test (append "asdasd" #("asd")) 'error)
(test (append (string->byte-vector "asdasd") '("asd")) 'error)
(test (append (string->byte-vector "asdasd") #("asd")) 'error)

(test (catch #t (lambda () (append  #r(1.0 2.0 3.0) #((a . 4)))) (lambda (type info) (apply format #f info))) "append argument, (a . 4), is a pair but should be a real number")
(test (catch #t (lambda () (append #i(1 2 3) #r(1.0 2.0 3.0))) (lambda (type info) (apply format #f info))) "can't append #r(1.0 2.0 3.0) to an int-vector")
(test (append (inlet) (hash-table :readable 123)) 'error)

(test (let ((h1 (hash-table 'a 1 'b 2)) (h2 (hash-table 'c 3))) (append h1 h2)) (hash-table 'c 3 'a 1 'b 2))
(test (let ((i1 (inlet 'a 1)) (i2 (inlet 'b 2 'c 3))) (append i1 i2)) (inlet 'a 1 'c 3 'b 2))
(test (let ((s1 "abc") (s2 "def")) (append s1 s2)) "abcdef")
(test (let ((v1 #(0 1)) (v2 #(2 3))) (append v1 v2)) #(0 1 2 3))
(test (let ((p1 '(1 2)) (p2 '(3 4))) (append p1 p2)) '(1 2 3 4))
(test (vector? (append #())) #t)
(test (float-vector? (append (float-vector))) #t)
(test (int-vector? (append (int-vector))) #t)
(test (append "12" '(1 . 2) "3") 'error)
(for-each
 (lambda (arg)
   (test (append arg) arg)
   (test (append () arg) arg))
 (list "" #u() () #() (int-vector) (float-vector) (inlet) (hash-table)
       "123" #u(101 102)
       '(1 2 3) '((e . 5) (f . 6))
       #(1 2 3) #((g . 8) (h . 9)) (int-vector 1 2 3) (float-vector 1 2 3)
       (inlet 'a 1 'b 2)
       (hash-table 'c 3 'd 4)))
(test (append #u() (int-vector 1 2 3)) #u(1 2 3))
(test (append #u() "123") #u(49 50 51))
(test (append "" "123") "123")
(test (append #() (hash-table)) #())
(test (append #() #u(101 102)) #(101 102))
(test (append (float-vector) #u(101 102)) (float-vector 101.0 102.0))
(test (append (int-vector) #u(101 102)) (int-vector 101 102))
(test (append (hash-table) '((e . 5) (f . 6))) (hash-table 'e 5 'f 6))
(test (append (inlet) #((g . 8) (h . 9))) (inlet 'g 8 'h 9))
(test (append '(1 2 3) #u()) '(1 2 3))
(test (append '(1 2 3) #u() #(4 5)) '(1 2 3 4 5))
(test (append '(1 2 3) #u(101 102)) '(1 2 3 101 102))
(test (append '(1 2 3) #() (inlet 'a 1 'b 2)) '(1 2 3 (a . 1) (b . 2)))
(test (let ((lst (append '((e . 5) (f . 6)) "" (hash-table 'c 3 'd 4))))
	(or (equal? lst '((e . 5) (f . 6) (c . 3) (d . 4)))
	    (equal? lst '((e . 5) (f . 6) (d . 4) (c . 3))))) #t)
(test (append (list 1) "hi") '(1 #\h #\i))
(test (append #(1 2 3) "123") #(1 2 3 #\1 #\2 #\3))
(test (append (int-vector 1 2 3) #(1 2 3)) (int-vector 1 2 3 1 2 3))
(test (append (int-vector 1 2 3) "123") (int-vector 1 2 3 49 50 51))
(test (append (float-vector 1.0 2.0 3.0) (int-vector 1 2 3)) (float-vector 1.0 2.0 3.0 1.0 2.0 3.0))
(test (append (int-vector 1 2 3) (float-vector 1.0 2.0 3.0)) 'error) ;(int-vector 1 2 3 1 2 3))
(test (append (inlet 'a 1 'b 2) '((e . 5) (f . 6))) (inlet 'b 2 'a 1 'e 5 'f 6))
(test (append (inlet 'a 1 'b 2) (hash-table 'c 3 'd 4)) (inlet 'b 2 'a 1 'c 3 'd 4))
(test (append "" #() #u(101 102)) "ef")
(test (append "" #u(101 102) (hash-table)) "ef")
(test (append #u() #() #u(101 102)) #u(101 102))
(test (append #u() (inlet) "") #u())
(test (append #u() #u(101 102) "123") #u(101 102 49 50 51))
(test (append () "" (int-vector 1 2 3)) (int-vector 1 2 3))
(test (let ((v (append #() #u() (hash-table 'c 3 'd 4))))
	(or (equal? v #((c . 3) (d . 4)))
	    (equal? v #((d . 4) (c . 3))))) #t)
(test (append #() #(1 2 3) (inlet)) #(1 2 3))
(test (append #() (float-vector 1.0 2.0 3.0) ()) #(1.0 2.0 3.0))
(test (append (float-vector) "" "123") (float-vector 49.0 50.0 51.0))
(test (append (float-vector) (int-vector 1 2 3) #u(101 102)) (float-vector 1.0 2.0 3.0 101.0 102.0))
(test (append (inlet) #() #((g . 8) (h . 9))) (inlet 'g 8 'h 9))
(test (append (inlet) '((e . 5) (f . 6)) (hash-table 'c 3 'd 4)) (inlet 'e 5 'f 6 'c 3 'd 4))
(test (append (hash-table) "" (inlet 'a 1 'b 2)) (hash-table 'b 2 'a 1))
(test (append (hash-table) '((e . 5) (f . 6)) (inlet 'a 1 'b 2)) (hash-table 'b 2 'e 5 'f 6 'a 1))
(test (append (hash-table) #((g . 8) (h . 9)) '((e . 5) (f . 6))) (hash-table 'e 5 'g 8 'f 6 'h 9))
(test (append "123" #u(101 102) (hash-table)) "123ef")
(test (append #u(101 102) #u() #u(101 102)) #u(101 102 101 102))
(test (append #u(101 102) "123" (int-vector 1 2 3)) #u(101 102 49 50 51 1 2 3))
(test (append #u(101 102) '(1 2 3) "") #u(101 102 1 2 3))
(test (append '(1 2 3) #u(101 102) #(1 2 3)) '(1 2 3 101 102 1 2 3))
(test (let ((lst (append '(1 2 3) (hash-table 'c 3 'd 4) "")))
	(or (equal? lst '(1 2 3 (c . 3) (d . 4)))
	    (equal? lst '(1 2 3 (d . 4) (c . 3))))) #t)
(test (append (int-vector 1 2 3) #u(101 102) (float-vector 1.0 2.0 3.0)) 'error) ; (int-vector 1 2 3 101 102 1 2 3))
(test (append (int-vector 1 2 3) '(1 2 3) #u(101 102)) (int-vector 1 2 3 1 2 3 101 102))
(test (append (hash-table 'c 3 'd 4) (hash-table 'c 3 'd 4) '((e . 5) (f . 6))) (hash-table 'e 5 'f 6 'c 3 'd 4))
(when with-block
  (test (append (block 1 2) (block 3 4)) (block 1 2 3 4))
  (test (let () (define (func) (append (list (list (list 1)) (setter car)) (vector-dimensions (block 0.0)))) (define (hi) (func)) (hi)) (list '((1)) set-car! 1))
  (test (append #i2d((101 201) (3 4)) (make-block 2)) 'error) ; #i(101 201 3 4 0 0))
  (test (let () (define (f1) (do ((i 0 (+ i 1))) ((= i 1)) (values (append "" (block)) 1))) (f1)) #t))

(let ((vvv (let ((v (make-vector '(2 2)))) (set! (v 0 0) "asd") (set! (v 0 1) #r(4 5 6)) (set! (v 1 0) '(1 2 3)) (set! (v 1 1) 32) v)))
(test (let ((x #f)) (do ((i 0 (+ i 1))) ((= i 1) x) (set! x (append vvv (byte-vector 255))))) #("asd" #r(4.0 5.0 6.0) (1 2 3) 32 255))
(test (let ((x #f)) (define (func) (do ((i 0 (+ i 1))) ((= i 1) x) (set! x (append vvv (byte-vector 255))))) (func)) #("asd" #r(4.0 5.0 6.0) (1 2 3) 32 255))
(test (let ((x #f)) (define (func) (do ((i 0 (+ i 1))) ((= i 1) x) (set! x (append vvv (byte-vector 255))))) (func)) #("asd" #r(4.0 5.0 6.0) (1 2 3) 32 255))
(test (let ((x #f)) (define (func) (do ((i 0 (+ i 1))) ((= i 1) x) (set! x (append vvv (int-vector 255))))) (func)) #("asd" #r(4.0 5.0 6.0) (1 2 3) 32 255))
(test (let ((x #f)) (define (func) (do ((i 0 (+ i 1))) ((= i 1) x) (set! x (append vvv (vector 255))))) (func)) #("asd" #r(4.0 5.0 6.0) (1 2 3) 32 255))
(test (let ((x #f)) (define (func) (do ((i 0 (+ i 1))) ((= i 1) x) (set! x (append vvv (string #\a))))) (func)) #("asd" #r(4.0 5.0 6.0) (1 2 3) 32 #\a))
(test (let ((x #f)) (define (func) (do ((i 0 (+ i 1))) ((= i 1) x) (set! x (append vvv (vector 1 2 3))))) (func)) #("asd" #r(4.0 5.0 6.0) (1 2 3) 32 1 2 3))
(test (let ((x #f)) (define (func) (do ((i 0 (+ i 1))) ((= i 1) x) (set! x (append vvv vvv)))) (func)) #("asd" #r(4.0 5.0 6.0) (1 2 3) 32 "asd" #r(4.0 5.0 6.0) (1 2 3) 32))
(test (let ((x #f)) (do ((i 0 (+ i 1))) ((= i 1) x) (set! x (append (vector 0) vvv (byte-vector 255))))) #(0 "asd" #r(4.0 5.0 6.0) (1 2 3) 32 255))
(test (let ((x #f)) (define (func) (do ((i 0 (+ i 1))) ((= i 1) x) (set! x (append (vector 0) vvv (byte-vector 255))))) (func)) #(0 "asd" #r(4.0 5.0 6.0) (1 2 3) 32 255)))
(when with-block
  (test (char? (append (make-float-vector '(2 3) 1) (immutable! (block 0.0 1.0 2.0)))) #f))

(test (let ((v1 #2d(((b . 2) (:rest . 123))))) (append (inlet :a 32) v1 ())) (inlet 'a 32 'b 2 'rest 123)) ; let_setter keyword check

(test (random-state? (cdr (append '(1) (random-state 123)))) #t)
(test (append '(1) (random-state 123) ()) 'error)
(test (random-state? (append () (random-state 123))) #t)

(when full-s7test
  (let ((seqs (list "" #u() () #() (int-vector) (float-vector) (inlet) (hash-table)
		    "123" #u(101 102)
		    '(1 2 3) '((e . 5) (f . 6))
		    #(1 2 3) #((g . 8) (h . 9)) (int-vector 1 2 3) (float-vector 1 2 3)
		    (inlet 'a 1 'b 2)
		    (hash-table 'c 3 'd 4)
		    1 #f '(1 . 2) (let ((lst (list 1))) (set-cdr! lst lst)))))
    (define (test-append)
      (for-each
       (lambda (s1)
	 (catch #t
	   (lambda ()
	     (append s1)
	     (for-each
	      (lambda (s2)
		(catch #t
		  (lambda ()
		    (append s1 s2)
		    (for-each
		     (lambda (s3)
		       (catch #t
			 (lambda ()
			   (append s1 s2 s3)
			   (for-each
			    (lambda (s4)
			      (catch #t
				(lambda ()
				  (append s1 s2 s3 s4)
				  (for-each
				   (lambda (s5)
				     (catch #t (lambda ()
						 (append s1 s2 s3 s4 s5))
					    (lambda args 'error)))
				   seqs))
				(lambda args 'error)))
			    seqs))
			 (lambda args 'error)))
		     seqs))
		  (lambda args 'error)))
	      seqs))
	   (lambda args 'error)))
       seqs))
    (test-append)))

(test (append (vector 1 2 3) (list 4 5 6)) #(1 2 3 4 5 6))
(test (append (list 1 2 3) (vector 4 5 6)) '(1 2 3 4 5 6))
(test (append (list) (vector 4 5 6)) #(4 5 6))
(test (append (vector 1 2 3) (list)) #(1 2 3))
(test (append (list 1 2 3) (vector)) '(1 2 3))
(test (append (vector #\f) "abc") #(#\f #\a #\b #\c))
(test (append "abc" (vector #\f)) "abcf")
(test (append (list #\a) "bc") '(#\a #\b #\c))
(test (append "ab" (list #\c)) "abc")
(test (append (hash-table) #(1 2 3)) 'error)
(test (append (hash-table 'a 1) #((b . 2))) (hash-table 'a 1 'b 2))
(test (append "" #()) "")
(test (append #() "") #())
(test (append #i(1 2 3) "asdf") #i(1 2 3 97 115 100 102))
(test (append "asdf" #i(90 91)) "asdfZ[")
(test (append "asdf" ())  "asdf")
(test (append "asdf" #()) "asdf")
(test (append () "asdf") "asdf")
(test (append #r(1.0 2.0 3.0) "asdf") #r(1.0 2.0 3.0 97.0 115.0 100.0 102.0))
(test (append #r(1.0 2.0 3.0) "") #r(1.0 2.0 3.0))
(test (append #r(1.0 2.0 3.0) (hash-table)) #r(1.0 2.0 3.0))
(test (append (hash-table) "asdf") 'error)
(test (append () "asdf" #()) '(#\a #\s #\d #\f))
(test (append "asdf" "asdf" #()) "asdfasdf")
(test (append (block) #r(1 2 3)) 'error)
(test (append #r(1 2 3) (block)) #r(1.0 2.0 3.0))
(test (append (block 1 2 3) #r()) 'error)
(test (append (hash-table 'a 1) #((b . 2))) (hash-table 'a 1 'b 2))
(test (append #((a . 1)) (hash-table 'b 2)) #((a . 1) (b . 2)))
(test (append #((a . 1)) (hash-table 'b 2) (inlet 'c 3)) #((a . 1) (b . 2) (c . 3)))
(test (append (hash-table 'a 1) #((b . 2)) (inlet 'c 3)) (hash-table 'c 3 'a 1 'b 2))
(test (append (inlet 'c 3) (hash-table 'a 1) #((b . 2))) (inlet 'b 2 'a 1 'c 3))
(test (append () (hash-table 'a 1)) (hash-table 'a 1))
(test (append '((b . 2)) (hash-table 'a 1)) '((b . 2) (a . 1)))
(test (append (hash-table 'a 1) '((b . 2))) (hash-table 'a 1 'b 2))
(test (append '((a . 1)) (inlet 'b 2)) '((a . 1) (b . 2)))
(test (append '(1 2) #i(3 4)) '(1 2 3 4))
(test (append '(1 2) (inlet 'a 1)) '(1 2 (a . 1)))
(test (append '(1 2) (hash-table 'a 1)) '(1 2 (a . 1)))
(test (append () (hash-table 'a 1)) (hash-table 'a 1))
(test (append (list 1 2) #(3) 4) '(1 2 3 . 4))
(let ((L (list 1))) (set-cdr! L L) (test (append (list 2) L) (cons 2 L)))
(test (append "" #() (list #\a) () (inlet) "b" (hash-table) #(#\c)) "abc")
(test (append (hash-table 'a 1) (hash-table 'a 2)) (hash-table 'a 2))
(test (append (hash-table 'a 1) (inlet 'a 2)) (hash-table 'a 2))
(test (append "asdf" (list #\a)) "asdfa")


;;; tree-cyclic --------
(test (tree-cyclic? '(1 2)) #f)
(test (tree-cyclic? 1) #f)
(test (tree-cyclic? (let ((lst (list 1))) (set-cdr! lst lst))) #t)
(test (tree-cyclic? (let ((lst (list 1))) (set-car! lst lst))) #t)
(test (tree-cyclic? (let ((lst (list 1 2))) (set-cdr! (cdr lst) lst))) #t)
(test (tree-cyclic? (let ((lst (list (list 1 3) 2))) (set-cdr! (car lst) lst))) #t)
(test (tree-cyclic? (let ((lst (list (list 1 3) 2))) (set-car! (car lst) lst))) #t)
(test (tree-cyclic? (let ((lst (list 1 (list 1 3) 2))) (set-car! (cdr lst) lst))) #t)
(test (tree-cyclic? (let ((lst (list 1 (list 1 3) 2))) (set-cdr! (cdr lst) lst))) #t)
(test (tree-cyclic? (let ((lst (list 1 (list 1 3) 2))) (set-cdr! (cddr lst) lst))) #t)
(test (tree-cyclic? '(1 2 (3 4))) #f)
(test (tree-cyclic? '(1 . 2)) #f)
(test (tree-cyclic? '(1 2 . 3)) #f)
(test (tree-cyclic? (let ((lst (list 1 (list 1 3) 2))) (set-cdr! (cddr lst) lst) (list 1 lst))) #t)
(test (tree-cyclic? (let ((lst (list 1 (list 1 3) 2))) (set-cdr! (cddr lst) lst) (list 1 (cdr lst)))) #t)
(test (tree-cyclic? (let ((lst (list (cons 1 3) 2))) (set-cdr! (cdr lst) lst))) #t)
(test (tree-cyclic? (let ((lst (list (list 1 4) (cons 2 3)))) (set-cdr! (car lst) lst))) #t)
(test (tree-cyclic? (let ((lst (list (list 1 4) (cons 2 3)))) (set-cdr! (cadr lst) lst))) #t)
(test (tree-cyclic? (let ((lst (list (list 1 2) (list 3) (list 4 5 6)))) (set-car! (car lst) lst))) #t)
(test (tree-cyclic? (let ((lst (list (list 1 2) (list 3) (list 4 5 6)))) (set-car! (cdar lst) lst))) #t)
(test (tree-cyclic? (let ((lst (list (list 1 2) (list 3) (list 4 5 6)))) (set-cdr! (car lst) lst))) #t)
(test (tree-cyclic? (let ((lst (list (list 1 2) (list 3) (list 4 5 6)))) (set-cdr! (cdar lst) lst))) #t)
(test (tree-cyclic? (let ((lst (list (list 1 2) (list 3) (list 4 5 6)))) (set-car! (cadr lst) lst))) #t)
(test (tree-cyclic? (let ((lst (list (list 1 2) (list 3) (list 4 5 6)))) (set-cdr! (cadr lst) lst))) #t)
(test (tree-cyclic? (let ((lst (list (list 1 2) (list 3) (list 4 5 6)))) (set-car! (cadr lst) (cdr lst)))) #t)
(test (tree-cyclic? (let ((lst (list (list 1 2) (list 3) (list 4 5 6)))) (set-cdr! (cadr lst) (cdr lst)))) #t)
(test (tree-cyclic? (let ((lst (list (list 1 2) (list 3) (list 4 5 6)))) (set-car! (caddr lst) lst))) #t)
(test (tree-cyclic? (let ((lst (list (list (list (list 4 5 6)))))) (set-car! (caaar lst) lst))) #t)
(test (tree-cyclic? (let ((lst (list (list (list (list (list 4 5 6))))))) (set-car! (caaaar lst) lst))) #t)
(test (tree-cyclic? (let ((lst (list (list 1 2) (list 3) (list 4 5 6)))) (set-cdr! (caddr lst) lst))) #t)
(test (tree-cyclic? (let ((lst (list (list 1 2) (list 3) (list 4 5 6)))) (set-car! (caddr lst) (cdr lst)))) #t)
(test (tree-cyclic? (let ((lst (list (list 1 2) (list 3) (list 4 5 6)))) (set-cdr! (caddr lst) (cdr lst)))) #t)
(test (tree-cyclic? (let ((lst (list (list 1 2) (list 3) (list 4 5 6)))) (set-car! (caddr lst) (cddr lst)))) #t)
(test (tree-cyclic? (let ((lst (list (list 1 2) (list 3) (list 4 5 6)))) (set-cdr! (caddr lst) (cddr lst)))) #t)
(test (tree-cyclic? (let ((lst (list (list 1 2) (list 3) (list 4 5 6)))) (set-car! (cdaddr lst) lst))) #t)
(test (tree-cyclic? (let ((lst (list (list 1 2) (list 3) (list 4 5 6)))) (set-cdr! (cdaddr lst) lst))) #t)
(test (tree-cyclic? (let ((lst (list (list 1 2) (list 3) (list 4 5 6)))) (set-car! (cdaddr lst) (cdr lst)))) #t)
(test (tree-cyclic? (let ((lst (list (list 1 2) (list 3) (list 4 5 6)))) (set-cdr! (cdaddr lst) (cdr lst)))) #t)
(test (tree-cyclic? (let ((lst (list (list 1 2) (list 3) (list 4 5 6)))) (set-car! (cdaddr lst) (cddr lst)))) #t)
(test (tree-cyclic? (let ((lst (list (list 1 2) (list 3) (list 4 5 6)))) (set-cdr! (cdaddr lst) (cddr lst)))) #t)

(let-temporarily (((*s7* 'safety) 1))
  (test (for-each
	 (define-bacro (_b1_ a) `(* ,a 2))
	 (let ((<1> (list #f)))
	   (set-cdr! <1> <1>)
	   (vector (let ((<L> (list #f #f)))
		     (set-cdr! (cdr <L>) <1>)
		     <L>)
		   #f #f)))
	'error))

(let ()
  (let* ((end '(2)) (tree (list end end))) (test (tree-cyclic? tree) #f))
  (test (let ((L (list 1 (list 6 7) 3))
	      (L1 (list 4 5 3)))
	  (set-car! (cdr L1) L)
	  (set-car! (cadr L) L1)
	  (tree-cyclic? (list 1 L 2)))
	#t)

  (define (setf top L)
    (if (pair? top)
	(begin
	  (if (not (car top))
	      (set-car! top L)
	      (setf (car top) L))
	  (setf (cdr top) L))))

  (test (let ((L1 (list (list #f))))
	  (let ((L2 (list (list 0) (list #f))))
	    (setf L1 L2)
	    (setf L2 L2)
	    (tree-cyclic? L1)))
	#t)

  (test (let ((L1 (list 1 2)))
	  (let ((L2 (cons L1 (cons L1 L1))))
	    (tree-cyclic? L2)))
	#f)

  (test (let ((L1 (list 1 2)))
	  (let ((L2 (cons (cons 1 L1) (cons 1 L1))))
	    (tree-cyclic? L2)))
	#f)
)


;;; tree-leaves --------
(test (tree-leaves '(lambda () 1)) 3)
(test (tree-leaves ()) 0)
(test (tree-leaves 1) 1)
(test (tree-leaves '(a . b)) 2)
(test (tree-leaves '(1 (2 3) (4 (5 . 6) . 7))) 7)
(test (tree-leaves '((() (1) (())))) 3)
(test (tree-leaves '(1 (2 (3 (4)) (5 . 6)) 7)) 7)

;;; tree-memq --------
(test (tree-memq 'a '(a b c)) #t)
(test (tree-memq 'a '(b c . a)) #t)
(test (tree-memq 'a '(b c . e)) #f)
(test (tree-memq 'a '(c b c)) #f)
(test (tree-memq 'a '(b c ((b a)))) #t)
(test (tree-memq 3 '(b c ((b 3)))) #t)

;;; tree-count --------
(test (tree-count 'x '(a b c)) 0)
(test (tree-count 'x '(a x c)) 1)
(test (tree-count 'x '(a x x)) 2)
(test (tree-count 'x '(a x x) 1) 1)
(test (tree-count 'x '(x x x) 2) 2)
(test (tree-count 'x 'x) 1)
(test (tree-count 'x 'x 'x) 'error)
(test (tree-count 'x '(a x x x) 1) 1)
(when with-bignums
  (test (tree-count 'x '(a x c) (bignum 10)) 1)
  (test (tree-count 'x () (bignum 10)) 0))

;; tree-count uses eq? which means these two can be different:
;;                        (do ((x 0) (i 0 (+ i 1))) ((= i 1) x) (set! x (tree-count 0 x)))
;;  (begin (define (func) (do ((x 0) (i 0 (+ i 1))) ((= i 1) x) (set! x (tree-count 0 x)))) (func))

;;; tree-set-memq --------
(test (tree-set-memq '(a b) '(1 (2 3 a) 4)) #t)
(test (tree-set-memq '(a b) '(1 (2 3 b) 4)) #t)
(test (tree-set-memq '(a b) '(1 (2 3 c) 4)) #f)
(test (tree-set-memq '(a) ()) #f)

(let-temporarily (((*s7* 'safety) 1))
  (test (tree-count 'a (let ((lst (list 1))) (set-cdr! lst lst))) 'error)
  (test (tree-leaves (let ((lst (list 1))) (set-cdr! lst lst))) 'error)
  (test (tree-memq 'a (let ((lst (list 1))) (set-cdr! lst lst))) 'error)
  (test (tree-set-memq '(a b c) (let ((lst (list 1))) (set-cdr! lst lst))) 'error)
  (test (tree-set-memq (let ((<1> (list #f))) (set-cdr! <1> <1>) (let ((<L> (list #f #f))) (set-cdr! (cdr <L>) <1>)  <L>)) ((lambda* ((a 1)) (+ a 1)) 1)) 'error)
  (test (let () (define (func) (tree-leaves (signature float-vector))) (define (hi) (func)) (hi)) 'error)

  ;; these two check that tree-cyclic? ignores quoted circular lists
  (test (member 1 (list 2 3) (lambda (a b) `(x ,(let ((lst (list 1))) (set-cdr! lst lst))) #f)) #f)
  (test (member 1 (list 3 2) (lambda (a b) (set-cdr! '(x) (let ((lst (list 1))) (set-cdr! lst lst))) (or))) #f))

(let-temporarily (((*s7* 'safety) 1))
  (let ((lst (list 1 2)))
    (set-cdr! (cdr lst) lst)
    (test (tree-leaves lst) 'error)
    (test (tree-memq 'a lst) 'error)
    (test (tree-set-memq '(a b c) lst) 'error)
    (test (tree-count 'a lst) 'error)))



;;; --------------------------------------------------------------------------------
;;; VECTORS
;;; --------------------------------------------------------------------------------


;;; --------------------------------------------------------------------------------
;;; vector?

(test (vector? (make-vector 6)) #t)
(test (vector? (make-vector 6 #\a)) #t)
(test (vector? (make-vector 0)) #t)
;; (test (vector? #*1011) #f)
(test (vector? #(0 (2 2 2 2) "Anna")) #t)
(test (vector? #()) #t)
(test (vector? #("hi")) #t)
(test (vector? (vector 1)) #t)
(test (let ((v (vector 1 2 3))) (vector? v)) #t)

(for-each
 (lambda (arg)
   (test (vector? arg) #f))
 (list #\a 1 () (list 1) '(1 . 2) #f "hi" 'a-symbol abs _ht_ _undef_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (vector?) 'error)
(test (vector? #() #(1)) 'error)
(test (vector? begin) #f)
(test (vector? vector?) #f)

;;; make a shared ref -- we'll check it later after enough has happened that an intervening GC is likely

(define check-subvector-after-gc #f)
(let ((avect (make-vector '(6 6) 32)))
  (do ((i 0 (+ i 1)))
      ((= i 6))
    (do ((j 0 (+ j 1)))
	((= j 6))
      (set! (avect i j) (cons i j))))
  (set! check-subvector-after-gc (avect 3)))

(if (not with-bignums)
    (test (vector? (make-float-vector 3 pi)) #t))
(test (vector? (make-vector 3 pi)) #t)
(test (vector? (subvector (make-int-vector '(2 3)) 0 6 '(3 2))) #t)
(test (vector? #r(+nan.0)) #t)
(test (vector? #r(+inf.0)) #t)
(test (vector? #(-nan.0 -inf.0)) #t)


;;; --------------------------------------------------------------------------------
;;; make-vector

(test (let ((v (make-vector 3 #f))) (and (vector? v) (= (vector-length v) 3) (eq? (vector-ref v 1) #f))) #t)
(test (let ((v (make-vector 1 1))) (and (vector? v) (= (vector-length v) 1) (vector-ref v 0))) 1)
(test (let ((v (make-vector 0 1))) (and (vector? v) (= (vector-length v) 0))) #t)
(test (do ((vec (make-vector 5)) (i 0 (+ i 1))) ((= i 5) vec) (vector-set! vec i i)) #(0 1 2 3 4))
(test (let ((v (make-vector 5))) (for-each (lambda (i) (vector-set! v i (* i i))) '(0 1 2 3 4)) v) #(0 1 4 9 16))
(test (make-vector 2 'hi) #(hi hi))
(test (make-vector 0) #())
(test (make-vector -0) #())
(test (make-vector 0 'hi) #())
(test (make-vector 3 (make-vector 1 'hi)) #(#(hi) #(hi) #(hi)))
(test (make-vector 3 #(hi)) #(#(hi) #(hi) #(hi)))
(test (make-vector 9/3 (list)) #(() () ()))
(test (make-vector 3/1 (make-vector 1 (make-vector 1 'hi))) #(#(#(hi)) #(#(hi)) #(#(hi))))
(test (make-float-vector 0 0.0) #())
(test (make-vector 0 0.0) #())

(test (let ((v (make-vector 3 0))) (set! (vector-ref v 1) 32) v) #(0 32 0))
(test (let ((v (make-int-vector 3))) (set! (vector-ref v 1) 0) v) (make-int-vector 3 0))

(for-each
 (lambda (arg)
   (test (vector-ref (make-vector 1 arg) 0) arg))
 (list #\a 1 () (list 1) '(1 . 2) #f "hi" 'a-symbol abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand (log 0)
       3.14 3/4 1.0+1.0i #t (vector 1 2 3) (lambda (a) (+ a 1))))

(test (make-vector) 'error)
;(test (make-vector 1 #f #t) 'error)
(test (make-vector 1 2 3) 'error)
(test (make-vector most-positive-fixnum) 'error)
(test (make-vector most-negative-fixnum) 'error)
(test (make-vector '(2 -2)) 'error)
(test (make-vector (list 2 -2 -3)) 'error)
(test (make-vector (cons 2 3)) 'error)
(test (make-vector '(2 3 . 4)) 'error)
(test (make-vector '(2 (3))) 'error)
(test (make-vector most-negative-fixnum) 'error)
(test (make-vector 3 0 #f 1) 'error)
(test (with-input-from-string "#0d()" read) 'error) ; in sbcl #0() -> #(), guile gives a confused error "too few elements", but #1() -> #()!
(test (make-vector 0) #())
(test (make-vector '(0)) #())
(test (make-vector ()) 'error)                      ; see above (in sbcl (make-array '()) -> #0A0)
(test (make-vector (list 8796093022208 8796093022208)) 'error)
(test (make-vector (list 8796093022208 2)) 'error)
(test (make-vector (list 8796093022208 -8796093022208)) 'error)

(test (make-vector 3 1 byte?) #u(1 1 1))
(test (make-vector 3 1 integer?) #i(1 1 1))
(test (make-vector 3 1 float?) #r(1 1 1))
(test (make-vector 3 1 #t) #(1 1 1))

(test (make-vector 3 #\a byte?) 'error)
(test (make-vector 3 256 byte?) 'error)
(test (make-vector 3 1.5 integer?) 'error)
(test (make-vector 3 1+i float?) 'error)

(let ((fv (make-vector 3 1.0 float?)))
  (test (vector-set! fv 0 1+i) 'error)
  (test (set! (fv 0) #\a) 'error)
  (copy #r(0 1 2) fv)
  (test fv #r(0 1 2))
  (test (sort! fv char<?) 'error)
  (set! fv (sort! fv >))
  (test fv #r(2 1 0))
  (test (float-vector? fv) #t)
  (float-vector-set! fv 0 32.0)
  (test (float-vector-ref fv 0) 32.0)
  (test (int-vector-ref fv 0) 'error))

(test (make-float-vector 0 1.0) #r())
(test (make-vector 0 1.0 float?) #r())
(test (make-vector 0 1+i float?) 'error)
(test (make-vector 0 1+i integer?) 'error)
(test (make-vector 0 1+i byte?) 'error)
(test (make-vector 0 1 cons) 'error)

(test (make-vector 1 1.0 symbol?) 'error)
(test (make-vector 1 'a symbol?) #(a))
(test (signature (make-vector 1 'a symbol?)) (cons 'symbol? (cons 'vector? (let ((L (list 'integer?))) (set-cdr! L L)))))
(test ((object->let (make-vector 1 'a symbol?)) 'signature) (signature (make-vector 1 'a symbol?)))
(test (arity (make-vector '(2 3) 'a symbol?)) '(1 . 2))
(test (arity (make-vector 0 'a symbol?)) #f)
(test (equal? (make-vector 1 'a symbol?) (make-vector 1 'a symbol?)) #t)
(test (equal? (make-vector 1 'a symbol?) (make-vector 1 1 number?)) #f)
(test (equivalent? (make-vector 1 'a symbol?) (make-vector 1 1 number?)) #f)

(let ((v (make-vector 1 'a symbol?)))
  (let ((v1 (copy v)))
    (test (vector-set! v1 0 123) 'error)
    (test (set! (v1 0) 123) 'error)
    (test (signature v1) (signature v)))
  (let ((sv (subvector v 0 1)))
    (test (vector-set! sv 0 123) 'error)
    (test (set! (sv 0) 123) 'error)
    (test (signature sv) (signature v))))

(let-temporarily (((*s7* 'safety) 1))
  (let ((v (make-vector 1 'a symbol?)))
    (vector-set! v 0 'b)
    (test (v 0) 'b)
    (test (vector-set! v 0 1.5) 'error))
  (let ((v (make-vector '(2 3) 'a symbol?)))
    (vector-set! v 0 0 'b)
    (test (v 0 0) 'b)
    (test (vector-set! v 0 0 1.5) 'error))
  (let ((v (make-vector 1 'a symbol?)))
    (define (fv)
      (vector-set! v 0 'b))
    (fv)
    (test (v 0) 'b)
    (test (let () (define (fv1) (vector-set! v 0 1.5)) (fv1)) 'error)
    (test (let () (define (fv2) (set! (v 0) 1.5)) (fv2)) 'error)
    (test (let () (define (fv1) (vector-set! v 0 'c)) (fv1) v) #(c))
    (test (let () (define (fv2) (set! (v 0) 'd)) (fv2) v) #(d))
    (test (let () (define (fv1) (vector-set! v 0 1.5)) (fv1) (fv1) v) 'error)
    (test (let () (define (fv2) (set! (v 0) 1.5)) (fv2) (fv2) v) 'error)
    (test (let () (define (fv1) (vector-set! v 0 'c)) (fv1) (fv1) v) #(c))
    (test (let () (define (fv2) (set! (v 0) 'd)) (fv2) (fv2) v) #(d)))

  (for-each (lambda (typ arg)
	      (catch #t
		(lambda ()
		  (make-vector 1 arg typ))
		(lambda (type info)
		  (format *stderr* ";(make-vector 1 ~S ~S): ~A~%" arg typ (apply format #f info)))))
	    (list undefined? unspecified? eof-object? boolean? gensym? syntax? symbol? let?
		  openlet? keyword? continuation? number? integer? byte? real? complex? rational? random-state?
		  char? string? input-port? output-port? iterator? null? pair? list? byte-vector? float-vector?
		  int-vector? subvector? vector? hash-table? weak-hash-table? procedure? macro? dilambda?
		  sequence? float? proper-list?)
	    (list #<undefined> #<unspecified> #<eof> #t (gensym) when 'a (inlet 'a 1)
		  (openlet (inlet 'a 1)) :key (call/cc (lambda (r) r)) 1 1 1 1 1+i 1/2 (random-state 1234)
		  #\a "a" (current-input-port) (current-output-port) (make-iterator '(1 2)) () '(1) '(1) #u(1) #r(1)
		  #i(1) (subvector #(1) 0 1) #(1) (hash-table 'a 1) (make-weak-hash-table) abs quasiquote (dilambda (lambda () 1) (lambda (val) val))
		  '(1) 1.0 '(1)))

;;; also bignum?

  (test (let ((v (make-vector 2 1 integer?))) (fill! v 3) v) #i(3 3))
  (test (let ((v (make-vector 2 1 integer?))) (fill! v 1/3) v) 'error)
  (test (let () (define (f) (vector-fill! (make-byte-vector 256 1) (string (integer->char 255)))) (f)) 'error)
  (test (let ((v (make-vector 3 1 integer?))) (copy #(1 2 3) v) v) #i(1 2 3))
  (test (let ((v (make-vector 3 1 integer?))) (copy #u(1 2 3) v) v) #i(1 2 3))
  (test (let ((v (make-vector 3 1 integer?))) (copy #(1 1/2 3) v) v) 'error)

  (test (let ((v (make-vector 2 'a symbol?))) (fill! v 3) v) 'error)
  (test (let ((v (make-vector 2 'a symbol?))) (fill! v 'b) v) #(b b))
  (test (let ((v (make-vector 3 'a symbol?))) (copy #(1 2 3) v) v) 'error)
  (test (let ((v (make-vector 3 'a symbol?))) (copy #(a b c) v) v) #(a b c))
  (test (let ((v (make-vector 3 'a symbol?))) (copy '(1 2 3) v) v) 'error)
  (test (let ((v (make-vector 3 'a symbol?))) (copy '(a b c) v) v) #(a b c))
  (test (let ((v (make-vector 3 'a symbol?))) (copy "123" v) v) 'error)
  (test (let ((v (make-vector 3 'a symbol?))) (copy (hash-table 'a 1) v) v) 'error)
  (test (let ((v (make-vector 3 'a symbol?))) (copy (inlet 'a 1) v) v) 'error)
  (test (let ((v (make-vector 3 'a symbol?))) (copy () v) v) #(a a a)) ; ??
  (test (let ((v (make-vector 3 'a symbol?))) (copy 1 v) v) 'error)
  (test (let ((v (make-vector 3 'a symbol?))) (copy #u(1 2 3) v) v) 'error)
  (test (let ((v (make-vector 3 'a symbol?))) (copy #i(1 2 3) v) v) 'error)
  (test (let ((v (make-vector 3 'a symbol?))) (copy #r(1 2 3) v) v) 'error)

  (test (let ((v (make-vector 3 1.0 real?))) (copy #r(2.0 3.0 1.0) v) (sort! v <) v) #(1.0 2.0 3.0))

  (test (let ((v (vector 1 2 3))) (+ (let-temporarily (((v 0) 4)) (v 0)) (v 1))) 6)
  (test (let ((v (make-vector 3 0 number?))) (copy #i(1 2 3) v) (+ (let-temporarily (((v 0) 4)) (v 0)) (v 1))) 6)
  (test (let ((v (make-vector 3 0 number?))) (let-temporarily (((v 0) #\a)) (v 0))) 'error)
  ) ; let-temp safety=1

(let ((x (make-vector 1 #() vector?)))
  (test (vector-set! x 0 #(1)) #(1))
  (test (vector-ref x 0) #(1))
  (test (vector-set! x 0 #i(1)) #i(1))
  (test (vector-ref x 0) #i(1)))

(when full-s7test
  (catch #t (lambda () (make-vector 1001 '(1) hash-table?)) (lambda args 'error))
  (gc)

  (for-each
   (lambda (typ1 arg1)
     (for-each
      (lambda (typ2 arg2)
	(catch #t
	  (lambda ()
	    (let ((v1 (make-vector 1 arg1 typ1))
		  (v2 (make-vector 1 arg2 typ2)))
	      (if (and (equal? v1 v2)
		       (not (equal? (v1 0) (v2 0))))
		  (format *stderr* ";(equal? ~S ~S ~S ~S): #t?~%" arg1 typ1 arg2 typ2)
		  (if (and (not (equal? v1 v2))
			   (equal? (v1 0) (v2 0)))
		      (format *stderr* ";(equal? ~S ~S ~S ~S): #f?~%" arg1 typ1 arg2 typ2)))
	      (if (and (equivalent? v1 v2)
		       (not (equivalent? (v1 0) (v2 0))))
		  (format *stderr* ";(equivalent? ~S ~S ~S ~S): #t?~%" arg1 typ1 arg2 typ2)
		  (if (and (not (equivalent? v1 v2))
			   (equivalent? (v1 0) (v2 0)))
		      (format *stderr* ";(equivalent? ~S ~S ~S ~S): #f?~%" arg1 typ1 arg2 typ2)))))
	  (lambda (type info)
	    (format *stderr* ";equal? ~S ~S ~S ~S: ~A~%" arg1 typ1 arg2 typ2 (apply format #f info)))))

      (list undefined? unspecified? eof-object? boolean? gensym? syntax? symbol? let?
	    openlet? keyword? continuation? number? integer? byte? float? real? complex? rational? random-state?
	    char? string? input-port? output-port? iterator? null? pair? list? byte-vector? float-vector?
	    int-vector? subvector? vector? hash-table? weak-hash-table? procedure? macro? dilambda?
	    sequence? proper-list?)
      (list #<undefined> #<unspecified> #<eof> #t (gensym) when 'a (inlet 'a 1)
	    (openlet (inlet 'a 1)) :key (call/cc (lambda (r) r)) 1 1 1 1 1 1 1 (random-state 1234)
	    #\a "a" (current-input-port) (current-output-port) (make-iterator '(1)) () '(1) '(1) #u(1) #r(1)
	    #i(1) (subvector #(1) 0 1) #(1) (hash-table 'a 1) (make-weak-hash-table) abs quasiquote
	    (dilambda (lambda () 1) (lambda (val) val)) '(1) '(1))))

   (list undefined? unspecified? eof-object? boolean? gensym? syntax? symbol? let?
	 openlet? keyword? continuation? number? integer? byte? float? real? complex? rational? random-state?
	 char? string? input-port? output-port? iterator? null? pair? list? byte-vector? float-vector?
	 int-vector? subvector? vector? hash-table? weak-hash-table? procedure? macro? dilambda?
	 sequence? proper-list?)
   (list #<undefined> #<unspecified> #<eof> #t (gensym) when 'a (inlet 'a 1)
	 (openlet (inlet 'a 1)) :key (call/cc (lambda (r) r)) 1 1 1 1 1 1 1 (random-state 1234)
	 #\a "a" (current-input-port) (current-output-port) (make-iterator '(1)) () '(1) '(1) #u(1) #r(1)
	 #i(1) (subvector #(1) 0 1) #(1) (hash-table 'a 1) (make-weak-hash-table) abs quasiquote
	 (dilambda (lambda () 1) (lambda (val) val)) '(1) '(1))))

;; --------
(for-each
 (lambda (arg)
   (test (make-vector arg) 'error)
   (test (make-vector (list 2 arg)) 'error))
 (list #\a () -1 #f "hi" 'a-symbol abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t (vector 1 2 3) (lambda (a) (+ a 1))))

(test (eval-string "#2147483649D()") 'error)
(test (type-of (eval-string "#-9223372036854775808D()")) 'error)
(test (eval-string "#922D()") 'error)
(test (eval-string "#(1 2 . 3)") 'error)
(test (eval-string "#(1 2 . ())") #(1 2))
(test (eval-string "#i(1 2 . 3)") 'error)
(test (eval-string "#u(1 2 . 3)") 'error)
(test (eval-string "#r(1.0 2.0 . 3.0)") 'error)
(test (eval-string "#2d((1 2) (3 4) . 5)") 'error)
(test (eval-string "#2d((1 2) (3 . 4))") 'error)
(test (eval-string "#2d((1 2) (3 4 . ()))") #2d((1 2) (3 4)))
(test (eval-string "#(1 2 . (3 4))") #(1 2 3 4))

;;; subvector
(test (let ((v1 #2d((1 2 3) (4 5 6)))) (let ((v2 (subvector v1 0 6 '(6)))) v2)) #(1 2 3 4 5 6))
(test (let ((v1 #(1 2 3 4 5 6))) (let ((v2 (subvector v1 0 6 '(3 2)))) v2)) #2d((1 2) (3 4) (5 6)))
(test (subvector #2d() 0 0 '(0)) #())
(test (subvector '(1) 0 1 '(1)) 'error)
(test (subvector #(1) '(2)) 'error)
(test (subvector #(1) 0 2 '(1 2)) 'error)
(test (subvector #(1 2 3 4) ()) 'error)
(test (subvector #(1 2 3 4) 0 most-positive-fixnum) 'error)
(test (subvector #(1 2 3 4) 0 most-negative-fixnum) 'error)
(test (subvector #(1 2 3 4) 0 -1) 'error)
(test (subvector #(1 2 3 4) 0 5) 'error)
(test (subvector #(1 2 3 4) 0 0) #())
(test (subvector #(1 2 3 4) 0 2 '(2)) #(1 2))
(test (subvector #(1 2 3 4) 0 2 '(2 1)) #2d((1) (2)))
(test (subvector #(1 2 3 4) 0 0 '(0)) #())
(test (subvector #() 1) 'error)
(test (subvector #() 0) #())
(test (subvector #(1) 0 0) #())
(let ((v #(1))) (test (subvector-vector (subvector v 0 0)) v))
(test (subvector #(1) 1 2) 'error)
(test (subvector (make-vector 3) 1 most-positive-fixnum) 'error)
(test (subvector (make-vector 3) 1 4) 'error)
(test (subvector (make-vector 3) 0 4) 'error)
(test (subvector (make-vector 3) -1 2) 'error)
(test (subvector (vector 1 2 3 4 5 6) 0 3 '(3 2)) 'error)
(test (subvector (vector 1 2 3 4 5 6) 0 6 '(2 2)) 'error)

(test (subvector (make-vector 2 1) 0 (bignum 2)) #(1 1))
(test (subvector (make-vector (list 2 3) #f) 0 (bignum 3)) #(#f #f #f))

(test (subvector) 'error)
(for-each
 (lambda (arg)
   (test (subvector arg) 'error)
   (test (subvector #(1 2 3) arg) 'error))
 (list #\a () -1 #f "hi" 'a-symbol abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t (lambda (a) (+ a 1))))

(let ((v #2d((1 2) (3 4))))
  (test (subvector v 0 2 '((1 2) (3 4))) 'error)
  (test (subvector v 0 0 ()) 'error)
  (test (subvector v 0 1 '(1.4)) 'error)
  (test (subvector v 0 2 '(14 15)) 'error)
  (test (subvector v 0 4 (list most-positive-fixnum)) 'error)
  (test (subvector v 0 4 '(-1 0)) 'error)
  (test (subvector v most-positive-fixnum '(1)) 'error))

(let ((v (float-vector 0.0 1.0 2.0)))
  (let ((v1 (subvector v 0 3 (list 1 3))))
    (test (float-vector? v1) #t)
    (test (equivalent? (v 0) (v1 0 0)) #t)))

(let ((v (vector 0.0 1.0 2.0)))
  (let ((v1 (subvector v 0 3 (list 1 3))))
    (test (vector? v1) #t)
    (test (equivalent? (v 0) (v1 0 0)) #t)))

(let ((v (int-vector 0 1 2)))
  (let ((v1 (subvector v 0 3 (list 1 3))))
    (test (int-vector? v1) #t)
    (test (equivalent? (v 0) (v1 0 0)) #t)))

(let ((v (byte-vector 0 1 2)))
  (let ((v1 (subvector v 0 3 (list 1 3))))
    (test (byte-vector? v1) #t)
    (test (equivalent? (v 0) (v1 0 0)) #t)))

(let ((v (make-int-vector 3)))
  (set! (v 1) 1)
  (set! (v 2) 2)
  (let ((v1 (subvector v 0 3 (list 1 3))))
    (test (float-vector? v1) #f)
    (test (int-vector? v1) #t)
    (test (integer? (v1 0 2)) #t)
    (test (= (v 2) (v1 0 2)) #t)))

(let ((v (vector 0 1 2 3 4 5 6 7 8)))
  (test (subvector v 1 7 (list 3 2)) #2d((1 2) (3 4) (5 6)))
  (test (subvector v 2 8 (list 3 2)) #2d((2 3) (4 5) (6 7)))
  (test (subvector v 2 5 (list 3)) #(2 3 4))
  (test (subvector v 0 3 (list 3)) (subvector v 0 3 (list 3)))
  (test (subvector v -1 3 (list 3)) 'error)
  (test (subvector v 10 3 (list 3)) 'error)
  (test (subvector v 3.2 3 (list 3)) 'error)
  (test (subvector v "0" 3 (list 3)) 'error)
  )

(test (subvector #() 1) 'error)
(test (subvector #() 0 1) 'error)
(test (subvector #(1) 1 2) 'error)
(test (subvector #(1) 0 1) #(1))

(let ((a (vector 1 2 3))
      (b (vector 4 5 6)))
  (test (subvector (append a b) 0 6 '(2 3)) #2d((1 2 3) (4 5 6))))

(let ((a #2d((1 2) (3 4)))
      (b #2d((5 6) (7 8))))
  (test (subvector (append a b) 0 8 '(2 4))                     #2d((1 2 3 4) (5 6 7 8)))
  (test (subvector (append a b) 0 8 '(4 2))                     #2d((1 2) (3 4) (5 6) (7 8)))
  (test (subvector (append (a 0) (b 0) (a 1) (b 1)) 0 8 '(2 4)) #2d((1 2 5 6) (3 4 7 8)))
  (test (subvector (append (a 0) (b 0) (a 1) (b 1)) 0 8 '(4 2)) #2d((1 2) (5 6) (3 4) (7 8))))

(test (subvector (subvector (float-vector 1.0 2.0 3.0 4.0) 0 4 '(2 2)) 0 0 '(0)) #())
(test (subvector (subvector (float-vector 1.0 2.0 3.0 4.0) 0 4 '(2 2)) 0 1 '(1)) (float-vector 1.0))
(test ((subvector (subvector (float-vector 1.0 2.0 3.0 4.0) 0 4 '(2 2)) 0 4 '(4 1)) 2 0) 3.0)

(let ((v (subvector (float-vector 1 2 3) 0 3)))
  (test (copy v) (subvector (float-vector 1 2 3) 0 3))
  (test (length v) 3)
  (test (reverse v) (float-vector 3 2 1))
  (test v #r(1 2 3))
  (set! v (reverse! v))
  (test v #r(3 2 1)))

;;; subvector?
(let ((v (vector 0 1 2 3 4 5 6 7 8)))
  (test (subvector? (subvector v 1 3)) #t)
  (test (subvector? v) #f))
(for-each
 (lambda (arg)
   (test (subvector? arg) #f)
   (test (subvector-position arg) 'error)
   (test (subvector-vector arg) 'error))
 (list #\a () -1 #f "hi" 'a-symbol abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t (vector 1 2 3) (lambda (a) (+ a 1))))
(test (subvector?) 'error)
(test (subvector? (subvector-vector (subvector #(1) 0 1))) #f)
(test (vector? (subvector-vector (subvector #(1) 0 1))) #t)

;;; subvector-position
(let ((v (vector 0 1 2 3 4 5 6 7 8)))
  (test (subvector-position (subvector v 1 3)) 1)
  (test (subvector-position (subvector v 0 1)) 0)
  (test (subvector-position (subvector v 1 5 '(2 2))) 1))

(let ((b (make-byte-vector '(2 3))))
  (do ((i 0 (+ i 1)))
      ((= i 2))
    (do ((j 0 (+ j 1)))
	((= j 3))
      (set! (b i j) (+ (* i 3) j))))
  (test b #u2d((0 1 2) (3 4 5)))
  (let ((b1 (byte-vector-ref b 1)))
    (test b1 #u(3 4 5))
    (test (subvector? b1) #t)
    (test (subvector-vector b1) b)
    (test (subvector-position (byte-vector-ref b 0)) 0)
    (test (subvector-position b1) 3)))

(test (subvector-position (vector-ref #2d((1 2 3) (4 5 6)) 1)) 3)
(test (subvector-position (float-vector-ref #r2d((1 2 3) (4 5 6)) 1)) 3)
(test (subvector-position (int-vector-ref #i2d((1 2 3) (4 5 6)) 1)) 3)

(test (subvector-position v) 'error)
(test (subvector-position) 'error)
(test (subvector-position (subvector #(1 2 3) 1 2)) 1) ; 2 = new dims, 1 = offset
(test (subvector-position (subvector #(0 1 2 3) 2 3)) 2)
(test (subvector-position (subvector #r(0 1 2 3) 2 2)) 2)
(test (subvector-position (subvector #i(0 1 2 3) 2 3)) 2)
(test (subvector-position (subvector #u(0 1 2 3) 2 3)) 2)
(test (subvector-position (subvector #2d() 0)) 0)
(test (subvector-position (subvector #u() 0)) 0)

;;; subvector-vector
(let ((v (vector 0 1 2 3 4 5 6 7 8)))
  (test (subvector-vector (subvector v 0 2)) v)
  (test (subvector-vector v) 'error))
(let ((v (int-vector 0 1 2 3 4 5 6 7 8))) (test (subvector-vector (subvector v 0 2)) v))
(let ((v (float-vector 0 1 2 3 4 5 6 7 8))) (test (subvector-vector (subvector v 0 2)) v))
(let ((v (byte-vector 0 1 2 3 4 5 6 7 8))) (test (subvector-vector (subvector v 0 2)) v))
(let ((v (subvector #u() 0))) (test (subvector-vector v) #u()))

(let ((v #(3 1 2 4 0)))
  (sort! (subvector v 2 5) <)
  (test v #(3 1 0 2 4)))

(when full-s7test
  ;; this hits a gc mark bug (now fixed)
  (let ((v (subvector (make-int-vector '(2 3)) 0 6))) (gc) (gc) (object->let v)))



;;; --------
(let-temporarily (((*s7* 'print-length) 123123123))
  (test (object->string (make-vector 2048 #f)) "(make-vector 2048 #f)")
  (test (object->string (make-vector '(12 2048) #<unspecified>)) "(make-vector '(12 2048) #<unspecified>)")
  (test (object->string (make-float-vector 2048 1.0)) "(make-float-vector 2048 1)")
  (test (object->string (make-int-vector 2048 32)) "(make-int-vector 2048 32)")
  (test (object->string (make-int-vector '(12 2048) 2)) "(make-int-vector '(12 2048) 2)")
  (test (object->string (make-string 20000 #\space)) "(make-string 20000 #\\space)")
  (test (object->string (make-byte-vector 2000 12)) "(make-byte-vector 2000 12)"))

(when with-bignums
  (let ((v (float-vector (bignum "1.0") (bignum "2.0"))))
    (test (float-vector? v) #t)
    (test (v 0) 1.0)))

(test (vector? (float-vector)) #t)
(test (vector? (int-vector)) #t)
(when with-block (test (float-vector? _c_obj_) #t))
(test (float-vector? 1 2) 'error)
(test (float-vector?) 'error)
(test (int-vector? 1 2) 'error)
(test (int-vector?) 'error)
(for-each
 (lambda (arg)
   (if (float-vector? arg) (format *stderr* ";~A is a float-vector?~%" arg))
   (test (float-vector arg) 'error)
   (if (int-vector? arg) (format *stderr* ";~A is an int-vector?~%" arg))
   (test (int-vector arg) 'error))
 (list #\a () #f "hi" 'a-symbol abs _ht_ _undef_ _null_ quasiquote macroexpand #() #t (vector 1 2 3) (lambda (a) (+ a 1))))


;;; make-float-vector
(test (float-vector? (make-float-vector 3)) #t)
(test (float-vector? (make-float-vector 3 pi)) #t)
(test ((make-float-vector 3) 1) 0.0)
(test (float-vector? (float-vector)) #t)
(test (float-vector? (make-float-vector 0)) #t)
(test (float-vector? (int-vector)) #f)
(test (equal? (float-vector) (int-vector)) #t)
(test (equal? (vector) (int-vector)) #t)

(test (equal? (make-float-vector 3 1.0) (float-vector 1.0 1.0 1.0)) #t)
(test (equal? (make-float-vector 3 1/2) (float-vector 0.5 0.5 0.5)) #t)
(test ((make-float-vector '(2 3) 2.0) 1 2) 2.0)
(test (nan? ((make-float-vector 3 1/0) 0)) #t)
(for-each
 (lambda (arg)
   (test (make-float-vector arg) 'error)
   (test (make-float-vector 3 arg) 'error))
 (list #\a () #f "hi" 'a-symbol abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand (log 0) 1.0+1.0i #t (vector 1 2 3) (lambda (a) (+ a 1))))
(test (equal? (vector) (float-vector)) #t)
(test (float-vector? (make-float-vector 3 0)) #t)
(test (float-vector? (make-float-vector 3 1/2)) #t)
(test (float-vector? #r(1.0)) #t)
(let ((v (make-vector 1 2.0 float?)))
  (test (float-vector? v) #t)
  (test (float-vector-ref v 0) 2.0))
(test (make-float-vector -12) 'error)
(test (make-float-vector -12 1.0) 'error)
(test (make-float-vector 4294967298) 'error)
(test (make-float-vector 4294967298 1.0) 'error)
(unless with-bignums
  (test (s7-optimize '((float-vector-ref (make-float-vector 4294967298 1.0) 0))) 'error) ; try to hit make_float_vector_p_pp
  (test (s7-optimize '((float-vector-ref (make-float-vector -8 1.0) 0))) 'error))

;;; make-int-vector
(test (int-vector? (make-int-vector 3)) #t)
(test (int-vector? (make-int-vector 3 2)) #t)
(test ((make-int-vector 3) 1) 0)
(test (int-vector? (int-vector)) #t)
(test (int-vector? (make-int-vector 0)) #t)
(test (int-vector? (float-vector)) #f)
(test (int-vector? (vector)) #f)
(test (int-vector? #i(1)) #t)
(unless with-bignums
  (let ((v (make-vector 1 2 integer?))) ; this is big-integer in gmp case
    (test (int-vector? v) #t)
    (test (int-vector-ref v 0) 2)))
(when with-bignums
  (let ((v (make-int-vector (bignum "1"))))
    (test (int-vector? v) #t)
    (test (length v) 1)
    (test (make-int-vector 3 (bignum "1")) #i(1 1 1))
    (test (make-int-vector (bignum "3") 1) #i(1 1 1))
    (test (make-int-vector (bignum "3") (bignum "1")) #i(1 1 1))
    (test (make-float-vector (bignum "2") 1.0) #r(1.0 1.0))
    (test (make-float-vector 3 (bignum "1.0")) #r(1.0 1.0 1.0))
    (test (make-float-vector (bignum "3") 1.0) #r(1.0 1.0 1.0))
    (test (make-float-vector (bignum "3") (bignum "1.0")) #r(1.0 1.0 1.0))
    (test (make-byte-vector 3 (bignum "1")) #u(1 1 1))
    (test (make-byte-vector (bignum "3") 1) #u(1 1 1))
    (test (make-byte-vector (bignum "3") (bignum "1")) #u(1 1 1))
    (set! v (make-byte-vector 3))
    (fill! v (bignum "1"))))

(test ((lambda () (make-vector (call-with-exit (lambda (return) (let ((x 1) (y 2)) (return x y)))) (vector? (make-float-vector '(2 3) 1))))) #(2))
(when with-block
  (test ((lambda () (list (call-with-exit (lambda (return) (let ((x 1) (y 2)) (return x y)))) (append (block) (block))))) (list 1 2 (block))))

(test (equal? (make-int-vector 3 1) (int-vector 1 1 1)) #t)
(test ((make-int-vector '(2 3) 2) 1 2) 2)

(test (make-int-vector -3) 'error)
(test (make-float-vector -3) 'error)
(test (make-vector -3) 'error)
(test (make-byte-vector -3) 'error)
(test (make-vector (list 1 pi)) 'error)
(test (eval-string "#r2d((1 2) (3 #\\a))") 'error)
(test (eval-string "#i2d((1 2) (3 #\\a))") 'error)
(let ((v (make-vector 1 2 byte?)))
  (test (byte-vector? v) #t)
  (test (byte-vector-ref v 0) 2))
(test (make-int-vector 1 integer?) 'error)
(test (make-int-vector 1 0 integer?) 'error)
(test (make-int-vector -12) 'error)
(test (make-int-vector -12 1.0) 'error)
(test (make-int-vector 4294967298) 'error)
(test (make-int-vector 4294967298 1.0) 'error)
(unless with-bignums
  (test (s7-optimize '((int-vector-ref (make-int-vector 4294967298 1.0) 0))) 'error)
  (test (s7-optimize '((int-vector-ref (make-int-vector -8 1.0) 0))) 'error))

(for-each
 (lambda (arg)
   (test (make-int-vector arg) 'error)
   (test (make-int-vector 3 arg) 'error))
 (list #\a () #f "hi" 'a-symbol abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand (log 0) 1.0+1.0i 1/2 pi #t (vector 1 2 3) (lambda (a) (+ a 1))))
(test (equal? (vector) (int-vector)) #t)

(test (catch #t
	(lambda () (make-float-vector 3.0))
	(lambda args
	  (let ((type (car args))
		(errmsg (apply format #f (cadr args))))
	    (list type errmsg))))
      '(wrong-type-arg "make-float-vector first argument, 3.0, is a real but should be an integer or a list of integers"))

(test (catch #t
	(lambda () (make-int-vector 3.0))
	(lambda args
	  (let ((type (car args))
		(errmsg (apply format #f (cadr args))))
	    (list type errmsg))))
      '(wrong-type-arg "make-int-vector first argument, 3.0, is a real but should be an integer or a list of integers"))

(test (catch #t
	(lambda () (make-vector 3.0))
	(lambda args
	  (let ((type (car args))
		(errmsg (apply format #f (cadr args))))
	    (list type errmsg))))
      '(wrong-type-arg "make-vector first argument, 3.0, is a real but should be an integer or a list of integers"))

(test (eval-string "#(0 1 . 2)") 'error)
(test (eval-string "#r(0 1 . 2)") 'error)
(test (eval-string "#i(0 1 . 2)") 'error)

;;; float-vector-ref
;;; float-vector-set!

(test (float-vector-ref (float-vector 1.0 2.0) 1) 2.0)
(for-each
 (lambda (arg)
   (test (float-vector-ref arg 0) 'error)
   (test (float-vector-ref (float-vector 1.0) arg) 'error))
 (list #\a () #f "hi" 'a-symbol abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand (log 0) 1.0+1.0i #t (vector 1 2 3) (lambda (a) (+ a 1))))
(let ((v (make-float-vector (list 2 3) 1.0))
      (v1 (make-float-vector 3)))
  (set! (v 1 1) 2.0)
  (test (v 1 1) 2.0)
  (test (v 0 1) 1.0)
  (test (float-vector-ref v 1 1) 2.0)
  (test (float-vector-ref #r2d((1 2) (3 4)) 1 1) 4.0)
  (test (float-vector-ref v 0) (float-vector 1.0 1.0 1.0))
  (test (float-vector-set! v 0 0 3.0) 3.0)
  (test (float-vector-ref v 0 0) 3.0)
  (test (float-vector-ref v1 3) 'error)
  (test (float-vector-ref v 1 3) 'error)
  (test (float-vector-ref v 2 2) 'error)
  (test (float-vector-ref v1 most-positive-fixnum) 'error)
  (test (float-vector-ref v1 most-negative-fixnum) 'error)
  (test (float-vector-set! v1 3 0.0) 'error)
  (test (float-vector-set! v 1 3 0.0) 'error)
  (test (float-vector-set! v 2 2 0.0) 'error)
  (test (float-vector-set! v1 most-positive-fixnum 0.0) 'error)
  (test (float-vector-set! v1 most-negative-fixnum 0.0) 'error)
  (test (float-vector-set! v1 0 0+i) 'error)
  (for-each
   (lambda (arg)
     (test (float-vector-ref v 0 arg) 'error)
     (test (float-vector-set! arg 0 1.0) 'error)
     (test (float-vector-set! v1 arg) 'error)
     (test (float-vector-set! v1 0 arg) 'error)
     (test (float-vector-set! v 0 arg 1.0) 'error))
   (list #\a () #f "hi" 1+i 'a-symbol abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand #t (vector 1 2 3) (lambda (a) (+ a 1))))
  (test (float-vector-ref v) 'error)
  (test (float-vector-set! v) 'error)
  (test (float-vector-ref v1 0 1) 'error)
  (test (float-vector-ref v 0 1 0) 'error)
  (test (float-vector-ref v1 -1) 'error)
  (float-vector-set! v1 0 2/5)
  (test (float-vector-ref v1 0) 0.4)
  (test (float-vector-set! v1 1 4.0) 4.0)
  (test (float-vector-ref v1 1) 4.0)
  (test (float-vector-ref v 3 0) 'error)
  (test (float-vector-ref v 1 3) 'error)
  (test (fill! v 0.0) 0.0))
(test (float-vector-ref (float-vector) 0) 'error)
(let ((v (float-vector 1 2 3)))
  (set! (float-vector-ref v 1) 32)
  (test v (float-vector 1 32 3))
  (set! (v 0) 64)
  (test v (float-vector 64 32 3))
  (test (float-vector-set! v 2 (float-vector-set! v 1 0.0)) 0.0)
  (test v (float-vector 64 0 0)))
(let ((v0 (make-float-vector '(3 0)))
      (v1 (make-float-vector '(0 3)))
      (v2 (make-float-vector '(2 3)))
      (v3 (make-float-vector '(1 3)))
      (v4 (make-float-vector '(3 1))))
  (test (float-vector? v0) #t)
  (test (float-vector-ref v0 0 0) 'error)
  (test (vector? v0) #t)
  (test (vector-ref v0 0 0) 'error)
  (test (v0 0 0) 'error)
  (test (float-vector? v1) #t)
  (test (float-vector-ref v1 0 0) 'error)
  (test (vector? v1) #t)
  (test (vector-ref v1 0 0) 'error)
  (test (v1 0 0) 'error)
  (test (equal? v0 v1) #f)
  (test (float-vector? (float-vector-ref v2 1)) #t)
  (test (float-vector-set! v2 1 32.0) 'error)
  (test (float-vector-set! (float-vector-ref v2 1) 1 32.0) 32.0)
  (test (float-vector-ref v2 1 1) 32.0)
  (test (float-vector-ref v3 0) (float-vector 0 0 0))
  (test (float-vector-ref v4 0) (float-vector 0))
  )
(let ()
  (define (hi) (let ((v2 (make-float-vector '(2 3)))) (float-vector-set! v2 1 12.0) v2))
  (test (hi) 'error))

(let ()
  (define (f1) ; opt_d_7piid_sfff
    (let ((fv (make-float-vector '(2 3))))
      (do ((i 0 (+ i 1)))
	  ((= i 2) fv)
        (float-vector-set! fv (+ i 0) (+ i 1) (* 2.0 3.0)))))
  (test (f1) #r2d((0.0 6.0 0.0) (0.0 0.0 6.0)))

  (define (f2) ; opt_d_7pii_sff
    (let ((iv (make-float-vector '(2 3) 1.0))
   	  (sum 0.0))
      (do ((i 0 (+ i 1)))
	  ((= i 2) sum)
        (set! sum (+ sum (iv (- (+ i 1) 1) (+ i 1)))))))
  (test (f2) 2.0)

  (define (f3) ; opt_d_7pii_sff
    (let ((iv (make-float-vector '(2 3) 1.0))
  	  (sum 0.0))
      (do ((i 0 (+ i 1)))
	  ((= i 2) sum)
        (set! sum (+ sum (float-vector-ref iv (- (+ i 1) 1) (+ i 1)))))))
  (test (f3) 2.0))

(let () ; regression test for optimizer safe_c_opcq_opcq bug
  (define (fx n x y)
    (make-float-vector (if x (+ n 1) n)
		       (if y 0 (/ pi 2))))
  (test (equivalent? (fx 3 #f #f) (make-float-vector 3 (/ pi 2))) #t)
  (test (equivalent? (fx 3 #f #t) (make-float-vector 3)) #t)
  (test (equivalent? (fx 3 #t #f) (make-float-vector 4 (/ pi 2))) #t)
  (test (equivalent? (fx 3 #t #t) (make-float-vector 4)) #t)

  (define (fx1 n x y)
    (make-float-vector (if x (+ n 1) (- n 1))
		       (if y (* pi 2) (/ pi 2))))
  (test (equivalent? (fx1 3 #f #f) (make-float-vector 2 (/ pi 2))) #t)
  (test (equivalent? (fx1 3 #f #t) (make-float-vector 2 (* pi 2))) #t)
  (test (equivalent? (fx1 3 #t #f) (make-float-vector 4 (/ pi 2))) #t)
  (test (equivalent? (fx1 3 #t #t) (make-float-vector 4 (* pi 2))) #t)

  (define (fx2 n x y)
    (make-float-vector (if x (+ n 1) n)
		       (if y (* pi 2) 0.0)))
  (test (equivalent? (fx2 3 #f #f) (make-float-vector 3)) #t)
  (test (equivalent? (fx2 3 #f #t) (make-float-vector 3 (* pi 2))) #t)
  (test (equivalent? (fx2 3 #t #f) (make-float-vector 4)) #t)
  (test (equivalent? (fx2 3 #t #t) (make-float-vector 4 (* pi 2))) #t)

  (define (fx3 n y) ; same for safe_c_opssq_opcq
    (make-float-vector (+ n n)
		       (if y 0.0 (/ pi 2))))
  (test (equivalent? (fx3 3 #f) (make-float-vector 6 (/ pi 2))) #t)
  (test (equivalent? (fx3 3 #t) (make-float-vector 6)) #t)

  (let ((v (make-float-vector 6))) (test (equivalent? (fx3 3 #t) v) #t))
  )

(let ((fv (make-float-vector 10))
      (ten 10))
  (define (fvf)
    (do ((i 0 (+ i 1)))
	((= i ten))
      (float-vector-set! fv i 1.0)))
  (fvf)
  (test fv #r(1 1 1 1 1 1 1 1 1 1))
  (define (fzf)
    (do ((i 3 (+ i 1)))
	((= i 8))
      (float-vector-set! fv i 0)))
  (fzf)
  (test fv #r(1 1 1 0 0 0 0 0 1 1)))

(let () ; optimizer type check
  (define (f1)
    (let ((v (float-vector 1 2 3))
	  (s "asdf"))
      (do ((i 0 (+ i 1)))
	  ((= i 3) v)
	(float-vector-set! v i 123)
	(set! v s))))
  (test (f1) 'error)

  (define (f2)
    (let ((v (float-vector 1 2 3)))
      (do ((i 0 (+ i 1)))
	  ((= i 3) v)
	(float-vector-set! v i 123)
	(set! v "asdf"))))
  (test (f2) 'error)

  (define (f3)
    (let ((v (float-vector 1 2 3)))
      (do ((i 0 (+ i 1)))
	  ((= i 3) v)
	(float-vector-set! v i 123)
	(set! v (substring "asdfg" 0 4)))))
  (test (f3) 'error))

(let ((rv #r(.0 ; a comment
	     +inf.0 #| another |# 1e5 #xa.a)))
  (test (equivalent? rv (float-vector 0.0 +inf.0 100000.0 10.625)) #t))
(let ((iv #i(9223372036854775807 -0 0000)))
  (test iv (int-vector most-positive-fixnum 0 0)))

(let ((fv (float-vector 0 1)))
  (test (float-vector-ref fv 0 0) 'error)
  (test (vector-ref fv 0 0) 'error)
  (test (fv 0 0) 'error))
(let ((fv #r2d((1 2) (3 4))))
  (test (float-vector-ref fv 0 0 0) 'error)
  (test (float-vector-ref fv 0 0 0 0) 'error)
  (test (vector-ref fv 0 0 0) 'error)
  (test (fv 0 0 0) 'error))

(let ()
  (define (h14)
    (let ((v (make-float-vector (list 10 10)))
	  (v1 (make-vector 10)))
      (do ((i 0 (+ i 1)))
	  ((= i 10) v1)
	(vector-set! v1 0 (float-vector-set! v 0 0 2.0))))) ; cell_optimize d_7piid d_to_p fixup */
  (let ((v (h14))) (test (v 0) 2.0)))

(unless with-bignums
  (test (let ((alias float-vector-ref)) (s7-optimize (list (list 'alias (list 'hash-table) 'x)))) #<undefined>))


;;; int-vector-ref
;;; int-vector-set!

(test (int-vector-ref (int-vector 1 2) 1) 2)
(for-each
 (lambda (arg)
   (test (int-vector-ref arg 0) 'error)
   (test (int-vector-ref (int-vector 1) arg) 'error))
 (list #\a () #f "hi" 'a-symbol abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand (log 0) 1.0+1.0i #t (vector 1 2 3) (lambda (a) (+ a 1))))
(let ((v (make-int-vector (list 2 3) 1))
      (v1 (make-int-vector 3)))
  (set! (v 1 1) 2)
  (test (v 1 1) 2)
  (test (v 0 1) 1)
  (test (int-vector-ref v 1 1) 2)
  (test (int-vector-ref #i2d((1 2) (3 4)) 1 1) 4)
  (test (int-vector-ref v 0) (int-vector 1 1 1))
  (test (int-vector-set! v 0 0 3) 3)
  (test (int-vector-ref v 0 0) 3)
  (test (int-vector-ref v1 3) 'error)
  (test (int-vector-ref v 1 3) 'error)
  (test (int-vector-ref v 2 2) 'error)
  (test (int-vector-ref v1 most-positive-fixnum) 'error)
  (test (int-vector-ref v1 most-negative-fixnum) 'error)
  (test (int-vector-set! v1 3 0) 'error)
  (test (int-vector-set! v 1 3 0) 'error)
  (test (int-vector-set! v 2 2 0) 'error)
  (test (int-vector-set! v1 most-positive-fixnum 0) 'error)
  (test (int-vector-set! v1 most-negative-fixnum 0) 'error)
  (test (int-vector-set! v1 0 0+i) 'error)
  (for-each
   (lambda (arg)
     (test (int-vector-ref v 0 arg) 'error)
     (test (int-vector-set! arg 0 1) 'error)
     (test (int-vector-set! v1 arg) 'error)
     (test (int-vector-set! v1 0 arg) 'error)
     (test (int-vector-set! v 0 arg 1) 'error))
   (list #\a () #f "hi" 1+i 'a-symbol abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand #t (vector 1 2 3) (lambda (a) (+ a 1))))
  (test (int-vector-ref v) 'error)
  (test (int-vector-set! v) 'error)
  (test (int-vector-ref v1 0 1) 'error)
  (test (int-vector-ref v 0 1 0) 'error)
  (test (int-vector-ref v1 -1) 'error)
  (int-vector-set! v1 0 2)
  (test (int-vector-ref v1 0) 2)
  (test (int-vector-set! v1 1 4) 4)
  (test (int-vector-ref v1 1) 4)
  (test (int-vector-ref v 3 0) 'error)
  (test (int-vector-ref v 1 3) 'error)
  (test (fill! v 0) 0))
(test (int-vector-ref (int-vector) 0) 'error)
(let ((v (int-vector 1 2 3)))
  (set! (int-vector-ref v 1) 32)
  (test v (int-vector 1 32 3))
  (set! (v 0) 64)
  (test v (int-vector 64 32 3))
  (test (int-vector-set! v 2 (int-vector-set! v 1 0)) 0)
  (test v (int-vector 64 0 0)))
(let ((v0 (make-int-vector '(3 0)))
      (v1 (make-int-vector '(0 3)))
      (v2 (make-int-vector '(2 3)))
      (v3 (make-int-vector '(1 3)))
      (v4 (make-int-vector '(3 1))))
  (test (int-vector? v0) #t)
  (test (int-vector-ref v0 0 0) 'error)
  (test (vector? v0) #t)
  (test (vector-ref v0 0 0) 'error)
  (test (v0 0 0) 'error)
  (test (int-vector? v1) #t)
  (test (int-vector-ref v1 0 0) 'error)
  (test (vector? v1) #t)
  (test (vector-ref v1 0 0) 'error)
  (test (v1 0 0) 'error)
  (test (equal? v0 v1) #f)
  (test (int-vector? (int-vector-ref v2 1)) #t)
  (test (int-vector-set! v2 1 32) 'error)
  (test (int-vector-set! (int-vector-ref v2 1) 1 32) 32)
  (test (int-vector-ref v2 1 1) 32)
  (test (int-vector-ref v3 0) (int-vector 0 0 0))
  (test (int-vector-ref v4 0) (int-vector 0))
  )
(let ((fv (int-vector 0 1)))
  (test (int-vector-ref fv 0 0) 'error)
  (test (vector-ref fv 0 0) 'error)
  (test (fv 0 0) 'error))
(let ((fv #i2d((1 2) (3 4))))
  (test (int-vector-ref fv 0 0 0) 'error)
  (test (int-vector-ref fv 0 0 0 0) 'error)
  (test (vector-ref fv 0 0 0) 'error)
  (test (fv 0 0 0) 'error))
(let ()
  (define (hi) (let ((v2 (make-int-vector '(2 3)))) (int-vector-set! v2 1 12) v2))
  (test (hi) 'error))

(let ()
  (define (f1)
    (let ((x (float-vector 0.0)))
      (set! (x 0) (complex 1 2))))
  (test (let ((x (float-vector 0.0))) (set! (x 0) 1+i)) 'error)
  (test (f1) 'error)

  (define (f2)
    (let ((x (int-vector 0)))
      (int-vector-set! x 0 (complex 1 2))))
  (test (let ((x (int-vector 0))) (set! (x 0) 0+i)) 'error)
  (test (f2) 'error)

  (define (f3)
    (let ((x (float-vector 0.0)))
      (float-vector-set! x 0 (complex 1 2))))
  (test (f3) 'error)

  (define (f4)
    (let ((x (int-vector 0)))
      (set! (x 0) (complex 1 2))))
  (test (f4) 'error))

(when with-bignums
  (test (int-vector 1 (bignum "2")) #i(1 2))
  (test (float-vector 1.0 (bignum "2.0")) #r(1.0 2.0)))

(let ((iv (make-int-vector 10))
      (ten 10))
  (define (ivf)
    (do ((i 0 (+ i 1)))
	((= i ten))
      (int-vector-set! iv i 1)))
  (ivf)
  (test iv #i(1 1 1 1 1 1 1 1 1 1))
  (define (izf)
    (do ((i 3 (+ i 1)))
	((= i 8))
      (int-vector-set! iv i 0)))
  (izf)
  (test iv #i(1 1 1 0 0 0 0 0 1 1)))

(let ()
  (define (f) ; opt_i_7pii_sff
    (let ((iv (make-int-vector '(2 3) 1))
	  (sum 0))
      (do ((i 0 (+ i 1)))
	  ((= i 2) sum)
        (set! sum (+ sum (iv (- (+ i 1) 1) (+ i 1)))))))
  (test (f) 2)

  (define (g) ; opt_i_7pii_sff
    (let ((iv (make-byte-vector '(2 3) 1))
  	(sum 0))
      (do ((i 0 (+ i 1)))
	  ((= i 2) sum)
        (set! sum (+ sum (iv (- (+ i 1) 1) (+ i 1)))))))
  (test (g) 2))

(let ((iv (make-int-vector (list 2 3) 1)) ; optimizer bug
      (c 3))
  (define (f1) (let ((v (vector #f))) (do ((i 0 (+ i 1))) ((= i 1) (vector-ref v 0)) (vector-set! v 0 (int-vector-set! iv 0 0 c)))))
  (define (f2) (let ((v (vector #f))) (do ((i 0 (+ i 1))) ((= i 1) (vector-ref v 0)) (vector-set! v 0 (int-vector-set! iv 0 0 3)))))
  (test (f1) 3)
  (test (f2) 3))

(let ()
  (define fvref float-vector-ref)
  (define ivref int-vector-ref)
  (define bvref byte-vector-ref)
  (define vref vector-ref)
  (test (let ((a7 (subvector #i2d((1 2) (3 4)) 1 3 '(2 1)))) (define (func) (do ((x 0) (i 0 (+ i 1))) ((= i 1) x) (set! x (ivref a7 0)))) (func)) #i(2))
  (test (let ((a7 (subvector #u2d((1 2) (3 4)) 1 3 '(2 1)))) (define (func) (do ((x 0) (i 0 (+ i 1))) ((= i 1) x) (set! x (bvref a7 0)))) (func)) #u(2))
  (test (let ((a7 (subvector #r2d((1 2) (3 4)) 1 3 '(2 1)))) (define (func) (do ((x 0) (i 0 (+ i 1))) ((= i 1) x) (set! x (fvref a7 0)))) (func)) #r(2))
  (test (let ((a7 (subvector #2d((1 2) (3 4)) 1 3 '(2 1)))) (define (func) (do ((x 0) (i 0 (+ i 1))) ((= i 1) x) (set! x (vref a7 0)))) (func)) #(2)))

  (when full-s7test
    (define (iota-iv len)
      (let ((b (make-int-vector len)))
        (do ((i 0 (+ i 1)))
            ((= i len) b)
          (int-vector-set! b i i))))
    (do ((i 0 (+ i 1)))
        ((= i 512))
      (let* ((b (iota-iv i))
             (b1 (reverse b))
             (b2 (reverse! b)))
        (unless (equal? b1 b2)
          (format *stderr* "reverse iota-iv ~D: ~A~%    ~A~%" i b1 b2)))))

  (when full-s7test
    (define (iota-fv len)
      (let ((b (make-float-vector len)))
        (do ((i 0 (+ i 1))
             (x 0.0 (+ x 1.0)))
            ((= i len) b)
          (float-vector-set! b i x))))
    (do ((i 0 (+ i 1)))
        ((= i 512))
      (let* ((b (iota-fv i))
             (b1 (reverse b))
             (b2 (reverse! b)))
        (unless (equal? b1 b2)
          (format *stderr* "reverse iota-fv ~D: ~A~%    ~A~%" i b1 b2)))))

  (when full-s7test
    (define (iota-v len)
      (let ((b (make-vector len)))
        (do ((i 0 (+ i 1)))
            ((= i len) b)
          (vector-set! b i i))))
    (do ((i 0 (+ i 1)))
        ((= i 512))
      (let* ((b (iota-v i))
             (b1 (reverse b))
             (b2 (reverse! b)))
        (unless (equal? b1 b2)
          (format *stderr* "reverse iota-v ~D: ~A~%    ~A~%" i b1 b2)))))

  (when full-s7test
    (define (iota-bv len)
      (let ((b (make-byte-vector len)))
        (do ((i 0 (+ i 1)))
            ((= i len) b)
          (byte-vector-set! b i (modulo i 256)))))
    (do ((i 0 (+ i 1)))
        ((= i 512))
      (let* ((b (iota-bv i))
             (b1 (reverse b))
             (b2 (reverse! b)))
        (unless (equal? b1 b2)
          (format *stderr* "reverse iota-bv ~D: ~A~%    ~A~%" i b1 b2)))))


;;; --------------------------------------------------------------------------------
;;; vector

(test (vector 1 2 3) #(1 2 3))
(test (vector 1 '(2) 3) #(1 (2) 3))
(test (vector) #())
(test (vector (vector (vector))) #(#(#())))
(test (vector (vector) (vector) (vector)) #(#() #() #()))
(test (vector (list)) #(()))
(test #(1 #\a "hi" hi) (vector 1 #\a "hi" 'hi))
(test (let ((v (make-vector 4 "hi")))
	(vector-set! v 0 1)
	(vector-set! v 1 #\a)
	(vector-set! v 3 'hi)
	v)
      #(1 #\a "hi" hi))
(let ((x 34))
  (test (vector x 'x) #(34 x)))

(for-each
 (lambda (arg)
   (test (vector-ref (vector arg) 0) arg))
 (list #\a 1 () (list 1) '(1 . 2) #f "hi" 'a-symbol abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand (log 0)
       3.14 3/4 1.0+1.0i #t (vector 1 2 3) (lambda (a) (+ a 1))))
(test (vector 1 . 2) 'error)
(test (apply vector (cons 1 2)) 'error)

(when with-bignums
  (test (append #(922337203685477580123123) #(2)) (vector 922337203685477580123123 2)))


;;; --------------------------------------------------------------------------------
;;; vector->list
;;; list->vector

(test (vector->list #(0)) (list 0))
(test (vector->list (vector)) ())
(test (vector->list #(a b c)) '(a b c))
(test (vector->list #(#(0) #(1))) '(#(0) #(1)))
(test (vector? (list-ref (let ((v (vector 1 2))) (vector-set! v 1 v) (vector->list v)) 1)) #t)
(test (vector->list #i(1 2)) '(1 2))

(test (list->vector ()) #())
(test (list->vector '(a b c)) #(a b c))
(test (list->vector (list (list 1 2) (list 3 4))) #((1 2) (3 4)))
(test (list->vector ''foo) #(quote foo))
(test (list->vector (list)) #())
(test (list->vector (list 1)) #(1))
(test (list->vector (list (list))) #(()))
(test (list->vector (list 1 #\a "hi" 'hi)) #(1 #\a "hi" hi))
(test (list->vector ''1) #(quote 1))
(test (list->vector '''1) #(quote '1))

(for-each
 (lambda (arg)
   (if (proper-list? arg)
       (test (vector->list (list->vector arg)) arg)))
 lists)
(set! lists ())

(test (list->vector (vector->list (vector))) #())
(test (list->vector (vector->list (vector 1))) #(1))
(test (vector->list (list->vector (list))) ())
(test (vector->list (list->vector (list 1))) '(1))

(test (reinvert 12 vector->list list->vector #(1 2 3)) #(1 2 3))

(test (vector->list) 'error)
(test (list->vector) 'error)
(test (vector->list #(1) #(2)) 'error)
(test (list->vector '(1) '(2)) 'error)

(for-each
 (lambda (arg)
   (test (vector->list arg) 'error))
 (list #\a 1 () (list 1) '(1 . 2) #f 'a-symbol "hi" abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (let ((x (cons #\a #\b))) (set-cdr! x x) (list->vector x)) 'error)
(test (list->vector (cons 1 2)) 'error)
(test (list->vector '(1 2 . 3)) 'error)
(test (let ((lst (list #\a #\b))) (set! (cdr (cdr lst)) lst) (list->vector lst)) 'error)
(test (let ((lst (list #\a #\b))) (set! (cdr (cdr lst)) lst) (apply vector lst)) 'error)

(for-each
 (lambda (arg)
   (test (list->vector arg) 'error))
 (list "hi" #\a 1 '(1 . 2) (cons #\a #\b) 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (vector->list #(1 2 3 4) 0) '(1 2 3 4))
(test (vector->list #(1 2 3 4) 2) '(3 4))
(test (vector->list #(1 2 3 4) 0 4) '(1 2 3 4))
(test (vector->list #(1 2 3 4) 4 4) ())
(test (vector->list #(1 2 3 4) 1 2) '(2))

(test (vector->list #(1 2 3 4) -1 4) 'error)
(test (vector->list #(1 2 3 4) 1 0) 'error)
(test (vector->list #(1 2 3 4) 5) 'error)
(test (vector->list #(1 2 3 4) 1 5) 'error)
(test (vector->list #(1 2 3 4) 1 2 3) 'error)
(unless pure-s7 (test (vector->list #(1 2 3 4) 1 #f) 'error))
(test (vector->list #(1 2 3 4) #f) 'error)
(test (vector->list #(1 2 3 4) #f 1) 'error)

(test (vector->list #() 0 10) 'error)
(test (vector->list #(1) 0 2) 'error)
(test (vector->list #() 0 0) ())
(test (vector->list #(1) 1) ())
(test (vector->list #(1) 0) '(1))
(test (vector->list #() #\null) 'error)
(test (vector->list #() 0 #\null) 'error)
(test (vector->list #() -1) 'error)
(test (vector->list #(1) -1) 'error)
(test (vector->list #(1) 0 -1) 'error)
(test (vector->list #(1) -2 -1) 'error)
(test (vector->list #(1) most-negative-fixnum) 'error)
(test (vector->list #(1) 2) 'error)

(test (vector->list (make-int-vector 3)) '(0 0 0))
(test (vector->list (make-float-vector 3)) '(0.0 0.0 0.0))

(for-each
 (lambda (arg)
   (test (vector->list #(0 1 2 3 4 5) arg) 'error)
   (test (vector->list #(0 1 2 3 4 5) 1 arg) 'error))
 (list #\a "hi" () (list 1) '(1 . 2) 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(let () ; vector_to_list_p_p
  (define (f)
    (let ((v (vector 1 2 3)))
      (do ((i 0 (+ i 1)))
	  ((= i 1) v)
        (vector-set! v 0 (vector->list #()))
        (vector-set! v 1 (vector->list #(4 5 6))))))
  (test (f) #(() (4 5 6) 3)))

(when full-s7test ; GC checks
  (let ()
    (define (func) (map values (vector->list (make-int-vector '(4 128) 100000)))) ; check GC protection
    (do ((i 0 (+ i 1))) ((= i 1000)) (func)))
  (let ()
    (define (func)
      (with-output-to-string
        (lambda ()
          (do ((i 0 (+ i 1)))
	      ((= i 1))
	    (display (make-float-vector '(128 3) (/ pi 2)))))))
    (do ((i 0 (+ i 1)))
        ((= i 10000))
      (func)))
  (let () ; GC protection in fx_cons_ac
    (define (f) (do ((i 0 (+ i 1))) ((= i 10000))
      (let () (define (func) (let () (apply values (cons (vector 0 1 2 3 (make-list 256 1)) ())))) (func) (func))))
    (f))
  (let () ; GC protection in list_p_p
    (define (f) (do ((i 0 (+ i 1))) ((= i 10000))
      (let () (define (func) (let () (apply values (list (vector -inf.0 case (symbol "(\")") 0+i (make-list 256 1)))))) (func) (func))))
    (f))
  (let () ; GC protection in list_ppp_p
    (define (f) (do ((i 0 (+ i 1))) ((= i 10000))
      (let () (define (func) (let () (apply values (list 1 (vector -inf.0 case (symbol "(\")") 0+i (make-list 256 1)) 2)))) (func) (func))))
    (f))
  (let () ; fx_cons_aa
    (define (f) (do ((i 0 (+ i 1))) ((= i 10000))
      (let () (define (func) (let () (apply values (cons (vector 0 1 2 3 4) (make-list 256))))) (func) (func))))
    (f))
  (let ()
    (define (f) ; fx_cons_as
      (define (func) (let ((L (list 32))) (apply values (cons (vector 0 1 2 3 (make-list 256 1)) L))))
      (do ((i 0 (+ i 1))) ((= i 10000)) (func) (func)))
    (f))
  (let ()
    (define (f) (do ((i 0 (+ i 1))) ((= i 10000))
      (let () (define (func) (let () (apply values (cons (cons (vector -inf.0 case (symbol "(\")") 0+i (make-list 256 1)) ()) ())))) (func) (func))))
    (f))
  (let ()
    (define (func x) (do ((j 0 (+ j 1))) ((= j 1)) (do ((i 0 (+ i 1))) ((= i 10000)) (list x (make-int-vector '(2 3) 1) (make-list 256 1)))))
    (func 2)) ; opt_p_ppp_sff
  (let () ; make_closure_unchecked bug
    (define (func)
      (do ((j 0 (+ j 1))) ((= j 100))
        (do ((i 0 (+ i 1))) ((= i 1000))
	  (char? (lambda (a) (values a (+ a 1)))))))
    (func))

   (let () ; opt_p_pp_ff
     (define (func)
       (do ((j 0 (+ j 1)))
	   ((= j 1))
	 (do ((i 0 (+ i 1)))
	     ((= i 10000))
	   (let ((x #f))
	     (sort! (vector 3 2 4 5 1)
		    (lambda (a b)
		      (let-temporarily ((x (list (logand) (list-ref (make-list 512) (logior)))))
			(> a b))))))))
     (func))

   (let ()
     (define* (sym5 a :rest b) (cons a (copy b)))
     (define (f1) (do ((i 0 (+ i 1))) ((= i 10000)) (sym5 '(()) (string #\c #\null #\b) (make-list 512)))) ; 1M is a better limit
     (f1))

   (test (let () (define (func) (do ((j 0 (+ j 1))) ((= j 1)) (do ((i 0 (+ i 1))) ((= i 10000)) (write (make-float-vector '(128 3) pi) (open-output-string))))) (func)) #t)
   ;; this doesn't hit the GC problem in this context, but maybe it's useful -- opt_p_pp_ff

   (test (let () (define (func) (do ((i 0 (+ i 1))) ((= i 10000)) (#_vector (apply-values (make-list 512)) #f))) (func) (func)) #t) ; pair_append

  (let () ; g_sort float_vector (data is the free cell, not vec)
    (define (f)
      (let ((x #f) (i 0))
        (do ((j 0 (+ j 1)))
	    ((= j 1))
 	  (do ((i 0 (+ i 1)))
	      ((= i 100))
            (apply values (sort! (make-float-vector '(128 3) pi) (lambda a (copy a))) ())))))
    (f))

  (do ((i 0 (+ i 1)))
      ((= i 100))
    (let () (define (func) (let ((x #f) (i 0)) (do ((j 0 (+ j 1))) ((= j 1)) (do ((i 0 (+ i 1))) ((= i 100)) (apply values (inlet ) (make-list 512)  ()))))) (func) (func))
    (let () (define (func) (let ((x #f) (i 0)) (do ((i 0 (+ i 1))) ((= i 100)) (apply values (#_open-output-string )    (make-list 512) #\\\\7  ())))) (func) (func)))
  (let ((H_4 (make-hash-table 8 (let ((eqf (lambda (a b) (equal? a b)))
				      (mapf (lambda (a) (hash-code a))))
                                  (cons eqf mapf)))))
    (do ((i 0 (+ i 1)))
        ((= i 10))
      (let () (define (func) (let ((x #f) (i 0)) (do ((i 0 (+ i 1))) ((= i 100)) (copy (hash-table +nan.0 1) H_4)))) (func) (func))))

  (test (let () 
          (define* (sym6 a b :rest c) (list a b (copy c)))
          (define (func) (do ((i 0 (+ i 1))) ((= i 100)) (sym6 (values 1.0 letrec* (cons i i) (log 1.0) (log 2.0) (log 3.0) (log 4.0) (log 5.0) (make-list 512)))))
          (func))
        #t)
)



;;; --------------------------------------------------------------------------------
;;; vector-length

(test (vector-length (vector)) 0)
(test (vector-length (vector 1)) 1)
(test (vector-length (make-vector 128)) 128)
(test (vector-length #(a b c d e f)) 6)
(test (vector-length #()) 0)
(test (vector-length (vector #\a (list 1 2) (vector 1 2))) 3)
(test (vector-length #(#(#(hi)) #(#(hi)) #(#(hi)))) 3)
(test (vector-length (vector 1 2 3 4)) 4)
(test (vector-length (let ((v (vector 1 2))) (vector-set! v 1 v) v)) 2)
(test (vector-length (let ((v (vector 1 2))) (vector-set! v 1 v) (vector-ref v 1))) 2)
(test (vector-length (make-int-vector 3 0)) 3)
(if (not with-bignums) (test (vector-length (make-float-vector 3 pi)) 3))
(if (not with-bignums) (test (vector-length (make-float-vector '(2 3) pi)) 6))
(test (vector-length #r(1 2)) 2)

(test (vector-length) 'error)
(test (vector-length #(1) #(2)) 'error)

(for-each
 (lambda (arg)
   (test (vector-length arg) 'error))
 (list "hi" #\a 1 () '(1 . 2) (cons #\a #\b) #f 'a-symbol abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))


;;; --------------------------------------------------------------------------------
;;; vector-rank
;;; vector-dimension
;;; vector-dimensions

(test (vector-rank #()) 1)
(test (vector-dimensions #()) '(0))
(test (vector-dimension #() 0) 0)
(test (vector-dimension #() 0 0) 'error)
(test (vector-dimension #() 1) 'error)
(test (vector-dimension #()) 'error)
(test (vector-rank #(1 2)) 1)
(test (vector-dimensions #(1 2)) '(2))
(test (vector-dimension #(1 2) 0) 2)
(test (vector-dimension #(1 2) -1) 'error)
(test (vector-dimension #(1 2) 1) 'error)
(test (vector-rank (make-vector '(2 3))) 2)
(test (vector-dimensions (make-vector '(2 3))) '(2 3))
(test (vector-dimension (make-vector '(2 3)) 0) 2)
(test (vector-dimension (make-vector '(2 3)) 1) 3)
(test (vector-rank (make-vector '(2 1 3))) 3)
(test (vector-dimensions (make-vector '(2 1 3))) '(2 1 3))
(test (vector-rank (make-vector '(2 0 3))) 3)
(test (vector-dimensions (make-vector '(2 0 3))) '(2 0 3))
(test (vector-dimension (make-vector '(2 0 3)) 1) 0)
(test (vector-dimension (make-vector '(2 0 3)) 2) 3)
(test (vector-rank (make-vector 0)) 1)
(test (vector-dimensions (make-vector 0)) '(0))
(test (vector-dimensions _c_obj_) '(16))
(let ((v (make-vector 24)))
  (test (vector-rank (subvector v 0 24 '(3 8))) 2)
  (test (vector-rank (subvector v 2 10 '(2 2 2))) 3))

(for-each
 (lambda (arg)
   (test (vector-rank arg) 'error)
   (test (vector-dimension arg 0) 'error)
   (test (vector-dimensions arg) 'error))
 (list "hi" #\a 1 () '(1 . 2) (cons #\a #\b) #f 'a-symbol abs _ht_ _undef_ _null_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (vector-rank (make-int-vector '(2 3))) 2)
(test (vector-dimensions (make-int-vector '(2 3))) '(2 3))
(test (vector-dimension (make-int-vector '(2 3)) 1) 3)
(test (vector-rank (make-byte-vector '(2 3))) 2)
(test (vector-dimensions (make-byte-vector '(2 3))) '(2 3))
(test (vector-dimension (make-byte-vector '(2 3)) 0) 2)
(test (vector-dimension (make-byte-vector '(2 3)) 1) 3)
(test (vector-rank (make-float-vector '(2 3))) 2)
(test (vector-dimensions (make-float-vector '(2 3))) '(2 3))

(test (vector-dimensions) 'error)
(test (vector-dimensions #() #()) 'error)
(test (vector-dimensions (vector)) '(0))
(test (vector-dimensions (vector 0)) '(1))
(test (vector-dimensions (vector-ref #2d((1 2 3) (3 4 5)) 0)) '(3))
(test (vector-dimensions (vector-ref #3d(((1 2 3) (3 4 5)) ((5 6 1) (7 8 2))) 0)) '(2 3))
(test (vector-dimensions (vector-ref #3d(((1 2 3) (3 4 5)) ((5 6 1) (7 8 2))) 0 1)) '(3))
(test (set! (vector-dimensions #(1 2)) 1) 'error)
(test (let ((v #(1 2 3))) (set! (car (vector-dimensions v)) 0) v) #(1 2 3))
(test (hash-table 1 (vector-dimensions (block))) (hash-table 1 '(0)))


;;; --------------------------------------------------------------------------------
;;; vector-typer

(let ((v #(1 2)))
  (for-each
   (lambda (arg)
     (test (vector-typer arg) 'error)
     (test (set! (vector-typer arg) integer?) 'error)
     (test (set! (vector-typer v) arg) 'error))
   (list "hi" #\a 1 () '(1 . 2) (cons #\a #\b) 'a-symbol abs _ht_ _undef_ _null_ quasiquote macroexpand 1/0 (log 0)
         3.14 3/4 1.0+1.0i :hi (if #f #f) (lambda (a) (+ a 1)))))

(let ()
  (define (typer x) (symbol? x))
  (define v (make-vector 9 'a typer))
  (define v1 (make-vector 3 1))

  (test (equal? typer (vector-typer v)) #t)
  (test (vector-typer (make-int-vector 3)) integer?)
  (test (vector-typer (make-vector 9 1 integer?)) integer?)

  (test (vector-typer v1) #f)
  (test (set! (vector-typer v1) integer?) integer?)
  (test (vector-typer v1) integer?)
  (test (set! (v1 0) pi) 'error) ; vector-set! third argument 3.141592653589793, is a real, but the vector's element type checker, integer?, rejects it

  (test (set! (vector-typer (int-vector 1 2 3)) integer?) integer?)
  (test (catch #t (lambda () (set! (vector-typer (int-vector 1 2 3)) (lambda (x) (float? x)))) (lambda (t i) (apply format #f i)))
        "vector-typer can't set #i(1 2 3) typer to #<lambda (x)>")
  (test (catch #t (lambda () (set! (vector-typer (vector 1 2 3)) (lambda (x) (float? x)))) (lambda (t i) (apply format #f i)))
        "vector-typer second argument, #<lambda (x)>, is a function but should be a named function")
  (test (catch #t (lambda () (set! (vector-typer (vector 1)) (macro (mac a) `(+ 1 ,a)))) (lambda (t i) (apply format #f i)))
        "vector-typer second argument, #<macro (mac a)>, is a macro but should be a built-in procedure, a closure, #f or #t")
  (test (catch #t (lambda () (set! (vector-typer (vector 1)) (lambda (a) #f))) (lambda (t i) (apply format #f i)))
        "vector-typer second argument, #<lambda (a)>, is a function but should be a named function")
  (test (catch #t (lambda () (set! (vector-typer (vector 1 2 3)) abs)) (lambda (t i) (apply format #f i)))
        "vector-typer second argument, abs, is a c-function but should be a boolean procedure")
  (test (set! (vector-typer (vector 1 2 3)) typer) typer)  ; current contents are not checked
  )

(let ()
  (define (vector-type v)
    (let* ((data (object->let v))
	   (sig (and (defined? 'signature data #t)
		     (let-ref data 'signature))))
      (and (pair? sig)
	   (car sig))))

  (define (vtyper v) #t)
  (let ((V1 (make-vector 3 1 vtyper)))
    (set! (V1 0) 0)
    (test (object->string V1 :readable) "(let ((<v> (vector 0 1 1))) (set! (vector-typer <v>) vtyper) <v>)")
    (test (vector-type V1) 'vtyper))

  (test (vector-type (make-vector 3 'a symbol?)) 'symbol?)
  (test (vector-type (vector 0 1)) #f))

(let ()
  (define (vtyper v) #t)
  (let ((V1 (make-vector 1024 #f vtyper)))
    (test (object->string V1 :readable) "(make-vector 1024 #f vtyper)")))

(test (object->string (make-int-vector 1024 1) :readable) "(make-int-vector 1024 1)")
(test (let ((v (make-vector 3 'a symbol?))) (object->string v :readable)) "(let ((<v> (vector 'a 'a 'a))) (set! (vector-typer <v>) symbol?) <v>)")
(let ((v (immutable! (make-vector '(2 3) 'a symbol?))))
  (test (object->string v :readable) "(let ((<v> (subvector (immutable! (vector 'a 'a 'a 'a 'a 'a)) 0 6 '(2 3)))) (set! (vector-typer <v>) symbol?) <v>)"))


(let ()
  ;; typers expect s7 to raise error
  (define (vtyper val) (or (integer? val) (boolean? val)))
  (define v (make-vector 3 0 vtyper))
  (set! (v 0) #f)
  (test v #(#f 0 0))
  (set! (v 1) 21)
  (test v #(#f 21 0))
  (test (catch #t (lambda () (set! (v 2) "asdf")) (lambda (typ info) (apply format #f info)))
	"vector-set! third argument \"asdf\", is a string, but the vector's element type checker, vtyper, rejects it")

  (define h (make-hash-table 3 #f (cons vtyper vtyper)))
  (set! (h 0) #t)
  (test h (hash-table 0 #t))
  (set! (h 1) 21)
  (test h (hash-table 0 #t 1 21))
  (test (catch #t (lambda () (set! (h 2) "asdf")) (lambda (typ info) (apply format #f info)))
	"hash-table-set! third argument \"asdf\", is a string, but the hash-table's value type checker, vtyper, rejects it")
  (test (catch #t (lambda () (set! (h "asdf") 2)) (lambda (typ info) (apply format #f info)))
	"hash-table-set! second argument \"asdf\", is a string, but the hash-table's key type checker, vtyper, rejects it")
  (set! (h 0) #f)
  (test h (hash-table 1 21))

  ;; symbol setters handle errors themselves
  (define int #t)
  (set! (setter 'int) boolean?)
  (set! int #f)
  (test int #f)
  (test (catch #t (lambda () (set! int 21)) (lambda (t info) (apply format #f info))) "set! int, 21 is an integer but should be boolean")
  (set! (setter 'int) (lambda (s v)
			(if (or (boolean? v) (integer? v))
			    v
			    (error 'wrong-type-arg "can't set! ~S to ~S" s v))))
  (set! int #f)
  (test int #f)
  (set! int 21)
  (test int 21)
  (test (catch #t (lambda () (set! int "asdf")) (lambda (typ info) (apply format #f info))) "can't set! int to \"asdf\"")

  ;; function setters also handle errors themselves
  (define f1 (dilambda (lambda () 3) (lambda (v) 32)))
  (test (f1) 3)
  (test (set! (f1) 12) 32)
  (set! (setter f1) #f)
  (test (catch #t (lambda () (set! (f1) 12)) (lambda (typ info) (apply format #f info))) "f1 (a function) does not have a setter: (set! (f1) 12)"))


;;; --------------------------------------------------------------------------------
;;; vector-ref

(test (vector-ref #(1 1 2 3 5 8 13 21) 5) 8)
(test (vector-ref #(1 1 2 3 5 8 13 21) (let ((i (round (* 2 (acos -1))))) (if (inexact? i) (inexact->exact i)  i))) 13)
(test (let ((v (make-vector 1 0))) (vector-ref v 0)) 0)
(test (let ((v (vector 1 (list 2) (make-vector 3 #\a)))) (vector-ref v 1)) (list 2))
(test (let ((v (vector 1 (list 2) (make-vector 3 #\a)))) (vector-ref v 2)) #(#\a #\a #\a))
(test (let ((v (vector 1 (list 2) (make-vector 3 #\a)))) (vector-ref (vector-ref v 2) 1)) #\a)
(test (vector-ref #(a b c) 1) 'b)
(test (vector-ref #(()) 0) ())
(test (vector-ref #(#()) 0) #())
(test (vector-ref (vector-ref (vector-ref #(1 (2) #(3 (4) #(5))) 2) 2) 0) 5)
(test (let ((v (vector 1 2))) (vector-set! v 1 v) (eq? (vector-ref v 1) v)) #t)
(test (let ((v (make-int-vector 3))) (vector-ref v 1)) 0)
(test (let ((v (make-vector 3 0))) (vector-ref v 1)) 0)
(test (let ((v (make-float-vector 3 1.0))) (vector-ref v 1)) 1.0)
(test (let ((v (make-int-vector 6 0))) (vector-set! v 3 32) (let ((v1 (subvector v 0 6 '(2 3)))) (vector-ref v1 1 0))) 32)

(test (vector-ref) 'error)
(test (vector-ref #(1)) 'error)
(test (vector-ref #(1) 0 0) 'error)
(test (vector-ref () 0) 'error)
(test (vector-ref #(1) 1) 'error)
(test (vector-ref #2d((1 2) (3 4)) 3 0) 'error)
(test (vector-ref #2d((1 2) (3 4)) 0 3) 'error)
(test (vector-ref #2d((1 2) (3 4)) 0 0 0) 'error)

(test (let ((v (make-vector 1 0))) (vector-ref v 1)) 'error)
(test (let ((v (make-vector 1 0))) (vector-ref v -1)) 'error)
(test (let ((v (vector 1 (list 2) (make-vector 3 #\a)))) (vector-ref (vector-ref v 2) 3)) 'error)
(test (let ((v (vector 1 (list 2) (make-vector 3 #\a)))) (vector-ref (vector-ref v 3) 0)) 'error)
(test (vector-ref (vector) 0) 'error)
(test (vector-ref #() 0) 'error)
(test (vector-ref #() -1) 'error)
(test (vector-ref #() 1) 'error)
(test (vector-ref #(1 2 3) (floor .1)) 1)
(test (vector-ref #(1 2 3) (floor 0+0i)) 1)
(test (vector-ref #10d((((((((((0 1)))))))))) 0 0 0 0 0 0 0 0 0 1) 1)

(test (#(1 2) 1) 2)
(test (#(1 2) 1 2) 'error)
(test ((#("hi" "ho") 0) 1) #\i)
(test (((vector (list 1 2) (cons 3 4)) 0) 1) 2)
(test ((#(#(1 2) #(3 4)) 0) 1) 2)
(test ((((vector (vector (vector 1 2) 0) 0) 0) 0) 0) 1)
(test ((((list (list (list 1 2) 0) 0) 0) 0) 0) 1)
(test ((((list (list (list 1 2) 0) 0) 0) 0) ((((vector (vector (vector 1 2) 0) 0) 0) 0) 0)) 2)
(test (#(1 2) -1) 'error)
(test (#()) 'error)
(test (#(1)) 'error)
(test (#2d((1 2) (3 4))) 'error)
(test (apply (make-vector '(1 2))) 'error)

(test (type-of (eval-string "#2/3d(1 2)")) 'error)
(test (type-of (eval-string "#2.1d(1 2)")) 'error)
(test (eval-string "#(1 2 . 3)") 'error)
(test (#(#(#(#t))) 0 0 0) #t)
(test (let ((v (make-vector 3 0 #t))) (v 0 0)) 'error)
(test (let ((v (make-int-vector '(2 2)))) (v 0 0 0)) 'error)
(test (let ((v (make-float-vector 3))) (vector-ref v 0 0)) 'error)
(test (let ((v (make-vector '(2 2) 0.0 #t))) (vector-ref v 0 0 0)) 'error)
(test (let ((v (make-vector 3 0))) (v 0 0)) 'error)
(test (let ((v (make-vector '(2 2) 0))) (v 0 0 0)) 'error)

(let ((v #(1 2 3)))
  (for-each
   (lambda (arg)
     ; (format *stderr* "~A~%" arg)
     (test (vector-ref arg 0) 'error)
     (test (v arg) 'error)
     (test (v arg 0) 'error)
     (test (vector-ref v arg) 'error)
     (test (vector-ref v arg 0) 'error)
     (test (vector-ref #2d((1 2) (3 4)) 0 arg) 'error))
   (list "hi" () #() #\a -1 '(1 . 2) (cons #\a #\b) #f 'a-symbol abs _ht_ _undef_ _null_ quasiquote macroexpand 1/0 (log 0)
	 3.14 3/4 1.0+1.0i #t (lambda (a) (+ a 1)) (make-hash-table))))


(test (vector-ref #(#(1 2 3) #(4 5 6)) 1) #(4 5 6))
(test (vector-ref #(#(1 2 3) #(4 5 6)) 1 2) 6)
(test (vector-ref #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1) #(#(7 8 9) #(10 11 12)))
(test (vector-ref #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1 0) #(7 8 9))
(test (vector-ref #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1 0 2) 9)
(test (vector-ref #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1 0 3) 'error)
(test (vector-ref #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1 0 2 0) 'error)

(test (#(#(1 2 3) #(4 5 6)) 1) #(4 5 6))
(test (#(#(1 2 3) #(4 5 6)) 1 2) 6)
(test (#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1) #(#(7 8 9) #(10 11 12)))
(test (#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1 0) #(7 8 9))
(test (#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1 0 2) 9)
(test (#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1 0 3) 'error)
(test (#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1 0 2 0) 'error)

(test (let ((L #(#(1 2 3) #(4 5 6)))) (L 1)) #(4 5 6))
(test (let ((L #(#(1 2 3) #(4 5 6)))) (L 1 2)) 6)
(test (let ((L #(#(1 2 3) #(4 5 6)))) (L 1 2 3)) 'error)
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (L 1)) #(#(7 8 9) #(10 11 12)))
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (L 1 0)) #(7 8 9))
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (L 1 0 2)) 9)
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (L 1 0 2 3)) 'error)

(test (let ((L #(#(1 2 3) #(4 5 6)))) ((L 1) 2)) 6)
(test (let ((L #(#(1 2 3) #(4 5 6)))) (((L 1) 2) 3)) 'error)
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) ((L 1) 0)) #(7 8 9))
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (((L 1) 0) 2)) 9)
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) ((L 1 0) 2)) 9)
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) ((L 1) 0 2)) 9)
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) ((((L 1) 0) 2) 3)) 'error)

(test (let ((L #(#(1 2 3) #(4 5 6)))) (vector-ref (L 1) 2)) 6)
(test (let ((L #(#(1 2 3) #(4 5 6)))) (vector-ref ((L 1) 2) 3)) 'error)
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (vector-ref (L 1) 0)) #(7 8 9))
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) ((vector-ref (L 1) 0) 2)) 9)
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (vector-ref (((L 1) 0) 2) 3)) 'error)


(let ((zero 0)
      (one 1)
      (two 2)
      (three 3)
      (thirty-two 32))
  (test (vector-ref #(#(1 2 3) #(4 5 6)) one) #(4 5 6))
  (test (vector-ref #(#(1 2 3) #(4 5 6)) one two) 6)
  (test (vector-ref #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) one) #(#(7 8 9) #(10 11 12)))
  (test (vector-ref #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) one zero) #(7 8 9))
  (test (vector-ref #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) one zero two) 9)

  (test (#(#(1 2 3) #(4 5 6)) one) #(4 5 6))
  (test (#(#(1 2 3) #(4 5 6)) one two) 6)
  (test (#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) one) #(#(7 8 9) #(10 11 12)))
  (test (#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) one zero) #(7 8 9))
  (test (#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) one zero two) 9)

  (test (let ((L #(#(1 2 3) #(4 5 6)))) (L one)) #(4 5 6))
  (test (let ((L #(#(1 2 3) #(4 5 6)))) (L one two)) 6)
  (test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (L one)) #(#(7 8 9) #(10 11 12)))
  (test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (L one zero)) #(7 8 9))
  (test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (L one zero two)) 9)

  (test (let ((L #(#(1 2 3) #(4 5 6)))) ((L one) two)) 6)
  (test (let ((L #(#(1 2 3) #(4 5 6)))) (((L one) two) 3)) 'error)
  (test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) ((L one) zero)) #(7 8 9))
  (test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (((L one) zero) two)) 9)
  (test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) ((L one zero) two)) 9)
  (test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) ((L one) zero two)) 9)

  (test (let ((L #(#(1 2 3) #(4 5 6)))) (vector-ref (L one) two)) 6)
  (test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (vector-ref (L one) zero)) #(7 8 9))
  (test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) ((vector-ref (L one) zero) two)) 9))

(test ((#(#(:hi) #\a (3)) (#("hi" 2) 1)) (#2d((#() ()) (0 #(0))) 1 ('(cons 0) 1))) 3)
(test (#(1 2 3) (#(1 2 3) 1)) 3)
(test ((#(#(1 2)) (#(1 0) 1)) (#(3 2 1 0) 2)) 2)
(test (apply min (#(1 #\a (3)) (#(1 2) 1))) 3) ; i.e vector ref here 2 levels -- (#(1 2) 1) is 2 and (#(1 #\a (3)) 2) is (3)

;;; vector-ref optimizer checks
(define global_vector (vector 1 2 3))
(let ()
  (define (hi i) (vector-ref global_vector i))
  (test (hi 1) 2))
(let ()
  (define (hi i) (vector-ref global_vector (vector-ref global_vector i)))
  (test (hi 0) 2))

(test (let ((v #(0 1 2 3 4 5))) (define (f1) (v 4/3)) (f1)) 'error)
(test (let ((v "012345")) (define (f1) (v 4/3)) (f1)) 'error)
(test (let ((v (list 0 1 2 3 4 5))) (define (f1) (v 4/3)) (f1)) 'error)

(define-constant -a-global-vector- (vector 1 2 3))
(let ()
  (define (fg a) (vector-ref -a-global-vector- a))
  (test (fg 0) 1))

(let ()
  (define (f1)
    (let ((v (vector #f)) (X #2d((1 2) (3 4))))
      (do ((i 0 (+ i 1))) ((= i 1) v)
	(vector-set! v 0 (vector-ref X 1)))))
  (test (f1) #(#(3 4))))

(let ()
  (define (f1)
    (let ((v (vector #f)) (X #2d((1 2) (3 4))))
      (do ((i 0 (+ i 1))) ((= i 1) v)
	(vector-set! v 0 (vector-ref X 2)))))
  (test (f1) 'error))

(let ()
  (define (f1)
    (let ((I 0) (v (vector #f)) (X #2d((1 2) (3 4))))
      (do ((i 0 (+ i 1))) ((= i 1) v)
	(vector-set! v 0 (vector-ref X (+ I 1))))))
  (test (f1) #(#(3 4))))

(let ()
  (define (f1)
    (let ((I 1) (v (vector #f)) (X #2d((1 2) (3 4))))
      (do ((i 0 (+ i 1))) ((= i 1) v)
	(vector-set! v 0 (vector-ref X (+ I 1))))))
  (test (f1) 'error))

(set! global_vector #2d((1 2) (3 4)))
(let ()
  (define (f1)
    (let ((I 1) (v (vector #f)))
      (do ((i 0 (+ i 1))) ((= i 1) v)
	(vector-set! v 0 (vector-ref global_vector I)))))
  (test (f1) #(#(3 4))))

(let ()
  (define (f1)
    (let ((I 2) (v (vector #f)))
      (do ((i 0 (+ i 1))) ((= i 1) v)
	(vector-set! v 0 (vector-ref global_vector I)))))
  (test (f1) 'error))

(test (#("asdf") 0 1) #\s)
(test (#("asdf") 0 1 0) 'error)

(test (let () (define (px x) (vector-ref x 0)) (define (fpx x y) (+ (px x) (px y))) (define (tfpx) (let ((v1 #(1 2 3)) (v2 #(2 3 1))) (fpx v1 v2))) (tfpx)) 3) ; fx_c_ff
(test (let ((v #2d((3 2 1) (6 5 4))) (b1 123) (b2 0)) (define (f) (do ((i 0 (+ i 1))) ((= i 1)) (imag-part (v b1 b2)))) (f)) 'error) ; opt_p_pii_sss bug

(let ()
;; coverage tests (not otherwise hit in this file)
  (define (f1)
    (let ((v1 (list (list (list 1)) (list (list 2)) (list (list 3)) (list (list 4))))
	  (sum 0))
      (do ((i 0 (+ i 1)))
	  ((= i 4) sum)
	(set! sum (+ sum (v1 i 0 0))))))
  (test (f1) 10)

  (define (f2)
    (let ((v2 (make-vector '(10 2 2) 1.0))
	  (sum 0.0))
      (set! (v2 1 0 0) 2.0)
      (set! (v2 2 0 0) 3.0)
      (set! (v2 3 0 0) 4.0)
      (do ((i 0 (+ i 1)))
	  ((= i 4) sum)
	(set! sum (+ sum (v2 i 0 0))))))
  (test (f2) 10.0)

  (define (f3)
    (let ((v3 (make-float-vector '(10 2 2) 1.0))
	  (sum 0.0))
      (set! (v3 1 0 0) 2.0)
      (set! (v3 2 0 0) 3.0)
      (set! (v3 3 0 0) 4.0)
      (do ((i 0 (+ i 1)))
	  ((= i 4) sum)
	(set! sum (+ sum (v3 i 0 0))))))
  (test (f3) 10.0)

  (define (f4)
    (let ((v4 (make-int-vector '(10 2 2) 1))
	  (sum 0))
      (set! (v4 1 0 0) 2)
      (set! (v4 2 0 0) 3)
      (set! (v4 3 0 0) 4)
      (do ((i 0 (+ i 1)))
	  ((= i 4) sum)
	(set! sum (+ sum (v4 i 0 0))))))
  (test (f4) 10)

  (define (fi)
    (let ((v4 (make-int-vector (expt 2 9)))
	  (sum 0))
      (do ((i 0 (+ i 1)))
	  ((= i 9))
	(vector-set! v4 (expt 2 i) i))
      (let ((v4a (subvector v4 0 (length v4) (make-list 9 2))))
	(do ((i 1 (+ i 1)))
	    ((= i 2) sum)
	  (set! sum (+ sum
		       (v4a i 0 0 0  0 0 0 0  0)
		       (v4a 0 i 0 0  0 0 0 0  0)
		       (v4a 0 0 i 0  0 0 0 0  0)
		       (v4a 0 0 0 i  0 0 0 0  0)))))))
  (test (fi) 26)

  (define (f5)
    (let ((v5 (make-byte-vector '(10 2 2) 1))
	  (sum 0))
      (set! (v5 1 0 0) 2)
      (set! (v5 2 0 0) 3)
      (set! (v5 3 0 0) 4)
      (do ((i 0 (+ i 1)))
	  ((= i 4) sum)
	(set! sum (+ sum (v5 i 0 0))))))
  (test (f5) 10)

  (define (f6)
    (let ((v6 (hash-table 0 (list (list 1)) 1 (list (list 2)) 2 (list (list 3)) 3 (list (list 4))))
	  (sum 0))
      (do ((i 0 (+ i 1)))
	  ((= i 4) sum)
	(set! sum (+ sum (v6 i 0 0))))))
  (test (f6) 10))


;;; --------------------------------------------------------------------------------
;;; vector-set!

(test (let ((vec (vector 0 '(2 2 2 2) "Anna"))) (vector-set! vec 1 '("Sue" "Sue")) vec) #(0 ("Sue" "Sue") "Anna"))
(test (let ((v (vector 1 2 3))) (vector-set! v 1 32) v) #(1 32 3))
(let ((v (make-vector 8 #f)))
  (for-each
   (lambda (arg)
     (vector-set! v 1 arg)
     (test (vector-ref v 1) arg))
   (list #\a 1 () (list 1) '(1 . 2) #f "hi" 'a-symbol abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand (log 0)
	 3.14 3/4 1.0+1.0i #t (vector 1 2 3) (lambda (a) (+ a 1)))))
(test (let ((v (vector 1 2 3))) (vector-set! v 1 0) v) #(1 0 3))
(test (let ((v (vector #f))) (vector-set! v 0 (vector)) v) #(#()))
(test (let ((v (vector 1 (list 2) (vector 1 2 3)))) (vector-set! (vector-ref v 2) 0 21) v) #(1 (2) #(21 2 3)))
(test (let ((v (make-int-vector 3))) (vector-set! v 1 32) (vector->list v)) '(0 32 0))
(test (let ((v (make-int-vector 3 0))) (set! (v 1) 32) (vector->list v)) '(0 32 0))

(test (vector-set! (vector 1 2) 0 4) 4)
(test (vector-set!) 'error)
(test (vector-set! #(1)) 'error)
(test (vector-set! #(1) 0) 'error)
(test (vector-set! #(1) 0 0 1) 'error)
(test (vector-set! #(1) 0 0 1 2 3) 'error)
(test (vector-set! #(1) #(0) 1) 'error)
(test (vector-set! #(1 2) 0 2) 2)
(test (let ((x 2) (v (vector 1 2))) (vector-set! (let () (set! x 3) v) 1 23) (list x v)) '(3 #(1 23)))
(test (let ((v #(1 2))) (vector-set! v 0 32)) 32)
(test (let ((v #(1 2))) (set! (v 0) 32)) 32)
(test (let ((v #(1 2))) (set! (vector-ref v 0) 32)) 32)
(test (let ((v (make-vector '(2 3) 0))) (set! (v (values 0 1)) 23) v) #2d((0 23 0) (0 0 0)))

(for-each
 (lambda (arg)
   (test (vector-set! arg 0 0) 'error))
 (list "hi" () #\a -1 '(1 . 2) (cons #\a #\b) #f 'a-symbol abs _ht_ _undef_ _null_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t (lambda (a) (+ a 1)) (make-hash-table)))

(let ((v (vector 1 2 3)))
  (for-each
   (lambda (arg)
     (test (vector-set! v arg 0) 'error))
   (list "hi" () #() #\a -1 '(1 . 2) (cons #\a #\b) #f 'a-symbol abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
	 3.14 3/4 1.0+1.0i #t (make-vector 3) (lambda (a) (+ a 1)))))

(for-each
 (lambda (arg)
   (test (vector-set! arg 0 0) 'error))
 (list "hi" () #\a 1 '(1 . 2) (cons #\a #\b) #f 'a-symbol abs _ht_ _undef_ _null_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(let ((v #(#(0 1) #(2 3))))
  (vector-set! (vector-ref v 1) 1 4)
  (test (v 1 1) 4)
  (set! ((vector-ref v 1) 1) 5)
  (test (v 1 1) 5)
  (set! ((v 1) 1) 6)
  (test (v 1 1) 6)
  (vector-set! (v 1) 1 7)
  (test (v 1 1) 7)
  (set! (v 1 1) 8)
  (test (v 1 1) 8))

(let ((v (vector)))
  (test (vector-set! v 0 0) 'error)
  (test (vector-set! v 1 0) 'error)
  (test (vector-set! v -1 0) 'error))
(test (vector-set! #() 0 123) 'error)
(test (vector-set! #(1 2 3) 0 123) 123)
(test (let ((v #(1 2 3))) (set! (v 0) '(+ 1 2)) v) #((+ 1 2) 2 3))
(test (let ((v #(1 2 3))) (set! (v '(+ 1 1)) 2) v) 'error)
(test (let ((v #(1 2 3))) (set! (v (+ 1 1)) 2) v) #(1 2 2))

(test (let ((g (lambda () #(1 2 3)))) (vector-set! (g) 0 #\?) (g)) #(#\? 2 3))
(test (let ((g (lambda () '(1 . 2)))) (set-car! (g) 123) (g)) '(123 . 2))
(test (let ((g (lambda () '(1 2)))) (list-set! (g) 0 123) (g)) '(123 2))
(test (let ((g (lambda () (symbol->string 'hi)))) (string-set! (g) 1 #\a) (symbol->string 'hi)) "hi")

(test (let ((L #(#(1 2 3) #(4 5 6)))) (vector-set! L 1 32) L) #(#(1 2 3) 32))
(test (let ((L #(#(1 2 3) #(4 5 6)))) (vector-set! L 1 0 32) L) #(#(1 2 3) #(32 5 6)))
(test (let ((L #(#(1 2 3) #(4 5 6)))) (vector-set! L 1 0 2 32) L) 'error)
(test (let ((L #(#(1 2 3) #(4 5 6)))) (vector-set! L 1 3 32) L) 'error)
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (vector-set! L 1 32) L) #(#(#(1 2 3) #(4 5 6)) 32))
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (vector-set! L 1 0 32) L) #(#(#(1 2 3) #(4 5 6)) #(32 #(10 11 12))))
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (vector-set! L 1 0 2 32) L) #(#(#(1 2 3) #(4 5 6)) #(#(7 8 32) #(10 11 12))))
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (vector-set! L 1 0 2 1 32) L) 'error)
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (vector-set! L 1 4 2 32) L) 'error)

(test (let ((L #(#(1 2 3) #(4 5 6)))) (set! (L 1) 32) L) #(#(1 2 3) 32))
(test (let ((L #(#(1 2 3) #(4 5 6)))) (set! (L 1 0) 32) L) #(#(1 2 3) #(32 5 6)))
(test (let ((L #(#(1 2 3) #(4 5 6)))) (set! (L 1 0 2) 32) L) 'error)
(test (let ((L #(#(1 2 3) #(4 5 6)))) (set! (L 1 3) 32) L) 'error)
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! (L 1) 32) L) #(#(#(1 2 3) #(4 5 6)) 32))
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! (L 1 0) 32) L) #(#(#(1 2 3) #(4 5 6)) #(32 #(10 11 12))))
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! (L 1 0 2) 32) L) #(#(#(1 2 3) #(4 5 6)) #(#(7 8 32) #(10 11 12))))
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! (L 1 0 2 1) 32) L) 'error)
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! (L 1 4 2) 32) L) 'error)

(test (let ((L #(#(1 2 3) #(4 5 6)))) (set! ((L 1) 0) 32) L) #(#(1 2 3) #(32 5 6)))
(test (let ((L #(#(1 2 3) #(4 5 6)))) (set! (((L 1) 0) 2) 32) L) 'error)
(test (let ((L #(#(1 2 3) #(4 5 6)))) (set! ((L 1) 3) 32) L) 'error)
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! ((L 1) 0) 32) L) #(#(#(1 2 3) #(4 5 6)) #(32 #(10 11 12))))
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! (((L 1) 0) 2) 32) L) #(#(#(1 2 3) #(4 5 6)) #(#(7 8 32) #(10 11 12))))
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! ((((L 1) 0) 2) 1) 32) L) 'error)
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! (((L 1) 4) 2) 32) L) 'error)
(test (let ((L #(#(#(1 2 3))))) (set! ((L 0) 0 1) 32) L) #(#(#(1 32 3))))
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! ((L 1 0) 2) 32) L) #(#(#(1 2 3) #(4 5 6)) #(#(7 8 32) #(10 11 12))))

(test (let ((L #(#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) #(13 14 15))))
	(set! (L 0 0 1) 32)
	L)
      #(#(#(#(1 2 3) 32) #(#(7 8 9) #(10 11 12))) #(13 14 15)))
(test (let ((L #(#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) #(13 14 15))))
	(set! ((L 0) 0 1 2) 32)
	L)
      #(#(#(#(1 2 3) #(4 5 32)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))
(test (let ((L #(#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) #(13 14 15))))
	(set! ((L 0 0) 1 2) 32)
	L)
      #(#(#(#(1 2 3) #(4 5 32)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))
(test (let ((L #(#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) #(13 14 15))))
	(set! ((L 0 0 1) 2) 32)
	L)
      #(#(#(#(1 2 3) #(4 5 32)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))
(test (let ((L #(#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) #(13 14 15))))
	(set! (((L 0) 0) 1 2) 32)
	L)
      #(#(#(#(1 2 3) #(4 5 32)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))
(test (let ((L #(#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) #(13 14 15))))
	(set! (((L 0 0) 1) 2) 32)
	L)
      #(#(#(#(1 2 3) #(4 5 32)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))
(test (let ((L #(#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) #(13 14 15))))
	(set! ((((L 0) 0) 1) 2) 32)
	L)
      #(#(#(#(1 2 3) #(4 5 32)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))


(test (eq? (car (catch #t (lambda () (set! (#(1)) 2)) (lambda args args))) 'wrong-number-of-args) #t)
(test (eq? (car (catch #t (lambda () (set! (#(1) 0 0) 2)) (lambda args args))) 'no-setter) #t)
(test (eq? (car (catch #t (lambda () (set! ((#(1) 0) 0) 2)) (lambda args args))) 'no-setter) #t) ; (set! (1 ...))
(test (let ((L #(#(1 2 3) #(4 5 6)))) (eq? (car (catch #t (lambda () (set! ((L) 1) 32) L) (lambda args args))) 'wrong-number-of-args)) #t)
(test (let ((L #(#(1 2 3) #(4 5 6)))) (eq? (car (catch #t (lambda () (set! ((L)) 32) L) (lambda args args))) 'wrong-number-of-args)) #t)
(test (let ((L #(#(1 2 3) #(4 5 6)))) (eq? (car (catch #t (lambda () (set! ((L 1) 2)) L) (lambda args args))) 'syntax-error)) #t)

(let ((v #(1 2 3))) (define (vr v a) (vector-ref v (+ a 1))) (test (vr v 1) 3))
(let () (define (fillv) (let ((v (make-vector 10))) (do ((i 0 (+ i 1))) ((= i 10) v) (vector-set! v i i)))) (test (fillv) #(0 1 2 3 4 5 6 7 8 9)))
(let () (define (vv) (let ((v #(0 1)) (i 0) (x 2)) (vector-set! v i (+ (vector-ref v i) x)))) (test (vv) 2))
(let () (define (hi) (let ((v1 #(0 1)) (i 0) (j 1)) (vector-set! v1 i (vector-ref v1 j)))) (hi) 1)
(let ()
  (define (fillv) (let ((v (make-vector '(3 3) #f boolean?))) (do ((i 0 (+ i 1))) ((= i 3) v) (vector-set! v i i #t))))
  (test (fillv) #2d((#t #f #f) (#f #t #f) (#f #f #t))))
(let ()
  (define (fillv) (let ((v (make-vector '(3 3) '- symbol?))) (do ((i 0 (+ i 1))) ((= i 3) v) (vector-set! v i i '+))))
  (test (fillv) #2d((+ - -) (- + -) (- - +))))
(let ()
  (define (h111)
    (let ((v (make-vector (list 3 3))))
      (do ((k 0 (+ k 1)))
	  ((= k 3) v)
	(do ((i 0 (+ i 1)))
	    ((= i 3))
	  (vector-set! v k i (+ i (* k 3)))))))
  (test (h111) #2d((0 1 2) (3 4 5) (6 7 8))))

(test (let ((v (immutable! (make-vector '(2 2) 0)))) (define (func) (vector-set! v 1 0 1)) (func)) 'error)

;; check that we can back out when a type changes
(let ((v (make-vector 10 1))
      (s (make-string 10 #\a)))
  (define (f x)
    (set! (x 0) #\b)
    (set! (x 1) #\b)
    x)
  (test (f v) #(#\b #\b 1 1 1 1 1 1 1 1)) ; unopt -> op_vector_set_3
  (test (f v) #(#\b #\b 1 1 1 1 1 1 1 1)) ; op_vector_set_3
  (test (f s) "bbaaaaaaaa"))              ; op_vector_set_3 -> set_unchecked -> implicit_set

(let ((v (make-vector '(2 4) 1))          ; same but op_vector_set_4
      (s (hash-table 0 #(0 0) 1 #(0 0))))
  (define (f x)
    (set! (x 0 0) #\b)
    (set! (x 1 0) #\b)
    x)
  (test (f v) #2d((#\b 1 1 1) (#\b 1 1 1)))
  (test (f v) #2d((#\b 1 1 1) (#\b 1 1 1)))
  (test (f s) (hash-table 0 #(#\b 0) 1 #(#\b 0))))

(test (let () (define (f) (do ((i 0 (+ i 1))) ((= i 2)) (set! (#(a 0 (3)) 1) 0))) (f)) #t)
(test (let () (define (f) (do ((i 0 (+ i 1))) ((= i 2)) (let ((v (make-vector 1 'a symbol?))) (define (fv2) (set! (v 0) 'd)) (fv2) v))) (f))#t)


;;; --------------------------------------------------------------------------------
;;; vector-fill!

(test (fill! (vector 1 2) 4) 4)

(test (let ((v (vector 1 2 3))) (vector-fill! v 0) v) #(0 0 0))
(test (let ((v (vector))) (vector-fill! v #f) v) #())
(let ((v (make-vector 8 #f)))
  (for-each
   (lambda (arg)
     (vector-fill! v arg)
     (test (vector-ref v 1) arg))
   (list #\a 1 () (list 1) '(1 . 2) #f "hi" 'a-symbol abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand (log 0)
	 3.14 3/4 1.0+1.0i #t (vector 1 2 3) (lambda (a) (+ a 1)))))

(test (let ((str "hi") (v (make-vector 3))) (vector-fill! v str) (string-set! (vector-ref v 0) 1 #\a) str) "ha")
(test (let ((lst (list 1 2)) (v (make-vector 3))) (vector-fill! v lst) (list-set! (vector-ref v 0) 1 #\a) lst) '(1 #\a))

(test (let ((v (vector 1 2 3))) (vector-set! v -1 0)) 'error)
(test (let ((v (vector 1 2 3))) (vector-set! v 3 0)) 'error)
(test (vector-fill! #(1 2) 2) 2)
(test (vector-fill! #() 0) 0)
(test (vector-fill! (vector) 0) 0)
(test (let ((v (vector 1))) (vector-fill! v 32) (v 0)) 32)
(test (let ((v (make-vector 11 0))) (vector-fill! v 32) (v 10)) 32)
(test (let ((v (make-vector 16 0))) (vector-fill! v 32) (v 15)) 32)
(test (let ((v (make-vector 3 0))) (vector-fill! v 32) (v 1)) 32)
(test (let ((v (make-vector 3 0))) (fill! v 32) (v 1)) 32)
(test (let ((v #2d((1 2 3) (4 5 6)))) (vector-fill! (v 1) 12) v) #2d((1 2 3) (12 12 12)))
(test (let ((v #i(1 2))) (fill! v 3) v) #i(3 3))

(for-each
 (lambda (arg)
   (test (vector-fill! arg 0) 'error))
 (list "hi" #\a () 1 '(1 . 2) (cons #\a #\b) #f 'a-symbol abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(when with-bignums
  (let ((v (make-vector 2 0.0)))
    (vector-fill! v 1180591620717411303424)
    (num-test (v 1) (expt 2 70))
    (vector-fill! v 3/1180591620717411303424)
    (num-test (v 0) 3/1180591620717411303424)
    (vector-fill! v 1180591620717411303424.0)
    (num-test (v 1) 1180591620717411303424.0)
    (vector-fill! v (complex (expt 2 70) 1.0))
    (num-test (v 0) (complex (expt 2 70) 1.0))
    (set! v (float-vector 1.0))
    (vector-fill! v (bignum "2.0"))
    (num-test (v 0) 2.0)
    (vector-fill! v pi)
    (num-test (v 0) pi)
    (set! v (float-vector 0.0 0.0 0.0))
    (vector-fill! v (bignum "2.0") 1 2)
    (num-test (v 0) 0.0)
    (num-test (v 1) 2.0)
    (set! v (make-int-vector 1))
    (vector-fill! v (bignum "2"))
    (num-test (v 0) 2)
    (set! v (make-int-vector 3 0))
    (vector-fill! v (bignum "2") 1 2)
    (num-test (v 0) 0)
    (num-test (v 1) 2))
  (test (let () (define (func) (vector-fill! (make-vector 3 'a symbol?) (bignum 1/3))) (func)) 'error))

(let ((v (make-vector 3)))
  (vector-fill! v v)
  (test (v 0) v)
  (set! (v 1) 32)
  (test ((v 0) 1) 32))

(test (let ((v (vector 1 2 3 4 5))) (vector-fill! v 21 0) v) #(21 21 21 21 21))
(test (let ((v (vector 1 2 3 4 5))) (vector-fill! v 21 0 5) v) #(21 21 21 21 21))
(test (let ((v (vector 1 2 3 4 5))) (vector-fill! v 21 0 3) v) #(21 21 21 4 5))
(test (let ((v (vector 1 2 3 4 5))) (vector-fill! v 21 2 3) v) #(1 2 21 4 5))
(test (let ((v (vector 1 2 3 4 5))) (vector-fill! v 21 3 3) v) #(1 2 3 4 5))

(if (not with-bignums) (test (let ((v (make-float-vector 3 pi))) (vector-fill! v 0.0) (vector->list v)) '(0.0 0.0 0.0)))
(test (let ((v (make-int-vector 3 1))) (vector-fill! v "2.5")) 'error)
(test (let ((v (make-float-vector 3 pi))) (vector-fill! v #\a)) 'error)
(test (let ((v (make-float-vector 3))) (vector-fill! v 1+i) v) 'error)
(test (let ((v (make-float-vector 3 0.0))) (vector-fill! v 3/4) (vector->list v)) '(0.75 0.75 0.75))
(test (let ((v (make-float-vector 3 0.0))) (vector-fill! v 3) (vector->list v)) '(3.0 3.0 3.0))
(test (let ((v (make-int-vector 3))) (vector-fill! v 1+i) v) 'error)
(test (let ((v (make-int-vector 3 0))) (vector-fill! v 3/4) v) 'error)
(test (let ((v (make-int-vector 3 0))) (vector-fill! v 3.0) v) 'error)

(test (let ((v (make-int-vector 3 1))) (vector-fill! v 2 1) (vector->list v)) '(1 2 2))
(test (let ((v (make-int-vector 3 1))) (vector-fill! v 2 1 1) (vector->list v)) '(1 1 1))
(test (let ((v (make-int-vector 3 1))) (vector-fill! v 2 1 2) (vector->list v)) '(1 2 1))
(test (let ((v (make-int-vector 3 1))) (vector-fill! v 2 1 3) (vector->list v)) '(1 2 2))
(test (let ((v (make-int-vector 3 1))) (vector-fill! v 2 0 3) (vector->list v)) '(2 2 2))
(test (let ((v (make-int-vector 3 1))) (vector-fill! v 2 1 4) (vector->list v)) 'error)
(test (let ((v (make-int-vector 3 1))) (vector-fill! v 2 -1) (vector->list v)) 'error)
(test (let ((v (make-float-vector 3 0.0))) (vector-fill! v 1.0 1) (vector->list v)) '(0.0 1.0 1.0))
(test (let ((v (make-int-vector 3 1))) (vector-fill! v "2.5" 1)) 'error)
(test (let ((v (make-float-vector 3 pi))) (vector-fill! v #\a 0 1)) 'error)
(test (let ((v (make-float-vector 3 0.0))) (vector-fill! v 1+i 1) v) 'error)
(test (let ((v (make-float-vector 3 0.0))) (vector-fill! v 3/4 1) (vector->list v)) '(0.0 0.75 0.75))
(test (let ((v (make-float-vector 3))) (vector-fill! v 3 2) (vector->list v)) '(0.0 0.0 3.0))
(test (let ((v (make-int-vector 3 0))) (vector-fill! v 1+i 2) v) 'error)
(test (let ((v (make-int-vector 3 0))) (vector-fill! v 3/4 0 1) v) 'error)
(test (let ((v (make-int-vector 3))) (vector-fill! v 3.0 2) v) 'error)
(test (vector-fill! #() 0 "hi") 'error)
(test (vector-fill! #() 0 -1 3) 'error)
(test (vector-fill! #() 0 1) 'error)
(test (vector-fill! #() 0 0 4/3) 'error)
(test (vector-fill! (float-vector 1 2) 1+i) 'error)
(test (vector-fill! (int-vector 1 2) 1+i) 'error)
(test (let ((v1 (make-float-vector 20 0.0)) (v2 (make-float-vector 20 1.0))) (vector-fill! v1 1.0) (equivalent? v1 v2)) #t)

(let ()
  (define (boolean|integer? x)
    (or (boolean? x)
	(integer? x)))
  (let ((v (make-vector 3 #f boolean|integer?)))
    (vector-set! v 0 #t)
    (test v #(#t #f #f))
    (vector-set! v 1 1)
    (test v #(#t 1 #f))
    (test (vector-set! v 0 #\a) 'error)
    (test v #(#t 1 #f))
    (test (signature v) (let ((L (list 'boolean|integer? 'vector? 'integer?)))
			  (set-cdr! (cddr L) (cddr L))
			  L))
    (fill! v 1)
    (test v #(1 1 1))
    (test (fill! v #\a) 'error)
    (let ((v1 (make-vector 3)))
      (copy v v1)
      (test v1 #(1 1 1))
    (let ((v2 (make-int-vector 3 3)))
      (copy v2 v)
      (test v #(3 3 3))
    (let ((v3 (make-float-vector 3 3.0)))
      (test (copy v3 v) 'error))))))

(let ()
  (define (vtype? x)
    (and (symbol? x)
         (char=? ((symbol->string x) 0) #\a)))
  (let ((v (make-vector 3 'a vtype?)))
    (test v #(a a a))
    (set! (v 0) 'a1)
    (test v #(a1 a a))
    (test (set! (v 1) :a2) 'error) ; first char is #\:
    (test (vector-set! v 2 'b) 'error)
    (test (fill! v 123) 'error)))


;;; --------------------------------------------------------------------------------
;;; vector-append

(test (vector-append #() #2d()) #())
(test (vector-append) #())
(test (vector-append #()) #())
(test (vector-append #(1 2)) #(1 2))
(test (vector-append #(1) #(2 3) #() #(4)) #(1 2 3 4))
(test (vector-append #(1) #2d((2 3) (4 5)) #3d()) #(1 2 3 4 5))
(test (vector-append #2d((1 2) (3 4)) #3d(((5 6) (7 8)) ((9 10) (11 12)))) #(1 2 3 4 5 6 7 8 9 10 11 12))

(test (vector-append (vector 1 2) (make-int-vector 1 3) #(4)) #(1 2 3 4))
(test (vector-append (vector 1 2) (make-float-vector 1) #(4)) #(1 2 0.0 4))
(test (vector->list (vector-append (make-int-vector 1 3) (make-int-vector 2 1))) '(3 1 1))
(test (vector->list (vector-append (make-float-vector 1 0.0) (make-float-vector 2 1.0))) '(0.0 1.0 1.0))

(test (vector-append (byte-vector 0) (vector 1)) #u(0 1))
(unless pure-s7 (test (vector-append (vector 1) (openlet (inlet 'vector-append (lambda args (#_vector-append (car args) #(2 3)))))) #(1 2 3)))

(unless pure-s7
  (for-each
   (lambda (arg)
     (test (vector-append arg) 'error)
     (test (vector-append #(1 2) arg) 'error))
   (list "hi" #\a () 1 '(1 . 2) (cons #\a #\b) #f 'a-symbol abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
	 3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1)))))

(test (equal? (make-vector 3 1) (make-int-vector 3 1)) #t)

(let ((iv (make-int-vector 3 1))
      (fv (make-float-vector 3 2.0))
      (vv (make-vector 3 #f)))
  (test (equal? (vector-append iv iv iv) (make-int-vector 9 1)) #t)
  (test (vector-append iv iv vv) 'error)
  (test (vector-append iv fv iv) 'error) ;(int-vector 1 1 1 2 2 2 1 1 1))
  (test (vector-append iv fv fv) 'error) ;(int-vector 1 1 1 2 2 2 2 2 2))
  (test (vector-append iv fv vv) 'error)
  (test (vector-append iv vv iv) 'error)
  (test (vector-append fv iv iv) (float-vector 2.0 2.0 2.0 1 1 1 1 1 1))       ; #(2.0 2.0 2.0 1 1 1 1 1 1))
  (test (vector-append fv iv fv) (float-vector 2.0 2.0 2.0 1 1 1 2.0 2.0 2.0)) ; #(2.0 2.0 2.0 1 1 1 2.0 2.0 2.0))
  (test (vector-append fv fv iv) (float-vector 2.0 2.0 2.0 2.0 2.0 2.0 1 1 1)) ; #(2.0 2.0 2.0 2.0 2.0 2.0 1 1 1))
  (test (vector-append fv fv fv) (make-float-vector 9 2.0))
  (test (vector-append fv fv vv) 'error)
  (test (vector-append vv iv iv) #(#f #f #f 1 1 1 1 1 1))
  (test (vector-append vv iv fv) #(#f #f #f 1 1 1 2.0 2.0 2.0))
  (test (vector-append vv iv vv) #(#f #f #f 1 1 1 #f #f #f))
  (test (vector-append vv fv iv) #(#f #f #f 2.0 2.0 2.0 1 1 1))
  (test (vector-append vv fv fv) #(#f #f #f 2.0 2.0 2.0 2.0 2.0 2.0))
  (test (vector-append vv fv vv) #(#f #f #f 2.0 2.0 2.0 #f #f #f))
  (test (vector-append vv vv iv) #(#f #f #f #f #f #f 1 1 1))
  (test (vector-append vv vv fv) #(#f #f #f #f #f #f 2.0 2.0 2.0))
  (test (vector-append vv vv vv) #(#f #f #f #f #f #f #f #f #f)))

(test (equal? (vector-append (float-vector 1 2 3) #()) (float-vector 1 2 3)) #t)
(test (equal? (vector-append (float-vector) #(1 2 3) #() (make-int-vector 0 0)) (float-vector 1 2 3)) #t)
(test (equal? (float-vector) (vector-append (float-vector))) #t)
(test (equal? (vector-append #() (float-vector) (make-int-vector 3 1) (vector)) (make-vector 3 1)) #t)
(test (equal? (vector-append (int-vector 1 2 3) #()) (int-vector 1 2 3)) #t)
(test (equal? (vector-append (int-vector) #(1 2 3) #() (make-int-vector 0 0)) (int-vector 1 2 3)) #t)
(test (equal? (int-vector) (vector-append (int-vector))) #t)
(test (equal? (vector-append #() (int-vector) (make-int-vector 3 1) (vector)) (make-vector 3 1)) #t)

(when full-s7test
  (define (test-append size)
    (let ((strs ())
	  (vecs ())
	  (fvecs ())
	  (ivecs ())
	  (ifvecs ())
	  (allvecs ())
	  (bvecs ())
	  (lsts ()))
      (do ((i 0 (+ i 1)))
	  ((= i size))
	(set! strs (cons (make-string size (integer->char (+ 1 (random 255)))) strs))
	(set! bvecs (cons (string->byte-vector (make-string size (integer->char (random 256)))) bvecs))
	(set! vecs (cons (make-vector size i) vecs))
	(set! ivecs (cons (make-int-vector size i) ivecs))
	(set! fvecs (cons (make-float-vector size (* i 1.0)) fvecs))
	(set! ifvecs (cons (make-vector size (if (even? i) (* i 1.0) i)) ifvecs))
	(set! allvecs (cons (make-vector size (if (even? i) (* i 1.0) i)) allvecs))
	(set! lsts (cons (make-list size i) lsts)))
      (let ((lst (apply append lsts))
	    (vec (apply vector-append vecs))
	    (fvec (apply vector-append fvecs))
	    (ivec (apply vector-append ivecs))
	    (ifvec (apply vector-append ifvecs))
	    (allvec (apply vector-append allvecs))
	    (str (apply string-append strs))
	    (bvec (apply append bvecs)))
	(test (vector? vec) #t)
	(test (length vec) (* size size))
	(test (float-vector? fvec) #t)
	(test (length fvec) (* size size))
	(test (int-vector? ivec) #t)
	(test (length ivec) (* size size))
	(test (vector? allvec) #t)
	(test (length allvec) (* size size))
	(test (vector? ifvec) #t)
	(test (length ifvec) (* size size))
	(test (pair? lst) #t)
	(test (length lst) (* size size))
	(test (string? str) #t)
	(test (length str) (* size size))
	(test (byte-vector? bvec) #t)
	(test (length bvec) (* size size))
	)))

  (do ((i 1 (* i 10)))
      ((> i 1000))
    (test-append i)))

(test (vector-append #i(2 1) #(2 1) #r(2.0 1.5)) 'error)


;;; --------------------------------------------------------------------------------
;;; miscellaneous vectors

(test (let ((sum 0)) (for-each (lambda (n) (set! sum (+ sum n))) (vector 1 2 3)) sum) 6)
(test (let ((sum 0)) (for-each (lambda (n m) (set! sum (+ sum n (- m)))) (vector 1 2 3) (vector 4 5 6)) sum) -9)
(test (let () (for-each (lambda (n) (error 'wrong-type-arg "oops")) (vector)) #f) #f)
(test (let ((sum 0)) (for-each (lambda (n m p) (set! sum (+ sum n (- m) (* 2 p)))) (vector 1 2 3) (vector 4 5 6) (vector 6 7 8)) sum) 33)
(test (let ((sum 0)) (for-each (lambda (n) (for-each (lambda (m) (set! sum (+ sum (* m n)))) (vector 1 2 3))) (vector 4 5 6)) sum) 90)
(test (call/cc (lambda (return) (for-each (lambda (n) (return "oops")) (vector 1 2 3)))) "oops")
(test (call/cc (lambda (return) (for-each (lambda (n) (if (even? n) (return n))) (vector 1 3 8 7 9 10)))) 8)

(for-each
 (lambda (data)
   (let ((v data)
	 (c #f)
	 (y 0))

     (do ((i 0 (+ i 1)))
	 ((= i 10))
       (set! (v i) i))

     (let ((tag
	    (call/cc
	     (lambda (exit)

	       (for-each
		(lambda (x)

		  (call/cc
		   (lambda (return)
		     (set! c return)))

		  (if (and (even? (inexact->exact x))
			   (> x y)
			   (< x 10))
		      (begin
			(set! (v (inexact->exact y)) 100)
			(set! y x)
			(exit x))
		      (set! y x)))
		v)))))

       (if (and (number? tag) (< tag 10))
	   (c)))

     (let ((correct (vector 0 100 2 100 4 100 6 100 8 9)))
       (do ((i 0 (+ i 1)))
	   ((= i (length v)))
	 (if (not (= (correct i) (inexact->exact (v i))))
	     (format #t ";for-each call/cc data: ~A~%" v))))))

 (list (make-vector 10)
       (make-list 10)))


(test (map (lambda (n) (+ 1 n)) (vector 1 2 3)) '(2 3 4))
(test (map (lambda (n m) (- n m)) (vector 1 2 3) (vector 4 5 6)) '(-3 -3 -3))
(test (map (lambda (n m p) (+ n m p)) (vector 1 2 3) (vector 4 5 6) (vector 6 7 8)) '(11 14 17))
(test (map (lambda (n) (map (lambda (m) (* m n)) (vector 1 2 3))) (vector 4 5 6)) '((4 8 12) (5 10 15) (6 12 18)))
(test (call/cc (lambda (return) (map (lambda (n) (return "oops")) (vector 1 2 3)))) "oops")
(test (call/cc (lambda (return) (map (lambda (n) (if (even? n) (return n))) (vector 1 3 8 7 9 10)))) 8)

(test (map (lambda (x) x) (make-int-vector 3 0)) '(0 0 0))
(test (map (lambda (x) x) (let ((v (make-int-vector 3 0))) (set! (v 1) 1) (set! (v 2) 2) v)) '(0 1 2))
(test (map (lambda (x) x) (make-float-vector 3 0.0)) '(0.0 0.0 0.0))
(test (let ((lst ())) (for-each (lambda (n) (set! lst (cons n lst))) (let ((v (make-int-vector 3 0))) (set! (v 1) 1) v)) lst) '(0 1 0))

(test (vector? (symbol-table)) #t)

(let ((v (make-vector 3 (vector 1 2))))
  (test (equal? (v 0) (v 1)) #t)
  (test (eq? (v 0) (v 1)) #t)
  (test (eqv? (v 0) (v 1)) #t))

(let ((v (vector (vector 1 2) (vector 1 2) (vector 1 2))))
  (test (equal? (v 0) (v 1)) #t)
  (test (eq? (v 0) (v 1)) #f)
  (test (eqv? (v 0) (v 1)) #f))

(let ((v (vector (vector (vector (vector 1 2) 3) 4) 5)))
  (test (v 0) #(#(#(1 2) 3) 4))
  (test (v 1) 5)
  (test (((v 0) 0) 1) 3)
  (test ((((v 0) 0) 0) 1) 2))

(test (make-vector 1 (make-vector 1 (make-vector 1 0))) #(#(#(0))))
(test (vector->list (let ((v (make-int-vector 3 0))) (set! (v 0) 32) (set! (v 1) -1) (set! (v 2) 2) (sort! v <))) '(-1 2 32))

(let ((v1 (make-vector 3 1)))
  (num-test (v1 1) 1)
  (set! (v1 1) 2)
  (num-test (v1 1) 2)
  (let ((i0 0)
	(i2 2))
    (num-test (v1 i0) 1)
    (num-test (vector-ref v1 i2) 1)
    (set! (v1 i0) 0)
    (num-test (v1 0) 0)
    (set! (v1 i0) i2)
    (num-test (v1 i0) i2))
  (test (vector-dimensions v1) '(3))
  (set! v1 (make-vector '(3 2)))
  (test (vector-dimensions v1) '(3 2))
  (vector-set! v1 1 1 0)
  (num-test (vector-ref v1 1 1) 0)
  (let ((i0 1)
	(i1 1)
	(i2 32))
    (set! (v1 i0 i1) i2)
    (num-test (vector-ref v1 1 1) 32)
    (num-test (v1 i0 i1) i2)
    (vector-set! v1 0 1 3)
    (num-test (v1 0 1) 3)
    (num-test (v1 1 1) 32))
  (set! v1 (make-vector '(2 4 3) 1))
  (test (vector-dimensions v1) '(2 4 3))
  (num-test (vector-ref v1 1 1 1) 1)
  (vector-set! v1 0 0 0 32)
  (num-test (v1 0 0 0) 32)
  (set! (v1 0 1 1) 3)
  (num-test (v1 0 1 1) 3))

(let-temporarily (((*s7* 'print-length) 32))
  (let ((vect1 #3d(((1 2 3) (3 4 5)) ((5 6 1) (7 8 2))))
	(vect2 #2d((1 2 3 4 5 6) (7 8 9 10 11 12)))
	(vect3 #(1 2 3 4 5 6 7 8 9 10 11 12 13 14))
	(vect4 #3d(((1 2) (3 4) (5 6)) ((7 8) (9 10) (11 12))))
	(vect1t (make-int-vector '(2 2 3) 0)))
    (let ((v (subvector vect1t 0 12 '(12))))
      (set! (v 0) 1) (set! (v 1) 2) (set! (v 2) 3) (set! (v 3) 3) (set! (v 4) 4) (set! (v 5) 5)
      (set! (v 6) 5) (set! (v 7) 6) (set! (v 8) 1) (set! (v 9) 7) (set! (v 10) 8) (set! (v 11) 2))
    (do ((i 1 (+ i 1)))
	((= i 15))
      (set! (*s7* 'print-length) i)
      (let ((str (object->string vect1)))
	(test str (case i
		    ((1) "#3d(((1 ...)...)...)")
		    ((2) "#3d(((1 2 ...)...)...)")
		    ((3) "#3d(((1 2 3)...)...)")
		    ((4) "#3d(((1 2 3) (3 ...))...)")
		    ((5) "#3d(((1 2 3) (3 4 ...))...)")
		    ((6) "#3d(((1 2 3) (3 4 5))...)")
		    ((7) "#3d(((1 2 3) (3 4 5)) ((5 ...)...))")
		    ((8) "#3d(((1 2 3) (3 4 5)) ((5 6 ...)...))")
		    ((9) "#3d(((1 2 3) (3 4 5)) ((5 6 1)...))")
		    ((10) "#3d(((1 2 3) (3 4 5)) ((5 6 1) (7 ...)))")
		    ((11) "#3d(((1 2 3) (3 4 5)) ((5 6 1) (7 8 ...)))")
		    ((12) "#3d(((1 2 3) (3 4 5)) ((5 6 1) (7 8 2)))")
		    ((13) "#3d(((1 2 3) (3 4 5)) ((5 6 1) (7 8 2)))")
		    ((14) "#3d(((1 2 3) (3 4 5)) ((5 6 1) (7 8 2)))"))))

      (let ((str (object->string vect1t)))
	(test str (case i
		    ((1) "#i3d(((1 ...)...)...)")
		    ((2) "#i3d(((1 2 ...)...)...)")
		    ((3) "#i3d(((1 2 3)...)...)")
		    ((4) "#i3d(((1 2 3) (3 ...))...)")
		    ((5) "#i3d(((1 2 3) (3 4 ...))...)")
		    ((6) "#i3d(((1 2 3) (3 4 5))...)")
		    ((7) "#i3d(((1 2 3) (3 4 5)) ((5 ...)...))")
		    ((8) "#i3d(((1 2 3) (3 4 5)) ((5 6 ...)...))")
		    ((9) "#i3d(((1 2 3) (3 4 5)) ((5 6 1)...))")
		    ((10) "#i3d(((1 2 3) (3 4 5)) ((5 6 1) (7 ...)))")
		    ((11) "#i3d(((1 2 3) (3 4 5)) ((5 6 1) (7 8 ...)))")
		    ((12) "#i3d(((1 2 3) (3 4 5)) ((5 6 1) (7 8 2)))")
		    ((13) "#i3d(((1 2 3) (3 4 5)) ((5 6 1) (7 8 2)))")
		    ((14) "#i3d(((1 2 3) (3 4 5)) ((5 6 1) (7 8 2)))"))))

      (let ((str (object->string vect4)))
	(test str (case i
		    ((1) "#3d(((1 ...)...)...)")
		    ((2) "#3d(((1 2)...)...)")
		    ((3) "#3d(((1 2) (3 ...)...)...)")
		    ((4) "#3d(((1 2) (3 4)...)...)")
		    ((5) "#3d(((1 2) (3 4) (5 ...))...)")
		    ((6) "#3d(((1 2) (3 4) (5 6))...)")
		    ((7) "#3d(((1 2) (3 4) (5 6)) ((7 ...)...))")
		    ((8) "#3d(((1 2) (3 4) (5 6)) ((7 8)...))")
		    ((9) "#3d(((1 2) (3 4) (5 6)) ((7 8) (9 ...)...))")
		    ((10) "#3d(((1 2) (3 4) (5 6)) ((7 8) (9 10)...))")
		    ((11) "#3d(((1 2) (3 4) (5 6)) ((7 8) (9 10) (11 ...)))")
		    ((12) "#3d(((1 2) (3 4) (5 6)) ((7 8) (9 10) (11 12)))")
		    ((13) "#3d(((1 2) (3 4) (5 6)) ((7 8) (9 10) (11 12)))")
		    ((14) "#3d(((1 2) (3 4) (5 6)) ((7 8) (9 10) (11 12)))"))))

      (let ((str (object->string vect2)))
	(test str (case i
		    ((1) "#2d((1 ...)...)")
		    ((2) "#2d((1 2 ...)...)")
		    ((3) "#2d((1 2 3 ...)...)")
		    ((4) "#2d((1 2 3 4 ...)...)")
		    ((5) "#2d((1 2 3 4 5 ...)...)")
		    ((6) "#2d((1 2 3 4 5 6)...)")
		    ((7) "#2d((1 2 3 4 5 6) (7 ...))")
		    ((8) "#2d((1 2 3 4 5 6) (7 8 ...))")
		    ((9) "#2d((1 2 3 4 5 6) (7 8 9 ...))")
		    ((10) "#2d((1 2 3 4 5 6) (7 8 9 10 ...))")
		    ((11) "#2d((1 2 3 4 5 6) (7 8 9 10 11 ...))")
		    ((12) "#2d((1 2 3 4 5 6) (7 8 9 10 11 12))")
		    ((13) "#2d((1 2 3 4 5 6) (7 8 9 10 11 12))")
		    ((14) "#2d((1 2 3 4 5 6) (7 8 9 10 11 12))"))))

      (let ((str (object->string vect3)))
	(test str (case i
		    ((1) "#(1 ...)")
		    ((2) "#(1 2 ...)")
		    ((3) "#(1 2 3 ...)")
		    ((4) "#(1 2 3 4 ...)")
		    ((5) "#(1 2 3 4 5 ...)")
		    ((6) "#(1 2 3 4 5 6 ...)")
		    ((7) "#(1 2 3 4 5 6 7 ...)")
		    ((8) "#(1 2 3 4 5 6 7 8 ...)")
		    ((9) "#(1 2 3 4 5 6 7 8 9 ...)")
		    ((10) "#(1 2 3 4 5 6 7 8 9 10 ...)")
		    ((11) "#(1 2 3 4 5 6 7 8 9 10 11 ...)")
		    ((12) "#(1 2 3 4 5 6 7 8 9 10 11 12 ...)")
		    ((13) "#(1 2 3 4 5 6 7 8 9 10 11 12 13 ...)")
		    ((14) "#(1 2 3 4 5 6 7 8 9 10 11 12 13 14)")))))

    (let ((vect5 (make-vector '(2 3))))
      (set! (vect5 0 0) vect1)
      (set! (vect5 0 1) vect2)
      (set! (vect5 0 2) vect3)
      (set! (vect5 1 0) vect4)
      (set! (vect5 1 1) (vector 1 2 3))
      (set! (vect5 1 2) #2d())

      (do ((i 1 (+ i 1)))
	  ((= i 15))
	(set! (*s7* 'print-length) i)
	(let ((str (object->string vect5)))
	  (test str (case i
		      ((1) "#2d((#3d(((1 ...)...)...) ...)...)")
		      ((2) "#2d((#3d(((1 2 ...)...)...) #2d((1 2 ...)...) ...)...)")
		      ((3) "#2d((#3d(((1 2 3)...)...) #2d((1 2 3 ...)...) #(1 2 3 ...))...)")
		      ((4) "#2d((#3d(((1 2 3) (3 ...))...) #2d((1 2 3 4 ...)...) #(1 2 3 4 ...)) (#3d(((1 2) (3 4)...)...) ...))")
		      ((5) "#2d((#3d(((1 2 3) (3 4 ...))...) #2d((1 2 3 4 5 ...)...) #(1 2 3 4 5 ...)) (#3d(((1 2) (3 4) (5 ...))...) #(1 2 3) ...))")
		      ((6) "#2d((#3d(((1 2 3) (3 4 5))...) #2d((1 2 3 4 5 6)...) #(1 2 3 4 5 6 ...)) (#3d(((1 2) (3 4) (5 6))...) #(1 2 3) #2d()))")
		      ((7) "#2d((#3d(((1 2 3) (3 4 5)) ((5 ...)...)) #2d((1 2 3 4 5 6) (7 ...)) #(1 2 3 4 5 6 7 ...)) (#3d(((1 2) (3 4) (5 6)) ((7 ...)...)) #(1 2 3) #2d()))")
		      ((8) "#2d((#3d(((1 2 3) (3 4 5)) ((5 6 ...)...)) #2d((1 2 3 4 5 6) (7 8 ...)) #(1 2 3 4 5 6 7 8 ...)) (#3d(((1 2) (3 4) (5 6)) ((7 8)...)) #(1 2 3) #2d()))")
		      ((9) "#2d((#3d(((1 2 3) (3 4 5)) ((5 6 1)...)) #2d((1 2 3 4 5 6) (7 8 9 ...)) #(1 2 3 4 5 6 7 8 9 ...)) (#3d(((1 2) (3 4) (5 6)) ((7 8) (9 ...)...)) #(1 2 3) #2d()))")
		      ((10) "#2d((#3d(((1 2 3) (3 4 5)) ((5 6 1) (7 ...))) #2d((1 2 3 4 5 6) (7 8 9 10 ...)) #(1 2 3 4 5 6 7 8 9 10 ...)) (#3d(((1 2) (3 4) (5 6)) ((7 8) (9 10)...)) #(1 2 3) #2d()))")
		      ((11) "#2d((#3d(((1 2 3) (3 4 5)) ((5 6 1) (7 8 ...))) #2d((1 2 3 4 5 6) (7 8 9 10 11 ...)) #(1 2 3 4 5 6 7 8 9 10 11 ...)) (#3d(((1 2) (3 4) (5 6)) ((7 8) (9 10) (11 ...))) #(1 2 3) #2d()))")
		      ((12) "#2d((#3d(((1 2 3) (3 4 5)) ((5 6 1) (7 8 2))) #2d((1 2 3 4 5 6) (7 8 9 10 11 12)) #(1 2 3 4 5 6 7 8 9 10 11 12 ...)) (#3d(((1 2) (3 4) (5 6)) ((7 8) (9 10) (11 12))) #(1 2 3) #2d()))")
		      ((13) "#2d((#3d(((1 2 3) (3 4 5)) ((5 6 1) (7 8 2))) #2d((1 2 3 4 5 6) (7 8 9 10 11 12)) #(1 2 3 4 5 6 7 8 9 10 11 12 13 ...)) (#3d(((1 2) (3 4) (5 6)) ((7 8) (9 10) (11 12))) #(1 2 3) #2d()))")
		      ((14) "#2d((#3d(((1 2 3) (3 4 5)) ((5 6 1) (7 8 2))) #2d((1 2 3 4 5 6) (7 8 9 10 11 12)) #(1 2 3 4 5 6 7 8 9 10 11 12 13 14)) (#3d(((1 2) (3 4) (5 6)) ((7 8) (9 10) (11 12))) #(1 2 3) #2d()))"))))))))

(test (object->string (make-int-vector 3 0)) "#i(0 0 0)")

(let ((v (make-vector '(2 2))))
  (set! (v 0 0) 1)
  (set! (v 0 1) 2)
  (set! (v 1 0) 3)
  (set! (v 1 1) 4)
  (set! (v 0 1) #2d((1 2) (3 4)))
  (test (object->string v) "#2d((1 #2d((1 2) (3 4))) (3 4))"))

(let ((v (make-int-vector '(2 2))))
  (set! (v 0 0) 1)
  (set! (v 0 1) 2)
  (set! (v 1 0) 3)
  (set! (v 1 1) 4)
  (test (object->string v) "#i2d((1 2) (3 4))"))

(let ((v #2d((1 2) (3 4))))
  (set! (v 0 1) #2d((1 2) (3 4)))
  (test (object->string v) "#2d((1 #2d((1 2) (3 4))) (3 4))"))

(let ((v (make-vector '(2 3))))
  (do ((i 0 (+ i 1)))
      ((= i 2))
    (do ((j 0 (+ j 1)))
	((= j 3))
      (set! (v i j) (list i j))))
  (test (v 0 0) '(0 0))
  (test ((v 1 2) 0) 1)
  (test (v 1 2 0) 1)
  (test (v 1 2 0 0) 'error)
  (test (object->string v) "#2d(((0 0) (0 1) (0 2)) ((1 0) (1 1) (1 2)))"))

(test (object->string (make-float-vector 3 1.0)) "#r(1.0 1.0 1.0)")
(test (object->string (make-float-vector 3 -1.5)) "#r(-1.5 -1.5 -1.5)")
(test (object->string (make-int-vector 3 1)) "#i(1 1 1)")
(test (object->string (make-int-vector 3 -1)) "#i(-1 -1 -1)")
(test (object->string (make-int-vector 0 0)) "#i()")
(test (object->string #r()) "#r()")
(test (object->string (make-float-vector '(3 2 0) 0.0)) "#r3d()")

(test (let ((v1 (make-vector '(3 2) 1))
	    (v2 (make-vector '(3 2) 2))
	    (sum 0))
	(for-each (lambda (n m) (set! sum (+ sum n m))) v1 v2)
	sum)
      18)
(test (vector->list (make-vector '(2 3) 1)) '(1 1 1 1 1 1))
(test (vector->list #2d((1 2) (3 4))) '(1 2 3 4))
(test (list->vector '((1 2) (3 4))) #((1 2) (3 4)))
(test (vector->list (make-vector (list 2 0))) ())
(test (vector-dimensions #2d((1 2 3))) '(1 3))

(test (#2d((1 2 3) (4 5 6)) 0 0) 1)
(test (#2d((1 2 3) (4 5 6)) 0 1) 2)
(test (#2d((1 2 3) (4 5 6)) 1 1) 5)
(test (#3d(((1 2) (3 4)) ((5 6) (7 8))) 0 0 0) 1)
(test (#3d(((1 2) (3 4)) ((5 6) (7 8))) 1 1 0) 7)
(test (#4d((((1) (2)) ((3) (4)) ((5) (6)))) 0 0 0 0) 1)
(test (vector? #2d((1 2) (3 4))) #t)
(test ((#2d((1 #2d((2 3) (4 5))) (6 7)) 0 1) 1 0) 4)
(test ((((((((((#10d((((((((((1) (1))) (((1) (1))))) (((((1) (1))) (((1) (1))))))) (((((((1) (1))) (((1) (1))))) (((((1) (1))) (((1) (1))))))))) (((((((((1) (1))) (((1) (1))))) (((((1) (1))) (((1) (1))))))) (((((((1) (1))) (((1) (1))))) (((((1) (1))) (((1) (1)))))))))) 0) 0) 0) 0) 0) 0) 0) 0) 0) 0) 1)
(test (#10d((((((((((1) (1))) (((1) (1))))) (((((1) (1))) (((1) (1))))))) (((((((1) (1))) (((1) (1))))) (((((1) (1))) (((1) (1))))))))) (((((((((1) (1))) (((1) (1))))) (((((1) (1))) (((1) (1))))))) (((((((1) (1))) (((1) (1))))) (((((1) (1))) (((1) (1)))))))))) 0 0 0 0 0 0 0 0 0 0) 1)
(let ((v (make-vector (make-list 100 1) 0)))
  (test (equal? v #100d((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((0))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))) #t)
  (test (apply v (make-list 100 0)) 0)
  (test (v 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0) 0))

;; eval-string here else these are read errors
(test (eval-string "#3d(((1 2) (3 4)) ((5 6) (7)))") 'error)
(test (eval-string "#3d(((1 2) (3 4)) ((5) (7 8)))") 'error)
(test (eval-string "#3d(((1 2) (3 4)) (() (7 8)))") 'error)
(test (eval-string "#3d(((1 2) (3 4)) ((5 6) (7 8 9)))") 'error)
(test (eval-string "#3d(((1 2) (3 4)) (5 (7 8 9)))") 'error)
(test (eval-string "#3d(((1 2) (3 4)) ((5 6) (7 . 8)))") 'error)
(test (eval-string "#3d(((1 2) (3 4)) ((5 6) (7 8 . 9)))") 'error)
(test (eval-string "#3d(((1 2) (3 4)) ((5 6) ()))") 'error)
(test (eval-string "#3d(((1 2) (3 4)) ((5 6)))") 'error)

(test (vector-dimensions #3d(((1 2) (3 4)) ((5 6) (7 8)))) '(2 2 2))
(test (vector-dimensions #2d((1 2 3) (4 5 6))) '(2 3))
(test (vector-dimensions #4d((((1) (2)) ((3) (4)) ((5) (6))))) '(1 3 2 1))

(test (vector-length #3d(((1 2) (3 4)) ((5 6) (7 8)))) 8)
(test (length #2d((1 2 3) (4 5 6))) 6)

(test (#2d((1 (2) 3) (4 () 6)) 0 1) '(2))
(test (#2d((1 (2) 3) (4 () 6)) 1 1) ())
(test (#2d((1 (2) 3) (4 6 ())) 1 2) ())
(test (#2d((() (2) ()) (4 5 6)) 0 2) ())

(test (equal? (make-vector 0) (make-vector '(0))) #t)
(test (equal? #() (make-vector '(0))) #t)

(test (equal? #2d((1 2) (3 4)) #2d((1 2) (3 4))) #t)
(test (eq? #2d((1 2) (3 4)) #2d((1 2) (3 4))) #f)
(test (eqv? #2d((1 2) (3 4)) #2d((1 2) (3 4))) #f)
(test (make-vector (1 . 2) "hi") 'error)
(test (make-vector (cons 1 2) "hi") 'error)
(test (equal? (make-vector 0) (vector)) #t)
(test (equal? #() (vector)) #t)
(test (equal? (make-int-vector 0 0) (make-int-vector 0 0)) #t)
(test (equal? #() (make-int-vector 0 0)) #t)
(test (equal? (make-vector '(2 0)) (make-int-vector '(2 0) 0)) #t)
(test (equal? (make-vector '(2 0)) (make-int-vector '(0 2) 0)) #f)

(let ((v (make-vector '(2 3) 0)))
  (num-test (vector-length v) 6)
  (test (vector-dimensions v) '(2 3))
  (num-test (v 0 0) 0)
  (num-test (v 1 2) 0)
  (test (v 2 2) 'error)
  (test (v 2 -1) 'error)
  (test (v 2 0) 'error)
  (set! (v 0 1) 1)
  (num-test (v 0 1) 1)
  (num-test (v 1 0) 0)
  (set! (v 1 2) 2)
  (num-test (v 1 2) 2)
  (test (set! (v 2 2) 32) 'error)
  (test (set! (v 1 -1) 0) 'error)
  (test (set! (v 2 0) 0) 'error)
  (num-test (vector-ref v 0 1) 1)
  (num-test (vector-ref v 1 2) 2)
  (test (vector-ref v 2 2) 'error)
  (test (vector-ref v 1 -1) 'error)
  (vector-set! v 1 1 64)
  (num-test (vector-ref v 1 1) 64)
  (num-test (vector-ref v 0 0) 0)
  (test (vector-ref v 1 2 3) 'error)
  (test (vector-set! v 1 2 3 4) 'error)
  (test (v 1 1 1) 'error)
  (test (set! (v 1 1 1) 1) 'error))


(let ((v1 (make-vector '(3 2) 0))
      (v2 (make-vector '(2 3) 0))
      (v3 (make-vector '(2 3 4) 0))
      (v4 (make-vector 6 0))
      (v5 (make-vector '(2 3) 0)))
  (test (equal? v1 v2) #f)
  (test (equal? v1 v3) #f)
  (test (equal? v1 v4) #f)
  (test (equal? v2 v2) #t)
  (test (equal? v3 v2) #f)
  (test (equal? v4 v2) #f)
  (test (equal? v5 v2) #t)
  (test (equal? v4 v3) #f)
  (test (vector-dimensions v3) '(2 3 4))
  (test (vector-dimensions v4) '(6))
  (num-test (v3 1 2 3) 0)
  (set! (v3 1 2 3) 32)
  (num-test (v3 1 2 3) 32)
  (num-test (vector-length v3) 24)
  (num-test (vector-ref v3 1 2 3) 32)
  (vector-set! v3 1 2 3 -32)
  (num-test (v3 1 2 3) -32)
  (test (v3 1 2) #(0 0 0 -32))
  (test (set! (v3 1 2) 3) 'error)
  (test (vector-ref v3 1 2) #(0 0 0 -32))
  (test (vector-set! v3 1 2 32) 'error))

(test (let ((v #2d((1 2) (3 4)))) (vector-fill! v #t) v) #2d((#t #t) (#t #t)))

(test (eval-string "#2d((1 2) #2d((3 4) 5 6))") 'error)
(test (string=? (object->string #2d((1 2) (3 #2d((3 4) (5 6))))) "#2d((1 2) (3 #2d((3 4) (5 6))))") #t)
(test (string=? (object->string #3d(((#2d((1 2) (3 4)) #(1)) (#3d(((1))) 6)))) "#3d(((#2d((1 2) (3 4)) #(1)) (#3d(((1))) 6)))") #t)

(test (make-vector '(2 -2)) 'error)
(test (make-vector '(2 1/2)) 'error)
(test (make-vector '(2 1.2)) 'error)
(test (make-vector '(2 2+i)) 'error)
(test (make-vector '(2 "hi")) 'error)

(let ((v (make-vector '(1 1 1) 32)))
  (test (vector? v) #t)
  (test (equal? v #()) #f)
  (test (vector->list v) '(32))
  (test (vector-ref v 0) '#2d((32)))
  (test (vector-set! v 0 0) 'error)
  (test (vector-ref v 0 0) #(32))
  (test (vector-set! v 0 0 0) 'error)
  (test (vector-ref v 0 0 0) 32)
  (test (let () (vector-set! v 0 0 0 31) (vector-ref v 0 0 0)) 31)
  (test (vector-length v) 1)
  (test (vector-dimensions v) '(1 1 1))
  (test (object->string v) "#3d(((31)))")
  )

(test (vector? #3d(((32)))) #t)
(test (equal? #3d(((32))) #()) #f)
(test (vector->list #3d(((32)))) '(32))
(test (#3d(((32))) 0) '#2d((32)))
(test (set! (#3d(((32))) 0) 0) 'error)
(test (#3d(((32))) 0 0) #(32))
(test (set! (#3d(((32))) 0 0) 0) 'error)
(test (#3d(((32))) 0 0 0) 32)
(test (vector-length #3d(((32)))) 1)
(test (vector-dimensions #3d(((32)))) '(1 1 1))
(test (object->string #3d(((32)))) "#3d(((32)))")


(let ((v1 (make-vector '(1 0))))
  (test (vector? v1) #t)
  (test (equal? v1 #()) #f)
  (test (vector->list v1) ())
  (test (vector-ref v1 0) 'error)
  (test (vector-set! v1 0 0) 'error)
  (test (vector-ref v1 0 0) 'error)
  (test (vector-set! v1 0 0 0) 'error)
  (test (vector-length v1) 0)
  (test (vector-dimensions v1) '(1 0))
  (test (object->string v1) "#2d()")
  )

(let ((v2 (make-vector '(10 3 0))))
  (test (vector? v2) #t)
  (test (equal? v2 #()) #f)
  (test (vector->list v2) ())
  (test (vector-ref v2) 'error)
  (test (vector-set! v2 0) 'error)
  (test (vector-ref v2 0) 'error)
  (test (vector-set! v2 0 0) 'error)
  (test (vector-ref v2 0 0) 'error)
  (test (vector-set! v2 0 0 0) 'error)
  (test (vector-ref v2 1 2 0) 'error)
  (test (vector-set! v2 1 2 0 0) 'error)
  (test (vector-length v2) 0)
  (test (vector-dimensions v2) '(10 3 0))
  (test (object->string v2) "#3d()")
  )

(let ((v3 (make-vector '(10 0 3))))
  (test (vector? v3) #t)
  (test (equal? v3 #()) #f)
  (test (vector->list v3) ())
  (test (vector-ref v3) 'error)
  (test (vector-set! v3 0) 'error)
  (test (vector-ref v3 0) 'error)
  (test (vector-set! v3 0 0) 'error)
  (test (vector-ref v3 0 0) 'error)
  (test (vector-set! v3 0 0 0) 'error)
  (test (vector-ref v3 1 0 2) 'error)
  (test (vector-set! v3 1 0 2 0) 'error)
  (test (vector-length v3) 0)
  (test (vector-dimensions v3) '(10 0 3))
  (test (object->string v3) "#3d()")
  )

(test (((#(("hi") ("ho")) 0) 0) 1) #\i)
(test (string-ref (list-ref (vector-ref #(("hi") ("ho")) 0) 0) 1) #\i)

(test (equal? #2d() (make-vector '(0 0))) #t)
(test (equal? #2d() (make-vector '(1 0))) #f)
(test (equal? (make-vector '(2 2) 2) #2d((2 2) (2 2))) #t)
(test (equal? (make-vector '(2 2) 2) #2d((2 2) (1 2))) #f)
(test (equal? (make-vector '(1 2 3) 0) (make-vector '(1 2 3) 0)) #t)
(test (equal? (make-vector '(1 2 3) 0) (make-vector '(1 3 2) 0)) #f)
(test (make-vector '1 2 3) 'error)

(test (set! (vector) 1) 'error)
(test (set! (make-vector 1) 1) 'error)
(test (equal? (make-vector 10 ()) (make-hash-table 10)) #f)
(test (equal? #() (copy #())) #t)
(test (equal? #2d() (copy #2d())) #t)
(test (fill! #() 1) 1)
(test (fill! #2d() 1) 1)

(test (equal? #2d((1 2) (3 4)) (copy #2d((1 2) (3 4)))) #t)
(test (equal? #3d() #3d(((())))) #f)
(test (equal? #3d() #3d()) #t)
(test (equal? #1d() #1d()) #t)
(test (equal? #3d() #2d()) #f)
(test (equal? #3d() (copy #3d())) #t)
(test (equal? #2d((1) (2)) #2d((1) (3))) #f)
(test (equal? #2d((1) (2)) (copy #2d((1) (2)))) #t)
(test (equal? (make-vector '(3 0 1)) (make-vector '(3 0 2))) #f)
(test (eval-string "#0d()") 'error)

(let ((v #2d((1 2 3) (4 5 6))))
  (let ((v1 (v 0))
	(v2 (v 1)))
    (if (not (equal? v1 #(1 2 3)))
	(format #t ";(v 0) subvector: ~A~%" v1))
    (if (not (equal? v2 #(4 5 6)))
	(format #t ";(v 1) subvector: ~A~%" v2))
    (let ((v3 (copy v1)))
      (if (not (equal? v3 #(1 2 3)))
	  (format #t ";(v 0) copied subvector: ~A~%" v3))
      (if (not (= (length v3) 3))
	  (format #t ";(v 0) copied length: ~A~%" (length v3)))
      (if (not (equal? v3 (copy (v 0))))
	  (format #t ";(v 0) copied subvectors: ~A ~A~%" v3 (copy (v 0)))))))

(let ((v1 (make-vector '(3 2 1) #f))
      (v2 (make-vector '(3 2 1) #f)))
  (test (equal? v1 v2) #t)
  (set! (v2 0 0 0) 1)
  (test (equal? v1 v2) #f))
(test (equal? (make-vector '(3 2 1) #f) (make-vector '(1 2 3) #f)) #f)

(test (map (lambda (n) n) #2d((1 2) (3 4))) '(1 2 3 4))
(test (let ((vals ())) (for-each (lambda (n) (set! vals (cons n vals))) #2d((1 2) (3 4))) vals) '(4 3 2 1))
(test (map (lambda (x y) (+ x y)) #2d((1 2) (3 4)) #1d(4 3 2 1)) '(5 5 5 5))
(test (let ((vals ())) (for-each (lambda (x y) (set! vals (cons (+ x y) vals))) #2d((1 2) (3 4)) #1d(4 3 2 1)) vals) '(5 5 5 5))

(let ((v #2d((#(1 2) #(3 4)) (#2d((5 6) (7 8)) #2d((9 10 11) (12 13 14))))))
  (test (v 0 0) #(1 2))
  (test (v 0 1) #(3 4))
  (test (v 1 0) #2d((5 6) (7 8)))
  (test (v 1 1) #2d((9 10 11) (12 13 14)))
  (test ((v 1 0) 0 1) 6)
  (test ((v 0 1) 1) 4)
  (test ((v 1 1) 1 2) 14))

(let ((v #2d((#((1) #(2)) #(#(3) (4))) (#2d(((5) #(6)) (#(7) #(8))) #2d((#2d((9 10) (11 12)) (13)) (14 15))))))
  (test (v 0 0) #((1) #(2)))
  (test (v 0 1) #(#(3) (4)))
  (test (v 1 0) #2d(((5) #(6)) (#(7) #(8))))
  (test (v 1 1) #2d((#2d((9 10) (11 12)) (13)) (14 15)))
  (test ((v 1 0) 0 1) #(6))
  (test (((v 1 0) 0 1) 0) 6)
  (test ((v 0 1) 1) '(4))
  (test (((v 1 1) 0 0) 1 0) 11))


(test (let ((V #2d((1 2 3) (4 5 6)))) (V 0)) #(1 2 3))
(test (let ((V #2d((1 2 3) (4 5 6)))) (V 1)) #(4 5 6))
(test (let ((V #2d((1 2 3) (4 5 6)))) (V 2)) 'error)
(test (let ((V #2d((1 2 3) (4 5 6)))) (set! (V 1) 0)) 'error)
(test (let ((V #2d((1 2 3) (4 5 6)))) (let ((V1 (V 0))) (set! (V1 1) 32) V)) '#2d((1 32 3) (4 5 6)))
(test (let ((V #2d((1 2 3) (4 5 6)))) (let ((V1 (V 0))) (set! (V1 3) 32) V)) 'error)

(test (let ((V '#3d(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (V 1)) '#2d((7 8 9) (10 11 12)))
(test (let ((V '#3d(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (V 1 1)) #(10 11 12))
(test (let ((V '#3d(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (V 0 1)) #(4 5 6))
(test (let ((V '#3d(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (V 2 1)) 'error)
(test (let ((V '#3d(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((V 0) 1)) #(4 5 6))
(test (let ((V '#3d(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (((V 0) 1) 1) 32) V) '#3d(((1 2 3) (4 32 6)) ((7 8 9) (10 11 12))))
(test (let ((V '#3d(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (vector-set! V 0 1 1 32) V) '#3d(((1 2 3) (4 32 6)) ((7 8 9) (10 11 12))))
(test (let ((V '#3d(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (vector-set! V 1 1 0 32) V) '#3d(((1 2 3) (4 5 6)) ((7 8 9) (32 11 12))))
(test (let ((V '#3d(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (vector-length (V 1))) 6)
(test (let ((V '#3d(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (vector-dimensions (V 1))) '(2 3))
(test (let ((V '#3d(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (vector-length (V 0 1))) 3)
(test (let ((V '#3d(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (vector-dimensions (V 0 1))) '(3))
(test (let ((V '#3d(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12)))) (one 1) (zero 0))
	(let ((V1 (V one zero))
	      (sum 0))
	  (for-each (lambda (n) (set! sum (+ sum n))) V1)
	  sum))
      24) ; 7 8 9
(test (let ((V '#3d(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12)))) (two 2) (one 1) (zero 0))
	(let ((V10 (V one zero))
	      (V00 (V zero zero))
	      (V01 (V zero one))
	      (V11 (V one one))
	      (sum 0))
	  (for-each (lambda (n0 n1 n2 n3) (set! sum (+ sum n0 n1 n2 n3))) V00 V01 V10 V11)
	  sum))
      78)

(let-temporarily (((*s7* 'print-length) 32))
  (test (object->string (make-vector '(8 8) 0)) "#2d((0 0 0 0 0 0 0 0) (0 0 0 0 0 0 0 0) (0 0 0 0 0 0 0 0) (0 0 0 0 0 0 0 0)...)")
  (test (object->string (make-vector 64 0)) "#(0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ...)")
  (test (object->string (make-vector 32 0)) "#(0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0)")
  (test (object->string (make-vector 33 0)) "#(0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ...)")
  (test (object->string (make-vector '(8 4) 0)) "#2d((0 0 0 0) (0 0 0 0) (0 0 0 0) (0 0 0 0) (0 0 0 0) (0 0 0 0) (0 0 0 0) (0 0 0 0))"))

(let-temporarily (((*s7* 'print-length) 1024))
  (test (object->string (make-vector '(2 1 2 1 2 1 2 1 2 1 2 1 2 1) 0)) "#14d((((((((((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))) (((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))))) (((((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))) (((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))))))) (((((((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))) (((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))))) (((((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))) (((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))))))))) (((((((((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))) (((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))))) (((((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))) (((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))))))) (((((((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))) (((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))))) (((((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))) (((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))))))))))")

  (test (object->string (make-vector '(16 1 1 1 1 1 1 1 1 1 1 1 1 1) 0)) "#14d((((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))))")

;;; now see if our shared vector has survived...
  (test (and (vector? check-subvector-after-gc)
	     (= (length check-subvector-after-gc) 6)
	     (do ((i 0 (+ i 1))
		  (happy #t))
		 ((= i 6) happy)
	       (if (or (not (pair? (check-subvector-after-gc i)))
		       (not (equal? (check-subvector-after-gc i) (cons 3 i))))
		   (set! happy #f))))
	#t)
  (set! check-subvector-after-gc #f))



;;; -------- circular structures -------

;;; here's an oddity:

(let ((l1 (make-list 1 #f))
      (l2 (make-list 3 #f)))
  (set-cdr! l1 l1)
  (set-cdr! (list-tail l2 2) l2)
  (test (equal? l1 l2) #t))  ; but (eq? l1 (cdr l1)): #t, and (eq? l2 (cdr l2)): #f

(let ((l1 (make-list 1 #f))
      (l2 (make-list 3 #f)))
  (set-car! l1 #t)
  (set-car! l2 #t)
  (set-cdr! l1 l1)
  (set-cdr! (list-tail l2 2) l2)
  (test (equal? l1 l2) #f))

;;; Guile agrees on the first, but hangs on the second
;;; CL says the first is false, but hangs on the second
;;; r7rs agrees with s7 here, to my dismay.

;;; other cases:
(let ((l1 (list #f #f))
      (l2 (list #f #f)))
  (set-cdr! l1 l1)
  (set-cdr! (cdr l2) l2)
  (test (equal? l1 l2) #t))

(let ((l1 (list #f #f #f))
      (l2 (list #f #f #f)))
  (set-cdr! (cdr l1) l1)
  (set-cdr! (cddr l2) l2)
  (test (equal? l1 l2) #t)) ; r7rs says #t I think, this was #f until 16-Jan-20

(let ((l1 (list #f #f #f))
      (l2 (list #f #f #f)))
  (set-cdr! (cdr l1) l1)
  (set-cdr! (cddr l2) (cdr l2))
  (test (equal? l1 l2) #t))

;;; Gauche says #t #f #t #t #t, as does chibi
;;; Guile-2.0 hangs on all, as does Chicken

(let ((l1 (list #t #f #f))
      (l2 (list #t #f #t)))
  (set-cdr! (cdr l1) l1)
  (set-cdr! (cddr l2) (cdr l2))
  (test (equal? l1 l2) #t))

(let ((l1 (list #t #f #f))
      (l2 (list #t #f #f #t)))
  (set-cdr! (cddr l1) l1)
  (set-cdr! (cdddr l2) (cdr l2))
  (test (equal? l1 l2) #t))


;;; cyclic-sequences

(define* (make-circular-list n init)
  (let ((l (make-list n init)))
    (set-cdr! (list-tail l (- n 1)) l)))

(define (cyclic? obj) (not (null? (cyclic-sequences obj))))

(for-each
 (lambda (arg)
   (test (cyclic? arg) #f))
  (list "hi" "" #\null #\a () #() 1 '(1 . 2) (cons #\a #\b) #f 'a-symbol abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))
       (let ((x '(1 2))) (list x x))
       (let ((x #(1 2))) (vector x x))
       (let ((x "a")) (list (vector x) x))
       (let ((x (hash-table 'a 1))) (vector x (list x x) (inlet 'b x)))
       (let ((x '(1))) (let ((y (list x))) (list x (list y))))))

(test (cyclic-sequences) 'error)
(test (cyclic-sequences (list 1 2) (list 3 4)) 'error)

(test (let ((y (make-circular-list 3))) (let ((x (cyclic-sequences y))) (list (length x) (eq? (car x) y)))) '(1 #t))
(test (let ((y (make-circular-list 3))) (let ((x (cyclic-sequences (vector y)))) (list (length x) (eq? (car x) y)))) '(1 #t))
(test (let ((y (make-circular-list 3))) (let ((x (cyclic-sequences (list y (vector y))))) (list (length x) (eq? (car x) y)))) '(1 #t))
(test (let* ((y (list 1)) (x (vector y))) (set! (y 0) x) (eq? x (car (cyclic-sequences x)))) #t)
(test (let ((x (hash-table 'a (make-circular-list 1)))) (eq? (car (cyclic-sequences x)) (x 'a))) #t)
(test (let ((x (list (make-circular-list 1) (make-circular-list 2)))) (length (cyclic-sequences x))) 2)
(test (let ((l1 '(1))) (let ((l2 (cons l1 l1))) (cyclic-sequences l2))) ())
(test (let ((l1 '(1))) (let ((l2 (list l1 l1))) (cyclic-sequences l2))) ())
(test (let ((y '(1)))
	(let ((x (list (make-circular-list 1) y y)))
	  (set-cdr! (cddr x) (cdr x))
	  (let ((z (cyclic-sequences x)))
	    (list (length z) (and (memq (cdr x) z) #t))))) ; "and" here just to make the result easier to check
      '(2 #t))
(test (let ((z (vector 1 2)))
	(let ((y (list 1 z 2)))
	  (let ((x (hash-table 'x y)))
	    (set! (z 1) x)
	    (length (cyclic-sequences z)))))
      1)
(test (let ((x '(1 2)))
	(let ((y (list x x)))
	  (let ((z (vector x y)))
	    (null? (cyclic-sequences z)))))
      #t)
(test (let ((v (vector 1 2 3 4)))
	(let ((lst (list 1 2)))
	  (set-cdr! (cdr lst) lst)
	  (set! (v 0) v)
	  (set! (v 3) lst)
	  (length (cyclic-sequences v))))
      2)

(test (infinite? (length (make-circular-list 3))) #t)
(test (object->string (make-circular-list 3)) "#1=(#f #f #f . #1#)")

(let ((lst (list 1 2 3)))
   (set! (cdr (cddr lst)) lst)
   (test (apply + lst) 'error)
   (test (cyclic? lst) #t)
   (test (eq? (car (cyclic-sequences lst)) lst) #t))

(let ((l1 (list 1)))
  (test (object->string (list l1 1 l1)) "((1) 1 (1))") ; was "(#1=(1) 1 #1#)"
  (test (cyclic? (list l1 1 l1)) #f))

(let ((lst (list 1 2)))
   (set! (cdr (cdr lst)) (cdr lst))
   (test (object->string lst) "(1 . #1=(2 . #1#))")
   (test-wi (object->string lst :readable)
	 "(let ((<1> (list 2)))
            (set-cdr! <1> <1>)
            (let ((<L> (list 1)))
              (set-cdr! <L> <1>)
              <L>))"))       ; (1 . #1=(2 . #1#))

(let ((lst (list 1 2 3)))
   (set! (cdr (cddr lst)) lst)
   (test (object->string (append '(1) lst)) "(1 . #1=(1 2 3 . #1#))"))
(let ((lst (list 1 2 3)))
   (set! (cdr (cddr lst)) lst)
   (test (append lst ()) 'error))

(let ((lst (list 1 2 3)))
   (set! (cdr (cddr lst)) lst)
   (test (sort! lst <) 'error))

(let ((lst (list 1 2 3)))
   (set! (cdr (cddr lst)) lst)
   (test (object->string (list lst)) "(#1=(1 2 3 . #1#))"))

(let ((lst (list 1 2 3)))
   (set! (cdr (cddr lst)) lst)
   (test (object->string (make-list 4 lst)) "(#1=(1 2 3 . #1#) #1# #1# #1#)"))

(let ((lst (list 1 2 3)))
   (set! (cdr (cddr lst)) lst)
   (test (object->string (vector lst lst)) "#(#1=(1 2 3 . #1#) #1#)"))

(let ((lst `(+ 1 2 3)))
   (set! (cdr (cdddr lst)) (cddr lst))
   (test (object->string lst) "(+ 1 . #1=(2 3 . #1#))"))


(let ((x (list 1 2)))
  (test (equal? x x) #t)
  (test (equal? x (cdr x)) #f)
  (test (equal? x ()) #f))
(let ((x (list 1 (list 2 3) (list (list 4 (list 5)))))
      (y (list 1 (list 2 3) (list (list 4 (list 5))))))
  (test (equal? x y) #t))
(let ((x (list 1 (list 2 3) (list (list 4 (list 5)))))
      (y (list 1 (list 2 3) (list (list 4 (list 5) 6)))))
  (test (equal? x y) #f))

(test (length ()) 0)
(test (length (cons 1 2)) -1)
(test (length '(1 2 3)) 3)

(let ((lst1 (list 1 2)))
  (test (length lst1) 2)
  (list-set! lst1 0 lst1)
  (test (length lst1) 2) ; its car is a circular list, but it isn't
  (test (eq? ((cyclic-sequences lst1) 0) lst1) #t)
  (test (list->string lst1) 'error)
  (let ((lst2 (list 1 2)))
    (set-car! lst2 lst2)
    (test (equal? lst1 lst2) #t)
    (test (equivalent? lst1 lst2) #t)
    (test (eq? lst1 lst2) #f)
    (test (eqv? lst1 lst2) #f)
    (test (pair? lst1) #t)
    (test (null? lst1) #f)
    (test (car lst2) lst2)
    (test (car lst1) lst1)
    (test (let ()
	    (fill! lst1 32)
	    lst1)
	  '(32 32))))

(let ((lst1 (list 1)))
  (test (length lst1) 1)
  (set-cdr! lst1 lst1)
  (test (infinite? (length lst1)) #t)
  (test (eq? (cdr ((cyclic-sequences lst1) 0)) lst1) #t)
  (test (null? lst1) #f)
  (test (pair? lst1) #t)
  (let ((lst2 (cons 1 ())))
    (set-cdr! lst2 lst2)
    (test (equal? lst1 lst2) #t)
    (test (equivalent? lst1 lst2) #t)
    (set-car! lst2 0)
    (test (equal? lst1 lst2) #f)
    (test (equivalent? lst1 lst2) #f)
    (test (infinite? (length lst2)) #t)))

(let ((lst1 (list 1)))
  (set-cdr! lst1 lst1)
  (test (list-tail lst1 0) lst1)
  (test (list-tail lst1 3) lst1)
  (test (list-tail lst1 10) lst1))

(let ((lst1 (let ((lst (list 'a)))
	      (set-cdr! lst lst)
	      lst)))
  (test (apply lambda lst1 (list 1)) 'error)) ; lambda parameter 'a is used twice in the lambda argument list !

(let ((lst1 (list 1))
      (lst2 (list 1)))
  (set-car! lst1 lst2)
  (set-car! lst2 lst1)
  (test (equal? lst1 lst2) #t)
  (test (equivalent? lst1 lst2) #t)
  (test (length lst1) 1)
  (let ((lst3 (list 1)))
    (test (equal? lst1 lst3) #f)
    (test (equivalent? lst1 lst3) #f)
    (set-cdr! lst3 lst3)
    (test (equal? lst1 lst3) #f)
    (test (equivalent? lst1 lst3) #f)))

(let ((lst1 (list 'a 'b 'c)))
  (set! (cdr (cddr lst1)) lst1)
  (test (infinite? (length lst1)) #t)
  (test (memq 'd lst1) #f)
  (test (memq 'a lst1) lst1)
  (test (memq 'b lst1) (cdr lst1)))

(let ((lst1 (list 1 2 3)))
  (list-set! lst1 1 lst1)
  (test (object->string lst1) "#1=(1 #1# 3)"))

(let ((lst1 (let ((lst (list 1)))
	      (set-cdr! lst lst)
	      lst)))
  (test (list-ref lst1 9223372036854775807) 'error)
  (test (list-set! lst1 9223372036854775807 2) 'error)
  (test (list-tail lst1 9223372036854775807) 'error)
  (test (make-vector lst1 9223372036854775807) 'error)
  (let-temporarily (((*s7* 'safety) 1))
    (test (not (member (map (lambda (x) x) lst1) '(() (1)))) #f) (newline) ; geez -- just want to allow two possible ok results, so "not" makes it boolean
    (test (not (member (map (lambda (x y) x) lst1 lst1) '(() (1)))) #f)
    (test (for-each (lambda (x) x) lst1) #<unspecified>) ; was 'error
    (test (for-each (lambda (x y) x) lst1 lst1) #<unspecified>) ; was 'error
    (test (not (member (map (lambda (x y) (+ x y)) lst1 '(1 2 3)) (list () '(2)))) #f)))

(test (map abs (let ((lst (list 0))) (set! (cdr lst) lst))) '(0))
(test (map (lambda (x) x) (let ((lst (list 0))) (set! (cdr lst) lst))) '(0))
(test (map abs (let ((lst (list 0 1))) (set-cdr! (cdr lst) lst))) '(0 1 0))
(test (map (lambda (x) x) (let ((lst (list 0 1))) (set-cdr! (cdr lst) lst))) '(0 1))

(let ((lst1 (list 1 -1)))
  (set-cdr! (cdr lst1) lst1)
  (let ((vals (map * '(1 2 3 4) lst1)))
    (test vals '(1 -2 3)))) ; was '(1 -2 3 -4), then later (1 -2) -- as in other cases above, map/for-each stop when a cycle is encountered

(test (let ((lst '(a b c)))
	(set! (cdr (cddr lst)) lst)
	(map cons lst '(0 1 2 3 4 5)))
      '((a . 0) (b . 1) (c . 2) (a . 3) (b . 4))) ; as above

(test (object->string (let ((l1 (list 0 1))) (set! (l1 1) l1) (copy l1))) "(0 #1=(0 #1#))")

;;; this changed 11-Mar-15
;;;(test (object->string (let ((lst (list 1 2))) (set! (cdr lst) lst) (copy lst))) "(1 . #1=(1 . #1#))")
(test (object->string (let ((lst (list 1 2))) (set! (cdr lst) lst) (copy lst))) "#1=(1 . #1#)")
(test (object->string (let ((lst (list 1 2))) (set! (cdr lst) lst) lst))        "#1=(1 . #1#)")

(test (object->string (let ((l1 (list 1 2))) (copy (list l1 4 l1)))) "((1 2) 4 (1 2))") ; was "(#1=(1 2) 4 #1#)"
;;;(test (object->string (let ((lst (list 1 2 3))) (set! (cdr (cddr lst)) (cdr lst)) (copy lst))) "(1 2 3 . #1=(2 3 . #1#))")
(test (object->string (let ((lst (list 1 2 3))) (set! (cdr (cddr lst)) (cdr lst)) (copy lst))) "(1 . #1=(2 3 . #1#))")
(test (object->string (let ((lst (list 1 2 3))) (set! (cdr (cddr lst)) (cdr lst)) lst))        "(1 . #1=(2 3 . #1#))")

(test (object->string (let ((lst (list 1 2 3))) (set! (cdr (cddr lst)) (cddr lst)) (copy lst))) "(1 2 . #1=(3 . #1#))")
(test (object->string (let ((lst (list 1 2 3))) (set! (cdr (cddr lst)) (cddr lst)) lst))        "(1 2 . #1=(3 . #1#))")

;;;(test (object->string (let ((lst (list 1 2 3 4))) (set! (cdr (cdddr lst)) (cddr lst)) (copy lst))) "(1 2 3 4 . #1=(3 4 . #1#))")
(test (object->string (let ((lst (list 1 2 3 4))) (set! (cdr (cdddr lst)) (cddr lst)) (copy lst))) "(1 2 . #1=(3 4 . #1#))")
(test (object->string (let ((lst (list 1 2 3 4))) (set! (cdr (cdddr lst)) (cddr lst)) lst))        "(1 2 . #1=(3 4 . #1#))")

;;;(test (object->string (let ((lst (list 1 2 3 4))) (set! (cdr (cdddr lst)) (cdr lst)) (copy lst))) "(1 2 3 4 . #1=(2 3 4 . #1#))")
(test (object->string (let ((lst (list 1 2 3 4))) (set! (cdr (cdddr lst)) (cdr lst)) (copy lst))) "(1 . #1=(2 3 4 . #1#))")
(test (object->string (let ((lst (list 1 2 3 4))) (set! (cdr (cdddr lst)) (cdr lst)) lst))        "(1 . #1=(2 3 4 . #1#))")

(test (object->string (vector (let ((lst (list 1))) (set-cdr! lst lst)))) "#(#1=(1 . #1#))")
(test (object->string (let ((lst (list 1 2))) (set! (cdr (cdr lst)) lst) (set! (car lst) (vector lst)) lst)) "#1=(#(#1#) 2 . #1#)")

(test-wi (object->string (vector (let ((lst (list 1))) (set-cdr! lst lst))) :readable) ; #(#1=(1 . #1#))
      "(let ((<1> (list 1)))
         (set-cdr! <1> <1>)
         (vector <1>))")  ; #(#1=(1 . #1#))

(test-wi (object->string (let ((lst (list 1 2))) (set! (cdr (cdr lst)) lst) (set! (car lst) (vector lst)) lst) :readable) ; #1=(#(#1#) 2 . #1#)
      "(let ((<1> (list #f 2)))
         (set-car! <1> (vector <1>))
         (set-cdr! (cdr <1>) <1>)
         <1>)")  ;  #1=(#(#1#) 2 . #1#)

(test-wi (let ((v (vector 1 2))) (set! (v 0) v) (object->string v :readable)) ; #1=#(#1# 2)
      "(let ((<1> (vector #f 2)))
         (set! (<1> 0) <1>)
         <1>)")  ;  #1=#(#1# 2)

(test-wi (let ((v (make-vector '(2 2) 0))) (set! (v 1 1) v) (object->string v :readable)) ; #1=#2d((0 0) (0 #1#))
      "(let ((<1> (subvector (vector 0 0 0 #f) 0 4 '(2 2))))
         (set! (<1> 1 1) <1>)
         <1>)")  ;  #1=#2d((0 0) (0 #1#))

(test (reverse '(1 2 (3 4))) '((3 4) 2 1))
(test (reverse '(1 2 3)) '(3 2 1))
(test (reverse ()) ())
(test (let ((lst (list 1 2 3))) (set! (lst 2) lst) (object->string (reverse lst))) "(#1=(1 2 #1#) 2 1)")
(test (let ((l1 (cons 1 ()))) (set-cdr! l1 l1) (object->string (reverse l1))) "(#1=(1 . #1#) 1 1 1)")


(test (equal? (vector 0) (vector 0)) #t)
(test (equal? (vector 0 #\a "hi" (list 1 2 3)) (vector 0 #\a "hi" (list 1 2 3))) #t)
(test (let ((v (vector 0))) (equal? (vector v) (vector v))) #t)

(let ((v1 (make-vector 1 0)))
  (set! (v1 0) v1)
  (test (vector? v1) #t)
  (let ((v2 (vector 0)))
    (vector-set! v2 0 v2)
    (test (vector-length v1) 1)
    (test (equal? v1 v2) #t)
    (test (equal? (vector-ref v1 0) v1) #t)
    (test (equal? (vector->list v1) (list v1)) #t)
    (vector-fill! v1 0)
    (test (equal? v1 (vector 0)) #t)
    (let ((v3 (copy v2)))
      (test (equal? v2 v3) #t)
      (vector-set! v3 0 0)
      (test (equal? v3 (vector 0)) #t))
    ))

(let ((v1 (make-vector 1 0))
      (v2 (vector 0)))
  (set! (v1 0) v2)
  (set! (v2 0) v1)
  (test (equal? v1 v2) #t))

(test (vector? (let ((v (vector 0))) (set! (v 0) v) (v 0 0 0 0))) #t) ; ?

(let* ((l1 (list 1 2))
       (v1 (vector 1 2))
       (l2 (list 1 l1 2))
       (v2 (vector l1 v1 l2)))
  (vector-set! v1 0 v2)
  (list-set! l1 1 l2)
  (test (equal? v1 v2) #f))

(let ((v1 (make-vector 1 0)))
  (set! (v1 0) v1)
  (let ((v2 (vector 0)))
    (vector-set! v2 0 v2)
    (test (equal? v1 v2) #t)))

(let ((v1 (make-vector 1 0)))
  (set! (v1 0) v1)
  (test (eq? ((cyclic-sequences v1) 0) v1) #t)
  (test (object->string v1) "#1=#(#1#)"))

(let ((l1 (cons 0 ())))
  (set-cdr! l1 l1)
  (test (list->vector l1) 'error))

(let ((lst (list "nothing" "can" "go" "wrong")))
  (let ((slst (cddr lst))
	(result ()))
    (set! (cdr (cdddr lst)) slst)
    (test (do ((i 0 (+ i 1))
	       (l lst (cdr l)))
	      ((or (null? l) (= i 12))
	       (reverse result))
	    (set! result (cons (car l) result)))
	  '("nothing" "can" "go" "wrong" "go" "wrong" "go" "wrong" "go" "wrong" "go" "wrong"))))

#|
;;; here is a circular function
(let ()
  (define (cfunc)
    (begin
      (display "cfunc! ")
      #f))

  (let ((clst (procedure-source cfunc)))
    (set! (cdr (cdr (car (cdr (cdr clst)))))
	  (cdr (car (cdr (cdr clst))))))

  (cfunc))
|#

(test (let ((l (list 1 2)))
	(list-set! l 0 l)
	(string=? (object->string l) "#1=(#1# 2)"))
      #t)
(test (let ((lst (list 1)))
	(set! (car lst) lst)
	(set! (cdr lst) lst)
	(string=? (object->string lst) "#1=(#1# . #1#)"))
      #t)
(test (let ((lst (list 1)))
	(set! (car lst) lst)
	(set! (cdr lst) lst)
	(equal? (car lst) (cdr lst)))
      #t)
(test (let ((lst (cons 1 2)))
	(set-cdr! lst lst)
	(string=? (object->string lst) "#1=(1 . #1#)"))
      #t)
(test (let ((lst (cons 1 2)))
	(set-car! lst lst)
	(string=? (object->string lst) "#1=(#1# . 2)"))
      #t)
(test (let ((lst (cons (cons 1 2) 3)))
	(set-car! (car lst) lst)
	(string=? (object->string lst) "#1=((#1# . 2) . 3)"))
      #t)
(test (let ((v (vector 1 2)))
	(vector-set! v 0 v)
	(string=? (object->string v) "#1=#(#1# 2)"))
      #t)
(test (let* ((l1 (list 1 2)) (l2 (list l1)))
	(list-set! l1 0 l1)
	(string=? (object->string l2) "(#1=(#1# 2))"))
      #t)

(test (let ((lst (list 1 2 3))) (set! (cdr (cdr (cdr lst))) lst) (object->string lst)) "#1=(1 2 3 . #1#)")
(test (let ((lst (list 1 2 3))) (set! (cdr (cdr (cdr lst))) (cdr lst)) (object->string lst)) "(1 . #1=(2 3 . #1#))")
(test (let ((lst (list 1 2 3))) (set! (cdr (cdr (cdr lst))) (cdr (cdr lst))) (object->string lst)) "(1 2 . #1=(3 . #1#))")
(test (let ((lst (list 1 2 3))) (set! (car lst) (cdr lst)) (object->string lst)) "((2 3) 2 3)") ; was "(#1=(2 3) . #1#)"
(test (let ((lst (list 1 2 3))) (set! (car (cdr lst)) (cdr lst)) (object->string lst)) "(1 . #1=(#1# 3))")
(test (let ((lst (list 1 2 3))) (set! (car (cdr lst)) lst) (object->string lst)) "#1=(1 #1# 3)")
(test (let ((l1 (list 1))) (let ((l2 (list l1 l1))) (object->string l2))) "((1) (1))") ; was "(#1=(1) #1#)"

(let ((v (vector #f)))
  (vector-set! v 0 v)
  (test (with-output-to-string
	  (lambda ()
	    (display v)
	    (newline)
	    (display (subvector v 0 1))
	    (newline)))
	"#1=#(#1#)\n#(#1=#(#1#))\n"))

(let ((v (make-vector '(2 2) #f)))
  (fill! v v)
  (test (with-output-to-string
	  (lambda ()
	    (display v)
	    (newline)
	    (display (subvector v 0 4))
	    (newline)))
	"#1=#2d((#1# #1#) (#1# #1#))\n#(#1=#2d((#1# #1#) (#1# #1#)) #1# #1# #1#)\n"))

(test (let* ((v1 (vector 1 2)) (v2 (vector v1)))
	(vector-set! v1 1 v1)
	(string=? (object->string v2) "#(#1=#(1 #1#))"))
      #t)
(test (let ((v1 (make-vector 3 1)))
	(vector-set! v1 0 (cons 3 v1))
	(string=? (object->string v1) "#1=#((3 . #1#) 1 1)"))
      #t)
(test (let ((h1 (make-hash-table 11))
	    (old-print-length (*s7* 'print-length)))
	(set! (*s7* 'print-length) 32)
	(hash-table-set! h1 "hi" h1)
	(let ((result (object->string h1)))
	  (set! (*s7* 'print-length) old-print-length)
	  (let ((val (string=? result "#1=(hash-table \"hi\" #1#)")))
	    (unless val
	      (format #t ";hash display:~%  ~A~%" result))
	    val)))
      #t)

(test (let* ((l1 (list 1 2))
	     (v1 (vector 1 2))
	     (l2 (list 1 l1 2))
	     (v2 (vector l1 v1 l2)))
	(vector-set! v1 0 v2)
	(list-set! l1 1 l2)
	(string=? (object->string v2) "#2=#(#1=(1 #3=(1 #1# 2)) #(#2# 2) #3#)"))
      #t)

(test (let ((l1 (list 1 2))
	    (l2 (list 1 2)))
	(set! (car l1) l2)
	(set! (car l2) l1)
	(object->string (list l1 l2)))
      "(#1=(#2=(#1# 2) 2) #2#)")

(test (let* ((l1 (list 1 2))
	     (l2 (list 3 4))
	     (l3 (list 5 l1 6 l2 7)))
	(set! (cdr (cdr l1)) l1)
	(set! (cdr (cdr l2)) l2)
	(string=? (object->string l3) "(5 #1=(1 2 . #1#) 6 #2=(3 4 . #2#) 7)"))
      #t)
(test (let* ((lst1 (list 1 2))
	     (lst2 (list (list (list 1 (list (list (list 2 (list (list (list 3 (list (list (list 4 lst1 5))))))))))))))
	(set! (cdr (cdr lst1)) lst1)
	(string=? (object->string lst2) "(((1 (((2 (((3 (((4 #1=(1 2 . #1#) 5))))))))))))"))
      #t)


(test (equal? '(a) (list 'a)) #t)
(test (equal? '(a b . c) '(a b . c)) #t)
(test (equal? '(a b (c . d)) '(a b (c . d))) #t)
(test (equal? (list "hi" "hi" "hi") '("hi" "hi" "hi")) #t)
(let ((l1 (list "hi" "hi" "hi"))
      (l2 (list "hi" "hi" "hi")))
  (fill! l1 "ho")
  (test (equal? l1 l2) #f)
  (fill! l2 (car l1))
  (test (equal? l1 l2) #t))
(let ((lst (list 1 2 3 4)))
  (fill! lst "hi")
  (test (equal? lst '("hi" "hi" "hi" "hi")) #t))
(let ((vect (vector 1 2 3 4)))
  (fill! vect "hi")
  (test (equal? vect #("hi" "hi" "hi" "hi")) #t))
(let ((lst (list 1 2 (list 3 4) (list (list 5) 6))))
  (test (equal? lst '(1 2 (3 4) ((5) 6))) #t)
  (fill! lst #f)
  (test (equal? lst '(#f #f #f #f)) #t))
(let ((lst (list 1 2 3 4)))
  (set! (cdr (cdddr lst)) lst)
  (test (equal? lst lst) #t)
  (test (eq? lst lst) #t)
  (test (eqv? lst lst) #t)
  (fill! lst #f)
  (test (object->string lst) "#1=(#f #f #f #f . #1#)")
  (let ((l1 (copy lst)))
    (test (equal? lst l1) #t)
    (test (eq? lst l1) #f)
    (test (eqv? lst l1) #f)))


(let ((lst '(#\( #\) #\* #\+ #\, #\- #\. #\/ #\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9 #\: #\; #\< #\= #\> #\? #\@ #\A #\B #\C #\D #\E #\F #\G #\H #\I #\J #\K #\L #\M #\N #\O #\P #\Q #\R #\S #\T #\U #\V #\W #\X #\Y #\Z #\[ #\\ #\] #\^ #\_ #\` #\a #\b #\c #\d #\e #\f #\g #\h #\i #\j #\k #\l #\m #\n #\o #\p #\q #\r #\s #\t #\u #\v #\w #\x #\y #\z #\{ #\| #\} #\~)))
  (let ((str (apply string lst)))
    (let ((lstr (list->string lst)))
      (let ((strl (string->list str)))
	(test (eq? str str) #t)
	(test (eq? str lstr) #f)
	(test (eqv? str str) #t)
	(test (eqv? str lstr) #f)
	(test (equal? str lstr) #t)
	(test (equal? str str) #t)
	(test (eq? lst strl) #f)
	(test (eqv? lst strl) #f)
	(test (equal? lst strl) #t)
	(let ((l2 (copy lst))
	      (s2 (copy str)))
	  (test (eq? l2 lst) #f)
	  (test (eq? s2 str) #f)
	  (test (eqv? l2 lst) #f)
	  (test (eqv? s2 str) #f)
	  (test (equal? l2 lst) #t)
	  (test (equal? s2 str) #t))))))


(let ((vect #(#\( #\) #\* #\+ #\, #\- #\. #\/ #\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9 #\: #\; #\< #\= #\> #\? #\@ #\A #\B #\C #\D #\E #\F #\G #\H #\I #\J #\K #\L #\M #\N #\O #\P #\Q #\R #\S #\T #\U #\V #\W #\X #\Y #\Z #\[ #\\ #\] #\^ #\_ #\` #\a #\b #\c #\d #\e #\f #\g #\h #\i #\j #\k #\l #\m #\n #\o #\p #\q #\r #\s #\t #\u #\v #\w #\x #\y #\z #\{ #\| #\} #\~)))
  (let ((lst (vector->list vect)))
    (let ((vect1 (list->vector lst)))
	(test (eq? lst lst) #t)
	(test (eq? lst vect) #f)
	(test (eqv? lst lst) #t)
	(test (eqv? lst vect) #f)
	(test (equal? vect1 vect) #t)
	(test (equal? lst lst) #t)
	(test (eq? vect vect1) #f)
	(test (eqv? vect vect1) #f)
	(test (equal? vect vect1) #t)
	(let ((l2 (copy vect))
	      (s2 (copy lst)))
	  (test (eq? l2 vect) #f)
	  (test (eq? s2 lst) #f)
	  (test (eqv? l2 vect) #f)
	  (test (eqv? s2 lst) #f)
	  (test (equal? l2 vect) #t)
	  (test (equal? s2 lst) #t)))))

(let* ((vals (list "hi" #\A 1 'a #(1) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand (log 0)
		   3.14 3/4 1.0+1.0i #\f '(1 . 2)))
       (vlen (length vals)))
  (define (fillv size vect)
    (do ((n 0 (+ n 1)))
	((= n size))
      (let ((choice (random 4))
	    (len (random 4)))
	(if (= choice 0)
	    (let ((v (make-vector len)))
	      (do ((k 0 (+ k 1)))
		  ((= k len))
		(vector-set! v k (list-ref vals (random vlen))))
	      (vector-set! vect n v))
	    (if (= choice 1)
		(let ((lst (make-list len #f)))
		  (do ((k 0 (+ k 1)))
		      ((= k len))
		    (list-set! lst k (list-ref vals (random vlen))))
		  (vector-set! vect n lst))
		(vector-set! vect n (list-ref vals (random vlen))))))))
  (do ((i 0 (+ i 1)))
      ((= i 20))
    (let* ((size (max 1 (random 20)))
	   (vect (make-vector size ())))
      (fillv size vect)
      (test (eq? vect vect) #t)
      (test (eqv? vect vect) #t)
      (test (equal? vect vect) #t)
      (let ((lst1 (vector->list vect)))
	(let ((lst2 (copy lst1)))
	  (test (eq? lst1 lst2) #f)
	  (test (eqv? lst1 lst2) #f)
	  (test (equal? lst1 lst2) #t))))))

(let* ((lst1 (list 1 2 3))
       (vec1 (vector 1 2 lst1)))
  (list-set! lst1 2 vec1)
  (let* ((lst2 (list 1 2 3))
	 (vec2 (vector 1 2 lst2)))
    (list-set! lst2 2 vec2)
    (test (equal? lst1 lst2) #t)
    (test (equal? vec1 vec2) #t)
    (vector-set! vec1 1 vec1)
    (test (equal? lst1 lst2) #f)
    (test (equal? vec1 vec2) #f)
    ))

(let* ((base (list #f))
       (lst1 (list 1 2 3))
       (vec1 (vector 1 2 base)))
  (list-set! lst1 2 vec1)
  (let* ((lst2 (list 1 2 3))
	 (vec2 (vector 1 2 base)))
    (list-set! lst2 2 vec2)
    (set! (car lst1) lst1)
    (set! (car lst2) lst2)
    (set! (cdr (cddr lst1)) base)
    (set! (cdr (cddr lst2)) base)
    (test (length (cyclic-sequences lst2)) 1)
    (test (equal? lst1 lst2) #t)
    (test (equal? vec1 vec2) #t)
    (test (object->string lst1) "#1=(#1# 2 #(1 2 (#f)) #f)"))) ; was "#1=(#1# 2 #(1 2 #2=(#f)) . #2#)"

(let ((base (list 0 #f)))
  (let ((lst1 (list 1 base 2))
	(lst2 (list 1 base 2)))
    (set! (cdr (cdr base)) base)
    (test (equal? lst1 lst2) #t)))

(let ((base1 (list 0 #f))
      (base2 (list 0 #f)))
  (let ((lst1 (list 1 base1 2))
	(lst2 (list 1 base2 2)))
    (set! (cdr (cdr base1)) lst2)
    (set! (cdr (cdr base2)) lst1)
    (test (equal? lst1 lst2) #t)
    (test (object->string lst1) "#1=(1 (0 #f 1 (0 #f . #1#) 2) 2)")))

(let ()
  (define-macro (c?r path)

    (define (X-marks-the-spot accessor tree)
      (if (eq? tree 'X)
          accessor
          (and (pair? tree)
	       (or (X-marks-the-spot (cons 'car accessor) (car tree))
	           (X-marks-the-spot (cons 'cdr accessor) (cdr tree))))))

    (let ((body 'lst))
      (for-each
       (lambda (f)
	 (set! body (list f body)))
       (reverse (X-marks-the-spot () path)))

      `(dilambda
	(lambda (lst)
	  ,body)
	(lambda (lst val)
	  (set! ,body val)))))

  (define (copy-tree lis)
    (if (pair? lis)
	(cons (copy-tree (car lis))
	      (copy-tree (cdr lis)))
	lis))

  (let* ((l1 '(0 (1 (2 (3 (4 (5 (6 (7 (8))))))))))
	 (l2 (list 0 (list 1 (list 2 (list 3 (list 4 (list 5 (list 6 (list 7 (list 8))))))))))
	 (l3 (copy-tree l1))
	 (cxr (c?r (0 (1 (2 (3 (4 (5 (6 (7 (X))))))))))))
    (set! (cxr l1) 3)
    (set! (cxr l2) 4)
    (test (equal? l1 l2) #f)
    (test (equal? l1 l3) #f)
    (set! (cxr l2) 3)
    (test (cxr l2) 3)
    (test (cxr l1) 3)
    (test (cxr l3) 8)
    (test (equal? l1 l2) #t)
    (test (equal? l2 l3) #f))

  (let* ((l1 '(0 (1 (2 (3 (4 (5 (6 (7 (8))))))))))
	 (l2 (list 0 (list 1 (list 2 (list 3 (list 4 (list 5 (list 6 (list 7 (list 8))))))))))
	 (l3 (copy-tree l1))
	 (cxr (c?r (0 (1 (2 (3 (4 (5 (6 (7 (8 . X))))))))))))
    (set! (cxr l1) l1)
    (set! (cxr l2) l2)
    (test (equal? l1 l2) #t)
    (test (equal? l1 l3) #f)
    (test (object->string l2) "#1=(0 (1 (2 (3 (4 (5 (6 (7 (8 . #1#)))))))))"))

  (let* ((l1 '(0 ((((((1))))))))
	 (l2 (copy-tree l1))
	 (cxr (c?r (0 ((((((1 . X))))))))))
    (set! (cxr l1) l2)
    (set! (cxr l2) l1)
    (test (equal? l1 l2) #t))

  (let* ((l1 '(0 1 (2 3) 4 5))
	 (cxr (c?r (0 1 (2 3 . X) 4 5))))
    (set! (cxr l1) (cdr l1))
    (test (object->string l1) "(0 . #1=(1 (2 3 . #1#) 4 5))"))

  (let* ((l1 '(0 1 (2 3) 4 5))
	 (l2 '(6 (7 8 9) 10))
	 (cxr1 (c?r (0 1 (2 3 . X) 4 5)))
	 (cxr2 (c?r (6 . X)))
	 (cxr3 (c?r (6 (7 8 9) 10 . X)))
	 (cxr4 (c?r (0 . X))))
    (set! (cxr1 l1) (cxr2 l2))
    (set! (cxr3 l2) (cxr4 l1))
    (test (object->string l1) "(0 . #1=(1 (2 3 (7 8 9) 10 . #1#) 4 5))")
    (test (cadr l1) 1)
    (test (cadddr l1) 4)
    )

  (let ((l1 '((a . 2) (b . 3) (c . 4)))
	(cxr (c?r ((a . 2) (b . 3) (c . 4) . X))))
    (set! (cxr l1) (cdr l1))
    (test (assq 'a l1) '(a . 2))
    (test (assv 'b l1) '(b . 3))
    (test (assoc 'c l1) '(c . 4))
    (test (object->string l1) "((a . 2) . #1=((b . 3) (c . 4) . #1#))")
    (test (assq 'asdf l1) #f)
    (test (assv 'asdf l1) #f)
    (test (assoc 'asdf l1) #f)
    )

  (let ((l1 '(a b c d e))
	(cxr (c?r (a b c d e . X))))
    (set! (cxr l1) (cddr l1))
    (test (memq 'b l1) (cdr l1))
    (test (memv 'c l1) (cddr l1))
    (test (member 'd l1) (cdddr l1))
    (test (object->string l1) "(a b . #1=(c d e . #1#))")
    (test (memq 'asdf l1) #f)
    (test (memv 'asdf l1) #f)
    (test (member 'asdf l1) #f)
    (test (pair? (member 'd l1)) #t) ; #1=(d e c . #1#)
    )

  (let ((x 0))
    (let ((lst `(call-with-exit (lambda (return) (set! x (+ x 1)) (if (> x 10) (return x) 0)))))
      (let ((acc1 (c?r (call-with-exit (lambda (return) . X))))
	    (acc2 (c?r (call-with-exit (lambda (return) (set! x (+ x 1)) (if (> x 10) (return x) 0) . X)))))
	(set! (acc2 lst) (acc1 lst))
	(test (eval lst (curlet)) 11))))
  )


(let ()
  ;; anonymous recursion...
  (define (fc?r path)
    (define (X-marks-the-spot accessor tree)
      (if (pair? tree)
	  (or (X-marks-the-spot (cons 'car accessor) (car tree))
	      (X-marks-the-spot (cons 'cdr accessor) (cdr tree)))
	  (if (eq? tree 'X) accessor #f)))
    (let ((body 'lst))
      (for-each
       (lambda (f)
	 (set! body (list f body)))
       (reverse (X-marks-the-spot () path)))
      (let ((getter (apply lambda '(lst) body ()))
	    (setter (apply lambda '(lst val) `(set! ,body val) ())))
	(dilambda getter setter))))

  (let ((body '(if (not (pair? (cdr lst))) lst (begin (set! lst (cdr lst)) X)))) ; X is where we jump back to the start
    (let ((recurse (fc?r body)))
      (set! (recurse body) body)
      (test ((apply lambda '(lst) body ()) '(1 2 3)) '(3)))))


(let ((v #2d((1 2) (3 4))))
  (set! (v 1 0) v)
  (test (object->string v) "#1=#2d((1 2) (#1# 4))")
  (test (length v) 4)
  (test ((((v 1 0) 1 0) 1 0) 0 0) 1))

(let ((lst (list 1 2 3)))
  (set! (cdr (cddr lst)) lst)
  (test (lst 100) 2)
  (test ((cdddr (cdddr (cdddr lst))) 100) 2)
  (set! (lst 100) 32)
  (test (object->string lst) "#1=(1 32 3 . #1#)"))

(let* ((l1 (list 1 2))
       (l2 (list l1 l1)))
  (set! (l1 0) 32)
  (test (equal? l2 '((32 2) (32 2))) #t))

(let ((q (list 1 2 3 4)))
  (set! (cdr (cdddr q)) q)
  (test (car q) 1)
  (set! (car q) 5)
  (set! q (cdr q))
  (test (car q) 2)
  (test (object->string q) "#1=(2 3 4 5 . #1#)"))

(let ()
  (define make-node vector)
  (define prev (dilambda (lambda (node) (node 0)) (lambda (node val) (set! (node 0) val))))
  (define next (dilambda (lambda (node) (node 2)) (lambda (node val) (set! (node 2) val))))
  (define data (dilambda (lambda (node) (node 1)) (lambda (node val) (set! (node 1) val))))
  (let* ((head (make-node () 0 ()))
	 (cur head))
    (do ((i 1 (+ i 1)))
	((= i 8))
      (let ((next-node (make-node cur i ())))
	(set! (next cur) next-node)
	(set! cur (next cur))))
    (set! (next cur) head)
    (set! (prev head) cur)
    (test (object->string head) "#1=#(#7=#(#6=#(#5=#(#4=#(#3=#(#2=#(#8=#(#1# 1 #2#) 2 #3#) 3 #4#) 4 #5#) 5 #6#) 6 #7#) 7 #1#) 0 #8#)")
#|
    ;; in CL:
    (let* ((head (vector nil 0 nil))
	   (cur head))
      (do ((i 1 (+ i 1)))
	  ((= i 8))
	(let ((node (vector nil i nil)))
	  (setf (aref node 0) cur)
	  (setf (aref cur 2) node)
	  (setf cur node)))
      (setf (aref head 0) cur)
      (setf (aref cur 2) head)
      (format t "~A~%" head)) -> "#1=#(#2=#(#3=#(#4=#(#5=#(#6=#(#7=#(#8=#(#1# 1 #7#) 2 #6#) 3 #5#) 4 #4#) 5 #3#) 6 #2#) 7 #1#) 0 #8#)"
|#
    (let ((ahead (do ((cur head (next cur))
		      (dat () (cons (data cur) dat)))
		     ((member (data cur) dat)
		      (reverse dat)))))
      (let ((behind (do ((cur (prev head) (prev cur))
			 (dat () (cons (data cur) dat)))
			((member (data cur) dat)
			 dat))))
	(test (equal? ahead behind) #t)))))

(let ()
  (define make-node list)
  (define prev (dilambda (lambda (node) (node 0)) (lambda (node val) (set! (node 0) val))))
  (define next (dilambda (lambda (node) (node 2)) (lambda (node val) (set! (node 2) val))))
  (define data (dilambda (lambda (node) (node 1)) (lambda (node val) (set! (node 1) val))))
  (let* ((head (make-node () 0 ()))
	 (cur head))
    (do ((i 1 (+ i 1)))
	((= i 8))
      (let ((next-node (make-node cur i ())))
	(set! (next cur) next-node)
	(set! cur (next cur))))
    (set! (next cur) head)
    (set! (prev head) cur)
    (test (object->string head) "#1=(#7=(#6=(#5=(#4=(#3=(#2=(#8=(#1# 1 #2#) 2 #3#) 3 #4#) 4 #5#) 5 #6#) 6 #7#) 7 #1#) 0 #8#)")
    (let ((ahead (do ((cur head (next cur))
		      (dat () (cons (data cur) dat)))
		     ((member (data cur) dat)
		      (reverse dat)))))
      (let ((behind (do ((cur (prev head) (prev cur))
			 (dat () (cons (data cur) dat)))
			((member (data cur) dat)
			 dat))))
	(test (equal? ahead behind) #t))))
  (let* ((head (make-node () 0 ()))
	 (cur head))
    (do ((i 1 (+ i 1)))
	((= i 32))
      (let ((next-node (make-node cur i ())))
	(set! (next cur) next-node)
	(set! cur (next cur))))
    (set! (next cur) head)
    (set! (prev head) cur)
    (let-temporarily (((*s7* 'print-length) 128))
      (test (object->string head) "#1=(#31=(#30=(#29=(#28=(#27=(#26=(#25=(#24=(#23=(#22=(#21=(#20=(#19=(#18=(#17=(#16=(#15=(#14=(#13=(#12=(#11=(#10=(#9=(#8=(#7=(#6=(#5=(#4=(#3=(#2=(#32=(#1# 1 #2#) 2 #3#) 3 #4#) 4 #5#) 5 #6#) 6 #7#) 7 #8#) 8 #9#) 9 #10#) 10 #11#) 11 #12#) 12 #13#) 13 #14#) 14 #15#) 15 #16#) 16 #17#) 17 #18#) 18 #19#) 19 #20#) 20 #21#) 21 #22#) 22 #23#) 23 #24#) 24 #25#) 25 #26#) 26 #27#) 27 #28#) 28 #29#) 29 #30#) 30 #31#) 31 #1#) 0 #32#)"))))

(let ((x (list '+ 1)))
  (set-cdr! (cdr x) x)
  (do ((i 0 (+ i 1)))
      ((= i 6))
    (set! x (cons x x)))
  (test (object->string x) "(#6=(#5=(#4=(#3=(#2=(#1=(+ 1 . #1#) . #1#) . #2#) . #3#) . #4#) . #5#) . #6#)"))

(test (let ((lst (list 1 2 3))) (set! (cdr (cddr lst)) lst) (append lst lst ())) 'error)
(test (let ((lst (list 1 2 3))) (set! (cdr (cddr lst)) lst) (object->string (append (list lst) (list lst) ()))) "(#1=(1 2 3 . #1#) #1#)")

(let ((ht (make-hash-table 3)))
  (set! (ht "hi") ht)
  (test (object->string ht) "#1=(hash-table \"hi\" #1#)")
  (test (equal? (ht "hi") ht) #t))

(let ((l1 '(0)) (l2 '(0)))
  (set! (car l1) l1) (set! (cdr l1) l1) (set! (car l2) l2) (set! (cdr l2) l2)
  (test (object->string l1) "#1=(#1# . #1#)")
  (test (equal? l1 l2) #t)
  (set! (cdr l1) l2)
  (test (object->string l1) "#1=(#1# . #2=(#2# . #2#))")
  (test (equal? l1 l2) #t)
  (set! (cdr l1) ())
  (test (equal? l1 l2) #f))

(let ((lst (list 1 2 3)))
  (set! (cdr (cddr lst)) lst)
  (test (map (lambda (a b)
	       (+ a b))
	     (list 4 5 6)
	     lst)
	'(5 7 9)))
(test (let ((lst (list 1 2 3))
	    (result ()))
	(set! (cdr (cddr lst)) lst)
	(for-each (lambda (a b)
		    (set! result (cons (+ a b) result)))
		  (list 4 5 6)
		  lst)
	result)
      '(9 7 5))
(let ((lst (list 1 2 3)))
  (set! (cdr (cddr lst)) lst)
  (test (map (lambda (a b)
	       (+ a b))
	     (vector 4 5 6)
	     lst)
	'(5 7 9)))
(test (let ((lst (list 1 2 3)))
	(set! (cdr (cddr lst)) lst)
	(map (lambda (a b)
	       (+ a b))
	     (vector 4 5 6 7 8 9 10)
	     lst))
      '(5 7 9 8)) ; this now quits when it sees the cycle
(test (map (lambda (a) a) '(0 1 2 . 3)) '(0 1 2))
(test (let ((ctr 0)) (for-each (lambda (a) (set! ctr (+ ctr a))) '(1 2 . 3)) ctr) 3)
(let ((lst (list 1 2 3)))
  (set! (cdr (cddr lst)) lst)
  (test (map (lambda (a b)
	       (+ a b))
	     ()
	     lst)
	()))
(test (let ((lst (list 1 2 3))
	    (ctr 0))
	(set! (cdr (cddr lst)) lst)
	(for-each (lambda (a b)
		    (set! ctr (+ ctr (+ a b))))
		  lst ())
	ctr)
      0)

(test (let ((lst (list 1))) (set! (cdr lst) (car lst)) (object->string lst)) "(1 . 1)")
(test (let ((lst (list 1))) (set! (car lst) (cdr lst)) (object->string lst)) "(())")

(let ((ctr 0) (lst `(let ((x 3)) (set! ctr (+ ctr 1)) (set! (cdr (cddr lst)) `((+ x ctr))) (+ x 1))))
  (test (eval lst) 4)
  (test (eval lst) 5)
  (test (eval lst) 6))

(when (= (*s7* 'debug) 0) ; fact assumes below that procedure-source is unaltered
  (let ()
    (define fact          ; Reini Urban, http://autocad.xarch.at/lisp/self-mod.lsp.txt
      (let ((old ())
	    (result ()))

	(define (last lst)
	  (list-tail lst (- (length lst) 1)))

	(define (butlast lis)
	  (let ((len (length lis)))
	    (if (<= len 1) ()
		(let ((result ()))
		  (do ((i 0 (+ i 1))
		       (lst lis (cdr lst)))
		      ((= i (- len 1)) (reverse result))
		    (set! result (cons (car lst) result)))))))

	(lambda (n)
	  (cond ((zero? n) 1)
		(#t
		 (set! old (procedure-source fact))
		 (set! fact (apply lambda '(n)
				   `((cond
				      ,@(butlast (cdr (car (cdr (cdr old)))))
				      ((= n ,n) ,(let ()
						   (set! result (* n (fact (- n 1))))
						   result))
				      ,@(last (cdr (car (cdr (cdr old)))))))))
		 result)))))

    (test (fact 3) 6)
    (test (fact 5) 120)
    (test (fact 2) 2)))

(test (let ((f #f))
	(set! f (lambda ()
		  (let* ((code (procedure-source f))
			 (pos (- (length code) 1)))
		    (set! (code pos) (+ (code pos) 1)))
		  1))
	(f) (f) (f))
      4)

(let* ((x (list 1 2 3)) ; from Lambda the Ultimate I think -- I lost the reference
       (y (list 4 5))
       (z (cons (car x) (cdr y)))
       (w (append y z))
       (v (cons (cdr x) (cdr y))))
  (set-car! x 6)
  (set-car! y 7)
  (set-cdr! (cdr x) (list 8))
  (test (object->string (list x y z w v)) "((6 2 8) (7 5) (1 5) (4 5 1 5) ((2 8) 5))"))
;; was "((6 . #3=(2 8)) (7 . #1=(5)) #2=(1 . #1#) (4 5 . #2#) (#3# . #1#))"

;; circular eval
(test (let ((e (list (list '+ 1)))) (set-cdr! (car e) e) (eval e)) 'error)
(test (let ((e (list (list '+ 1 2)))) (set-cdr! (cdar e) e) (eval e)) 'error)
(test (let ((e (list (list '+ 1 2) 3))) (set-cdr! (cdar e) e) (eval e)) 'error)
(test (let ((e (list (list '+ 1) 3 4))) (set-cdr! (cdar e) e) (eval e)) 'error)
(test (let ((x '(1 2 3)))
	(set! (x 0) (cons x 2))
	(eval (list-values 'let () (list-values 'define '(f1) (list-values 'list-set! x 0 (list-values 'cons x 2))) '(catch #t f1 (lambda a 'error)))))
      'error)
(test (let ((x '(car (list 1 2 3))))
	(set! (x 0) x)
	(eval (list-values 'let () (list-values 'define '(f1) x) '(catch #t f1 (lambda a 'error)))))
      'error)

#|
(define (for-each-permutation func vals)          ; for-each-combination -- use for-each-subset below
  "(for-each-permutation func vals) applies func to every permutation of vals"
  ;;   (for-each-permutation (lambda args (format #t "~{~A~^ ~}~%" args)) '(1 2 3))
  (define (pinner cur nvals len)
    (if (= len 1)
	(apply func (cons (car nvals) cur))
	(do ((i 0 (+ i 1)))                       ; I suppose a named let would be more Schemish
	    ((= i len))
	  (let ((start nvals))
	    (set! nvals (cdr nvals))
	    (let ((cur1 (cons (car nvals) cur)))  ; add (car nvals) to our arg list
	      (set! (cdr start) (cdr nvals))      ; splice out that element and
	      (pinner cur1 (cdr start) (- len 1)) ;   pass a smaller circle on down
	      (set! (cdr start) nvals))))))       ; restore original circle
  (let ((len (length vals)))
    (set-cdr! (list-tail vals (- len 1)) vals)    ; make vals into a circle
    (pinner () vals len)
    (set-cdr! (list-tail vals (- len 1)) ())))   ; restore its original shape

;; a slightly faster version (avoids consing and some recursion)
(define (for-each-permutation func vals)          ; for-each-combination -- use for-each-subset below
  "(for-each-permutation func vals) applies func to every permutation of vals"
  ;;   (for-each-permutation (lambda args (format #t "~A~%" args)) '(1 2 3))
  (let ((cur (make-list (length vals))))

    (define (pinner nvals len)
      (if (= len 2)
	  (begin
	    (set! (cur 0) (car nvals))
	    (set! (cur 1) (cadr nvals))
	    (apply func cur)
	    (set! (cur 1) (car nvals))
	    (set! (cur 0) (cadr nvals))
	    (apply func cur))

	(do ((i 0 (+ i 1)))
	    ((= i len))
	  (let ((start nvals))
	    (set! nvals (cdr nvals))
	    (set! (cur (- len 1)) (car nvals))
	    (set! (cdr start) (cdr nvals))        ; splice out that element and
	    (pinner (cdr start) (- len 1))        ;   pass a smaller circle on down
	    (set! (cdr start) nvals)))))          ; restore original circle

  (let ((len (length vals)))
    (set-cdr! (list-tail vals (- len 1)) vals)    ; make vals into a circle
    (pinner vals len)
    (set-cdr! (list-tail vals (- len 1)) ()))))  ; restore its original shape
|#

;; and continuing down that line...
(define (for-each-permutation func vals)          ; for-each-combination -- use for-each-subset below
  "(for-each-permutation func vals) applies func to every permutation of vals"
  ;;   (for-each-permutation (lambda args (format #t "~A~%" args)) '(1 2 3))
  (let ((cur (make-list (length vals))))

    (define (pinner nvals len)
      (if (= len 3)
	  (let ((a0 (car nvals))
		(a1 (cadr nvals))
		(a2 (caddr nvals))
		(c1 (cdr cur))
		(c2 (cddr cur)))
	    (set-car! cur a2)
	    (set-car! c1 a0)
	    (set-car! c2 a1)
	    (apply func cur)
	    (set-car! cur a0)
	    (set-car! c1 a2)
	    ;(set-car! c2 a1)
	    (apply func cur)
	    ;(set-car! cur a0)
	    (set-car! c1 a1)
	    (set-car! c2 a2)
	    (apply func cur)
	    (set-car! cur a1)
	    (set-car! c1 a0)
	    ;(set-car! c2 a2)
	    (apply func cur)
	    ;(set-car! cur a1)
	    (set-car! c1 a2)
	    (set-car! c2 a0)
	    (apply func cur)
	    (set-car! cur a2)
	    (set-car! c1 a1)
	    ;(set-car! c2 a0)
	    (apply func cur)
	    )

	(do ((i 0 (+ i 1)))
	    ((= i len))
	  (let ((start nvals))
	    (set! nvals (cdr nvals))
	    (list-set! cur (- len 1) (car nvals))
	    (set-cdr! start (cdr nvals))        ; splice out that element and
	    (pinner (cdr start) (- len 1))      ;   pass a smaller circle on down
	    (set-cdr! start nvals)))))          ; restore original circle

  (let ((len (length vals)))
    (if (< len 2)
	(apply func vals)
	(if (= len 2)
	    (let ((c1 (cdr cur)))
	      (set-car! cur (car vals))
	      (set-car! c1 (cadr vals))
	      (apply func cur)
	      (set-car! c1 (car vals))
	      (set-car! cur (cadr vals))
	      (apply func cur))
	    (begin
	      (set-cdr! (list-tail vals (- len 1)) vals)    ; make vals into a circle
	      (pinner vals len)
	      (set-cdr! (list-tail vals (- len 1)) ())))))))  ; restore its original shape

#|
;; much slower:
(define (for-each-permutation func vals)
  (define (heap-permutation size n)
    (if (= size 1)
	(apply func vals)
	(do ((i 0 (+ i 1)))
	    ((= i size))
	  (heap-permutation (- size 1) n)
	  (if (odd? size)
	      (let ((temp (vals 0)))
		(set! (vals 0) (vals (- size 1)))
		(set! (vals (- size 1)) temp))
	      (let ((temp (vals i)))
		(set! (vals i) (vals (- size 1)))
		(set! (vals (- size 1)) temp))))))
  (heap-permutation (length vals) (length vals)))
|#

(when full-s7test
  (let()
    (define ops '(+ *))
    (define args '(1 pi 1+i 2/3 x y))

    (define (listify lst)
      ((if (memq (car lst) ops) list
	   (if (null? (cdr lst)) append values))
       (if (null? (cdr lst))
	   (car lst)
	   (values (car lst) (listify (cdr lst))))))

    (call-with-output-file "t923.scm"
      (lambda (p)
        (format p "(define t923-old-eps (*s7* 'equivalent-float-epsilon))~%(set! (*s7* 'equivalent-float-epsilon) 1e-15)~%~%")
        ;; 1e-15 here depends on add_p_ppp order -- it needs to mimic g_add, (+ (+ x y) z), else limit needs to be 1e-14
	(let ((fctr 0))
	  (for-each-permutation
	   (lambda lst
	     (let ((expr (list (listify lst))))
	       (format p "(define (f~D x y) ~{~^~S ~})~%" fctr expr)
	       (format p "(let ((e1 (f~D 3 4)))~%" fctr)
	       (format p "  (let ((e2 (let ((x 3) (y 4)) ~{~^~S ~})))~%" expr)
	       (format p "    (unless (equivalent? e1 e2)~%      (format *stderr* \"~{~^~S ~}: ~~A ~~A~~%\" e1 e2))))~%~%" expr))
	     (set! fctr (+ fctr 1)))
	   (append ops args)))
         (format p "(set! (*s7* 'equivalent-float-epsilon) t923-old-eps)~%")))

    (load "t923.scm")))

  ;; t224 also applies this to +/*

(let ((perms '((3 1 2) (1 3 2) (1 2 3) (2 1 3) (2 3 1) (3 2 1)))
      (pos ()))
  (for-each-permutation
   (lambda args
     (call-with-exit
      (lambda (ok)
	(let ((ctr 0))
	  (for-each
	   (lambda (a)
	     (if (equal? a args)
		 (begin
		   (set! pos (cons ctr pos))
		   (ok)))
	     (set! ctr (+ ctr 1)))
	   perms)))))
   '(1 2 3))
  (test pos '(5 4 3 2 1 0)))

(test (let ((v1 (make-vector 16 0))
	    (v2 (make-vector 16 0)))
	(set! (v2 12) v2)
	(set! (v1 12) v1)
	(equal? v1 v2))        ; hmmm -- not sure this is correct
      #t)
(test (let ((lst1 (list 1))
	    (lst2 (list 1)))
	(set-cdr! lst1 lst1)
	(set-cdr! lst2 lst2)
	(equal? lst1 lst2))
      #t)


(test (let ((hi 3))
	(let ((e (curlet)))
	  (set! hi (curlet))
	  (object->string e)))
      "#1=(inlet 'hi #2=(inlet 'e #1#))") ; was "#1=(inlet 'hi (inlet 'e #1#))")
(let ((e (inlet 'a 0 'b 1)))
  (let ((e1 (inlet 'a e)))
    (set! (e 'b) e1)
    (test (equal? e (copy e)) #t)
    (test (object->string e) "#1=(inlet 'a 0 'b (inlet 'a #1#))")))

;; eval circles -- there are many more of these that will cause stack overflow
(test (let ((x '(1 2 3))) (set! (x 0) (cons x 2)) (eval `(let () (define (f1) (list-set! ,x 0 (cons ,x 2))) (catch #t f1 (lambda a 'error))))) 'error)
(test (let ((x '(car (list 1 2 3)))) (set! (x 0) x) (eval `(let () (define (f1) ,x) (catch #t f1 (lambda a 'error))))) 'error)


(test (apply + (cons 1 2)) 'error)
(test (let ((L (list 0))) (set-cdr! L L) (apply + L)) 'error)
(test (let ((L (list 0))) (set-cdr! L L) (format #f "(~S~{~^ ~S~})~%" '+ L)) "(+ 0)\n") ; 28-Nov-18
(test (apply + (list (let ((L (list 0 1))) (set-cdr! L L) L))) 'error)
(test (apply + (let ((L (list 0 1))) (set-cdr! L L) L)) 'error)
(test (length (let ((E (inlet 'value 0))) (varlet E 'self E))) 2)
;(test (apply case 2 (list (let ((L (list (list 0 1)))) (set-cdr! L L) L))) 'error)
;(test (apply cond (list (let ((L (list 0 1))) (set-cdr! L L) L))) 'error)
;(test (apply quote (let ((L (list 0 1))) (set-car! L L) L)) 'error)
;(test (apply letrec (hash-table) (let ((L (list 0 1))) (set-car! L L) L)) 'error)
;I now think the caller should check for these, not s7



;;; --------------------------------------------------------------------------------
;;; HOOKS
;;; make-hook
;;; hook-functions
;;; --------------------------------------------------------------------------------

(let-temporarily (((hook-functions *error-hook*) ())
		  ((hook-functions *load-hook*) ())
		  ((hook-functions *unbound-variable-hook*) ())
		  ((hook-functions *missing-close-paren-hook*) ()))
  (for-each
   (lambda (arg)
     (test (set! *unbound-variable-hook* arg) 'error)
     (test (set! *missing-close-paren-hook* arg) 'error)
     (test (set! *load-hook* arg) 'error)

     (test (set! (hook-functions *unbound-variable-hook*) arg) 'error)
     (test (set! (hook-functions *missing-close-paren-hook*) arg) 'error)
     (test (set! (hook-functions *error-hook*) arg) 'error)
     (test (set! (hook-functions *load-hook*) arg) 'error)

     (test (set! (hook-functions *unbound-variable-hook*) (list arg)) 'error)
     (test (set! (hook-functions *missing-close-paren-hook*) (list arg)) 'error)
     (test (set! (hook-functions *error-hook*) (list arg)) 'error)
     (test (set! (hook-functions *load-hook*) (list arg)) 'error))
   (list -1 #\a #(1 2 3) 3.14 3/4 1.0+1.0i 'hi :hi #<eof> #(1 2 3) #(()) "hi" '(1 . 2) '(1 2 3))))

(let ((hook-val #f))
  (let-temporarily (((hook-functions *unbound-variable-hook*)
		     (list (lambda (hook)
			     (set! hook-val (hook 'variable))
			     (set! (hook 'result) 123)))))
    (let ((val (catch #t
		 (lambda ()
		   (+ 1 one-two-three))
		 (lambda args
		   (apply format *stderr* (cadr args))
		   'error))))
      (test val 124))
    (test (equal? one-two-three 123) #t)
    (test (equal? hook-val 'one-two-three) #t)))

(let-temporarily (((hook-functions *unbound-variable-hook*) (list (lambda (hook)
								    (set! (hook 'result) 32)))))
  (let ((val (+ 1 _an_undefined_variable_i_hope_)))
    (test val 33))
  (let ((val (+ 1 _an_undefined_variable_i_hope_)))
    (test (call/cc (lambda (_a_) (_a_ val))) 33))
  (let ((val (* _an_undefined_variable_i_hope_ _an_undefined_variable_i_hope_)))
    (test val 1024)))

(let ((x #f))
  (let-temporarily (((hook-functions *unbound-variable-hook*)
		     (list (lambda (hook)
			     (set! x 0)
			     (set! (hook 'result) #<undefined>))
			   (lambda (hook)
			     (set! (hook 'result) 32))
			   (lambda (hook)
			     (if (not (number? (hook 'result)))
				 (format *stderr* "oops -- *unbound-variable-hook* func called incorrectly~%"))))))
    (let ((val (+ 1 _an_undefined_variable_i_hope_)))
      (test val 33))
    (test x 0)
    (test (+ 1 _an_undefined_variable_i_hope_) 33)))


(define (-a-rootlet-entry- x) (- x (abs x)))
(define -a-rootlet-entry-value- #f)
(set! (hook-functions *rootlet-redefinition-hook*)
      (list (lambda (hook)
	      (set! -a-rootlet-entry-value- (hook 'value)))))
(define (-a-rootlet-entry- x) (+ x (abs x)))
(unless (and (procedure? -a-rootlet-entry-value-)
             (equal? (procedure-source -a-rootlet-entry-value-)
                     (if (positive? (*s7* 'debug))
                         '(lambda (x) (trace-in (curlet)) (+ x (abs x)))
                         '(lambda (x) (+ x (abs x))))))
    (format *stderr* "rootlet redef: ~W~%" -a-rootlet-entry-value-))
(set! (hook-functions *rootlet-redefinition-hook*) ())


;;; optimizer bug involving unbound variable
(let ()
  (define (opt1)
    (let ((val (let ()
		 (define (hi x y) (let ((m (memq x y)) (loc (and m (- x (length m))))) loc))
		 (hi 'a '(a b c)))))
      (format #t "~A: opt1 got ~S but expected 'error~%~%" (port-line-number) val)))
  (catch #t opt1
	 (lambda (type info)
	   (if (not (eq? type 'unbound-variable))
	       (format *stderr* "opt1 type: ~A, info: ~A~%" type info))
	   'error)))
(let ()
  (define (opt2)
    (let ((val (let ()
		 (define (hi x y) (let* ((n (memq x y)) (loc (and m (- x (length m)))) (m (+ n 1))) loc))
		 (hi 'a '(a b c)))))
      (format #t "~A: opt2 got ~S but expected 'error~%~%" (port-line-number) val)))
  (catch #t opt2
	 (lambda (type info)
	   (if (not (eq? type 'unbound-variable))
	       (format *stderr* "opt2 type: ~A, info: ~A~%" type info))
	   'error)))

(let ()
  (define (opt3)
    (let ((val (let ()
		 (define (hi x y) (do ((m (memq x y) 0) (loc (and m (- x (length m))) 0)) (loc #t)))
		 (hi 'a '(a b c)))))
      (format #t "~A: opt3 got ~S but expected 'error~%~%" (port-line-number) val)))
  (catch #t opt3
	 (lambda (type info)
	   (if (not (eq? type 'unbound-variable))
	       (format *stderr* "opt3 type: ~A, info: ~A~%" type info))
	   'error)))

(let ()
  (define (opt4)
    (let ()
      (define (hi x y) (letrec ((m (memq x y)) (loc (and m (length m)))) loc))
      (hi 'a '(a b c))))
  (catch #t opt4
	 (lambda (type info)
	   'error)))

(let ()
  (define (opt5)
    (let ((val (let ()
		 (define (hi x y) (letrec* ((n (memq x y)) (loc (and m (- x (length m)))) (m (+ n 1))) loc))
		 (hi 'a '(a b c)))))
      (format #t "~A: opt5 got ~S but expected 'error~%~%" (port-line-number) val)))
  (catch #t opt5
	 (lambda (type info)
	   'error)))

(let ()
  (define (opt6)
    (let ((val (let ()
		 (define (hi x) (let ((m (memq n x)) (loc (and m (- x (length m))))) (define n 1) loc))
		 (hi '(a b c)))))
      (format #t "~A: opt6 got ~S but expected 'error~%~%" (port-line-number) val)))
  (catch #t opt6
	 (lambda (type info)
	   (if (not (eq? type 'unbound-variable))
	       (format *stderr* "opt6 type: ~A, info: ~A~%" type info))
	   'error)))

(let ()
  (define (opt7)
    (let ((val (let ()
		 (define* (f1 (a (+ m 1)) (m (+ a 1))) (+ a m))
		 (f1))))
      (format #t "~A: opt7 got ~S but expected 'error~%~%" (port-line-number) val)))
  (catch #t opt7
	 (lambda (type info)
	   'error)))

(let ()
  (define (opt8)
    (let ((val (let ()
		 (let ((x 1))
		   (set! x (+ m 1))
		   (define m 2)
		   x))))
      (format #t "~A: opt8 got ~S but expected 'error~%~%" (port-line-number) val)))
  (catch #t opt8
	 (lambda (type info)
	   (if (not (eq? type 'unbound-variable))
	       (format *stderr* "opt8 type: ~A, info: ~A~%" type info))
	   'error)))

(let ()
  (define (opt9)
    (let ((val (let ()
		 (let ((x 1)) (set! x (and m (length m))) (define m 2) x))))
      (format #t "~A: opt9 got ~S but expected 'error~%~%" (port-line-number) val)))
  (catch #t opt9
	 (lambda (type info)
	   (if (not (eq? type 'unbound-variable))
	       (format *stderr* "opt9 type: ~A, info: ~A~%" type info))
	   'error)))

(let ()
  (define (opt10)
    (let ()
      (define* (f1 (a (and m (length m))) (m 1)) (+ a m))
      (f1))) ; but not unbound var error!  isn't this a bug?
  (catch #t opt10
	 (lambda (type info)
	   'error)))

(let ()
  (define (makel depth)
    (if (= depth 10)
	(list depth)
	(set! ___lst (list (makel (+ depth 1))))))
  (test (makel 0) 'error))


(let ((val #f))
  (let-temporarily (((hook-functions *load-hook*) (list (lambda (hook)
							  (if (or val
								  (defined? 'load-hook-test))
							      (format #t ";*load-hook*: ~A ~A?~%" val load-hook-test))
							  (set! val (hook 'name))))))
    (with-output-to-file "load-hook-test.scm"
      (lambda ()
	(format #t "(define (load-hook-test val) (+ val 1))")))
    (load "load-hook-test.scm")
    (if (or (not (string? val))
	    (not (string=? val "load-hook-test.scm")))
	(format #t ";*load-hook-test* file: ~S~%" val))
    (if (not (defined? 'load-hook-test))
	(format #t ";load-hook-test function not defined?~%")
	(if (not (= (load-hook-test 1) 2))
	    (format #t ";load-hook-test: ~A~%" (load-hook-test 1))))))

(let-temporarily (((hook-functions *error-hook*) ()))
  (test (hook-functions *error-hook*) ())
  (set! (hook-functions *error-hook*) (list (lambda (hook) #f)))
  (test (list? (hook-functions *error-hook*)) #t))

(let-temporarily (((hook-functions *missing-close-paren-hook*) (list (lambda (h) (set! (h 'result) 'incomplete-expr)))))
  (test (catch #t (lambda () (eval-string "(+ 1 2")) (lambda args (car args))) 'incomplete-expr)
  (test (catch #t (lambda () (eval-string "(")) (lambda args (car args))) 'incomplete-expr)
  (test (catch #t (lambda () (eval-string "(abs ")) (lambda args (car args))) 'incomplete-expr))

(let ((h (make-hook 'x)))
  (test (procedure? h) #t)
  (test (eq? h h) #t)
  (test (eqv? h h) #t)
  (test (equal? h h) #t)
  (test (equivalent? h h) #t)
  (test (arity h) (cons 0 1))
  (let ((h1 (copy h)))
    (test (eq? h h1) #f) ; fluctutates...
    (test (equivalent? h h1) #t))
  (test (hook-functions h) ())
  (test (h) #<unspecified>)
  (test (h 1) #<unspecified>)
  (test (h 1 2) 'error)
  (let ((f1 (lambda (hook) (set! (hook 'result) (hook 'x)))))
    (set! (hook-functions h) (list f1))
    (test (member f1 (hook-functions h)) (list f1))
    (test (hook-functions h) (list f1))
    (test (h 1) 1)
    (set! (hook-functions h) ())
    (test (hook-functions h) ())
    (let ((f2 (lambda* args (set! ((car args) 'result) ((car args) 'x)))))
      (set! (hook-functions h) (list f2))
      (test (hook-functions h) (list f2))
      (test (h 1) 1)))
  (for-each
   (lambda (arg)
     (test (set! (hook-functions h) arg) 'error))
   (list "hi" #f (integer->char 65) 1 (list 1 2) '#t '3 (make-vector 3) 3.14 3/4 1.0+1.0i #\f :hi #<eof> #<undefined> #<unspecified>))
  (set! (hook-functions h) (list (lambda (hk) (set! (hk 'result) (hk 'xyzzy)))))
  (test (h 123) #<undefined>))

(let ((h (make-hook)))
  (test (procedure? h) #t)
  (test (documentation h) "")
  (test (hook-functions h) ())
  (test (h) #<unspecified>)
  (test (arity h) (cons 0 0))
  (test (h 1) 'error)
  (let ((f1 (lambda (hook) (set! (hook 'result) 123))))
    (set! (hook-functions h) (list f1))
    (test (member f1 (hook-functions h)) (list f1))
    (test (hook-functions h) (list f1))
    (test (h) 123)
    (set! (hook-functions h) ())
    (test (hook-functions h) ())
    (let ((f2 (lambda* args (set! ((car args) 'result) 321))))
      (set! (hook-functions h) (list f2))
      (test (hook-functions h) (list f2))
      (test (h) 321))))

(let ((h (make-hook '(a 32) 'b)))
  (test (procedure? h) #t)
  (test (hook-functions h) ())
  (test (arity h) (cons 0 2))
  (test (h) #<unspecified>)
  (test (h 1) #<unspecified>)
  (test (h 1 2) #<unspecified>)
  (test (h 1 2 3) 'error)
  (let ((f1 (lambda (hook) (set! (hook 'result) (+ (hook 'a) (or (hook 'b) 0))))))
    (set! (hook-functions h) (list f1))
    (test (member f1 (hook-functions h)) (list f1))
    (test (hook-functions h) (list f1))
    (test (h) 32)
    (test (h 1) 1)
    (test (h 1 2) 3)
    (set! (hook-functions h) ())
    (test (hook-functions h) ())))

(test (let () (define h (make-hook 'x)) (set! (hook-functions h) (list (lambda (hk) (set! (hk 'result) (hk 'abs))))) (h 123)) #<undefined>) ; new version of make-hook

(let ()
  (for-each
   (lambda (arg)
     (test (make-hook arg) 'error))
   (list "hi" #f 1 (list 1 2) '#t '3 (make-vector 3) 3.14 3/4 1.0+1.0i #\f :hi #<eof> #<undefined> #<unspecified>)))

(let ((h (make-hook)))
  (let ((f1 (lambda (hook) (if (number? (hook 'result)) (set! (hook 'result) (+ (hook 'result) 1)) (set! (hook 'result) 0)))))
    (test (h) #<unspecified>)
    (set! (hook-functions h) (list f1))
    (test (h) 0)
    (set! (hook-functions h) (list f1 f1 f1))
    (test (h) 2)))

(if (not (defined? 'hook-push))
    (define (hook-push hook func)
      (set! (hook-functions hook) (cons func (hook-functions hook)))))

(let ((h (make-hook)))
  (hook-push h (lambda (hook) (set! (hook 'result) 32)))
;  (test (dynamic-wind h h h) 32)
  (test (catch h h h) 32)
  )

(let ((h (make-hook 'x)))
  (hook-push h (lambda (hook) (set! (hook 'result) (hook 'x))))
  (test (continuation? (call/cc h)) #t)
  (set! (hook-functions h) (list (lambda (hook) (set! (hook 'result) (+ 1 (hook 'x))))))
  (test (map h '(1 2 3)) '(2 3 4))
  )

(let ()
  (define-macro (hook . body)
    `(let ((h (make-hook)))
       (set! (hook-functions h)
	     (list (lambda (h)
		     (set! (h 'result) (begin ,@body)))))
       h))
  (let ((x 0))
    (define hi (hook (set! x 32) (+ 2 3 1)))
    (test (hi) 6)
    (test x 32)))

(let ()
  (define-macro (hooked-catch hook . body)
    `(catch #t
       (lambda ()
	 ,@body)
       (lambda args
	 (let ((val (apply ,hook args)))
	   (if (eq? val #<unspecified>) ; hook did not do anything
	       (apply error args)       ; so re-raise the error
	       val)))))

 (let ((a-hook (make-hook 'error-type :rest 'error-info)))
   (set! (hook-functions a-hook)
         (list (lambda (hook)
		 ;(format #t "hooked-catch: ~A~%" (apply format #t (car (hook 'error-info))))
		 (set! (hook 'result) 32))))
   (test (hooked-catch a-hook (abs "hi")) 32)

   (set! (hook-functions a-hook) ())

   (test (catch #t
	   (lambda ()
	     (hooked-catch a-hook (abs "hi")))
	   (lambda args
	     123))
	 123)
   ))

(let ((loaded #f))
(let-temporarily (((hook-functions *autoload-hook*)
                    (list (lambda (h) (set! loaded (cons (h 'name) (h 'file)))))))
  (autoload 'pp "write.scm")
  (pp '(1 2))
  (test loaded (cons 'pp "write.scm"))))

(let ()
  (define *breaklet* #f)
  (define *step-hook* (make-hook 'code 'e))

  (define-macro* (trace/break code . break-points)
    (define (caller tree)
      (if (pair? tree)
	  (cons
	   (if (pair? (car tree))
	       (if (and (symbol? (caar tree))
			(procedure? (symbol->value (caar tree))))
		   (if (member (car tree) break-points)
		       `(__break__ ,(caller (car tree)))
		       `(__call__ ,(caller (car tree))))
		   (caller (car tree)))
	       (car tree))
	   (caller (cdr tree)))
	  tree))
    `(call-with-exit (lambda (__top__) ,(caller code))))

  (define (go . args)
    (and (let? *breaklet*)
	 (apply (*breaklet* 'go) args)))

  (define (clear-break)
    (set! *breaklet* #f))

  (define-macro (__call__ code)
    `(*step-hook* ',code (curlet)))

  (define-macro (__break__ code)
    `(begin
       (call/cc
	(lambda (go)
	  (set! *breaklet* (curlet))
	  (__top__ (format #f "break at: ~A~%" ',code))))
       ,code))

  (set! (hook-functions *step-hook*)
	(list (lambda (hook)
		(set! (hook 'result) (eval (hook 'code) (hook 'e))))
	      (lambda (hook)
		(define (uncaller tree)
		  (if (pair? tree)
		      (cons
		       (if (and (pair? (car tree))
				(memq (caar tree) '(__call__ __break__)))
			   (uncaller (cadar tree))
			   (uncaller (car tree)))
		       (uncaller (cdr tree)))
		      tree))
		(format (current-output-port) ": ~A -> ~A~40T~A~%"
			(uncaller (hook 'code))
			(hook 'result)
			(if (and (not (eq? (hook 'e) (rootlet)))
				 (not (defined? '__top__ (hook 'e))))
			    (map values (hook 'e))
			    "")))))

  (let ((str (with-output-to-string
	       (lambda ()
		 (trace/break (let ((a (+ 3 1)) (b 2)) (if (> (* 2 a) b) 2 3)))))))
    (test (or (string-wi=? str ": (+ 3 1) -> 4
: (* 2 a) -> 8                         ((a . 4) (b . 2))
: (> (* 2 a) b) -> #t                  ((a . 4) (b . 2))
")
	      (string-wi=? str ": (+ 3 1) -> 4
: (* 2 a) -> 8                         ((b . 2) (a . 4))
: (> (* 2 a) b) -> #t                  ((b . 2) (a . 4))
")) #t)))


;;; #_ stuff
(test ((lambda () (if (#_round pi) #f))) #f)
(test ((lambda () (when (#_round pi) #f))) #f)
(test ((lambda () (#_cond (* 1)))) 1)
(test (let () (define (f1) (abs (#_logand))) (f1)) 1)
(test ((lambda () (abs (#_logand)))) 1)
(test ((lambda () (abs (#_logand 2 3)))) 2)
(test (call-with-exit (lambda (g) (abs (#_logand)))) 1)
(test (let () (define (func) (append (#_begin (tree-cyclic?)))) (func)) 'error)
(test (let () (define (func) (do ((x 0.0 (+ x 0.1)) (i 0 (+ i 1))) ((>= x .1) (#_with-baffle (inlet (values 1 2) (symbol? x)))))) (func)) 'error)
(test (let () (define (func) (do ((i 0 (+ i 1))) ((= i 1)) (#_provide :readable))) (func)) #t)
(test (pair? (let ()
               (define (func)
                 (list-values (#_quasiquote (odd?)) (let ((<1> (list 1 #f))) (set! (<1> 1) (let ((<L> (list #f 3))) (set-car! <L> <1>) <L>)) <1>)))
               (func)))
      #t)
(test (let () (define (hi) (let ((x 0) (i 3)) (do ((i i (+ i 1))) ((= i 6)) (do ((i i (+ i 1))) ((= i 7)) (set! x (+ x i)))) x)) (hi)) 44)

;;; *error-hook* + let-temporarily

(catch #t
  (lambda ()
    (let-temporarily (((hook-functions *error-hook*) (list (lambda (hook) 'hook))))
      (+ 1 #())
      322))
  (lambda (type info)
    'catch?))

(test (catch #t
        (lambda ()
          (let-temporarily (((*s7* 'print-length) 123123))
            (+ 1 #())
            323))
        (lambda (type info) 'catch1))
      'catch1)

(test (let-temporarily (((hook-functions *error-hook*) (list (lambda (hook) 'hook))))
        (catch #t
          (lambda ()
            (+ 1 #())
            324)
          (lambda (type info)
            'catch2)))
      'catch2)

(let ((x (let-temporarily (((hook-functions *error-hook*) (list (lambda (hook) 'hook))))
	   (catch #t
	     (lambda ()
	       (+ 1 #())
	       325)
	     (lambda (type info)
	       'catch3)))))
  (test x 'catch3)
  (test (hook-functions *error-hook*) ()))

(test (hook-functions *error-hook*) ())

(test (catch #t
	(lambda ()
	  (let-temporarily (((hook-functions *error-hook*)
			     (list (lambda (hook)
				     (+ 1 #\a))))) ; not called: error-hook is not triggered
	    322))
	(lambda (type info)
	  'catch4))
      322)

(test (catch #t
	(lambda ()
	  (let-temporarily (((hook-functions *error-hook*)
			     (list (lambda (hook)
				     (+ 1 #\a))))) ; this goes to the catch -- error-hook is not enabled within its body to avoid infinite recursion
	    (+ 1 #\b) ; this triggers error-hook, but the error is: "+ second argument, #\a, is a character but should be a number"
	    322))
	(lambda (type info)
	  (apply format #f info)))
      "+ second argument, #\\a, is a character but should be a number")

;;; *read-error-hook*

(test (hook-functions *read-error-hook*) ())

(let ()
  (define (reader-hooks h)
    (let ((type (h 'type))
	  (data (h 'data)))
      (if type
	  (set! (h 'result)
		(case (data 0)
		  ((#\T)
		   (and (string=? data "T")
			;(format #t "#T should be #t~%")
			#t))
		  ((#\F)
		   (and (string=? data "F")
			;(format #t "#F should be #f~%")
			''#f)))))))

  (set! (hook-functions *read-error-hook*) (list reader-hooks))
  (test (eval-string "#T") #t)
  (test (eval-string "(list #F)") '(#f))
  (set! (hook-functions *read-error-hook*) ())

  (let-temporarily (((hook-functions *read-error-hook*) (list reader-hooks)))
    (test (eval-string "#T") #t))

  (test (hook-functions *read-error-hook*) ()))



;;; --------------------------------------------------------------------------------
;;; HASH-TABLES
;;; --------------------------------------------------------------------------------
;;; make-hash-table
;;; make-weak-hash-table
;;; hash-table
;;; weak-hash-table
;;; hash-table?
;;; weak-hash-table?
;;; hash-table-entries
;;; hash-table-ref
;;; hash-table-set!
;;; hash-code
;;; hash-table-key-typer
;;; hash-table-value-typer

(let ((ht (make-hash-table)))
  (test (hash-table? ht) #t)
  (test (equal? ht ht) #t)
  (test (let () (hash-table-set! ht 'key 3.14) (hash-table-ref ht 'key)) 3.14)
  (test (let () (hash-table-set! ht "ky" 3.14) (hash-table-ref ht "ky")) 3.14)
  (test (let () (hash-table-set! ht 123 "hiho") (hash-table-ref ht 123)) "hiho")
  (test (let () (hash-table-set! ht 3.14 "hi") (hash-table-ref ht 3.14)) "hi")
  (test (let () (hash-table-set! ht pi "hiho") (hash-table-ref ht pi)) "hiho")
  (test (hash-table-ref ht "123") #f)
  (let ((ht1 (copy ht)))
    (test (hash-table? ht1) #t)
    (test (iterator? ht1) #f)
    (test (iterator? (make-iterator ht1)) #t)
    (test (= (length ht) (length ht1)) #t)
    (test (equal? ht ht1) #t)
    (test (eq? ht ht) #t)
    (test (eqv? ht ht) #t)
    (set! (ht 'key) 32)
    (set! (ht1 'key) 123)
    (test (and (= (ht 'key) 32) (= (ht1 'key) 123)) #t)
    (set! (ht "key") 321)
    (test (ht "key") 321)
    (test (ht 'key) 32)
    (set! (ht 123) 43)
    (set! (ht "123") 45)
    (test (ht 123) 43)
    (test (ht "123") 45)
    (test (hash-table-set! ht "1" 1) 1)
    (test (set! (ht "2") 1) 1)
    (test (set! (hash-table-ref ht "3") 1) 1)
    (test (hash-table-ref ht "3") 1))
  (test (let () (set! (hash-table-ref ht 'key) 32) (hash-table-ref ht 'key)) 32)

  (for-each
   (lambda (arg)
     (test (let () (hash-table-set! ht 'key arg) (hash-table-ref ht 'key)) arg))
   (list "hi" -1 #\a 1 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2))))

(for-each
 (lambda (arg)
   (test (hash-table-set! arg 'key 32) 'error))
 (list "hi" () -1 #\a 1 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (integer? (hash-code arg)) #t))
 (list "hi" () -1 #\a 1 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2) '(1 2) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       #f (lambda (a) (+ a 1)) (macro (a) `(+ ,a 1)) :hi (if #f #f) #<eof> #<undefined>))

(for-each
 (lambda (arg)
   (test (hash-code 123 arg) 'error))
 (list "hi" () -1 #\a 1 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2) '(1 2) _ht_ quasiquote macroexpand 1/0 (log 0)
       #f (macro (a) `(+ ,a 1)) :hi (if #f #f) #<eof> #<undefined>))

(test (catch #t (lambda () (hash-table-ref (hash-table 'a 1) 'b 2)) (lambda (typ info) (apply format #f info)))
      "(hash-table-ref (hash-table 'a 1) 'b 2) becomes (#f 2), but #f can't take arguments")

(test (catch #t (lambda () (let ((h (hash-table 'b 1))) (h 'a 'asdf))) (lambda (typ info) (apply format #f info)))
      "((hash-table 'b 1) 'a 'asdf) becomes (#f 'asdf), but #f can't take arguments")

(test (catch #t (lambda () (let ((h (hash-table 'a (hash-table 'b 1)))) (h 'a 'c 'd))) (lambda (typ info) (apply format #f info)))
      "((hash-table 'b 1) 'c 'd) becomes (#f 'd), but #f can't take arguments")

(test (catch #t (lambda () (let ((h (hash-table))) (hash-table-ref h 'a 'asdf))) (lambda (typ info) (apply format #f info)))
      "(hash-table-ref (hash-table) 'a 'asdf) becomes (#f 'asdf), but #f can't take arguments")

(test (catch #t (lambda () (let ((L (list 1))) (list-ref L 0 2))) (lambda (typ info) (apply format #f info)))
      "(list-ref (1) 0 2) becomes (1 2), but 1 can't take arguments")

(test (catch #t (lambda () (let ((L (list 1))) (L 0 2))) (lambda (typ info) (apply format #f info)))
      "((1) 0 2) becomes (1 2), but 1 can't take arguments")

(test (catch #t (lambda () (let ((L (list (list 0)))) (L 0 0 2))) (lambda (typ info) (apply format #f info)))
      "((0) 0 2) becomes (0 2), but 0 can't take arguments")

(test (catch #t (lambda () (let ((V (vector 1 2))) (V 0 1))) (lambda (typ info) (apply format #f info)))
      "(#(1 2) 0 1) becomes (1 1), but 1 can't take arguments")

(test (catch #t (lambda () (let ((V (vector 1 2))) (vector-ref V 0 1))) (lambda (typ info) (apply format #f info)))
      "(#(1 2) 0 1) becomes (1 1), but 1 can't take arguments")

(test (catch #t (lambda () (let ((V (vector (vector 0 12)))) (V 0 1 0))) (lambda (typ info) (apply format #f info)))
      "(#(0 12) 1 0) becomes (12 0), but 12 can't take arguments")

(test (catch #t (lambda () (let ((V (int-vector 1 2))) (V 0 1))) (lambda (typ info) (apply format #f info)))
      "vector-ref: too many indices: (0 1)")

(test (catch #t (lambda () (let ((L (inlet))) (L 'a :asdf))) (lambda (typ info) (apply format #f info)))
      "((inlet) 'a :asdf) becomes (#<undefined> :asdf), but #<undefined> can't take arguments")

(test (catch #t (lambda () (let ((L (inlet 'a (inlet 'b 1)))) (L 'a 'b 'c))) (lambda (typ info) (apply format #f info)))
      "((inlet 'b 1) 'b 'c) becomes (1 'c), but 1 can't take arguments")

(test (catch #t (lambda () (let ((L (list 1))) (set! (L 0 2) 32))) (lambda (typ info) (apply format #f info)))
      "in (set! (L 0 2) 32), ((1) 0) is 1 which can't take arguments")

(test (catch #t (lambda () (let ((L (list (list 0)))) (set! (L 0 0 2) 32))) (lambda (typ info) (apply format #f info)))
      "in (set! (L 0 0 2) 32), ((0) 0) is 0 which can't take arguments")

(test (catch #t (lambda () (let ((h (hash-table 'b 1))) (set! (h 'a 'asdf) 32))) (lambda (typ info) (apply format #f info)))
      "in (set! (h 'a 'asdf) 32), 'a does not exist in (hash-table 'b 1)")

(test (catch #t (lambda () (let ((h (hash-table 'b 1))) (set! (h 'b 'asdf) 32))) (lambda (typ info) (apply format #f info)))
      "in (set! (h 'b 'asdf) 32), ((hash-table 'b 1) 'b) is 1 which can't take arguments")

(test (catch #t (lambda () (let ((h (hash-table 'a (hash-table 'b 1)))) (set! (h 'a 'c 'd) 32))) (lambda (typ info) (apply format #f info)))
      "in (set! (h 'a 'c 'd) 32), 'c does not exist in (hash-table 'b 1)")

(test (catch #t (lambda () (let ((v (hash-table 'a (list 1 2)))) (set! (v 'a 1) 5))) (lambda (typ info) (apply format #f info)))
      5)

(test (catch #t (lambda () (let ((v (hash-table 'a (hash-table 'b 1)))) (set! (v 'a 'b 'b) 32) v)) (lambda (typ info) (apply format #f info)))
      "in (set! (v 'a 'b 'b) 32), ((hash-table 'b 1) 'b) is 1 which can't take arguments")

(test (catch #t (lambda () (let ((L (inlet 'a (inlet 'b 1)))) (set! (L 'a 'b 'c) 32))) (lambda (typ info) (apply format #f info)))
      "in (set! (L 'a 'b 'c) 32), ((inlet 'b 1) 'b) is 1 which can't take arguments")

(test (catch #t (lambda () (let ((L (inlet))) (set! (L 'a :asdf) 32))) (lambda (typ info) (apply format #f info)))
      "in (set! (L 'a :asdf) 32), ((inlet) 'a) is #<undefined> which can't take arguments")

(test (catch #t (lambda () (set! (abs 1) 2)) (lambda (typ info) (apply format #f info)))
      "abs (a c-function) does not have a setter: (set! (abs 1) 2)")

(test (catch #t (lambda () (set! (when #t 3) 21) ) (lambda (type info) (apply format #f info)))
      "when (syntactic) does not have a setter: (set! (when #t 3) 21)")

(test (catch #t (lambda () (call-with-exit (lambda (go) (set! (go 1) 2)))) (lambda (type info) (apply format #f info)))
      "go (a goto (from call-with-exit)) does not have a setter: (set! (go 1) 2)")

(test (catch #t (lambda () (eval '(call/cc (lambda (go) (set! (go 1) 2))))) (lambda (type info) (apply format #f info)))
      "go (a continuation) does not have a setter: (set! (go 1) 2)")

(test (catch #t (lambda () (let ((L (inlet))) (let-ref L 'a :asdf))) (lambda (type info) (apply format #f info)))
      "let-ref: too many arguments: (let-ref (inlet) a :asdf)")

(test (catch #t (lambda () (let ((V (vector 1 2))) (set! (vector-ref V 0 1) 32))) (lambda (type info) (apply format #f info)))
      "too many arguments for vector-set!: (#(1 2) 0 1 32)")

(test (catch #t (lambda () (let ((V (vector 1 2))) (vector-set! V 0 1 32))) (lambda (type info) (apply format #f info)))
      "too many arguments for vector-set!: (#(1 2) 0 1 32)")

(test (catch #t (lambda () (let ((V (vector 1 2))) (set! (V 0 1) 32))) (lambda (type info) (apply format #f info)))
      "in (set! (V 0 1) 32), (#(1 2) 0) is 1 which can't take arguments")

(test (catch #t (lambda () (set! (:asdf 3) 2)) (lambda (type info) (apply format #f info)))
      "in (set! (:asdf 3) 2), :asdf has no setter")

(test (catch #t (lambda () (set! (_asdf_ 3) 3)) (lambda (type info) (apply format #f info)))
      "unbound variable _asdf_ in (set! (_asdf_ 3) 3)")

(test (catch #t (lambda () (make-hash-table 8 eq? #t)) (lambda (type info) (apply format #f info)))
        "make-hash-table third argument, #t, is boolean but should be either #f or (cons key-type-check value-type-check)")

(test (catch #t (lambda () (make-hash-table 8 eq? (cons integer? ()))) (lambda (type info) (apply format #f info)))
        "make-hash-table third argument, (integer?), is a pair but should be (key-type . value-type)")

(test (catch #t (lambda () (make-hash-table 8 eq? (cons (lambda (x) x) integer?))) (lambda (type info) (apply format #f info)))
        "make-hash-table: in the third argument, (#<lambda (x)> . integer?), (the key/value type checkers) the first function is anonymous")

(test (catch #t (lambda () (make-hash-table 8 eq? (cons integer? (lambda (x) x)))) (lambda (type info) (apply format #f info)))
        "make-hash-table: in the third argument, (integer? . #<lambda (x)>), (the key/value type checkers) the second function is anonymous")

(test (catch #t (lambda () (make-hash-table 8 eq? (cons (lambda (a b) a) integer?))) (lambda (type info) (apply format #f info)))
        "make-hash-table: in the third argument, (#<lambda (a b)> . integer?), (the key/value type checkers) both functions should take one argument")

(test (catch #t (lambda () (make-hash-table 8 eq? (cons integer? (lambda (a b) a)))) (lambda (type info) (apply format #f info)))
        "make-hash-table: in the third argument, (integer? . #<lambda (a b)>), (the key/value type checkers) both functions should take one argument")

(test (catch #t (lambda () (make-hash-table 8 eq? (cons expt integer?))) (lambda (type info) (apply format #f info)))
        "make-hash-table: in the third argument, (expt . integer?), (the key/value type checkers) both functions should take one argument")

(test (catch #t (lambda () (make-hash-table 8 eq? (cons integer? expt))) (lambda (type info) (apply format #f info)))
        "make-hash-table: in the third argument, (integer? . expt), (the key/value type checkers) both functions should take one argument")

(test (catch #t (lambda () (make-hash-table 8 string=? (cons symbol? symbol?))) (lambda (type info) (apply format #f info)))
        "make-hash-table: in the third argument, the key type function is not compatible with the equality function: (symbol? . symbol?)")

(define (hash-table-equalizer h)
  (let ((data (object->let h)))
    (and (defined? 'function data #t)
	 (let-ref data 'function))))

(define (hash-table-key/value-types h)
  (let* ((data (object->let h))
	 (sig (and (defined? 'signature data #t)
		   (let-ref data 'signature))))
    (and (pair? sig)
	 (list (caddr sig)
	       (car sig)))))

(test (hash-table-equalizer (make-hash-table 8 eq? (cons symbol? integer?))) 'eq?)
(test (hash-table-key/value-types (make-hash-table 8 eq? (cons symbol? integer?))) '(symbol? integer?))

(test (let ((H (hash-table 'a (hash-table 'b 1)))) (apply H (list 'a 'b))) 1)

(let ((ht1 (make-hash-table 31))
      (ht2 (make-hash-table 31)))
  (if (not (equal? ht1 ht2))
      (format #t ";ht1 and ht2 are empty, but not equal??~%"))

      ;; these first tests take advantage of s7's hashing function
  (hash-table-set! ht1 'abc 1)
  (hash-table-set! ht1 'abcabc 2)
  (hash-table-set! ht1 'abcabcabc 3)
  (hash-table-set! ht2 'abcabcabc 3)
  (hash-table-set! ht2 'abcabc 2)
  (hash-table-set! ht2 'abc 1)
  (if (not (equal? ht1 ht2))
      (format #t ";ht1 and ht2 have the same key value pairs, but are not equal??~%"))

  (test (make-hash-table 1 (call-with-exit (lambda (goto) goto))) 'error)
  (test (make-hash-table 1 atan) 'error)

  (set! ht2 (make-hash-table 31))
  (hash-table-set! ht2 'abc 1)
  (hash-table-set! ht2 'abcabc 2)
  (hash-table-set! ht2 'abcabcabc 3)
  (if (not (equal? ht1 ht2))
      (format #t ";ht1 and ht2 have the same key value pairs in the same order, but are not equal??~%"))

  (hash-table-set! ht2 'abc "1")
  (if (equal? ht1 ht2)
      (format #t ";ht1 and ht2 are equal but values are not~%"))
  (hash-table-set! ht2 'abc 1)
  (if (not (equal? ht1 ht2))
      (format #t ";after reset ht1 and ht2 have the same key value pairs in the same order, but are not equal??~%"))
  (hash-table-set! ht2 1 'abc)
  (if (equal? ht1 ht2)
      (format #t ";ht1 and ht2 are equal but entries are not~%"))
  (hash-table-set! ht1 1 'abc)
  (if (not (equal? ht1 ht2))
      (format #t ";after add ht1 and ht2 have the same key value pairs, but are not equal??~%"))

      ;; these should force chaining in any case
  (set! ht1 (make-hash-table 31))
  (set! ht2 (make-hash-table 60))
  (do ((i 0 (+ i 1)))
      ((= i 100))
    (hash-table-set! ht1 i (* i 2))
    (hash-table-set! ht2 i (* i 2)))
  (if (not (equal? ht1 ht2))
      (format #t ";ht1 and ht2 have the same (integer) key value pairs in the same order, but are not equal??~%"))

  (let ((h1 (hash-table "a" 1))
          (h2 (hash-table 'a 1)))
     (set! (h2 "a") 1)
     (set! (h2 'a) #f)
     test (equal? h1 h2) #t)

  (let ((ht (make-hash-table)))
    (set! (ht (expt 2 40)) 40)
    (set! (ht (expt 2 50)) 50)
    (set! (ht (- (expt 2 60))) -60) ; these all hash into 0 unfortunately -- maybe fold halves?
    (test (ht (expt 2 40)) 40)
    (test (ht (expt 2 50)) 50)
    (test (ht (expt 2 60)) #f)
    (test (ht (- (expt 2 60))) -60)
    (test (ht (expt 2 41)) #f))

  (set! ht2 (make-hash-table 31))
  (do ((i 99 (- i 1)))
      ((< i 0))
    (hash-table-set! ht2 i (* i 2)))
  (test (hash-table-entries ht2) 100)
  (if (not (equal? ht1 ht2))
      (format #t ";ht1 and ht2 have the same (integer) key value pairs, but are not equal??~%"))

  (fill! ht1 ())
  (test (hash-table-entries ht1) 100)
  (test (ht1 32) ()))

(let ((h (make-hash-table)))
  (test (hash-table-entries h) 0)
  (set! (h 'a) 1)
  (test (hash-table-entries h) 1)
  (set! (h 'a) #f)
  (test (hash-table-entries h) 0)
  (set! (h 'a) 'b)
  (test (h 'a) 'b))

(let ((h (make-hash-table)))
  (define (f t) (set! (t 'a) 'b))
  (f h)
  (f h)
  (test (h 'a) 'b))

(let ((ht (make-hash-table))
      (l1 '(x y z))
      (l2 '(y x z)))
  (set! (hash-table-ref ht 'x) 123)
  (define (hi)
    (hash-table-ref ht (cadr l1))) ; 123
  (test (hi) #f))

(let () ; hash-code
  (define (make-hash size)
    (make-vector size ()))

  (define (hash-ref table key)
    (let ((loc (modulo (hash-code key) (length table))))
      (cond ((assoc key (table loc)) => cdr)
	    (else #f))))

  (define (hash-set! table key value)
    (let ((loc (modulo (hash-code key) (length table))))
      (cond ((assoc key (table loc)) => (lambda (key/value)
					  (set-cdr! key/value value)))
	    (else (set! (table loc) (cons (cons key value) (table loc)))))))

  (let ((h (make-hash 8)))
    (hash-set! h 'abc 1)
    (test (hash-ref h 'abc) 1)
    (hash-set! h "cba" #\c)
    (test (hash-ref h "cba") #\c)
    (do ((i 0 (+ i 1)))
	((= i 12))
      (hash-set! h i (* i 2)))
    (test (hash-ref h 3) 6)
    (test (hash-ref h 'abc) 1)
    (test (hash-ref h "asdf") #f)))

(test (make-hash-table most-positive-fixnum) 'error)
;(test (make-hash-table (+ 1 (expt 2 31))) 'error)  ; out-of-memory error except in clang
(test (make-hash-table most-negative-fixnum) 'error)
(test (make-hash-table (* 8796093022208 8796093022208)) 'error)
(test (make-hash-table 8796093022208) 'error)
(test (make-hash-table 21 eq? 12) 'error)
(test (make-hash-table 21 12) 'error)
(test (make-hash-table 21 eq? #f 12) 'error)
(test (hash-table? (make-hash-table 8 #f #f)) #t)
(test (make-hash-table eq? eq?) 'error)
(test (make-hash-table eq? eq? 12) 'error)
(test (make-hash-table ()) 'error)
(test (make-hash-table 3 ()) 'error)
(test (make-hash-table eq? ()) 'error)
(test (make-hash-table 0) 'error)
(test (make-hash-table -4) 'error)
(test (make-hash-table (ash 1 32)) 'error)
(test (let ((imh (immutable! (hash-table 'a 1 'b 2))))
	(define (func) (do ((j 0 (+ j 1))) ((= j 1)) (hash-table-set! (or imh) 't 1)))
	(define (hi) (func)) (hi))
      'error)

(test (let ((h (hash-table 'a (hash-table 'b 2)))) (h 'a 'b)) 2)
(let ((h (hash-table)))
  (hash-table-set! h 'a (+ (or (hash-table-ref h 'a) 0) 1))
  (test (hash-table-ref h 'a) 1)
  (hash-table-set! h 'a (+ (or (hash-table-ref h 'a) 0) 1))
  (test (hash-table-ref h 'a) 2))
(let ((h (hash-table)))
  (define (hash-inc)
    (hash-table-set! h 'a (+ 1 (or (hash-table-ref h 'a) 0)))
    (hash-table-set! h 'a (+ (or (hash-table-ref h 'a) 0) 1)))
  (hash-inc)
  (test (hash-table-ref h 'a) 2))

(for-each
 (lambda (arg)
   (test (make-hash-table arg) 'error))
 (list "hi" #\a 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test (catch 'wrong-type-arg
	(lambda () (make-hash-table 8 (cons 32 symbol?)))
	(lambda (type info) (apply format #f info)))
      "make-hash-table: first entry of type info, 32, is an integer, but should be a function")
(test (catch 'wrong-type-arg
	(lambda () (make-hash-table 8 (cons symbol? 32)))
	(lambda (type info) (apply format #f info)))
      "make-hash-table: second entry of type info, 32, is an integer, but should be a function")
(test (catch 'wrong-type-arg
	(lambda () (make-hash-table 8 (cons symbol? symbol?)))
	(lambda (type info) (apply format #f info)))
      "make-hash-table second argument, symbol?, is a c-function but should be a function of two arguments")
(test (catch 'wrong-type-arg
	(lambda () (make-hash-table 8 (cons eq? symbol?)))
	(lambda (type info) (apply format #f info)))
      "make-hash-table mapper function, symbol?, should return an integer")
(test (catch 'wrong-type-arg
	(lambda () (make-hash-table 8 (cons string-ref symbol?)))
	(lambda (type info) (apply format #f info)))
      "make-hash-table checker function, string-ref, should return a boolean value")
(test (catch 'wrong-type-arg
	(lambda () (make-hash-table 8 (cons symbol? symbol?)))
	(lambda (type info) (apply format #f info)))
      "make-hash-table second argument, symbol?, is a c-function but should be a function of two arguments")
(test (copy (inlet 'a 1) (make-hash-table 8 #f (cons symbol? cons))) 'error)
(test (copy (inlet 'a 1) (make-hash-table 8 #f (cons cons symbol?))) 'error)

(let ((h (hash-table)))
  (hash-table-set! h 0/0 1)
  (test (object->string h) "(hash-table +nan.0 1)")
  (hash-table-set! h 0/0 #f)
  (test (object->string h) "(hash-table +nan.0 1)"))
(let ((h (hash-table +nan.0 1)))
  (test (h +nan.0) #f))

(let ((ht (hash-table :a 1/0)))
  (test (nan? (ht :a)) #t)
  (set! (ht 1/0) :a)
  (test (ht 1/0) #f)) ; NaNs aren't equal?

(test (hash-table 'a #f 'b 1) (hash-table 'b 1))
(test (hash-table 'a #f) (hash-table))

(let ((ht (make-hash-table)))
  (define (f1) (do ((i 0 (+ i 1))) ((= i 100)) (hash-table-set! ht i #t)))
  (f1)
  (test (hash-table-entries ht) 100)
  (set! ht (make-hash-table))
  (define (f2) (do ((i 0 (+ i 1))) ((= i 100)) (hash-table-set! ht i 0)))
  (f2)
  (test (hash-table-entries ht) 100)
  (set! ht (make-hash-table))
  (define (f3) (do ((i 0 (+ i 1))) ((= i 100)) (hash-table-set! ht i i)))
  (f3)
  (test (hash-table-entries ht) 100))

(let ((ht (make-hash-table)))
  (define (f1) (do ((i 0 (+ i 1))) ((= i 1000)) (hash-table-set! ht i #t)))
  (f1)
  (test (hash-table-entries ht) 1000))

(let ((hi (make-hash-table 7)))
  (test (object->string hi) "(hash-table)")
  (set! (hi 1) "1")
  (test (object->string hi) "(hash-table 1 \"1\")")
  (set! (hi -1) "-1")

  (test (or (string=? (object->string hi) "(hash-table -1 \"-1\" 1 \"1\")")
	    (string=? (object->string hi) "(hash-table 1 \"1\" -1 \"-1\")"))
	#t)
  (set! (hi 9) "9")
  (test (or (string=? (object->string hi) "(hash-table 9 \"9\" -1 \"-1\" 1 \"1\")")
	    (string=? (object->string hi) "(hash-table 9 \"9\" 1 \"1\" -1 \"-1\")"))
	#t)
  (set! (hi -9) "-9")
  (test (or (string=? (object->string hi) "(hash-table -9 \"-9\" 9 \"9\" -1 \"-1\" 1 \"1\")")
	    (string=? (object->string hi) "(hash-table 9 \"9\" 1 \"1\" -9 \"-9\" -1 \"-1\")"))
	#t)
  (test (hi 1) "1")
  (test (hi -1) "-1")
  (test (hi -9) "-9")
  (set! (hi 2) "2")
  (test (or (string=? (object->string hi) "(hash-table -9 \"-9\" 9 \"9\" -1 \"-1\" 1 \"1\" 2 \"2\")")
	    (string=? (object->string hi) "(hash-table 9 \"9\" 1 \"1\" 2 \"2\" -9 \"-9\" -1 \"-1\")"))
	#t)
  (let-temporarily (((*s7* 'print-length) 3))
    (test (or (string=? (object->string hi) "(hash-table -9 \"-9\" 9 \"9\" -1 \"-1\" ...)")
	      (string=? (object->string hi) "(hash-table 9 \"9\" 1 \"1\" 2 \"2\" ...)"))
	  #t)
    (set! (*s7* 'print-length) 0)
    (test (object->string hi) "(hash-table ...)")
    (test (object->string (hash-table)) "(hash-table)")))

(let ((ht (make-hash-table 277)))
  (test (hash-table? ht) #t)
  (test (>= (length ht) 277) #t)
  (test (hash-table-entries ht) 0)
  (test (let () (hash-table-set! ht 'key 3.14) (hash-table-ref ht 'key)) 3.14)
  (test (let () (hash-table-set! ht "ky" 3.14) (hash-table-ref ht "ky")) 3.14)
  (for-each
   (lambda (arg)
     (test (let () (hash-table-set! ht 'key arg) (hash-table-ref ht 'key)) arg))
   (list "hi" -1 #\a 1 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2))))

(for-each
 (lambda (arg)
   (test (hash-table? arg) #f))
 (list "hi" -1 #\a 1 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #t #f () #(()) (list 1 2 3) '(1 . 2)))

(test (hash-table? (make-vector 3 ())) #f)
(test (let ((ht (make-hash-table))) (set! (ht 'a) 123) (map values ht)) '((a . 123)))

(let ((ht (make-hash-table)))
  (test (hash-table-ref ht 'not-a-key) #f)
  (test (hash-table-ref ht "not-a-key") #f)
  (hash-table-set! ht 'key 3/4)
  (hash-table-set! ht "key" "hi")
  (test (hash-table-ref ht "key") "hi")
  (test (hash-table-ref ht 'key) 3/4)

  (hash-table-set! ht 'asd 'hiho)
  (test (hash-table-ref ht 'asd) 'hiho)
  (hash-table-set! ht 'asd 1234)
  (test (hash-table-ref ht 'asd) 1234))

(let ((ht (make-hash-table)))
  (define (ht-add h)
    (+ (h 1) (h 2)))
  (hash-table-set! ht 1 2)
  (hash-table-set! ht 2 3)
  (test (ht-add ht) 5))

(let ()
  (define h4 (make-hash-table 8 =))
  (test (set! (h4 'a) 3) (if (> (*s7* 'safety) 0) 'error 3))
  (define h5 (make-hash-table 8 string=?))
  (test (set! (h5 'a) 3) (if (> (*s7* 'safety) 0) 'error 3))
  (define h6 (make-hash-table 8 eq?))
  (test (set! (h6 21) 3) (if (> (*s7* 'safety) 0) 'error 3))
  (test (make-hash-table 8 >) 'error))

(let ((let1 (inlet 'a 1))
      (let2 (inlet 'a 1))
      (let3 (inlet 'a 2))
      (let4 (inlet 'b 1))
      (let5 (inlet 'a 1 'a 2)))
  (test (equal? let1 let2) #t)
  (test (equal? let1 let3) #f)
  (test (equal? let1 let5) #t)
  (let ((hash1 (hash-table let1 32)))
    (test (integer? (hash1 let1)) #t)
    (test (integer? (hash1 let2)) #t)
    (test (integer? (hash1 let3)) #f)
    (test (integer? (hash1 let4)) #f)
    (test (integer? (hash1 let5)) #t)))

(test ((hash-table 1.5 #t #f #t) #f) #t) ; this is checking hash_float if debugging
(test ((hash-table 1.5 #t 1 #t) 1) #t)

(let ((let1 (inlet 'a 1 'b 2))
      (let2 (inlet 'b 2 'a 1))
      (let3 (inlet 'a 1 'b 1)))
  (test (equal? let1 let2) #t)
  (let ((hash1 (hash-table let1 32)))
    (test (integer? (hash1 let1)) #t)
    (test (integer? (hash1 let2)) #t)
    (test (integer? (hash1 let3)) #f)))

(let ((hash1 (hash-table 'a 1 'b 2))
      (hash2 (hash-table 'b 2 'a 1)))
  (test (equal? hash1 hash2) #t)
  (let ((hash3 (hash-table hash1 32)))
    (test (integer? (hash3 hash1)) #t)
    (test (integer? (hash3 hash2)) #t)))

(let ((hash1 (hash-table 'b 2 'a 1)))
  (let ((hash2 (make-hash-table (* (length hash1) 2))))
    (set! (hash2 'a) 1)
    (set! (hash2 'b) 2)
    (test (equal? hash1 hash2) #t)
    (let ((hash3 (make-hash-table (* 2 (length hash2)))))
      (set! (hash3 hash1) 32)
      (test (integer? (hash3 hash1)) #t)
      (test (integer? (hash3 hash2)) #t))))

(for-each
 (lambda (arg)
   (test (hash-table-ref arg 'key) 'error))
 (list "hi" -1 #\a 1 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(let ((ht1 (make-hash-table 653))
      (ht2 (make-hash-table 277)))
  (test (equal? ht1 ht2) #t) ; equal? because both are empty
  (hash-table-set! ht1 'key 'hiho)
  (hash-table-set! ht2 (hash-table-ref ht1 'key) 3.14)
  (test (>= (length ht1) 653) #t)
  (test (hash-table-ref ht2 'hiho) 3.14)
  (test (hash-table-ref ht2 (hash-table-ref ht1 'key)) 3.14))

(let ((ht1 (make-hash-table)))
   (set! (ht1 1) 'hi)
   (let ((ht2 (make-hash-table)))
      (set! (ht2 1) ht1)
      (test ((ht2 1) 1) 'hi)))

(let ((ht1 (make-hash-table)))
   (set! (ht1 1/0) "NaN!")
   (let ((nan 1/0))
      (test (ht1 nan) #f)
      (set! (ht1 nan) 0)
      (test (ht1 nan) #f)
      (if (not with-windows)
	  (test (object->string ht1) "(hash-table +nan.0 0 +nan.0 \"NaN!\")"))))
(let ((h (make-hash-table 8 eqv?)))
  (set! (h +nan.0) 'ok)
  (test (h +nan.0) #f))

(let ((h1 (make-hash-table 8 equivalent?))
      (h2 (make-hash-table 8 equivalent?)))
  (hash-table-set! h2 (let ((<1> (vector #f))) (set! (<1> 0) <1>) <1>) 1)
  (test (object->string (append h1 h2)) "(hash-table #1=#(#1#) 1)"))

(unless with-bignums
  (let ((ht1 (make-hash-table)))
    (set! (ht1 1) "1")
    (set! (ht1 1.0) "1.0")
    (test (ht1 1) "1")
    (set! (ht1 1/0) "nan")
    (test (ht1 0/0) #f)
    (test (ht1 1/0) #f)
    (set! (ht1 (/ (log 0) (log 0))) "nan-nani")
    (test (ht1 (/ (log 0) (log 0))) #f)
    (test (ht1 (- 0/0)) #f)
    (test (ht1 (real-part (/ (log 0) (log 0)))) #f)
    (test (ht1 (complex 0/0 1/0)) #f)
    (set! (ht1 (real-part (log 0))) "-inf")
    (test (ht1 (real-part (log 0))) "-inf")
    (set! (ht1 (- (real-part (log 0)))) "inf")
    (test (ht1 (- (real-part (log 0)))) "inf")
    (set! (ht1 (log 0)) "log(0)")
    (test (ht1 (log 0)) "log(0)")
    (set! (ht1 (complex 80143857/25510582 1)) "pi+i")
    (test (ht1 (complex pi (- 1.0 1e-16))) #f)))

(when with-bignums
  (test (hash-table-ref (let ((h (make-hash-table 8 eqv?))) (set! (h 1.0) 'a) (set! (h 2.0) 'b) (set! (h 3.0) 'c) h) (bignum 2.0)) 'b)
  (test (hash-table-ref (let ((h (make-hash-table 8 eqv?))) (set! (h 1.0) 'a) (set! (h 2.0) 'b) h) (bignum 2.0)) 'b)
  (test (hash-table-ref (hash-table 1.0 'a 2.0 'b 3.0 'c) 2.0) 'b)
  (test (hash-table-ref (hash-table 1.0 'a (bignum 2.0) 'b 3.0 'c) 2.0) 'b)
  (test (hash-table-ref (hash-table 1.0 'a 2.0 'b 3.0 'c) (bignum 2.0)) 'b)
  (test (hash-table-ref (hash-table 1 'a 2 'b 3 'c) (bignum 2)) 'b)
  (test (hash-table-ref (hash-table 1/3 'a 2/3 'b 3/2 'c) (bignum 2/3)) 'b)
  (test (hash-table-ref (hash-table 1+i 'a 2/3+i 'b 3+i 'c) (bignum 2/3+i)) 'b)
  (test (hash-table-ref (hash-table (bignum 1) 'a (bignum 2) 'b (bignum 3) 'c) (bignum 2)) 'b)
  (test (hash-table-ref (hash-table (bignum 1) 'a (bignum 2) 'b (bignum 3) 'c) 2) 'b)
  (test (hash-table-ref (hash-table (bignum 1.0) 'a (bignum 2.0) 'b (bignum 3.0) 'c) (bignum 2.0)) 'b)
  (test (hash-table-ref (hash-table (bignum 1.0) 'a (bignum 2.0) 'b (bignum 3.0) 'c) 2.0) 'b)
  (test (hash-table-ref (let ((h (make-hash-table 8 =))) (set! (h 1.0) 'a) (set! (h 2.0) 'b) (set! (h 3.0) 'c) h) (bignum 2.0)) 'b)
  (test (hash-table-ref (let ((h (make-hash-table 8 equal?))) (set! (h 1.0) 'a) (set! (h 2.0) 'b) (set! (h 3.0) 'c) h) (bignum 2.0)) 'b)
  (test (hash-table-ref (let ((h (make-hash-table 8 eqv?))) (set! (h 1.0) 'a) (set! (h 2.0) 'b) (set! (h 3.0) 'c) h) (bignum 2.0)) 'b)
  (test (hash-table-ref (let ((h (make-hash-table 8 eqv?))) (set! (h 1.0) 'a) (set! (h 2.0) 'b) h) (bignum 2.0)) 'b)
  (test (hash-table-ref (let ((h (make-hash-table 8 equivalent?))) (set! (h 1.0) 'a) (set! (h 2.0) 'b) (set! (h 3.0) 'c) h) (bignum 2.0)) 'b))

(test (hash-table-ref (let ((h (make-hash-table 8 equivalent?))) (set! (h 10000000001/10000000000) 'a) h) 1) #f)
(let-temporarily (((*s7* 'hash-table-float-epsilon) 1.0e-9))
  (test (hash-table-ref (let ((h (make-hash-table 8 equivalent?))) (set! (h 10000000001/10000000000) 'a) h) 1) 'a)
  (test (hash-table-ref (let ((h (make-hash-table 8 equivalent?))) (set! (h 10000000001/10000000000) 'a) h) 1.0) 'a)
  (test (hash-table-ref (let ((h (make-hash-table 8 equivalent?))) (set! (h 10000000001/10000000000) 'a) h) 1.0+1.0e-10i) 'a))
(let-temporarily (((*s7* 'hash-table-float-epsilon) 1.0e-9))
  (test (hash-table-ref (let ((h (make-hash-table 8 equivalent?))) (set! (h -10000000001/10000000000) 'a) h) -1) 'a)
  (test (hash-table-ref (let ((h (make-hash-table 8 equivalent?))) (set! (h -10000000001/10000000000) 'a) h) -1.0) 'a)
  (test (hash-table-ref (let ((h (make-hash-table 8 equivalent?))) (set! (h -10000000001/10000000000) 'a) h) -1.0+1.0e-10i) 'a))
(let-temporarily (((*s7* 'hash-table-float-epsilon) 1.0e-9))
  (test (hash-table-ref (let ((h (make-hash-table 8 equivalent?))) (set! (h 1+1.0e-10i) 'a) h) 1) 'a)
  (test (hash-table-ref (let ((h (make-hash-table 8 equivalent?))) (set! (h 1+1.0e-10i) 'a) h) 1.0) 'a)
  (test (hash-table-ref (let ((h (make-hash-table 8 equivalent?))) (set! (h 1+1.0e-10i) 'a) h) 1.0-1.0e-10i) 'a))

(let-temporarily (((*s7* 'hash-table-float-epsilon) 1.0e-9))
  (let ((eps5 5.0e-10)
	(eps1 1.0e-10)
	(eps9 9.0e-10)
	(eps10 1.0001e-9)
	(h (make-hash-table 8 equivalent?))
	(keys (list 0 1.0 2.0 -2.0 1/2 7/2 1.5 2.5 3 (+ 1.0 5.0e-8) -1.0e-8 -11/2 0+i 1-i 8-1.0e-10i)))
    (for-each
     (lambda (key)
       (hash-table-set! h key (number->string key)))
     keys)
    (for-each
     (lambda (key)
       (let ((val (number->string key)))
	 (test (hash-table-ref h key) val)
         (when (real? key)
  	   (test (hash-table-ref h (inexact->exact key)) val)
	   (test (hash-table-ref h (exact->inexact key)) val))
	 (test (hash-table-ref h (+ key eps1)) val)
	 (test (hash-table-ref h (+ key eps5)) val)
	 (test (hash-table-ref h (+ key eps9)) val)
	 (test (hash-table-ref h (+ key (* (max (real-part key) 1.0) eps10))) #f)
	 (test (hash-table-ref h (- key eps1)) val)
	 (test (hash-table-ref h (- key eps5)) val)
	 (test (hash-table-ref h (- key eps9)) val)
	 (test (hash-table-ref h (- key (* (max (real-part key) 1.0) eps10))) #f)))
     keys)
    (set! h (make-hash-table 8 =))
    (for-each
     (lambda (key)
       (hash-table-set! h key (number->string key)))
     keys)
    (for-each
     (lambda (key)
       (let ((val (number->string key)))
	 (test (hash-table-ref h key) val)
         (when (real? key)
	   (test (hash-table-ref h (inexact->exact key)) (and (= key (inexact->exact key)) val))
	   (test (hash-table-ref h (exact->inexact key)) (and (= key (exact->inexact key)) val)))
	 (test (hash-table-ref h (+ key eps1)) #f)
	 (test (hash-table-ref h (+ key eps5)) #f)
	 (test (hash-table-ref h (+ key eps9)) #f)
	 (test (hash-table-ref h (+ key (* (max (real-part key) 1.0) eps10))) #f)
	 (test (hash-table-ref h (- key eps1)) #f)
	 (test (hash-table-ref h (- key eps5)) #f)
	 (test (hash-table-ref h (- key eps9)) #f)
	 (test (hash-table-ref h (- key (* (max (real-part key) 1.0) eps10))) #f)))
     keys)
    (set! h (make-hash-table 8 eqv?))
    (for-each
     (lambda (key)
       (hash-table-set! h key (number->string key)))
     keys)
    (for-each
     (lambda (key)
       (let ((val (number->string key)))
	 (test (hash-table-ref h key) val)
         (when (real? key)
	   (test (hash-table-ref h (inexact->exact key)) (and (exact? key) val))
	   (test (hash-table-ref h (exact->inexact key)) (and (inexact? key) val)))
	 (test (hash-table-ref h (+ key eps1)) #f)
	 (test (hash-table-ref h (+ key eps5)) #f)
	 (test (hash-table-ref h (+ key eps9)) #f)
	 (test (hash-table-ref h (+ key (* (max (real-part key) 1.0) eps10))) #f)
	 (test (hash-table-ref h (- key eps1)) #f)
	 (test (hash-table-ref h (- key eps5)) #f)
	 (test (hash-table-ref h (- key eps9)) #f)
	 (test (hash-table-ref h (- key (* (max (real-part key) 1.0) eps10))) #f)))
     keys)
    (set! h (make-hash-table 8 equal?)) ; like eqv?
    (for-each
     (lambda (key)
       (hash-table-set! h key (number->string key)))
     keys)
    (for-each
     (lambda (key)
       (let ((val (number->string key)))
	 (test (hash-table-ref h key) val)
         (when (real? key)
	   (test (hash-table-ref h (inexact->exact key)) (and (exact? key) val))
	   (test (hash-table-ref h (exact->inexact key)) (and (inexact? key) val)))
	 (test (hash-table-ref h (+ key eps1)) #f)
	 (test (hash-table-ref h (+ key eps5)) #f)
	 (test (hash-table-ref h (+ key eps9)) #f)
	 (test (hash-table-ref h (+ key (* (max (real-part key) 1.0) eps10))) #f)
	 (test (hash-table-ref h (- key eps1)) #f)
	 (test (hash-table-ref h (- key eps5)) #f)
	 (test (hash-table-ref h (- key eps9)) #f)
	 (test (hash-table-ref h (- key (* (max (real-part key) 1.0) eps10))) #f)))
     keys)))

(when with-bignums
  (let-temporarily (((*s7* 'hash-table-float-epsilon) 1.0e-9))
    (let ((eps5 5.0e-10)
	  (eps1 1.0e-10)
	  (eps9 9.0e-10)
	  (eps10 1.0001e-9)
	  (h (make-hash-table 8 equivalent?))
	  (keys (map bignum (list 0 1.0 2.0 -2.0 1/2 7/2 1.5 2.5 3 (+ 1.0 5.0e-8) -1.0e-8 -11/2 0+i 1-i))))
      (for-each
       (lambda (key)
	 (hash-table-set! h key (number->string key)))
       keys)
      (for-each
       (lambda (key)
	 (let ((val (number->string key)))
	   (test (hash-table-ref h key) val)
	   (when (real? key)
	     (test (hash-table-ref h (inexact->exact key)) val)
	     (test (hash-table-ref h (exact->inexact key)) val))
	   (test (hash-table-ref h (+ key eps1)) val)
	   (test (hash-table-ref h (+ key eps5)) val)
	   (test (hash-table-ref h (+ key eps9)) val)
	   (test (hash-table-ref h (+ key (* (max (real-part key) 1.0) eps10))) #f)
	   (test (hash-table-ref h (- key eps1)) val)
	   (test (hash-table-ref h (- key eps5)) val)
	   (test (hash-table-ref h (- key eps9)) val)
	   (test (hash-table-ref h (- key (* (max (real-part key) 1.0) eps10))) #f)))
       keys)

      (set! h (make-hash-table 8 =))
      (for-each
       (lambda (key)
	 (hash-table-set! h key (number->string key)))
       keys)
      (for-each
       (lambda (key)
	 (let ((val (number->string key)))
	   (test (hash-table-ref h key) val)
	   (when (real? key)
	     (test (hash-table-ref h (inexact->exact key)) (and (= key (inexact->exact key)) val))
	     (test (hash-table-ref h (exact->inexact key)) (and (= key (exact->inexact key)) val)))
	   (test (hash-table-ref h (+ key eps1)) #f)
	   (test (hash-table-ref h (+ key eps5)) #f)
	   (test (hash-table-ref h (+ key eps9)) #f)
	   (test (hash-table-ref h (+ key (* (max (real-part key) 1.0) eps10))) #f)
	   (test (hash-table-ref h (- key eps1)) #f)
	   (test (hash-table-ref h (- key eps5)) #f)
	   (test (hash-table-ref h (- key eps9)) #f)
	   (test (hash-table-ref h (- key (* (max (real-part key) 1.0) eps10))) #f)))
       keys)
      (set! h (make-hash-table 8 eqv?))
      (for-each
       (lambda (key)
	 (hash-table-set! h key (number->string key)))
       keys)
      (for-each
       (lambda (key)
	 (let ((val (number->string key)))
	   (test (hash-table-ref h key) val)
	   (when (real? key)
	     (test (hash-table-ref h (inexact->exact key)) (and (exact? key) val))
	     (test (hash-table-ref h (exact->inexact key)) (and (inexact? key) val)))
	   (test (hash-table-ref h (+ key eps1)) #f)
	   (test (hash-table-ref h (+ key eps5)) #f)
	   (test (hash-table-ref h (+ key eps9)) #f)
	   (test (hash-table-ref h (+ key (* (max (real-part key) 1.0) eps10))) #f)
	   (test (hash-table-ref h (- key eps1)) #f)
	   (test (hash-table-ref h (- key eps5)) #f)
	   (test (hash-table-ref h (- key eps9)) #f)
	   (test (hash-table-ref h (- key (* (max (real-part key) 1.0) eps10))) #f)))
       keys)
      (set! h (make-hash-table 8 equal?)) ; like eqv?
      (for-each
       (lambda (key)
	 (hash-table-set! h key (number->string key)))
       keys)
      (for-each
       (lambda (key)
	 (let ((val (number->string key)))
	   (test (hash-table-ref h key) val)
	   (when (real? key)
	     (test (hash-table-ref h (inexact->exact key)) (and (exact? key) val))
	     (test (hash-table-ref h (exact->inexact key)) (and (inexact? key) val)))
	   (test (hash-table-ref h (+ key eps1)) #f)
	   (test (hash-table-ref h (+ key eps5)) #f)
	   (test (hash-table-ref h (+ key eps9)) #f)
	   (test (hash-table-ref h (+ key (* (max (real-part key) 1.0) eps10))) #f)
	   (test (hash-table-ref h (- key eps1)) #f)
	   (test (hash-table-ref h (- key eps5)) #f)
	   (test (hash-table-ref h (- key eps9)) #f)
	   (test (hash-table-ref h (- key (* (max (real-part key) 1.0) eps10))) #f)))
       keys)))


  (define (big-hash-table-ref table key)
    (hash-table-ref table (bignum key)))

  (let-temporarily (((*s7* 'hash-table-float-epsilon) 1.0e-9))
    (let ((eps5 5.0e-10)
	  (eps1 1.0e-10)
	  (eps9 9.0e-10)
	  (eps10 1.0001e-9)
	  (h (make-hash-table 8 equivalent?))
	  (keys (list 0 1.0 2.0 -2.0 1/2 7/2 1.5 2.5 3 (+ 1.0 5.0e-8) -1.0e-8 -11/2 0+i 1-i)))
      (for-each
       (lambda (key)
	 (hash-table-set! h key (number->string key)))
       keys)
      (for-each
       (lambda (key)
	 (let ((val (number->string key)))
	   (test (hash-table-ref h key) val)
	   (when (real? key)
	     (test (big-hash-table-ref h (inexact->exact key)) val)
	     (test (big-hash-table-ref h (exact->inexact key)) val))
	   (test (big-hash-table-ref h (+ key eps1)) val)
	   (test (big-hash-table-ref h (+ key eps5)) val)
	   (test (big-hash-table-ref h (+ key eps9)) val)
	   (test (big-hash-table-ref h (+ key (* (max (real-part key) 1.0) eps10))) #f)
	   (test (big-hash-table-ref h (- key eps1)) val)
	   (test (big-hash-table-ref h (- key eps5)) val)
	   (test (big-hash-table-ref h (- key eps9)) val)
	   (test (big-hash-table-ref h (- key (* (max (real-part key) 1.0) eps10))) #f)))
       keys)
      (set! h (make-hash-table 8 =))
      (for-each
       (lambda (key)
	 (hash-table-set! h key (number->string key)))
       keys)
      (for-each
       (lambda (key)
	 (let ((val (number->string key)))
	   (test (hash-table-ref h key) val)
	   (when (real? key)
	     (test (big-hash-table-ref h (inexact->exact key)) (and (= key (inexact->exact key)) val))
	     (test (big-hash-table-ref h (exact->inexact key)) (and (= key (exact->inexact key)) val)))
	   (test (big-hash-table-ref h (+ key eps1)) #f)
	   (test (big-hash-table-ref h (+ key eps5)) #f)
	   (test (big-hash-table-ref h (+ key eps9)) #f)
	   (test (big-hash-table-ref h (+ key (* (max (real-part key) 1.0) eps10))) #f)
	   (test (big-hash-table-ref h (- key eps1)) #f)
	   (test (big-hash-table-ref h (- key eps5)) #f)
	   (test (big-hash-table-ref h (- key eps9)) #f)
	   (test (big-hash-table-ref h (- key (* (max (real-part key) 1.0) eps10))) #f)))
       keys)
      (set! h (make-hash-table 8 eqv?))
      (for-each
       (lambda (key)
	 (hash-table-set! h key (number->string key)))
       keys)
      (for-each
       (lambda (key)
	 (let ((val (number->string key)))
	   (test (hash-table-ref h key) val)
	   (when (real? key)
	     (test (big-hash-table-ref h (inexact->exact key)) (and (exact? key) val))
	     (test (big-hash-table-ref h (exact->inexact key)) (and (inexact? key) val)))
	   (test (big-hash-table-ref h (+ key eps1)) #f)
	   (test (big-hash-table-ref h (+ key eps5)) #f)
	   (test (big-hash-table-ref h (+ key eps9)) #f)
	   (test (big-hash-table-ref h (+ key (* (max (real-part key) 1.0) eps10))) #f)
	   (test (big-hash-table-ref h (- key eps1)) #f)
	   (test (big-hash-table-ref h (- key eps5)) #f)
	   (test (big-hash-table-ref h (- key eps9)) #f)
	   (test (big-hash-table-ref h (- key (* (max (real-part key) 1.0) eps10))) #f)))
       keys)
      (set! h (make-hash-table 8 equal?)) ; like eqv?
      (for-each
       (lambda (key)
	 (hash-table-set! h key (number->string key)))
       keys)
      (for-each
       (lambda (key)
	 (let ((val (number->string key)))
	   (test (hash-table-ref h key) val)
	   (when (real? key)
	     (test (big-hash-table-ref h (inexact->exact key)) (and (exact? key) val))
	     (test (big-hash-table-ref h (exact->inexact key)) (and (inexact? key) val)))
	   (test (big-hash-table-ref h (+ key eps1)) #f)
	   (test (big-hash-table-ref h (+ key eps5)) #f)
	   (test (big-hash-table-ref h (+ key eps9)) #f)
	   (test (big-hash-table-ref h (+ key (* (max (real-part key) 1.0) eps10))) #f)
	   (test (big-hash-table-ref h (- key eps1)) #f)
	   (test (big-hash-table-ref h (- key eps5)) #f)
	   (test (big-hash-table-ref h (- key eps9)) #f)
	   (test (big-hash-table-ref h (- key (* (max (real-part key) 1.0) eps10))) #f)))
       keys)))

  (let-temporarily (((*s7* 'hash-table-float-epsilon) 1.0e-9))
    (let ((eps5 5.0e-10)
	  (eps1 1.0e-10)
	  (eps9 9.0e-10)
	  (eps10 1.0001e-9)
	  (h (make-hash-table 8 equivalent?))
	  (keys (list 0 1.0 2.0 -2.0 1/2 7/2 1.5 2.5 3 (+ 1.0 5.0e-8) -1.0e-8 -11/2 0+i 1-i)))
      (for-each
       (lambda (key)
	 (hash-table-set! h (bignum key) (number->string (bignum key))))
       keys)
      (for-each
       (lambda (key)
	 (let ((val (number->string (bignum key))))
	   (test (hash-table-ref h key) val)
	   (when (real? key)
	     (test (hash-table-ref h (inexact->exact key)) val)
	     (test (hash-table-ref h (exact->inexact key)) val))
	   (test (hash-table-ref h (+ key eps1)) val)
	   (test (hash-table-ref h (+ key eps5)) val)
	   (test (hash-table-ref h (+ key eps9)) val)
	   (test (hash-table-ref h (+ key (* (max (real-part key) 1.0) eps10))) #f)
	   (test (hash-table-ref h (- key eps1)) val)
	   (test (hash-table-ref h (- key eps5)) val)
	   (test (hash-table-ref h (- key eps9)) val)
	   (test (hash-table-ref h (- key (* (max (real-part key) 1.0) eps10))) #f)))
       keys)
      (set! h (make-hash-table 8 =))
      (for-each
       (lambda (key)
	 (hash-table-set! h (bignum key) (number->string (bignum key))))
       keys)
      (for-each
       (lambda (key)
	 (let ((val (number->string (bignum key))))
	   (test (hash-table-ref h key) val)
	   (when (real? key)
	     (test (hash-table-ref h (inexact->exact key)) (and (= key (inexact->exact key)) val))
	     (test (hash-table-ref h (exact->inexact key)) (and (= key (exact->inexact key)) val)))
	   (test (hash-table-ref h (+ key eps1)) #f)
	   (test (hash-table-ref h (+ key eps5)) #f)
	   (test (hash-table-ref h (+ key eps9)) #f)
	   (test (hash-table-ref h (+ key (* (max (real-part key) 1.0) eps10))) #f)
	   (test (hash-table-ref h (- key eps1)) #f)
	   (test (hash-table-ref h (- key eps5)) #f)
	   (test (hash-table-ref h (- key eps9)) #f)
	   (test (hash-table-ref h (- key (* (max (real-part key) 1.0) eps10))) #f)))
       keys)
      (set! h (make-hash-table 8 eqv?))
      (for-each
       (lambda (key)
	 (hash-table-set! h (bignum key) (number->string (bignum key))))
       keys)
      (for-each
       (lambda (key)
	 (let ((val (number->string (bignum key))))
	   (test (hash-table-ref h key) val)
	   (when (real? key)
	     (test (hash-table-ref h (inexact->exact key)) (and (exact? key) val))
	     (test (hash-table-ref h (exact->inexact key)) (and (inexact? key) val)))
	   (test (hash-table-ref h (+ key eps1)) #f)
	   (test (hash-table-ref h (+ key eps5)) #f)
	   (test (hash-table-ref h (+ key eps9)) #f)
	   (test (hash-table-ref h (+ key (* (max (real-part key) 1.0) eps10))) #f)
	   (test (hash-table-ref h (- key eps1)) #f)
	   (test (hash-table-ref h (- key eps5)) #f)
	   (test (hash-table-ref h (- key eps9)) #f)
	   (test (hash-table-ref h (- key (* (max (real-part key) 1.0) eps10))) #f)))
       keys)
      (set! h (make-hash-table 8 equal?)) ; like eqv?
      (for-each
       (lambda (key)
	 (hash-table-set! h (bignum key) (number->string (bignum key))))
       keys)
      (for-each
       (lambda (key)
	 (let ((val (number->string (bignum key))))
	   (test (hash-table-ref h key) val)
	   (when (real? key)
	     (test (hash-table-ref h (inexact->exact key)) (and (exact? key) val))
	     (test (hash-table-ref h (exact->inexact key)) (and (inexact? key) val)))
	   (test (hash-table-ref h (+ key eps1)) #f)
	   (test (hash-table-ref h (+ key eps5)) #f)
	   (test (hash-table-ref h (+ key eps9)) #f)
	   (test (hash-table-ref h (+ key (* (max (real-part key) 1.0) eps10))) #f)
	   (test (hash-table-ref h (- key eps1)) #f)
	   (test (hash-table-ref h (- key eps5)) #f)
	   (test (hash-table-ref h (- key eps9)) #f)
	   (test (hash-table-ref h (- key (* (max (real-part key) 1.0) eps10))) #f)))
       keys))))

(let-temporarily (((*s7* 'hash-table-float-epsilon) 1.0e-9))
  (let ((eps5 5.0e-10)
	(eps1 1.0e-10)
	(eps9 9.0e-10)
	(eps10 1.0001e-9)
	(h (make-hash-table 8 equivalent?))
	(keys (list +inf.0 -inf.0 +nan.0 -nan.0 +inf.0+i 1-inf.0i)))
    (for-each
     (lambda (key)
       (hash-table-set! h key (number->string key)))
     keys)
    (for-each
     (lambda (key)
       (let ((val (number->string key)))
	 (test (hash-table-ref h key) val)
	 (test (hash-table-ref h (+ key eps1)) val)
	 (test (hash-table-ref h (+ key eps5)) val)
	 (test (hash-table-ref h (+ key eps9)) val)
	 (test (hash-table-ref h (- key eps1)) val)
	 (test (hash-table-ref h (- key eps5)) val)
	 (test (hash-table-ref h (- key eps9)) val)
	 ))
     keys)
    (set! h (make-hash-table 8 =)) ; infinities are =: (= +inf.0 (+ +inf.0 1.0e-10))
    (for-each
     (lambda (key)
       (hash-table-set! h key (number->string key)))
     keys)
    (for-each
     (lambda (key)
       (let ((val (and (not (nan? key)) (number->string key))))
	 (test (hash-table-ref h key) val)
	 (test (hash-table-ref h (+ key eps1)) (and (infinite? (real-part key)) val)) ; eps is applied to real part
	 (test (hash-table-ref h (+ key eps5)) (and (infinite? (real-part key)) val))
	 (test (hash-table-ref h (+ key eps9)) (and (infinite? (real-part key)) val))
	 (test (hash-table-ref h (- key eps1)) (and (infinite? (real-part key)) val))
	 (test (hash-table-ref h (- key eps5)) (and (infinite? (real-part key)) val))
	 (test (hash-table-ref h (- key eps9)) (and (infinite? (real-part key)) val))
	 ))
     keys)))

(when with-bignums
 (let-temporarily (((*s7* 'hash-table-float-epsilon) 1.0e-9))
  (let ((eps5 5.0e-10)
	(eps1 1.0e-10)
	(eps9 9.0e-10)
	(eps10 1.0001e-9)
	(h (make-hash-table 8 equivalent?))
	(keys (list
	            18446744073709551614
		    18446744073709551614/3
		    18446744073709551615.0
		    18446744073709551614+i
		    -18446744073709551616
		    -18446744073709551616/3
		    -18446744073709551617.0
		    -18446744073709551616-18446744073709551614i
		    9223372036854775807
		    -9223372036854775807
		    9007199254740992
		    9223372036854775800.1
		    -9223372036854775800.1+i
		    92233720368547758/19
		    92233720368547758/5
		    )))
    (for-each
     (lambda (key)
       (hash-table-set! h key (number->string key)))
     keys)
    (for-each
     (lambda (key)
       (let ((val (number->string key)))
	 (test (hash-table-ref h key) val)
         (when (real? key)
  	   (test (hash-table-ref h (inexact->exact key)) val)
	   (test (hash-table-ref h (exact->inexact key)) val))
	 (test (hash-table-ref h (+ key eps1)) val)
	 (test (hash-table-ref h (+ key eps5)) val)
	 (test (hash-table-ref h (+ key eps9)) val)
	 (test (hash-table-ref h (- key eps1)) val)
	 (test (hash-table-ref h (- key eps5)) val)
	 (test (hash-table-ref h (- key eps9)) val)
	 ))
     keys)
    (set! h (make-hash-table 8 =))
    (for-each
     (lambda (key)
       (hash-table-set! h key (number->string key)))
     keys)
    (for-each
     (lambda (key)
       (let ((val (number->string key)))
	 (test (hash-table-ref h key) val)
         (when (real? key)
  	   (test (hash-table-ref h (inexact->exact key)) (and (= key (inexact->exact key)) val))
	   (test (hash-table-ref h (exact->inexact key)) (and (= key (exact->inexact key)) val)))
	 (test (hash-table-ref h (+ key eps1)) #f)
	 (test (hash-table-ref h (+ key eps5)) #f)
	 (test (hash-table-ref h (+ key eps9)) #f)
	 (test (hash-table-ref h (- key eps1)) #f)
	 (test (hash-table-ref h (- key eps5)) #f)
	 (test (hash-table-ref h (- key eps9)) #f)
	 ))
     keys)
    (set! h (make-hash-table 8))
    (for-each
     (lambda (key)
       (hash-table-set! h key (number->string key)))
     keys)
    (for-each
     (lambda (key)
       (let ((val (number->string key)))
	 (test (hash-table-ref h key) val)
         (when (real? key)
  	   (test (hash-table-ref h (inexact->exact key)) (and (exact? key) val))
	   (test (hash-table-ref h (exact->inexact key)) (and (inexact? key) val)))
	 (test (hash-table-ref h (+ key eps1)) #f)
	 (test (hash-table-ref h (+ key eps5)) #f)
	 (test (hash-table-ref h (+ key eps9)) #f)
	 (test (hash-table-ref h (- key eps1)) #f)
	 (test (hash-table-ref h (- key eps5)) #f)
	 (test (hash-table-ref h (- key eps9)) #f)
	 ))
     keys)
    )))

(let ((ht (make-hash-table)))
  (set! (ht (string #\a #\null #\b)) 1)
  (test (ht (string #\a #\null #\b)) 1)
  (test (ht (string #\a)) #f)
  (set! (ht (string #\a #\null #\b)) 12)
  (test (ht (string #\a #\null #\b)) 12)
  (fill! ht #f)
  (test (hash-table-entries ht) 0)
  (set! (ht #u(3 0 21)) 1)
  (test (ht #u(3 0 21)) 1))

(let ((hash (make-hash-table)))
  (hash-table-set! hash "01234567" 1)
  (hash-table-set! hash "012345678" 2)
  (hash-table-set! hash "12345678" 3)
  (hash-table-set! hash "012345670" 4)
  (hash-table-set! hash "0123456701234567" 5)
  (hash-table-set! hash "0123456" 6)
  (hash-table-set! hash "012345" 7)
  (hash-table-set! hash "01234" 8)
  (hash-table-set! hash "0123" 9)
  (hash-table-set! hash "012" 10)
  (hash-table-set! hash "01235" 11)
  (test (hash-table-ref hash "01234567") 1)
  (test (hash-table-ref hash "012345678") 2)
  (test (hash-table-ref hash "12345678") 3)
  (test (hash-table-ref hash "012345670") 4)
  (test (hash-table-ref hash "0123456701234567") 5)
  (test (hash-table-ref hash "0123456") 6)
  (test (hash-table-ref hash "012345") 7)
  (test (hash-table-ref hash "01234") 8)
  (test (hash-table-ref hash "0123") 9)
  (test (hash-table-ref hash "012") 10)
  (test (hash-table-ref hash "01235") 11))

(let ((ht (hash-table 'a #t)))
  (test (hash-table-entries ht) 1)
  (do ((i 0 (+ i 1))) ((= i 10)) (set! (ht 'a) #f) (set! (ht 'a) #t))
  (test (hash-table-entries ht) 1))

(when with-bignums
  (let ((ht (make-hash-table)))
    (set! (ht pi) 1)
    (test (ht pi) 1)
    (set! (ht (bignum "1")) 32)
    (test (ht (bignum "1")) 32)
    (set! (ht (/ (bignum "11") (bignum "3"))) 12)
    (test (ht (/ (bignum "11") (bignum "3"))) 12)
    (set! (ht (bignum "1+i")) -1)
    (test (ht (bignum "1+i")) -1)
    (set! (ht 3) 2)
    (test (ht 3) 2)
    (set! (ht 3.0) 3)
    (test (ht 3.0) 3)))

(test (hash-table?) 'error)
(test (hash-table? 1 2) 'error)

(test (make-hash-table most-positive-fixnum) 'error)
(test (make-hash-table most-negative-fixnum) 'error)
(test (make-hash-table 10 1) 'error)

(let () ; size bug noticed by K.M. -- libasan reports it
  (define hash (make-hash-table 1)) ;; Size must be 1.
  (set! (hash :hello) "a50")
  (gc))

(let ((ht (make-hash-table)))
  (test (hash-table? ht ht) 'error)
  (test (hash-table-ref ht #\a #\b) 'error)
  (test (hash-table-ref ht) 'error)
  (test (hash-table-ref) 'error)
  (test (hash-table-set!) 'error)
  (test (hash-table-set! ht) 'error)
  (test (hash-table-set! ht #\a) 'error)
  (test (hash-table-set! ht #\a #\b #\c) 'error)
  (set! (ht 'key) 32)
  (test (fill! ht 123) 123)
  (test (ht 'key) 123)
  (set! (ht 'key) 32)
  (test (ht 'key) 32)
  (set! (ht :key) 123)
  (test (ht 'key) 32)
  (test (ht :key) 123)
  (fill! ht ())
  (test (ht 'key) ()))

(let ((ht (make-hash-table)))
  (test (hash-table-set! ht #\a 'key) 'key)
  (for-each
   (lambda (arg)
     (test (hash-table-set! ht arg 3.14) 3.14))
   (list #\a #(1 2 3) 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))
  (for-each
   (lambda (arg)
     (test (hash-table-ref ht arg) 3.14))
   (list #\a #(1 2 3) 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))
  (test (length ht 123) 'error))

(for-each
 (lambda (arg)
   (test (make-hash-table arg) 'error))
 (list "hi" -1 0 #\a 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(let ((lst (list 1 2)))
  (set-cdr! (cdr lst) lst)
  (test (object->string (hash-table 'a lst)) "(hash-table 'a #1=(1 2 . #1#))")
  (test (object->string (hash-table lst lst)) "(hash-table #1=(1 2 . #1#) #1#)")
  (test (object->string (hash-table lst 1)) "(hash-table #1=(1 2 . #1#) 1)"))

(let ()
 (define ht (make-hash-table))
 (set! (ht 123) "123")
 (set! (ht 456) "456")
 (define hti (make-iterator ht))
 (test (iterator? hti) #t)
 (test (object->string hti) "#<iterator: hash-table>")
 (test (equal? hti hti) #t)
 (test (eq? hti hti) #t)
 (test (eqv? hti hti) #t)
 (test (equivalent? hti hti) #t)

 (let ((hti2 hti))
   (test (equal? hti2 hti) #t)
   (test (equivalent? hti2 hti) #t)
   (set! hti2 (copy hti))
   (test (equal? hti2 hti) #t)
   (test (equivalent? hti2 hti) #t)
   (test (let ((val (hti2))) (or (equal? val '(123 . "123")) (equal? val '(456 . "456")))) #t) ; order depends on table size
   (test (equal? hti2 hti) #f)
   (test (equivalent? hti2 hti) #f)
   )

 (let ((vals (list (hti) (hti))))
   (if (not (equal? (sort! vals (lambda (a b) (< (car a) (car b)))) '((123 . "123") (456 . "456"))))
       (format #t ";iterator: ~A~%" vals))
   (let ((val (hti)))
     (if (not (eof-object? val))
	 (format #t ";iterator at end: ~A~%" val)))
   (let ((val (hti)))
     (if (not (eof-object? val))
	 (format #t ";iterator at end (2): ~A~%" val)))))

(test (make-iterator) 'error)
(test (make-iterator (make-hash-table) 1) 'error)
(test (iterator?) 'error)
(test (iterator? 1 2) 'error)

(let ()
  (define (get-iter)
    (let ((ht (hash-table 'a 1 'b 2)))
      (test (hash-table-entries ht) 2)
      (make-iterator ht)))
  (let ((hti (get-iter)))
    (gc)
    (let ((a (hti)))
      (let ((b (hti)))
	(let ((c (hti)))
	  (test (let ((lst (list a b c)))
		  (or (equal? lst '((a . 1) (b . 2) #<eof>))
		      (equal? lst '((b . 2) (a . 1) #<eof>))))
		#t))))))

(let ((ht1 (make-hash-table))
      (ht2 (make-hash-table)))
  (test (equal? ht1 ht2) #t)
  (test (equal? ht1 (make-vector (length ht1) ())) #f)
  (hash-table-set! ht1 'key 'hiho)
  (test (equal? ht1 ht2) #f)
  (hash-table-set! ht2 'key 'hiho)
  (test (equal? ht1 ht2) #t)

  (hash-table-set! ht1 'a ())
  (test (ht1 'a) ())
  )

(let ((ht (make-hash-table 1)))
  (test (>= (length ht) 1) #t)
  (set! (ht 1) 32)
  (test (>= (length ht) 1) #t))

(let ((ht (hash-table "hi" 32 "ho" 1)))
  (test (hash-table-entries ht) 2)
  (test (ht "hi") 32)
  (test (ht "ho") 1))

(let ((ht (hash-table "hi" 32 "ho" 1)))
  (test (hash-table-entries ht) 2)
  (test (ht "hi") 32)
  (test (ht "ho") 1))

(let ((ht (hash-table)))
  (test (hash-table? ht) #t)
  (test (>= (length ht) 1) #t)
  (test (ht 1) #f))

(let ((ht (hash-table)))
  (test (hash-table? ht) #t)
  (test (>= (length ht) 1) #t)
  (test (ht 1) #f))

(test (let () (define gakk (make-hash-table 10 (cons equal? (lambda (x) #t)))) (gakk #f)) 'error)
(test (let ((h (make-hash-table 8 (cons equal? (lambda (x) pi))))) (hash-table-set! h 'a 0)) 'error)

(for-each
 (lambda (arg)
   (test (hash-table arg) 'error)
   (test ((hash-table 'a arg) 'a) arg)
   (test ((hash-table arg 'a) arg) 'a))
 (list "hi" -1 0 #\a 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #t abs #<eof> #<unspecified> (lambda () 1)))

(let ((ht (make-hash-table))
      (lst (list 1)))
  (set-cdr! lst lst)
  (set! (ht lst) lst)
  (let ((lst1 (list 1 2)))
    (set-cdr! (cdr lst1) lst1)
    (set! (ht lst1) lst1)
    (test (ht lst) lst)
    (test (ht lst1) lst1)
    (test (or (string=? (object->string ht) "(hash-table #1=(1 2 . #1#) #1# #2=(1 . #2#) #2#)")
	      (string=? (object->string ht) "(hash-table #1=(1 . #1#) #1# #2=(1 2 . #2#) #2#)"))
	  #t)))

(test (set! (hash-table) 1) 'error)
(test (set! (hash-table) 1) 'error)
(test (set! (make-hash-table) 1) 'error)

;; no null hash-tables?

(let ((ht (make-hash-table)))
  (test (map (lambda (x) x) ht) ())
  (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) ht) ctr) 0)
  (test (map (lambda (x y) (cons x y)) (list 1 2 3) ht) ())
  ;(test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) #(1 2 3) ht) ctr) 0) ; this is now an error 15-Jan-15
  (test (map (lambda (x y) (cons x y)) ht "123") ())
  (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) ht ()) ctr) 'error) ; 2 args

  (let ((rt (reverse ht)))
    (test (map (lambda (x) x) rt) ())
    (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) rt) ctr) 0))

  (set! (ht 1) 32)
  ;; these need to be independent of entry order

  (test (sort! (map (lambda (x) (cdr x)) ht) <) '(32))
  (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) ht) ctr) 1)
  (test (map (lambda (x y) (cons x y)) () ht) ())
  (test (let ((ctr 0)) (for-each (lambda (x y) (set! ctr (+ ctr 1))) ht "") ctr) 0)
  (test (sort! (map (lambda (x y) (max (cdr x) y)) ht (list 1 2 3)) <) '(32))
  (test (let ((ctr 0)) (for-each (lambda (x y) (set! ctr (max (cdr x) y))) ht #(1 2 3)) ctr) 32)

  (let ((rt (reverse ht)))
    (test (equal? (rt 32) 1) #t)
    (test (equal? (rt 1) #f) #t)
    (test (ht (rt 32)) 32)
    (test (sort! (map (lambda (x) (cdr x)) rt) <) '(1))
    (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) rt) ctr) 1)
    (for-each (lambda (x) (test (ht (rt (cdr x))) (cdr x)) (test (rt (ht (car x))) (car x))) ht)
    (set! (rt 32) 123)
    (test (rt 32) 123)
    (test (ht 32) #f)
    (test (ht 1) 32))

  (set! (ht 2) 1)
  (test (ht (ht 2)) 32)
  (test (sort! (map (lambda (x) (cdr x)) ht) <) '(1 32))
  (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) ht) ctr) 2)
  (set! (ht 3) 123)
  (test (sort! (map (lambda (x) (cdr x)) ht) <) '(1 32 123))
  (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) ht) ctr) 3)
  (test (let ((ctr 0)) (for-each (lambda (x y) (set! ctr (+ ctr 1))) ht '(1)) ctr) 1)
  (test (let ((ctr 0)) (for-each (lambda (x y z) (set! ctr (+ ctr 1))) "12" ht '(1)) ctr) 1)
  (test (let ((ctr 0)) (for-each (lambda (x y z) (set! ctr (+ ctr 1))) "12" ht '(1 2)) ctr) 2)
  (test (let ((ctr 0)) (for-each (lambda (x y z) (set! ctr (+ ctr 1))) "12345" ht '(1 2 3 4 5 6)) ctr) 3)

  (test (sort! (map (lambda (x y) (max x (cdr y))) (list -1 -2 -3 -4) ht) <) '(1 32 123))
  (test (let ((sum 0)) (for-each (lambda (x y) (set! sum (+ sum x (cdr y)))) #(10 20 30) ht) sum) 216)


  (let ((rt (reverse ht)))
    (for-each (lambda (x) (test (ht (rt (cdr x))) (cdr x)) (test (rt (ht (car x))) (car x))) ht))

  (set! (ht (list 1 2 3)) "hi")
  (test (ht '(1 2 3)) "hi")
  (test (ht 2) 1)
  (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) ht) ctr) 4)
  (set! (ht "hi") 2)
  (test (ht "hi") 2)
  (test (ht (ht (ht "hi"))) 32)

  (let ((rt (reverse ht)))
    (test (rt "hi") '(1 2 3))
    (test (rt 2) "hi")
    (for-each (lambda (x) (test (ht (rt (cdr x))) (cdr x)) (test (rt (ht (car x))) (car x))) ht)
    (set! (rt 2) "ho")
    (test (rt 2) "ho")
    (test (ht '(1 2 3)) "hi")
    (set! (rt 123) 321)
    (test (rt 123) 321)
    (test (ht 3) 123))

  (fill! ht 0)
  (set! (ht "hi") 1)
  (set! (ht "hoi") 2)
  (test (sort! (map (lambda (x) (cdr x)) ht) <) '(0 0 0 0 1 2))
  (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) ht) ctr) 6)

  (let ((rt (reverse ht)))
    (test (rt 2) "hoi")
    (set! (rt 2) "ha")
    (test (ht "hoi") 2))

  (set! (ht #\a) #\b)
  (test (ht #\a) #\b)
  (test (ht "hi") 1)

  (set! ht (hash-table))
  (set! (ht #(1)) #(2))
  (test (ht #(1)) #(2))
  (set! (ht '(1)) '(3))
  (set! (ht "1") "4")
  ;(set! (ht ht) "5")
  ;(test (ht ht) "5")
  (test (ht '(1)) '(3))
  (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) ht) ctr) 3)

  (let ((rt (reverse ht)))
    ;(test (rt "5") ht)
    (test (rt "4") "1")
    (for-each (lambda (x)
		(test (ht (rt (cdr x))) (cdr x))
		(test (rt (ht (car x))) (car x)))
	      ht))
  )

(let ((ht (make-hash-table)))
  (let ((str (string (integer->char 255)))
	(u8 #u(254 0))
	(rl 1e18)
	(int most-negative-fixnum)
	(rat (/ 1 most-negative-fixnum)))
    (set! (ht str) 1)
    (set! (ht u8) 2)
    (set! (ht rl) 3)
    (set! (ht int) 4)
    (set! (ht rat) 5)
    (test (ht str) 1)
    (test (ht u8) 2)
    (test (ht rl) 3)
    (test (ht int) 4)
    (test (ht rat) 5)))

(let ((ht1 (make-hash-table 32))
      (ht2 (make-hash-table 1024)))
  (do ((i 0 (+ i 1)))
      ((= i 256))
    (let ((str (number->string i)))
      (set! (ht1 str) i)
      (set! (ht2 i) str)))

  (let ((cases 0))
    (for-each
     (lambda (a b)
       (if (not (equal? (string->number (car a)) (cdr a)))
	   (format #t ";hash-table for-each (str . i): ~A?~%" a))
       (if (not (equal? (number->string (car b)) (cdr b)))
	   (format #t ";hash-table for-each (i . str): ~A?~%" b))
       (set! cases (+ cases 1)))
     ht1 ht2)
    (if (not (= cases 256))
	(format #t ";hash-table for-each cases: ~A~%" cases)))

  (let ((iter1 (make-iterator ht1))
	(iter2 (make-iterator ht2)))
    (test (equal? iter1 iter2) #f)
    (test (equivalent? iter1 iter2) #f)
    (test (iterator? iter2) #t)
    (let ((cases 0))
      (do ((a (iter1) (iter1))
	   (b (iter2) (iter2)))
	  ((or (eof-object? a)
	       (eof-object? b)))
	(if (not (equal? (string->number (car a)) (cdr a)))
	    (format #t ";hash-table iter1 (str . i): ~A?~%" a))
	(if (not (equal? (number->string (car b)) (cdr b)))
	    (format #t ";hash-table iter2 (i . str): ~A?~%" b))
	(set! cases (+ cases 1)))
      (if (not (= cases 256))
	  (format #t ";hash-table iter1/2 cases: ~A~%" cases)))))

(let ((ht (make-hash-table 31)))
  (let ((ht1 (make-hash-table 31)))
    (set! (ht1 'a1) 'b1)
    (set! (ht 'a0) ht1)
    (test ((ht 'a0) 'a1) 'b1)
    (test (hash-table-ref ht 'a0 'a1) 'b1)
    (test (ht 'a0 'a1) 'b1)))

(let ((ht (make-hash-table 31))
      (e (curlet)))
  (define (a-func a) (+ a 1))
  (define-macro (a-macro a) `(+ 1 , a))
  (define (any-func a) (let ((x a)) (lambda () x)))

  (set! (ht abs) 1)
  (set! (ht begin) 2)
  (set! (ht quasiquote) 3)
  (set! (ht a-func) 4)
  (set! (ht a-macro) 5)
  (set! (ht (any-func 6)) 6)
  (set! (ht e) 7)
  (test (ht e) 7)
  (set! (ht (rootlet)) 8)
  (test (ht abs) 1)
  (test (ht round) #f)
  (test (ht quasiquote) 3)
  (test (ht begin) 2)
  (test (ht lambda) #f)
  (test (ht a-func) 4)
  (test (ht a-macro) 5)
  (test (ht (any-func 6)) #f)
  (test (ht (rootlet)) 8)
  (call-with-exit
   (lambda (return)
     (set! (ht return) 9)
     (test (ht return) 9)))
  ;(set! (ht ht) 10)
  ;(test (ht ht) 10)
  )

;;; weak-hash-table?
(for-each
 (lambda (arg)
   (test (weak-hash-table? arg) #f))
 (list "hi" () -1 #\a 1 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test (weak-hash-table? (hash-table)) #f)
(test (weak-hash-table? (make-weak-hash-table)) #t)
(test (let ((h (make-weak-hash-table))) (set! (h 'a) 1) (object->string h :readable)) "(weak-hash-table 'a 1)")
(test (equal? (weak-hash-table 'a 1) (weak-hash-table 'a 1)) #t)
(test (object->string (weak-hash-table)) "(weak-hash-table)")
(test (object->string (make-iterator (weak-hash-table 'a 1))) "#<iterator: weak-hash-table>")
(test (object->string (make-iterator (weak-hash-table 'a 1)) :readable) "(make-iterator (weak-hash-table 'a 1))")

(when full-s7test
  (do ((z 0 (+ z 1)))
      ((= z 10))
    (let ((keys (make-vector 100))
	  (wht (make-weak-hash-table)))
      (do ((i 0 (+ i 1)))
	  ((= i 100))
	(set! (keys i) (list (random 100) (random 100)))
	(set! (wht (keys i)) i))
      (do ((i 0 (+ i 1)))
	  ((= i 10000))
	(let ((key (random 100)))
	  (set! (keys key) (list (random 100) (random 100)))
	  (if (> (random 100) 50)
	      (set! (wht (keys key)) i)
	      (if (> (random 100) 90)
		  (do ((k 0 (+ k 1)))
		      ((= k 100))
		    (set! (wht (keys k)) k)))))
	(when (zero? (hash-table-entries wht))
	  (do ((k 0 (+ k 1)))
	      ((= k 100))
	    (set! (wht (keys k)) k))))))
  (let ((wht (make-weak-hash-table)))
    (do ((i 0 (+ i 1)))
	((= i 1000))
      (set! (wht (list i)) i))
    (do ((i 0 (+ i 1)))
	((= i 1000))
      (for-each
       (lambda (p)
	 (unless (pair? p)
	   (format *stderr* "p: ~S~%" p)))
       wht))))

(test (let ((h1 (hash-table 'a 1 'b 2)) (h2 (make-hash-table 31))) (set! (h2 'a) 1) (set! (h2 'b) 2.0) (equivalent? h1 h2)) #t)
(test (let ((h1 (hash-table 'a 1 'b 2)) (h2 (make-hash-table 31))) (set! (h2 'a) 1.0) (set! (h2 'b) 2) (equivalent? (list h1) (list h2))) #t)

;(test (let ((ht (make-hash-table))) (hash-table-set! ht ht 1) (ht ht)) #f)
; this is #f now because the old ht is not equal to the new one (different number of entries)
;(test (let ((ht (make-hash-table))) (hash-table-set! ht ht ht) (equal? (ht ht) ht)) #t)

(test (let ((ht (make-hash-table))) (hash-table-set! ht 'a ht) (object->string ht)) "#1=(hash-table 'a #1#)")
(test (let ((h1 (make-hash-table))) (hash-table-set! h1 "hi" h1) (object->string h1)) "#1=(hash-table \"hi\" #1#)")
(test (let ((ht (make-hash-table))) (hash-table-set! ht 'a ht) (equivalent? ht (copy ht))) #t)
(test (let ((ht (make-hash-table))) (hash-table-set! ht 'a ht) (equal? ht (copy ht))) #t)

(test (hash-table 'a 1 'b) 'error)

;; there's no real need for multidim hashes:

(let ((ht (make-hash-table)))
   (set! (ht (cons 'a 1)) 'b)
   (set! (ht (cons 'a 2)) 'c)
   (set! (ht (cons 'b 1)) 'd)
   (test (ht '(a . 1)) 'b)
   (test (ht '(b . 1)) 'd)
   (set! (ht '(a . 2)) 32)
   (test (ht '(a . 2)) 32)
   (let ((lst1 (list 1))
	 (lst2 (list 1)))
     (set-car! lst1 lst2)
     (set-car! lst2 lst1)
     (set! (ht lst1) 32)
     (set! (ht lst2) 3)
     (test (equal? lst1 lst2) #t)
     (test (ht lst1) 3)
     (test (ht lst2) 3)))

(let ((ht (make-hash-table)))
  (set! (ht 1.0) 'a)
  (set! (ht 2.0) 'b)
  (set! (ht 3.0) 'c)
  (test (ht 2.0) 'b)
  (set! (ht 2.0) 'd)
  (test (ht 2.0) 'd)
  (test (ht 0.0) #f)
  (test (ht 1.0) 'a))

(let ((ht (make-hash-table)))
  (test (ht) 'error)
  (test (ht 0 1) 'error))

(let ((h (hash-table 'a (hash-table 'b 2 'c 3)
		      'b (hash-table 'b 3 'c 4))))
  (test (h 'a 'b) 2)
  (test (h 'b 'b) 3)
  (test (h 'a 'c) 3))

(let ()
  (define-macro (memoize f)
    `(define ,f (let ((ht (make-hash-table))
		      (old-f ,f))
		  (lambda args
		    (or (ht args)
			(let ((new-val (apply old-f args)))
			  (set! (ht args) new-val)
			  new-val))))))

  (define (our-abs num) (abs num))
  (memoize our-abs)
  (num-test (our-abs -1) 1)
  (with-let (funclet our-abs)
    (test (ht '(-1)) 1)))

(let ()
  (define-macro (define-memoized name&arg . body)
    (let ((arg (cadr name&arg))
	  (memo (gensym "memo")))
      `(define ,(car name&arg)
	 (let ((,memo (make-hash-table)))
	   (lambda (,arg)
	     (or (,memo ,arg)
		 (set! (,memo ,arg) (begin ,@body))))))))

  (define-memoized (f1 abc) (+ abc 2))
  (test (f1 3) 5)
  (test (f1 3) 5)
  (test (f1 2) 4)
  (let ((ht (call-with-exit
	     (lambda (return)
	       (for-each (lambda (x)
			   (if (hash-table? (cdr x))
			       (return (cdr x))))
			 (outlet (funclet f1)))
	       #f))))
    (if (not (hash-table? ht))
	(format #t ";can't find memo? ~A~%" (let->list (outlet (funclet f1))))
	(test (length (map (lambda (x) x) ht)) 2))))

(let ()
  (define-macro (define-memoized name&args . body)
    (let ((args (cdr name&args))
	  (memo (gensym "memo")))
      `(define ,(car name&args)
	 (let ((,memo (make-hash-table)))
	   (lambda ,args
	     (or (,memo (list ,@args))
		 (set! (,memo (list ,@args)) (begin ,@body))))))))

  (define (ack m n)
    (cond ((= m 0) (+ n 1))
	  ((= n 0) (ack (- m 1) 1))
	  (else (ack (- m 1) (ack m (- n 1))))))

  (define-memoized (ack1 m n)
    (cond ((= m 0) (+ n 1))
	  ((= n 0) (ack1 (- m 1) 1))
	  (else (ack1 (- m 1)
		      (ack1 m (- n 1))))))

  (test (ack 2 3) (ack1 2 3)))


(let ((ht (make-hash-table)))
  (test (eq? (car (catch #t (lambda () (set! (ht) 2)) (lambda args args))) 'wrong-number-of-args) #t)
  ;;(test (eq? (car (catch #t (lambda () (set! (ht 0 0) 2)) (lambda args args))) 'syntax-error) #t)
  (test (eq? (car (catch #t (lambda () (set! ((ht 0) 0) 2)) (lambda args args))) 'no-setter) #t))

(let ()
  (define merge-hash-tables append)
  (let ((ht (merge-hash-tables (hash-table 'a 1 'b 2) (hash-table 'c 3))))
    (test (ht 'c) 3))
  (test ((append (hash-table 'a 1 'b 2) (hash-table 'c 3)) 'c) 3))

;;; test the eq-func business
(test (make-hash-table 8 '(atan abs)) 'error)
(test (make-hash-table 8 (cons atan abs)) 'error)
(test (make-hash-table 8 (cons eq? sqrt)) 'error)

(let ((ht (make-hash-table 8 eq?)))
  (test (hash-table-ref ht 'a) #f)
  (hash-table-set! ht 'a 1)
  (hash-table-set! ht 'c 'd)
  (test (hash-table-ref ht 'a) 1)
  (hash-table-set! ht "hi" 3)
  (test (hash-table-ref ht "hi") #f)
  (set! (ht '(a . 1)) "ho")
  (test (ht '(a . 1)) #f)
  (let ((ht1 (copy ht)))
    (test (ht1 'a) 1)
    (test (ht1 "hi") #f)
    (set! (ht1 #\a) #\b)
    (test (ht1 #\a) #\b)
    (test (ht #\a) #f)
    (let ((ht2 (reverse ht1)))
      (test (ht1 #\a) #\b)
      (test (ht2 #\b) #\a)
      (test (ht2 'd) 'c)))
  (do ((i 0 (+ i 1)))
      ((= i 32))
    (set! (ht (symbol "g" (number->string i))) i))
  (test (ht 'a) 1)
  (test (ht 'g3) 3)
  (set! (ht ht) 123)
  (test (ht ht) 123))

(let ((ht (make-hash-table 31 string=?)))
  (test (length ht) 32)
  (set! (ht "hi") 'a)
  (test (ht "hi") 'a)
  (test (ht "Hi") #f)
  (test (set! (ht 32) 'b) (if (> (*s7* 'safety) 0) 'error 'b))
  (test (ht 32) #f)
  )

(let ((ht (make-hash-table 8 string=?)))
  (set! (ht "a string longer than 8 chars") 32)
  (test (ht "a string longer than 8 chars") 32)
  (set! (ht "") 3)
  (test (ht "") 3)
  (set! (ht "") #f)
  (test (ht "") #f))

(let ((ht (make-hash-table 31 char=?)))
  (test (length ht) 32)
  (set! (ht #\a) 'a)
  (test (ht #\a) 'a)
  (test (ht #\A) #f)
  (test (set! (ht 32) 'b) (if (> (*s7* 'safety) 0) 'error 'b))
  (test (ht 32) #f)
  )

(unless pure-s7
  (let ((ht (make-hash-table 31 string-ci=?)))
    (test (length ht) 32)
    (set! (ht "hi") 'a)
    (test (ht "hi") 'a)
    (test (ht "Hi") 'a)
    (test (ht "HI") 'a)
    (test (set! (ht 32) 'b) (if (> (*s7* 'safety) 0) 'error 'b))
    (test (ht 32) #f)
    )

  (let ((ht (make-hash-table 31 char-ci=?)))
    (test (length ht) 32)
    (set! (ht #\a) 'a)
    (test (ht #\a) 'a)
    (test (ht #\A) 'a)
    (test (set! (ht 32) 'b) (if (> (*s7* 'safety) 0) 'error 'b))
    (test (ht 32) #f)
    ))

(let ((ht (make-hash-table 31 =)))
  (test (length ht) 32)
  (set! (ht 1) 'a)
  (test (ht 1.0) 'a)
  (test (ht 1+i) #f)
  (set! (ht 32) 'b)
  (test (ht 32) 'b)
  (set! (ht 1/2) 'c)
  (test (ht 0.5) 'c)
  )

(let ((ht (make-hash-table 31 eqv?)))
  (test (length ht) 32)
  (set! (ht 1) 'a)
  (test (ht 1.0) #f)
  (set! (ht 2.0) 'b)
  (test (ht 2.0) 'b)
  (set! (ht 32) 'b)
  (test (ht 32) 'b)
  (set! (ht #\a) 1)
  (test (ht #\a) 1)
  (set! (ht ()) 2)
  (test (ht ()) 2)
  (set! (ht abs) 3)
  (test (ht abs) 3)
  )

(let ((ht (make-hash-table 8 (cons string=? (lambda (a) (string-length a))))))
  (set! (ht "a") 'a)
  (test (ht "a") 'a)
  (set! (ht "abc") 'abc)
  (test (ht "abc") 'abc))

(let ((ht (make-hash-table 8 (cons eq? (lambda (a) (hash-table-ref a a))))))
  (test (set! (ht ht) 1) 'error)) ; ;hash-table mapper called recursively

(let ((ht (make-hash-table 8 (cons (lambda (a b) (string=? a b)) string-length))))
  (set! (ht "a") 'a)
  (test (ht "a") 'a)
  (set! (ht "abc") 'abc)
  (test (ht "abc") 'abc))

(let ((ht (make-hash-table 8 (cons (lambda (a b) (string=? a b)) (lambda (a) (string-length a))))))
  (set! (ht "a") 'a)
  (test (ht "a") 'a)
  (set! (ht "abc") 'abc)
  (test (ht "abc") 'abc))

(let ((ht (make-hash-table 8 (cons string=? string-length))))
  (set! (ht "a") 'a)
  (test (ht "a") 'a)
  (set! (ht "abc") 'abc)
  (test (ht "abc") 'abc))

(let-temporarily (((*s7* 'equivalent-float-epsilon) 1e-15))
  (let ((h (make-hash-table 8 equal?)))
    (set! (h (make-int-vector 3 0)) 3)
    (test (h (make-int-vector 3 0)) 3)
    (test (h (make-vector 3 0)) 3) ; vector equality changed 19-Sep-18
    (test (h (make-float-vector 3 0)) #f)
    (let ((x 1.0)
	  (y (+ 1.0 (* 0.5 (*s7* 'equivalent-float-epsilon))))
	  (z (+ 1.0 (* 1000 (*s7* 'equivalent-float-epsilon))))) ; !
      (set! (h x) 12)
      (test (h x) 12)
      (test (h y) #f)
      (test (h z) #f))))

(let ((h (make-hash-table 8 equivalent?)))
  (set! (h (make-int-vector 3 0)) 3)
  (test (h (make-int-vector 3 0)) 3)
  (test (h (make-vector 3 0)) 3)
  (test (h (make-float-vector 3 0)) 3)
  (let ((x 1.0)
	(y (+ 1.0 (* 0.5 (*s7* 'hash-table-float-epsilon))))
	(z (+ 1.0 (* 2 (*s7* 'hash-table-float-epsilon)))))
    (set! (h x) 12)
    (test (h x) 12)
    (test (h y) 12)
    (test (h z) #f)
    (set! (h 1/10) 3)
    (test (h 0.1) 3)
    (set! (h #(1 2.0)) 4)
    (test (h (vector 1 2)) 4)
    (set! (h 1.0) 5)
    (test (h 1) 5)
    (set! (h (list 3)) 6)
    (test (h (list 3.0)) 6)
    ))

(when with-block
  (let ((ht (make-hash-table 31 (cons hash_heq hash_hloc))))
    (test (length ht) 32)
    (set! (ht 'a) 'b)
    (test (ht 'a) 'b)
    (test (ht 1) #f)
    (let ((ht1 (reverse ht)))
      (test (ht1 'b) 'a)))
  (let () ; Radium op_x_aa needs_copied_args bugs (collides with c_object equal test)
    (define (f h b ind)
      (h b ind))
    (define (ftest)
      (let ((h1 (make-hash-table 8 equal?)))
        (let ((b1 (block 0 1 2))
	      (b2 (block 3 4 5)))
	  (hash-table-set! h1 b1 (hash-table :a 1))
 	  (hash-table-set! h1 b2 (inlet :a 2))
 	  (test (f h1 b1 :a) 1) ; #f if bug
	  (test (f h1 b2 :a) 2))))
    (ftest)))

(let ((ht (make-hash-table 31 equivalent?))
      (ht1 (make-hash-table 31)))
  (test (length ht) 32)
  (test (equal? ht ht1) #t)
  (set! (ht 3) 1)
  (test (ht 3) 1)
  (set! (ht1 3) 1)
  (test (equal? ht ht1) #f)
  (set! (ht 3.14) 'a)
  (test (ht 3.14) 'a)
  (set! (ht "hi") 123)
  (test (ht "hi") 123)
  (set! (ht 1/0) #<eof>)
  (test (ht 1/0) #<eof>))

(let ((ht (make-hash-table 31 (cons (lambda (a b) (eq? a b)) (lambda (a) 0)))))
  (test (hash-table-ref ht 'a) #f)
  (hash-table-set! ht 'a 1)
  (hash-table-set! ht 'c 'd)
  (test (hash-table-ref ht 'a) 1)
  (hash-table-set! ht "hi" 3)
  (test (hash-table-ref ht "hi") #f)
  (set! (ht '(a . 1)) "ho")
  (test (ht '(a . 1)) #f)
  (let ((ht1 (copy ht)))
    (test (ht1 'a) 1)
    (test (ht1 "hi") #f)
    (test (equal? ht ht1) #t)
    (test (equal? ht1 ht) #t)
    (test (equivalent? ht ht1) #t)
    (test (equivalent? ht1 ht) #t)
    (set! (ht1 #\a) #\b)
    (test (ht1 #\a) #\b)
    (test (ht #\a) #f)))

(when (provided? 'snd)
  (let ((ht (make-hash-table 31 (cons equal? mus-type))))
    (let ((g1 (make-oscil 100))
	  (g2 (make-oscil 100)))
      (set! (ht g1) 32)
      (test (ht g1) 32)
      (test (ht g2) 32)
      (test (equal? g1 g2) #t))))

;;; hash-table typers

(test ((object->let (make-hash-table 8 #f (cons symbol? integer?))) 'signature) '(integer? hash-table? symbol?))
(let ((x (copy (make-hash-table 8 #f (cons symbol? integer?))))) (test ((object->let x) 'signature) '(integer? hash-table? symbol?)))
(test (let ((h (make-hash-table 3 #f (list symbol? integer?)))) (set! (h 'a) 1) (fill! h #\a) h) 'error)
(test (let ((h (make-hash-table 3 #f (cons symbol? integer?)))) (set! (h 'a) 1) (fill! h #\a) h) 'error)
(test (let ((h (make-hash-table 3 #f (cons symbol? integer?)))) (set! (h 'a) 1) (fill! h #f) h) (hash-table))
(test (hash-table? (make-hash-table 8 eqv? (cons #t integer?))) #t)
(test (hash-table? (make-hash-table 8 = (cons #t integer?))) #t)
(test (make-hash-table 8 = (cons char? symbol?)) 'error)
(test (make-hash-table 8 string=? (cons char? symbol?)) 'error)
(test (hash-table? (make-hash-table 8 equal? (cons char? symbol?))) #t)

(let ((ht (make-hash-table 8 #f (cons symbol? integer?))))
  (test (hash-table-set! ht 'a 1) 1)
  (test (hash-table-ref ht 'a) 1)
  (test (fill! ht 32) 32)
  (test (hash-table-ref ht 'a) 32)
  (test (fill! ht 1/2) 'error)
  (test (hash-table-ref ht 'a) 32)
  (test (fill! ht #f) #f)
  (test (hash-table-ref ht 'a) #f)
  (test (hash-table-set! ht 'a 1) 1)
  (let-temporarily (((ht 'a) 2))
    (test (ht 'a) 2))
  (test (ht 'a) 1)
  (test (let-temporarily (((ht 'a) 1/2)) 21) 'error)
  (test (ht 'a) 1)
  (test (hash-table-set! ht 'a 1/2) 'error)
  (test (hash-table-set! ht 123 1) 'error)
  (test (ht 'a) 1)
  (let ((ht1 (hash-table 'a 2)))
    (test (copy ht1 ht) ht)
    (test (ht 'a) 2))
  (let ((ht1 (hash-table 'a 1/2)))
    (test (copy ht1 ht) 'error)
    (test (ht 'a) 2))
  (let ((ht1 (hash-table 123 1)))
    (test (copy ht1 ht) 'error)
    (test (ht 'a) 2))
  (test (set! (ht 'a) 1/2) 'error)
  (test (set! (ht "a") 1) 'error)
  (test (ht 'a) 2)
  (test (set! (ht 'a) 3) 3)
  (test (ht 'a) 3)
  (let ()
    (define (st)
      (hash-table-set! ht 'a 4))
    (st) (st)
    (test (ht 'a) 4)
    (define (st1)
      (hash-table-set! ht 'a 1/2))
    (test (st1) 'error)
    (test (st1) 'error)
    (test (ht 'a) 4)))

(let ()
  (define (boolean-or-integer? x)
    (or (boolean? x)
	(integer? x)))
  (let ((ht (make-hash-table 8 #f (cons symbol? boolean-or-integer?))))
    (test (signature ht) '(boolean-or-integer? hash-table? symbol?))
    (test (hash-table-set! ht 'a 21) 21)
    (test (hash-table-set! ht 'b #t) #t)
    (test (hash-table-set! ht 'c pi) 'error)
    (test (fill! ht 1) 1)
    (test (hash-table-ref ht 'a) 1)
    (test (fill! ht #\a) 'error)
    (let ((ht1 (copy ht)))
      (test (hash-table-set! ht1 'a #\a) 'error)
      (test (signature ht1) '(boolean-or-integer? hash-table? symbol?))))

  (define (symbol-or-pi? x)
    (or (symbol? x)
	(eq? x pi)))
  (let ((ht (make-hash-table 8 #f (cons symbol-or-pi? boolean-or-integer?))))
    (test (signature ht) '(boolean-or-integer? hash-table? symbol-or-pi?))
    (test (hash-table-set! ht 'a 21) 21)
    (test (hash-table-set! ht 'b #t) #t)
    (test (hash-table-set! ht 'c pi) 'error)
    (test (hash-table-set! ht pi 0) 0)
    (test (fill! ht 1) 1)
    (test (hash-table-ref ht 'a) 1)
    (test (fill! ht #\a) 'error)
    (let ((ht1 (copy ht)))
      (test (hash-table-set! ht1 'a #\a) 'error)
      (test (signature ht1) '(boolean-or-integer? hash-table? symbol-or-pi?)))))

#|
(define constants (vector 1)) ; etc -- see tauto.scm
(define ops (list eq? eqv? equal? equivalent? = char=? string=? char-ci=? string-ci=?
		  (cons string=? (lambda (a) (string-length a)))
		  (cons (lambda (a b) (string=? a b)) string-length)))
(for-each
 (lambda (op)
   (for-each
    (lambda (val)
      (let ((h (make-hash-table 8 op)))
	(catch #t
	  (lambda ()
	    (set! (h val) #t)
	    (if (not (eq? (h val) (op val val)))
		(format *stderr* "~A ~A: ~A ~A~%" op val (h val) (op val val))))
	  (lambda any #f))))
    constants))
 ops)
|#

(let ((H1 (make-hash-table)))
  (set! (H1 H1) H1)
  (let ((H2 (hash-table H1 H1)))
    (test (string-wi=? (object->string (append H2 (make-hash-table 8 #f (cons symbol? integer?))) :readable)
                       "(let ((<1> (hash-table))) (set! (<1> <1>) <1>) (hash-table <1> <1>))")
          #t)
    (test (object->string (append H2 (hash-table))) "(hash-table #1=(hash-table #1# #1#) #1#)")))

(let ()
  (let ((ht (make-hash-table 31 (cons (lambda (a b) (eq? a b)) (lambda (a) 0)))))
    (hash-table-set! ht 'a 1)
    (test (ht 'a) 1))
  (let ((ht (make-hash-table 31 (cons (lambda* (a b) (eq? a b)) (lambda (a) 0)))))
    (hash-table-set! ht 'a 1)
    (test (ht 'a) 1))
  (let ((ht (make-hash-table 31 (cons (lambda* (a (b 0)) (eq? a b)) (lambda (a) 0)))))
    (hash-table-set! ht 'a 1)
    (test (ht 'a) 1))
  (test (let ((ht (make-hash-table 31 (list (eq? a b)))))
	  (hash-table-set! ht 'a 1)
	  (ht 'a))
	'error)
  (test (let ((ht (make-hash-table 31 (cons abs +))))
	  (hash-table-set! ht 'a 1)
	  (ht 'a))
	'error)
  (test (let ((ht (make-hash-table 31 (cons eq? float-vector-ref))))
	  (hash-table-set! ht 'a 1)
	  (ht 'a))
	'error)
  (test (let ((ht (make-hash-table 31 (dilambda (lambda (a) (eq? a b)) (lambda (a) 0)))))
	  (hash-table-set! ht 'a 1)
	  (ht 'a))
	'error)
  (test (let ((ht (make-hash-table 31 (lambda a (eq? car a) (cadr s)))))
	  (hash-table-set! ht 'a 1)
	  (ht 'a))
	'error)
  (test (let ((ht (make-hash-table 31 (cons (lambda (a b c) (eq? a b)) (lambda (a) 0)))))
	  (hash-table-set! ht 'a 1)
	  (ht 'a))
	'error)
  (test (let ((ht (make-hash-table 31 (define-macro (_m_ . args) #f))))
	  (hash-table-set! ht 'a 1)
	  (ht 'a))
	'error)
  (test (let ((ht (make-hash-table 31 abs)))
	  (hash-table-set! ht 'a 1)
	  (ht 'a))
	'error)
  (test (let ((ht (make-hash-table 31 list-set!)))
	  (hash-table-set! ht 'a 1)
	  (ht 'a))
	'error))

(let ()
  (define (test-hash size)
    (let ((c #t))
      (let ((int-hash (make-hash-table (max size 511) (cons (lambda (a b) (= a b)) (lambda (a) a)))))
	(do ((i 0 (+ i 1)))
	    ((= i size))
	  (hash-table-set! int-hash i i))
	(do ((i 0 (+ i 1)))
	    ((= i size))
	  (let ((x (hash-table-ref int-hash i)))
	    (if (not (= x i)) (format *stderr* ";test-hash(0) ~D -> ~D~%" i x)))))
      (let ((int-hash (make-hash-table (max size 511) (cons (lambda (a b) (and c (= a b))) (lambda (a) a)))))
	(do ((i 0 (+ i 1)))
	    ((= i size))
	  (hash-table-set! int-hash i i))
	(do ((i 0 (+ i 1)))
	    ((= i size))
	  (let ((x (hash-table-ref int-hash i)))
	    (if (not (= x i)) (format *stderr* ";test-hash(1) ~D -> ~D~%" i x)))))
      (let ((int-hash (make-hash-table (max size 511) (let ((c #f)) (cons (lambda (a b) (and (not c) (= a b))) (lambda (a) a))))))
	(do ((i 0 (+ i 1)))
	    ((= i size))
	  (hash-table-set! int-hash i i))
	(do ((i 0 (+ i 1)))
	    ((= i size))
	  (let ((x (hash-table-ref int-hash i)))
	    (if (not (= x i)) (format *stderr* ";test-hash(2) ~D -> ~D~%" i x)))))
      ))

  (test-hash 10))

(let () ; check hash-table-increment internals
  (define (g word)
    (let ((ht (hash-table)))
      (hash-table-set! ht word (+ (or (hash-table-ref ht word) 0) 1))
      (let ((x (hash-table-ref ht word)))
        (do ((i 3 (+ i 1)))
  	    ((= i 6))
  	  (set! x i))
        (hash-table-ref ht word))))
  (test (g 'i) 1)
  (define (g1 word)
    (let ((ht (hash-table)))
      (hash-table-set! ht word (+ (or (hash-table-ref ht word) 0) 1))
      (let ((x (hash-table-ref ht word)))
        (hash-table-set! ht word (+ (or (hash-table-ref ht word) 0) 1))
        x)))
  (test (g1 'i) 1))

#|
;; another problem
(let ((ht (make-hash-table))
      (lst (list 1)))
  (set! (ht lst) 32)
  (let ((v1 (ht '(1)))
	(v2 (ht '(2))))
    (set-car! lst 2)        ; can't copy key unless equal?
    (let ((v3 (ht '(1)))
	  (v4 (ht '(2))))
      (list v1 v2 v3 v4)))) ; 32 #f #f #f

(let ((ht (make-hash-table))
      (lst (list 1)))
  (set! (ht (copy lst)) 32)
  (let ((v1 (ht '(1)))
	(v2 (ht '(2))))
    (set-car! lst 2)
    (let ((v3 (ht '(1)))
	  (v4 (ht '(2))))
      (list v1 v2 v3 v4)))) ; 32 #f 32 #f
|#

(test (hash-table-set! (copy (make-hash-table 8 equivalent? (cons symbol? #t))) #\A (inlet 'a (inlet 'b 1))) 'error)

(let ((source (hash-table #() "asdf"))
      (dest (make-hash-table 8 #f (cons symbol? integer?))))
  (test (copy source dest) 'error))

(let ((H_3 (make-hash-table 8 (cons equal? hash-code)))
      (H_4 (make-hash-table 8 (let ((eqf (lambda (a b) (equal? a b)))
				    (mapf (lambda (a) (hash-code a))))
				(cons eqf mapf)))))
  (test (hash-table-set! H_3 #<unspecified> ()) ())
  (test (hash-table? (copy H_3 H_4)) #t)
  (test (hash-table-set! H_4 'a 1) 1))

;;; hash-table-key|value-typer

(let ((h (hash-table 'a 1)))
  (for-each
   (lambda (arg)
     (test (hash-table-key-typer arg) 'error)
     (test (hash-table-value-typer arg) 'error)
     (test (set! (hash-table-key-typer arg) integer?) 'error)
     (test (set! (hash-table-key-typer h) arg) 'error)
     (test (set! (hash-table-value-typer h) arg) 'error))
   (list "hi" #\a 1 () '(1 . 2) (cons #\a #\b) 'a-symbol #(1 2) _undef_ _null_ quasiquote macroexpand 1/0 (log 0)
         3.14 3/4 1.0+1.0i :hi (if #f #f))))

(test (hash-table-key-typer (hash-table)) #f)
(test (hash-table-key-typer (make-hash-table 8 eq? (cons symbol? integer?))) symbol?)
(test (hash-table-value-typer (make-hash-table 8 eq? (cons symbol? integer?))) integer?)
(test (hash-table-value-typer (hash-table)) #f)
(let ()
  (define H (make-hash-table 8 eq? (cons symbol? integer?)))
  (test (set! (hash-table-key-typer H) integer?) integer?)
  (test (hash-table-key-typer H) integer?)
  (test (set! (hash-table-key-typer H) #f) #f)
  (test (hash-table-key-typer H) #t)
  (test (set! (hash-table-key-typer H) symbol?) symbol?)
  (test (hash-table-key-typer H) symbol?)
  (test (set! (hash-table-value-typer H) integer?) integer?)
  (test (hash-table-value-typer H) integer?)
  (test (hash-table-key-typer H) symbol?))

(let ()
  (define (typer x) (symbol? x))
  (define h (make-hash-table 8 eq? (cons typer typer)))
  (test (equal? typer (hash-table-key-typer h)) #t)
  (test (catch #t (lambda () (set! (h 'a) 32)) (lambda (t i) (apply format #f i)))
        "hash-table-set! third argument 32, is an integer, but the hash-table's value type checker, typer, rejects it")
  (test (set! (hash-table-value-typer h) #f) #f)
  (test (catch #t (lambda () (set! (h 'a) 32)) (lambda (t i) (apply format #f i))) 32))

(test (object->string (hash-table) :readable) "(hash-table)")
(test (object->string (make-hash-table 31 eqv?) :readable) "(make-hash-table 32 eqv?)")

(test (let ((H (make-hash-table 8 eq?))) (object->string H :readable)) "(make-hash-table 8 eq?)")
(test (let ((H (make-hash-table 8 eq? (cons symbol? integer?)))) (object->string H :readable)) "(make-hash-table 8 eq? (cons symbol? integer?))")
(test (let ((H (make-hash-table 8 eq?))) (set! (H 'a) 1) (object->string H :readable))
      "(let ((<h> (make-hash-table 8 eq?))) (copy (hash-table 'a 1) <h>))")
(test (let ((H (make-hash-table 8 eq? (cons symbol? integer?)))) (set! (H 'a) 1) (object->string H :readable))
      "(let ((<h> (make-hash-table 8 eq? (cons symbol? integer?)))) (copy (hash-table 'a 1) <h>))")
(test (let ((H1 (immutable! (hash-table 'a 1)))) (object->string H1 :readable))  "(immutable! (hash-table 'a 1))")
(test (let () (define (10-12? val) (memv val '(10 12))) (object->string (make-hash-table 16 #f (cons #t 10-12?)) :readable))
      "(make-hash-table 16 #f (cons #t 10-12?))")
(let ((H (make-hash-table 8 eq? (cons symbol? integer?))))
  (test (object->string H :readable) "(make-hash-table 8 eq? (cons symbol? integer?))"))
(let ((H (make-hash-table 8 eqv? (cons symbol? integer?))))
  (test (object->string H :readable) "(make-hash-table 8 eqv? (cons symbol? integer?))"))
(let ((H (make-hash-table 8 (cons (lambda (a b) (eq? a b)) (lambda (a) 0)))))
  (test (object->string H :readable) "(make-hash-table 8 (cons (lambda (a b) (eq? a b)) (lambda (a) 0)))"))
(let ((H (make-hash-table 8 (cons (lambda (a b) (eq? a b)) (lambda (a) 0)) (cons symbol? integer?))))
  (test (object->string H :readable) "(make-hash-table 8 (cons (lambda (a b) (eq? a b)) (lambda (a) 0)) (cons symbol? integer?))"))
(let ((H (make-hash-table 8 (cons (lambda (a b) (eqv? a b)) (lambda (a) 0)) (cons symbol? integer?))))
  (test (object->string H :readable) "(make-hash-table 8 (cons (lambda (a b) (eqv? a b)) (lambda (a) 0)) (cons symbol? integer?))"))
(let ((H (make-hash-table 8 eqv? (cons integer? integer?))))
  (test (object->string H :readable) "(make-hash-table 8 eqv? (cons integer? integer?))"))
(let ((H (make-hash-table 8 eqv?))) (test (object->string H :readable) "(make-hash-table 8 eqv?)"))
(let ()
  (define (10-12? val) (memv val '(10 12)))
  (define hash (make-hash-table 8 #f (cons #t 10-12?)))
  (test (object->string hash :readable) "(make-hash-table 8 #f (cons #t 10-12?))"))
(let ()
  (define (10-12? val) (memv val '(10 12)))
  (define hash (make-hash-table 8 = (cons #t 10-12?)))
  (test (object->string hash :readable) "(make-hash-table 8 = (cons #t 10-12?))"))
(test (object->string (make-hash-table 8 (cons string=? string-length)) :readable) "(make-hash-table 8 (cons string=? string-length))")

(let ((H2 (make-hash-table 8 #f (cons string? integer?))))
  (test (object->string H2 :readable) "(make-hash-table 8 #f (cons string? integer?))"))
(let ((H2 (make-hash-table 8 equivalent? (cons string? integer?))))
  (test (object->string H2 :readable) "(make-hash-table 8 equivalent? (cons string? integer?))"))
(let ((H2 (make-hash-table 8 (cons string=? string-length) (cons string? integer?))))
  (test (object->string H2 :readable) "(make-hash-table 8 (cons string=? string-length) (cons string? integer?))"))
(let ((H2 (make-hash-table 8 (cons string=? string-length))))
  (test (object->string H2 :readable) "(make-hash-table 8 (cons string=? string-length))"))
(let ((H2 (make-hash-table 8 #f)))
  (test (object->string H2 :readable) "(hash-table)"))

(let ((H2 (immutable! (make-hash-table 8 #f (cons string? integer?)))))
  (test (object->string H2 :readable) "(immutable! (make-hash-table 8 #f (cons string? integer?)))"))
(let ((H2 (immutable! (make-hash-table 8 equivalent? (cons string? integer?)))))
  (test (object->string H2 :readable) "(immutable! (make-hash-table 8 equivalent? (cons string? integer?)))"))
(let ((H2 (immutable! (make-hash-table 8 (cons string=? string-length) (cons string? integer?)))))
  (test (object->string H2 :readable) "(immutable! (make-hash-table 8 (cons string=? string-length) (cons string? integer?)))"))
(let ((H2 (immutable! (make-hash-table 8 (cons string=? string-length)))))
  (test (object->string H2 :readable) "(immutable! (make-hash-table 8 (cons string=? string-length)))"))
(let ((H2 (immutable! (make-hash-table 8 #f))))
  (test (object->string H2 :readable) "(immutable! (hash-table))"))

(let ((H2 (make-hash-table 8 (cons (lambda (a b) (eq? a b)) (lambda (a) 0)))))
  (test (object->string H2 :readable) "(make-hash-table 8 (cons (lambda (a b) (eq? a b)) (lambda (a) 0)))"))
(let ((H2 (make-hash-table 8 (cons eq? (lambda (a) 0)))))
  (test (object->string H2 :readable) "(make-hash-table 8 (cons eq? (lambda (a) 0)))"))
(let ((H2 (make-hash-table 8 (cons (lambda (a b) (eq? a b)) hash-code))))
  (test (object->string H2 :readable) "(make-hash-table 8 (cons (lambda (a b) (eq? a b)) hash-code))"))

(test (object->string (immutable! (hash-table))) "(hash-table)")
(test (object->string (immutable! (hash-table)) :readable) "(immutable! (hash-table))")
(test (object->string (immutable! (hash-table 'a 1)) :readable) "(immutable! (hash-table 'a 1))")

(test (string-wi=? (object->string (immutable! (let ((H (hash-table))) (set! (H 'a) H) H)) :readable)
		   "(let ((<1> (hash-table))) (set! (<1> 'a) <1>) (immutable! <1>))")
      #t)

(test (object->string (make-hash-table 8 #f #f) :readable) "(hash-table)")
(test (immutable? (copy (immutable! (hash-table 'a 1)))) #f)

(let ((H (make-hash-table 32 (cons string=? string-length))))
  (test (object->string H :readable) "(make-hash-table 32 (cons string=? string-length))")

  (let ((H1 (copy H)))
    (test (object->string H1 :readable) "(make-hash-table 32 (cons string=? string-length))"))

  (let ((H2 (make-hash-table 32)))
    (copy H H2)
    (test (object->string H2 :readable) "(hash-table)"))

  (set! (H "a") 32)
  (test (object->string H :readable) "(let ((<h> (make-hash-table 32 (cons string=? string-length)))) (copy (hash-table \"a\" 32) <h>))")

  (let ((H1 (copy H)))
    (test (object->string H1 :readable) "(let ((<h> (make-hash-table 32 (cons string=? string-length)))) (copy (hash-table \"a\" 32) <h>))")
    (test (set! (H1 'a) "asdf") 'error))

  (let ((H2 (make-hash-table 32)))
    (copy H H2)
    (test (object->string H2 :readable) "(hash-table \"a\" 32)"))

  (set! H (make-hash-table 8 #f (cons symbol? integer?)))
  (test (object->string H :readable) "(make-hash-table 8 #f (cons symbol? integer?))")

  (let ((H1 (copy H)))
    (test (object->string H1 :readable) "(make-hash-table 8 #f (cons symbol? integer?))"))

  (let ((H2 (make-hash-table 32)))
    (copy H H2)
    (test (object->string H2 :readable) "(hash-table)"))

  (set! (H 'a) 32)

  (let ((H1 (copy H)))
    (test (object->string H1 :readable) "(let ((<h> (make-hash-table 8 #f (cons symbol? integer?)))) (copy (hash-table 'a 32) <h>))"))

  (let ((H2 (make-hash-table 32)))
    (copy H H2)
    (test (object->string H2 :readable) "(hash-table 'a 32)"))

  (set! H (make-hash-table 8 (cons eq? hash-code) (cons symbol? integer?)))
  (test (object->string H :readable) "(make-hash-table 8 (cons eq? hash-code) (cons symbol? integer?))")

  (let ((H1 (copy H)))
    (test (object->string H1 :readable) "(make-hash-table 8 (cons eq? hash-code) (cons symbol? integer?))"))

  (let ((H2 (make-hash-table 32)))
    (copy H H2)
    (test (object->string H2 :readable) "(hash-table)"))

  (set! (H 'a) 32)

  (let ((H1 (copy H)))
    (test (hash-table-key-typer H1) symbol?)
    (test (object->string H1 :readable) "(let ((<h> (make-hash-table 8 (cons eq? hash-code) (cons symbol? integer?)))) (copy (hash-table 'a 32) <h>))"))

  (let ((H2 (make-hash-table 32)))
    (copy H H2)
    (test (object->string H2 :readable) "(hash-table 'a 32)"))

  (let ((H1 (make-hash-table)))
    (set! (hash-table-value-typer H1) integer?)
    (test (object->string H1 :readable) "(make-hash-table 8 #f (cons #t integer?))"))

  (let ((H1 (make-hash-table)))
    (set! (H1 'a) 1)
    (set! (hash-table-value-typer H1) integer?)
    (set! (H1 'b) 2)
    (let ((str (object->string H1 :readable)))
      (test (or (string-wi=? str "(let ((<h> (make-hash-table 8 #f (cons #t integer?)))) (copy (hash-table 'a 1 'b 2) <h>))")
                (string-wi=? str "(let ((<h> (make-hash-table 8 #f (cons #t integer?)))) (copy (hash-table 'b 2 'a 1) <h>))"))
            #t)))

  (set! H (make-hash-table 8))
  (let ((H1 (make-hash-table 32 (cons string=? string-length))))
    (copy H H1)
    (test (object->string H1 :readable) "(make-hash-table 32 (cons string=? string-length))"))

  (let ((H2 (make-hash-table 8 #f (cons string? integer?))))
    (test (hash-table-value-typer H2) integer?)
    (copy H H2)
    (test (object->string H2 :readable) "(make-hash-table 8 #f (cons string? integer?))"))

  (set! (H "a") 32)
  (let ((H1 (make-hash-table 32 (cons string=? string-length))))
    (copy H H1)
    (test (object->string H1 :readable) "(let ((<h> (make-hash-table 32 (cons string=? string-length)))) (copy (hash-table \"a\" 32) <h>))"))

  (let ((H2 (make-hash-table 8 #f (cons string? integer?))))
    (copy H H2)
    (test (object->string H2 :readable) "(let ((<h> (make-hash-table 8 string=? (cons string? integer?)))) (copy (hash-table \"a\" 32) <h>))"))

  (test (string-wi=? (let ((H (hash-table))) (set! (H 'a) H) (object->string H :readable)) "(let ((<1> (hash-table))) (set! (<1> 'a) <1>) <1>)") #t)

  (test (let ((H (make-hash-table 8 eq? (cons symbol? integer?)))) (set! (H 'a) 1) (object->string H :readable))
	"(let ((<h> (make-hash-table 8 eq? (cons symbol? integer?)))) (copy (hash-table 'a 1) <h>))")

  (test (let ((H (make-hash-table 8 eq?))) (set! (H 'a) 1) (object->string H :readable))
	"(let ((<h> (make-hash-table 8 eq?))) (copy (hash-table 'a 1) <h>))")

  (set! H (make-hash-table 8 eq?))
  (set! (H 'a) H)
  (test (string-wi=? (object->string H :readable) "(let ((<1> (let ((<h> (make-hash-table 8 eq?))) <h>))) (set! (<1> 'a) <1>) <1>)") #t)
  ;; TODO: remove the let? also below

  (set! H (make-hash-table 8 (cons eq? hash-code) (cons symbol? hash-table?)))
  (set! (H 'a) H)
  (test (string-wi=? (object->string H :readable)
		     "(let ((<1> (let ((<h> (make-hash-table 8 (cons eq? hash-code) (cons symbol? hash-table?))))
                                   <h>)))
                        (set! (<1> 'a) <1>)
                        <1>)") #t)

  (let ((H1 (copy H)))
    (test (or (procedure? (hash-table-key-typer H1)) (procedure? (hash-table-value-typer H1))) #t)
    (test (string-wi=? (object->string H :readable)
		     "(let ((<1> (let ((<h> (make-hash-table 8 (cons eq? hash-code) (cons symbol? hash-table?))))
                                   <h>)))
                        (set! (<1> 'a) <1>)
                        <1>)") #t))

  (let ((H2 (make-hash-table 32)))
    (copy H H2)
    (test (or (procedure? (hash-table-key-typer H2)) (procedure? (hash-table-value-typer H2))) #f)
    (test (string-wi=? (object->string H2 :readable)
		       "(let ((<1> (let ((<h> (make-hash-table 8 (cons eq? hash-code) (cons symbol? hash-table?)))) <h>)))
                          (set! (<1> 'a) <1>)
                          (hash-table 'a <1>))")
	  #t))

  (set! H (make-hash-table 8 (cons equivalent? hash-code) (cons symbol? #t)))
  (set! (H 'a) H) (set! (H 'b) 2) (set! (H 'c) (list H)) (set! (H 'd) "hi")
  (let ((str (object->string H :readable)))
    (test H (eval-string str)))
  )

(let ()
  (define H (hash-table 'a 1 'b 2 'c 3))

  (let ((last-key #f))

    (define (valtyp val)
      (or (not last-key)
	  (eq? last-key 'a)
	  (and (eq? last-key 'b)
	       (<= 0 val 32))))

    (define (keytyp key)
      (set! last-key key)
      #t)

    (set! (hash-table-key-typer H) keytyp)
    (set! (hash-table-value-typer H) valtyp))

  ;; now a can be set but b must between 0..32 and c is immutable

  (test H (hash-table 'a 1 'b 2 'c 3))

  (test (catch #t (lambda () (hash-table-set! H 'a 11)) (lambda (t i) (apply format #f i))) 11)
  (test (catch #t (lambda () (hash-table-set! H 'b 12)) (lambda (t i) (apply format #f i))) 12)
  (test (catch #t (lambda () (hash-table-set! H 'c 13)) (lambda (t i) (apply format #f i)))
	"hash-table-set! third argument 13, is an integer, but the hash-table's value type checker, valtyp, rejects it")

  (test H (hash-table 'a 11 'b 12 'c 3))

  (test (catch #t (lambda () (hash-table-set! H 'a 111)) (lambda (t i) (apply format #f i))) 111)
  (test (catch #t (lambda () (hash-table-set! H 'b 112)) (lambda (t i) (apply format #f i)))
	"hash-table-set! third argument 112, is an integer, but the hash-table's value type checker, valtyp, rejects it")
  (test (catch #t (lambda () (hash-table-set! H 'c 113)) (lambda (t i) (apply format #f i)))
	"hash-table-set! third argument 113, is an integer, but the hash-table's value type checker, valtyp, rejects it")

  (test H (hash-table 'a 111 'b 12 'c 3))

  (set! (H 'c) #f)
  (test H (hash-table 'a 111 'b 12)))

(let ()
  (define H_2 (make-hash-table 8 (cons equal? hash-code)))
  (set! (H_2 'a) 32)
  (test (H_2 'a) 32)
  (set! (H_2 'a) 12)
  (test (H_2 'a) 12)
  (set! (H_2 ()) #(1))
  (test (H_2 ()) #(1))

  (define H_3 (make-hash-table 8
			       (let ((eqf (lambda (a b) (equal? a b)))
				     (mapf (lambda (a) (hash-code a))))
				 (cons eqf mapf))))
  (set! (H_3 'a) 32)
  (test (H_3 'a) 32)
  (set! (H_3 'a) 12)
  (test (H_3 'a) 12)
  (set! (H_3 ()) #(1))
  (test (H_3 ()) #(1))

  (let ((H_4 (make-hash-table 8
			      (let ((eqf (lambda (a b) (equal? a b)))
				    (mapf (lambda (a) (hash-code a))))
				(cons eqf mapf))))
	(last-key #f))
    (define (valtyp val)
      (or (not last-key)
	  (eq? last-key 'v1)
	  (and (eq? last-key 'v2)
	       (<= 0 val 32))))
    (define (keytyp key)
      (set! last-key key)
      #t)
    (set! (hash-table-key-typer H_4) keytyp)
    (set! (hash-table-value-typer H_4) valtyp)

    (set! (H_4 'v1) "asdf")
    (set! (H_4 'v2) 12)
    (test (catch #t (lambda () (set! (H_4 'v3) 0)) (lambda (t i) (apply format #f i)))
          "hash-table-set! third argument 0, is an integer, but the hash-table's value type checker, valtyp, rejects it")))

(let ((L (openlet (inlet 'hash-table-key-typer (lambda (h) 123)
			 'hash-table-value-typer (lambda (h) 321)
			 'vector-typer (lambda (v) 231)))))
  (test (hash-table-key-typer L) 123)
  (test (hash-table-value-typer L) 321)
  (test (vector-typer L) 231))

(unless (provided? 'debug.scm)
  (let ((H_4 (make-hash-table 8
	       (let ((eqf (lambda (a b) (equal? a b)))
	             (mapf (lambda (a) (hash-code a))))
			     (cons eqf mapf)))))
    (hash-table-set! H_4 #\7 begin)
    ;; (H_4 #\7) -> begin, (begin 0) -> 0! but this clobbered sc->args in implicit_index
    (let ()
      (define (func)
        (let ((x #f) (i 0))
  	  (let ((x (list (hash-table-ref H_4 #\7 (logxor))))) x)))
      (func)))) ; reuse permanent let in op_let1

(let ((H_2 (make-hash-table 8 eq? (cons symbol? integer?)))) (define (func) (copy (hash-table 'a 1) H_2)) (func) (func)
  (test (and (= (hash-table-entries H_2) 1) (eqv? (hash-table-ref H_2 'a) 1)) #t))

(let ()
  (define (valtyp val) (integer? val))
  (define (keytyp key) (string? key))
  (define H (make-hash-table 8 (cons string=? string-length)))
  (set! (hash-table-key-typer H) keytyp)
  (set! (hash-table-value-typer H) valtyp)
  (hash-table-set! H "a" 1)
  (let ((HL (object->let H)))
    (test (HL 'signature) '(valtyp hash-table? keytyp))
    (test (HL 'function) (list string=? string-length)))
  (test (object->string H :readable)
	"(let ((<h> (make-hash-table 8 (cons string=? string-length) (cons keytyp valtyp)))) (copy (hash-table \"a\" 1) <h>))")
  (test (hash-table-key-typer H) keytyp)
  (test (hash-table-value-typer H) valtyp))

(let ()
  (define H (make-hash-table 8 eqv? (cons string? integer?)))
  (hash-table-set! H "a" 1)
  (let ((HL (object->let H)))
    (test (HL 'signature) '(integer? hash-table? string?))
    (test (HL 'function) 'eqv?))
  (test (object->string H :readable)
	"(let ((<h> (make-hash-table 8 eqv? (cons string? integer?)))) (copy (hash-table \"a\" 1) <h>))")
  (test (hash-table-key-typer H) string?)
  (test (hash-table-value-typer H) integer?))

(let ()
  (define H (make-hash-table 8 string=? (cons string? integer?)))
  (hash-table-set! H "a" 1)
  (let ((HL (object->let H)))
    (test (HL 'signature) '(integer? hash-table? string?))
    (test (HL 'function) 'string=?))
  (test (object->string H :readable)
        "(let ((<h> (make-hash-table 8 string=? (cons string? integer?)))) (copy (hash-table \"a\" 1) <h>))")
  (test (hash-table-key-typer H) string?)
  (test (hash-table-value-typer H) integer?))

(let ()
  (define H (make-hash-table))
  (define (keytyp key) (string? key))
  (set! (hash-table-key-typer H) keytyp)
  (hash-table-set! H "a" 1)
  (let ((HL (object->let H)))
    (test (HL 'signature) '(#t hash-table? keytyp))
    (test (HL 'function) 'string=?))
  (test (object->string H :readable)
	"(let ((<h> (make-hash-table 8 string=? (cons keytyp #t)))) (copy (hash-table \"a\" 1) <h>))")
  (test (hash-table-key-typer H) keytyp)
  (test (hash-table-value-typer H) #t))

(let ()
  (define H (make-hash-table 8 #f (cons integer? integer?)))
  (hash-table-set! H 0 1)
  (let ((HL (object->let H)))
    (test (HL 'signature) '(integer? hash-table? integer?))
    (test (HL 'function) '=))
  (test (object->string H :readable)
	"(let ((<h> (make-hash-table 8 #f (cons integer? integer?)))) (copy (hash-table 0 1) <h>))")
  (test (hash-table-key-typer H) integer?)
  (test (hash-table-value-typer H) integer?))

(let-temporarily ((*#readers*
		   (cons (cons #\h (lambda (str)
				     (and (string=? str "h") ; #h(...)
					  (if (> (*s7* 'safety) 0)
					      (immutable! (apply hash-table (read)))
					      (apply hash-table (read))))))
			 *#readers*)))
  (test (eval-string "#h(:a 1)") (hash-table :a 1))
  (test (eval-string "#h(:a 1 :b \"asdf\")") (hash-table :a 1 :b "asdf"))
  (test (eval-string "#h(a 1 b 2)") (hash-table 'a 1 'b 2)) ; 'a and :a are different keys in hash-tables:
  (test ((hash-table 'a 1) 'a) 1) ; but...
  (test ((hash-table :a 1) 'a) #f)) ; was this intentional?


;;; --------------------------------------------------------------------------------
;;; some implicit index tests

(test (#(#(1 2) #(3 4)) 1 1) 4)
(test (#("12" "34") 0 1) #\2)
(test (#((1 2) (3 4)) 1 0) 3)
(test (#((1 (2 3))) 0 1 0) 2)
(test ((vector (hash-table 'a 1 'b 2)) 0 'a) 1)
(test ((list (lambda (x) x)) 0 "hi") 'error) ; "hi")
(test (let ((lst '("12" "34"))) (lst 0 1)) #\2)
(test (let ((lst (list #(1 2) #(3 4)))) (lst 0 1)) 2)
(test (#2d(("hi" "ho") ("ha" "hu")) 1 1 0) #\h)
(test ((list (lambda (a) (+ a 1)) (lambda (b) (* b 2))) 1 2) 'error) ; 4)
(test ((lambda (arg) arg) "hi" 0) 'error)

(let ((L1 (list 1 2 3))
      (V1 (vector 1 2 3))
      (M1 #2d((1 2 3) (4 5 6) (7 8 9)))
      (S1 "123")
      (H1 (hash-table 1 1 2 2 3 3)))
  (let ((L2 (list L1 V1 M1 S1 H1))
	(V2 (vector L1 V1 M1 S1 H1))
	(H2 (hash-table 0 L1 1 V1 2 M1 3 S1 4 H1))
	(M2 (let ((v (make-vector '(3 3))))
	      (set! (v 0 0) L1)
	      (set! (v 0 1) V1)
	      (set! (v 0 2) M1)
	      (set! (v 1 0) S1)
	      (set! (v 1 1) H1)
	      (set! (v 1 2) L1)
	      (set! (v 2 0) S1)
	      (set! (v 2 1) H1)
	      (set! (v 2 2) L1)
	      v)))
#|
    ;; this code generates the tests below
    (for-each
     (lambda (arg)
       (let* ((val (symbol->value arg))
	      (len (min 5 (length val))))
	 (do ((i 0 (+ i 1)))
	     ((= i len))
	   (format *stderr* "(test (~S ~S) ~S)~%" arg i
		   (catch #t (lambda () (val i)) (lambda args 'error)))
	   (let ((len1 (catch #t (lambda () (min 5 (length (val i)))) (lambda args 0))))
	     (if (> len1 0)
		 (do ((k 0 (+ k 1)))
		     ((= k len1))
		   (format *stderr* "(test (~S ~S ~S) ~S)~%" arg i k
			   (catch #t (lambda () (val i k)) (lambda args 'error)))
		   (let ((len2 (catch #t (lambda () (min 5 (length (val i k)))) (lambda args 0))))
		     (if (> len2 0)
			 (do ((m 0 (+ m 1)))
			     ((= m len2))
			   (format *stderr* "(test (~S ~S ~S ~S) ~S)~%" arg i k m
				   (catch #t (lambda () (val i k m)) (lambda args 'error)))
			   (let ((len3 (catch #t (lambda () (min 5 (length (val i k m)))) (lambda args 0))))
			     (if (> len3 0)
				 (do ((n 0 (+ n 1)))
				     ((= n len3))
				   (format *stderr* "(test (~S ~S ~S ~S ~S) ~S)~%" arg i k m n
					   (catch #t (lambda () (val i k m n)) (lambda args 'error)))))))))))))))
     (list 'L2 'V2 'M2 'H2))
|#

    (test (L2 0) '(1 2 3))
    (test (L2 0 0) 1)
    (test (L2 0 1) 2)
    (test (L2 0 2) 3)
    (test (L2 1) #(1 2 3))
    (test (L2 1 0) 1)
    (test (L2 1 1) 2)
    (test (L2 1 2) 3)
    (test (L2 2) #2d((1 2 3) (4 5 6) (7 8 9)))
    (test (L2 2 0) #(1 2 3))
    (test (L2 2 0 0) 1)
    (test (L2 2 0 1) 2)
    (test (L2 2 0 2) 3)
    (test (L2 2 1) #(4 5 6))
    (test (L2 2 1 0) 4)
    (test (L2 2 1 1) 5)
    (test (L2 2 1 2) 6)
    (test (L2 2 2) #(7 8 9))
    (test (L2 2 2 0) 7)
    (test (L2 2 2 1) 8)
    (test (L2 2 2 2) 9)
    (test (L2 2 3) 'error)
    (test (L2 2 4) 'error)
    (test (L2 3) "123")
    (test (L2 3 0) #\1)
    (test (L2 3 1) #\2)
    (test (L2 3 2) #\3)
    (test (L2 4) H1)
    (test (L2 4 0) #f)
    (test (L2 4 1) 1)
    (test (L2 4 2) 2)
    (test (L2 4 3) 3)
    (test (L2 4 4) #f)
    (test (V2 0) '(1 2 3))
    (test (V2 0 0) 1)
    (test (V2 0 1) 2)
    (test (V2 0 2) 3)
    (test (V2 1) #(1 2 3))
    (test (V2 1 0) 1)
    (test (V2 1 1) 2)
    (test (V2 1 2) 3)
    (test (V2 2) #2d((1 2 3) (4 5 6) (7 8 9)))
    (test (V2 2 0) #(1 2 3))
    (test (V2 2 0 0) 1)
    (test (V2 2 0 1) 2)
    (test (V2 2 0 2) 3)
    (test (V2 2 1) #(4 5 6))
    (test (V2 2 1 0) 4)
    (test (V2 2 1 1) 5)
    (test (V2 2 1 2) 6)
    (test (V2 2 2) #(7 8 9))
    (test (V2 2 2 0) 7)
    (test (V2 2 2 1) 8)
    (test (V2 2 2 2) 9)
    (test (V2 2 3) 'error)
    (test (V2 2 4) 'error)
    (test (V2 3) "123")
    (test (V2 3 0) #\1)
    (test (V2 3 1) #\2)
    (test (V2 3 2) #\3)
    (test (V2 4) H1)
    (test (V2 4 0) #f)
    (test (V2 4 1) 1)
    (test (V2 4 2) 2)
    (test (V2 4 3) 3)
    (test (V2 4 4) #f)
    (test (M2 0) #((1 2 3) #(1 2 3) #2d((1 2 3) (4 5 6) (7 8 9))))
    (test (M2 0 0) '(1 2 3))
    (test (M2 0 0 0) 1)
    (test (M2 0 0 1) 2)
    (test (M2 0 0 2) 3)
    (test (M2 0 1) #(1 2 3))
    (test (M2 0 1 0) 1)
    (test (M2 0 1 1) 2)
    (test (M2 0 1 2) 3)
    (test (M2 0 2) #2d((1 2 3) (4 5 6) (7 8 9)))
    (test (M2 0 2 0) #(1 2 3))
    (test (M2 0 2 0 0) 1)
    (test (M2 0 2 0 1) 2)
    (test (M2 0 2 0 2) 3)
    (test (M2 0 2 1) #(4 5 6))
    (test (M2 0 2 1 0) 4)
    (test (M2 0 2 1 1) 5)
    (test (M2 0 2 1 2) 6)
    (test (M2 0 2 2) #(7 8 9))
    (test (M2 0 2 2 0) 7)
    (test (M2 0 2 2 1) 8)
    (test (M2 0 2 2 2) 9)
    (test (M2 0 2 3) 'error)
    (test (M2 0 2 4) 'error)
    (test (M2 1) (vector "123" H1 '(1 2 3)))
    (test (M2 1 0) "123")
    (test (M2 1 0 0) #\1)
    (test (M2 1 0 1) #\2)
    (test (M2 1 0 2) #\3)
    (test (M2 1 1) H1)
    (test (M2 1 1 0) #f)
    (test (M2 1 1 1) 1)
    (test (M2 1 1 2) 2)
    (test (M2 1 1 3) 3)
    (test (M2 1 1 4) #f)
    (test (M2 1 2) '(1 2 3))
    (test (M2 1 2 0) 1)
    (test (M2 1 2 1) 2)
    (test (M2 1 2 2) 3)
    (test (M2 2) (vector "123" H1 '(1 2 3)))
    (test (M2 2 0) "123")
    (test (M2 2 0 0) #\1)
    (test (M2 2 0 1) #\2)
    (test (M2 2 0 2) #\3)
    (test (M2 2 1) H1)
    (test (M2 2 1 0) #f)
    (test (M2 2 1 1) 1)
    (test (M2 2 1 2) 2)
    (test (M2 2 1 3) 3)
    (test (M2 2 1 4) #f)
    (test (M2 2 2) '(1 2 3))
    (test (M2 2 2 0) 1)
    (test (M2 2 2 1) 2)
    (test (M2 2 2 2) 3)
    (test (M2 3) 'error)
    (test (M2 4) 'error)
    (test (H2 0) '(1 2 3))
    (test (H2 0 0) 1)
    (test (H2 0 1) 2)
    (test (H2 0 2) 3)
    (test (H2 1) #(1 2 3))
    (test (H2 1 0) 1)
    (test (H2 1 1) 2)
    (test (H2 1 2) 3)
    (test (H2 2) #2d((1 2 3) (4 5 6) (7 8 9)))
    (test (H2 2 0) #(1 2 3))
    (test (H2 2 0 0) 1)
    (test (H2 2 0 1) 2)
    (test (H2 2 0 2) 3)
    (test (H2 2 1) #(4 5 6))
    (test (H2 2 1 0) 4)
    (test (H2 2 1 1) 5)
    (test (H2 2 1 2) 6)
    (test (H2 2 2) #(7 8 9))
    (test (H2 2 2 0) 7)
    (test (H2 2 2 1) 8)
    (test (H2 2 2 2) 9)
    (test (H2 2 3) 'error)
    (test (H2 2 4) 'error)
    (test (H2 3) "123")
    (test (H2 3 0) #\1)
    (test (H2 3 1) #\2)
    (test (H2 3 2) #\3)
    (test (H2 4) H1)
    (test (H2 4 0) #f)
    (test (H2 4 1) 1)
    (test (H2 4 2) 2)
    (test (H2 4 3) 3)
    (test (H2 4 4) #f)
     ))

(let* ((L1 (cons 1 2))
       (L2 (list L1 3)))
  (test (L1 0) 1)
  (test (L1 1) 'error)
  (test (L1 2) 'error)
  (test (L2 0 0) 1)
  (test (L2 0 1) 'error)
  (test ((cons "123" 0) 0 1) #\2))

(let ((L1 (list "123" "456" "789")))
  (set-cdr! (cdr L1) L1)
  (test (L1 0 1) #\2)
  (test (L1 1 1) #\5)
  (test (L1 2 1) #\2)
  (test (L1 12 0) #\1))

(let ((L1 (list "123" "456" "789")))
  (set-car! (cdr L1) L1)
  (test (L1 1 1 1 1 1 0 0) #\1))

(test ((list (list) "") 1 0) 'error)
(test ((list (list) "") 0 0) 'error)
(test (#(1 2) 0 0) 'error)
(test (#(1 #()) 1 0) 'error)

(test ('(((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((12))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))) 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0) 12)

;;; implicit index as expression (_A cases)

(let ((L1 (list 1 2 3))
      (V1 (vector 1 2 3))
      (S1 "123")
      (H1 (hash-table 1 1 2 2 3 3))
      (E1 (inlet :a 1 :b 2)))
  (define (f1 i s L V S H E)
    (vector (L (+ i 1)) (V (+ i 1)) (S (+ i 1)) (H (+ i 1)) (E (string->symbol s))))
  (test (f1 0 "a" L1 V1 S1 H1 E1) (vector 2 2 #\2 1 1))
  (test (f1 1 "b" L1 V1 S1 H1 E1) (vector 3 3 #\3 2 2))
  (define (f2 i s L V S H E)
    (vector (L (abs i)) (V (abs i)) (S (abs i)) (H (abs i)) (E (vector-ref s 0))))
  (test (f2 -2 #(b a) L1 V1 S1 H1 E1) (vector 3 3 #\3 2 2)))

(when with-block
  (define (f3 B i) (B (+ i 1)))
  (define (f4 B i) (B (abs i)))
  (let ((b (make-block 4)))
    (set! (b 0) 1.0)
    (set! (b 1) 2.0)
    (test (f3 b -1) 1.0)
    (test (f4 b -1) 2.0)))

(let ((v1 #(0 1 2 3 4 5 6 7))
      (v2 #2d((0 1 2 3) (4 5 6 7)))
      (e1 (inlet :a 1))
      (p1 (list 0 1 2 3))
      (s1 "0123")
      )
  (define (call-1 func arg1 arg2) (func arg1 arg2))
  (define (call-2 func arg1 arg2) (func arg1 arg2))
  (define (call-3 func arg1 arg2) (func arg1 arg2))
  (define (call-4 func arg1 arg2) (func arg1 arg2))
  (define (call-5 func arg1 arg2) (func arg1 arg2))
  (define (call-6 func) (func 'a))
  (define (call-7 func) (func 'a))
  (define (call-8 func arg) (func (* 2 (+ arg 1))))
  (define (call-9 func arg) (func (* 2 (+ arg 1))))
  (define (call-10 func arg) (func arg))
  (define (call-11 func arg) (func arg))
  (define (call-12 func) (func 0))
  (define (call-13 func) (func 0))
  (define (call-14 func arg) (func arg 2))
  (define (call-15 func arg) (func 2 arg))

  (define (f+ x y) (+ x y))
  (define (f- x y) (- x y))
  (define* (f++ (x 0) y) (+ x y))
  (define* (f-- x (y 0)) (- x y))
  (define (fabs x) (abs x))
  (define-macro (m+ x y) `(+ ,x ,y))

  (test (call-1 + 5 2) 7)
  (test (call-1 f- 5 2) 3)
  (test (call-2 f+ 5 2) 7)
  (test (call-2 - 5 2) 3)
  (test (call-3 v2 0 3) 3)
  (test (call-3 list 0 3) (list 0 3))
  (test (call-4 f++ 5 2) 7)
  (test (call-4 f-- 5 2) 3)
  (test (call-5 m+ 5 2) 7)
  (test (call-5 - 5 2) 3)
  (test (call-6 e1) 1)
  (test (call-6 symbol?) #t)
  (test (call-7 symbol?) #t)
  (test (call-7 list) (list 'a))
  (test (call-8 abs -3) 4)
  (test (call-8 f-- 10) 22)
  (test (call-9 fabs -3) 4)
  (test (call-9 list -3) (list -4))
  (test (call-10 e1 'a) 1)
  (test (call-10 list 'a) (list 'a))
  (test (call-11 symbol? 'a) #t)
  (test (call-11 e1 'a) 1)
  (test (call-12 p1) 0)
  (test (call-12 s1) #\0)
  (test (call-13 v1) 0)
  (test (call-13 (lambda (x) (+ x 1))) 1)
  (test (call-14 * 3) 6)
  (test (call-14 (lambda (x y) (- x y)) 3) 1)
  (test (call-15 (lambda (x y) (- x y)) 3) -1)
  (test (call-15 - 3) -1)
  )

;; multi-index get/set
(let ((v (vector (hash-table 'a 1 'b 2)))) (test (v 0 'a) 1) (set! (v 0 'a) 5) (test (v 0 'a) 5))
(let ((v (vector (inlet 'a 1 'b 2)))) (test (v 0 'a) 1) (set! (v 0 'a) 5) (test (v 0 'a) 5))
(let ((v (vector (list 1 2)))) (test (v 0 1) 2) (set! (v 0 1) 5) (test (v 0 1) 5))
(let ((v (vector (string #\1 #\2)))) (test (v 0 1) #\2) (set! (v 0 1) #\5) (test (v 0 1) #\5))
(let ((v (vector (vector 1 2)))) (test (v 0 1) 2) (set! (v 0 1) 5) (test (v 0 1) 5))
(let ((v (vector (byte-vector 1 2)))) (test (v 0 1) 2) (set! (v 0 1) 5) (test (v 0 1) 5))
(when with-block (let ((v (vector (block 1 2)))) (test (v 0 1) 2.0) (set! (v 0 1) 5) (test (v 0 1) 5.0)))
(let ((v (vector (float-vector 1 2)))) (test (v 0 1) 2.0) (set! (v 0 1) 5) (test (v 0 1) 5.0))
(let ((v (vector (int-vector 1 2)))) (test (v 0 1) 2) (set! (v 0 1) 5) (test (v 0 1) 5))
(let ((v (list (hash-table 'a 1 'b 2)))) (test (v 0 'a) 1) (set! (v 0 'a) 5) (test (v 0 'a) 5))
(let ((v (list (inlet 'a 1 'b 2)))) (test (v 0 'a) 1) (set! (v 0 'a) 5) (test (v 0 'a) 5))
(let ((v (list (list 1 2)))) (test (v 0 1) 2) (set! (v 0 1) 5) (test (v 0 1) 5))
(let ((v (list (string #\1 #\2)))) (test (v 0 1) #\2) (set! (v 0 1) #\5) (test (v 0 1) #\5))
(let ((v (list (vector 1 2)))) (test (v 0 1) 2) (set! (v 0 1) 5) (test (v 0 1) 5))
(let ((v (list (byte-vector 1 2)))) (test (v 0 1) 2) (set! (v 0 1) 5) (test (v 0 1) 5))
(when with-block (let ((v (list (block 1 2)))) (test (v 0 1) 2.0) (set! (v 0 1) 5) (test (v 0 1) 5.0)))
(let ((v (list (float-vector 1 2)))) (test (v 0 1) 2.0) (set! (v 0 1) 5) (test (v 0 1) 5.0))
(let ((v (list (int-vector 1 2)))) (test (v 0 1) 2) (set! (v 0 1) 5) (test (v 0 1) 5))
(let ((v (hash-table 'a (hash-table 'a 1 'b 2)))) (test (v 'a 'a) 1) (set! (v 'a 'a) 5) (test (v 'a 'a) 5))
(let ((v (hash-table 'a (inlet 'a 1 'b 2)))) (test (v 'a 'a) 1) (set! (v 'a 'a) 5) (test (v 'a 'a) 5))
(let ((v (hash-table 'a (list 1 2)))) (test (v 'a 1) 2) (set! (v 'a 1) 5) (test (v 'a 1) 5))
(let ((v (hash-table 'a (string #\1 #\2)))) (test (v 'a 1) #\2) (set! (v 'a 1) #\5) (test (v 'a 1) #\5))
(let ((v (hash-table 'a (vector 1 2)))) (test (v 'a 1) 2) (set! (v 'a 1) 5) (test (v 'a 1) 5))
(let ((v (hash-table 'a (byte-vector 1 2)))) (test (v 'a 1) 2) (set! (v 'a 1) 5) (test (v 'a 1) 5))
(when with-block (let ((v (hash-table 'a (block 1 2)))) (test (v 'a 1) 2.0) (set! (v 'a 1) 5) (test (v 'a 1) 5.0)))
(let ((v (hash-table 'a (float-vector 1 2)))) (test (v 'a 1) 2.0) (set! (v 'a 1) 5) (test (v 'a 1) 5.0))
(let ((v (hash-table 'a (int-vector 1 2)))) (test (v 'a 1) 2) (set! (v 'a 1) 5) (test (v 'a 1) 5))
(let ((v (inlet 'a (hash-table 'a 1 'b 2)))) (test (v 'a 'a) 1) (set! (v 'a 'a) 5) (test (v 'a 'a) 5))
(let ((v (inlet 'a (inlet 'a 1 'b 2)))) (test (v 'a 'a) 1) (set! (v 'a 'a) 5) (test (v 'a 'a) 5))
(let ((v (inlet 'a (list 1 2)))) (test (v 'a 1) 2) (set! (v 'a 1) 5) (test (v 'a 1) 5))
(let ((v (inlet 'a (string #\1 #\2)))) (test (v 'a 1) #\2) (set! (v 'a 1) #\5) (test (v 'a 1) #\5))
(let ((v (inlet 'a (vector 1 2)))) (test (v 'a 1) 2) (set! (v 'a 1) 5) (test (v 'a 1) 5))
(let ((v (inlet 'a (byte-vector 1 2)))) (test (v 'a 1) 2) (set! (v 'a 1) 5) (test (v 'a 1) 5))
(when with-block (let ((v (inlet 'a (block 1 2)))) (test (v 'a 1) 2.0) (set! (v 'a 1) 5) (test (v 'a 1) 5.0)))
(let ((v (inlet 'a (float-vector 1 2)))) (test (v 'a 1) 2.0) (set! (v 'a 1) 5) (test (v 'a 1) 5.0))
(let ((v (inlet 'a (int-vector 1 2)))) (test (v 'a 1) 2) (set! (v 'a 1) 5) (test (v 'a 1) 5))

(let ((ind 0) (sym 'a) (v (vector (hash-table 'a 1 'b 2)))) (test (v ind sym) 1) (set! (v (+ ind ind) sym) (+ ind 5)) (test (v 0 'a) 5))
(let ((v (vector (hash-table 'a "123" 'b 2)))) (test (v 0 'a 1) #\2) (set! (v 0 'a 1) #\5) (test (v 0 'a) "153"))

(let ((iv (make-vector '(2 2))))
  (set! (iv 1 0) 2)
  (set! (iv 1 1) 4)
  (let ((v (vector iv)))
    (test (v 0 1 0) 2)
    (set! (v 0 1 0) 5)
    (test (v 0 1) #(5 4))))

(let ((ov (make-vector '(2 2)))
      (iv (make-vector '(2 2))))
  (set! (ov 1 0) iv)
  (set! (iv 0 1) 3)
  (test (ov 1 0 0 1) 3)
  (set! (ov 1 0 0 1) 5)
  (test (ov 1 0 0 1) 5))

(test (let () (define (func) (abs ((list #f #r(1 2 3)) 1 0))) (define (hi) (func)) (hi)) 1.0)
(test (let () (define (func) (abs ((list #f #i(1 2 3)) 1 0))) (define (hi) (func)) (hi)) 1)
(test (let () (define (func) (abs ((list #f #(1 2 3)) 1 0))) (define (hi) (func)) (hi)) 1)
(test (let () (define (func) (abs ((list #f #u(1 2 3)) 1 0))) (define (hi) (func)) (hi)) 1)
(test (let () (define (func) (char->integer ((list #f "123") 1 0))) (define (hi) (func)) (hi)) (char->integer #\1))
(test (let () (define (func) (abs ((list #f (lambda (x) (+ x 10))) 1 0))) (define (hi) (func)) (hi)) 'error) ; 10)
(test (let () (define (func) (abs ((list #f ceiling) 1 1.1))) (define (hi) (func)) (hi)) 2)
(test (let () (define (func) (abs ((list #f quasiquote) 1 2))) (define (hi) (func)) (hi)) 2)
(test (let () (define (func) (abs ((list #f (define-macro (_m_ x) `(+ ,x 1))) 1 2))) (define (hi) (func)) (hi)) 3)
(test (let () (define (func) (abs ((list #f (inlet 'a -2)) 1 :a))) (define (hi) (func)) (hi)) 2)
(test (let () (define (func) (abs ((list #f (inlet :a -2)) 1 :a))) (define (hi) (func)) (hi)) 2)
(test (let () (define (func) (abs ((list #f (inlet 'a -2)) 1 'a))) (define (hi) (func)) (hi)) 2)
(test (let () (define (func) (abs ((list #f (inlet :a -2)) 1 'a))) (define (hi) (func)) (hi)) 2)
(test (let () (define (func) (abs ((list #f (hash-table :a -2)) 1 :a))) (define (hi) (func)) (hi)) 2)
(test (let () (define (func) (abs ((list #f (hash-table 'a -2)) 1 'a))) (define (hi) (func)) (hi)) 2)
(test (let () (define (func) (abs ((list #f begin) 1 3))) (define (hi) (func)) (hi)) 3)
(test (let () (define (func) (abs ((list #f when) 1 #t 4))) (define (hi) (func)) (hi)) 4)
(test (let () (define (func) (abs (list-ref (list #f (list #i(1 2 3))) 1 0 0))) (define (hi) (func)) (hi)) 1) ; 'error -- implicit_index
(test (let () (define (func) (abs (list-ref (list #f #i(1 2 3)) 1 0))) (define (hi) (func)) (hi)) 1)          ;  'error -- same
(test (let () (define (func) (abs (vector-ref (vector #f '(1 2 3)) 1 0))) (define (hi) (func)) (hi)) 1)       ; same
(test (let () (define (func) (abs ((inlet 'a (inlet 'b 2)) 'a 'b))) (define (hi) (func)) (hi)) 2)
(test (let () (define (func) (abs ((inlet 'a (vector 1 2 3)) 'a 1))) (define (hi) (func)) (hi)) 2)
(test (let () (define (func) (abs (hash-table-ref (hash-table 'a (hash-table 'b 2)) 'a 'b))) (define (hi) (func)) (hi)) 2)
(test (let () (define (func) (abs ((hash-table 'a (hash-table 'b 2)) 'a 'b))) (define (hi) (func)) (hi)) 2)
(test (let () (define (func) (abs ((hash-table 'a (vector 1 2 3)) 'a 1))) (define (hi) (func)) (hi)) 2)

;;; (hash-table 'a 1) is different from (hash-table :a 1):
(test (let () (define (func) (abs ((list #f (hash-table 'a -2)) 1 :a))) (define (hi) (func)) (hi)) 'error)
(test (let () (define (func) (abs ((list #f (hash-table :a -2)) 1 'a))) (define (hi) (func)) (hi)) 'error)

(test (let () (define (func) (abs (let-ref (inlet 'a (inlet 'b 2)) 'a 'b))) (define (hi) (func)) (hi)) 'error)
(test (let () (define (func) (abs (hash-table-ref (hash-table 'a (vector 1 2 3)) 'a 1))) (define (hi) (func)) (hi)) 2) ; see above 'error



;;; --------------------------------------------------------------------------------
;;; PORTS
;;; --------------------------------------------------------------------------------

;;; with-output-to-string
;;; with-output-to-file
;;; call-with-output-string
;;; call-with-output-file
;;; with-input-from-string
;;; with-input-from-file
;;; call-with-input-string
;;; call-with-input-file
;;; read-char
;;; peek-char
;;; close-input-port
;;; close-output-port
;;; flush-output-port
;;; open-input-file
;;; open-output-file
;;; open-input-string
;;; open-output-string
;;; get-output-string
;;; open-input-function
;;; open-output-function
;;; write-char
;;; write-string
;;; read-byte
;;; write-byte
;;; read-line
;;; read-string
;;; read
;;; write

(define start-input-port (current-input-port))
(define start-output-port (current-output-port))

(test (input-port? (current-input-port)) #t)
(test (input-port? *stdin*) #t)
(test (input-port? (current-output-port)) #f)
(test (input-port? *stdout*) #f)
(test (input-port? (current-error-port)) #f)
(test (input-port? *stderr*) #f)

(for-each
 (lambda (arg)
   (if (input-port? arg)
       (format #t ";(input-port? ~A) -> #t?~%" arg)))
 (list "hi" #f (integer->char 65) 1 (list 1 2) '#t '3 (make-vector 3) 3.14 3/4 1.0+1.0i #\f :hi #<eof> #<undefined> #<unspecified>))

(test (call-with-input-file "s7test.scm" input-port?) #t)
(if (not (eq? start-input-port (current-input-port)))
    (format #t "call-with-input-file did not restore current-input-port? ~A from ~A~%" start-input-port (current-input-port)))

(test (let ((this-file (open-input-file "s7test.scm"))) (let ((res (input-port? this-file))) (close-input-port this-file) res)) #t)
(if (not (eq? start-input-port (current-input-port)))
    (format #t "open-input-file clobbered current-input-port? ~A from ~A~%" start-input-port (current-input-port)))

(test (call-with-input-string "(+ 1 2)" input-port?) #t)
(test (let ((this-file (open-input-string "(+ 1 2)"))) (let ((res (input-port? this-file))) (close-input-port this-file) res)) #t)
(test (let ((this-file (open-input-string "(+ 1 2)"))) (let ((len (length this-file))) (close-input-port this-file) len)) 7)

;;; (test (let ((str "1234567890")) (let ((p (open-input-string str))) (string-set! str 0 #\a) (let ((c (read-char p))) (close-input-port p) c))) #\1)
;;; is that result demanded by the scheme spec? perhaps make str immutable if so?

(test (+ 100 (call-with-input-string "123" (lambda (p) (values (read p) 1)))) 224)

(let ((p (open-input-string "asdf")))
  (test (display pi p) 'error)
  (test (write pi p) 'error)
  (test (close-output-port p) 'error)
  (test (flush-output-port p) 'error)
  (close-input-port p)
  (test (read p) 'error))
(let ((p (open-output-string)))
  (test (read-char p) 'error)
  (test (read-byte p) 'error)
  (test (read-line p) 'error)
  (test (read-string 1 p) 'error)
  (test (read-line p) 'error)
  (test (close-input-port p) 'error)
  (close-output-port p)
  (test (display pi p) 'error)
  (test (write pi p) 'error))

(test (call-with-input-string
       "1234567890"
       (lambda (p)
	 (call-with-input-string
	  "0987654321"
	  (lambda (q)
            (+ (read p) (read q))))))
      2222222211)

(test (call-with-input-string
       "12345 67890"
       (lambda (p)
	 (call-with-input-string
	  "09876 54321"
	  (lambda (q)
            (- (+ (read p) (read q)) (read p) (read q))))))
      -99990)

(call-with-output-file "empty-file" (lambda (p) #f))
(test (call-with-input-file "empty-file" (lambda (p) (eof-object? (read-char p)))) #t)
(test (call-with-input-file "empty-file" (lambda (p) (eof-object? (read p)))) #t)
(test (call-with-input-file "empty-file" (lambda (p) (eof-object? (read-byte p)))) #t)
(test (call-with-input-file "empty-file" (lambda (p) (eof-object? (read-line p)))) #t)
(test (load "empty-file") #<unspecified>)
(test (call-with-input-file "empty-file" (lambda (p) (port-closed? p))) #f)
(test (eof-object? (call-with-input-string "" (lambda (p) (read p)))) #t)
(test (eof-object? #<eof>) #t)
(test (let () (define (hi a) (eof-object? a)) (hi #<eof>)) #t)

(let ()
  (define (io-func) (lambda (p) (eof-object? (read-line p))))
  (test (call-with-input-file (let () "empty-file") (io-func)) #t))

(let ((p1 #f))
  (call-with-output-file "empty-file" (lambda (p) (set! p1 p) (write-char #\a p)))
  (test (port-closed? p1) #t))
(test (call-with-input-file "empty-file" (lambda (p) (and (char=? (read-char p) #\a) (eof-object? (read-char p))))) #t)
(test (call-with-input-file "empty-file" (lambda (p) (and (string=? (symbol->string (read p)) "a") (eof-object? (read p))))) #t) ; Guile also returns a symbol here
(test (call-with-input-file "empty-file" (lambda (p) (and (char=? (integer->char (read-byte p)) #\a) (eof-object? (read-byte p))))) #t)
(test (call-with-input-file "empty-file" (lambda (p) (and (string=? (read-line p) "a") (eof-object? (read-line p))))) #t)

(test (call-with-input-string "(lambda (a) (+ a 1))" (lambda (p) (let ((f (eval (read p)))) (f 123)))) 124)
(test (call-with-input-string "(let ((x 21)) (+ x 1))" (lambda (p) (eval (read p)))) 22)
(test (call-with-input-string "(1 2 3) (4 5 6)" (lambda (p) (list (read p) (read p)))) '((1 2 3) (4 5 6)))

(test (let ()
	(call-with-output-file "empty-file" (lambda (p) (write '(lambda (a) (+ a 1)) p)))
	(call-with-input-file "empty-file" (lambda (p) (let ((f (eval (read p)))) (f 123)))))
      124)
(test (let ()
	(call-with-output-file "empty-file" (lambda (p) (write '(let ((x 21)) (+ x 1)) p)))
	(call-with-input-file "empty-file" (lambda (p) (eval (read p)))))
      22)
(test (let ()
	(call-with-output-file "empty-file" (lambda (p) (write '(1 2 3) p) (write '(4 5 6) p)))
	(call-with-input-file "empty-file" (lambda (p) (list (read p) (read p)))))
      '((1 2 3) (4 5 6)))

(call-with-output-file "empty-file" (lambda (p) (for-each (lambda (c) (write-char c p)) "#b11")))
(test (call-with-input-file "empty-file" (lambda (p)
					   (and (char=? (read-char p) #\#)
						(char=? (read-char p) #\b)
						(char=? (read-char p) #\1)
						(char=? (read-char p) #\1)
						(eof-object? (read-char p)))))
      #t)
(test (call-with-input-file "empty-file" (lambda (p)
					   (and (= (read p) 3)
						(eof-object? (read p)))))
      #t)
(test (call-with-input-file "empty-file" (lambda (p)
					   (and (= (read-byte p) (char->integer #\#))
						(= (read-byte p) (char->integer #\b))
						(= (read-byte p) (char->integer #\1))
						(= (read-byte p) (char->integer #\1))
						(eof-object? (read-byte p)))))
      #t)
(test (call-with-input-file "empty-file" (lambda (p)
					   (and (string=? (read-line p) "#b11")
						(eof-object? (read-line p)))))
      #t)
(test (load "empty-file") 3)
(let ((p1 (dilambda (lambda (p) (and (= (read p) 3) (eof-object? (read p)))) (lambda (p x) #f))))
  (test (call-with-input-file "empty-file" p1) #t))


;;; load
(for-each
 (lambda (arg)
   (test (load arg) 'error)
   (test (load "empty-file" arg) 'error))
 (list () (list 1) '(1 . 2) #f #\a 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #f #t :hi (if #f #f) (lambda (a) (+ a 1))))
(test (load) 'error)
(test (load "empty-file" (curlet) 1) 'error)
(test (load "not a file") 'error)
(test (load "") 'error)
(test (load (append "/home/" username "/cl")) 'error)
(test (call-with-input-string "(display (+ 1 2))" load) 'error)
(test (let () (define (func) (#_call-with-input-string (make-vector 3 'a symbol?) (lambda (x) x))) (func)) 'error)
(test (let () (define (func) (#_call-with-input-file (openlet (inlet 'a 1)) (lambda (x) x))) (func)) 'error)

(call-with-output-file "empty-file" (lambda (p) (write '(+ 1 2 3) p)))
(let ((x 4))
  (test (+ x (load "empty-file")) 10))

(call-with-output-file "empty-file" (lambda (p) (write '(list 1 2 3) p)))
(let ((x 4))
  (test (cons x (load "empty-file")) '(4 1 2 3)))

(call-with-output-file "empty-file" (lambda (p) (write '(values 1 2 3) p)))
(let ((x 4))
  (test (+ x (load "empty-file")) 10))
(test (+ 4 (eval (call-with-input-file "empty-file" (lambda (p) (read p))))) 10)

(call-with-output-file "empty-file" (lambda (p) (write '(+ x 1) p)))
(let ((x 2))
  (test (load "empty-file" (curlet)) 3))

(call-with-output-file "empty-file" (lambda (p) (write '(set! x 1) p)))
(let ((x 2))
  (load "empty-file" (curlet))
  (test x 1))

(call-with-output-file "empty-file" (lambda (p) (write '(define (hi a) (values a 2)) p) (write '(hi x) p)))
(let ((x 4))
  (test (+ x (load "empty-file" (curlet))) 10))

(let ((x 1)
      (e #f))
  (set! e (curlet))
  (let ((x 4))
    (test (+ x (load "empty-file" e)) 7)))

(let ()
  (let ()
    (call-with-output-file "empty-file" (lambda (p) (write '(define (load_hi a) (+ a 1)) p)))
    (load "empty-file" (curlet))
    (test (load_hi 2) 3))
  (test (defined? 'load_hi) #f))

(let ()
  (apply load '("empty-file"))
  (test (load_hi 2) 3))

(call-with-output-file "empty-file" (lambda (p) (display "\"empty-file\"" p)))
(test (load (load "empty-file")) "empty-file")

;;; *cload-directory*
(test (set! *cload-directory* 123) 'error)
(let ((old-dir *cload-directory*))
  (set! *cload-directory* (append "/home/" username "/cl/"))
  (test *cload-directory* (append "/home/" username "/cl/"))
  (set! *cload-directory* old-dir))

;;; autoload
(test (autoload) 'error)
(test (autoload 'abs) 'error)
(test (autoload :abs "dsp.scm") 'error)
(for-each
 (lambda (arg)
   (test (autoload arg "dsp.scm") 'error)
   (test (autoload 'hi arg) 'error))
 (list #f () (integer->char 65) 1 (list 1 2) _ht_ _undef_ _null_ _c_obj_ '#t '3 (make-vector 3) 3.14 3/4 1.0+1.0i #\f))
(test (autoload 'abs "dsp.scm" 123) 'error)
(test (autoload "" "dsp.scm") 'error)

(autoload 'auto_test_var "empty-file")
(test (defined? 'auto_test_var) #f)
(call-with-output-file "empty-file" (lambda (p) (format p "(define auto_test_var 123)~%")))
(load "empty-file")
(test (+ 1 auto_test_var) 124)

(autoload 'auto_test_var_2 (lambda (e) (varlet e (cons 'auto_test_var_2 1))))
(test (let () (+ 1 auto_test_var_2)) 2)

(autoload 'auto_test_var_3 (lambda (e) (varlet e (cons 'auto_test_var_3 1))))
(autoload 'auto_test_var_4 (lambda (e) (varlet e (cons 'auto_test_var_4 (+ auto_test_var_3 1)))))
(test (let () (+ auto_test_var_4 1)) 3)

(test (autoload 'auto_test_var_1 (lambda () #f)) 'error)
(test (autoload 'auto_test_var_1 (lambda (a b) #f)) 'error)

(let ()
  (test (autoload 'ho "s7test.scm") "s7test.scm")
  (test (*autoload* 'ho) "s7test.scm")
  (set! (*autoload* 'ho) "dsp.scm")
  (test (*autoload* 'ho) "dsp.scm"))

(let ((str3 #f))
  ;; IO tests mainly

  (set! str3 "0123456789")
  (set! str3 (string-append str3 str3 str3 str3 str3 str3 str3 str3 str3 str3))
  (set! str3 (string-append str3 str3 str3 str3 str3 str3 str3 str3 str3 str3))
  (set! str3 (string-append str3 str3 str3))

  (call-with-output-file "test.scm"
    (lambda (p)
      (format p "(define (big-string)~%")
      (format p "  \"")
      (display str3 p)
      (format p "\\\n") ; this becomes \<newline> in the midst of a string which we ignore
      (display str3 p)
      (format p "\"")
      (format p ")~%")))

  (load "test.scm")
  (let ((str (big-string)))
    (test (length str) 6000))

  (let ((big-string (eval (call-with-input-string
			   (call-with-output-string
			    (lambda (p)
			      (format p "(lambda ()~%")
			      (format p "  \"")
			      (display str3 p)
			      (format p "\\\n") ; this becomes \<newline> in the midst of a string which we ignore
			      (display str3 p)
			      (format p "\"")
			      (format p ")~%")))
			   read))))
    (let ((str (big-string)))
      (test (length str) 6000)))

  (call-with-output-file "test.scm"
    (lambda (p)
      (format p "(define (big-string)~%")
      (format p "  \"")
      (display str3 p)
      (format p "\\\"")
      (display str3 p)
      (format p "\"")
      (format p ")~%")))

  (load "test.scm")
  (let ((str (big-string)))
    (test (length str) 6001))

  (let ((big-string (eval (call-with-input-string
			   (call-with-output-string
			    (lambda (p)
			      (format p "(lambda ()~%")
			      (format p "  \"")
			      (display str3 p)
			      (format p "\\\"")
			      (display str3 p)
			      (format p "\"")
			      (format p ")~%")))
			   read))))
    (let ((str (big-string)))
      (test (length str) 6001)))

  (call-with-output-file "test.scm"
    (lambda (p)
      (format p "")))
  (load "test.scm") ; #<unspecified>

  (let ()
    (define (write-stuff p)
      (format p ";")
      (do ((i 0 (+ i 1)))
	  ((= i 3000))
	(let ((c (integer->char (random 128))))
	  (if (char<? c #\space)
	      (display #\space p)
	      (display c p))))
      (format p "~%32~%"))
    (call-with-output-file "test.scm" write-stuff)
    (test (load "test.scm") 32))

  (let ()
    (define (write-stuff p)
      (format p "(define (big-list)~%  (list ")
      (do ((i 0 (+ i 1)))
	  ((= i 2000))
	(format p "~D " i))
      (format p "))~%"))
    (call-with-output-file "test.scm" write-stuff)
    (load "test.scm")
    (let ((lst (big-list)))
      (test (length lst) 2000)))

  (let ()
    (define (write-stuff p)
      (format p "(define (big-list)~%  ")
      (do ((i 0 (+ i 1)))
	  ((= i 2000))
	(format p "(cons ~D " i))
      (format p "()")
      (do ((i 0 (+ i 1)))
	  ((= i 2000))
	(format p ")"))
      (format p ")~%"))
  (call-with-output-file "test.scm" write-stuff)
  (load "test.scm")
  (let ((lst (big-list)))
    (test (length lst) 2000)))

  (call-with-output-file "test.scm"
    (lambda (p)
      (format p "(define (a-char)~%  #\\a)~%")))

  (load "test.scm")
  (test (a-char) #\a)

  (call-with-output-file "test.scm"
    (lambda (p)
      (let ((a (char->integer #\a)))
	(format p "(define (big-char)~%  (string ")
	(do ((i 0 (+ i 1)))
	    ((= i 2000))
	  (format p "#\\~C " (integer->char (+ a (modulo i 26)))))
	(format p "))~%"))))

  (load "test.scm")
  (let ((chars (big-char)))
    (test (length chars) 2000))

  (call-with-output-file "test.scm"
    (lambda (p)
      (let ((a (char->integer #\a)))
	(format p "(define (big-xchar)~%  (string ")
	(do ((i 0 (+ i 1)))
	    ((= i 2000))
	  (format p "#\\x~X " (+ a (modulo i 26))))
	(format p "))~%"))))

  (load "test.scm")
  (let ((chars (big-xchar)))
    (test (length chars) 2000))

  (call-with-output-file "test.scm"
    (lambda (p)
      (format p "(define (ychar) #\\~C)" (integer->char 255))))
  (load "test.scm")
  (test (ychar) (integer->char 255))

  (let ()
    (define (write-stuff p)
      (do ((i 0 (+ i 1)))
	  ((= i 1000))
	(format p "~D" i))
      (format p "~%")
      (do ((i 0 (+ i 1)))
	  ((= i 1000))
	(format p "~D" i)))
    (call-with-output-file "test.scm" write-stuff))

  (call-with-input-file "test.scm"
    (lambda (p)
      (let ((s1 (read-line p))
	    (s2 (read-line p)))
	(test (and (string=? s1 s2)
		   (= (length s1) 2890))
	      #t))))

  (call-with-output-file "test.scm"
    (lambda (p)
      (format p "(define (big-int)~%")
      (do ((i 0 (+ i 1)))
	  ((= i 3000))
	(format p "0"))
      (format p "123)~%")))

  (load "test.scm")
  (test (big-int) 123)

  (call-with-output-file "test.scm"
    (lambda (p)
      (format p "(define (big-rat)~%")
      (do ((i 0 (+ i 1)))
	  ((= i 3000))
	(format p "0"))
      (format p "123/")
      (do ((i 0 (+ i 1)))
	  ((= i 3000))
	(format p "0"))
      (format p "2)~%")))

  (load "test.scm")
  (test (big-rat) 123/2)

  (call-with-output-file "test.scm"
    (lambda (p)
      (format p "(define (big-hash)~%  (hash-table ")
      (do ((i 0 (+ i 1)))
	  ((= i 2000))
	(format p "~D ~D " i (+ i 1)))
      (format p "))~%")))

  (load "test.scm")
  (let ((ht (big-hash)))
    (let ((entries 0))
      (for-each
       (lambda (htv)
	 (set! entries (+ entries 1))
	 (if (not (= (+ (car htv) 1) (cdr htv)))
	     (format *stderr* ";hashed: ~A~%" htv)))
       ht)
      (test entries 2000)))

  (call-with-output-file "test.scm"
    (lambda (p)
      (format p "(define (big-hash)~%  (apply hash-table (list ")
      (do ((i 0 (+ i 1)))
	  ((= i 2000))
	(format p "~D ~D " i (+ i 1)))
      (format p ")))~%")))

  (load "test.scm")
  (let ((ht (big-hash)))
    (let ((entries 0))
      (for-each
       (lambda (htv)
	 (set! entries (+ entries 1))
	 (if (not (= (+ (car htv) 1) (cdr htv)))
	     (format *stderr* ";hashed: ~A~%" htv)))
       ht)
      (test entries 2000)))

  (call-with-output-file "test.scm"
    (lambda (p)
      (let ((a (char->integer #\a)))
	(format p "(define (big-env)~%  (inlet ")
	(do ((i 0 (+ i 1)))
	    ((= i 2000))
	  (format p "'(~A . ~D) "
		  (string (integer->char (+ a (modulo i 26)))
			  (integer->char (+ a (modulo (floor (/ i 26)) 26)))
			  (integer->char (+ a (modulo (floor (/ i (* 26 26))) 26))))
		  i))
	(format p "))~%"))))

  (load "test.scm")
  (let ((E (big-env))
	(a (char->integer #\a)))
    (do ((i 0 (+ i 1)))
	((= i 2000))
      (let ((sym (string->symbol
		  (string (integer->char (+ a (modulo i 26)))
			  (integer->char (+ a (modulo (floor (/ i 26)) 26)))
			  (integer->char (+ a (modulo (floor (/ i (* 26 26))) 26)))))))
	(let ((val (E sym)))
	  (if (not (equal? val i))
	      (format *stderr* ";env: ~A -> ~A, not ~D~%" sym val i))))))

  (call-with-output-file "test.scm"
    (lambda (p)
      (format p "")))

  (let ((val (call-with-input-file "test.scm"
	       (lambda (p)
		 (read p)))))
    (if (not (eof-object? val))
	(format *stderr* ";read empty file: ~A~%" val)))

  (call-with-output-file "test.scm"
    (lambda (p)
      (format p " ;")
      (do ((i 0 (+ i 1)))
	  ((= i 3000))
	(let ((c (integer->char (random 128))))
	  (if (char<? c #\space)
	      (display #\space p)
	      (display c p))))
      (format p "~%")))

  (let ((val (call-with-input-file "test.scm"
	       (lambda (p)
		 (read p)))))
    (if (not (eof-object? val))
	(format *stderr* ";read comment file: ~A~%" val)))

  (call-with-output-file "test.scm"
    (lambda (p)
      (format p "\"~3001TT\"~%")))

  (let ((str (call-with-input-file "test.scm" read)))
    (test (length str) 3000)))

(if (defined? 'big-list)
    (set! big-list #f))
#|
(let ((c #f)
      (i 0)
      (e #f))
  (set! e (curlet))
  (call-with-output-file "empty-file" (lambda (p) (write '(call/cc (lambda (c1) (set! c c1) (set! i (+ i 1)))) p)))
  (load "empty-file" e)
  (test (c) 'error)) ; ;read-error ("our input port got clobbered!")
|#

(let ((stdin-wrapper (open-input-function
		       (lambda (choice)
			 (case choice
			   ((read peek-char) #\?)
			   ((char-ready?) #f)
			   ((read-char) #\a)
			   ((read-line) "a line"))))))
  (test (port-filename stdin-wrapper) "")
  (test (port-line-number stdin-wrapper) 0)
  (let-temporarily (((current-input-port) stdin-wrapper))
    (test (read-char) #\a)
    ;(test (read-byte) 65)
    (test (read-line) "a line")
    (test (char-ready?) #f)
    (test (read) #\?)
    (test (peek-char) #\?)
    (test (object->string (current-input-port)) "#<input-function-port>")
    (test (object->string
            (object->let (current-input-port)))
          "(inlet 'function #<lambda (choice)> 'value #<input-function-port> 'type input-port? 'port-type function 'closed #f 'mutable? #t)")
    (test (input-port? (current-input-port)) #t)
    (test (reverse (current-input-port)) 'error)
    (test (fill! (current-input-port)) 'error)
    (test (length (current-input-port)) #f))
  (test (port-closed? stdin-wrapper) #f)
  (close-input-port stdin-wrapper)
  (test (read-char stdin-wrapper) 'error)
  (test (port-closed? stdin-wrapper) #t))

(let ((str ())
      (obj #f))
  (let ((stdout-wrapper (open-output-function
			 (lambda (c)
			   (set! str (cons c str))))))
    (let-temporarily (((current-output-port) stdout-wrapper))
      (write-char #\a)
      (display #\b)
      (write-byte 123)
      (write-string "a line")
      (newline)
      (test (object->string (current-output-port)) "#<output-function-port>")
      (set! obj (object->let (current-output-port)))
      (test (output-port? (current-output-port)) #t))
    (test (reverse str) '(97 98 123 97 32 108 105 110 101 10))
    (close-output-port stdout-wrapper)
    (test (write "a test" stdout-wrapper) 'error)
    (test (object->string obj)
          "(inlet 'function #<lambda (c)> 'value #<output-function-port:closed> 'type output-port? 'port-type function 'closed #f 'mutable? #t)")))

(test (read-char (open-input-function vector)) 'error)
;; (write-char #\a (open-output-function list)) -> a?? but string: error: string argument 1, 97, is an integer but should be a character
(test (peek-char (open-input-function cons)) 'error)
(test (write-char #\a (open-output-function cons)) 'error)
(test (open-input-function (vector)) 'error)
(test (open-output-function (vector)) 'error)

(unless pure-s7
  (test ((lambda (w) 1) (char-ready? (open-input-function (lambda (x) (values 1 2 3 4 5 6 7))))) 'error))
(test ((lambda (w) 1) (read-char (open-input-function (lambda (x) (values 1 2 3 4 5 6 7))))) 'error)
(test ((lambda (w) 1) (read-line (open-input-function (lambda (x) (values 1 2 3 4 5 6 7))))) 'error)
(test ((lambda (w) 1) (read (open-input-function (lambda (x) (values 1 2 3 4 5 6 7))))) 'error)

(test (reverse *stdin*) 'error)
(test (fill! (current-output-port)) 'error)
(test (length *stderr*) #f)

(test (output-port? (current-input-port)) #f)
(test (output-port? *stdin*) #f)
(test (output-port? (current-output-port)) #t)
(test (output-port? *stdout*) #t)
(test (output-port? (current-error-port)) #t)
(test (output-port? *stderr*) #t)

;(write-char #\space (current-output-port))
;(write " " (current-output-port))
(newline (current-output-port))
(for-each
 (lambda (p)
   (test (write-char #\a p) 'error)
   (test (write-byte 0 p) 'error)
   (test (write-string "a" p) 'error)
   (test (write "a" p) 'error)
   (test (display "a" p) 'error)
   (test (format p "~a" 2) 'error))
 (list *stdin* (current-input-port)
       (call-with-output-string (lambda (p) p))))

(for-each
 (lambda (p)
   (test (read-char p) 'error)
   (test (read-byte p) 'error)
   (test (read-line p) 'error)
   (test (read-string 5 p) 'error)
   (test (read p) 'error))
 (list *stdout* #f (current-output-port)
       (call-with-input-file "s7test.scm" (lambda (p) p))
       (call-with-input-string "0123456" (lambda (p) p))))

(test (write #\1 *stdin*) 'error)
(test (write-char #\1 *stdin*) 'error)
(test (write-byte 1 *stdin*) 'error)
(test (write-string "123" *stdin*) 'error)
(test (display #\1 *stdin*) 'error)

(for-each
 (lambda (p)
   (test (write-char #\a p) 'error)
   (test (write-byte 0 p) 'error)
   (test (write-string "a" p) 'error)
   (test (write "a" p) 'error)
   (test (display "a" p) 'error)
   (test (format p "~a" 2) 'error)
   (test (read-char p) 'error)
   (test (read-byte p) 'error)
   (test (read-line p) 'error)
   (test (read-string p) 'error)
   (test (read p) 'error))
 (list (let ((p (open-output-string))) (close-output-port p) p)
       (let ((p (open-input-string "123"))) (close-input-port p) p)))

(for-each
 (lambda (arg)
   (if (output-port? arg)
       (format #t ";(output-port? ~A) -> #t?~%" arg)))
 (list "hi" #f () 'hi (integer->char 65) 1 (list 1 2) _ht_ _undef_ _null_ _c_obj_ '#t '3 (make-vector 3) 3.14 3/4 1.0+1.0i #\f))

(for-each
 (lambda (arg)
   (test (read-line () arg) 'error)
   (test (read-line arg) 'error))
 (list "hi" (integer->char 65) 1 #f _ht_ _undef_ _null_ _c_obj_ (list) (cons 1 2) (list 1 2) (make-vector 3) 3.14 3/4 1.0+1.0i #\f))

(test (call-with-output-file tmp-output-file output-port?) #t)
(if (not (eq? start-output-port (current-output-port)))
    (format #t "call-with-output-file did not restore current-output-port? ~A from ~A~%" start-output-port (current-output-port)))

(test (let ((this-file (open-output-file tmp-output-file))) (let ((res (output-port? this-file))) (close-output-port this-file) res)) #t)
(if (not (eq? start-output-port (current-output-port)))
    (format #t "open-output-file clobbered current-output-port? ~A from ~A~%" start-output-port (current-output-port)))

(test (let ((val #f)) (call-with-output-string (lambda (p) (set! val (output-port? p)))) val) #t)
(test (let ((res #f)) (let ((this-file (open-output-string))) (set! res (output-port? this-file)) (close-output-port this-file) res)) #t)

(test (with-output-to-string (lambda () (display _undef_))) "#_asdf")
(test (with-output-to-string (lambda () (write _undef_))) "#_asdf")
(test (with-output-to-string (lambda () (make-string (+ (*s7* 'max-string-length) 10)))) 'error)

(test (open-input-file "tools") 'error)
(let ()
  (call-with-output-file "tmp.r5rs"
    (lambda (p)
      (format p "(values 2 3 4)")))
  (test (+ 1 (load "tmp.r5rs") 5) 15))

(for-each
 (lambda (arg)
   (if (eof-object? arg)
       (format #t ";(eof-object? ~A) -> #t?~%" arg)))
 (list "hi" () '(1 2) -1 #\a 1 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #f #t (if #f #f) #<undefined> (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (let ((val (catch #t
		     (lambda () (port-closed? arg))
		     (lambda args 'error))))
     (if (not (eq? val 'error))
	 (format #t ";(port-closed? ~A) -> ~S?~%" arg val))))
 (list "hi" '(1 2) -1 #\a 1 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
       3.14 3/4 1.0+1.0i #f #t (if #f #f) #<undefined> #<eof> (lambda (a) (+ a 1))))

(test (port-closed?) 'error)
(test (port-closed? (current-input-port) (current-output-port)) 'error)
(test (let-temporarily (((current-output-port) #f)) (port-closed? (current-output-port))) #f)


;;; port-position
(test (call-with-input-string "0123456789" (lambda (p) (set! (port-position p) 3) (list (read-char p) (port-position p)))) '(#\3 4))
(test (call-with-input-file "s7test.scm" (lambda (p) (set! (port-position p) 88) (list (read-string 10 p) (port-position p)))) '(";;   Paul " 98))
(test (call-with-input-string "0123456789" (lambda (p) (set! (port-position p) -3))) 'error)
(test (port-position (call-with-input-file (string #\c #\null #\b) quasiquote)) 'error) ; closed port
(test (call-with-input-string "" (lambda (p) (set! (port-position p) 2) (port-position p))) 0)
(test (call-with-input-string "123" (lambda (p) (set! (port-position p) 12) (port-position p))) 3)

(let ()
  (define (pos1)
    (let* ((in (open-input-string "#xff"))
           (val (read in))
	   (pos (port-position in)))
      (close-input-port in)
      pos))
  (test (pos1) 4)

  (define (pos2)
    (let* ((in (open-input-string "#xff "))
           (val (read in))
	   (pos (port-position in)))
      (close-input-port in)
      pos))
  (test (pos2) 4)

  (define (pos3)
    (let* ((in (open-input-string "(+ 1 xyz)"))
           (val (read in))
	   (pos (port-position in)))
      (close-input-port in)
      pos))
  (test (pos3) 9)

  (define (pos4)
    (let* ((in (open-input-string "(+ 1 xyz) ; a comment"))
           (val (read in))
	   (pos (port-position in)))
      (close-input-port in)
      pos))
  (test (pos4) 9)

  (define (pos5)
    (let* ((in (open-input-string " xyz"))
           (val (read in))
	   (pos (port-position in)))
      (close-input-port in)
      pos))
  (test (pos5) 4)

  (define (pos6)
    (let* ((in (open-input-string " xyz "))
           (val (read in))
	   (pos (port-position in)))
      (close-input-port in)
      pos))
  (test (pos6) 4)

  (define (pos7)
    (let* ((in (open-input-string (format #f "1234~%5678")))
           (val1 (read-line in))
	   (pos1 (port-position in))
           (val2 (read-line in))
	   (pos2 (port-position in)))
      (close-input-port in)
      (list pos1 pos2)))
  (test (pos7) '(5 9))

  (define (pos8)
    (let* ((in (open-input-string "12345678"))
           (val1 (read-string 4 in))
	   (pos1 (port-position in))
           (val11 (read-string 0 in))
	   (pos11 (port-position in))
           (val2 (read-string 5 in))
	   (pos2 (port-position in))
           (val3 (read-string 1 in))
	   (pos3 (port-position in)))
      (close-input-port in)
      (list pos1 pos11 pos2 pos3)))
  (test (pos8) '(4 4 8 8))

  (define (pos9)
    (let* ((in (open-input-string "123456789"))
	   (vals ()))
      (do ((i 0 (+ i 3))
	   (k 1 (+ k 2)))
	  ((>= i 9))
	(set! vals (cons (read-string 3 in) vals))
	(set! (port-position in) k)
	(test (port-position in) k))
      (close-input-port in)
      (reverse vals)))
  (test (pos9) '("123" "234" "456"))

  (define (pos10)
    (let ((f (open-output-file "hi")))
      (display "1234567879" f)
      (flush-output-port f)
      (close-output-port f)
      (let* ((in (open-input-file "hi"))
             (vals ()))
        (do ((i 0 (+ i 3))
	     (k 1 (+ k 2)))
	    ((>= i 9))
	  (set! vals (cons (read-string 3 in) vals))
	  (set! (port-position in) k)
	  (test (port-position in) k))
        (close-input-port in)
        (reverse vals))))
  (test (pos10) '("123" "234" "456")))

(test (call-with-input-file "s7test.scm" (lambda (p) (c-pointer? (port-file p)))) #t)

(call-with-output-file tmp-output-file (lambda (p) (display "3.14" p)))
(test (call-with-input-file tmp-output-file (lambda (p) (read p) (let ((val (read p))) (eof-object? val)))) #t)

(test (call-with-input-file tmp-output-file (lambda (p) (read-char p))) #\3)
(test (call-with-input-file tmp-output-file (lambda (p) (peek-char p))) #\3)
(test (call-with-input-file tmp-output-file (lambda (p) (peek-char p) (read-char p))) #\3)
(test (call-with-input-file tmp-output-file (lambda (p) (list->string (list (read-char p) (read-char p) (read-char p) (read-char p))))) "3.14")
(test (call-with-input-file tmp-output-file (lambda (p) (list->string (list (read-char p) (peek-char p) (read-char p) (read-char p) (peek-char p) (read-char p))))) "3..144")

(for-each
 (lambda (arg)
   (call-with-output-file tmp-output-file (lambda (p) (write arg p)))
   (if (not (equivalent? (call-with-input-file tmp-output-file (lambda (p) (read p))) arg))
       (format *stderr* "~A different after write~%" arg)))
 (list "hi" -1 #\a 1 'a-symbol (make-vector 3 0) 3.14 3/4 .6 1.0+1.0i #f #t (list 1 2 3) (cons 1 2)
       '(1 2 . 3) () '((1 2) (3 . 4)) '(()) (list (list 'a "hi") #\b 3/4) ''a
       (string #\a #\null #\b) "" "\"hi\""
       (integer->char 128) (integer->char 127) (integer->char 255) #\space #\null #\newline #\tab
       #() #2d((1 2) (3 4)) #3d()
       :hi #<eof> #<undefined> #<unspecified>
       most-negative-fixnum
       (if with-bignums 1239223372036854775808 123)
       (if with-bignums 144580536300674537151081081515762353325831/229154728370723013560448485454219755525522 11/10)
       (if with-bignums 221529797579218180403518826416685087012.0 1000.1)
       (if with-bignums 1239223372036854775808+1239223372036854775808i 1000.1-1234i)))

(for-each
 (lambda (arg)
   (call-with-output-file tmp-output-file (lambda (p) (write arg p)))
   (test (call-with-input-file tmp-output-file (lambda (p) (eval (read p)))) arg)) ; so read -> symbol?
 (list *stdout* *stdin* *stderr*
       abs + quasiquote

;       (hash-table 'a 1 'b 2) (hash-table)
;       0/0 (real-part (log 0))
;;; for these we need nan? and infinite? since equal? might be #f
;       (lambda (a) (+ a 1))
; pws?
;       (current-output-port)
;       (random-state 1234)
;       (symbol ":\"")
; (let () (define-macro (hi1 a) `(+ ,a 1)) hi1)
;;; and how could a continuation work in general?
       ))

;;; (call-with-input-file tmp-output-file (lambda (p) (read p))) got (symbol ":\"") but expected (symbol ":\"")


;;; r4rstest
(let* ((write-test-obj '(#t #f a () 9739 -3 . #((test) "te \" \" st" "" test #() b c)))
       (load-test-obj (list 'define 'foo (list 'quote write-test-obj))))

  (define (check-test-file name)
    (let ((val (call-with-input-file
		   name
		 (lambda (test-file)
		   (test (read test-file) load-test-obj)
		   (test (eof-object? (peek-char test-file)) #t)
		   (test (eof-object? (read-char test-file)) #t)
		   (input-port? test-file)))))
      (if (not (eq? val #t))
	  (format #t "input-port? in call-with-input-file? returned ~A from ~A~%" val name))))

  (test (call-with-output-file
	    tmp-output-file
	  (lambda (test-file)
	    (write-char #\; test-file)
	    (display #\; test-file)
	    (display ";" test-file)
	    (write write-test-obj test-file)
	    (newline test-file)
	    (write load-test-obj test-file)
	    (output-port? test-file))) #t)
  (check-test-file tmp-output-file)

  (let ((test-file (open-output-file "tmp2.r5rs")))
    (test (port-closed? test-file) #f)
    (write-char #\; test-file)
    (display #\; test-file)
    (display ";" test-file)
    (write write-test-obj test-file)
    (newline test-file)
    (write load-test-obj test-file)
    (test (output-port? test-file) #t)
    (close-output-port test-file)
    (check-test-file "tmp2.r5rs")))

(test (with-input-from-string "" read) #<eof>) ; ?

(call-with-output-file tmp-output-file (lambda (p) (display "3.14" p)))
(test (with-input-from-file tmp-output-file read) 3.14)
(if (not (eq? start-input-port (current-input-port)))
    (format #t "with-input-from-file did not restore current-input-port? ~A from ~A~%" start-input-port (current-input-port)))

(test (with-input-from-file tmp-output-file (lambda () (eq? (current-input-port) start-input-port))) #f)
(test (char->integer ((with-input-from-string (string (integer->char 255))(lambda () (read-string 1))) 0)) 255)

(test (with-output-to-file tmp-output-file (lambda () (eq? (current-output-port) start-output-port))) #f)
(if (not (eq? start-output-port (current-output-port)))
    (format #t "with-output-to-file did not restore current-output-port? ~A from ~A~%" start-output-port (current-output-port)))


(let ((newly-found-sonnet-probably-by-shakespeare
       "This is the story, a sad tale but true \
        Of a programmer who had far too little to do.\
        One day as he sat in his hut swilling stew, \
        He cried \"CLM takes forever, it's stuck in a slough!,\
        Its C code is slow, too slow by a few.\
        Why, with just a small effort, say one line or two,\
        It could outpace a no-op, you could scarcely say 'boo'\"!\
        So he sat in his kitchen and worked like a dog.\
        He typed and he typed 'til his mind was a fog. \
        Now 6000 lines later, what wonders we see!  \
        CLM is much faster, and faster still it will be!\
        In fact, for most cases, C beats the DSP!  \
        But bummed is our coder; he grumbles at night.  \
        That DSP code took him a year to write.  \
        He was paid many dollars, and spent them with glee,\
        But his employer might mutter, this result were he to see."))

  (call-with-output-file tmp-output-file
    (lambda (p)
      (write newly-found-sonnet-probably-by-shakespeare p)))

  (let ((sonnet (with-input-from-file tmp-output-file
		  (lambda ()
		    (read)))))
    (if (or (not (string? sonnet))
	    (not (string=? sonnet newly-found-sonnet-probably-by-shakespeare)))
	(format #t "write/read long string returned: ~A~%" sonnet)))

  (let ((file (open-output-file tmp-output-file)))
    (let ((len (string-length newly-found-sonnet-probably-by-shakespeare)))
      (write-char #\" file)
      (do ((i 0 (+ i 1)))
	  ((= i len))
	(let ((chr (string-ref newly-found-sonnet-probably-by-shakespeare i)))
	  (if (char=? chr #\")
	      (write-char #\\ file))
	  (write-char chr file)))
      (write-char #\" file)
      (close-output-port file)))

  (let ((file (open-input-file tmp-output-file)))
    (let ((sonnet (read file)))
      (close-input-port file)
      (if (or (not (string? sonnet))
	      (not (string=? sonnet newly-found-sonnet-probably-by-shakespeare)))
	  (format #t "write-char/read long string returned: ~A~%" sonnet)))))

(test (let () (define (func) (vector (call-with-output-file "/dev/null" quasiquote))) (define (hi) (func)) (vector? (hi))) #t)

(let ((file (open-output-file tmp-output-file)))
  (for-each
   (lambda (arg)
     (write arg file)
     (write-char #\space file))
   (list "hi" -1 #\a 1 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #f #t (list 1 2 3) '(1 . 2)))
  (close-output-port file))

(let ((file (open-input-file tmp-output-file)))
  (for-each
   (lambda (arg)
     (let ((val (read file)))
       (if (not (equal? val arg))
	   (format #t "read/write ~A returned ~A~%" arg val))))
   (list "hi" -1 #\a 1 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #f #t (list 1 2 3) '(1 . 2)))
  (close-input-port file))

(with-output-to-file tmp-output-file
  (lambda ()
    (write lists)))

(let ((val (with-input-from-file tmp-output-file
	     (lambda ()
	       (read)))))
  (if (not (equal? val lists))
      (format #t "read/write lists returned ~A~%" val)))

(if (not (string=? "" (with-output-to-string (lambda () (display "")))))
    (format #t "with-output-to-string null string?"))

(let ((str (with-output-to-string
	     (lambda ()
	       (with-input-from-string "hiho123"
		 (lambda ()
		   (do ((c (read-char) (read-char)))
		       ((eof-object? c))
		     (display c))))))))
  (if (not (string=? str "hiho123"))
      (format #t "with string ports 0: ~S?~%" str)))

(let ((str1 (let ((x #f)) (with-output-to-string (lambda () (display `(+ x 1)) (display (object->string '(1 2 3) #f 1)))))))
  (let ((str2 (let ((x #f)) (with-output-to-string (lambda () (display `(+ x 1)) (display (object->string '(1 2 3) #f 1)))))))
    (test str1 str2))) ; both "(+ x 1)..."

(let ((p1 (open-input-string "123"))
      (p2 (open-input-string "123")))
  (test (equivalent? p1 p2) #t)
  (read-char p1)
  (test (equivalent? p1 p2) #f)
  (read-char p2)
  (test (equivalent? p1 p2) #t)
  (close-input-port p1)
  (close-input-port p2))

(let ((p1 (open-input-string "1234"))
      (p2 (open-input-string "123")))
  (test (equivalent? p1 p2) #f)
  (read-char p1)
  (test (equivalent? p1 p2) #f)
  (close-input-port p1)
  (close-input-port p2))

(let ((p1 (open-output-string))
      (p2 (open-output-string)))
  (test (equivalent? p1 p2) #t)
  (write-char #\a p1)
  (test (equivalent? p1 p2) #f)
  (close-output-port p1)
  (close-output-port p2))

(let ()
  (define* (f1 (b 123)) (display b))
  (test (with-output-to-string f1) "123")
  (define (f2) (display "123"))
  (test (with-output-to-string f2) "123")
  (define (f3 . args) (display 123))
  (test (with-output-to-string f3) "123")
  (define-macro (m1) `(write 123))
  (test (with-output-to-string m1) "123")
  (define-macro (m2) (write 123))
  (test (with-output-to-string m2) "123")
  (define (f4 a b) (display 123))
  (test (with-output-to-string f4) 'error)
  (test (with-output-to-string (lambda () (*s7* 'version))) "")) ; the output is a string -- not written to stdout or whatever

(let ()
  (define* (f1 a (b 123)) (display b a))
  (test (call-with-output-string f1) "123")
  (define (f2 a) (display "123" a))
  (test (call-with-output-string f2) "123")
  (define (f3 . args) (display 123 (car args)))
  (test (call-with-output-string f3) "123")
  (define-macro (m1 p) `(write 123 ,p))
  (test (call-with-output-string m1) "123")
  (define-macro* (m2 (p #f)) (write 123 p))
  (test (call-with-output-string m2) "123")
  (define (f4 a b) (display 123 a))
  (test (call-with-output-string f4) 'error)
  (test (call-with-output-string (lambda () (*s7* 'version))) 'error))

(let ()
  (define* (f1 (a #f)) (read))
  (test (with-input-from-string "(+ 1 2 3)" f1) '(+ 1 2 3))
  (define* (f2 . args) (read))
  (test (with-input-from-string "(+ 1 2 3)" f2) '(+ 1 2 3))
  (define f3 read)
  (test (with-input-from-string "(+ 1 2 3)" f3) '(+ 1 2 3))
  (define (f4) (read))
  (test (with-input-from-string "(+ 1 2 3)" f4) '(+ 1 2 3))
  (define-macro (m1) `(read))
  (test (with-input-from-string "(+ 1 2 3)" m1) '(+ 1 2 3))
  (define-macro (m2) (read))
  (test (with-input-from-string "(+ 1 2 3)" m2) 6)
  (define (f5 a) (read a))
  (test (with-input-from-string "(+ 1 2 3)" f5) 'error)
  (test (with-input-from-string "(+ 1 2 3)" (lambda () (*s7* 'version))) (*s7* 'version)))

(let ()
  (define* (f1 (a #f)) (read a))
  (test (call-with-input-string "(+ 1 2 3)" f1) '(+ 1 2 3))
  (define* (f2 . args) (read (car args)))
  (test (call-with-input-string "(+ 1 2 3)" f2) '(+ 1 2 3))
  (define f3 read)
  (test (call-with-input-string "(+ 1 2 3)" f3) '(+ 1 2 3))
  (define-macro (m1 p) `(read ,p))
  (test (call-with-input-string "(+ 1 2 3)" m1) '(+ 1 2 3))
  (define-macro* (m2 (p #f)) (read p))
  (test (call-with-input-string "(+ 1 2 3)" m2) 6)
  (define (f4) (read))
  (test (call-with-input-string "(+ 1 2 3)" f4) 'error)
  (test (call-with-input-string "(+ 1 2 3)" (lambda () (*s7* 'version))) 'error))


(let ()
  (with-output-to-file tmp-output-file
    (lambda ()
      (display "(+ 1 2 3)")))
  (define* (f1 (a #f)) (read))
  (test (with-input-from-file tmp-output-file f1) '(+ 1 2 3))
  (define* (f2 . args) (read))
  (test (with-input-from-file tmp-output-file f2) '(+ 1 2 3))
  (define f3 read)
  (test (with-input-from-file tmp-output-file f3) '(+ 1 2 3))
  (define (f4) (read))
  (test (with-input-from-file tmp-output-file f4) '(+ 1 2 3))
  (define-macro (m1) `(read))
  (test (with-input-from-file tmp-output-file m1) '(+ 1 2 3))
  (define-macro (m2) (read))
  (test (with-input-from-file tmp-output-file m2) 6)
  (define (f5 a) (read a))
  (test (with-input-from-file tmp-output-file f5) 'error)
  (test (with-input-from-file tmp-output-file (lambda () (*s7* 'version))) (*s7* 'version)))

(let ()
  (define (eval-from-string-1 str)
    (define-macro (m) (read))
    (with-input-from-string str m))
  (test (eval-from-string-1 "(+ 1 2 3)") 6)
  (define (eval-from-string str)
    (with-input-from-string str (define-macro (m) (read))))
  (test (eval-from-string "(+ 1 2 3)") 6))

(let ()
  (define* (f1 (a #f)) (read a))
  (test (call-with-input-file tmp-output-file f1) '(+ 1 2 3))
  (define* (f2 . args) (read (car args)))
  (test (call-with-input-file tmp-output-file f2) '(+ 1 2 3))
  (define f3 read)
  (test (call-with-input-file tmp-output-file f3) '(+ 1 2 3))
  (define-macro (m1 p) `(read ,p))
  (test (call-with-input-file tmp-output-file m1) '(+ 1 2 3))
  (define-macro* (m2 (p #f)) (read p))
  (test (call-with-input-file tmp-output-file m2) 6)
  (define (f4) (read))
  (test (call-with-input-file tmp-output-file f4) 'error)
  (test (call-with-input-file tmp-output-file (lambda () (*s7* 'version))) 'error))

(let ((ofile tmp-output-file))
  (define (get-file-contents)
    (with-input-from-file ofile read-line))

  (define* (f1 (b 123)) (display b))
  (test (let () (with-output-to-file ofile f1) (get-file-contents)) "123")
  (define (f2) (display "123"))
  (test (let () (with-output-to-file ofile f2) (get-file-contents)) "123")
  (define (f3 . args) (display 123))
  (test (let () (with-output-to-file ofile f3) (get-file-contents)) "123")
  (define-macro (m1) `(write 123))
  (test (let () (with-output-to-file ofile m1) (get-file-contents)) "123")
  (define-macro (m2) (write 123))
  (test (let () (with-output-to-file ofile m2) (get-file-contents)) "123")
  (define (f4 a b) (display 123))
  (test (let () (with-output-to-file ofile f4) (get-file-contents)) 'error)
  (test (let () (with-output-to-file ofile (lambda () (*s7* 'version))) (get-file-contents)) #<eof>)

  (define* (f11 a (b 123)) (display b a))
  (test (let () (call-with-output-file ofile f11) (get-file-contents)) "123")
  (define (f21 a) (display "123" a))
  (test (let () (call-with-output-file ofile f21) (get-file-contents)) "123")
  (define (f31 . args) (display 123 (car args)))
  (test (let () (call-with-output-file ofile f31) (get-file-contents)) "123")
  (define-macro (m3 p) `(write 123 ,p))
  (test (let () (call-with-output-file ofile m3) (get-file-contents)) "123")
  (define-bacro* (m2 (p 123)) `(write 123 ,p))
  (test (let () (call-with-output-file ofile m2) (get-file-contents)) "123")
  (define (f41 a b) (display 123 a))
  (test (let () (call-with-output-file ofile f41) (get-file-contents)) 'error)
  (test (let () (call-with-output-file ofile (lambda () (*s7* 'version))) (get-file-contents)) 'error))

(if (not (eof-object? (with-input-from-string "" (lambda () (read-char)))))
    (format #t ";input from null string not #<eof>?~%")
    (let ((EOF (with-input-from-string "" (lambda () (read-char)))))
      (if (not (eq? (with-input-from-string "" (lambda () (read-char)))
		    (with-input-from-string "" (lambda () (read-char)))))
	  (format #t "#<eof> is not eq? to itself?~%"))
      (if (char? EOF)
	  (do ((c 0 (+ c 1)))
	      ((= c 256))
	    (if (char=? EOF (integer->char c))
		(format #t "#<eof> is char=? to ~C~%" (integer->char c)))))))

(test (+ 100 (call-with-output-file "tmp.r5rs" (lambda (p) (write "1" p) (values 1 2)))) 103)
(test (+ 100 (with-output-to-file "tmp.r5rs" (lambda () (write "2") (values 1 2)))) 103)

(if (not pure-s7)
    (let ((str (with-output-to-string
		 (lambda ()
		   (with-input-from-string "hiho123"
		     (lambda ()
		       (do ((c (read-char) (read-char)))
			   ((or (not (char-ready?))
				(eof-object? c)))
			 (display c))))))))
      (if (not (string=? str "hiho123"))
	  (format #t "with string ports 1: ~S?~%" str))))

(let ((str (with-output-to-string
	     (lambda ()
	       (with-input-from-string ""
		 (lambda ()
		   (do ((c (read-char) (read-char)))
		       ((eof-object? c))
		     (display c))))))))
  (if (not (string=? str ""))
      (format #t "with string ports and null string: ~S?~%" str)))

(let ((str (with-output-to-string ; this is from the guile-user mailing list, I think -- don't know who wrote it
	     (lambda ()
	       (with-input-from-string "A2B5E3426FG0ZYW3210PQ89R."
		 (lambda ()
		   (call/cc
		    (lambda (hlt)
		      (define (nextchar)
			(let ((c (read-char)))
			  (if (eq? c #\space)
			      (nextchar)
			      c)))

		      (define inx
			(lambda()
			  (let in1 ()
			    (let ((c (nextchar)))
			      (if (char-numeric? c)
				  (let ((r (nextchar)))
				    (let out*n ((n (- (char->integer c) (char->integer #\0))))
				      (out r)
				      (if (not (zero? n))
					  (out*n (- n 1)))))
				  (out c))
			      (in1)))))

		      (define (move-char c)
			(write-char c)
			(if (char=? c #\.)
			    (begin (hlt))))

		      (define outx
			(lambda()
			  (let out1 ()
			    (let h1 ((n 16))
			      (move-char (in))
			      (move-char (in))
			      (move-char (in))
			      (if (= n 1)
				  (begin (out1))
				  (begin (write-char #\space) (h1 (- n 1))) )))))

		      (define (in)
			(call/cc (lambda(return)
				   (set! outx return)
				   (inx))))

		      (define (out c)
			(call/cc (lambda(return)
				   (set! inx return)
				   (outx c))))
		      (outx)))))))))
  (if (not (string=? str "ABB BEE EEE E44 446 66F GZY W22 220 0PQ 999 999 999 R."))
      (format #t "call/cc with-input-from-string str: ~A~%" str)))

(let ((badfile tmp-output-file))
  (let ((p (open-output-file badfile)))
    (close-output-port p))
  (load badfile))

(test (let ((str1 (let ((port #f))
		    (dynamic-wind
			(lambda ()
			  (set! port (open-input-string (format #f "~S" (call-with-input-string "" and)))))
			(lambda ()
			  (read port))
			(lambda ()
			  (close-input-port port)))))
	    (str2 (with-input-from-string
		      (object->string (call-with-input-string "" and))
		    read)))
	(equivalent? str1 str2))
      #t)

(for-each
 (lambda (str)
   ;;(test (eval-string str) 'error)
   ;; eval-string is confused somehow
   (test (with-input-from-string str read) 'error))
 (list "\"\\x" "\"\\x0" "`(+ ," "`(+ ,@" "#2d("))

(let ((loadit tmp-output-file))
  (let ((p1 (open-output-file loadit)))
    (display "(define s7test-var 314) (define (s7test-func) 314) (define-macro (s7test-mac a) `(+ ,a 2))" p1)
    (newline p1)
    (close-output-port p1)
    (load loadit)
    (test (= s7test-var 314) #t)
    (test (s7test-func) 314)
    (test (s7test-mac 1) 3)
    (let ((p2 (open-output-file loadit))) ; hopefully this starts a new file
      (display "(define s7test-var 3) (define (s7test-func) 3) (define-macro (s7test-mac a) `(+ ,a 1))" p2)
      (newline p2)
      (close-output-port p2)
      (load loadit)
      (test (= s7test-var 3) #t)
      (test (s7test-func) 3)
      (test (s7test-mac 1) 2)
      (test (equivalent? p1 p2) #t)))) ; undefined constants

(test (+ 100 (with-input-from-string "123" (lambda () (values (read) 1)))) 224)

(for-each
 (lambda (op)
   (for-each
    (lambda (arg) ;(format #t ";(~A ~A)~%" op arg)
      (test (op arg) 'error))
    (list (integer->char 65) 1 0 -1 (list 1) (cons 1 2) 'a-symbol (make-vector 3) abs lambda with-let
	  _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
	  3.14 3/4 1.0+1.0i #\f #t :hi (if #f #f) (lambda (a) (+ a 1)))))
 (list char-ready? set-current-output-port set-current-input-port set-current-error-port
       close-input-port close-output-port open-input-file open-output-file
       read-char peek-char read
       (lambda (arg) (write-char #\a arg))
       (lambda (arg) (write "hi" arg))
       (lambda (arg) (display "hi" arg))
       call-with-input-file with-input-from-file call-with-output-file with-output-to-file))

(unless pure-s7 (test (char-ready? (open-input-string "")) #t)) ; humph
(with-output-to-file tmp-output-file
  (lambda ()
    (display "this is a test")
    (newline)))

(test (call-with-input-file tmp-output-file (lambda (p) (integer->char (read-byte p)))) #\t)
(test (with-input-from-string "123" (lambda () (read-byte))) 49)
;(test (with-input-from-string "1/0" read) 'error) ; this is a reader error in CL
;;; this test causes trouble when s7test is called from snd-test -- I can't see why

(let ((bytes (vector #o000 #o000 #o000 #o034 #o000 #o001 #o215 #o030 #o000 #o000 #o000 #o022 #o000
		     #o000 #o126 #o042 #o000 #o000 #o000 #o001 #o000 #o000 #o000 #o000 #o000 #o001)))
  (with-output-to-file tmp-output-file
    (lambda ()
      (for-each
       (lambda (b)
	 (write-byte b))
       bytes)))

  (let ((ctr 0))
    (call-with-input-file tmp-output-file
      (lambda (p)
	(if (not (string=? (port-filename p) tmp-output-file)) (display (port-filename p)))
	(let loop ((val (read-byte p)))
	  (if (eof-object? val)
	      (if (not (= ctr 26))
		  (format #t "read-byte done at ~A~%" ctr))
	      (begin
		(if (not (= (bytes ctr) val))
		    (format #t "read-byte bytes[~D]: ~A ~A~%" ctr (bytes ctr) val))
		(set! ctr (+ 1 ctr))
		(loop (read-byte p))))))))

  (let ((ctr 0))
    (call-with-input-file tmp-output-file
      (lambda (p)
	(let loop ((val (read-char p)))
	  (if (eof-object? val)
	      (if (not (= ctr 26))
		  (format #t "read-char done at ~A~%" ctr))
	      (begin
		(if (not (= (bytes ctr) (char->integer val)))
		    (format #t "read-char bytes[~D]: ~A ~A~%" ctr (bytes ctr) (char->integer val)))
		(set! ctr (+ 1 ctr))
		(loop (read-char p))))))))
  )

(with-output-to-file tmp-output-file
  (lambda ()
    (if (not (string=? (port-filename (current-output-port)) tmp-output-file)) (display (port-filename (current-output-port))))
    (display "(+ 1 2) 32")
    (newline)
    (display "#\\a  -1")))

(with-input-from-file tmp-output-file
  (lambda ()
    (if (not (string=? (port-filename (current-input-port)) tmp-output-file)) (display (port-filename (current-input-port))))
    (let ((val (read)))
      (if (not (equal? val (list '+ 1 2)))
	  (format #t ";file read +: ~A~%" val)))
    (let ((val (read)))
      (if (not (equal? val 32))
	  (format #t "file read 32: ~A~%" val)))
    (let ((val (read)))
      (if (not (equal? val #\a))
	  (format #t "file read a: ~A~%" val)))
    (let ((val (read)))
      (if (not (equal? val -1))
	  (format #t "file read -1: ~A~%" val)))
    (let ((val (read)))
      (if (not (eof-object? val))
	  (format #t "file read #<eof>: ~A~%" val)))
    (let ((val (read)))
      (if (not (eof-object? val))
	  (format #t "file read #<eof> again: ~A~%" val)))))

(let ()
  (call-with-input-string "012"
    (lambda (p)
      (do ((i 0 (+ i 1)))
	  ((= i 4))
	(let ((c (peek-char p)))
	  (let ((r (read-char p)))
	    (if (not (equal? c r))
		(format #t ";peek-char: ~A ~A~%" c r))))))))

(let ((port #f))
  (call-with-exit
   (lambda (go)
     (call-with-input-string "0123456789"
       (lambda (p)
	 (set! port p)
	 (if (not (char=? (peek-char p) #\0))
	     (format #t ";peek-char input-string: ~A~%" (peek-char p)))
	 (go)))))
  (if (not (input-port? port))
      (format #t ";c/e-> c/is -> port? ~A~%" port)
      (if (not (port-closed? port))
	  (begin
	    (format #t ";c/e -> c/is -> closed? ~A~%" port)
	    (close-input-port port)))))

(call-with-output-file tmp-output-file (lambda (p) (display "0123456789" p)))

(let ((port #f))
  (call-with-exit
   (lambda (go)
     (call-with-input-file tmp-output-file
       (lambda (p)
	 (set! port p)
	 (if (not (char=? (peek-char p) #\0))
	     (format #t ";peek-char input-file: ~A~%" (peek-char p)))
	 (go)))))
  (if (not (input-port? port))
      (format #t ";c/e -> c/if -> port? ~A~%" port)
      (if (not (port-closed? port))
	  (begin
	    (format #t ";c/e -> c/if -> closed? ~A~%" port)
	    (close-input-port port)))))

(let ((port #f))
  (call-with-exit
   (lambda (go)
     (dynamic-wind
	 (lambda () #f)
	 (lambda ()
	   (call-with-input-string "0123456789"
             (lambda (p)
	       (set! port p)
	       (if (not (char=? (peek-char p) #\0))
		   (format #t ";peek-char input-string 1: ~A~%" (peek-char p)))
	       (go))))
	 (lambda ()
	   (close-input-port port)))))
  (if (not (input-port? port))
      (format #t ";c/e -> dw -> c/is -> port? ~A~%" port)
      (if (not (port-closed? port))
	  (begin
	    (format #t ";c/e -> dw -> c/is -> closed? ~A~%" port)
	    (close-input-port port)))))

(let ((port #f))
  (call-with-exit
   (lambda (go)
     (dynamic-wind
	 (lambda () #f)
	 (lambda ()
	   (call-with-input-file tmp-output-file
            (lambda (p)
	      (set! port p)
	      (if (not (char=? (peek-char p) #\0))
		  (format #t ";peek-char input-file: ~A~%" (peek-char p)))
	      (go))))
	 (lambda ()
	   (close-input-port port)))))
  (if (not (input-port? port))
      (format #t ";c/e -> dw -> c/if -> port? ~A~%" port)
      (if (not (port-closed? port))
	  (begin
	    (format #t ";c/e -> dw -> c/if -> closed? ~A~%" port)
	    (close-input-port port)))))

(let ((port #f))
  (catch #t
    (lambda ()
     (call-with-input-string "0123456789"
       (lambda (p)
	 (set! port p)
	 (if (not (char=? (peek-char p) #\0))
	     (format #t ";peek-char input-string: ~A~%" (peek-char p)))
	 (error 'oops))))
    (lambda args #f))
  (if (not (input-port? port))
      (format #t ";catch -> c/is -> error -> port? ~A~%" port)
      (if (not (port-closed? port))
	  (begin
	    (format #t ";catch -> c/is -> error -> closed? ~A~%" port)
	    (close-input-port port)))))

(let ((port #f))
  (catch #t
    (lambda ()
     (call-with-input-file tmp-output-file
       (lambda (p)
	 (set! port p)
	 (if (not (char=? (peek-char p) #\0))
	     (format #t ";peek-char input-file: ~A~%" (peek-char p)))
	 (error 'oops))))
    (lambda args #f))
  (if (not (input-port? port))
      (format #t ";catch -> c/if -> error -> port? ~A~%" port)
      (if (not (port-closed? port))
	  (begin
	    (format #t ";catch -> c/if -> error -> closed? ~A~%" port)
	    (close-input-port port)))))

(test (with-output-to-string (lambda () (write (string (integer->char 4) (integer->char 8) (integer->char 20) (integer->char 30))))) "\"\\x04;\\b\\x14;\\x1e;\"")
(test (string-length "\x04;\x08;\x14;\x1e;") 4)
(test (char->integer (string-ref "\x0;" 0)) 0)
(test (char->integer (string-ref "\x0e;" 0)) 14)
(test (char->integer (string-ref "\x1e;" 0)) 30)
(test (char->integer (string-ref "\xff;" 0)) 255)
(test (string=? "\x61;\x42;\x63;" "aBc") #t)
(test (string=? "\"\\x01;\\x02;\\x03;\\x04;\\x05;\\x06;\\x07;\\x08;\\x09;x\\x0b;\\x0c;\\x0d;\\x0e;\\x0f;\\x10;\\x11;\\x12;\\x13;\\x14;\\x15;\\x16;\\x17;\\x18;\\x19;\\x1a;\\x1b;\\x1c;\\x1d;\\x1e;\\x1f; !\\\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\\\]^_`abcdefghijklmnopqrstuvwxyz{|}~\\x7f;\\x80;\\x81;\\x82;\\x83;\\x84;\\x85;\\x86;\\x87;\\x88;\\x89;\\x8a;\\x8b;\\x8c;\\x8d;\\x8e;\\x8f;\\x90;\\x91;\\x92;\\x93;\\x94;\\x95;\\x96;\\x97;\\x98;\\x99;\\x9a;\\x9b;\\x9c;\\x9d;\\x9e;\\x9f;\\xa0Â¡Â¢Â£Â¤Â¥Â¦Â§Â¨Â©ÂªÂ«Â¬\\xadÂ®Â¯Â°Â±Â²Â³Â´ÂµÂ¶Â·Â¸Â¹ÂºÂ»Â¼Â½Â¾Â¿Ã\x80;Ã\x81;Ã\x82;Ã\x83;Ã\x84;Ã\x85;Ã\x86;Ã\x87;Ã\x88;Ã\x89;Ã\x8a;Ã\x8b;Ã\x8c;Ã\x8d;Ã\x8e;Ã\x8f;Ã\x90;Ã\x91;Ã\x92;Ã\x93;Ã\x94;Ã\x95;Ã\x96;Ã\x97;Ã\x98;Ã\x99;Ã\x9a;Ã\x9b;Ã\x9c;Ã\x9d;Ã\x9e;Ã\x9f;Ã Ã¡Ã¢Ã£Ã¤Ã¥Ã¦Ã§Ã¨Ã©ÃªÃ«Ã¬Ã­Ã®Ã¯Ã°Ã±Ã²Ã³Ã´ÃµÃ¶Ã·Ã¸Ã¹ÃºÃ»Ã¼Ã½Ã¾Ã¿\""
		"\"\\x01;\\x02;\\x03;\\x04;\\x05;\\x06;\\x07;\\x08;\\x09;x\\x0b;\\x0c;\\x0d;\\x0e;\\x0f;\\x10;\\x11;\\x12;\\x13;\\x14;\\x15;\\x16;\\x17;\\x18;\\x19;\\x1a;\\x1b;\\x1c;\\x1d;\\x1e;\\x1f; !\\\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\\\]^_`abcdefghijklmnopqrstuvwxyz{|}~\\x7f;\\x80;\\x81;\\x82;\\x83;\\x84;\\x85;\\x86;\\x87;\\x88;\\x89;\\x8a;\\x8b;\\x8c;\\x8d;\\x8e;\\x8f;\\x90;\\x91;\\x92;\\x93;\\x94;\\x95;\\x96;\\x97;\\x98;\\x99;\\x9a;\\x9b;\\x9c;\\x9d;\\x9e;\\x9f;\\xa0Â¡Â¢Â£Â¤Â¥Â¦Â§Â¨Â©ÂªÂ«Â¬\\xadÂ®Â¯Â°Â±Â²Â³Â´ÂµÂ¶Â·Â¸Â¹ÂºÂ»Â¼Â½Â¾Â¿Ã\x80;Ã\x81;Ã\x82;Ã\x83;Ã\x84;Ã\x85;Ã\x86;Ã\x87;Ã\x88;Ã\x89;Ã\x8a;Ã\x8b;Ã\x8c;Ã\x8d;Ã\x8e;Ã\x8f;Ã\x90;Ã\x91;Ã\x92;Ã\x93;Ã\x94;Ã\x95;Ã\x96;Ã\x97;Ã\x98;Ã\x99;Ã\x9a;Ã\x9b;Ã\x9c;Ã\x9d;Ã\x9e;Ã\x9f;Ã Ã¡Ã¢Ã£Ã¤Ã¥Ã¦Ã§Ã¨Ã©ÃªÃ«Ã¬Ã­Ã®Ã¯Ã°Ã±Ã²Ã³Ã´ÃµÃ¶Ã·Ã¸Ã¹ÃºÃ»Ã¼Ã½Ã¾Ã¿\"") #t)

(when (provided? 'system-extras)
  ;; directory?
  (test (directory? tmp-output-file) #f)
  (test (directory? ".") #t)
  (test (directory?) 'error)
  (test (directory? "." 0) 'error)
  (for-each
   (lambda (arg)
     (test (directory? arg) 'error))
   (list -1 #\a 1 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
	 3.14 3/4 1.0+1.0i #f #t :hi (if #f #f) (lambda (a) (+ a 1))))

  ;; file-exists?
  (test (file-exists? tmp-output-file) #t)
  (test (file-exists? "not-a-file-I-hope") #f)
  (test (file-exists?) 'error)
  (test (file-exists? tmp-output-file 0) 'error)
  (for-each
   (lambda (arg)
     (test (file-exists? arg) 'error))
   (list -1 #\a 1 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
	 3.14 3/4 1.0+1.0i #f #t :hi (if #f #f) (lambda (a) (+ a 1))))

  ;; delete-file
  (test (delete-file tmp-output-file) 0)
  (test (file-exists? tmp-output-file) #f)
  (test (delete-file "not-a-file-I-hope") -1)
  (test (delete-file) 'error)
  (test (delete-file tmp-output-file 0) 'error)
  (for-each
   (lambda (arg)
     (test (delete-file arg) 'error))
   (list -1 #\a 1 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
	 3.14 3/4 1.0+1.0i #f #t :hi (if #f #f) (lambda (a) (+ a 1))))

  ;; getenv
  (test (pair? (member (getenv "HOME") (list (append "/usr/home/" username)
                                             (append "/Users/" username)
                                             (append "/home/" username)) string=?)) #t)
  (test (getenv "NO-ENV") #f) ; was ""
  (test (getenv) 'error)
  (test (getenv "HOME" #t) 'error)
  (for-each
   (lambda (arg)
     (test (getenv arg) 'error))
   (list -1 #\a 1 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
	 3.14 3/4 1.0+1.0i #f #t :hi (if #f #f) (lambda (a) (+ a 1))))

  ;; directory->list
  (test (directory->list) 'error)
  (test (directory->list "." 0) 'error)
  (for-each
   (lambda (arg)
     (test (directory->list arg) 'error))
   (list -1 #\a 1 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
	 3.14 3/4 1.0+1.0i #f #t :hi (if #f #f) (lambda (a) (+ a 1))))
  (test (pair? (directory->list "tools")) #t)

  ;; file-mtime
  (test (integer? (file-mtime "s7test.scm")) #t)
  (test (file-mtime "asdf.data") 'error)
  (test (file-mtime #\a) 'error)
  (test (file-mtime) 'error)
  (test (file-mtime "asdf" "a") 'error)

  ;; system
  (test (system "test -f s7test.scm") 0)
  (test (system) 'error)
  (test (let ((str (system "man fgrep" #t)))
	  (and (string? str)
	       (> (length str) 10000))) ; osx: 14479, linux: 40761
	#t)
  (for-each
   (lambda (arg)
     (test (system arg) 'error))
   (list -1 #\a 1 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
	 3.14 3/4 1.0+1.0i #f #t :hi (if #f #f) (lambda (a) (+ a 1)))))

(if (not pure-s7)
    (for-each
     (lambda (arg)
       (test (char-ready? arg) 'error))
     (list "hi" -1 #\a 1 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0)
	   3.14 3/4 1.0+1.0i #f #t :hi (if #f #f) (lambda (a) (+ a 1)))))

;;; newline
(test (newline 0) 'error)
(test (newline *stdout* 0) 'error)
(test (newline #f) #\newline)
(test (with-output-to-string (lambda () (newline))) "\n")
(test (with-output-to-string (lambda () (newline #f))) "")
(test (call-with-output-string (lambda (p) (newline p))) "\n")
(test (newline *stdin*) 'error)

(test (display 0 0) 'error)
(test (write 0 0) 'error)
(test (write-char 0 0) 'error)
(test (write-string 0 0) 'error)
(test (write-line 0 0) 'error)
(test (read-char 0) 'error)
(test (read 0) 'error)
(test (read-string 1 0) 'error)
(test (read-line 0) 'error)


;;; -------- format --------
;;; format

(test (format #f "hiho") "hiho")
(test (format #f "") "")
(test (format #f "" 1) 'error)
(test (format #f "a") "a")
;(test (format #f "a\x00;b") "a")

(test (format #f "~~") "~") ; guile returns this, but clisp thinks it's an error
(test (format #f "~~~~") "~~")
(test (format #f "a~~") "a~")
(test (format #f "~~a") "~a")
(test (format #f "~A" "") "")
(test (format #f "~{~^~A~}" ()) "")
(test (format #f "~{~^~{~^~A~}~}" '(())) "")
(test (format #f "~P" 1) "")
(test (format #f "~P" #\a) 'error)
(test (format #f "~0T") "")
(test (format #f "") "")
(test (format #f "~*~*" 1 2) "")
(test (format #f "~20,'~D" 3) "~~~~~~~~~~~~~~~~~~~3")
(test (format #f "~0D" 123) "123")
(test (format #f "~{~S~}" ()) "")
(test (format #f "~-1D" 123) 'error)
(test (format #f "~+1D" 123) 'error)
(test (format #f "~1.D" 123) 'error)
(test (format #f "~1+iD" 123) 'error)
(test (format #f "~1/2D" 123) 'error)
(test (format #f "~1/1D" 123) 'error)
(test (format #f "~20,'-1D" 123) 'error)

(for-each
 (lambda (arg)
   (test (format arg "~D" 1) 'error))
 (list "hi" #\a 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand :hi (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (directive)
   (for-each
    (lambda (arg)
      (test (format #f directive arg) 'error)
      (test (format #f directive) 'error))
    (list "hi" #\a 'a-symbol (make-vector 3) abs _ht_ _undef_ _null_ _c_obj_ quasiquote macroexpand
	  #f #t :hi (if #f #f) (lambda (a) (+ a 1)))))
 (list "~D" "~F" "~G" "~X" "~B" "~O" "~E" "~P"))

(test (format #f "~,1" 123) 'error)
;format "~,1" 123: numeric argument, but no directive!
;    (format #f "~,1" 123)

(test (format #f "~,123456789123456789123456789d" 1) 'error)
;format "~,123456789123456789123456789d" 1: numeric argument too large
;    (format #t "~,123456789123456789123456789d" 1)
(test (format #f "~969424987x" 12) 'error)

(test (format #f "~D" 1 2) 'error)
;format: "~D" 1 2
;           ^: too many arguments
;    (format #f "~D" 1 2)

(test (format #f "~D~" 1) 'error)
;format: "~D~" 1
;           ^: control string ends in tilde
;    (format #f "~D~" 1)
(test (format #f "~") 'error)
(test (format #f " ~") 'error)
(test (format #f "~~~") 'error)
(test (format #f " ~~~") 'error)

(test (format #f "~@D" 1) 'error)
;format "~@D" 1: unknown '@' directive
;    (format #f "~@D" 1)

(test (format #f "~@p" #\a) 'error)
;format "~@p" #\a: '@P' directive argument is not an integer
;    (format #f "~@p" #\a)

(test (format #f "~@p") 'error)
;format "~@p": '@' directive argument missing

(test (format #f "~P" 1+i) 'error)
;format "~P" 1+1i: 'P' directive argument is not a real number
;    (format #f "~P" 1+1i)

(test (format #f "~P" (real-part (log 0))) "s")

(test (format #f "~@p" 0+i) 'error)
;format "~@p" 0+1i: '@P' directive argument is not a real number
;    (format #f "~@p" 0+1i)

(test (format #f "~{~}") 'error)
;format "~{~}": missing argument
;    (format #f "~{~}")

(test (format #f "~{~a" '(1 2 3)) 'error)
;format "~{~a" (1 2 3): '{' directive, but no matching '}'
;    (format #f "~{~a" '(1 2 3))

(test (format #f "~{~a~}" '(1 . 2)) 'error) ; changed 28-Nov-18, then again 11-Dec-18
;format "~{~a~}" (1 . 2): '{' directive argument should be a proper list or something we can turn into a list
;    (format #f "~{~a~}" '(1 . 2))

(test (let ((lst (cons 1 2))) (set-cdr! lst lst) (format #f "~{~A~}" lst)) "1")
;format "~{~A~}" #1=(1 . #1#): '{' directive argument should be a proper list or something we can turn into a list
;    (format #f "~{~A~}" lst)

(test (format #f "~{~a~}" 'asdf) 'error)
;format "~{~a~}" asdf: '{' directive argument should be a proper list or something we can turn into a list
;    (format #f "~{~a~}" 'asdf)

(test (format #f "~{~a~}" ()) "")
(test (format #f "~{asd~}" '(1 2 3)) 'error)
;format: "~{asd~}" (1 2 3)
;             ^: '{...}' doesn't consume any arguments!
;    (format #f "~{asd~}" '(1 2 3))

(test (format #f "~}" '(1 2 3)) 'error)
;format "~}" (1 2 3): unmatched '}'
;    (format #f "~}" '(1 2 3))

(test (format #f "~C") 'error)
;format "~C": ~C: missing argument
;    (format #f "~C")

(test (format #f "~A ~C" #\a) 'error)
;format: "~A ~C" #\a
;            ^: ~C: missing argument
;    (format #f "~A ~C" #\a)

(test (format #f "~C" 1) 'error)
;format "~C" 1: 'C' directive requires a character argument
;    (format #f "~C" 1)

(test (format #f "~C" #<eof>) 'error)
;format "~C" #<eof>: 'C' directive requires a character argument
;    (format #f "~C" #<eof>)

(test (format #f "~1,9223372036854775807f" 1) 'error)
;format "~1,9223372036854775807f" 1: numeric argument too large
;    (format #f "~1,9223372036854775807f" 1)

(test (format #f "~1,2A" 1) 'error)
;format "~1,2A" 1: extra numeric argument
;    (format #f "~1,2A" 1)

(test (format #f "~F" #\a) 'error)
;format "~F" #\a: ~F: numeric argument required
;    (format #f "~F" #\a)

(test (format #f "~1,") 'error)
;format "~1,": format directive does not take a numeric argument
;    (format #f "~1,")

(test (format #f "~-1,") 'error)
;format "~-1,": unimplemented format directive
;    (format #f "~-1,")

(test (format #f "~L" 1) 'error)
;format "~L" 1: unimplemented format directive
;    (format #f "~L" 1)

(test (format #f "~*") 'error)
;format "~*": can't skip argument!

(test (format #f "~*~A") 'error)
(test (format #f "~*~*" 1) 'error)
(test (format #f "~N") 'error)
(test (format #f "~N" 2) 'error)
(test (format #f "~N." 2) 'error)
(test (format #f "~NT" 2.1) 'error)
(test (format #f "~NT" #\a) 'error)
(test (format #f "~N," 1) 'error)
(test (format #f "~N,N" 1 2) 'error)
(test (format #f "~N,N." 1 2) 'error)
(test (format #f "~,N" 1) 'error)
(test (format #f "~,N." 1) 'error)
(test (format #f "~ND" 123456789) 'error)
(test (format #f "~ND" -1) 'error)

(test (format #f "~{~a~e~}" (cons 1 2)) 'error)
;format: "~{~a~e~}" (1 . 2): ~{} argument is a dotted list

(for-each
 (lambda (c)
   (test (apply format #f (string-append "~" (string c)) '(a)) 'error))
 (list #\H #\I #\J #\K #\L #\M #\Q #\R #\U #\V #\Y #\Z
       #\[ #\\ #\] #\_ #\` #\# #\! #\" #\' #\( #\) #\+ #\, #\- #\. #\/ #\< #\= #\> #\?
       #\h #\i #\j #\k #\l #\m #\q #\r #\u #\v #\y #\z))

(test (format #f "~A" 1 2) 'error)
;format: "~A" 1 2
;           ^: too many arguments
;    (format #f "~A" 1 2)

(test (format #f "hiho~%ha") (string-append "hiho" (string #\newline) "ha"))
(test (format #f "~%") (string #\newline))
(test (format #f "~%ha") (string-append (string #\newline) "ha"))
(test (format #f "hiho~%") (string-append "hiho" (string #\newline)))

(test (eq? #\tab ((format #f "\t") 0)) #t)
(test (eq? #\newline ((format #f "\n") 0)) #t)
(test (eq? #\\ ((format #f "\\") 0)) #t)
(test (eq? #\" ((format #f "\"") 0)) #t)

(for-each
 (lambda (arg res)
   (let ((val (catch #t (lambda () (format #f "~A" arg)) (lambda args 'error))))
     (if (or (not (string? val))
	     (not (string=? val res)))
	 (begin (display "(format #f \"~A\" ") (display arg)
		(display " returned \"") (display val)
		(display "\" but expected \"") (display res) (display "\"")
		(newline)))))
 (list "hiho"  -1  #\a  1   #f   #t  #(1 2 3)   3.14   3/4  1.5+1.5i ()  #(())  (list 1 2 3) '(1 . 2) 'hi)
 (list "hiho" "-1" "a" "1" "#f" "#t" "#(1 2 3)" "3.14" "3/4" "1.5+1.5i"   "()" "#(())" "(1 2 3)"    "(1 . 2)" "hi"))

(test (format #f "hi ~A ho" 1) "hi 1 ho")
(test (format #f "hi ~a ho" 1) "hi 1 ho")
(test (format #f "~a~A~a" 1 2 3) "123")
(test (format #f "~a~~~a" 1 3) "1~3")
(test (format #f "~a~%~a" 1 3) (string-append "1" (string #\newline) "3"))

(for-each
 (lambda (arg res)
   (let ((val (catch #t (lambda () (format #f "~S" arg)) (lambda args 'error))))
     (if (or (not (string? val))
	     (not (string=? val res)))
	 (begin (display "(format #f \"~S\" ") (display arg)
		(display " returned \"") (display val)
		(display "\" but expected \"") (display res) (display "\"")
		(newline)))))
 (list "hiho"  -1  #\a  1   #f   #t  #(1 2 3)   3.14   3/4  1.5+1.5i ()  #(())  (list 1 2 3) '(1 . 2) 'hi)
 (list "\"hiho\"" "-1" "#\\a" "1" "#f" "#t" "#(1 2 3)" "3.14" "3/4" "1.5+1.5i"   "()" "#(())" "(1 2 3)"    "(1 . 2)" "hi"))

(test (format #f "hi ~S ho" 1) "hi 1 ho")
(test (format #f "hi ~S ho" "abc") "hi \"abc\" ho")
(test (format #f "~s~a" #\a #\b) "#\\ab")
(test (format #f "~C~c~C" #\a #\b #\c) "abc")
;(test (format #f "1 2~C 3 4" #\null) "1 2") ; ?? everyone does something different here
;; s7 used to return "1 2 3 4" because it treated ~C as a string (empty in this case)
(test  (format #f "1 2~C 3 4" #\null) "1 2\x00; 3 4")
(test (format #f "~nc" 3 #\a) "aaa")
(test (format #f "~nc" 0 #\a) "")
(test (format #f "~0c" #\a) "")
(test (format #f "~01c" #\a) "a")
(test (format #f "~002c" #\a) "aa")
(test (format #f "~nc" -1 #\a) 'error)
(test (format #f "~nc" most-positive-fixnum #\a) 'error)
(test (format #f "~nc" 1.0 #\a) 'error)
(test (format #f "~n~nc" 1 2 #\a) 'error)
(test (format #f "~na" 1 #\a) 'error)
(test (format #f "[~NC]" 0 #\a) "[]")
(test (format #f "[~NC]" -1 #\a) 'error)
(test (format #f "[~NC]" 1 #\a) "[a]")

(test (format #f "~{~A~}" '(1 2 3)) "123")
(test (format #f "asb~{~A ~}asb" '(1 2 3 4)) "asb1 2 3 4 asb")
(test (format #f "asb~{~A ~A.~}asb" '(1 2 3 4)) "asb1 2.3 4.asb")
(test (format #f ".~{~A~}." ()) "..")

(test (format #f "~{~A ~A ~}" '(1 "hi" 2 "ho")) "1 hi 2 ho ")
(test (format #f "~{.~{+~A+~}.~}" (list (list 1 2 3) (list 4 5 6))) ".+1++2++3+..+4++5++6+.")
(test (format #f "~{~s ~}" '(fred jerry jill)) "fred jerry jill ")
(test (format #f "~{~s~^ ~}" '(fred jerry jill)) "fred jerry jill")
(test (format #f "~{~s~^~^ ~}" '(fred jerry jill)) "fred jerry jill")
(test (format #f "~{.~{~A~}+~{~A~}~}" '((1 2) (3 4 5) (6 7 8) (9))) ".12+345.678+9")
(test (format #f "~{.~{+~{-~A~}~}~}" '(((1 2) (3 4 5)))) ".+-1-2+-3-4-5")
(test (format #f "~{.~{+~{-~A~}~}~}" '(((1 2) (3 4 5)) ((6) (7 8 9)))) ".+-1-2+-3-4-5.+-6+-7-8-9")

(test (format #f "~A ~* ~A" 1 2 3) "1  3")
(test (format #f "~*" 1) "")
(test (format #f "~{~* ~}" '(1 2 3)) "   ")
(test (format #f "~A" catch) "catch")
(test (format #f "this is a ~
             sentence") "this is a sentence")
(test (format #f "~{~C~}" "hi") "hi")
(test (format #f "~{~C~}" #(#\h #\i)) "hi")
(test (format #f "~S" #(a b)) "#(a b)")
(test (format #f "~S" #(a 'b)) "#(a 'b)")

(test (format #f "~{.~{~C+~}~}" '((#\h #\i) (#\h #\o))) ".h+i+.h+o+")
(test (format #f "~{.~{~C+~}~}" '("hi" "ho")) ".h+i+.h+o+")
(test (format #f "~{.~{~C+~}~}" #("hi" "ho")) ".h+i+.h+o+")
(test (format #f "~{.~{~C+~}~}" #(#(#\h #\i) #(#\h #\o))) ".h+i+.h+o+")

; (format #f "~{.~{~C~+~}~}" #2d((#\h #\i) (#\h #\o))) error?? -- this is documented...
(test (format #f "~{~A~}" #2d((1 2) (3 4))) "1234") ; this seems inconsistent with:
(test (format #f "~{~A~}" '((1 2) (3 4))) "(1 2)(3 4)")
(test (format #f "~{~A ~}" #2d((1 2) (3 4))) "1 2 3 4 ")
(test (format #f "1~\
a2" 3) "132")
(test (format #f "1~
                 ~a2" 3) "132")

(test (format #f "~{~{~C~^ ~}~^...~}" (list "hiho" "test")) "h i h o...t e s t")

;; ~nT handling is a mess -- what are the defaults?  which is column 1? do we space up to or up to and including?

(test (format #f "~A:~8T~A" 100 'a)   "100:   a")
(test (format #f "~A:~nT~A" 100 8 'a)   "100:   a")
(test (format #f "~A:~8T~A" 0 'a)     "0:     a")
(test (format #f "~A:~8T~A" 10000 'a) "10000: a")
(test (format #f "~8T~A" 'a)      "       a")
(test (format #f "1212:~8T~A" 'a) "1212:  a")
(test (format #f "~D:~8T~A" 100 'a)   "100:   a")
(test (format #f "~D:~8T~A" 0 'a)     "0:     a")
(test (format #f "~D:~8T~A" 10000 'a) "10000: a")
(test (format #f "~a~10,7Tb" 1)     "1               b")
(test (format #f "~a~10,7Tb" 10000) "10000           b")
(test (format #f "~a~10,12Tb" 1)     "1                    b")
(test (format #f "~a~10,12Tb" 10000) "10000                b")
(test (format #f "~a~n,nTb" 10000 10 12) "10000                b")
(test (format #f "~n,'xT" 8) "xxxxxxx")
(test (format #f "~n,' T" 8) "       ")

(test (length (format #f "~{~A~}~40T." '(1 2 3))) 40)
(test (length (format #f "~{~A ~}~40T." '(1 2 3))) 40)
(test (length (format #f "~{~,3F ~}~40T." '(1.0 2.0 3.0))) 40)
(test (length (format #f "~S~40T." pi)) (if with-bignums 44 40))

(test (format #f "asdh~20Thiho") "asdh               hiho")
(test (format #f "asdh~2Thiho") "asdhhiho")
(test (format #f "a~Tb") "ab")
(test (format #f "0123456~4,8Tb") "0123456    b")
(test (format #f "0123456~0,8Tb") "0123456b")
(test (format #f "0123456~10,8Tb") "0123456          b")
(test (format #f "0123456~1,0Tb") "0123456b")
(test (format #f "0123456~1,Tb") "0123456b")
(test (format #f "0123456~1,Tb") "0123456b")
(test (format #f "0123456~,Tb") "0123456b")
(test (format #f "0123456~7,10Tb") "0123456         b")
(test (format #f "0123456~8,10tb") "0123456          b")
(test (format #f "0123456~3,12tb") "0123456       b")
(test (format #f "~40TX") "                                       X")
(test (format #f "X~,8TX~,8TX") "X      X       X")
(test (format #f "X~8,TX~8,TX") "X      XX")
(test (format #f "X~8,10TX~8,10TX") "X                X         X")
(test (format #f "X~8,0TX~8,0TX") "X      XX")
(test (format #f "X~0,8TX~0,8TX") "X      X       X")
(test (format #f "X~1,8TX~1,8TX") "X       X       X")
(test (format #f "X~,8TX~,8TX") "X      X       X") ; ??
(test (format #f "X~TX~TX") "XXX") ; clisp and sbcl say "X X X" here and similar differences elsewhere -- is it colnum or colinc as default if no comma?
(test (format #f "X~2TX~4TX") "XX X")
(test (format #f "X~0,0TX~0,0TX") "XXX")
(test (format #f "X~0,TX~0,TX") "XXX")
(test (format #f "X~,0TX~,0TX") "XXX")

(test (format #f "~0D" 123) "123")
(test (format #f "~0F" 123.123) "123.123000")
(test (format #f "~,0D" 123) "123")
(test (format #f "~,0F" 123.123) "123.0")
(test (format #f "~,D" 123) "123")
(test (format #f "~,F" 123.123) "123.123000")
(test (format #f "~0,D" 123) "123")
(test (format #f "~0,F" 123.123) "123.123000")
(test (format #f "~0,0D" 123) "123")
(test (format #f "~n,nD" 0 0 123) "123")
(test (format #f "~0,0F" 123.123) "123.0")
(test (format #f "~0,0,D" 123) 'error)
(test (format #f "~n,n,D" 0 0 123) 'error)
(test (format #f "~0,0,F" 123.123) 'error)

(test (format #f "~,3F" 1+i) "1.000+1.000i")
(test (format #f "~,nF" 3 1+i) "1.000+1.000i")
(test (format #f "~,3G" 1+i) "1.0+1.0i")
(test (format #f "~,3E" 1+i) "1.000e+00+1.000e+00i")
(test (format #f "~,3F" 1-i) "1.000-1.000i")
(test (format #f "~,3G" 1-i) "1.0-1.0i")
(test (format #f "~,3E" 1-i) "1.000e+00-1.000e+00i")

;; not sure about these:
(test (format #f "~X" 1-i) "1.0-1.0i")
(test (format #f "~,3D" 1-i) "1.000e+00-1.000e+00i")
(test (format #f "~A" 1-i) "1.0-1.0i")

(test (format #f "~W" 3) "3")
(test (format #f "~W" 3/4) "3/4")
(test (format #f "~W" 3.4) "3.4")
(test (format #f "~W" 3+4i) "3.0+4.0i")
(test (format #f "~W" 3-4i) "3.0-4.0i")
(if with-bignums
    (test (format #f "~W" pi) "3.141592653589793238462643383279502884195E0")
    (test (format #f "~W" pi) "3.141592653589793"))
(unless with-bignums
  (test (format #f "~W" (complex 1/0 0)) "+nan.0")
  (test (format #f "~W" (complex 1/0 1)) "+nan.0+1.0i")
  (test (format #f "~W" (complex +inf.0 1/0)) "+inf.0+nan.0i")
  (test (format #f "~W" (log 0)) "-inf.0+3.141592653589793i"))

(test (catch #t (lambda () (let () (define (c1 a b) (+ a b)) (c1 1))) (lambda (t i) (apply format #f i))) "c1: not enough arguments: ((lambda (a b) ...) 1)")
(test (catch #t (lambda () (let () (define (c1 a b) (+ a b)) (c1 1 2 3))) (lambda (t i) (apply format #f i))) "c1: too many arguments: ((lambda (a b) ...) 1 2 3)")
(test (catch #t (lambda () (let () (define-macro (m1 a b) (+ a b)) (m1 1))) (lambda (t i) (apply format #f i))) "m1: not enough arguments: ((macro (a b) ...) 1)")
(test (catch #t (lambda () (let () (define-macro (m1 a b) (+ a b)) (m1 1 2 3))) (lambda (t i) (apply format #f i))) "m1: too many arguments: ((macro (a b) ...) 1 2 3)")
(test (catch #t (lambda () (let () (define-bacro (m2 a b) (+ a b)) (m2 1))) (lambda (t i) (apply format #f i))) "m2: not enough arguments: ((bacro (a b) ...) 1)")
(test (catch #t (lambda () (let () (define-bacro (m2 a b) (+ a b)) (m2 1 2 3))) (lambda (t i) (apply format #f i))) "m2: too many arguments: ((bacro (a b) ...) 1 2 3)")

;; see also object->string with :readable

(test (format #f "~000000000000000000000000000000000000000000003F" 123.123456789) "123.123457")
(test (format #f "~922337203685477580F" 123.123) 'error)   ; numeric argument too large
(test (format #f "~,922337203685477580F" 123.123) 'error)
(test (format #f "~1,922337203685477580F" 123.123) 'error)
(test (format #f "~1 ,2F" 123.456789) 'error)
(test (format #f "~1, 2F" 123.456789) 'error)
(test (format #f "~1, F" 123.456789) 'error)

(if with-bignums
    (begin
      (test (format #f "~o" 1e19) "1.053071060221172E21")
      (test (format #f "~o" -1e19) "-1.053071060221172E21")
      (test (format #f "~x" 1e19) "8.ac7230489e8@15")
      (test (format #f "~b" 1e19) "1.00010101100011100100011000001001000100111101E63")
      (test (format #f "~o" 1e308) "1.071474702753621177617256074117252375235444E341")
      (test (format #f "~o" -1e308) "-1.071474702753621177617256074117252375235444E341")
      (test (format #f "~x" 1e308) "8.e679c2f5e44ff8f570f09eaa7ea7648@255")
      (test (format #f "~x" 9.22e18) "7.ff405267d1a@15")
      (test (format #f "~b" 1e308) "1.0001110011001111001110000101111010111100100010011111111100011110101011100001111000010011110101010100111111010100111011001001E1023")
      (test (format #f "~,791o" 1e308) "1.071474702753621177617256074117252375235444E341")
      (test (format #f "~1200,2g" 1e308) "                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   9.999999999999999999999999999999999999982E307")
      (test (format #f "~o" 1e19-1e20i) "1.053071060221172E21-1.2657072742654304E22i")
      (test (format #f "~x" 1e308+1e300i) "8.e679c2f5e44ff8f570f09eaa7ea7648@255+1.7e43c8800759ba59c08e14c7cd7aad86@249i"))
    (begin
      (test (format #f "~o" 1e19) "1.053071e21")
      (test (format #f "~o" -1e19) "-1.053071e21")
      (test (format #f "~x" 1e19) "8.ac723@15")
      (test (format #f "~b" 1e19) "1.000101e63")
      (test (format #f "~o" 1e308) "1.071474e341")
      (test (format #f "~o" -1e308) "-1.071474e341")
      (test (format #f "~x" 1e308) "8.e679c2@255")
      (test (or (string=? (format #f "~x" 9.22e18) "7ff405267d1a0000.0")
		(string=? (format #f "~x" 9.22e18) "7.ff4052@15"))
	    #t) ; this depends on a cutoff point in s7.c, L8850, number_to_string_with_radix
      (test (format #f "~b" 1e308) "1.000111e1023")
      (test (format #f "~,791o" 1e308) "1.0714747027536212e341")
      (test (format #f "~1200,2g" 1e308) "                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          1e+308")
      (test (format #f "~o" 1e19-1e20i) "1.053071e21-1.265707e22i")
      (test (format #f "~x" 1e308+1e300i) "8.e679c2@255+1.7e43c8@249i")))

(test (= (length (substring (format #f "~%~10T.") 1)) (length (format #f "~10T."))) #t)
(test (= (length (substring (format #f "~%-~10T.~%") 1)) (length (format #f "-~10T.~%"))) #t)
(test (string=? (format #f "~%|0 1 2|~21T|5  8  3  2|~%~
                              |1 2 3| |0 1 2 3|~21T|8 14  8  6|~%~
                              |2 3 0| |1 2 3 0| = ~21T|3  8 13  6|~%~
                              |3 0 1| |2 3 0 1|~21T|2  6  6 10|~%")
		"
|0 1 2|             |5  8  3  2|
|1 2 3| |0 1 2 3|   |8 14  8  6|
|2 3 0| |1 2 3 0| = |3  8 13  6|
|3 0 1| |2 3 0 1|   |2  6  6 10|
") #t)


(if (not with-windows) (test (format #f "~S" '(+ 1/0 1/0)) "(+ +nan.0 +nan.0)")) ; !?
(if (not with-windows) (test (format #f "~S" '(+ '1/0 1/0)) "(+ '+nan.0 +nan.0)")) ; !?
(test (format #f "~S" '(+ 1/0 1.0/0.0)) (format #f "~S" (list '+ '1/0 '1.0/0.0)))
(test (format #f "~S" (quote (+ '1 1))) "(+ '1 1)")


(test (format #f "~12,''D" 1) "'''''''''''1")
(test (let ((str "~12,'xD")) (set! (str 5) #\space) (format #f str 1)) "           1")
(test (format #f "~12,' D" 1) "           1")
(test (format #f "~12,'\\D" 1) "\\\\\\\\\\\\\\\\\\\\\\1")
(test (format #f "~12,'\"D" 1) "\"\"\"\"\"\"\"\"\"\"\"1")
(test (format #f "~12,'~D" 1) "~~~~~~~~~~~1")
(test (format #f "~12,',d" 1) ",,,,,,,,,,,1")
(test (format #f "~12,',,d" 1) 'error)
(test (format #f "~12,,d" 1) 'error)
(test (format #f "~n,,d" 12 1) 'error)
(test (format #f "hiho~\n") "hiho")

(test (string=? (format #f "~%~&" ) (string #\newline)) #t)
(test (string=? (format #f "~%a~&" ) (string #\newline #\a #\newline)) #t)
(test (string=? (format #f "~%~%") (string #\newline #\newline)) #t)
(test (string=? (format #f "~10T~%~&~10T.") (format #f "~10T~&~&~10T.")) #t)
(test (string=? (format #f "~10T~&~10T.") (format #f "~10T~%~&~&~&~&~10T.")) #t)
(test (length (format #f "~%~&~%")) 2)
(test (length (format #f "~%~&~&~&~&~%")) 2)
(test (length (format #f "~&~%")) 1)

(test (format #f "~2,1F" 0.5) "0.5")
(test (format #f "~:2T") 'error)
(test (format #f "~2,1,3F" 0.5) 'error)
(test (format #f "~<~W~>" 'foo) 'error)
(test (format #f "~{12") 'error)
(test (format #f "~{}") 'error)
(test (format #f "~{}" '(1 2)) 'error)
(test (format #f "{~}" '(1 2)) 'error)
(test (format #f "~{~{~}}" '(1 2)) 'error)
(test (format #f "~}" ) 'error)
;(test (format #f "#|~|#|") 'error) ; ~| is ~^+ now
(test (format #f "~1.5F" 1.5) 'error)
(test (format #f "~1+iF" 1.5) 'error)
(test (format #f "~1,1iF" 1.5) 'error)
(test (format #f "~0" 1) 'error)
(test (format #f "~1") 'error)
(test (format #f "~^" 1) 'error)
(test (format #f "~.0F" 1.0) 'error)
(test (format #f "~1.0F" 1.0) 'error)
(test (format #f "~-1F" 1.0) 'error)
(test (format #f "~^") "")
(test (format #f "~A ~A~|this is not printed" 1 2) "1 2")
(test (format #f "~^~A~^~A~^this is not printed" 1 2) "12")
(test (format #f "~|") "")
(test (format #f "~D~" 9) 'error)
(test (format #f "~&" 9) 'error)
(test (format #f "~D~100T~D" 1 1) "1                                                                                                  1")
(test (format #f ".~P." 1) "..")
(test (format #f ".~P." 1.0) "..")
(test (format #f ".~P." 1.2) ".s.")
(test (format #f ".~P." 2/3) ".s.")
(test (format #f ".~P." 2) ".s.")
(test (format #f ".~p." 1) "..")
(test (format #f ".~p." 1.0) "..")
(test (format #f ".~p." 1.2) ".s.")
(test (format #f ".~p." 2) ".s.")
(test (format #f ".~@P." 1) ".y.")
(test (format #f ".~@P." 1.0) ".y.")
(test (format #f ".~@P." 1.2) ".ies.")
(test (format #f ".~@P." 2) ".ies.")
(test (format #f ".~@p." 1) ".y.")
(test (format #f ".~@p." 1.0) ".y.")
(test (format #f ".~@p." 1.2) ".ies.")
(test (format #f ".~@p." 2) ".ies.")
(test (format #f ".~P." 1.0+i) 'error)
(test (format #f ".~P." 1/0) ".s.")
(test (format #f "~P" 1) "") ; Clisp does this
(if (not with-windows) (test (format #f ".~P." (real-part (log 0))) ".s."))

(test (format #f (string #\~ #\a) 1) "1")
(test (format #f (format #f "~~a") 1) "1")
(test (format #f (format #f "~~a") (format #f "~D" 1)) "1")
(test (format #f "~A" (quasiquote quote)) "quote")

(test (format #f "~f" (/ 1 3)) "1/3") ; hmmm -- should it call exact->inexact?
(test (format #f "~f" 1) "1")
(test (format #f "~F" most-positive-fixnum) "9223372036854775807")

(test (format () "") #f)
(test (with-output-to-string (lambda () (display (format () "")))) "#f")
(test (with-output-to-string (lambda () (display #f))) "#f")

(unless with-bignums
  (test (format #f "~,20F" 1e-20) "0.00000000000000000001")
  (test (format #f "~,40F" 1e-40) "0.0000000000000000000000000000000000000001"))
;; if with bignums, these needs more bits

;;; the usual troubles here with big floats:
;;; (format #f "~F" 922337203685477580.9) -> "922337203685477632.000000"
;;; (format #f "~F" 9223372036854775.9) -> "9223372036854776.000000"
;;; (format #f "~F" 1e25) -> "10000000000000000905969664.000000"
;;; or small:
;;; (format #f "~,30F" 1e-1) -> "0.100000000000000005551115123126"

(when with-bignums
  (test (format #f "~A" -7043009959286724629649270926654940933664689003233793014518979272497911394287216967075767325693021717277238746020477538876750544587281879084559996466844417586093291189295867052594478662802691926547232838591510540917276694295393715934079679531035912244103731582711556740654671309980075069010778644542022/670550434139267031632063192770201289106737062379324644110801846820471752716238484923370056920388400273070254958650831435834503195629325418985020030706879602898158806736813101434594805676212779217311897830937606064579213895527844045511878668289820732425014254579493444623868748969110751636786165152601) "-7043009959286724629649270926654940933664689003233793014518979272497911394287216967075767325693021717277238746020477538876750544587281879084559996466844417586093291189295867052594478662802691926547232838591510540917276694295393715934079679531035912244103731582711556740654671309980075069010778644542022/670550434139267031632063192770201289106737062379324644110801846820471752716238484923370056920388400273070254958650831435834503195629325418985020030706879602898158806736813101434594805676212779217311897830937606064579213895527844045511878668289820732425014254579493444623868748969110751636786165152601")
  (test (format #f "~D" -7043009959286724629649270926654940933664689003233793014518979272497911394287216967075767325693021717277238746020477538876750544587281879084559996466844417586093291189295867052594478662802691926547232838591510540917276694295393715934079679531035912244103731582711556740654671309980075069010778644542022/670550434139267031632063192770201289106737062379324644110801846820471752716238484923370056920388400273070254958650831435834503195629325418985020030706879602898158806736813101434594805676212779217311897830937606064579213895527844045511878668289820732425014254579493444623868748969110751636786165152601) "-7043009959286724629649270926654940933664689003233793014518979272497911394287216967075767325693021717277238746020477538876750544587281879084559996466844417586093291189295867052594478662802691926547232838591510540917276694295393715934079679531035912244103731582711556740654671309980075069010778644542022/670550434139267031632063192770201289106737062379324644110801846820471752716238484923370056920388400273070254958650831435834503195629325418985020030706879602898158806736813101434594805676212779217311897830937606064579213895527844045511878668289820732425014254579493444623868748969110751636786165152601"))

(unless with-bignums
  (test (format #f "~,1024F" pi)
        "3.1415926535897931159979634685441851615905761718750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"))

(test (format #f "~100,30000F" pi) 'error)
(test (format #f "~N,1F" most-positive-fixnum pi) 'error) ; "~N value is too big"
(test (format #f "~N,1F" (- most-positive-fixnum 1) pi) 'error) ; "~N value is too big"
(test (format #f "~N,1F" most-negative-fixnum pi) 'error) ; "~N value is negative"
(test (format #f "~NC" (ash 1 32) #\c) 'error) ; "~N value is too big"
(test (format #f "~,9223272036854775807F" pi) 'error) ; "precision is too big"

(test (format #f "~@F" 1.23) 'error)
(test (format #f "~{testing ~D ~C ~}" (list 0 #\( 1 #\) 2 #\* 3 #\+ 4 #\, 5 #\- 6 #\. 7 #\/ 8 #\0 9 #\1 10 #\2 11 #\3 12 #\4 13 #\5 14 #\6 15 #\7 16 #\8 17 #\9 18 #\: 19 #\; 20 #\< 21 #\= 22 #\> 23 #\? 24 #\@ 25 #\A 26 #\B 27 #\C 28 #\D 29 #\E 30 #\F 31 #\G 32 #\H 33 #\I 34 #\J 35 #\K 36 #\L 37 #\M 38 #\N 39 #\O 40 #\P 41 #\Q 42 #\R 43 #\S 44 #\T 45 #\U 46 #\V 47 #\W 48 #\X 49 #\Y 50 #\( 51 #\) 52 #\* 53 #\+ 54 #\, 55 #\- 56 #\. 57 #\/ 58 #\0 59 #\1 60 #\2 61 #\3 62 #\4 63 #\5 64 #\6 65 #\7 66 #\8 67 #\9 68 #\: 69 #\; 70 #\< 71 #\= 72 #\> 73 #\? 74 #\@ 75 #\A 76 #\B 77 #\C 78 #\D 79 #\E 80 #\F 81 #\G 82 #\H 83 #\I 84 #\J 85 #\K 86 #\L 87 #\M 88 #\N 89 #\O 90 #\P 91 #\Q 92 #\R 93 #\S 94 #\T 95 #\U 96 #\V 97 #\W 98 #\X 99 #\Y))
      "testing 0 ( testing 1 ) testing 2 * testing 3 + testing 4 , testing 5 - testing 6 . testing 7 / testing 8 0 testing 9 1 testing 10 2 testing 11 3 testing 12 4 testing 13 5 testing 14 6 testing 15 7 testing 16 8 testing 17 9 testing 18 : testing 19 ; testing 20 < testing 21 = testing 22 > testing 23 ? testing 24 @ testing 25 A testing 26 B testing 27 C testing 28 D testing 29 E testing 30 F testing 31 G testing 32 H testing 33 I testing 34 J testing 35 K testing 36 L testing 37 M testing 38 N testing 39 O testing 40 P testing 41 Q testing 42 R testing 43 S testing 44 T testing 45 U testing 46 V testing 47 W testing 48 X testing 49 Y testing 50 ( testing 51 ) testing 52 * testing 53 + testing 54 , testing 55 - testing 56 . testing 57 / testing 58 0 testing 59 1 testing 60 2 testing 61 3 testing 62 4 testing 63 5 testing 64 6 testing 65 7 testing 66 8 testing 67 9 testing 68 : testing 69 ; testing 70 < testing 71 = testing 72 > testing 73 ? testing 74 @ testing 75 A testing 76 B testing 77 C testing 78 D testing 79 E testing 80 F testing 81 G testing 82 H testing 83 I testing 84 J testing 85 K testing 86 L testing 87 M testing 88 N testing 89 O testing 90 P testing 91 Q testing 92 R testing 93 S testing 94 T testing 95 U testing 96 V testing 97 W testing 98 X testing 99 Y ")


(let ((vect1 #3d(((1 2 3) (3 4 5)) ((5 6 1) (7 8 2))))
      (vect2 #2d((1 2 3 4 5 6) (7 8 9 10 11 12)))
      (vect3 #(1 2 3 4 5 6 7 8 9 10 11 12 13 14))
      (vect4 #3d(((1 2) (3 4) (5 6)) ((7 8) (9 10) (11 12)))))
  (do ((i 0 (+ i 2)))
      ((>= i 10))
    (set! (*s7* 'print-length) i)
    (test (object->string vect1) (format #f "~A" vect1))
    (test (object->string vect2) (format #f "~A" vect2))
    (test (object->string vect3) (format #f "~A" vect3))
    (test (object->string vect4) (format #f "~A" vect4))))

(let-temporarily (((*s7* 'print-length) 0))
  (test (format #f "~A" #()) "#()")
  (test (format #f "~S" (list (cons 1 1))) "((...))"))

(let-temporarily (((*s7* 'print-length) 3))
  (let ((lst (list 1)))
    (set-car! lst lst)
    (let ((v (vector 1 1 1 1 1 1 1 1 1 lst)))
      (let ((str (format #f "~A" v)))
	(test (string=? str "#(1 1 1 ...)") #t)))))

(let ()
  (catch #t (lambda () (object->string (make-iterator "1234") :readable 3)) (lambda args #f))
  (test (format #f "~S" (list (cons 1 1))) "((1 . 1))")) ; check that bogus object->string truncation doesn't affect subsequent output

(test (format #f "~D" 123) "123")
(test (format #f "~X" 123) "7b")
(test (format #f "~B" 123) "1111011")
(test (format #f "~O" 123) "173")

(test (format #f "~10D" 123) "       123")
(test (format #f "~nD" 10 123) "       123")
(test (format #f "~10X" 123) "        7b")
(test (format #f "~10B" 123) "   1111011")
(test (format #f "~10O" 123) "       173")

(test (format #f "~D" -123) "-123")
(test (format #f "~X" -123) "-7b")
(test (format #f "~B" -123) "-1111011")
(test (format #f "~O" -123) "-173")

(test (format #f "~10D" -123) "      -123")
(test (format #f "~10X" -123) "       -7b")
(test (format #f "~10B" -123) "  -1111011")
(test (format #f "~10O" -123) "      -173")

(test (format #f "~d" 123) "123")
(test (format #f "~x" 123) "7b")
(test (format #f "~b" 123) "1111011")
(test (format #f "~o" 123) "173")

(test (format #f "~10d" 123) "       123")
(test (format #f "~10x" 123) "        7b")
(test (format #f "~10b" 123) "   1111011")
(test (format #f "~10o" 123) "       173")

(test (format #f "~d" -123) "-123")
(test (format #f "~x" -123) "-7b")
(test (format #f "~b" -123) "-1111011")
(test (format #f "~o" -123) "-173")

(test (format #f "~10d" -123) "      -123")
(test (format #f "~10x" -123) "       -7b")
(test (format #f "~10b" -123) "  -1111011")
(test (format #f "~10o" -123) "      -173")

(test (format #f "~D" most-positive-fixnum) "9223372036854775807")
(test (format #f "~D" (+ 1 most-negative-fixnum)) "-9223372036854775807")

(test (format #f "~X" most-positive-fixnum) "7fffffffffffffff")
(test (format #f "~X" (+ 1 most-negative-fixnum)) "-7fffffffffffffff")

(test (format #f "~O" most-positive-fixnum) "777777777777777777777")
(test (format #f "~O" (+ 1 most-negative-fixnum)) "-777777777777777777777")

(test (format #f "~B" most-positive-fixnum) "111111111111111111111111111111111111111111111111111111111111111")
(test (format #f "~B" (+ 1 most-negative-fixnum)) "-111111111111111111111111111111111111111111111111111111111111111")

(num-test (inexact->exact most-positive-fixnum) most-positive-fixnum)

(test (format #f "~0D" 123) "123")
(test (format #f "~0X" 123) "7b")
(test (format #f "~0B" 123) "1111011")
(test (format #f "~0O" 123) "173")

(test (format #f "" 1) 'error)
(test (format #f "hiho" 1) 'error)
(test (format #f "a~%" 1) 'error) ; some just ignore extra args

(for-each
 (lambda (arg)
   (let ((result (catch #t (lambda () (format arg "hiho")) (lambda args 'error))))
     (if (not (eq? result 'error))
	 (begin (display "(format ") (display arg) (display " \"hiho\")")
		(display " returned ") (display result)
		(display " but expected 'error")
		(newline)))))
 (list -1 #\a 1 #(1 2 3) 3.14 3/4 1.0+1.0i 'hi :hi #<eof> abs (lambda () 1) #(()) (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (let ((result (catch #t (lambda () (format #f arg)) (lambda args 'error))))
     (if (not (eq? result 'error))
	 (begin (display "(format #f ") (display arg) (display ")")
		(display " returned ") (display result)
		(display " but expected 'error")
		(newline)))))
 (list -1 #\a 1 #f #t #(1 2 3) 3.14 3/4 1.0+1.0i () 'hi abs (lambda () 1) #(()) (list 1 2 3) '(1 . 2)))

(test (format #f "hi ~A ho" 1 2) 'error)
(test (format #f "hi ~A ho") 'error)
(test (format #f "hi ~S ho") 'error)
(test (format #f "hi ~S ho" 1 2) 'error)
(test (format #f "~C" 1) 'error)
(test (format #f "123 ~R 321" 1) 'error)
(test (format #f "123 ~,3R 321" 1) 'error)
(test (format #f "~,2,3,4D" 123) 'error)

(test (format #f "hi ~Z ho") 'error)
(test (format #f "hi ~+ ho") 'error)
(test (format #f "hi ~# ho") 'error)
(test (format #f "hi ~, ho") 'error)

(test (format #f "hi ~} ho") 'error)
(test (format #f "hi {ho~}") 'error)

(test (format #f "asb~{~A asd" '(1 2 3)) 'error)
(test (format #f "~{~A~}" 1 2 3) 'error)
(test (format #f "asb~{~}asd" '(1 2 3)) 'error)
(test (format #f "asb~{ ~}asd" '(1 2 3)) 'error)
(test (format #f "asb~{ . ~}asd" '(1 2 3)) 'error)
(test (format #f "asb~{ hiho~~~}asd" '(1 2 3)) 'error)

(test (format #f "~12C" #\a) "aaaaaaaaaaaa")
(test (format #f ".~0C." #\a) "..")
(test (format #f "~10C" #\space) "          ")

(test (format #f "~12P" #\a) 'error)
(test (format #f "~12*" #\a) 'error)
(test (format #f "~12%" #\a) 'error)
(test (format #f "~12^" #\a) 'error)
(test (format #f "~12{" #\a) 'error)
(test (format #f "~12,2A" #\a) 'error)

(test (format #f "~12,A" #\a) 'error) ; s7 misses padding errors such as (format #f "~12,' A" #\a)

(for-each
 (lambda (arg)
   (test (format #f "~F" arg) 'error))
 (list "hi" #\a 'a-symbol (make-vector 3) abs #f #t :hi (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (format #f "~D" arg) 'error))
 (list "hi" #\a 'a-symbol (make-vector 3) abs #f #t :hi (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (format #f "~P" arg) 'error))
 (list "hi" #\a 'a-symbol (make-vector 3) abs #f #t :hi (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (format #f "~X" arg) 'error))
 (list "hi" #\a 'a-symbol (make-vector 3) abs #f #t :hi (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (format #f "~C" arg) 'error))
 (list "hi" 1 1.0 1+i 2/3 'a-symbol (make-vector 3) abs #f #t :hi (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (format #f arg 123) 'error))
 (list 1 1.0 1+i 2/3 'a-symbol (make-vector 3) abs #f #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (format #f "~{~A ~A ~}" '(1 "hi" 2)) 'error)
(for-each
 (lambda (arg)
   (let ((result (catch #t (lambda () (format #f "~F" arg)) (lambda args 'error))))
     (if (not (eq? result 'error))
	 (begin (display "(format #f \"~F\" ") (display arg)
		(display ") returned ") (display result)
		(display " but expected 'error")
		(newline)))))
 (list #\a #(1 2 3) "hi" () 'hi abs (lambda () 1) #(()) (list 1 2 3) '(1 . 2)))

(test (format #f "~D") 'error)
(test (format () "hi") #f) ; not an error now -- will print "hi" also
(test (format #f "~F" "hi") 'error)
(test (format #f "~D" #\x) 'error)
(test (format #f "~C" (list 1 2 3)) 'error)
(test (format #f "~1/4F" 1.4) 'error)
(test (format #f "~1.4F" 1.4) 'error)
(test (format #f "~F" (real-part (log 0.0))) "-inf.0")
(test (let ((val (format #f "~F" (/ (real-part (log 0.0)) (real-part (log 0.0)))))) (string=? val "+nan.0")) #t)
(test (format #f "~1/4T~A" 1) 'error)
(test (format #f "~T") "")
(test (format #f "~@P~S" 1 '(1)) "y(1)")
(test (format #f ".~A~*" 1 '(1)) ".1")
(test (format #f "~*~*~T" 1 '(1)) "")

(test (format #f "~A" 'AB\c) "(symbol \"AB\\\\c\")")
(test (format #f "~S" 'AB\c) "(symbol \"AB\\\\c\")")
(test (format #f "~A" '(AB\c () xyz)) "((symbol \"AB\\\\c\") () xyz)")
(test (format #f "~,2f" 1234567.1234) "1234567.12")
(test (format #f "~5D" 3) "    3")
(test (format #f "~5,'0D" 3) "00003")
(test (format #f "++~{-=~s=-~}++" (quote (1 2 3))) "++-=1=--=2=--=3=-++")

(test (format) 'error)
(for-each
 (lambda (arg)
   (test (format arg) 'error))
 (list 1 1.0 1+i 2/3 'a-symbol (make-vector 3) '(1 2) (cons 1 2) abs #f #t :hi (if #f #f) (lambda (a) (+ a 1))))
(test (format "hi") 'error)
(test (format "~A ~D" 1/3 2) 'error)
(test (format "") 'error)

(test (format #f "~:D" 23) "23rd")
(test (format #f "~:D" 101) "101st")
(test (format #f "~:D" 11) "11th")
(test (format #f "~:D" 1) "first")
(test (format #f "~:D" 8) "eighth")
(test (format #f "~:D" 42) "42nd")
(test (format #f "~:D" 2) "second")
(test (format #f "~:D" -2) 'error)
(test (format #f "~:D" #()) 'error)
(test (format #f "~:F" pi) 'error)
(test (format #f "~:" pi) 'error)

;; from slib/formatst.scm
(test (string=? (format #f "abc") "abc") #t)
(test (string=? (format #f "~a" 10) "10") #t)
(test (string=? (format #f "~a" -1.2) "-1.2") #t)
(test (string=? (format #f "~a" 'a) "a") #t)
(test (string=? (format #f "~a" #t) "#t") #t)
(test (string=? (format #f "~a" #f) "#f") #t)
(test (string=? (format #f "~a" "abc") "abc") #t)
(test (string=? (format #f "~a" #(1 2 3)) "#(1 2 3)") #t)
(test (string=? (format #f "~a" ()) "()") #t)
(test (string=? (format #f "~a" '(a)) "(a)") #t)
(test (string=? (format #f "~a" '(a b)) "(a b)") #t)
(test (string=? (format #f "~a" '(a (b c) d)) "(a (b c) d)") #t)
(test (string=? (format #f "~a" '(a . b)) "(a . b)") #t)
(test (string=? (format #f "~a ~a" 10 20) "10 20") #t)
(test (string=? (format #f "~a abc ~a def" 10 20) "10 abc 20 def") #t)
(test (string=? (format #f "~d" 100) "100") #t)
(test (string=? (format #f "~x" 100) "64") #t)
(test (string=? (format #f "~o" 100) "144") #t)
(test (string=? (format #f "~b" 100) "1100100") #t)
(test (string=? (format #f "~10d" 100) "       100") #t)
(test (string=? (format #f "~10,'*d" 100) "*******100") #t)
(test (string=? (format #f "~c" #\a) "a") #t)
(test (string=? (format #f "~c" #\space) " ") #t)
(test (string=? (format #f "~C" #\x91) "\x91;") #t)
(test (string=? (format #f "~C" #\x9) "\x09;") #t)
(test (string=? (format #f "~C" #\~) "~") #t)
(test (string=? (format #f "~A" #\x91) "\x91;") #t)
(test (string=? (format #f "~S" #\x91) "#\\x91") #t)
(test (string=? (format #f "~A" (string->symbol "hi")) "hi") #t)
(test (string=? (format #f "~S" (string->symbol "hi")) "hi") #t)
(test (string=? (format #f "~A" (string->symbol ";\\\";")) "(symbol \";\\\\\\\";\")") #t)
(test (string=? (format #f "~S" (string->symbol ";\\\";")) "(symbol \";\\\\\\\";\")") #t)
(test (string=? (format #f "~A" (string->symbol (string #\, #\. #\# #\; #\" #\\ #\' #\`))) "(symbol \",.#;\\\"\\\\'`\")") #t)

(test (string=? (format #f "~~~~") "~~") #t)
(test (string=? (format #f "~s" "abc") "\"abc\"") #t)
(test (string=? (format #f "~s" "abc \\ abc") "\"abc \\\\ abc\"") #t)
(test (string=? (format #f "~a" "abc \\ abc") "abc \\ abc") #t)
(test (string=? (format #f "~s" "abc \" abc") "\"abc \\\" abc\"") #t)
(test (string=? (format #f "~a" "abc \" abc") "abc \" abc") #t)
(test (string=? (format #f "~s" #\space) "#\\space") #t)
(test (string=? (format #f "~s" #\newline) "#\\newline") #t)
(test (string=? (format #f "~s" #\a) "#\\a") #t)
(test (string=? (format #f "~a" '(a "b" c)) "(a \"b\" c)") #t)
(test (string=? (format #f "abc~
         123") "abc123") #t)
(test (string=? (format #f "abc~
123") "abc123") #t)
(test (string=? (format #f "abc~
") "abc") #t)
(test (string=? (format #f "~{ ~a ~}" '(a b c)) " a  b  c ") #t)
(test (string=? (format #f "~{ ~a ~}" ()) "") #t)
(test (string=? (format #f "~{ ~a ~}" "") "") #t)
(test (string=? (format #f "~{ ~a ~}" #()) "") #t)
(test (string=? (format #f "~{ ~a,~a ~}" '(a 1 b 2 c 3)) " a,1  b,2  c,3 ") #t)
(test (string=? (format #f "abc ~^ xyz") "abc ") #t)
(test (format (values #f "~A ~D" 1 2)) "1 2")
(test (format #f "~A~^" 1) "1") ; clisp agrees here
(test (format #f "~A~*~* ~A" (values 1 2 3 4)) "1 4")
(test (format #f "~^~A~^~*~*~^ ~^~A~^" (values 1 2 3 4)) "1 4")

(test (string=? (format #f "~B" 123) "1111011") #t)
(test (string=? (format #f "~B" 123/25) "1111011/11001") #t)
(test (string=? (format #f "~B" 123.25) "1111011.01") #t)
(test (string=? (format #f "~B" 123+i) "1111011.0+1.0i") #t)

(test (string=? (format #f "~D" 123) "123") #t)
(test (string=? (format #f "~D" 123/25) "123/25") #t)

(test (string=? (format #f "~O" 123) "173") #t)
(test (string=? (format #f "~O" 123/25) "173/31") #t)
(test (string=? (format #f "~O" 123.25) "173.2") #t)
(test (string=? (format #f "~O" 123+i) "173.0+1.0i") #t)

(test (string=? (format #f "~X" 123) "7b") #t)
(test (string=? (format #f "~X" 123/25) "7b/19") #t)
(test (string=? (format #f "~X" 123.25) "7b.4") #t)
(test (string=? (format #f "~X" 123+i) "7b.0+1.0i") #t)

(test (string=? (format #f "~A" "hi") (format #f "~S" "hi")) #f)
(test (string=? (format #f "~A" #\a) (format #f "~S" #\a)) #f)
(for-each
 (lambda (arg)
   (test (string=? (format #f "~A" arg) (format #f "~S" arg)) #t))
 (list 1 1.0 #(1 2 3) '(1 2 3) '(1 . 2) () #f #t abs #<eof> #<unspecified> 'hi '\a))
(test (length (format #f "~S" (string #\\))) 4)                  ; "\"\\\\\""
(test (length (format #f "~S" (string #\a))) 3)                  ; "\"a\""
(test (length (format #f "~S" (string #\null))) 7)               ; "\"\\x00;\""
(test (length (format #f "~S" (string (integer->char #xf0)))) 3) ; "\"ð\""
(test (length (format #f "~S" (string #\"))) 4)                  ; "\""

(test (format #f "~F" 3.0) "3.000000")
(test (format #f "~G" 3.0) "3.0")
(test (format #f "~E" 3.0) (if (not with-windows) "3.000000e+00" "3.000000e+000"))
(test (format #f "~F" 3.14159) "3.141590")
(test (format #f "~G" 3.14159) "3.14159")
(test (format #f "~E" 3.14159) (if (not with-windows) "3.141590e+00" "3.141590e+000"))
(test (format #f "~,2F" 3.14159) "3.14")
(test (format #f "~,2G" 3.14159) "3.1")
(test (format #f "~,2E" 3.14159) (if (not with-windows) "3.14e+00" "3.14e+000"))
(test (format #f "~12F" 3.14159) "    3.141590")
(test (format #f "~12G" 3.14159) "     3.14159")
(test (format #f "~12E" 3.14159) (if (not with-windows) "3.141590e+00" "3.141590e+000"))
(test (format #f "~12,3F" 3.14159) "       3.142")
(test (format #f "~n,nF" 12 3 3.14159) "       3.142")
(test (format #f "~12,nF" 3 3.14159) "       3.142")
(test (format #f "~12,3G" 3.14159) "        3.14")
(test (format #f "~12,3E" 3.14159) (if (not with-windows) "   3.142e+00" "  3.142e+000"))
(test (format #f "~12,'xD" 1) "xxxxxxxxxxx1")
(test (format #f "~12,'xF" 3.14159) "xxxx3.141590")
(test (format #f "~12,'xG" 3.14159) "xxxxx3.14159")
(test (format #f "~12,'xE" 3.14159) (if (not with-windows) "3.141590e+00" "3.141590e+000"))
(test (format #f "~12,'\\F" 3.14159) "\\\\\\\\3.141590")
(test (format #f "~20,20G" 3.0) "                   3.0")
(test (format #f "~20,20F" 3.0) "3.00000000000000000000")
(test (format #f "~20,20E" 3.0) (if (not with-windows) "3.00000000000000000000e+00" "3.00000000000000000000e+000"))

(test (format #f "~,3B" 0.99999) "0.111")
(test (format #f "~,3O" 0.99999) "0.777")
(test (format #f "~,3F" 0.99999) "1.000")
(test (format #f "~,3X" 0.99999) "0.fff")

(test (format #f "~-2F" 0.0) 'error)
(test (format #f "~,-2F" 0.0) 'error)
(test (format #f "~2/3F" 0.0) 'error)
(test (format #f "~2.3F" 0.0) 'error)
(test (format #f "~2,1,3,4F" 0.0) 'error)
(test (format #f "~'xF" 0.0) 'error)
(test (format #f "~3,3" pi) 'error)
(test (format #f "~3," pi) 'error)
(test (format #f "~3" pi) 'error)
(test (format #f "~," pi) 'error)
(test (format #f "~'," pi) 'error)
(test (format #f "~'" pi) 'error)

(test (format #f "~*" 1.0) "")
(test (format #f "~D" 1.0) (if (not with-windows) "1.000000e+00" "1.000000e+000"))
(test (format #f "~O" 1.0) "1.0")
(test (format #f "~P" 1.0) "")
(test (format #f "~P" '(1 2 3)) 'error)
(test (format #f "~\x00;T") 'error)
(test (format #f "~9,'(T") "((((((((")
(test (format #f "~0F" 1+1i) "1.000000+1.000000i")
(test (format #f "~9F" 1) "        1")
(test (format #f "~,0F" 3.14) "3.0")
(test (format #f "~,0F" 1+1i) "1.0+1.0i")
(test (format #f "~,0X" 1+1i) "1.0+1.0i")
(test (format #f "~,9g" 1+1i) "1.0+1.0i")
(test (format #f "~,1e" 3.14) (if (not with-windows) "3.1e+00" "3.1e+000"))
(test (format #f "~9,0F" 3.14) "        3.0")
(test (format #f "~9,1F" 3.14) "      3.1")
(test (format #f "~9,2F" 3.14) "     3.14")
(test (format #f "~9,3F" 3.14) "    3.140")
(test (format #f "~9,4F" 3.14) "   3.1400")
(test (format #f "~n,4F" 9 3.14) "   3.1400")
(test (format #f "~9,nF" 4 3.14) "   3.1400")
(test (format #f "~n,nF" 9 4 3.14) "   3.1400")
(test (format #f "~9,5F" 3.14) "  3.14000")
(test (format #f "~9,6F" 3.14) " 3.140000")
(test (format #f "~9,7F" 3.14) "3.1400000")
(test (format #f "~9,8F" 3.14) "3.14000000")
(test (format #f "~9,9F" 3.14) "3.140000000")
(test (format #f "~9,9G" 1+1i) " 1.0+1.0i")
(if (not with-windows)
    (begin
      (test (format #f "~9,0e" 1+1i) "1e+00+1e+00i")
      (test (format #f "~9,1e" 1+1i) "1.0e+00+1.0e+00i")
      (test (format #f "~9,2e" 1+1i) "1.00e+00+1.00e+00i")
      (test (format #f "~9,3e" 1+1i) "1.000e+00+1.000e+00i")
      (test (format #f "~9,4e" 1+1i) "1.0000e+00+1.0000e+00i")
      (test (format #f "~9,5e" 1+1i) "1.00000e+00+1.00000e+00i")
      (test (format #f "~9,6e" 1+1i) "1.000000e+00+1.000000e+00i")
      (test (format #f "~9,7e" 1+1i) "1.0000000e+00+1.0000000e+00i")
      (test (format #f "~9,8e" 1+1i) "1.00000000e+00+1.00000000e+00i")
      (test (format #f "~9,9e" 1+1i) "1.000000000e+00+1.000000000e+00i"))
    (begin
      (test (format #f "~9,0e" 1+1i) "1e+000+1e+000i")
      (test (format #f "~9,1e" 1+1i) "1.0e+000+1.0e+000i")
      (test (format #f "~9,2e" 1+1i) "1.00e+000+1.00e+000i")
      (test (format #f "~9,3e" 1+1i) "1.000e+000+1.000e+000i")
      (test (format #f "~9,4e" 1+1i) "1.0000e+000+1.0000e+000i")
      (test (format #f "~9,5e" 1+1i) "1.00000e+000+1.00000e+000i")
      (test (format #f "~9,6e" 1+1i) "1.000000e+000+1.000000e+000i")
      (test (format #f "~9,7e" 1+1i) "1.0000000e+000+1.0000000e+000i")
      (test (format #f "~9,8e" 1+1i) "1.00000000e+000+1.00000000e+000i")
      (test (format #f "~9,9e" 1+1i) "1.000000000e+000+1.000000000e+000i")))
(test (format #f "~9,0x" 3.14) "      3.0")
(test (format #f "~9,1x" 3.14) "      3.2")
(test (format #f "~9,2x" 3.14) "     3.23")
(test (format #f "~9,3x" 3.14) "    3.23d")
(test (format #f "~9,4x" 3.14) "   3.23d7")
(test (format #f "~9,5x" 3.14) "   3.23d7")
(test (format #f "~9,6x" 3.14) " 3.23d70a")
(test (format #f "~9,7x" 3.14) "3.23d70a3")
(test (format #f "~9,8x" 3.14) "3.23d70a3d")
(test (format #f "~9,9x" 3.14) "3.23d70a3d7")
(test (format #f "~9,0b" 3.14) "     11.0")
(test (format #f "~9,1b" 3.14) "     11.0")
(test (format #f "~9,2b" 3.14) "     11.0")
(test (format #f "~9,3b" 3.14) "   11.001")
(test (format #f "~9,4b" 3.14) "   11.001")
(test (format #f "~9,5b" 3.14) "   11.001")
(test (format #f "~9,6b" 3.14) "   11.001")
(test (format #f "~9,7b" 3.14) "11.0010001")
(test (format #f "~9,8b" 3.14) "11.00100011")
(test (format #f "~9,9b" 3.14) "11.001000111")
(test (format #f "~0,'xf" 1) "1")
(test (format #f "~1,'xf" 1) "1")
(test (format #f "~2,'xf" 1) "x1")
(test (format #f "~3,'xf" 1) "xx1")
(test (format #f "~4,'xf" 1) "xxx1")
(test (format #f "~5,'xf" 1) "xxxx1")
(test (format #f "~6,'xf" 1) "xxxxx1")
(test (format #f "~7,'xf" 1) "xxxxxx1")
(test (format #f "~8,'xf" 1) "xxxxxxx1")
(test (format #f "~9,'xf" 1) "xxxxxxxx1")
(test (format #f "~11,'xf" 3.14) "xxx3.140000")
(test (format #f "~12,'xf" 3.14) "xxxx3.140000")
(test (format #f "~13,'xf" 3.14) "xxxxx3.140000")
(test (format #f "~14,'xf" 3.14) "xxxxxx3.140000")
(test (format #f "~15,'xf" 3.14) "xxxxxxx3.140000")
(test (format #f "~16,'xf" 3.14) "xxxxxxxx3.140000")
(test (format #f "~17,'xf" 3.14) "xxxxxxxxx3.140000")
(test (format #f "~18,'xf" 3.14) "xxxxxxxxxx3.140000")
(test (format #f "~19,'xf" 3.14) "xxxxxxxxxxx3.140000")
(test (format #f "~,f" 1.0) "1.000000")
(test (format #f "~,,f" 1.0) 'error)
(test (format #f "~p" '(1 2 3)) 'error) ; these are not errors in CL
(test (format #f "~p" #(())) 'error)
(test (format #f "~p" 'hi) 'error)
(test (format #f "~p" abs) 'error)
(test (format #f "~p" 1+i) 'error)
(test (format #f "~@p" '(1 2 3)) 'error)
(test (format #f "~@p" #(())) 'error)
(test (format #f "~@p" 'hi) 'error)
(test (format #f "~@p" abs) 'error)
(let-temporarily (((*s7* 'print-length) 3))
  (test (format #f "~{~A~| ~}" '(1 2 3 4 5 6)) "1 2 3 ...")
  (test (format #f "~{~A~| ~}" #(1 2 3 4 5 6)) "1 2 3 ...")
  (test (format #f "~{~A~| ~}" #(1 2)) "1 2")
  (test (format #f "~{~A~| ~}" #(1 2 3)) "1 2 3")
  (test (format #f "~{~A~| ~}" #(1 2 3 4)) "1 2 3 ...")
  (test (format #f "~{~A~| ~}" (inlet 'a 1 'b 2 'c 3 'd 4 'e 5)) "(a . 1) (b . 2) (c . 3) ...")
  (test (format #f "~{~{~A~| ~}~}" '((1 2 3 4 5 6))) "1 2 3 ...")
  (test (format #f "~{~{~A~| ~}~|~}" '((1 2) (3 4 5 6 7 8) (15) (16) ())) "1 23 4 5 ...15 ...")
  (test (format #f "~{~|~|~|~A ~}" '(1 2 3 4 5)) "1 2 3  ...")
  (test (format #f "~{~C~| ~}" "1234567") "1 2 3 ...")
  (test (format #f "~{~{~A~|~} ~}" '((1 2) (3 4))) "12 34 ")
  (test (format #f "~C ~^" #\a) "a ")
  (test (format #f "~{~{~{~A~| ~}~| ~}~}" '(((1 2) (3 4)))) "1 2 3 4")
  (test (format #f "~{~{~{~A~| ~}~| ~}~}" '((#(1 2) #(3 4)))) "1 2 3 4")
  (test (format #f "~{~{~{~A~| ~}~| ~}~}" #(((1 2) (3 4)))) "1 2 3 4")
  (test (format #f "~{~{~{~A~| ~}~| ~}~}" #(#((1 2) (3 4)))) "1 2 3 4")
  (test (format #f "~{~{~C~| ~}~| ~}" (list "hiho" "xxx")) "h i h ... x x x"))
(test (format #f "~{~{~A~^~} ~}" '((hi 1))) "hi1 ")
(test (format #f "~{~{~A~^~} ~}" '((1 2) (3 4))) "12 34 ")
(test (format #f "~{~{~A~} ~}" '((1 2) (3 4))) "12 34 ")
(test (format #f "~{~{~A~} ~}" '(())) " ")
(test (format #f "~{~{~A~} ~}" '((()))) "() ")
(test (format #f "~{~{~F~} ~}" '(())) " ")
(test (format #f "~{~{~C~} ~}" '(())) " ")
(test (format #f "~{~C ~}" ()) "")
(test (format #f "~C ~^" #\a) "a ") ; CL ignores pointless ~^
(test (format #f "~^~A" #f) "#f")
(test (format #f "~^~^~A" #f) "#f")
(test (format #f "~*~*~A~*" 1 2 3 4) "3")
(test (format #f "~{~*~A~}" '(1 2 3 4)) "24")
(test (let ((lst (list 1 2 3))) (set! (cdr (cddr lst)) lst) (format #f "~A" lst)) "#1=(1 2 3 . #1#)")
(test (let ((lst (list 1 2 3))) (set! (cdr (cddr lst)) lst) (format #f "~{~A~}" lst)) "12312")
(test (format #f "~{~A~}" (cons 1 2)) 'error)
(test (format #f "~{~A~}" '(1 2 3 . 4)) 'error)
(test (format #f "~20,vF" 3.14) 'error)
(test (format #f "~{~C~^ ~}" "hiho") "h i h o")
(test (format #f "~{~{~C~^ ~}~}" (list "hiho")) "h i h o")
(test (format #f "~{~A ~}" #(1 2 3 4)) "1 2 3 4 ")
(test (let ((v (vector 1))) (set! (v 0) v) (format #f "~A" v)) "#1=#(#1#)")
(test (let ((v (vector 1))) (set! (v 0) v) (format #f "~{~A~}" v)) "#1=#(#1#)")
(test (format #f "~{~{~{~A~^ ~}~^ ~}~}" '(((1 2) (3 4)))) "1 2 3 4")
(test (format #f "~{~{~{~A~^ ~}~^ ~}~}" '((#(1 2) #(3 4)))) "1 2 3 4")
(test (format #f "~{~{~{~A~^ ~}~^ ~}~}" #(((1 2) (3 4)))) "1 2 3 4")
(test (format #f "~{~{~{~A~^ ~}~^ ~}~}" #(#((1 2) (3 4)))) "1 2 3 4")
(test (format #f "~{~{~{~A~^ ~}~^ ~}~}" #(#(#(1 2) (3 4)))) "1 2 3 4")
(test (format #f "~{~{~{~A~^ ~}~^ ~}~}" #(#(#(1 2) #(3 4)))) "1 2 3 4")
(test (format #f "~{~{~C~^ ~}~^ ~}" (list "hiho" "xxx")) "h i h o x x x")
(test (format #f "~{~{~A~}~}" '((1 . 2) (3 . 4))) 'error)
(test (format #f "~{~A~^ ~}" '((1 . 2) (3 . 4))) "(1 . 2) (3 . 4)")
(test (format #f "~{~A ~}" (hash-table)) "")
(test (format #f "~{~^~S ~}" (make-iterator '(1 2 3))) "1 2 3 ")
(test (format #f "~{~^~S ~}" (make-iterator (let ((lst (list 1))) (set-cdr! lst lst)))) "1 ")
(test (format #f "~{~^~S ~}" (make-iterator "")) "")
(test (format #f "~{~^~S ~}" (make-iterator #(1 2 3))) "1 2 3 ")

(test (format #f "~10,'-T") "---------")
(test (format #f "~10,'\\T") "\\\\\\\\\\\\\\\\\\")
(test (format #f "~10,'\"T") "\"\"\"\"\"\"\"\"\"")
(test (format #f "~10,'-T12345~20,'-T") "---------12345-----")
(test (format #f "~10,')T") ")))))))))")

(test (format #f "~,0F" 1.4) "1.0")
(test (format #f "~,0F" 1.5) "2.0")
(test (format #f "~,0F" 1.6) "2.0")
(test (format #f "~,0F" 0.4) "0.0")
(test (format #f "~,0F" 0.5) (if (not with-windows) "0.0" "1.0")) ; !!
(test (format #f "~,0F" 0.6) "1.0")
(test (format #f "~,-0F" 1.4) 'error)
(test (format #f "~, 0F" 1.4) 'error)
(test (format #f "~*1~*" 1) 'error)
(test (format #f "~*1~A" 1) 'error)
(test (format #f #u()) 'error)
(test (format #f #u(65 90)) 'error)

;; optimizer bug
(test (let () (define (func) (format `((x)) "")) (define (hi) (func)) (hi)) 'error)
(test (let () (define (func) (format (make-iterator #(10 20)) #u())) (define (hi) (func)) (hi)) 'error)
(test (let () (define (func) (format (list 1 2) "")) (define (hi) (func)) (hi)) 'error)
(test (format :rest "") 'error)

(let* ((str1 #t) (str2 (with-output-to-string (lambda () (set! str1 (format () "~D" 1)))))) (test (and (not str1) (equal? str2 "1")) #t))

(test (format #f "~,'") 'error)
(if with-bignums
    (begin
      (test (format #f "~F" 1e300) "9.999999999999999999999999999999999999987E299")
      (test (format #f "~F" 1e308) "9.999999999999999999999999999999999999982E307")
      (test (format #f "~G" 1e308) "9.999999999999999999999999999999999999982E307")
      (test (format #f "~E" 1e308)  "9.999999999999999999999999999999999999982E307")
      (test (format #f "~E" 1e308+1e308i) "9.999999999999999999999999999999999999982E307+9.999999999999999999999999999999999999982E307i")
      (test  (format #f "~F" 1e308+1e308i) "9.999999999999999999999999999999999999982E307+9.999999999999999999999999999999999999982E307i")
      (test (format #f "~F" -1e308-1e308i) "-9.999999999999999999999999999999999999982E307-9.999999999999999999999999999999999999982E307i")
      (test (format #f "~,32f" (/ 1.0 most-positive-fixnum)) "1.084202172485504434125002235952170462235E-19")
      (test (format #f "~{~^~f ~}" (vector 1e308)) "9.999999999999999999999999999999999999982E307 ")
      (test (object->string (vector 1e308)) "#(9.999999999999999999999999999999999999982E307)"))
    (begin
      (test (format #f "~F" 1e300) "1000000000000000052504760255204420248704468581108159154915854115511802457988908195786371375080447864043704443832883878176942523235360430575644792184786706982848387200926575803737830233794788090059368953234970799945081119038967640880074652742780142494579258788820056842838115669472196386865459400540160.000000")
      (test (format #f "~F" 1e308) "100000000000000001097906362944045541740492309677311846336810682903157585404911491537163328978494688899061249669721172515611590283743140088328307009198146046031271664502933027185697489699588559043338384466165001178426897626212945177628091195786707458122783970171784415105291802893207873272974885715430223118336.000000")
      (test (format #f "~G" 1e308) "1e+308")
      (test (format #f "~E" 1e308) "1.000000e+308")
      (test (format #f "~E" 1e308+1e308i) "1.000000e+308+1.000000e+308i")
      (test  (format #f "~F" 1e308+1e308i) "100000000000000001097906362944045541740492309677311846336810682903157585404911491537163328978494688899061249669721172515611590283743140088328307009198146046031271664502933027185697489699588559043338384466165001178426897626212945177628091195786707458122783970171784415105291802893207873272974885715430223118336.000000+100000000000000001097906362944045541740492309677311846336810682903157585404911491537163328978494688899061249669721172515611590283743140088328307009198146046031271664502933027185697489699588559043338384466165001178426897626212945177628091195786707458122783970171784415105291802893207873272974885715430223118336.000000i")
      (test (format #f "~F" -1e308-1e308i) "-100000000000000001097906362944045541740492309677311846336810682903157585404911491537163328978494688899061249669721172515611590283743140088328307009198146046031271664502933027185697489699588559043338384466165001178426897626212945177628091195786707458122783970171784415105291802893207873272974885715430223118336.000000-100000000000000001097906362944045541740492309677311846336810682903157585404911491537163328978494688899061249669721172515611590283743140088328307009198146046031271664502933027185697489699588559043338384466165001178426897626212945177628091195786707458122783970171784415105291802893207873272974885715430223118336.000000i")
      (test (format #f "~,32f" (/ 1.0 most-positive-fixnum)) "0.00000000000000000010842021724855")
      (test (format #f "~{~^~f ~}" (vector 1e308)) "100000000000000001097906362944045541740492309677311846336810682903157585404911491537163328978494688899061249669721172515611590283743140088328307009198146046031271664502933027185697489699588559043338384466165001178426897626212945177628091195786707458122783970171784415105291802893207873272974885715430223118336.000000 ")
      (test (object->string (vector 1e308)) "#(1e+308)")))


(when full-s7test
  (let ()
    (define ctrl-chars (vector ;#\A #\S #\C #\F #\E #\G #\O #\D #\B #\X #\W
			#\, #\{ #\} #\@ #\P #\* #\< #\>
			#\a #\s #\c #\f #\e #\g #\o #\d #\b #\x #\p #\n #\w
			#\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9
			#\~ #\T #\& #\% #\^ #\|
			#\~ #\~ #\~ #\~
			#\, #\, #\, #\, #\" #\" #\\ #\'
			#\+ #\- #\@ #\. #\/ #\; #\:
			))
    (define ctrl-chars-len (length ctrl-chars))

    (define (test-chars)
      (do ((size 1 (+ size 1)))
	  ((= size 7))
	(let ((tries (* size size 10000)))
	  (format *stderr* "~D " size)
	  (let ((ctrl-str (make-string (+ size 1)))
		(x 12)
		(y '(1 2))
		(z #\a))
	    (string-set! ctrl-str 0 #\~)
	    (do ((i 0 (+ i 1)))
		((= i tries))
	      (do ((j 1 (+ j 1)))
		  ((> j size))
		(string-set! ctrl-str j (vector-ref ctrl-chars (random ctrl-chars-len))))
					;(format *stderr* "~S " ctrl-str)
					;(catch #t (lambda () (format *stderr* "~S: ~A~%" ctrl-str (format #f ctrl-str))) (lambda arg 'error))
					;(catch #t (lambda () (format *stderr* "~S ~A: ~A~%" ctrl-str x (format #f ctrl-str x))) (lambda arg 'error))
					;(catch #t (lambda () (format *stderr* "~S ~A: ~A~%" ctrl-str y (format #f ctrl-str y))) (lambda arg 'error))
					;(catch #t (lambda () (format *stderr* "~S ~A: ~A~%" ctrl-str z (format #f ctrl-str z))) (lambda arg 'error)))))
	      (catch #t (lambda () (format #f ctrl-str)) (lambda arg 'error))
	      (catch #t (lambda () (format #f ctrl-str x)) (lambda arg 'error))
	      (catch #t (lambda () (format #f ctrl-str y)) (lambda arg 'error))
	      (catch #t (lambda () (format #f ctrl-str z)) (lambda arg 'error))
	      (catch #t (lambda () (format #f ctrl-str x x)) (lambda arg 'error))
	      (catch #t (lambda () (format #f ctrl-str x y)) (lambda arg 'error))
	      (catch #t (lambda () (format #f ctrl-str y z)) (lambda arg 'error))
	      (catch #t (lambda () (format #f ctrl-str x y z)) (lambda arg 'error)))))
	))
    (test-chars)))


(test (reverse (format #f "~{~A~}" '((1 2) (3 4)))) ")4 3()2 1(")
(test (string->symbol (format #f "~A" '(1 2))) (symbol "(1 2)"))

(test (string->number (format #f "~A" -1)) -1)
(test (string->number (format #f "~S" -1)) -1)
(test (string->number (format #f "~F" -1)) -1)
(test (string->number (format #f "~D" -1)) -1)
(test (string->number (format #f "~G" -1)) -1)
(test (string->number (format #f "~E" -1)) -1)
(test (string->number (format #f "~B" -1)) -1)
(test (string->number (format #f "~X" -1)) -1)
(test (string->number (format #f "~O" -1)) -1)
(num-test (string->number (format #f "~A" 1.5)) 1.5)
(num-test (string->number (format #f "~S" 1.5)) 1.5)
(num-test (string->number (format #f "~F" 1.5)) 1.5)
(num-test (string->number (format #f "~D" 1.5)) 1.5)
(num-test (string->number (format #f "~G" 1.5)) 1.5)
(num-test (string->number (format #f "~E" 1.5)) 1.5)
(num-test (string->number (format #f "~B" 1.5)) 1.1)
(num-test (string->number (format #f "~X" 1.5)) 1.8)
(num-test (string->number (format #f "~O" 1.5)) 1.4)
(num-test (string->number (format #f "~A" 1+1i)) 1+1i)
(num-test (string->number (format #f "~S" 1+1i)) 1+1i)
(num-test (string->number (format #f "~F" 1+1i)) 1+1i)
(num-test (string->number (format #f "~D" 1+1i)) 1+1i)
(num-test (string->number (format #f "~G" 1+1i)) 1+1i)
(num-test (string->number (format #f "~E" 1+1i)) 1+1i)
(num-test (string->number (format #f "~B" 1+1i)) 1+1i)
(num-test (string->number (format #f "~X" 1+1i)) 1+1i)
(num-test (string->number (format #f "~O" 1+1i)) 1+1i)
(test (string->number (format #f "~A" 3/4)) 3/4)
(test (string->number (format #f "~S" 3/4)) 3/4)
(test (string->number (format #f "~F" 3/4)) 3/4)
(test (string->number (format #f "~D" 3/4)) 3/4)
(test (string->number (format #f "~G" 3/4)) 3/4)
(test (string->number (format #f "~E" 3/4)) 3/4)
(test (string->number (format #f "~B" 3/4)) 11/100)
(test (string->number (format #f "~X" 3/4)) 3/4)
(test (string->number (format #f "~O" 3/4)) 3/4)
(num-test (string->number (format #f "~A" 0+1i)) 0+1i)
(num-test (string->number (format #f "~S" 0+1i)) 0+1i)
(num-test (string->number (format #f "~F" 0+1i)) 0+1i)
(num-test (string->number (format #f "~D" 0+1i)) 0+1i)
(num-test (string->number (format #f "~G" 0+1i)) 0+1i)
(num-test (string->number (format #f "~E" 0+1i)) 0+1i)
(num-test (string->number (format #f "~B" 0+1i)) 0+1i)
(num-test (string->number (format #f "~X" 0+1i)) 0+1i)
(num-test (string->number (format #f "~O" 0+1i)) 0+1i)

(test (format #f "~P{T}'" 1) "{T}'")
(test (format #f "~") 'error)
(test (format #f "~B&B~X" 1.5 1.5) "1.1&B1.8")
(test (format #f ",~~~A~*1" 1 1) ",~11")
(test (format #f "~D~20B" 0 0) "0                   0")
(test (format #f "~D~20B" 1 1) "1                   1")
(test (format #f "~10B" 1) "         1")
(test (format #f "~10B" 0) "         0")
(test (format #f "~100B" 1) "                                                                                                   1")
(test (length (format #f "~1000B" 1)) 1000)
(test (format #f "~D~20D" 3/4 3/4) "3/4                 3/4")
(test (length (format #f "~20D" 3/4)) 20)
(test (format #f "~20B" 3/4) "              11/100")
(test (length (format #f "~20B" 3/4)) 20)
(test (format #f "~D~20B" 3/4 3/4) "3/4              11/100")
(test (format #f "~X~20X" 21/33 21/33) "7/b                 7/b")
(test (format #f "~D~20,'.B" 3/4 3/4) "3/4..............11/100")
(test (format #f "~20g" 1+i) "            1.0+1.0i")
(test (length (format #f "~20g" 1+i)) 20)
(test (format #f "~20f" 1+i) "  1.000000+1.000000i")
(test (length (format #f "~20f" 1+i)) 20)
(test (format #f "~20x" 17+23i) "          11.0+17.0i")
(test (length (format #f "~20x" 17+23i)) 20)

(test (format #f "~{~{~A~^~} ~}" (hash-table '(a . 1) '((b . 2)))) "(a . 1)(b . 2) ")
(test (format #f "~{~{~A~^~}~^~}" (hash-table '(a . 1) '((b . 2)))) "(a . 1)(b . 2)")
(test (format #f "~{~{~A~^ ~}~^~}" (hash-table '(a . 1) '((b . 2)))) "(a . 1) (b . 2)")
(test (format #f "~{~{~{~A~^~} ~}~}" #(())) "")
(test (format #f "~{~{~{~P~^~} ~}~}" '((()))) " ")
(test (format #f "~{~{~{~P~^~}~}~}" '(((2 3 4)))) "sss")
(test (apply format #f "~T~~{~{~{~*~~0~1~*~}~@~}" '(())) "~{")
(test (format #f "~{~S}%~}" '(a b c)) "a}%b}%c}%")
(test (format #f "~&~^%~F." 0) "%0.")
(test (format #f "1~^2") "1")
(test (apply format #f "~P~d~B~~" '(1 2 3)) "211~")
(test (format #f "~T1~~^~P" 0) "1~^s")
(test (format #f "~S~^~{~^" '(+ x 1)) "(+ x 1)")
(test (format #f "1~^~{2") "1")
(test (format #f "~A~{~0~g~@~B~}" () ()) "()")
(test (format #f "1~^~^~^2") "1")
(test (format #f "~{~{~~}~~,~}~*" '(()) '(())) "~,")
(test (format #f "~~S~S~T~~C~g~~" 0 0) "~S0~C0~")
(test (format #f "~{~~e~}~~{~*~~" "" "") "~{~")

(let ()
  (define* (clean-string e (precision 3))
    (format #f (format #f "(~~{~~,~DF~~^ ~~})" precision) e))
  (test (clean-string '(1.123123 -2.31231323 3.141592653589 4/3) 1) "(1.1 -2.3 3.1 4/3)")
  (test (clean-string '(1.123123 -2.31231323 3.141592653589 4/3)) "(1.123 -2.312 3.142 4/3)")
  (test (clean-string '(1.123123 -2.31231323 3.141592653589 4/3) 6) "(1.123123 -2.312313 3.141593 4/3)"))

(when with-bignums
  (test (format #f "~P" (bignum "1")) "")
  (test (format #f "~P" (bignum "1.0")) "")
  (test (format #f "~P" (bignum "2")) "s")
  (test (format #f "~P" (bignum "2.0")) "s")
  (test (format #f "~10,' D" (bignum "1")) "         1")
  (test (format #f "~10,' D" (bignum "3/4")) "       3/4")
  (test (format #f "~10,'.D" (bignum "3/4")) ".......3/4")
  (test (format #f "~10D" (bignum "3/4")) "       3/4")
  (test (length (format #f "~100D" (bignum "34"))) 100)
  (test (format #f "~50F" (bignum "12345678.7654321")) "                                1.23456787654321E7"))

(test (format #f "~W" (float-vector +nan.0)) "#r(+nan.0)")
(test (format #f "~W" (float-vector -3/4 +nan.0)) "#r(-0.75 +nan.0)")
(test (format #f "~W" (float-vector -nan.0 +nan.0)) "#r(+nan.0 +nan.0)")
(test (format #f "~W" (float-vector +nan.0 +inf.0)) "#r(+nan.0 +inf.0)")
(test (format #f "~W" (float-vector +nan.0 -3/4)) "#r(+nan.0 -0.75)")
(test (format #f "~W" (float-vector -inf.0 +inf.0)) "#r(-inf.0 +inf.0)")


(call-with-output-file tmp-output-file (lambda (p) (format p "this ~A ~C test ~D" "is" #\a 3)))
(let ((res (call-with-input-file tmp-output-file (lambda (p) (read-line p)))))
  (if (not (string=? res "this is a test 3"))
      (begin
	(display "call-with-input-file + format to \"tmp1.r5rs\" ... expected \"this is a test 3\", but got \"")
	(display res) (display "\"?") (newline))))

(let ((val (format #f "line 1~%line 2~%line 3")))
  (with-input-from-string val
    (lambda ()
      (let ((line1 (read-line)))
	(test (string=? line1 "line 1") #t))
      (let ((line2 (read-line)))
	(test (string=? line2 "line 2") #t))
      (let ((line3 (read-line)))
	(test (string=? line3 "line 3") #t))
      (let ((eof (read-line)))
	(test (eof-object? eof) #t))
      (let ((eof (read-line)))
	(test (eof-object? eof) #t)))))

(test (display 3 #f) 3)
(test (write 3 #f) 3)

(let ((val (format #f "line 1~%line 2~%line 3")))
  (call-with-input-string val
			  (lambda (p)
			    (let ((line1 (read-line p #t)))
			      (test (string=? line1 (string-append "line 1" (string #\newline))) #t))
			    (let ((line2 (read-line p #t)))
			      (test (string=? line2 (string-append "line 2" (string #\newline))) #t))
			    (let ((line3 (read-line p #t)))
			      (test (string=? line3 "line 3") #t))
			    (let ((eof (read-line p #t)))
			      (test (eof-object? eof) #t))
			    (let ((eof (read-line p #t)))
			      (test (eof-object? eof) #t)))))

(let ((res #f))
  (let ((this-file (open-output-string)))
    (format this-file "this ~A ~C test ~D" "is" #\a 3)
    (set! res (get-output-string this-file))
    (close-output-port this-file))
  (if (not (string=? res "this is a test 3"))
      (begin
	(display "open-output-string + format ... expected \"this is a test 3\", but got \"")
	(display res) (display "\"?") (newline))))

(test (with-output-to-string (lambda () (display 123) (flush-output-port))) "123")
(test (with-output-to-string (lambda () (display 123) (flush-output-port) (display 124))) "123124")

(when (provided? 'linux) ; gets "operation not permitted" in osx
  (test (catch #t (lambda ()
                    (call-with-output-file "/dev/full"
                      (lambda (p) (display 123 p) (flush-output-port p))))
                  (lambda (typ info)
                    (apply format #f info)))
        "flush-output-port: No space left on device \"/dev/full\""))

(test (call-with-output-string
       (lambda (p)
	 (write 1 p)
	 (display 2  p)
	 (format p "~D" 3)
	 (write-byte (char->integer #\4) p)
	 (write-char #\5 p)
	 (write-string "6" p)
	 (write 1 #f)
	 (display 2 #f)
	 (format #f "~D" 3)
	 (write-byte (char->integer #\4) #f)
	 (write-char #\5 #f)
	 (write-string "6" #f)))
      "123456")

(test (write-byte most-positive-fixnum #f) 'error)
(test (write-byte -1 #f) 'error)
(test (write-byte 256 #f) 'error)

(let ((res #f))
  (let ((this-file (open-output-string)))
    (format this-file "this is a test")
    (set! res (get-output-string this-file))
    (if (not (string=? res "this is a test"))
	(format #t "open-output-string + format expected \"this is a test\", but got ~S~%" res))
    (flush-output-port this-file)
    (set! res (get-output-string this-file))
    (if (not (string=? res "this is a test"))
	(format #t "flush-output-port of string port expected \"this is a test\", but got ~S~%" res))
    (format this-file "this is a test")
    (set! res (get-output-string this-file))
    (if (not (string=? res "this is a testthis is a test"))
	(format #t "open-output-string after flush expected \"this is a testthis is a test\", but got ~S~%" res))
    (close-output-port this-file)
    (test (flush-output-port this-file) this-file)))

(test (flush-output-port "hiho") 'error)
(test (flush-output-port *stdin*) 'error)

(call-with-output-file tmp-output-file
  (lambda (p)
    (format p "123456~%")
    (format p "67890~%")
    (flush-output-port p)
    (test (call-with-input-file tmp-output-file
	    (lambda (p)
	      (read-line p)))
	  "123456")
    (close-output-port p)))

(let ((res1 #f)
      (res2 #f)
      (res3 #f))
  (let ((p1 (open-output-string)))
    (format p1 "~D" 0)
    (let ((p2 (open-output-string)))
      (format p2 "~D" 1)
      (let ((p3 (open-output-string)))
	(if (not (string=? (get-output-string p1) "0"))
	    (format #t ";format to nested ports, p1: ~S~%" (get-output-string p1)))
	(if (not (string=? (get-output-string p2) "1"))
	    (format #t ";format to nested ports, p2: ~S~%" (get-output-string p2)))
	(format p3 "~D" 2)
	(format p2 "~D" 3)
	(format p1 "~D" 4)
	(format p3 "~D" 5)
	(set! res3 (get-output-string p3))
	(close-output-port p3)
	(if (not (string=? (get-output-string p1) "04"))
	    (format #t ";format to nested ports after close, p1: ~S~%" (get-output-string p1)))
	(if (not (string=? (get-output-string p2) "13"))
	    (format #t ";format to nested ports after close, p2: ~S~%" (get-output-string p2))))
      (format (or p1 p3) "~D" 6)
      (format (and p1 p2) "~D" 7)
      (set! res1 (get-output-string p1))
      (close-output-port p1)
      (if (not (string=? (get-output-string p2) "137"))
	  (format #t ";format to nested ports after 2nd close, p2: ~S~%" (get-output-string p2)))
      (format p2 "~D" 8)
      (set! res2 (get-output-string p2))
      (test (get-output-string p1) 'error)
      (test (get-output-string p2 "hi") 'error)
      (close-output-port p2)))
  (if (not (string=? res1 "046"))
      (format #t ";format to nested ports, res1: ~S~%" res1))
  (if (not (string=? res2 "1378"))
      (format #t ";format to nested ports, res2: ~S~%" res2))
  (if (not (string=? res3 "25"))
      (format #t ";format to nested ports, res3: ~S~%" res3)))

(test (call/cc (lambda (return)
		 (let ((val (format #f "line 1~%line 2~%line 3")))
		   (call-with-input-string val
					   (lambda (p) (return "oops"))))))
      "oops")

(test (get-output-string #f 64) 'error)

;(format #t "format #t: ~D" 1)
;(format (current-output-port) " output-port: ~D! (this is testing output ports)~%" 2)

(call-with-output-file tmp-output-file
  (lambda (p)
    (display 1 p)
    (write 2 p)
    (write-char #\3 p)
    (format p "~D" 4)
    (write-byte (char->integer #\5) p)
    (call-with-output-file "tmp2.r5rs"
      (lambda (p)
	(display 6 p)
	(write 7 p)
	(write-char #\8 p)
	(format p "~D" 9)
	(write-byte (char->integer #\0) p)
	(newline p)))
    (call-with-input-file "tmp2.r5rs"
      (lambda (pin)
	(display (read-line pin) p)))
    (newline p)))

(test (call-with-input-file tmp-output-file
	(lambda (p)
	  (read-line p)))
      "1234567890")

(call-with-output-file tmp-output-file
  (lambda (p)
    (format p "12345~%")
    (format p "67890~%")))

(call-with-input-file tmp-output-file
  (lambda (p)
    (test (read-char p) #\1)
    (test (read-byte p) (char->integer #\2))
    (test (peek-char p) #\3)
    (if (not pure-s7) (test (char-ready? p) #t))
    (test (read-line p) "345")
    (test (read-line p) "67890")))

(call-with-output-file tmp-output-file
  (lambda (p)
    (write-string "123" p)
    (write-string "" p)
    (write-string "456\n789" p)))

(call-with-input-file tmp-output-file
  (lambda (p)
    (test (read-line p) "123456")
    (test (read-char p) #\7)
    (test (read-char p) #\8)
    (test (read-char p) #\9)
    (test (eof-object? (read-char p)) #t)))

(test (with-output-to-string
	(lambda ()
	  (write-string "123")
	  (write-string "")
	  (write-string "456")))
      "123456")

(test (with-output-to-string
	(lambda ()
	  (write-string "123" (current-output-port))
	  (write-string "" (current-output-port))
	  (write-string "456" (current-output-port))
	  (write-string "678" (current-output-port) 1)
	  (write-string "679" (current-output-port) 2 3)
	  (write-string "079" (current-output-port) 0 1)
	  (write-string "123" (current-output-port) 0 3)
	  (write-string "123" (current-output-port) 3 3)
	  (write-string "" (current-output-port) 0 0)
	  (write-string "1423" (current-output-port) 1 1) ; 1.3.3: end is exclusive, if start=end, empty result
	  (write-string "1423" (current-output-port) 1 4/2)
	  (write-string "5423" (current-output-port) -0 1)))
      "123456789012345")

(test (write-string "12345" -1) 'error)
(test (write-string "12345" 0 -1) 'error)
(test (write-string "12345" 0 18) 'error)
(test (write-string "12345" 18) 'error)
(test (write-string "12345" 2 1) 'error)
(test (write-string "12345" 5 5) 'error)
(test (write-string "12345" 0.0 2) 'error)
(test (write-string "12345" 0 2.0) 'error)
(test (write-string "12345" 0 1+i) 'error)
(test (write-string "12345" 0 2/3) 'error)
(test (write-string "12345" 0 #\a) 'error)
(test (write-string "12345" #\null) 'error)
(test (write-string "12345" most-negative-fixnum) 'error)
(test (write-string "12345" 0 most-positive-fixnum) 'error)
(test (write-string "12345" 0 4294967296) 'error)
(test (write-string "a" #f 1) "")
(test (write-string "abc" #f 3) "")
(test (write-string "ab" #f 1) "b")
(test (write-string "ab" #f 2) "")
(test (write-string "abc" #f 1 2) "b")
(test (write-string "abc" #f 1 3) "bc")

(test (with-input-from-string "12345" (lambda () (read-string 3))) "123")
(test (with-input-from-string "" (lambda () (read-string 3))) #<eof>)
(test (with-input-from-string "" (lambda () (read-string 0))) "")
(test (with-input-from-string "1" (lambda () (read-string 0))) "")
(test (with-input-from-string "1" (lambda () (read-string -1))) 'error)
(test (with-input-from-string "1" (lambda () (read-string #f))) 'error)
(test (with-input-from-string "123" (lambda () (read-string 10))) "123")
(test (call-with-input-string "123" (lambda (p) (read-string 2 p))) "12")
(test (call-with-input-string "123" (lambda (p) (read-string 2 #f))) 'error)
(test (call-with-input-string "123" (lambda (p) (read-string 2 (current-output-port)))) 'error)
(test (call-with-input-string "123" (lambda (p) (read-string 0 #<unspecified>))) 'error)
(test (call-with-input-string "123" (lambda (p) (read-string 0 123))) 'error)

(test (read-string most-positive-fixnum) 'error)
(test (read-string -1) 'error)
(test (read-string most-negative-fixnum) 'error)
;(test (read-string 0) "")
; (test (read-string 123) "")
; s7 considers this file (during load) to be the current-input-file, so the above read-string ruins the load
; the other choice is to hang (waiting for stdin)
; perhaps this choice should be documented since it is specifically contrary to r7rs

(test (write 1 (current-input-port)) 'error)
(test (write-char #\a (current-input-port)) 'error)
(test (write-byte 0 (current-input-port)) 'error)
(test (read (current-output-port)) 'error)
(test (read-char (current-output-port)) 'error)
(test (read-byte (current-output-port)) 'error)
(test (read-line (current-output-port)) 'error)
(test (display 3) 3)
(test (display 3 #f) 3)

(unless pure-s7
  (let ((op1 (set-current-output-port (open-output-file tmp-output-file))))
    (display 1)
    (write 2)
    (write-char #\3)
    (format #t "~D" 4) ; #t -> output port
    (write-byte (char->integer #\5))
    (let ((op2 (set-current-output-port (open-output-file "tmp2.r5rs"))))
      (display 6)
      (write 7)
      (write-char #\8)
      (format #t "~D" 9)
      (write-byte (char->integer #\0))
      (newline)
      (close-output-port (current-output-port))
      (set-current-output-port op2)
      (let ((ip1 (set-current-input-port (open-input-file "tmp2.r5rs"))))
	(display (read-line))
	(close-input-port (current-input-port))
	(set-current-input-port ip1))
      (newline)
      (close-output-port (current-output-port)))

    (set-current-output-port #f)
    (test (string? (format #t "~%")) #t)
    (write "write: should not appear" #f) (newline #f)
    (display "display: should not appear" #f) (newline #f)
    (format #f "format: should not appear") (newline #f)
    (write-string "write-string: should not appear" #f) (newline #f)
    (write-char #\! #f)
    (write-byte 123 #f)

    (write "write: should not appear" (current-output-port)) (newline (current-output-port))
    (display "display: should not appear" (current-output-port)) (newline (current-output-port))
    (format (current-output-port) "format: should not appear") (newline (current-output-port))
    (write-string "write-string: should not appear" (current-output-port)) (newline (current-output-port))
    (write-char #\! (current-output-port))
    (write-byte 123 (current-output-port))

    (write "write: should not appear") (newline)
    (display "display: should not appear") (newline)
    (format #t "format: should not appear") (newline)
    (write-string "write-string: should not appear") (newline)
    (write-char #\!)
    (write-byte 123)

    (set-current-output-port op1))

  (let ((op1 (open-output-file tmp-output-file)))
    (let-temporarily (((current-output-port) op1))
      (display 1)
      (write 2)
      (write-char #\3)
      (format #t "~D" 4) ; #t -> output port
      (write-byte (char->integer #\5))
      (let ((op2 (open-output-file "tmp2.r5rs")))
	(let-temporarily (((current-output-port) op2))
	  (display 6)
	  (write 7)
	  (write-char #\8)
	  (format #t "~D" 9)
	  (write-byte (char->integer #\0))
	  (newline)
	  (close-output-port (current-output-port)))
	(let ((ip1 (open-input-file "tmp2.r5rs")))
	  (let-temporarily (((current-input-port) ip1))
	    (display (read-line))
	    (close-input-port (current-input-port))))
	(newline)
	(close-output-port (current-output-port)))))

  (test (call-with-input-file tmp-output-file
	  (lambda (p)
	    (read-line p)))
	"1234567890"))

(for-each
 (lambda (op)
   (for-each
    (lambda (arg)
      (test (op arg display) 'error))
    (list 1 1.0 1+i 2/3 'a-symbol (make-vector 3) '(1 2) (cons 1 2) abs #f #t :hi (if #f #f) (lambda (a) (+ a 1)))))
 (list call-with-output-file call-with-input-file
       call-with-output-string call-with-input-string
       with-input-from-string with-input-from-file
       with-output-to-file))

(for-each
 (lambda (op)
   (for-each
    (lambda (arg)
      (test (op arg) 'error))
    (list 1 1.0 1+i 2/3 'a-symbol (make-vector 3) '(1 2) (cons 1 2) abs #f #t :hi (if #f #f) (lambda (a) (+ a 1)))))
 (list open-output-file open-input-file
       open-input-string))

(for-each
 (lambda (op)
   (for-each
    (lambda (arg)
      (test (op "hi" arg) 'error))
    (list "hi" 1 1.0 1+i 2/3 'a-symbol (make-vector 3) '(1 2) (cons 1 2) abs #t :hi (if #f #f) (lambda (a) (+ a 1)))))
 (list write display write-byte newline write-char
       read read-char read-byte peek-char char-ready? read-line))

(for-each
 (lambda (arg)
   (test (write-char arg) 'error)
   (test (write-byte arg) 'error)
   (test (read-char arg) 'error)
   (test (read-byte arg) 'error)
   (test (peek-char arg) 'error)
   (test (write-char #\a arg) 'error)
   (test (write-byte 1 arg) 'error))
 (list "hi" 1.0 1+i 2/3 'a-symbol (make-vector 3) '(1 2) (cons 1 2) abs #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (write-byte -1) 'error)
(test (write-byte most-positive-fixnum) 'error)
(test (write-byte 300) 'error)

(for-each
 (lambda (arg)
   (test (write-string arg) 'error)
   (test (write-string "hi" arg) 'error))
 (list 1.0 1+i 2/3 'a-symbol (make-vector 3) '(1 2) (cons 1 2) abs #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (with-output-to-string (lambda () (newline #f))) "")
(test (with-output-to-string (lambda () (write-byte 95 #f))) "")
(test (with-output-to-string (lambda () (write-char #\a #f))) "")
(test (with-output-to-string (lambda () (write-string "a" #f))) "")
(test (with-output-to-string (lambda () (write "hiho" #f))) "")
(test (with-output-to-string (lambda () (display "hiho" #f))) "")
(test (with-output-to-string (lambda () (format #f "hiho"))) "")

(unless pure-s7
  (test (with-output-to-string
	  (lambda ()
	    (set! (current-output-port) #f)
	    (newline (current-output-port))
	    (write-byte 95 (current-output-port))
	    (write-char #\a (current-output-port))
	    (write-string "a" (current-output-port))
	    (write "hiho" (current-output-port))
	    (display "hiho" (current-output-port))
	    (format (current-output-port) "hiho")))
	"")
  (set! (current-output-port) *stdout*))

(let-temporarily (((current-output-port) #f)
                  ((*s7* 'max-string-length) 32))
  (catch #t
    (lambda ()
      (with-output-to-string
	(lambda ()
	  (display
	   (symbol
	    (make-string (*s7* 'max-string-length)))))))
    (lambda args
      'error))
  (when (current-output-port)
    (format *stderr* "current-output-port is ~S~%" (current-output-port))))

(for-each
 (lambda (op)
   (for-each
    (lambda (arg)
      (test (op arg) 'error))
    (list "hi" 1 1.0 1+i 2/3 'a-symbol (make-vector 3) '(1 2) (cons 1 2) abs :hi (if #f #f) (lambda (a) (+ a 1)))))
 (list set-current-input-port set-current-error-port set-current-output-port close-input-port close-output-port))

(let ((hi (open-output-string)))
  (test (get-output-string hi) "")
  (close-output-port hi)
  (test (get-output-string hi) 'error))

(test (open-output-string "hiho") 'error)
(test (with-output-to-string "hi") 'error)
(test (call-with-output-string "hi") 'error)

(test (get-output-string 1 2) 'error)
(test (get-output-string) 'error)
(for-each
 (lambda (arg)
   (test (get-output-string arg) 'error))
 (list "hi" 1 1.0 1+i 2/3 'a-symbol (make-vector 3) '(1 2) (cons 1 2) abs :hi (if #f #f) (lambda (a) (+ a 1))))

(let ((p (open-output-string)))
  (write 123 p)
  (test (get-output-string p) "123")
  (write 124 p)
  (test (get-output-string p #t) "123124")
  (test (get-output-string p #t) "")
  (write 123 p)
  (test (get-output-string p) "123"))

;; since read of closed port will generate garbage, it needs to be an error,
;;   so I guess write of closed port should also be an error

(let ((hi (open-output-string)))
  (close-output-port hi)
  (for-each
   (lambda (op)
     (test-e (op hi) (object->string op) 'closed-port))
   (list (lambda (p) (display 1 p))
	 (lambda (p) (write 1 p))
	 (lambda (p) (write-char #\a p))
	 (lambda (p) (write-byte 0 p))
	 (lambda (p) (format p "hiho"))
	 (if pure-s7 newline set-current-output-port)
	 (if pure-s7 newline set-current-input-port)
	 set-current-error-port
	 newline)))

(let ((hi (open-input-string "hiho")))
  (test (get-output-string hi) 'error)
  (close-input-port hi)
  (for-each
   (lambda (op)
     (test-e (op hi) (object->string op) 'closed-port))
   (list read read-char read-byte peek-char read-line
	 port-filename port-line-number
	 (if pure-s7 read-line char-ready?)
	 (if pure-s7 read-line set-current-output-port)
	 (if pure-s7 read-line set-current-input-port)
	 set-current-error-port
	 )))

(test (close-output-port (open-input-string "hiho")) 'error)
(test (close-input-port (open-output-string)) 'error)
(test (set! (port-filename) "hiho") 'error)
(test (set! (port-closed? (current-output-port)) "hiho") 'error)
(test (begin (close-output-port *stderr*) (port-closed? *stderr*)) #f)
(test (begin (close-output-port *stdout*) (port-closed? *stdout*)) #f)
(test (begin (close-input-port *stdin*) (port-closed? *stdin*)) #f)

(test (call-with-output-file "test.data" port-filename) "test.data")
(test (call-with-input-file "s7test.scm" port-filename) "s7test.scm")
(if (provided? 'linux) (test (with-input-from-file "/proc/cpuinfo" port-filename) "/proc/cpuinfo"))
(test (call-with-output-file "/dev/null" port-filename) "/dev/null")
(test (port-filename (open-output-string)) "") ; malloc garbage if not cleared in open_output_string

(test (let ((str ""))
	(with-input-from-string "1234567890" (lambda ()
          (with-input-from-string "1234567890" (lambda ()
            (with-input-from-string "1234567890" (lambda ()
              (with-input-from-string "1234567890" (lambda ()
                (with-input-from-string "1234567890" (lambda ()
                  (with-input-from-string "1234567890" (lambda ()
                    (with-input-from-string "1234567890" (lambda ()
	              (set! str (string-append str (string (read-char))))))
	            (set! str (string-append str (string (read-char) (read-char))))))
	          (set! str (string-append str (string (read-char) (read-char) (read-char))))))
	        (set! str (string-append str (string (read-char) (read-char) (read-char) (read-char))))))
              (set! str (string-append str (string (read-char) (read-char) (read-char) (read-char) (read-char))))))
            (set! str (string-append str (string (read-char) (read-char) (read-char) (read-char) (read-char) (read-char))))))
          (set! str (string-append str (string (read-char) (read-char) (read-char) (read-char) (read-char) (read-char) (read-char))))))
	  str)
      "1121231234123451234561234567")

(let* ((new-error-port (open-output-string))
       (old-error-port (set-current-error-port new-error-port)))
  (catch #t
    (lambda ()
      (format #f "~R" 123))
    (lambda args
      (format (current-error-port) "oops")))
  (let ((str (get-output-string new-error-port)))
    (set-current-error-port old-error-port)
    (test str "oops")))

(let ((hi (open-input-string "hiho")))
  (for-each
   (lambda (op)
     (test-e (op hi) (object->string op) 'input-port))
   (list (lambda (p) (display 1 p))
	 (lambda (p) (write 1 p))
	 (lambda (p) (write-char #\a p))
	 (lambda (p) (write-byte 0 p))
	 (lambda (p) (format p "hiho"))
	 newline))
  (close-input-port hi))

(let ((hi (open-output-file tmp-output-file)))
  (write-byte 1 hi)
  (close-output-port hi)
  (test (write-byte 1 hi) 'error))

(let ((hi (open-output-string)))
  (for-each
   (lambda (op)
     (test-e (op hi) (object->string op) 'output-port))
   (list read read-char read-byte peek-char char-ready? read-line))
  (close-output-port hi))

(test (output-port? (current-error-port)) #t)
(test (and (not (null? (current-error-port))) (input-port? (current-error-port))) #f)

(call-with-output-file tmp-output-file
  (lambda (p)
    (test (get-output-string p) 'error)
    (do ((i 0 (+ i 1)))
	((= i 256))
      (write-byte i p))))

(call-with-input-file tmp-output-file
  (lambda (p)
    (test (get-output-string p) 'error)
    (call-with-exit
     (lambda (quit)
       (do ((i 0 (+ i 1)))
	   ((= i 256))
	 (let ((b (read-byte p)))
	   (if (or (not (number? b))
		   (not (= b i)))
	       (begin
		 (format #t "read-byte got ~A, expected ~A~%" b i)
		 (quit)))))))
    (let ((eof (read-byte p)))
      (if (not (eof-object? eof))
	  (format #t "read-byte at end: ~A~%" eof)))
    (let ((eof (read-byte p)))
      (if (not (eof-object? eof))
	  (format #t "read-byte at end: ~A~%" eof)))))

(call-with-output-file tmp-output-file
  (lambda (p)
    (do ((i 0 (+ i 1)))
	((= i 256))
      (write-char (integer->char i) p))))

(define our-eof #f)

(call-with-input-file tmp-output-file
  (lambda (p)
    (call-with-exit
     (lambda (quit)
       (do ((i 0 (+ i 1)))
	   ((= i 256))
	 (let ((b (read-char p)))
	   (if (or (not (char? b))
		   (not (char=? b (integer->char i))))
	       (begin
		 (format #t "read-char got ~A, expected ~A (~D: char? ~A)~%" b (integer->char i) i (char? (integer->char i)))
		 (quit)))))))
    (let ((eof (read-char p)))
      (if (not (eof-object? eof))
	  (format #t "read-char at end: ~A~%" eof))
      (set! our-eof eof))
    (let ((eof (read-char p)))
      (if (not (eof-object? eof))
	  (format #t "read-char again at end: ~A~%" eof)))))

(test (eof-object? (integer->char 255)) #f)
(test (eof-object? our-eof) #t)
(test (char->integer our-eof) 'error)
(test (char? our-eof) #f)
(test (eof-object? ((lambda () our-eof))) #t)

(for-each
 (lambda (op)
   (test (op *stdout*) 'error)
   (test (op *stderr*) 'error)
   (test (op (current-output-port)) 'error)
   (test (op (current-error-port)) 'error)
   (test (op ()) 'error))
 (list read read-line read-char read-byte peek-char char-ready?))

(for-each
 (lambda (op)
   (test (op #\a *stdin*) 'error)
   (test (op #\a (current-input-port)) 'error)
   (test (op #\a ()) 'error))
 (list write display write-char))

(test (write-byte 0 *stdin*) 'error)
(test (write-byte (char->integer #\space) *stdout*) (char->integer #\space))
(test (write-byte (char->integer #\space) *stderr*) (char->integer #\space))
(test (newline *stdin*) 'error)
(test (format *stdin* "hiho") 'error)

(test (port-filename *stdin*) "*stdin*")
(test (port-filename *stdout*) "*stdout*")
(test (port-filename *stderr*) "*stderr*")

(test (input-port? *stdin*) #t)
(test (output-port? *stdin*) #f)
(test (port-closed? *stdin*) #f)
(test (input-port? *stdout*) #f)
(test (output-port? *stdout*) #t)
(test (port-closed? *stdout*) #f)
(test (input-port? *stderr*) #f)
(test (output-port? *stderr*) #t)
(test (port-closed? *stderr*) #f)

(test (port-line-number *stdin*) 0)
(test (port-line-number *stdout*) 'error)
(test (port-line-number *stderr*) 'error)
(test (port-line-number ()) 'error) ; this used to be *stdin*?

(test (open-input-file "[*not-a-file!*]-") 'error)
(test (call-with-input-file "[*not-a-file!*]-" (lambda (p) p)) 'error)
(test (with-input-from-file "[*not-a-file!*]-" (lambda () #f)) 'error)

(test (open-input-file "") 'error)
(test (call-with-input-file "" (lambda (p) p)) 'error)
(test (with-input-from-file "" (lambda () #f)) 'error)

;(test (open-output-file "/bad-dir/badness/[*not-a-file!*]-") 'error)
;(test (call-with-output-file "/bad-dir/badness/[*not-a-file!*]-" (lambda (p) p)) 'error)
;(test (with-output-to-file "/bad-dir/badness/[*not-a-file!*]-" (lambda () #f)) 'error)

(with-output-to-file "tmp.r5rs"
  (lambda ()
    (write-char #\a)
    (with-output-to-file tmp-output-file
      (lambda ()
	(format #t "~C" #\b)
	(with-output-to-file "tmp2.r5rs"
	  (lambda ()
	    (display #\c)))
	(display (with-input-from-file "tmp2.r5rs"
		   (lambda ()
		     (read-char))))))
    (with-input-from-file tmp-output-file
      (lambda ()
	(write-byte (read-byte))
	(write-char (read-char))))))

(with-input-from-file "tmp.r5rs"
  (lambda ()
    (test (read-line) "abc")))

(with-input-from-file "tmp.r5rs" ; this assumes tmp.r5rs has "abc" as above
  (lambda ()
    (test (read-char) #\a)
    (test (eval-string "(+ 1 2)") 3)
    (test (read-char) #\b)
    (with-input-from-string "(+ 3 4)"
      (lambda ()
	(test (read) '(+ 3 4))))
    (test (read-char) #\c)))

(test (eval-string (object->string (with-input-from-string "(+ 1 2)" read))) 3)
(test (eval (eval-string "(with-input-from-string \"(+ 1 2)\" read)")) 3)
(test (eval-string "(eval (with-input-from-string \"(+ 1 2)\" read))") 3)
(test (eval-string (object->string (eval-string (format #f "(+ 1 2)")))) 3)


;;; -------- test that we can plow past errors --------

(if (and (defined? 'file-exists?) ; (ifdef name ...)?
	 (file-exists? "tests.data"))
    (delete-file "tests.data"))

(call-with-output-file "tests.data"
  (lambda (p)
    (format p "start ")
    (catch #t
      (lambda ()
	(format p "next ") (abs "hi") (format p "oops "))
      (lambda args
	'error))
    (format p "done\n")))

(let ((str (call-with-input-file "tests.data"
             (lambda (p)
	       (read-line p)))))
  (if (or (not (string? str))
	  (not (string=? str "start next done")))
      (format #t ";call-with-output-file + error -> ~S~%" str)))

(let ((str (call-with-input-file "tests.data"
             (lambda (p)
	       (catch #t
		      (lambda ()
			(read-char p)
			(abs "hi")
			(read-char p))
		      (lambda args "s"))))))
  (if (or (not (string? str))
	  (not (string=? str "s")))
      (format #t ";call-with-input-file + error -> ~S~%" str)))

(if (and (defined? 'file-exists?)
	 (file-exists? "tests.data"))
    (delete-file "tests.data"))

(with-output-to-file "tests.data"
  (lambda ()
    (format #t "start ")
    (catch #t
      (lambda ()
	(format #t "next ") (abs "hi") (format #t "oops "))
      (lambda args
	'error))
    (format #t "done\n")))

(let ((str (with-input-from-file "tests.data"
             (lambda ()
	       (read-line)))))
  (if (or (not (string? str))
	  (not (string=? str "start next done")))
      (format #t ";with-output-to-file + error -> ~S~%" str)))

(let ((str (with-input-from-file "tests.data"
             (lambda ()
	       (catch #t
		      (lambda ()
			(read-char)
			(abs "hi")
			(read-char))
		      (lambda args "s"))))))
  (if (or (not (string? str))
	  (not (string=? str "s")))
      (format #t ";with-input-from-file + error -> ~S~%" str)))

(test (call-with-output-string newline) (string #\newline))
(test (call-with-output-string append) "")

(let ((str (call-with-output-string
	    (lambda (p)
	      (format p "start ")
	      (catch #t
		     (lambda ()
		       (format p "next ") (abs "hi") (format p "oops "))
		     (lambda args
		       'error))
	      (format p "done")))))
  (if (or (not (string? str))
	  (not (string=? str "start next done")))
      (format #t ";call-with-output-string + error -> ~S~%" str)))

(let ((str (with-output-to-string
	    (lambda ()
	      (format #t "start ")
	      (catch #t
		     (lambda ()
		       (format #t "next ") (abs "hi") (format #t "oops "))
		     (lambda args
		       'error))
	      (format #t "done")))))
  (if (or (not (string? str))
	  (not (string=? str "start next done")))
      (format #t ";with-output-to-string + error -> ~S~%" str)))

(test (with-output-to-string (lambda () (format (current-output-port) "a test ~D" 123))) "a test 123")
;(test (with-output-to-string (lambda () (format *stdout* "a test ~D" 1234))) "a test 1234")

(test (string=? (with-output-to-string (lambda () (write #\null))) "#\\null") #t)
(test (string=? (with-output-to-string (lambda () (write #\space))) "#\\space") #t)
(test (string=? (with-output-to-string (lambda () (write #\return))) "#\\return") #t)
(test (string=? (with-output-to-string (lambda () (write #\escape))) "#\\escape") #t)
(test (string=? (with-output-to-string (lambda () (write #\tab))) "#\\tab") #t)
(test (string=? (with-output-to-string (lambda () (write #\newline))) "#\\newline") #t)
(test (string=? (with-output-to-string (lambda () (write #\backspace))) "#\\backspace") #t)
(test (string=? (with-output-to-string (lambda () (write #\alarm))) "#\\alarm") #t)
(test (string=? (with-output-to-string (lambda () (write #\delete))) "#\\delete") #t)

(test (string=? (with-output-to-string (lambda () (write-char #\space))) " ") #t)  ; weird -- the name is backwards
(test (string=? (with-output-to-string (lambda () (display #\space))) " ") #t)

(let ((str (call-with-input-string "12345"
	    (lambda (p)
	      (catch #t
		     (lambda ()
		       (read-char p)
		       (abs "hi")
		       (read-char p))
		     (lambda args "s"))))))
  (if (or (not (string? str))
	  (not (string=? str "s")))
      (format #t ";call-with-input-string + error -> ~S~%" str)))

(let ((str (with-input-from-string "12345"
	    (lambda ()
	      (catch #t
		     (lambda ()
		       (read-char)
		       (abs "hi")
		       (read-char))
		     (lambda args "s"))))))
  (if (or (not (string? str))
	  (not (string=? str "s")))
      (format #t ";with-input-from-string + error -> ~S~%" str)))

(for-each
 (lambda (arg)
   (test (port-line-number arg) 'error)
   (test (port-filename arg) 'error))
 (list "hi" -1 0 #\a 'a-symbol #(1 2 3) '(1 . 2) '(1 2 3) 3.14 3/4 1.0+1.0i #t abs #<eof> #<unspecified> (lambda () 1)))

(test (catch #t (lambda () (eval-string (port-filename))) (lambda args #f)) #f)
(test (symbol? (string->symbol (port-filename))) #t)

(for-each
 (lambda (arg)
   (test
    (with-input-from-string (format #f "~A" arg)
      (lambda ()
	(read)))
    arg))
 (list 1 3/4 '(1 2) #(1 2) :hi #f #t))

(num-test (with-input-from-string "3.14" read) 3.14)
(num-test (with-input-from-string "3.14+2i" read) 3.14+2i)
(num-test (with-input-from-string "#x2.1" read) 2.0625)
(test (with-input-from-string "'hi" read) ''hi)
(test (with-input-from-string "'(1 . 2)" read) ''(1 . 2))

(test
 (let ((cin #f)
       (cerr #f))
   (catch #t
	  (lambda ()
	    (with-input-from-string "123"
	      (lambda ()
		(set! cin (current-input-port))
		(error 'testing "jump out"))))
	  (lambda args
	    (set! cerr #t)))
   (format #f "~A ~A" cin cerr))
 "#<input-string-port:closed> #t")

;;; old form:  "<port string input (closed)> #t")

(test
 (let ((cp (current-output-port))
       (cout #f)
       (cerr #f))
   (catch #t
	  (lambda ()
	    (with-output-to-string
	      (lambda ()
		(set! cout (current-output-port))
		(error 'testing "jump out"))))
	  (lambda args
	    (set! cerr #t)))
   (format #f "~A ~A" cout cerr))
 "#<output-string-port:closed> #t")
(if (not (eq? *stdout* old-stdout))
    (format *stderr* ";~D: stdout clobbered~%" (port-line-number)))

;;; old form:  "<port string output (closed)> #t")

(test (open-input-file #u(115 55 116 101 115 116 46 115 99 109 0) #u(114 0 98)) 'error) ; "s7test.scm" "r\x00b"

(call-with-output-file tmp-output-file
  (lambda (p)
    (display "1" p)
    (newline p)
    (newline p)
    (display "2345" p)
    (newline p)))

(call-with-input-file tmp-output-file
  (lambda (p)
    (test (read-line p) "1")
    (test (read-line p) "")
    (test (read-line p) "2345")
    (test (eof-object? (read-line p)) #t)))

(let ((p (open-output-file tmp-output-file "a")))
  (display "678" p)
  (newline p)
  (close-output-port p))

(if (not with-windows) ; "xyzzy" is legit in windows??
    (begin
      (test (let ((p (open-output-file tmp-output-file "xyzzy"))) (close-output-port p)) 'error)
      (test (let ((p (open-input-file tmp-output-file "xyzzy"))) (close-input-port p)) 'error)))

(call-with-input-file tmp-output-file
  (lambda (p)
    (test (read-line p) "1")
    (test (read-line p) "")
    (test (read-line p) "2345")
    (test (read-line p) "678")
    (test (eof-object? (read-line p)) #t)))

(test (let ((a 1))
	(define-macro (m1) `(set! a (read)))
	(with-input-from-string "123" m1)
	a)
      123)

(test (let ((a 1))
	(define-macro (m3 p) `(set! a (read ,p)))
	(call-with-input-string "123" m3)
	a)
      123)

(test (let ()
	(define-macro (m1) `(define a (read)))
	(with-input-from-string "123" m1)
	a)
      123)

(test (let ()
	(define-macro (m3 p) `(define a (read ,p)))
	(call-with-input-string "123" m3)
	a)
      123)

(for-each
 (lambda (arg)
   (test (port-filename arg) 'error))
 (list "hi" -1 #\a 1 0 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #f #t () (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (port-filename arg) 'error))
 (list "hi" -1 #\a 1 0 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #f #t () (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (open-input-file "s7test.scm" arg) 'error)
   (test (open-output-file tmp-data-file arg) 'error))
 (list -1 #\a 1 0 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #f #t () (list 1 2 3) '(1 . 2)))

(test (current-input-port ()) 'error)
(test (current-output-port ()) 'error)
(test (current-error-port ()) 'error)

(for-each
 (lambda (op)
   (let ((tag (catch #t (lambda () (op)) (lambda args 'error))))
     (if (not (eq? tag 'error))
	 (format #t ";(~A) -> ~A (expected 'error)~%" op tag))))
 (list set-current-input-port set-current-error-port set-current-output-port
       close-input-port close-output-port
       write display write-byte write-char format                     ; newline
       ;read read-char read-byte peek-char char-ready? read-line      ; these can default to current input
       call-with-output-file call-with-input-file
       call-with-output-string call-with-input-string
       with-input-from-string with-input-from-file
       with-output-to-file
       open-output-file open-input-file
       open-input-string))

(for-each
 (lambda (op)
   (let ((tag (catch #t (lambda () (op 1 2 3 4 5)) (lambda args 'error))))
     (if (not (eq? tag 'error))
	 (format #t ";(~A 1 2 3 4 5) -> ~A (expected 'error)~%" op tag))))
 (list set-current-input-port set-current-error-port set-current-output-port
       close-input-port close-output-port
       write display write-byte write-char format newline
       read read-char read-byte peek-char char-ready? read-line
       call-with-output-file call-with-input-file
       call-with-output-string call-with-input-string
       with-input-from-string with-input-from-file
       with-output-to-file
       open-output-file open-input-file
       open-input-string))

;;; (string-set! (with-input-from-string "\"1234\"" read) 1 #\a)
(test (with-input-from-string "(+ 1 2)" read) '(+ 1 2))

(when (and (provided? 'system-extras)
	   (file-exists? "gad1.data")) ; file too big to treat as in-core string
  (let ((p (catch #t (lambda () (open-input-file "gad1.data")) (lambda (type info) #f))))
    (when p
      (test (port-filename p) "gad1.data")
      (test (string? (read-line p)) #t)
      (test (char? (read-char p)) #t)
      (test (integer? (read-byte p)) #t)
      (read p) ; could be symbol, number etc
      (test (string? (read-string 10 p)) #t)
      (close-input-port p))))

(test (>= (length (with-output-to-string (lambda () (write (make-string 512 #\tab))))) 512) #t)
(test (>= (length (with-output-to-string (lambda () (write (make-string 512 #\newline))))) 512) #t)
(test (>= (length (with-output-to-string (lambda () (write (make-string 512 #\"))))) 512) #t)
(test (>= (length (with-output-to-string (lambda () (write (make-string 512 #\x65))))) 512) #t)

(if (and (defined? 'file-exists?)
	 (file-exists? (append "/home/" username "/test")))
    (let-temporarily ((*load-path* (cons (append "/home/" username "/test") *load-path*)))
      (with-output-to-file (append "/home/" username "/test/load-path-test.scm")
	(lambda ()
	  (format #t "(define (load-path-test) *load-path*)~%")))
      (load "load-path-test.scm")
      (if (or (not (defined? 'load-path-test))
	      (not (equal? *load-path* (load-path-test))))
	  (format #t ";*load-path*: ~S, but ~S~%" *load-path* (load-path-test)))))

;;; function ports
(when with-block

  (let ((p (function-open-output)))
    (write-char #\a p)
    (let ((val (function-get-output p)))
      (function-close-output p)
      (if (not (string=? val "a"))
	  (format *stderr* ";function port write #\\a: ~S (~D, ~A)~%" val (length val) (string->vector val)))))

  (let ((p (function-open-output)))
    (display "123" p)
    (format p "4~D6" 5)
    (write-string "789" p)
    (write-byte (char->integer #\0) p)
    (newline p)
    (let ((val (function-get-output p)))
      (function-close-output p)
      (close-output-port p)
      (if (not (string=? val "1234567890\n"))
	  (format *stderr* ";function port outputs: ~S (~D, ~A)~%" val (length val) (string->vector val)))))

  (let ((str "0123"))
    (let ((p (function-open-input str)))
      (let ((val (read-char p)))
	(if (not (char=? val #\0))
	    (format *stderr* ";function port read #\\0: ~S~%" val)))))

  (let ((str "0123\n45678"))
    (let ((p (function-open-input str)))
      (let ((val (read-line p)))
	(if (not (string=? val "0123"))
	    (format *stderr* ";function port read-line: ~S~%" val))
	(set! val (read-byte p))
	(if (not (= val (char->integer #\4)))
	    (format *stderr* ";function port read-byte: ~S~%" val))
	(set! val (peek-char p))
	(if (not (char=? val #\5))
	    (format *stderr* ";function port peek-char: ~S~%" val))
	(set! val (read-string 2 p))
	(if (not (string=? val "56"))
	    (format *stderr* ";function port read-string: ~S~%" val))
	(if (and (not pure-s7)
		 (not (char-ready? p)))
	    (format *stderr* ";function port has no char ready?~%"))
	(close-input-port p)))))


;;; run integer? code before integer? is localized below
(test (let ((x 1)) ((if (integer? x) + -)  1  1/2 1234)) 2471/2)
(test (let () (define (func) (let ((x 1)) ((if (integer? x) + -)  1  1/2 1234))) (func) (func)) 2471/2)
(test (let ((x 1)) ((if (integer? x) + *))) 0)
(test (let () (define (func) (let ((x 1)) ((if (integer? x) + *)))) (func) (func)) 0)


;;; -------- poke at the reader --------

(test (cdr '(1 ."a")) "a")
(test '(1 .(2 3)) '(1 2 3))
(test '(1 .(2 3)) '(1 . (2 3)))
(test (+ .(2 .(3))) 5)
(test (cadr '(1 '0,)) ''0,)
(test (equal? 3 ' 3) #t)
(test (equal? '
	             3 3) #t)
(test (equal? '"hi" ' "hi") #t)
(test (equal? '#\a '    #\a) #t)
(test (let ((nam()e 1)) 1) 'error)
(test (let ((nam""e 1)) nam""e) 'error) ; this was 1 originally
(test (cadr '(1 ']x)) '']x)
(test `1 1)
(test (equal? '(1 .(1 .())) '(1 1)) #t)
(test (equal? '("hi"."ho") ' ("hi" . "ho")) #t)
(test (equal? '("hi""ho") '("hi" "ho")) #t)
(test '("""""") '("" "" ""))
(test '(#|;"();|#) ())
(test '(#||##\# #||##b1) '(#\# 1))
(test (#|s'!'|#*) 1)
(test (#|==|#) ())
(test -#|==|#1 'error) ; unbound variable
(test '((). '()) '(() quote ()))
(test '(1. . .2) '(1.0 . 0.2))
(test (equal? '(().()) '(())) #t)
(test (equal? '(()()) '(() ())) #t)
(test (equal? '(()..()) '(() .. ())) #t)
(test '((().()).()) '((())))
(test '(((().()).()).()) '(((()))))
(test '((().(().())).()) '((() ())))
(test '((()().(().()))) '((() () ())))
(test '(1 .;
	  2) '(1 . 2))
(test (vector .(1 .(2))) #(1 2))
(test (vector 0. .(.1)) #(0.0 0.1))
(test '(a #|foo||# b) '(a b)) ; from bug-guile
(test '(a #|foo|||# b) '(a b))
(test '(a #|foo||||# b) '(a b))
(test '(a #|foo|||||# b) '(a b))

(test (let () (define (f' x) (+ x x)) (f' 10)) 20) ; from /r/scheme
(test (let () (define (f'' a'b) (+ a'b a'b)) (f'' 10)) 20)
(test (symbol? 'a'b) #t)

(test (char? #\#) #t)
(test (type-of (eval-string "'#<float-vector>")) 'undefined?)
(test (type-of (eval-string "'(#<float-vector>)")) 'pair?)
(test (car `(,.1e0)) .1)
(test (car `(,.1E0)) .1)
(test (let ((x "hi")) (set! x"asdf") x) "asdf")
(test (let* ((x "hi") (y x)) (set! x "asdf") y) "hi")
(test (let ((x 1)) (set! x(list 1 2)) x) '(1 2))
(num-test (let ((x 1)) (set!;"
			x;)
			12.;(
			);#|
	       x) 12.0)
(test (let ((\x00}< 1) (@:\t{ 2)) (+ \x00}< @:\t{)) 3)
(test (let ((| 1) (|| 2) (||| 3)) (+ || | |||)) 6)
(test (let ((|a#||#b| 1)) |a#||#b|) 1)
(test (let ((@,@'[1] 1) (\,| 2)) (+ @,@'[1] \,|)) 3)
(test (list"0"0()#()#\a"""1"'x(list)+(cons"""")#f) (list "0" 0 () #() #\a "" "1" 'x (list) + '("" . "") #f))
(test (let ((x, 1)) x,) 1)
(test (length (eval-string (string #\' #\( #\1 #\space #\. (integer->char 200) #\2 #\)))) 2) ; will be -1 if dot is for improper list, 3 if dot is a symbol
(test (+ `,0(angle ```,`11)) 0)
(test (map . (char->integer "123")) '(49 50 51))
(test (map .(values "0'1")) '(#\0 #\' #\1))
(test (map /""'(123)) ())
(num-test (+ 1 .()) 1)
(test (let () (define (x .()) (list .())) (x)) ())

(test '(1 . ()) '(1))
(test '(1 . (2)) '(1 2))
(test '(1 . (2) 3) '(1 2 3)) ; Guile says "missing close paren", sbcl says "More than one object follows . in list.", clisp: "illegal end of dotted list"
(test '(1 . (2 3) 4 5) '(1 2 3 4 5))
(test '(1 . (2) (3)) '(1 2 (3)))
(test '(1 2 . 'x 3) '(1 2 quote x 3))
(test (eval-string "'(1 . 2 3)") 'error)  ; eval-string here wraps up the read-error
(test '(1 . (2 . 3)) '(1 2 . 3))
(test (eval-string "'(1 . . ((2 3)))") 'error)
(test '((1 . 2) . (3 . 4)) '((1 . 2) 3 . 4))
(test (eval-string "'(1 . () 2)") 'error)
(test '(1 . (2) . (3)) '(1 2 3))
(test '(1 . (2 . (3))) '(1 2 3))

;; how is ...#(... parsed?
;(test (eval-string "'(# (1))") 'error)
(test (let ((lst (eval-string "'(#(1))"))) (and (= (length lst) 1) (vector? (car lst)))) #t)                     ; '(#(1))
(test (let ((lst (eval-string "'(-#(1))"))) (and (= (length lst) 2) (symbol? (car lst)) (pair? (cadr lst)))) #t) ; '(-# (1))
(test (let ((lst (eval-string "'(1#(1))"))) (and (= (length lst) 2) (symbol? (car lst)) (pair? (cadr lst)))) #t) ; '(1# (1))
(test (let ((lst (eval-string "'('#(1))"))) (and (= (length lst) 1) (vector? (cadar lst)))) #t)                  ; '((quote #(1)))
(test (let ((lst (eval-string "'(()#())"))) (and (= (length lst) 2) (null? (car lst)) (vector? (cadr lst)))) #t) ; '(() #())
(test (let ((lst (eval-string "'(().())"))) (and (= (length lst) 1) (null? (car lst)))) #t)                      ; '(())
(test (let ((lst (eval-string "'(()-())"))) (and (= (length lst) 3) (null? (car lst)) (null? (caddr lst)))) #t)  ; '(() - ())
(test (let ((lst (eval-string "'(().#())"))) (and (= (length lst) 3) (null? (car lst)) (null? (caddr lst)))) #t) ; '(() .# ())
(test (let ((lst (eval-string "'((). #())"))) (and (= (length lst) -1) (null? (car lst)) (vector? (cdr lst)))) #t) ; '(() . #())
(test (let ((lst (eval-string "'(\"\"#())"))) (and (= (length lst) 2) (string? (car lst)) (vector? (cadr lst)))) #t) ; '("" #())
(test (length (car '("#\\("))) 3)
(test (length (car '("#\\\""))) 3)
(test (char=? ((car '("#\\\"")) 2) #\") #t)
(test (length '(()#\(())) 3)
(test (length (eval-string "'(()#\\(())")) 3)
(test (char=? ((eval-string "'(()#\\#())") 1) #\#) #t)
(test (length (list""#t())) 3)
(test (length (list""#())) 2)
(test (length (eval-string "'(#xA(1))")) 2)
(test (length '(#xA""#(1))) 3)
(test (length (eval-string "'(#xA\"\"#(1))")) 3)
(test (length (eval-string "'(1#f)")) 1)
;(test (eval-string "'(#f#())") 'error)
(test (length '(#f())) 2)
(test (length '(#f"")) 2)
(test (eq? #f (eval-string "#F")) #f)
;(test (eval-string "'(#<eof>#<eof>)") 'error)
;(test (eval-string "'(#<eof>#())") 'error)
(test (equal? '('#()) '(#())) #f)
(test (equal? (list #()) '(#())) #t)
(test (equal? '(#()) '(#())) #t)
(test (equal? '('#()) '(`#())) #f) ;  [guile agrees]
(test (equal? '('()) '(`())) #f) ; ! quote != quasiquote [guile agrees]
(test (equal? '('(1)) '(`(1))) #t) ;  but lists are different? [guile says #f]
(test (equal? '('#(1)) '(`#(1))) #f) ; [guile agrees]
(test (equal? '('#()) '(#())) #f)
(test (equal? '(`#()) '(`#())) #t)
(test (equal? #() `#()) #t)
(test (equal? (list #()) (list `#())) #t)
(test (equal? (list #()) '(`#())) #t)
(test (equal? '(`#()) '(#())) #t)
(test (equal? `#() #()) #t) ; and also (1) () #(1) etc
(test (equal? `#() '#()) #t) ; "
(test (equal? '`#() ''#()) #f) ; it equals #() -- this is consistent -- see below
(test (equal? '`#() ``#()) #t)

(test (catch #t (lambda () (with-input-from-string "#0d()" read)) (lambda (type info) (apply format #f info)))
         "#nD(...) dimensions, 0, should be 1 or more")
(test (catch #t (lambda () (with-input-from-string "#1230d()" read)) (lambda (type info) (apply format #f info)))
         "reading #1230...: 1230 is too large, (*s7* 'max-vector-dimensions): 512")

(test (equal? '() '()) #t)
(test (equal? (quote ()) '()) #t)
(test (equal? '() (quote ())) #t)
(test (equal? (quote ()) (quote ())) #t)
(test (equal? `(1) '(1)) #t)
(test (equal? (quasiquote (1)) '(1)) #t)
(test (equal? `(1) (quote (1))) #t)
(test (equal? (quasiquote (1)) (quote (1))) #t)
(test (equal? ``''1 '``'1) #t)
(test (equal? (quasiquote `(quote (quote 1))) '``'1) #t)
(test (equal? ``''1 (quote ``(quote 1))) #t)
(test (equal? (quasiquote `(quote (quote 1))) (quote ``(quote 1))) #t)
(test (equal? '``'#f ```'#f) #t)
(test (equal? (quote ``(quote #f)) ```'#f) #t)
(test (equal? '``'#f (quasiquote ``(quote #f))) #t)
(test (equal? (quote ``(quote #f)) (quasiquote ``(quote #f))) #t)
;;; etc:

#|
(equal? (quote `1) (quote (quasiquote 1))) -> #f
the reader sees `1 and turns it into 1 in the first case, but does not collapse the 2nd case to 1
  (who knows, quasiquote might have been redefined in context... but ` can't be redefined):
:(define (` a) a)
;define: define a non-symbol? 'a
;    (define ('a) a)

this is different from guile which does not handle ` at read time except to expand it:

guile> (quote `1)
(quasiquote 1)

:(quote `1)
1

so anything that quotes ` is not going to equal quote quasiquote

(define (check-strs str1 str2)
  (for-each
   (lambda (arg)
     (let ((expr (format #f "(equal? ~A~A ~A~A)" str1 arg str2 arg)))
       (let ((val (catch #t
			 (lambda () (eval-string expr))
			 (lambda args 'error))))
	 (format #t "--------~%~S -> ~S" expr val)
	 (let* ((parens3 0)
		(parens4 0)
		(str3 (apply string-append (map (lambda (c)
						 (if (char=? c #\`)
						     (if (= parens3 0)
							 (begin
							   (set! parens3 (+ parens3 1))
							   "(quasiquote ")
							 "`")
						     (if (char=? c #\')
							 (begin
							   (set! parens3 (+ parens3 1))
							   "(quote ")
							 (string c))))
						str1)))
		(str4 (apply string-append (map (lambda (c)
						 (if (char=? c #\`)
						     (if (= parens4 0)
							 (begin
							   (set! parens4 (+ parens4 1))
							   "(quasiquote ")
							 "`")
						     (if (char=? c #\')
							 (begin
							   (set! parens4 (+ parens4 1))
							   "(quote ")
							 (string c))))
						str2))))
	   (let ((expr (format #f "(equal? ~A~A~A ~A~A)" str3 arg (make-string parens3 #\)) str2 arg)))
	     (let* ((val1 (catch #t
			       (lambda () (eval-string expr))
			       (lambda args 'error)))
		    (trouble (and (not (eq? val1 'error))
				  (not (eq? val1 val)))))
	       (if trouble
		   (format #t "~%~8T~A~S -> ~S~A" bold-text expr val1 unbold-text)
		   (format #t "~%~8T~S -> ~S" expr val1))))
	   (let ((expr (format #f "(equal? ~A~A ~A~A~A)" str1 arg str4 arg (make-string parens4 #\)))))
	     (let* ((val1 (catch #t
			       (lambda () (eval-string expr))
			       (lambda args 'error)))
		    (trouble (and (not (eq? val1 'error))
				  (not (eq? val1 val)))))
	       (if trouble
		   (format #t "~%~8T~A~S -> ~S~A" bold-text expr val1 unbold-text)
		   (format #t "~%~8T~S -> ~S" expr val1))))
	   (let ((expr (format #f "(equal? ~A~A~A ~A~A~A)" str3 arg (make-string parens3 #\)) str4 arg (make-string parens4 #\)))))
	     (let* ((val1 (catch #t
			       (lambda () (eval-string expr))
			       (lambda args 'error)))
		    (trouble (and (not (eq? val1 'error))
				  (not (eq? val1 val)))))
	       (if trouble
		   (format #t "~%~8T~A~S -> ~S~A~%" bold-text expr val1 unbold-text)
		   (format #t "~%~8T~S -> ~S~%" expr val1))))
	   ))))
   (list "()" "(1)" "#()" "#(1)" "1" "#f")))
   ;; (list ",(+ 1 2)" "\"\"" "(())" "#\\1" "3/4" ",1")

(check-strs "'" "'")
(check-strs "`" "'")
(check-strs "'" "`")
(check-strs "`" "`")

(let ((strs ()))
  (do ((i 0 (+ i 1)))
      ((= i 4))
    (let ((c1 ((vector #\' #\` #\' #\`) i))
	  (c2 ((vector #\' #\' #\` #\`) i)))
      (do ((k 0 (+ k 1)))
	  ((= k 4))
	(let ((d1 ((vector #\' #\` #\' #\`) k))
	      (d2 ((vector #\' #\' #\` #\`) k)))
	  (let ((str1 (string c1 c2))
		(str2 (string d1 d2)))
	    (if (not (member (list str1 str2) strs))
		(begin
		  (check-strs str1 str2)
		  (set! strs (cons (list str1 str2) strs))
		  (set! strs (cons (list str2 str1) strs))))))))))

(let ((strs ()))
  (do ((i 0 (+ i 1)))
      ((= i 8))
    (let ((c1 ((vector #\' #\` #\' #\` #\' #\` #\' #\`) i))
	  (c2 ((vector #\' #\' #\` #\` #\' #\' #\` #\`) i))
	  (c3 ((vector #\' #\' #\' #\' #\` #\` #\` #\`) i)))
      (do ((k 0 (+ k 1)))
	  ((= k 8))
	(let ((d1 ((vector #\' #\` #\' #\` #\' #\` #\' #\`) k))
	      (d2 ((vector #\' #\' #\` #\` #\' #\' #\` #\`) k))
	      (d3 ((vector #\' #\' #\' #\' #\` #\` #\` #\`) k)))
	  (let ((str1 (string c1 c2 c3))
		(str2 (string d1 d2 d3)))
	    (if (not (member (list str1 str2) strs))
		(begin
		  (check-strs str1 str2)
		  (set! strs (cons (list str1 str2) strs))
		  (set! strs (cons (list str2 str1) strs))))))))))


;;; --------------------------------

(do ((i 0 (+ i 1)))
    ((= i 256))
  (if (and (not (= i (char->integer #\))))
	   (not (= i (char->integer #\"))))
      (let ((str (string #\' #\( #\1 #\space #\. (integer->char i) #\2 #\))))
	(catch #t
	       (lambda ()
		 (let ((val (eval-string str)))
		   (format #t "[~D] ~A -> ~S (~S ~S)~%" i str val (car val) (cdr val))))
	       (lambda args
		 (format #t "[~D] ~A -> ~A~%" i str args))))))

(let ((chars (vector (integer->char 0) #\newline #\space #\tab #\. #\, #\@ #\= #\x #\b #\' #\`
		     #\# #\] #\[ #\} #\{ #\( #\) #\1 #\i #\+ #\- #\e #\_ #\\ #\" #\: #\; #\> #\<)))
  (let ((nchars (vector-length chars)))
    (do ((len 2 (+ len 1)))
	((= len 3))
      (let ((str (make-string len))
	    (ctrs (make-vector len 0)))

	(do ((i 0 (+ i 1)))
	    ((= i (expt nchars len)))

	  (let ((carry #t))
	    (do ((k 0 (+ k 1)))
		((or (= k len)
		     (not carry)))
	      (vector-set! ctrs k (+ 1 (vector-ref ctrs k)))
	      (if (= (vector-ref ctrs k) nchars)
		  (vector-set! ctrs k 0)
		  (set! carry #f)))
	    (do ((k 0 (+ k 1)))
		((= k len))
	      (string-set! str k (vector-ref chars (vector-ref ctrs k)))))

	  (format #t "~A -> " str)
	  (catch #t
		 (lambda ()
		   (let ((val (eval-string str)))
		     (format #t " ~S -> ~S~%" str val)))
		 (lambda args
		   ;(format #t " ~A~%" args)
		   #f
		   )))))))
|#

(let ((äåæéîå define)
      (ìåîçôè length)
      (äï do)
      (ìåô* let*)
      (éæ if)
      (áâó abs)
      (ìïç log)
      (óåô! set!))

  (äåæéîå (óòã-äõòáôéïî å)
    (ìåô* ((ìåî (ìåîçôè å))
           (åø0 (å 0))
           (åø1 (å (- ìåî 2)))
           (áìì-ø (- åø1 åø0))
           (äõò 0.0))
      (äï ((é 0 (+ é 2)))
          ((>= é (- ìåî 2)) äõò)
        (ìåô* ((ø0 (å é))
               (ø1 (å (+ é 2)))
               (ù0 (å (+ é 1))) ; 1/ø ø ðïéîôó
               (ù1 (å (+ é 3)))
               (áòåá (éæ (< (áâó (- ù0 ù1)) .0001)
                         (/ (- ø1 ø0) (* ù0 áìì-ø))
                         (* (/ (- (ìïç ù1) (ìïç ù0))
                               (- ù1 ù0))
                            (/ (- ø1 ø0) áìì-ø)))))
         (óåô! äõò (+ äõò (áâó áòåá)))))))

  (num-test (óòã-äõòáôéïî (list 0 1 1 2)) 0.69314718055995)
  (num-test (óòã-äõòáôéïî (vector 0 1 1 2)) 0.69314718055995))

(test (let ((ÿa 1)) ÿa) 1)
(test (+ (let ((!a 1)) !a) (let (($a 1)) $a) (let ((%a 1)) %a) (let ((&a 1)) &a) (let ((*a 1)) *a) (let ((+a 1)) +a) (let ((-a 1)) -a) (let ((.a 1)) .a) (let ((/a 1)) /a) (let ((0a 1)) 0a) (let ((1a 1)) 1a) (let ((2a 1)) 2a) (let ((3a 1)) 3a) (let ((4a 1)) 4a) (let ((5a 1)) 5a) (let ((6a 1)) 6a) (let ((7a 1)) 7a) (let ((8a 1)) 8a) (let ((9a 1)) 9a) (let ((<a 1)) <a) (let ((=a 1)) =a) (let ((>a 1)) >a) (let ((?a 1)) ?a) (let ((@a 1)) @a) (let ((Aa 1)) Aa) (let ((Ba 1)) Ba) (let ((Ca 1)) Ca) (let ((Da 1)) Da) (let ((Ea 1)) Ea) (let ((Fa 1)) Fa) (let ((Ga 1)) Ga) (let ((Ha 1)) Ha) (let ((Ia 1)) Ia) (let ((Ja 1)) Ja) (let ((Ka 1)) Ka) (let ((La 1)) La) (let ((Ma 1)) Ma) (let ((Na 1)) Na) (let ((Oa 1)) Oa) (let ((Pa 1)) Pa) (let ((Qa 1)) Qa) (let ((Ra 1)) Ra) (let ((Sa 1)) Sa) (let ((Ta 1)) Ta) (let ((Ua 1)) Ua) (let ((Va 1)) Va) (let ((Wa 1)) Wa) (let ((Xa 1)) Xa) (let ((Ya 1)) Ya) (let ((Za 1)) Za) (let (([a 1)) [a) (let ((\a 1)) \a) (let ((]a 1)) ]a) (let ((^a 1)) ^a) (let ((_a 1)) _a) (let ((aa 1)) aa) (let ((ba 1)) ba) (let ((ca 1)) ca) (let ((da 1)) da) (let ((ea 1)) ea) (let ((fa 1)) fa) (let ((ga 1)) ga) (let ((ha 1)) ha) (let ((ia 1)) ia) (let ((ja 1)) ja) (let ((ka 1)) ka) (let ((la 1)) la) (let ((ma 1)) ma) (let ((na 1)) na) (let ((oa 1)) oa) (let ((pa 1)) pa) (let ((qa 1)) qa) (let ((ra 1)) ra) (let ((sa 1)) sa) (let ((ta 1)) ta) (let ((ua 1)) ua) (let ((va 1)) va) (let ((wa 1)) wa) (let ((xa 1)) xa) (let ((ya 1)) ya) (let ((za 1)) za) (let (({a 1)) {a) (let ((|a 1)) |a) (let ((}a 1)) }a) (let ((~a 1)) ~a) (let (( a 1))  a) (let ((¡a 1)) ¡a) (let ((¢a 1)) ¢a) (let ((£a 1)) £a) (let ((¤a 1)) ¤a) (let ((¥a 1)) ¥a) (let ((¦a 1)) ¦a) (let ((§a 1)) §a) (let ((¨a 1)) ¨a) (let ((©a 1)) ©a) (let ((ªa 1)) ªa) (let ((«a 1)) «a) (let ((¬a 1)) ¬a) (let ((­a 1)) ­a) (let ((®a 1)) ®a) (let ((¯a 1)) ¯a) (let ((°a 1)) °a) (let ((±a 1)) ±a) (let ((²a 1)) ²a) (let ((³a 1)) ³a) (let ((´a 1)) ´a) (let ((µa 1)) µa) (let ((¶a 1)) ¶a) (let ((·a 1)) ·a) (let ((¸a 1)) ¸a) (let ((¹a 1)) ¹a) (let ((ºa 1)) ºa) (let ((»a 1)) »a) (let ((¼a 1)) ¼a) (let ((½a 1)) ½a) (let ((¾a 1)) ¾a) (let ((¿a 1)) ¿a) (let ((Àa 1)) Àa) (let ((Áa 1)) Áa) (let ((Âa 1)) Âa) (let ((Ãa 1)) Ãa) (let ((Äa 1)) Äa) (let ((Åa 1)) Åa) (let ((Æa 1)) Æa) (let ((Ça 1)) Ça) (let ((Èa 1)) Èa) (let ((Éa 1)) Éa) (let ((Êa 1)) Êa) (let ((Ëa 1)) Ëa) (let ((Ìa 1)) Ìa) (let ((Ía 1)) Ía) (let ((Îa 1)) Îa) (let ((Ïa 1)) Ïa) (let ((Ða 1)) Ða) (let ((Ña 1)) Ña) (let ((Òa 1)) Òa) (let ((Óa 1)) Óa) (let ((Ôa 1)) Ôa) (let ((Õa 1)) Õa) (let ((Öa 1)) Öa) (let ((×a 1)) ×a) (let ((Øa 1)) Øa) (let ((Ùa 1)) Ùa) (let ((Úa 1)) Úa) (let ((Ûa 1)) Ûa) (let ((Üa 1)) Üa) (let ((Ýa 1)) Ýa) (let ((Þa 1)) Þa) (let ((ßa 1)) ßa) (let ((àa 1)) àa) (let ((áa 1)) áa) (let ((âa 1)) âa) (let ((ãa 1)) ãa) (let ((äa 1)) äa) (let ((åa 1)) åa) (let ((æa 1)) æa) (let ((ça 1)) ça) (let ((èa 1)) èa) (let ((éa 1)) éa) (let ((êa 1)) êa) (let ((ëa 1)) ëa) (let ((ìa 1)) ìa) (let ((ía 1)) ía) (let ((îa 1)) îa) (let ((ïa 1)) ïa) (let ((ða 1)) ða) (let ((ña 1)) ña) (let ((òa 1)) òa) (let ((óa 1)) óa) (let ((ôa 1)) ôa) (let ((õa 1)) õa) (let ((öa 1)) öa) (let ((÷a 1)) ÷a) (let ((øa 1)) øa) (let ((ùa 1)) ùa) (let ((úa 1)) úa) (let ((ûa 1)) ûa) (let ((üa 1)) üa) (let ((ýa 1)) ýa) (let ((þa 1)) þa) (let ((ÿa 1)) ÿa)) 181)

;;; there are about 50 non-printing chars, some of which would probably work as well


;; (eval-string "(eval-string ...)") is not what it appears to be -- the outer call
;;    still sees the full string when it evaluates, not the string that results from
;;    the inner call.


(let () ; from scheme bboard
  (define (maxlist list)
    (define (maxlist' l max)
      (if (null? l) max
	  (if (> (car l) max)
	      (maxlist' (cdr l) (car l))
	      (maxlist' (cdr l) max))))
    (if (null? list) 'undef
	(maxlist' list (car list))))
  (test (maxlist '(1 2 3)) 3) ; quote is ok in s7 if not the initial char (sort of like a number)

  (let ((h'a 3))
    (test h'a 3))
  (let ((1'2 32))
    (test 1'2 32))
  (let ((1'`'2 32))
    (test 1'`'2 32))
  (let ((1'`,@2 32))
    (test 1'`,@2 32))

;  (test (define '3 32) 'error) ;define quote: syntactic keywords tend to behave badly if redefined
  )

(let ((|,``:,*|',## 1)
      (0,,&:@'>>.<# 2)
      (@.*0`#||\<,, 3)
      (*&:`&'>#,*<` 4)
      (*0,,`&|#*:`> 5)
      (>:|<*.<@:\|` 6)
      (*',>>:.'@,** 7)
      (0|.'@<<:,##< 8)
      (<>,\',\.>>#` 9)
      (@#.>|&#&,\0* 10)
      (0'.`&<','<<. 11)
      (&@@*<*\'&|., 12)
      (|0*&,':|0\** 13)
      (<:'*@<>*,<&` 14)
      (>@<@<|>,`&'. 15)
      (@#,00:<:@*.\ 16)
      (*&.`\>#&,&., 17)
      (0|0|`,,..<@, 18)
      (0@,'>\,,&.@# 19)
      (>@@>,000`\#< 20)
      (|>*'',<:&@., 21)
      (|>,0>0|,@'|. 22)
      (0,`'|'`,:`@` 23)
      (<>#'>,,\'.'& 24)
      (*..,|,.,&&@0 25))
  (test (+ |,``:,*|',## 0,,&:@'>>.<# @.*0`#||\<,, *&:`&'>#,*<` *0,,`&|#*:`> >:|<*.<@:\|` *',>>:.'@,**
	   0|.'@<<:,##< <>,\',\.>>#` @#.>|&#&,\0* 0'.`&<','<<.  &@@*<*\'&|., |0*&,':|0\** <:'*@<>*,<&`
           >@<@<|>,`&'. @#,00:<:@*.\ *&.`\>#&,&., 0|0|`,,..<@, 0@,'>\,,&.@# >@@>,000`\#<
           |>*'',<:&@., |>,0>0|,@'|. 0,`'|'`,:`@` <>#'>,,\'.'& *..,|,.,&&@0)
	325))

(when full-s7test
  (let ((first-chars (list #\. #\0 #\@ #\! #\& #\| #\* #\< #\>))
	(rest-chars (list #\. #\0 #\@ #\! #\| #\, #\# #\' #\\ #\` #\, #\: #\& #\* #\< #\>)))
    (let ((first-len (length first-chars))
	  (rest-len (length rest-chars)))
      (let ((n 100)
	    (size 12))
	(let ((str (make-string size #\space)))
	  (do ((i 0 (+ i 1)))
	      ((= i n))
	    (set! (str 0) (first-chars (random first-len)))
	    (do ((k 1 (+ 1 k)))
		((= k size))
	      (set! (str k) (rest-chars (random rest-len))))
	    (catch #t (lambda ()
			(let ((val (eval-string (format #f "(let () (define ~A 3) ~A)" str str))))
			  (format #f "~A -> ~A~%" str val)))
		   (lambda args
		     (format #f "~A error: ~A~%" str args)))))))))

(let ((List 1)
      (LIST 2)
      (lIsT 3)
      (-list 4)
      (_list 5)
      (+list 6))
  (test (apply + (list List LIST lIsT -list _list +list)) 21))

(let ()
  (define (\ arg) (+ arg 1))
  (test (+ 1 (\ 2)) 4)
  (define (@\ arg) (+ arg 1))
  (test (+ 1 (@\ 2)) 4)
  (define (@,\ arg) (+ arg 1))
  (test (+ 1 (@,\ 2)) 4)
  (define (\,@\ arg) (+ arg 1))
  (test (+ 1 (\,@\ 2)) 4)
  )

;;; these are from the r7rs discussions
(test (let ((a'b 3)) a'b) 3) ; allow variable names like "can't-go-on" or "don't-ask"
(test (let () (define (f x y) (+ x y)) (let ((a 3) (b 4)) (f a, b))) 'error) ; unbound variable a,
(test (let () (define (f x y) (+ x y)) (let ((a 3) (b 4)) (f a ,b))) 'error) ; unquote outside quasiquote

(test (vector? (owlet 0. 3/4 #(reader-cond ))) 'error)
(test (vector? #(reader-cond)) #t)


;;; -------- object->string
;;; object->string

(test (string=? (object->string 32) "32") #t)
(test (string=? (object->string 32.5) "32.5") #t)
(test (string=? (object->string 32/5) "32/5") #t)
(test (object->string 1+i) "1.0+1.0i")
(test (string=? (object->string "hiho") "\"hiho\"") #t)
(test (string=? (object->string 'symb) "symb") #t)
(test (string=? (object->string (list 1 2 3)) "(1 2 3)") #t)
(test (string=? (object->string (cons 1 2)) "(1 . 2)") #t)
(test (string=? (object->string #(1 2 3)) "#(1 2 3)") #t)
(test (string=? (object->string +) "+") #t)
(test (object->string (object->string (object->string "123"))) "\"\\\"\\\\\\\"123\\\\\\\"\\\"\"")
(test (object->string #<eof>) "#<eof>")
(test (object->string (if #f #f)) "#<unspecified>")
(test (object->string #<undefined>) "#<undefined>")
(test (object->string _undef_) "#_asdf")
(test (object->string #f) "#f")
(test (object->string #t) "#t")
(test (object->string ()) "()")
(test (object->string #()) "#()")
(test (object->string "") "\"\"")
(test (object->string abs) "abs")
(test (object->string lambda) "lambda")
(test (object->string (lambda () a)) "#<lambda ()>")
(test (object->string (lambda a a)) "#<lambda a>")
(test (object->string (lambda (a) a)) "#<lambda (a)>")
(test (object->string (lambda (a . b) a)) "#<lambda (a . b)>")
(test (object->string (lambda (a b) a)) "#<lambda (a b)>")
(test (object->string (lambda (a b c) a)) "#<lambda (a b ...)>")
(test (object->string (lambda (a b . c) a)) "#<lambda (a b ...)>")
(test (object->string (lambda* (a :rest b) a)) "#<lambda* (a :rest b)>")
(test (object->string (lambda* (:rest a) a)) "#<lambda* (:rest a)>")
(test (object->string (lambda* (a b :rest c) a)) "#<lambda* (a b ...)>")
(let () (define-macro (mac a) a) (test (object->string mac) "mac"))
(let ((m (macro (a) a))) (test (object->string m) "#<macro (a)>"))
(let ((m (macro* (a) a))) (test (object->string m) "#<macro* (a)>"))
(let ((m (bacro (a) a))) (test (object->string m) "#<bacro (a)>"))
(let ((m (bacro* (a) a))) (test (object->string m) "#<bacro* (a)>"))
(let ((_?_m (define-expansion (_?_mac a) a))) (test (object->string _?_m) "_?_mac"))
(let ((_?_m1 (define-expansion* (_?_mac (a 0)) a))) (test (object->string _?_m1) "_?_mac"))
(test (object->string +) "+")
(test (object->string +) "+")
(test (object->string '''2) "''2")
(test (object->string (lambda () #f)) "#<lambda ()>") ;"#<closure>"
(test (call-with-exit (lambda (return) (object->string return))) "#<goto return>")
(test (call/cc (lambda (return) (object->string return))) "#<continuation return>")
(test (let () (define-macro (hi a) `(+ 1 ,a)) (object->string hi)) "hi")
(test (let () (define (hi a) (+ 1 a)) (object->string hi)) "hi")
(test (let () (define* (hi a) (+ 1 a)) (object->string hi)) "hi")
(test (object->string dynamic-wind) "dynamic-wind")
(test (object->string (dilambda (lambda () 1) (lambda (val) val))) "#<lambda ()>") ;"#<closure>"
(test (object->string object->string) "object->string")
(test (object->string 'if) "if")
(test (object->string begin) "begin")
(test (object->string let) "let")

(test (object->string #\n #f) "n")
(test (object->string #\n) "#\\n")
(test (object->string #\r) "#\\r")
(test (object->string #\r #f) "r")
(test (object->string #\t #f) "t")
(test (object->string #\t) "#\\t")

(test (object->string #\a) "#\\a")
(test (object->string #\a #t) "#\\a")
(test (object->string #\a :write) "#\\a")
(test (object->string #\a #f) "a")
(test (object->string #\a :display) "a")

(test (object->string "a\x00;b" #t) "\"a\\x00;b\"")
(test (object->string "a\x00;b" #f) "a\x00;b")

(let-temporarily (((*s7* 'print-length) 3))
  (test (object->string (inlet :a 1 :b 2 :c 3 :d 4)) "(inlet 'a 1 'b 2 'c 3 ...)")
  (test (object->string (vector 1 2 3 4)) "#(1 2 3 ...)")
  (test (object->string #r(1 2 3 4)) "#r(1.0 2.0 3.0 ...)")
  (test (object->string #i(1 2 3 4)) "#i(1 2 3 ...)")
  (test (object->string (list 1 2 3 4)) "(1 2 3 ...)")
  (test (object->string (list (list 1 2 3 4 5 6)
			      (list 1 2 3 4 5 6)
			      (list 1 2 3 4 5 6)
			      (list 1 2 3 4 5 6)
			      (list 1 2 3 4 5 6)))
	"((1 2 3 ...) (1 2 3 ...) (1 2 3 ...) ...)"))
;;; hash-tables with entries>1 are hard to check -- order of entries is not known
(test (object->string (hash-table 'a 1) :readable) "(hash-table 'a 1)")
(test (object->string (hash-table) :readable) "(hash-table)")
(test (let ((h (object->string (hash-table 'a 1 'b 2) :readable)))
	(or (string=? h "(hash-table 'a 1 'b 2)")
	    (string=? h "(hash-table 'b 2 'a 1)")))
      #t)
(test (object->string (make-float-vector '(2 3) 1)) "#r2d((1.0 1.0 1.0) (1.0 1.0 1.0))")
(test (object->string (immutable! #r(0.0)) :readable) "(immutable! #r(0.0))")
(test (object->string (immutable! #i(0 1)) :readable) "(immutable! #i(0 1))")
(test (object->string (immutable! #u(0)) :readable) "(immutable! #u(0))")
(test (let-temporarily (((*s7* 'print-length) 0)) (object->string #u(1 2 3))) "#u(...)")
(test (object->string (immutable! (make-string 1001 #\space)) :readable) "(immutable! (make-string 1001 #\\space))")
(test (object->string #r(1 2 3 4) :readable 4) 'error)
(test (object->string #r(1 2 3 4) :readable -4) 'error)
(test (object->string #r(1 2 3 4) :readable most-negative-fixnum) 'error)
(test (object->string #r(1 2 3 4) :readable 40) "#r(1.0 2.0 3.0 4.0)")
(test (object->string #r(1 2 3 4) :readable most-positive-fixnum) "#r(1.0 2.0 3.0 4.0)")
(test (object->string #r(1 2 3 4) :readable "hi") 'error)
(test (let ((s (immutable! "abc"))) (define (f) (object->string (string-append (substring s 3)) :readable)) (f)) "\"\"")

#|
(do ((i 0 (+ i 1)))
    ((= i 256))
  (let ((c (integer->char i)))
    (let ((str (object->string c)))
      (if (and (not (= (length str) 3))       ; "#\\a"
	       (or (not (char=? (str 2) #\x))
		   (not (= (length str) 5)))) ; "#\\xee"
	  (format #t "(#t) ~C: ~S~%" c str))
      (set! str (object->string c #f))
      (if (not (= (length str) 1))
	  (format #t "(#f) ~C: ~S~%" c str)))))
this prints:
(#t) : "#\\null"
(#f) : ""
(#t) : "#\\x1"
(#t) : "#\\x2"
(#t) : "#\\x3"
(#t) : "#\\x4"
(#t) : "#\\x5"
(#t) : "#\\x6"
(#t) : "#\\x7"
(#t): "#\\x8"
(#t) 	: "#\\tab"
(#t)
: "#\\newline"
(#t)
     : "#\\xb"
(#t)
     : "#\\xc"
: "#\\return"
(#t) : "#\\xe"
(#t) : "#\\xf"
(#t)  : "#\\space"
|#

(test (object->string #\x30) "#\\0")
(test (object->string #\x91) "#\\x91")
(test (object->string #\x10) "#\\x10")
(test (object->string #\xff) "#\\xff")
(test (object->string #\x55) "#\\U")
(test (object->string #\x7e) "#\\~")
(test (object->string #\newline) "#\\newline")
(test (object->string #\return) "#\\return")
(test (object->string #\tab) "#\\tab")
(test (object->string #\null) "#\\null")
(test (object->string #\space) "#\\space")
(test (object->string (integer->char 8)) "#\\backspace")
(test (object->string ''#\a) "'#\\a")
(test (object->string (list 1 '.' 2)) "(1 .' 2)")
(test (object->string (quote (quote))) "(quote)")
(test (object->string (quote quote)) "quote")
(test (object->string (quote (quote (quote)))) "'(quote)")

(test (object->string) 'error)
(test (object->string 1 2) 'error)
(test (object->string 1 #f #t) 'error)
;(test (object->string 1 #t -123) 'error)
(test (object->string 1 #t pi) 'error)
(test (object->string abs) "abs")
(test(let ((val 0)) (cond (else (set! val (object->string else)) 1)) val) "else")
(test (cond (else (object->string else))) "else")
(test (object->string (string->symbol (string #\; #\" #\)))) "(symbol \";\\\")\")")

(test (object->string "hi" #f) "hi")
(test (object->string "h\\i" #f) "h\\i")
(test (object->string -1.(list? -1e0)) "-1.0")

(test (object->string catch) "catch")
(test (object->string lambda) "lambda")
(test (object->string dynamic-wind) "dynamic-wind")
(test (object->string quasiquote) "quasiquote")
;(test (object->string else) "else") ; this depends on previous code
(test (object->string do) "do")

(for-each
 (lambda (arg)
   (test (object->string 1 arg) 'error)
   (test (object->string arg) (with-output-to-string (lambda () (write arg))))
   (test (object->string arg #t) (with-output-to-string (lambda () (write arg))))
   (test (object->string arg #f) (with-output-to-string (lambda () (display arg)))))
 (list "hi" -1 #\a 1 0 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i () (list 1 2 3) '(1 . 2)))

(test (string->symbol (object->string #(1 #\a (3)) #f)) (symbol "#(1 #\\a (3))"))
(test (string->list (object->string #(1 2) #f)) '(#\# #\( #\1 #\space #\2 #\)))
(test (string->list (object->string #(1 #\a (3)) #f)) '(#\# #\( #\1 #\space #\# #\\ #\a #\space #\( #\3 #\) #\)))
(test (reverse (object->string #2d((1 2) (3 4)) #f))  "))4 3( )2 1((d2#")

;; write readably (this affects ~W in format as well)
;; :readable special things
(for-each
 (lambda (n)
   (let ((str (object->string n :readable)))
     (let ((obj (with-input-from-string str
		  (lambda ()
		    (eval (read))))))
       (if (not (eq? n obj))
	   (format *stderr* "~A not eq? ~A (~S)~%" n obj str)))))
 (list #<eof> #<undefined> #<unspecified> #t #f #true #false else ()
       lambda lambda* begin case if do quote set! let let* letrec
       cond and or define define* define-constant define-macro
       define-macro* define-bacro define-bacro*
       with-baffle
       *stdin* *stdout* *stderr*))

;; :readable characters
(do ((i 0 (+ i 1)))
    ((= i 256))
  (let ((c (integer->char i)))
    (let ((str (object->string c :readable)))
      (let ((nc (with-input-from-string str
		  (lambda ()
		    (eval (read)))))) ; no need for eval here or in some other cases, but might as well be consistent
	(if (not (eq? c nc))
	    (format *stderr* "~C (~D) != ~C (~S)~%" c i nc str))))))

;; :readable integers
(for-each
 (lambda (n)
   (let ((str (object->string n :readable)))
     (let ((nn (with-input-from-string str
		 (lambda ()
		   (eval (read))))))
       (if (or (not (integer? n))
	       (not (integer? nn))
	       (not (= n nn)))
	   (format *stderr* "~D != ~D (~S)~%" n nn str)))))
 (list 0 1 3 most-positive-fixnum -0 -1 -3 most-negative-fixnum
       -9223372036854775808 9223372036854775807))
;; but unless gmp at read end we'll fail with most-positive-fixnum+1
;; -> check *features* at start of read

;; :readable ratios
(for-each
 (lambda (n)
   (let ((str (object->string n :readable)))
     (let ((nn (with-input-from-string str
		 (lambda ()
		   (eval (read))))))
       (if (or (not (rational? n))
	       (not (rational? nn))
	       (not (= n nn)))
	   (format *stderr* "~A != ~A (~S)~%" n nn str)))))
 (list 1/2 -1/2 123456789/2 -2/123456789 2147483647/2147483646 312689/99532
       -9223372036854775808/3 9223372036854775807/2  1/1428571428571429 1/1152921504606846976))

(unless (provided? 'solaris)
  ;; :readable reals
  (for-each
   (lambda (n)
     (let ((str (object->string n :readable)))
       (let ((nn (with-input-from-string str
		   (lambda ()
		     (eval (read))))))
	 (if (or (not (real? n))
		 (not (real? nn))
		 (not (equivalent? n nn)))
	     (format *stderr* "~A != ~A (~S)~%" n nn str)))))
   (list 1.0 0.0 -0.0 pi 0.1 -0.1 0.9999999995 9007199254740993.1 (sqrt 2) 1/100000000000
	 1.5e-16 1.5e16 3.141592653589793238462643383279502884197169399375105820 1e-300 8.673617379884e-19
	 1/0 (- 1/0) (real-part (log 0)) (- (real-part (log 0)))))

  ;; :readable complex
  (for-each
   (lambda (n)
     (let ((str (object->string n :readable)))
       (let ((nn (with-input-from-string str
		   (lambda ()
		     (eval (read))))))
	 (if (or (not (complex? n))
		 (not (complex? nn))
		 (not (equivalent? n nn)))
	     (format *stderr* "~A != ~A (~S)~%" n nn str)))))
   (list 0+i 0-i 1+i 1.4+i 3.0+1.5i
;	 (log 0) (- (log 0))
;	 (complex 1/0 1.0) (complex 1/0 1/0) (complex 1.0 1/0) ; default: nan+1i nannani 1nani!
;	 (complex 1/0 (real-part (log 0))) (complex (real-part (log 0)) 1/0)
	 1e-14+1e14i 0+1e-16i (complex pi pi))))


;; :readable strings/byte-vectors
(for-each
 (lambda (n)
   (let ((str (object->string n :readable)))
     (let ((obj (with-input-from-string str
		  (lambda ()
		    (eval (read))))))
       (if (or (not (string? n))
	       (not (string? obj))
	       (not (string=? n obj)))
	   (format *stderr* "~S not string=? ~S (~S)~%" n obj str)))))
 (list "" "abc" (string #\newline) "#<abc>" "a\"b\"c" "a\\b\nc" "aBc"
       (let ((s (make-string 4 #\space))) (set! (s 3) #\null) s) ; writes as "   \x00"
       "ab
c"
       (string #\a #\b #\null #\c #\escape #\newline)
       (string #\x (integer->char #xf0) #\x)
       (string #\null)
       ;#u() #u(0 1 2 3)
       (let ((str (make-string 256 #\null)))
	 (do ((i 0 (+ i 1)))
	     ((= i 256) str)
	   (set! (str i) (integer->char i))))))

;; :readable symbols/keywords
(catch #t
  (lambda()
    (for-each
     (lambda (n)
       (let ((str (object->string n :readable)))
	 (let ((obj (with-input-from-string str
		      (lambda ()
			(eval (read))))))
	   (if (or (not (symbol? n))
		   (not (symbol? obj))
		   (not (eq? n obj)))
	       (format *stderr* "~A not eq? ~A (~S)~%" n obj str)))))
     (list 'abc :abc abc:
	   (symbol "a") (symbol "#<>")
	   (gensym "|") (gensym "#<>") (gensym "}")
	   ':: ':abc
	   (gensym "\\"))))
  (lambda (type info)
    (format *stderr* "readable symbols: ~A ~A~%" type info)))

;; :readable environments
(for-each
 (lambda (n)
   (let ((str (object->string n :readable)))
     (let ((obj (with-input-from-string str
		  (lambda ()
		    (eval (read))))))
       (if (or (not (let? n))
	       (not (let? obj))
	       (not (equal? n obj)))
	   (format *stderr* "~A not equal?~%~A~%    (~S)~%" n obj str)))))
 (list (inlet '(a . 1))
       (inlet)
       (rootlet)
       (inlet (cons 't12 "12") (cons (symbol "#<") 12))
       (inlet 'a 1 'a 2)))

;(test (object->string (list (owlet)) :readable) "(list (owlet))")

;; :readable hash-tables
(for-each
 (lambda (n)
   (let ((str (object->string n :readable)))
     (let ((obj (with-input-from-string str
		  (lambda ()
		    (eval (read))))))
       (if (or (not (hash-table? n))
	       (not (hash-table? obj))
	       (not (equal? n obj)))
	   (format *stderr* ";readable hash-tables, ~A not equal? ~A (~S)~%" n obj str)))))
 (list (hash-table 'a 1)
       (hash-table 'a 1 'b "hi")
       (let ((ht (make-hash-table 31)))
	 (set! (ht 1) 321)
	 (set! (ht 2) 123)
	 ht)
       (let ((ht (make-hash-table)))
	 (set! (ht 'b) 1)
	 (set! (ht 'a) ht)
	 ht)
       ;(let ((ht (make-hash-table))) (set! (ht ht) 123) ht)
       ;(let ((ht (make-hash-table))) (set! (ht ht) ht)  ht)
       (hash-table)))

;; :readable vectors
(let-temporarily (((*s7* 'print-length) 8))
  (for-each
   (lambda (p)
     (set! (*s7* 'print-length) p)
     (for-each
      (lambda (n)
	(let ((str (object->string n :readable)))
	  (let ((obj (with-input-from-string str
		       (lambda ()
			 (eval (read))))))
	    (if (or (not (vector? n))
		    (not (vector? obj))
		    (not (equal? n obj)))
		(format *stderr* ";readable vectors, ~A not equal? ~A (~S)~%" n obj str)))))
      (list #() #(1) #(1 #(2)) #2d((1 2) (3 4))
	    #3d(((1 2 3) (4 5 6) (7 8 9)) ((9 8 7) (6 5 4) (3 2 1)))
	    #2d()
	    #(1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0)
	    (let ((v (vector 1 2 3))) (set! (v 1) v) v)
	    (let ((v (vector 1 #(2) 3))) (set! ((v 1) 0) v) v)
	    (let ((v #2d((1 2 3) (4 5 6)))) (set! (v 1 1) v) v)
	    (make-int-vector 3 0)
	    (make-float-vector 3 0.0)
	    (make-int-vector '(2 3) 1))))
   (list 8 2 1)))

(test (object->string (vector 1 2 3) :readable) "(vector 1 2 3)")
(let ((v (make-vector '(2 3) #f)))
  (set! (v 1 0) (v 0))
  (test (object->string v) "#2d((#f #f #f) (#(#f #f #f) #f #f))")
  (set! (v 0 1) 3)
  (test (object->string v) "#2d((#f 3 #f) (#(#f 3 #f) #f #f))"))

;; :readable lists (circular, dotted)
(for-each
 (lambda (n)
   (let ((str (object->string n :readable)))
     (let ((obj (with-input-from-string str
		  (lambda ()
		    (eval (read))))))
       (if (or (not (pair? n))
	       (not (pair? obj))
	       (not (equal? n obj)))
	   (format *stderr* ";readable lists, ~A not equal? ~A (~S)~%" n obj str)))))
 (list '(1) '(1 . 2) '((1 ()) 3) '((1 2) (3 4))
       '(1 2 . 3) '(1 2 3 . 4) '(())
       (let ((lst (cons 1 2))) (set-cdr! lst lst) lst)
       (let ((lst (list 1 2 3))) (set-cdr! (cddr lst) lst) lst)
       (let ((lst (list 1 2 3))) (set-car! (cddr lst) lst) lst)
       ))

;; :readable macros
(let ()
  (define-macro (mac1) `(+ 1 2))
  (test ((eval-string (object->string mac1 :readable))) 3)
  (define-macro (mac2 a) `(+ ,a 2))
  (test ((eval-string (object->string mac2 :readable)) 1) 3)
  (define-macro* (mac3 (a 1)) `(+ ,a 2))
  (test ((eval-string (object->string mac3 :readable))) 3)
  (define-macro (mac4 . a) `(+ ,@a 2))
  (test ((eval-string (object->string mac4 :readable)) 1 3) 6)
  (define-macro (mac5 a b . c) `(+ ,a ,b ,@c 2))
  (test ((eval-string (object->string mac5 :readable)) 1 5 3 4) 15)
  (define-macro (mac7 a) (let ((b (+ a 1))) `(+ ,b ,a)))
  (test ((eval-string (object->string mac7 :readable)) 2) 5)
  )

;; :readable closures/functions/built-in (C) functions + the setters thereof
(for-each
 (lambda (n)
   (let ((str (object->string n :readable)))
     (let ((obj (with-input-from-string str
		  (lambda ()
		    (eval