/*
 * This file is part of the sn package.
 * Distribution of sn is covered by the GNU GPL. See file COPYING.
 * Copyright  1998-2000 Harold Tay.
 * Copyright  2000- Patrik Rdman.
 */

/*
 * Read-only portion of disk hashing.  See dhash.c
 */

#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <string.h>
#include "config.h"
#include "dhash.h"
#include "allocate.h"
#include "newsgroup.h"

#include <out.h>
#include <nap.h>

int dh_fd = -1;

char dh_tablefile[sizeof (DH_TABLEFILE) + 32];
char dh_chainfile[sizeof (DH_TABLEFILE) + 32];
char dh_groupfile[sizeof (DH_GROUPFILE) + 32];
char dh_tmp[sizeof (DH_GROUPFILE) + 32];
int dh_isreadonly = 0;

static const char rcsid[] = "$Id$";

struct table *dh_table = NULL;

/* General string hashing function, taken from djb */

unsigned int dhhash (char *buf)
{
   unsigned int h;
   int len = strlen(buf);

   h = 5381;
   while (len)
   {
      --len;
      h += (h << 5);
      h ^= (unsigned int) *buf++;
   }
   return (h % DH_SIZE);
}

/*
 * TODO: Maybe use a different kind of lock?  Since most of the time
 * we expect to gain the lock, it might make sense to use a flag in
 * the map.  But then we would for sure be faulting the first page,
 * even for readonly access.
 */

int dhlock (void)
{
   int i;

   for (i = 100; i--;)
   {
      if (0 == lockf(dh_fd, F_TLOCK, 0))
         return (0);
      if (EAGAIN != errno)
         return (-1);
      nap(0, 500);
   }
   return (-1);
}

int dhunlock (void)
{
   lseek(dh_fd, 0, SEEK_SET);
   lockf(dh_fd, F_ULOCK, 0);
   return (0);
}

struct chain *dhlocatechain (char *messageid)
{
   unsigned index;
   unsigned char *x;
   unsigned h;
   struct chain *chp;

   h = dhhash(messageid);
   x = dh_table->next + h * 3;
   index = char3toint(x);
   if (!index)
      return (NULL);

   chp = allo_deref(index);
   if (NULL == chp)
   {
      LOG("dhlocatechain:bad allo deref");
   }
   return (chp);
}

static int initfile (void)
{
   int fd;
   int integer = DH_MAGIC;
   int i;
   char foo[3] = { '\0', };

   fd = open(dh_tablefile, O_RDWR | O_CREAT | O_EXCL, 0644);
   if (-1 == fd)
      return (-1);
   if (-1 == lockf(fd, F_TLOCK, 0))
   {
      do
      { /* wait until other process has initialized it */
         if (EAGAIN == errno)
            nap(0, 200);
         else
            goto fail;
      }
      while (-1 == lockf(fd, F_TLOCK, 0));
      lockf(fd, F_ULOCK, 0);
      return (fd);
   }
   if (sizeof (int) != write(fd, &integer, sizeof (int)))
      goto fail;
   for (i = 0; i < DH_SIZE; i++)
      if (sizeof (foo) != write(fd, foo, sizeof (foo)))
         goto fail;
   lseek(fd, 0, SEEK_SET);
   lockf(fd, F_ULOCK, 0);
   return (fd);

fail:
   if (fd > -1)
      close(fd);
   return (-1);
}

int dh_find (struct data *dp, int readonly)
{
   struct chain *chp;
   unsigned char *x;

   if (!readonly)   /* lockf() fails on read-only file, and locking not needed anyway for R/O? */
      if (-1 == dhlock())
         return (-1);
   chp = dhlocatechain(dp->messageid);
   if (!chp)
      goto fail;
   while (strcmp(dp->messageid, chp->messageid))
   {
      unsigned index = chp->next;

      if (!index)
         goto fail;
      chp = allo_deref(index);
      if (!chp)
      {
         LOG2("dh_find:bad allo deref");
         goto fail;
      }
   }
   dp->serial = chp->serial;
   x = (unsigned char *) chp->newsgroup;
   dp->newsgroup = ng_newsgroup(char2toint(x));
   if (!dp->newsgroup)
      goto fail;
   if (!readonly)
      dhunlock();
   return (0);

fail:
   if (!readonly)
      dhunlock();
   return -1;
}

int dh_open (char *pre, int readonly)
{
   struct stat st;
   int prot;
   int openflag;

   if (pre && *pre)
   {
      if (strlen(pre) > 27)
      {
         LOG("SNROOT too long");
         return (-1);
      }
      strcpy(dh_tablefile, pre);
      strcat(dh_tablefile, DH_TABLEFILE);
      strcpy(dh_chainfile, pre);
      strcat(dh_chainfile, DH_CHAINFILE);
      strcpy(dh_groupfile, pre);
      strcat(dh_groupfile, DH_GROUPFILE);
   }
   else
   {
      strcpy(dh_tablefile, DH_TABLEFILE);
      strcpy(dh_chainfile, DH_CHAINFILE);
      strcpy(dh_groupfile, DH_GROUPFILE);
   }

   prot = PROT_READ;
   if (!(dh_isreadonly = !!readonly))
   {
      openflag = O_RDWR;
      prot |= PROT_WRITE;
   }
   else
      openflag = O_RDONLY;

   if (-1 == (dh_fd = open(dh_tablefile, openflag)))
   {
      if (ENOENT == errno)
      {
         if (-1 == (dh_fd = initfile()))
         {
            LOG("dh_open:Can't create %s: %m", dh_tablefile);
            return (-1);
         }
      }
      else
      {
         LOG("dh_open:Can't open %s: %m", dh_tablefile);
         return (-1);
      }
   }

   if (-1 == ng_init())
      goto fail;

   dh_table = (struct table *) mmap(0, sizeof (struct table), prot, MAP_SHARED, dh_fd, 0);

   if (!dh_table || dh_table == MAP_FAILED)
   {
      LOG("dh_open:mmap:%m?");
      dh_table = 0;
   }
   else if (dh_table->magic != DH_MAGIC)
      LOG("dh_open:Bad magic");
   else if (-1 == fstat(dh_fd, &st))
      LOG("dh_open:fstat:%m");
   else if (sizeof (struct table) != st.st_size)
      LOG("dh_open:table has wrong size!");
   else if (-1 == allo_init(dh_chainfile, openflag) &&
            -1 == allo_init(dh_chainfile, openflag | O_CREAT))
   {
      LOG("dh_open:allo_init(%s):%m?", dh_chainfile);
   }
   else
      return (0);
   if (dh_table)
   {
      munmap(dh_table, sizeof (struct table));
      dh_table = 0;
   }
   ng_fin();

fail:
   close(dh_fd);
   return (dh_fd = -1);
}

int dh_close (void)
{
   if (dh_fd > -1)
   {
      close(dh_fd);
      dh_fd = -1;
   }
   if (dh_table)
   {
      munmap((caddr_t) dh_table, sizeof (struct table));
      dh_table = NULL;
   }
   allo_destroy();
   ng_fin();
   return (0);
}
