/*==================================================================
 * seq_oss.c - OSS sequencer routines (also used by OSS wavetable drivers)
 *
 * Smurf Sound Font Editor
 * Copyright (C) 1999-2001 Josh Green
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 * To contact the author of this program:
 * Email: Josh Green <jgreen@users.sourceforge.net>
 * Smurf homepage: http://smurf.sourceforge.net
 *==================================================================*/
#include "config.h"

/* only compile if OSS_SUPPORT */
#ifdef OSS_SUPPORT

#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <glib.h>
#include "seq_oss.h"
#include "sequencer.h"
#include "wavetable.h"
#include "util.h"
#include "i18n.h"

#if defined(HAVE_SYS_SOUNDCARD_H)
#include <sys/soundcard.h>
#elif defined(HAVE_MACHINE_SOUNDCARD_H)
#include <machine/soundcard.h>
#endif

gint seq_oss_fd;		/* OSS Sequencer file descriptor */
gboolean seq_oss_fdactive = FALSE;	/* file descriptor active? */
gint seq_oss_dev;		/* Sequencer synth device */

SEQ_DEFINEBUF (2048);		/* define sequencer buffer size */

void
seqbuf_dump ()
{				/* routine to dump sequencer buffer */
  if (_seqbufptr)
    if (write (seq_oss_fd, _seqbuf, _seqbufptr) == -1)
      {
	logit (LogFubar | LogErrno,_("Failed to write to OSS sequencer device %s, disabling"),
	  wtbl_oss_devname);
	seq_close ();
      }
  _seqbufptr = 0;
}

gint
seq_oss_init (void)
{
  if (!seq_oss_fdactive)
    return (logit (LogFubar, _("OSS sequencer driver requires that an OSS"
			      " wavetable driver be active")));
  return (OK);
}

/* OSS wavetable drivers use this function */
gint
seq_oss_init_synth (int subtype, char *name)
{				/* OSS based synth init */
  gint i;
  struct synth_info card_info;
  gint nrsynths;

  if (!seq_oss_fdactive)
    if ((seq_oss_fd = open (wtbl_oss_devname, O_RDWR, 0)) < 0)
      return (logit (LogFubar | LogErrno,
		     _("Failed to open OSS sequencer device %s"),
		     wtbl_oss_devname));

  if (ioctl (seq_oss_fd, SNDCTL_SEQ_NRSYNTHS, &nrsynths) == -1)
    {
      logit (LogFubar | LogErrno, _("Failed to detect synth on device %s"),
	   wtbl_oss_devname);
      close (seq_oss_fd);
      return (FAIL);
    }
  seq_oss_dev = -1;
  for (i = 0; i < nrsynths; i++)
    {
      card_info.device = i;
      if (ioctl (seq_oss_fd, SNDCTL_SYNTH_INFO, &card_info) == -1)
	{
	  logit (LogFubar | LogErrno,
		 _("Can't get soundcard info for device %s"), wtbl_oss_devname);
	  close (seq_oss_fd);
	  return (FAIL);
	}
      if (card_info.synth_type == SYNTH_TYPE_SAMPLE
	&& card_info.synth_subtype == subtype)
	{
	  seq_oss_dev = i;
	  break;
	}
    }

  if (seq_oss_dev < 0)
    {
      logit (LogInfo, _("Synth of type \"%s\" not found on device %s"),
	name, wtbl_oss_devname);
      close (seq_oss_fd);
      return (FAIL);
    }

  seq_oss_fdactive = TRUE;

  return (OK);
}

/* close OSS file descriptor */
void
seq_oss_fd_close (void)
{
  if (!seq_oss_fdactive) return;
  seq_oss_fdactive = FALSE;
  close (seq_oss_fd);
}

/* set bank for specified channel */
void
seq_oss_set_bank (gint chan, gint bank)
{
  SEQ_CONTROL (seq_oss_dev, chan, CTL_BANK_SELECT, bank);
  SEQ_DUMPBUF ();
}

/* Set preset for specified channel */
void
seq_oss_set_preset (gint chan, gint preset)
{
  SEQ_SET_PATCH (seq_oss_dev, chan, preset);
  SEQ_DUMPBUF ();
}

/* Note on (note) on channel (chan) with velocity (vel) */
void
seq_oss_note_on (gint chan, gint note, gint vel)
{
  SEQ_START_NOTE (seq_oss_dev, chan, note, vel);
  SEQ_DUMPBUF ();
}

/* Note off (note) on channel (chan) with velocity (vel) */
void
seq_oss_note_off (gint chan, gint note, gint vel)
{
  SEQ_STOP_NOTE (seq_oss_dev, chan, note, vel);
  SEQ_DUMPBUF ();
}

/* Set midi bender control on channel (chan) to (bendval) */
void
seq_oss_pitch_bender (gint chan, gint bendval)
{
  SEQ_BENDER (seq_oss_dev, chan, bendval);
  SEQ_DUMPBUF ();
}

/* bender range (bend range val is in semitones) */
void
seq_oss_pitch_bend_range (gint chan, gint val)
{
  /* OSS bender range value is in semitones * 100 */
  SEQ_BENDER_RANGE (seq_oss_dev, chan, val * 100);
  SEQ_DUMPBUF ();
}

/* main volume */
void
seq_oss_main_volume (gint chan, gint val)
{
  SEQ_CONTROL (seq_oss_dev, chan, CTL_MAIN_VOLUME, val);
  SEQ_DUMPBUF ();
}

/* Set chorus amount */
void
seq_oss_chorus (gint chan, gint val)
{
  SEQ_CONTROL (seq_oss_dev, chan, CTL_CHORUS_DEPTH, val);
  SEQ_DUMPBUF ();
}

/* Set reverb amount */
void
seq_oss_reverb (gint chan, gint val)
{
  SEQ_CONTROL (seq_oss_dev, chan, CTL_EXT_EFF_DEPTH, val);
  SEQ_DUMPBUF ();
}

#endif /* #ifdef OSS_SUPPORT */
