/*==================================================================
 * midi_oss.c - OSS sequencer MIDI thru driver
 *
 * Smurf Sound Font Editor
 * Copyright (C) 1999-2001 Josh Green
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 * To contact the author of this program:
 * Email: Josh Green <jgreen@users.sourceforge.net>
 * Smurf homepage: http://smurf.sourceforge.net
 *==================================================================*/
#include "config.h"

#ifdef OSS_SUPPORT

#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <glib.h>
#include "midi_oss.h"
#include "midi.h"
#include "sequencer.h"
#include "seq_oss.h"
#include "smurfcfg.h"
#include "util.h"
#include "i18n.h"

#if defined(HAVE_SYS_SOUNDCARD_H)
#include <sys/soundcard.h>
#elif defined(HAVE_MACHINE_SOUNDCARD_H)
#include <machine/soundcard.h>
#endif

static gint midioss_inptag;	/* GDK watch tag for MIDI fd */

/* init for OSS midi driver */
gint
midi_oss_init (void)
{
  if (!seq_oss_fdactive)	/* OSS sequencer file descriptor active? */
    return (logit (LogFubar, _("OSS MIDI Thru driver requires that an OSS"
			      " wavetable driver be active")));

  midioss_inptag = gdk_input_add (seq_oss_fd, GDK_INPUT_READ,
				  (GdkInputFunction)midi_read, NULL);

  return (OK);
}

/* close routine for OSS midi driver */
void
midi_oss_close (void)
{
  if (midi_driver == MIDI_OSS && midi_active)
    gdk_input_remove (midioss_inptag);
}

/* OSS sequencer read routines, converts to RAW midi data */
gint
midi_oss_read (guint8 * buf, guint size)
{
  gint c, i;
  guint8 *p;
  static gboolean pc = 0;	/* # of bytes in partial 4 byte block */
  static guint8 parts[3];	/* stores partial 4 byte block */

  p = buf;
  if (pc)
    {				/* left over partial 4 byte block? */
      for (i = 0; i < pc; i++)	/* copy it to buffer */
	*p++ = parts[i];
    }
  else
    i = 0;

  if ((c = read (seq_oss_fd, p, size)) < 0)
    {
      logit (LogWarn, _("Failed to read from OSS MIDI device: %s"),
	strerror (c));
      return (-1);
    }

  c += i;			/* # of bytes read + partial block */

  if ((pc = c & 3))
    {				/* any partial 4 byte block? */
      p = buf + (c & ~3);
      for (i = 0; i < pc; i++)	/* save last partial 4 byte block */
	parts[i] = *p++;
    }

  p = buf;
  if ((c &= ~3))
    {				/* any 4 byte blocks to process? */
      for (i = 0; i < c; i += 4)
	{			/* convert to RAW midi bytes */
	  if (buf[i] == SEQ_MIDIPUTC && buf[i + 3] == 0)	/* MIDI only */
	    *p++ = buf[i + 1];
	}
    }

  return ((gint) (p - buf));
}

#endif /* #ifdef OSS_SUPPORT */
