/***************************************************************************
           rex.cpp  -  The little dinosaur
                             -------------------
    copyright            : (C) 2004 - 2007 by Florian Richter
 ***************************************************************************/
/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "../enemies/rex.h"
#include "../core/game_core.h"
#include "../video/animation.h"
#include "../gui/hud.h"
#include "../player/player.h"
#include "../video/gl_surface.h"

/* *** *** *** *** *** cRex *** *** *** *** *** *** *** *** *** *** *** *** */

cRex :: cRex( float x, float y )
: cEnemy( x, y )
{
	Init();
}

cRex :: cRex( XMLAttributes &attributes )
: cEnemy()
{
	Init();
	Create_from_Stream( attributes );
}

cRex :: ~cRex( void )
{
	//
}

void cRex :: Init( void  )
{
	type = TYPE_REX;
	posz = 0.093f;

	state = STA_WALK;
	rex_state = REX_WALK;

	speed = 6;

	images.push_back( pVideo->Get_Surface( "enemy/rex/big_1.png" ) );
	images.push_back( pVideo->Get_Surface( "enemy/rex/big_2.png" ) );
	images.push_back( pVideo->Get_Surface( "enemy/rex/small_1.png" ) );
	images.push_back( pVideo->Get_Surface( "enemy/rex/small_2.png" ) );
	images.push_back( pVideo->Get_Surface( "enemy/rex/dead.png" ) );

	Set_Direction( DIR_RIGHT );

	Set_Image( 0, 1 );

	kill_sound = "enemy/rex/die.ogg";
	kill_points = 20;
}

cRex *cRex :: Copy( void )
{
	cRex *rex = new cRex( startposx, startposy );
	rex->Set_Direction( start_direction );

	return rex;
}

void cRex :: Create_from_Stream( XMLAttributes &attributes )
{
	// position
	Set_Pos( (float)attributes.getValueAsInteger( "posx" ), (float)attributes.getValueAsInteger( "posy" ), 1 );
	// direction
	Set_Direction( Get_Direction_id( attributes.getValueAsString( "direction", Get_Direction_name( start_direction ) ).c_str() ) );
}

void cRex :: Save_to_Stream( ofstream &file )
{
	// begin enemy
	file << "\t<enemy>" << std::endl;

	// name
	file << "\t\t<Property name=\"type\" value=\"rex\" />" << std::endl;
	// position
	file << "\t\t<Property name=\"posx\" value=\"" << (int)startposx << "\" />" << std::endl;
	file << "\t\t<Property name=\"posy\" value=\"" << (int)startposy << "\" />" << std::endl;
	// direction
	file << "\t\t<Property name=\"direction\" value=\"" << Get_Direction_name( start_direction ) << "\" />" << std::endl;

	// end enemy
	file << "\t</enemy>" << std::endl;
}

void cRex :: Set_Direction( ObjectDirection dir )
{
	// already set
	if( direction == dir )
	{
		return;
	}

	cEnemy::Set_Direction( dir, 1 );

	if( direction == DIR_RIGHT )
	{
		velx = 2.5f;
		name = "Rex Right";
	}
	else
	{
		velx = -2.5f;
	    name = "Rex Left";
	}

	Update_Rotation_velx( 1 );
}

void cRex :: DownGrade( bool force /* = 0 */ )
{
	// default stomp downgrade
	if( !force )
	{
		// to small walking
		if( rex_state == REX_WALK )
		{
			rex_state = REX_MINI;
			state = STA_RUN;
			velx = ( direction == DIR_RIGHT ) ? (speed) : (-speed);
			counter = 2;
			kill_points = 40;

			Set_Image( (int)counter );
			Col_Move( 0, images[1]->col_h - images[2]->col_h, 1, 1 );
			Update_Direction();

			// animation
			cParticleAnimation *anim = new cParticleAnimation( posx + ( col_rect.w / 2 ), posy + ( col_rect.h / 4 ) );
			Generate_Hit_Animation( anim );

			anim->Set_Speed( 3.5f, 0.6f );
			anim->Set_Fading_Alpha( 1 );
			// add animation
			pAnimationManager->Add( anim );
		}
		else if( rex_state == REX_MINI )
		{
			Col_Move( 0, images[3]->col_h - images[4]->col_h, 1, 1, 0 );
			Set_Image( 4, 0, 0 );
			dead = 1;

			// animation
			cParticleAnimation *anim = new cParticleAnimation( posx + ( col_rect.w / 2 ), posy + ( col_rect.h / 2 ) );
			Generate_Hit_Animation( anim );

			anim->Set_Speed( 4.5f, 1.6f );
			anim->Set_Scale( 0.6f );
			// add animation
			pAnimationManager->Add( anim );
		}
	}
	// falling death
	else
	{
		dead = 1;
		Set_RotationZ( 180 );
	}

	if( dead )
	{
		massivetype = MASS_PASSIVE;
		state = STA_STAY;
		counter = 0;
		rex_state = REX_DEAD;
	}
}

void cRex :: DieStep( void )
{
	counter += pFramerate->speedfactor;

	// stomp death
	if( rotz != 180 )
	{
		float speed = pFramerate->speedfactor * 0.05f;

		Add_Scale( -speed );
		Move( ( image->w / 2 ) * speed, ( image->h ) * speed, 1 );

		if( scalex < 0.1f )
		{
			visible = 0;
			Set_Scale( 1 );
		}
	}
	// falling death
	else
	{
		// a little bit upwards first
		if( counter < 5 )
		{
			Move( 0, -5 );
		}
		// if not below the screen fall
		else if( posy < GAME_RES_H + col_rect.h )
		{
			Move( 0, 20 );
		}
		// if below disable
		else
		{
			rotz = 0;
			visible = 0;
		}
	}
}

void cRex :: Update( void )
{
	cEnemy::Update();

	if( dead || freeze_counter || !is_Player_range() )
	{
		return;
	}

	if( rex_state == REX_WALK )
	{
		counter += pFramerate->speedfactor * 0.2f;

		if( counter > 1.99f )
		{
			counter = 0;
		}
	}
	else
	{
		counter += pFramerate->speedfactor * 0.4f;

		if( counter > 3.99f )
		{
			counter = 2;
		}
	}

	Set_Image( (int)counter );

	// gravity
	Update_Gravity();

	CollideMove();

	Update_Rotation_velx();
}

unsigned int cRex :: Validate_Collision( cSprite *obj )
{
	// basic validation checking
	int basic_valid = Validate_Collision_Ghost( obj );

	// found valid collision
	if( basic_valid > -1 )
	{
		return basic_valid;
	}

	if( obj->massivetype == MASS_MASSIVE )
	{
		if( obj->type == TYPE_JPIRANHA )
		{
			return 0;
		}
		if( obj->type == TYPE_ROKKO )
		{
			return 0;
		}
		if( obj->type == TYPE_GEE )
		{
			return 0;
		}

		return 2;
	}
	if( obj->type == TYPE_ENEMYSTOPPER )
	{
		return 2;
	}
	if( obj->massivetype == MASS_HALFMASSIVE )
	{
		// if moving downwards and object is on top
		if( vely >= 0 && is_onTop( obj ) )
		{
			return 2;
		}
	}

	return 0;
}

void cRex :: Handle_Collision_Player( ObjectDirection cdirection )
{
	// invalid
	if( cdirection == DIR_UNDEFINED )
	{
		return;
	}

	if( cdirection == DIR_TOP && pPlayer->state != STA_FLY )
	{
		pointsdisplay->Add_Points( kill_points, pPlayer->posx, pPlayer->posy, "", (Uint8)255, 1 );
		pAudio->PlaySound( kill_sound );


		// big walking
		if( rex_state == REX_WALK )
		{
			DownGrade();
		}
		// small walking
		else if( rex_state == REX_MINI )
		{
			DownGrade();
			pPlayer->Add_Kill_Multiplier();
		}

		pPlayer->start_enemyjump = 1;
	}
	else
	{
		pPlayer->DownGrade();
		Turn_Around( cdirection );
	}
}

void cRex :: Handle_Collision_Enemy( cObjectCollision *collision )
{
	Turn_Around( collision->direction );
	Send_Collision( collision );
}

void cRex :: Handle_Collision_Massive( cObjectCollision *collision )
{
	Turn_Around( collision->direction );
}
