#include <stddef.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <pthread.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "list.h"
#include "common.h"
#include "buffer.h"
#include "array.h"
#include "charset.h"
#include "smbctx.h"
#include "function.h"
#include "smbitem.h"

time_t		last_smbitem_scan_time	= (time_t) 0;
int		smbitem_update_interval	= 300;
int		smbitem_time_to_live	= 900;
int		query_browser_flag	= 1;
LIST		user_tree		= STATIC_LIST_INITIALIZER(user_tree);
LIST		auto_tree		= STATIC_LIST_INITIALIZER(auto_tree);
BUFFERMANAGER	ConvertBuffer		= {"Convert", 15, 0, 2048,
					    STATIC_LIST_INITIALIZER(ConvertBuffer.list),
					    PTHREAD_MUTEX_INITIALIZER
					  };
pthread_mutex_t m_smbitem		= PTHREAD_MUTEX_INITIALIZER;

int SetSmbItemUpdateInterval(int interval){
    DPRINT(7, "interval=%d\n", interval);
    if (interval < GetUpdateTimeDelta()) return 0;
    pthread_mutex_lock(&m_smbitem);
    smbitem_update_interval = interval;
    pthread_mutex_unlock(&m_smbitem);
    return 1;
}

int isTimeForRescanSambaTree(time_t time){
    int result = 0;

    pthread_mutex_lock(&m_smbitem);
    result = (time - last_smbitem_scan_time >= smbitem_update_interval);
    pthread_mutex_unlock(&m_smbitem);
    return result;
}

int SetSmbItemTime2Live(int ttl){
    int result;

    DPRINT(7, "ttl=%d\n", ttl);
    pthread_mutex_lock(&m_smbitem);
    result = (ttl > smbitem_update_interval);
    if (result) smbitem_time_to_live = ttl;
    pthread_mutex_unlock(&m_smbitem);
    return result;
}

int SetQueryBrowserFlag(int flag){
    DPRINT(7, "flag=%d\n", flag);
    pthread_mutex_lock(&m_smbitem);
    query_browser_flag = flag;
    pthread_mutex_unlock(&m_smbitem);
    return 1;
}

int GetQueryBrowserFlag(){
    int flag;
    
    pthread_mutex_lock(&m_smbitem);
    flag = query_browser_flag;
    pthread_mutex_unlock(&m_smbitem);
    DPRINT(7, "flag=%d\n", flag);
    return flag;
}

smbitem* MakeItem(hstring *name, int type, hstring *link){
    int		alloc_len;
    smbitem	*item;
    char	*ptr;

    alloc_len = sizeof(smbitem) + name->length + 1;
    switch(type){
	case SMBITEM_COMP:
	    break;
	case SMBITEM_LINK:
	    alloc_len += link->length + 1;
	    break;
	case SMBITEM_GROUP:
	    alloc_len += sizeof(LIST);
	    break;
	default:
	    return NULL;
    };
    
    if ((item = malloc(alloc_len)) == NULL) return NULL;
    memset(item, 0, alloc_len);
    item->ctime = time(NULL);
    item->type = type;
    
    ptr = item->data;
    if (type == SMBITEM_GROUP){
	init_list((LIST*)item->data);
	ptr += sizeof(LIST);
    }else 
    if (type == SMBITEM_LINK){
	strncpy(ptr, link->string, link->length);
	memcpy(&item->link, link, sizeof(hstring));
	item->link.string = ptr;
	ptr += link->length + 1;
    }

    strncpy(ptr, name->string, name->length);
    memcpy(&item->name, name, sizeof(hstring));
    item->name.string = ptr;

    return item;
}

void DeleteSmbItem(smbitem *item){
    if (item->type == SMBITEM_GROUP){
	LIST	*list = (LIST*)item->data;
	smbitem *elem, *tmp;
	
	elem = first_elem(list);
	while(is_valid_elem(list, elem)){
	    tmp = elem; elem = next_elem(elem);
	    remove_from_list(list, tmp);
	    DeleteSmbItem(tmp);
	}
    }
    DPRINT(6, "%s[%d]\n", item->name.string, item->ref_count);
    if (item->ref_count > 0) item->ref_count--;
    else free(item);
}

smbitem* FindSmbItem(LIST *list, hstring *name){
    smbitem *item = first_elem(list);
    while(is_valid_elem(list, item)){
	if (hstring_casecmp(&item->name, name) == 0) return item;
	item = next_elem(item);
    }
    return NULL;
}

smbitem* mkitem(LIST *list, hstring *name, int type, hstring *link){
    smbitem *item;
    
    if ((item = FindSmbItem(list, name)) != NULL){
	if (item->type == type){
	    if ((type != SMBITEM_LINK) || (hstring_cmp(&item->link, link) == 0)){
		item->ctime = time(NULL);
		return item;
	    }
	}
	remove_from_list(list, item);
	DeleteSmbItem(item);
    }
    if ((item = MakeItem(name, type, link)) == NULL) return NULL;
    add_to_list_back(list, item);
    return item;
}

LIST* finddir(LIST *list, hstring *path){
    smbitem 	*item;

    if (strcmp(path->string, "") == 0) return list;
    if ((item = FindSmbItem(list, path)) == NULL) return NULL;
    if (item->type != SMBITEM_GROUP) return NULL;
    return (LIST*)item->data;
}

void print_list(LIST *list, char *s){
    smbitem *item = first_elem(list);
    while(is_valid_elem(list, item)){
	DPRINT(6, "%s%s[%d, %x], type=%d, link=%s[%d, %x], ref=%d, ctime=%d\n", s,
	    item->name.string, item->name.length, item->name.hash,
	    item->type, 
	    item->link.string, item->link.length, item->link.hash,
	    item->ref_count, (int)item->ctime);
	    if (item->type == SMBITEM_GROUP)
		print_list((LIST*)item->data, "  ");
	item = next_elem(item);
    }
}

int mkgroup(const char *path, int flag){
    LIST	*list = NULL;
    smbitem	*item;
    hstring	name;
    
    DPRINT(6, "path=%s, flag=%d\n", path, flag);
    path = filename_begin(path);
    if (*next_filename(path) != '\0') return 0;
    if (*path == '\0') return 0;
    if (strcmp(path, ".") == 0) return 0;
    if (strcmp(path, "..") == 0) return 0;
    if (flag & AUTOPATH) list = &auto_tree;
    if (flag & USERPATH) list = &user_tree;
    if (list == NULL) return 0;

    make_casehstring(&name, path, filename_len(path));

    pthread_mutex_lock(&m_smbitem);
    item = mkitem(list, &name, SMBITEM_GROUP, NULL);
    pthread_mutex_unlock(&m_smbitem);
    return (item != NULL);
}

int mkhost(const char *path, int flag){
    LIST	*list = NULL;
    smbitem	*item;
    hstring	name;

    DPRINT(6, "path=%s, flag=%d\n", path, flag);
    path = filename_begin(path);
    if (*next_filename(path) != '\0') return 0;
    if (*path == '\0') return 0;
    if (strcmp(path, ".") == 0) return 0;
    if (strcmp(path, "..") == 0) return 0;
    if (flag & AUTOPATH) list = &auto_tree;
    if (flag & USERPATH) list = &user_tree;
    if (list == NULL) return 0;    

    make_casehstring(&name, path, filename_len(path));

    pthread_mutex_lock(&m_smbitem);
    item = mkitem(list, &name, SMBITEM_COMP, NULL);
    pthread_mutex_unlock(&m_smbitem);
    return (item != NULL);
}

int mklink(const char *path, const char *linkpath, int flag){
    LIST	*list = NULL;
    smbitem	*item = NULL;
    hstring	name, link;
    const char	*linkname;
    
    DPRINT(6, "path=%s, link=%s, flag=%d\n", path, linkpath, flag);
    path = filename_begin(path);
    linkname = next_filename(path);
    if (*linkname == '\0'){
	linkname = path; path = "";
    }
    if (*next_filename(linkname) != '\0') return 0;
    if (*linkname == '\0') return 0;
    if (strcmp(linkname, ".") == 0) return 0;
    if (strcmp(linkname, "..") == 0) return 0;
    if (flag & AUTOPATH) list = &auto_tree;
    if (flag & USERPATH) list = &user_tree;
    if (list == NULL) return 0;    

    make_casehstring(&name, path, filename_len(path));
    
    pthread_mutex_lock(&m_smbitem);
    if ((list = finddir(list, &name)) != NULL){
	make_casehstring(&name, linkname, filename_len(linkname));
	make_hstring(&link, linkpath, strlen(linkpath));
	item = mkitem(list, &name, SMBITEM_LINK, &link);
    }
    pthread_mutex_unlock(&m_smbitem);
    return (item != NULL);
}


void delete_old_item(LIST *list, time_t ctime){
    smbitem	*tmp, *item;

    item = first_elem(list);
    while(is_valid_elem(list, item)){
	tmp = item; item = next_elem(item);
	if (tmp->type == SMBITEM_GROUP){
	    delete_old_item((LIST*)tmp->data, ctime);
	    if (first_elem((LIST*)tmp->data) != NULL) continue;
	}
	if (tmp->ctime < ctime){
	    remove_from_list(list, tmp);
	    DeleteSmbItem(tmp);
	}
    }
}

void DeleteOldSmbItem(time_t ctime, int flag){
    pthread_mutex_lock(&m_smbitem);
    if (flag & AUTOPATH) delete_old_item(&auto_tree, ctime);
    if (flag & USERPATH) delete_old_item(&user_tree, ctime);
    pthread_mutex_unlock(&m_smbitem);
}

inline void RequestElem(smbitem *item){
    item->ref_count++;
}

inline void ReleaseElem(smbitem *item){
    if (item->ref_count > 0) item->ref_count--;
    else free(item);
}

void DeleteSmbItemArray(ARRAY *array){
    int			i;

    if (array == NULL) return;
    pthread_mutex_lock(&m_smbitem);
    for(i = 0; i < array_count(array); i++)
	ReleaseElem((smbitem*)array_elem(array, i));
    pthread_mutex_unlock(&m_smbitem);
    destroy_array(array);
}

void RequestSmbItem(smbitem *item){
    pthread_mutex_lock(&m_smbitem);
    RequestElem(item);
    pthread_mutex_unlock(&m_smbitem);
}

void ReleaseSmbItem(smbitem* item){
    pthread_mutex_lock(&m_smbitem);
    ReleaseElem(item);
    pthread_mutex_unlock(&m_smbitem);
}

int find_in_array(ARRAY *array, hstring *name){
    int 	i;
    smbitem	*item;

    for(i = 0; i < array_count(array); i++){
	item = array_elem(array, i);
	if (item == NULL){
	    DPRINT(0, "WARNING!!! *smbitem == NULL.\n");
	    continue;
	}
        if (hstring_casecmp(&item->name, name) == 0) return i;
    }
    return -1;
}

void add_list_to_array(LIST *list, ARRAY *array){
    int		i;
    smbitem	*item;
    
    if (list == NULL) return;
    item = first_elem(list);
    while(is_valid_elem(list, item)){
	if ((i = find_in_array(array, &item->name)) >= 0){
	    ReleaseElem((smbitem*)array_elem(array, i));
	    remove_from_array(array, i);
	}
	if (add_to_array_back(array, item)) RequestElem(item);
	item = next_elem(item);
    }
}

int getdir(ARRAY *array, const char *path, int flag){
    hstring	name;
    
    init_array(array);
    path = filename_begin(path);
    if (*next_filename(path) != '\0') return 0;

    make_casehstring(&name, path, filename_len(path));
    
    pthread_mutex_lock(&m_smbitem);
    if (flag & AUTOPATH){
	add_list_to_array(finddir(&auto_tree, &name), array);
    }	
    if (flag & USERPATH){
	add_list_to_array(finddir(&user_tree, &name), array);
    }
    pthread_mutex_unlock(&m_smbitem);
    return 1;
}

int GetPathType(const char *path, BUFFER *buf){
    smbitem	*item, *item1;
    int		tree;
    hstring	name, name1;

    DPRINT(6, "path=%s, buf=%p\n", path, buf);

    if (buf != NULL) buf->data[0] = '\0';
    
    path = filename_begin(path);
    if (*path == '\0') return PATH_TO_DIR;

    make_casehstring(&name, path, filename_len(path));

    pthread_mutex_lock(&m_smbitem);
    tree = USERPATH;
    item = FindSmbItem(&user_tree, &name);
    if (item == NULL){
	tree = AUTOPATH;
	item = FindSmbItem(&auto_tree, &name);
    }
    if (item != NULL) RequestElem(item);
    pthread_mutex_unlock(&m_smbitem);

    if ((item == NULL) || (item->type == SMBITEM_COMP)){
	if (item != NULL) ReleaseSmbItem(item);
	if (buf != NULL){
	    strcpy(buf->data, "smb://");
	    if (!local2smb(buf->data + 6, path, BufferSize(buf) - 6))
		return PATH_UNKNOWN;
	}
	if (*(path = next_filename(path)) == '\0') return PATH_TO_COMP;
	if (*next_filename(path) == '\0') return PATH_TO_SHARE;
	return PATH_TO_SMBFILE;
    }

    if (item->type == SMBITEM_LINK){
    	if (*next_filename(path) != '\0'){
	    ReleaseSmbItem(item);
	    return PATH_UNKNOWN;
	}
	if (buf != NULL) safe_copy(buf->data, item->data, BufferSize(buf));
	ReleaseSmbItem(item);
	return PATH_TO_LINK;
    }

    if (item->type == SMBITEM_GROUP){
	if (*(path = next_filename(path)) == '\0'){
	    ReleaseSmbItem(item);
	    return PATH_TO_DIR;
	}
	
	make_casehstring(&name1, path, filename_len(path));
	
	pthread_mutex_lock(&m_smbitem);
	item1 = FindSmbItem((LIST*)item->data, &name1);
	ReleaseElem(item);
	if ((item1 == NULL) && (tree != AUTOPATH)){
	    item = FindSmbItem(&auto_tree, &name);
	    if ((item != NULL) && (item->type == SMBITEM_GROUP))
		item1 = FindSmbItem((LIST*)item->data, &name1);
	}
	if ((item = item1) != NULL) RequestElem(item);
	pthread_mutex_unlock(&m_smbitem);

	if ((item == NULL) || (item->type != SMBITEM_LINK)){
	    if (item != NULL){
		DPRINT(0, "WARNING!!! BAD SMBITEM: %s\n", item->name.string);
		ReleaseSmbItem(item);
	    }
	    return PATH_UNKNOWN;
	}
	if (buf != NULL) safe_copy(buf->data, item->data, BufferSize(buf));
	ReleaseSmbItem(item);
	return PATH_TO_LINK;
    };
    
    DPRINT(0, "WARNING!!! BAD SMBITEM: %s\n", item->name.string);
    ReleaseSmbItem(item);
    return PATH_UNKNOWN;
}

void UpdateGroup(const char *group, BUFFER *buf1){
    BUFFER		*buf2;
    SmbCtx		*smb;
    struct smbc_dirent	*dirent;
    SMBCFILE		*fd;

    DPRINT(5, "(group=%s)\n", group);
    if (BufferSize(buf1) < strlen(group) + 7) return;
    strcpy(buf1->data, "smb://");
    if (! local2smb(buf1->data + 6, group, BufferSize(buf1) - 6)) return;
    if ((buf2 = GetBuffer(buf1->man)) == NULL) return;

    if ((smb = GetSmbCtx(buf1->data,
	SMBCTX_DO_NOT_UPDATE | SMBCTX_NEEDS_CLEANUP)) == NULL) goto error1;

    buf1->data[0] = '/';
    strcpy(buf1->data + 1, group);
    buf1->data[strlen(group) + 1] = '/';
    strcpy(buf2->data, "../");
    
    LockSmbCtx(smb);
    if ((fd = smb->ctx->opendir(smb->ctx, smb->name.string)) == NULL){
	SetSmbCtxFlags(smb, SMBCTX_IS_BAD);
	goto error2;
    }

    while ((dirent = smb->ctx->readdir(smb->ctx, fd)) != NULL){
	if (strcmp(dirent->name, "") == 0) continue;
	if (strcmp(dirent->name, ".") == 0) continue;
	if (strcmp(dirent->name, "..") == 0) continue;
	if (dirent->smbc_type != SMBC_SERVER) continue;
		
	if (! smb2local(buf1->data + strlen(group) + 2, dirent->name,
	    BufferSize(buf1) - strlen(group) - 2)) continue;
	if (! smb2local(buf2->data + 3, dirent->name, BufferSize(buf2) - 3))
	    continue;
	mklink(buf1->data, buf2->data, AUTOPATH);
    }
    smb->ctx->closedir(smb->ctx, fd);

  error2:
    UnlockSmbCtx(smb);
    ReleaseSmbCtx(smb);
  error1:
    ReleaseBuffer(buf2);
}

void UpdateRoot(){
    BUFFER		*buf;
    SmbCtx		*smb;
    struct smbc_dirent	*dirent;
    ARRAY		array;
    SMBCFILE		*fd;
    int			i;

    if ((buf = GetBuffer(&ConvertBuffer)) == NULL) return;
    if (!GetQueryBrowserFlag()) goto update_group;

    DPRINT(5, "reading group list\n");
    if ((smb = GetSmbCtx("smb://",
	SMBCTX_DO_NOT_UPDATE | SMBCTX_NEEDS_CLEANUP)) == NULL) goto error0;
    LockSmbCtx(smb);
    if ((fd = smb->ctx->opendir(smb->ctx, smb->name.string)) == NULL){
	SetSmbCtxFlags(smb, SMBCTX_IS_BAD);
	goto error1;
    }

    while ((dirent = smb->ctx->readdir(smb->ctx, fd)) != NULL){
	if (strcmp(dirent->name, "") == 0) continue;
	if (strcmp(dirent->name, ".") == 0) continue;
	if (strcmp(dirent->name, "..") == 0) continue;
	if (dirent->smbc_type != SMBC_WORKGROUP) continue;
	if (! smb2local(buf->data, dirent->name, BufferSize(buf))) continue;
	mkgroup(buf->data, AUTOPATH);
    }
    smb->ctx->closedir(smb->ctx, fd);

  error1:
    UnlockSmbCtx(smb);
    ReleaseSmbCtx(smb);
    
  update_group:
    getdir(&array, "", AUTOPATH);
    for(i = 0; i < array_count(&array); i++)
	UpdateGroup(((smbitem*)array_elem(&array, i))->name.string, buf);
    DeleteSmbItemArray(&array);
    
  error0:
    ReleaseBuffer(buf);    
}

void UpdateSambaTree(){
    time_t die_time;

    pthread_mutex_lock(&m_smbitem);
    last_smbitem_scan_time = time(NULL);
    die_time = last_smbitem_scan_time - smbitem_time_to_live;
    pthread_mutex_unlock(&m_smbitem);

    UpdateRoot();
    DeleteOldSmbItem(die_time, AUTOPATH);
}
