/***************************************************************************
    smb4k.cpp  -  The main class of Smb4K.
                             -------------------
    begin                : Sam M�  1 14:57:21 CET 2003
    copyright            : (C) 2003 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// Qt includes
#include <qtooltip.h>

// KDE includes
#include <kmenubar.h>
#include <kstatusbar.h>
#include <klocale.h>
#include <kstdaction.h>
#include <kmessagebox.h>
#include <kpopupmenu.h>
#include <kiconloader.h>
#include <kdebug.h>

// application specific includes
#include "smb4k.h"
#include "widgets/smb4kbookmarkmenuwidget.h"
#include "widgets/smb4koptionsdlg.h"
#include "widgets/smb4kbookmarkeditor.h"
#include "core/smb4kdefs.h"
#include "core/smb4kglobal.h"

using namespace Smb4KGlobal;

#define ID_STATUS_MSG 1


Smb4KApp::Smb4KApp( QWidget* , const char* name ) : KDockMainWindow( 0, name )
{
  m_startDocked = false;

  // The caption.
  setPlainCaption( i18n( "Smb4K - The SMB/CIFS Share Browser" ) );

  ///////////////////////////////////////////////////////////////////
  // call inits to invoke all other construction parts
  initCore();
  initStatusBar();
  initView();
  initActions();
  setupMenuBar();
  setupSystemTray();

  // Resize the main window. It'll start with a resolution
  // of 800x600 pixels the first time. Later, this will be
  // overwritten by readOptions().
  resize( 800, 600 );

  // Read the options for the main window
  readOptions();
}


Smb4KApp::~Smb4KApp()
{
}


void Smb4KApp::initActions()
{
  // Add the action collections of the widgets to the main action collection.
  setStandardToolBarMenuEnabled( true );
  createStandardStatusBarAction();

  (void) KStdAction::quit( this, SLOT( slotQuit() ), actionCollection(), "quit_smb4k" );
  (void) KStdAction::preferences( this, SLOT( slotSmb4KOptionsDlg() ), actionCollection(), "configure_smb4k" );
  (void) new KAction( i18n( "&Abort" ) , "stop", CTRL+Key_A, this, SLOT( slotAbort() ), actionCollection(), "abort" );

  // Import the actions from the child widgets.
  // KActionCollection::operator+=() is deprecated. So, we have to write
  // our own code.
  KActionPtrList list = m_network_view->action_collection()->actions();

  for ( KActionPtrList::Iterator it = list.begin(); it != list.end(); it++ )
  {
    actionCollection()->insert( *it );
  }

  list = m_share_view->action_collection()->actions();

  for ( KActionPtrList::Iterator it = list.begin(); it != list.end(); it++ )
  {
    actionCollection()->insert( *it );
  }

  // We need some KAccelActions to jump to the dock widgets.
  accel()->insert( i18n( "Jump to Network Browser" ), CTRL+Key_1, this, SLOT( slotShowNetworkBrowser() ), false, true );
  accel()->insert( i18n( "Jump to Search Dialog (Alternative)" ), CTRL+Key_2, this, SLOT( slotShowSearchDialog() ), false, true );
  accel()->insert( i18n( "Jump to Search Dialog" ), CTRL+Key_S, this, SLOT( slotShowSearchDialog() ), false, true );
  accel()->insert( i18n( "Jump to Shares View" ), CTRL+Key_3, this, SLOT( slotJumpToSharesView() ), false, true );

  // For KDE versions bigger than or equal to 3.3, we use setupGUI(), for all
  // others we stick with createGUI().
#if KDE_VERSION_MAJOR >= 3 && KDE_VERSION_MINOR >= 3
  setupGUI();
#else
  createGUI();
#endif
}


void Smb4KApp::initStatusBar()
{
  statusBar()->insertItem( i18n( "Ready." ), ID_STATUS_MSG, 1, false );
  statusBar()->setItemAlignment( ID_STATUS_MSG, AlignLeft );

  // Labels:
  m_winsLabel = new QLabel( statusBar() );
  m_winsLabel->setAlignment( Qt::AlignCenter );
  m_listLabel = new QLabel( statusBar() );
  m_listLabel->setAlignment( Qt::AlignCenter );
  QLabel *version = new QLabel( QString( "Smb4K %1" ).arg( VERSION ), statusBar() );
  version->setAlignment( Qt::AlignCenter );
  version->setFixedWidth( version->sizeHint().width() + 5 );

  m_progress = new KProgress( statusBar(), 0, 0 );
  m_progress->setFixedWidth( 75 );
  m_progress->setFixedHeight( version->fontMetrics().height() );
  m_progress->setPercentageVisible( false );

  m_progTimer = new QTimer( this );
  connect( m_progTimer, SIGNAL( timeout() ),
           this,        SLOT( slotSetProgress() ) );

  statusBar()->addWidget( m_winsLabel, false );
  statusBar()->addWidget( m_listLabel, false );
  statusBar()->addWidget( m_progress, false );
  statusBar()->addWidget( version, false );

  QToolTip::add( m_winsLabel, i18n( "WINS server" ) );
  QToolTip::add( m_listLabel, i18n( "Browse list retrieval" ) );
  QToolTip::add( version, i18n( "Version" ) );
}


void Smb4KApp::initCore()
{
  connect( smb4k_core, SIGNAL( runStateChanged() ),
           this,       SLOT( slotCoreRunStateChanged() ) );
}


void Smb4KApp::initView()
{
  KDockWidget *main = createDockWidget( "ShareView", SmallIcon( "hdd_mount" ), 0L );
  m_share_view = new Smb4KShareWidget( main );
  main->setWidget( m_share_view );
  main->setDockSite( KDockWidget::DockCorner );
  main->setEnableDocking( KDockWidget::DockNone );
  setView( main );
  setMainDockWidget( main );

  KDockWidget *network = createDockWidget( "NetworkBrowser", SmallIcon( "network" ), 0L, i18n( "Network Browser" ) );
  m_network_view = new Smb4KBrowserWidget( network );
  network->setWidget( m_network_view );
  network->manualDock( main, KDockWidget::DockLeft, 40 );

  KDockWidget *search = createDockWidget( "SearchDialog", SmallIcon( "find" ), 0L, i18n( "Search Dialog" ) );
  m_search_dialog = new Smb4KSearchDialog( search );
  search->setWidget( m_search_dialog );
  search->manualDock( network, KDockWidget::DockCenter, 0 );

  connect( m_network_view,  SIGNAL( searchRequest() ),
           this,            SLOT( slotShowSearchDialog() ) );
  connect( m_search_dialog, SIGNAL( searchResult( Smb4KHostItem * ) ),
           m_network_view,  SLOT( slotInsertItem( Smb4KHostItem * ) ) );

  connect( kapp,            SIGNAL( iconChanged( int ) ),
           this,            SLOT( slotIconChanged( int ) ) );
}


void Smb4KApp::setupMenuBar()
{
  Smb4KBookmarkMenuWidget *bookmark_menu = new Smb4KBookmarkMenuWidget( this, "BookmarkMenu" );

  actionCollection()->action( "add_bookmark_action" )->plug( bookmark_menu->popupMenu(), 0 );

  menuBar()->insertItem( i18n( "&Bookmarks" ), bookmark_menu->popupMenu(), 3, 3 );

  ((QPopupMenu *)(this->child( "settings" ) ))->insertItem( i18n( "&Dock Widgets" ), manager()->dockHideShowMenu(), 2, 2 );
}


void Smb4KApp::setupSystemTray()
{
  sys_tray = new Smb4KSystemTray( true, this, "SystemTray" );
}


void Smb4KApp::saveOptions()
{
  saveMainWindowSettings( config(), "Main Window" );
  writeDockConfig( config(), "Main Window" );

  config()->setGroup( "Docking" );
  config()->writeEntry( "Start Docked", m_startDocked );
}


void Smb4KApp::readOptions()
{
  applyMainWindowSettings( config(), "Main Window" );
  readDockConfig( config(), "Main Window" );

  // We have to set the tab labels here. The writeDockConfig()
  // readDockConfig() combo will ignore changes of the localization,
  // because the first labels will be saved and then loaded again
  // and again.
  KDockWidget *network_view = manager()->getDockWidgetFromName( "NetworkBrowser" );
  network_view->setTabPageLabel( i18n( "Network Browser" ) );

  KDockWidget *search_dialog = manager()->getDockWidgetFromName( "SearchDialog" );
  search_dialog->setTabPageLabel( i18n( "Search Dialog" ) );

  KDockTabGroup *network_parent = NULL;
  KDockTabGroup *search_parent = NULL;

  if ( (network_parent = network_view->parentDockTabGroup()) )
  {
    network_parent->changeTab( network_view, i18n( "Network Browser" ) );
  }

  if ( (search_parent = search_dialog->parentDockTabGroup()) )
  {
    search_parent->changeTab( search_dialog,  i18n( "Search Dialog" ) );
  }

  // Set-up the status bar.
  config()->setGroup( "Browse Options" );
  QString method = config()->readEntry( "Browse List", "lookup" );

  if ( QString::compare( method, "lookup" ) == 0 )
  {
    m_listLabel->setText( i18n( "Network" ) );
  }
  else if ( QString::compare( method, "master" ) == 0 )
  {
    m_listLabel->setText( i18n( "Master" ) );
  }
  else if ( QString::compare( method, "host" ) == 0 )
  {
    m_listLabel->setText( config()->readEntry( "Query Host" ) );
  }
  else if ( QString::compare( method, "ipscan" ) == 0 )
  {
    m_listLabel->setText( i18n( "IP Scan" ) );
  }
  else
  {
    m_listLabel->setText( i18n( "Unknown" ) );
  }

  QString wins = optionsHandler()->winsServer();

  if ( !wins.isEmpty() )
  {
    m_winsLabel->setText( wins );
  }
  else
  {
    m_winsLabel->setText( i18n( "None" ) );
  }

  m_listLabel->setFixedWidth( m_listLabel->sizeHint().width() + 5 );
  m_winsLabel->setFixedWidth( m_winsLabel->sizeHint().width() + 5 );

  // EXTERNAL configuration options:
  m_network_view->readOptions();
  m_share_view->readOptions();
}


bool Smb4KApp::queryExit()
{
  m_startDocked = !this->isVisible();
  hide();

  if ( sys_tray )
  {
    delete sys_tray;
    sys_tray = NULL;
  }

  saveOptions();

  return true;
}


bool Smb4KApp::queryClose()
{
  if ( !kapp->sessionSaving() && this->isVisible() && sys_tray->isDocked() )
  {
    // This part has been 'stolen' from JuK application.
    KMessageBox::information(this,
            i18n( "Closing the main window will keep Smb4K running in the system tray. "
                 "Use 'Quit' from the 'File' menu to quit the application."),
            i18n("Docking in System Tray"), "Hide On Close Info" );
    hide();
    return false;

  }
  else
  {
    return true;
  }
}


/////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////

void Smb4KApp::slotQuit()
{
  slotStatusMsg(i18n("Exiting..."));

  if ( sys_tray->isDocked() )
  {
    sys_tray->undock();
  }

  KApplication::exit( 0 );
}


void Smb4KApp::slotAbort()
{
  slotStatusMsg( i18n( "Aborting..." ) );
  smb4k_core->abort();
  slotStatusMsg( i18n( "Done." ) );
}


void Smb4KApp::slotCoreRunStateChanged()
{
  int state = smb4k_core->currentState();

  if ( smb4k_core->isRunning() )
  {
    actionCollection()->action( "abort" )->setEnabled( true );

    switch ( state )
    {
      case SCANNER_INIT:
        slotStatusMsg( i18n( "Searching network..." ) );
        break;
      case SCANNER_OPENING_WORKGROUP:
        slotStatusMsg( i18n( "Opening workgroup..." ) );
        break;
      case SCANNER_OPENING_HOST:
        slotStatusMsg( i18n( "Opening host..." ) );
        break;
      case SCANNER_RETRYING_OPENING_HOST:
        slotStatusMsg( i18n( "Retrying to open host..." ) );
        break;
      case SCANNER_RETRIEVING_INFO:
        slotStatusMsg( i18n( "Retrieving information..." ) );
        break;
      case SCANNER_RETRIEVING_PREVIEW:
        slotStatusMsg( i18n( "Retrieving preview..." ) );
        break;
      case SCANNER_SEARCHING:
        slotStatusMsg( i18n( "Searching for host..." ) );
        break;
      case MOUNTER_MOUNTING:
        slotStatusMsg( i18n( "Mounting share..." ) );
        break;
      case MOUNTER_UNMOUNTING:
        slotStatusMsg( i18n( "Unmounting share..." ) );
        break;
      case MOUNTER_UNMOUNTING_ALL:
        slotStatusMsg( i18n( "Unmounting all shares..." ) );
        break;
      case PRINT_START:
        slotStatusMsg( i18n( "Printing file..." ) );
        break;
      case SYNCER_START:
        slotStatusMsg( i18n( "Synchronizing data..." ) );
        break;
      default:
        break;
    }

    if ( !m_progTimer->isActive() )
    {
      // Start the busy progress bar:
      m_progress->setTotalSteps( 0 );
      m_progTimer->start( 25, false );
    }
  }
  else
  {
    m_progTimer->stop();
    m_progress->setTotalSteps( 100 );
    m_progress->reset();
    actionCollection()->action( "abort" )->setEnabled( false );
    slotStatusMsg( i18n( "Done." ) );
  }
}


void Smb4KApp::slotSmb4KOptionsDlg()
{
  if ( child( "ConfigDialog", "KDialogBase", true ) == 0 )
  {
    Smb4KOptionsDlg *configDlg = new Smb4KOptionsDlg( this, "ConfigDialog", true );
    connect( configDlg, SIGNAL( savedOptions() ), this, SLOT( slotReadOptions() ) );
    configDlg->exec();
  }
}


void Smb4KApp::slotStatusMsg(const QString &text)
{
  ///////////////////////////////////////////////////////////////////
  // change status message permanently
  statusBar()->clear();
  statusBar()->changeItem(text, ID_STATUS_MSG);
}


void Smb4KApp::slotSetProgress()
{
  m_progress->setProgress( m_progress->progress()+1 );
}


void Smb4KApp::slotReadOptions()
{
  // NOTE: You can put the readOptions() functions here,
  // but then there will be some unwanted flickering. This
  // is the better way.

  m_network_view->readOptions();
  m_share_view->readOptions();

  // Set-up the status bar.
  config()->setGroup( "Browse Options" );
  QString method = config()->readEntry( "Browse List", "lookup" );

  if ( QString::compare( method, "lookup" ) == 0 )
  {
    m_listLabel->setText( i18n( "Network" ) );
  }
  else if ( QString::compare( method, "master" ) == 0 )
  {
    m_listLabel->setText( i18n( "Master" ) );
  }
  else if ( QString::compare( method, "host" ) == 0 )
  {
    m_listLabel->setText( config()->readEntry( "Query Host" ) );
  }
  else if ( QString::compare( method, "ipscan" ) == 0 )
  {
    m_listLabel->setText( i18n( "IP Scan" ) );
  }
  else
  {
    m_listLabel->setText( i18n( "Unknown" ) );
  }

  QString wins = optionsHandler()->winsServer();

  if ( QString::compare( wins, QString::null ) != 0 )
  {
    m_winsLabel->setText( wins );
  }
  else
  {
    m_winsLabel->setText( i18n( "None" ) );
  }

  m_listLabel->setFixedWidth( m_listLabel->sizeHint().width() + 5 );
  m_winsLabel->setFixedWidth( m_winsLabel->sizeHint().width() + 5 );
}


void Smb4KApp::slotShowSearchDialog()
{
  if ( manager()->getDockWidgetFromName( "SearchDialog" )->isHidden() )
  {
    manager()->getDockWidgetFromName( "SearchDialog" )->changeHideShowState();
  }

  m_search_dialog->comboBox()->lineEdit()->setFocus();
}


void Smb4KApp::slotShowNetworkBrowser()
{
  if ( manager()->getDockWidgetFromName( "NetworkBrowser" )->isHidden() )
  {
    manager()->getDockWidgetFromName( "NetworkBrowser" )->changeHideShowState();
  }

  m_network_view->setFocus();
}


void Smb4KApp::slotJumpToSharesView()
{
  m_share_view->setFocus();
}


void Smb4KApp::slotIconChanged( int group )
{
  // We do not need to switch the icon size, etc., for Smb4K*
  // child widgets here, because they will do it by themselves.

  switch ( group )
  {
    case KIcon::Small:
    {
      // Reset the icons of the tabs of the dock widgets:

      KDockWidget *dock = NULL;

      if ( (dock = manager()->getDockWidgetFromName( "ShareView" )) != NULL )
      {
        dock->setPixmap( SmallIcon( "hdd_mount" ) );
      }

      if ( (dock = manager()->getDockWidgetFromName( "NetworkBrowser" )) != NULL )
      {
        dock->setPixmap( SmallIcon( "network" ) );
      }

      if ( (dock = manager()->getDockWidgetFromName( "SearchDialog" )) != NULL )
      {
        dock->setPixmap( SmallIcon( "find" ) );
      }

      // Resize the actions:

      // FIXME

      break;
    }
    default:
    {
      break;
    }
  }
}


#include "smb4k.moc"
