/***************************************************************************
    smb4knetworkitems  -  Network items used by the Smb4KScanner class
    to pass and store data.
                             -------------------
    begin                : Mi Jun 2 2004
    copyright            : (C) 2004 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

#ifndef SMB4KNETWORKITEMS_H
#define SMB4KNETWORKITEMS_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// Qt includes
#include <qobject.h>
#include <qstring.h>

/**
 * This class provides a container for a workgroup found in the network
 * neighborhood.
 */

class Smb4KWorkgroupItem
{
  public:
    /**
     * The constructor.
     */
    Smb4KWorkgroupItem( const QString &name, const QString &master, const QString &masterIP = QString::null );

    /**
     * Empty constructor.
     */
    Smb4KWorkgroupItem() {}

    /**
     * The destructor.
     */
    ~Smb4KWorkgroupItem();

    /**
     * Returns the workgroup name.
     */
    const QString &workgroup() const { return m_name; }

    /**
     * Returns the name of the master browser.
     */
    const QString &master() const { return m_master; }

    /**
     * Returns the IP address of the master browser.
     */
    const QString &ip() const { return m_ip; }

    /**
     * You can mark the master as 'pseudo' with this function.
     */
    void setPseudo();

    /**
     * Returns TRUE, if the master is a 'pseudo'-master.
     */
    bool pseudo() const { return m_pseudo; }

    /**
     * This function sets the IP address of the master browser.
     *
     * @param ip            The IP address of the master browser
     */
    void setMasterIPAddress( const QString &ip );

  private:
    /**
     * The name of the workgroup.
     */
    QString m_name;

    /**
     * The name of the workgroup master.
     */
    QString m_master;

    /**
     * The IP address of the master.
     */
    QString m_ip;

    /**
     * Determines whether the master is a 'pseudo'-master.
     */
    bool m_pseudo;
};


/**
 * This class provides a container for a host found in the network
 * neighborhood.
 */

class Smb4KHostItem
{
  public:
    /**
     * The constructor.
     */
    Smb4KHostItem( const QString &workgroup, const QString &name, const QString &comment = QString::null, const QString &ip = QString::null );

    /**
     * The empty constructor.
     */
    Smb4KHostItem() {}

    /**
     * The destructor.
     */
    ~Smb4KHostItem();

    /**
     * Returns the workgroup the host is in.
     */
    const QString &workgroup() const { return m_workgroup; }

    /**
     * Returns the name of the host.
     */
    const QString &name() const { return m_name; }

    /**
     * Returns the IP address of the host.
     */
    const QString &ip() const { return m_ip; }

    /**
     * Returns the comment for this host.
     */
    const QString &comment() const { return m_comment; }

    /**
     * Sets the Server string that is reported by the host.
     */
    void setServerString( const QString &server );

    /**
     * Returns the Server string.
     */
    const QString &serverString() const { return m_server_string; }

    /**
     * Sets the OS string that is reported by the host.
     */
    void setOSString( const QString &os );

    /**
     * Returns the OS string.
     */
    const QString &osString() const { return m_os_string; }

    /**
     * This functions determines whether this host should be
     * registered as a master browser.
     *
     * @param master        Set this to TRUE if the host is a master browser.
     */
    void setMaster( bool master );

    /**
     * This function tells you if the host is a master or not.
     *
     * @returns             TRUE if the host is a master browser.
     */
    bool isMaster() { return m_master; }

    /**
     * This function sets the IP address of a host.
     *
     * @param ip            The IP address of a host
     */
    void setIPAddress( const QString &ip );

    /**
     * This function sets the command for a host.
     *
     * @param comment       The comment
     */
    void setComment( const QString &comment );

    /**
     * This function is used to tell the host item, if a
     * check for the IP address has already been performed.
     *
     * @param yes           Should be set to TRUE if a check was performed.
     */
    void setIPAddressChecked( bool yes );

    /**
     * Tells if a check for the IP address has already been performed.
     *
     * @returns             TRUE if a check was performed, and FALSE otherwise.
     */
    bool ipAddressChecked() { return m_ip_checked; }

    /**
     * This function is used to tell the host item, if a
     * check for the information (OS and Server string ) has already been performed.
     *
     * @param yes           Should be set to TRUE if a check was performed.
     */
    void setInfoChecked( bool yes );

    /**
     * Tells if a check for the information (OS and Server string) has already
     * been performed.
     *
     * @returns             TRUE is the check was performed previously.
     */
    bool infoChecked() { return m_info_checked; }

  private:
    /**
     * The name of the workgroup.
     */
    QString m_workgroup;

    /**
     * The name of the host.
     */
    QString m_name;

    /**
     * The comment for this host.
     */
    QString m_comment;

    /**
     * The IP address of the host.
     */
    QString m_ip;

    /**
     * The Server string as reported by the host.
     */
    QString m_server_string;

    /**
     * The operating system string as reported by the host.
     */
    QString m_os_string;

    /**
     * This boolian determines if the host is a master browser
     * or not.
     */
    bool m_master;

    /**
     * This boolean tells if a check for the IP address was already
     * performed.
     */
    bool m_ip_checked;

    /**
     * This boolean tells if a check for the information (OS and server string)
     * was already performed.
     */
    bool m_info_checked;
};


/**
 * This class provides a container for a share found in the
 * network neighborhood.
 */

class Smb4KShareItem
{
  public:
    /**
     * The constructor.
     */
    Smb4KShareItem( const QString &workgroup, const QString &host, const QString &name, const QString &type, const QString &comment );

    /**
     * Empty constructor.
     */
    Smb4KShareItem() {}

    /**
     * The destructor.
     */
    ~Smb4KShareItem();

    /**
     * Returns the workgroup of the host where the share is located.
     */
    const QString &workgroup() const { return m_workgroup; }

    /**
     * Returns the name of the host where the share is located.
     */
    const QString &host() const { return m_host; }

    /**
     * Returns the name of the share.
     */
    const QString &name() const { return m_name; }

    /**
     * Returns the type of the share.
     */
    const QString &plainType() const { return m_type; }

    /**
     * Returns a translated version of the type of the share. If
     *
     * @returns           The translated share type
     */
    const QString translatedType() const;

    /**
     * Returns the comment for this share.
     */
    const QString &comment() const { return m_comment; }

  private:
    /**
     * The workgroup object.
     */
    QString m_workgroup;

    /**
     * The host name.
     */
    QString m_host;

    /**
     * The name of the share.
     */
    QString m_name;

    /**
     * The type of the share.
     */
    QString m_type;

    /**
     * The comment for this share.
     */
    QString m_comment;
};


/**
 * This class provides a container for the preview
 * of the contents of a share.
 */

class Smb4KPreviewItem
{
  public:
    /**
     * The default constructor.
     */
    Smb4KPreviewItem( const QString &host, const QString &share, const QString &path, const QString &name, bool file = true );
    /**
     * The empty constructor.
     */
    Smb4KPreviewItem() {}
    /**
     * The destructor.
     */
    ~Smb4KPreviewItem();
    /**
     * Return the name of the host where the share is located.
     */
    const QString &host() const { return m_host; }
    /**
     * Return the name of the share that is to be previewed.
     */
    const QString &share() const { return m_share; }
    /**
     * Return the path that is to be previewed.
     */
    const QString &path() const { return m_path; }
    /**
     * Return the name of the item.
     */
    const QString &name() const { return m_name; }
    /**
     * Returns TRUE if this item is a file.
     */
    bool isFile() const { return m_isFile; }
    /**
     * Sets the item to be a file or a directory.
     */
    void setIsFile( bool sw );

  private:
    /**
     * The host name.
     */
    QString m_host;
    /**
     * The share name.
     */
    QString m_share;
    /**
     * The path that has to be previewed.
     */
    QString m_path;
    /**
     * The name of the item.
     */
    QString m_name;
    /**
     * Determines whether this is a file or a directory.
     */
    bool m_isFile;
};

#endif
