/***************************************************************************
    smb4ksharewidget.cpp  -  The widget, that displays the mounted shares.
                             -------------------
    begin                : Sam Mr 1 2003
    copyright            : (C) 2003 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// Qt includes
#include <qdir.h>
#include <qregexp.h>

// KDE includes
#include <klocale.h>
#include <kprocess.h>
#include <kmessagebox.h>
#include <kapplication.h>
#include <kactionclasses.h>
#include <kpopupmenu.h>
#include <kiconloader.h>

// system includes
#include <unistd.h>
#include <sys/types.h>

// application specific includes
#include "smb4ksharewidget.h"
#include "smb4kmountdialog.h"
#include "../core/smb4kcore.h"


Smb4KShareWidget::Smb4KShareWidget( QWidget *parent, const char *name ) : KIconView( parent, name )
{
  setFrameShape( KIconView::PopupPanel );
  setFrameShadow( KIconView::Sunken );
  setSelectionMode( KIconView::Single );
  setResizeMode( KIconView::Adjust );
  setShowToolTips( true );
  setAutoArrange( true );
  setSorting( true, true );

  m_collection = new KActionCollection( this, "SharesView_ActionCollection", KGlobal::instance() );

  initActions();

  // Internal connections
  connect( this,                  SIGNAL( mouseButtonPressed( int, QIconViewItem *, const QPoint & ) ), this, SLOT( slotButtonPressed( int, QIconViewItem *, const QPoint & ) ) );
  connect( this,                  SIGNAL( executed( QIconViewItem * ) ),                                this, SLOT( slotOpenFilemanager() ) );
  connect( this,                  SIGNAL( selectionChanged( QIconViewItem * ) ),                        this, SLOT( slotSelectionChanged( QIconViewItem * ) ) );
  connect( smb4k_core->mounter(), SIGNAL( updated() ),                                           this, SLOT( slotMountedShares() ) );
}


Smb4KShareWidget::~Smb4KShareWidget()
{
}


/****************************************************************************
   Initialize the actions needed by this class.
****************************************************************************/

void Smb4KShareWidget::initActions()
{
  m_unmount_action = new KAction( i18n( "&Unmount" ), "hdd_unmount", CTRL+Key_U, this,            SLOT( slotUnmountShare() ), m_collection, "shares_unmount_share" );
  m_unmount_action->setGroup( "ShareWidget" );
  m_unmount_action->setEnabled( false );

  m_force_action = new KAction( i18n( "&Force Unmounting" ), "hdd_unmount", CTRL+Key_F, this,     SLOT( slotForceUnmountShare() ), m_collection, "shares_force_unmount" );
  m_force_action->setGroup( "ShareWidget" );
  m_force_action->setEnabled( false );

  m_unmount_all_action = new KAction( i18n( "U&nmount All" ), "gear", CTRL+Key_N, this,           SLOT( slotUnmountAllShares() ), m_collection, "shares_unmount_all" );
  m_unmount_all_action->setGroup( "ShareWidget" );
  m_unmount_all_action->setEnabled( false );

  m_manual_action = new KAction( i18n( "M&ount Manually" ), "connect_creating", CTRL+Key_O, this, SLOT( slotManualMount() ), m_collection, "shares_manual_mount" );
  m_manual_action->setGroup( "ShareWidget" );
  m_manual_action->setEnabled( true );

  m_filemanager_action = new KAction( i18n( "&Konqueror" ), "kfm_home", CTRL+Key_K, this,         SLOT( slotOpenFilemanager() ), m_collection, "shares_start_filemanager" );
  m_filemanager_action->setGroup( "ShareWidget" );
  m_filemanager_action->setEnabled( false );
}


/****************************************************************************
   Reads the options for the share view widget.
****************************************************************************/

void Smb4KShareWidget::readOptions()
{
  // Which appearance does the user want?
  config()->setGroup( "Appearance" );
  m_display = config()->readEntry( "Show Shares", "icons" );

  if ( QString::compare( m_display, "icons" ) == 0 )
  {
    setItemTextPos( KIconView::Bottom );
    setMaxItemWidth( 150 );
    setArrangement( KIconView::LeftToRight );
    setWordWrapIconText( true );
  }
  else if ( QString::compare( m_display, "list" ) == 0 )
  {
    setItemTextPos( KIconView::Right );
    setMaxItemWidth( 300 );
    setArrangement( KIconView::TopToBottom );
    setWordWrapIconText( false );
  }

  m_mountpoint = config()->readBoolEntry( "Show Mount Point", false );

  // Should all share be shown, that were mounted on the system?
  m_showExternal = config()->readBoolEntry( "Show All Shares", false );

  if ( count() != 0 )
  {
    for ( QIconViewItem *item = this->firstItem(); item; item = item->nextItem() )
    {
      if ( QString::compare( m_display, "icons" ) == 0 )
      {
        item->setPixmap( DesktopIcon( "hdd_mount" ) );
      }
      else if ( QString::compare( m_display, "list" ) == 0 )
      {
        item->setPixmap( SmallIcon( "hdd_mount" ) );
      }

      if ( m_mountpoint )
      {
        item->setText( ((Smb4KShareWidgetItem *)item)->mountpoint() );
      }
      else
      {
        item->setText( ((Smb4KShareWidgetItem *)item)->name() );
      }
    }
  }

  slotMountedShares();

  adjustItems();

  // Mount options.
  config()->setGroup( "Programs" );
  QString super = config()->readPathEntry( "super" );
  QString sudo = config()->readPathEntry( "sudo" );

  config()->setGroup( "Super User Privileges" );
  if ( !super.isEmpty() || !sudo.isEmpty() )
  {
    m_forcedUnmount = config()->readBoolEntry( "Force Unmount", false );
  }
  else
  {
    m_forcedUnmount = false;
  }
}


/////////////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////


/****************************************************************************
   Internal slots.
****************************************************************************/

void Smb4KShareWidget::slotButtonPressed( int button, QIconViewItem *item, const QPoint &pos )
{
  if ( item )
  {
    Smb4KShareWidgetItem *i = (Smb4KShareWidgetItem *)item;

    // Enable the actions:
    if ( m_forcedUnmount )
    {
      m_force_action->setEnabled( true );
    }

    if ( !i->isBroken() )
    {
      m_filemanager_action->setEnabled( true );
      smb4k_core->setShareForUsageInfo( i->shareObject() );
    }
    else
    {
      m_filemanager_action->setEnabled( false );
      smb4k_core->setShareForUsageInfo( NULL );
    }

    m_unmount_action->setEnabled( true );
  }
  else
  {
    m_unmount_action->setEnabled( false );
    m_filemanager_action->setEnabled( false );
    m_force_action->setEnabled( false );

    smb4k_core->setShareForUsageInfo( NULL );
    clearSelection();
  }

  if ( button == KIconView::RightButton )
  {
    KActionMenu *menu = new KActionMenu( this, "SharesViewPopupMenu" );

    menu->popupMenu()->insertTitle( SmallIcon( "hdd_mount" ), i18n( "Shares" ), 0, 0 );
    menu->insert( m_unmount_action );
    menu->insert( m_force_action );
    menu->insert( m_unmount_all_action );
    menu->popupMenu()->insertSeparator( -1 );
    menu->insert( m_manual_action );
    menu->popupMenu()->insertSeparator( -1 );
    menu->insert( m_filemanager_action );

    menu->popupMenu()->exec( pos, 0 );
  }
}


void Smb4KShareWidget::slotSelectionChanged( QIconViewItem *item )
{
  if ( item )
  {
    Smb4KShareWidgetItem *i = (Smb4KShareWidgetItem *)item;

    if ( !i->isBroken() )
    {
      m_filemanager_action->setEnabled( true );
      smb4k_core->setShareForUsageInfo( i->shareObject() );
    }
    else
    {
      m_filemanager_action->setEnabled( false );
      smb4k_core->setShareForUsageInfo( NULL );
    }

    if ( m_forcedUnmount )
    {
      m_force_action->setEnabled( true );
    }

    m_unmount_action->setEnabled( true );
  }
}


/****************************************************************************
   External slots.
****************************************************************************/

void Smb4KShareWidget::slotMountedShares()
{
  QValueList<Smb4KShare *> shares = smb4k_core->mounter()->getShares();
  Smb4KShareWidgetItem *current = (Smb4KShareWidgetItem *)currentItem();

  if ( !shares.isEmpty() )
  {
    // First remove those, that are not mounted anymore:
    for ( Smb4KShareWidgetItem *item = (Smb4KShareWidgetItem *)this->firstItem(); item; item = (Smb4KShareWidgetItem *)item->nextItem() )
    {
      if( !smb4k_core->mounter()->findShareByPath( item->mountpoint() ) ||
          ( item->shareObject()->isForeign() && !m_showExternal ) )
      {
        if ( current == item )
        {
          smb4k_core->setShareForUsageInfo( NULL );
        }

        delete item;
        this->sort( this->sortDirection() );
      }
    }

    // Now process the entries in the list:
    for ( QValueListIterator<Smb4KShare *> it = shares.begin(); it != shares.end(); ++it )
    {
      // Check, whether the share is already in the list. Look for the
      // mount point.
      bool alreadyInView = false;

      for ( Smb4KShareWidgetItem *item = (Smb4KShareWidgetItem *)this->firstItem(); item; item = (Smb4KShareWidgetItem *)item->nextItem() )
      {
        if ( QString::compare( item->mountpoint(), (*it)->getPath() ) == 0 ||
             QString::compare( item->mountpoint(), (*it)->getCanonicalPath() ) == 0 )
        {
          if ( (*it)->isBroken() && !item->isBroken() )
          {
            item->setBroken( true );
          }
          else if ( !(*it)->isBroken() && item->isBroken() )
          {
            item->setBroken( false );
          }

          alreadyInView = true;
          break;
        }
        else
        {
          continue;
        }
      }

      if ( !alreadyInView )
      {
        if ( QString::compare( m_display, "icons" ) == 0 && !m_showExternal && (*it)->isForeign() )
        {
          continue;
        }
        else if ( QString::compare( m_display, "icons" ) == 0 )
        {
          new Smb4KShareWidgetItem( *it, this, Smb4KShareWidgetItem::Icons, m_mountpoint );
        }

        if ( QString::compare( m_display, "list" ) == 0 && !m_showExternal && (*it)->isForeign() )
        {
          continue;
        }
        else if ( QString::compare( m_display, "list" ) == 0 )
        {
          new Smb4KShareWidgetItem( *it, this, Smb4KShareWidgetItem::List, m_mountpoint );
        }

        // Do the sorting:
        this->sort( this->sortDirection() );
      }
    }

    if ( current != (Smb4KShareWidgetItem *)currentItem() )
    {
      m_unmount_action->setEnabled( false );
      m_force_action->setEnabled( false );
      m_filemanager_action->setEnabled( false );

      clearSelection();
      emit clearedSelection();
    }

    m_unmount_all_action->setEnabled( true );
  }
  else
  {
    smb4k_core->setShareForUsageInfo( NULL );
    delete current;
    clear();

    m_unmount_action->setEnabled( false );
    m_force_action->setEnabled( false );
    m_unmount_all_action->setEnabled( false );
    m_filemanager_action->setEnabled( false );

    emit cleared();
  }
}


/***************************************************************************
   Slots for the menu.
***************************************************************************/


void Smb4KShareWidget::slotOpenFilemanager()
{
  if ( currentItem() && !((Smb4KShareWidgetItem *)currentItem())->isBroken() )
  {
    smb4k_core->open( ((Smb4KShareWidgetItem *)currentItem())->shareObject() );
  }
}


void Smb4KShareWidget::slotUnmountShare()
{
  if ( currentItem() )
  {
    Smb4KShareWidgetItem *i = (Smb4KShareWidgetItem *)currentItem();
    smb4k_core->mounter()->unmountShare( i->mountpoint(), QString( "%1" ).arg( i->uid() ), QString( "%1" ).arg( i->gid() ) );
  }
}


void Smb4KShareWidget::slotUnmountAllShares()
{
  smb4k_core->mounter()->unmountAllShares();
}


void Smb4KShareWidget::slotForceUnmountShare()
{
  if ( currentItem() )
  {
    Smb4KShareWidgetItem *i = (Smb4KShareWidgetItem *)currentItem();
    smb4k_core->mounter()->forceUnmountShare( i->mountpoint(), QString( "%1" ).arg( i->uid() ), QString( "%1" ).arg( i->gid() ) );
  }
}


void Smb4KShareWidget::slotManualMount()
{
  Smb4KMountDialog *dlg = new Smb4KMountDialog( 0, "ManualMountDialog" );
  dlg->show();
}


/***************************************************************************
 *                                                                         *
 *   The Smb4KShareWidgetItem class.                                       *
 *                                                                         *
 ***************************************************************************/


Smb4KShareWidgetItem::Smb4KShareWidgetItem( Smb4KShare *share, Smb4KShareWidget *parent, int alignment, bool mp )
: KIconViewItem( parent, QString::null ), m_share( *share ), m_alignment( alignment )
{
  setBroken( m_share.isBroken() );

  if ( mp )
  {
    setText( m_share.getPath() );
  }
  else
  {
    setText( m_share.getName() );
  }
}


Smb4KShareWidgetItem::~Smb4KShareWidgetItem()
{
}

/****************************************************************************
   Paints an item.
****************************************************************************/


void Smb4KShareWidgetItem::paintItem( QPainter *p, const QColorGroup &cg )
{
  QColorGroup colorgrp( cg );

  if ( m_share.isForeign() || m_share.isBroken() )
  {
    colorgrp.setColor( QColorGroup::Text, Qt::gray );
  }

  QIconViewItem::paintItem( p, colorgrp );
}


void Smb4KShareWidgetItem::setBroken( bool broken )
{
  m_share.setBroken( broken );

  if ( broken )
  {
    if ( !m_share.isForeign() && m_alignment == Icons )
    {
      setPixmap( DesktopIcon( "file_broken" ) );
    }
    else if ( !m_share.isForeign() && m_alignment == List )
    {
      setPixmap( SmallIcon( "file_broken" ) );
    }
    else if ( m_share.isForeign() && m_alignment == Icons )
    {
      setPixmap( kapp->iconLoader()->loadIcon( "file_broken", KIcon::Desktop, 0, KIcon::DisabledState, 0L, false ) );
    }
    else if ( m_share.isForeign() && m_alignment == List )
    {
      setPixmap( kapp->iconLoader()->loadIcon( "file_broken", KIcon::Small, 0, KIcon::DisabledState, 0L, false ) );
    }
  }
  else
  {
    if ( !m_share.isForeign() && m_alignment == Icons )
    {
      setPixmap( DesktopIcon( "hdd_mount" ) );
    }
    else if ( !m_share.isForeign() && m_alignment == List )
    {
      setPixmap( SmallIcon( "hdd_mount" ) );
    }
    else if ( m_share.isForeign() && m_alignment == Icons )
    {
      setPixmap( kapp->iconLoader()->loadIcon( "hdd_mount", KIcon::Desktop, 0, KIcon::DisabledState, 0L, false ) );
    }
    else if ( m_share.isForeign() && m_alignment == List )
    {
      setPixmap( kapp->iconLoader()->loadIcon( "hdd_mount", KIcon::Small, 0, KIcon::DisabledState, 0L, false ) );
    }
  }
}


#include "smb4ksharewidget.moc"

