/***************************************************************************
    smb4kbrowserwidget.cpp  -  Network browser widget class of Smb4K.
                             -------------------
    begin                : Sam Feb 22 2003
    copyright            : (C) 2003 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// Qt includes
#include <qlayout.h>
#include <qheader.h>
#include <qvaluelist.h>
#include <qlabel.h>

// KDE includes
#include <kapplication.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kdialogbase.h>
#include <kglobalsettings.h>
#include <kactionclasses.h>
#include <kpopupmenu.h>
#include <kiconloader.h>

// application specific includes
#include "smb4kbrowserwidget.h"
#include "smb4kpreviewdialog.h"
#include "smb4kprintdialog.h"
#include "../core/smb4kshare.h"
#include "../core/smb4kcore.h"
#include "../core/smb4kpasswordhandler.h"
#include "../core/smb4kglobal.h"

using namespace Smb4K_Global;


Smb4KBrowserWidget::Smb4KBrowserWidget( QWidget *parent, const char *name ) : KListView( parent, name )
{
  setFrameShape( KListView::PopupPanel );
  setFrameShadow( KListView::Sunken );
  setRootIsDecorated( true );

  addColumn( i18n( "Network" ), -1 );
  addColumn( i18n( "Type" ), -1 );
  addColumn( i18n( "IP Address" ), -1 );
  addColumn( i18n( "Comment" ), -1 );

  m_collection = new KActionCollection( this, "BrowserWidget_ActionCollection", KGlobal::instance() );

  initActions();

  // Widget specific connections.
  connect( this, SIGNAL( executed( QListViewItem * ) ),
           this, SLOT( slotItemExecuted( QListViewItem * ) ) );
  connect( this, SIGNAL( expanded( QListViewItem * ) ),
           this, SLOT( slotItemExpandedCollapsed( QListViewItem * ) ) );
  connect( this, SIGNAL( collapsed( QListViewItem * ) ),
           this, SLOT( slotItemExpandedCollapsed( QListViewItem * ) ) );
  connect( this, SIGNAL( selectionChanged( QListViewItem * ) ),
           this, SLOT( slotSelectionChanged( QListViewItem * ) ) );
  connect( this, SIGNAL( clicked( QListViewItem * ) ),
           this, SLOT( slotItemClicked( QListViewItem * ) ) );
  connect( this, SIGNAL( rightButtonPressed( QListViewItem *, const QPoint &, int ) ),
           this, SLOT( slotRightButtonPressed( QListViewItem *, const QPoint &, int ) ) );

  // External connections.
  connect( smb4k_core->scanner(),  SIGNAL( workgroups( const QValueList<Smb4KWorkgroupItem *> & ) ),
           this,                   SLOT( slotWorkgroups( const QValueList<Smb4KWorkgroupItem *> & ) ) );
  connect( smb4k_core->scanner(),  SIGNAL( members( const QValueList<Smb4KHostItem *> & ) ),
           this,                   SLOT( slotMembers( const QValueList<Smb4KHostItem *> & ) ) );
  connect( smb4k_core->scanner(),  SIGNAL( shares( const QValueList<Smb4KShareItem *> & ) ),
           this,                   SLOT( slotShares( const QValueList<Smb4KShareItem *> & ) ) );
  connect( smb4k_core->ipFinder(), SIGNAL( ip_address( Smb4KHostItem * ) ),
           this,                   SLOT( slotAddIPAddress( Smb4KHostItem * ) ) );
  connect( smb4k_core->mounter(),  SIGNAL( updated() ),
           this,                   SLOT( slotMarkShares() ) );
}


Smb4KBrowserWidget::~Smb4KBrowserWidget()
{
}


void Smb4KBrowserWidget::initActions()
{
  m_preview_action = new KAction( i18n( "Pre&view" ), "view_icon", CTRL+Key_V, this,          SLOT( slotPreview() ), m_collection, "network_preview" );
  m_preview_action->setGroup( "BrowserWidget" );
  m_preview_action->setEnabled( false );

  m_askpass_action = new KAction( i18n( "Au&thentication" ), "identity", CTRL+Key_T, this,    SLOT( slotAskPass() ), m_collection, "network_askpass" );
  m_askpass_action->setGroup( "BrowserWidget" );
  m_askpass_action->setEnabled( false );

  m_print_action = new KAction( i18n( "&Print File" ), "printer1", CTRL+Key_P, this,          SLOT( slotPrint() ), m_collection, "network_print" );
  m_print_action->setGroup( "BrowserWidget" );
  m_print_action->setEnabled( false );

  m_mount_action = new KAction( i18n( "&Mount" ), "hdd_mount", CTRL+Key_M, this,              SLOT( slotMountShare() ), m_collection, "network_mount" );
  m_mount_action->setGroup( "BrowserWidget" );
  m_mount_action->setEnabled( false );

  m_bookmark_action = new KAction( i18n( "&Add Bookmark" ), "bookmark_add", CTRL+Key_B, this, SLOT( slotAddBookmark() ), m_collection, "network_add_bookmark" );
  m_bookmark_action->setGroup( "BrowserWidget" );
  m_bookmark_action->setEnabled( false );

  m_rescan_action = new KAction( i18n( "&Rescan" ), "reload", CTRL+Key_R, this,               SLOT( slotRescan() ), m_collection, "network_rescan" );
  m_rescan_action->setGroup( "BrowserWidget" );
  m_rescan_action->setEnabled( true );

  m_abort_action = new KAction( i18n( "&Abort" ) , "stop", CTRL+Key_A, this,                  SLOT( slotAbort() ), m_collection, "abort" );
  m_abort_action->setGroup( "BrowserWidget" );
  m_abort_action->setEnabled( false );

  m_search_action = new KAction( i18n( "&Search" ), "find", CTRL+Key_S, this,                 SLOT( slotSearch() ), m_collection, "network_search" );
  m_search_action->setGroup( "BrowserWidget" );
  m_search_action->setEnabled( true );
}


void Smb4KBrowserWidget::setOpen( QListViewItem *item, bool open )
{
  if ( open && item->depth() == 0 )
  {
    Smb4KWorkgroupItem *i = smb4k_core->scanner()->getWorkgroup( item->text( Network ) );

    smb4k_core->scanner()->getWorkgroupMembers( item->text( Network ), i->master(), i->ip() );
  }
  else if ( open && item->depth() == 1 )
  {
    smb4k_core->scanner()->getShares( item->parent()->text( Network ), item->text( Network ), item->text( IP ) );
    smb4k_core->scanner()->getInfo( item->parent()->text( Network ), item->text( Network ), item->text( IP ) );
  }

  KListView::setOpen( item, open );
}


void Smb4KBrowserWidget::insertItem( Smb4KHostItem *item )
{
  if ( smb4k_core->scanner()->getWorkgroup( item->workgroup() ) == 0 )
  {
    smb4k_core->scanner()->appendWorkgroup( new Smb4KWorkgroupItem( item->workgroup(), item->name(), item->ip() ) );
  }

  // Now put the host into the list.
  QListViewItem *workgroupItem = findItem( item->workgroup(), Network, ExactMatch|CaseSensitive );

  if ( workgroupItem != 0 )
  {
    QListViewItem *hostItem = findItem( item->name(), Network, ExactMatch|CaseSensitive );

    if ( hostItem == 0 )
    {
      Smb4KBrowserWidgetItem *newItem = new Smb4KBrowserWidgetItem( workgroupItem, item );
      newItem->setExpandable( true );
    }
    else if ( hostItem != 0 && QString::compare( hostItem->parent()->text( Network ), item->workgroup() ) != 0 )
    {
      Smb4KBrowserWidgetItem *newItem = new Smb4KBrowserWidgetItem( workgroupItem, item );
      newItem->setExpandable( true );
    }
    else
    {
      KMessageBox::sorry( this, i18n( "This computer is already in the list." ), QString::null, KMessageBox::Notify );
    }
  }
  else
  {
    Smb4KBrowserWidgetItem *newItem = new Smb4KBrowserWidgetItem( this, new Smb4KWorkgroupItem( item->workgroup(), item->name(), item->ip() ) );
    newItem->setExpandable( true );

    Smb4KBrowserWidgetItem *hostItem = new Smb4KBrowserWidgetItem( newItem, item );
    hostItem->setExpandable( true );
  }
}


void Smb4KBrowserWidget::readOptions()
{
  // Read the configuration.
  config()->setGroup( "Appearance" );
  bool showType = config()->readBoolEntry( "Show Type", true );
  bool showComment = config()->readBoolEntry( "Show Comment", true );
  bool showIP = config()->readBoolEntry( "Show IP", true );

  // Now put everything back in according to the wishes of the user.
  if ( showIP )
  {
    setColumnWidth( IP, 10 );
    setColumnWidthMode( IP, QListView::Maximum );
  }
  else
  {
    setColumnWidth( IP, 0 );
    setColumnWidthMode( IP, QListView::Manual );
  }

  if ( showType )
  {
    setColumnWidth( Type, 10 );
    setColumnWidthMode( Type, QListView::Maximum );
  }
  else
  {
    setColumnWidth( Type, 0 );
    setColumnWidthMode( Type, QListView::Manual );
  }

  if ( showComment )
  {
    setColumnWidth( Comment, 10 );
    setColumnWidthMode( Comment, QListView::Maximum );
  }
  else
  {
    setColumnWidth( Comment, 0 );
    setColumnWidthMode( Comment, QListView::Manual );
  }

  // Adjust the columns. Avoid those that have a width of 0.
  int index = 0;

  while ( index < this->columns() )
  {
    if ( this->columnWidth( index ) != 0 )
    {
      adjustColumn( index );
    }

    index++;
  }

  // Initialize the rest.
  m_hidden = config()->readBoolEntry( "Show Hidden", true );
  m_ipc = config()->readBoolEntry( "Show IPC", false );
  m_admin = config()->readBoolEntry( "Show ADMIN", false );
  m_printer = config()->readBoolEntry( "Show Printer", true );
}


const QString Smb4KBrowserWidget::specifyUser( const QString &host )
{
  KDialogBase *dialog = new KDialogBase( KDialogBase::Plain, i18n( "Specify User" ), KDialogBase::Ok|KDialogBase::Cancel, KDialogBase::Ok, this, 0, false, true );

  // Set up the ask pass dialog.
  QFrame *frame = dialog->plainPage();
  QGridLayout *layout = new QGridLayout( frame );
  layout->setSpacing( 5 );

  QLabel *pic = new QLabel( frame );
  pic->setPixmap( DesktopIcon( "personal" ) );
  pic->setMargin( 10 );

  QLabel *text = new QLabel( i18n( "Please specify a user name." ), frame );

  QLabel *user = new QLabel( i18n( "User:" ), frame );
  KComboBox *username = new KComboBox( true, frame, 0 );
  username->setDuplicatesEnabled( false );

  QSpacerItem *spacer1 = new QSpacerItem( 10, 10, QSizePolicy::Expanding, QSizePolicy::Preferred );

  layout->addWidget( pic, 0, 0, 0 );
  layout->addMultiCellWidget( text, 0, 0, 1, 3, 0 );
  layout->addWidget( user, 1, 0, 0 );
  layout->addMultiCellWidget( username, 1, 1, 1, 4, 0 );
  layout->addItem( spacer1, 0, 2 );

  // Read the list of logins, that were already defined
  // for this 'homes' share.
  QStringList list;

  if ( config()->hasGroup( "Homes Shares" ) )
  {
    config()->setGroup( "Homes Shares" );

    if ( config()->hasKey( host ) )
    {
      list = config()->readListEntry( host, ',' );
    }
  }

  if ( !list.isEmpty() )
  {
    username->insertStringList( list, -1 );
  }

  username->setCurrentText( QString::null );

  // Do the last things before showing.
  username->setFocus();
  dialog->setFixedSize( dialog->sizeHint() );

  if ( dialog->exec() == KDialogBase::Accepted )
  {
    // Write the new list of logins to the config file.
    if ( list.contains( username->currentText() ) == 0 )
    {
      list.append( username->currentText() );
    }

    int index = 0;
    while ( index < username->count() )
    {
      if ( list.contains( username->text( index ) ) == 0 )
      {
        list.append( username->text( index ) );
      }

      index++;
    }

    list.sort();

    config()->setGroup( "Homes Shares" );
    config()->writeEntry( host, list, ',' );

    return username->currentText();
  }
  else
  {
    return QString::null;
  }
}


/////////////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////


void Smb4KBrowserWidget::slotItemExecuted( QListViewItem *item )
{
  if ( item )
  {
    if ( item->depth() == 2 )
    {
      if ( item->text( Type ).contains( "Printer" ) == 0 )
      {
        slotMountShare();
      }
      else
      {
        slotPrint();
      }
    }
    else
    {
      setOpen( item, !item->isOpen() );
    }
  }
}


void Smb4KBrowserWidget::slotItemExpandedCollapsed( QListViewItem *item )
{
  setSelected( item, true );
}


void Smb4KBrowserWidget::slotSelectionChanged( QListViewItem *item )
{
  if ( item && item->depth() == 2 )
  {
    m_preview_action->setEnabled( true );

    if ( item->text( Type ).contains( "Printer" ) == 0 )
    {
      m_bookmark_action->setEnabled( true );
      m_preview_action->setEnabled( true );
      m_mount_action->setEnabled( true );
      m_print_action->setEnabled( false );
    }
    else
    {
      m_bookmark_action->setEnabled( false );
      m_preview_action->setEnabled( false );
      m_mount_action->setEnabled( false );
      m_print_action->setEnabled( true );
    }
  }
  else if ( item && item->depth() == 1 )
  {
    m_bookmark_action->setEnabled( false );
    m_askpass_action->setEnabled( true );
    m_preview_action->setEnabled( false );
    m_print_action->setEnabled( false );
    m_mount_action->setEnabled( false );
  }
  else
  {
    m_bookmark_action->setEnabled( false );
    m_preview_action->setEnabled( false );
    m_print_action->setEnabled( false );
    m_mount_action->setEnabled( false );
    m_askpass_action->setEnabled( false );
  }

  if ( item )
  {
    emit highlightedItem( item );
  }
}


void Smb4KBrowserWidget::slotRightButtonPressed( QListViewItem *, const QPoint &point, int )
{
  KActionMenu *menu = new KActionMenu( this, "BrowserPopupMenu" );

  menu->popupMenu()->insertTitle( SmallIcon( "network" ), i18n( "Network" ), 0, 0 );
  menu->insert( m_rescan_action );
  menu->insert( m_abort_action );
  menu->popupMenu()->insertSeparator( -1 );
  menu->insert( m_search_action );
  menu->popupMenu()->insertSeparator( -1 );
  menu->insert( m_bookmark_action );
  menu->insert( m_askpass_action );
  menu->insert( m_preview_action );
  menu->insert( m_print_action );
  menu->insert( m_mount_action );

  menu->popupMenu()->exec( point, 0 );
}


void Smb4KBrowserWidget::slotItemClicked( QListViewItem *item )
{
  // We only need this for the case, that the user clicks somewhere, where
  // NO item is. Smb4KNetworkTab needs it to clear the tab and the main window
  // to disable some actions in the tool bar.
  if ( !item && !selectedItem() )
  {
    m_bookmark_action->setEnabled( false );
    m_preview_action->setEnabled( false );
    m_print_action->setEnabled( false );
    m_mount_action->setEnabled( false );
    m_askpass_action->setEnabled( false );

    emit highlightedItem( item );
  }
}


void Smb4KBrowserWidget::slotWorkgroups( const QValueList<Smb4KWorkgroupItem *> &list )
{
  // Clear the list view:
  clear();

  emit cleared();

  // Adjust the columns.
  for ( int index = 0; index < columns(); index++ )
  {
    if ( this->columnWidth( index ) != 0 )
    {
      adjustColumn( index );
    }
  }

  // Put the workgroups into the list view.
  if ( !list.isEmpty() )
  {
    for ( QValueList<Smb4KWorkgroupItem *>::ConstIterator it = list.begin(); it != list.end(); ++it )
    {
      if ( this->findItem( (*it)->workgroup(), Network, CaseSensitive|ExactMatch ) == 0 )
      {
        Smb4KBrowserWidgetItem *workgroup = new Smb4KBrowserWidgetItem( this, *it );
        workgroup->setExpandable( true );
      }
      else
      {
        continue;
      }
    }
  }
}



void Smb4KBrowserWidget::slotMembers( const QValueList<Smb4KHostItem *> &list )
{
  if ( !list.isEmpty() )
  {
    for ( QValueList<Smb4KHostItem *>::ConstIterator it = list.begin(); it != list.end(); ++it )
    {
      QListViewItem *item = this->findItem( (*it)->name(), Network, CaseSensitive|ExactMatch );

      if ( item != 0 )
      {
        if ( !((*it)->ip()).isEmpty() && QString::compare( item->text( IP ), (*it)->ip() ) != 0 )
        {
          item->setText( IP, (*it)->ip() );
        }

        if ( !((*it)->comment()).isEmpty() && QString::compare( item->text( Comment ), (*it)->comment() ) != 0 )
        {
          item->setText( Comment, (*it)->comment() );
        }

        continue;
      }
      else
      {
        Smb4KBrowserWidgetItem *host = new Smb4KBrowserWidgetItem( this->findItem( (*it)->workgroup(), Network, ExactMatch|CaseSensitive ), *it );
        host->setExpandable( true );
      }
    }
  }
}


void Smb4KBrowserWidget::slotShares( const QValueList<Smb4KShareItem *> &list )
{
  if ( !list.isEmpty() )
  {
    for ( QValueList<Smb4KShareItem *>::ConstIterator it = list.begin(); it != list.end(); ++it )
    {
      QListViewItem *host = this->findItem( (*it)->host(), Network, ExactMatch|CaseSensitive );

      if ( host != 0 )
      {
        if ( !m_hidden && (*it)->name().endsWith( "$" ) )
        {
          continue;
        }
        else
        {
          if ( !m_ipc && (*it)->name().contains( "IPC" ) != 0 )
            continue;

          if ( !m_admin && (*it)->name().contains( "ADMIN" ) != 0 )
            continue;
        }

        if ( !m_printer && QString::compare( (*it)->name(), "Printer" ) == 0 )
        {
          continue;
        }

        QListViewItem *test = findItem( (*it)->name(), Network, ExactMatch|CaseSensitive );

        if ( test == 0 || test->parent() != host )
        {
          Smb4KBrowserWidgetItem *share = new Smb4KBrowserWidgetItem( host, *it );
          share->setExpandable( false );

          continue;
        }
        else if ( test != 0 && test->parent() == host )
        {
          if ( !((*it)->comment()).isEmpty() && QString::compare( test->text( Comment ), (*it)->comment() ) != 0 )
          {
            test->setText( Comment, (*it)->comment() );
          }

          continue;
        }
      }
    }
  }
}


void Smb4KBrowserWidget::slotMountShare()
{
  if ( currentItem() && currentItem()->depth() == 2 )
  {
    if ( QString::compare( currentItem()->text( Network ).stripWhiteSpace(), "homes" ) == 0 )
    {
      QString name = specifyUser( currentItem()->parent()->text( Network ) );

      if ( !name.isEmpty() )
      {
        smb4k_core->mounter()->mountShare( currentItem()->parent()->parent()->text( Network ), currentItem()->parent()->text( Network ), currentItem()->parent()->text( IP ), name );
      }
    }
    else
    {
      smb4k_core->mounter()->mountShare( currentItem()->parent()->parent()->text( Network ), currentItem()->parent()->text( Network ), currentItem()->parent()->text( IP ), currentItem()->text( Network ) );
    }
  }
}


void Smb4KBrowserWidget::slotAskPass()
{
  if ( currentItem() && currentItem()->depth() == 1 )
  {
    QString workgroup = currentItem()->parent()->text( Network );
    QString host = currentItem()->text( Network );
    QString share = QString::null;

    smb4k_core->passwordHandler()->askpass( workgroup, host, share, Smb4KPasswordHandler::NewData );
  }
  else if ( currentItem() && currentItem()->depth() == 2 )
  {
    QString workgroup = currentItem()->parent()->parent()->text( Network );
    QString host = currentItem()->parent()->text( Network );
    QString share = currentItem()->text( Network );

    smb4k_core->passwordHandler()->askpass( workgroup, host, share, Smb4KPasswordHandler::NewData );
  }
}


void Smb4KBrowserWidget::slotPreview()
{
  QString workgroup = currentItem()->parent()->parent()->text( Network );
  QString host = currentItem()->parent()->text( Network );
  QString ip = currentItem()->parent()->text( IP );
  QString share;

  if ( QString::compare( currentItem()->text( Network ).stripWhiteSpace(), "homes" ) == 0 )
  {
    share = specifyUser( currentItem()->parent()->text( Network ) );
  }
  else
  {
    share = currentItem()->text( Network );
  }

  Smb4KPreviewDialog *preview = new Smb4KPreviewDialog( workgroup, host, ip, share, this, 0, false );
  preview->show();
}


void Smb4KBrowserWidget::slotAddBookmark()
{
  if ( currentItem() && currentItem()->depth() == 2 )
  {
    if ( QString::compare( currentItem()->text( Network ).stripWhiteSpace(), "homes" ) != 0 )
    {
      Smb4KBookmark *bookmark = new Smb4KBookmark( QString( "//%1/%2" ).arg( currentItem()->parent()->text( Network ), currentItem()->text( Network ) ), currentItem()->text( Type ) );
      smb4k_core->bookmarkHandler()->addBookmark( bookmark );
    }
    else
    {
      QString share = specifyUser( currentItem()->parent()->text( Network ) );
      Smb4KBookmark *bookmark = new Smb4KBookmark( QString( "//%1/%2" ).arg( currentItem()->parent()->text( Network ), share ), currentItem()->text( Type ) );
      smb4k_core->bookmarkHandler()->addBookmark( bookmark );
    }
  }
}


void Smb4KBrowserWidget::slotRescan()
{
  smb4k_core->scanner()->rescan();
}


void Smb4KBrowserWidget::slotAddIPAddress( Smb4KHostItem *item )
{
  // First find the host and check wether it is the right one.
  QListViewItem *hostItem = findItem( item->name(), Network, ExactMatch|CaseSensitive );

  // FIXME: This might be a potential problem, if there are several hosts
  // with the same name in a domain/LAN.
  // *** Question: Is this case at all possible? ***
  if ( hostItem && hostItem->parent() && QString::compare( hostItem->parent()->text( Network ), item->workgroup() ) == 0 )
  {
    hostItem->setText( IP, item->ip() );

    if ( this->columnWidth( IP ) != 0 )
    {
      adjustColumn( IP );
    }
  }
}


void Smb4KBrowserWidget::slotPrint()
{
  if ( currentItem() && currentItem()->depth() == 2 )
  {
    Smb4KPrintDialog *print = new Smb4KPrintDialog( currentItem()->parent()->parent()->text( Network ), currentItem()->parent()->text( Network ), currentItem()->parent()->text( IP ), currentItem()->text( Network ), currentItem()->text( Comment ), this, "PrintDialog" );
    print->exec();
  }
}


void Smb4KBrowserWidget::slotMarkShares()
{
  config()->setGroup( "Appearance" );
  bool show_all = config()->readBoolEntry( "Show All Shares", false );

  QListViewItemIterator it( this );
  QListViewItem *item;

  while ((item = it.current()) != 0 )
  {
    ++it;

    if ( item->depth() == 2 )
    {
      Smb4KShare *share = smb4k_core->mounter()->findShareByName( QString( "//%1/%2" ).arg( item->parent()->text( Network ), item->text( Network ) ) );

      if ( smb4k_core->mounter()->isMounted( QString( "//%1/%2" ).arg( item->parent()->text( Network ), item->text( Network ) ) ) &&
           (!share->isForeign() || show_all) )
      {
        if ( !((Smb4KBrowserWidgetItem *)item)->isMounted() )
        {
          ((Smb4KBrowserWidgetItem *)item)->setMounted( true );
        }

        continue;
      }
      else
      {
        ((Smb4KBrowserWidgetItem *)item)->setMounted( false );
        continue;
      }
    }
    else
    {
      continue;
    }
  }
}


void Smb4KBrowserWidget::slotSearch()
{
  emit searchRequest();
}


void Smb4KBrowserWidget::slotInsertItem( Smb4KHostItem *item )
{
  insertItem( item );
}


void Smb4KBrowserWidget::slotAbort()
{
  smb4k_core->abort();
}

#include "smb4kbrowserwidget.moc"
