/***************************************************************************
    smb4k.cpp  -  The main class of Smb4K.
                             -------------------
    begin                : Sam Mr  1 14:57:21 CET 2003
    copyright            : (C) 2003 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// Qt includes
#include <qtooltip.h>
#include <qmap.h>

// KDE includes
#include <kmenubar.h>
#include <kstatusbar.h>
#include <klocale.h>
#include <kstdaction.h>
#include <kmessagebox.h>
#include <kpopupmenu.h>
#include <kiconloader.h>

// application specific includes
#include "smb4k.h"
#include "widgets/smb4kbookmarkmenuwidget.h"
#include "widgets/smb4koptionsdlg.h"
#include "widgets/smb4kbookmarkeditor.h"
#include "core/smb4kdefs.h"
#include "core/smb4kglobal.h"

using namespace Smb4K_Global;

#define ID_STATUS_MSG 1


Smb4KApp::Smb4KApp( QWidget* , const char* name ) : KDockMainWindow( 0, name )
{
  m_config = kapp->config();
  m_startDocked = false;

  // The caption.
  setPlainCaption( i18n( "Smb4K - The SMB/CIFS Share Browser" ) );

  ///////////////////////////////////////////////////////////////////
  // call inits to invoke all other construction parts
  initCore();
  initStatusBar();
  initView();
  initActions();
  setupMenuBar();
  setupSystemTray();

  readOptions();
}


Smb4KApp::~Smb4KApp()
{
}


void Smb4KApp::initActions()
{
  // Add the action collections of the widgets to the main action collection.

  quitApp = KStdAction::quit( this, SLOT( slotQuit() ), actionCollection(), "quit_smb4k" );

  viewToolBar = KStdAction::showToolbar(this, SLOT(slotViewToolBar()), actionCollection());
  viewStatusBar = KStdAction::showStatusbar(this, SLOT(slotViewStatusBar()), actionCollection());

  settingsSmb4KOptions = KStdAction::preferences( this, SLOT( slotSmb4KOptionsDlg() ), actionCollection(), "configure_smb4k" );

  actionCollection()->insert( m_network_view->action_collection()->action( "abort" ) );
  abort = m_network_view->action_collection()->action( "abort" );

  actionCollection()->insert( m_network_view->action_collection()->action( "network_rescan" ) );
  networkRescan = m_network_view->action_collection()->action( "network_rescan" );

  actionCollection()->insert( m_network_view->action_collection()->action("network_search" ) );
  networkSearch = m_network_view->action_collection()->action("network_search" );

  actionCollection()->insert( m_network_view->action_collection()->action( "network_preview" ) );
  networkPreview = m_network_view->action_collection()->action( "network_preview" );

  actionCollection()->insert( m_network_view->action_collection()->action( "network_askpass" ) );
  networkAskPass = m_network_view->action_collection()->action( "network_askpass" );

  actionCollection()->insert( m_network_view->action_collection()->action( "network_print" ) );
  networkPrint = m_network_view->action_collection()->action( "network_print" );

  actionCollection()->insert( m_network_view->action_collection()->action( "network_mount" ) );
  networkMount = m_network_view->action_collection()->action( "network_mount" );

  actionCollection()->insert( m_network_view->action_collection()->action( "network_add_bookmark" ) );
  networkAddBookmark = m_network_view->action_collection()->action( "network_add_bookmark" );

  actionCollection()->insert( m_share_view->action_collection()->action( "shares_unmount_share" ) );
  sharesUnmount = m_share_view->action_collection()->action( "shares_unmount_share" );

  actionCollection()->insert( m_share_view->action_collection()->action( "shares_force_unmount" ) );
  sharesForceUnmount = m_share_view->action_collection()->action( "shares_force_unmount" );

  actionCollection()->insert( m_share_view->action_collection()->action( "shares_unmount_all" ) );
  sharesUnmountAll = m_share_view->action_collection()->action( "shares_unmount_all" );

  actionCollection()->insert( m_share_view->action_collection()->action( "shares_manual_mount" ) );
  sharesManualMount = m_share_view->action_collection()->action( "shares_manual_mount" );

  actionCollection()->insert( m_share_view->action_collection()->action( "shares_start_filemanager" ) );
  sharesStartFilemanager = m_share_view->action_collection()->action( "shares_start_filemanager" );

  // use the absolute path to your smb4kui.rc file for testing purpose in createGUI();
  createGUI();
}


void Smb4KApp::initStatusBar()
{
  statusBar()->insertItem( i18n( "Ready." ), ID_STATUS_MSG, 1, false );
  statusBar()->setItemAlignment( ID_STATUS_MSG, AlignLeft );

  // Labels:
  m_winsLabel = new QLabel( statusBar() );
  m_winsLabel->setAlignment( Qt::AlignCenter );
  m_listLabel = new QLabel( statusBar() );
  m_listLabel->setAlignment( Qt::AlignCenter );
  QLabel *version = new QLabel( QString( "Smb4K %1" ).arg( VERSION ), statusBar() );
  version->setAlignment( Qt::AlignCenter );
  version->setFixedWidth( version->sizeHint().width() + 5 );

  m_progress = new KProgress( statusBar(), 0, 0 );
  m_progress->setFixedWidth( 75 );
  m_progress->setFixedHeight( version->fontMetrics().height() );
  m_progress->setPercentageVisible( false );

  m_progTimer = new QTimer( this );
  connect( m_progTimer, SIGNAL( timeout() ), SLOT( slotSetProgress() ) );

  statusBar()->addWidget( m_winsLabel, false );
  statusBar()->addWidget( m_listLabel, false );
  statusBar()->addWidget( m_progress, false );
  statusBar()->addWidget( version, false );

  QToolTip::add( m_winsLabel, i18n( "WINS server" ) );
  QToolTip::add( m_listLabel, i18n( "Browse list retrieval" ) );
  QToolTip::add( version, i18n( "Version" ) );
}


void Smb4KApp::initCore()
{
  connect( smb4k_core, SIGNAL( runStateChanged() ), this, SLOT( slotCoreRunStateChanged() ) );
}


void Smb4KApp::initView()
{
  KDockWidget *main = createDockWidget( "ShareView", SmallIcon( "hdd_mount" ), 0L );
  m_share_view = new Smb4KShareWidget( main );
  main->setWidget( m_share_view );
  main->setDockSite( KDockWidget::DockCorner );
  main->setEnableDocking( KDockWidget::DockNone );
  setView( main );
  setMainDockWidget( main );

  KDockWidget *network = createDockWidget( "NetworkView", SmallIcon( "network" ), 0L, i18n( "Network Browser" ) );
  m_network_view = new Smb4KBrowserWidget( network );
  network->setWidget( m_network_view );
  network->manualDock( main, KDockWidget::DockLeft, 30 );

  KDockWidget *tabs = createDockWidget( "TabsView", SmallIcon( "tab_new_raised" ), 0L, i18n( "Tab Widget" ) );
  KDockTabGroup *info = new KDockTabGroup( tabs, "TabGroup" );
  tabs->setWidget( info );
  tabs->manualDock( main, KDockWidget::DockBottom, 80 );

  m_network_tab = new Smb4KNetworkTab( info );
  info->insertTab( m_network_tab, QIconSet( SmallIcon( "network" ), QIconSet::Automatic ), i18n( "Network" ), 0 );

  m_share_tab = new Smb4KShareTab( info );
  info->insertTab( m_share_tab, QIconSet( SmallIcon( "hdd_mount" ), QIconSet::Automatic ), i18n( "Share" ), 1 );

  m_search_tab = new Smb4KSearchTab( info );
  info->insertTab( m_search_tab, QIconSet( SmallIcon( "find" ), QIconSet::Automatic ), i18n( "Search" ), 2 );

  connect( m_network_view, SIGNAL( searchRequest() ), this,                             SLOT( slotShowSearchTab() ) );
  connect( m_network_view, SIGNAL( highlightedItem( QListViewItem * ) ), m_network_tab, SLOT( slotGetListViewItem( QListViewItem * ) ) );
  connect( m_network_view, SIGNAL( cleared() ), m_network_tab,                          SLOT( slotClearWidget() ) );
  connect( m_share_view,   SIGNAL( cleared() ), m_share_tab,                            SLOT( slotClearWidget() ) );
  connect( m_share_view,   SIGNAL( clearedSelection() ), m_share_tab,                   SLOT( slotClearWidget() ) );
  connect( m_search_tab,   SIGNAL( searchResult(Smb4KHostItem* ) ), m_network_view,     SLOT( slotInsertItem(Smb4KHostItem* ) ) );

  accel()->insert( i18n( "Show network tab" ), CTRL+Key_1, this, SLOT( slotShowNetworkTab() ), false, true );
  accel()->insert( i18n( "Show Share tab" ),   CTRL+Key_2, this, SLOT( slotShowShareTab() ),   false, true );
  accel()->insert( i18n( "Show Search tab" ),  CTRL+Key_3, this, SLOT( slotShowSearchTab() ),  false, true );
}


void Smb4KApp::setupMenuBar()
{
  Smb4KBookmarkMenuWidget *bookmark_menu = new Smb4KBookmarkMenuWidget( this, "BookmarkMenu" );

  networkAddBookmark->plug( bookmark_menu->popupMenu(), 0 );

  menuBar()->insertItem( i18n( "&Bookmarks" ), bookmark_menu->popupMenu(), 3, 3 );

  ((QPopupMenu *)(this->child( "settings" ) ))->insertItem( i18n( "&Dock Widgets" ), manager()->dockHideShowMenu(), 2, 2 );
}


void Smb4KApp::setupSystemTray()
{
  sys_tray = new Smb4KSystemTray( true, this, "SystemTray" );
}


void Smb4KApp::saveOptions()
{
  m_config->setGroup("General Options");
  m_config->writeEntry("Show Toolbar", viewToolBar->isChecked());
  m_config->writeEntry("Show Statusbar",viewStatusBar->isChecked());
  m_config->writeEntry("Toolbar Position", (int)toolBar("mainToolBar")->barPos());
  m_config->writeEntry("Toolbar Text Position", toolBar("mainToolBar")->iconText());
  m_config->writeEntry("Toolbar Icon Size", toolBar("mainToolBar")->iconSize());

  writeDockConfig( m_config, "General Options" );

  KConfigGroup settingsConfig( KGlobal::config(), "Settings" );
  settingsConfig.writeEntry( "StartDocked", m_startDocked );
}


void Smb4KApp::readOptions()
{
  m_config->setGroup("General Options");

  // bar status settings
  bool bViewToolbar = m_config->readBoolEntry("Show Toolbar", true);
  viewToolBar->setChecked(bViewToolbar);
  slotViewToolBar();

  bool bViewStatusbar = m_config->readBoolEntry("Show Statusbar", true);
  viewStatusBar->setChecked(bViewStatusbar);
  slotViewStatusBar();


  // bar position settings
  KToolBar::BarPosition toolBarPos;
  toolBarPos=(KToolBar::BarPosition) m_config->readNumEntry("Toolbar Position", KToolBar::Top);
  toolBar("mainToolBar")->setBarPos(toolBarPos);

  KToolBar::IconText toolBarIconText;
  toolBarIconText=(KToolBar::IconText) m_config->readNumEntry("Toolbar Text Position", KToolBar::IconOnly);
  toolBar("mainToolBar")->setIconText(toolBarIconText);
  toolBar("mainToolBar")->setIconSize( m_config->readNumEntry( "Toolbar Icon Size", 0 ) );

  readDockConfig( m_config, "General Options" );

  // Set-up the status bar.
  m_config->setGroup( "Browse Options" );
  
  if ( QString::compare( m_config->readEntry( "Browse List", "nmblookup" ), "nmblookup" ) == 0 )
  {
    m_listLabel->setText( i18n( "Network" ) );
  }
  else
  {
    m_listLabel->setText( m_config->readEntry( "Query Host" ) );
  }
    
  QString wins = getWINSServer();

  if ( QString::compare( wins, QString::null ) != 0 )
  {
    m_winsLabel->setText( wins );
  }
  else
  {
    m_winsLabel->setText( i18n( "None" ) );
  }

  m_listLabel->setFixedWidth( m_listLabel->sizeHint().width() + 5 );
  m_winsLabel->setFixedWidth( m_winsLabel->sizeHint().width() + 5 );

  // EXTERNAL configuration options:
  m_network_view->readOptions();
  m_share_view->readOptions();
}


bool Smb4KApp::queryExit()
{
  m_startDocked = !this->isVisible();
  hide();

  if ( sys_tray )
  {
    delete sys_tray;
    sys_tray = NULL;
  }

  saveOptions();

  return true;
}


bool Smb4KApp::queryClose()
{
  if ( !kapp->sessionSaving() && this->isVisible() && sys_tray->isDocked() )
  {
    // This part has been 'stolen' from JuK application.
    KMessageBox::information(this,
            i18n( "Closing the main window will keep Smb4K running in the system tray. "
                 "Use 'Quit' from the 'File' menu to quit the application."),
            i18n("Docking in System Tray"), "Hide On Close Info" );
    hide();
    return false;

  }
  else
  {
    return true;
  }
}


/////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////

void Smb4KApp::slotQuit()
{
  slotStatusMsg(i18n("Exiting..."));

  if ( sys_tray->isDocked() )
  {
    sys_tray->undock();
  }

  KApplication::exit( 0 );
}


void Smb4KApp::slotAbort()
{
  slotStatusMsg( i18n( "Aborting..." ) );
  smb4k_core->abort();
  slotStatusMsg( i18n( "Done." ) );
}


void Smb4KApp::slotCoreRunStateChanged()
{
  int mode = smb4k_core->currentState();

  if ( smb4k_core->isRunning() )
  {
    abort->setEnabled( true );

    if ( smb4k_core->scannerIsRunning() )
      networkRescan->setEnabled( false );
    else
      networkRescan->setEnabled( true );

    switch ( mode )
    {
      case SCANNER_INIT:
        slotStatusMsg( i18n( "Searching network..." ) );
        break;
      case SCANNER_OPENING_WORKGROUP:
        slotStatusMsg( i18n( "Opening workgroup..." ) );
        break;
      case SCANNER_OPENING_HOST:
        slotStatusMsg( i18n( "Opening host..." ) );
        break;
      case SCANNER_RETRIEVING_INFO:
        slotStatusMsg( i18n( "Retrieving information..." ) );
        break;
      case SCANNER_RETRIEVING_PREVIEW:
        slotStatusMsg( i18n( "Retrieving preview..." ) );
        break;
      case SCANNER_SEARCHING:
        slotStatusMsg( i18n( "Searching for host..." ) );
        break;
      case IPFINDER_WORKING:
        slotStatusMsg( i18n( "Looking for IP addresses..." ) );
        break;
      case MOUNTER_MOUNTING:
        slotStatusMsg( i18n( "Mounting share..." ) );
        break;
      case MOUNTER_UNMOUNTING:
        slotStatusMsg( i18n( "Unmounting share..." ) );
        break;
      case MOUNTER_UNMOUNTING_ALL:
        slotStatusMsg( i18n( "Unmounting all shares..." ) );
        break;
      case PRINT_SEND_FILE:
        slotStatusMsg( i18n( "Sending file to printer..." ) );
        break;
      case PRINT_CONVERT_FILE:
        slotStatusMsg( i18n( "Converting file to Postscript..." ) );
        break;
      default:
        break;
    }

    if ( !m_progTimer->isActive() )
    {
      // Start the busy progress bar:
      m_progress->setTotalSteps( 0 );
      m_progTimer->start( 25, false );
    }
  }
  else
  {
    m_progTimer->stop();
    m_progress->setTotalSteps( 100 );
    m_progress->reset();
    networkRescan->setEnabled( true );
    abort->setEnabled( false );
    slotStatusMsg( i18n( "Done." ) );
  }
}


void Smb4KApp::slotSmb4KOptionsDlg()
{
  if ( child( "ConfigDialog", "KDialogBase", true ) == 0 )
  {
    Smb4KOptionsDlg *configDlg = new Smb4KOptionsDlg( this, "ConfigDialog", false );
    connect( configDlg, SIGNAL( read() ), this, SLOT( slotReadOptions() ) );
    configDlg->show();
  }
}



void Smb4KApp::slotViewToolBar()
{
  ///////////////////////////////////////////////////////////////////
  // turn Toolbar on or off
  if(!viewToolBar->isChecked())
  {
    toolBar("mainToolBar")->hide();
  }
  else
  {
    toolBar("mainToolBar")->show();
  }
}


void Smb4KApp::slotViewStatusBar()
{
  ///////////////////////////////////////////////////////////////////
  //turn Statusbar on or off
  if(!viewStatusBar->isChecked())
  {
    statusBar()->hide();
  }
  else
  {
    statusBar()->show();
  }
}


void Smb4KApp::slotStatusMsg(const QString &text)
{
  ///////////////////////////////////////////////////////////////////
  // change status message permanently
  statusBar()->clear();
  statusBar()->changeItem(text, ID_STATUS_MSG);
}


void Smb4KApp::slotSetProgress()
{
  m_progress->setProgress( m_progress->progress()+1 );
}


void Smb4KApp::slotReadOptions()
{
  m_network_view->readOptions();
  m_share_view->readOptions();

  // Set-up the status bar.
  m_config->setGroup( "Browse Options" );
  
  if ( QString::compare( m_config->readEntry( "Browse List", "nmblookup" ), "nmblookup" ) == 0 )
  {
    m_listLabel->setText( i18n( "Network" ) );
  }
  else
  {
    m_listLabel->setText( m_config->readEntry( "Query Host" ) );
  }
  
  QString wins = getWINSServer();

  if ( QString::compare( wins, QString::null ) != 0 )
  {
    m_winsLabel->setText( wins );
  }
  else
  {
    m_winsLabel->setText( i18n( "None" ) );
  }

  m_listLabel->setFixedWidth( m_listLabel->sizeHint().width() + 5 );
  m_winsLabel->setFixedWidth( m_winsLabel->sizeHint().width() + 5 );
}


void Smb4KApp::slotShowNetworkTab()
{
  if ( (KDockTabGroup *)child( "TabGroup", "KDockTabGroup", true ) && ((KDockTabGroup *)child( "TabGroup", "KDockTabGroup", true ))->isShown() )
  {
    ((KDockTabGroup *)child( "TabGroup", "KDockTabGroup", true ))->showPage( m_network_tab );
  }
}


void Smb4KApp::slotShowShareTab()
{
  if ( (KDockTabGroup *)child( "TabGroup", "KDockTabGroup", true ) && ((KDockTabGroup *)child( "TabGroup", "KDockTabGroup", true ))->isShown() )
  {
    ((KDockTabGroup *)child( "TabGroup", "KDockTabGroup", true ))->showPage( m_share_tab );
  }
}


void Smb4KApp::slotShowSearchTab()
{
  if ( (KDockTabGroup *)child( "TabGroup", "KDockTabGroup", true ) && ((KDockTabGroup *)child( "TabGroup", "KDockTabGroup", true ))->isShown() )
  {
    ((KDockTabGroup *)child( "TabGroup", "KDockTabGroup", true ))->showPage( m_search_tab );
  }
}

#include "smb4k.moc"
