/***************************************************************************
    smb4kpasswordhandler  -  This class handles the passwords for Smb4K.
                             -------------------
    begin                : So Jan 16 2005
    copyright            : (C) 2005 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef SMB4KPASSWORDHANDLER_H
#define SMB4KPASSWORDHANDLER_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// Qt includes
#include <qobject.h>
#include <qcheckbox.h>

// KDE includes
#include <kwallet.h>
#include <kdialogbase.h>
#include <kconfig.h>
#include <klineedit.h>
#include <kcombobox.h>
#include <kprocess.h>

// application specific includes
#include "smb4kauthinfo.h"


/**
 * This class handles the passwords used by Smb4K to authenticate to
 * the network shares.
 *
 * @author                 Alexander Reinholdt <dustpuppy@mail.berlios.de>
 */

class Smb4KPasswordHandler : public QObject
{
  Q_OBJECT

  public:
    /**
     * The constructor.
     *
     * @param parent        The parent object.
     *
     * @param name          The name this class should carry.
     */
    Smb4KPasswordHandler( QObject *parent = 0, const char *name = 0 );
    /**
     * The destructor.
     */
    ~Smb4KPasswordHandler();
    /**
     * Opens the askpass dialog.
     *
     * @param workgroup     The workgroup where the share is located in (may be empty).
     *
     * @param host          The host the share belongs to.
     *
     * @param name          The name of the share.
     *
     * @param desc          Determines which descriptive label the askpass dialog should carry.
     *                      Values are taken from the Smb4KPasswordHandler::AskPass::Description
     *                      enumeration.
     *
     * @returns             TRUE if a new password has been supplied and FALSE otherwise.
     */
    bool askpass( const QString &workgroup, const QString &host, const QString &share, int desc );
    /**
     * Returns the authentication data that was found for the defined share.
     *
     * @param workgroup     The workgroup the share is located in (may be empty).
     *
     * @param host          The name of the host the share belongs to.
     *
     * @param share         The name of the share.
     *
     * @returns             The authentication data for the share.
     */
    Smb4KAuthInfo *readAuth( const QString &workgroup, const QString &host, const QString &share );
    /**
     * Writes the provided authentication data to the wallet.
     *
     * @param auth          The Smb4KAuthInfo object that holds the authentication information.
     */
    void writeAuth( Smb4KAuthInfo *auth );
    /**
     * Reads the default authentication from the wallet. This function seems only to
     * be useful for the configuration dialog.
     *
     * @returns             pointer to the authentication data object if a wallet is used or NULL otherwise
     */
    Smb4KAuthInfo *readDefaultAuth();
    /**
     * Writes the default authentication data to the wallet. This function seems only to
     * be useful for the configuration dialog.
     *
     * @param auth          The authentication data for the default login if a wallet is
     *                      used or NULL otherwise.
     */
    void writeDefaultAuth( Smb4KAuthInfo *auth );
    /**
     * This enumeration destermines which destriptive label the askpass
     * dialog should carry.
     */
    enum Description{ NewData, AccessDenied, BadPassword, PermDenied, AuthError, LogonFailure, None };
    /**
     * This function returns a static pointer to this class.
     *
     * @returns             A static pointer to this class
     */
    static Smb4KPasswordHandler *password_handler() { return m_this_class; }

  signals:
    /**
     * Emits an error code and error message, when something went wrong.
     *
     * @param error_code    The error code like defined in smb4kdefs.h.
     *
     * @param error_message The error message.
     */
    void error( int error_code, const QString &error_message );


  protected slots:
    /**
     * Get the password for a specific share and user name.
     */
    void slotGetPassword( const QString &username );
    /**
     * FreeBSD specific: This slot receives output from the process that encrypts
     * the passwords.
     *
     * @param proc          The process that sends the output
     *
     * @param buffer        The buffer that contains the output
     *
     * @param buflen        The length of the buffer
     */
    void slotReceivePassword( KProcess *proc, char *buffer, int buflen );
    /**
     * FreeBSD specific: This slot is invoked if the process exited.
     *
     * @param proc          The process that exited
     */
    void slotWritePassword( KProcess *proc );

  private:
    /**
     * This function opens the wallet in which the passwords should be
     * stored.
     */
    void open_close_wallet();
    /**
     * A pointer to this class.
     */
    static Smb4KPasswordHandler *m_this_class;
    /**
     * The KWallet object.
     */
    KWallet::Wallet *m_wallet;
    /**
     * The Smb4KAuthInfo object. For use with the askpass dialog.
     */
    Smb4KAuthInfo *m_auth;
    /**
     * The ask pass dialog class. It provides only the widget, the functionality
     * will be given to it by the askpass() funtion.
     */
    class AskPass : public KDialogBase
    {
      public:
        /**
         * The constructor.
         *
         * @param auth      The authentication information object that should be filled by this ask
         *                  pass dialog.
         *
         * @param desc      Determines which descriptive label the askpass dialog should carry.
         *                  Values are taken from the Smb4KPasswordHandler::Description
         *                  enumeration.
         *
         * @param parent    The parent widget.
         *
         * @param name      The name the class should carry.
         *
         */
        AskPass( Smb4KAuthInfo *auth, int desc, QWidget *name = 0, const char *name = 0 );
        /**
         * The destructor.
         */
        ~AskPass();
        /**
         * This function returns a pointer to the line edit that holds the user name.
         *
         * @returns         The pointer to the line edit that holds the user name.
         */
        KLineEdit *userEdit() { return m_user_edit; }
        /**
         * This function returns a pointer to the line edit that holds the password.
         *
         * @returns         The pointer to the line edit that holds the password.
         */
        KLineEdit *passwdEdit() { return m_pass_edit; }
        /**
         * This function returns a pointer to the combo box that holds the user names
         * if the share is a "homes" share.
         *
         * @returns         The pointer to the combo box that holds the user names.
         */
        KComboBox *userCombo() { return m_user_combo; }

      private:
        /**
         * The KLineEdit which holds the user name.
         */
        KLineEdit *m_user_edit;
        /**
         * The KLineEdit which holds the password.
         */
        KLineEdit *m_pass_edit;
        /**
         * The KComboBox which holds the user names when the share is
         * a "homes" share.
         */
        KComboBox *m_user_combo;
    };
    /**
     * The askpass dialog.
     */
    AskPass::AskPass *m_dlg;
    /**
     * This funtion imports the authentication data from the old password file
     * into the wallet.
     */
    void import();
    /**
     * This list holds the authentication data the user supplied if he/she
     * does not want to use KWallet.
     */
    QValueList<Smb4KAuthInfo *> m_auth_list;
    /**
     * If the user neither wants to store the passwords in a wallet nor in a
     * temporary list, the authentication data received from the askpass dialog
     * will the stored in this object.
     */
    Smb4KAuthInfo *m_temp_auth;
#ifdef __FreeBSD__
    /**
     * FreeBSD specific: This function writes authentication information
     * to the ~/.nsmbrc file, which is used by mount_smbfs.
     *
     * @param auth          The authentication information that should be written
     *                      to ~/.nsmbrc file.
     */
    void writeToSMBConfFile( Smb4KAuthInfo *auth );
    /**
     * FreeBSD specific: Holds the authentication information that should
     * be written to ~/.nsmbrc.
     */
    Smb4KAuthInfo m_nsmbrc_auth;
    /**
     * The buffer for the process that encrypts the password.
     */
    QString m_buffer;
#endif
};

#endif
