/***************************************************************************
    smb4kfileio  -  Does file IO operations for Smb4K
                             -------------------
    begin                : Do Jan 1 2004
    copyright            : (C) 2004 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


// Qt includes
#include <qfile.h>
#include <qdir.h>
#include <qtextstream.h>
#include <qfileinfo.h>

// KDE includes
#include <kmessagebox.h>
#include <klocale.h>
#include <kapplication.h>

// system includes
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>

// application specific includes
#include "smb4kfileio.h"
#include "smb4kdefs.h"
#include "smb4kglobal.h"

using namespace Smb4K_Global;

// static varibles
static Smb4KFileIO::FileItem sudoers_item;
static QFileInfo tmp_sudoers_info;
static QFile lock_file;
static bool shutdown = false;

// create a unique temporary directory.
static int error_number;
static bool failed = false;

char *createtmpdir( char *dirname )
{
  char tmpd_name[] = "/tmp/smb4k.XXXXXX";

  if ( mkdtemp( tmpd_name ) == NULL )
  {
    error_number = errno;
    failed = true;
  }

  return strcpy( dirname, tmpd_name );
}



Smb4KFileIO::Smb4KFileIO( QObject *parent, const char *name ) : QObject( parent, name )
{
  m_proc = new KProcess( this, "FileIOProcess" );
  m_proc->setUseShell( true );

  char tmp_dir[18];
  m_temp_dir = QDir( createtmpdir( tmp_dir ) );

  m_state = Idle;

  connect( m_proc, SIGNAL( receivedStderr( KProcess *, char *, int ) ),
           this,   SLOT( slotReceivedStderr( KProcess *, char *, int ) ) );
  connect( m_proc, SIGNAL( processExited( KProcess * ) ),
           this,   SLOT( slotProcessExited( KProcess * ) ) );
  connect( kapp,   SIGNAL( shutDown() ),
           this,   SLOT( slotShutdown() ) );
}


Smb4KFileIO::~Smb4KFileIO()
{
}


/****************************************************************************
   TEMPORARY FUNCTION: Converts the super.tab file to the new style.
****************************************************************************/

void Smb4KFileIO::convertSuperTab()
{
  FileItem item = readConfigFile( "super.tab" );

  if ( write_lock_file( item.path() ) )
  {
    config()->setGroup( "Super User Privileges" );
    bool run_suid = config()->readBoolEntry( "Run SUID", false );
    bool force_umount = config()->readBoolEntry( "Force Unmount", false );

    if ( run_suid || force_umount )
    {
      if ( item.exists() )
      {
        QStringList contents = item.contents();

        if ( contents.grep( "# Entries for Smb4K users." ).count() != 0 && contents.grep( ":define Smb4KUsers" ).count() == 0 )
        {
          KMessageBox::information( new QWidget(), i18n( "As of version 0.5.0, the entries in the super.tab file are stored in a different format. Smb4K is now going to convert the old ones." ) );

          QStringList::Iterator insert_point;
          QString users;
          bool ok = false;
          int entries = 0;
          config()->setGroup( "Programs" );

          for ( QStringList::Iterator it = contents.begin(); it != contents.end(); ++it )
          {
            if ( (*it).startsWith( "# Entries for Smb4K users." ) )
            {
              ok = true;
              continue;
            }
            else if ( ok && ((*it).startsWith( "kill" ) ) )
            {
              (*it).replace( (*it).section( "\t", 1, 1 ).stripWhiteSpace(), config()->readPathEntry( "smb4k_kill" ), true );
              (*it).replace( (*it).section( "\t", 2, 2 ).stripWhiteSpace(), "$(Smb4KUsers)", true );
              QString tmp = "smb4k_kill\t";
              *it = tmp.append( (*it).section( "\t", 1, -1 ).stripWhiteSpace() );
              insert_point = it;
              entries++;
              continue;
            }
            else if ( ok && (*it).startsWith( "umount" ) )
            {
              (*it).replace( (*it).section( "\t", 1, 1 ).stripWhiteSpace(), config()->readPathEntry( "smb4k_umount" ), true );
              users = (*it).section( "\t", 2, 2 ).stripWhiteSpace();
              (*it).replace( users, "$(Smb4KUsers)", true );
              QString tmp = "smb4k_umount\t";
              *it = tmp.append( (*it).section( "\t", 1, -1 ).stripWhiteSpace() );
              entries++;
              continue;
            }
            else if ( ok && (*it).startsWith( "mount" ) )
            {
              (*it).replace( (*it).section( "\t", 1, 1 ).stripWhiteSpace(), config()->readPathEntry( "smb4k_mount" ), true );
              (*it).replace( (*it).section( "\t", 2, 2 ).stripWhiteSpace(), "$(Smb4KUsers)", true );
              QString tmp = "smb4k_mount\t";
              *it = tmp.append( (*it).section( "\t", 1, -1 ).stripWhiteSpace()+"\tenv=PASSWD,USER" );
              entries++;
              continue;
            }
            else if ( (*it).startsWith( "# End of Smb4K user entries." ) )
            {
              if ( entries < 3 )
              {
#ifndef __FreeBSD__
                contents.insert( it, "smb4k_mount\t"+config()->readPathEntry( "smb4k_mount" )+"\t$(Smb4KUsers)\tuid=root\tgid=root\tenv=PASSWD,USER" );
#else
                contents.insert( it, "smb4k_mount\t"+config()->readPathEntry( "smb4k_mount" )+"\t$(Smb4KUsers)\tuid=root\tgid=wheel\tsetenv=HOME=$CALLER_HOME\tenv=PASSWD,USER" );
#endif
              }

              break;
            }
            else
            {
              continue;
            }
          }

          contents.insert( insert_point, ":define Smb4KUsers "+users );

          item.replaceContents( contents );

          writeFile( &item );
        }
        else if ( contents.grep( "# Entries for Smb4K users." ).count() != 0 && contents.grep( ":define Smb4KUsers" ).count() != 0
                  && contents.grep( "smb4k_mount" ).first().contains( "env" ) == 0 )
        {
          KMessageBox::information( new QWidget(), i18n( "Smb4K is going to modify the entries in the super.tab file in order to work properly." ) );

          for ( QStringList::Iterator it = contents.begin(); it != contents.end(); ++it )
          {
            if ( (*it).startsWith( "smb4k_mount" ) )
            {
              (*it).append( "\tenv=PASSWD,USER" );
              break;
            }
          }

          item.replaceContents( contents );

          writeFile( &item );
        }
      }
      else
      {
        emit error( ERROR_FILE_NOT_FOUND, "super.tab" );
      }
    }

    remove_lock_file();
  }
}


/****************************************************************************
   Writes entries to the super.tab file.
****************************************************************************/

bool Smb4KFileIO::writeSuperUserEntries( const QString &program )
{
  FileItem item;

  if ( QString::compare( program, "super" ) == 0 )
  {
    item = readConfigFile( "super.tab" );
  }
  else if ( QString::compare( program, "sudo" ) == 0 )
  {
    item = readConfigFile( "sudoers", false, true );
  }

  if ( item.exists() && write_lock_file( item.path() ) )
  {
    config()->setGroup( "Programs" );

    QStringList contents;
    bool write = true;

    if ( QString::compare( program, "super" ) == 0 )
    {
      QStringList contents = item.contents();

      if ( contents.grep( "# Entries for Smb4K users." ).count() == 0 )
      {
        // Create the new entries:
        contents.append( "# Entries for Smb4K users." );
        contents.append( "# Generated by Smb4K. Please do not modify!" );
        contents.append( ":define Smb4KUsers "+QString( "%1" ).arg( getenv( "USER" ) ) );
#ifndef __FreeBSD__
        contents.append( "smb4k_kill\t"+config()->readPathEntry( "smb4k_kill" )+"\t$(Smb4KUsers)\tuid=root\tgid=root" );
        contents.append( "smb4k_umount\t"+config()->readPathEntry( "smb4k_umount" )+"\t$(Smb4KUsers)\tuid=root\tgid=root" );
        contents.append( "smb4k_mount\t"+config()->readPathEntry( "smb4k_mount" )+"\t$(Smb4KUsers)\tuid=root\tgid=root\tenv=PASSWD,USER" );
#else
        contents.append( "smb4k_kill\t"+config()->readPathEntry( "smb4k_kill" )+"\t$(Smb4KUsers)\tuid=root\tgid=wheel" );
        contents.append( "smb4k_umount\t"+config()->readPathEntry( "smb4k_umount" )+"\t$(Smb4KUsers)\tuid=root\tgid=wheel" );
        contents.append( "smb4k_mount\t"+config()->readPathEntry( "smb4k_mount" )+"\t$(Smb4KUsers)\tuid=root\tgid=wheel\tsetenv=HOME=$CALLER_HOME\tenv=PASSWD,USER" );
#endif
        contents.append( "# End of Smb4K user entries." );
      }
      else
      {
        if ( contents.grep( ":define Smb4KUsers" ).count() != 0 )
        {
          // Append the user name, if necessary.
          for ( QStringList::Iterator it = contents.begin(); it != contents.end(); ++it )
          {
            if ( (*it).contains( ":define Smb4KUsers" ) != 0 )
            {
              QString users = (*it).section( "Smb4KUsers", 1, 1 ).stripWhiteSpace();
              QStringList user_list = QStringList::split( ",", users, false );

              if ( user_list.find( QString( "%1" ).arg( getenv( "USER" ) ) ) == user_list.end() )
              {
                user_list.append( QString( "%1" ).arg( getenv( "USER" ) ) );
                (*it).replace( users, user_list.join( "," ) );
              }
              else
                write = false;

              break;
            }
            else
            {
              continue;
            }
          }
        }
        else
        {
          // Convert the old entries.
          QStringList::Iterator insert_point;
          QString users;
          bool ok = false;
          int entries = 0;
          config()->setGroup( "Programs" );

          for ( QStringList::Iterator it = contents.begin(); it != contents.end(); ++it )
          {
            if ( (*it).startsWith( "# Entries for Smb4K users." ) )
            {
              ok = true;
              continue;
            }
            else if ( ok && ((*it).startsWith( "kill" ) ) )
            {
              (*it).replace( (*it).section( "\t", 1, 1 ).stripWhiteSpace(), config()->readPathEntry( "smb4k_kill" ), true );
              (*it).replace( (*it).section( "\t", 2, 2 ).stripWhiteSpace(), "$(Smb4KUsers)", true );
              QString tmp = "smb4k_kill\t";
              *it = tmp.append( (*it).section( "\t", 1, -1 ).stripWhiteSpace() );
              insert_point = it;
              entries++;
              continue;
            }
            else if ( ok && (*it).startsWith( "umount" ) )
            {
              (*it).replace( (*it).section( "\t", 1, 1 ).stripWhiteSpace(), config()->readPathEntry( "smb4k_umount" ), true );
              users = (*it).section( "\t", 2, 2 ).stripWhiteSpace();
              (*it).replace( users, "$(Smb4KUsers)", true );
              QString tmp = "smb4k_umount\t";
              *it = tmp.append( (*it).section( "\t", 1, -1 ).stripWhiteSpace() );
              entries++;
              continue;
            }
            else if ( ok && (*it).startsWith( "mount" ) )
            {
              (*it).replace( (*it).section( "\t", 1, 1 ).stripWhiteSpace(), config()->readPathEntry( "smb4k_mount" ), true );
              (*it).replace( (*it).section( "\t", 2, 2 ).stripWhiteSpace(), "$(Smb4KUsers)", true );
              QString tmp = "smb4k_mount\t";
              *it = tmp.append( (*it).section( "\t", 1, -1 ).stripWhiteSpace()+"\tenv=PASSWD,USER" );
              entries++;
              continue;
            }
            else if ( (*it).startsWith( "# End of Smb4K user entries." ) )
            {
              if ( entries < 3 )
              {
#ifndef __FreeBSD__
                contents.insert( it, "smb4k_mount\t"+config()->readPathEntry( "smb4k_mount" )+"\t$(Smb4KUsers)\tuid=root\tgid=root\tenv=PASSWD,USER" );
#else
                contents.insert( it, "smb4k_mount\t"+config()->readPathEntry( "smb4k_mount" )+"\t$(Smb4KUsers)\tuid=root\tgid=wheel\tsetenv=HOME=$CALLER_HOME\tenv=PASSWD,USER" );
#endif
              }

              break;
            }
            else
            {
              continue;
            }
          }

          QStringList user_list = QStringList::split( ",", users, false );

          if ( user_list.find( QString( "%1" ).arg( getenv( "USER" ) ) ) == user_list.end() )
            user_list.append( QString( "%1" ).arg( getenv( "USER" ) ) );

          contents.insert( insert_point, ":define Smb4KUsers "+user_list.join( "," ) );
        }
      }

      if ( write )
      {
        item.replaceContents( contents );

        if ( !writeFile( &item ) )
        {
          emit no_suid_program();
        }
      }
      else
      {
        emit finished_suid_writing();
      }

      remove_lock_file();
    }
    else if ( QString::compare( program, "sudo" ) == 0 )
    {
      if ( !failed && m_temp_dir.exists() )
      {
        char tmpf[] = "XXXXXX";

        if ( mktemp( tmpf ) == NULL )
        {
          int error_code = errno;

          emit error( ERROR_CREATING_TEMP_FILE, strerror( error_code ) );
          return false;
        }

        QFile tmp_file( tmpf );
        QDir::setCurrent( m_temp_dir.canonicalPath() );
        QFileInfo tmp_file_info( tmp_file );
        tmp_file_info.setCaching( false );

        if ( !tmp_file.exists() )
        {
          m_todo_sudoers = Add;
          m_state = copySudoers;
          sudoers_item = item;
          tmp_sudoers_info = tmp_file_info;
          tmp_sudoers_info.setCaching( false );

          *m_proc << "kdesu "+KProcess::quote( "cp "+sudoers_item.path()+" "+tmp_file_info.absFilePath()+" && chown "+QString( "%1:%2" ).arg( getuid() ).arg( getgid() )+" "+tmp_file_info.absFilePath()+" && chmod 0600 "+tmp_file_info.absFilePath() );

          m_proc->start( KProcess::NotifyOnExit, KProcess::Stderr );
        }
        else
        {
          emit error( ERROR_WRITING_FILE, tmp_file_info.absFilePath() );
          return false;
        }
      }
      else
      {
        if ( failed )
        {
          emit error( ERROR_CREATING_TEMP_DIR, strerror( error_number ) );
        }
        else
        {
          emit error( ERROR_CREATING_TEMP_DIR, m_temp_dir.canonicalPath() );
        }

        return false;
      }
    }
  }
  else
  {
    if ( !item.exists() )
    {
      emit no_suid_program();
      emit finished_suid_writing();
      remove_lock_file();
      return false;
    }
  }

  return true;
}


/****************************************************************************
   Removes all Smb4K specific entries of the user from /etc/super.tab.
****************************************************************************/

bool Smb4KFileIO::removeSuperUserEntries()
{
  config()->setGroup( "Super User Privileges" );
  QString program = config()->readEntry( "SUID Program", QString::null );

  FileItem item;

  if ( QString::compare( program, "super" ) == 0 )
  {
    item = readConfigFile( "super.tab" );
  }
  else if ( program = "sudo" )
  {
    item = readConfigFile( "sudoers", false, true );
  }

  if ( item.exists() && write_lock_file( item.path() ) )
  {
    if ( QString::compare( program, "super" ) == 0 )
    {
      QStringList contents = item.contents();
      QStringList::Iterator j;

      for ( QStringList::Iterator it = contents.begin(); it != contents.end(); ++it )
      {
        if ( (*it).startsWith( "# Entries for Smb4K users." ) )
        {
          j = it;
        }
        else if ( (*it).startsWith( ":define Smb4KUsers" ) )
        {
          QString users = (*it).section( "Smb4KUsers", 1, 1 ).stripWhiteSpace();

          if ( users.contains( "," ) == 0 )
          {
            while ( j != contents.end() )
            {
              if ( !(*j).startsWith( "# End of Smb4K user entries." ) )
              {
                j = contents.remove( j );
              }
              else
              {
                contents.remove( j );
                break;
              }
            }
          }
          else
          {
            QStringList list = QStringList::split( ",", users, false );
            list.remove( QString( "%1" ).arg( getenv( "USER" ) ) );

            (*it).replace( users, list.join( "," ) );
          }

          break;
        }
        else
        {
          continue;
        }
      }

      item.replaceContents( contents );

      if ( !writeFile( &item ) )
      {
        emit no_suid_program();
      }

      remove_lock_file();
    }
    else if ( QString::compare( program, "sudo" ) == 0 )
    {
      if ( !failed && m_temp_dir.exists() )
      {
        char tmpf[] = "XXXXXX";

        if ( mktemp( tmpf ) == NULL )
        {
          int error_code = errno;

          emit error( ERROR_CREATING_TEMP_FILE, strerror( error_code ) );
          return false;
        }

        QFile tmp_file( tmpf );
        QDir::setCurrent( m_temp_dir.canonicalPath() );
        QFileInfo tmp_file_info( tmp_file );
        tmp_file_info.setCaching( false );

        if ( !tmp_file.exists() )
        {
          m_todo_sudoers = Remove;
          m_state = copySudoers;
          sudoers_item = item;
          tmp_sudoers_info = tmp_file_info;
          tmp_sudoers_info.setCaching( false );

          *m_proc << "kdesu "+KProcess::quote( "cp "+sudoers_item.path()+" "+tmp_file_info.absFilePath()+" && chown "+QString( "%1:%2" ).arg( getuid() ).arg( getgid() )+" "+tmp_file_info.absFilePath()+" && chmod 0600 "+tmp_file_info.absFilePath() );

          m_proc->start( KProcess::NotifyOnExit, KProcess::Stderr );
        }
        else
        {
          emit error( ERROR_WRITING_FILE, tmp_file_info.absFilePath() );
          return false;
        }
      }
      else
      {
        if ( failed )
        {
          emit error( ERROR_CREATING_TEMP_DIR, strerror( error_number ) );
        }
        else
        {
          emit error( ERROR_CREATING_TEMP_DIR, m_temp_dir.canonicalPath() );
        }

        return false;
      }
    }
  }
  else
  {
    if ( !item.exists() )
    {
      emit no_suid_program();
      emit finished_suid_writing();
      return false;
    }
  }

  return true;
}


/****************************************************************************
   Extracts the non-system users from the /etc/passwd and /etc/group
****************************************************************************/


const QValueList<Smb4KUser *> Smb4KFileIO::getUsers()
{
  FileItem item = readConfigFile( "passwd" );
  QStringList contents = item.contents();

  QValueList<Smb4KUser *> users;

  for ( QStringList::ConstIterator it = contents.begin(); it != contents.end(); ++it )
  {
    if ( !(*it).stripWhiteSpace().startsWith( "#" ) && ( (*it).section( ":", 2, 2 ).toInt() >= 500 || (*it).section( ":", 2, 2 ).toInt() == (int)getuid() ) )
    {
      users.append( new Smb4KUser( (*it).section( ":", 2, 2 ).toInt(), (*it).section( ":", 3, 3 ).toInt() ) );
    }
    else
    {
      continue;
    }
  }

  return users;
}


/****************************************************************************
   Returns the paper size defined on the system.
****************************************************************************/

const QString Smb4KFileIO::getPaperSize()
{
  FileItem item = readConfigFile( "papersize" );
  QStringList contents = item.contents();

  return contents.isEmpty() ? QString( "a4" ) : contents.join( " " ).stripWhiteSpace();
}


/****************************************************************************
   Locates a system config file and returns its path
****************************************************************************/

Smb4KFileIO::FileItem Smb4KFileIO::readConfigFile( const QString &fileName, bool strip, bool quiet )
{
  QStringList paths;
  paths << "/etc";
  paths << "/etc/samba";
  paths << "/usr/local/etc";
  paths << "/usr/local/etc/samba";

  QStringList contents;
  bool exists = false;
  bool read_failed = false;

  QString path;
  QString abs_file_path;
  QString perms;

  for ( QStringList::ConstIterator it = paths.begin(); it != paths.end(); ++it )
  {
    QDir::setCurrent( *it );

    if ( QFile::exists( fileName ) )
    {
      exists = true;
      path = QDir::currentDirPath();
      abs_file_path = path+"/"+fileName;

      QFile file( fileName );

      if ( file.open( IO_ReadOnly ) )
      {
        QTextStream ts( &file );

        while ( !ts.atEnd() )
        {
          if ( !strip )
          {
            contents.append( ts.readLine() );
          }
          else
          {
            contents.append( ts.readLine().stripWhiteSpace() );
          }
        }

        file.close();
      }
      else
      {
        if ( !quiet )
        {
          read_failed = true;
          emit error( ERROR_READING_FILE, fileName );
        }
      }

      struct stat file_stat;

      if ( stat( abs_file_path.ascii(), &file_stat ) == -1 )
      {
        int error_number = errno;

        emit error( ERROR_GETTING_PERMISSIONS, strerror( error_number ) );

        return Smb4KFileIO::FileItem();
      }

      perms = QString( "%1" ).arg( (int)file_stat.st_mode, 0, 8 );
      perms = perms.right( 4 );

      break;
    }
    else
    {
      continue;
    }
  }

  if ( !exists )
  {
    emit error( ERROR_FILE_NOT_FOUND, fileName );
  }

  if ( read_failed )
  {
    exists = false;
  }

  return Smb4KFileIO::FileItem( abs_file_path, perms, contents, exists );
}


/****************************************************************************
   Writes a file to its destination
****************************************************************************/

bool Smb4KFileIO::writeFile( Smb4KFileIO::FileItem *item )
{
  QStringList contents = item->contents();

  if ( !failed && m_temp_dir.exists() )
  {
    char tmpf[] = "XXXXXX";

    if ( mktemp( tmpf ) == NULL )
    {
      int error_code = errno;

      emit error( ERROR_CREATING_TEMP_FILE, strerror( error_code ) );
      return false;
    }

    QFile tmp_file( tmpf );
    QDir::setCurrent( m_temp_dir.canonicalPath() );
    QFileInfo tmp_file_info( tmp_file );
    tmp_file_info.setCaching( false );

    if ( !tmp_file.exists() && tmp_file.open( IO_WriteOnly ) )
    {
      QTextStream ts( &tmp_file );

      for ( QStringList::ConstIterator it = contents.begin(); it != contents.end(); ++it )
      {
        ts << *it << endl;
      }

      tmp_file.close();
    }
    else
    {
      emit error( ERROR_WRITING_FILE, tmp_file_info.absFilePath() );
      return false;
    }

    // Now move it to the right location.
    if ( tmp_file.exists() && tmp_file_info.isFile() )
    {
      m_state = writeSU;

#ifndef __FreeBSD__
      QString command( "chown root:root "+tmp_file_info.absFilePath()+" && chmod "+item->permissions()+" "+tmp_file_info.absFilePath()+" && mv "+tmp_file_info.absFilePath()+" "+item->path() );
#else
      QString command( "chown root:wheel "+tmp_file_info.absFilePath()+" && chmod "+item->permissions()+" "+tmp_file_info.absFilePath()+" && mv "+tmp_file_info.absFilePath()+" "+item->path() );
#endif

      *m_proc << QString( "kdesu %1 ; rm -f %2" ).arg( KProcess::quote( command ), KProcess::quote( tmp_file_info.absFilePath() ) );
      m_proc->start( KProcess::NotifyOnExit, KProcess::Stderr );
    }
    else
    {
      if ( !tmp_file.exists() )
      {
        emit error( ERROR_FILE_NOT_FOUND, tmp_file_info.absFilePath() );
      }
      else
      {
        emit error( ERROR_WRITING_FILE, tmp_file_info.absFilePath() );
      }
      return false;
    }
  }
  else
  {
    if ( failed )
    {
      emit error( ERROR_CREATING_TEMP_DIR, strerror( error_number ) );
    }
    else
    {
      emit error( ERROR_CREATING_TEMP_DIR, m_temp_dir.canonicalPath() );
    }

    return false;
  }

  return true;
}


/****************************************************************************
   Process the sudoers file.
****************************************************************************/


void Smb4KFileIO::processSudoers()
{
  QDir::setCurrent( m_temp_dir.canonicalPath() );
  QFile file( tmp_sudoers_info.fileName() );
  tmp_sudoers_info.refresh();

  QStringList contents;

  // In case the user canceled or did not provide the right
  // password.
  if ( !file.exists() )
  {
    remove_lock_file();
    return;
  }

  if ( file.open( IO_ReadOnly ) )
  {
    QTextStream ts( &file );

    while ( !ts.atEnd() )
    {
      contents.append( ts.readLine() );
    }

    file.close();
  }
  else
  {
    emit error( ERROR_READING_FILE, file.name() );
    return;
  }

  file.remove();

  bool write = true;

  if ( m_todo_sudoers == Add )
  {
    char *hn;
    hn = (char *)malloc( 200*sizeof(char) );

    if ( gethostname( hn, 200*sizeof(char) ) != 0 )
    {
      emit error( ERROR_GETTING_HOSTNAME, QString::null );
      return;
    }

    QString hostname( hn );

    free( hn );

    config()->setGroup( "Programs" );

    if ( contents.grep( "# Entries for Smb4K users." ).count() == 0 )
    {
      contents.append( "# Entries for Smb4K users." );
      contents.append( "# Generated by Smb4K. Please do not modify!" );
      contents.append( "User_Alias\tSMB4KUSERS = "+QString( "%1" ).arg( getenv( "USER" ) ) );
      contents.append( "SMB4KUSERS\t"+hostname+" = NOPASSWD: "+config()->readPathEntry( "smb4k_kill" ) );
      contents.append( "SMB4KUSERS\t"+hostname+" = NOPASSWD: "+config()->readPathEntry( "smb4k_umount" ) );
      contents.append( "SMB4KUSERS\t"+hostname+" = NOPASSWD: "+config()->readPathEntry( "smb4k_mount" ) );
      contents.append( "Defaults:SMB4KUSERS\tenv_keep=PASSWD" );
      contents.append( "# End of Smb4K user entries." );
    }
    else
    {
      for ( QStringList::Iterator it = contents.find( "# Entries for Smb4K users." ); it != contents.end(); ++it )
      {
        if ( (*it).startsWith( "User_Alias\tSMB4KUSERS" ) )
        {
          if ( (*it).contains( getenv( "USER" ), true ) == 0 )
          {
            (*it).append( ","+QString( getenv( "USER") ) );
          }
          else
          {
            write = false;
          }
          break;
        }
        else
        {
          continue;
        }
      }
    }
  }
  else if ( m_todo_sudoers == Remove )
  {
    QStringList::Iterator j = contents.find( "# Entries for Smb4K users." );
    bool remove_all = false;

    for ( QStringList::Iterator it = j; it != contents.end(); ++it )
    {
      if ( (*it).startsWith( "User_Alias" ) && (*it).contains( "SMB4KUSERS" ) != 0 )
      {
        if ( (*it).contains( getenv( "USER" ), true ) != 0 )
        {
          QStringList users = QStringList::split( ",", (*it).section( "=", 1, 1 ).stripWhiteSpace(), false );
          users.remove( QString( "%1" ).arg( getenv( "USER" ) ) );

          if ( users.isEmpty() )
          {
            remove_all = true;
          }
          else
          {
            (*it).replace( (*it).section( "=", 1, 1 ).stripWhiteSpace(), users.join( "," ), true );
          }
        }
        else
        {
          write = false;
        }

        break;
      }
      else if ( (*it).startsWith( "# End of Smb4K user entries." ) )
      {
        break;
      }
      else
      {
        continue;
      }
    }

    if ( remove_all )
    {
      while ( j != contents.end() )
      {
        if ( !(*j).startsWith( "# End of Smb4K user entries." ) )
        {
          j = contents.remove( j );
        }
        else
        {
          contents.remove( j );
          break;
        }
      }
    }
  }

  if ( write )
  {
    sudoers_item.replaceContents( contents );

    if ( !writeFile( &sudoers_item ) )
    {
      emit no_suid_program();
    }
  }
  else
  {
    emit finished_suid_writing();
  }

  remove_lock_file();
}


/****************************************************************************
   Write the lock file
****************************************************************************/

bool Smb4KFileIO::write_lock_file( const QString &filename )
{
  QDir dir( "/tmp" );
  bool lock_file_exists = false;
  bool ok = false;

  const QFileInfoList *list = dir.entryInfoList( QDir::Files, QDir::Name );

  if ( list != 0 )
  {
    for ( QFileInfoList::ConstIterator it = list->begin(); it != list->end(); ++it )
    {
      (*it)->setCaching( false );

      if ( (*it)->exists() && (*it)->fileName().compare( "smb4k.lock" ) == 0 && (*it)->fileName().length() == 10 )
      {
        if ( !(*it)->isSymLink() && (*it)->isReadable() && (*it)->isWritable() )
        {
          lock_file.setName( (*it)->absFilePath() );
          lock_file_exists = true;
        }
        else
        {
          // We will stop at the first error that occurrs.
          if ( (*it)->isSymLink() )
          {
            emit error( ERROR_LOCK_FILE_IS_SYMLINK, (*it)->absFilePath() );
          }
          else if ( !(*it)->isReadable() )
          {
            emit error( ERROR_READING_FILE, (*it)->absFilePath() );
          }
          else if ( !(*it)->isWritable() )
          {
            emit error( ERROR_WRITING_FILE, (*it)->absFilePath() );
          }

          emit finished_suid_writing();
          return ok;
        }

        break;
      }
      else
      {
        continue;
      }
    }
  }

  if ( !lock_file_exists )
  {
    lock_file.setName( "/tmp/smb4k.lock" );

    if ( lock_file.open( IO_WriteOnly ) )
    {
      QTextStream ts( &lock_file );

      ts << getenv( "USER" ) << ":" << filename << endl;
    }
    else
    {
      emit error( ERROR_WRITING_FILE, lock_file.name() );

      return ok;
    }

    lock_file.close();
    ok = true;
  }
  else
  {
    if ( lock_file.open( IO_ReadWrite ) )
    {
      QTextStream ts( &lock_file );
      QStringList contents = QStringList::split( '\n', ts.read(), false );
      QString test_string = QString( ":%2" ).arg( filename );
      QString entry = contents.grep( test_string, true ).join( "\n" ).stripWhiteSpace();

      if ( !entry.isEmpty() )
      {
        emit error( ERROR_LOCKED, entry );

        return ok;
      }
      else
      {
        ts << getenv( "USER" ) << ":" << filename << endl;
      }

      lock_file.close();
      ok = true;
    }
    else
    {
      emit error( ERROR_READING_FILE, lock_file.name() );

      return ok;
    }
  }

  return ok;
}


bool Smb4KFileIO::remove_lock_file()
{
  QDir dir( "/tmp" );
  bool lock_file_exists = false;
  bool ok = true;

  const QFileInfoList *list = dir.entryInfoList( QDir::Files, QDir::Name );

  if ( list != 0 )
  {
    for ( QFileInfoList::ConstIterator it = list->begin(); it != list->end(); ++it )
    {
      (*it)->setCaching( false );

      if ( (*it)->exists() && (*it)->fileName().compare( "smb4k.lock" ) == 0 && (*it)->fileName().length() == 10 )
      {
        if ( !(*it)->isSymLink() && (*it)->isReadable() && (*it)->isWritable() )
        {
          lock_file.setName( (*it)->absFilePath() );
          lock_file_exists = true;
        }
        else
        {
          if ( !shutdown )
          {
            // We will stop at the first error that occurrs.
            if ( (*it)->isSymLink() )
            {
              emit error( ERROR_LOCK_FILE_IS_SYMLINK, (*it)->absFilePath() );
            }
            else if ( !(*it)->isReadable() )
            {
              emit error( ERROR_READING_FILE, (*it)->absFilePath() );
            }
            else if ( !(*it)->isWritable() )
            {
              emit error( ERROR_WRITING_FILE, (*it)->absFilePath() );
            }
          }

          emit finished_suid_writing();
          return !ok;
        }

        break;
      }
      else
      {
        continue;
      }
    }
  }

  if ( lock_file_exists )
  {
    QStringList contents;

    if ( lock_file.open( IO_ReadOnly ) )
    {
      QTextStream ts( &lock_file );
      contents = QStringList::split( '\n', ts.read().stripWhiteSpace(), false );

      lock_file.close();
    }

    for ( QStringList::Iterator it = contents.begin(); it != contents.end(); ++it )
    {
      if ( (*it).startsWith( getenv( "USER" ) ) )
      {
        *it = QString::null;
        continue;
      }
      else
      {
        continue;
      }
    }

    contents.remove( QString::null );

    if ( !contents.isEmpty() )
    {
      if ( lock_file.open( IO_WriteOnly ) )
      {
        QTextStream ts( &lock_file );
        ts << contents.join( "\n" ) << endl;
        lock_file.close();
      }
    }
    else
    {
      ok = lock_file.remove();
    }
  }

  return ok;
}


/////////////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////

void Smb4KFileIO::slotReceivedStderr( KProcess *, char *buf, int len )
{
  QString buffer( QString::fromLocal8Bit( buf, len ) );

//   if ( buffer.contains( "KIconLoader" ) == 0 && buffer.contains( "(kdelibs)" ) == 0 )
//   {
//     emit error( ERROR_UNKNOWN, buffer );
//   }
}


void Smb4KFileIO::slotProcessExited( KProcess * )
{
  switch ( m_state )
  {
    case writeSU:
      m_state = Idle;
      m_proc->clearArguments();
      emit finished_suid_writing();
      break;
    case copySudoers:
      m_state = Idle;
      m_proc->clearArguments();
      processSudoers();
      break;
    default:
      m_state = Idle;
      m_proc->clearArguments();
      break;
  }
}


void Smb4KFileIO::slotShutdown()
{
  shutdown = true;
  remove_lock_file();
  m_temp_dir.rmdir( m_temp_dir.canonicalPath() );
}


#include "smb4kfileio.moc"
