/***************************************************************************
    smb4ksystemtray.cpp - The widget that handles the docking
              to the system tray.
                             -------------------
    begin                : Mon Jan 12 2004
    copyright            : (C) 2004 by Franck Babin
    email                : babinfranck@yahoo.ca
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

//KDE includes
#include <kapplication.h>
#include <kpopupmenu.h>
#include <kiconloader.h>
#include <klocale.h>

//Qt includes
#include <qtooltip.h>
#include <qiconset.h>
#include <qstringlist.h>
#include <qvariant.h>
#include <qpopupmenu.h>
#include <qpixmap.h>
#include <qpainter.h>

// application specific includes
#include "smb4ksystemtray.h"
#include "smb4kbookmarkmenuwidget.h"
#include "../core/smb4kbookmark.h"

class Smb4KSystemTrayItem : public QCustomMenuItem
{
public:
  Smb4KSystemTrayItem( const QString& name, const QString &path, const QFont& font, int item_type ):
      m_font(font), m_type(item_type), m_name(name), m_path(path) {};
  ~Smb4KSystemTrayItem(){}

  enum item_types {Share, Bookmark};

  bool isItemType(int type){return (m_type == type);}

  void paint( QPainter* p, const QColorGroup& /*cg*/, bool /*act*/, bool /*enabled*/, int x, int y, int w, int h )
  {
    p->setFont ( m_font );
    p->drawText( x, y, w, h, AlignLeft | AlignVCenter | DontClip | ShowPrefix, m_name );
  }

  QSize sizeHint()
  {
    return QFontMetrics( m_font ).size( AlignLeft | AlignVCenter | ShowPrefix | DontClip,  m_name );
  }

  QString getName()
  {
    return m_name;
  }

  QString getPath()
  {
    return m_path;
  }

private:
  QFont m_font;
  int m_type;
  QString m_name;
  QString m_path;
};

Smb4KSystemTray::Smb4KSystemTray(QWidget* parent, const char* name)
          : KSystemTray(parent,name), docked()
{
  m_app = (Smb4KApp *)parent;

  //Sets the icon to display in the system tray using the main icon
  setPixmap( BarIcon( "smb4k" ) );
  setAlignment( Qt::AlignCenter );
  initContextMenu();
  QToolTip::add(this, "Smb4K");
}

Smb4KSystemTray::~Smb4KSystemTray()
{
  if(popup_shares){delete popup_shares;}
  if(popup_bookmarks){delete popup_bookmarks;}
}

/**
 * Docks the main widget to the system tray,
 */
void Smb4KSystemTray::dock()
{
  if (!docked)
  {
    docked = true;
    this->show();
  }
}

/**
 * Undocks the window from the system tray,
 */
void Smb4KSystemTray::undock()
{
  if (docked)
  {
    docked = false;
    this->hide();
  }
}

void Smb4KSystemTray::initContextMenu()
{
  contextMenu()->changeTitle(contextMenu()->idAt(0), "Smb4K");

  //add share header
  popup_shares = new QPopupMenu( this );
  Q_CHECK_PTR(popup_shares);
  contextMenu()->insertItem( QIconSet(SmallIcon("hdd_unmount")), i18n("Mounted &Shares"), popup_shares );

  //add bookmark header
  popup_bookmarks = new Smb4KBookmarkMenuWidget(Smb4KBookmarkMenuWidget::SMALL, m_app);
  Q_CHECK_PTR(popup_bookmarks);
  contextMenu()->insertItem( QIconSet(SmallIcon("folder_green")), i18n("&Bookmarks"), popup_bookmarks );

  //add configuration menu
  contextMenu()->insertItem(QIconSet(SmallIcon("configure"), QIconSet::Small),
                            i18n("Configure Smb4K..."),
                            this, SLOT(slotDisplayConfig()),
                            0, Configure, -1);

  //connect for listing to mount/unmount share
  connect( m_app->core()->mounter(), SIGNAL( importedShares() ), this,                  SLOT( slotMountedShares() ) );
  connect( m_app->core()->mounter(), SIGNAL( mountedShare(Smb4KShare *) ), this,        SLOT( slotMountedShare(Smb4KShare *) ) );
  connect( m_app->core()->mounter(), SIGNAL( unmountedShare(QString, QString) ), this,  SLOT( slotUnmountedShare(QString, QString) ) );
  connect( m_app->core()->mounter(), SIGNAL( unmountedAllShares() ), this,              SLOT( slotUnmountedAllShares() ) );
  connect( this, SIGNAL( quitSelected() ), m_app,                                       SLOT( slotQuit() ) );
}

void Smb4KSystemTray::slotMountedShares()
{
  QValueList<Smb4KShare *> shares = m_app->core()->mounter()->getShares();

  // Due to sorting problems we have to do this workaround:
  QStringList list;
  
  for ( QValueListIterator<Smb4KShare *> it = shares.begin(); it != shares.end(); ++it )
    list.append( (*it)->getName() );
    
  list.sort();
  
  for ( QStringList::ConstIterator it = list.begin(); it != list.end(); ++it )
  {
    mountShare( *it, (m_app->core()->mounter()->findShareByName( *it ))->getPath() );
  }
}

void Smb4KSystemTray::slotMountedShare(Smb4KShare *share)
{
  mountShare(share->getName(), share->getPath());
}

void Smb4KSystemTray::slotUnmountedShare(QString share, QString path)
{
  for( QMap<int, QString>::Iterator it = mounted_shares_id.begin(); it != mounted_shares_id.end(); ++it)
  {
    if( path == it.data() )
    {
      popup_shares->removeItem(it.key());
      mounted_shares.remove(share);
      mounted_shares_id.remove(it.key());

      return;
    }
  }
}

void Smb4KSystemTray::slotUnmountedAllShares()
{
  popup_shares->clear();
  mounted_shares.clear();
  mounted_shares_id.clear();
}

void Smb4KSystemTray::mountShare(const QString name, const QString path)
{
  if( mounted_shares.contains(name) )
  {
    return;
  }

  mounted_shares[name] = path;

  Smb4KSystemTrayItem *item = new Smb4KSystemTrayItem( name, path, popup_shares->font(), Smb4KSystemTrayItem::Share );
  int id = popup_shares->insertItem( QIconSet(SmallIcon("hdd_mount")), item );
  mounted_shares_id[id] = path;
  popup_shares->connectItem( id, this, SLOT(slotDisplayShare(const QVariant &)) );
}

void Smb4KSystemTray::slotDisplayConfig()
{
  m_app->slotSmb4KOptionsDlg();
}

void Smb4KSystemTray::slotMountBookmark(int id)
{
  QString bookmark = popup_bookmarks->text(id);
  if ( bookmark.startsWith( "//" ) )
  {
    QString host = bookmark.section( "/", 2, 2 );
    QString share = bookmark.section( "/", 3, 3 );
    // FIXME: should not try to cut the string. Instead ask the bookmark
    // handler to do this and call the mounter itself.
    m_app->core()->mounter()->mountShare( QString::null, host, QString::null, share );
  }
}

void Smb4KSystemTray::slotDisplayShare(const QVariant &item_id)
{
  int id = item_id.toInt();
  for( QMap<int, QString>::Iterator it = mounted_shares_id.begin(); it != mounted_shares_id.end(); ++it)
  {
    if( id == it.key() )
    {
      m_app->core()->open( it.data() );
      return;
    }
  }
}

#include "smb4ksystemtray.moc"
