/***************************************************************************
    smb4kbrowserwidget.cpp  -  Network browser widget class of Smb4K.
                             -------------------
    begin                : Sam Feb 22 2003
    copyright            : (C) 2003 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// Qt includes
#include <qlayout.h>
#include <qpainter.h>
#include <qheader.h>

// KDE includes
#include <kapplication.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kmessagebox.h>
#include <kdialogbase.h>

// application specific includes
#include "smb4kbrowserwidget.h"
#include "../smb4k.h"
#include "smb4kpreviewdialog.h"
#include "smb4kprintdialog.h"


Smb4KBrowserWidget::Smb4KBrowserWidget( QWidget *parent, const char *name ) : KListView( parent, name )
{
  m_config = kapp->config();

  setFrameShape( KListView::PopupPanel );
  setFrameShadow( KListView::Sunken );
  setRootIsDecorated( true );

  addColumn( i18n( "Network" ), -1 );
  addColumn( i18n( "Type" ), -1 );
  addColumn( i18n( "IP Address" ), -1 );
  addColumn( i18n( "Comment" ), -1 );

  m_menu = new KPopupMenu( this );

  // Widget specific connections.
  connect( this, SIGNAL( executed( QListViewItem * ) ), SLOT( slotItemExecuted( QListViewItem * ) ) );
  connect( this, SIGNAL( expanded( QListViewItem * ) ), SLOT( slotItemExpandedCollapsed( QListViewItem * ) ) );
  connect( this, SIGNAL( collapsed( QListViewItem * ) ), SLOT( slotItemExpandedCollapsed( QListViewItem * ) ) );
  connect( this, SIGNAL( selectionChanged( QListViewItem * ) ), SLOT( slotSelectionChanged( QListViewItem * ) ) );
  connect( this, SIGNAL( clicked( QListViewItem * ) ), SLOT( slotItemClicked( QListViewItem * ) ) );
  connect( this, SIGNAL( rightButtonPressed( QListViewItem *, const QPoint &, int ) ), SLOT( slotRightButtonPressed( QListViewItem *, const QPoint &, int ) ) );

  // External connections.
  connect( sapp->core()->scanner(), SIGNAL( workgroups( const QValueList<Smb4KWorkgroupItem *> & ) ), this, SLOT( slotWorkgroups( const QValueList<Smb4KWorkgroupItem *> & ) ) );
  connect( sapp->core()->scanner(), SIGNAL( members( const QValueList<Smb4KHostItem *> & ) ), this, SLOT( slotMembers( const QValueList<Smb4KHostItem *> & ) ) );
  connect( sapp->core()->scanner(), SIGNAL( shares( const QValueList<Smb4KShareItem *> & ) ), this, SLOT( slotShares( const QValueList<Smb4KShareItem *> & ) ) );
  connect( sapp->core()->ipFinder(), SIGNAL( ip_address( Smb4KHostItem * ) ), this, SLOT( slotAddIPAddress( Smb4KHostItem * ) ) );

  connect( sapp->core(), SIGNAL( runStateChanged() ), this, SLOT( slotCoreRunStateChanged() ) );
}


Smb4KBrowserWidget::~Smb4KBrowserWidget()
{
}


void Smb4KBrowserWidget::setOpen( QListViewItem *item, bool open )
{
  if ( open && item->depth() == 0 )
  {
    Smb4KWorkgroupItem *i = sapp->core()->scanner()->getWorkgroup( item->text( Network ) );
    
    sapp->core()->scanner()->getWorkgroupMembers( item->text( Network ), i->master(), i->ip() );
  }
  else if ( open && item->depth() == 1 && item->childCount() == 0 )
  {
    sapp->core()->scanner()->getShares( item->parent()->text( Network ), item->text( Network ), item->text( IP ) );
  }
  KListView::setOpen( item, open );
}


void Smb4KBrowserWidget::insertItem( Smb4KHostItem *item )
{
  if ( sapp->core()->scanner()->getWorkgroup( item->workgroup() ) == 0 )
    sapp->core()->scanner()->appendWorkgroup( new Smb4KWorkgroupItem( item->workgroup(), item->name(), item->ip() ) );  
  
  // Now put the host into the list.
  QListViewItem *workgroupItem = findItem( item->workgroup(), Network, ExactMatch|CaseSensitive );

  if ( workgroupItem != 0 )
  {
    QListViewItem *hostItem = findItem( item->name(), Network, ExactMatch|CaseSensitive );

    if ( hostItem == 0 )
    {
      Smb4KBrowserWidgetItem *newItem = new Smb4KBrowserWidgetItem( workgroupItem, item );
      newItem->setExpandable( true );
    }
    else if ( hostItem != 0 && hostItem->parent()->text( Network ) != item->workgroup() )
    {
      Smb4KBrowserWidgetItem *newItem = new Smb4KBrowserWidgetItem( workgroupItem, item );
      newItem->setExpandable( true );
    }
    else
      KMessageBox::sorry( this, i18n( "This computer is already in the list!" ), QString::null, KMessageBox::Notify );
  }
  else
  {
    Smb4KBrowserWidgetItem *newItem = new Smb4KBrowserWidgetItem( this, new Smb4KWorkgroupItem( item->workgroup(), item->name(), item->ip() ) );
    newItem->setExpandable( true );

    Smb4KBrowserWidgetItem *hostItem = new Smb4KBrowserWidgetItem( newItem, item );
    hostItem->setExpandable( true );
  }
}


void Smb4KBrowserWidget::readOptions()
{
  // Read the configuration.
  m_config->setGroup( "Appearance" );
  bool showType = m_config->readBoolEntry( "Show Type", true );
  bool showComment = m_config->readBoolEntry( "Show Comment", true );
  bool showIP = m_config->readBoolEntry( "Show IP", true );

  // Now put everything back in according to the wishes of the user.
  if ( showIP )
  {
    setColumnWidth( IP, 10 );
    setColumnWidthMode( IP, QListView::Maximum );
  }
  else
  {
    setColumnWidth( IP, 0 );
    setColumnWidthMode( IP, QListView::Manual );
  }

  if ( showType )
  {
    setColumnWidth( Type, 10 );
    setColumnWidthMode( Type, QListView::Maximum );
  }
  else
  {
    setColumnWidth( Type, 0 );
    setColumnWidthMode( Type, QListView::Manual );
  }

  if ( showComment )
  {
    setColumnWidth( Comment, 10 );
    setColumnWidthMode( Comment, QListView::Maximum );
  }
  else
  {
    setColumnWidth( Comment, 0 );
    setColumnWidthMode( Comment, QListView::Manual );
  }

  // Adjust the columns. Avoid those that have a width of 0.
  int index = 0;
  while ( index < this->columns() )
  {
    if ( this->columnWidth( index ) != 0 )
      adjustColumn( index );

    index++;
  }

  // Initialize the rest.
  m_hidden = m_config->readBoolEntry( "Show Hidden", true );
  m_ipc = m_config->readBoolEntry( "Show IPC", false );
  m_admin = m_config->readBoolEntry( "Show ADMIN", false );
  m_printer = m_config->readBoolEntry( "Show Printer", true );
}


const QString Smb4KBrowserWidget::specifyUser( const QString &host )
{
  KDialogBase *dialog = new KDialogBase( KDialogBase::Plain, i18n( "Specify User" ), KDialogBase::Ok|KDialogBase::Cancel, KDialogBase::Ok, this, 0, false, true );

  // Set up the ask pass dialog.
  QFrame *frame = dialog->plainPage();
  QGridLayout *layout = new QGridLayout( frame );
  layout->setSpacing( 5 );

  QLabel *pic = new QLabel( frame );
  pic->setPixmap( DesktopIcon( "personal" ) );
  pic->setMargin( 10 );

  QLabel *text = new QLabel( i18n( "Please specify a user name." ), frame );

  QLabel *user = new QLabel( i18n( "User:" ), frame );
  KComboBox *username = new KComboBox( true, frame, 0 );
  username->setDuplicatesEnabled( false );

  QSpacerItem *spacer1 = new QSpacerItem( 10, 10, QSizePolicy::Expanding, QSizePolicy::Preferred );

  layout->addWidget( pic, 0, 0, 0 );
  layout->addMultiCellWidget( text, 0, 0, 1, 3, 0 );
  layout->addWidget( user, 1, 0, 0 );
  layout->addMultiCellWidget( username, 1, 1, 1, 4, 0 );
  layout->addItem( spacer1, 0, 2 );

  // Read the list of logins, that were already defined
  // for this 'homes' share.
  QStringList list;

  if ( m_config->hasGroup( "Homes Shares" ) )
  {
    m_config->setGroup( "Homes Shares" );
    if ( m_config->hasKey( host ) )
      list = m_config->readListEntry( host, ',' );
  }

  if ( !list.isEmpty() )
    username->insertStringList( list, -1 );

  username->setCurrentText( QString::null );

  // Do the last things before showing.
  username->setFocus();
  dialog->setFixedSize( dialog->sizeHint() );

  if ( dialog->exec() == KDialogBase::Accepted )
  {
    // Write the new list of logins to the config file.
    if ( list.contains( username->currentText() ) == 0 )
      list.append( username->currentText() );

    int index = 0;
    while ( index < username->count() )
    {
      if ( list.contains( username->text( index ) ) == 0 )
        list.append( username->text( index ) );

      index++;
    }

    list.sort();

    m_config->setGroup( "Homes Shares" );
    m_config->writeEntry( host, list, ',' );

    return username->currentText();
  }
  else
    return QString::null;
}


/////////////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////


void Smb4KBrowserWidget::slotItemExecuted( QListViewItem *item )
{
  if ( item && item->depth() == 2 )
  {
    if ( item->text( Type ).contains( "Printer" ) == 0 )
    {
      slotMountShare();
    }
    else
    {
      slotPrint();
    }
  }
}


void Smb4KBrowserWidget::slotItemExpandedCollapsed( QListViewItem *item )
{
  setSelected( item, true );
}


void Smb4KBrowserWidget::slotSelectionChanged( QListViewItem *item )
{
  if ( item && item->depth() == 2 )
  {
    sapp->bookmarksAddBookmark->setEnabled( true );
    sapp->networkAskPass->setEnabled( true );
    
    if ( item->text( Type ).contains( "Printer" ) == 0 )
    {
      sapp->networkPreview->setEnabled( true );
      sapp->sharesMount->setEnabled( true );
      sapp->networkPrint->setEnabled( false );
    }
    else
    {
      sapp->networkPreview->setEnabled( false );
      sapp->sharesMount->setEnabled( false );
      sapp->networkPrint->setEnabled( true );
    }
  }
  else if ( item && item->depth() == 1 )
  {
    sapp->bookmarksAddBookmark->setEnabled( false );
    sapp->networkAskPass->setEnabled( true );
    sapp->networkPreview->setEnabled( false );
    sapp->networkPrint->setEnabled( false );
    sapp->sharesMount->setEnabled( false );
  }
  else
  {
    sapp->bookmarksAddBookmark->setEnabled( false );
    sapp->networkPreview->setEnabled( false );
    sapp->networkPrint->setEnabled( false );
    sapp->sharesMount->setEnabled( false );
    sapp->networkAskPass->setEnabled( false );
  }

  if ( item )
    sapp->view()->networkTab()->display( item );
}


void Smb4KBrowserWidget::slotRightButtonPressed( QListViewItem *item, const QPoint &point, int col )
{
  m_menu->clear();

  // The icon sets:
  QIconSet reloadSet( SmallIcon( "reload" ), QIconSet::Automatic );
  QIconSet abortSet( SmallIcon( "stop" ), QIconSet::Automatic );
  QIconSet searchSet( SmallIcon( "find" ), QIconSet::Automatic );
  QIconSet bookmarkSet( SmallIcon( "bookmark_add" ), QIconSet::Automatic );
  QIconSet authSet( SmallIcon( "personal" ), QIconSet::Automatic );
  QIconSet previewSet( SmallIcon( "view_icon" ), QIconSet::Automatic );  
  QIconSet printSet( SmallIcon( "printer1" ), QIconSet::Automatic );
  QIconSet mountSet( SmallIcon( "hdd_mount" ), QIconSet::Automatic );    
  
  // Menu items
  m_menu->insertTitle( SmallIcon( "network" ), i18n( "Network" ), Title, 0 );
  m_menu->insertItem( reloadSet, i18n( "Rescan" ), sapp->core()->scanner(), SLOT( slotInit() ), CTRL+Key_R, Rescan, 1 );
  m_menu->insertItem( abortSet, i18n( "Abort" ), sapp, SLOT( slotAbort() ), CTRL+Key_A, Abort, 2 );
  m_menu->insertSeparator( 3 );
  m_menu->insertItem( searchSet, i18n( "Search" ), sapp->view(), SLOT( slotShowSearchTab() ), CTRL+Key_S, Search, 4 );
  m_menu->insertSeparator( 5 );
  m_menu->insertItem( bookmarkSet, i18n( "Add Bookmark" ), this, SLOT( slotAddBookmark() ), CTRL+Key_B, Bookmark, 6 );
  m_menu->insertItem( authSet, i18n( "Authentication" ), this, SLOT( slotAskPass() ), CTRL+Key_T, Password, 7 );
  m_menu->insertItem( previewSet, i18n( "Preview" ), this, SLOT( slotPreview() ), CTRL+Key_V, Preview, 8 );
  m_menu->insertItem( printSet, i18n( "Print File" ), this, SLOT( slotPrint() ), CTRL+Key_P, Print, 9 );
  m_menu->insertItem( mountSet, i18n( "Mount" ), this, SLOT( slotMountShare() ), CTRL+Key_M, Mount, 10 );

  // Enable and disable menu items.
  if ( sapp->core()->scannerIsRunning() || sapp->core()->mounterIsRunning() )
  {
    m_menu->setItemEnabled( Abort, true );
      
    if ( sapp->core()->scannerIsRunning() && sapp->core()->scannerState() != SCANNER_STOP )
    {
      m_menu->setItemEnabled( Rescan, false );
    }
    else
      m_menu->setItemEnabled( Rescan, true );
  }
  else
  {
    m_menu->setItemEnabled( Rescan, true );
    m_menu->setItemEnabled( Abort, false );
  }
  
  if ( item && item->depth() == 2 && col < columns() )
  {
    m_menu->setItemEnabled( Bookmark, true );
    m_menu->setItemEnabled( Password, true );
    
    if ( item->text( Type ).contains( "Printer" ) == 0 )
    {
      m_menu->setItemEnabled( Preview, true );
      m_menu->setItemEnabled( Print, false );
      m_menu->setItemEnabled( Mount, true );
    }
    else
    {
      m_menu->setItemEnabled( Preview, false );
      m_menu->setItemEnabled( Print, true );
      m_menu->setItemEnabled( Mount, false );    
    }
  }
  else if ( item && item->depth() == 1 && col < columns() )
  {
    m_menu->setItemEnabled( Bookmark, false );
    m_menu->setItemEnabled( Password, true );
    m_menu->setItemEnabled( Preview, false );
    m_menu->setItemEnabled( Print, false );
    m_menu->setItemEnabled( Mount, false );
  }
  else
  {
    m_menu->setItemEnabled( Bookmark, false );
    m_menu->setItemEnabled( Preview, false );
    m_menu->setItemEnabled( Print, false );
    m_menu->setItemEnabled( Mount, false );
    m_menu->setItemEnabled( Password, false );
  }

  m_menu->exec( point, 0 );
}


void Smb4KBrowserWidget::slotItemClicked( QListViewItem *item )
{
  // We only need this for the case, that the user clicks somewhere, where
  // NO item is. Smb4KNetworkTab needs it to clear the tab and the main window
  // to disable some actions in the tool bar.
  if ( !item && !selectedItem() )
  {
    sapp->bookmarksAddBookmark->setEnabled( false );
    sapp->networkPreview->setEnabled( false );
    sapp->networkPrint->setEnabled( false );
    sapp->sharesMount->setEnabled( false );
    sapp->networkAskPass->setEnabled( false );

    sapp->view()->networkTab()->display( item );
  }
}


void Smb4KBrowserWidget::slotWorkgroups( const QValueList<Smb4KWorkgroupItem *> &list )
{
  // Clear the network tab completely.
  sapp->view()->networkTab()->clear( 0 );

  // Clear the list view:
  clear();

  // Adjust the columns.
  for ( int index = 0; index < columns(); index++ )
  {
    if ( this->columnWidth( index ) != 0 )
      adjustColumn( index );
  }
  
  // Put the workgroups into the list view.
  if ( !list.isEmpty() )
  {
    for ( QValueList<Smb4KWorkgroupItem *>::ConstIterator it = list.begin(); it != list.end(); ++it )
    {
      if ( this->findItem( (*it)->workgroup(), Network, CaseSensitive|ExactMatch ) == 0 )
      {
        Smb4KBrowserWidgetItem *workgroup = new Smb4KBrowserWidgetItem( this, *it );
        workgroup->setExpandable( true );
      }
      else
      {
        continue;
      }
    }
  }
}



void Smb4KBrowserWidget::slotMembers( const QValueList<Smb4KHostItem *> &list )
{
  if ( !list.isEmpty() )
  {
    for ( QValueList<Smb4KHostItem *>::ConstIterator it = list.begin(); it != list.end(); ++it )
    {      
      QListViewItem *item = this->findItem( (*it)->name(), Network, CaseSensitive|ExactMatch );
        
      if ( item != 0 )
      {
        if ( !((*it)->ip()).isEmpty() && item->text( IP ) != (*it)->ip() )
          item->setText( IP, (*it)->ip() );
          
        if ( !((*it)->comment()).isEmpty() && item->text( Comment ) != (*it)->comment() )
          item->setText( Comment, (*it)->comment() );
            
        continue;
      }
      else
      {
        Smb4KBrowserWidgetItem *host = new Smb4KBrowserWidgetItem( this->findItem( (*it)->workgroup(), Network, ExactMatch|CaseSensitive ), *it );
        host->setExpandable( true );
      }            
    }
  }
}


void Smb4KBrowserWidget::slotShares( const QValueList<Smb4KShareItem *> &list )
{
  if ( !list.isEmpty() )
  {
    for ( QValueList<Smb4KShareItem *>::ConstIterator it = list.begin(); it != list.end(); ++it )
    {      
      QListViewItem *host = this->findItem( (*it)->host(), Network, ExactMatch|CaseSensitive );
      
      if ( host != 0 )
      {
        if ( !m_hidden && (*it)->name().endsWith( "$" ) )
        {
          continue;
        }
        else
        {
          if ( !m_ipc && (*it)->name().contains( "IPC" ) != 0 )
            continue;
            
          if ( !m_admin && (*it)->name().contains( "ADMIN" ) != 0 )
            continue;
        }
        
        if ( !m_printer && (*it)->name() == "Printer" )
          continue;
      
        QListViewItem *test = findItem( (*it)->name(), Network, ExactMatch|CaseSensitive );
        
        if ( test == 0 || test->parent() != host )
        {
          Smb4KBrowserWidgetItem *share = new Smb4KBrowserWidgetItem( host, *it );
          share->setExpandable( false );
          
          continue;
        }
        else if ( test != 0 && test->parent() == host )
        {
          if ( !((*it)->comment()).isEmpty() && test->text( Comment ) != (*it)->comment() )
            test->setText( Comment, (*it)->comment() );
            
          continue;
        }      
      }
    }
  }
}


void Smb4KBrowserWidget::slotCoreRunStateChanged()
{
  // If the menu is shown, enable or disable the
  // items on the fly.
  if ( m_menu->isShown() && ( sapp->core()->scannerIsRunning() || sapp->core()->mounterIsRunning() ) )
  {
    m_menu->setItemEnabled( Abort, true );
      
    if ( sapp->core()->scannerIsRunning() && sapp->core()->scannerState() != SCANNER_STOP )
    {
      m_menu->setItemEnabled( Rescan, false );
    }
    else
      m_menu->setItemEnabled( Rescan, true );
     
  }
  else
  {
    m_menu->setItemEnabled( Rescan, true );
    m_menu->setItemEnabled( Abort, false );
  }
}


void Smb4KBrowserWidget::slotMountShare()
{
  if ( currentItem() && currentItem()->depth() == 2 )
  {
    if ( currentItem()->text( Network ).stripWhiteSpace() == "homes" )
    {
      QString name = specifyUser( currentItem()->parent()->text( Network ) );
      if ( !name.isEmpty() )
        sapp->core()->mounter()->mountShare( currentItem()->parent()->parent()->text( Network ), currentItem()->parent()->text( Network ), currentItem()->parent()->text( IP ), name );
    }
    else
      sapp->core()->mounter()->mountShare( currentItem()->parent()->parent()->text( Network ), currentItem()->parent()->text( Network ), currentItem()->parent()->text( IP ), currentItem()->text( Network ) );
  }
}


void Smb4KBrowserWidget::slotAskPass()
{
  if ( currentItem() && currentItem()->depth() == 1 )
  {
    QString workgroup = currentItem()->parent()->text( Network );
    QString host = currentItem()->text( Network );
    QString share = QString::null;

    sapp->core()->passwdReader()->askpass( workgroup, host, share );
  }
  else if ( currentItem() && currentItem()->depth() == 2 )
  {
    QString workgroup = currentItem()->parent()->parent()->text( Network );
    QString host = currentItem()->parent()->text( Network );
    QString share = currentItem()->text( Network );

    sapp->core()->passwdReader()->askpass( workgroup, host, share );
  }
}


void Smb4KBrowserWidget::slotPreview()
{
  QString workgroup = currentItem()->parent()->parent()->text( Network );
  QString host = currentItem()->parent()->text( Network );
  QString ip = currentItem()->parent()->text( IP );
  QString share;
   
  if ( currentItem()->text( Network ).stripWhiteSpace() == "homes" )
    share = specifyUser( currentItem()->parent()->text( Network ) );
  else
    share = currentItem()->text( Network );
      
  Smb4KPreviewDialog *preview = new Smb4KPreviewDialog( workgroup, host, ip, share, this, 0, false );
  preview->show();
}


void Smb4KBrowserWidget::slotAddBookmark()
{
  if ( currentItem() && currentItem()->depth() == 2 )
  {
    if ( currentItem()->text( Network ).stripWhiteSpace() != "homes" )
      sapp->core()->bookmarkHandler()->writeBookmark( currentItem()->parent()->text( Network ), currentItem()->text( Network ) );
    else
    {
      QString share = specifyUser( currentItem()->parent()->text( Network ) );
      sapp->core()->bookmarkHandler()->writeBookmark( currentItem()->parent()->text( Network ), share );
    }
  }
}


void Smb4KBrowserWidget::slotAddIPAddress( Smb4KHostItem *item )
{
  // First find the host and check wether it is the right one.
  QListViewItem *hostItem = findItem( item->name(), Network, ExactMatch|CaseSensitive );

  if ( hostItem->parent()->text( Network ) == item->workgroup() )
  {
    hostItem->setText( IP, item->ip() );
    
    if ( this->columnWidth( IP ) != 0 )
      adjustColumn( IP );
  }
}


void Smb4KBrowserWidget::slotPrint()
{
  if ( currentItem() && currentItem()->depth() == 2 )
  {
    Smb4KPrintDialog *print = new Smb4KPrintDialog( currentItem()->parent()->parent()->text( Network ), currentItem()->parent()->text( Network ), currentItem()->parent()->text( IP ), currentItem()->text( Network ), currentItem()->text( Comment ), this, "PrintDialog" );
    print->exec();
  }
}



/***************************************************************************
 *                                                                         *
 *   The Smb4KBrowserWidgetItem class.                                     *
 *                                                                         *
 ***************************************************************************/


Smb4KBrowserWidgetItem::Smb4KBrowserWidgetItem( QListView *parent, Smb4KWorkgroupItem *item ) : QListViewItem( parent, item->workgroup() ), m_workgroup( m_workgroup ), m_host( NULL ), m_share( NULL )
{
  setPixmap( 0, SmallIcon( "network" ) );
}


Smb4KBrowserWidgetItem::Smb4KBrowserWidgetItem( QListViewItem *parent, Smb4KHostItem *item ) : QListViewItem( parent, item->name(), QString::null, item->ip(), item->comment() ), m_workgroup( NULL ), m_host( item ), m_share( NULL )
{
  setPixmap( 0, SmallIcon( "network_local" ) );
}
  

Smb4KBrowserWidgetItem::Smb4KBrowserWidgetItem( QListViewItem *parent, Smb4KShareItem *item ) : QListViewItem( parent, item->name(), item->type(), QString::null, item->comment() ), m_workgroup( NULL ), m_host( NULL ), m_share( item )
{
  if ( item->type().contains( "Printer" ) )
    setPixmap( 0, SmallIcon( "printer1" ) );
  else
    setPixmap( 0, SmallIcon( "folder_green" ) );
}


Smb4KBrowserWidgetItem::~Smb4KBrowserWidgetItem()
{
}


void Smb4KBrowserWidgetItem::paintCell( QPainter *p, const QColorGroup &cg, int column, int width, int align )
{
  QListViewItem::paintCell( p, cg, column, width, align );
}


#include "smb4kbrowserwidget.moc"
