/***************************************************************************
                          smb4kprint  -  description
                             -------------------
    begin                : Tue Mar 30 2004
    copyright            : (C) 2004 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// Qt includes
#include <qtimer.h>
#include <qfile.h>

// KDE includes
#include <klocale.h>
#include <kapplication.h>
#include <kmessagebox.h>
#include <kurl.h>

// system includes
#include <stdlib.h>

// application specific includes
#include "smb4kprint.h"
#include "smb4kcore.h"
#include "smb4kdefs.h"


Smb4KPrint::Smb4KPrint( QObject *parent, const char *name ) : QObject( parent, name )
{
  m_proc = new KProcess( this, "PrintProcess" );
  m_proc->setUseShell( true );
  
  m_config = kapp->config();
  
  m_item = new KFileItem( KFileItem::Unknown, KFileItem::Unknown, KURL(), false );
  
  connect( m_proc, SIGNAL( receivedStdout( KProcess *, char *, int ) ), this, SLOT( slotReceivedStdout( KProcess *, char *, int ) ) );
  connect( m_proc, SIGNAL( receivedStderr( KProcess *, char *, int ) ), this, SLOT( slotReceivedStderr( KProcess *, char *, int ) ) );
  connect( m_proc, SIGNAL( processExited( KProcess * ) ),               this, SLOT( slotProcessExited( KProcess * ) ) );
}


Smb4KPrint::~Smb4KPrint()
{
  abort();
  delete m_item;
}


/****************************************************************************
   Initializes the printing of a file. This function looks for the mime 
   type and takes care of the further processing.   
****************************************************************************/


bool Smb4KPrint::init( Smb4KPrintInfo *info )
{
  m_info = info;
  
  bool ok = true;
  
  if ( QFile::exists( m_info->path() ) )
  {  
    // We want to determine the mimetype of the incoming 
    // file. Let's use KFileItem for this, because KFileMetaInfo
    // does not work very reliable.
    KURL u;
    u.setPath( m_info->path() );
    
    m_item->assign( KFileItem( KFileItem::Unknown, KFileItem::Unknown, u, false ) );
    
    QString type = m_item->mimetype();
    
//    qDebug( type );
    
    if ( type == "application/postscript" || 
         type == "application/pdf" ||
         type.startsWith( "image" ) )
    {
      startPrinting();
    } 
    else if ( type == "application/x-dvi" && m_dvips )
    {
      convertDVIToPS();
    }
    else if ( ( type.startsWith( "text" ) || type.startsWith( "message" ) ) && m_enscript )
    {
      if ( KMessageBox::warningContinueCancel( (QWidget *)this, QString( i18n( "The mimetype (%1) indicates that this is a text file. Printing it, however, might fail. Do you want to continue?" ) ).arg( type ) ) == KMessageBox::Continue )
      {
        convertTextToPS();
      }
    }
    else
      emit error( ERROR_MIMETYPE_NOT_SUPPORTED, type );
  }
  else
  {
    emit error( ERROR_FILE_NOT_FOUND, m_info->path() );
    ok = false; 
  }
  
  return ok;
}


/****************************************************************************
   Aborts the current process.
****************************************************************************/

void Smb4KPrint::abort()
{
  if ( m_proc->isRunning() )
    m_proc->kill();
}


/****************************************************************************
   Starts the actual printing.
****************************************************************************/

void Smb4KPrint::startPrinting()
{ 
  QString command( "for ((i=0;i<"+QString( "%1" ).arg( m_info->copies() )+";i++)) ; do " );
  
  command.append( "cat "+m_info->path()+" | smbclient //"+KProcess::quote( m_info->host() )+"/"+KProcess::quote( m_info->printer() )+" -N -W "+KProcess::quote( m_info->workgroup() ) );
    
  // Get the authentication data:
  Smb4KAuthInfo *auth = ((Smb4KCore *)parent())->passwdReader()->getAuth( m_info->workgroup(), m_info->host(), m_info->printer() );
    
  if ( !auth->user().isEmpty() )
  {
    command.append( " -U "+KProcess::quote( auth->user() ) );
      
    if ( !auth->password().isEmpty() )
      command.append( "%"+auth->maskedPassword() );
  }
  else
    command.append( " -U%" );
     
  if ( !m_info->ip().isEmpty() )
    command.append( " -I "+KProcess::quote( m_info->ip() ) );
      
  // Advanced options. 
  if ( !m_SMBOptions.isEmpty() )
    command.append( m_SMBOptions );
    
  if ( m_item->mimetype().startsWith( "image" ) )
    command.append( " -c 'printmode graphics ; print -'" );
  else
    command.append( " -c 'printmode text ; print -'" );
  
  command.append( " ; done" );
  
  *m_proc << command;
  
  startProcess( Print );
}


/****************************************************************************
   Converts DVI files to PS using dvips.
****************************************************************************/

void Smb4KPrint::convertDVIToPS()
{
  QString path = m_info->path().section( "/", 0, -2 );
  QString file = m_info->path().section( "/", -1, -1 );
  
  *m_proc << "cd "+KProcess::quote( path )+" && dvips -P pdf -o /tmp/smb4k_print_$USER.ps "+KProcess::quote( file );
  startProcess( Convert );
}


/****************************************************************************
   Converts text files to PS using enscript.
****************************************************************************/

void Smb4KPrint::convertTextToPS()
{
  *m_proc << "enscript -1 -B --ps-level=2 -o /tmp/smb4k_print_$USER.ps "+KProcess::quote( m_info->path() );
  startProcess( Convert );
}


/****************************************************************************
   Starts any process of this class.
****************************************************************************/

void Smb4KPrint::startProcess( int state )
{
  m_state = state;
  m_buffer = QString::null;
  
  if ( m_state == Print )
    emit running( PRINT_SEND_FILE, true );
  else
    emit running( PRINT_CONVERT_FILE, true );
    
  m_proc->start( KProcess::NotifyOnExit, KProcess::AllOutput );
}


/****************************************************************************
   If the KProcess ended, this function takes care of the further
   processing.
****************************************************************************/

void Smb4KPrint::endProcess()
{
  switch ( m_state )
  {
    case Print:
      endPrintProcess();
      emit running( PRINT_STOP, false );
      break;
    case Convert:
      endConversionProcess();
      break;
    default:
      break;
  }
  
  m_state = Idle;
  m_proc->clearArguments();
}


/****************************************************************************
   Post-processing of any print job.
****************************************************************************/

void Smb4KPrint::endPrintProcess()
{
  if ( m_buffer.contains( "NT_STATUS", true ) != 0 || m_buffer.contains( "Connection to", true ) != 0  || 
       m_buffer.contains( "tree connect failed:", true ) != 0 )
  {
    if ( m_buffer.contains( "NT_STATUS_ACCESS_DENIED" ) != 0 || m_buffer.contains( "NT_STATUS_LOGON_FAILURE" ) != 0 )
    {
      if ( ((Smb4KCore *)parent())->passwdReader()->askpass( m_info->workgroup(), m_info->host(), m_info->printer() ) )
      {
        QTimer::singleShot( 50, this, SLOT( slotRetry() ) );
      }
    }
    else
    {
//      qDebug( m_buffer );
    }
  }
  else
  {
    QStringList list = QStringList::split( '\n', m_buffer, false );
    
    if ( list.last().startsWith( "putting file" ) )
    {
      QFile::remove( "/tmp/smb4k_print_"+QString( "%1" ).arg( getenv( "USER" ) )+".ps" );
      emit running( PRINT_STOP, false );
    }
  }
}


/****************************************************************************
   Ends the conversion process.
****************************************************************************/

void Smb4KPrint::endConversionProcess()
{
  if ( m_buffer.contains( "command not found", true ) != 0 )
  {
    QString missing = m_buffer.section( ":", -2, -2 ).section( ":", -1, -1 ).stripWhiteSpace();
    KMessageBox::error( (QWidget *)this, QString( i18n( "The command %1 could not be found!\n%2 won't be printed." ) ).arg( missing, m_info->path() ) );
    
    emit running( PRINT_STOP, false );
    m_state = Idle;
    m_proc->clearArguments();
  }
  else
  {  
    // Set the URL to the newly created PS file:
    m_info->setPath( "/tmp/smb4k_print_"+QString( "%1" ).arg( getenv( "USER" ) )+".ps" );
  
    QTimer::singleShot( 50, this, SLOT( slotRetry() ) );
  }
}


/****************************************************************************
   Reads the options.
****************************************************************************/

void Smb4KPrint::readOptions()
{
  //
  // smbclient's options
  //
  m_config->setGroup( "Samba" );

  m_SMBOptions = QString::null;

  if ( !m_config->readEntry( "Client Resolve Order", QString::null ).isEmpty() )
    m_SMBOptions.append( QString( " -R \"%1\"" ).arg( m_config->readEntry( "Client Resolve Order", QString::null ) ) );

  if ( !m_config->readEntry( "Port", QString::null ).isEmpty() )
    m_SMBOptions.append( QString( " -p %1" ).arg( m_config->readNumEntry( "Port", 139 ) ) );

  if ( !m_config->readEntry( "Client Buffer Size", QString::null ).isEmpty() )
    m_SMBOptions.append( QString( " -b %1" ).arg( m_config->readNumEntry( "Client Buffer Size", 65520 ) ) );

  if ( m_config->readBoolEntry( "Use Kerberos", false ) )
  {
    m_config->setGroup( "System" );

    if ( m_config->readEntry( "Samba" ).startsWith( "3" ) )
      m_SMBOptions.append( " -k" );

    m_config->setGroup( "Samba" );
  }

  if ( !m_config->readEntry( "NetBIOS Name", QString::null ).isEmpty() )
    m_SMBOptions.append( QString( " -n \"%1\"" ).arg( m_config->readEntry( "NetBIOS Name", QString::null ) ) );

  if ( !m_config->readEntry( "NetBIOS Scope", QString::null ).isEmpty() )
    m_SMBOptions.append( QString( " -i \"%1\"" ).arg( m_config->readEntry( "NetBIOS Scope", QString::null ) ) );

  if ( !m_config->readEntry( "Socket Options", QString::null ).isEmpty() )
    m_SMBOptions.append( QString( " -O \"%1\"" ).arg( m_config->readEntry( "Socket Options", QString::null ) ) );
    
  m_config->setGroup( "Printing" );
  m_enscript = m_config->readBoolEntry( "Enscript", false );
  m_dvips = m_config->readBoolEntry( "DVIPS", false );
}


/////////////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////

void Smb4KPrint::slotReceivedStdout( KProcess *, char *buf, int len )
{
  m_buffer.append( QString::fromLocal8Bit( buf, len ) );
}


void Smb4KPrint::slotReceivedStderr( KProcess *, char *buf, int len )
{
  m_buffer.append( QString::fromLocal8Bit( buf, len ) );
}


void Smb4KPrint::slotProcessExited( KProcess * )
{
  endProcess();
}


void Smb4KPrint::slotRetry()
{
  startPrinting();
}

#include "smb4kprint.moc"
