/***************************************************************************
    smb4kpasswdreader.cpp  -  Reads passwords. Used by several classes.
                             -------------------
    begin                : Mit Mai 14 2003
    copyright            : (C) 2003 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// Qt includes
#include <qlayout.h>
#include <qlabel.h>
#include <qcheckbox.h>

#ifdef __FreeBSD__
#include <qdir.h>
#include <qfile.h>
#include <qstringlist.h>
#include <qurlinfo.h>
#endif

// KDE includes
#include <kapplication.h>
#include <kstandarddirs.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kapplication.h>
#include <kurl.h>
#include <klineedit.h>
#include <kcombobox.h>

// system includes
#include <cstdlib>

// application specific includes
#include "smb4kpasswdreader.h"
#include "smb4kcore.h"


Smb4KPasswdReader::Smb4KPasswdReader( QWidget *parent, const char *name ) : QObject( parent, name )
{
  // First we need the directory.
  KStandardDirs *stddir = new KStandardDirs();
  QString dir = locateLocal( "appdata", QString::null, KGlobal::instance() );
  if ( !stddir->exists( dir ) )
    stddir->makeDir( dir );

  // Do not use the KConfig object, that is provided by kapp->config().
  // We want a separate one for the passwords.
  m_config = new KConfig( "passwords", false, true, "appdata" );
  // Give it strict permissions:
  m_config->setFileWriteMode( 00600 );

  // Get or create the key and convert the not obfuscated passwords,
  // if there are any:
  KConfig *global_config = kapp->config();
  global_config->setGroup( "Authentication" );

  m_key = global_config->readNumEntry( "Random Number", rand() );
  global_config->writeEntry( "Random Number", m_key );

  read();
  
#ifdef __FreeBSD__
  writeToSMBConfFile();
#endif
}


Smb4KPasswdReader::~Smb4KPasswdReader()
{
  // Delete all pointers in the auth data list.
  for ( QValueList<Smb4KAuthInfo *>::Iterator it = m_list.begin(); it != m_list.end(); ++it )
    delete *it;
}


/****************************************************************************
   Functions that read the authentication info.
****************************************************************************/

Smb4KAuthInfo *Smb4KPasswdReader::getAuth( const QString &workgroup, const QString &host, const QString &share )
{
  Smb4KAuthInfo *auth = new Smb4KAuthInfo( workgroup, host, share ); 
  
  if ( !m_list.isEmpty() )
  {
    for ( QValueList<Smb4KAuthInfo *>::ConstIterator it = m_list.begin(); it != m_list.end(); ++it )
    {
      if ( !workgroup.isEmpty() )
      {
        if ( (*it)->workgroup() == workgroup && (*it)->host() == host )
        {
          if ( !share.isEmpty() )
          {
            if ( (*it)->share().upper() == share.upper() )
            {
              auth->setUser( (*it)->user() );
              auth->setPassword( (*it)->password() );
          
              break;
            }
            else if ( (*it)->share() == "*" && auth->user().isEmpty() )
            {
              auth->setUser( (*it)->user() );
              auth->setPassword( (*it)->password() );
              
              continue;            
            }
            else
              continue;
          }
          else
          {
            if ( (*it)->share() == "*" )
            {
              auth->setUser( (*it)->user() );
              auth->setPassword( (*it)->password() );
              
              break;
            }
            else
              continue;
          }
        }
        else
          continue;
      }
      else
      {
        if ( (*it)->host() == host )
        {
          auth->setUser( (*it)->user() );
          auth->setPassword( (*it)->password() );
          
          if ( (*it)->share() == "*" )
            break;
          else
            continue;
        }
        else
          continue;
      }
    }
  }

  if ( auth->user().isEmpty() )
  {
    kapp->config()->setGroup( "Authentication" );
    
    if ( kapp->config()->readBoolEntry( "Default Authentication", false ) )
    {
      m_config->setGroup( "Default" );
      QString pass = obfuscate( m_config->readEntry( "Password", QString::null ), false );
      auth->setUser( m_config->readEntry( "User", QString::null ) );
      auth->setPassword( pass );
    }
    else
    {
      auth->setUser( QString::null );
      auth->setPassword( QString::null );
    }
  }
  
  return auth; 
}


/****************************************************************************
   Reads the list of all authentication entries that are saved in the
   password file.
****************************************************************************/

void Smb4KPasswdReader::read()
{
  if ( m_config->hasGroup( "Logins" ) )
  {
    QStringList loginData;
    m_config->setGroup( "Logins" );
    for ( int pos = 0; ; pos++ )
    {
      loginData = m_config->readListEntry( QString( "%1" ).arg( pos, 0, 10 ).stripWhiteSpace(), ',' );

      // "Decrypt" the password.
      if ( !loginData.isEmpty() )
      {
        m_list.append( new Smb4KAuthInfo( loginData[0], loginData[1], loginData[2], loginData[3], obfuscate( loginData[4], false ) ) );
        
        loginData.clear();
      }
      else
        break;
    }
  }
}


/****************************************************************************
   Reads the default auth data.
****************************************************************************/

Smb4KAuthInfo *Smb4KPasswdReader::readDefaultAuthData()
{
  m_config->setGroup( "Default" );
  Smb4KAuthInfo *auth = new Smb4KAuthInfo( QString::null, QString::null, QString::null, m_config->readEntry( "User", QString::null ), obfuscate( m_config->readEntry( "Password", QString::null ), false ) );
  
  return auth;  
}


/****************************************************************************
   Adds authentication information to the existing list or modifies 
   existing ones.
****************************************************************************/

void Smb4KPasswdReader::commit( Smb4KAuthInfo *auth )
{
  bool exists = false;
  
  for ( QValueList<Smb4KAuthInfo *>::Iterator it = m_list.begin(); it != m_list.end(); ++it )
  {
    if ( (*it)->workgroup() == auth->workgroup() && (*it)->host() == auth->host() && (*it)->share() == auth->share() )
    {
      (*it)->setUser( auth->user() );
      (*it)->setPassword( auth->password() );
      
      exists = true;
      
      break;
    }
    else
      continue;
  }
  
  if ( !exists )
    m_list.append( new Smb4KAuthInfo( auth->workgroup(), auth->host(), auth->share(), auth->user(), auth->password() ) );
  
  write();
}


/****************************************************************************
   Writes the auth data that is passed by the configuration dialog on
   exit.
****************************************************************************/

void Smb4KPasswdReader::write()
{
  // Clear all entries.
  if ( m_config->hasGroup( "Logins" ) )
  {
    m_config->deleteGroup( "Logins", true, false );
    m_config->sync();
  }

  m_config->setGroup( "Logins" );
  
  int index = 0;
  
  for ( QValueList<Smb4KAuthInfo *>::ConstIterator it = m_list.begin(); it != m_list.end(); ++it )
  {
    QStringList item;
    item << (*it)->workgroup() << (*it)->host() << (*it)->share() << (*it)->user() << obfuscate( (*it)->password(), true );
    m_config->writeEntry( QString( "%1" ).arg( index++ ), item, ',' );
  }
  
  m_config->sync();
  
#ifdef __FreeBSD__
  writeToSMBConfFile();
#endif
}


/****************************************************************************
   Writes the default auth data that is passed by the configuration dialog
   on exit.
****************************************************************************/

void Smb4KPasswdReader::writeDefaultAuthData( Smb4KAuthInfo *auth )
{
  // Obfuscate the password:
  QString pass = obfuscate( auth->password(), true );
  
  // Now write the data:
  m_config->setGroup( "Default" );
  m_config->writeEntry( "User", auth->user() );
  m_config->writeEntry( "Password", pass );
  
  m_config->sync();
}


/****************************************************************************
   This function shows the askpass dialog.
****************************************************************************/

bool Smb4KPasswdReader::askpass( const QString &workgroup, const QString &host, const QString &share )
{
  passwdDlg = new KDialogBase( KDialogBase::Plain, i18n( "Authentication" ), KDialogBase::Ok|KDialogBase::Cancel, KDialogBase::Ok, (QWidget *)parent(), "PasswordDlg", false, true );

  //
  // Set up the ask pass dialog.
  //
  QFrame *frame = passwdDlg->plainPage();
  QGridLayout *layout = new QGridLayout( frame );
  layout->setSpacing( 5 );

  QLabel *pic = new QLabel( frame );
  pic->setPixmap( DesktopIcon( "personal" ) );
  pic->setMargin( 10 );
  
  layout->addWidget( pic, 0, 0, 0 );

  QWidget *shareData = new QWidget( frame );
  QGridLayout *grid = new QGridLayout( shareData );
  grid->setSpacing( 5 );
  grid->setMargin( 0 );

  QLabel *label1 = new QLabel( i18n( "Workgroup:" ), shareData );
  QLabel *workgroupLabel = new QLabel( shareData, "Workgroup" );
  QLabel *label2 = new QLabel( i18n( "Host:" ), shareData );
  QLabel *hostLabel = new QLabel( shareData, "Host" );
  QLabel *label3 = new QLabel( i18n( "Share:" ), shareData );
  QLabel *shareLabel = new QLabel( shareData );

  grid->addWidget( label1, 1, 0, 0 );
  grid->addWidget( workgroupLabel, 1, 1, 0 );
  grid->addWidget( label2, 2, 0, 0 );
  grid->addWidget( hostLabel, 2, 1, 0 );
  grid->addWidget( label3, 3, 0, 0 );
  grid->addWidget( shareLabel, 3, 1, 0 );
  
  layout->addMultiCellWidget( shareData, 0, 0, 1, 3, 0 );

  QLabel *user = new QLabel( i18n( "User:" ), frame );
  
  layout->addWidget( user, 1, 0, 0 );
 
  // This is perhaps a workaround, but it works...
  KLineEdit *userEdit = new KLineEdit( frame );
  KComboBox *userCombo = new KComboBox( frame, "Combo" );
 
    
  if ( share != "homes" )
  {
    // We don't need this anymore.
    delete userCombo;    
    
    layout->addMultiCellWidget( userEdit, 1, 1, 1, 4, 0 );
  }
  else
  {
    // We don't need this anymore.
    delete userEdit;
    
    userCombo->setEditable( true );
    layout->addMultiCellWidget( userCombo, 1, 1, 1, 4, 0 );
  }
  
  QLabel *pass = new QLabel( i18n( "Password:" ), frame );
  layout->addWidget( pass, 2, 0, 0 );
  
  m_passwdEdit = new KLineEdit( frame );
  m_passwdEdit->setEchoMode( KLineEdit::Password );
  layout->addMultiCellWidget( m_passwdEdit, 2, 2, 1, 4, 0 );

  QCheckBox *use = new QCheckBox( i18n( "Use authentication for all shares." ), frame );
  layout->addMultiCellWidget( use, 3, 3, 0, 3, 0 );

  QSpacerItem *spacer1 = new QSpacerItem( 10, 10, QSizePolicy::Expanding, QSizePolicy::Preferred );
  layout->addItem( spacer1, 0, 4 );

  //
  // Data processing and completion of the dialog.
  //
 
  if ( !workgroup.isEmpty() )
    workgroupLabel->setText( workgroup );
  else
    workgroupLabel->setText( "---" );
    
  hostLabel->setText( host );
  
  if ( share.isEmpty() )
  {
    use->setEnabled( false );
    use->setChecked( true );
    shareLabel->setText( "---" );
  }
  else
  {
    use->setEnabled( true );
    shareLabel->setText( share );
  }
  
  // The new authentication information:
  Smb4KAuthInfo *newAuth = new Smb4KAuthInfo( workgroup, host, share );
    
  // List for the logins, if the specified share was a 'homes'
  // share.
  QStringList logins;
    
  if ( share != "homes" )
  {
    // Get the auth data.
    Smb4KAuthInfo *auth = getAuth( workgroup, host, share );
 
    // If this share is one derived from a homes share we
    // have to get the user name from the config file.     
    if ( auth->user().isEmpty() && auth->password().isEmpty() )
    {
      if ( kapp->config()->hasGroup( "Homes Shares" ) )
      {
        kapp->config()->setGroup( "Homes Shares" );
        QString name = *( kapp->config()->readListEntry( host, ',' ).find( share ) );
        if ( !name.isEmpty() )
          auth->setUser( name );
      }
    }
       
    // Fill the line edits:
    if ( !auth->user().isEmpty() )
      userEdit->setText( auth->user() );
       
    if ( !auth->password().isEmpty() )
      m_passwdEdit->setText( auth->password() );
        
    // Set the cursor to the right line edit:
    if ( userEdit->text().stripWhiteSpace().isEmpty() || ( !userEdit->text().stripWhiteSpace().isEmpty() && !m_passwdEdit->text().stripWhiteSpace().isEmpty() ) )
      userEdit->setFocus();
    else
      m_passwdEdit->setFocus();
  }
  else
  {
    // Read the list of logins, that were already defined
    // for this 'homes' share.
    if ( kapp->config()->hasGroup( "Homes Shares" ) )
    {
      kapp->config()->setGroup( "Homes Shares" );
      if ( kapp->config()->hasKey( host ) )
        logins = kapp->config()->readListEntry( host, ',' );
    }

    if ( !logins.isEmpty() )
      userCombo->insertStringList( logins, 0 );

    userCombo->setCurrentText( QString::null );

    // Do the last things before showing.
    userCombo->setFocus();
    
    connect( userCombo, SIGNAL( activated( const QString & ) ), SLOT( slotGetPassword( const QString & ) ) );
  }
  
  passwdDlg->setFixedSize( passwdDlg->sizeHint() );
  
  bool new_passwd = false;
    
  // Show it.
  if ( passwdDlg->exec() == KDialogBase::Accepted )
  {
    if ( share != "homes" )
    {
      if ( share.isEmpty() || use->isChecked() )
        newAuth->setShare( "*" );
      else
        newAuth->setShare( share );
        
      // Since we want to allow the user to provide an empty password,
      // only the user name is checked to be not empty.
      if ( !userEdit->text().stripWhiteSpace().isEmpty() )
      {
        newAuth->setUser( userEdit->text().stripWhiteSpace() );
        newAuth->setPassword( m_passwdEdit->text().stripWhiteSpace() );
        commit( newAuth );     
        new_passwd = true;
      }
    }
    else
    {
      // Same as above:
      if ( !userCombo->currentText().stripWhiteSpace().isEmpty() )
      {
        // Save the list of users to the config file.
        kapp->config()->setGroup( "Homes Shares" );
        
        if ( logins.contains( userCombo->currentText().stripWhiteSpace() ) == 0 )
          logins.append( userCombo->currentText().stripWhiteSpace() );
          
        kapp->config()->writeEntry( host, logins, ',' );
          
        // Put the share in the authentication data list:
        newAuth->setShare( userCombo->currentText().stripWhiteSpace() );
        
        // Save the authentication data:
        newAuth->setUser( userCombo->currentText().stripWhiteSpace() );
        newAuth->setPassword( m_passwdEdit->text().stripWhiteSpace() );
        commit( newAuth );
        
        new_passwd = true;
      }
    }
  }  
  
  return new_passwd;
}


/****************************************************************************
   Updates the authentication information list.
****************************************************************************/

void Smb4KPasswdReader::updateAuthList( const QValueList<Smb4KAuthInfo *> &list )
{
  m_list = list;
  
  write();
}


/****************************************************************************
   Obfuscation of the passwords.
****************************************************************************/

const QString Smb4KPasswdReader::obfuscate( const QString &password, bool encrypt )
{
  QString passwd;
  
  if ( encrypt )
  {
    uint index( 0 );
    while ( index < password.length() )
    {
      passwd[index] = (char)( (int)password[index]+m_key%62 );
      index++;
    }
  }
  else if ( !encrypt )
  {
    uint index( 0 );
    while ( index < password.length() )
    {
      passwd[index] = (char)( (int)password[index]-m_key%62 );
      index++;
    }
  }
    
  return passwd;
}


#ifdef __FreeBSD__

/****************************************************************************
   Write to the ~/.nsmbrc file
****************************************************************************/

void Smb4KPasswdReader::writeToSMBConfFile()
{
  QDir::setCurrent( QDir::homeDirPath() );
  
  QFile file( ".nsmbrc" );
  
  QStringList contents;
  
  if ( file.exists() )
  {
    if ( file.open( IO_ReadOnly ) )
    {
      QTextStream ts( &file );

      while ( !ts.atEnd() )
        contents.append( ts.readLine().stripWhiteSpace() );

      file.close();
    }
     
    if ( contents.grep( "[default]" ).count() == 0 )
    {
      contents.append( "[default]" );
      
      kapp->config()->setGroup( "Browse Options" );
      
      if ( !kapp->config()->readEntry( "WINS Server" ).isEmpty() )
        contents.append( "nbns="+kapp->config()->readEntry( "WINS Server" ) );
        
      QMap<QString, QString> options = ((Smb4KCore *)parent())->fileIO()->getSMBOptions();
      
      if ( !options["workgroup"].isEmpty() )
        contents.append( "workgroup="+options["workgroup"] );      
        
      contents.append( QString::null );
    }    
  }
  else
  {
    contents.append( "[default]" );
    
    kapp->config()->setGroup( "Browse Options" );
    
    if ( !kapp->config()->readEntry( "WINS Server" ).isEmpty() )
      contents.append( "nbns="+kapp->config()->readEntry( "WINS Server" ) );
      
    QMap<QString, QString> options = ((Smb4KCore *)parent())->fileIO()->getSMBOptions();
    
    if ( !options["workgroup"].isEmpty() )
      contents.append( "workgroup="+options["workgroup"] );
      
    contents.append( QString::null );
  }
  
  // Now write the new data to ~/.nsmbrc. Passwords will be stored in plain text, but I DON'T 
  // CARE! I tried several hours to make encryption work with smbutil in Smb4KShellIO, but 
  // I got only crashes.
  
  for ( QValueListIterator<Smb4KAuthInfo *> it = m_list.begin(); it != m_list.end(); ++it )
  {
    if ( (*it)->share().isEmpty() )
    {
      continue;
    }
    else
    {
      if ( contents.grep( "["+(*it)->host()+":"+(*it)->user()+":"+(*it)->share()+"]", false ).count() == 0 )
      {
        contents.append( "["+(*it)->host().upper()+":"+(*it)->user().upper()+":"+(*it)->share().upper()+"]" );
        contents.append( "workgroup="+(*it)->workgroup().upper() );
        contents.append( "password="+(*it)->password() );
        contents.append( QString::null );      
      }
      else
      {
        QStringList::Iterator i = contents.find( contents.grep( "["+(*it)->host()+":"+(*it)->user()+":"+(*it)->share()+"]", false ).first() );
        ++i;
        
        while ( !(*i).stripWhiteSpace().isEmpty() && !(*i).stripWhiteSpace().startsWith( "[" ) )
        {
          if( (*i).startsWith( "password", false ) )
          {
            QString pwd = (*i).section( "=", 1, 1 ).stripWhiteSpace();
            
            if ( pwd != (*it)->password() )
              (*i).replace( pwd, (*it)->password() );
          }
          
          ++i;
        }
      }
    }
  }
  
  QDir::setCurrent( QDir::homeDirPath() );
  
  // Write to the file.
  if ( file.open( IO_WriteOnly ) )
  {
    QTextStream ts( &file );
    
    for ( QStringList::ConstIterator it = contents.begin(); it != contents.end(); ++it )
      ts << *it << endl;
    
    file.close();
  }  
    
  // Get minimal security: Fix permissions.
  QString path( QDir::homeDirPath()+"/"+file.name() );
  chmod( path.ascii(), 00600 ) != 0;    
}

#endif


/////////////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////


/****************************************************************************
   Returns the password for a given user. This is mainly used by the 
   askpass dialog.
****************************************************************************/

void Smb4KPasswdReader::slotGetPassword( const QString &user )
{
  if ( passwdDlg )
  {
    QString wg = ((QLabel *)passwdDlg->child( "Workgroup", "QLabel", true ))->text();
    QString host = ((QLabel *)passwdDlg->child( "Host", "QLabel", true ))->text();
    Smb4KAuthInfo *auth = getAuth( wg, host, user );
    
    m_passwdEdit->clear();
    
    if ( !auth->password().isEmpty() )
      m_passwdEdit->setText( auth->password() );
  }
}


#include "smb4kpasswdreader.moc"
