-- This file is part of SmartEiffel The GNU Eiffel Compiler Tools and Libraries.
-- See the Copyright notice at the end of this file.
--
class LOOP_VARIANT
	--
	-- For the variant part of a loop instruction.
	--

inherit
	VISITABLE
insert
	GLOBALS

creation {ANY}
	make

feature {E_LOOP, LOOP_VARIANT, LOOP_VARIANT_VISITOR}
	comment1: COMMENT
			-- If any the one after the %"variant%" keyword.

	tag: TAG_NAME
			-- If any, the optional tag name.

	comment2: COMMENT
			-- If any the one which can be between `tag' and `expression'.

	expression: EXPRESSION
			-- The variant INTEGER expression itself.

	start_position: POSITION is
		do
			if tag /= Void then
				Result := tag.start_position
			else
				Result := expression.start_position
			end
		end

feature {E_LOOP, LOOP_VARIANT}
	collect (t: TYPE) is
		require
			has_been_specialized
			t.feature_collection_done
			smart_eiffel.status.is_collecting
			not smart_eiffel.status.is_specializing
		local
			exp: TYPE
		do
			exp := expression.collect(t)
			if not exp.is_integer then
				error_handler.append("%"variant%" must be followed by an INTEGER expression.")
				error_handler.add_position(expression.start_position)
				error_handler.print_as_fatal_error
			end
		end

	adapt_for (t: TYPE): like Current is
		local
			exp: like expression
		do
			exp := expression.adapt_for(t)
			if exp /= expression then
				Result := twin
				Result.init(exp)
			else
				Result := Current
			end
		end

	use_current (type: TYPE): BOOLEAN is
		do
			Result := expression.use_current(type)
		end

	pretty (indent_level: INTEGER) is
		do
			if comment1 /= Void then
				comment1.pretty(indent_level)
			end
			if tag /= Void then
				pretty_printer.set_indent_level(indent_level)
				pretty_printer.put_string(tag.to_string)
				pretty_printer.put_string(": ")
			end
			if comment2 /= Void then
				comment2.pretty(indent_level)
			end
			expression.pretty(indent_level)
		end

	specialize_in (type: TYPE): like Current is
		require
			type /= Void
		local
			exp: like expression
		do
			exp := expression.specialize_in(type)
			if exp /= expression then
				Result := twin
				Result.init(exp)
			else
				Result := Current
			end
		ensure
			Result /= Current implies expression /= Result.expression
			Result.has_been_specialized
		end

	specialize_thru (parent_type: TYPE; parent_edge: PARENT_EDGE; new_type: TYPE): like Current is
		require
			parent_type /= Void
			parent_edge /= Void
			new_type /= Void
			new_type /= parent_type
			has_been_specialized
		local
			exp: like expression
		do
			exp := expression.specialize_thru(parent_type, parent_edge, new_type)
			if exp /= expression then
				Result := twin
				Result.init(exp)
			else
				Result := Current
			end
		ensure
			Result /= Current implies expression /= Result.expression
			has_been_specialized
			Result.has_been_specialized
		end

	specialize_2 (type: TYPE): like Current is
		require
			has_been_specialized
			not smart_eiffel.status.is_specializing
		local
			exp: like expression
		do
			exp := expression.specialize_2(type)
			if exp /= expression then
				Result := twin
				Result.init(exp)
			else
				Result := Current
			end
		ensure
			has_been_specialized
			Result.has_been_specialized
		end

	has_been_specialized: BOOLEAN is
		do
			Result := expression.has_been_specialized
		ensure
			Result
		end

feature {ANY}
	accept (visitor: LOOP_VARIANT_VISITOR) is
		do
			visitor.visit_loop_variant(Current)
		end

feature {LOOP_VARIANT}
	init (exp: like expression) is
		require
			exp /= Void
		do
			expression := exp
		ensure
			expression = exp
		end

feature {}
	make (c1: COMMENT; t: like tag; c2: COMMENT; exp: like expression) is
		require
			exp /= Void
		do
			comment1 := c1
			tag := t
			comment2 := c2
			expression := exp
		ensure
			comment1 = c1
			tag = t
			comment2 = c2
			expression = exp
		end

invariant
	expression /= Void

end -- class LOOP_VARIANT
--
-- ------------------------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- SmartEiffel is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License,
-- as published by the Free Software Foundation; either version 2, or (at your option) any later version.
-- SmartEiffel is distributed in the hope that it will be useful but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. You should have
-- received a copy of the GNU General Public License along with SmartEiffel; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
--
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2004: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------------------------
