-- This file is part of SmartEiffel The GNU Eiffel Compiler Tools and Libraries.
-- See the Copyright notice at the end of this file.
--
class INSPECT_STATEMENT
	--
	-- The Eiffel inspect instruction.
	--
	-- Note: values used inside when clauses are always static.
	--

inherit
	INSTRUCTION

creation {EIFFEL_PARSER, INTROSPECTION_HANDLER, MANIFEST_STRING_INSPECTOR}
	make

creation {INTROSPECTION_HANDLER}
	make_strippable

feature {ANY}
	start_position: POSITION
			-- Of keyword `inspect'.

	expression: EXPRESSION
			-- Heading expression after keyword `inspect'.

	when_list: FAST_ARRAY[WHEN_CLAUSE]
			-- List of when clauses.

	else_position: POSITION
			-- Of the keyword `else' if any.

	else_compound: INSTRUCTION
			-- Else compound if any.

	end_mark_comment: BOOLEAN is True

	manifest_string_inspector: MANIFEST_STRING_INSPECTOR
			-- Non Void if manifest strings are used. Some extra code is generated by this object.

	safety_check (type: TYPE) is
		local
			i: INTEGER
		do
			expression.safety_check(type)
			if when_list /= Void then
				from
					i := when_list.upper
				until
					i < when_list.lower
				loop
					when_list.item(i).safety_check(type)
					i := i - 1
				end
			end
			if else_compound /= Void then
				else_compound.safety_check(type)
			end
		end

	compile_to_c (type: TYPE) is
		local
			no_check, all_check: BOOLEAN
		do
			check
				first_one.item /= Void
			end
			if manifest_string_inspector /= Void then
				manifest_string_inspector.c_compile(type, Current)
			else
				no_check := ace.no_check
				all_check := ace.all_check
				cpp.inspect_tmp_increment
				cpp.pending_c_function_body.append(once "/*[INSPECT*/%N{int ")
				cpp.put_inspect_tmp
				cpp.pending_c_function_body.extend('=')
				if all_check then
					cpp.pending_c_function_body.extend('(')
					cpp.put_trace_or_sedb_expression(expression.start_position)
					cpp.pending_c_function_body.extend(',')
				end				
				expression.compile_to_c_with_internal_c_local_saving(type)
				if all_check then
					cpp.pending_c_function_body.extend(')')
				end
				cpp.pending_c_function_body.append(once ";%N")
				if when_list = Void then
					if else_position.is_unknown then
						if no_check then
							exceptions_handler.bad_inspect_value(start_position)
						end
					elseif else_compound /= Void then
						else_compound.compile_to_c_with_internal_c_local_saving(type)
					end
				elseif use_c_switch_statement then
					cpp.pending_c_function_body.append(once "switch(")
					cpp.put_inspect_tmp
					cpp.pending_c_function_body.append(once "){%N")
					compile_to_c_switch(type)
					if else_position.is_unknown then
						if no_check then
							cpp.pending_c_function_body.append(once "default:;%N")
							exceptions_handler.bad_inspect_value(start_position)
						end
					elseif else_compound /= Void then
						cpp.pending_c_function_body.append(once "default:;%N")
						else_compound.compile_to_c_with_internal_c_local_saving(type)
					end
					cpp.pending_c_function_body.append(once "}%N")
				else
					compile_to_c_if(type)
					if else_position.is_unknown then
						if no_check then
							cpp.pending_c_function_body.append(once " else{")
							exceptions_handler.bad_inspect_value(start_position)
							cpp.pending_c_function_body.extend('}')
						end
					elseif else_compound /= Void then
						cpp.pending_c_function_body.append(once " else{")
						else_compound.compile_to_c_with_internal_c_local_saving(type)
						cpp.pending_c_function_body.extend('}')
					end
				end
				cpp.pending_c_function_body.append(once "}/*INSPECT]*/%N")
				cpp.inspect_tmp_decrement
			end
		end

	compile_to_jvm (type: TYPE) is
		do
			check
				first_one.item /= Void
			end
			if manifest_string_inspector /= Void then
				manifest_string_inspector.jvm_compile(type, Current)
			else
				expression.compile_to_jvm(type)
				when_list_compile_to_jvm(type)
				if else_compound /= Void then
					else_compound.compile_to_jvm(type)
				elseif else_position.is_unknown then
					if ace.no_check then
						code_attribute.runtime_error_inspect(type, expression)
					end
				end
				when_list_compile_to_jvm_resolve_branch
				code_attribute.opcode_pop
			end
		end

	simplify (type: TYPE): INSTRUCTION is
		local
			exp: like expression; wl: like when_list; ec: like else_compound; i: INTEGER; wc1, wc2: WHEN_CLAUSE
			cc: CHARACTER_CONSTANT; ic: INTEGER_CONSTANT; selected: BOOLEAN; value: INTEGER
		do
			if manifest_string_inspector /= Void then
				Result := manifest_string_inspector.simplify(type, Current, Void)
			else
				exp := expression.simplify(type)
				cc ?= exp
				ic ?= exp
				if cc /= Void or else ic /= Void then
					if cc /= Void then
						value := cc.value.code
					else
						value := ic.value_memory.to_integer_32
					end
					smart_eiffel.magic_count_increment
					if when_list = Void then
						Result := else_compound
					else
						from
							i := when_list.upper
						until
							selected or else i < when_list.lower
						loop
							wc1 := when_list.item(i)
							if wc1.match_value(value) then
								selected := True
								Result := wc1.compound
							end
							i := i - 1
						end
						if not selected then
							if else_position.is_unknown then
								error_handler.append("In the context %"")
								error_handler.append(type.name.to_string)
								error_handler.append("%" (i.e. when the type of Current is %"")
								error_handler.append(type.name.to_string)
								error_handler.append("%") this expression is the ")
								if cc /= Void then
									error_handler.append("CHARACTER ")
									error_handler.append(cc.to_string)
								else
									error_handler.append("INTEGER ")
									error_handler.append(ic.to_string)
								end
								error_handler.append(" but there is _no_ when clause selected. (Also note that there is %
						  %no else part for this inspect statment, hence this error message.)")
								error_handler.add_position(expression.start_position)
								error_handler.print_as_fatal_error
							end
							Result := else_compound
						end
					end
					if Result /= Void then
						Result := Result.simplify(type)
					end
				else
					-- The general scheme:
					if when_list /= Void then
						from
							i := when_list.upper
						until
							i < when_list.lower or else wc1 /= wc2
						loop
							wc1 := when_list.item(i)
							wc2 := simplify_when_clause(type, wc1)
							i := i - 1
						end
						if wc1 = wc2 then
							wl := when_list
						else
							from
								wl := when_list.twin
								wl.put(wc2, i + 1)
							until
								i < wl.lower
							loop
								wl.put(simplify_when_clause(type, wl.item(i)), i)
								i := i - 1
							end
						end
					end
					if else_compound /= Void then
						ec := else_compound.simplify(type)
					end
					Result := current_or_twin_init(exp, wl, ec)
				end
			end
		end

	side_effect_free (type: TYPE): BOOLEAN is
		do
			--|*** Could be better. ***
		end

	use_current (type: TYPE): BOOLEAN is
		local
			i: INTEGER
		do
			Result := expression.use_current(type)
			if not Result and then when_list /= Void then
				from
					i := when_list.upper
				until
					Result or else i < when_list.lower
				loop
					Result := when_list.item(i).use_current(type)
					i := i - 1
				end
			end
			if else_compound /= Void then
				Result := Result or else else_compound.use_current(type)
			end
		end

	specialize_in (new_type: TYPE): like Current is
		local
			exp: like expression; wl: like when_list; ec: like else_compound; i: INTEGER; ew1, ew2: WHEN_CLAUSE
		do
			exp := expression.specialize_in(new_type)
			if when_list /= Void then
				from
					i := when_list.upper
				until
					i < when_list.lower or else ew1 /= ew2
				loop
					ew1 := when_list.item(i)
					ew2 := ew1.specialize_in(new_type)
					i := i - 1
				end
				if ew1 = ew2 then
					wl := when_list
				else
					from
						wl := when_list.twin
						wl.put(ew2, i + 1)
					until
						i < wl.lower
					loop
						wl.put(when_list.item(i).specialize_in(new_type), i)
						i := i - 1
					end
				end
			end
			if else_compound /= Void then
				ec := else_compound.specialize_in(new_type)
			end
			Result := current_or_twin_init(exp, wl, ec)
		end

	specialize_thru (parent_type: TYPE; parent_edge: PARENT_EDGE; new_type: TYPE): like Current is
		local
			exp: like expression; wl: like when_list; ec: like else_compound; i: INTEGER; ew1, ew2: WHEN_CLAUSE
		do
			exp := expression.specialize_thru(parent_type, parent_edge, new_type)
			if when_list /= Void then
				from
					i := when_list.upper
				until
					i < when_list.lower or else ew1 /= ew2
				loop
					ew1 := when_list.item(i)
					ew2 := ew1.specialize_thru(parent_type, parent_edge, new_type)
					i := i - 1
				end
				if ew1 = ew2 then
					wl := when_list
				else
					from
						wl := when_list.twin
						wl.put(ew2, i + 1)
					until
						i < wl.lower
					loop
						wl.put(when_list.item(i).specialize_thru(parent_type, parent_edge, new_type), i)
						i := i - 1
					end
				end
			end
			if else_compound /= Void then
				ec := else_compound.specialize_thru(parent_type, parent_edge, new_type)
			end
			Result := current_or_twin_init(exp, wl, ec)
		end

	specialize_2 (type: TYPE): like Current is
		local
			exp: like expression; wl: like when_list; ec: like else_compound; te: TYPE_MARK; i: INTEGER
			ew1, ew2: WHEN_CLAUSE
		do
			exp := expression.specialize_2(type)
			if when_list /= Void then
				te := exp.declaration_type.canonical_type_mark
				if te.is_character then
					from
						i := when_list.upper
					until
						i < when_list.lower or else ew1 /= ew2
					loop
						ew1 := when_list.item(i)
						ew2 := ew1.specialize_2_character(type)
						i := i - 1
					end
					if ew1 = ew2 then
						wl := when_list
					else
						from
							wl := when_list.twin
							wl.put(ew2, i + 1)
						until
							i < wl.lower
						loop
							wl.put(when_list.item(i).specialize_2_character(type), i)
							i := i - 1
						end
					end
				elseif te.is_integer then
					from
						i := when_list.upper
					until
						i < when_list.lower or else ew1 /= ew2
					loop
						ew1 := when_list.item(i)
						ew2 := ew1.specialize_2_integer(type)
						i := i - 1
					end
					if ew1 = ew2 then
						wl := when_list
					else
						from
							wl := when_list.twin
							wl.put(ew2, i + 1)
						until
							i < wl.lower
						loop
							wl.put(when_list.item(i).specialize_2_integer(type), i)
							i := i - 1
						end
					end
				elseif te.is_string then
					create manifest_string_inspector.make(Current)
					from
						i := when_list.upper
					until
						i < when_list.lower or else ew1 /= ew2
					loop
						ew1 := when_list.item(i)
						ew2 := ew1.specialize_2_string(type)
						i := i - 1
					end
					if ew1 = ew2 then
						wl := when_list
					else
						from
							wl := when_list.twin
							wl.put(ew2, i + 1)
						until
							i < wl.lower
						loop
							wl.put(when_list.item(i).specialize_2_string(type), i)
							i := i - 1
						end
					end
				else
					error_handler.add_position(expression.start_position)
					error_handler.append("For inspect statement, the expression type can be only INTEGER, %
				    %CHARACTER or STRING. (Actually ")
					error_handler.add_expression(expression)
					error_handler.append(" is of type ")
					error_handler.add_type_mark(te)
					error_handler.append(" which is not allowed.)")
					error_handler.print_as_error
				end
			end
			if else_compound /= Void then
				ec := else_compound.specialize_2(type)
			end
			Result := current_or_twin_init(exp, wl, ec)
			if first_one.item = Void then
				Result.validity_check_and_compute_values(type)
				check
					first_one.item = Result
				end
			else
				Result.share_values_of(first_one.item)
			end
		ensure then
			first_one.item /= Void
			first_one.item = Result.first_one.item
		end

	has_been_specialized: BOOLEAN is
		local
			i: INTEGER
		do
			Result := expression.has_been_specialized
			if Result and then when_list /= Void then
				from
					i := when_list.upper
					Result := True
				until
					i < when_list.lower or not Result
				loop
					Result := when_list.item(i).has_been_specialized
					i := i - 1
				end
			end
			if Result and then else_compound /= Void then
				Result := else_compound.has_been_specialized
			end
		end

	pretty (indent_level: INTEGER) is
		local
			i: INTEGER
		do
			pretty_printer.set_indent_level(indent_level)
			pretty_printer.put_string(once "inspect")
			pretty_printer.set_indent_level(indent_level + 1)
			pretty_printer.set_semi_colon_flag(False)
			expression.pretty(indent_level + 1)
			pretty_printer.set_indent_level(0)
			if when_list /= Void then
				from
					i := when_list.lower
				until
					i > when_list.upper
				loop
					when_list.item(i).pretty(indent_level)
					i := i + 1
				end
				pretty_printer.set_indent_level(0)
			end
			if else_compound = Void then
				if not else_position.is_unknown then
					pretty_printer.set_indent_level(indent_level)
					pretty_printer.put_string(once "else")
					pretty_printer.set_indent_level(0)
				end
			else
				pretty_printer.set_indent_level(indent_level)
				pretty_printer.put_string(once "else")
				else_compound.pretty(indent_level + 1)
			end
			pretty_printer.set_indent_level(indent_level)
			pretty_printer.keyword(once "end")
			if pretty_printer.print_end_of_statement then
				pretty_printer.put_end_of(once "inspect")
			end
		end

	accept (visitor: INSPECT_STATEMENT_VISITOR) is
		do
			visitor.visit_inspect_statement(Current)
		end

	collect (t: TYPE) is
		local
			i: INTEGER; dummy: TYPE; lt: LIVE_TYPE
		do
			dummy := expression.collect(t)
			if when_list /= Void then
				if condition_type /= Void then
					lt := condition_type.live_type
				end
				from
					i := when_list.upper
				until
					i < when_list.lower
				loop
					when_list.item(i).collect(t, lt)
					i := i - 1
				end
			end
			if else_compound /= Void then
				else_compound.collect(t)
			end
		end

	adapt_for (t: TYPE): like Current is
		local
			exp: like expression; wl: like when_list; ew1, ew2: like simplify_when_clause; ec: like else_compound
			source_index, destination_index: INTEGER
		do
			exp := expression.adapt_for(t)
			if when_list /= Void then
				from
					source_index := when_list.lower
					check
						source_index = 0
					end
				until
					source_index > when_list.upper or else ew1 /= ew2
				loop
					ew1 := when_list.item(source_index)
					source_index := source_index + 1
					ew2 := ew1.strip_or_set_value(source_index, t, condition_type)
					if ew2 /= Void then
						ew2 := ew2.adapt_for(t)
					end
				end
				if ew1 = ew2 then
					wl := when_list
				else
					wl := when_list.twin
					if ew2 /= Void then
						wl.put(ew2, source_index - 1)
						destination_index := source_index
					else
						destination_index := source_index - 1
					end
					from
					until
						source_index > wl.upper
					loop
						ew1 := wl.item(source_index)
						ew2 := ew1.strip_or_set_value(destination_index + 1, t, condition_type)
						if ew2 /= Void then
							ew2 := ew2.adapt_for(t)
							wl.put(ew2, destination_index)
							destination_index := destination_index + 1
						end
						source_index := source_index + 1
					end
					if destination_index = wl.lower then
						wl := Void
					elseif destination_index < wl.upper + 1 then
						wl.remove_tail(wl.upper - destination_index + 1)
					end
				end
			end
			if else_compound /= Void then
				ec := else_compound.adapt_for(t)
			end
			Result := current_or_twin_init(exp, wl, ec)
			if when_list /= Void and then condition_type /= Void then
				Result.first_one.set_item(Void)
			end
			if Result.first_one.item = Void then
				check condition_type /= Void
					or start_position.class_text.name.to_string = as_internals_handler
				end
				Result.validity_check_and_compute_values(t)
			end
		end

feature {WHEN_CLAUSE}
	always_selected (compound: INSTRUCTION) is
		do
			else_compound := compound
			when_list := Void
		end

feature {EIFFEL_PARSER, INTROSPECTION_HANDLER, MANIFEST_STRING_INSPECTOR}
	set_else_compound (sp: like else_position; ec: like else_compound) is
		do
			else_position := sp
			else_compound := ec
		end

feature {WHEN_CLAUSE}
	add_when (when_clause: WHEN_CLAUSE) is
		require
			when_clause /= Void
		do
			if when_list = Void then
				create when_list.with_capacity(4)
			end
			when_list.add_last(when_clause)
		end

	condition_type: TYPE

feature {INSPECT_STATEMENT, EXTERNAL_ROUTINE}
	init (exp: like expression; wl: like when_list; ec: like else_compound) is
		require
			exp /= Void
			ec /= Void implies not else_position.is_unknown
		do
			expression := exp
			else_compound := ec
			when_list := wl
			if when_list = Void then
				manifest_string_inspector := Void
			end
		ensure
			expression = exp
			when_list = wl
			else_compound = ec
		end

	share_values_of (first: like Current) is
		require
			first /= Void
		local
			i: INTEGER
		do
			if when_list /= Void then
				from
					i := when_list.upper
				until
					i < when_list.lower
				loop
					when_list.item(i).share_values_of(first.when_list.item(i))
					i := i - 1
				end
			end
		end

feature {INTROSPECTION_HANDLER}
	prelude (type: TYPE): TUPLE[INSTRUCTION, INTERNAL_LOCAL, CALL_1_C, EXPRESSION] is
		local
			e_local, s_local: INTERNAL_LOCAL
			eval_exp: ASSIGNMENT
			i_call: CALL_1_C; c_call: CALL_0_C
			simplified: EXPRESSION
		do
			simplified := expression.simplify(type)
			e_local := exp_local(type, simplified)
			s_local := state_local(type)
			create eval_exp.inline_make(e_local, simplified)
			i_call := item_call(type, e_local)
			c_call := count_call(type, e_local)
			Result := [eval_exp, s_local, i_call, c_call]
		end

	simplify_with_prelude (type: TYPE; prel: like prelude): INSTRUCTION is
		require
			manifest_string_inspector /= Void
			prel /= Void
		do
			Result := manifest_string_inspector.simplify(type, Current, prel)
		end

feature {MANIFEST_STRING_INSPECTOR}
	exp_local (type: TYPE; exp: EXPRESSION): INTERNAL_LOCAL is
		local
			er: E_ROUTINE
		do
			er ::= smart_eiffel.context_feature
			Result := er.typed_internal_local(type, exp.resolve_in(type), start_position, once "string inspect expression")
		end

	state_local (type: TYPE): INTERNAL_LOCAL is
		local
			er: E_ROUTINE
		do
			er ::= smart_eiffel.context_feature
			Result := er.typed_internal_local(type, smart_eiffel.type_integer_32, start_position,
																		  once "string inspect state")
		end

	item_call (type: TYPE; e_local: like exp_local): CALL_1_C is
		local
			fn_item: FEATURE_NAME
			args: EFFECTIVE_ARG_LIST
		do
			create args.make_1(create {INTEGER_CONSTANT}.make(0, start_position))
			create fn_item.ordinary_name(item_name, start_position)
			create Result.make_specialized(e_local, e_local.resolve_in(type), fn_item, args)
		end

	count_call (type: TYPE; e_local: like exp_local): CALL_0_C is
		local
			fn_count: FEATURE_NAME
		do
			create fn_count.ordinary_name(count_name, start_position)
			create Result.make_specialized(e_local, e_local.resolve_in(type), fn_count)
		end

	largest_range_of_values: INTEGER is
		local
			i, range: INTEGER
		do
			from
				i := when_list.lower
			until
				i > when_list.upper
			loop
				range := when_list.item(i).largest_range_of_values
				if range > Result then
					Result := range
				end
				i := i + 1
			end
		end

	compile_to_c_switch (type: TYPE) is
		local
			i: INTEGER; last_compound: INSTRUCTION
		do
			from
				i := when_list.lower
			until
				i = when_list.upper
			loop
				when_list.item(i).compile_to_c_switch(type)
				i := i + 1
			end
			if else_position.is_unknown and then ace.boost then
				cpp.pending_c_function_body.append(once "default:;%N")
				last_compound := when_list.item(i).compound
				if last_compound /= Void then
					last_compound.compile_to_c_with_internal_c_local_saving(type)
				end
			else
				when_list.item(i).compile_to_c_switch(type)
			end
		end

	compile_to_c_if (type: TYPE) is
		local
			i: INTEGER; last_compound: INSTRUCTION
		do
			from
				i := when_list.lower
			until
				i = when_list.upper
			loop
				when_list.item(i).compile_to_c(type)
				i := i + 1
				if i < when_list.upper then
					cpp.pending_c_function_body.append(once " else ")
				end
			end
			if i > when_list.lower then
				cpp.pending_c_function_body.append(once " else ")
			end
			if else_position.is_unknown and then ace.boost then
				cpp.pending_c_function_body.extend('{')
				last_compound := when_list.item(i).compound
				if last_compound /= Void then
					last_compound.compile_to_c_with_internal_c_local_saving(type)
				end
				cpp.pending_c_function_body.extend('}')
			else
				when_list.item(i).compile_to_c(type)
			end
		end

	when_list_compile_to_jvm_resolve_branch is
		local
			i: INTEGER
		do
			if when_list /= Void then
				from
					i := when_list.upper
				until
					i < when_list.lower
				loop
					when_list.item(i).compile_to_jvm_resolve_branch
					i := i - 1
				end
			end
		end

	when_list_compile_to_jvm (type: TYPE) is
		local
			remainder, i: INTEGER
		do
			if when_list /= Void then
				from
					remainder := when_list.count
					i := when_list.lower
				until
					remainder = 0
				loop
					remainder := remainder - 1
					when_list.item(i).compile_to_jvm(type, else_position, remainder)
					i := i + 1
				end
			end
		end

feature {WHEN_ITEM}
	validity_check_continued (occurrence_1: WHEN_ITEM) is
		require
			occurrence_1 /= Void
		local
			i: INTEGER
		do
			from
				i := when_list.lower
			until
				i > when_list.upper or else when_list.item(i).validity_check_continued(occurrence_1)
			loop
				i := i + 1
			end
		end

feature {INSPECT_STATEMENT, WHEN_CLAUSE, WHEN_ITEM}
	first_one: REFERENCE[like Current]
			-- To memorize the first one for which `validity_check_and_compute_values' is called. (Only the `first_one' gets its
			-- `values' computed.

feature {MANIFEST_STRING_INSPECTOR}
	force_internal_values (type: TYPE) is
			-- Force the creation of values for this internally generated inspect statement
		local
			i: INTEGER; t: TYPE; te: TYPE_MARK
		do
			t := expression.declaration_type
			te := t.canonical_type_mark
			if te.is_character then
				from
					i := when_list.lower
				until
					i > when_list.upper
				loop
					when_list.item(i).force_internal_character_values(type)
					i := i + 1
				end
			elseif te.is_integer then
				from
					i := when_list.lower
				until
					i > when_list.upper
				loop
					when_list.item(i).force_internal_integer_values(type)
					i := i + 1
				end
			else
				error_handler.add_position(start_position)
				error_handler.append("A generated inspect should test integers or characters.")
				error_handler.print_as_internal_error
			end
			first_one.set_item(Current)
		end

feature {INSPECT_STATEMENT}
	validity_check_and_compute_values (type: TYPE) is
			-- Note: the `when_list' part is completely static and this validity check must be performed once.
		require
			first_one.item = Void
		local
			i: INTEGER
		do
			first_one.set_item(Current)
			if when_list /= Void then
				-- Validity check first:
				from
					i := when_list.lower
				until
					i > when_list.upper
				loop
					when_list.item(i).validity_check(Current, type)
					i := i + 1
				end
				-- Now compute values:
				from
					i := when_list.lower
				until
					i > when_list.upper
				loop
					when_list.item(i).compute_values
					i := i + 1
				end
			end
		ensure
			first_one.item = Current
		end

feature {}
	make (sp: like start_position; exp: like expression) is
		require
			not sp.is_unknown
			exp /= Void
		do
			create first_one
			start_position := sp
			expression := exp
		ensure
			start_position = sp
			expression = exp
		end

	make_strippable (sp: like start_position; exp: like expression; c_type: like condition_type) is
		require
			not sp.is_unknown
			exp /= Void
		do
			condition_type := c_type
			make(sp, exp)
		ensure
			start_position = sp
			expression = exp
		end

	use_c_switch_statement: BOOLEAN is
			-- To decide if the generated C code is a True C "switch" or
			-- a sequence of "if... else if ... else ...").
		do
			-- For the time being, this a naive implementation, because
			-- we may also  consider wich C compiler is used (see in
			-- `system_tools'). If someone want to do this, I think we should add a
			-- new function named `use_c_switch_statement' in system_tools.
			-- Just post the fix in the mailing list.
			if largest_range_of_values <= 32 then
				Result := True
			end
		end

	current_or_twin_init (exp: like expression; wl: like when_list; ec: like else_compound): like Current is
		require
			exp /= Void
		do
			if exp = expression and then wl = when_list and then ec = else_compound then
				Result := Current
			else
				Result := twin
				Result.init(exp, wl, ec)
			end
		ensure
			Result.expression = exp
			Result.when_list = wl
			Result.else_compound = ec
		end

	simplify_when_clause (type: TYPE; wc: WHEN_CLAUSE): WHEN_CLAUSE is
		require
			type /= Void
			wc /= Void
		local
			compound1, compound2: INSTRUCTION
		do
			compound1 := wc.compound
			if compound1 = Void then
				Result := wc
			else
				compound2 := compound1.simplify(type)
				if compound1 = compound2 then
					Result := wc
				else
					Result := wc.twin
					Result.set_compound(compound2)
				end
			end
		ensure
			Result /= Void
		end

invariant
	expression /= Void

end -- class INSPECT_STATEMENT
--
-- ------------------------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- SmartEiffel is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License,
-- as published by the Free Software Foundation; either version 2, or (at your option) any later version.
-- SmartEiffel is distributed in the hope that it will be useful but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. You should have
-- received a copy of the GNU General Public License along with SmartEiffel; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
--
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2004: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------------------------
