-- This file is part of SmartEiffel The GNU Eiffel Compiler Tools and Libraries.
-- See the Copyright notice at the end of this file.
--
class IFTHEN
	--
	-- Used when we have an single if-then INSTRUCTION with no else part or when it is a component of
	-- an IFTHENELSE (see also IFTHENELSE).
	--

inherit
	INSTRUCTION
		redefine
			simplify_routine_body_first
		end

insert
	IF_SUPPORT

creation {ANY}
	make

feature {ANY}
	accept (visitor: IFTHEN_VISITOR) is
		do
			visitor.visit_ifthen(Current)
		end

	collect (type: TYPE) is
		local
			dummy: TYPE
		do
			dummy := expression.collect(type)
			if then_compound /= Void then
				then_compound.collect(type)
			end
		end

	simplify (type: TYPE): INSTRUCTION is
		local
			bc: BOOLEAN_CONSTANT; e: like expression; tc: like then_compound
		do
			e := expression.simplify(type)
			bc ?= e
			if bc /= Void then
				smart_eiffel.magic_count_increment
				if bc.value then
					if then_compound /= Void then
						Result := then_compound.simplify(type)
					end
				else
					-- The `then_compound' cannot be reached.
				end
			else
				if then_compound /= Void then
					tc := then_compound.simplify(type)
				end
				Result := current_or_twin_init(e, tc)
			end
		end

	adapt_for (type: TYPE): like Current is
		local
			e: like expression; tc: like then_compound
		do
			e := expression.adapt_for(type)
			if then_compound /= Void then
				tc := then_compound.adapt_for(type)
			end
			Result := current_or_twin_init(e, tc)
		end

	safety_check (type: TYPE) is
		do
			expression.safety_check(type)
			if then_compound /= Void then
				then_compound.safety_check(type)
			end
		end

	compile_to_c (type: TYPE) is
		do
			cpp.pending_c_function_body.append(once "if(")
			if ace.no_check then
				cpp.trace_boolean_expression(type, expression)
			else
				expression.compile_to_c_with_internal_c_local_saving(type)
			end
			cpp.pending_c_function_body.append(once "){%N")
			if then_compound /= Void then
				then_compound.compile_to_c_with_internal_c_local_saving(type)
			end
			cpp.pending_c_function_body.append(once "}%N")
		end

	compile_to_jvm (type: TYPE) is
		local
			point: INTEGER
		do
			point := expression.jvm_branch_if_false(type)
			if then_compound /= Void then
				then_compound.compile_to_jvm(type)
			end
			code_attribute.resolve_u2_branch(point)
		end

	use_current (type: TYPE): BOOLEAN is
		do
			Result := expression.use_current(type)
			if not Result and then then_compound /= Void then
				Result := then_compound.use_current(type)
			end
		end

	pretty (indent_level: INTEGER) is
		do
			pretty_(indent_level, once "if")
			pretty_end_if(indent_level)
		end

	specialize_in (type: TYPE): like Current is
		local
			e: like expression; tc: like then_compound
		do
			e := expression.specialize_in(type)
			if then_compound /= Void then
				tc := then_compound.specialize_in(type)
			end
			Result := current_or_twin_init(e, tc)
		end

	specialize_thru (parent_type: TYPE; parent_edge: PARENT_EDGE; new_type: TYPE): like Current is
		local
			e: like expression; tc: like then_compound
		do
			e := expression.specialize_thru(parent_type, parent_edge, new_type)
			if then_compound /= Void then
				tc := then_compound.specialize_thru(parent_type, parent_edge, new_type)
			end
			Result := current_or_twin_init(e, tc)
		end

	specialize_2 (type: TYPE): like Current is
		local
			e: like expression; tc: like then_compound
		do
			e := expression.specialize_2(type)
			if then_compound /= Void then
				tc := then_compound.specialize_2(type)
			end
			Result := current_or_twin_init(e, tc)
			Result.specialize_2_check_(type)
		end

	has_been_specialized: BOOLEAN is
		do
			Result := expression.has_been_specialized
			if then_compound /= Void then
				Result := then_compound.has_been_specialized
			end
		end

feature {ANONYMOUS_FEATURE, INSTRUCTION}
	simplify_routine_body_first (type: TYPE; return_type: TYPE): INSTRUCTION is
		local
			assignment: ASSIGNMENT; e_true: E_TRUE; result_variable: RESULT
			call_infix_and_then: CALL_INFIX_AND_THEN
		do
			Result := Current
			if return_type /= Void and then return_type.is_boolean then
				assignment ?= then_compound
				if assignment /= Void then
					if assignment.left_side.is_result then
						smart_eiffel.magic_count_increment
						e_true ?= assignment.right_side
						if e_true /= Void then
							create {ASSIGNMENT} Result.inline_make(assignment.left_side, expression)
						else
							create result_variable.make(start_position)
							result_variable.set_type_mark_memory(return_type.canonical_type_mark)
							create call_infix_and_then.make(expression, start_position, assignment.right_side)
							call_infix_and_then.set_feature_stamp(boolean_and_then_fs)
							create {ASSIGNMENT} Result.inline_make(result_variable, call_infix_and_then)
						end
					end
				end
			end
		end

feature {IFTHEN}
	init (e: like expression; tc: like then_compound) is
		require
			e /= Void
		do
			expression := e
			then_compound := tc
		ensure
			expression = e
			then_compound = tc
		end

feature {}
	current_or_twin_init (e: like expression; tc: like then_compound): like Current is
		do
			if e = expression and then tc = then_compound then
				Result := Current
			else
				Result := twin
				Result.init(e, tc)
			end
		end

	make (sp: like start_position; e: like expression; tc: like then_compound) is
		require
			not sp.is_unknown
			e /= Void
		do
			start_position := sp
			expression := e
			then_compound := tc
		ensure
			start_position = sp
			expression = e
			then_compound = tc
		end

end -- class IFTHEN
--
-- ------------------------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- SmartEiffel is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License,
-- as published by the Free Software Foundation; either version 2, or (at your option) any later version.
-- SmartEiffel is distributed in the hope that it will be useful but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. You should have
-- received a copy of the GNU General Public License along with SmartEiffel; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
--
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2004: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------------------------
