-- This file is part of SmartEiffel The GNU Eiffel Compiler Tools and Libraries.
-- See the Copyright notice at the end of this file.
--
class CECIL_POOL
	--
	-- Unique global object in charge of CECIL calls.
	--

insert
	GLOBALS

feature {ACE, COMMAND_LINE_TOOLS, C_PLUGIN}
	add_cecil_file (path: STRING) is
			-- Add `path' as a new -cecil file to be considered.
		require
			path /= Void
			not string_aliaser.registered_one(path)
			may_report_an_error: error_handler.is_empty
		local
			file: CECIL_FILE
		do
			echo.put_string(once "Adding Cecil file: ")
			echo.put_string(path)
			echo.put_new_line
			if cecil_files = Void then
				create {HASHED_DICTIONARY[CECIL_FILE, STRING]} cecil_files.make
			end
			if not cecil_files.has(path) then
				create file.make(path)
				cecil_files.add(file, path)
				-- because plugins add cecil files only at collect time
				inspect state
				when State_initial then
					-- nothing
				when State_parsed then
					file.parse
				when State_collected then
					file.parse
					file.collect
				end
			end
		ensure
			may_report_an_error: error_handler.is_empty
		end

feature {SMART_EIFFEL}
	parse_cecil_files is
		require
			state = State_initial
			may_report_an_error: error_handler.is_empty
		local
			i: INTEGER
		do
			if cecil_files /= Void then
				from
					i := cecil_files.lower
				until
					i > cecil_files.upper
				loop
					cecil_files.item(i).parse
					i := i + 1
				end
			end
			state := State_parsed
		ensure
			state = State_parsed
			may_report_an_error: error_handler.is_empty
		end

	collect is
		require
			(state = State_parsed) xor (state = State_collected)
		local
			i: INTEGER
		do
			if cecil_files /= Void then
				echo.put_string(once "Collecting Cecil features.%N")
				from
					i := cecil_files.lower
				until
					i > cecil_files.upper
				loop
					cecil_files.item(i).collect
					i := i + 1
				end
			end
			state := State_collected
		ensure
			state = State_collected
		end

feature {ANY}
	state: INTEGER

	State_initial: INTEGER is 0
	State_parsed: INTEGER is 1
	State_collected: INTEGER is 2

feature {C_PRETTY_PRINTER}
	c_define_users is
		local
			i: INTEGER
		do
			if cecil_files /= Void then
				from
					i := cecil_files.lower
				until
					i > cecil_files.upper
				loop
					cecil_files.item(i).c_define_users
					i := i + 1
				end
			end
		end

feature {RUN_FEATURE}
	c_define_body_of (rf: RUN_FEATURE; is_creation: BOOLEAN) is
		require
			cpp.pending_c_function
		local
			type_of_current: TYPE; current_type_mark, result_type_mark: TYPE_MARK; fake_target: FAKE_TARGET; eal: EFFECTIVE_ARG_LIST
			run_time_set: RUN_TIME_SET; buffer: STRING; internal_c_local: INTERNAL_C_LOCAL
		do
			type_of_current := rf.type_of_current
			current_type_mark := rf.type_of_current.canonical_type_mark
			result_type_mark := rf.result_type
			if result_type_mark /= Void or else is_creation then
				if is_creation then
					current_type_mark.c_type_for_external_in(cpp.pending_c_function_body)
				else
					result_type_mark.c_type_for_external_in(cpp.pending_c_function_body)
				end
				cpp.pending_c_function_body.append(once " R;%N")
			end
			run_time_set := current_type_mark.type.live_type.run_time_set
			if run_time_set.count = 0 then
				check
					not is_creation
				end
				error_handler.add_type_mark(current_type_mark)
				error_handler.append(" not created (type is not alive). Empty Cecil/Wrapper function ")
				error_handler.append(current_type_mark.written_mark)
				error_handler.extend('.')
				buffer := ".... local buffer ...."
				buffer.clear_count
				rf.name.mapping_c_in(buffer)
				error_handler.append(buffer)
				error_handler.extend('.')
				error_handler.print_as_warning
				if ace.no_check then
					cpp.pending_c_function_body.append(once "[
               error0("Deferred feature call (Void target).", NULL);

                                ]")
				end
			else
				if not gc_handler.is_off then
					cpp.pending_c_function_body.append(once "[
												#ifndef FIXED_STACK_BOTTOM
												int valid_stack_bottom = stack_bottom != NULL;
												#endif
												
												]")
				end
				if ace.no_check then
					cpp.pending_c_function_body.append(once "[
                  se_dump_stack ds={NULL,NULL,0,NULL,NULL,NULL};
                  ds.caller=se_dst;
                  ds.exception_origin=NULL;
                  ds.locals=NULL;

                  ]")
					rf.c_set_dump_stack_top(once "&ds", once "link")
				end
				if not gc_handler.is_off then
					cpp.pending_c_function_body.append(once "[
												#ifndef FIXED_STACK_BOTTOM
												if(!valid_stack_bottom) stack_bottom = (void**)(void*)&valid_stack_bottom;
												#endif
												
												]")
				end
				if result_type_mark /= Void then
					cpp.pending_c_function_body.append(once "R=")
				end
				if rf.arguments /= Void then
					eal := effective_arg_list(rf)
				end
				if is_creation then
					cpp.pending_c_function_body.append(once "/* CECIL creation */%N{%N")
					internal_c_local := cpp.pending_c_function_lock_local(type_of_current)
					if type_of_current.is_reference then
						gc_handler.allocation_of(internal_c_local, type_of_current.live_type)
					else
						internal_c_local.append_in(cpp.pending_c_function_body)
						cpp.pending_c_function_body.append(once "=M")
						type_of_current.live_type.id.append_in(cpp.pending_c_function_body)
						cpp.pending_c_function_body.append(once ";%N")
					end
					cpp.push_create_instruction(type_of_current, rf, eal, internal_c_local)
					rf.mapping_c
					cpp.pop
					cpp.pending_c_function_body.append(once "R=")
					internal_c_local.append_in(cpp.pending_c_function_body)
					internal_c_local.un_lock
					cpp.pending_c_function_body.append(once ";%N}%N")
				else
					create fake_target.make(rf.start_position, rf.type_of_current.canonical_type_mark)
					if current_type_mark.is_expanded then
						cpp.push_direct(rf, type_of_current, fake_target, eal)
						rf.mapping_c
						cpp.pop
					else
						cpp.push_pop_cpc(rf, run_time_set, type_of_current, Void, fake_target, eal)
					end
				end
				if result_type_mark /= Void then
					cpp.pending_c_function_body.append(once ";%N")
				end
				if not gc_handler.is_off then
					cpp.pending_c_function_body.append(once "[
												#ifndef FIXED_STACK_BOTTOM
												if(!valid_stack_bottom) if(!valid_stack_bottom) stack_bottom = NULL;
												#endif
												
												]")
				end
				if ace.no_check then
					rf.c_set_dump_stack_top(once "ds.caller", once "unlink")
				end
				if result_type_mark /= Void or else is_creation then
					cpp.pending_c_function_body.append(once "return R;%N")
				end
			end
		end

feature {CECIL_FILE}
	c_define_for (cecil_entry: CECIL_ENTRY) is
		require
			not cpp.pending_c_function
		local
			current_type_mark, result_type_mark: TYPE_MARK; rfargs: FORMAL_ARG_LIST; c_name: STRING; rf: RUN_FEATURE; is_creation: BOOLEAN
		do
			c_name := cecil_entry.c_name
			rf := cecil_entry.run_feature
			is_creation := cecil_entry.is_creation
			current_type_mark := rf.type_of_current.canonical_type_mark
			result_type_mark := rf.result_type
			rfargs := rf.arguments -- (1) ------------------------- Define Cecil heading :
			cpp.prepare_c_function
			if is_creation then
				current_type_mark.c_type_for_external_in(cpp.pending_c_function_signature)
			elseif result_type_mark /= Void then
				result_type_mark.c_type_for_external_in(cpp.pending_c_function_signature)
			else
				cpp.pending_c_function_signature.append(once "void")
			end
			cpp.pending_c_function_signature.extend(' ')
			cpp.pending_c_function_signature.append(c_name)
			cpp.pending_c_function_signature.extend('(')
			if is_creation then
				if rfargs = Void then
					cpp.pending_c_function_signature.append(once "void")
				else
					rfargs.external_prototype_in(cpp.pending_c_function_signature)
				end
			else
				current_type_mark.c_type_for_external_in(cpp.pending_c_function_signature)
				cpp.pending_c_function_signature.append(once " C")
				if rfargs /= Void then
					cpp.pending_c_function_signature.extend(',')
					rfargs.external_prototype_in(cpp.pending_c_function_signature)
				end
			end
			cpp.pending_c_function_signature.extend(')')
			c_define_body_of(rf, cecil_entry.is_creation)
			cpp.dump_pending_c_function(True)
		end

feature {}
	effective_arg_list (rf: RUN_FEATURE): EFFECTIVE_ARG_LIST is
		local
			first: FAKE_ARGUMENT; remainder: FAST_ARRAY[EXPRESSION]; fake_argument: FAKE_ARGUMENT
			fal: FORMAL_ARG_LIST; i: INTEGER
		do
			fal := rf.arguments
			create first.make(fal.type_mark(1), 1)
			if fal.count > 1 then
				from
					create remainder.with_capacity(fal.count - 1)
					i := 2
				until
					i > fal.count
				loop
					create fake_argument.make(fal.type_mark(i), i)
					remainder.add_last(fake_argument)
					i := i + 1
				end
				create Result.make_n(first, remainder)
			else
				create Result.make_1(first)
			end
		end

	cecil_files: DICTIONARY[CECIL_FILE, STRING]
			-- Non Void if some -cecil option is used.

	singleton_memory: CECIL_POOL is
		once
			Result := Current
		end

invariant
	is_real_singleton: Current = singleton_memory
	valid_state: state.in_range(State_initial, State_collected)

end -- class CECIL_POOL
--
-- ------------------------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- SmartEiffel is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License,
-- as published by the Free Software Foundation; either version 2, or (at your option) any later version.
-- SmartEiffel is distributed in the hope that it will be useful but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. You should have
-- received a copy of the GNU General Public License along with SmartEiffel; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
--
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2004: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------------------------
