-- See the Copyright notice at the end of this file.
--
class MICROSOFT_PATH_NAME
	-- Operating system path name, Microsoft notation (DOS, Win9x, WinNT)

inherit
	PATH_NAME
		redefine copy, is_equal end

create
	make_empty, make_root, make_current, make_from_string, copy

feature -- Creation

	make_empty is
		do
			if path = Void then path := "" else path.clear_count end
			if to_string_cache = Void then
				to_string_cache := ""
			else
				to_string_cache.clear_count
			end
			drive := no_drive
			valid_cache := True
		ensure then
			to_string_cache /= Void
			path /= Void
			drive = no_drive
		end

	make_root is
		do
			make_empty
			path.extend (directory_separator)
			valid_cache := False
		ensure then
			to_string.is_equal (once "\")
		end

	make_current is
		do
			make_empty
			path.extend ('.')
			valid_cache := False
		ensure then
			to_string.is_equal (once ".")
		end

	make_from_string (s: STRING) is
		do
			make_empty
			if s.count >=2 and then s.item (2) = drive_separator then
				drive := s.first
				path.clear_count
				path.append_substring (s, 3, s.count)
			else
				drive := no_drive
				path.copy (s)
			end
			valid_cache := False
		end

feature -- Constants

	extension_separator: CHARACTER is '.'

	directory_separator: CHARACTER is '\'

	drive_separator: CHARACTER is ':'

feature -- Access

	to_string: STRING is
		do
			if drive = no_drive then
				Result := path
			elseif valid_cache then
				Result := to_string_cache
			else
				Result := to_string_cache
				Result.clear_count
				Result.extend (drive)
				Result.extend (drive_separator)
				Result.append (path)
				valid_cache := True
			end
		end

	drive_specification: STRING is
		do
			if drive = no_drive then
				Result := once ""
			else
				Result := "X:"
				Result.put (drive, 1)
			end
		ensure then
			Result.count = 0 or Result.count = 2
			(drive = '%U') = Result.is_empty
			not Result.is_empty implies Result.first = drive
			not Result.is_empty implies Result.item (2) = ':'
		end

	count: INTEGER is
		local
			p: INTEGER
			sep: BOOLEAN
		do
			from 
				p := 1
				sep := True
			until p > path.count loop
				if not sep and is_separator (path.item (p)) then
					sep := True
				elseif sep and not is_separator (path.item (p)) then
					sep := False
					Result := Result + 1
				end
				p := p + 1
			end
			if sep and Result > 0 then -- trailing \
				Result := Result + 1
			end
		end

	last: STRING is
		local
			p: INTEGER
		do
			from p := path.count until
				p = 0 or else is_separator (path.item (p))
			loop p := p - 1 end
			Result := path.substring (p+1, path.count)
		ensure then
			to_string.has_suffix (Result)
		end

	extension: STRING is
		local
			p: INTEGER
		do
			Result := once ""
			p := path.reverse_index_of (extension_separator, path.count)
			if p /= 0 then
				if path.index_of ('/', p) = 0 and then path.index_of (directory_separator, p) = 0 then
					Result := path.substring (p, path.count)
				end
			end
		end

	is_absolute: BOOLEAN is
		do
			Result := not path.is_empty and then is_separator(path.first)
		end

	is_normalized: BOOLEAN is
		local
			elem: STRING
			scan: STRING
		do
			elem := once "path_element"
			scan := once ""
			scan.copy (to_string)
			-- Some basic checks
			Result := not path.has ('/') and then -- UNIX slash not allowed
				not path.is_empty and then -- Empty pathis not normalized
				(drive /= no_drive implies not path.has_prefix (once "\\")) -- Double slash allowed only without drive
			-- Remove initial slashes
			from until scan.is_empty or else scan.first /= directory_separator loop
				scan.remove_head (1)
			end
			-- Check for trailing slashes, double slashes
			Result := Result and then
			          (scan.is_empty or else scan.last /= directory_separator) and then
			          not scan.has_substring (once "\\")
			-- Remove initial sequences of ".."
			if not is_absolute and Result then
				from until
					scan.is_empty or else
					not scan.has_prefix (up_directory)
				loop
					if scan.count >= 3 and then scan.item (3) = directory_separator then
						scan.remove_head (3)
					else
						scan.remove_head (2)
					end
				end
			else
				Result := Result and then
				          not scan.has_prefix (once "..\") and then
				          not scan.is_equal (up_directory)
			end
			-- Make sure that there is no '..' remaining
			Result := Result and then
				not scan.has_substring (once "\..\") and then
				not scan.has_suffix (once "\..")
			-- Make sure that there is no '.' remaining except alones
			Result := Result and then
				not scan.has_substring (once "\.\") and then
				not scan.has_suffix (once "\.") and then
				not scan.has_prefix (once ".\")
			Result := Result and then 
			          (is_absolute implies not scan.is_equal (this_directory))
		ensure
			Result implies not to_string.has_substring (once "\.\")
			Result implies not to_string.has_suffix (once "\.")
			Result implies not to_string.is_empty
		end

	exists: BOOLEAN is
--		local
--			i: FILE_INFORMATION
		do
			crash
--			i.update (to_string)
--			Result := i.exists
-- FIXME: No way to do this
		end

	same_file (other: like Current): BOOLEAN is
--		local
--			i, j: FILE_INFORMATION
		do
			crash
--			i.update (to_string)
--			j.update (other.to_string)
--			Result := i.exists and then j.exists and then
--				(i.inode = j.inode) and (i.device = j.device)
-- FIXME: No way to do this
		end

feature -- Operations

	to_absolute is
		local
			bd: BASIC_DIRECTORY
		do
			if not is_absolute then
				tmp.copy (Current)
				make_from_string (bd.current_working_directory.twin)
				join (tmp)
				tmp.make_empty
			end
			normalize
		end

	normalize_case is
		do
			drive := drive.to_lower
			path.to_lower
			path.replace_all ('/', directory_separator)
			valid_cache := False
		end

	normalize is
		local
			p: INTEGER
			components: ARRAY [STRING]
			initial_slashes: INTEGER
			elem: STRING
		do
			path.replace_all ('/', directory_separator)
			-- Count initial slashes; only one accepted when there is a drive
			-- letter, but many otherwise (to handle UNC paths, \\server\path)
			from initial_slashes := 0 until
				initial_slashes = path.count or else 
				not is_separator (path.item (initial_slashes + 1)) or else
				(initial_slashes = 1 and drive /= no_drive)
			loop initial_slashes := initial_slashes + 1 end

			-- Splitting
			from
				create components.make (1, 0)
				p := 1
			until p > path.count loop
				-- Skip slashes
				from until p > path.count or else not is_separator (path.item (p)) loop
					p := p + 1
				end
				-- Get path component
				from
					create elem.make (8)
				until p > path.count or else is_separator (path.item (p)) loop
					elem.extend (path.item (p))
					p := p + 1
				end
				components.add_last (elem)
			end
			-- Joining
			path.make_filled (directory_separator, initial_slashes)
			from p := 1 variant components.upper-p until
				p > components.upper
			loop
				if (components.item (p)).is_empty then
					-- nothing to do, "P\" == "P"
				elseif (components.item (p)).is_equal (this_directory) then
					-- nothing to do, "P\." == "P"
				elseif not (components.item (p)).is_equal (up_directory) then
					add_last (components.item (p))
				elseif path.count = 1 and then is_separator (path.first) then
					-- Nothing to do, "\.." == "\"
				elseif not is_empty and then not last.is_equal (up_directory) then
					remove_last -- "P\x\.." == "P"
				else
					add_last (components.item (p))
				end
				p := p + 1
			end
			if path.is_empty then
				path.extend ('.')
			end
		end

	remove_last is
		local
			p: INTEGER
		do
			-- Find last separator
			p := path.reverse_index_of ('/', path.count).max (
			     path.reverse_index_of (directory_separator, path.count))
			-- Remove all trailing slashes, leaving one if it is root
			from until p <= 1 or else not is_separator (path.item (p)) loop
				p := p - 1
			end
			to_string.keep_head (p)
		ensure then
			(old to_string.twin).has_prefix (to_string)
		end

	add_last (elem: STRING) is
		do
			if not path.is_empty and then not is_separator (path.last) then
				path.extend (directory_separator)
			end
			path.append (elem)
			valid_cache := False
		end

	join (other: like Current) is
		local
			other_wins: BOOLEAN
		do
			if path.is_empty and drive = no_drive then
				other_wins := True
			elseif other.is_absolute then
				other_wins := (drive = no_drive or other.drive /= no_drive) or else
					(path.count > 1 or (path.count = 1 and then not is_separator (path.last)))
			end
			if other_wins then
				copy (other)
			else
				check not path.is_empty or drive /= no_drive end
				if not path.is_empty and then is_separator (path.last) then
					if not other.path.is_empty and then is_separator(other.path.first) then
						path.append_substring (other.path, 2, other.path.count)
					else
						path.append (other.path)
					end
				elseif path.is_empty then
					path.copy (other.path)
				elseif not other.path.is_empty then
					path.extend_unless (directory_separator)
					path.append (other.path)
				else
					path.extend(directory_separator)
				end
			end
		end

	expand_user is
		local
			sys: SYSTEM
			home: STRING
			p: INTEGER
		do
			if drive = no_drive and then not path.is_empty and then path.first = '~' then
				if path.count = 1 or else is_separator (path.item (2)) then
					home := sys.get_environment_variable (once "HOME")
					if home = Void then
						home := sys.get_environment_variable (once "HOMEPATH")
						if home /= Void then
							p := path.first_index_of (directory_separator).min (
								path.first_index_of ('\'))
							if p = 0 then p := path.count + 1 end
							path.remove_head (p-1)
							path.prepend (home)
							home := sys.get_environment_variable (once "HOMEDRIVE")
							if home /= Void and not home.is_empty then
								drive := home.first
							end
						end
					else
						p := path.first_index_of (directory_separator).min (
								path.first_index_of ('\'))
						if p = 0 then p := path.count + 1 end
						path.remove_head (p-1)
						path.prepend (home)
					end
				else
					-- MS paths do not support ~user. Return original path
				end
			end
		end

	expand_shellouts is
		do
			not_yet_implemented
		end

feature -- Copying, comparison

	copy (other: like Current) is
		do
			make_empty
			drive := other.drive
			path.copy (other.path)
			valid_cache := False
		end

	is_equal (other: like Current): BOOLEAN is
		do
			-- Note: case insensitive
			Result := drive.same_as (other.drive) and then path.same_as (other.path)
		end

feature {MICROSOFT_PATH_NAME} -- Representation

	drive: CHARACTER
		-- Drive letter, or '%U' if none

	path: STRING
		-- path inside drive

feature {} -- Representation

	no_drive: CHARACTER is '%U'

	to_string_cache: STRING

	valid_cache: BOOLEAN

feature {} -- Internal

	is_separator (ch: CHARACTER): BOOLEAN is
			-- Is `ch' a possible path separator? ( '/'  or '\' )
		do
			Result := ch = '/' or ch = directory_separator
		end

	up_directory: STRING is ".."

	this_directory: STRING is "."

	tmp: MICROSOFT_PATH_NAME is
		once
			create Result.make_empty
		end

invariant
	to_string_cache /= Void
	path /= Void
	drive = no_drive or drive.is_letter;
	(valid_cache and drive = no_drive) implies to_string.is_equal (path)
	(valid_cache and drive /= no_drive) implies to_string.is_equal (drive.to_string+":"+path)
end -- class MICROSOFT_PATH_NAME    
--
-- ------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- This file is part of the SmartEiffel standard library.
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2006: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated
-- documentation files (the "Software"), to deal in the Software without restriction, including without
-- limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
-- the Software, and to permit persons to whom the Software is furnished to do so, subject to the following
-- conditions:
--
-- The above copyright notice and this permission notice shall be included in all copies or substantial
-- portions of the Software.
--
-- THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT
-- LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO
-- EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
-- AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE
-- OR OTHER DEALINGS IN THE SOFTWARE.
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------
