/* NVClock 0.8 - Linux overclocker for NVIDIA cards
 *
 * site: http://nvclock.sourceforge.net
 *
 * Copyright(C) 2001-2005 Roderick Colenbrander
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#ifndef NVCLOCK_H
#define NVCLOCK_H

#include "i2c.h"

#define MAX_CARDS 4

#define NV5  (1<<0)
#define NV10 (1<<1)
#define NV17 (1<<2)
#define NV1X (NV10 | NV17)
#define NV20 (1<<3)
#define NV25 (1<<4)
#define NV2X (NV20 | NV25)
#define NV30 (1<<5)
#define NV31 (1<<6)
#define NV35 (1<<7)
#define NV3X (NV30 | NV31 | NV35)
#define NV40 (1<<8)
#define NV43 (1<<9)
#define NV4X (NV40 | NV43)

#define NV_ERR_NO_DEVICES_FOUND 1
#define NV_ERR_NO_DRIVERS_FOUND 2
#define NV_ERR_NOT_ENOUGH_PERMISSIONS 3
#define NV_ERR_OTHER 4

#define CARD_SUPPORTED (1<<0)
#define GPU_OVERCLOCKING_SUPPORTED (1<<1)
#define MEM_OVERCLOCKING_SUPPORTED (1<<2)
#define PIPELINE_MODDING_SUPPORTED (1<<3)
#define GPU_FANSPEED_MONITORING_SUPPORTED (1<<4) /* Fanspeed monitoring based on fan voltage */
#define I2C_TEMP_MONITORING_SUPPORTED (1<<5)
#define I2C_FANSPEED_MONITORING_SUPPORTED (1<<6) /* Fanspeed monitoring using a i2c sensor chip */

typedef struct
{
    void *next;
    char *section;
    char *name;
    unsigned int value;
} cfg_entry;

typedef enum
{
    SDR,
    DDR
} mem_type;

typedef enum
{
    AGP,
    PCI,
    PCI_EXPRESS
} bus_type;

typedef enum
{
    UNKNOWN,
    DESKTOP,
    NFORCE,
    MOBILE
} gpu_type;

struct pci_ids {
    short id;
    const char *name;
    gpu_type gpu;
};

struct voltage
{
    unsigned char VID;
    float voltage;
};

struct performance
{
    int nvclk;
    int memclk;
    float voltage;
};

struct nvbios
{
    char *signon_msg;
    char *vendor_name;
    unsigned short device_id;
    char* version;
    unsigned char major;
    unsigned char minor;

    short volt_entries;
    short volt_mask;
    struct voltage volt_lst[4];

    short perf_entries;
    struct performance perf_lst[3];
};

typedef struct
{
    short device_id;
    short arch; /* Architecture NV10, NV15, NV20 .. */
    unsigned int reg_address;
    short number; /* internal card number */
    unsigned short devbusfn; /* needed for pciReadLong; further it will be used for X code */
    int irq;
    gpu_type gpu;
    char *card_name; /* Name of the card */
    char *dev_name; /* file in /dev to open */
    struct nvbios *bios; /* nvidia bios information; needed for geforcefx overclocking */
    short num_busses; /* number of available i2c busses */
    I2CBusPtr busses[3]; /* I2C busses on the videocard; this bus is needed for communication with sensor chips */
    I2CDevPtr sensor; /* When a sensor chip is available, this device pointer can be used to access it */ 
    volatile unsigned int mpll; /* default memory speed */
    volatile unsigned int mpll2;
    volatile unsigned int nvpll; /* default gpu speed */
    volatile unsigned int nvpll2;
} card_list;

typedef struct {
    char *card_name; /* Name of the card */
    short number; /* internal card number; used by the gtk client and set_card to see if we really need to switch cards */
    short supported; /* A bitmask that contains what card features are supported; A normal gpu can do gpu/memory overclocking but a nforce can do only gpu. */
    short device_id;
    short arch;
    unsigned int reg_address;
    unsigned short devbusfn;
    int irq; /* We might need the IRQ to sync NV-CONTROL info with nvclock */
    short base_freq;
    gpu_type gpu; /* Tells what type of gpu is used: mobile, nforce .. */
    short debug; /* Enable/Disable debug information */

    struct nvbios *bios; /* pointer to bios information */

    /* card registers */
    volatile unsigned int *PFB;
    volatile unsigned int *PBUS;
    volatile unsigned int *PMC;
    volatile unsigned int *PRAMDAC;
    volatile unsigned int *PEXTDEV;
    volatile unsigned char *PROM; /* Nvidia bios */
    volatile unsigned char *PCIO;

    /* Overclock range of speeds */
    short nvclk_min; 
    short nvclk_max; 
    short memclk_min; 
    short memclk_max; 
    
    /* Various GeforceFX/Geforce6 boards use different clocks in 3d. We need to store those clocks */
    short nvclk_3d;
    short memclk_3d;

    /* Memory info */
    int mem_type; /* needs to replace memory_type ?? */
    char* (*get_memory_type)(); /* Memory type: SDR/DDR */
    short (*get_memory_width)(); /* Memory width 64bit or 128bit */
    short (*get_memory_size)(); /* Amount of memory between 4 and 128 MB */

    /* AGP info */
    char* (*get_bus_type)(); /* Bus type: AGP/PCI/PCI-Express */
    short (*get_agp_rate)(); /* Current AGP rate: 1, 2, 4  or 8*/
    char* (*get_agp_status)(); /* Current AGP status: Enabled/Disabled */
    char* (*get_fw_status)(); /* Current FW status: Enabled/Disabled */
    char* (*get_sba_status)(); /* Current SBA status: Enabled/Disabled */
    char* (*get_supported_agp_rates)(); /* Supported AGP rates */

    /* Hardware monitoring */
    I2CDevPtr sensor; /* When a sensor chip is available, this device pointer can be used to access it */    
    int (*get_board_temp)(I2CDevPtr dev); /* Temperature of the sensor chip or for example the ram chips */
    int (*get_gpu_temp)(I2CDevPtr dev); /* Internal gpu temperature */
    float (*get_fanspeed)(); /* Get the speed in % from the pwm register in %*/
    void (*set_fanspeed)(float speed); /* Set the speed of the fan using the pwm register of the gpu */
    int (*get_i2c_fanspeed_rpm)(I2CDevPtr dev); /* Speed of the fan in rpm */
    float (*get_i2c_fanspeed_pwm)(I2CDevPtr dev); /* Duty cycle of the pwm signal that controls the fan */
    int (*set_i2c_fanspeed_pwm)(I2CDevPtr dev, float speed); /* By adjusting the duty cycle of the pwm signal, the fanspeed can be adjusted. */

    /* Pipeline stuff; On various Geforce6 boards disabled pixel/vertex pipelines can be re-enabled. */
    int (*get_pixel_pipelines)(unsigned char *mask);
    void (*set_pixel_pipelines)(unsigned char mask);
    int (*get_vertex_pipelines)(unsigned char *mask);
    void (*set_vertex_pipelines)(unsigned char mask);
    

    /* Overclocking */
    volatile unsigned int mpll; /* default memory speed */
    volatile unsigned int mpll2;
    volatile unsigned int nvpll; /* default gpu speed */
    volatile unsigned int nvpll2;
    float (*get_gpu_speed)();
    void (*set_gpu_speed)(unsigned int nvclk);
    float (*get_memory_speed)();
    void (*set_memory_speed)(unsigned int memclk);
    void (*reset_gpu_speed)();
    void (*reset_memory_speed)();
} Card, *CardPtr;

typedef struct
{
    int num_cards;
    card_list card[MAX_CARDS];
    cfg_entry *cfg;
    char *path; /* path to home directory */
    int nv_errno;
    char *nv_err_str;
} NVClock;

extern NVClock nvclock;
extern Card nv_card;

#ifdef __cplusplus
extern "C" {
#endif

int init_nvclock();
int set_card(int number);
void unset_card();

/* config file stuff */
int open_config();
int create_config(char *file);
int read_config(cfg_entry **cfg, char *file);
int parse_config(char *file);
void write_config(cfg_entry *cfg, char *file);
void add_entry(cfg_entry **cfg, char *section, char *name, int value);
void change_entry(cfg_entry **cfg, char *section, char *name, int value);
cfg_entry* lookup_entry(cfg_entry **cfg, char *section, char *name);

/* error handling */
char *get_error(char *buf, int size);
void set_error(int code);
void set_error_str(const char *err);

#ifdef __cplusplus    
};
#endif

#endif /* NVCLOCK_H */
