%  SLgtk: S-Lang language bindings for GTK+ widget set {{{
%
%  Copyright (C) 2003-2007 Massachusetts Institute of Technology
%  Copyright (C) 2002 Michael S. Noble <mnoble@space.mit.edu>
% 
%  SLgtk was partially developed at the MIT Center for Space Research,
%  under contract SV1-61010 from the Smithsonian Institution.
% 
%  Permission to use, copy, modify, distribute, and sell this software
%  and its documentation for any purpose is hereby granted without fee,
%  provided that the above copyright notice appear in all copies and
%  that both that copyright notice and this permission notice appear in
%  the supporting documentation, and that the name of the Massachusetts
%  Institute of Technology not be used in advertising or publicity
%  pertaining to distribution of the software without specific, written
%  prior permission.  The Massachusetts Institute of Technology makes
%  no representations about the suitability of this software for any
%  purpose.  It is provided "as is" without express or implied warranty.
%  
%  THE MASSACHUSETTS INSTITUTE OF TECHNOLOGY DISCLAIMS ALL WARRANTIES
%  WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
%  MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL THE MASSACHUSETTS
%  INSTITUTE OF TECHNOLOGY BE LIABLE FOR ANY SPECIAL, INDIRECT OR
%  CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
%  OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
%  NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
%  WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
% }}}

variable _current_namespace = current_namespace();
#ifnexists gtk_main
import("gtk", _current_namespace);
#endif

require("animio");

#ifexists __is_same % {{{
define __eqs()
{
   variable a = __pop_args(_NARGS);
   return __is_same(__push_args(a));
}
#endif % }}}

% Miscellaneous {{{

private variable Empty = "";

define norm_array() % {{{
{
   variable arr, max_value, type;
   switch(_NARGS)
   {  case 1: arr = (); max_value = 255; type = UChar_Type; }
   {  case 3: (arr, max_value, type) = (); }
   {  usage("Array_Type  norm_array(array, [maximum_value, return_type])"); }

   if (_typeof(arr) == type)
	return arr;

   %max_value *= 1.0;
   variable min_arr = min(arr);
   variable span = 1.0 * (max(arr) - min_arr);

   if (span != 0.0)
	arr = max_value * ((arr - min_arr)/span);
   else if (min_arr != 0) {
	arr = @arr;
	arr[*] = max_value;
   }
   return typecast (arr, type);
} % }}}

#ifnexists _is_numeric % {{{
define _is_numeric(datum)
{
   switch(_typeof(datum))
	{orelse	{case Char_Type} {case UChar_Type}
		{case Short_Type} {case UShort_Type}
		{case Integer_Type} {case UInteger_Type}
		{case Long_Type} {case ULong_Type}
		{case Float_Type} {case Double_Type} : return 1;}
	{ return 0;}
}
#endif % }}}

define _gtk_window_destroy_with(gtk_window, toplevel_gtk_window) % {{{
{
   gtk_window_set_transient_for(gtk_window,toplevel_gtk_window);
   gtk_window_set_destroy_with_parent(gtk_window,TRUE);
} % }}}

% }}}

% Window chaining % {{{

private variable North = 1, South = 2, East  = 3, West  = 4;
private variable WM_PLACEMENT_BROKEN = NULL;
define _gtk_window_chain();

private define test_move(win, event, status, signal) % {{{
{
   g_signal_handler_disconnect(win, signal.id);
   signal.id = 0;

   variable x, y, nx, ny;

   @status = 0;
   gtk_window_get_position(win, &x, &y);

   loop(50) {

	% Check window manager (WM) placement algorithm by "moving" win
	% to its current position; if new position results, WM is broken.
	gtk_window_move(win, x, y);
	gtk_window_get_position(win, &nx, &ny);

	if (nx != x or ny != y) {
	   @status = 1;
	   break;
	}
   }

   gtk_main_quit();
   return TRUE;
} % }}}

define __wm_placement_test() % {{{
{
   if (WM_PLACEMENT_BROKEN != NULL)
	return;

   variable win = gtk_window_new(GTK_WINDOW_TOPLEVEL), status = 0;
   gtk_widget_set_size_request(win, 1, 1);
   variable s = struct { id }; 
   s.id = g_signal_connect(win, "expose_event", &test_move,
						&WM_PLACEMENT_BROKEN, s);
   gtk_quit_add_destroy(gtk_main_level() + 1, win);
   gtk_widget_show_all(win);
   gtk_main();
} % }}}

private define gravitate_chain(chain) % {{{
{
   variable mx, my, mw, mh, sx, sy, sw, sh;
   gtk_window_get_position(chain.master, &mx, &my);
   gtk_window_get_size(chain.master, &mw, &mh);
   gtk_window_get_position(chain.slave, &sx, &sy);
   gtk_window_get_size(chain.slave, &sw, &sh);

   if ( (sy+sh) < my) return North;
   if (sy > (my+mh) ) return South;
   if ((sx + (sw/2.0)) > (mx + (mw/2.0)) ) return East;
   return West;

} % }}}

define forget_chain(master, chain)  % {{{
{
   variable slaves = g_object_get_data(master, "slaves");
   if (slaves == NULL) return;

   if (slaves.scid == chain.scid) {
	slaves = slaves.next;
	g_object_set_data(master, "slaves", slaves);
   }
   else {

	if (chain.prev != NULL)
	   chain.prev.next = chain.next;

	if (chain.next != NULL)
	   chain.next.prev = chain.prev;
   }

   variable last_slave = g_object_get_data(master, "last_slave");
   if (last_slave.scid == chain.scid)
	g_object_set_data(master, "last_slave", last_slave.prev);

} % }}}

private define break_chain() % {{{
{
   variable chain, forget;
   switch(_NARGS) 
   { case 1:  chain = (); forget = 1; }
   { (chain, forget) = (); }

   if (chain.mcid == 0) return;

   g_signal_handler_disconnect(chain.master, chain.mcid);
   g_signal_handler_disconnect(chain.master, chain.mdid);
   g_signal_handler_disconnect(chain.slave, chain.scid);
   g_signal_handler_disconnect(chain.slave, chain.sdid);
   g_signal_handler_disconnect(chain.slave, chain.kpid);
   g_signal_handler_disconnect(chain.slave, chain.krid);
   chain.mcid = 0;

   if (forget)
	forget_chain(chain.master, chain);
} % }}}

private define test_chain(slave, event, chain) % {{{
{
   if (chain.shift)
	break_chain(chain);

   % Potentially modify master gravity when user repositions slave manually
   if (gtk_window_is_active(slave))
	chain.grav = gravitate_chain(chain);

   return FALSE;
} % }}}

private define shift_chain(slave, event, chain) % {{{
{
   if (event.keyval != GDK_Shift_L and event.keyval != GDK_Shift_R)
	return FALSE;

   chain.shift = (event.type == GDK_KEY_PRESS);

   return TRUE;
} % }}}

private define pull_chain(master, event, chain) % {{{
{
   if (chain.master == NULL or chain.slave == NULL) return FALSE;

   g_signal_handler_block(chain.slave, chain.scid);

   variable mx, my, mw, mh, sw, sh, x, y;
   gtk_window_get_size(chain.master, &mw, &mh);
   gtk_window_get_position(master, &mx, &my);
   gtk_window_get_size(chain.slave, &sw, &sh);

   % FIXME: ultimately better to use frame dimensions here
   switch(chain.grav)
   { case North: x = mx + mw/2.0 - sw/2.0; y = my - sh - 40; }
   { case South: x = mx + mw/2.0 - sw/2.0; y = my + mh + 40; }
   { case East : x = mx + mw + 20; y = my + mh/2.0 - sh/2.0; }
   { x = mx - sw - 20; y = my + mh/2.0 - sh/2.0; }

   x = max([0, int(x)]);
   y = max([15, int(y)]);

   if (x + sw > chain.scrw)
	x = chain.scrw - sw;

   if (y + sh > chain.scrh)
	y = chain.scrh - sh;

   gtk_window_move(chain.slave, x, y);

   g_signal_handler_unblock(chain.slave, chain.scid);

   return FALSE;
} % }}}

private define delete_chain(thing, event, chain, is_slave) % {{{
{
   variable slaves = g_object_get_data(thing, "slaves");

   break_chain(chain, is_slave);

   chain = g_object_get_data(thing, "chain");
   if (chain == NULL) {
	% thing has no master: migrate slaves 2, 3, ... to slave 1
	variable master = slaves.slave;
	g_object_set_data(master, "chain", NULL);
	slaves = slaves.next;
   }
   else
	% thing has a master: migrate any slaves it has to that master
	master = chain.master;

   % Do the migration when slaves != NULL
   foreach chain (slaves)
	   _gtk_window_chain(master, chain.slave, chain.grav);

   gtk_widget_grab_focus(master);
   g_object_set_data(thing, "slaves", NULL);

   return FALSE;
} % }}}

private define remember_chain(master, chain)  % {{{
{
   variable slaves = g_object_get_data(master, "slaves");

   if (slaves == NULL) {
	slaves = chain;
	g_object_set_data(master, "slaves", slaves);
	g_object_set_data(master, "last_slave", chain);
   }
   else {
	variable last_slave = g_object_get_data(master, "last_slave");
	last_slave.next = chain;
	chain.prev = last_slave;
	last_slave = chain;
	g_object_set_data(master, "last_slave", last_slave);
   }

   chain.mdid = g_signal_connect(master,"delete_event", &delete_chain, chain,0);
} % }}}

define _gtk_window_chain() % {{{
{
   variable master, slave, gravity;
   switch(_NARGS)
   { case 2: (master, slave) = (); gravity = South; }
   { case 3: (master, slave, gravity) = (); }
   { usage("_gtk_window_chain(master_win, slave_win [, gravity=2(South)] );"); }

   variable chain = g_object_get_data(slave, "chain");
   if (chain != NULL)
	break_chain(chain);		% A slave may be in at most 1 chain

   if (master == NULL)			% Pass NULL master to unchain
	return;

   gtk_widget_add_events(master, GDK_POINTER_MOTION_MASK);

   chain = struct {master, slave, shift, mcid, mdid, scid, sdid, kpid, krid,
				grav, scrw, scrh, next, prev, data};

   chain.mcid = g_signal_connect(master, "configure_event", &pull_chain, chain);
   chain.kpid = g_signal_connect(slave, "key_press_event", &shift_chain, chain);
   chain.krid = g_signal_connect(slave, "key_release_event",&shift_chain,chain);
   chain.scid = g_signal_connect(slave, "configure_event", &test_chain, chain);
   chain.sdid = g_signal_connect(slave, "delete_event", &delete_chain,chain,1);

   chain.shift = 0;
   chain.master = master;
   chain.slave = slave;
   chain.grav = gravity;
   chain.scrw = gdk_screen_width();
   chain.scrh = gdk_screen_height();

   g_object_set_data(slave, "chain", chain);
   remember_chain(master, chain);

   if (WM_PLACEMENT_BROKEN == NULL)
	__wm_placement_test();

   if (WM_PLACEMENT_BROKEN) {
	gtk_window_set_gravity(master, GDK_GRAVITY_STATIC);
	gtk_window_set_gravity(slave, GDK_GRAVITY_STATIC);
   }

} % }}}

% }}}

% Documentation {{{
define _get_slgtk_doc_string()
{
   variable topic = ();
   variable content = get_doc_string_from_file(topic);
   if (content == NULL)
      content = sprintf("Error: could not retrieve help for <%S>:\n", topic);
   return content;
}

$1 = path_concat (path_concat (path_dirname (__FILE__), "help"),"slgtkfun.hlp");
if (NULL != stat_file ($1)) {
#ifexists add_help_file
   add_help_file($1);
#endif
#ifexists add_doc_file
   add_doc_file ($1);
#endif
}

private variable rseed = getpid();
private define random()
{
   rseed = (rseed * 0x5DEECEDL + 0xB) & ((1 shl 30) - 1);
   (rseed shr 6) / ((1 shl 24) * 1.0);
}

define _tmp_file_name(base)		% adapted from JED editor
{
   base = path_sans_extname (base);
   variable pid = getpid();
   loop(100) {
	variable fname = sprintf("%s%d.%d",base,int(10e5*random),pid);
	if (stat_file(fname) == NULL) return fname;
   }
   error("Unable to generate unique temporary file name.");
}
% }}}

% Menus {{{

private define _menu_new_with_signal()
{
   variable signal, labels, callback;
   variable cbargs = __pop_args(_NARGS-3);
   (signal,labels, callback) = ();

   variable menu = gtk_menu_new();
   foreach (labels) {

	variable item = ();

	if (not strcmp( strlow(item), "<separator>"))
	   item = gtk_separator_menu_item_new();
	else
	   item = gtk_menu_item_new_with_label(item);

	% Render the item now, for smarter allocation of menu width (later on)
	gtk_widget_show(item);
	gtk_menu_shell_append(menu,item);

	if (callback != NULL)
	   () = g_signal_connect(item,signal,callback,__push_args(cbargs));
   }
   return menu;
}

define _menu_new()
{
   if (_NARGS < 2)
      usage(_function_name + ": (menu_entry_labels, entry_selection_callback"+
						" [, callback_arg1, ...])");
   variable args = __pop_args(_NARGS);
   return _menu_new_with_signal("activate", __push_args(args));
}

define _option_menu_new(labels,default)
{
   % Create an option menu from given array of label strings,
   % and visually reflecting the given default selection
   variable omenu = gtk_option_menu_new();
   gtk_option_menu_set_menu (omenu, _menu_new(labels,NULL));
   gtk_option_menu_set_history (omenu, default);
   return omenu;
}
% }}}

% Preferences {{{

private variable PrefSelector = struct {
	description,
	chooser,			% widget displaying the prefs content
	content,			% either: list of menu labels or func
	value,				% initial value for this preference
	prev_value,			% previous value
	typeclass,			% is value a scalar, vector, etc?
	equals,				% comparison func
	dirty,				% is current value != previous value?

	widget,				% hbox encapsulating the description
					% label, a widget visualizing the curr
					% pref value, and the chooser; this
					% simplifies layout w/in other widgets,
					% as well as [en|dis]abling sensitivity

	omenu,				% pertinent to menu-based selectors
	floating,

	args				% pertinent to function-based selectors
};

private variable _pref_callback_hook = NULL;
private variable _pref_callback_hook_args = NULL;

private define _pref_callback(item,event,pref)
{
   if (pref.omenu != NULL)
      pref.value = g_list_index(gtk_container_get_children(pref.chooser),item);
   % else assume pref.value has been set by callback attached to chooser

   pref.dirty = not( (@pref.equals)(pref.value,pref.prev_value) );

   if (_pref_callback_hook != NULL)
	(@_pref_callback_hook) (pref, __push_args(_pref_callback_hook_args));
   
   return FALSE;
}

private define _pref_assign_value(pref,value)
{
   if (pref.typeclass == 1)
	return value;
   return @value;
}

define _pref_sync(pref,revert)
{
   if (revert)
	pref.value = _pref_assign_value(pref,pref.prev_value);
   else
	pref.prev_value = _pref_assign_value(pref,pref.value);

   pref.dirty = 0;
   if (pref.floating) {
	pref.floating = 0;
	if (gtk_menu_get_attach_widget(pref.chooser) != NULL)
	   gtk_menu_detach(pref.chooser);
	gtk_option_menu_set_menu (pref.omenu, pref.chooser);
	g_object_unref(pref.chooser);
   }
}

define _pref_set_callback_hook()
{
   _pref_callback_hook_args = __pop_args(_NARGS - 1);
   _pref_callback_hook = ();
}

define _pref_unset_callback_hook()
{
   _pref_callback_hook_args = NULL;
   _pref_callback_hook = NULL;
}

define _pref_steal_menu(pref)
{
   if (pref.omenu != NULL) {
	pref.floating = 1;
	() = g_object_ref(pref.chooser);
	gtk_menu_detach(pref.chooser);
	return pref.chooser;
   }
   return NULL;
}

private define _pref_destroy();
private define _pref_make_widget(pref)
 {
   variable hbox = gtk_hbox_new(FALSE,5);
   gtk_container_set_border_width(hbox,5);
   variable label = gtk_label_new(pref.description);
   gtk_box_pack_start(hbox,label,FALSE,FALSE,0);

   if (pref.args == NULL) {

      	% Assume array of strings:  chooser will be a menu
	pref.chooser = _menu_new_with_signal("button-release-event",
				pref.content, &_pref_callback, pref);
	pref.omenu = gtk_option_menu_new();
	gtk_option_menu_set_menu (pref.omenu, pref.chooser);
	gtk_option_menu_set_history (pref.omenu, pref.value);
	gtk_box_pack_end(hbox,pref.omenu,FALSE,FALSE,0);
   }
   else {
	
	pref.chooser = (@pref.content) (pref.value, __push_args(pref.args));
	() = g_signal_connect(pref.chooser,"expose_event",
			      		&_pref_callback, pref);
	gtk_box_pack_end(hbox,pref.chooser,FALSE,FALSE,0);
   }
	
   () = g_signal_connect_swapped(hbox,"destroy", &_pref_destroy, pref);
   pref.typeclass = __class_type( typeof(pref.value) );

   pref.prev_value = _pref_assign_value(pref,pref.value);
   pref.floating = 0;
   pref.dirty  = 0;
   return hbox;
}

private define _pref_destroy(pref)
{
   pref.widget = _pref_make_widget(pref);
}

define _pref_new()
{
   if (_NARGS != 3)
	usage(_function_name+ "(description, menu_labels, initial_value);");

   variable pm = @PrefSelector;

   (pm.description, pm.content, pm.value) = ();
   pm.equals = &__eqs;
   pm.widget = _pref_make_widget(pm);

   return pm;
}

define _pref_new_from_func()
{
   if (_NARGS < 4)
	usage(_function_name+ "(description, initial_value, compare_func,"+
				"generate_func [, gfarg2, gfarg3, ...]);");

   variable pm = @PrefSelector;
   pm.args = __pop_args(_NARGS - 4);
   (pm.description, pm.value, pm.equals, pm.content) = (); 
   pm.widget = _pref_make_widget(pm);
   return pm;
}
% }}}

% Printing {{{

private define activate_entry_box(ignored, which, dialog_ctx)
{
   if (which == 0) {
      	gtk_widget_set_sensitive(dialog_ctx.file.box, FALSE);
      	gtk_widget_set_sensitive(dialog_ctx.cmd.box, TRUE);
	gtk_widget_grab_focus(dialog_ctx.cmd.value);
   }
   else {
      	gtk_widget_set_sensitive(dialog_ctx.cmd.box, FALSE);
      	gtk_widget_set_sensitive(dialog_ctx.file.box, TRUE);
	gtk_widget_grab_focus(dialog_ctx.file.value);
   }
}

private define print_action_cb(win, dialog_ctx, user_cb, user_cb_args)
{
   variable print_ctx = dialog_ctx.print_ctx;
   print_ctx.dest = gtk_widget_is_sensitive(dialog_ctx.file.box);
   print_ctx.cmd = gtk_entry_get_text(dialog_ctx.cmd.value);
   print_ctx.fname  = gtk_entry_get_text(dialog_ctx.file.value);
   print_ctx.orient = dialog_ctx.orient.value;
   print_ctx.size   = dialog_ctx.size.value;

   (@user_cb) ( print_ctx, __push_args(user_cb_args) );
   gtk_widget_destroy(win);
}

private define make_entry_box(container,label_text,initial_value)
{
   variable eb = struct { box, value};
   eb.box = gtk_hbox_new(FALSE,10);
   eb.value = gtk_entry_new();
   gtk_entry_set_width_chars(eb.value, 20);
   gtk_entry_set_max_length(eb.value, 255);
   gtk_entry_set_text(eb.value,initial_value);
   gtk_box_pack_end(eb.box, eb.value, TRUE, TRUE, 10);
   gtk_box_pack_start(eb.box, gtk_label_new(label_text), FALSE, FALSE, 10);
   gtk_container_add(container,eb.box);
   return eb;
}

#ifnexists GtkPrintContext
typedef struct {
   title,		% window title for print dialog
   dest,		% will output be sent to printer, or only to file?
   cmd,			% o/s command for creating a print job
   fname,		% output filename
   orient,		% 0 = PORTRAIT, 1 = LANDSCAPE
   size			% 0 = LETTER, 1 = LEGAL, 2 = A4
} GtkPrintContext;
#endif

define _print_context_set_defaults()
{
   variable ctx = NULL;
   switch(_NARGS)
   { case 1: ctx = ();  if (ctx == NULL) ctx = @GtkPrintContext; }

   if (typeof(ctx) != GtkPrintContext)
	usage("_print_context_set_defaults(GtkPrintContext);");

   if (ctx.title == NULL)
	ctx.title = "Print";
 
   if (ctx.cmd == NULL)
	ctx.cmd = "lpr";

   if (ctx.fname == NULL)
	ctx.fname = "slgtk.ps";

   if (ctx.dest == NULL)
	ctx.dest = 0;		% default to printer, not file

   if (ctx.orient == NULL)
	ctx.orient = 0;		% default to PORTRAIT

   if (ctx.size == NULL)
	ctx.size = 0;		% default to LETTER

   return ctx;
}

define _print_context_new()
{
   return _print_context_set_defaults(NULL);
}

define _print_dialog()
{
   variable ctx, user_cb, user_cb_args;

   if (_NARGS < 2)
	usage("_print_dialog( print_context, callback, [, cbarg1, ...]]");

   user_cb_args = __pop_args(_NARGS-2);
   (ctx, user_cb) = ();

   !if (_is_callable(user_cb))
	error(sprintf("_print_dialog: %S is not a callable function",user_cb));

   if (ctx == NULL)
	ctx = _print_context_new();
   else
	ctx = _print_context_set_defaults(ctx);

   variable win = gtk_dialog_new();
   gtk_window_set_title(win, ctx.title);
   gtk_container_set_border_width(win,5);
   gtk_window_resize(win,300,180);
   gtk_window_set_modal(win,TRUE);
   gtk_dialog_set_has_separator(win,FALSE);
   () = g_signal_connect (win, "destroy", &gtk_main_quit);

   variable frame = gtk_frame_new("Print To");
   gtk_box_pack_start(gtk_dialog_get_vbox(win),frame,FALSE,FALSE,5);

   variable vbox = gtk_vbox_new(TRUE,5);
   gtk_container_add(frame,vbox);

   variable align = gtk_alignment_new(0.5,0.5,0.5,0.5); 
   variable hbox = gtk_hbox_new(FALSE,0);
   gtk_container_add(align,hbox);
   variable prb = gtk_radio_button_new_with_label(NULL,"Printer");
   gtk_box_pack_start(hbox,prb,FALSE,FALSE,60);
   variable frb = gtk_radio_button_new_with_label_from_widget(prb,"File");
   gtk_box_pack_start(hbox,frb,FALSE,FALSE,0);
   gtk_box_pack_start(vbox,align,FALSE,FALSE,0);

   if (ctx.dest == 0)
	gtk_toggle_button_set_active(prb,TRUE);
   else
	gtk_toggle_button_set_active(frb,TRUE);

   variable dialog_ctx = struct { print_ctx, cmd, file, orient, size };
   dialog_ctx.print_ctx = ctx;
   dialog_ctx.cmd = make_entry_box(vbox,"Print Command", ctx.cmd);
   dialog_ctx.file = make_entry_box(vbox,"File Name", ctx.fname);
   activate_entry_box(NULL, ctx.dest, dialog_ctx);
   () = g_signal_connect (prb, "clicked", &activate_entry_box, 0, dialog_ctx);
   () = g_signal_connect (frb, "clicked", &activate_entry_box, 1, dialog_ctx);

   variable b = gtk_dialog_add_button(win,GTK_STOCK_PRINT,0);
   () = g_signal_connect_swapped(b,"clicked",&print_action_cb,win,dialog_ctx,
						     user_cb, user_cb_args);
   gtk_widget_grab_focus(b);

   % UI nicety: automove input focus to print button after input
   () = g_signal_connect_swapped(dialog_ctx.cmd.value, "activate",
				 		&gtk_widget_grab_focus,b);
   () = g_signal_connect_swapped(dialog_ctx.file.value, "activate",
				 		&gtk_widget_grab_focus,b);

   dialog_ctx.orient = _pref_new("Orientation:", ["Portrait","LandScape"],
	 							ctx.orient);
   dialog_ctx.size   = _pref_new("Paper Size :", ["Letter","Legal","A4"],
	 							ctx.size);
   hbox = gtk_hbox_new(TRUE,10);
   gtk_container_add(hbox, dialog_ctx.orient.widget);
   gtk_container_add(hbox, dialog_ctx.size.widget);
   gtk_box_pack_start(vbox,hbox,TRUE,TRUE,10);

   b = gtk_dialog_add_button(win,GTK_STOCK_CANCEL,0);
   () = g_signal_connect_swapped(b,"clicked",&gtk_widget_destroy,win);

   gtk_widget_show_all(win);
   gtk_main();
}
% }}}

% Colors {{{
private define alloc_color(name)
{
   eval( sprintf("gdk_color_parse(\"%s\");",  name), _current_namespace);
   variable value = ();
   if (value == NULL)
	error("Could not allocate color %s",name);
   value;
   eval( sprintf("variable gdk_%s = ();", name), _current_namespace);
   value;
}

% White and black given first (and must remain so!), so they're easy to cull
private variable color_names = ["white", "black","red",  "green", "blue",
"magenta", "orange", "purple", "brown", "yellow", "grey" ];
private variable gdk_colors = array_map(Struct_Type, &alloc_color, color_names);
define _get_predefined_colors()
{
   return gdk_colors;
}

private define change_color(dialog_window, colorsel, color, swatch)
{
   variable new_color = gtk_color_selection_get_current_color(colorsel);
   color.pixel = new_color.pixel;
   color.red   = new_color.red;
   color.green = new_color.green;
   color.blue  = new_color.blue;
   gtk_widget_modify_bg (swatch, GTK_STATE_NORMAL, new_color);
   gtk_widget_destroy(dialog_window);
}

private define launch_color_selector(gdkcolor,swatch)
{
   variable win = gtk_color_selection_dialog_new ("Select Color");
   gtk_window_set_position (win, GTK_WIN_POS_MOUSE);

   variable colorsel = gtk_color_selection_dialog_get_colorsel(win);
   gtk_color_selection_set_has_palette (colorsel, TRUE);
   gtk_color_selection_set_current_color (colorsel,gdkcolor);

   () = g_signal_connect_swapped (
		gtk_color_selection_dialog_get_ok_button(win),
		"clicked",
		&change_color,
		win,
		colorsel,
		gdkcolor,
		swatch);

   () = g_signal_connect_swapped (
		gtk_color_selection_dialog_get_cancel_button(win),
		"clicked",
		&gtk_widget_destroy,
		win);

   gtk_window_set_modal(win,TRUE);		% allow no other activity
   gtk_widget_show (win);
}

define _color_button_new(gdkcolor)
{
   % Create a button which displays the given color and supports modifying
   % it through a callback on the color selector launched at button press.
   variable button = gtk_button_new();

   variable frame = gtk_aspect_frame_new(NULL,0.5,0.5,3,FALSE);
   gtk_frame_set_shadow_type(frame,GTK_SHADOW_IN);
   gtk_widget_set_size_request(frame,60,10);

   variable swatch = gtk_drawing_area_new();
   gtk_widget_modify_bg (swatch, GTK_STATE_NORMAL, gdkcolor);
   gtk_container_add(frame,swatch);
   gtk_container_add(button,frame);

   % How does this work?  Well, recall that gdkcolor is a struct, and thus
   % a reference, which gives the useful property of allowing its fields
   % to be modified in place when a new color selection is applied.
   () = g_signal_connect_swapped(button, "clicked", &launch_color_selector,
				 			gdkcolor, swatch);
   return button;
}

private variable _display_depths = NULL;
define _gdk_query_depths() {
   if (_display_depths == NULL)
	_display_depths = gdk_query_depths();
   return @_display_depths;
}
% }}}

% Dialogs {{{
define _info_window(title,text) % {{{
{
   if (orelse {_NARGS != 2} {title == NULL} {text == NULL})
      usage("_info_window(title,text_to_display)");

   variable win = gtk_dialog_new();
   gtk_window_set_title(win,title);
   gtk_container_set_border_width (win, 10);
   gtk_window_set_default_size(win,520,600);	% approximate an 80x40 terminal

   variable tview = gtk_text_view_new();
   gtk_text_view_set_editable (tview, FALSE);
   gtk_text_view_set_cursor_visible(tview,FALSE);
   gtk_text_buffer_insert_at_cursor(gtk_text_view_get_buffer(tview), text, -1);

   variable scroller = gtk_scrolled_window_new(NULL,NULL);
   gtk_scrolled_window_set_policy(scroller,
				GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
   gtk_box_pack_start ( gtk_dialog_get_vbox(win), scroller, TRUE, TRUE, 0);
   gtk_container_add(scroller,tview);

   variable button = gtk_button_new_from_stock(GTK_STOCK_OK);
   gtk_box_pack_start ( gtk_dialog_get_action_area(win), button, TRUE, TRUE, 0);
   () = g_signal_connect_swapped(button,"clicked", &gtk_widget_destroy, win);
   gtk_widget_show_all(win);
} % }}}

define _error_dialog(title, message) % {{{
{
   variable d = gtk_dialog_new();
   if (title == NULL) title = Empty;
   gtk_window_set_title(d, "Error: "+title);
   gtk_window_set_default_size(d, 200, 150);

   variable w = gtk_dialog_get_vbox(d);
   gtk_container_add(w, gtk_image_new_from_stock(GTK_STOCK_DIALOG_ERROR,
							GTK_ICON_SIZE_DIALOG));
   gtk_container_add(w, gtk_label_new(message));
   () = gtk_dialog_add_button(d, GTK_STOCK_OK,1);
   gtk_widget_show_all(d);
   () = gtk_dialog_run(d);
   gtk_widget_destroy(d);
} % }}}

define _input_dialog() % {{{
{
   variable label, title = NULL, description = NULL;

   switch(_NARGS)
   { case 1: label = (); }
   { case 2: (label,title) = (); }
   { case 3: (label,title,description) = (); }
   { usage("input_dialog(label [, title [, description ]]"); }

   if (label == NULL) label = Empty;
   if (title == NULL) title = "Input Dialog";
   if (description == NULL) description = Empty;

   variable d = gtk_dialog_new();
   gtk_window_set_title(d, title);
   gtk_container_set_border_width(d, 10);
   gtk_window_set_default_size(d, 200, 150);

   variable entry = gtk_entry_new();
   gtk_entry_set_width_chars(entry, 32);
   gtk_entry_set_max_length(entry, 255);

   label = gtk_label_new(label);
   variable hbox = gtk_hbox_new(FALSE, FALSE);
   gtk_container_add(hbox, label);
   gtk_container_add(hbox, entry);

   variable w = gtk_dialog_get_vbox(d);
   if (description != Empty)
	gtk_container_add(w, gtk_label_new(description));

   gtk_container_add(w, hbox);
   () = gtk_dialog_add_button(d, GTK_STOCK_OK,1);
   gtk_widget_show_all(d);
   () = gtk_dialog_run(d);
   variable result = gtk_entry_get_text(entry);
   gtk_widget_destroy(d);
   return result;
} % }}}

% Image save dialog support code % {{{
private variable Image_Formats = NULL;
private variable Image_Formats_Writable = Assoc_Type[String_Type, Empty];;

private define get_image_file_formats() 
{
   if (Image_Formats != NULL) return;

   Image_Formats = Assoc_Type[String_Type, Empty];
   variable list = gdk_pixbuf_get_formats();
   variable names = String_Type[0];

   foreach( [0 : g_slist_length(list) -1] ) {

	variable i = (), format = g_slist_nth_data(list,i);
	variable name = gdk_pixbuf_format_get_name(format);
	variable exts = gdk_pixbuf_format_get_extensions(format);
	variable writable = gdk_pixbuf_format_is_writable(format);

	foreach(exts) {
	   variable ext = ();
	   Image_Formats[name] = ext;
	   if (writable)
		Image_Formats_Writable[name] = "." + ext;
	}
   }

   % GdkPixbuf doesn't support GIF output, but SLgtk-bundled gif module does
   Image_Formats_Writable["gif"] = "." + "gif";
}

private define get_file_selection(selector)
{
   variable file = gtk_file_selection_get_filename (selector);
   variable st = stat_file (file);
   if (andelse {st != NULL} { stat_is ("dir", st.st_mode)})
	return Empty;
   return file;
}

private define choose_callback(button, ctx, kind)
{
   ctx.kind = kind;

   variable selector = ctx.win;
   variable file = get_file_selection(selector);
   if (file == Empty) return;
  
   variable dir = path_dirname (file);
   file = path_sans_extname (path_basename (file));

   if (file == Empty) return;

   file += Image_Formats_Writable[kind];
   file = path_concat (dir, file);
   gtk_file_selection_set_filename (selector, file);
}

private define _image_choice_button(box, group, kind, ctx)
{
   variable button = gtk_radio_button_new_with_label (group, kind);
   () = g_signal_connect (button, "clicked", &choose_callback, ctx, kind);
   gtk_box_pack_start (box, button, 1, 1, 0);
   group = gtk_radio_button_get_group (button);

   return (button, group);
}

private define _image_save_ok (ctx)
{
   variable file = get_file_selection(ctx.win);

   if (file != Empty) {
	variable dir = path_dirname (file);
	if (stat_file(dir) == NULL) {
		_error_dialog("during file save",
		      		sprintf("Directory %s does not exist!",dir));
		return;
	}
	variable ext = path_extname (file);
	if (ext == Empty) 
	   file += Image_Formats_Writable[ctx.kind];
   }

   ctx.filename = file;
   gtk_widget_destroy (ctx.win);
   ctx.win = NULL;
}
% }}}

define _image_save_dialog() % {{{
{
   variable kind = qualifier("kind", "Image");
   variable format_names = NULL;
   if (_NARGS == 1)
      format_names = ();
   else 
      format_names = [ "png", "jpeg" ];

   if (typeof(format_names) == String_Type)
	format_names = [ format_names ];
   else if (_typeof(format_names) != String_Type)
	usage("(filename, kind) = _image_save_dialog( [format_name, ...] )");

   if (Image_Formats == NULL)
	get_image_file_formats();

   variable ctx = struct { filename, kind, win};
   ctx.win = gtk_file_selection_new ("Save " + kind);

   gtk_file_selection_hide_fileop_buttons (ctx.win);
   gtk_window_set_position (ctx.win, GTK_WIN_POS_MOUSE);
   gtk_window_set_modal (ctx.win, 1);

   () = g_signal_connect (ctx.win, "destroy", &gtk_main_quit);
   () = g_signal_connect_swapped (gtk_file_selection_get_ok_button(ctx.win),
				  "clicked", &_image_save_ok, ctx);

   variable button = gtk_file_selection_get_cancel_button (ctx.win);
   () = g_signal_connect_swapped (button,"clicked",&gtk_widget_destroy,ctx.win);

   variable frame = gtk_frame_new ("File Type");
   variable box = gtk_hbox_new (1, 0);
   gtk_container_add (frame, box);

   variable group = NULL, first = 1;
   foreach(format_names) {
	variable format = ();
	if (Image_Formats_Writable[format] == Empty) continue;
	(button, group) = _image_choice_button (box, group, format, ctx);
	if (first) {
	   ctx.kind = format;
	   gtk_toggle_button_set_active (button, TRUE);
	   first = 0;
	}
   }

   box = gtk_file_selection_get_action_area (ctx.win);
   gtk_box_pack_start (box, frame, 1, 1, 10);

   gtk_widget_show_all (ctx.win);
   gtk_main ();

   if (ctx.filename == Empty) ctx.filename = NULL;
   return (ctx.filename, ctx.kind);
} %}}}

define _pixbuf_save(pixbuf, file, kind) % {{{
{
   variable e;
   try (e) {
	if (kind == "gif")
	   gif_write(file, gdk_pixbuf_get_pixels(pixbuf));
	else
	   gdk_pixbuf_save(pixbuf, file, kind);
   }
   catch AnyError: {
	_error_dialog("gdk_pixbuf_save",
	      	sprintf("Could not save to %s\n\n%s",file, e.message));
   }
} %}}}

define _gdk_pixbuf_get_formats() % {{{
{
   if (Image_Formats == NULL)
	get_image_file_formats();

   return assoc_get_keys(Image_Formats);
} % }}}

define _image_format_is_writable() % {{{
{
   % Indicate whether images in the given format
   % (e.g. "png" or "gif") can be output by SLgtk
   if (_NARGS == 1)
	variable format = ();
   else
	usage("_image_format_writable(image_format_name)");

   if (Image_Formats == NULL)
	get_image_file_formats();

   return (Image_Formats_Writable[strlow(format)] != Empty);
} % }}}
% }}}

#ifdef GTK_MODULE_STATIC_BINARY % {{{
   try { () = evalfile("gtk_binary_customize"); }
   catch OpenError;
#endif % }}}

provide ("gtk");
