
private variable _version_string = "0.1.5";

private variable intro_string =  % {{{
   "gPrompt version " + _version_string + "\n" +
   "Synopsis: a lightweight S-Lang prompt in Gtk (mnoble@space.mit.edu)\n"+
   "Type arbitrary S-Lang expression above, then hit ENTER to evaluate.\n"+
   "Use:\n"+
   "   - help <topic> for basic S-Lang help\n"+
   "   - Arrow keys for simple command history navigation\n"+
   "   - Shift-Mouse2 to see current namespace, Shift-Mouse3 to change it\n"+
   "Type or click 'quit' to terminate.\n";
%
% See ../examples/gprompt_example.sl for sample usages.
% }}}

require("gtk");

private define help () % {{{
{
   if (_NARGS == 0)
	return "Usage:  help <topic>";

   variable name = ();
   variable text = get_doc_string_from_file (name);
   if (text == NULL)
	text = sprintf("No help available for %s", name);

   return text;
} % }}} 

private define quit(prompt) % {{{
{
   variable top = gtk_widget_get_toplevel (prompt.input);
   if (gtk_widget_toplevel(top))
	gtk_widget_destroy(top);
} % }}}

private define log_write(prompt, msg) % {{{
{
  variable iter = gtk_text_buffer_get_iter_at_mark(prompt.buffer, prompt.mark);
  if (msg != "")
	gtk_text_buffer_insert (prompt.buffer, iter, msg + "\n", -1);
  gtk_text_view_scroll_mark_onscreen(prompt.output, prompt.mark);
} % }}}

private define log() % {{{
{
   variable args = __pop_args(max([0,_NARGS - 1])), this = ();

   switch(length(args))
   { case 0: return; }
   { case 1: variable msg = args[0].value; }
   { msg = sprintf( __push_args(args) ); }

   this.logger( msg, __push_args(this.logger_args) );

} % }}}

private define intro(this) % {{{
{
   this.log_write(intro_string);
} % }}}

private define evaluate(prompt, expr) % {{{
{
   variable ad = _auto_declare, err;

   % Strip off leading prompt, if present, for more permissive cut/paste
   (expr, ) = strreplace(expr, prompt.label, "", 999);
#ifexists get_prompt
   (expr, ) = strreplace(expr, get_prompt(), "", 999);
#endif

   variable e2 = sprintf("_auto_declare=1; %s; _auto_declare = %d;", expr, ad);

   try (err) { eval(e2, prompt.nspace); }
   catch AnyError: { prompt.log(err.message); return 0; }

   return 1;
} % }}}

private define visible(prompt) % {{{
{
   andelse {prompt.input != NULL} {gtk_widget_visible(prompt.input)};
} % }}}

private define set_prompt_from_history(prompt, direction) % {{{
{
   %if (prompt.hist_size == 0) return;

   variable new_index = prompt.hist_index + direction;

   if (direction == -1)
	variable cmd = prompt.hist[prompt.hist_index];
   else if (new_index <= prompt.hist_size)
	cmd = prompt.hist[new_index];
   else
	cmd = "";

   gtk_entry_set_text(prompt.input, cmd);
   gtk_editable_set_position(prompt.input, -1);

   if (new_index > 0 and new_index <= prompt.hist_size)
	prompt.hist_index = new_index;
} % }}}

private define mouse_callback(entry, event, prompt) % {{{
{
  !if (event.state & GDK_SHIFT_MASK) return FALSE;
  if (event.button == 1) return FALSE;

  if (event.button == 3) {

	variable new_ns = _input_dialog("Namespace ","Change S-Lang Namespace",
	"Enter name of the namespace in which subsequent S-Lang statements\n"+
	"will be evaluated.  If the namespace does not exist it will be "+
	"created.\n"+
	"Leave the field blank if you decide not to change the namespace.\n\n"+
	"NOTE: Only expert users should change namespaces, because existing\n"+
	"      symbols will not be propagated to the new namespace.\n");

	new_ns = strtrim(new_ns);
	if (new_ns !=  "" and new_ns != prompt.nspace)
	   prompt.nspace = new_ns;
  }
  prompt.log("Namespace: %S", prompt.nspace);
  return TRUE;
} % }}}

private define key_callback(entry, event, prompt) % {{{
{
   if (event.keyval == GDK_Up)
	set_prompt_from_history(prompt, -1);
   else if (event.keyval == GDK_Down)
	set_prompt_from_history(prompt, 1);
   else if (event.keyval == GDK_Tab)  {
	variable focus_next = g_object_get_data(entry, "focus_next");
	if (focus_next != NULL)
	   gtk_widget_grab_focus(focus_next);
	else
	   return FALSE;
   }
   else
	return FALSE;

   return TRUE;
} % }}}

private define eval_callback(entry, prompt) % {{{
{
   variable expr = strcompress(gtk_entry_get_text(prompt.input)," \t");
   if (expr == "") return;

   prompt.log("%s%s", prompt.label, expr);	% log what was typed

   prompt.hist = [prompt.hist, expr];		% save it in history buffer
   prompt.hist_size++;
   prompt.hist_index = prompt.hist_size;

   expr = strtrim_end(expr, ";");		% semicolons are ok!

   gtk_entry_set_text(prompt.input, "");	% clear prompt

   variable prev_stack_depth = _stkdepth();

   if (strncmp(expr,"help",4) == 0) {
	if (strlen(expr) > 4) {
	   if (expr[4] == ' ')
		help( expr[[5:]] );
	   else
		() = prompt.eval(expr);
	}
	else
	   help();
   }
   else if (expr == "quit")
	return @(prompt.quit)(prompt);
   else
	() = prompt.eval(expr);

   variable stack_leftovers = _stkdepth() - prev_stack_depth;
   if (stack_leftovers > 0) {
	variable top = string(());
	prompt.log(top);
	_pop_n(stack_leftovers-1);
   }
} % }}}

private define make_output_object() % {{{
{
   variable prompt = ();

   prompt.log = &log;
   prompt.log_write = &log_write;
   prompt.intro = &intro;

   if (_NARGS > 1) {
	prompt.logger_args = __pop_args( max([0, _NARGS-2]) );
	prompt.logger = ();
	!if (_is_callable(prompt.logger))
	    throw UsageError, sprintf("%S is not callable", prompt.logger);
   }
   else {
	prompt.logger = prompt.log_write;
	prompt.logger_args = Struct_Type[0];
   }

   variable swin = gtk_scrolled_window_new (NULL, NULL);
   gtk_scrolled_window_set_policy(swin,
	 			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
   gtk_scrolled_window_set_shadow_type(swin, GTK_SHADOW_ETCHED_IN);


   variable textview = gtk_text_view_new ();
   gtk_text_view_set_editable(textview, FALSE);
   gtk_widget_unset_flags (textview, GTK_CAN_FOCUS);
   gtk_container_add ( swin, textview);

   () = g_signal_connect(textview, "button_press_event",&mouse_callback,prompt);

   prompt.output_container = swin;
   prompt.output = textview;

   prompt.buffer = gtk_text_view_get_buffer(textview);
   variable iter = gtk_text_buffer_get_iter_at_offset(prompt.buffer, -1);
   prompt.mark = gtk_text_buffer_create_mark (prompt.buffer, "end", iter, 0);

} % }}}

private define make_input_object(prompt) % {{{
{
   variable hbox = gtk_hbox_new(FALSE, 0);
   variable ebox = gtk_event_box_new();			  % put label in event
   gtk_box_pack_start(hbox, ebox, FALSE, FALSE, 0);	  % box so that it can
   gtk_widget_modify_bg(ebox,GTK_STATE_NORMAL,gdk_white); % have colored bg
   gtk_container_add(ebox, gtk_label_new(prompt.label));

   variable entry = gtk_entry_new();
   gtk_entry_set_width_chars(entry, 20);
   gtk_entry_set_has_frame(entry, FALSE);
   gtk_box_pack_start(hbox, entry, TRUE, TRUE, 0);

   () = g_signal_connect(entry, "activate", &eval_callback, prompt);
   () = g_signal_connect(entry, "button_press_event", &mouse_callback, prompt);
   () = g_signal_connect(entry, "key_press_event", &key_callback, prompt);

   prompt.input = entry;

   variable frame = gtk_frame_new(NULL);
   gtk_container_add(frame, hbox);
   prompt.input_container = frame;

} % }}}

define gprompt_new() % {{{
{
   variable p = struct {

	input,			% widget into which text is typed
	input_container,	%   and its parent widget, for layout

	output,			% scrolling widget into which output is logged
	output_container,	%   and its parent widget, for layout

	logger,			% customizable func which formats/emits log
				% entries to scrollable output widget
	logger_args,		% extra args to pass to each invocation

	log,			% wrapper to simplify logger invocation/use
				% accepts sprintf-style combination of args

	log_write,		% func which causes log message to actually
				% appear in logger widget; may be called
				% by custom logger function

	buffer, mark,		% anchors logging to end of scroll window

      	eval,			% func used to evaluate input commands
	nspace,			% the namespace in which they'll be evaluated
	intro,			% func which emits startup message to log

	hist,			% simple up/down arrow history mechanism
	hist_index, hist_size,
	label,			% the text of the prompt itself!
	quit,			% function to call when "quit" is typed

	visible,		% func which indicates if prompt is visible
	vbox			% top-level vbox containing all widgets
	};

   p.eval = &evaluate;
   p.quit = &quit;
   p.visible = &visible;
   p.nspace = "gprompt";
   p.hist = [""];
   p.hist_size = 0;
   p.hist_index = 0;
   p.label = "slang> ";

   variable args = __pop_args(_NARGS);
   make_output_object( __push_args(args), p );
   make_input_object(p);

   p.vbox = gtk_vbox_new (FALSE, 0);
   gtk_box_pack_start(p.vbox, p.input_container, FALSE, FALSE, 0);
   gtk_box_pack_start(p.vbox, p.output_container, TRUE, TRUE, 0);

   return p;
} % }}}

provide("gprompt");
