/*
 * ewfcommon
 * Common functions for the libewf tools
 *
 * Copyright (c) 2006, Joachim Metz <forensics@hoffmannbv.nl>,
 * Hoffmann Investigations. All rights reserved.
 *
 * Refer to AUTHORS for acknowledgements.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * - Redistributions of source code must retain the above copyright notice,
 *   this list of conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 * - Neither the name of the creator, related organisations, nor the names of
 *   its contributors may be used to endorse or promote products derived from
 *   this software without specific prior written permission.
 * - All advertising materials mentioning features or use of this software
 *   must acknowledge the contribution by people stated in the acknowledgements.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER, COMPANY AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "libewf_common.h"

#include <stdlib.h>
#include <string.h>

#include "libewf_definitions.h"
#include "libewf_notify.h"
#include "libewf_string.h"

/* Determines the units strings of a certain factor value
 */
char *determine_units_string( int factor )
{
	switch( factor )
	{
		case 0:
			return( "B" );
		case 1:
			return( "kB" );
		case 2:
			return( "MB" );
		case 3:
			return( "GB" );
		case 4:
			return( "TB" );
		case 5:
			return( "PB" );
		case 6:
			return( "EB" );
		case 7:
			return( "ZB" );
		default :
			break;
	}
	return( "?B" );
}

/* Determines the human readable size as a string
 * Returns a pointer to the new instance, NULL on error
 */
char *determine_human_readable_size_string( uint64_t size )
{
	char *size_string  = NULL;
	char *units_string = NULL;
	int8_t remainder   = -1;
	uint8_t factor     = 0;
	uint64_t new_size  = 0;

	while( size > 1024 )
	{
		factor++;

		new_size = size / 1024;

		if( new_size < 10 )
		{
			remainder = ( size % 1024 ) / 100;
		}
		size = new_size;
	}
	if( factor > 7 )
	{
		LIBEWF_WARNING_PRINT( "determine_human_readable_size_string: a size with a factor larger than 7 currently not supported.\n" );

		return( NULL );
	}
	/* The string has a maximum of 7 characters + end of string '\0'
	 */
	size_string  = (char *) libewf_string_alloc( 8 );
	units_string = determine_units_string( factor );

	if( remainder > 9 )
	{
		LIBEWF_WARNING_PRINT( "determine_human_readable_size_string: remainder is larger than 9.\n" );

		return( NULL );
	}
	else if( remainder >= 0 )
	{
		snprintf( size_string, 8, "%" PRIu64 ".%" PRIu8 " %s", size, remainder, units_string );
	}
	else
	{
		snprintf( size_string, 8, "%" PRIu64 " %s", size, units_string );
	}
	return( size_string );
}

/* Get variable input from the user
 * with a maximum of 1023 characters
 */
char *get_user_input_variable( FILE *stream, char *request_string )
{
	char user_input_buffer[ 1024 ];

	char *user_input_buffer_ptr = &user_input_buffer[ 0 ];
	char *user_input            = NULL;
	char *end_of_input          = NULL;
	void *data_set              = NULL;
	uint32_t input_length       = 0;
	uint32_t string_iterator    = 0;
	uint8_t string_valid        = 1;

	if( stream == NULL )
	{
		LIBEWF_WARNING_PRINT( "get_user_input_variable: Invalid output stream.\n" );

		return( NULL );
	}
	if( request_string == NULL )
	{
		LIBEWF_WARNING_PRINT( "get_user_input_variable: Invalid request string.\n" );

		return( NULL );
	}
	while( 1 )
	{
		fprintf( stream, "%s: ", request_string );

		user_input_buffer_ptr = fgets( user_input_buffer_ptr, 1023, stdin );

		if( user_input_buffer_ptr != NULL )
		{
			end_of_input = memchr( user_input_buffer_ptr, '\n', 1024 );

			if( end_of_input == NULL )
			{
				return( NULL );
			}
			input_length = end_of_input - user_input_buffer_ptr;

			if( input_length <= 0 )
			{
				return( NULL );
			}
			for( string_iterator = 0; string_iterator < input_length; string_iterator++ )
			{
				if( user_input_buffer[ string_iterator ] < 0x20 )
				{
					fprintf( stream, "Invalid character in input, please try again or terminate using Ctrl^C.\n" );

					string_valid = 0;
				}
				else if( user_input_buffer[ string_iterator ] >= 0x7f )
				{
					fprintf( stream, "Invalid character in input, please try again or terminate using Ctrl^C.\n" );

					string_valid = 0;
				}
			}
			if( string_valid == 1 )
			{
				user_input = (char *) libewf_string_alloc( ( input_length + 1 ) );

				if( user_input == NULL )
				{
					LIBEWF_WARNING_PRINT( "get_user_input_variable: Unable to allocate memory for string.\n" );

					return( NULL );
				}
				data_set = libewf_common_memcpy( (void *) user_input, (void *) user_input_buffer_ptr, input_length );

				if( data_set == NULL )
				{
					LIBEWF_WARNING_PRINT( "get_user_input_variable: Unable to set string.\n" );

					libewf_common_free( user_input );

					return( NULL );
				}
				user_input[ input_length ] = '\0';

				break;
			}
		}
		else
		{
			fprintf( stream, "Error reading input, please try again or terminate using Ctrl^C.\n" );
		}
	}
	return( user_input );
}

/* Get variable containing a size definnition input from the user
 * with a maximum of 1023 characters
 */
uint64_t get_user_input_size_variable( FILE *stream, char *request_string, uint64_t minimum, uint64_t maximum, uint64_t default_value )
{
	char user_input_buffer[ 1024 ];

	char *user_input_buffer_ptr = &user_input_buffer[ 0 ];
	char *last_character        = NULL;
	uint64_t input_length       = 0;
	uint64_t size_value         = 0;

	if( stream == NULL )
	{
		LIBEWF_WARNING_PRINT( "get_user_input_size_variable: Invalid output stream.\n" );

		return( 0 );
	}
	if( request_string == NULL )
	{
		LIBEWF_WARNING_PRINT( "get_user_input_size_variable: Invalid request string.\n" );

		return( 0 );
	}
	while( 1 )
	{
		fprintf( stream, "%s (%" PRIu64 " >= value >= %" PRIu64 ") [%" PRIu64 "]: ", request_string, minimum, maximum, default_value );

		user_input_buffer_ptr = fgets( user_input_buffer_ptr, 1023, stdin );

		if( user_input_buffer_ptr != NULL )
		{
			/* Remove the trailing newline character
			 */
			input_length = libewf_common_strlen( user_input_buffer_ptr ) - 1;

			if( input_length <= 0 )
			{
				return( default_value );
			}
			last_character = &user_input_buffer_ptr[ input_length ];
			size_value     = strtoul( user_input_buffer_ptr, &last_character, 0 );

			if( ( size_value >= minimum ) && ( size_value <= maximum ) )
			{
				break;
			}
			else
			{
				fprintf( stream, "Value not within specified range, please try again or terminate using Ctrl^C.\n" );
			}
		}
		else
		{
			fprintf( stream, "Error reading input, please try again or terminate using Ctrl^C.\n" );
		}
	}
	return( size_value );
}

/* Get fixed value input from the user
 * The first value is considered the default value
 */
char *get_user_input_fixed_value( FILE *stream, char *request_string, char **values, uint8_t amount, uint8_t default_value )
{
	char user_input_buffer[ 1024 ];

	char *user_input_buffer_ptr = &user_input_buffer[ 0 ];
	void *data_set              = NULL;
	uint32_t input_length       = 0;
	uint8_t iterator            = 0;
	uint8_t value_match         = 0;
	uint8_t value_size          = 0;
	char *user_input            = NULL;

	if( stream == NULL )
	{
		LIBEWF_WARNING_PRINT( "get_user_input_fixed_value: Invalid output stream.\n" );

		return( NULL );
	}
	if( request_string == NULL )
	{
		LIBEWF_WARNING_PRINT( "get_user_input_fixed_value: Invalid request string.\n" );

		return( NULL );
	}
	if( default_value >= amount )
	{
		LIBEWF_WARNING_PRINT( "get_user_input_fixed_value: Default value cannot be larger than or equal as amount.\n" );

		return( NULL );
	}
	while( 1 )
	{
		fprintf( stream, "%s (", request_string );

		for( iterator = 0; iterator < amount; iterator++ )
		{
			if( iterator > 0 )
			{
				fprintf( stream, ", " );
			}
			fprintf( stream, "%s", values[ iterator ] );
		}
		fprintf( stream, ") [%s]: ", values[ default_value ] );

		user_input_buffer_ptr = fgets( user_input_buffer_ptr, 1023, stdin );

		if( user_input_buffer_ptr != NULL )
		{
			iterator = 0;

			/* Remove the trailing newline character
			 */
			input_length = libewf_common_strlen( user_input_buffer_ptr ) - 1;

			/* Check if the default value was selected
			 */
			if( input_length == 0 )
			{
				iterator     = default_value;
				input_length = libewf_common_strlen( values[ iterator ] );
				value_match  = 1;
			}
			else
			{
				while( iterator < amount )
				{
					value_size = libewf_common_strlen( values[ iterator ] );

					if( libewf_common_strncmp( user_input_buffer_ptr, values[ iterator ], value_size ) == 0 )
					{
						/* Make sure no trailing characters were given
						 */
						if( user_input_buffer_ptr[ value_size ] == '\n' )
						{
							value_match = 1;

							break;
						}
					}
					iterator++;
				}
			}
		}
		else
		{
			fprintf( stream, "Error reading input, please try again or terminate using Ctrl^C.\n" );
		}
		if( value_match == 1 )
		{
			value_size = libewf_common_strlen( values[ iterator ] );

			user_input = (char *) libewf_string_alloc( value_size + 1 );

			if( user_input == NULL )
			{
				LIBEWF_WARNING_PRINT( "get_user_input_fixed_value: Unable to allocate memory for string.\n" );

				return( NULL );
			}
			data_set = libewf_common_memcpy( (void *) user_input, (void *) values[ iterator ], input_length );

			if( data_set == NULL )
			{
				LIBEWF_WARNING_PRINT( "get_user_input_fixed_value: Unable to set string.\n" );

				libewf_common_free( user_input );

				return( NULL );
			}
			break;
		}
		else
		{
			fprintf( stream, "Selected option not supported, please try again or terminate using Ctrl^C.\n" );
		}
	}
	return( user_input );
}

