/* necessary for POSIX conform readdir_r API */
#if defined(__sun)
#define _POSIX_PTHREAD_SEMANTICS
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <unistd.h>
#include <errno.h>
#include <assert.h>

#include "directory.h"
#include "slatevm.h"

#define DIRS_MAXIMUM 256

static DIR *dirs[DIRS_MAXIMUM];

#define BYTEARRAY_LEN(x)	(PSObject_payloadSize((struct Object *) (x)))

void initDirectoryModule(void) {
  int i;
  for (i = 0; i < DIRS_MAXIMUM; i++) {
    dirs[i] = NULL;
  }
}

static int findDirectorySlot(void) {
  int i;
  for (i = 0; i < DIRS_MAXIMUM; i++) {
    if (dirs[i] == NULL)
      return i;
  }
  return -EMFILE;
}

int openDirectory(struct ByteArray *dirName) {
  int dirHandle = findDirectorySlot();
  if (dirHandle < 0) {
    return dirHandle;
  } else {
    size_t nameLen = BYTEARRAY_LEN(dirName);
    char *name = malloc(nameLen + 1);
    if (name == NULL) {
      return -errno;
    } else {
      memcpy(name, dirName->elements, nameLen);
      name[nameLen] = '\0';
      dirs[dirHandle] = opendir(name);
      if (dirs[dirHandle] == NULL) {
        int savedErrno = errno;
        free(name);
        return -savedErrno;
      } else {
        free(name);
        return dirHandle;
      }
    }
  }
}

int closeDirectory(int dirHandle) {
  if (dirs[dirHandle] != NULL) {
    closedir(dirs[dirHandle]);
    dirs[dirHandle] = NULL;
    return 0;
  } else {
    return -EINVAL;
  }
}

int readDirectory(int dirHandle, struct ByteArray *entNameBuffer) {
  struct dirent ent;
  struct dirent *result = &ent;
  int resultCode, entLen;

  if (dirHandle < 0)
    return -EINVAL;

#if defined(__CYGWIN__)
  result = readdir(dirs[dirHandle]);
  resultCode = errno;
#else
  resultCode = readdir_r(dirs[dirHandle], result, &result);
#endif
  if (resultCode != 0) {
    /* An error situation. */
    assert(resultCode >= 0);	/* TODO: read SUSv2. The Darwin manpage is unclear about this */
    return -resultCode;
  }

  if (result == NULL) {
    /* End of the directory. */
    return 0;
  }

  entLen = strlen(result->d_name);
  if (entLen == 0) {
    /* Bizarre! Make up a reasonable response. */
    return -ENOENT;
  }

  assert(BYTEARRAY_LEN(entNameBuffer) >= entLen);
  memcpy(entNameBuffer->elements, result->d_name, entLen);
  return entLen;
}

int getCurrentDirectory(struct ByteArray *wdBuffer) {
  if (getcwd(wdBuffer->elements, BYTEARRAY_LEN(wdBuffer)) == NULL)
    return -errno;

  return strlen(wdBuffer->elements);
}

int setCurrentDirectory(struct ByteArray *newWd) {
  size_t wdLen = BYTEARRAY_LEN(newWd);
  char *wd = malloc(wdLen + 1);
  if (wd == NULL)
    return -errno;

  memcpy(wd, newWd->elements, wdLen);
  wd[wdLen] = '\0';
  if (chdir(wd) == -1) {
    int savedErrno = errno;
    free(wd);
    return -savedErrno;
  } else {
    free(wd);
    return 0;
  }
}
