#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>

#include "file.h"
#include "directory.h"
#include "image.h"
#include "slate.h"

extern const unsigned long int CardSize;
extern unsigned long long int currentDispatchID;

void
error (const char * message, ...)
{
	va_list args;

	va_start (args, message);
	fprintf (stderr, "Slate: ");
    vfprintf (stderr, message, args);
	fputc ('\n', stderr);
	va_end (args);
    exit (EXIT_FAILURE);
}

#include "slatevm.h"

void
printSymbol (FILE * file, ObjectPointer symbol)
{
        fwrite (PSObject_arrayElements ((struct Object *) symbol),
                1,
                PSObject_payloadSize ((struct Object *) symbol),
                file);
}

void
unhandledSignal (ObjectPointer symbol, int n, ObjectPointer * args)
{
        fprintf (stderr, "Unhandled signal: ");
        printSymbol (stderr, symbol);
        fputc ('\n', stderr);
        if (symbol == PSObjectHeap_specialAt_ (CurrentMemory, NotFoundOnSymbol))
        {
           fprintf (stderr, "Selector: ");
           printSymbol (stderr, args [0]);
           fputc ('\n', stderr);
        }
        exit (EXIT_FAILURE);
}

void
growMemoryBy (struct ObjectHeap * heap, unsigned long int delta)
{
    if (heap -> memoryEnd + delta > heap -> memoryLimit)
      error ("Attempted to grow heap beyond limits");
    
    heap -> memoryEnd += delta;

    fprintf (stderr, "Slate: Growing heap to %d bytes.\n", (unsigned) heap -> memoryEnd - (unsigned) heap -> memory);
}

void
shrinkMemoryBy (struct ObjectHeap * heap, unsigned long int delta)
{
    if (heap -> memoryEnd - delta <= (unsigned long int) heap -> memory)
      error ("Attempted to shrink heap to zero size");

    heap -> memoryEnd -= delta;

    fprintf (stderr, "Slate: Shrinking heap to %d bytes.\n", (unsigned) heap -> memoryEnd - (unsigned) heap -> memory);
}


int
extractCString (struct ByteArray * array, char * buffer, size_t bufferSize)
{
	int arrayLength = PSByteArray_extractInto_sized_ (array, buffer, bufferSize - 1);

	if (arrayLength < 0)
		return -1;

	buffer [arrayLength] = '\0';
		
	return arrayLength;
}


void
initModules ()
{
	initFileModule ();
	initDirectoryModule ();
}

static int
strtobkm(const char *str)
// For parsing mega/kilo on the command-line heap-size-limit argument.
{
  char *suffix;
  int result= strtol(str, &suffix, 10);
  switch (*suffix)
    {
    case 'k': case 'K':
      result*= 1024;
      break;
    case 'm': case 'M':
      result*= 1024*1024;
      break;
    }
  return result;
}

void
usage (char *progName)
{
  // this will grow.
  fprintf(stderr, "usage: %s [option] [<image>]\n", progName);
  //fprintf(stderr, "usage: %s [option] [<image> [<argument>...]]\n", progName);
  fprintf(stderr, "-h, -help		print this help message, then exit\n");
  fprintf(stderr, "-v, -version		print the VM version\n");
  fprintf(stderr, "-memory <size>[mk]	set the maximum heap size (beyond image size)\n");
  fprintf(stderr, "\nNotes:\n");
  fprintf(stderr, "<image> defaults to `slate.image' in the current directory\n");
}

int
slateMain (const int argc, char ** argv)
{
  FILE * imageFile;
  ImageHeader header;
  struct ObjectHeap heap;
  unsigned long int stackBottom;
  unsigned long int heapLimit = DEFAULT_HEAP_LIMIT;
  int i;
  char *imageName = "slate.image";

  // Parse the command-line arguments
  for (i=1 ; i<argc ; i++)
    {
      if (argv[i][0] != '-')
	imageName = argv[i];
      else if (!strcmp(argv[i], "-h") || !strcmp(argv[i], "-help")) {
	usage (argv[0]);
	exit (EXIT_SUCCESS);
      } else if (!strcmp(argv[i], "-v") || !strcmp(argv[i], "-version")) {
	error ("VM versions are not internally tracked yet.");
      } else if (!strcmp(argv[i], "-memory"))
	heapLimit = strtobkm (argv[++i]);
    }

  initModules ();

  imageFile = fopen (imageName, "rb");
  if (imageFile == NULL)
    error ("%s: Could not open image file", imageName);

  if (fread (& header, sizeof (ImageHeader), 1, imageFile) != 1 ||
      header.magicNumber != MAGIC_NUMBER)
    {
      fclose (imageFile);
      error ("%s: Invalid image header", imageName);
    }

  heapLimit += header.imageSize;
  heap.memory = (unsigned long int *) calloc (heapLimit, 1);
  heap.pinnedCards = (unsigned long int *) calloc ((heapLimit + (CardSize * 8) - 1) / (CardSize * 8), 1);
  if (heap.memory == NULL ||
      heap.pinnedCards == NULL)
    {
      fclose (imageFile);
      error ("%s: Failed to allocate memory for image heap", imageName);
    }

  if (fread (heap.memory, 1, header.imageSize, imageFile) != header.imageSize)
    {
      fclose (imageFile);
      error ("%s: Could not read image heap", imageName);
    }

  fclose (imageFile);

  heap.memoryEnd = (unsigned long int) heap.memory + header.imageSize;
  heap.memoryLimit = (unsigned long int) heap.memory + heapLimit;
  heap.lastHash = header.nextHash;
  heap.specialObjectsOop = header.specialOops;
  heap.stackBottom = & stackBottom;

  CurrentMemory = & heap;

  currentDispatchID = header.currentDispatchID;

  PSObjectHeap_initializeWithShift_ (CurrentMemory, (unsigned long int) heap.memory);

  PSInterpreter_interpret ((struct Interpreter *) PSObjectHeap_specialAt_ (CurrentMemory, InterpreterObject));

  return EXIT_SUCCESS;
}

int
saveImageNamed (struct ByteArray * name)
{
        char nameString [SLATE_FILE_NAME_LENGTH];
        size_t nameLength = PSObject_payloadSize ((struct Object *) name);
        FILE * imageFile;
        ImageHeader header;

        if (nameLength >= sizeof (nameString))
          return -1;

        memcpy (nameString, name -> elements, nameLength);
        nameString [nameLength] = '\0';

        imageFile = fopen (nameString, "wb");
        if (imageFile == NULL)
          return -1;

       PSObjectHeap_compact_ (CurrentMemory, 0);
       
       if (CurrentMemory -> markColor == 0)
         PSObjectHeap_garbageCollect (CurrentMemory);

       PSObjectHeap_sweep (CurrentMemory);
       
       PSObjectHeap_adjustAllOopsBy_ (CurrentMemory, - (unsigned) CurrentMemory -> memory);

       header.magicNumber = MAGIC_NUMBER;
       header.imageSize = CurrentMemory -> lastAllocated - (unsigned) CurrentMemory -> memory;
       header.nextHash = CurrentMemory -> lastHash;
       header.specialOops = CurrentMemory -> specialObjectsOop - (unsigned) CurrentMemory -> memory; 
       header.currentDispatchID = currentDispatchID;

       fwrite (& header, sizeof (header), 1, imageFile);
       fwrite (CurrentMemory -> memory, 1, header.imageSize, imageFile);

       fclose (imageFile);

       PSObjectHeap_adjustAllOopsBy_ (CurrentMemory, (unsigned) CurrentMemory -> memory);

       return 0;
}

