/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A skrooge plugin for monthlu report.
 *
 * @author Stephane MANKOWSKI
 */
#include "skgmonthlypluginwidget.h"
#include "skgmainpanel.h"
#include "skgtraces.h"
#include "skgdocumentbank.h"
#include "skgaccountobject.h"
#include "skgunitobject.h"
#include "skgtransactionmng.h"

#include <kstandarddirs.h>

#include <QDomDocument>
#include <QPrinter>
#include <QPrintDialog>
#include <QTextStream>
#include <QFile>
#include <QDir>

#include <math.h>

SKGMonthlyPluginWidget::SKGMonthlyPluginWidget(SKGDocument* iDocument)
    : SKGTabPage(iDocument)
{
    SKGTRACEIN(1, "SKGMonthlyPluginWidget::SKGMonthlyPluginWidget");
    if(!iDocument) return;

    ui.setupUi(this);

    ui.kRefresh->setIcon(KIcon("view-refresh"));

    connect(ui.kMonth, SIGNAL(currentIndexChanged(int)), this, SLOT(onMonthChanged()));

    //Refresh
    connect((const QObject*) getDocument(), SIGNAL(tableModified(QString, int)), this, SLOT(dataModified(QString, int)), Qt::QueuedConnection);
    dataModified("", 0);
}

SKGMonthlyPluginWidget::~SKGMonthlyPluginWidget()
{
    SKGTRACEIN(1, "SKGMonthlyPluginWidget::~SKGMonthlyPluginWidget");
}

QString SKGMonthlyPluginWidget::getState()
{
    SKGTRACEIN(10, "SKGMonthlyPluginWidget::getState");
    QDomDocument doc("SKGML");
    QDomElement root = doc.createElement("parameters");
    doc.appendChild(root);

    //Get state
    root.setAttribute("month", ui.kMonth->text());
    root.setAttribute("web", ui.kWebView->getState());

    return doc.toString();
}

void SKGMonthlyPluginWidget::setState(const QString& iState)
{
    SKGTRACEIN(10, "SKGMonthlyPluginWidget::setState");
    QDomDocument doc("SKGML");
    doc.setContent(iState);
    QDomElement root = doc.documentElement();

    //Set state
    QString month = root.attribute("month");
    if(!month.isEmpty()) ui.kMonth->setText(month);

    QString webS = root.attribute("web");
    if(!webS.isEmpty()) ui.kWebView->setState(webS);
}

QString SKGMonthlyPluginWidget::getDefaultStateAttribute()
{
    return "SKGMONTHLY_DEFAULT_PARAMETERS";
}

QWidget* SKGMonthlyPluginWidget::mainWidget() const
{
    return ui.kWebView;
}

void SKGMonthlyPluginWidget::dataModified(const QString& iTableName, int iIdTransaction)
{
    SKGTRACEIN(1, "SKGMonthlyPluginWidget::dataModified");
    Q_UNUSED(iIdTransaction);

    QSqlDatabase* db = getDocument()->getDatabase();
    setEnabled(db != NULL);
    if(db != NULL && (iTableName == "v_operation_display" || iTableName.isEmpty())) {
        //Fill combo
        disconnect(ui.kMonth, SIGNAL(currentIndexChanged(int)), this, SLOT(onMonthChanged()));

        QStringList list;
        getDocument()->getDistinctValues("v_operation_display", "d_DATEMONTH", "d_date<=CURRENT_DATE", list);
        list.removeOne(QDate::currentDate().toString("yyyy-MM"));
        qSort(list.begin(), list.end(), qGreater<QString>());

        QString month = ui.kMonth->text();
        ui.kMonth->clear();
        ui.kMonth->addItem(i18nc("The month before the current month", "Last month"));
        ui.kMonth->addItems(list);
        if(!month.isEmpty()) ui.kMonth->setText(month);
        ui.kRefresh->setEnabled(list.count() > 0);
        connect(ui.kMonth, SIGNAL(currentIndexChanged(int)), this, SLOT(onMonthChanged()));

        onMonthChanged();
    }
}

QString SKGMonthlyPluginWidget::getMonth()
{
    QString month;
    if(ui.kMonth->currentIndex() == 0) {
        //Last month
        if(ui.kMonth->count() > 1) month = ui.kMonth->itemText(1);
    } else {
        //Specific month
        month = ui.kMonth->text();
    }
    return month;
}

void SKGMonthlyPluginWidget::onMonthChanged()
{
    SKGTRACEIN(1, "SKGMonthlyPluginWidget::onMonthChanged");
    QString month = getMonth();
    if(!month.isEmpty()) {
        //Display report
        QString htmlReport = getDocument()->getParameter("SKG_MONTHLY_REPORT_" % month);
        if(htmlReport.isEmpty() || sender() == ui.kRefresh) {
            SKGError err;
            SKGBEGINTRANSACTION(*getDocument(), i18nc("Noun, name of the user action", "Compute monthly report for '%1'", month), err);
            htmlReport = getReport();
            err = getDocument()->setParameter("SKG_MONTHLY_REPORT_" % month, htmlReport);
        }

        //Report generated
        //Set style sheet
        QDir dirCss(KStandardDirs::locate("data", QString::fromLatin1("skrooge/css/")));
        QFile file(dirCss.absoluteFilePath("skrooge_monthly-default.css"));
        if(!file.open(QIODevice::ReadOnly | QIODevice::Text))
            return;

        QString style;
        QTextStream in(&file);
        while(!in.atEnd()) {
            style += in.readLine();
        }

        htmlReport.replace("##STYLE##", style);

        //Display html report
        ui.kWebView->setHtml(htmlReport);
    }
}

QString SKGMonthlyPluginWidget::getReport()
{
    QString html;
    SKGError err;
    SKGTRACEINRC(10, "SKGMonthlyPluginWidget::getReport", err);
    QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

    //Get month
    QString month = getMonth();
    if(!month.isEmpty()) {
        QDate date = QDate::fromString(month, "yyyy-MM");
        QDate date1 = date.addDays(-1);
        QDate date2 = date.addMonths(1).addDays(-1);
        QDate date3 = date2.addYears(-1);
        QString previousmonth = date1.toString("yyyy-MM");
        SKGDocumentBank* doc = static_cast<SKGDocumentBank*>(getDocument());
        KLocale* locale = KGlobal::locale();
        if(doc && locale) {
            SKGServices::SKGUnitInfo primary = doc->getPrimaryUnit();

            //Generate HTML report
            html = "<?xml version=\"1.0\" encoding=\"utf-8\"?>"
                   "<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Strict//EN\" \"http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd\">"
                   "<html xmlns=\"http://www.w3.org/1999/xhtml\">"
                   "<head>"
                   "<meta http-equiv=\"content-type\" content=\"text/html;charset=utf-8\" />"
                   "<meta http-equiv=\"Content-Style-Type\" content=\"text/css\" />"
                   "<style type=\"text/css\">##STYLE##</style>"
                   "</head>"
                   "<body>\n";

            //Main title
            html += "<h1>" % i18nc("Report header",  "Report for %1", month) % "</h1>";

            //Income & Expenditure
            if(!err) {
                SKGStringListList listTmp;
                err = doc->executeSelectSqliteOrder(
                          "SELECT TOTAL(f_CURRENTAMOUNT), d_DATEMONTH  from v_operation_display WHERE d_DATEMONTH IN ('" % month % "', '" % previousmonth % "') AND i_group_id=0 group by d_DATEMONTH, t_TYPEEXPENSE",
                          listTmp);
                if(!err) {
                    html += "<h2>" % i18nc("Report header",  "Income & Expenditure") % "</h2><table class=\"table\">";
                    html += "<tr class=\"tabletitle\"><td align=\"center\"></td>"
                            "<td align=\"center\">" % previousmonth % "</td>"
                            "<td align=\"center\">" % month % "</td>"
                            "</tr>";
                    double income_previous_month = 0;
                    double expense_previous_month = 0;
                    double income_month = 0;
                    double expense_month = 0;

                    int nbval = listTmp.count();
                    for(int i = 1; i < nbval; ++i) {  //Ignore header
                        QString m = listTmp.at(i).at(1);
                        double v = SKGServices::stringToDouble(listTmp.at(i).at(0));
                        if(v > 0 && m == month) income_month = v;
                        else if(v < 0 && m == month) expense_month = v;
                        else if(v > 0 && m == previousmonth) income_previous_month = v;
                        else if(v < 0 && m == previousmonth) expense_previous_month = v;
                    }
                    double saving_previous_month = income_previous_month + expense_previous_month;
                    double saving_month = income_month + expense_month;

                    html += QString("<tr><td>") % SKGServices::stringToHtml(doc->getDisplay("f_CURRENTAMOUNT_INCOME")) % "</td>"
                            "<td align=\"right\">" % doc->formatMoney(income_previous_month, primary) % "</td>"
                            "<td align=\"right\">" % doc->formatMoney(income_month, primary) % "</td>"
                            "</tr>";
                    html += QString("<tr><td>") % SKGServices::stringToHtml(doc->getDisplay("f_CURRENTAMOUNT_EXPENSE")) % "</td>"
                            "<td align=\"right\">" % doc->formatMoney(abs(expense_previous_month), primary) % "</td>"
                            "<td align=\"right\">" % doc->formatMoney(abs(expense_month), primary) % "</td>"
                            "</tr>";
                    html += "<tr class=\"tabletotal\"><td>" % i18nc("Noun",  "Savings possible") % "</td>"
                            "<td align=\"right\">" % doc->formatMoney(saving_previous_month, primary) % "</td>"
                            "<td align=\"right\">" % doc->formatMoney(saving_month, primary) % "</td>"
                            "</tr>";
                    html += "</table>";
                    QString range = "0," % SKGServices::doubleToString(qMax(abs(income_month), abs(expense_month)));
                    QString img = "http://chart.apis.google.com/chart?cht=bvs&chbh=100&chxt=x,y&chxr=1," % range % "&chco=FF0000|00FF00&chd=t:" %
                                  SKGServices::doubleToString(abs(expense_month)) % ',' %
                                  SKGServices::doubleToString(abs(income_month)) %
                                  "&chds=" % range %
                                  "&chs=300x200&chl=" % SKGServices::stringToHtml(doc->getDisplay("f_CURRENTAMOUNT_EXPENSE")) % '|' % SKGServices::stringToHtml(doc->getDisplay("f_CURRENTAMOUNT_INCOME")) % "&chtt=" % i18nc("Noun",  "Expenditure vs Income") % '|' % month;
                    // SKGTRACE << img << endl;
                    html += "<img src=\"" % img % "\" />";
                }
            }

            //Budget
            QString budget = doc->getBudget(month);
            if(!err && !budget.isEmpty()) {
                html += "<h2>" % i18nc("Report header",  "Budget") % "</h2>";
                html += budget;
            }

            //5 main categories of expenditure
            if(!err) {
                html += "<h2>" % i18nc("Report header",  "5 main categories of expenditure") % "</h2>";
                html += "<table>";
                html += "<tr><td align=\"center\">" % doc->get5MainCategories(previousmonth) % "</td>"
                        "<td align=\"center\">" % doc->get5MainCategories(month) % "</td>"
                        "</tr></table>";
            }

            //5 main categories variations
            if(!err) {
                html += "<h2>" % i18nc("Report header",  "5 main variations") % "</h2>";
                html += doc->get5MainCategoriesVariation(month, previousmonth);
            }

            //Account
            if(!err) {
                SKGObjectBase::SKGListSKGObjectBase accounts;
                SKGError err = doc->getObjects("v_account_display", "1=1 ORDER BY t_TYPENLS", accounts);
                if(!err) {
                    html += "<h2>" % i18nc("Report header",  "Amounts in accounts") % "</h2><table class=\"table\">";
                    html += "<tr class=\"tabletitle\"><td align=\"center\">" % SKGServices::stringToHtml(doc->getDisplay("t_ACCOUNT")) % "</td>"
                            "<td align=\"center\">" % SKGServices::stringToHtml(locale->formatDate(date1, KLocale::ShortDate)) % "</td>"
                            "<td align=\"center\">" % SKGServices::stringToHtml(locale->formatDate(date2, KLocale::ShortDate)) % "</td>"
                            "<td align=\"center\">%</td>"
                            "<td align=\"center\">" % SKGServices::stringToHtml(locale->formatDate(date3, KLocale::ShortDate)) % "</td>"
                            "<td align=\"center\">" % SKGServices::stringToHtml(locale->formatDate(date2, KLocale::ShortDate)) % "</td>"
                            "<td align=\"center\">%</td>"
                            "</tr>";
                    double sumTypeV1 = 0;
                    double sumTypeV2 = 0;
                    double sumTypeV3 = 0;
                    double sumV1 = 0;
                    double sumV2 = 0;
                    double sumV3 = 0;
                    QString currentType;
                    int nb = accounts.count();
                    for(int i = 0; !err && i < nb; ++i) {
                        SKGAccountObject account = accounts.at(i);
                        double v1 = account.getAmount(date1);
                        double v2 = account.getAmount(date2);
                        double v3 = account.getAmount(date3);
                        QString type = account.getAttribute("t_TYPENLS");
                        bool closed = account.isClosed();
                        if(type != currentType) {
                            if(!currentType.isEmpty()) {
                                html += "<tr class=\"tabletotal\"><td>" % SKGServices::stringToHtml(i18nc("Noun",  "Total of %1", currentType)) % "</td>"
                                        "<td align=\"right\">" % doc->formatMoney(sumTypeV1, primary) % "</td>"
                                        "<td align=\"right\">" % doc->formatMoney(sumTypeV2, primary) % "</td>"
                                        "<td align=\"right\">" % doc->formatPercentage(100.0 * (sumTypeV2 - sumTypeV1) / fabs(sumTypeV1)) % "</td>"
                                        "<td align=\"right\">" % doc->formatMoney(sumTypeV3, primary) % "</td>"
                                        "<td align=\"right\">" % doc->formatMoney(sumTypeV2, primary) % "</td>"
                                        "<td align=\"right\">" % doc->formatPercentage(100.0 * (sumTypeV2 - sumTypeV3) / fabs(sumTypeV3)) % "</td>"
                                        "</tr>";
                                sumTypeV1 = 0;
                                sumTypeV2 = 0;
                                sumTypeV3 = 0;
                            }
                            currentType = type;
                        }
                        if(!closed || abs(v1) > 0.01 || abs(v2) > 0.01 || abs(v3) > 0.01) {
                            html += QString("<tr><td>") % (closed ? "<i>" : "") % SKGServices::stringToHtml(account.getName()) % (closed ? "</i>" : "") % "</td>"
                                    "<td align=\"right\">" % doc->formatMoney(v1, primary) % "</td>"
                                    "<td align=\"right\">" % doc->formatMoney(v2, primary) % "</td>"
                                    "<td align=\"right\">" % doc->formatPercentage(100.0 * (v2 - v1) / fabs(v1)) % "</td>"
                                    "<td align=\"right\">" % doc->formatMoney(v3, primary) % "</td>"
                                    "<td align=\"right\">" % doc->formatMoney(v2, primary) % "</td>"
                                    "<td align=\"right\">" % doc->formatPercentage(100.0 * (v2 - v3) / fabs(v3)) % "</td>"
                                    ;
                        }
                        sumTypeV1 += v1;
                        sumTypeV2 += v2;
                        sumTypeV3 += v3;
                        sumV1 += v1;
                        sumV2 += v2;
                        sumV3 += v3;
                    }

                    html += "<tr class=\"tabletotal\"><td>" % SKGServices::stringToHtml(i18nc("Noun",  "Total of %1", currentType)) % "</td>"
                            "<td align=\"right\">" % doc->formatMoney(sumTypeV1, primary) % "</td>"
                            "<td align=\"right\">" % doc->formatMoney(sumTypeV2, primary) % "</td>"
                            "<td align=\"right\">" % doc->formatPercentage(100.0 * (sumTypeV2 - sumTypeV1) / fabs(sumTypeV1)) % "</td>"
                            "<td align=\"right\">" % doc->formatMoney(sumTypeV3, primary) % "</td>"
                            "<td align=\"right\">" % doc->formatMoney(sumTypeV2, primary) % "</td>"
                            "<td align=\"right\">" % doc->formatPercentage(100.0 * (sumTypeV2 - sumTypeV3) / fabs(sumTypeV3)) % "</td>"
                            "</tr>";
                    html += "<tr class=\"tabletotal\"><td>" % SKGServices::stringToHtml(i18nc("Noun, the numerical total of a sum of values", "Total")) % "</td>"
                            "<td align=\"right\">" % doc->formatMoney(sumV1, primary) % "</td>"
                            "<td align=\"right\">" % doc->formatMoney(sumV2, primary) % "</td>"
                            "<td align=\"right\">" % doc->formatPercentage(100.0 * (sumV2 - sumV1) / fabs(sumV1)) % "</td>"
                            "<td align=\"right\">" % doc->formatMoney(sumV3, primary) % "</td>"
                            "<td align=\"right\">" % doc->formatMoney(sumV2, primary) % "</td>"
                            "<td align=\"right\">" % doc->formatPercentage(100.0 * (sumV2 - sumV3) / fabs(sumV3)) % "</td>"
                            "</tr>";
                    html += "</table>";
                }
            }

            //Unit
            if(!err) {
                SKGObjectBase::SKGListSKGObjectBase units;
                SKGError err = doc->getObjects("v_unit_display", "1=1 ORDER BY t_TYPENLS", units);
                if(!err) {
                    html += "<h2>" % i18nc("Report header",  "Amounts of units") % "</h2><table class=\"table\">";
                    html += "<tr class=\"tabletitle\"><td align=\"center\">" % SKGServices::stringToHtml(doc->getDisplay("t_UNIT")) % "</td>"
                            "<td align=\"center\">" % SKGServices::stringToHtml(locale->formatDate(date1, KLocale::ShortDate)) % "</td>"
                            "<td align=\"center\">" % SKGServices::stringToHtml(locale->formatDate(date2, KLocale::ShortDate)) % "</td>"
                            "<td align=\"center\">%</td></tr>";
                    foreach(const SKGUnitObject & unit, units) {
                        double v1 = unit.getAmount(date1);
                        double v2 = unit.getAmount(date2);
                        html += "<tr><td>" % SKGServices::stringToHtml(unit.getName()) % "</td>"
                                "<td align=\"right\">" % doc->formatMoney(v1, primary) % "</td>"
                                "<td align=\"right\">" % doc->formatMoney(v2, primary) % "</td>"
                                "<td align=\"right\">" % doc->formatPercentage(100.0 * (v2 - v1) / fabs(v1)) % "</td>"
                                "</tr>";
                    }
                    html += "</table>";
                }
            }
            html += "</body></html>";
        }
    }
    QApplication::restoreOverrideCursor();


    //status bar
    if(!err)  err = SKGError(0, i18nc("Successful message after an user action", "Report computed."));
    else err.addError(ERR_FAIL, i18nc("Error message",  "Report computation failed."));

    //Display error
    SKGMainPanel::displayErrorMessage(err);
    return html;
}

#include "skgmonthlypluginwidget.moc"


