/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is Skrooge plugin for operation management.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgoperationplugin.h"
#include "skgoperationpluginwidget.h"
#include "skgoperationboardwidget.h"
#include "skgbookmarkboardwidget.h"
#include "skgtraces.h"
#include "skgoperationobject.h"
#include "skgaccountobject.h"
#include "skgsuboperationobject.h"
#include "skgtransactionmng.h"
#include "skgmainpanel.h"
#include "skgtableview.h"
#include "skgruleobject.h"
#include "skgoperation_settings.h"
#include "skgdocumentbank.h"

#include <kactioncollection.h>
#include <kstandardaction.h>

#include <QDomDocument>

/**
 * This plugin factory.
 */
K_PLUGIN_FACTORY ( SKGOperationPluginFactory, registerPlugin<SKGOperationPlugin>(); )
/**
 * This plugin export.
 */
K_EXPORT_PLUGIN ( SKGOperationPluginFactory ( "skrooge_operation", "skrooge_operation" ) )

SKGOperationPlugin::SKGOperationPlugin ( QObject* iParent, const QVariantList& /*iArg*/ ) : SKGInterfacePlugin ( iParent )
{
    SKGTRACEIN ( 10, "SKGOperationPlugin::SKGOperationPlugin" );
}

SKGOperationPlugin::~SKGOperationPlugin()
{
    SKGTRACEIN ( 10, "SKGOperationPlugin::~SKGOperationPlugin" );
    currentBankDocument=NULL;
    duplicateAction=NULL;
    switchToPointedAction=NULL;
    switchHighLight=NULL;
    openHighLights=NULL;
    openLastModified=NULL;
    groupOperation=NULL;
    ungroupOperation=NULL;
    createTemplateAction=NULL;
    openSuboperations=NULL;
}

bool SKGOperationPlugin::setupActions ( SKGDocument* iDocument, const QStringList& iArgument )
{
    SKGTRACEIN ( 10, "SKGOperationPlugin::setupActions" );
    Q_UNUSED ( iArgument );
    currentBankDocument=dynamic_cast<SKGDocumentBank*>(iDocument);
    if (currentBankDocument==NULL) return false;

    setComponentData ( SKGOperationPluginFactory::componentData() );
    setXMLFile ( "skrooge_operation.rc" );

    //Menu
    duplicateAction = new KAction ( KIcon ( "skrooge_duplicate" ), i18nc ("Verb, duplicate an object",  "Duplicate" ), this );
    connect ( duplicateAction, SIGNAL ( triggered ( bool ) ), this, SLOT ( actionDuplicate() ) );
    actionCollection()->addAction ( QLatin1String ( "edit_duplicate_operation" ), duplicateAction );
    duplicateAction->setShortcut ( Qt::CTRL+Qt::Key_D );

    if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction ( "edit_duplicate_operation", duplicateAction );

    createTemplateAction = new KAction ( KIcon ( "skrooge_template" ), i18nc ("Verb", "Create template" ), this );
    connect ( createTemplateAction, SIGNAL ( triggered ( bool ) ), this, SLOT ( actionCreateTemplate() ) );
    actionCollection()->addAction ( QLatin1String ( "edit_template_operation" ), createTemplateAction );
    createTemplateAction->setShortcut ( Qt::CTRL+Qt::SHIFT+Qt::Key_T );

    if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction ( "edit_template_operation", createTemplateAction );

    switchToPointedAction = new KAction ( KIcon ( "dialog-ok" ), i18nc ("Verb, mark an object", "Point" ), this );
    connect ( switchToPointedAction, SIGNAL ( triggered ( bool ) ), this, SLOT ( actionSwitchToPointed() ) );
    actionCollection()->addAction ( QLatin1String ( "edit_point_selected_operation" ), switchToPointedAction );
    switchToPointedAction->setShortcut ( Qt::CTRL+Qt::Key_R );

    if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction ( "edit_point_selected_operation", switchToPointedAction );

    KAction* fastEditionAction = new KAction ( KIcon ( "games-solve" ), i18nc ("Verb", "Fast edit" ), this );
    actionCollection()->addAction ( QLatin1String ( "fast_edition" ), fastEditionAction );
    fastEditionAction->setEnabled ( false );
    fastEditionAction->setShortcut ( Qt::Key_F10 );

    if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction ( "fast_edition", fastEditionAction );

    switchHighLight = new KAction ( KIcon ( "rating" ), i18nc ("Verb", "Switch highlight" ), this );
    connect ( switchHighLight, SIGNAL ( triggered ( bool ) ), this, SLOT ( actionSwitchHighLight() ) );
    actionCollection()->addAction ( QLatin1String ( "edit_switch_highlight" ), switchHighLight );
    switchHighLight->setShortcut ( Qt::CTRL+Qt::Key_H );

    if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction ( "edit_switch_highlight", switchHighLight );

    QStringList overlayopen;
    overlayopen.push_back("skg_open");
    openHighLights = new KAction(KIcon( "rating", NULL, overlayopen), i18nc ("Verb", "Open highlights..." ), this );
    connect ( openHighLights, SIGNAL ( triggered ( bool ) ), this, SLOT ( actionOpenHighLights() ) );
    actionCollection()->addAction ( QLatin1String ( "view_open_highlight" ), openHighLights );
    openHighLights->setShortcut ( Qt::META+Qt::Key_H );

    if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction ( "view_open_highlight", openHighLights );

    openLastModified = new KAction ( KIcon("view-refresh", NULL, overlayopen), i18nc ("Verb", "Open last modified..." ), this);
    connect ( openLastModified, SIGNAL ( triggered ( bool ) ), this, SLOT ( actionOpenLastModified() ) );
    actionCollection()->addAction ( QLatin1String ( "view_open_last_modified" ), openLastModified );
    openLastModified->setShortcut ( Qt::META+Qt::Key_L );

    if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction ( "view_open_last_modified", openLastModified );

    openSuboperations = new KAction ( KIcon("skrooge_split", NULL, overlayopen), i18nc ("Verb", "Open sub operations..." ), this);
    connect ( openSuboperations, SIGNAL ( triggered ( bool ) ), this, SLOT ( actionOpenSubOperations() ) );
    actionCollection()->addAction ( QLatin1String ( "view_open_suboperations" ), openSuboperations );
    openSuboperations->setShortcut ( Qt::META+Qt::Key_S );

    if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction ( "view_open_suboperations", openSuboperations );

    groupOperation = new KAction ( KIcon ( "skrooge_transfer" ), i18nc ("Verb", "Group operations" ), this );
    connect ( groupOperation, SIGNAL ( triggered ( bool ) ), this, SLOT ( actionGroupOperation() ) );
    actionCollection()->addAction ( QLatin1String ( "edit_group_operation" ), groupOperation );
    groupOperation->setShortcut ( Qt::CTRL+Qt::Key_G );

    if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction ( "edit_group_operation", groupOperation );

    QStringList overlay;
    overlay.push_back("edit-delete");
    ungroupOperation = new KAction ( KIcon ( "skrooge_transfer", NULL, overlay ), i18nc ("Verb", "Ungroup operations" ), this );
    connect ( ungroupOperation, SIGNAL ( triggered ( bool ) ), this, SLOT ( actionUngroupOperation() ) );
    actionCollection()->addAction ( QLatin1String ( "edit_ungroup_operation" ), ungroupOperation );
    ungroupOperation->setShortcut ( Qt::CTRL+Qt::SHIFT+Qt::Key_G );

    if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction ( "edit_ungroup_operation", ungroupOperation );
    return true;
}

void SKGOperationPlugin::refresh()
{
    SKGTRACEIN ( 10, "SKGOperationPlugin::refresh" );
    if ( currentBankDocument && SKGMainPanel::getMainPanel() ) {
        bool test=(currentBankDocument->getDatabase()!=NULL);
        if ( openHighLights ) openHighLights->setEnabled ( test );
        if ( openSuboperations ) openSuboperations->setEnabled ( test );
        if ( openLastModified ) openLastModified->setEnabled ( test && currentBankDocument->getNbTransaction()>0 );

        SKGObjectBase::SKGListSKGObjectBase selection=SKGMainPanel::getMainPanel()->getSelectedObjects();
        if ( selection.count() >0 ) {
            bool onOperation= ( selection.at ( 0 ).getRealTable() =="operation" &&  selection.at ( 0 ).getTable()!="v_operation_consolidated");
            bool onAccount= ( selection.at ( 0 ).getRealTable() =="account");
            bool onRule= ( selection.at ( 0 ).getRealTable() =="rule");
            if ( duplicateAction ) duplicateAction->setEnabled ( onOperation );
            if ( createTemplateAction ) createTemplateAction->setEnabled ( onOperation );
            if ( switchHighLight ) switchHighLight->setEnabled ( onOperation || onAccount || onRule);
            if ( switchToPointedAction ) switchToPointedAction->setEnabled ( onOperation );
            if ( groupOperation ) groupOperation->setEnabled ( onOperation && selection.count()>=2);
            if ( ungroupOperation ) ungroupOperation->setEnabled ( onOperation );
        } else {
            if ( duplicateAction ) duplicateAction->setEnabled ( false );
            if ( createTemplateAction ) createTemplateAction->setEnabled ( false );
            if ( switchHighLight ) switchHighLight->setEnabled ( false );
            if ( switchToPointedAction ) switchToPointedAction->setEnabled ( false );
            if ( groupOperation ) groupOperation->setEnabled ( false );
            if ( ungroupOperation ) ungroupOperation->setEnabled ( false );
        }
    }
}

void SKGOperationPlugin::close()
{
    SKGTRACEIN ( 10, "SKGOperationPlugin::close" );
}

int SKGOperationPlugin::getNbDashboardWidgets()
{
    return 2;
}

QString SKGOperationPlugin::getDashboardWidgetTitle(int iIndex)
{
    if (iIndex==0) return i18nc("Noun, the title of a section", "Income && Expenditure");
    else if (iIndex==1) return i18nc("Noun, the title of a section", "Highlighted operations");
    return "";
}

SKGWidget* SKGOperationPlugin::getDashboardWidget(int iIndex)
{
    if (iIndex==0) return new SKGOperationBoardWidget(currentBankDocument);
    else if (iIndex==1) return new SKGBookmarkBoardWidget(currentBankDocument);
    return NULL;
}

SKGTabPage* SKGOperationPlugin::getWidget()
{
    SKGTRACEIN ( 10, "SKGOperationPlugin::getWidget" );
    return new SKGOperationPluginWidget(currentBankDocument);
}

QWidget* SKGOperationPlugin::getPreferenceWidget()
{
    SKGTRACEIN ( 10, "SKGMonthlyPlugin::getPreferenceWidget" );
    QWidget* widget=new QWidget();
    ui.setupUi ( widget );
    return widget;
}

KConfigSkeleton* SKGOperationPlugin::getPreferenceSkeleton()
{
    return skgoperation_settings::self();
}

QString SKGOperationPlugin::title() const
{
    return i18nc("Noun", "Operations" );
}

QString SKGOperationPlugin::icon() const
{
    return "view-pim-tasks";
}

QString SKGOperationPlugin::toolTip () const
{
    return i18nc("Noun", "Operation management" );
}


int SKGOperationPlugin::getOrder() const
{
    return 15;
}

QStringList SKGOperationPlugin::tips() const
{
    QStringList output;
    output.push_back(i18nc("Description of a tips", "<p>... you can press <strong>+</strong>, <strong>-</strong>, <strong>CTRL +</strong> or <strong>CTRL -</strong> to quickly change dates.</p>" ) );
    output.push_back(i18nc("Description of a tips", "<p>... you can update many operations in one shot.</p>" ) );
    output.push_back(i18nc("Description of a tips", "<p>... you can double click on an operation to show or edit sub operations.</p>" ) );
    output.push_back(i18nc("Description of a tips", "<p>... you can duplicate an operation including complex operations (split, grouped, ...).</p>" ) );
    output.push_back(i18nc("Description of a tips", "<p>... you can create template of operations.</p>" ) );
    output.push_back(i18nc("Description of a tips", "<p>... you can group and ungroup operations.</p>" ) );
    return output;
}

bool SKGOperationPlugin::isInContext() const
{
    return true;
}

void SKGOperationPlugin::actionGroupOperation()
{
    SKGError err;
    SKGTRACEINRC ( 10, "SKGOperationPlugin::actionGroupOperation",err );
    //Get Selection
    if ( SKGMainPanel::getMainPanel() && currentBankDocument ) {
        SKGObjectBase::SKGListSKGObjectBase selection=SKGMainPanel::getMainPanel()->getSelectedObjects();
        int nb=selection.count();
        if (nb>=2) {
            SKGBEGINPROGRESSTRANSACTION(*currentBankDocument, i18nc("Noun, name of the user action","Group operations" ), err, nb );
            QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
            SKGOperationObject main=selection.at(0);
            if ( err.isSucceeded() ) err=currentBankDocument->stepForward(1);
            for ( int i=1; err.isSucceeded() && i<nb; ++i ) {
                SKGOperationObject operationObj=selection.at(i);
                if ( err.isSucceeded() ) err=operationObj.setGroupOperation(main);
                if ( err.isSucceeded() ) err=operationObj.save();

                if ( err.isSucceeded() ) err=currentBankDocument->stepForward ( i+1 );
            }
            QApplication::restoreOverrideCursor();
        }

        //status bar
        if ( err.isSucceeded() ) err=SKGError(0, i18nc("Successful message after an user action", "Operations grouped." ) );
        else err.addError ( ERR_FAIL, i18nc("Error message",  "Group creation failed" ) );

        //Display error
        SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
    }
}

void SKGOperationPlugin::actionUngroupOperation()
{
    SKGError err;
    SKGTRACEINRC ( 10, "SKGOperationPlugin::actionUngroupOperation",err );
    //Get Selection
    if ( SKGMainPanel::getMainPanel() && currentBankDocument ) {
        SKGObjectBase::SKGListSKGObjectBase selection=SKGMainPanel::getMainPanel()->getSelectedObjects();
        int nb=selection.count();
        {
            SKGBEGINPROGRESSTRANSACTION(*currentBankDocument, i18nc("Noun, name of the user action","Ungroup operation" ), err, nb );

            QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
            for ( int i=0; err.isSucceeded() && i<nb; ++i ) {
                SKGOperationObject operationObj=selection.at ( i );
                if ( err.isSucceeded() ) err=operationObj.setGroupOperation(operationObj);
                if ( err.isSucceeded() ) err=operationObj.save();

                if ( err.isSucceeded() ) err=currentBankDocument->stepForward ( i+1 );
            }
            QApplication::restoreOverrideCursor();
        }

        //status bar
        if ( err.isSucceeded() ) err=SKGError(0, i18nc("Successful message after an user action", "Operation ungrouped." ) );
        else err.addError ( ERR_FAIL, i18nc("Error message",  "Group deletion failed" ) );

        //Display error
        SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
    }
}

void SKGOperationPlugin::actionSwitchToPointed()
{
    SKGError err;
    SKGTRACEINRC ( 10, "SKGOperationPlugin::actionSwitchToPointed",err );
    //Get Selection
    if ( SKGMainPanel::getMainPanel() && currentBankDocument ) {
        SKGObjectBase::SKGListSKGObjectBase selection=SKGMainPanel::getMainPanel()->getSelectedObjects();
        int nb=selection.count();
        {
            SKGBEGINPROGRESSTRANSACTION(*currentBankDocument, i18nc("Noun, name of the user action","Switch to pointed" ), err, nb );

            QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
            for ( int i=0; err.isSucceeded() && i<nb; ++i ) {
                SKGOperationObject operationObj=selection.at ( i );
                if ( err.isSucceeded() ) err=operationObj.setStatus ( SKGOperationObject::POINTED );
                if ( err.isSucceeded() ) err=operationObj.save();

                if ( err.isSucceeded() ) err=currentBankDocument->stepForward ( i+1 );
            }
            QApplication::restoreOverrideCursor();
        }

        //status bar
        if ( err.isSucceeded() ) err=SKGError(0, i18nc("Successful message after an user action", "Operation pointed." ) );
        else err.addError ( ERR_FAIL, i18nc("Error message",  "Switch failed" ) );

        //Display error
        SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
    }
}

void SKGOperationPlugin::actionSwitchHighLight()
{
    SKGError err;
    SKGTRACEINRC ( 10, "SKGOperationPlugin::actionSwitchHighLight",err );
    //Get Selection
    if ( SKGMainPanel::getMainPanel() && currentBankDocument ) {
        SKGObjectBase::SKGListSKGObjectBase selection=SKGMainPanel::getMainPanel()->getSelectedObjects();
        int nb=selection.count();
        {
            SKGBEGINPROGRESSTRANSACTION(*currentBankDocument, i18nc("Noun, name of the user action","Highlight" ), err, nb );

            QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
            for ( int i=0; err.isSucceeded() && i<nb; ++i ) {
                if ( selection.at ( i ).getRealTable() =="account")
                {
                    SKGAccountObject obj=selection.at ( i );
                    if ( err.isSucceeded() ) err=obj.bookmark ( !obj.isBookmarked() );
                    if ( err.isSucceeded() ) err=obj.save();
                }
                else if ( selection.at ( i ).getRealTable() =="operation")
                {
                    SKGOperationObject obj=selection.at ( i );
                    if ( err.isSucceeded() ) err=obj.bookmark ( !obj.isBookmarked() );
                    if ( err.isSucceeded() ) err=obj.save();
                }
                else if ( selection.at ( i ).getRealTable() =="rule")
                {
                    SKGRuleObject obj=selection.at ( i );
                    if ( err.isSucceeded() ) err=obj.bookmark ( !obj.isBookmarked() );
                    if ( err.isSucceeded() ) err=obj.save();
                }
                if ( err.isSucceeded() ) err=currentBankDocument->stepForward ( i+1 );
            }
            QApplication::restoreOverrideCursor();
        }

        //status bar
        if ( err.isSucceeded() )  err=SKGError(0, i18nc("Successful message after an user action", "Highlighted." ) );
        else err.addError ( ERR_FAIL, i18nc("Error message",  "Highlight failed" ) );

        //Display error
        SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
    }
}

void SKGOperationPlugin::actionDuplicate()
{
    SKGError err;
    SKGTRACEINRC ( 10, "SKGOperationPlugin::actionDuplicate",err );
    //Get Selection
    if ( SKGMainPanel::getMainPanel() && currentBankDocument ) {
        QStringList listUUID;
        SKGObjectBase::SKGListSKGObjectBase selection=SKGMainPanel::getMainPanel()->getSelectedObjects();
        int nb=selection.count();
        {
            SKGBEGINPROGRESSTRANSACTION(*currentBankDocument, i18nc("Noun, name of the user action","Duplicate operation" ), err, nb );

            QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
            for ( int i=0; err.isSucceeded() && i<nb; ++i ) {
                SKGOperationObject operationObj=selection.at ( i );
                SKGOperationObject dup;
                if ( err.isSucceeded() ) err=operationObj.duplicate ( dup );
                if ( err.isSucceeded() ) err=currentBankDocument->stepForward ( i+1 );

                listUUID.push_back ( dup.getUniqueID() );
            }
            QApplication::restoreOverrideCursor();
        }

        //status bar
        if ( err.isSucceeded() ) {
            err=SKGError(0, i18nc("Successful message after an user action", "Operation duplicated." ) );
            SKGOperationPluginWidget* w=dynamic_cast<SKGOperationPluginWidget*>(SKGMainPanel::getMainPanel()->currentPage());
            if (w) w->getTableView()->selectObjects(listUUID, true);
        } else err.addError ( ERR_FAIL, i18nc("Error message",  "Duplicate operation failed" ) );

        //Display error
        SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
    }
}

void SKGOperationPlugin::actionCreateTemplate()
{
    SKGError err;
    SKGTRACEINRC ( 10, "SKGOperationPlugin::actionCreateTemplate",err );
    //Get Selection
    if ( SKGMainPanel::getMainPanel() && currentBankDocument ) {
        QStringList listUUID;
        SKGObjectBase::SKGListSKGObjectBase selection=SKGMainPanel::getMainPanel()->getSelectedObjects();
        int nb=selection.count();
        {
            SKGBEGINPROGRESSTRANSACTION(*currentBankDocument, i18nc("Noun, name of the user action","Create template" ), err, nb );

            QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
            for ( int i=0; err.isSucceeded() && i<nb; ++i ) {
                SKGOperationObject operationObj=selection.at ( i );
                SKGOperationObject dup;
                if ( err.isSucceeded() ) err=operationObj.duplicate ( dup , QDate::currentDate(), true );
                if ( err.isSucceeded() ) err=currentBankDocument->stepForward ( i+1 );

                listUUID.push_back ( dup.getUniqueID() );
            }
            QApplication::restoreOverrideCursor();
        }

        //status bar
        if ( err.isSucceeded() ) {
            err=SKGError(0, i18nc("Successful message after an user action", "Template created." ) );
            SKGOperationPluginWidget* w=dynamic_cast<SKGOperationPluginWidget*>(SKGMainPanel::getMainPanel()->currentPage());
            if (w) {
                w->setTemplateMode(true);
                w->getTableView()->selectObjects(listUUID, true);
            }
        } else err.addError ( ERR_FAIL, i18nc("Error message",  "Creation template failed" ) );

        //Display error
        SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
    }
}

void SKGOperationPlugin::actionOpenHighLights()
{
    SKGTRACEIN ( 10, "SKGOperationPlugin::actionOpenHighLights");
    QString wc="t_bookmarked='Y'";
    QString title=i18nc("Noun, a list of items", "Highlighted operations" );

    //Call operation plugin
    QDomDocument doc("SKGML");
    if (currentBankDocument) doc.setContent ( currentBankDocument->getParameter("SKGOPERATION_DEFAULT_PARAMETERS") );
    QDomElement root = doc.documentElement();
    if (root.isNull()) {
        root=doc.createElement("parameters");
        doc.appendChild(root);
    }

    root.setAttribute ( "account", i18nc("For all accounts", "All") );
    root.setAttribute ( "operationTable", "v_operation_display" );
    root.setAttribute ( "operationWhereClause", wc );
    root.setAttribute ( "title", title );
    root.setAttribute ( "title_icon", "rating" );

    if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->openPage ( SKGMainPanel::getMainPanel()->getPluginByName ( "Skrooge operation plugin" ), -1, doc.toString() );
}

void SKGOperationPlugin::actionOpenLastModified()
{
    SKGTRACEIN ( 10, "SKGOperationPlugin::actionOpenLastModified");
    QString wc="id in (SELECT i_object_id FROM doctransactionitem di, doctransaction dt WHERE dt.t_mode='U' AND dt.id=di.rd_doctransaction_id AND di.t_object_table='operation'AND NOT EXISTS(select 1 from doctransaction B where B.i_parent=dt.id))";
    QString title=i18nc("Noun, a list of items", "Operations modified or created during last action" );

    //Call operation plugin
    QDomDocument doc("SKGML");
    if (currentBankDocument) doc.setContent ( currentBankDocument->getParameter("SKGOPERATION_DEFAULT_PARAMETERS") );
    QDomElement root = doc.documentElement();
    if (root.isNull()) {
        root=doc.createElement("parameters");
        doc.appendChild(root);
    }

    root.setAttribute ( "account", i18nc("For all accounts", "All") );
    root.setAttribute ( "operationTable", "v_operation_display" );
    root.setAttribute ( "operationWhereClause", wc );
    root.setAttribute ( "title", title );
    root.setAttribute ( "title_icon", "rating" );

    if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->openPage ( SKGMainPanel::getMainPanel()->getPluginByName ( "Skrooge operation plugin" ), -1, doc.toString() );
}

void SKGOperationPlugin::actionOpenSubOperations()
{
    SKGTRACEIN ( 10, "SKGOperationPlugin::actionOpenLastModified");
    //Call operation plugin
    QDomDocument doc("SKGML");
    if (currentBankDocument) doc.setContent ( currentBankDocument->getParameter("SKGOPERATION_CONSOLIDATED_DEFAULT_PARAMETERS") );
    QDomElement root = doc.documentElement();
    if (root.isNull()) {
        root=doc.createElement("parameters");
        doc.appendChild(root);
    }

    root.setAttribute ( "account", i18nc("For all accounts", "All") );
    root.setAttribute ( "operationTable", "v_operation_consolidated" );
    root.setAttribute ( "operationWhereClause", "" );
    root.setAttribute ( "title", i18nc("Noun, a list of items", "Sub operations") );
    root.setAttribute ( "title_icon", icon() );

    if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->openPage ( SKGMainPanel::getMainPanel()->getPluginByName ( "Skrooge operation plugin" ), -1, doc.toString() );
}

QList< SKGInterfacePlugin::SKGAdvice > SKGOperationPlugin::advices() const
{
    QList< SKGInterfacePlugin::SKGAdvice > output;

    //Search duplicate number on operation
    SKGStringListList result;
    SKGServices::executeSelectSqliteOrder(currentBankDocument, "SELECT count(1), t_ACCOUNT, i_number FROM v_operation_display WHERE i_number!=0 GROUP BY t_ACCOUNT, i_number HAVING count(1)>1 ORDER BY count(1) DESC", result);
    int nb=result.count();
    for (int i=1; i<nb; ++i) //Ignore header
    {
        //Get parameters
        QStringList line=result.at(i);
        QString account=line.at(1);
        QString number=line.at(2);

        SKGInterfacePlugin::SKGAdvice ad;
        ad.uuid="skgoperationplugin_duplicate_"+number+';'+account;
        ad.priority=7;
        ad.shortMessage=i18nc("Advice on making the best (short)", "Duplicate number %1 in account '%2'", number, account);
        ad.longMessage=i18nc("Advice on making the best (long)", "Your account '%1' contains more than one operation with number %2.The operation number should be unique (check number, transaction reference...)", account, number);
        ad.autoCorrection.push_back(i18nc("Advice on making the best (action)", "Edit operations with duplicate number"));
        output.push_back(ad);
    }

    //Check operations not reconciliated
    SKGServices::executeSelectSqliteOrder(currentBankDocument, "SELECT count(1), t_ACCOUNT FROM v_operation_display WHERE t_status='N' GROUP BY t_ACCOUNT HAVING count(1)>100 ORDER BY count(1) DESC", result);
    nb=result.count();
    for (int i=1; i<nb; ++i) //Ignore header
    {
        //Get parameters
        QStringList line=result.at(i);
        QString account=line.at(1);

        SKGInterfacePlugin::SKGAdvice ad;
        ad.uuid="skgoperationplugin_notreconciliated_"+account;
        ad.priority=9;
        ad.shortMessage=i18nc("Advice on making the best (short)", "Many operations of '%1' not reconciliated", account);
        ad.longMessage=i18nc("Advice on making the best (long)", "Don't forget to reconciliate your accounts. By doing so, you acknowledge that your bank has indeed processed these operations on your account. This is how you enforce compliance with your bank's statements. See online help for more details");
        ad.autoCorrection.push_back(i18nc("Advice on making the best (action)", "Open account '%1' for reconciliation", account));
        output.push_back(ad);
    }

    //Check operations without category
    SKGObjectBase::getNbObjects(currentBankDocument, "v_operation_display","EXISTS (SELECT 1 FROM suboperation WHERE rd_operation_id=v_operation_display.id AND r_category_id=0)", nb);
    if (nb)
    {
        SKGInterfacePlugin::SKGAdvice ad;
        ad.uuid="skgoperationplugin_nocategory";
        ad.priority=5;
        ad.shortMessage=i18nc("Advice on making the best (short)", "Many operations don't have category");
        ad.longMessage=i18nc("Advice on making the best (long)", "Don't forget to associate a category for each operation. This will allow you to generate better reports.");
        ad.autoCorrection.push_back(i18nc("Advice on making the best (action)", "Open operations without category"));
        output.push_back(ad);
    }

    //Check operations without payee
    SKGObjectBase::getNbObjects(currentBankDocument, "v_operation_display","r_payee_id=0", nb);
    if (nb)
    {
        SKGInterfacePlugin::SKGAdvice ad;
        ad.uuid="skgoperationplugin_nopayee";
        ad.priority=5;
        ad.shortMessage=i18nc("Advice on making the best (short)", "Many operations don't have payee");
        ad.longMessage=i18nc("Advice on making the best (long)", "Don't forget to associate a payee for each operation. This will allow you to generate better reports.");
        ad.autoCorrection.push_back(i18nc("Advice on making the best (action)", "Open operations without payee"));
        output.push_back(ad);
    }

    //Too many money on a current account
    SKGServices::executeSelectSqliteOrder(currentBankDocument, "SELECT t_name, f_RATE FROM v_account_display WHERE t_close='N' AND f_RATE>0 ORDER BY f_RATE DESC", result);
    nb=result.count();
    if (nb>1)
    {
        //Get better interest account
        QString target=result.at(1).at(0);
        QString rate=result.at(1).at(1);

        //Get accounts with too much money
        SKGServices::executeSelectSqliteOrder(currentBankDocument, "SELECT t_name FROM v_account_display WHERE f_RATE<"+rate+" AND t_type='C' AND t_close='N' AND f_CURRENTAMOUNT>-2*(SELECT TOTAL(s.f_CURRENTAMOUNT) FROM v_operation_display s WHERE s.rd_account_id=v_account_display.id AND s.t_TYPEEXPENSE='-' AND s.d_DATEMONTH = strftime('%Y-%m',date('now','start of month', '-1 MONTH')))", result);
        nb=result.count();
        for (int i=1; i<nb; ++i) //Ignore header
        {
            //Get parameters
            QString account=result.at(i).at(0);

            SKGInterfacePlugin::SKGAdvice ad;
            ad.uuid="skgoperationplugin_too_much_money_"+account;
            ad.priority=6;
            ad.shortMessage=i18nc("Advice on making the best (short)", "Too much money in your account '%1'", account);
            ad.longMessage=i18nc("Advice on making the best (long)", "You could save money on an account with a better rate. Example: '%1' (%2%)",target, rate);
            output.push_back(ad);
        }
    }
    return output;
}

SKGError SKGOperationPlugin::executeAdviceCorrection(const QString& iAdviceIdentifier, int iSolution) const
{
    if (currentBankDocument && iAdviceIdentifier.startsWith(QLatin1String("skgoperationplugin_duplicate_")))
    {
        //Get parameters
        QString parameters=iAdviceIdentifier.right(iAdviceIdentifier.length()-29);
        int pos=parameters.indexOf(';');
        QString num=parameters.left(pos);
        QString account=parameters.right(parameters.length()-1-pos);

        //Call operation plugin
        QDomDocument doc ( "SKGML" );
        doc.setContent ( currentBankDocument->getParameter ( "SKGOPERATION_DEFAULT_PARAMETERS" ) );
        QDomElement root = doc.documentElement();
        if ( root.isNull() )
        {
            root=doc.createElement ( "parameters" );
            doc.appendChild ( root );
        }

        root.setAttribute ( "operationTable", "v_operation_display" );
        root.setAttribute ( "operationWhereClause", "i_number="+SKGServices::stringToSqlString(num)+" AND t_ACCOUNT='"+SKGServices::stringToSqlString(account)+'\'' );
        root.setAttribute ( "title", i18nc("Noun, a list of items", "Operations of '%1' with duplicate number %2",account, num ) );
        root.setAttribute ( "title_icon", "security-low" );
        SKGMainPanel::getMainPanel()->openPage ( SKGMainPanel::getMainPanel()->getPluginByName ( "Skrooge operation plugin" ), -1, doc.toString() );
        return SKGError();
    }
    else if (currentBankDocument && iAdviceIdentifier.startsWith(QLatin1String("skgoperationplugin_notreconciliated_")))
    {
        //Get parameters
        QString account=iAdviceIdentifier.right(iAdviceIdentifier.length()-36);

        //Call operation plugin
        QDomDocument doc ( "SKGML" );
        doc.setContent ( currentBankDocument->getParameter ( "SKGOPERATION_DEFAULT_PARAMETERS" ) );
        QDomElement root = doc.documentElement();
        if ( root.isNull() )
        {
            root=doc.createElement ( "parameters" );
            doc.appendChild ( root );
        }

        root.setAttribute ( "account", account );
        root.setAttribute ( "modeInfoZone", "1" );
        root.setAttribute ( "currentPage", "-1" );
        SKGMainPanel::getMainPanel()->openPage ( SKGMainPanel::getMainPanel()->getPluginByName ( "Skrooge operation plugin" ), -1, doc.toString() );
        return SKGError();
    }
    else if (currentBankDocument && iAdviceIdentifier=="skgoperationplugin_nocategory")
    {
        //Get parameters
        QString account=iAdviceIdentifier.right(iAdviceIdentifier.length()-36);

        //Call operation plugin
        QDomDocument doc ( "SKGML" );
        doc.setContent ( currentBankDocument->getParameter ( "SKGOPERATION_DEFAULT_PARAMETERS" ) );
        QDomElement root = doc.documentElement();
        if ( root.isNull() )
        {
            root=doc.createElement ( "parameters" );
            doc.appendChild ( root );
        }

        root.setAttribute ( "operationTable", "v_operation_display" );
        root.setAttribute ( "operationWhereClause", "EXISTS (SELECT 1 FROM suboperation WHERE rd_operation_id=v_operation_display.id AND r_category_id=0)" );
        root.setAttribute ( "title", i18nc("Noun, a list of items", "Operations without category") );
        root.setAttribute ( "title_icon", "skrooge_category" );
        SKGMainPanel::getMainPanel()->openPage ( SKGMainPanel::getMainPanel()->getPluginByName ( "Skrooge operation plugin" ), -1, doc.toString() );
        return SKGError();
    }
    else if (currentBankDocument && iAdviceIdentifier=="skgoperationplugin_nopayee")
    {
        //Get parameters
        QString account=iAdviceIdentifier.right(iAdviceIdentifier.length()-36);

        //Call operation plugin
        QDomDocument doc ( "SKGML" );
        doc.setContent ( currentBankDocument->getParameter ( "SKGOPERATION_DEFAULT_PARAMETERS" ) );
        QDomElement root = doc.documentElement();
        if ( root.isNull() )
        {
            root=doc.createElement ( "parameters" );
            doc.appendChild ( root );
        }

        root.setAttribute ( "operationTable", "v_operation_display" );
        root.setAttribute ( "operationWhereClause", "r_payee_id=0" );
        root.setAttribute ( "title", i18nc("Noun, a list of items", "Operations without payee") );
        root.setAttribute ( "title_icon", "skrooge_payee" );
        SKGMainPanel::getMainPanel()->openPage ( SKGMainPanel::getMainPanel()->getPluginByName ( "Skrooge operation plugin" ), -1, doc.toString() );
        return SKGError();
    }

    return SKGInterfacePlugin::executeAdviceCorrection(iAdviceIdentifier, iSolution);
}

#include "skgoperationplugin.moc"
