/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
#ifndef SKGIMPORTEXPORTMANAGER_H
#define SKGIMPORTEXPORTMANAGER_H
/** @file
 * This file defines classes SKGImportExportManager.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include <QObject>
#include <QStringList>
#include <QMap>

#include "skgerror.h"
#include "skgbankmodeler_export.h"

#ifdef SKG_ENABLE_OFX
#include <libofx/libofx.h>
#endif

class SKGDocumentBank;
class SKGAccountObject;
class SKGUnitObject;
class QDate;

/**
 *Manage import and export
 */
class SKGBANKMODELER_EXPORT SKGImportExportManager : public QObject
{
    Q_OBJECT;
    /**
     * This enumerate defines dump modes
     */
    Q_ENUMS ( ImportExportMode );

public:
    /**
     * This enumerate defines dump modes
     */
    enum ImportExportMode
    {
        AUTO, 		/**< To export or inport in the right mode based on extension file */
        CSV, 		/**< To export or inport in CSV mode */
        CSVUNIT, 	/**< To inport in CSV mode values for a unit */
        GNUCASH,	/**< To inport in GNUCASH file */
        GSB,		/**< To inport in GRISBI file */
        XHB,		/**< To inport in HOMEBANK file */
        SKG,		/**< To export or inport in SKROOGE file */
        SQLITE,		/**< To export or inport in SQLITE file */
        KMY,		/**< To export or inport in KMYMONEY file */
        XML,  		/**< To export in XML mode */
        QIF,  		/**< To export or inport in QIF mode */
        OFX 		/**< To export or inport in OFX mode */
    };

    /**
     * Constructor.
     * @param iDocument the document
     * @param iFileName the file name
     * @param iMode the mode
     */
    explicit SKGImportExportManager ( SKGDocumentBank* iDocument,
                                      const QString& iFileName="",
                                      ImportExportMode iMode=SKGImportExportManager::AUTO );

    /**
     * Destructor
     */
    virtual ~SKGImportExportManager();

    /**
     * Get the import/export mode.
     * @return the import/export mode
     */
    virtual ImportExportMode getMode();

    /**
     * Set the codec used for imports.
     * @param iCodec the codec name.
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError setCodec ( const QString& iCodec );

    /**
     * Set the default account for import in case of account is not detected in imported file.
     * @param iAccount the account where to import. NULL if you want to create a specific account for that.
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError setDefaultAccount ( SKGAccountObject* iAccount );

    /**
     * Set the default unit for import in case of unit is not detected in imported file.
     * @param iUnit the unit where to import. NULL if you want to create a specific unit for that.
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError setDefaultUnit ( SKGUnitObject* iUnit );

    /**
     * Enable/disable the automatically validation of imported operations.
     * @param iValidation the mode.
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual void setAutomaticValidation ( bool iValidation );
    /**
     * Enable/disable the automatically rules application on imported operations.
     * @param iApply the mode.
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual void setAutomaticApplyRules ( bool iApply );

    /**
     * Set the CSV mapping rule.
     * A mapping rule is a list of regular expression allowing to automatically map each attribute:
     * the operation object.
     * List of supported attribute:
     * date
     * number
     * mode
     * payee
     * comment
     * status
     * bookmarked
     * account
     * category
     * amount
     * quantity
     * sign
     * unit
     * This list is a list of operation attributes.
     * @param iCSVMappingRules the mapping. NULL to build automatically the CSV mapping.
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError setCSVMappingRules ( const QMap<QString, QString>& iCSVMappingRules );

    /**
     * Set the CSV mapping.
     * A mapping is ordered list to described the mapping between the csv file and
     * the operation object.
     * List of supported key word:
     * date
     * number
     * mode
     * payee
     * comment
     * status
     * bookmarked
     * account
     * category
     * amount
     * quantity
     * sign
     * unit
     * This list is a list of operation attributes.
     * @param iCSVMapping the mapping. NULL to build automatically the CSV mapping.
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError setCSVMapping ( const QStringList* iCSVMapping=NULL );

    /**
     * Get the CSV mapping.
     * @return the CSV mapping
     */
    virtual QStringList getCSVMapping() const;

    /**
     * Set the index of the header in the CSV file.
     * @param iIndex the index. -1 to search automatically the index of the header.
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError setCSVHeaderIndex ( int iIndex=-1 );

    /**
     * Get the index of the header in the CSV file.
     * @return the index
     */
    virtual int getCSVHeaderIndex();

    /**
     * Import the file in the document
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError importFile();

    /**
     * Export the file in the document
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError exportFile();

    /**
     * Anonymize the document.
     * This function must not be launched into a transaction
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError anonymize();

    /**
     * Find and group operations
     * @param oNbOperationsMerged returns the number of operations merged.
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError findAndGroupTransfers ( int& oNbOperationsMerged );

    /**
     * Clean operations after an import coming from bank's web sites
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError cleanBankImport();

    /**
     * Get the document
     * @return the document.
     */
    virtual SKGDocumentBank* getDocument();

    /**
     * Get file name
     * @return the file name.
     */
    virtual QString getFileName();

    /**
     * Return the default account for import
     * @param oAccount the default account for import.
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError getDefaultAccount ( SKGAccountObject& oAccount );

    /**
     * Return the preferred unit for a date for import
     * @param oUnit the default unit for import.
     * @param iDate the date.
     * @brief
     * If @see setDefaultUnit is used then getDefaultUnit will return this unit.
     * else return the unit compatible with entry date and with a value nearest than 1
     * else a new unit is created and returned
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError getDefaultUnit ( SKGUnitObject& oUnit, const QDate* iDate=NULL );

    /**
     * To know if OFX i supported
     * @return true or false.
     */
    static bool isOFXSupported();

private:
    Q_DISABLE_COPY ( SKGImportExportManager );

    SKGError exportQIF();
    SKGError exportCSV();
    SKGError exportOFX();
    SKGError importQIF();
    SKGError importCSV();
#ifdef SKG_ENABLE_OFX
    SKGError importOFX();
#endif
    SKGError importCSVUnit();

#ifdef SKG_ENABLE_OFX
    static int ofxAccountCallback ( struct OfxAccountData data, void * pv );
    static int ofxTransactionCallback ( struct OfxTransactionData data, void * pv );
    static int ofxStatementCallback ( struct OfxStatementData data, void * pv );
    static QList<OfxStatementData> ofxInitialBalance;
    static SKGError getAccount(OfxAccountData* iAccountData, SKGDocumentBank* iDoc, SKGAccountObject& oAccount);
#endif

    static int nbOperationsNotImported;
    static int nbOperationsImported;
    static SKGError ofxError;

    SKGError finalizedImportedOperations();

    QStringList getCSVMappingFromLine ( const QString& iLine );

    SKGDocumentBank* document;
    ImportExportMode mode;
    QString fileName;
    QStringList csvMapping;
    QMap<QString, QString>  csvMapper;
    QChar csvSeparator;
    int csvHeaderIndex;
    SKGAccountObject* defaultAccount;
    SKGUnitObject* defaultUnit;
    QString codec;
    bool automaticValidationOfImportedOperation;
    bool automaticApplyRulesOfImportedOperation;
};

#endif
