/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A plugin to manage properties on objects
 *
 * @author Stephane MANKOWSKI
 */
#include "skgpropertiesplugin.h"
#include "skgpropertiesplugindockwidget.h"
#include "skgtraces.h"
#include "skgmainpanel.h"
#include "skgtransactionmng.h"

#include <QDockWidget>

#include <kactioncollection.h>
#include <kstandardaction.h>
#include <ktoolbarpopupaction.h>
/**
 * This plugin factory.
 */
K_PLUGIN_FACTORY(SKGPropertiesPluginFactory, registerPlugin<SKGPropertiesPlugin>();)
/**
 * This plugin export.
 */
K_EXPORT_PLUGIN(SKGPropertiesPluginFactory("skg_properties", "skg_properties"))

SKGPropertiesPlugin::SKGPropertiesPlugin(QObject* iParent, const QVariantList& /*iArg*/) :
        SKGInterfacePlugin(iParent), dockWidget(NULL), dockContent(NULL)
{
    SKGTRACEIN(10, "SKGPropertiesPlugin::SKGPropertiesPlugin");
}

SKGPropertiesPlugin::~SKGPropertiesPlugin()
{
    SKGTRACEIN(10, "SKGPropertiesPlugin::~SKGPropertiesPlugin");
    currentDocument=NULL;
    dockWidget=NULL;
    dockContent=NULL;
    addPropertyMenu=NULL;
    addPropertyAction=NULL;
}

bool SKGPropertiesPlugin::setupActions(SKGDocument* iDocument, const QStringList& iArgument)
{
    SKGTRACEIN(10, "SKGPropertiesPlugin::setupActions");
    Q_UNUSED(iArgument);

    currentDocument=iDocument;


    setComponentData( SKGPropertiesPluginFactory::componentData() );
    setXMLFile("skg_properties.rc");

    dockContent = new SKGPropertiesPluginDockWidget(currentDocument);
    if (dockContent) {
        dockWidget = new QDockWidget(SKGMainPanel::getMainPanel());
        if (dockWidget) {
            dockWidget->setObjectName(QString::fromUtf8("skg_properties_docwidget"));
            dockWidget->setAllowedAreas(Qt::LeftDockWidgetArea|Qt::RightDockWidgetArea);
            dockWidget->setWindowTitle(title());
            dockWidget->setWidget(dockContent);

            // add action to control hide / display of Bookmarks
            dockWidget->toggleViewAction()->setShortcut(Qt::SHIFT+Qt::Key_F12);
            actionCollection()->addAction("view_properties", dockWidget->toggleViewAction());
        }
    }

    //Menu
    addPropertyAction = new KToolBarPopupAction ( KIcon ( "feed-subscribe" ), i18nc ( "Verb, action to add a property", "Add property" ), this );
    //connect ( addPropertyAction, SIGNAL ( triggered ( bool ) ), this, SLOT ( actionUndo() ) );
    actionCollection()->addAction ( QLatin1String ( "add_property" ), addPropertyAction );
    addPropertyMenu= static_cast<KMenu*>(addPropertyAction->menu());
    connect ( addPropertyMenu , SIGNAL ( aboutToShow() ), this, SLOT ( onShowAddPropertyMenu() ) );

    addPropertyAction->setStickyMenu ( false );
    addPropertyAction->setDelayed ( false );

    //redoAction=KStandardAction::redo(this, SLOT(actionRedo()), actionCollection());
    if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction ( "add-property", addPropertyAction );
    return true;
}

void SKGPropertiesPlugin::actionAddProperty()
{
    SKGTRACEIN(10, "SKGPropertiesPluginDockWidget::actionAddProperty");
    SKGError err;
    //Scope for the transaction
    QAction* act=dynamic_cast<QAction*>(sender());
    if (act && currentDocument)
    {
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
        //Get parameters
        QStringList list=act->data().toStringList();
        QString name=list.at(0);
        QString value=list.at(1);

        //Create properties
        if (err.isSucceeded()) {
            SKGObjectBase::SKGListSKGObjectBase selection=SKGMainPanel::getMainPanel()->getSelectedObjects();
            int nb=selection.count();
            SKGBEGINPROGRESSTRANSACTION(*currentDocument, i18nc("Noun, name of the user action", "Property creation"), err, nb);
            for (int i=0; err.isSucceeded() && i<nb; ++i) {
                err=selection[i].setProperty(name, value);
                if (err.isSucceeded())  err=currentDocument->stepForward(i+1);
            }
        }
        QApplication::restoreOverrideCursor();
    }

    //status bar
    if (err.isSucceeded()) {
        err=SKGError(0, i18nc("Successful message after an user action", "Property created"));
    }
    SKGMainPanel::getMainPanel()->displayErrorMessage(err);
}

void SKGPropertiesPlugin::onShowAddPropertyMenu()
{
    if ( addPropertyMenu ) {
        addPropertyMenu->clear();

        SKGObjectBase::SKGListSKGObjectBase sels=SKGMainPanel::getMainPanel()->getSelectedObjects();
        if (sels.count())
        {
            QString table=sels.at(0).getRealTable();
            SKGStringListList listTmp;
            SKGServices::executeSelectSqliteOrder ( currentDocument,
                                                    "SELECT t_name, t_value FROM (SELECT t_name, t_value, COUNT(1) AS nb FROM parameters WHERE t_uuid_parent like '%-"+table+"' GROUP BY t_name, t_value) ORDER BY nb DESC LIMIT 7",
                                                    listTmp );
            int nb=listTmp.count();
            for ( int i=1; i<nb; ++i ) {
                QAction* act=addPropertyMenu->addAction ( KIcon("feed-subscribe"), i18nc("Add a property (attribute=value)", "%1=%2",listTmp.at ( i ).at ( 0 ), listTmp.at ( i ).at ( 1 )));
                if ( act ) {
                    act->setData ( listTmp.at ( i ) );
                    connect ( act, SIGNAL ( triggered() ), this, SLOT ( actionAddProperty() ) );
                }
            }
        }
    }
}

void SKGPropertiesPlugin::refresh()
{
    SKGTRACEIN(10, "SKGPropertiesPlugin::refresh");
    if (dockContent) dockContent->refresh();
    if (addPropertyAction && SKGMainPanel::getMainPanel()) addPropertyAction->setEnabled(SKGMainPanel::getMainPanel()->getSelectedObjects().count()>0);
}

void SKGPropertiesPlugin::close()
{
    SKGTRACEIN(10, "SKGPropertiesPlugin::close");
}

QDockWidget* SKGPropertiesPlugin::getDockWidget()
{
    return dockWidget;
}

QString SKGPropertiesPlugin::title() const
{
    return i18nc("Noun, an item's properties","Properties");
}

int SKGPropertiesPlugin::getOrder() const
{
    return 6;
}

QStringList SKGPropertiesPlugin::tips() const
{
    QStringList output;
    output.push_back(i18nc("Description of a tips", "<p>... you can manage properties on all objects.</p>"));
    output.push_back(i18nc("Description of a tips", "<p>... you can add files or Internet links as property.</p>"));
    return output;
}

bool SKGPropertiesPlugin::isInContext() const
{
    return false;
}

#include "skgpropertiesplugin.moc"
