/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is Skrooge plugin to generate report.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgreportpluginwidget.h"
#include "skgmainpanel.h"
#include "skgbankincludes.h"
#include "skgtraces.h"

#include <klocale.h>
#include <kmenu.h>

#include <QtGui/QGraphicsScene>

/**
 * The size of the forecast
 */
#define FORECASTMAX 400

SKGReportPluginWidget::SKGReportPluginWidget ( SKGDocumentBank* iDocument, bool iMinimmumMode )
        :SKGTabPage ( iDocument ), nbLevelLines ( 0 ), nbLevelColumns ( 0 )
{
    SKGTRACEIN ( 10, "SKGReportPluginWidget::SKGReportPluginWidget" );

    timer = new QTimer ( this );
    timer->setSingleShot ( true );
    connect ( timer, SIGNAL ( timeout() ), this, SLOT ( refreshActivated() ) );

    ui.setupUi ( this );

    ui.kTitle->hide();
    ui.kPeriod->setCurrentIndex ( 1 );
    ui.kInterval->setCurrentIndex ( 2 );

    ui.kSetupReport->setIcon ( KIcon ( "configure" ) );
    ui.kLineUp->setIcon ( KIcon ( "list-add" ) );
    ui.kLineDown->setIcon ( KIcon ( "list-remove" ) );
    ui.kColUp->setIcon ( KIcon ( "list-add" ) );
    ui.kColDown->setIcon ( KIcon ( "list-remove" ) );

    ui.kMode->addItem ( KIcon ( "skg-chart-pie" ), i18nc ( "Noun, the numerical sum of a list of values","Sum" ) );
    ui.kMode->addItem ( KIcon ( "skg-chart-line" ), i18nc ( "Noun","History" ) );

    ui.kForecastCmb->addItem ( i18nc ( "Noun","None" ) );
    ui.kForecastCmb->addItem ( KIcon ( "chronometer" ), i18nc ( "Noun","Schedule" ) );
    ui.kForecastCmb->addItem ( KIcon ( "applications-education-mathematics" ), i18nc ( "Noun","Moving average" ) );

    //Build contextual menu
    KMenu* tableMenu=ui.kTableWithGraph->getTableContextualMenu();
    if ( tableMenu )
    {
        tableMenu->addSeparator();
        QStringList overlayopen;
        overlayopen.push_back ( "skg_open" );
        openReportAction = tableMenu->addAction ( KIcon ( "view-statistics", NULL, overlayopen ), i18nc ( "Verb", "Open report..." ) );
    }

    KMenu* graphMenu=ui.kTableWithGraph->getGraphContextualMenu();
    if ( graphMenu )
    {
        graphMenu->addSeparator();
        graphMenu->addAction ( openReportAction );
    }

    connect ( openReportAction, SIGNAL ( triggered ( bool ) ), this, SLOT ( onOpenReport() ) );

    //Init comboboxes
    AttsForColumns << "d_date" << "d_DATEWEEK" << "d_DATEMONTH" << "d_DATEQUARTER" << "d_DATESEMESTER" << "d_DATEYEAR" << "";
    AttsForLines << "t_REALCATEGORY" << "t_payee" << "t_mode" << "t_TYPEEXPENSENLS" << "t_status" << "t_ACCOUNTTYPE" << "t_UNITTYPE" << "t_REALREFUND";
    AttsForColumns+=AttsForLines;


    if ( iDocument )
    {
        foreach ( const QString& att, AttsForColumns )
        ui.kColumns->addItem ( iDocument->getIcon ( att ), iDocument->getDisplay ( att ) );

        foreach ( const QString& att, AttsForLines )
        ui.kLines->addItem ( iDocument->getIcon ( att ), iDocument->getDisplay ( att ) );

        ui.kColumns->setCurrentIndex ( 2 );
    }

    connect ( ui.kTableWithGraph->table(), SIGNAL ( currentItemChanged ( QTableWidgetItem*, QTableWidgetItem* ) ), this, SLOT ( onSelectionChanged() ) );

    //Refresh
    connect ( ui.kColumns, SIGNAL ( currentIndexChanged ( int ) ), this, SLOT ( refresh() ), Qt::QueuedConnection );
    connect ( ui.kLines, SIGNAL ( currentIndexChanged ( int ) ), this, SLOT ( refresh() ), Qt::QueuedConnection );
    connect ( ui.kMode, SIGNAL ( currentIndexChanged ( int ) ), this, SLOT ( refresh() ), Qt::QueuedConnection );
    connect ( ui.kPeriod, SIGNAL ( currentIndexChanged ( int ) ), this, SLOT ( refresh() ), Qt::QueuedConnection );
    connect ( ui.kInterval, SIGNAL ( currentIndexChanged ( int ) ),this,SLOT ( refresh() ), Qt::QueuedConnection );
    connect ( ui.kDateBegin, SIGNAL ( dateChanged ( QDate ) ),this,SLOT ( refresh() ),Qt::QueuedConnection );
    connect ( ui.kDateEnd, SIGNAL ( dateChanged ( QDate ) ),this,SLOT ( refresh() ),Qt::QueuedConnection );
    connect ( ui.kNbIntervals, SIGNAL ( valueChanged ( int ) ),this,SLOT ( refresh() ),Qt::QueuedConnection );
    connect ( ui.kIncomes, SIGNAL ( stateChanged ( int ) ),this,SLOT ( refresh() ),Qt::QueuedConnection );
    connect ( ui.kExpenses, SIGNAL ( stateChanged ( int ) ),this,SLOT ( refresh() ),Qt::QueuedConnection );
    connect ( ui.kTransfers,SIGNAL ( stateChanged ( int ) ) ,this,SLOT ( refresh() ),Qt::QueuedConnection );
    connect ( ui.kForecastCmb,SIGNAL ( currentIndexChanged ( int ) ) ,this,SLOT ( refresh() ),Qt::QueuedConnection );
    connect ( ui.kForecastValue,SIGNAL ( valueChanged ( int ) ) ,this,SLOT ( refresh() ),Qt::QueuedConnection );
    connect ( ui.kTableOnly,SIGNAL ( toggled ( bool ) ),ui.kTableWithGraph,SLOT ( showTable() ) );
    connect ( ui.kGraphOnly,SIGNAL ( toggled ( bool ) ),ui.kTableWithGraph,SLOT ( showGraph() ) );
    connect ( ui.kTableAndGraph,SIGNAL ( toggled ( bool ) ),ui.kTableWithGraph,SLOT ( showTableAndGraph() ) );

    connect ( ( const QObject* ) getDocument(), SIGNAL ( tableModified ( QString,int ) ), this, SLOT ( dataModified ( QString, int ) ) );

    if ( iMinimmumMode )
    {
        ui.kGraphOnly->toggle();
        setCurrentMode ( -1 );
    }
    else
    {
        setCurrentMode ( 0 );
    }
}

SKGReportPluginWidget::~SKGReportPluginWidget()
{
    SKGTRACEIN ( 10, "SKGReportPluginWidget::~SKGReportPluginWidget" );
    timer=NULL;
}

int SKGReportPluginWidget::getCurrentMode()
{
    return mode;
}

void SKGReportPluginWidget::onBtnModeClicked()
{
    QWidget* sender=static_cast<QWidget*> ( this->sender() );

    int currentMode=getCurrentMode();
    int newMode=0;

    if ( sender==ui.kSetupReport ) newMode=0;

    if ( currentMode==newMode ) newMode=-1;
    setCurrentMode ( newMode );
}

void SKGReportPluginWidget::setCurrentMode ( int iMode )
{
    mode=iMode;

    ui.setupWidget->setVisible ( mode==0 );

    if ( mode==0 )
    {
        ui.kSetupReport->setChecked ( true );
    }
}

QString SKGReportPluginWidget::getState()
{
    SKGTRACEIN ( 10, "SKGReportPluginWidget::getState" );
    QDomDocument doc ( "SKGML" );
    QDomElement root;
    if ( lastState.hasChildNodes() )
    {
        doc=lastState;
        root = doc.documentElement();
    }
    else
    {
        root = doc.createElement ( "parameters" );
        doc.appendChild ( root );
    }

    QString representation;
    if ( ui.kGraphOnly->isChecked() )
    {
        representation = "graph";
    }
    else if ( ui.kTableOnly->isChecked() )
    {
        representation = "table";
    }
    else if ( ui.kTableAndGraph->isChecked() )
    {
        representation = "both";
    }
    root.setAttribute ( "representation",representation );

    root.setAttribute ( "columns", AttsForColumns.value ( ui.kColumns->currentIndex() ) );
    root.setAttribute ( "lines", AttsForLines.value ( ui.kLines->currentIndex() ) );
    root.setAttribute ( "mode", SKGServices::intToString ( ui.kMode->currentIndex() ) );
    root.setAttribute ( "period", SKGServices::intToString ( ui.kPeriod->currentIndex() ) );
    if(ui.kPeriod->currentIndex() ==4)
    {
	root.setAttribute ( "date_begin", SKGServices::intToString ( ui.kDateBegin->date().toJulianDay() ) );
	root.setAttribute ( "date_end", SKGServices::intToString ( ui.kDateEnd->date().toJulianDay() ) );
    }
    root.setAttribute ( "interval", SKGServices::intToString ( ui.kInterval->currentIndex() ) );
    root.setAttribute ( "nb_intervals",SKGServices::intToString ( ui.kNbIntervals->value() ) );
    root.setAttribute ( "incomes", ui.kIncomes->isChecked() ? "Y" : "N" );
    root.setAttribute ( "expenses", ui.kExpenses->isChecked() ? "Y" : "N" );
    root.setAttribute ( "transfers",ui.kTransfers->isChecked() ? "Y" : "N" );
    root.setAttribute ( "currentPage", SKGServices::intToString ( getCurrentMode() ) );
    root.setAttribute ( "tableAndGraphState", ui.kTableWithGraph->getState() );
    root.setAttribute ( "nbLevelLines", SKGServices::intToString ( nbLevelLines ) );
    root.setAttribute ( "nbLevelColumns", SKGServices::intToString ( nbLevelColumns ) );
    root.setAttribute ( "forecast", SKGServices::intToString ( ui.kForecastCmb->currentIndex() ) );
    root.setAttribute ( "forecastValue", SKGServices::intToString ( ui.kForecastValue->value() ) );

    if ( operationWhereClause.length() ) root.setAttribute ( "operationWhereClause", operationWhereClause );

    return doc.toString();
}

void SKGReportPluginWidget::setState ( const QString& iState )
{
    SKGTRACEIN ( 10, "SKGReportPluginWidget::setState" );
    QDomDocument doc ( "SKGML" );
    if ( doc.setContent ( iState ) )
    {
        QDomElement root = doc.documentElement();

        QString representation = root.attribute ( "representation" );
        QString columns=root.attribute ( "columns" );
        QString lines=root.attribute ( "lines" );
        QString mode=root.attribute ( "mode" );
        QString period=root.attribute ( "period" );
        QString interval = root.attribute ( "interval" );
        QString nb_interval = root.attribute ( "nb_intervals" );
        QString date_begin=root.attribute ( "date_begin" );
        QString date_end=root.attribute ( "date_end" );
        QString incomes = root.attribute ( "incomes" );
        QString expenses= root.attribute ( "expenses" );
        QString transfers = root.attribute ( "transfers" );
        QString currentPage=root.attribute ( "currentPage" );
        QString forecast=root.attribute ( "forecast" );
        QString forecastValue=root.attribute ( "forecastValue" );

        QString tableAndGraphState=root.attribute ( "tableAndGraphState" );
        QString title=root.attribute ( "title" );
        QString title_icon=root.attribute ( "title_icon" );
        operationWhereClause=root.attribute ( "operationWhereClause" );
        QString nbLevelLinesString=root.attribute ( "nbLevelLines" );
        QString nbLevelColumnsString=root.attribute ( "nbLevelColumns" );
        if ( representation == "table" )
        {
            ui.kTableOnly->toggle();
        }
        else if ( representation == "graph" )
        {
            ui.kGraphOnly->toggle();
        }
        else if ( representation == "both" )
        {
            ui.kTableAndGraph->toggle();
        }

        if ( !nbLevelLinesString.isEmpty() ) nbLevelLines=SKGServices::stringToInt ( nbLevelLinesString );
        if ( !nbLevelColumnsString.isEmpty() ) nbLevelColumns=SKGServices::stringToInt ( nbLevelColumnsString );
        if ( !columns.isEmpty() ) ui.kColumns->setCurrentIndex ( AttsForColumns.indexOf ( columns ) );
        if ( !lines.isEmpty() ) ui.kLines->setCurrentIndex ( AttsForLines.indexOf ( lines ) );
        if ( !mode.isEmpty() ) ui.kMode->setCurrentIndex ( SKGServices::stringToInt ( mode ) );
        if ( !period.isEmpty() ) ui.kPeriod->setCurrentIndex ( SKGServices::stringToInt ( period ) );
        if ( !interval.isEmpty() ) ui.kInterval->setCurrentIndex ( SKGServices::stringToInt ( interval ) );
        if ( !nb_interval.isEmpty() ) ui.kNbIntervals->setValue ( SKGServices::stringToInt ( nb_interval ) );
        if ( !date_begin.isEmpty() ) ui.kDateBegin->setDate ( QDate::fromJulianDay ( SKGServices::stringToInt ( date_begin ) ) );
        if ( !date_end.isEmpty() ) ui.kDateEnd->setDate ( QDate::fromJulianDay ( SKGServices::stringToInt ( date_end ) ) );
        if ( !incomes.isEmpty() ) ui.kIncomes->setChecked ( incomes!="N" );
        if ( !expenses.isEmpty() ) ui.kExpenses->setChecked ( expenses!="N" );
        if ( !transfers.isEmpty() ) ui.kTransfers->setChecked ( transfers!="N" );
        if ( !currentPage.isEmpty() ) setCurrentMode ( SKGServices::stringToInt ( currentPage ) );
        if ( !forecast.isEmpty() ) ui.kForecastCmb->setCurrentIndex ( SKGServices::stringToInt ( forecast ) );
        if ( !forecastValue.isEmpty() ) ui.kForecastValue->setValue ( SKGServices::stringToInt ( forecastValue ) );

        refresh();

        ui.kTableWithGraph->setState ( tableAndGraphState );
        if ( !title.isEmpty() )
        {
            ui.kTitle->setComment ( "<html><body><b>"+SKGServices::stringToHtml ( title ) +"</b></body></html>" );
            ui.kTitle->show();
        }
        else
        {
            ui.kTitle->hide();
        }
        if ( !title_icon.isEmpty() ) ui.kTitle->setPixmap ( KIcon ( title_icon ).pixmap ( 22, 22 ), KTitleWidget::ImageLeft );
        if ( !operationWhereClause.isEmpty() )
        {
            //We keep a copy of given state in case of bookmark
            lastState=doc;
            dataModified ( "", 0 );
        }
    }
    else
    {
        refresh();
        ui.kTableWithGraph->setState ( "" );
    }
}



QString SKGReportPluginWidget::getDefaultStateAttribute()
{
    return "SKGREPORT_DEFAULT_PARAMETERS";
}

QWidget* SKGReportPluginWidget::getWidgetForPrint()
{
    return this;
}

void SKGReportPluginWidget::getWhereClauseAndTitleForSelection ( int row, int column, QString& oWc, QString& oTitle )
{
    //Build where clause and title
    QString wc;
    QString title;
    if ( ui.kTitle->isVisible() ) oTitle=ui.kTitle->text() +'\n';

    SKGTableWithGraph::DisplayAdditionalFlag mode=ui.kTableWithGraph->getAdditionnalDisplayMode();

    //Condition on line attribute
    oTitle+=i18nc ( "Noun, a list of items", "Sub operations with " );
    if ( mode==SKGTableWithGraph::NONE || row+1!=ui.kTableWithGraph->table()->rowCount() )
    {
        QString att=AttsForLines[ui.kLines->currentIndex() ];
        oWc=att;
        QString lineVal=ui.kTableWithGraph->table()->item ( row, 0 )->text();
        if ( lineVal.isEmpty() )
        {
            oWc+=" IS NULL OR "+att+"=''";
            oWc='('+oWc+')';
            oTitle+=i18nc ( "Noun",  "%1 are empty", ui.kLines->currentText() );
        }
        else
        {
            oWc+=" = '"+SKGServices::stringToSqlString ( lineVal ) +"' OR "+
                 att+" like '"+SKGServices::stringToSqlString ( lineVal ) +OBJECTSEPARATOR+"%'";
            oWc='('+oWc+')';
            oTitle+=i18nc ( "Noun",  "%1 with '%2'", ui.kLines->currentText(), lineVal );
        }
    }
    //Condition on column attribute
    int nbCol=ui.kTableWithGraph->getNbColumns();
    if ( column!=0 && column<nbCol )
    {
        QString att=AttsForColumns[ui.kColumns->currentIndex() ];
        if ( !att.isEmpty() )
        {
            if ( !oWc.isEmpty() )
            {
                oWc+=" AND ";
                oTitle+=i18nc ( "Noun",  " and " );
            }

            oWc+='('+att;
            QString val=ui.kTableWithGraph->table()->horizontalHeaderItem ( column )->text();
            if ( val.isEmpty() )
            {
                oWc+=" IS NULL OR "+att+"=''";
                oTitle+=i18nc ( "Noun",  "%1 are empty", ui.kColumns->currentText() );
            }
            else
            {
                oWc+=" = '"+SKGServices::stringToSqlString ( val ) +"' OR "+
                     att+" like '"+SKGServices::stringToSqlString ( val ) +OBJECTSEPARATOR+"%'";
                oTitle+=i18nc ( "Noun",  "%1 with '%2'", ui.kColumns->currentText(), val );
            }
            oWc+=')';
        }
    }

    //Condition on other attribute
    if ( !oWc.isEmpty() )
    {
        oWc+=" AND ";
        oTitle+=i18nc ( "Noun",  " and " );
    }
    oWc+=getConsolidatedWhereClause();

    QString during=ui.kPeriod->text();
    if ( ui.kPeriod->currentIndex() ==4 )
    {
        during+=SKGServices::timeToString ( QDateTime ( ui.kDateBegin->date() ) ) +' '+SKGServices::timeToString ( QDateTime ( ui.kDateEnd->date() ) );
    }
    else if ( ui.kPeriod->currentIndex() >0 )
    {
        during+=' '+SKGServices::intToString ( ui.kNbIntervals->value() ) +' '+ui.kInterval->text();
    }

    QStringList types;
    if ( ui.kIncomes->isChecked() ) types.push_back ( i18nc ( "Noun",  "incomes" ) );
    if ( ui.kExpenses->isChecked() ) types.push_back ( i18nc ( "Noun",  "expenses" ) );
    if ( ui.kTransfers->isChecked() ) types.push_back ( i18nc ( "Noun",  "transfers" ) );

    oTitle+=i18nc ( "Noun",  "during '%1' for '%2'",during, types.join ( " " ) );
}

void SKGReportPluginWidget::onDoubleClick ( int row, int column )
{
    _SKGTRACEIN ( 10, "SKGReportPluginWidget::onDoubleClick" );

    QString wc;
    QString title;
    getWhereClauseAndTitleForSelection ( row, column, wc, title );

    //Open
    if ( QApplication::keyboardModifiers() &Qt::ControlModifier && QApplication::keyboardModifiers() &Qt::ShiftModifier )
    {
        //Call debug plugin
        QDomDocument doc ( "SKGML" );
        QDomElement root = doc.createElement ( "parameters" );
        doc.appendChild ( root );
        root.setAttribute ( "sqlOrder", "SELECT * from v_operation_consolidated WHERE "+wc );

        SKGMainPanel::getMainPanel()->setNewTabContent ( SKGMainPanel::getMainPanel()->getPluginByName ( "Skrooge debug plugin" ), -1, doc.toString() );
    }
    else
    {
        //Call operation plugin
        QDomDocument doc ( "SKGML" );
        doc.setContent ( getDocument()->getParameter ( "SKGOPERATION_CONSOLIDATED_DEFAULT_PARAMETERS" ) );
        QDomElement root = doc.documentElement();
        if ( root.isNull() )
        {
            root=doc.createElement ( "parameters" );
            doc.appendChild ( root );
        }

        root.setAttribute ( "operationTable", "v_operation_consolidated" );
        root.setAttribute ( "operationWhereClause", wc );
        root.setAttribute ( "title", title );
        root.setAttribute ( "title_icon", "view-statistics" );
        root.setAttribute ( "currentPage", "-1" );

        SKGMainPanel::getMainPanel()->setNewTabContent ( SKGMainPanel::getMainPanel()->getPluginByName ( "Skrooge operation plugin" ), -1, doc.toString() );
    }
}

void SKGReportPluginWidget::onOpenReport()
{
    SKGError err;
    SKGTRACEINRC ( 10, "SKGReportPluginWidget::onOpenReport",err );
    QList<QTableWidgetItem *> selection=ui.kTableWithGraph->table()->selectedItems();
    if ( selection.count() )
    {
        QString wc;
        QString title;
        getWhereClauseAndTitleForSelection ( selection.at ( 0 )->row(), selection.at ( 0 )->column(), wc, title );

        //Call report plugin
        QDomDocument doc ( "SKGML" );
        doc.setContent ( getState() );
        QDomElement root = doc.documentElement();
        root.setAttribute ( "operationWhereClause", wc );
        root.setAttribute ( "title", title );
        root.setAttribute ( "title_icon", "view-statistics" );
        SKGMainPanel::getMainPanel()->setNewTabContent ( SKGMainPanel::getMainPanel()->getPluginByName ( "Skrooge report plugin" ), -1, doc.toString() );
    }
}

void SKGReportPluginWidget::onOneLevelMore()
{
    _SKGTRACEIN ( 10, "SKGTableWithGraph::onOneLevelMore" );
    if ( sender() ==ui.kLineUp ) ++nbLevelLines;
    else ++nbLevelColumns;
    refresh();
}

void SKGReportPluginWidget::onOneLevelLess()
{
    _SKGTRACEIN ( 10, "SKGTableWithGraph::onOneLevelLess" );
    if ( sender() ==ui.kLineDown ) --nbLevelLines;
    else --nbLevelColumns;
    refresh();
}

QString SKGReportPluginWidget::getConsolidatedWhereClause ( QString* oWhereClausForPreviousData, QString*  oWhereClausForForecastData)
{
    //Build where clause
    QString wc;

    QString strfFormat;
    QString sqlInterval;
    double val=ui.kNbIntervals->value();
    double one=1;
    QDate a=QDate::currentDate();
    QDate b=a;

    switch ( ui.kInterval->currentIndex() )
    {
    case 0:
        // Interval is in days
        strfFormat = "'%Y-%m-%d'";
        sqlInterval = "DAY";
        break;
    case 1:
        // Interval is in weeks
        strfFormat = "'%Y-%W'";
        sqlInterval = "DAY";
        val*=7;
        one*=7;
        break;
    case 2:
        // Interval is in months
        strfFormat = "'%Y-%m'";
        sqlInterval = "MONTH";
        break;
    case 3:
        // Interval is in years
        strfFormat = "'%Y'";
        sqlInterval = "YEAR";
        break;
    }

    switch ( ui.kPeriod->currentIndex() )
    {
    case 1:
        // Current Interval
        switch ( ui.kInterval->currentIndex() )
        {
        case 0:
            // Interval is in days
            break;
        case 1:
            // Interval is in weeks
            a=a.addDays ( 1-a.dayOfWeek() );
            b=a.addDays ( 7-1 );
            break;
        case 2:
            // Interval is in months
            a=a.addDays ( 1-a.day() );
            b=a.addMonths ( 1 ).addDays ( -1 );
            break;
        case 3:
            // Interval is in years
            a=a.addDays ( 1-a.day() ).addMonths ( 1-a.month() );
            b=a.addYears ( 1 ).addDays ( -1 );
            break;
        }
        ui.kDateBegin->setDate ( a );
        ui.kDateEnd->setDate ( b );
        wc = "strftime(" + strfFormat + ",d_date) = strftime(" + strfFormat + ",'now')";
        if ( oWhereClausForPreviousData ) *oWhereClausForPreviousData="strftime(" + strfFormat + ",d_date) < strftime(" + strfFormat + ",'now')";
        if ( oWhereClausForForecastData ) *oWhereClausForForecastData="strftime(" + strfFormat + ",d_date) > strftime(" + strfFormat + ",'now')";
        break;
    case 2:
        // Previous Interval
        switch ( ui.kInterval->currentIndex() )
        {
        case 0:
            // Interval is in days
            b=b.addDays ( -1 );
            a=b.addDays ( -val+1 );
            break;
        case 1:
            // Interval is in weeks
            b=b.addDays ( -a.dayOfWeek() );
            a=b.addDays ( -val+1 );
            break;
        case 2:
            // Interval is in months
            b=b.addDays ( -b.day() );
            a=a.addDays ( 1-a.day() ).addMonths ( -val );
            break;
        case 3:
            // Interval is in years
            b=b.addDays ( -b.day() ).addMonths ( -b.month() );
            a=a.addMonths ( 1-a.month() ).addDays ( 1-a.day() ).addYears ( -val );
            break;
        }
        ui.kDateBegin->setDate ( a );
        ui.kDateEnd->setDate ( b );
        wc = "strftime("+ strfFormat + ",d_date)>=strftime(" + strfFormat + ",date('now', '-" + SKGServices::intToString ( val ) + ' ' + sqlInterval + "'))";
	if(ui.kForecastCmb->currentIndex()!=1) wc+=" AND strftime("+ strfFormat + ",d_date)<=strftime(" + strfFormat + ",date('now', '-" + SKGServices::intToString ( one ) + ' ' + sqlInterval + "'))"; //For forecast based on scheduled operations
        if ( oWhereClausForPreviousData ) *oWhereClausForPreviousData="strftime(" + strfFormat + ",d_date) < strftime(" + strfFormat + ",date('now', '-" + SKGServices::intToString ( val ) + ' ' + sqlInterval + "'))";
        if ( oWhereClausForForecastData ) *oWhereClausForForecastData="strftime(" + strfFormat + ",d_date) > strftime(" + strfFormat + ",date('now', '-" + SKGServices::intToString ( one ) + ' ' + sqlInterval + "'))";
        break;
    case 3:
        //Last Interval
        switch ( ui.kInterval->currentIndex() )
        {
        case 0:
            // Interval is in days
            a=a.addDays ( -val );
            break;
        case 1:
            // Interval is in weeks
            a=a.addDays ( -val );
            break;
        case 2:
            // Interval is in months
            a=a.addMonths ( -val );
            break;
        case 3:
            // Interval is in years
            a=a.addYears ( -val );
            break;
        }
        a=a.addDays ( 1 );
        ui.kDateBegin->setDate ( a );
        ui.kDateEnd->setDate ( b );
        wc = "d_date > date('now','-" + SKGServices::intToString ( val ) + ' ' + sqlInterval + "')";
	if(ui.kForecastCmb->currentIndex()!=1) wc+=" AND d_date<=date('now')"; //For forecast based on scheduled operations
        if ( oWhereClausForPreviousData ) *oWhereClausForPreviousData="d_date <= date('now','-" + SKGServices::intToString ( val ) + ' ' + sqlInterval + "')";
        if ( oWhereClausForForecastData ) *oWhereClausForForecastData="d_date > date('now')";
        break;
    case 4:
        // Custom Date
        wc="d_date>='"+SKGServices::dateToSqlString ( QDateTime ( ui.kDateBegin->date() ) ) + '\'';
        if(ui.kForecastCmb->currentIndex()!=1) wc+="AND d_date<='"+SKGServices::dateToSqlString ( QDateTime ( ui.kDateEnd->date() ) ) + '\''; //For forecast based on scheduled operations
        if ( oWhereClausForPreviousData ) *oWhereClausForPreviousData="d_date<'"+SKGServices::dateToSqlString ( QDateTime ( ui.kDateBegin->date() ) ) + '\'';
        if ( oWhereClausForForecastData ) *oWhereClausForForecastData="d_date>'"+SKGServices::dateToSqlString ( QDateTime ( ui.kDateEnd->date() ) ) + '\'';
        break;
    default:
        // Take all dates
	a=a.addYears ( -1 );
        ui.kDateBegin->setDate ( a );
        ui.kDateEnd->setDate ( b );
        wc = "1=1";
        if ( oWhereClausForPreviousData ) *oWhereClausForPreviousData="1=0";
        if ( oWhereClausForForecastData ) *oWhereClausForForecastData="d_date > date('now')";
        break;
    }

    wc="(("+wc+")"+" OR d_date='0000') AND d_date!='0000-00-00'";
    if ( oWhereClausForPreviousData ) *oWhereClausForPreviousData= "(("+*oWhereClausForPreviousData+")"+" OR d_date='0000-00-00')";

    QStringList operationTypes;
    if ( ui.kIncomes->isChecked() )
    {
        operationTypes << " t_TYPEEXPENSE='+'";
    }

    if ( ui.kExpenses->isChecked() )
    {
        operationTypes << " t_TYPEEXPENSE='-'";
    }
    if ( operationTypes.length() > 0 )
    {
        QString condition=" AND (" + operationTypes.join ( " OR " ) +')';
        wc += condition;
        if ( oWhereClausForPreviousData ) *oWhereClausForPreviousData+= condition;
    }

    if ( ui.kTransfers->isChecked() == false )
    {
        QString condition=" AND i_group_id=0";
        wc += condition;
        if ( oWhereClausForPreviousData ) *oWhereClausForPreviousData+= condition;
    }

    if ( operationWhereClause.length() )
    {
        QString condition=" AND (" + operationWhereClause +')';
        wc += condition;
        if ( oWhereClausForPreviousData ) *oWhereClausForPreviousData+= condition;
    }

    return wc;
}

void SKGReportPluginWidget::onSelectionChanged()
{
    openReportAction->setEnabled ( ui.kTableWithGraph->table()->selectedItems().count() >0 );
}

void SKGReportPluginWidget::refresh()
{
    ui.kForecastValue->setEnabled(ui.kForecastCmb->currentIndex()>0);

    //Check dates
    ui.kDateSelect->setVisible ( ui.kPeriod->currentIndex() >0 );
    ui.kDateSelect->setEnabled ( ui.kPeriod->currentIndex() ==4 );

    QDate d1=ui.kDateBegin->date();
    QDate d2=ui.kDateEnd->date();
    if ( d1>d2 )
    {
        ui.kDateBegin->setDate ( d2 );
        ui.kDateEnd->setDate ( d1 );
    }

    //Check income & expense
    if ( !ui.kIncomes->isChecked() && !ui.kExpenses->isChecked() )
    {
        if ( sender() ==ui.kIncomes ) ui.kExpenses->setChecked ( true );
        else ui.kIncomes->setChecked ( true );
    }

    bool current = ( ui.kPeriod->currentIndex() == 1 );
    bool previous = ( ui.kPeriod->currentIndex() == 2 );
    bool last = ( ui.kPeriod->currentIndex() == 3 );

    ui.kInterval->setVisible ( current || previous || last );
    ui.kNbIntervals->setVisible ( last || previous );

    if ( timer ) timer->start ( 300 );
    else refreshActivated();
}

void SKGReportPluginWidget::refreshActivated()
{
    dataModified ( "", 0 );
}

void SKGReportPluginWidget::dataModified ( const QString& iTableName, int iIdTransaction )

{
    SKGTRACEIN ( 10, "SKGReportPluginWidget::dataModified" );
    Q_UNUSED ( iIdTransaction );

    //Refresh panel
    QSqlDatabase* db = getDocument()->getDatabase();
    setEnabled ( db!=NULL );
    if ( db!=NULL )
    {
        //Check if needed
        if ( iTableName=="operation" || iTableName.isEmpty() )
        {
            //Check if update is needed
            QString ParametersUsed=getState() +';'+SKGServices::intToString ( getDocument()->getTransactionToProcess() );;
            if ( ParametersUsed==previousParametersUsed )
            {
                SKGTRACEL ( 10 ) << "Same parameters. Refresh ignored" << endl;
                return;
            }
            previousParametersUsed=ParametersUsed;

            QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );

            //Fill table
            SKGError err;

            //Get parameters
            int col =ui.kColumns->currentIndex();
            int line =ui.kLines->currentIndex();

            if ( col>=0 && line>=0 )
            {
                //Mode history ?
                bool modeHistory= ( ui.kMode->currentIndex() >0 );

		//Get condition (must be done before forecast on scheduled operations)
		int nbVirtualColumn=0;
                QString conditionPrevious;
                QString conditionForecast;
                QString condition=getConsolidatedWhereClause ( &conditionPrevious, &conditionForecast );

		//Compute forecast on scheduled operations
		if (ui.kForecastCmb->currentIndex() ==1 )
		{
		  //Create scheduled operation
		  getDocument()->beginTransaction ( "#INTERNAL#", 2 );
		  int nbInserted=0;
		  int nbDays=ui.kDateBegin->date().daysTo(QDate::currentDate());
		  nbDays=(nbDays*ui.kForecastValue->value())/FORECASTMAX;
		  QDate lastDate=QDate::currentDate().addDays(nbDays);
		  SKGRecurrentOperationObject::process(getDocument(), nbInserted, lastDate);
		  ui.kForecastValue->setToolTip(SKGServices::dateToSqlString(QDateTime(lastDate)));
		}
		else
		{
		  ui.kForecastValue->setToolTip("");
		  conditionForecast="1=0";
		}

                //Execute sql order
                SKGStringListList table;
                QString tableName="v_operation_consolidated";
                if ( modeHistory && !AttsForColumns.at ( col ).isEmpty() )
                {
                    tableName="(SELECT ";
                    if ( AttsForColumns.at ( col ) !="d_date" ) tableName+="(CASE WHEN "+conditionPrevious+" THEN '0000' ELSE d_date END) AS d_date, ";
                    tableName+="(CASE WHEN "+conditionPrevious+" THEN '0000' ELSE "+AttsForColumns.at ( col ) +"||(CASE WHEN "+conditionForecast+" THEN '999' ELSE '' END) END) AS "+AttsForColumns.at ( col ) +",* FROM v_operation_consolidated)";
                }
                err=getDocument()->getConsolidatedView ( tableName, AttsForColumns.at ( col ), AttsForLines.at ( line ), "f_REALCURRENTAMOUNT", "TOTAL", condition, table );
                IFSKGTRACEL ( 10 )
                {
                    QStringList dump=SKGServices::tableToDump ( table, SKGServices::DUMP_TEXT );
                    int nbl=dump.count();
                    for ( int i=0; i<nbl; ++i )
                    {
                        SKGTRACE << dump[i] << endl;
                    }
                }

		if (ui.kForecastCmb->currentIndex() ==1 )
		{
		  //Rollback create operations
		  getDocument()->endTransaction(false);

		  //Compute nb date in futur
                  QStringList line1=table.at ( 0 );
		  int nbCol=line1.count();
		  for ( int i=0; i<nbCol; ++i )
		  {
		      QString title=line1.at(i);
		      if(title.endsWith(QLatin1String("999")))
		      {
			title=title.left(title.count()-3);
			line1.replace ( i, title );
			++nbVirtualColumn;
		      }
		  }
		  table.replace ( 0, line1 );
		}

                if ( err.isSucceeded() )
                {
                    if ( table.count() )
                    {
                        //Change title
                        QStringList line1=table.at ( 0 );
                        line1.replace ( 0, ui.kLines->text() );
                        table.replace ( 0, line1 );
                    }

                    //Compute forecast on average
                    int nbLines=table.count();
                    if ( nbLines && ui.kForecastCmb->currentIndex() ==2 )
                    {
			QStringList newLine=table.at ( 0 );
			int nbVals=newLine.count()-1;
		        int pourcent=ui.kForecastValue->value();
			if(nbVals>=3 && pourcent>0)
			{
			    //Compute nb value to add
			    nbVirtualColumn=pourcent*nbVals/FORECASTMAX;

			    //Build header
			    for ( int i=0; i<nbVirtualColumn; ++i )
			    {
				newLine.push_back ( i18nc ( "Noun","N %1", (i+1) ) );
			    }
			    table.replace ( 0, newLine );

			    //Build values
			    for ( int j=1; j<nbLines; ++j )
			    {
				QStringList newLine=table.at ( j );
				for ( int i=0; i<nbVirtualColumn; ++i )
				{
				    //Moving average
				    double nv=0;
				    for(int k=0; k<nbVirtualColumn; ++k)
				    {
					nv+=SKGServices::stringToDouble ( newLine.at(nbVals+i-k) );
				    }
				    newLine.push_back (SKGServices::doubleToString(nv/nbVirtualColumn));
				}
				table.replace ( j, newLine );
			    }
			}
                    }

                    //Create history
                    if ( modeHistory ) table=SKGReportPluginWidget::getHistorizedTable ( table );

                    //Create grouped by lines table
                    int nbLevelLineMax=0;
                    {
                        int nbCols=-1;
                        SKGStringListList groupedByLineTable=table;
                        {
                            QString previousLineName="###";
                            if ( groupedByLineTable.count() )
                            {
                                nbCols=groupedByLineTable.at ( 0 ).count();
                            }
                            for ( int i=1; nbCols && i<groupedByLineTable.count(); ++i ) //Dynamically modified
                            {
                                QStringList line=groupedByLineTable.at ( i );
                                QString val=line.at ( 0 );

                                //Rebuild val for the number of level
                                QStringList vals=val.split ( OBJECTSEPARATOR );
                                int nbvals=vals.count();
                                if ( nbvals>nbLevelLines+1 )
                                {
                                    //Rebuild val
                                    val="";
                                    for ( int k=0; k<=nbLevelLines; ++k )
                                    {
                                        val+=vals[k];
                                        if ( k!=nbLevelLines ) val+=OBJECTSEPARATOR;
                                    }
                                }
                                nbLevelLineMax=qMax ( nbLevelLineMax, nbvals-1 );

                                if ( val==previousLineName )
                                {
                                    //Current line is merged with previous one
                                    QStringList newLine;
                                    newLine.push_back ( val );
                                    for ( int k=1; k<nbCols; ++k )
                                    {
                                        QString valstring1=line.at ( k );
                                        QString valstring2=groupedByLineTable.at ( i-1 ).at ( k );

                                        double sum2=0;
                                        if ( !valstring1.isEmpty() || !valstring2.isEmpty() )
                                        {
                                            if ( !valstring1.isEmpty() ) sum2+=SKGServices::stringToDouble ( valstring1 );
                                            if ( !valstring2.isEmpty() ) sum2+=SKGServices::stringToDouble ( valstring2 );
                                            newLine.push_back ( SKGServices::doubleToString ( sum2 ) );
                                        }
                                        else
                                        {
                                            newLine.push_back ( "" );
                                        }
                                    }

                                    groupedByLineTable.replace ( i-1, newLine );

                                    //Remove current line
                                    groupedByLineTable.removeAt ( i );
                                    --i;
                                }
                                else
                                {
                                    //Current line is just modified
                                    QStringList newLine;
                                    newLine.push_back ( val );
                                    for ( int k=1; k<nbCols; ++k )
                                    {
                                        newLine.push_back ( line.at ( k ) );
                                    }
                                    groupedByLineTable.replace ( i, newLine );

                                    previousLineName=val;
                                }
                            }
                        }
                        table=groupedByLineTable;
                    }

                    //Create grouped by columns table
                    int nbLevelColMax=0;
                    {
                        SKGStringListList groupedByColTable=table;
                        int nbLines=groupedByColTable.count();
                        if ( nbLines )
                        {
                            QString previousColumnName="###";
                            for ( int i=1; nbLines && i<groupedByColTable.at ( 0 ).count(); ++i ) //Dynamically modified
                            {
                                QString val=groupedByColTable.at ( 0 ).at ( i );

                                //Rebuild val for the number of level
                                QStringList vals=val.split ( OBJECTSEPARATOR );
                                int nbvals=vals.count();
                                if ( nbvals>nbLevelColumns+1 )
                                {
                                    //Rebuild val
                                    val="";
                                    for ( int k=0; k<=nbLevelColumns; ++k )
                                    {
                                        val+=vals[k];
                                        if ( k!=nbLevelColumns ) val+=OBJECTSEPARATOR;
                                    }
                                }
                                nbLevelColMax=qMax ( nbLevelColMax, nbvals-1 );

                                if ( val==previousColumnName )
                                {
                                    //Current column is merged with previous one
                                    QStringList newLine=groupedByColTable.at ( 0 );
                                    newLine.removeAt ( i );
                                    groupedByColTable.replace ( 0, newLine );

                                    for ( int k=1; k<nbLines; ++k )
                                    {
                                        newLine=groupedByColTable.at ( k );
                                        QString valstring1=newLine.at ( i );
                                        QString valstring2=newLine.at ( i-1 );

                                        double sum2=0;
                                        if ( !valstring1.isEmpty() || !valstring2.isEmpty() )
                                        {
                                            if ( !valstring1.isEmpty() ) sum2+=SKGServices::stringToDouble ( valstring1 );
                                            if ( !valstring2.isEmpty() ) sum2+=SKGServices::stringToDouble ( valstring2 );
                                            newLine.replace ( i-1, SKGServices::doubleToString ( sum2 ) );
                                        }
                                        else
                                        {
                                            newLine.removeAt ( i-1 );
                                        }
                                        newLine.removeAt ( i );
                                        groupedByColTable.replace ( k, newLine );
                                    }

                                    //Remove current line
                                    --i;
                                }
                                else
                                {
                                    //Current column is just modified
                                    QStringList newLine=groupedByColTable.at ( 0 );
                                    newLine.replace ( i, val );
                                    groupedByColTable.replace ( 0, newLine );

                                    previousColumnName=val;
                                }
                            }
                        }
                        table=groupedByColTable;
                    }

                    //Set data
                    SKGServices::SKGUnitInfo primaryUnit= ( ( SKGDocumentBank* ) getDocument() )->getPrimaryUnit();
                    SKGServices::SKGUnitInfo secondaryUnit= ( ( SKGDocumentBank* ) getDocument() )->getSecondaryUnit();

                    SKGTableWithGraph::DisplayAdditionalFlag dmode=SKGTableWithGraph::NONE;
                    if ( AttsForColumns.at ( col ).startsWith ( QLatin1String ( "d_" ) ) ) dmode|=SKGTableWithGraph::AVERAGE|SKGTableWithGraph::LIMITS;
                    if ( !modeHistory ) dmode|=SKGTableWithGraph::SUM;
                    ui.kTableWithGraph->setData ( table, primaryUnit, secondaryUnit, dmode, nbVirtualColumn );

                    //Enable/Disable buttons
                    nbLevelLines=qMin ( nbLevelLineMax, nbLevelLines );
                    ui.kLineDown->setEnabled ( nbLevelLines>0 );
                    ui.kLineUp->setEnabled ( nbLevelLines<nbLevelLineMax );

                    nbLevelColumns=qMin ( nbLevelColMax, nbLevelColumns );
                    ui.kColDown->setEnabled ( nbLevelColumns>0 );
                    ui.kColUp->setEnabled ( nbLevelColumns<nbLevelColMax );
                }
            }

            QApplication::restoreOverrideCursor();

            //Display error
            SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
        }
    }
}

SKGStringListList SKGReportPluginWidget::getHistorizedTable ( const SKGStringListList& iTable )
{
    //Build history
    SKGStringListList historizedTable;

    historizedTable.push_back ( iTable.at ( 0 ) );

    QStringList newLine;
    int nblines=iTable.count();
    int nbCols=0;
    if ( nblines )
    {
        nbCols=iTable.at ( 0 ).count();
    }
    for ( int i=1; i<nblines; ++i )
    {
        QStringList newLine;
        newLine.push_back ( iTable.at ( i ).at ( 0 ) );

        double sum=0;
        for ( int j=1; j<nbCols; ++j )
        {
            sum+=SKGServices::stringToDouble ( iTable.at ( i ).at ( j ) );
            newLine.push_back ( SKGServices::doubleToString ( sum ) );
        }
        historizedTable.push_back ( newLine );
    }

    return historizedTable;
}

#include "skgreportpluginwidget.moc"
