/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A skrooge plugin to manage properties on objects.
 *
 * @author Stephane MANKOWSKI
 */
#include "skgpropertiesplugindockwidget.h"
#include "skgmainpanel.h"
#include "skgtraces.h"
#include "skgdocument.h"
#include "skgservices.h"
#include "skgnamedobject.h"
#include "skgobjectmodelbase.h"
#include "skgsortfilterproxymodel.h"
#include "skgtransactionmng.h"

#include <kfiledialog.h>

#include <QFile>

SKGPropertiesPluginDockWidget::SKGPropertiesPluginDockWidget(SKGDocument* iDocument)
                :SKGTabPage(iDocument)
{
        SKGTRACEIN(1, "SKGPropertiesPluginDockWidget::SKGPropertiesPluginDockWidget");

        ui.setupUi(this);

        ui.kPicture->hide();

        ui.kAdd->setIcon(KIcon("list-add"));
        ui.kRemove->setIcon(KIcon("list-remove"));
        ui.kSelectFile->setIcon(KIcon("document-open"));

        //Add model
        SKGObjectModelBase* modelview = new SKGObjectModelBase(getDocument(), "parameters", "1=1 ORDER BY t_uuid_parent, t_name", this, "", false);
        SKGSortFilterProxyModel* modelproxy = new SKGSortFilterProxyModel(this);
        modelproxy->setSourceModel(modelview);
        ui.kView->setModel(modelproxy);

        ui.kFilterEdit->setProxy(modelproxy);

        ui.kView->setDefaultSaveParameters(getDocument(), "SKG_DEFAULT_PROPERTIES");

        connect(ui.kView, SIGNAL(selectionChangedDelayed()), this, SLOT(onSelectionChanged()));
}

SKGPropertiesPluginDockWidget::~SKGPropertiesPluginDockWidget()
{
        SKGTRACEIN(1, "SKGPropertiesPluginDockWidget::~SKGPropertiesPluginDockWidget");
}

void SKGPropertiesPluginDockWidget::refresh()
{
        SKGTRACEIN(1, "SKGPropertiesPluginDockWidget::refresh");

        //Change filter
        SKGObjectModelBase* model=(SKGObjectModelBase*) ui.kView->model();
        QSortFilterProxyModel *proxyModel = qobject_cast<QSortFilterProxyModel *>(model);
        if (proxyModel) model=(SKGObjectModelBase*) proxyModel->sourceModel();

        if (model) {
                QString filter="t_uuid_parent IN (";

                SKGObjectBase::SKGListSKGObjectBase selection=SKGMainPanel::getMainPanel()->getSelectedObjects();
                int nb=selection.count();
                if (nb) {
                        QString tableName;
                        for (int i=0; i<nb; ++i) {
                                if (i>0) filter+=',';
                                else tableName=selection.at(i).getRealTable();
                                filter+='\''+selection.at(i).getUniqueID()+'\'';
                        }

                        ui.kAdd->setEnabled(true);
                        ui.kSelectFile->setEnabled(true);
                        ui.kRemove->setEnabled(false);
                        ui.kAttribute->setEnabled(true);
                        ui.kValue->setEnabled(true);

                        //Fill combo box
                        SKGMainPanel::fillWithDistinctValue(ui.kAttribute, getDocument(), "parameters", "t_name", "t_uuid_parent like '%-"+tableName+'\'');
                        SKGMainPanel::fillWithDistinctValue(ui.kValue, getDocument(), "parameters", "t_value", "t_uuid_parent like '%-"+tableName+'\'');
                } else {
                        filter+="'XXX'"; //Always false
                        ui.kAdd->setEnabled(false);
                        ui.kSelectFile->setEnabled(false);
                        ui.kRemove->setEnabled(false);
                        ui.kAttribute->setEnabled(false);
                        ui.kValue->setEnabled(false);
                }
                filter+=") ORDER BY t_uuid_parent, t_name";

                ui.kView->saveSelection();

                model->setFilter(filter);
                model->refresh();

                ui.kView->resetSelection();
        }

        ui.kView->setState("");
        if (ui.kView->isAutoResized()) ui.kView->resizeColumnsToContents();

        onSelectionChanged();
}

SKGObjectBase::SKGListSKGObjectBase SKGPropertiesPluginDockWidget::getSelectedObjects()
{
        return ui.kView->getSelectedObjects();
}

int SKGPropertiesPluginDockWidget::getNbSelectedObjects()
{
        return ui.kView->getNbSelectedObjects();
}

void SKGPropertiesPluginDockWidget::onSelectionChanged()
{
        SKGTRACEIN(10, "SKGPropertiesPluginDockWidget::onSelectionChanged");
        int nbSelected=getNbSelectedObjects();
        ui.kPicture->hide();
        ui.kRemove->setEnabled(nbSelected>0);
        if (nbSelected>0) {
                SKGObjectBase::SKGListSKGObjectBase objs=getSelectedObjects();
                SKGObjectBase obj=objs.at(0);
                ui.kAttribute->setText(obj.getAttribute("t_name"));
                ui.kValue->setText(obj.getAttribute("t_value"));

                if (nbSelected==1) {
                        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
                        QStringList uuid=obj.getAttribute("t_uuid_parent").split('-');
                        if (uuid.count()==2) {

                                SKGObjectBase p(getDocument(), uuid.at(1), SKGServices::stringToInt(uuid.at(0)));
                                QVariant blob=p.getPropertyBlob(obj.getAttribute("t_name"));
                                if (!blob.isNull()) {
                                        QByteArray blob_bytes=blob.toByteArray();

                                        QPixmap img;
                                        img.loadFromData(blob_bytes);
                                        ui.kPicture->setPixmap(img.scaled(ui.kPicture->size(), Qt::KeepAspectRatio));
                                        ui.kPicture->show();
                                }

                        }
                        QApplication::restoreOverrideCursor();
                }
        }

        if (ui.kView->isAutoResized()) ui.kView->resizeColumnsToContents();
}

void SKGPropertiesPluginDockWidget::onAddProperty()
{
        SKGTRACEIN(10, "SKGPropertiesPluginDockWidget::onAddProperty");
        SKGError err;
        QStringList listUUID;
        //Scope for the transaction
        {
                QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
                //Get parameters
                QString name=ui.kAttribute->text();
                QString value=ui.kValue->text();
                QVariant blob;
                QFile file(value);
                if (file.exists()) {
                        //Value is a file name ==> blob
                        if (!file.open(QIODevice::ReadOnly)) {
                                err=SKGError(ERR_INVALIDARG, i18nc("Error message", "Open file [%1] failed", value));
                        } else {
                                QByteArray blob_bytes=file.readAll();
                                if (!blob_bytes.count()) {
                                        err=SKGError(ERR_INVALIDARG, i18nc("Error message", "Open file [%1] failed", value));
                                } else {
                                        blob=blob_bytes;
                                        value=QFileInfo(value).fileName();
                                }

                                //close file
                                file.close();
                        }
                }

                //Create properties
                if (err.isSucceeded()) {
                        SKGObjectBase::SKGListSKGObjectBase selection=SKGMainPanel::getMainPanel()->getSelectedObjects();
                        int nb=selection.count();
                        SKGBEGINPROGRESSTRANSACTION(*getDocument(), i18nc("Noun, name of the user action", "Property creation"), err, nb);
                        for (int i=0; err.isSucceeded() && i<nb; ++i) {
                                SKGObjectBase prop;
                                err=selection[i].setProperty(name, value, blob, &prop);

                                if (err.isSucceeded()) {
                                        listUUID.push_back(prop.getUniqueID());
                                        err=getDocument()->stepForward(i+1);
                                }
                        }
                }
                QApplication::restoreOverrideCursor();
        }

        //status bar
        if (err.isSucceeded()) {
                err=SKGError(0, i18nc("Successful message after an user action", "Property created"));
                ui.kView->selectObjects(listUUID, true);
        }
        SKGMainPanel::getMainPanel()->displayErrorMessage(err);
}

void SKGPropertiesPluginDockWidget::onSelectFile()
{
        QString fileName=KFileDialog::getOpenFileName( KUrl ( "kfiledialog:///IMPEXP" ), "image/png image/jpeg image/gif image/tiff" , this );
        ui.kValue->setText(fileName);
}

void SKGPropertiesPluginDockWidget::onRemoveProperty()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGPropertiesPluginDockWidget::onRemoveProperty",err);
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
        SKGObjectBase::SKGListSKGObjectBase selection=getSelectedObjects();
        int nb=selection.count();
        SKGBEGINPROGRESSTRANSACTION(*getDocument(), i18nc("Verb, delete an item","Delete"), err, nb);
        for (int i=0; err.isSucceeded() && i<nb; ++i) {
                err=selection[i].remove();
                if (err.isSucceeded()) err=getDocument()->stepForward(i+1);
        }

        //status bar
        if (err.isSucceeded()) err=SKGError(0, i18nc("Successful message after an user action", "Properties deleted."));
        else err.addError(ERR_FAIL, i18nc("Error message", "Delete failed"));
        QApplication::restoreOverrideCursor();

        //Display error
        SKGMainPanel::getMainPanel()->displayErrorMessage(err);
}

void SKGPropertiesPluginDockWidget::cleanEditor()
{
        if (getNbSelectedObjects()==0) {
                ui.kAttribute->setText("");
                ui.kValue->setText("");
        }
}
#include "skgpropertiesplugindockwidget.moc"


