/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is Skrooge plugin to generate categories.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgcategoriesplugin.h"
#include "skgcategoriespluginwidget.h"
#include "skgtraces.h"
#include "skgmainpanel.h"
#include "skgtransactionmng.h"
#include "skgcategoryobject.h"

#include <kmessagebox.h>
#include <kactioncollection.h>
#include <kaction.h>

/**
 * This plugin factory.
 */
K_PLUGIN_FACTORY(SKGCategoriesPluginFactory, registerPlugin<SKGCategoriesPlugin>(););
/**
 * This plugin export.
 */
K_EXPORT_PLUGIN(SKGCategoriesPluginFactory("skrooge_categories", "skrooge_categories"));

SKGCategoriesPlugin::SKGCategoriesPlugin(QObject* iParent, const QVariantList& /*iArg*/)
                :SKGInterfacePlugin(iParent), currentBankDocument(NULL), importStdCatAction(NULL)
{
        SKGTRACEIN(10, "SKGCategoriesPlugin::SKGCategoriesPlugin");
}

SKGCategoriesPlugin::~SKGCategoriesPlugin()
{
        SKGTRACEIN(10, "SKGCategoriesPlugin::~SKGCategoriesPlugin");
        currentBankDocument=NULL;
        importStdCatAction=NULL;
}

void SKGCategoriesPlugin::setupActions(SKGDocument* iDocument, const QStringList& iArgument)
{
        SKGTRACEIN(10, "SKGCategoriesPlugin::setupActions");
        Q_UNUSED(iArgument);

        currentBankDocument=iDocument;


        // Tell the host application to load my GUI component
        setComponentData( SKGCategoriesPluginFactory::componentData() );
        setXMLFile("skrooge_categories.rc");

        //Import categories
        QStringList overlaycategories;
        overlaycategories.push_back("skrooge_category");

        importStdCatAction = new KAction(KIcon("document-import", NULL, overlaycategories), i18nc("Verb", "Import standard categories"), this);
        connect(importStdCatAction, SIGNAL(triggered(bool)), SLOT(importStandardCategories()));
        actionCollection()->addAction( QLatin1String("import_standard_categories"), importStdCatAction );

        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->registedGlobalAction("import_standard_categories", importStdCatAction);
}

void SKGCategoriesPlugin::refresh()
{
        SKGTRACEIN(10, "SKGCategoriesPlugin::refresh");
        if (currentBankDocument) {
                bool test=(currentBankDocument->getDatabase()!=NULL);
                if (importStdCatAction) importStdCatAction->setEnabled(test);

                //Automatic categories creation
                if (currentBankDocument->getDatabase()!=NULL) {
                        QString doc_id=currentBankDocument->getUniqueIdentifier();
                        if (docUniqueIdentifier!=doc_id) {
                                docUniqueIdentifier=doc_id;

                                int nbCategory=0;
                                SKGError err=SKGObjectBase::getNbObjects(currentBankDocument, "category", "", nbCategory);
                                if (err.isSucceeded() && nbCategory==0) {
					/*QApplication::setOverrideCursor(QCursor(Qt::ArrowCursor));
					int rc=KMessageBox::questionYesNo(SKGMainPanel::getMainPanel(), i18nc("Question", "Do you want to import standard categories?"));
					QApplication::restoreOverrideCursor();
					if (rc==KMessageBox::Yes) {*/
					//To simplify gui, categories are always imported
					importStandardCategories();
					//}
                                }
                        }
                }
        }
}

void SKGCategoriesPlugin::importStandardCategories()
{
        SKGTRACEIN(10, "SKGCategoriesPlugin::importStandardCategories");
        SKGError err;
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

	QString cats=i18nc("List of categories.It is not needed to translate each item. You can set the list you want. ';' must be used to separate categories. ' > ' must be used to separate categorie and sub caterogie (no limit of level).","Alimony;Auto;Auto > Fuel;Auto > Insurance;Auto > Lease;Auto > Loan;Auto > Registration;Auto > Service;Bank Charges;Bank Charges > Interest Paid;Bank Charges > Service Charge;Bills;Bills > Electricity;Bills > Fuel Oil;Bills > Local Taxes;Bills > Mortgage;Bills > Natural Gas;Bills > Rent;Bills > TV;Bills > Telephone;Bills > Water & Sewage;Bonus;Business;Business > Auto;Business > Capital Goods;Business > Legal Expenses;Business > Office Rent;Business > Office Supplies;Business > Other;Business > Revenue;Business > Taxes;Business > Travel;Business > Utilities;Business > Wages & Salary;Car;Car > Fuel;Car > Insurance;Car > Lease;Car > Loan;Car > Registration;Car > Service;Cash Withdrawal;Charity;Charity > Donations;Child Care;Child Support;Clothing;Disability;Div Income;Div Income > Ord dividend;Div Income > Stock dividend;Education;Education > Board;Education > Books;Education > Fees;Education > Loans;Education > Tuition;Employment;Employment > Benefits;Employment > Foreign;Employment > Lump sums;Employment > Other employ;Employment > Salary & wages;Food;Food > Dining Out;Food > Groceries;Gardening;Gift Received;Gifts;Healthcare;Healthcare > Dental;Healthcare > Doctor;Healthcare > Hospital;Healthcare > Optician;Healthcare > Prescriptions;Holidays;Holidays > Accomodation;Holidays > Travel;Household;Household > Furnishings;Household > Repairs;Insurance;Insurance > Auto;Insurance > Disability;Insurance > Home and Contents;Insurance > Life;Insurance > Medical;Int Inc;Int Inc > Bank Interest;Int Inc > Gross;Int Inc > Net;Int Inc > Other savings;Invest. income;Invest. income > 1st option;Invest. income > Dividend;Invest. income > Foreign;Invest. income > Other savings;Invest. income > Other trusts;Invest. income > Other trusts#Capital;Invest. income > Other trusts#Dist. rec'd;Invest. income > Other trusts#Estate;Investment Income;Investment Income > Dividends;Investment Income > Interest;Investment Income > Long-Term Capital Gains;Investment Income > Short-Term Capital Gains;Investment Income > Tax-Exempt Interest;Job Expense;Job Expense > Non-Reimbursed;Job Expense > Reimbursed;Legal Fees;Leisure;Leisure > Books & Magazines;Leisure > Entertaining;Leisure > Films & Video Rentals;Leisure > Hobbies;Leisure > Sporting Events;Leisure > Sports Goods;Leisure > Tapes & CDs;Leisure > Theatre & Concerts etc;Leisure > Toys & Games;Loan;Loan > Loan Interest;Long-Term Capital gains;Mortgage;Mortgage > Interest;Mortgage > PMI;Mortgage > Principle;Motor;Motor > Fuel;Motor > Loan;Motor > Service;Other Expense;Other Expense > Unknown;Other Income;Other Income > Child Support;Other Income > Employee Share Option;Other Income > Gifts Received;Other Income > Loan Principal Received;Other Income > Lottery or Premium Bond Prizes;Other Income > Student loan;Other Income > Tax Refund;Other Income > Unemployment Benefit;Pension;Pension > Employer;Personal Care;Pet Care;Pet Care > Food;Pet Care > Supplies;Pet Care > Vet's Bills;Recreation;Retirement Accounts;Retirement Accounts > 401(k)403(b) Plan Contributions;Retirement Accounts > 529 Plan Contributions;Retirement Accounts > IRA Contributions;Retirement Income;Retirement Income > 401(k);Retirement Income > 401(k) > 403(b) Distributions;Retirement Income > IRA Distributions;Retirement Income > Pensions & Annuities;Retirement Income > State Pension Benefits;Short-Term Capital gains;Social Security Benefits;Taxes;Taxes > AMT;Taxes > Federal Tax;Taxes > Federal Taxes;Taxes > Local Tax;Taxes > Local Taxes;Taxes > Other Invest;Taxes > Other Tax;Taxes > Property Taxes;Taxes > Social Security;Taxes > State Tax;Taxes > State Taxes;Travel;Travel > Accomodations;Travel > Car Rental;Travel > Fares;Utilities;Utilities > Electricity;Utilities > Garbage & Recycling;Utilities > Gas;Utilities > Sewer;Utilities > Telephone;Utilities > Water;Wages & Salary;Wages & Salary > Benefits;Wages & Salary > Bonus;Wages & Salary > Commission;Wages & Salary > Employer Pension Contributions;Wages & Salary > Gross Pay;Wages & Salary > Net Pay;Wages & Salary > Overtime;Wages & Salary > Workman's Comp");

	SKGBEGINTRANSACTION(*currentBankDocument, i18nc("Noun, name of the user action", "Import standard categories"), err);

	foreach(const QString& item, SKGServices::splitCSVLine(cats, ';')) {
		QString line=item.trimmed();
		if (!line.isEmpty()) {
			SKGCategoryObject cat;
			err=SKGCategoryObject::createPathCategory((SKGDocumentBank*) currentBankDocument, line, cat);
		}
	}
        QApplication::restoreOverrideCursor();

        //status
        if (err.isSucceeded())  err=SKGError(0, i18nc("Successful message after an user action", "Standard categories imported."));
        else err.addError(ERR_FAIL, i18nc("Error message", "Importing standard categories failed."));

        //Display error
        if (SKGMainPanel::getMainPanel()) SKGMainPanel::getMainPanel()->displayErrorMessage(err);
}

void SKGCategoriesPlugin::close()
{
        SKGTRACEIN(10, "SKGCategoriesPlugin::close");
}

SKGTabPage* SKGCategoriesPlugin::getWidget()
{
        SKGTRACEIN(10, "SKGCategoriesPlugin::getWidget");
        return new SKGCategoriesPluginWidget((SKGDocumentBank*) currentBankDocument);
}

QString SKGCategoriesPlugin::title() const
{
        return i18nc("Noun, categories of items","Categories");
}

QString SKGCategoriesPlugin::icon() const
{
        return "skrooge_category";
}

QString SKGCategoriesPlugin::toolTip () const
{
        return i18nc("A tool tip", "Categories management");
}

QStringList SKGCategoriesPlugin::tips() const
{
        QStringList output;
        output.push_back(i18nc("Description of a tips", "<p>... categories can be reorganized by drag & drop.</p>"));
        return output;
}

int SKGCategoriesPlugin::getOrder() const
{
        return 30;
}

bool SKGCategoriesPlugin::isInContext() const
{
        return true;
}

#include "skgcategoriesplugin.moc"
