/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is Skrooge plugin to generate report.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgreportpluginwidget.h"
#include "skgmainpanel.h"
#include "skgbankincludes.h"
#include "skgtraces.h"

#include <klocale.h>
#include <kmenu.h>

#include <QtGui/QGraphicsScene>

SKGReportPluginWidget::SKGReportPluginWidget(SKGMainPanel* iParent, SKGDocumentBank* iDocument)
                : SKGTabWidget(iParent, iDocument)
{
        SKGTRACEIN(10, "SKGReportPluginWidget::SKGReportPluginWidget");

        ui.setupUi(this);

        ui.kTitle->hide();

        //Build contextual menu
        ui.kTableWithGraph->getTableContextualMenu()->addSeparator();
        openReportAction = ui.kTableWithGraph->getTableContextualMenu()->addAction(KIcon("skrooge_open_report"), i18n("Open &report"));

        ui.kTableWithGraph->getGraphContextualMenu()->addSeparator();
        ui.kTableWithGraph->getGraphContextualMenu()->addAction(openReportAction);

        connect(openReportAction, SIGNAL(triggered(bool)), this, SLOT(onOpenReport()));

        //Init comboboxes
        AttsForColumns << "d_DATEMONTH" << "d_DATEYEAR" << "";
        ui.kColumns->addItem(i18n("Months"));
        ui.kColumns->addItem(i18n("Years"));
        ui.kColumns->addItem(i18n("-- Nothing --"));

        AttsForLines << "t_REALCATEGORY" << "t_payee" << "t_mode" << "t_TYPEEXPENSENLS" << "t_status" << "t_ACCOUNTTYPE" << "t_UNITTYPE";
        ui.kLines->addItem(i18n("Categories"));
        ui.kLines->addItem(i18n("Payees"));
        ui.kLines->addItem(i18n("Modes"));
        ui.kLines->addItem(i18n("Types"));
        ui.kLines->addItem(i18n("Status"));
        ui.kLines->addItem(i18n("Accounts"));
        ui.kLines->addItem(i18n("Units"));

        ui.kMode->addItem(i18n("Sum"));
        ui.kMode->addItem(i18n("History"));

        ui.kPeriod->addItem(i18n("All dates"));
        ui.kPeriod->addItem(i18n("Current month"));
        ui.kPeriod->addItem(i18n("Previous month"));
        ui.kPeriod->addItem(i18n("Current year"));
        ui.kPeriod->addItem(i18n("Previous year"));
        ui.kPeriod->addItem(i18n("Last 30 days"));
        ui.kPeriod->addItem(i18n("Last 3 months"));
        ui.kPeriod->addItem(i18n("Last 6 months"));
        ui.kPeriod->addItem(i18n("Last 12 months"));
        ui.kPeriod->addItem(i18n("Last 2 years"));
        ui.kPeriod->addItem(i18n("Last 3 years"));
        ui.kPeriod->addItem(i18n("Last 5 years"));
        ui.kPeriod->setCurrentIndex(1);

        ui.kType->addItem(i18n("All without transfer"));
        ui.kType->addItem(i18n("Income without transfer"));
        ui.kType->addItem(i18n("Expenditure without transfer"));
        ui.kType->addItem(i18n("All with transfers"));
        ui.kType->addItem(i18n("Income with transfers"));
        ui.kType->addItem(i18n("Expenditure with transfers"));

        connect(ui.kTableWithGraph->table(), SIGNAL(currentItemChanged(QTableWidgetItem*, QTableWidgetItem*)), this, SLOT(onSelectionChanged()));

        //Refresh
        connect(ui.kColumns, SIGNAL(currentIndexChanged(int)), this, SLOT(refresh()), Qt::QueuedConnection);
        connect(ui.kLines, SIGNAL(currentIndexChanged(int)), this, SLOT(refresh()), Qt::QueuedConnection);
        connect(ui.kMode, SIGNAL(currentIndexChanged(int)), this, SLOT(refresh()), Qt::QueuedConnection);
        connect(ui.kPeriod, SIGNAL(currentIndexChanged(int)), this, SLOT(refresh()), Qt::QueuedConnection);
        connect(ui.kType, SIGNAL(currentIndexChanged(int)), this, SLOT(refresh()), Qt::QueuedConnection);
        connect((const QObject*) getDocument(), SIGNAL(tableModified(QString, int)), this, SLOT(dataModified(QString, int)));

        refresh();
}

SKGReportPluginWidget::~SKGReportPluginWidget()
{
        SKGTRACEIN(10, "SKGReportPluginWidget::~SKGReportPluginWidget");
}

QString SKGReportPluginWidget::getState()
{
        SKGTRACEIN(10, "SKGReportPluginWidget::getState");
        QDomDocument doc("SKGML");
        QDomElement root;
        if (lastState.hasChildNodes()) {
                doc=lastState;
                root = doc.documentElement();
        } else {
                root = doc.createElement("parameters");
                doc.appendChild(root);
        }

        root.setAttribute("columns", ui.kColumns->currentIndex());
        root.setAttribute("lines", ui.kLines->currentIndex());
        root.setAttribute("mode", ui.kMode->currentIndex());
        root.setAttribute("period", ui.kPeriod->currentIndex());
        root.setAttribute("type", ui.kType->currentIndex());
        root.setAttribute("tableAndGraphState", ui.kTableWithGraph->getState());
        if (operationWhereClause.length()) root.setAttribute("operationWhereClause", operationWhereClause);

        return doc.toString();
}

void SKGReportPluginWidget::setState(const QString& iState )
{
        SKGTRACEIN(10, "SKGReportPluginWidget::setState");
        QDomDocument doc("SKGML");
        if (doc.setContent(iState)) {
                QDomElement root = doc.documentElement();

                QString columns=root.attribute ( "columns");
                QString lines=root.attribute ( "lines");
                QString mode=root.attribute ( "mode");
                QString period=root.attribute ( "period");
                QString type=root.attribute ( "type");
                QString tableAndGraphState=root.attribute ( "tableAndGraphState");
                QString title=root.attribute ( "title");
                QString title_icon=root.attribute ( "title_icon");
                operationWhereClause=root.attribute ( "operationWhereClause");
                if (!columns.isEmpty()) ui.kColumns->setCurrentIndex(SKGServices::stringToInt(columns));
                if (!lines.isEmpty()) ui.kLines->setCurrentIndex(SKGServices::stringToInt(lines));
                if (!mode.isEmpty()) ui.kMode->setCurrentIndex(SKGServices::stringToInt(mode));
                if (!period.isEmpty()) ui.kPeriod->setCurrentIndex(SKGServices::stringToInt(period));
                if (!type.isEmpty()) ui.kType->setCurrentIndex(SKGServices::stringToInt(type));
                ui.kTableWithGraph->setState(tableAndGraphState);
                if (!title.isEmpty()) {
                        ui.kTitle->setComment("<html><body><b>"+title+"</b></body></html>");
                        ui.kTitle->show();
                } else {
                        ui.kTitle->hide();
                }
                if (!title_icon.isEmpty()) ui.kTitle->setPixmap(KIcon(title_icon).pixmap(22, 22), KTitleWidget::ImageLeft);
                if (!operationWhereClause.isEmpty()) {
                        //We keep a copy of given state in case of bookmark
                        lastState=doc;
                        dataModified("", 0);
                }
        } else ui.kTableWithGraph->setState("");
}

QString SKGReportPluginWidget::getDefaultStateAttibute()
{
        return "SKGREPORT_DEFAULT_PARAMETERS";
}

QWidget* SKGReportPluginWidget::getWidgetForPrint()
{
        return this;
}

void SKGReportPluginWidget::getWhereClauseAndTitleForSelection(int row, int column, QString& oWc, QString& oTitle)
{
        //Build where clause and title
        QString wc;
        QString title;
        if (ui.kTitle->isVisible()) oTitle=ui.kTitle->text()+'\n';

        bool modeHistory=(ui.kMode->currentIndex()==1);

        //Condition on line attribute
        oTitle+=i18n("Operations with ");
        if (modeHistory || row+1!=ui.kTableWithGraph->table()->rowCount()) {
                QString att=AttsForLines[ui.kLines->currentIndex()];
                oWc=att;
                QString lineVal=ui.kTableWithGraph->table()->item(row, 0)->text();
                if (lineVal.isEmpty()) {
                        oWc+=" IS NULL OR "+att+"=''";
                        oWc='('+oWc+')';
                        oTitle+=i18n("%1 are empty", ui.kLines->currentText());
                } else {
                        oWc+=" = '"+SKGServices::stringToSqlString(lineVal)+"' OR "+
                             att+" like '"+SKGServices::stringToSqlString(lineVal)+OBJECTSEPARATOR+"%'";
                        oWc='('+oWc+')';
                        oTitle+=i18n("%1 with '%2'", ui.kLines->currentText(), lineVal);
                }
        }
        //Condition on column attribute
        int nbCol=ui.kTableWithGraph->table()->columnCount();
        if (modeHistory || (column!=0 && (nbCol==2 || (nbCol>=4 && column<nbCol-2)))) {
                if (!AttsForColumns[ui.kColumns->currentIndex()].isEmpty()) {
                        if (!oWc.isEmpty()) {
                                oWc+=" AND ";
                                oTitle+=i18n(" and ");
                        }
                        QString val=ui.kTableWithGraph->table()->horizontalHeaderItem(column)->text();
                        oWc+=AttsForColumns[ui.kColumns->currentIndex()]+"='"+ SKGServices::stringToSqlString(val)+'\'';
                        oTitle+=i18n("%1 with '%2'", ui.kColumns->currentText(), val);
                }
        }

        //Condition on other attribute
        if (!oWc.isEmpty()) {
                oWc+=" AND ";
                oTitle+=i18n(" and ");
        }
        oWc+=getConsolidatedWhereClause();
        oTitle+=i18n("during '%1' for '%2'",ui.kPeriod->currentText(), ui.kType->currentText());

}

void SKGReportPluginWidget::onDoubleClick(int row, int column)
{
        _SKGTRACEIN(10, "SKGReportPluginWidget::onDoubleClick");

        QString wc;
        QString title;
        getWhereClauseAndTitleForSelection(row, column, wc, title);

        //Open
        if (QApplication::keyboardModifiers()&Qt::ControlModifier && QApplication::keyboardModifiers()&Qt::ShiftModifier) {
                //Call debug plugin
                QDomDocument doc("SKGML");
                QDomElement root = doc.createElement("parameters");
                doc.appendChild(root);
                root.setAttribute("sqlOrder", "SELECT * from v_operation_consolidated WHERE "+wc);

                getMainPanel()->setNewTabContent(getMainPanel()->getPluginByName("Skrooge debug plugin"), -1, doc.toString());
        } else {
                //Call operation plugin
                QDomDocument doc("SKGML");
                doc.setContent ( getDocument()->getParameter("SKGOPERATION_DEFAULT_PARAMETERS") );
                QDomElement root = doc.documentElement();
                if (root.isNull()) {
                        root=doc.createElement("parameters");
                        doc.appendChild(root);
                }

                root.setAttribute("operationTable", "v_operation_consolidated");
                root.setAttribute("operationWhereClause", wc);
                root.setAttribute("title", title);
                root.setAttribute("title_icon", "view-statistics");
                root.setAttribute ( "currentPage", "-1" );

                getMainPanel()->setNewTabContent(getMainPanel()->getPluginByName("Skrooge operation plugin"), -1, doc.toString());
        }
}

void SKGReportPluginWidget::onOpenReport()
{
        SKGError err;
        SKGTRACEINRC(10, "SKGReportPluginWidget::onOpenReport",err);
        QList<QTableWidgetItem *> selection=ui.kTableWithGraph->table()->selectedItems();
        if (selection.count()) {
                QString wc;
                QString title;
                getWhereClauseAndTitleForSelection(selection.at(0)->row(), selection.at(0)->column(), wc, title);

                //Call report plugin
                QDomDocument doc("SKGML");
                doc.setContent(getState());
                QDomElement root = doc.documentElement();
                root.setAttribute("operationWhereClause", wc);
                root.setAttribute("title", title);
                root.setAttribute("title_icon", "view-statistics");
                getMainPanel()->setNewTabContent(getMainPanel()->getPluginByName("Skrooge report plugin"), -1, doc.toString());
        }
}

QString SKGReportPluginWidget::getConsolidatedWhereClause()
{
        //Build where clause
        QString wc;
        switch ( ui.kPeriod->currentIndex() ) {
        case 1:
                //Current month
                wc="d_DATEMONTH=STRFTIME('%Y-%m',date('now'))";
                break;

        case 2:
                //Previous month
                wc="d_DATEMONTH=STRFTIME('%Y-%m',date('now','-1 month'))";
                break;

        case 3:
                //Current year
                wc="d_DATEYEAR=STRFTIME('%Y',date('now'))";
                break;

        case 4:
                //Previous year
                wc="d_DATEYEAR=STRFTIME('%Y',date('now','-1 year'))";
                break;
        case 5:
                //30 last days
                wc="d_date>date('now','-30 day') AND d_date<=date('now')";
                break;
        case 6:
                //3 last months
                wc="d_date>date('now','-3 month') AND d_date<=date('now')";
                break;
        case 7:
                //6 last months
                wc="d_date>date('now','-6 month') AND d_date<=date('now')";
                break;
        case 8:
                //12 last months
                wc="d_date>date('now','-12 month') AND d_date<=date('now')";
                break;
        case 9:
                //2 last years
                wc="d_date>date('now','-2 year') AND d_date<=date('now')";
                break;
        case 10:
                //3 last years
                wc="d_date>date('now','-3 year') AND d_date<=date('now')";
                break;
        case 11:
                //5 last years
                wc="d_date>date('now','-5 year') AND d_date<=date('now')";
                break;
        default:
                wc="1=1";
                break;
        }

        switch ( ui.kType->currentIndex() ) {
        case 1:
        case 4:
                //Income
                wc+=" AND t_TYPEEXPENSE='+'";
                break;

        case 2:
        case 5:
                //Expenditure
                wc+=" AND t_TYPEEXPENSE='-'";
                break;

        default:
                break;
        }

        if (ui.kType->currentIndex()<=2) {
                //Withou transfers
                wc+=" AND i_group_id=0";
        }

        if (operationWhereClause.length()) wc='('+wc+") AND ("+operationWhereClause+')';

        return wc;
}

void SKGReportPluginWidget::onSelectionChanged()
{
        openReportAction->setEnabled(ui.kTableWithGraph->table()->selectedItems().count()>0);
}

void SKGReportPluginWidget::refresh()
{
        dataModified("", 0);
}

void SKGReportPluginWidget::dataModified(const QString& iTableName, int iIdTransaction)

{
        SKGTRACEIN(10, "SKGReportPluginWidget::dataModified");
        Q_UNUSED(iIdTransaction);

        //Refresh panel
        QSqlDatabase* db = getDocument()->getDatabase();
        setEnabled(db!=NULL);
        if (db!=NULL) {
                //Check if needed
                if (iTableName=="operation" || iTableName.isEmpty()) {
                        //Check if update is needed
                        QString ParametersUsed=getState()+';'+SKGServices::intToString(getDocument()->getTransactionToTreat());;
                        if (ParametersUsed==previousParametersUsed) {
                                SKGTRACEL(10) << "Same parameters. Refresh ignored" << endl;
                                return;
                        }
                        previousParametersUsed=ParametersUsed;

                        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

                        //Fill table
                        SKGError err;

                        //Get parameters
                        int col =ui.kColumns->currentIndex();
                        int line =ui.kLines->currentIndex();

                        if (col>=0 && line>=0) {
                                //Execute sql order
                                SKGStringListList table;
                                err=getDocument()->getConsolidatedView("v_operation_consolidated", AttsForColumns.at(col), AttsForLines.at(line),
                                                                       "f_REALCURRENTAMOUNT", "TOTAL", getConsolidatedWhereClause(), table);

                                IFSKGTRACEL(10) {
                                        QStringList dump=SKGServices::tableToDump(table, SKGServices::DUMP_TEXT);
                                        int nbl=dump.count();
                                        for (int i=0; i<nbl; ++i) {
                                                SKGTRACE << dump[i] << endl;
                                        }
                                }

                                if (err.isSucceeded()) {
                                        if (table.count()) {
                                                //Change title
                                                QStringList line1=table.at(0);
                                                line1.replace(0, ui.kLines->text());
                                                table.replace(0, line1);
                                        }

                                        bool modeHistory=(ui.kMode->currentIndex()==1);
                                        if (modeHistory) {
                                                //Build history
                                                SKGStringListList historizedTable;

                                                historizedTable.push_back(table.at(0));

                                                QStringList newLine;
                                                int nblines=table.count();
                                                int nbCols=0;
                                                if (nblines) {
                                                        nbCols=table.at(0).count();
                                                }
                                                for (int i=1; i<nblines; ++i) {
                                                        QStringList newLine;
                                                        newLine.push_back(table.at(i).at(0));

                                                        double sum=0;
                                                        for (int j=1; j<nbCols; ++j) {
                                                                sum+=SKGServices::stringToDouble(table.at(i).at(j));
                                                                newLine.push_back(SKGServices::doubleToString(sum));
                                                        }
                                                        historizedTable.push_back(newLine);
                                                }

                                                table=historizedTable;
                                        }

                                        QString primaryUnit=((SKGDocumentBank*) getDocument())->getPrimaryUnit();
                                        QString secondaryUnit=((SKGDocumentBank*) getDocument())->getSecondaryUnit();
                                        double secondaryUnitValue=((SKGDocumentBank*) getDocument())->getSecondaryUnitValue();
                                        ui.kTableWithGraph->setData(table, primaryUnit, secondaryUnit, secondaryUnitValue, !modeHistory);
                                }
                        }


                        QApplication::restoreOverrideCursor();

                        //Display error
                        getMainPanel()->displayErrorMessage(err);
                }
        }
}

#include "skgreportpluginwidget.moc"
